// Copyright 2016 The Chromium Authors
// Use of this source code is governed by a BSD-style license that can be
// found in the LICENSE file.
/**
 * @fileoverview
 * Responds to route changes by "activating" the respective top-level page,
 * effectively making that page visible to the user and potentially hiding other
 * pages.
 */
import { assert, assertNotReached } from 'chrome://resources/js/assert.js';
import { beforeNextRender, dedupingMixin } from 'chrome://resources/polymer/v3_0/polymer/polymer_bundled.min.js';
import { castExists } from '../assert_extras.js';
import { RouteObserverMixin } from '../common/route_observer_mixin.js';
import { ensureLazyLoaded } from '../ensure_lazy_loaded.js';
import { Router, routes } from '../router.js';
/**
 * A categorization of every possible Settings URL, necessary for implementing
 * a finite state machine.
 */
var RouteState;
(function (RouteState) {
    // Initial state before anything has loaded yet.
    RouteState["INITIAL"] = "initial";
    // The root Settings page, '/'.
    RouteState["ROOT"] = "root";
    // A section, basically a scroll position within the root page.
    // After infinite scroll is removed, this is a top-level page.
    // e.g. /network, /bluetooth, /device
    RouteState["SECTION"] = "section";
    // A subpage, or nested subpage, e.g. /networkDetail.
    RouteState["SUBPAGE"] = "subpage";
    // A navigable dialog that has a dedicated URL. Currently unused in Settings.
    RouteState["DIALOG"] = "dialog";
})(RouteState || (RouteState = {}));
function classifyRoute(route) {
    if (!route) {
        return RouteState.INITIAL;
    }
    if (route === routes.BASIC) {
        return RouteState.ROOT;
    }
    if (route.isSubpage()) {
        return RouteState.SUBPAGE;
    }
    if (route.isNavigableDialog) {
        return RouteState.DIALOG;
    }
    return RouteState.SECTION;
}
const ALL_STATES = new Set([
    RouteState.DIALOG,
    RouteState.SECTION,
    RouteState.SUBPAGE,
    RouteState.ROOT,
]);
/**
 * A map holding all valid state transitions.
 */
const VALID_TRANSITIONS = new Map([
    [RouteState.INITIAL, ALL_STATES],
    [
        RouteState.DIALOG,
        new Set([
            RouteState.SECTION,
            RouteState.SUBPAGE,
            RouteState.ROOT,
        ]),
    ],
    [RouteState.SECTION, ALL_STATES],
    [RouteState.SUBPAGE, ALL_STATES],
    [RouteState.ROOT, ALL_STATES],
]);
/**
 * The route for the first page listed in the Settings menu.
 */
const FIRST_PAGE_ROUTE = routes.INTERNET;
export const MainPageMixin = dedupingMixin((superClass) => {
    const superclassBase = RouteObserverMixin(superClass);
    class MainPageMixinInternal extends superclassBase {
        constructor() {
            super(...arguments);
            this.lastScrollTop_ = 0;
        }
        /**
         * The scroller is derived from the #container ancestor element.
         */
        get scroller_() {
            const hostEl = this.getRootNode().host;
            return castExists(hostEl ? hostEl.closest('#container') : document.body);
        }
        /**
         * Method to be overridden by users of MainPageMixin.
         * @return Whether the given route is part of |this| page.
         */
        containsRoute(_route) {
            assertNotReached();
        }
        async enterSubpage(route) {
            // Immediately record the last scroll position before continuing.
            this.lastScrollTop_ = this.scroller_.scrollTop;
            // Make the parent page visible to ensure the subpage is visible
            if (route === routes.BLUETOOTH_DEVICES) {
                // The Bluetooth subpage (L2) acts as the top-level (L1) page, so
                // focus it when navigating to it. See crbug.com/328315423 for more
                // context.
                await this.activatePage(route, { focus: true });
            }
            else {
                await this.activatePage(route);
            }
            this.scroller_.scrollTop = 0;
            this.classList.add('showing-subpage');
            this.dispatchCustomEvent_('showing-subpage');
            // Explicitly load the lazy_load module, since all subpages reside in
            // the lazy loaded module.
            ensureLazyLoaded();
            this.dispatchCustomEvent_('show-container');
        }
        /**
         * Indicates the page transition of leaving a subpage and entering the
         * main page by emitting a `showing-main-page` event.
         * If the page transition was a pop state (e.g. clicking back button on
         * a subpage), then the cached scroll position on the main page is
         * restored.
         */
        enterMainPage() {
            this.classList.remove('showing-subpage');
            return new Promise((resolve) => {
                requestAnimationFrame(() => {
                    if (Router.getInstance().lastRouteChangeWasPopstate()) {
                        this.scroller_.scrollTop = this.lastScrollTop_;
                    }
                    this.dispatchCustomEvent_('showing-main-page');
                    resolve();
                });
            });
        }
        /**
         * Simple helper method to display a page/section.
         */
        showPage(route) {
            this.activatePage(route, { focus: true });
        }
        /**
         * Effectively displays the page for the given |route| by marking the
         * respective page-displayer element as active, and hides all other
         * pages by marking them as inactive. Also, optionally transfers focus
         * to the page content.
         */
        async activatePage(route, options = {}) {
            const page = await this.ensurePageForRoute(route);
            const previouslyActive = this.shadowRoot.querySelectorAll('page-displayer[active]');
            for (const prevPage of previouslyActive) {
                prevPage.active = false;
            }
            page.active = true;
            if (options.focus) {
                page.focus();
            }
            this.dispatchCustomEvent_('show-container');
        }
        /**
         * Activate and display the first page (Network page). This page
         * should be the default visible page when the root page is visited.
         */
        activateInitialPage() {
            // Note: This should not focus the Network page since the search box
            // should be the element initially focused after app load.
            this.activatePage(FIRST_PAGE_ROUTE, { focus: false });
        }
        /**
         * Detects which state transition is appropriate for the given new/old
         * routes.
         */
        getStateTransition_(newRoute, oldRoute) {
            const containsNew = this.containsRoute(newRoute);
            const containsOld = this.containsRoute(oldRoute);
            if (!containsNew && !containsOld) {
                // Nothing to do, since none of the old/new routes belong to this
                // page.
                return null;
            }
            // Case where going from |this| page to an unrelated page.
            // For example:
            //  |this| is main-page-container AND
            //  oldRoute is /searchEngines AND
            //  newRoute is /help.
            if (containsOld && !containsNew) {
                return [classifyRoute(oldRoute), RouteState.ROOT];
            }
            // Case where return from an unrelated page to |this| page.
            // For example:
            //  |this| is main-page-container AND
            //  oldRoute is /help AND
            //  newRoute is /searchEngines
            if (!containsOld && containsNew) {
                return [RouteState.ROOT, classifyRoute(newRoute)];
            }
            // Case where transitioning between routes that both belong to |this|
            // page.
            return [classifyRoute(oldRoute), classifyRoute(newRoute)];
        }
        currentRouteChanged(newRoute, oldRoute) {
            const transition = this.getStateTransition_(newRoute, oldRoute);
            if (transition === null) {
                return;
            }
            const [oldState, newState] = transition;
            assert(VALID_TRANSITIONS.get(oldState).has(newState));
            if (oldState === RouteState.INITIAL) {
                switch (newState) {
                    case RouteState.SECTION:
                        this.showPage(newRoute);
                        return;
                    case RouteState.SUBPAGE:
                        this.enterSubpage(newRoute);
                        return;
                    case RouteState.ROOT:
                        this.activateInitialPage();
                        return;
                    // Nothing to do here for the DIALOG case.
                    case RouteState.DIALOG:
                    default:
                        return;
                }
            }
            if (oldState === RouteState.ROOT) {
                switch (newState) {
                    case RouteState.SECTION:
                        this.showPage(newRoute);
                        return;
                    // Navigating directly to a subpage via search on the main page
                    case RouteState.SUBPAGE:
                        this.enterSubpage(newRoute);
                        return;
                    // Happens when clearing search results (Navigating from
                    // '/?search=foo' to '/')
                    case RouteState.ROOT:
                        this.activateInitialPage();
                        return;
                    // Nothing to do here for the DIALOG case.
                    case RouteState.DIALOG:
                    default:
                        return;
                }
            }
            if (oldState === RouteState.SECTION) {
                switch (newState) {
                    case RouteState.SECTION:
                        this.showPage(newRoute);
                        return;
                    case RouteState.SUBPAGE:
                        this.enterSubpage(newRoute);
                        return;
                    case RouteState.ROOT:
                        this.scroller_.scrollTop = 0;
                        this.activateInitialPage();
                        return;
                    // Nothing to do here for the case of DIALOG.
                    case RouteState.DIALOG:
                    default:
                        return;
                }
            }
            if (oldState === RouteState.SUBPAGE) {
                assert(oldRoute);
                switch (newState) {
                    case RouteState.SECTION:
                        this.enterMainPage().then(() => {
                            this.activatePage(newRoute, { focus: true });
                        });
                        return;
                    case RouteState.SUBPAGE:
                        // Handle case where the two subpages belong to
                        // different sections, but are linked to each other. For example
                        // /displayAndMagnification linking to /display
                        if (!oldRoute.contains(newRoute) &&
                            !newRoute.contains(oldRoute)) {
                            this.enterMainPage().then(() => {
                                this.enterSubpage(newRoute);
                            });
                            return;
                        }
                        // Handle case of subpage to nested subpage navigation.
                        if (oldRoute.contains(newRoute)) {
                            this.scroller_.scrollTop = 0;
                            return;
                        }
                        // When going from a nested subpage to its parent subpage,
                        // the scroll position is automatically restored because we
                        // focus the nested subpage's entry point.
                        return;
                    // Happens when the user navigates to a subpage via the search box
                    // on the root page, and then clicks the back button.
                    case RouteState.ROOT:
                        this.enterMainPage().then(() => {
                            this.activateInitialPage();
                        });
                        return;
                    // This is a supported case but there are currently no known
                    // examples of this transition in Settings.
                    case RouteState.DIALOG:
                        this.enterMainPage();
                        return;
                    default:
                        return;
                }
            }
            if (oldState === RouteState.DIALOG) {
                switch (newState) {
                    // There are currently no known examples of this transition
                    case RouteState.SUBPAGE:
                        this.enterSubpage(newRoute);
                        return;
                    // There are currently no known examples of these transitions.
                    // Update when a relevant use-case exists.
                    case RouteState.ROOT:
                    case RouteState.SECTION:
                    case RouteState.DIALOG:
                    default:
                        return;
                }
            }
        }
        /**
         * Finds the settings page corresponding to the given route.
         */
        ensurePageForRoute(route) {
            const section = this.queryPage(route.section);
            if (section) {
                return Promise.resolve(section);
            }
            // The function to use to wait for <dom-if>s to render.
            const waitFn = beforeNextRender.bind(null, this);
            return new Promise(resolve => {
                waitFn(() => {
                    resolve(castExists(this.queryPage(route.section)));
                });
            });
        }
        /**
         * Queries for a page visibility element with the given |section| from
         * the shadow DOM.
         */
        queryPage(section) {
            if (section === null) {
                return null;
            }
            return this.shadowRoot.querySelector(`page-displayer[section="${section}"]`);
        }
        dispatchCustomEvent_(name, options) {
            const event = new CustomEvent(name, { bubbles: true, composed: true, ...options });
            this.dispatchEvent(event);
        }
    }
    return MainPageMixinInternal;
});
