// Copyright 2020 The Chromium Authors
// Use of this source code is governed by a BSD-style license that can be
// found in the LICENSE file.
/**
 * @fileoverview
 * 'settings-nearby-share-subpage' is the settings subpage for managing the
 * Nearby Share feature.
 */
import 'chrome://resources/ash/common/cr_elements/cr_shared_style.css.js';
import 'chrome://resources/ash/common/cr_elements/cr_link_row/cr_link_row.js';
import 'chrome://resources/polymer/v3_0/iron-flex-layout/iron-flex-layout-classes.js';
import '../controls/settings_toggle_button.js';
import '../settings_shared.css.js';
import './nearby_share_contact_visibility_dialog.js';
import './nearby_share_device_name_dialog.js';
import './nearby_share_data_usage_dialog.js';
import './nearby_share_receive_dialog.js';
import { getContactManager } from '/shared/nearby_contact_manager.js';
import { PrefsMixin } from '/shared/settings/prefs/prefs_mixin.js';
import { I18nMixin } from 'chrome://resources/ash/common/cr_elements/i18n_mixin.js';
import { assertNotReached } from 'chrome://resources/js/assert.js';
import { loadTimeData } from 'chrome://resources/js/load_time_data.js';
import { DataUsage, FastInitiationNotificationState, Visibility } from 'chrome://resources/mojo/chromeos/ash/services/nearby/public/mojom/nearby_share_settings.mojom-webui.js';
import { flush, PolymerElement } from 'chrome://resources/polymer/v3_0/polymer/polymer_bundled.min.js';
import { DeepLinkingMixin } from '../common/deep_linking_mixin.js';
import { RouteObserverMixin } from '../common/route_observer_mixin.js';
import { Setting } from '../mojom-webui/setting.mojom-webui.js';
import { Router, routes } from '../router.js';
import { NearbyAccountManagerBrowserProxyImpl } from './nearby_account_manager_browser_proxy.js';
import { observeReceiveManager } from './nearby_share_receive_manager.js';
import { getTemplate } from './nearby_share_subpage.html.js';
import { dataUsageStringToEnum } from './types.js';
const DEFAULT_HIGH_VISIBILITY_TIMEOUT_LEGACY_S = 300;
const DEFAULT_HIGH_VISIBILITY_TIMEOUT_S = 600;
const SettingsNearbyShareSubpageElementBase = DeepLinkingMixin(PrefsMixin(RouteObserverMixin(I18nMixin(PolymerElement))));
export class SettingsNearbyShareSubpageElement extends SettingsNearbyShareSubpageElementBase {
    static get is() {
        return 'settings-nearby-share-subpage';
    }
    static get template() {
        return getTemplate();
    }
    static get properties() {
        return {
            profileName_: {
                type: String,
                value: '',
            },
            profileLabel_: {
                type: String,
                value: '',
            },
            settings: {
                type: Object,
                notify: true,
                value: {},
            },
            isSettingsRetreived: {
                type: Boolean,
                value: false,
            },
            showDeviceNameDialog_: {
                type: Boolean,
                value: false,
            },
            showVisibilityDialog_: {
                type: Boolean,
                value: false,
            },
            showDataUsageDialog_: {
                type: Boolean,
                value: false,
            },
            showReceiveDialog_: {
                type: Boolean,
                value: false,
            },
            manageContactsUrl_: {
                type: String,
                value: () => loadTimeData.getString('nearbyShareManageContactsUrl'),
            },
            inHighVisibility_: {
                type: Boolean,
                value: false,
            },
            /**
             * Determines whether the QuickShareV2 flag is enabled.
             */
            isQuickShareV2Enabled_: {
                type: Boolean,
                value: () => loadTimeData.getBoolean('isQuickShareV2Enabled'),
            },
            shouldShowFastInititationNotificationToggle_: {
                type: Boolean,
                computed: `computeShouldShowFastInititationNotificationToggle_(
                settings.isFastInitiationHardwareSupported)`,
            },
            yourDevicesLabel_: {
                type: String,
                value: 'Your devices',
            },
            contactsLabel_: {
                type: String,
                value: 'Contacts',
            },
        };
    }
    static get observers() {
        return [
            'enabledChange_(settings.enabled)',
        ];
    }
    constructor() {
        super();
        // DeepLinkingMixin override
        this.supportedSettingIds = new Set([
            Setting.kNearbyShareOnOff,
            Setting.kNearbyShareDeviceName,
            Setting.kNearbyShareDeviceVisibility,
            Setting.kNearbyShareContacts,
            Setting.kNearbyShareDataUsage,
            Setting.kDevicesNearbyAreSharingNotificationOnOff,
        ]);
        this.receiveObserver_ = null;
    }
    ready() {
        super.ready();
        this.addEventListener('onboarding-cancelled', this.onOnboardingCancelled_);
    }
    connectedCallback() {
        super.connectedCallback();
        // TODO(b/166779043): Check whether the Account Manager is enabled and fall
        // back to profile name, or just hide the row. This is not urgent because
        // the Account Manager should be available whenever Nearby Share is enabled.
        NearbyAccountManagerBrowserProxyImpl.getInstance().getAccounts().then(accounts => {
            if (accounts.length === 0) {
                return;
            }
            this.profileName_ = accounts[0].fullName;
            this.profileLabel_ = accounts[0].email;
        });
        this.receiveObserver_ = observeReceiveManager(this);
    }
    enabledChange_(newValue, oldValue) {
        if (oldValue === undefined && newValue) {
            // Trigger a contact sync whenever the Nearby subpage is opened and
            // nearby is enabled complete to improve consistency. This should help
            // avoid scenarios where a share is attempted and contacts are stale on
            // the receiver.
            getContactManager().downloadContacts();
        }
    }
    onDeviceNameClick_() {
        if (this.showDeviceNameDialog_) {
            return;
        }
        this.showDeviceNameDialog_ = true;
    }
    onVisibilityClick_() {
        this.showVisibilityDialog_ = true;
    }
    onDataUsageClick_() {
        this.showDataUsageDialog_ = true;
    }
    onDeviceNameDialogClose_() {
        this.showDeviceNameDialog_ = false;
    }
    onVisibilityDialogClose_() {
        this.showVisibilityDialog_ = false;
    }
    onDataUsageDialogClose_() {
        this.showDataUsageDialog_ = false;
    }
    onReceiveDialogClose_() {
        this.showReceiveDialog_ = false;
        this.inHighVisibility_ = false;
    }
    onManageContactsClick_() {
        window.open(this.manageContactsUrl_);
    }
    getManageContactsSubLabel_() {
        // Remove the protocol part of the contacts url.
        return this.manageContactsUrl_.replace(/(^\w+:|^)\/\//, '');
    }
    /**
     * Mojo callback when high visibility changes.
     */
    onHighVisibilityChanged(inHighVisibility) {
        this.inHighVisibility_ = inHighVisibility;
    }
    /**
     * Mojo callback when transfer status changes.
     */
    onTransferUpdate(_shareTarget, _metadata) {
        // Note: Intentionally left empty.
    }
    /**
     * Mojo callback when the Nearby utility process stops.
     */
    onNearbyProcessStopped() {
        this.inHighVisibility_ = false;
    }
    /**
     * Mojo callback when advertising fails to start.
     */
    onStartAdvertisingFailure() {
        this.inHighVisibility_ = false;
    }
    onInHighVisibilityToggledByUser_() {
        if (this.inHighVisibility_) {
            this.showHighVisibilityPage_();
        }
    }
    /**
     * @param state boolean state that determines which string to show
     * @param onstr string to show when state is true
     * @param offstr string to show when state is false
     * @return localized string
     */
    getOnOffString_(state, onstr, offstr) {
        return state ? onstr : offstr;
    }
    getEditNameButtonAriaDescription_(name) {
        return this.i18n('nearbyShareDeviceNameAriaDescription', name);
    }
    getVisibilityText_(visibility) {
        if (visibility === undefined) {
            return '';
        }
        if (this.isQuickShareV2Enabled_) {
            switch (visibility) {
                case Visibility.kAllContacts:
                    return this.i18n('nearbyShareContactVisiblityContactsButton');
                case Visibility.kNoOne:
                    return this.i18n('nearbyShareContactVisibilityNone');
                case Visibility.kUnknown:
                    return this.i18n('nearbyShareContactVisibilityUnknown');
                case Visibility.kYourDevices:
                    return this.i18n('nearbyShareContactVisibilityYourDevices');
                default:
                    assertNotReached();
            }
        }
        switch (visibility) {
            case Visibility.kAllContacts:
                return this.i18n('nearbyShareContactVisibilityAll');
            case Visibility.kSelectedContacts:
                return this.i18n('nearbyShareContactVisibilitySome');
            case Visibility.kNoOne:
                return this.i18n('nearbyShareContactVisibilityNone');
            case Visibility.kUnknown:
                return this.i18n('nearbyShareContactVisibilityUnknown');
            case Visibility.kYourDevices:
                return this.i18n('nearbyShareContactVisibilityYourDevices');
            default:
                assertNotReached();
        }
    }
    getVisibilityDescription_(visibility) {
        if (visibility === undefined) {
            return '';
        }
        switch (visibility) {
            case Visibility.kAllContacts:
                return this.i18n('nearbyShareContactVisibilityAllDescription');
            case Visibility.kSelectedContacts:
                return this.i18n('nearbyShareContactVisibilitySomeDescription');
            case Visibility.kNoOne:
                return this.i18n('nearbyShareContactVisibilityNoneDescription');
            case Visibility.kUnknown:
                return this.i18n('nearbyShareContactVisibilityUnknownDescription');
            case Visibility.kYourDevices:
                return this.i18n('nearbyShareContactVisibilityYourDevicesDescription');
            default:
                assertNotReached();
        }
    }
    getHighVisibilityToggleText_(inHighVisibility) {
        // TODO(crbug.com/40159645): Add logic to show how much time the user
        // actually has left.
        return inHighVisibility ?
            this.i18n('nearbyShareHighVisibilityOn', this.isQuickShareV2Enabled_ ? 10 : 5) :
            this.i18nAdvanced('nearbyShareHighVisibilityOff', { substitutions: [this.isQuickShareV2Enabled_ ? '10' : '5'] });
    }
    getDataUsageLabel_(dataUsageValue) {
        if (dataUsageStringToEnum(dataUsageValue) === DataUsage.kOnline) {
            return this.i18n('nearbyShareDataUsageDataLabel');
        }
        else if (dataUsageStringToEnum(dataUsageValue) === DataUsage.kOffline) {
            return this.i18n('nearbyShareDataUsageOfflineLabel');
        }
        else {
            return this.i18n('nearbyShareDataUsageWifiOnlyLabel');
        }
    }
    getDataUsageSubLabel_(dataUsageValue) {
        if (dataUsageStringToEnum(dataUsageValue) === DataUsage.kOnline) {
            return this.i18n('nearbyShareDataUsageDataDescription');
        }
        else if (dataUsageStringToEnum(dataUsageValue) === DataUsage.kOffline) {
            return this.i18n('nearbyShareDataUsageOfflineDescription');
        }
        else {
            return this.i18n('nearbyShareDataUsageWifiOnlyDescription');
        }
    }
    getEditDataUsageButtonAriaDescription_(dataUsageValue) {
        if (dataUsageStringToEnum(dataUsageValue) === DataUsage.kOnline) {
            return this.i18n('nearbyShareDataUsageDataEditButtonDescription');
        }
        else if (dataUsageStringToEnum(dataUsageValue) === DataUsage.kOffline) {
            return this.i18n('nearbyShareDataUsageOfflineEditButtonDescription');
        }
        else {
            return this.i18n('nearbyShareDataUsageWifiOnlyEditButtonDescription');
        }
    }
    currentRouteChanged(route) {
        // Does not apply to this page.
        if (route !== routes.NEARBY_SHARE) {
            return;
        }
        const router = Router.getInstance();
        const queryParams = router.getQueryParameters();
        if (queryParams.has('deviceName')) {
            this.showDeviceNameDialog_ = true;
        }
        if (queryParams.has('visibility')) {
            this.showVisibilityDialog_ = true;
        }
        if (queryParams.has('receive')) {
            this.showHighVisibilityPage_(Number(queryParams.get('timeout')));
        }
        if (queryParams.has('confirm')) {
            this.showReceiveDialog_ = true;
            flush();
            this.shadowRoot
                .querySelector('#receiveDialog').showConfirmPage();
        }
        if (queryParams.has('onboarding')) {
            this.showOnboarding_();
        }
        this.attemptDeepLink();
    }
    showHighVisibilityPage_(timeoutInSeconds) {
        const defaultTimeout = this.isQuickShareV2Enabled_ ?
            DEFAULT_HIGH_VISIBILITY_TIMEOUT_S :
            DEFAULT_HIGH_VISIBILITY_TIMEOUT_LEGACY_S;
        const shutoffTimeoutInSeconds = timeoutInSeconds || defaultTimeout;
        this.showReceiveDialog_ = true;
        flush();
        this.shadowRoot
            .querySelector('#receiveDialog').showHighVisibilityPage(shutoffTimeoutInSeconds);
    }
    getAccountRowLabel(profileName, profileLabel) {
        return this.i18n('nearbyShareAccountRowLabel', this.i18n('nearbyShareFeatureName'), profileName, profileLabel);
    }
    onOnboardingCancelled_() {
        // Return to main settings page multidevice section
        Router.getInstance().navigateTo(routes.MULTIDEVICE);
    }
    onFastInitiationNotificationToggledByUser_() {
        this.set('settings.fastInitiationNotificationState', this.isFastInitiationNotificationEnabled_() ?
            FastInitiationNotificationState.kDisabledByUser :
            FastInitiationNotificationState.kEnabled);
    }
    isFastInitiationNotificationEnabled_() {
        return this.get('settings.fastInitiationNotificationState') ===
            FastInitiationNotificationState.kEnabled;
    }
    shouldShowSubpageContent_(isNearbySharingEnabled, isOnboardingComplete, shouldShowFastInititationNotificationToggle) {
        if (!isOnboardingComplete) {
            return false;
        }
        return isNearbySharingEnabled ||
            shouldShowFastInititationNotificationToggle;
    }
    showOnboarding_() {
        this.showReceiveDialog_ = true;
        flush();
        this.shadowRoot
            .querySelector('#receiveDialog').showOnboarding();
    }
    computeShouldShowFastInititationNotificationToggle_(isHardwareSupported) {
        return isHardwareSupported;
    }
    setVisibility_(visibility) {
        this.set('settings.visibility', visibility);
    }
    getVisibilityByLabel_(visibilityString) {
        switch (visibilityString) {
            case this.yourDevicesLabel_:
                return Visibility.kYourDevices;
            case this.contactsLabel_:
                return Visibility.kAllContacts;
            default:
                return Visibility.kUnknown;
        }
    }
}
customElements.define(SettingsNearbyShareSubpageElement.is, SettingsNearbyShareSubpageElement);
