// Copyright 2024 The Chromium Authors
// Use of this source code is governed by a BSD-style license that can be
// found in the LICENSE file.
/**
 * @fileoverview 'facegaze-actions-add-dialog' is a dialog for
 * adding an action to FaceGaze.
 */
import 'chrome://resources/ash/common/cr_elements/cr_button/cr_button.js';
import 'chrome://resources/ash/common/cr_elements/cr_dialog/cr_dialog.js';
import 'chrome://resources/ash/common/cr_elements/cr_slider/cr_slider.js';
import 'chrome://resources/ash/common/shortcut_input_ui/shortcut_input.js';
import 'chrome://resources/polymer/v3_0/iron-flex-layout/iron-flex-layout-classes.js';
import 'chrome://resources/polymer/v3_0/iron-list/iron-list.js';
import 'chrome://resources/polymer/v3_0/paper-ripple/paper-ripple.js';
import '../settings_shared.css.js';
import './facegaze_icons.html.js';
import { PrefsMixin } from '/shared/settings/prefs/prefs_mixin.js';
import { MacroName } from 'chrome://resources/ash/common/accessibility/macro_names.js';
import { CrScrollableMixin } from 'chrome://resources/ash/common/cr_elements/cr_scrollable_mixin.js';
import { I18nMixin } from 'chrome://resources/ash/common/cr_elements/i18n_mixin.js';
import { WebUiListenerMixin } from 'chrome://resources/ash/common/cr_elements/web_ui_listener_mixin.js';
import { ModifierKeyCodes } from 'chrome://resources/ash/common/shortcut_input_ui/shortcut_utils.js';
import { assert } from 'chrome://resources/js/assert.js';
import { EventTracker } from 'chrome://resources/js/event_tracker.js';
import { PolymerElement } from 'chrome://resources/polymer/v3_0/polymer/polymer_bundled.min.js';
import { getShortcutInputProvider } from '../device_page/shortcut_input_mojo_interface_provider.js';
import { getTemplate } from './facegaze_actions_add_dialog.html.js';
import { AssignedKeyCombo, ComplexActions, ConflictingGestures, FACE_GAZE_GESTURE_TO_CONFIDENCE_PREF, FACE_GAZE_GESTURE_TO_CONFIDENCE_PREF_DICT, FACE_GAZE_GESTURE_TO_MACROS_PREF, FACEGAZE_COMMAND_PAIR_ADDED_EVENT_NAME, FaceGazeActions, FaceGazeCommandPair, FaceGazeGestures, FaceGazeLocationDependentActions, FaceGazeLookGestures, FaceGazeUtils } from './facegaze_constants.js';
import { FaceGazeSubpageBrowserProxyImpl } from './facegaze_subpage_browser_proxy.js';
export var AddDialogPage;
(function (AddDialogPage) {
    AddDialogPage[AddDialogPage["SELECT_ACTION"] = 0] = "SELECT_ACTION";
    AddDialogPage[AddDialogPage["SELECT_GESTURE"] = 1] = "SELECT_GESTURE";
    AddDialogPage[AddDialogPage["GESTURE_THRESHOLD"] = 2] = "GESTURE_THRESHOLD";
    AddDialogPage[AddDialogPage["CUSTOM_KEYBOARD"] = 3] = "CUSTOM_KEYBOARD";
})(AddDialogPage || (AddDialogPage = {}));
export var Navigation;
(function (Navigation) {
    Navigation[Navigation["PREVIOUS"] = 0] = "PREVIOUS";
    Navigation[Navigation["NEXT"] = 1] = "NEXT";
})(Navigation || (Navigation = {}));
export class PageNavigation {
}
export class FaceGazeGestureConfidence {
}
export const FACEGAZE_DEFINED_MACRO_FLOW = {
    [AddDialogPage.SELECT_ACTION]: {
        next: AddDialogPage.SELECT_GESTURE,
    },
    [AddDialogPage.SELECT_GESTURE]: {
        previous: AddDialogPage.SELECT_ACTION,
        next: AddDialogPage.GESTURE_THRESHOLD,
    },
    [AddDialogPage.GESTURE_THRESHOLD]: {
        previous: AddDialogPage.SELECT_GESTURE,
    },
    [AddDialogPage.CUSTOM_KEYBOARD]: {},
};
export const FACEGAZE_CUSTOM_KEYBOARD_SHORTCUT_FLOW = {
    [AddDialogPage.SELECT_ACTION]: {
        next: AddDialogPage.CUSTOM_KEYBOARD,
    },
    [AddDialogPage.SELECT_GESTURE]: {
        previous: AddDialogPage.CUSTOM_KEYBOARD,
        next: AddDialogPage.GESTURE_THRESHOLD,
    },
    [AddDialogPage.GESTURE_THRESHOLD]: {
        previous: AddDialogPage.SELECT_GESTURE,
    },
    [AddDialogPage.CUSTOM_KEYBOARD]: {
        previous: AddDialogPage.SELECT_ACTION,
        next: AddDialogPage.SELECT_GESTURE,
    },
};
export const FACEGAZE_CONFIDENCE_DEFAULT = 50;
export const FACEGAZE_CONFIDENCE_MIN = 1;
export const FACEGAZE_CONFIDENCE_MAX = 100;
export const FACEGAZE_CONFIDENCE_BUTTON_STEP = 5;
const FaceGazeAddActionDialogElementBase = PrefsMixin(I18nMixin(CrScrollableMixin(WebUiListenerMixin(PolymerElement))));
export class FaceGazeAddActionDialogElement extends FaceGazeAddActionDialogElementBase {
    static get is() {
        return 'facegaze-actions-add-dialog';
    }
    static get template() {
        return getTemplate();
    }
    static get properties() {
        return {
            currentPage_: {
                type: Number,
                observer: 'currentPageChanged_',
            },
            initialPage: {
                type: Object,
                observer: 'initialPageChanged_',
            },
            commandPairToConfigure: {
                type: Object,
                observer: 'commandPairToConfigureChanged_',
            },
            leftClickGestures: {
                type: Array,
                value: () => [],
            },
            showSelectAction_: {
                type: Boolean,
                computed: 'shouldShowSelectAction_(currentPage_)',
            },
            showCustomKeyboard_: {
                type: Boolean,
                computed: 'shouldShowCustomKeyboard_(currentPage_)',
            },
            showSelectGesture_: {
                type: Boolean,
                computed: 'shouldShowSelectGesture_(currentPage_)',
            },
            showGestureThreshold_: {
                type: Boolean,
                computed: 'shouldShowGestureThreshold_(currentPage_)',
            },
            displayedActions_: {
                type: Array,
                value: () => [],
            },
            selectedAction_: {
                type: Object,
                value: null,
                observer: 'onSelectedActionChanged_',
            },
            keyCombination_: {
                type: Object,
            },
            shortcutInputLabel_: {
                type: String,
                computed: 'getShortcutInputLabel_(keyCombination_)',
            },
            keyComboChangeButtonLabel_: {
                type: String,
                computed: 'getKeyComboChangeButtonLabel_(keyCombination_)',
            },
            localizedSelectGestureTitle_: {
                type: String,
                computed: 'getLocalizedSelectGestureTitle_(selectedAction_)',
            },
            displayedGestures_: {
                type: Array,
                computed: 'computeAllowedGestures_(selectedAction_, leftClickGestures)',
            },
            selectedGesture_: {
                type: Object,
                value: null,
                observer: 'onSelectedGestureChanged_',
            },
            localizedGestureThresholdTitle_: {
                type: String,
                computed: 'getLocalizedGestureThresholdTitle_(selectedGesture_)',
            },
            localizedGestureCountLabel_: {
                type: String,
                computed: 'getLocalizedGestureCountLabel_(detectedGestureCount_)',
            },
            gestureThresholdValue_: {
                type: Number,
                observer: 'onGestureThresholdChanged_',
            },
            detectedGestureCount_: {
                type: Number,
            },
            disableActionNextButton_: {
                type: Boolean,
                computed: 'shouldDisableActionNextButton_(selectedAction_)',
            },
            disableCustomKeyboardNextButton_: {
                type: Boolean,
                computed: 'shouldDisableCustomKeyboardNextButton_(keyCombination_)',
            },
            disableGestureNextButton_: {
                type: Boolean,
                computed: 'shouldDisableGestureNextButton_(selectedGesture_)',
            },
            displayGesturePreviousButton_: {
                type: Boolean,
                computed: 'shouldDisplayGesturePreviousButton_(initialPage)',
            },
            displayThresholdPreviousButton_: {
                type: Boolean,
                computed: 'shouldDisplayThresholdPreviousButton_(initialPage)',
            },
        };
    }
    constructor() {
        super();
        this.initialPage = AddDialogPage.SELECT_ACTION;
        this.commandPairToConfigure = null;
        this.leftClickGestures = [];
        this.keyCombination_ = null;
        this.selectedGesture_ = null;
        this.currentPage_ = AddDialogPage.SELECT_ACTION;
        this.pageNavigation_ = FACEGAZE_DEFINED_MACRO_FLOW;
        this.detectedGestureCount_ = 0;
        this.eventTracker_ = new EventTracker();
        this.holdingGesture_ = false;
        this.displayedActions_ = FaceGazeActions;
        this.faceGazeSubpageBrowserProxy_ =
            FaceGazeSubpageBrowserProxyImpl.getInstance();
        this.addWebUiListener('settings.sendGestureInfoToSettings', (gestureConfidences) => this.onGestureConfidencesReceived_(gestureConfidences));
        this.eventTracker_.add(this, 'shortcut-input-event', this.onShortcutInputEvent_);
    }
    getItemClass_(selected) {
        return selected ? 'selected' : '';
    }
    getShortcutInputProvider() {
        return getShortcutInputProvider();
    }
    onShortcutInputEvent_(e) {
        this.keyCombination_ = this.formatKeyCombination_(e.detail.keyEvent);
        if (this.keyCombination_) {
            this.shortcutInput.stopObserving();
            const changeButton = this.shadowRoot.querySelector('#faceGazeCustomKeyboardChangeButton');
            changeButton.focus();
        }
    }
    formatKeyCombination_(keyEvent) {
        if (!this.shortcutInput) {
            return null;
        }
        // Do not support a key combination consisting of only a modifier key.
        if (ModifierKeyCodes.includes(keyEvent.vkey)) {
            return null;
        }
        const newKeyCombination = { key: keyEvent.vkey, keyDisplay: keyEvent.keyDisplay };
        const modifiers = this.shortcutInput.getModifiers(keyEvent);
        if (modifiers.length > 0) {
            newKeyCombination.modifiers = {};
            modifiers.forEach((modifier) => {
                switch (modifier) {
                    case 'ctrl':
                        newKeyCombination.modifiers.ctrl = true;
                        break;
                    case 'alt':
                        newKeyCombination.modifiers.alt = true;
                        break;
                    case 'shift':
                        newKeyCombination.modifiers.shift = true;
                        break;
                    case 'meta':
                        newKeyCombination.modifiers.search = true;
                        break;
                    default:
                        break;
                }
            });
        }
        return newKeyCombination;
    }
    getKeyComboChangeButtonLabel_() {
        return this.keyCombination_ ?
            this.i18n('faceGazeActionsDialogKeyCombinationChangeButtonDescription', FaceGazeUtils.getKeyComboDisplayText(this.keyCombination_)) :
            this.i18n('faceGazeActionsDialogKeyCombinationChangeButtonLabel');
    }
    getShortcutInputLabel_() {
        // If there is a user-entered key combination, return the key combination as
        // a label. Otherwise, return instructions on how to input a key
        // combination.
        return this.keyCombination_ ?
            FaceGazeUtils.getKeyComboDisplayText(this.keyCombination_) :
            this.i18n('faceGazeActionsDialogKeyCombinationLabel');
    }
    getLocalizedSelectGestureTitle_() {
        return this.i18n('faceGazeActionsDialogSelectGestureTitle', this.selectedAction_ ? this.i18n(FaceGazeUtils.getMacroDisplayTextName(this.selectedAction_)) :
            '');
    }
    getLocalizedGestureThresholdTitle_() {
        return this.i18n('faceGazeActionsDialogGestureThresholdTitle', this.selectedGesture_ ?
            this.i18n(FaceGazeUtils.getGestureDisplayTextName(this.selectedGesture_)) :
            '');
    }
    getLocalizedGestureCountLabel_() {
        if (this.detectedGestureCount_ === 0) {
            return this.i18n('faceGazeActionsDialogGestureNotDetectedLabel');
        }
        else if (this.detectedGestureCount_ === 1) {
            return this.i18n('faceGazeActionsDialogGestureDetectedCountOneLabel');
        }
        else {
            return this.i18n('faceGazeActionsDialogGestureDetectedCountLabel', this.detectedGestureCount_);
        }
    }
    getActionDisplayText_(action) {
        return this.i18n(FaceGazeUtils.getMacroDisplayTextName(action));
    }
    getGestureDisplayText_(gesture) {
        return this.i18n(FaceGazeUtils.getGestureDisplayTextName(gesture));
    }
    getComplexActionDisplayText_(action) {
        if (!action || !ComplexActions[action]) {
            return null;
        }
        return this.i18n(ComplexActions[action]);
    }
    getWarningDisplayText_(gesture) {
        if (!gesture) {
            return null;
        }
        let alreadyAssigned = false;
        const bindings = this.get(FACE_GAZE_GESTURE_TO_MACROS_PREF);
        const potentialConflicts = ConflictingGestures[gesture];
        const conflicts = [];
        for (const assignedGesture of Object.keys(bindings)) {
            if (assignedGesture === gesture) {
                alreadyAssigned = true;
            }
            if (potentialConflicts && potentialConflicts.includes(assignedGesture)) {
                // Only show conflicts warning if the user has already assigned a
                // conflicting gesture.
                conflicts.push(assignedGesture);
            }
        }
        if (conflicts.length === 0 && !alreadyAssigned) {
            return null;
        }
        let conflictsString = '';
        if (conflicts.length > 0) {
            // Compute conflicting gestures string.
            const substitutions = [];
            for (const conflict of conflicts) {
                substitutions.push(this.getGestureDisplayText_(conflict));
            }
            substitutions.unshift(this.getGestureDisplayText_(gesture));
            // Decide which localized string to use based on the number of conflicting
            // gestures.
            if (conflicts.length === 1) {
                conflictsString = this.i18n('faceGazeWarningConflictingGesturesSingleLabel', ...substitutions);
            }
            else if (conflicts.length === 2) {
                conflictsString = this.i18n('faceGazeWarningConflictingGesturesDoubleLabel', ...substitutions);
            }
            else if (conflicts.length === 3) {
                conflictsString = this.i18n('faceGazeWarningConflictingGesturesTripleLabel', ...substitutions);
            }
            else {
                throw new Error(`Got an unexpected number of conflicting gestures: ${conflicts.length}`);
            }
        }
        const alreadyAssignedString = this.i18n('faceGazeWarningGestureAlreadyAssignedLabel');
        // There are three possible warning messages we can show. Note that we've
        // already handled the case where both `conflicts` and `assigned` are false.
        if (conflictsString && !alreadyAssigned) {
            return conflictsString;
        }
        else if (!conflictsString && alreadyAssigned) {
            return alreadyAssignedString;
        }
        else {
            return this.i18n('faceGazeWarningCombinedLabel', alreadyAssignedString, conflictsString);
        }
    }
    getGestureIconName_(gesture) {
        return `facegaze:${FaceGazeUtils.getGestureIconName(gesture)}`;
    }
    getActionAriaLabel_(macro, selected) {
        const label = selected ? 'faceGazeActionsDialogSelectedItemInstruction' :
            'faceGazeActionsDialogNotSelectedItemInstruction';
        return this.i18n(label, this.getActionDisplayText_(macro), this.displayedActions_.indexOf(macro) + 1, this.displayedActions_.length);
    }
    getGestureAriaLabel_(gesture, selected) {
        const displayedGestures = this.get('displayedGestures_');
        const label = selected ? 'faceGazeActionsDialogSelectedItemInstruction' :
            'faceGazeActionsDialogNotSelectedItemInstruction';
        return this.i18n(label, this.getGestureDisplayText_(gesture), displayedGestures.indexOf(gesture) + 1, displayedGestures.length);
    }
    // Dialog page navigation.
    shouldShowSelectAction_() {
        return this.currentPage_ === AddDialogPage.SELECT_ACTION;
    }
    shouldShowCustomKeyboard_() {
        return this.currentPage_ === AddDialogPage.CUSTOM_KEYBOARD;
    }
    shouldShowSelectGesture_() {
        return this.currentPage_ === AddDialogPage.SELECT_GESTURE;
    }
    shouldShowGestureThreshold_() {
        return this.currentPage_ === AddDialogPage.GESTURE_THRESHOLD;
    }
    // Disable navigation buttons.
    shouldDisableActionNextButton_() {
        return this.selectedAction_ === null;
    }
    shouldDisableCustomKeyboardNextButton_() {
        return this.keyCombination_ === null;
    }
    shouldDisableGestureNextButton_() {
        return this.selectedGesture_ === null;
    }
    shouldDisplayGesturePreviousButton_() {
        // Only show the previous button on the gesture page if we are starting from
        // the beginning of the dialog flow.
        return this.initialPage === AddDialogPage.SELECT_ACTION;
    }
    shouldDisplayThresholdPreviousButton_() {
        // Only show the previous button on the threshold page if we are starting
        // from an earlier dialog flow.
        return this.initialPage !== AddDialogPage.GESTURE_THRESHOLD;
    }
    // Dialog navigation button event handlers.
    onPreviousButtonClick_() {
        this.onNavigateButtonClick_(Navigation.PREVIOUS);
    }
    onNextButtonClick_() {
        this.onNavigateButtonClick_(Navigation.NEXT);
    }
    onNavigateButtonClick_(direction) {
        const newPage = direction === Navigation.PREVIOUS ?
            this.pageNavigation_[this.currentPage_].previous :
            this.pageNavigation_[this.currentPage_].next;
        if (newPage === undefined) {
            // This should never happen.
            throw new Error(`Tried to navigate in ${direction.toString()} direction from page index ${this.currentPage_.toString()}`);
        }
        this.currentPage_ = newPage;
    }
    onCancelButtonClick_() {
        this.close_();
    }
    onSaveButtonClick_() {
        if (!this.selectedAction_ || !this.selectedGesture_) {
            console.error('FaceGaze Add Dialog clicked save button but no action/gesture pair selected. Closing dialog.');
            this.close_();
            return;
        }
        const commandPair = new FaceGazeCommandPair(this.selectedAction_, this.selectedGesture_);
        if (this.selectedAction_ === MacroName.CUSTOM_KEY_COMBINATION) {
            if (!this.keyCombination_) {
                throw new Error('FaceGaze selected custom key combination action but no key combination set.');
            }
            commandPair.assignedKeyCombo =
                new AssignedKeyCombo(JSON.stringify(this.keyCombination_));
        }
        const event = new CustomEvent(FACEGAZE_COMMAND_PAIR_ADDED_EVENT_NAME, {
            bubbles: true,
            composed: true,
            detail: commandPair,
        });
        this.dispatchEvent(event);
        this.setPrefDictEntry(FACE_GAZE_GESTURE_TO_CONFIDENCE_PREF_DICT, this.selectedGesture_, Math.round(this.getThresholdSlider().value));
        this.close_();
    }
    currentPageChanged_(newPage, oldPage) {
        // Only toggle request for gesture information on if we are on the gesture
        // threshold page, which requires information about detected gestures,
        // or toggle request off if we are switching away from the gesture
        // threshold page.
        if (newPage === AddDialogPage.GESTURE_THRESHOLD ||
            oldPage === AddDialogPage.GESTURE_THRESHOLD) {
            this.faceGazeSubpageBrowserProxy_.toggleGestureInfoForSettings(newPage === AddDialogPage.GESTURE_THRESHOLD);
        }
        if (oldPage === AddDialogPage.CUSTOM_KEYBOARD && this.shortcutInput) {
            this.shortcutInput.stopObserving();
        }
    }
    // Handlers for initial state.
    initialPageChanged_(page) {
        this.currentPage_ = page;
    }
    commandPairToConfigureChanged_(newValue) {
        if (!newValue) {
            return;
        }
        this.selectedAction_ = newValue.action;
        this.selectedGesture_ = newValue.gesture;
        if (newValue.assignedKeyCombo) {
            this.keyCombination_ = newValue.assignedKeyCombo.keyCombo;
        }
    }
    computeAllowedGestures_() {
        let displayedGestures = FaceGazeGestures;
        // If left-click action is assigned to a singular gesture then remove it
        // from the list of available gestures to avoid losing left click
        // functionality.
        if (this.leftClickGestures.length === 1) {
            displayedGestures = displayedGestures.filter((gesture) => {
                return this.leftClickGestures[0] !== gesture;
            });
        }
        // If the selected action is dependent on location, then only allow gestures
        // where the user can be looking at their screen while performing it so they
        // can be certain of their mouse location.
        if (FaceGazeLocationDependentActions.includes(this.selectedAction_)) {
            displayedGestures = displayedGestures.filter((gesture) => {
                return !FaceGazeLookGestures.includes(gesture);
            });
        }
        return displayedGestures;
    }
    focusTitle_() {
        const title = this.shadowRoot.querySelector('#title');
        if (title) {
            title.focus();
        }
    }
    onActionPageDomChanged_() {
        if (this.currentPage_ !== AddDialogPage.SELECT_ACTION) {
            return;
        }
        this.focusTitle_();
    }
    onShortcutInputPageDomChanged_() {
        if (this.currentPage_ !== AddDialogPage.CUSTOM_KEYBOARD) {
            return;
        }
        this.keyCombination_ = null;
        this.focusTitle_();
        // Start observing for input events the moment `shortcutInput` is available.
        this.shortcutInput =
            this.shadowRoot.querySelector('#shortcutInput');
        if (this.shortcutInput) {
            this.shortcutInput.reset();
            this.shortcutInput.startObserving();
        }
    }
    onChangeButtonClick_() {
        if (this.currentPage_ !== AddDialogPage.CUSTOM_KEYBOARD) {
            return;
        }
        this.keyCombination_ = null;
        this.shortcutInput =
            this.shadowRoot.querySelector('#shortcutInput');
        if (this.shortcutInput) {
            this.shortcutInput.reset();
            this.shortcutInput.startObserving();
        }
    }
    onGesturePageDomChanged_() {
        if (this.currentPage_ !== AddDialogPage.SELECT_GESTURE) {
            return;
        }
        this.focusTitle_();
    }
    async onThresholdPageDomChanged_() {
        if (this.currentPage_ !== AddDialogPage.GESTURE_THRESHOLD) {
            return;
        }
        this.focusTitle_();
        const videoElement = this.shadowRoot.querySelector('#cameraStream');
        if (videoElement) {
            this.stream_ = await navigator.mediaDevices.getUserMedia({
                audio: false,
                video: { facingMode: 'user' },
            });
            this.streamTrack_ = this.stream_.getVideoTracks()[0];
            videoElement.srcObject = this.stream_;
        }
    }
    onDecreaseThresholdButtonClick_() {
        this.gestureThresholdValue_ = Math.max(FACEGAZE_CONFIDENCE_MIN, this.gestureThresholdValue_ - FACEGAZE_CONFIDENCE_BUTTON_STEP);
    }
    onIncreaseThresholdButtonClick_() {
        this.gestureThresholdValue_ = Math.min(FACEGAZE_CONFIDENCE_MAX, this.gestureThresholdValue_ + FACEGAZE_CONFIDENCE_BUTTON_STEP);
    }
    getThresholdSlider() {
        const slider = this.shadowRoot?.querySelector('#faceGazeGestureThresholdSlider');
        assert(slider);
        return slider;
    }
    onThresholdSliderChanged_() {
        this.gestureThresholdValue_ = this.getThresholdSlider().value;
    }
    onGestureConfidencesReceived_(info) {
        if (!this.selectedGesture_ ||
            this.currentPage_ !== AddDialogPage.GESTURE_THRESHOLD) {
            return;
        }
        const previewContainer = this.shadowRoot.querySelector('#cameraPreviewContainer');
        assert(previewContainer);
        const slider = this.getThresholdSlider();
        const sliderBar = slider.shadowRoot.querySelector('#bar');
        assert(sliderBar);
        const gestureInfo = info.find((entry) => entry.gesture === this.selectedGesture_);
        if (!gestureInfo) {
            this.holdingGesture_ = false;
            previewContainer.className = 'gesture-not-detected';
            sliderBar.style.width = `0%`;
            return;
        }
        if (gestureInfo.confidence >= this.gestureThresholdValue_) {
            if (!this.holdingGesture_) {
                this.detectedGestureCount_++;
                this.holdingGesture_ = true;
            }
            previewContainer.className = 'gesture-detected';
        }
        else {
            this.holdingGesture_ = false;
            previewContainer.className = 'gesture-not-detected';
        }
        // Show confidence values at all levels in dynamic bar.
        sliderBar.style.width = `${gestureInfo.confidence}%`;
    }
    onSelectedActionChanged_() {
        if (this.selectedAction_ === MacroName.CUSTOM_KEY_COMBINATION) {
            this.pageNavigation_ = FACEGAZE_CUSTOM_KEYBOARD_SHORTCUT_FLOW;
        }
        else {
            this.pageNavigation_ = FACEGAZE_DEFINED_MACRO_FLOW;
        }
    }
    onSelectedGestureChanged_() {
        this.detectedGestureCount_ = 0;
        this.gestureThresholdValue_ = FACEGAZE_CONFIDENCE_DEFAULT;
        const gesturesToConfidence = this.get(FACE_GAZE_GESTURE_TO_CONFIDENCE_PREF);
        if (this.selectedGesture_ &&
            this.selectedGesture_ in gesturesToConfidence) {
            this.gestureThresholdValue_ = gesturesToConfidence[this.selectedGesture_];
        }
    }
    onGestureThresholdChanged_() {
        this.detectedGestureCount_ = 0;
    }
    close_() {
        if (this.shortcutInput) {
            this.shortcutInput.stopObserving();
        }
        if (this.streamTrack_) {
            this.streamTrack_.stop();
        }
        if (this.stream_) {
            this.stream_ = null;
        }
        this.commandPairToConfigure = null;
        this.keyCombination_ = null;
        this.faceGazeSubpageBrowserProxy_.toggleGestureInfoForSettings(false);
        this.$.dialog.close();
    }
    onKeydown_(e) {
        // Close dialog if 'esc' is pressed.
        if (e.key === 'Escape') {
            this.close_();
        }
    }
    getCurrentPageForTest() {
        return this.currentPage_;
    }
}
customElements.define(FaceGazeAddActionDialogElement.is, FaceGazeAddActionDialogElement);
