// Copyright 2024 The Chromium Authors
// Use of this source code is governed by a BSD-style license that can be
// found in the LICENSE file.
/**
 * @fileoverview Foldable container for dictionary editor (for a single
 * dictionary).
 */
import './os_japanese_dictionary_entry_row.js';
import 'chrome://resources/ash/common/cr_elements/cr_input/cr_input.js';
import { I18nMixin } from 'chrome://resources/ash/common/cr_elements/i18n_mixin.js';
import { afterNextRender, PolymerElement } from 'chrome://resources/polymer/v3_0/polymer/polymer_bundled.min.js';
import { JpPosType } from '../mojom-webui/user_data_japanese_dictionary.mojom-webui.js';
import { getTemplate } from './os_japanese_dictionary_expand.html.js';
import { UserDataServiceProvider } from './user_data_service_provider.js';
class OsJapaneseDictionaryExpandElement extends I18nMixin(PolymerElement) {
    constructor() {
        super(...arguments);
        // Whether or not this container UI is expanded or folded.
        this.expanded_ = false;
        this.showingDeleteDialog_ = false;
        // Used for chromevox announcements.
        this.statusMessage_ = '';
        this.dictIndex = 0;
    }
    static get is() {
        return 'os-japanese-dictionary-expand';
    }
    static get template() {
        return getTemplate();
    }
    static get properties() {
        return {
            dict: {
                type: Object,
            },
            syncedEntriesCount: {
                type: Number,
            },
            showingDeleteDialog_: {
                type: Boolean,
            },
            statusMessage_: {
                type: String,
            },
            dictIndex: {
                type: Number,
            },
        };
    }
    ready() {
        super.ready();
        this.addEventListener('dictionary-entry-deleted', this.onEntryDelete_);
    }
    onEntryDelete_(event) {
        this.statusMessage_ = '';
        afterNextRender(this, () => {
            this.statusMessage_ = this.i18n('japaneseDictionaryEntryDeleted');
            if (event.detail.isLastEntry) {
                const newEntryButton = this.shadowRoot.querySelector('#newEntryButton');
                if (newEntryButton) {
                    // TODO(crbug.com/419677565): Find a way to queue this focus only
                    // after the hidden value has changed by itself. This is a hacky way
                    // to resolve a race condition where the hidden value has not changed
                    // yet before the focus is moved.
                    newEntryButton.removeAttribute('hidden');
                    newEntryButton.focus();
                }
            }
        });
    }
    // Adds a new entry locally to create an entry-row component.
    addEntry_() {
        // This changes the entries array from the parent component which it will
        // not be notified of. This is intentional.
        // We do not want to trigger a rerender in the parent component.
        this.push('dict.entries', { key: '', value: '', pos: JpPosType.kNoPos, comment: '' });
        afterNextRender(this, () => {
            this.shadowRoot
                .querySelector('os-japanese-dictionary-entry-row:last-of-type').shadowRoot
                .querySelector('cr-input').focus();
        });
    }
    // Renames the dictionary.
    async saveName_(e) {
        this.dict.name = e.target.value;
        const dictionarySaved = (await UserDataServiceProvider.getRemote().renameJapaneseDictionary(this.dict.id, this.dict.name))
            .status.success;
        if (dictionarySaved) {
            this.dispatchSavedEvent_();
        }
    }
    // Renames the dictionary.
    async deleteDictionary_() {
        const dictionarySaved = (await UserDataServiceProvider.getRemote().deleteJapaneseDictionary(this.dict.id))
            .status.success;
        if (dictionarySaved) {
            this.dispatchDictionaryDeletedEvent_();
            this.dispatchSavedEvent_();
        }
        this.showingDeleteDialog_ = false;
    }
    hideDeleteDialog_() {
        this.showingDeleteDialog_ = false;
    }
    showDeleteDialog_() {
        this.showingDeleteDialog_ = true;
    }
    // Export dictionary.
    exportDictionary_() {
        const a = document.createElement('a');
        a.href = `jp-export-dictionary/${this.dict.id}`;
        // In case there is no name, use a placeholder name.
        const fileName = this.dict.name || 'unnamed-dictionary';
        a.download = `${fileName}.txt`;
        a.click();
    }
    // Imports dictionary.
    importDictionary_() {
        this.$.selectFileDialog.dispatchEvent(new MouseEvent('click'));
    }
    async handleFileSelectChange_(e) {
        const fileInput = e.target;
        const fileData = fileInput.files[0];
        // Use bytes for now rather than shared memory for simplicity.
        // TODO(b/366101658): Use shared memory when file is too big.
        // The limit below is the max size that a mojo BigBuffer can handle via
        // directly using the bytes rather than shared memory.
        if (fileData.size >= 128 * 1048576) {
            // Clear value so that file select change will retrigger for the same
            // file.
            fileInput.value = '';
            return;
        }
        const fileDataView = new Uint8Array(await fileData.arrayBuffer());
        const fileMojomBigBuffer = {
            bytes: Array.from(fileDataView),
        };
        const fileMojomBigString = { data: fileMojomBigBuffer };
        const { status } = await UserDataServiceProvider.getRemote().importJapaneseDictionary(this.dict.id, fileMojomBigString);
        if (status.success) {
            this.dispatchSavedEvent_();
        }
        // Clear value so that file select change will retrigger for the same file.
        fileInput.value = '';
    }
    // Returns true if this entry is a locally added entry.
    locallyAdded_(entryIndex) {
        // This entry falls outside of the range of entries that were initially
        // synced, hence it must be added locally.
        return entryIndex > this.syncedEntriesCount;
    }
    // Returns true if this entry is the last one.
    isLastEntry_(entryIndex) {
        return entryIndex === this.dict.entries.length - 1;
    }
    // If there is currently an unsynced entry then hide the add button.
    // This is to prevent two "unadded" entries to cause issues with ordering when
    // synced. Users should only be able to add one entry at a time before a sync
    // occurs.
    shouldShowAddButton_(entriesLength) {
        return entriesLength - 1 <= this.syncedEntriesCount;
    }
    dispatchSavedEvent_() {
        this.dispatchEvent(new CustomEvent('dictionary-saved', { bubbles: true, composed: true }));
    }
    dispatchDictionaryDeletedEvent_() {
        this.dispatchEvent(new CustomEvent('dictionary-deleted', {
            bubbles: true,
            composed: true,
            detail: { dictIndex: this.dictIndex },
        }));
    }
    i18nDialogString_(dictName) {
        return this.i18n('japaneseDeleteDictionaryDetail', dictName);
    }
}
customElements.define(OsJapaneseDictionaryExpandElement.is, OsJapaneseDictionaryExpandElement);
