// Copyright 2023 The Chromium Authors
// Use of this source code is governed by a BSD-style license that can be
// found in the LICENSE file.
import './setup_pin_dialog.js';
import './pin_autosubmit_dialog.js';
import { LockScreenProgress, recordLockScreenProgress } from 'chrome://resources/ash/common/quick_unlock/lock_screen_constants.js';
import { fireAuthTokenInvalidEvent } from 'chrome://resources/ash/common/quick_unlock/utils.js';
import { CrActionMenuElement } from 'chrome://resources/ash/common/cr_elements/cr_action_menu/cr_action_menu.js';
import { CrButtonElement } from 'chrome://resources/ash/common/cr_elements/cr_button/cr_button.js';
import { CrIconButtonElement } from 'chrome://resources/ash/common/cr_elements/cr_icon_button/cr_icon_button.js';
import { WebUiListenerMixin } from 'chrome://resources/ash/common/cr_elements/web_ui_listener_mixin.js';
import { assert } from 'chrome://resources/js/assert.js';
import { focusWithoutInk } from 'chrome://resources/js/focus_without_ink.js';
import { loadTimeData } from 'chrome://resources/js/load_time_data.js';
import { AuthFactor, AuthFactorConfig, ConfigureResult, FactorObserverReceiver, PinFactorEditor } from 'chrome://resources/mojo/chromeos/ash/services/auth_factor_config/public/mojom/auth_factor_config.mojom-webui.js';
import { PolymerElement } from 'chrome://resources/polymer/v3_0/polymer/polymer_bundled.min.js';
import { SettingsToggleButtonElement } from '../controls/settings_toggle_button.js';
import { getTemplate } from './pin_settings.html.js';
const SettingsPinSettingsElementBase = WebUiListenerMixin(PolymerElement);
export class SettingsPinSettingsElement extends SettingsPinSettingsElementBase {
    static get is() {
        return 'settings-pin-settings';
    }
    static get template() {
        return getTemplate();
    }
    static get properties() {
        return {
            authToken: {
                type: String,
                value: null,
                observer: 'updatePinState_',
            },
            hasPin_: {
                type: Boolean,
                value: false,
            },
            showSetPinDialog_: {
                type: Boolean,
                value: false,
            },
            showPinAutosubmitDialog_: {
                type: Boolean,
                value: false,
            },
            /**
             * True if quick unlock settings are disabled by policy.
             */
            quickUnlockDisabledByPolicy_: {
                type: Boolean,
                value: loadTimeData.getBoolean('quickUnlockDisabledByPolicy'),
            },
        };
    }
    ready() {
        super.ready();
        // Register observer for auth factor updates.
        // TODO(crbug.com/40223898): Are we leaking |this| here because we never remove
        // the observer? We could close the pipe with |$.close()|, but not clear
        // whether that removes all references to |receiver| and then eventually to
        // |this|.
        const receiver = new FactorObserverReceiver(this);
        const remote = receiver.$.bindNewPipeAndPassRemote();
        AuthFactorConfig.getRemote().observeFactorChanges(remote);
    }
    connectedCallback() {
        super.connectedCallback();
        this.addWebUiListener('quick-unlock-disabled-by-policy-changed', (quickUnlockDisabledByPolicy) => {
            this.quickUnlockDisabledByPolicy_ = quickUnlockDisabledByPolicy;
        });
        chrome.send('RequestQuickUnlockDisabledByPolicy');
    }
    onFactorChanged(factor) {
        switch (factor) {
            case AuthFactor.kPrefBasedPin:
            case AuthFactor.kCryptohomePin:
            case AuthFactor.kCryptohomePinV2:
            case AuthFactor.kGaiaPassword:
            case AuthFactor.kLocalPassword:
                this.updatePinState_();
                break;
            default:
                return;
        }
    }
    // Remove pin is disabled when pin is the only factor available, or
    // in other words, when there is no password available.
    // Remove can also be disabled by then the QuickUnlock policy disallows it.
    removeDisabled_() {
        return !this.hasPassword_ || this.quickUnlockDisabledByPolicy_;
    }
    moreButton_() {
        const moreButton = this.shadowRoot.querySelector('#moreButton');
        assert(moreButton instanceof CrIconButtonElement);
        return moreButton;
    }
    moreMenu_() {
        const moreMenu = this.shadowRoot.querySelector('#moreMenu');
        assert(moreMenu instanceof CrActionMenuElement);
        return moreMenu;
    }
    setPinButton_() {
        // We enforce that there is precisely one .set-pin-button attached at any
        // moment.
        const elements = this.shadowRoot.querySelectorAll('.set-pin-button');
        assert(elements.length === 1);
        const setPinButton = elements[0];
        assert(setPinButton instanceof CrButtonElement);
        return setPinButton;
    }
    enablePinAutoSubmitToggle_() {
        const toggle = this.shadowRoot.querySelector('#enablePinAutoSubmit');
        assert(toggle instanceof SettingsToggleButtonElement);
        return toggle;
    }
    /**
     * Fetches the state of the pin factor and updates the corresponding
     * property. It also updates the fact that there is another knowledge factor
     * present or not. This will help with logic of removal.
     */
    async updatePinState_() {
        if (!this.authToken) {
            return;
        }
        if (typeof this.authToken !== 'string') {
            return;
        }
        const authToken = this.authToken;
        const afc = AuthFactorConfig.getRemote();
        const pfe = PinFactorEditor.getRemote();
        // clang-format off
        const [{ configured: hasGaiaPassword }, { configured: hasLocalPassword }, { pinFactor }] = await Promise.all([
            afc.isConfigured(authToken, AuthFactor.kGaiaPassword),
            afc.isConfigured(authToken, AuthFactor.kLocalPassword),
            pfe.getConfiguredPinFactor(authToken),
        ]);
        // clang-format off
        this.hasPin_ = pinFactor !== null;
        this.hasPassword_ = hasGaiaPassword || hasLocalPassword;
    }
    onSetPinButtonClicked_() {
        recordLockScreenProgress(LockScreenProgress.CHOOSE_PIN_OR_PASSWORD);
        this.showSetPinDialog_ = true;
    }
    onSetPinDialogClose_() {
        this.showSetPinDialog_ = false;
        focusWithoutInk(this.setPinButton_());
    }
    onPinAutosubmitChange_(event) {
        const target = event.target;
        assert(target instanceof SettingsToggleButtonElement);
        assert(target === this.enablePinAutoSubmitToggle_());
        if (typeof this.authToken !== 'string') {
            console.error('PIN autosubmit setting changed with expired token.');
            target.checked = !target.checked;
            return;
        }
        // Read-only preference. Changes will be reflected directly on the toggle.
        const autosubmitEnabled = target.checked;
        target.resetToPrefValue();
        if (autosubmitEnabled) {
            this.showPinAutosubmitDialog_ = true;
        }
        else {
            // Call quick unlock to disable the auto-submit option.
            chrome.quickUnlockPrivate.setPinAutosubmitEnabled(this.authToken, '' /* PIN */, false /*enabled*/, () => { });
        }
    }
    onPinAutosubmitDialogClose_() {
        this.showPinAutosubmitDialog_ = false;
        const toggle = this.enablePinAutoSubmitToggle_();
        focusWithoutInk(toggle);
    }
    onMoreButtonClicked_(event) {
        event.preventDefault(); // Prevent default browser action (navigation).
        const moreButton = this.moreButton_();
        if (moreButton === null) {
            return;
        }
        const moreMenu = this.moreMenu_();
        if (moreMenu === null) {
            return;
        }
        moreMenu.showAt(moreButton);
    }
    async onRemovePinButtonClicked_() {
        if (typeof this.authToken !== 'string') {
            console.error('Tried to remove PIN with expired token.');
            return;
        }
        const { result } = await PinFactorEditor.getRemote().removePin(this.authToken);
        switch (result) {
            case ConfigureResult.kSuccess:
                break;
            case ConfigureResult.kInvalidTokenError:
                fireAuthTokenInvalidEvent(this);
                break;
            case ConfigureResult.kFatalError:
                console.error('Error removing PIN');
                break;
            default:
                break;
        }
        // We always close the "more" menu, even when removePin call didn't work:
        // If the menu isn't closed but not attached anymore, then the user can't
        // interact with the whole settings UI at all anymore.
        const moreMenu = this.moreMenu_();
        if (moreMenu) {
            moreMenu.close();
        }
    }
}
customElements.define(SettingsPinSettingsElement.is, SettingsPinSettingsElement);
