// Copyright 2022 The Chromium Authors
// Use of this source code is governed by a BSD-style license that can be
// found in the LICENSE file.
/**
 * @fileoverview 'settings-cups-edit-printer-dialog' is a dialog to edit the
 * existing printer's information and re-configure it.
 */
import 'chrome://resources/ash/common/cr_elements/localized_link/localized_link.js';
import 'chrome://resources/ash/common/cr_elements/cr_button/cr_button.js';
import 'chrome://resources/ash/common/cr_elements/cr_input/cr_input.js';
import 'chrome://resources/ash/common/cr_elements/cr_searchable_drop_down/cr_searchable_drop_down.js';
import 'chrome://resources/ash/common/cr_elements/cr_shared_style.css.js';
import 'chrome://resources/polymer/v3_0/iron-icon/iron-icon.js';
import './cups_add_printer_dialog.js';
import './cups_printer_dialog_error.js';
import './cups_printer_shared.css.js';
import './cups_printers_browser_proxy.js';
import { I18nMixin } from 'chrome://resources/ash/common/cr_elements/i18n_mixin.js';
import { MojoInterfaceProviderImpl } from 'chrome://resources/ash/common/network/mojo_interface_provider.js';
import { NetworkListenerBehavior } from 'chrome://resources/ash/common/network/network_listener_behavior.js';
import { OncMojo } from 'chrome://resources/ash/common/network/onc_mojo.js';
import { FilterType, NO_LIMIT } from 'chrome://resources/mojo/chromeos/services/network_config/public/mojom/cros_network_config.mojom-webui.js';
import { NetworkType } from 'chrome://resources/mojo/chromeos/services/network_config/public/mojom/network_types.mojom-webui.js';
import { mixinBehaviors, PolymerElement } from 'chrome://resources/polymer/v3_0/polymer/polymer_bundled.min.js';
import { cast } from '../assert_extras.js';
import { getTemplate } from './cups_edit_printer_dialog.html.js';
import { getDisplayPpd, getErrorText, isNameAndAddressValid, isNetworkProtocol, isPPDInfoValid } from './cups_printer_dialog_util.js';
import { CupsPrintersBrowserProxyImpl } from './cups_printers_browser_proxy.js';
/**
 * The types of actions that can be performed with the edit dialog.  These
 * values are written to logs and used as metrics.  New enum values can be
 * added, but existing values must never be renumbered or deleted and reused.
 * See PrinterEditDialogActions enum in tools/metrics/hisograms/enums.xml.
 */
var DialogActions;
(function (DialogActions) {
    DialogActions[DialogActions["DIALOG_OPENED"] = 0] = "DIALOG_OPENED";
    DialogActions[DialogActions["VIEW_PPD_CLICKED"] = 1] = "VIEW_PPD_CLICKED";
})(DialogActions || (DialogActions = {}));
/** Keyword used for recording metrics */
const METRICS_KEYWORD = 'Printing.CUPS.PrinterEditDialogActions';
const SettingsCupsEditPrinterDialogElementBase = mixinBehaviors([NetworkListenerBehavior], I18nMixin(PolymerElement));
export class SettingsCupsEditPrinterDialogElement extends SettingsCupsEditPrinterDialogElementBase {
    static get is() {
        return 'settings-cups-edit-printer-dialog';
    }
    static get template() {
        return getTemplate();
    }
    static get properties() {
        return {
            /**
             * The currently saved printer.
             */
            activePrinter: Object,
            /**
             * Printer that holds the modified changes to activePrinter and only
             * applies these changes when the save button is clicked.
             */
            pendingPrinter_: Object,
            /**
             * If the printer needs to be re-configured.
             */
            needsReconfigured_: {
                type: Boolean,
                value: false,
            },
            /**
             * The current PPD in use by the printer.
             */
            userPPD_: String,
            /**
             * Tracks whether the dialog is fully initialized. This is required
             * because the dialog isn't fully initialized until Model and Manufacturer
             * are set. Allows us to ignore changes made to these fields until
             * initialization is complete.
             */
            arePrinterFieldsInitialized_: {
                type: Boolean,
                value: false,
            },
            /**
             * If the printer info has changed since loading this dialog. This will
             * only track the freeform input fields, since the other fields contain
             * input selected from dropdown menus.
             */
            printerInfoChanged_: {
                type: Boolean,
                value: false,
            },
            networkProtocolActive_: {
                type: Boolean,
                computed: 'isNetworkProtocol_(pendingPrinter_.printerProtocol)',
            },
            manufacturerList: Array,
            modelList: Array,
            /**
             * Whether the user selected PPD file is valid.
             */
            invalidPPD_: {
                type: Boolean,
                value: false,
            },
            /**
             * The base name of a newly selected PPD file.
             */
            newUserPPD_: String,
            /**
             * The URL to a printer's EULA.
             */
            eulaUrl_: {
                type: String,
                value: '',
            },
            isOnline_: {
                type: Boolean,
                value: true,
            },
            /**
             * The error text to be displayed on the dialog.
             */
            errorText_: {
                type: String,
                value: '',
            },
            /**
             * Indicates whether the value in the Manufacturer dropdown is a valid
             * printer manufacturer.
             */
            isManufacturerInvalid_: {
                type: Boolean,
                value: false,
            },
            /**
             * Indicates whether the value in the Model dropdown is a valid printer
             * model.
             */
            isModelInvalid_: {
                type: Boolean,
                value: false,
            },
        };
    }
    static get observers() {
        return [
            'printerPathChanged_(pendingPrinter_.*)',
            'selectedEditManufacturerChanged_(pendingPrinter_.ppdManufacturer)',
            'onModelChanged_(pendingPrinter_.ppdModel)',
        ];
    }
    constructor() {
        super();
        this.browserProxy_ = CupsPrintersBrowserProxyImpl.getInstance();
        this.networkConfig_ =
            MojoInterfaceProviderImpl.getInstance().getMojoServiceRemote();
    }
    connectedCallback() {
        super.connectedCallback();
        chrome.metricsPrivate.recordEnumerationValue(METRICS_KEYWORD, DialogActions.DIALOG_OPENED, Object.keys(DialogActions).length);
        // Create a copy of activePrinter so that we can modify its fields.
        this.pendingPrinter_ = Object.assign({}, this.activePrinter);
        this.refreshNetworks_();
        this.browserProxy_
            .getPrinterPpdManufacturerAndModel(this.pendingPrinter_.printerId)
            .then(this.onGetPrinterPpdManufacturerAndModel_.bind(this), this.onGetPrinterPpdManufacturerAndModelFailed_.bind(this));
        this.browserProxy_.getCupsPrinterManufacturersList().then(this.manufacturerListChanged_.bind(this));
        this.userPPD_ = getDisplayPpd(this.pendingPrinter_.printerPPDPath);
    }
    onActiveNetworksChanged(networks) {
        this.isOnline_ = networks.some((network) => {
            return OncMojo.connectionStateIsConnected(network.connectionState);
        });
    }
    printerPathChanged_(change) {
        if (change.path !== 'pendingPrinter_.printerName') {
            this.needsReconfigured_ = true;
        }
    }
    onProtocolChange_(event) {
        const selectEl = cast(event.target, HTMLSelectElement);
        this.set('pendingPrinter_.printerProtocol', selectEl.value);
        this.onPrinterInfoChange_();
    }
    onPrinterInfoChange_() {
        this.printerInfoChanged_ = true;
    }
    onCancelClick_() {
        this.shadowRoot.querySelector('add-printer-dialog').close();
    }
    /**
     * Handler for update|reconfigureCupsPrinter success.
     */
    onPrinterEditSucceeded_(result) {
        const showCupsPrinterToastEvent = new CustomEvent('show-cups-printer-toast', {
            bubbles: true,
            composed: true,
            detail: { resultCode: result, printerName: this.activePrinter.printerName },
        });
        this.dispatchEvent(showCupsPrinterToastEvent);
        this.shadowRoot.querySelector('add-printer-dialog').close();
    }
    /**
     * Handler for update|reconfigureCupsPrinter failure.
     */
    onPrinterEditFailed_(result) {
        this.errorText_ = getErrorText(result);
    }
    onSaveClick_() {
        this.updateActivePrinter_();
        if (!this.needsReconfigured_ || !this.isOnline_) {
            // If we don't need to reconfigure or we are offline, just update the
            // printer name.
            this.browserProxy_
                .updateCupsPrinter(this.activePrinter.printerId, this.activePrinter.printerName)
                .then(this.onPrinterEditSucceeded_.bind(this), this.onPrinterEditFailed_.bind(this));
        }
        else {
            this.browserProxy_.reconfigureCupsPrinter(this.activePrinter)
                .then(this.onPrinterEditSucceeded_.bind(this), this.onPrinterEditFailed_.bind(this));
        }
    }
    /**
     * @return Returns the i18n string for the dialog title.
     */
    getDialogTitle_() {
        return this.pendingPrinter_.isManaged ?
            this.i18n('viewPrinterDialogTitle') :
            this.i18n('editPrinterDialogTitle');
    }
    getPrinterUri_(printer) {
        if (!printer) {
            return '';
        }
        else if (printer.printerProtocol && printer.printerAddress &&
            printer.printerQueue) {
            return printer.printerProtocol + '://' + printer.printerAddress + '/' +
                printer.printerQueue;
        }
        else if (printer.printerProtocol && printer.printerAddress) {
            return printer.printerProtocol + '://' + printer.printerAddress;
        }
        else {
            return '';
        }
    }
    /**
     * Handler for getPrinterPpdManufacturerAndModel() success case.
     */
    onGetPrinterPpdManufacturerAndModel_(info) {
        this.set('pendingPrinter_.ppdManufacturer', info.ppdManufacturer);
        this.set('pendingPrinter_.ppdModel', info.ppdModel);
        // |needsReconfigured_| needs to reset to false after |ppdManufacturer| and
        // |ppdModel| are initialized to their correct values.
        this.needsReconfigured_ = false;
    }
    /**
     * Handler for getPrinterPpdManufacturerAndModel() failure case.
     */
    onGetPrinterPpdManufacturerAndModelFailed_() {
        this.needsReconfigured_ = false;
    }
    /**
     * Returns whether |protocol| is a network protocol
     */
    isNetworkProtocol_(protocol) {
        return isNetworkProtocol(protocol);
    }
    /**
     * Returns whether the current printer was auto configured.
     */
    isAutoconfPrinter_() {
        return this.pendingPrinter_.printerPpdReference.autoconf;
    }
    /**
     * Returns whether the Save button is enabled.
     */
    canSavePrinter_() {
        return this.printerInfoChanged_ &&
            (this.isPrinterConfigured_() || !this.isOnline_) &&
            !this.isManufacturerInvalid_ && !this.isModelInvalid_;
    }
    /**
     * @param manufacturer The manufacturer for which we are retrieving models.
     */
    selectedEditManufacturerChanged_(manufacturer) {
        // Reset model if manufacturer is changed.
        this.set('pendingPrinter_.ppdModel', '');
        this.modelList = [];
        if (!!manufacturer && manufacturer.length !== 0) {
            this.browserProxy_.getCupsPrinterModelsList(manufacturer)
                .then(this.modelListChanged_.bind(this));
        }
    }
    /**
     * Sets printerInfoChanged_ to true to show that the model has changed. Also
     * attempts to get the EULA Url if the selected printer has one.
     */
    onModelChanged_() {
        if (this.arePrinterFieldsInitialized_) {
            this.printerInfoChanged_ = true;
        }
        if (!this.pendingPrinter_.ppdManufacturer ||
            !this.pendingPrinter_.ppdModel) {
            // Do not check for an EULA unless both |ppdManufacturer| and |ppdModel|
            // are set. Set |eulaUrl_| to be empty in this case.
            this.onGetEulaUrlCompleted_('' /* eulaUrl */);
            return;
        }
        this.attemptPpdEulaFetch_();
    }
    /**
     * @param eulaUrl The URL for the printer's EULA.
     */
    onGetEulaUrlCompleted_(eulaUrl) {
        this.eulaUrl_ = eulaUrl;
    }
    onViewPpd_() {
        chrome.metricsPrivate.recordEnumerationValue(METRICS_KEYWORD, DialogActions.VIEW_PPD_CLICKED, Object.keys(DialogActions).length);
        // We always use the activePrinter (the printer when the dialog was first
        // displayed) when viewing the PPD.  Once the user has modified the dialog,
        // the view PPD button is no longer active.
        const eula = this.eulaUrl_ || '';
        this.browserProxy_.retrieveCupsPrinterPpd(this.activePrinter.printerId, this.activePrinter.printerName, eula);
    }
    onBrowseFile_() {
        this.browserProxy_.getCupsPrinterPpdPath().then(this.printerPpdPathChanged_.bind(this));
    }
    manufacturerListChanged_(manufacturersInfo) {
        if (!manufacturersInfo.success) {
            return;
        }
        this.manufacturerList = manufacturersInfo.manufacturers;
        if (this.pendingPrinter_.ppdManufacturer.length !== 0) {
            this.browserProxy_
                .getCupsPrinterModelsList(this.pendingPrinter_.ppdManufacturer)
                .then(this.modelListChanged_.bind(this));
        }
    }
    modelListChanged_(modelsInfo) {
        if (modelsInfo.success) {
            this.modelList = modelsInfo.models;
            // ModelListChanged_ is the final step of initializing pendingPrinter.
            this.arePrinterFieldsInitialized_ = true;
            // Fetch the EULA URL once we have PpdReferences from fetching the
            // |modelList|.
            this.attemptPpdEulaFetch_();
        }
    }
    /**
     * @param path The full path to the selected PPD file
     */
    printerPpdPathChanged_(path) {
        this.set('pendingPrinter_.printerPPDPath', path);
        this.invalidPPD_ = !path;
        if (!this.invalidPPD_) {
            // A new valid PPD file should be treated as a saveable change.
            this.onPrinterInfoChange_();
        }
        this.userPPD_ = getDisplayPpd(path);
    }
    /**
     * @return Returns true if the printer has valid name, address, and valid PPD
     *     or was
     * auto-configured.
     */
    isPrinterConfigured_() {
        return isNameAndAddressValid(this.pendingPrinter_) &&
            (this.isAutoconfPrinter_() ||
                isPPDInfoValid(this.pendingPrinter_.ppdManufacturer, this.pendingPrinter_.ppdModel, this.pendingPrinter_.printerPPDPath));
    }
    /**
     * Helper function to copy over modified fields to activePrinter.
     */
    updateActivePrinter_() {
        if (!this.isOnline_) {
            // If we are not online, only copy over the printerName.
            this.activePrinter.printerName = this.pendingPrinter_.printerName;
            return;
        }
        // Clone pendingPrinter_ into activePrinter_.
        this.activePrinter = Object.assign({}, this.pendingPrinter_);
        // Set ppdModel since there is an observer that clears ppdmodel's value when
        // ppdManufacturer changes.
        this.activePrinter.ppdModel = this.pendingPrinter_.ppdModel;
    }
    /**
     * Callback function when networks change.
     */
    refreshNetworks_() {
        this.networkConfig_
            .getNetworkStateList({
            filter: FilterType.kActive,
            networkType: NetworkType.kAll,
            limit: NO_LIMIT,
        })
            .then((responseParams) => {
            this.onActiveNetworksChanged(responseParams.result);
        });
    }
    /**
     * @return Returns true if the printer protocol select field should be
     *     enabled.
     */
    protocolSelectEnabled_() {
        if (this.pendingPrinter_) {
            // Print server printer's protocol should not be editable; disable the
            // drop down if the printer is from a print server.
            if (this.pendingPrinter_.printServerUri) {
                return false;
            }
            // Managed printers are not editable.
            if (this.pendingPrinter_.isManaged) {
                return false;
            }
        }
        return this.isOnline_ && this.networkProtocolActive_;
    }
    /**
     * Attempts fetching for the EULA Url based off of the current printer's
     * |ppdManufacturer| and |ppdModel|.
     */
    attemptPpdEulaFetch_() {
        if (!this.pendingPrinter_.ppdManufacturer ||
            !this.pendingPrinter_.ppdModel) {
            return;
        }
        this.browserProxy_
            .getEulaUrl(this.pendingPrinter_.ppdManufacturer, this.pendingPrinter_.ppdModel)
            .then(this.onGetEulaUrlCompleted_.bind(this));
    }
    /**
     * @return Returns true if we're on an active network and the printer
     * is not from a print server. If true, the input field is enabled.
     */
    isInputFieldEnabled_() {
        // Print server printers should not be editable (except for the name field).
        // Return false to disable the field.
        if (this.pendingPrinter_.printServerUri) {
            return false;
        }
        return this.networkProtocolActive_;
    }
    /**
     * @return Returns true if the printer is managed or not online.
     */
    isInputFieldReadonly_() {
        return !this.isOnline_ ||
            (this.pendingPrinter_ && this.pendingPrinter_.isManaged);
    }
    /**
     * @return Returns true if the printer is managed and both manufacturer and
     * model fields are empty.
     */
    shouldHideMakeAndModel_() {
        // If the printer is not managed, we should not hide the fields.
        if (!this.pendingPrinter_ || !this.pendingPrinter_.isManaged) {
            return false;
        }
        // Hide the fields if both are empty.
        return !this.pendingPrinter_.ppdManufacturer &&
            !this.pendingPrinter_.ppdModel;
    }
}
customElements.define(SettingsCupsEditPrinterDialogElement.is, SettingsCupsEditPrinterDialogElement);
