// Copyright 2022 The Chromium Authors
// Use of this source code is governed by a BSD-style license that can be
// found in the LICENSE file.
/**
 * @fileoverview 'settings-cups-nearby-printers' is a list container for
 * Nearby Printers.
 */
import 'chrome://resources/polymer/v3_0/iron-list/iron-list.js';
import '../settings_shared.css.js';
import './cups_printer_types.js';
import './cups_printers_browser_proxy.js';
import './cups_printers_entry.js';
import { WebUiListenerMixin } from 'chrome://resources/ash/common/cr_elements/web_ui_listener_mixin.js';
import { PolymerElement } from 'chrome://resources/polymer/v3_0/polymer/polymer_bundled.min.js';
import { recordSettingChange } from '../metrics_recorder.js';
import { Setting } from '../mojom-webui/setting.mojom-webui.js';
import { getTemplate } from './cups_nearby_printers.html.js';
import { matchesSearchTerm, sortPrinters } from './cups_printer_dialog_util.js';
import { CupsPrintersBrowserProxyImpl, PrinterSetupResult } from './cups_printers_browser_proxy.js';
import { CupsPrintersEntryListMixin } from './cups_printers_entry_list_mixin.js';
const SettingsCupsNearbyPrintersElementBase = CupsPrintersEntryListMixin(WebUiListenerMixin(PolymerElement));
export class SettingsCupsNearbyPrintersElement extends SettingsCupsNearbyPrintersElementBase {
    static get is() {
        return 'settings-cups-nearby-printers';
    }
    static get template() {
        return getTemplate();
    }
    static get properties() {
        return {
            /**
             * Search term for filtering |nearbyPrinters|.
             */
            searchTerm: {
                type: String,
                value: '',
            },
            /**
             * This value is set to true if UserPrintersAllowed policy is enabled.
             */
            userPrintersAllowed: {
                type: Boolean,
                value: false,
            },
            activePrinter: {
                type: Object,
                notify: true,
            },
            printersCount: {
                type: Number,
                computed: 'getFilteredPrintersLength_(filteredPrinters_.*)',
                notify: true,
            },
            activePrinterListEntryIndex_: {
                type: Number,
                value: -1,
            },
            /**
             * List of printers filtered through a search term.
             */
            filteredPrinters_: {
                type: Array,
                value: () => [],
            },
            /**
             * Used by FocusRowBehavior to track the last focused element on a row.
             */
            lastFocused_: Object,
            /**
             * Used by FocusRowBehavior to track if the list has been blurred.
             */
            listBlurred_: Boolean,
            /**
             * This is set to true while waiting for a response during a printer
             * setup.
             */
            savingPrinter_: {
                type: Boolean,
                value: false,
            },
        };
    }
    static get observers() {
        return ['onSearchOrPrintersChanged_(nearbyPrinters.*, searchTerm)'];
    }
    ready() {
        super.ready();
        this.addEventListener('add-automatic-printer', (event) => {
            this.onAddAutomaticPrinter_(event);
        });
        this.addEventListener('add-print-server-printer', (event) => {
            this.onAddPrintServerPrinter_(event);
        });
        this.addEventListener('query-discovered-printer', (event) => {
            this.onQueryDiscoveredPrinter_(event);
        });
    }
    /**
     * Redoes the search whenever |searchTerm| or |nearbyPrinters| changes.
     */
    onSearchOrPrintersChanged_() {
        if (!this.nearbyPrinters) {
            return;
        }
        // Filter printers through |searchTerm|. If |searchTerm| is empty,
        // |filteredPrinters_| is just |nearbyPrinters|.
        const updatedPrinters = this.searchTerm ?
            this.nearbyPrinters.filter((item) => matchesSearchTerm(item.printerInfo, this.searchTerm)) :
            this.nearbyPrinters.slice();
        updatedPrinters.sort(sortPrinters);
        this.updateList('filteredPrinters_', (printer) => printer.printerInfo.printerId, updatedPrinters);
    }
    onAddAutomaticPrinter_(e) {
        const item = e.detail.item;
        this.setActivePrinter_(item);
        this.savingPrinter_ = true;
        CupsPrintersBrowserProxyImpl.getInstance()
            .addDiscoveredPrinter(item.printerInfo.printerId)
            .then(this.onAddNearbyPrintersSucceeded_.bind(this, item.printerInfo.printerName), this.onAddNearbyPrinterFailed_.bind(this));
    }
    onAddPrintServerPrinter_(e) {
        const item = e.detail.item;
        this.setActivePrinter_(item);
        this.savingPrinter_ = true;
        CupsPrintersBrowserProxyImpl.getInstance()
            .addCupsPrinter(item.printerInfo)
            .then(this.onAddNearbyPrintersSucceeded_.bind(this, item.printerInfo.printerName), this.onAddNearbyPrinterFailed_.bind(this));
    }
    onQueryDiscoveredPrinter_(e) {
        const item = e.detail.item;
        this.setActivePrinter_(item);
        this.savingPrinter_ = true;
        CupsPrintersBrowserProxyImpl.getInstance()
            .addDiscoveredPrinter(item.printerInfo.printerId)
            .then(this.onQueryDiscoveredPrinterSucceeded_.bind(this, item.printerInfo.printerName), this.onQueryDiscoveredPrinterFailed_.bind(this));
    }
    /**
     * Retrieves the index of |item| in |nearbyPrinters_| and sets that printer as
     * the active printer.
     */
    setActivePrinter_(item) {
        this.activePrinterListEntryIndex_ = this.nearbyPrinters.findIndex((printer) => printer.printerInfo.printerId === item.printerInfo.printerId);
        this.activePrinter =
            this.get(['nearbyPrinters', this.activePrinterListEntryIndex_])
                .printerInfo;
    }
    showCupsPrinterToast_(resultCode, printerName) {
        const event = new CustomEvent('show-cups-printer-toast', {
            bubbles: true,
            composed: true,
            detail: {
                resultCode,
                printerName,
            },
        });
        this.dispatchEvent(event);
    }
    /**
     * Handler for addDiscoveredPrinter success.
     */
    onAddNearbyPrintersSucceeded_(printerName, result) {
        this.savingPrinter_ = false;
        this.showCupsPrinterToast_(result, printerName);
        recordSettingChange(Setting.kAddPrinter);
    }
    /**
     * Handler for addDiscoveredPrinter failure.
     */
    onAddNearbyPrinterFailed_(printer) {
        this.savingPrinter_ = false;
        this.showCupsPrinterToast_(PrinterSetupResult.PRINTER_UNREACHABLE, printer.printerName);
    }
    /**
     * Handler for queryDiscoveredPrinter success.
     */
    onQueryDiscoveredPrinterSucceeded_(printerName, result) {
        this.savingPrinter_ = false;
        this.showCupsPrinterToast_(result, printerName);
        chrome.metricsPrivate.recordEnumerationValue('Printing.CUPS.PrinterSetupResult.SettingsDiscoveredPrinters', result, Object.keys(PrinterSetupResult).length);
        recordSettingChange(Setting.kAddPrinter);
    }
    /**
     * Handler for queryDiscoveredPrinter failure.
     */
    onQueryDiscoveredPrinterFailed_(printer) {
        this.savingPrinter_ = false;
        const openManufacturerDialogEvent = new CustomEvent('open-manufacturer-model-dialog-for-specified-printer', {
            bubbles: true,
            composed: true,
            detail: { item: printer },
        });
        this.dispatchEvent(openManufacturerDialogEvent);
        chrome.metricsPrivate.recordEnumerationValue('Printing.CUPS.PrinterSetupResult.SettingsDiscoveredPrinters', PrinterSetupResult.MANUAL_SETUP_REQUIRED, Object.keys(PrinterSetupResult).length);
    }
    /**
     * @return Returns true if the no search message should be visible.
     */
    showNoSearchResultsMessage_() {
        return !!this.searchTerm && !this.filteredPrinters_.length;
    }
    getFilteredPrintersLength_() {
        return this.filteredPrinters_.length;
    }
    /**
     * Forces the printer list to re-render all items.
     */
    resizePrintersList() {
        this.shadowRoot.querySelector('#printerEntryList').notifyResize();
    }
}
customElements.define(SettingsCupsNearbyPrintersElement.is, SettingsCupsNearbyPrintersElement);
