// Copyright 2022 The Chromium Authors
// Use of this source code is governed by a BSD-style license that can be
// found in the LICENSE file.
/**
 * @fileoverview 'settings-cups-printers-entry' is a component that holds a
 * printer.
 */
import 'chrome://resources/ash/common/cr_elements/cr_button/cr_button.js';
import 'chrome://resources/polymer/v3_0/iron-icon/iron-icon.js';
import '../settings_shared.css.js';
import './cups_printer_types.js';
import { FocusRowMixin } from 'chrome://resources/ash/common/cr_elements/focus_row_mixin.js';
import { I18nMixin } from 'chrome://resources/ash/common/cr_elements/i18n_mixin.js';
import { assertNotReached } from 'chrome://resources/js/assert.js';
import { loadTimeData } from 'chrome://resources/js/load_time_data.js';
import { PolymerElement } from 'chrome://resources/polymer/v3_0/polymer/polymer_bundled.min.js';
import { PrinterType } from './cups_printer_types.js';
import { PrinterSettingsUserAction, recordPrinterSettingsUserAction } from './cups_printers.js';
import { getTemplate } from './cups_printers_entry.html.js';
import { computePrinterState, PrinterState, STATUS_REASON_STRING_KEY_MAP } from './printer_status.js';
const SettingsCupsPrintersEntryElementBase = FocusRowMixin(I18nMixin(PolymerElement));
export class SettingsCupsPrintersEntryElement extends SettingsCupsPrintersEntryElementBase {
    static get is() {
        return 'settings-cups-printers-entry';
    }
    static get template() {
        return getTemplate();
    }
    static get properties() {
        return {
            printerEntry: Object,
            /**
             * This value is set to true if the printer is in saving mode.
             */
            savingPrinter: Boolean,
            /**
             * This value is set to true if UserPrintersAllowed policy is enabled.
             */
            userPrintersAllowed: {
                type: Boolean,
                value: false,
            },
            /**
             * The cache of printer status reasons used to look up this entry's
             * current printer status. Populated and maintained by
             * cups_saved_printers.ts.
             */
            printerStatusReasonCache: (Map),
            hasHighSeverityError_: {
                type: Boolean,
                value: false,
                reflectToAttribute: true,
            },
            /** Number of printers in the respective list this row is part of. */
            numPrinters: Number,
        };
    }
    /**
     * Fires a custom event when the menu button is clicked. Sends the details of
     * the printer and where the menu should appear.
     */
    onOpenActionMenuClick_(e) {
        const openActionMenuEvent = new CustomEvent('open-action-menu', {
            bubbles: true,
            composed: true,
            detail: {
                target: e.target,
                item: this.printerEntry,
            },
        });
        this.dispatchEvent(openActionMenuEvent);
    }
    onAddDiscoveredPrinterClick_() {
        const queryDiscoveredPrinterEvent = new CustomEvent('query-discovered-printer', {
            bubbles: true,
            composed: true,
            detail: { item: this.printerEntry },
        });
        this.dispatchEvent(queryDiscoveredPrinterEvent);
        recordPrinterSettingsUserAction(PrinterSettingsUserAction.SAVE_PRINTER);
    }
    onAddAutomaticPrinterClick_() {
        const addAutomaticPrinterEvent = new CustomEvent('add-automatic-printer', {
            bubbles: true,
            composed: true,
            detail: { item: this.printerEntry },
        });
        this.dispatchEvent(addAutomaticPrinterEvent);
        recordPrinterSettingsUserAction(PrinterSettingsUserAction.SAVE_PRINTER);
    }
    onAddServerPrinterClick_() {
        const addPrintServer = new CustomEvent('add-print-server-printer', {
            bubbles: true,
            composed: true,
            detail: { item: this.printerEntry },
        });
        this.dispatchEvent(addPrintServer);
    }
    showActionsMenu_() {
        return this.printerEntry.printerType === PrinterType.SAVED ||
            this.printerEntry.printerType === PrinterType.ENTERPRISE;
    }
    isDiscoveredPrinter_() {
        return this.printerEntry.printerType === PrinterType.DISCOVERED;
    }
    isAutomaticPrinter_() {
        return this.printerEntry.printerType === PrinterType.AUTOMATIC;
    }
    isPrintServerPrinter_() {
        return this.printerEntry.printerType === PrinterType.PRINTSERVER;
    }
    isSavedPrinter_() {
        return this.printerEntry.printerType === PrinterType.SAVED;
    }
    isEnterprisePrinter_() {
        return this.printerEntry.printerType === PrinterType.ENTERPRISE;
    }
    isConfigureDisabled_() {
        return !this.userPrintersAllowed || this.savingPrinter;
    }
    getSaveButtonAria_() {
        return loadTimeData.getStringF('savePrinterAria', this.printerEntry.printerInfo.printerName);
    }
    getSetupButtonAria_() {
        return loadTimeData.getStringF('setupPrinterAria', this.printerEntry.printerInfo.printerName);
    }
    getPrinterIcon_() {
        // Only saved printers need to display an icon with printer status.
        if (!this.isSavedPrinter_()) {
            return 'os-settings:printer-plain';
        }
        const printerStatusIcon = 'os-settings:printer-status-illo';
        const printerStatusReason = this.printerStatusReasonCache.get(this.printerEntry.printerInfo.printerId);
        if (printerStatusReason === undefined || printerStatusReason === null) {
            return `${printerStatusIcon}-grey`;
        }
        let iconColor = '';
        switch (computePrinterState(printerStatusReason)) {
            case PrinterState.GOOD:
                iconColor = 'green';
                break;
            case PrinterState.LOW_SEVERITY_ERROR:
                iconColor = 'orange';
                break;
            case PrinterState.HIGH_SEVERITY_ERROR:
                iconColor = 'red';
                break;
            case PrinterState.UNKNOWN:
                iconColor = 'grey';
                break;
            default:
                assertNotReached('Invalid PrinterState');
        }
        return `${printerStatusIcon}-${iconColor}`;
    }
    getStatusReasonString_() {
        // Only saved printers need to display printer status text.
        if (!this.isSavedPrinter_()) {
            return window.trustedTypes.emptyHTML;
        }
        const printerStatusReason = this.printerStatusReasonCache.get(this.printerEntry.printerInfo.printerId);
        if (!printerStatusReason) {
            return window.trustedTypes.emptyHTML;
        }
        // Use the printer state to determine printer status text color.
        this.hasHighSeverityError_ = computePrinterState(printerStatusReason) ===
            PrinterState.HIGH_SEVERITY_ERROR;
        // Use the printer state to determine printer status text content.
        const statusReasonStringKey = STATUS_REASON_STRING_KEY_MAP.get(printerStatusReason);
        return statusReasonStringKey ? this.i18nAdvanced(statusReasonStringKey) :
            window.trustedTypes.emptyHTML;
    }
    getAriaLabel_() {
        if (!this.printerEntry) {
            return '';
        }
        return this.i18n('printerEntryAriaLabel', this.printerEntry.printerInfo.printerName, this.getStatusReasonString_().toString(), this.focusRowIndex + 1, this.numPrinters);
    }
}
customElements.define(SettingsCupsPrintersEntryElement.is, SettingsCupsPrintersEntryElement);
