// Copyright 2023 The Chromium Authors
// Use of this source code is governed by a BSD-style license that can be
// found in the LICENSE file.
import { ListPropertyUpdateMixin } from 'chrome://resources/ash/common/cr_elements/list_property_update_mixin.js';
import { assert } from 'chrome://resources/js/assert.js';
import { dedupingMixin } from 'chrome://resources/polymer/v3_0/polymer/polymer_bundled.min.js';
import { findDifference } from './cups_printer_dialog_util.js';
import { CupsPrintersEntryManager } from './cups_printers_entry_manager.js';
export const CupsPrintersEntryListMixin = dedupingMixin((superClass) => {
    const superclassBase = ListPropertyUpdateMixin(superClass);
    class CupsPrintersEntryListMixinInternal extends superclassBase {
        static get properties() {
            return {
                entryManager_: Object,
                savedPrinters: {
                    type: Array,
                    value: () => [],
                },
                nearbyPrinters: {
                    type: Array,
                    value: () => [],
                },
                enterprisePrinters: {
                    type: Array,
                    value: () => [],
                },
            };
        }
        constructor() {
            super();
            this.entryManager_ = CupsPrintersEntryManager.getInstance();
        }
        connectedCallback() {
            this.entryManager_.addOnSavedPrintersChangedListener(this.onSavedPrintersChanged_.bind(this));
            this.entryManager_.addOnNearbyPrintersChangedListener(this.onNearbyPrintersChanged_.bind(this));
            this.entryManager_.addOnEnterprisePrintersChangedListener(this.onEnterprisePrintersChanged_.bind(this));
            // Initialize saved and nearby printers list.
            this.onSavedPrintersChanged_(this.entryManager_.savedPrinters, [] /* printerAdded */, [] /* printerRemoved */);
            this.onNearbyPrintersChanged_(this.entryManager_.nearbyPrinters);
            this.onEnterprisePrintersChanged_(this.entryManager_.enterprisePrinters);
        }
        disconnectedCallback() {
            this.entryManager_.removeOnSavedPrintersChangedListener(this.onSavedPrintersChanged_.bind(this));
            this.entryManager_.removeOnNearbyPrintersChangedListener(this.onNearbyPrintersChanged_.bind(this));
            this.entryManager_.removeOnEnterprisePrintersChangedListener(this.onEnterprisePrintersChanged_.bind(this));
        }
        /**
         * Non-empty params indicate the applicable change to be notified.
         */
        onSavedPrintersChanged_(savedPrinters, addedPrinters, removedPrinters) {
            this.updateList('savedPrinters', printer => printer.printerInfo.printerId, savedPrinters);
            assert(!(addedPrinters.length && removedPrinters.length));
            if (addedPrinters.length) {
                this.onSavedPrintersAdded(addedPrinters);
            }
            else if (removedPrinters.length) {
                this.onSavedPrintersRemoved(removedPrinters);
            }
        }
        onNearbyPrintersChanged_(printerList) {
            // |printerList| consists of automatic and discovered printers that
            // have not been saved and are available. Add all unsaved print server
            // printers to |printerList|.
            this.entryManager_.printServerPrinters = findDifference(this.entryManager_.printServerPrinters, this.savedPrinters);
            printerList =
                printerList.concat(this.entryManager_.printServerPrinters);
            this.updateList('nearbyPrinters', printer => printer.printerInfo.printerId, printerList);
        }
        onEnterprisePrintersChanged_(enterprisePrinters) {
            this.updateList('enterprisePrinters', printer => printer.printerInfo.printerId, enterprisePrinters);
        }
        // Override in the custom element implementation
        onSavedPrintersAdded(_addedPrinters) { }
        // Override in the custom element implementation
        onSavedPrintersRemoved(_removedPrinters) { }
    }
    return CupsPrintersEntryListMixinInternal;
});
