// Copyright 2023 The Chromium Authors
// Use of this source code is governed by a BSD-style license that can be
// found in the LICENSE file.
/**
 * @fileoverview
 * 'settings-privacy-hub-geolocation-subpage' contains a detailed overview about
 * the state of the system geolocation access.
 */
import './privacy_hub_app_permission_row.js';
import 'chrome://resources/ash/common/cr_elements/cr_radio_button/cr_radio_button_style.css.js';
import 'chrome://resources/ash/common/cr_elements/icons.html.js';
import 'chrome://resources/polymer/v3_0/iron-icon/iron-icon.js';
import '../controls/controlled_button.js';
import '../os_settings_icons.html.js';
import { PrefsMixin } from '/shared/settings/prefs/prefs_mixin.js';
import { I18nMixin } from 'chrome://resources/ash/common/cr_elements/i18n_mixin.js';
import { WebUiListenerMixin } from 'chrome://resources/ash/common/cr_elements/web_ui_listener_mixin.js';
import { PermissionType } from 'chrome://resources/cr_components/app_management/app_management.mojom-webui.js';
import { isPermissionEnabled } from 'chrome://resources/cr_components/app_management/permission_util.js';
import { loadTimeData } from 'chrome://resources/js/load_time_data.js';
import { OpenWindowProxyImpl } from 'chrome://resources/js/open_window_proxy.js';
import { PolymerElement } from 'chrome://resources/polymer/v3_0/polymer/polymer_bundled.min.js';
import { assertExhaustive, castExists } from '../assert_extras.js';
import { DeepLinkingMixin } from '../common/deep_linking_mixin.js';
import { isSecondaryUser } from '../common/load_time_booleans.js';
import { RouteObserverMixin } from '../common/route_observer_mixin.js';
import { AppPermissionsObserverReceiver } from '../mojom-webui/app_permission_handler.mojom-webui.js';
import { Setting } from '../mojom-webui/setting.mojom-webui.js';
import { Router, routes } from '../router.js';
import { getAppPermissionProvider } from './mojo_interface_provider.js';
import { PrivacyHubBrowserProxyImpl } from './privacy_hub_browser_proxy.js';
import { getTemplate } from './privacy_hub_geolocation_subpage.html.js';
import { LOCATION_PERMISSION_CHANGE_FROM_SETTINGS_HISTOGRAM_NAME } from './privacy_hub_metrics_util.js';
/**
 * Geolocation access levels for the ChromeOS system.
 * This must be kept in sync with `GeolocationAccessLevel` in
 * ash/constants/geolocation_access_level.h
 */
export var GeolocationAccessLevel;
(function (GeolocationAccessLevel) {
    GeolocationAccessLevel[GeolocationAccessLevel["DISALLOWED"] = 0] = "DISALLOWED";
    GeolocationAccessLevel[GeolocationAccessLevel["ALLOWED"] = 1] = "ALLOWED";
    GeolocationAccessLevel[GeolocationAccessLevel["ONLY_ALLOWED_FOR_SYSTEM"] = 2] = "ONLY_ALLOWED_FOR_SYSTEM";
    GeolocationAccessLevel[GeolocationAccessLevel["MAX_VALUE"] = 2] = "MAX_VALUE";
})(GeolocationAccessLevel || (GeolocationAccessLevel = {}));
export var ScheduleType;
(function (ScheduleType) {
    ScheduleType[ScheduleType["NONE"] = 0] = "NONE";
    ScheduleType[ScheduleType["SUNSET_TO_SUNRISE"] = 1] = "SUNSET_TO_SUNRISE";
    ScheduleType[ScheduleType["CUSTOM"] = 2] = "CUSTOM";
})(ScheduleType || (ScheduleType = {}));
/**
 * Whether the app has location permission defined.
 */
function hasLocationPermission(app) {
    return app.permissions[PermissionType.kLocation] !== undefined;
}
const SettingsPrivacyHubGeolocationSubpageBase = RouteObserverMixin(DeepLinkingMixin(WebUiListenerMixin(PrefsMixin(I18nMixin(PolymerElement)))));
export class SettingsPrivacyHubGeolocationSubpage extends SettingsPrivacyHubGeolocationSubpageBase {
    static get is() {
        return 'settings-privacy-hub-geolocation-subpage';
    }
    static get template() {
        return getTemplate();
    }
    static get properties() {
        return {
            geolocationAccessLevelPrefValues_: {
                readOnly: true,
                type: Object,
                value: {
                    DISALLOWED: GeolocationAccessLevel.DISALLOWED,
                    ALLOWED: GeolocationAccessLevel.ALLOWED,
                    ONLY_ALLOWED_FOR_SYSTEM: GeolocationAccessLevel.ONLY_ALLOWED_FOR_SYSTEM,
                },
            },
            /**
             * Apps with location permission defined.
             */
            appList_: {
                type: Array,
                value: [],
            },
            automaticTimeZoneText_: {
                type: String,
                computed: 'computeAutomaticTimeZoneText_(' +
                    'systemGeolocationAccessLevel_,' +
                    'prefs.generated.resolve_timezone_by_geolocation_on_off.value,' +
                    'currentTimeZoneName_)',
            },
            isSecondaryUser_: {
                type: Boolean,
                value() {
                    return isSecondaryUser();
                },
                readOnly: true,
            },
            systemGeolocationAccessLevel_: {
                type: Number,
                value: GeolocationAccessLevel.DISALLOWED,
            },
            isGeolocationAllowedForApps_: {
                type: Boolean,
                computed: 'computedIsGeolocationAllowedForApps_(' +
                    'systemGeolocationAccessLevel_)',
            },
            currentTimeZoneName_: {
                type: String,
                notify: true,
            },
            currentSunRiseTime_: {
                type: String,
                notify: true,
            },
            currentSunSetTime_: {
                type: String,
                notify: true,
            },
            nightLightText_: {
                type: String,
                computed: 'computeNightLightText_(' +
                    'systemGeolocationAccessLevel_,' +
                    `prefs.ash.night_light.schedule_type.value,` +
                    'currentSunRiseTime_, currentSunSetTime_)',
            },
            darkThemeText_: {
                type: String,
                computed: 'computeDarkThemeText_(' +
                    'systemGeolocationAccessLevel_,' +
                    `prefs.ash.dark_mode.schedule_type.value,` +
                    'currentSunRiseTime_, currentSunSetTime_)',
            },
            localWeatherText_: {
                type: String,
                computed: 'computeLocalWeatherText_(' +
                    'systemGeolocationAccessLevel_,' +
                    'prefs.settings.ambient_mode.enabled.value)',
            },
        };
    }
    static get observers() {
        return [
            'onTimeZoneChanged_(prefs.cros.system.timezone.value)',
        ];
    }
    currentRouteChanged(route) {
        // Does not apply to this page.
        if (route !== routes.PRIVACY_HUB_GEOLOCATION_ADVANCED) {
            return;
        }
        this.attemptDeepLink();
    }
    constructor() {
        super();
        /**
         * Used by DeepLinkingMixin to focus this page's deep links.
         */
        this.supportedSettingIds = new Set([
            Setting.kGeolocationAdvanced,
        ]);
        this.mojoInterfaceProvider_ = getAppPermissionProvider();
        this.appPermissionsObserverReceiver_ = null;
        this.browserProxy_ = PrivacyHubBrowserProxyImpl.getInstance();
        // Assigning the initial time zone name.
        this.currentTimeZoneName_ = this.i18n('timeZoneName');
        this.currentSunRiseTime_ =
            this.i18n('privacyHubSystemServicesInitSunRiseTime');
        this.currentSunSetTime_ =
            this.i18n('privacyHubSystemServicesInitSunSetTime');
    }
    ready() {
        super.ready();
        this.addWebUiListener('system-geolocation-access-level-changed', (accessLevel) => {
            this.systemGeolocationAccessLevel_ = accessLevel;
        });
        this.browserProxy_.getInitialGeolocationAccessLevelState().then((accessLevel) => {
            this.systemGeolocationAccessLevel_ = accessLevel;
        });
    }
    connectedCallback() {
        super.connectedCallback();
        this.appPermissionsObserverReceiver_ =
            new AppPermissionsObserverReceiver(this);
        this.mojoInterfaceProvider_.addObserver(this.appPermissionsObserverReceiver_.$.bindNewPipeAndPassRemote());
        this.updateAppList_();
    }
    disconnectedCallback() {
        super.disconnectedCallback();
        this.appPermissionsObserverReceiver_.$.close();
    }
    isGeolocationPrefEnforced_() {
        return this.getPref('ash.user.geolocation_access_level').enforcement ===
            chrome.settingsPrivate.Enforcement.ENFORCED;
    }
    isGeolocationModifiable_() {
        if (isSecondaryUser() || this.isGeolocationPrefEnforced_()) {
            return false;
        }
        return true;
    }
    showManageGeolocationDialog_() {
        if (this.isGeolocationModifiable_()) {
            this.shouldShowManageGeolocationDialog_ = true;
        }
    }
    onCancelClicked_() {
        const radioGroup = this.shadowRoot.querySelector('#manageGeolocationRadioGroup');
        radioGroup.resetToPrefValue();
        this.shouldShowManageGeolocationDialog_ = false;
    }
    onConfirmClicked_() {
        // Reflect user choice to the underlying pref.
        const radioGroup = this.shadowRoot.querySelector('#manageGeolocationRadioGroup');
        radioGroup.sendPrefChange();
        // Record metrics.
        this.recordMetric_();
        // Dismiss the dialog.
        this.shouldShowManageGeolocationDialog_ = false;
    }
    settingControlledByPrimaryUserText_() {
        return this.i18n('geolocationControlledByPrimaryUserText', loadTimeData.getString('primaryUserEmail'));
    }
    /**
     * The function is used for sorting app names alphabetically.
     */
    alphabeticalSort_(first, second) {
        return first.name.localeCompare(second.name);
    }
    async updateAppList_() {
        const apps = (await this.mojoInterfaceProvider_.getApps()).apps;
        this.appList_ = apps.filter(hasLocationPermission);
    }
    isLocationPermissionEnabled_(app) {
        const permission = castExists(app.permissions[PermissionType.kLocation]);
        return isPermissionEnabled(permission.value);
    }
    /** Implements AppPermissionsObserver.OnAppUpdated */
    onAppUpdated(updatedApp) {
        if (!hasLocationPermission(updatedApp)) {
            return;
        }
        const idx = this.appList_.findIndex(app => app.id === updatedApp.id);
        if (idx === -1) {
            // New app installed.
            this.push('appList_', updatedApp);
        }
        else {
            // An already installed app is updated.
            this.splice('appList_', idx, 1, updatedApp);
        }
    }
    /** Implements AppPermissionsObserver.OnAppRemoved */
    onAppRemoved(appId) {
        const idx = this.appList_.findIndex(app => app.id === appId);
        if (idx !== -1) {
            this.splice('appList_', idx, 1);
        }
    }
    computedIsGeolocationAllowedForApps_() {
        const accessLevel = this.systemGeolocationAccessLevel_;
        switch (accessLevel) {
            case GeolocationAccessLevel.ALLOWED:
                return true;
            case GeolocationAccessLevel.DISALLOWED:
            case GeolocationAccessLevel.ONLY_ALLOWED_FOR_SYSTEM:
                return false;
            default:
                assertExhaustive(accessLevel);
        }
    }
    isAutomaticTimeZoneConfigured_() {
        return this.getPref('generated.resolve_timezone_by_geolocation_on_off')
            .value;
    }
    computeAutomaticTimeZoneText_() {
        if (!this.prefs) {
            return '';
        }
        if (!this.isAutomaticTimeZoneConfigured_()) {
            return this.i18n('privacyHubSystemServicesGeolocationNotConfigured');
        }
        return this.geolocationAllowedForSystem_() ?
            this.i18n('privacyHubSystemServicesAllowedText') :
            this.i18n('privacyHubSystemServicesAutomaticTimeZoneBlockedText', this.currentTimeZoneName_);
    }
    isNightLightConfiguredToUseGeolocation_() {
        return this.getPref('ash.night_light.schedule_type').value ===
            ScheduleType.SUNSET_TO_SUNRISE;
    }
    computeNightLightText_() {
        if (!this.prefs) {
            return '';
        }
        if (!this.isNightLightConfiguredToUseGeolocation_()) {
            return this.i18n('privacyHubSystemServicesGeolocationNotConfigured');
        }
        return this.geolocationAllowedForSystem_() ?
            this.i18n('privacyHubSystemServicesAllowedText') :
            this.i18n('privacyHubSystemServicesSunsetScheduleBlockedText', this.currentSunRiseTime_, this.currentSunSetTime_);
    }
    onManagePermissionsInChromeRowClick_() {
        this.mojoInterfaceProvider_.openBrowserPermissionSettings(PermissionType.kLocation);
    }
    computeGeolocationAccessLevelText_() {
        const accessLevel = this.systemGeolocationAccessLevel_;
        switch (accessLevel) {
            case GeolocationAccessLevel.ALLOWED:
                return this.i18nAdvanced('geolocationAccessLevelAllowed');
            case GeolocationAccessLevel.ONLY_ALLOWED_FOR_SYSTEM:
                return this.i18nAdvanced('geolocationAccessLevelOnlyAllowedForSystem');
            case GeolocationAccessLevel.DISALLOWED:
                return this.i18nAdvanced('geolocationAccessLevelDisallowed');
            default:
                assertExhaustive(accessLevel);
        }
    }
    computeGeolocationAccessLevelDescriptionText_() {
        const accessLevel = this.systemGeolocationAccessLevel_;
        switch (accessLevel) {
            case GeolocationAccessLevel.ALLOWED:
                return this.i18nAdvanced('geolocationAllowedModeDescription');
            case GeolocationAccessLevel.ONLY_ALLOWED_FOR_SYSTEM:
                return this.i18nAdvanced('geolocationOnlyAllowedForSystemModeDescription');
            case GeolocationAccessLevel.DISALLOWED:
                return this.i18nAdvanced('geolocationBlockedModeDescription');
            default:
                assertExhaustive(accessLevel);
        }
    }
    recordMetric_() {
        const accessLevel = this.getPref('ash.user.geolocation_access_level')
            .value;
        chrome.metricsPrivate.recordEnumerationValue(LOCATION_PERMISSION_CHANGE_FROM_SETTINGS_HISTOGRAM_NAME, accessLevel, GeolocationAccessLevel.MAX_VALUE + 1);
    }
    geolocationAllowedForSystem_() {
        return this.systemGeolocationAccessLevel_ !==
            GeolocationAccessLevel.DISALLOWED;
    }
    isLocalWeatherConfiguredToUseGeolocation_() {
        return this.getPref('settings.ambient_mode.enabled').value;
    }
    computeLocalWeatherText_() {
        if (!this.prefs) {
            return '';
        }
        if (!this.isLocalWeatherConfiguredToUseGeolocation_()) {
            return this.i18n('privacyHubSystemServicesGeolocationNotConfigured');
        }
        return this.geolocationAllowedForSystem_() ?
            this.i18n('privacyHubSystemServicesAllowedText') :
            this.i18n('privacyHubSystemServicesBlockedText');
    }
    isDarkThemeConfiguredToUseGeolocation_() {
        return this.getPref('ash.dark_mode.schedule_type').value ===
            ScheduleType.SUNSET_TO_SUNRISE;
    }
    computeDarkThemeText_() {
        if (!this.prefs) {
            return '';
        }
        if (!this.isDarkThemeConfiguredToUseGeolocation_()) {
            return this.i18n('privacyHubSystemServicesGeolocationNotConfigured');
        }
        return this.geolocationAllowedForSystem_() ?
            this.i18n('privacyHubSystemServicesAllowedText') :
            this.i18n('privacyHubSystemServicesSunsetScheduleBlockedText', this.currentSunRiseTime_, this.currentSunSetTime_);
    }
    onTimeZoneChanged_() {
        this.browserProxy_.getCurrentTimeZoneName().then((timeZoneName) => {
            this.currentTimeZoneName_ = timeZoneName;
        });
        this.browserProxy_.getCurrentSunriseTime().then((time) => {
            this.currentSunRiseTime_ = time;
        });
        this.browserProxy_.getCurrentSunsetTime().then((time) => {
            this.currentSunSetTime_ = time;
        });
    }
    onTimeZoneClick_() {
        Router.getInstance().navigateTo(routes.DATETIME_TIMEZONE_SUBPAGE);
    }
    onNightLightClick_() {
        Router.getInstance().navigateTo(routes.DISPLAY);
    }
    onDarkThemeClick_() {
        OpenWindowProxyImpl.getInstance().openUrl(loadTimeData.getString('personalizationAppUrl'));
    }
    onLocalWeatherClick_() {
        OpenWindowProxyImpl.getInstance().openUrl(loadTimeData.getString('personalizationAppUrl') +
            loadTimeData.getString('ambientSubpageRelativeUrl'));
    }
    onGeolocationAdvancedAreaClick_() {
        Router.getInstance().navigateTo(routes.PRIVACY_HUB_GEOLOCATION_ADVANCED);
    }
}
customElements.define(SettingsPrivacyHubGeolocationSubpage.is, SettingsPrivacyHubGeolocationSubpage);
