// Copyright 2019 The Chromium Authors
// Use of this source code is governed by a BSD-style license that can be
// found in the LICENSE file.
/**
 * @fileoverview
 * 'os-settings-menu' shows a menu with a hardcoded set of pages and subpages.
 */
import 'chrome://resources/ash/common/cr_elements/cr_button/cr_button.js';
import 'chrome://resources/ash/common/cr_elements/icons.html.js';
import 'chrome://resources/polymer/v3_0/iron-collapse/iron-collapse.js';
import 'chrome://resources/polymer/v3_0/iron-icon/iron-icon.js';
import 'chrome://resources/polymer/v3_0/iron-selector/iron-selector.js';
import '../settings_shared.css.js';
import '../os_settings_icons.html.js';
import './menu_item.js';
import { getDeviceNameUnsafe } from 'chrome://resources/ash/common/bluetooth/bluetooth_utils.js';
import { getBluetoothConfig } from 'chrome://resources/ash/common/bluetooth/cros_bluetooth_config.js';
import { I18nMixin } from 'chrome://resources/ash/common/cr_elements/i18n_mixin.js';
import { WebUiListenerMixin } from 'chrome://resources/ash/common/cr_elements/web_ui_listener_mixin.js';
import { MojoInterfaceProviderImpl } from 'chrome://resources/ash/common/network/mojo_interface_provider.js';
import { NetworkListenerBehavior } from 'chrome://resources/ash/common/network/network_listener_behavior.js';
import { OncMojo } from 'chrome://resources/ash/common/network/onc_mojo.js';
import { BluetoothSystemState, DeviceConnectionState, SystemPropertiesObserverReceiver as BluetoothPropertiesObserverReceiver } from 'chrome://resources/mojo/chromeos/ash/services/bluetooth_config/public/mojom/cros_bluetooth_config.mojom-webui.js';
import { FilterType, NO_LIMIT } from 'chrome://resources/mojo/chromeos/services/network_config/public/mojom/cros_network_config.mojom-webui.js';
import { NetworkType } from 'chrome://resources/mojo/chromeos/services/network_config/public/mojom/network_types.mojom-webui.js';
import { mixinBehaviors, PolymerElement } from 'chrome://resources/polymer/v3_0/polymer/polymer_bundled.min.js';
import { assertExists, castExists } from '../assert_extras.js';
import { androidAppsVisible } from '../common/load_time_booleans.js';
import { RouteObserverMixin } from '../common/route_observer_mixin.js';
import { FakeInputDeviceSettingsProvider } from '../device_page/fake_input_device_settings_provider.js';
import { getInputDeviceSettingsProvider } from '../device_page/input_device_mojo_interface_provider.js';
import { KeyboardSettingsObserverReceiver, MouseSettingsObserverReceiver, PointingStickSettingsObserverReceiver, TouchpadSettingsObserverReceiver } from '../mojom-webui/input_device_settings_provider.mojom-webui.js';
import * as routesMojom from '../mojom-webui/routes.mojom-webui.js';
import { MultiDeviceBrowserProxyImpl } from '../multidevice_page/multidevice_browser_proxy.js';
import { MultiDeviceSettingsMode } from '../multidevice_page/multidevice_constants.js';
import { AccountManagerBrowserProxyImpl } from '../os_people_page/account_manager_browser_proxy.js';
import { getTemplate } from './os_settings_menu.html.js';
const { Section } = routesMojom;
/**
 * Returns a copy of the given `str` with the first letter capitalized according
 * to the locale.
 */
function capitalize(str) {
    const firstChar = str.charAt(0).toLocaleUpperCase();
    const remainingStr = str.slice(1);
    return `${firstChar}${remainingStr}`;
}
function getPrioritizedConnectedNetwork(networkStateList) {
    // The priority of the network types. Both Cellular and Tether belongs to
    // the Mobile Data.
    const orderedNetworkTypes = [
        NetworkType.kEthernet,
        NetworkType.kWiFi,
        NetworkType.kCellular,
        NetworkType.kTether,
        NetworkType.kVPN,
    ];
    const networkStates = {};
    for (const networkType of orderedNetworkTypes) {
        networkStates[networkType] = [];
    }
    for (const networkState of networkStateList) {
        networkStates[networkState.type].push(networkState);
    }
    for (const type of orderedNetworkTypes) {
        for (const networkState of networkStates[type]) {
            if (OncMojo.connectionStateIsConnected(networkState.connectionState)) {
                return networkState;
            }
        }
    }
    return null;
}
const OsSettingsMenuElementBase = mixinBehaviors([NetworkListenerBehavior], WebUiListenerMixin(RouteObserverMixin(I18nMixin(PolymerElement))));
export class OsSettingsMenuElement extends OsSettingsMenuElementBase {
    static get is() {
        return 'os-settings-menu';
    }
    static get template() {
        return getTemplate();
    }
    static get properties() {
        return {
            /**
             * Determines which menu items are available for their respective pages
             */
            pageAvailability: {
                type: Object,
            },
            /**
             * If this menu exists in the drawer. Used to compute responsiveness in
             * smaller window sizes.
             */
            isDrawerMenu: {
                type: Boolean,
                value: false,
            },
            menuItems_: {
                type: Array,
                computed: 'computeMenuItems_(pageAvailability.*,' +
                    'accountsMenuItemDescription_,' +
                    'bluetoothMenuItemDescription_,' +
                    'deviceMenuItemDescription_,' +
                    'internetMenuItemDescription_,' +
                    'multideviceMenuItemDescription_)',
                readOnly: true,
            },
            /**
             * The path of the currently selected menu item. e.g. '/internet'.
             */
            selectedItemPath_: {
                type: String,
                value: '',
            },
            accountsMenuItemDescription_: {
                type: String,
                value() {
                    return this.i18n('primaryUserEmail');
                },
            },
            bluetoothMenuItemDescription_: {
                type: String,
                value: '',
            },
            hasKeyboard_: Boolean,
            hasMouse_: Boolean,
            /**
             * Whether a pointing stick (such as a TrackPoint) is connected.
             */
            hasPointingStick_: Boolean,
            hasTouchpad_: Boolean,
            deviceMenuItemDescription_: {
                type: String,
                value: '',
                computed: 'computeDeviceMenuItemDescription_(hasKeyboard_,' +
                    'hasMouse_, hasPointingStick_, hasTouchpad_)',
            },
            multideviceMenuItemDescription_: {
                type: String,
                value: '',
            },
            internetMenuItemDescription_: {
                type: String,
                value: '',
            },
            isRtl_: {
                type: Boolean,
                value: false,
            },
        };
    }
    constructor() {
        super();
        this.inputDeviceSettingsProvider_ = getInputDeviceSettingsProvider();
        this.multideviceBrowserProxy_ = MultiDeviceBrowserProxyImpl.getInstance();
    }
    connectedCallback() {
        super.connectedCallback();
        // Accounts menu item is not available in guest mode.
        if (this.pageAvailability[Section.kPeople]) {
            this.updateAccountsMenuItemDescription_();
            this.addWebUiListener('accounts-changed', this.updateAccountsMenuItemDescription_.bind(this));
        }
        // Bluetooth menu item.
        this.observeBluetoothProperties_();
        // Device menu item.
        this.observeKeyboardSettings_();
        this.observeMouseSettings_();
        this.observePointingStickSettings_();
        this.observeTouchpadSettings_();
        // Internet menu item.
        this.networkConfig_ =
            MojoInterfaceProviderImpl.getInstance().getMojoServiceRemote();
        this.computeIsDeviceCellularCapable_().then(() => {
            this.updateInternetMenuItemDescription_();
        });
        // Multidevice menu item is not available in guest mode.
        if (this.pageAvailability[Section.kMultiDevice]) {
            this.addWebUiListener('settings.updateMultidevicePageContentData', this.updateMultideviceMenuItemDescription_.bind(this));
            this.multideviceBrowserProxy_.getPageContentData().then(this.updateMultideviceMenuItemDescription_.bind(this));
        }
        this.isRtl_ = window.getComputedStyle(this).direction === 'rtl';
    }
    disconnectedCallback() {
        super.disconnectedCallback();
        this.bluetoothPropertiesObserverReceiver_?.$.close();
        // The following receivers are undefined in tests.
        this.keyboardSettingsObserverReceiver_?.$.close();
        this.mouseSettingsObserverReceiver_?.$.close();
        this.pointingStickSettingsObserverReceiver_?.$.close();
        this.touchpadSettingsObserverReceiver_?.$.close();
    }
    ready() {
        super.ready();
        // Force render menu items so the matching item can be selected when the
        // page initially loads.
        this.$.topMenuRepeat.render();
    }
    currentRouteChanged(newRoute) {
        this.setSelectedItemPathForRoute_(newRoute);
    }
    /**
     * The selected menu item should be the menu item whose path matches the path
     * of the section ancestor route for the given `route`. For example, the
     * BLUETOOTH_DEVICES_SUBPAGE route's section ancestor is the BLUETOOTH route,
     * whose path matches the bluetooth menu item path.
     */
    setSelectedItemPathForRoute_(route) {
        const sectionAncestorRoute = route.getSectionAncestor();
        if (sectionAncestorRoute) {
            const menuItems = this.shadowRoot.querySelectorAll('os-settings-menu-item');
            for (const menuItem of menuItems) {
                if (sectionAncestorRoute.path === menuItem.path) {
                    this.setSelectedItemPath_(menuItem.path);
                    return;
                }
            }
        }
        // Nothing is selected.
        this.setSelectedItemPath_('');
    }
    computeMenuItems_() {
        const menuItems = [
            {
                section: Section.kNetwork,
                path: `/${routesMojom.NETWORK_SECTION_PATH}`,
                icon: 'os-settings:network-wifi',
                label: this.i18n('internetPageTitle'),
                sublabel: this.internetMenuItemDescription_,
            },
            {
                section: Section.kBluetooth,
                path: `/${routesMojom.BLUETOOTH_SECTION_PATH}`,
                icon: 'cr:bluetooth',
                label: this.i18n('bluetoothPageTitle'),
                sublabel: this.bluetoothMenuItemDescription_,
            },
            {
                section: Section.kMultiDevice,
                path: `/${routesMojom.MULTI_DEVICE_SECTION_PATH}`,
                icon: 'os-settings:connected-devices',
                label: this.i18n('multidevicePageTitle'),
                sublabel: this.multideviceMenuItemDescription_,
            },
            {
                section: Section.kPeople,
                path: `/${routesMojom.PEOPLE_SECTION_PATH}`,
                icon: 'os-settings:account',
                label: this.i18n('osPeoplePageTitle'),
                sublabel: this.accountsMenuItemDescription_,
            },
            {
                section: Section.kKerberos,
                path: `/${routesMojom.KERBEROS_SECTION_PATH}`,
                icon: 'os-settings:auth-key',
                label: this.i18n('kerberosPageTitle'),
                sublabel: null,
            },
            {
                section: Section.kDevice,
                path: `/${routesMojom.DEVICE_SECTION_PATH}`,
                icon: 'os-settings:laptop-chromebook',
                label: this.i18n('devicePageTitle'),
                sublabel: this.deviceMenuItemDescription_,
            },
            {
                section: Section.kPersonalization,
                path: `/${routesMojom.PERSONALIZATION_SECTION_PATH}`,
                icon: 'os-settings:personalization-menu',
                label: this.i18n('personalizationPageTitle'),
                sublabel: this.i18n('personalizationMenuItemDescription'),
            },
            {
                section: Section.kPrivacyAndSecurity,
                path: `/${routesMojom.PRIVACY_AND_SECURITY_SECTION_PATH}`,
                icon: 'cr:security',
                label: this.i18n('privacyPageTitle'),
                sublabel: this.i18n('privacyMenuItemDescription'),
            },
            {
                section: Section.kApps,
                path: `/${routesMojom.APPS_SECTION_PATH}`,
                icon: 'os-settings:apps',
                label: this.i18n('appsPageTitle'),
                sublabel: androidAppsVisible() ?
                    this.i18n('appsMenuItemDescription') :
                    this.i18n('appsmenuItemDescriptionArcUnavailable'),
            },
            {
                section: Section.kAccessibility,
                path: `/${routesMojom.ACCESSIBILITY_SECTION_PATH}`,
                icon: 'os-settings:accessibility',
                label: this.i18n('a11yPageTitle'),
                sublabel: this.i18n('a11yMenuItemDescription'),
            },
            {
                section: Section.kSystemPreferences,
                path: `/${routesMojom.SYSTEM_PREFERENCES_SECTION_PATH}`,
                icon: 'os-settings:system-preferences',
                label: this.i18n('systemPreferencesTitle'),
                sublabel: this.i18n('systemPreferencesMenuItemDescription'),
            },
            {
                section: Section.kAboutChromeOs,
                path: `/${routesMojom.ABOUT_CHROME_OS_SECTION_PATH}`,
                icon: 'os-settings:chrome',
                label: this.i18n('aboutOsPageTitle'),
                sublabel: this.i18n('aboutChromeOsMenuItemDescription'),
            },
        ];
        return menuItems.filter(({ section }) => !!this.pageAvailability[section]);
    }
    /**
     * @param path The path of the menu item to be selected. This path should be
     * the pathname portion of a URL, not the full URL. e.g. `/internet`, not
     * `chrome://os-settings/internet`.
     */
    setSelectedItemPath_(path) {
        this.selectedItemPath_ = path;
    }
    /**
     * Called when a selectable item from <iron-selector> is clicked. This is
     * fired before the selected item is changed.
     */
    onItemActivated_(event) {
        this.setSelectedItemPath_(event.detail.selected);
    }
    onItemSelected_(e) {
        e.detail.item.setAttribute('aria-current', 'true');
    }
    onItemDeselected_(e) {
        e.detail.item.removeAttribute('aria-current');
    }
    /**
     * @param opened Whether the menu is expanded.
     * @return Which icon to use.
     */
    arrowState_(opened) {
        return opened ? 'cr:arrow-drop-up' : 'cr:arrow-drop-down';
    }
    boolToString_(bool) {
        return bool.toString();
    }
    getMenuItemTooltipPosition_() {
        if (this.isDrawerMenu) {
            return 'bottom';
        }
        return this.isRtl_ ? 'left' : 'right';
    }
    /**
     * Updates the "Accounts" menu item description to one of the following:
     * - If there are multiple accounts (> 1), show "N accounts".
     * - If there is only one account, show the account email.
     */
    async updateAccountsMenuItemDescription_() {
        const accounts = await AccountManagerBrowserProxyImpl.getInstance().getAccounts();
        if (accounts.length > 1) {
            this.accountsMenuItemDescription_ =
                this.i18n('accountsMenuItemDescription', accounts.length);
            return;
        }
        const deviceAccount = accounts.find(account => account.isDeviceAccount);
        assertExists(deviceAccount, 'No device account found.');
        this.accountsMenuItemDescription_ = deviceAccount.email;
    }
    observeBluetoothProperties_() {
        this.bluetoothPropertiesObserverReceiver_ =
            new BluetoothPropertiesObserverReceiver(this);
        getBluetoothConfig().observeSystemProperties(this.bluetoothPropertiesObserverReceiver_.$.bindNewPipeAndPassRemote());
    }
    /** Implements SystemPropertiesObserverInterface */
    onPropertiesUpdated(properties) {
        const isBluetoothOn = properties.systemState === BluetoothSystemState.kEnabled ||
            properties.systemState === BluetoothSystemState.kEnabling;
        const connectedDevices = properties.pairedDevices.filter((device) => device.deviceProperties.connectionState ===
            DeviceConnectionState.kConnected);
        this.updateBluetoothMenuItemDescription_(isBluetoothOn, connectedDevices);
    }
    /**
     * Updates the "Bluetooth" menu item description to one of the following:
     * - If bluetooth is off, show "Off".
     * - If bluetooth is on but no bluetooth devices are connected, show "On".
     * - If one device is connected, show the name of the device.
     * - If there are multiple devices connected, show "N devices connected".
     */
    updateBluetoothMenuItemDescription_(isBluetoothOn, connectedDevices) {
        if (connectedDevices.length === 0) {
            this.bluetoothMenuItemDescription_ =
                isBluetoothOn ? this.i18n('deviceOn') : this.i18n('deviceOff');
            return;
        }
        if (connectedDevices.length === 1) {
            const device = castExists(connectedDevices[0]);
            this.bluetoothMenuItemDescription_ = getDeviceNameUnsafe(device);
            return;
        }
        this.bluetoothMenuItemDescription_ = this.i18n('bluetoothMenuItemDescriptionMultipleDevicesConnected', connectedDevices.length);
    }
    /** NetworkListenerBehavior override */
    onNetworkStateListChanged() {
        this.updateInternetMenuItemDescription_();
    }
    /** NetworkListenerBehavior override */
    onDeviceStateListChanged() {
        this.updateInternetMenuItemDescription_();
    }
    /** NetworkListenerBehavior override */
    onActiveNetworksChanged() {
        this.updateInternetMenuItemDescription_();
    }
    async computeIsDeviceCellularCapable_() {
        const { result: deviceStateList } = await this.networkConfig_.getDeviceStateList();
        const cellularDeviceState = deviceStateList.find(deviceState => deviceState.type === NetworkType.kCellular);
        this.isDeviceCellularCapable_ = !!cellularDeviceState;
    }
    async isInstantHotspotAvailable_() {
        const { result: deviceStateList } = await this.networkConfig_.getDeviceStateList();
        const tetherDeviceState = deviceStateList.find(deviceState => deviceState.type === NetworkType.kTether);
        return !!tetherDeviceState;
    }
    /**
     * Updates the "Internet" menu item description to one of the followings:
     * - If there are networks connected, show the name of one connected network
     *   with the priority: Ethernet, Wi-Fi, mobile(Cellular, Tether) and VPN.
     * - If there is no networks connected but instant hotspot is available, show
     * "Instant hotspot available".
     * - If there is no networks connected and mobile data is not supported, show
     * "Wi-Fi".
     * - If there is no networks connected but mobile data is supported, show
     * "Wi-Fi, mobile data".
     */
    async updateInternetMenuItemDescription_() {
        const { result: networkStateList } = await this.networkConfig_.getNetworkStateList({
            filter: FilterType.kVisible,
            limit: NO_LIMIT,
            networkType: NetworkType.kAll,
        });
        const prioritizedConnectedNetwork = getPrioritizedConnectedNetwork(networkStateList);
        if (prioritizedConnectedNetwork) {
            this.internetMenuItemDescription_ = prioritizedConnectedNetwork.name;
            return;
        }
        const tetherNetworkState = networkStateList.find(networkState => networkState.type === NetworkType.kTether);
        if (tetherNetworkState && await this.isInstantHotspotAvailable_()) {
            this.internetMenuItemDescription_ =
                this.i18n('internetMenuItemDescriptionInstantHotspotAvailable');
            return;
        }
        if (this.isDeviceCellularCapable_) {
            this.internetMenuItemDescription_ =
                this.i18n('internetMenuItemDescriptionWifiAndMobileData');
            return;
        }
        this.internetMenuItemDescription_ =
            this.i18n('internetMenuItemDescriptionWifi');
    }
    /**
     * Updates the "Multidevice" menu item description to one of the following:
     * - If there is a phone connected, show "Connected to <phone name>".
     * - If there is a phone connected but the device name is missing, show
     *   "Connected to Android phone".
     * - If there is no phone connected, show "Phone Hub, Nearby Share".
     */
    updateMultideviceMenuItemDescription_(pageContentData) {
        if (pageContentData.mode === MultiDeviceSettingsMode.HOST_SET_VERIFIED) {
            if (pageContentData.hostDeviceName) {
                this.multideviceMenuItemDescription_ = this.i18n('multideviceMenuItemDescriptionPhoneConnected', pageContentData.hostDeviceName);
            }
            else {
                this.multideviceMenuItemDescription_ =
                    this.i18n('multideviceMenuItemDescriptionDeviceNameMissing');
            }
            return;
        }
        this.multideviceMenuItemDescription_ =
            this.i18n('multideviceMenuItemDescription');
    }
    observeKeyboardSettings_() {
        if (this.inputDeviceSettingsProvider_ instanceof
            FakeInputDeviceSettingsProvider) {
            this.inputDeviceSettingsProvider_.observeKeyboardSettings(this);
            return;
        }
        this.keyboardSettingsObserverReceiver_ =
            new KeyboardSettingsObserverReceiver(this);
        this.inputDeviceSettingsProvider_.observeKeyboardSettings(this.keyboardSettingsObserverReceiver_.$.bindNewPipeAndPassRemote());
    }
    /** Implements KeyboardSettingsObserverInterface */
    onKeyboardListUpdated(keyboards) {
        this.hasKeyboard_ = keyboards.length > 0;
    }
    /** Implements KeyboardSettingsObserverInterface */
    onKeyboardPoliciesUpdated() {
        // Not handled.
    }
    observeMouseSettings_() {
        if (this.inputDeviceSettingsProvider_ instanceof
            FakeInputDeviceSettingsProvider) {
            this.inputDeviceSettingsProvider_.observeMouseSettings(this);
            return;
        }
        this.mouseSettingsObserverReceiver_ =
            new MouseSettingsObserverReceiver(this);
        this.inputDeviceSettingsProvider_.observeMouseSettings(this.mouseSettingsObserverReceiver_.$.bindNewPipeAndPassRemote());
    }
    /** Implements MouseSettingsObserverInterface */
    onMouseListUpdated(mice) {
        this.hasMouse_ = mice.length > 0;
    }
    /** Implements MouseSettingsObserverInterface */
    onMousePoliciesUpdated() {
        // Not handled.
    }
    observePointingStickSettings_() {
        if (this.inputDeviceSettingsProvider_ instanceof
            FakeInputDeviceSettingsProvider) {
            this.inputDeviceSettingsProvider_.observePointingStickSettings(this);
            return;
        }
        this.pointingStickSettingsObserverReceiver_ =
            new PointingStickSettingsObserverReceiver(this);
        this.inputDeviceSettingsProvider_.observePointingStickSettings(this.pointingStickSettingsObserverReceiver_.$
            .bindNewPipeAndPassRemote());
    }
    /** Implements PointingStickSettingsObserverInterface */
    onPointingStickListUpdated(pointingSticks) {
        this.hasPointingStick_ = pointingSticks.length > 0;
    }
    observeTouchpadSettings_() {
        if (this.inputDeviceSettingsProvider_ instanceof
            FakeInputDeviceSettingsProvider) {
            this.inputDeviceSettingsProvider_.observeTouchpadSettings(this);
            return;
        }
        this.touchpadSettingsObserverReceiver_ =
            new TouchpadSettingsObserverReceiver(this);
        this.inputDeviceSettingsProvider_.observeTouchpadSettings(this.touchpadSettingsObserverReceiver_.$.bindNewPipeAndPassRemote());
    }
    /** Implements TouchpadSettingsObserverInterface */
    onTouchpadListUpdated(touchpads) {
        this.hasTouchpad_ = touchpads.length > 0;
    }
    /**
     * Only show at most 3 strings in order of priority:
     * - "keyboard" (if available)
     * - "mouse" OR "touchpad" (if available, prioritize mouse)
     * - "print"
     * - "display"
     */
    computeDeviceMenuItemDescription_() {
        const wordOptions = [];
        if (this.hasKeyboard_) {
            wordOptions.push(this.i18n('deviceMenuItemDescriptionKeyboard'));
        }
        if (this.hasMouse_ || this.hasPointingStick_) {
            wordOptions.push(this.i18n('deviceMenuItemDescriptionMouse'));
        }
        else if (this.hasTouchpad_) {
            wordOptions.push(this.i18n('deviceMenuItemDescriptionTouchpad'));
        }
        wordOptions.push(this.i18n('deviceMenuItemDescriptionPrint'), this.i18n('deviceMenuItemDescriptionDisplay'));
        const words = wordOptions.slice(0, 3);
        return capitalize(words.join(this.i18n('listSeparator')));
    }
}
customElements.define(OsSettingsMenuElement.is, OsSettingsMenuElement);
