import { TestImportManager } from '../../../../../../../../../../../../../../../../../../common/testing/test_import_manager.js';
import { Flags, FlagName } from '../../../../../../../../../../../../../../../../../../common/flags.js';
import { InstanceChecker } from '../../../../../../../../../../../../../../../../../../common/mv2/instance_checker.js';
import '../../../../../../../../../../../../../../../../../../common/array_util.js';
import { AutomationPredicate } from '../../../../../../../../../../../../../../../../../../common/automation_predicate.js';
import '../../../../../../../../../../../../../../../../../../common/tree_walker.js';
import '../../../../../../../../../../../../../../../../../../common/automation_util.js';
import { RectUtil } from '../../../../../../../../../../../../../../../../../../common/rect_util.js';
import '../../../../../../../../../../../../../../../../../../common/cursors/cursor.js';
import '../../../../../../../../../../../../../../../../../../common/cursors/range.js';
import { EventGenerator } from '../../../../../../../../../../../../../../../../../../common/event_generator.js';
import '../../../../../../../../../../../../../../../../../../common/keep_alive.js';
import '../../../../../../../../../../../../../../../../../../common/local_storage.js';
import { EventHandler } from '../../../../../../../../../../../../../../../../../../common/event_handler.js';
import { Context } from '../../../../../../../../../../../../../../../../../../common/action_fulfillment/context_checker.js';
import { MacroName } from '../../../../../../../../../../../../../../../../../../common/action_fulfillment/macros/macro_names.js';
import { AsyncUtil } from '../../../../../../../../../../../../../../../../../../common/async_util.js';
import { InputController } from '../../../../../../../../../../../../../../../../../../common/action_fulfillment/input_controller.js';
import { InputTextViewMacro, NewLineMacro } from '../../../../../../../../../../../../../../../../../../common/action_fulfillment/macros/input_text_view_macro.js';
import { DeletePrevSentMacro } from '../../../../../../../../../../../../../../../../../../common/action_fulfillment/macros/delete_prev_sent_macro.js';
import { NavPrevSentMacro, NavNextSentMacro } from '../../../../../../../../../../../../../../../../../../common/action_fulfillment/macros/nav_sent_macro.js';
import { RepeatMacro } from '../../../../../../../../../../../../../../../../../../common/action_fulfillment/macros/repeat_macro.js';
import * as RepeatableKeyPressMacro from '../../../../../../../../../../../../../../../../../../common/action_fulfillment/macros/repeatable_key_press_macro.js';
import { SmartDeletePhraseMacro } from '../../../../../../../../../../../../../../../../../../common/action_fulfillment/macros/smart_delete_phrase_macro.js';
import { SmartInsertBeforeMacro } from '../../../../../../../../../../../../../../../../../../common/action_fulfillment/macros/smart_insert_before_macro.js';
import { SmartReplacePhraseMacro } from '../../../../../../../../../../../../../../../../../../common/action_fulfillment/macros/smart_replace_phrase_macro.js';
import { SmartSelectBetweenMacro } from '../../../../../../../../../../../../../../../../../../common/action_fulfillment/macros/smart_select_between_macro.js';
import { ToggleDictationMacro } from '../../../../../../../../../../../../../../../../../../common/action_fulfillment/macros/toggle_dictation_macro.js';
import { Macro, ToggleDirection, MacroError } from '../../../../../../../../../../../../../../../../../../common/action_fulfillment/macros/macro.js';
import { CustomCallbackMacro } from '../../../../../../../../../../../../../../../../../../common/action_fulfillment/macros/custom_callback_macro.js';
import { KeyPressMacro } from '../../../../../../../../../../../../../../../../../../common/action_fulfillment/macros/key_press_macro.js';
import { MouseClickLeftTripleMacro, MouseClickLeftDoubleMacro, MouseClickMacro } from '../../../../../../../../../../../../../../../../../../common/action_fulfillment/macros/mouse_click_macro.js';
import { KeyCode } from '../../../../../../../../../../../../../../../../../../common/key_code.js';
import { FaceLandmarker } from '../../../../../../../../../../../../../../../../../../accessibility_common/mv2/third_party/mediapipe_task_vision/vision_bundle.mjs';
import { ChromeEventHandler } from '../../../../../../../../../../../../../../../../../../common/chrome_event_handler.js';

// Copyright 2020 The Chromium Authors
// Use of this source code is governed by a BSD-style license that can be
// found in the LICENSE file.
// Required for running accessibility/common tests. Without this,
// necessary exports are not available in the test enviropment.
// Required for AccessibilityExtensionArrayUtilTest.
/**
 * The hex color for the focus rings.
 */
const AUTOCLICK_FOCUS_RING_COLOR = '#aac9fa';
/**
 * The amount of time to wait before hiding the focus rings from the display.
 */
const AUTOCLICK_FOCUS_RING_DISPLAY_TIME_MS = 250;
/**
 * Class to manage Automatic Clicks' interaction with the accessibility tree.
 */
class Autoclick {
    /**
     * Whether to blink the focus rings. Disabled during tests due to
     * complications with callbacks.
     */
    blinkFocusRings_ = true;
    desktop_ = null;
    scrollableBoundsListener_ = null;
    hitTestHandler_;
    onLoadDesktopCallbackForTest_ = null;
    constructor() {
        this.hitTestHandler_ = new EventHandler([], chrome.automation.EventType.MOUSE_PRESSED, event => this.onAutomationHitTestResult_(event), {
            capture: true,
            exactMatch: false,
            listenOnce: false,
            predicate: undefined,
        });
        this.init_();
    }
    setNoBlinkFocusRingsForTest() {
        this.blinkFocusRings_ = false;
    }
    /**
     * Destructor to remove any listeners.
     */
    onAutoclickDisabled() {
        if (this.scrollableBoundsListener_) {
            chrome.accessibilityPrivate.onScrollableBoundsForPointRequested
                .removeListener(this.scrollableBoundsListener_);
            this.scrollableBoundsListener_ = null;
        }
        this.hitTestHandler_.stop();
    }
    /**
     * Initializes Autoclick.
     */
    init_() {
        this.scrollableBoundsListener_ = (x, y) => this.findScrollingContainerForPoint_(x, y);
        chrome.automation.getDesktop(desktop => {
            this.desktop_ = desktop;
            // We use a hit test at a point to determine what automation node is
            // at that point, in order to find the scrollable area.
            this.hitTestHandler_.setNodes(this.desktop_);
            this.hitTestHandler_.start();
            if (this.onLoadDesktopCallbackForTest_) {
                this.onLoadDesktopCallbackForTest_();
                this.onLoadDesktopCallbackForTest_ = null;
            }
        });
        chrome.accessibilityPrivate.onScrollableBoundsForPointRequested.addListener(this.scrollableBoundsListener_);
    }
    /**
     * Sets the focus ring to |rects|.
     */
    setFocusRings_(rects) {
        // TODO(katie): Add a property to FocusRingInfo to set FocusRingBehavior
        // to fade out.
        chrome.accessibilityPrivate.setFocusRings([{
                rects,
                type: chrome.accessibilityPrivate.FocusType.SOLID,
                color: AUTOCLICK_FOCUS_RING_COLOR,
                secondaryColor: AUTOCLICK_FOCUS_RING_COLOR,
            }], chrome.accessibilityPrivate.AssistiveTechnologyType.AUTO_CLICK);
    }
    /**
     * Calculates whether a node should be highlighted as scrollable.
     * Returns True  if the node should be highlighted as scrollable.
     */
    shouldHighlightAsScrollable_(node) {
        if (node.scrollable === undefined || !node.scrollable) {
            return false;
        }
        // Check that the size of the scrollable area is larger than the node's
        // location. If it is not larger, then scrollbars are not shown.
        return node.scrollXMax - node.scrollXMin > node.location.width ||
            node.scrollYMax - node?.scrollYMin > node.location.height;
    }
    /**
     * Processes an automation hit test result.
     */
    onAutomationHitTestResult_(event) {
        // Walk up to the nearest scrollale area containing the point.
        let node = event.target;
        while (node.parent && node.role !== chrome.automation.RoleType.WINDOW &&
            node.role !== chrome.automation.RoleType.ROOT_WEB_AREA &&
            node.role !== chrome.automation.RoleType.DESKTOP &&
            node.role !== chrome.automation.RoleType.DIALOG &&
            node.role !== chrome.automation.RoleType.ALERT_DIALOG &&
            node.role !== chrome.automation.RoleType.TOOLBAR) {
            if (this.shouldHighlightAsScrollable_(node)) {
                break;
            }
            node = node.parent;
        }
        if (!node.location) {
            return;
        }
        const bounds = node.location;
        this.setFocusRings_([bounds]);
        if (this.blinkFocusRings_) {
            // Blink the focus ring briefly per UX spec, using timeouts to turn it
            // off, on, and off again. The focus ring is only used to show the user
            // where the scroll might occur, but is not persisted after the blink.
            // Turn off after 500 ms.
            setTimeout(() => {
                this.setFocusRings_([]);
            }, AUTOCLICK_FOCUS_RING_DISPLAY_TIME_MS * 2);
            // Back on after an additional 250 ms.
            setTimeout(() => {
                this.setFocusRings_([bounds]);
            }, AUTOCLICK_FOCUS_RING_DISPLAY_TIME_MS * 3);
            // And off after another 500 ms.
            setTimeout(() => {
                this.setFocusRings_([]);
            }, AUTOCLICK_FOCUS_RING_DISPLAY_TIME_MS * 5);
        }
        chrome.accessibilityPrivate.handleScrollableBoundsForPointFound(bounds);
    }
    /**
     * Initiates finding the nearest scrolling container for the given point.
     */
    findScrollingContainerForPoint_(x, y) {
        // The hit test will come back through onAutomationHitTestResult_,
        // which will do the logic for finding the scrolling container.
        this.desktop_.hitTest(x, y, chrome.automation.EventType.MOUSE_PRESSED);
    }
    /**
     * Used by C++ tests to ensure Autoclick JS load is completed.
     * `callback` Callback for when desktop is loaded from
     * automation.
     */
    setOnLoadDesktopCallbackForTest(callback) {
        if (!this.desktop_) {
            this.onLoadDesktopCallbackForTest_ = callback;
            return;
        }
        // Desktop already loaded.
        callback();
    }
}

// Copyright 2022 The Chromium Authors
// Use of this source code is governed by a BSD-style license that can be
// found in the LICENSE file.
var EventType$2 = chrome.automation.EventType;
class FocusHandler {
    active_ = false;
    /** The currently focused editable node. */
    editableNode_ = null;
    deactivateTimeoutId_ = null;
    eventHandler_ = null;
    onActiveChangedForTesting_ = null;
    onEditableNodeChangedForTesting_ = null;
    onFocusChangedForTesting_ = null;
    /**
     * Starts listening to focus events and sets a timeout to deactivate after
     * a certain amount of inactivity.
     */
    async refresh() {
        if (this.deactivateTimeoutId_ !== null) {
            clearTimeout(this.deactivateTimeoutId_);
        }
        this.deactivateTimeoutId_ = setTimeout(() => this.deactivate_(), FocusHandler.DEACTIVATE_TIMEOUT_MS_);
        await this.activate_();
    }
    /** Gets the current focus and starts listening for focus events. */
    async activate_() {
        if (this.active_) {
            return;
        }
        const desktop = await AsyncUtil.getDesktop();
        const focus = await AsyncUtil.getFocus();
        if (focus && AutomationPredicate.editText(focus)) {
            this.setEditableNode_(focus);
        }
        if (!this.eventHandler_) {
            this.eventHandler_ = new EventHandler([], EventType$2.FOCUS, event => this.onFocusChanged_(event));
        }
        this.eventHandler_.setNodes(desktop);
        this.eventHandler_.start();
        this.setActive_(true);
    }
    deactivate_() {
        // TODO(b/314203187): Determine if not null assertion is acceptable.
        this.eventHandler_.stop();
        this.eventHandler_ = null;
        this.setActive_(false);
        this.setEditableNode_(null);
    }
    /** Saves the focused node if it's an editable. */
    onFocusChanged_(event) {
        const node = event.target;
        if (!node || !AutomationPredicate.editText(node)) {
            this.setEditableNode_(null);
            return;
        }
        this.setEditableNode_(node);
        if (this.onFocusChangedForTesting_) {
            this.onFocusChangedForTesting_();
        }
    }
    getEditableNode() {
        return this.editableNode_;
    }
    setActive_(value) {
        this.active_ = value;
        if (this.onActiveChangedForTesting_) {
            this.onActiveChangedForTesting_();
        }
    }
    setEditableNode_(node) {
        this.editableNode_ = node;
        if (this.onEditableNodeChangedForTesting_) {
            this.onEditableNodeChangedForTesting_();
        }
    }
    isReadyForTesting(expectedClassName) {
        return this.active_ && this.editableNode_ !== null &&
            this.editableNode_.className === expectedClassName;
    }
}
(function (FocusHandler) {
    FocusHandler.DEACTIVATE_TIMEOUT_MS_ = 45 * 1000;
})(FocusHandler || (FocusHandler = {}));
TestImportManager.exportForTesting(FocusHandler);

// Copyright 2022 The Chromium Authors
// Use of this source code is governed by a BSD-style license that can be
// found in the LICENSE file.
/** Contains all locale-related information for Dictation. */
class LocaleInfo {
    static allowSmartCapAndSpacing() {
        const language = LocaleInfo.locale.split('-')[0];
        return LocaleInfo.SMART_CAP_AND_SPACING_LANGUAGES_.has(language);
    }
    static allowSmartEditing() {
        // Restrict smart editing commands to left-to-right locales.
        // TODO(crbug.com/1331351): Add support for RTL locales.
        return !LocaleInfo.isRTLLocale();
    }
    /* eslint-disable-next-line @typescript-eslint/naming-convention */
    static isRTLLocale() {
        const locale = LocaleInfo.locale;
        return LocaleInfo.RTL_LOCALES_.has(locale);
    }
    /* eslint-disable-next-line @typescript-eslint/naming-convention */
    static getUILanguage() {
        const locale = LocaleInfo.locale.toLowerCase();
        return LocaleInfo.LOCALE_TO_UI_LANGUAGE_MAP_[locale];
    }
    /**
     * Determines whether commands are supported for this Dictation language
     * and UI system language.
     * @return Whether or not commands are supported.
     */
    static areCommandsSupported() {
        // Currently Dictation cannot support commands when the UI language
        // doesn't match the Dictation language. See crbug.com/1340590.
        const systemLocale = chrome.i18n.getUILanguage().toLowerCase();
        const systemLanguage = systemLocale.split('-')[0];
        const dictationLanguage = LocaleInfo.locale.toLowerCase().split('-')[0];
        if (systemLanguage === dictationLanguage) {
            return true;
        }
        return LocaleInfo.alwaysEnableCommandsForTesting ||
            Boolean(LocaleInfo.getUILanguage()) &&
                (LocaleInfo.getUILanguage() === systemLanguage ||
                    LocaleInfo.getUILanguage() === systemLocale);
    }
    /** Returns true if we should consider spaces, false otherwise. */
    static considerSpaces() {
        const language = LocaleInfo.locale.toLowerCase().split('-')[0];
        return !LocaleInfo.NO_SPACE_LANGUAGES_.has(language);
    }
}
(function (LocaleInfo) {
    /** The current Dictation locale. */
    /* eslint-disable-next-line prefer-const */
    LocaleInfo.locale = '';
    LocaleInfo.alwaysEnableCommandsForTesting = false;
    LocaleInfo.SMART_CAP_AND_SPACING_LANGUAGES_ = new Set(['en', 'fr', 'it', 'de', 'es']);
    /** All RTL locales from Dictation::GetAllSupportedLocales. */
    LocaleInfo.RTL_LOCALES_ = new Set([
        'ar-AE', 'ar-BH', 'ar-DZ', 'ar-EG', 'ar-IL', 'ar-IQ', 'ar-JO',
        'ar-KW', 'ar-LB', 'ar-MA', 'ar-OM', 'ar-PS', 'ar-QA', 'ar-SA',
        'ar-TN', 'ar-YE', 'fa-IR', 'iw-IL', 'ur-IN', 'ur-PK',
    ]);
    /**
     * TODO: get this data from l10n or i18n.
     * Hebrew in Dictation is 'iw-IL' but 'he' in UI languages.
     * yue-Hant-HK can map to 'zh-TW' because both are written as traditional
     * Chinese. Norwegian in Dictation is 'no-NO' but 'nb' in UI languages.
     */
    LocaleInfo.LOCALE_TO_UI_LANGUAGE_MAP_ = {
        'iw-il': 'he',
        'yue-hant-hk': 'zh-tw',
        'no-no': 'nb',
    };
    /**
     * TODO(akihiroota): Follow-up with an i18n expert to get a full list of
     * languages.
     * All languages that don't use spaces.
     */
    LocaleInfo.NO_SPACE_LANGUAGES_ = new Set(['ja', 'zh']);
})(LocaleInfo || (LocaleInfo = {}));
TestImportManager.exportForTesting(LocaleInfo);

// Copyright 2022 The Chromium Authors
// Use of this source code is governed by a BSD-style license that can be
// found in the LICENSE file.
/**
 * EditingUtil provides utility and helper methods for editing-related
 * operations.
 */
class EditingUtil {
    /**
     * TODO(https://crbug.com/1331351): Add RTL support.
     * Returns data needed by inputController.replacePhrase(). Calculates the new
     * caret index and number of characters to be deleted. Only operates on the
     * text to the left of the text caret. If multiple instances of `deletePhrase`
     * are present, this function will operate on the one closest one to the text
     * caret.
     * @param value The current value of the text field.
     * @param deletePhrase The phrase to be deleted.
     */
    static getReplacePhraseData(value, caretIndex, deletePhrase) {
        const leftOfCaret = value.substring(0, caretIndex);
        deletePhrase = deletePhrase.trim();
        // Find the right-most occurrence of `deletePhrase`. If we're deleting text,
        // prefer the RegExps that include a leading/trailing space to preserve
        // spacing.
        let re;
        if (LocaleInfo.considerSpaces()) {
            re = EditingUtil.getPhraseRegex_(deletePhrase);
        }
        else {
            re = EditingUtil.getPhraseRegexNoWordBoundaries_(deletePhrase);
        }
        const reWithLeadingSpace = EditingUtil.getPhraseRegexLeadingSpace_(deletePhrase);
        const reWithTrailingSpace = EditingUtil.getPhraseRegexTrailingSpace_(deletePhrase);
        const leadingSpaceResult = EditingUtil.getIndexFromRegex_(reWithLeadingSpace, leftOfCaret);
        const trailingSpaceResult = EditingUtil.getIndexFromRegex_(reWithTrailingSpace, leftOfCaret);
        const noSpacesResult = EditingUtil.getIndexFromRegex_(re, leftOfCaret);
        let newIndex = caretIndex;
        let deleteLength = 0;
        if (leadingSpaceResult !== -1) {
            // Delete one extra character to preserve spacing.
            newIndex = leadingSpaceResult;
            deleteLength = deletePhrase.length + 1;
        }
        else if (trailingSpaceResult !== -1) {
            // Delete one extra character to preserve spacing.
            newIndex = trailingSpaceResult;
            deleteLength = deletePhrase.length + 1;
        }
        else if (noSpacesResult !== -1) {
            // Matched with no spacing.
            newIndex = noSpacesResult;
            deleteLength = deletePhrase.length;
        }
        else {
            // No match.
            return null;
        }
        return { newIndex, deleteLength };
    }
    /**
     * TODO(https://crbug.com/1331351): Add RTL support.
     * Calculates the new caret index for `inputController.insertBefore`. Only
     * operates on the text to the left of the text caret. If multiple instances
     * of `beforePhrase` are present, this function will operate on the one
     * closest one to the text caret.
     * @param value The current value of the text field.
     */
    static getInsertBeforeIndex(value, caretIndex, beforePhrase) {
        const result = EditingUtil.getReplacePhraseData(value, caretIndex, beforePhrase);
        return result ? result.newIndex : -1;
    }
    /**
     * Returns a selection starting at `startPhrase` and ending at `endPhrase`
     * (inclusive). The function operates on the text to the left of the text
     * caret. If multiple instances of `startPhrase` or `endPhrase` are present,
     * the function will use the ones closest to the text caret.
     * @param value The current value of the text field.
     */
    static selectBetween(value, caretIndex, startPhrase, endPhrase) {
        const leftOfCaret = value.substring(0, caretIndex);
        startPhrase = startPhrase.trim();
        endPhrase = endPhrase.trim();
        let startRe;
        let endRe;
        if (LocaleInfo.considerSpaces()) {
            startRe = EditingUtil.getPhraseRegex_(startPhrase);
            endRe = EditingUtil.getPhraseRegex_(endPhrase);
        }
        else {
            startRe = EditingUtil.getPhraseRegexNoWordBoundaries_(startPhrase);
            endRe = EditingUtil.getPhraseRegexNoWordBoundaries_(endPhrase);
        }
        const start = leftOfCaret.search(startRe);
        let end = leftOfCaret.search(endRe);
        if (start === -1 || end === -1) {
            return null;
        }
        // Adjust `end` so that we return an inclusive selection.
        end += endPhrase.length;
        if (start > end) {
            return null;
        }
        return { start, end };
    }
    /**
     * TODO(https://crbug.com/1331351): Add RTL support.
     * Returns the start index of the sentence to the right of the caret.
     * Indices are relative to `value`. Assumes that sentences are separated by
     * punctuation specified in `EditingUtil.END_OF_SENTENCE_REGEX_`. If no next
     * sentence can be found, returns `value.length`.
     * @param value The current value of the text field.
     */
    static navNextSent(value, caretIndex) {
        const rightOfCaret = value.substring(caretIndex);
        const index = rightOfCaret.search(EditingUtil.END_OF_SENTENCE_REGEX_);
        if (index === -1) {
            return value.length;
        }
        // `index` should be relative to `value`;
        return index + caretIndex + 1;
    }
    /**
     * TODO(https://crbug.com/1331351): Add RTL support.
     * Returns the start index of the sentence to the left of the caret. Indices
     * are relative to `value`. Assumes that sentences are separated by
     * punctuation specified in `EditingUtil.END_OF_SENTENCE_REGEX_`. If no
     * previous sentence can be found, returns 0.
     * @param value The current value of the text field.
     */
    static navPrevSent(value, caretIndex) {
        let encounteredText = false;
        if (caretIndex === value.length) {
            --caretIndex;
        }
        while (caretIndex >= 0) {
            const valueAtCaret = value[caretIndex];
            if (encounteredText &&
                EditingUtil.END_OF_SENTENCE_REGEX_.test(valueAtCaret)) {
                // Adjust if there is whitespace immediately to the right of the caret.
                return EditingUtil.BEGINS_WITH_WHITESPACE_REGEX_.test(value[caretIndex + 1]) ?
                    caretIndex + 1 :
                    caretIndex;
            }
            if (!EditingUtil.BEGINS_WITH_WHITESPACE_REGEX_.test(valueAtCaret) &&
                !EditingUtil.PUNCTUATION_REGEX_.test(valueAtCaret)) {
                encounteredText = true;
            }
            --caretIndex;
        }
        return 0;
    }
    /**
     * TODO(https://crbug.com/1331351): Add RTL support.
     * This function analyzes the context and adjusts the spacing of `commitText`
     * to maintain proper spacing between text.
     * @param value The current value of the text field.
     */
    static smartSpacing(value, caretIndex, commitText) {
        // There is currently a bug in SODA (b/213934503) where final speech results
        // do not start with a space. This results in a Dictation bug
        // (crbug.com/1294050), where final speech results are not separated by a
        // space when committed to a text field. This is a temporary workaround
        // until the blocking SODA bug can be fixed. Note, a similar strategy
        // already exists in Dictation::OnSpeechResult().
        if (!value || EditingUtil.BEGINS_WITH_WHITESPACE_REGEX_.test(commitText) ||
            EditingUtil.BEGINS_WITH_PUNCTUATION_REGEX_.test(commitText)) {
            return commitText;
        }
        commitText = commitText.trim();
        const leftOfCaret = value.substring(0, caretIndex);
        const rightOfCaret = value.substring(caretIndex);
        if (leftOfCaret &&
            !EditingUtil.ENDS_WITH_WHITESPACE_REGEX_.test(leftOfCaret)) {
            // If there is no whitespace before the caret index, prepend a space.
            commitText = ' ' + commitText;
        }
        if (rightOfCaret &&
            (!EditingUtil.BEGINS_WITH_WHITESPACE_REGEX_.test(rightOfCaret) ||
                EditingUtil.BEGINS_WITH_PUNCTUATION_REGEX_.test(rightOfCaret))) {
            // If there are no spaces or there is punctuation after the caret index,
            // append a space.
            commitText = commitText + ' ';
        }
        return commitText;
    }
    /**
     * TODO(https://crbug.com/1331351): Add RTL support.
     * This function analyzes the context and adjusts the capitalization of
     * `commitText` as needed. See below for sample input and output: value:
     * 'Hello world.' caretIndex: value.length commitText: 'goodnight world'
     * return value: 'Goodnight world'
     * @param value The current value of the text field.
     */
    static smartCapitalization(value, caretIndex, commitText) {
        if (EditingUtil.BEGINS_WITH_PUNCTUATION_REGEX_.test(commitText)) {
            // If `commitText` begins with punctuation, then it's assumed that it's
            // already correctly capitalized.
            return commitText;
        }
        if (!value) {
            return EditingUtil.capitalize_(commitText);
        }
        const leftOfCaret = value.substring(0, caretIndex).trim();
        if (!leftOfCaret) {
            return EditingUtil.capitalize_(commitText);
        }
        return EditingUtil.ENDS_WITH_END_OF_SENTENCE_REGEX_.test(leftOfCaret) ?
            EditingUtil.capitalize_(commitText) :
            EditingUtil.lowercase_(commitText);
    }
    /** Returns a string where the first character is capitalized. */
    static capitalize_(text) {
        return text.charAt(0).toUpperCase() + text.substring(1);
    }
    /** Returns a string where the first character is lowercase. */
    static lowercase_(text) {
        return text.charAt(0).toLowerCase() + text.substring(1);
    }
    /**
     * TODO(akihiroota): Break regex construction into smaller chunks.
     * Returns a RegExp that matches on the right-most occurrence of a phrase.
     * The returned RegExp is case insensitive and requires that `phrase` is
     * separated by word boundaries.
     */
    static getPhraseRegex_(phrase) {
        return new RegExp(`(\\b${phrase}\\b)(?!.*\\b\\1\\b)`, 'i');
    }
    /**
     * Similar to above, but doesn't include word boundaries. This is useful for
     * languages that don't use spaces e.g. Japanese.
     */
    static getPhraseRegexNoWordBoundaries_(phrase) {
        return new RegExp(`(${phrase})(?!.*\\1)`, 'i');
    }
    /** Similar to above, but accounts for a leading space. */
    static getPhraseRegexLeadingSpace_(phrase) {
        return new RegExp(`( \\b${phrase}\\b)(?!.*\\b\\1\\b)`, 'i');
    }
    /** Similar to above, but accounts for a trailing space. */
    static getPhraseRegexTrailingSpace_(phrase) {
        return new RegExp(`(\\b${phrase}\\b )(?!.*\\b\\1\\b)`, 'i');
    }
    static getIndexFromRegex_(re, str) {
        const result = re.exec(str);
        return result ? result.index : -1;
    }
}
(function (EditingUtil) {
    /** Includes full-width symbols that are commonly used in Japanese. */
    EditingUtil.END_OF_SENTENCE_REGEX_ = /[;!.?。．？！]/;
    /** Similar to above, but looks for a match at the end of a string. */
    EditingUtil.ENDS_WITH_END_OF_SENTENCE_REGEX_ = /[;!.?。．？！]$/;
    EditingUtil.BEGINS_WITH_WHITESPACE_REGEX_ = /^\s/;
    EditingUtil.ENDS_WITH_WHITESPACE_REGEX_ = /\s$/;
    EditingUtil.PUNCTUATION_REGEX_ = /[-$#"()*;:<>\\\/\{\}\[\]+='~`!@_.,?%。．？！\u2022\u25e6\u25a0]/g;
    EditingUtil.BEGINS_WITH_PUNCTUATION_REGEX_ = /^[-$#"()*;:<>\\\/\{\}\[\]+='~`!@_.,?%。．？！\u2022\u25e6\u25a0]/;
})(EditingUtil || (EditingUtil = {}));
TestImportManager.exportForTesting(EditingUtil);

// Copyright 2021 The Chromium Authors
// Use of this source code is governed by a BSD-style license that can be
// found in the LICENSE file.
var EventType$1 = chrome.automation.EventType;
var PositionType = chrome.automation.PositionType;
var StateType$1 = chrome.automation.StateType;
/** A helper class that waits for automation and IME events. */
class AutomationImeEventWaiter {
    node_;
    event_;
    constructor(node, event) {
        this.node_ = node;
        this.event_ = event;
    }
    /**
     * Calls |doAction|, then waits for |this.event_| and a
     * chrome.input.ime.onSurroundingTextChanged event. We need to wait for both
     * since we use the automation and IME APIs to retrieve the editable node
     * data.
     */
    async doActionAndWait(doAction) {
        let surroundingTextChanged = false;
        let eventSeen = false;
        return new Promise(resolve => {
            const onSurroundingTextChanged = () => {
                surroundingTextChanged = true;
                chrome.input.ime.onSurroundingTextChanged.removeListener(onSurroundingTextChanged);
                if (eventSeen) {
                    resolve();
                }
            };
            let handler = new EventHandler([this.node_], this.event_, () => {
                eventSeen = true;
                // TODO(b/314203187): Determine if not null assertion is acceptable.
                handler.stop();
                handler = null;
                if (surroundingTextChanged) {
                    resolve();
                }
            });
            handler.start();
            chrome.input.ime.onSurroundingTextChanged.addListener(onSurroundingTextChanged);
            doAction();
        });
    }
}
/** InputController handles interaction with input fields for Dictation. */
// TODO(b/307904022): Remove dependency on chrome.input.ime and depend fully on
// Automation instead, then
// TODO(b/324316493): Refactor logic that isn't specific to Dictation into
// common InputController.
class InputControllerImpl extends InputController {
    stopDictationCallback_;
    focusHandler_;
    activeImeContextId_ = InputControllerImpl.NO_ACTIVE_IME_CONTEXT_ID_;
    onConnectCallback_ = null;
    onFocusListener_ = null;
    onBlurListener_ = null;
    /** A listener for chrome.input.ime.onSurroundingTextChanged events. */
    onSurroundingTextChangedListener_ = null;
    surroundingInfo_ = null;
    onSurroundingTextChangedForTesting_ = null;
    onSelectionChangedForTesting_ = null;
    /**
     * The engine ID of the previously active IME input method. Used to
     * restore the previous IME after Dictation is deactivated.
     */
    previousImeEngineId_ = '';
    constructor(stopDictationCallback, focusHandler) {
        super();
        this.stopDictationCallback_ = stopDictationCallback;
        this.focusHandler_ = focusHandler;
        this.initialize_();
    }
    /** Sets up Dictation's various IME listeners. */
    initialize_() {
        this.onFocusListener_ = context => this.onImeFocus_(context);
        this.onBlurListener_ = contextId => this.onImeBlur_(contextId);
        this.onSurroundingTextChangedListener_ = (engineID, surroundingInfo) => this.onSurroundingTextChanged_(engineID, surroundingInfo);
        chrome.input.ime.onFocus.addListener(this.onFocusListener_);
        chrome.input.ime.onBlur.addListener(this.onBlurListener_);
        chrome.input.ime.onSurroundingTextChanged.addListener(this.onSurroundingTextChangedListener_);
    }
    /** Removes IME listeners. */
    removeListeners() {
        if (this.onFocusListener_) {
            chrome.input.ime.onFocus.removeListener(this.onFocusListener_);
        }
        if (this.onBlurListener_) {
            chrome.input.ime.onBlur.removeListener(this.onBlurListener_);
        }
        if (this.onSurroundingTextChangedListener_) {
            chrome.input.ime.onSurroundingTextChanged.removeListener(this.onSurroundingTextChangedListener_);
        }
    }
    /** Whether this is the active IME and has a focused input. */
    isActive() {
        return this.activeImeContextId_ !==
            InputControllerImpl.NO_ACTIVE_IME_CONTEXT_ID_;
    }
    /**
     * Connect as the active Input Method Manager.
     * @param callback The callback to run after IME is connected.
     */
    connect(callback) {
        this.onConnectCallback_ = callback;
        chrome.inputMethodPrivate.getCurrentInputMethod((method) => this.saveCurrentInputMethodAndStart_(method));
    }
    /**
     * Called when InputController has received the current input method. We save
     * the current method to reset when InputController toggles off, then continue
     * with starting up Dictation after the input gets focus (onImeFocus_()).
     * @param method The currently active IME ID.
     */
    saveCurrentInputMethodAndStart_(method) {
        this.previousImeEngineId_ = method;
        // Add AccessibilityCommon as an input method and activate it.
        chrome.languageSettingsPrivate.addInputMethod(InputControllerImpl.IME_ENGINE_ID);
        chrome.inputMethodPrivate.setCurrentInputMethod(InputControllerImpl.IME_ENGINE_ID);
    }
    /**
     * Disconnects as the active Input Method Manager. If any text was being
     * composed, commits it.
     */
    disconnect() {
        // Clean up IME state and reset to the previous IME method.
        this.activeImeContextId_ = InputControllerImpl.NO_ACTIVE_IME_CONTEXT_ID_;
        chrome.inputMethodPrivate.setCurrentInputMethod(this.previousImeEngineId_);
        this.previousImeEngineId_ = '';
        this.surroundingInfo_ = null;
    }
    /**
     * Commits the given text to the active IME context.
     * @param text The text to commit
     */
    commitText(text) {
        if (!this.isActive() || !text) {
            return;
        }
        const data = this.getEditableNodeData();
        if (LocaleInfo.allowSmartCapAndSpacing() &&
            this.checkEditableNodeData_(data)) {
            const { value, selStart } = data;
            text = EditingUtil.smartCapitalization(value, selStart, text);
            text = EditingUtil.smartSpacing(value, selStart, text);
        }
        if (!LocaleInfo.considerSpaces()) {
            // Remove spaces from within `text`, if there are any. In practice, it's
            // possible for the speech recognition service to return text with spaces
            // within the utterance, even if that language doesn't separate words
            // with spaces (e.g. Japanese).
            let textNoSpaces = '';
            for (const char of text) {
                if (char === ' ') {
                    continue;
                }
                textNoSpaces += char;
            }
            text = textNoSpaces;
        }
        chrome.input.ime.commitText({ contextID: this.activeImeContextId_, text });
    }
    /**
     * chrome.input.ime.onFocus callback. Save the active context ID, and
     * finish starting speech recognition if needed. This needs to be done
     * before starting recognition in order for browser tests to know that
     * Dictation is already set up as an IME.
     * @param context Input field context.
     */
    onImeFocus_(context) {
        this.activeImeContextId_ = context.contextID;
        if (this.onConnectCallback_) {
            this.onConnectCallback_();
            this.onConnectCallback_ = null;
        }
    }
    /**
     * chrome.input.ime.onFocus callback. Stops Dictation if the active
     * context ID lost focus.
     */
    onImeBlur_(contextId) {
        if (contextId === this.activeImeContextId_) {
            // Clean up context ID immediately. We can no longer use this context.
            this.activeImeContextId_ = InputControllerImpl.NO_ACTIVE_IME_CONTEXT_ID_;
            this.surroundingInfo_ = null;
            this.stopDictationCallback_();
        }
    }
    /**
     * Called when the editable string around the caret is changed or when the
     * caret position is moved.
     */
    onSurroundingTextChanged_(engineID, surroundingInfo) {
        if (engineID !==
            InputControllerImpl.ON_SURROUNDING_TEXT_CHANGED_ENGINE_ID) {
            return;
        }
        this.surroundingInfo_ = surroundingInfo;
        if (this.onSurroundingTextChangedForTesting_) {
            this.onSurroundingTextChangedForTesting_();
        }
    }
    /**
     * Deletes the sentence to the left of the text caret. If the caret is in the
     * middle of a sentence, it will delete a portion of the sentence it
     * intersects.
     */
    deletePrevSentence() {
        const data = this.getEditableNodeData();
        if (!this.checkEditableNodeData_(data)) {
            return;
        }
        const { value, selStart } = data;
        const prevSentenceStart = EditingUtil.navPrevSent(value, selStart);
        const length = selStart - prevSentenceStart;
        this.deleteSurroundingText_(length, -length);
    }
    /**
     * @param length The number of characters to be deleted.
     * @param offset The offset from the caret position where deletion will start.
     *     This value can be negative.
     */
    async deleteSurroundingText_(length, offset) {
        const editableNode = this.focusHandler_.getEditableNode();
        if (!editableNode) {
            throw new Error('deleteSurroundingText_ requires a valid editable node');
        }
        const deleteSurroundingText = () => {
            chrome.input.ime.deleteSurroundingText({
                contextID: this.activeImeContextId_,
                engineID: InputControllerImpl.IME_ENGINE_ID,
                length,
                offset,
            });
        };
        // Delete the surrounding text and wait for events to propagate.
        const waiter = new AutomationImeEventWaiter(editableNode, EventType$1.VALUE_IN_TEXT_FIELD_CHANGED);
        await waiter.doActionAndWait(deleteSurroundingText);
    }
    /**
     * Deletes a phrase to the left of the text caret. If multiple instances of
     * `phrase` are present, it deletes the one closest to the text caret.
     * @param phrase The phrase to be deleted.
     */
    deletePhrase(phrase) {
        this.replacePhrase(phrase, '');
    }
    /**
     * Replaces a phrase to the left of the text caret with another phrase. If
     * multiple instances of `deletePhrase` are present, this function will
     * replace the one closest to the text caret.
     * @param deletePhrase The phrase to be deleted.
     * @param insertPhrase The phrase to be inserted.
     */
    async replacePhrase(deletePhrase, insertPhrase) {
        const data = this.getEditableNodeData();
        if (!this.checkEditableNodeData_(data)) {
            return;
        }
        const { value, selStart } = data;
        const replacePhraseData = EditingUtil.getReplacePhraseData(value, selStart, deletePhrase);
        if (!replacePhraseData) {
            return;
        }
        const { newIndex, deleteLength } = replacePhraseData;
        await this.setSelection_(newIndex, newIndex);
        await this.deleteSurroundingText_(deleteLength, deleteLength);
        if (insertPhrase) {
            this.commitText(insertPhrase);
        }
    }
    /**
     * Sets the selection within the editable node. `selStart` and `selEnd` are
     * relative to the value of the editable node. Works in all types of text
     * fields, including content editables.
     */
    async setSelection_(selStart, selEnd) {
        const editableNode = this.focusHandler_.getEditableNode();
        if (!editableNode) {
            return;
        }
        let anchorObject = editableNode;
        let anchorOffset = selStart;
        let focusObject = editableNode;
        let focusOffset = selEnd;
        // TODO(b/314203187): Determine if not null assertion is acceptable.
        const isContentEditable = editableNode.state[StateType$1.RICHLY_EDITABLE];
        if (isContentEditable) {
            // Contenteditables can contain multiple inline text nodes, so we need to
            // translate `selStart` and `selEnd` to a node and index within the
            // contenteditable.
            let data = this.textNodeAndIndex_(selStart);
            if (data) {
                anchorObject = data.node;
                anchorOffset = data.index;
            }
            data = this.textNodeAndIndex_(selEnd);
            if (data) {
                focusObject = data.node;
                focusOffset = data.index;
            }
        }
        const setDocumentSelection = () => {
            chrome.automation.setDocumentSelection({ anchorObject, anchorOffset, focusObject, focusOffset });
        };
        // Set selection and wait for events to propagate.
        const waiter = new AutomationImeEventWaiter(editableNode, EventType$1.TEXT_SELECTION_CHANGED);
        await waiter.doActionAndWait(setDocumentSelection);
        if (this.onSelectionChangedForTesting_) {
            this.onSelectionChangedForTesting_();
        }
    }
    /**
     * Inserts `insertPhrase` directly before `beforePhrase` (and separates them
     * with a space). This function operates on the text to the left of the caret.
     * If multiple instances of `beforePhrase` are present, this function will
     * use the one closest to the text caret.
     */
    async insertBefore(insertPhrase, beforePhrase) {
        const data = this.getEditableNodeData();
        if (!this.checkEditableNodeData_(data)) {
            return;
        }
        const { value, selStart } = data;
        const newIndex = EditingUtil.getInsertBeforeIndex(value, selStart, beforePhrase);
        if (newIndex === -1) {
            return;
        }
        await this.setSelection_(newIndex, newIndex);
        this.commitText(insertPhrase);
    }
    /**
     * Sets selection starting at `startPhrase` and ending at `endPhrase`
     * (inclusive). The function operates on the text to the left of the text
     * caret. If multiple instances of `startPhrase` or `endPhrase` are present,
     * the function will use the ones closest to the text caret.
     */
    selectBetween(startPhrase, endPhrase) {
        const data = this.getEditableNodeData();
        if (!this.checkEditableNodeData_(data)) {
            return;
        }
        const { value, selStart } = data;
        const selection = EditingUtil.selectBetween(value, selStart, startPhrase, endPhrase);
        if (!selection) {
            return;
        }
        this.setSelection_(selection.start, selection.end);
    }
    /** Moves the text caret to the next sentence. */
    async navNextSent() {
        const data = this.getEditableNodeData();
        if (!this.checkEditableNodeData_(data)) {
            return;
        }
        const { value, selStart } = data;
        const newCaretIndex = EditingUtil.navNextSent(value, selStart);
        await this.setSelection_(newCaretIndex, newCaretIndex);
    }
    /** Moves the text caret to the previous sentence. */
    async navPrevSent() {
        const data = this.getEditableNodeData();
        if (!this.checkEditableNodeData_(data)) {
            return;
        }
        const { value, selStart } = data;
        const newCaretIndex = EditingUtil.navPrevSent(value, selStart);
        await this.setSelection_(newCaretIndex, newCaretIndex);
    }
    /**
     * Returns the editable node, its value, the selection start, and the
     * selection end.
     * TODO(crbug.com/1353871): Only return text that is visible on-screen.
     */
    getEditableNodeData() {
        const node = this.focusHandler_.getEditableNode();
        if (!node) {
            return null;
        }
        let value;
        let selStart;
        let selEnd;
        // TODO(b/314203187): Determine if not null assertion is acceptable.
        const isContentEditable = node.state[StateType$1.RICHLY_EDITABLE];
        if (isContentEditable && this.surroundingInfo_) {
            const info = this.surroundingInfo_;
            // Use IME data only in contenteditables.
            value = info.text;
            selStart = Math.min(info.anchor, info.focus);
            selEnd = Math.max(info.anchor, info.focus);
            return { node, value, selStart, selEnd };
        }
        // Fall back to data from Automation.
        value = node.value || '';
        selStart = (node.textSelStart !== undefined && node.textSelStart !== -1) ?
            node.textSelStart :
            value.length;
        selEnd = (node.textSelEnd !== undefined && node.textSelEnd !== -1) ?
            node.textSelEnd :
            value.length;
        return {
            node,
            value,
            selStart: Math.min(selStart, selEnd),
            selEnd: Math.max(selStart, selEnd),
        };
    }
    /**
     * Returns whether or not `data` meets the prerequisites for performing an
     * editing command.
     */
    checkEditableNodeData_(data) {
        if (!data || data.selStart !== data.selEnd) {
            // TODO(b:259353226): Move this selection check into checkContext()
            // method.
            return false;
        }
        return true;
    }
    /**
     * Translates `index`, which is relative to the editable's value, to an inline
     * text node and index within the editable. Only returns valid data when the
     * editable node is a contenteditable.
     */
    textNodeAndIndex_(index) {
        const editableNode = this.focusHandler_.getEditableNode();
        // TODO(b/314203187): Determine if not null assertion is acceptable.
        if (!editableNode || !editableNode.state[StateType$1.RICHLY_EDITABLE]) {
            throw new Error('textNodeAndIndex_ requires a content editable node');
        }
        const position = editableNode.createPosition(PositionType.TEXT, index);
        position.asLeafTextPosition();
        if (!position || !position.node || position.textOffset === undefined) {
            return null;
        }
        return {
            node: position.node,
            index: position.textOffset,
        };
    }
}
(function (InputControllerImpl) {
    /** The IME engine ID for AccessibilityCommon. */
    InputControllerImpl.IME_ENGINE_ID = '_ext_ime_egfdjlfmgnehecnclamagfafdccgfndpdictation';
    /**
     * The engine ID that is passed into `onSurroundingTextChanged_` when
     * Dictation modifies the text field.
     */
    InputControllerImpl.ON_SURROUNDING_TEXT_CHANGED_ENGINE_ID = 'dictation';
    InputControllerImpl.NO_ACTIVE_IME_CONTEXT_ID_ = -1;
})(InputControllerImpl || (InputControllerImpl = {}));
TestImportManager.exportForTesting(InputControllerImpl);

// Copyright 2021 The Chromium Authors
// Use of this source code is governed by a BSD-style license that can be
// found in the LICENSE file.
var SpeechRecognitionType$1 = chrome.speechRecognitionPrivate.SpeechRecognitionType;
/** A class used record metrics for the dictation feature. */
let MetricsUtils$1 = class MetricsUtils {
    onDevice_;
    locale_;
    speechRecognitionStartTime_ = null;
    constructor(type, locale) {
        this.onDevice_ = type === SpeechRecognitionType$1.ON_DEVICE;
        this.locale_ = locale;
    }
    static recordMacroRecognized(macro) {
        chrome.metricsPrivate.recordSparseValue(MetricsUtils.MACRO_RECOGNIZED_METRIC, macro.getName());
    }
    static recordMacroSucceeded(macro) {
        chrome.metricsPrivate.recordSparseValue(MetricsUtils.MACRO_SUCCEEDED_METRIC, macro.getName());
    }
    static recordMacroFailed(macro) {
        chrome.metricsPrivate.recordSparseValue(MetricsUtils.MACRO_FAILED_METRIC, macro.getName());
    }
    static recordPumpkinUsed(used) {
        chrome.metricsPrivate.recordBoolean(MetricsUtils.PUMPKIN_USED_METRIC, used);
    }
    static recordPumpkinSucceeded(succeeded) {
        chrome.metricsPrivate.recordBoolean(MetricsUtils.PUMPKIN_SUCCEEDED_METRIC, succeeded);
    }
    /** Records metrics when speech recognition starts. */
    recordSpeechRecognitionStarted() {
        chrome.metricsPrivate.recordBoolean(MetricsUtils.ON_DEVICE_SPEECH_METRIC, this.onDevice_);
        chrome.metricsPrivate.recordSparseValueWithHashMetricName(MetricsUtils.LOCALE_METRIC, this.locale_);
        this.speechRecognitionStartTime_ = new Date();
    }
    /**
     * Records metrics when speech recognition stops. Must be called after
     * `recordSpeechRecognitionStarted` is called.
     */
    recordSpeechRecognitionStopped() {
        if (this.speechRecognitionStartTime_ === null) {
            // Check that we have called `recordSpeechRecognitionStarted` by
            // checking `speechRecognitionStartTime_`.
            console.warn(`Failed to record metrics when speech recognition stopped, valid
          speech recognition start time required.`);
            return;
        }
        const metricName = this.onDevice_ ?
            MetricsUtils.LISTENING_DURATION_METRIC_ON_DEVICE :
            MetricsUtils.LISTENING_DURATION_METRIC_NETWORK;
        const listeningDuration = new Date().getTime() - this.speechRecognitionStartTime_.getTime();
        chrome.metricsPrivate.recordLongTime(metricName, listeningDuration);
    }
};
(function (MetricsUtils) {
    /**
     * The metric used to record whether on-device or network speech recognition
     * was used.
     */
    MetricsUtils.ON_DEVICE_SPEECH_METRIC = 'Accessibility.CrosDictation.UsedOnDeviceSpeech';
    /**
     * The metric used to record which locale Dictation used for speech
     * recognition.
     */
    MetricsUtils.LOCALE_METRIC = 'Accessibility.CrosDictation.Language';
    /**
     * The metric used to record the listening duration for on-device speech
     * recognition.
     */
    MetricsUtils.LISTENING_DURATION_METRIC_ON_DEVICE = 'Accessibility.CrosDictation.ListeningDuration.OnDeviceRecognition';
    /**
     * The metric used to record the listening duration for network speech
     * recognition.
     */
    MetricsUtils.LISTENING_DURATION_METRIC_NETWORK = 'Accessibility.CrosDictation.ListeningDuration.NetworkRecognition';
    /** The metric used to record that a macro was recognized. */
    MetricsUtils.MACRO_RECOGNIZED_METRIC = 'Accessibility.CrosDictation.MacroRecognized';
    /** The metric used to record that a macro succeeded. */
    MetricsUtils.MACRO_SUCCEEDED_METRIC = 'Accessibility.CrosDictation.MacroSucceeded';
    /** The metric used to record that a macro failed. */
    MetricsUtils.MACRO_FAILED_METRIC = 'Accessibility.CrosDictation.MacroFailed';
    /**
     * The metric used to record whether or not Pumpkin was used for command
     * parsing.
     */
    MetricsUtils.PUMPKIN_USED_METRIC = 'Accessibility.CrosDictation.UsedPumpkin';
    /**
     * The metric used to record whether or not Pumpkin succeeded in parsing a
     * command.
     */
    MetricsUtils.PUMPKIN_SUCCEEDED_METRIC = 'Accessibility.CrosDictation.PumpkinSucceeded';
})(MetricsUtils$1 || (MetricsUtils$1 = {}));

// Copyright 2021 The Chromium Authors
// Use of this source code is governed by a BSD-style license that can be
// found in the LICENSE file.
/**
 * Represents a strategy for parsing speech input and converting it into a
 * Macro.
 */
class ParseStrategy {
    inputController_;
    enabled = false;
    constructor(inputController) {
        this.inputController_ = inputController;
    }
    getInputController() {
        return this.inputController_;
    }
    isEnabled() {
        return this.enabled;
    }
    setEnabled(enabled) {
        this.enabled = enabled;
    }
    /** Refreshes this strategy when the locale changes. */
    refresh() { }
    /** Accepts text, parses it, and returns a Macro. */
    async parse(text) {
        throw new Error(`The parse() function must be implemented by each subclass. Trying to parse text: ${text}`);
    }
}

// Copyright 2021 The Chromium Authors
// Use of this source code is governed by a BSD-style license that can be
// found in the LICENSE file.
/** A parsing strategy that tells text to be input as-is. */
class InputTextStrategy extends ParseStrategy {
    constructor(inputController) {
        super(inputController);
        // InputTextStrategy is always enabled.
        this.enabled = true;
    }
    async parse(text) {
        return new InputTextViewMacro(text, this.getInputController());
    }
}

// Copyright 2021 The Chromium Authors
// Use of this source code is governed by a BSD-style license that can be
// found in the LICENSE file.
/**
 * Class that implements a macro to list Dictation commands (by opening a Help
 * Center article)
 */
class ListCommandsMacro extends Macro {
    constructor() {
        super(MacroName.LIST_COMMANDS);
    }
    run() {
        // Note that this will open a new tab, ending the current Dictation session
        // by changing the input focus.
        globalThis.open('https://support.google.com/chromebook?p=text_dictation_m100', '_blank');
        return this.createRunMacroResult_(/*isSuccess=*/ true);
    }
}

// Copyright 2022 The Chromium Authors
// Use of this source code is governed by a BSD-style license that can be
// found in the LICENSE file.
/**
 * @fileoverview Defines constants used for Pumpkin.
 */
/** The types of commands that can come from SandboxedPumpkinTagger. */
var FromPumpkinTaggerCommand;
(function (FromPumpkinTaggerCommand) {
    FromPumpkinTaggerCommand["READY"] = "ready";
    FromPumpkinTaggerCommand["FULLY_INITIALIZED"] = "fullyInitialized";
    FromPumpkinTaggerCommand["TAG_RESULTS"] = "tagResults";
    FromPumpkinTaggerCommand["REFRESHED"] = "refreshed";
})(FromPumpkinTaggerCommand || (FromPumpkinTaggerCommand = {}));
/** The types of commands that can be sent to SandboxedPumpkinTagger. */
var ToPumpkinTaggerCommand;
(function (ToPumpkinTaggerCommand) {
    ToPumpkinTaggerCommand["LOAD"] = "load";
    ToPumpkinTaggerCommand["TAG"] = "tagAndGetNBestHypotheses";
    ToPumpkinTaggerCommand["REFRESH"] = "refresh";
})(ToPumpkinTaggerCommand || (ToPumpkinTaggerCommand = {}));
/** Supported Pumpkin locales. */
var PumpkinLocale;
(function (PumpkinLocale) {
    PumpkinLocale["EN_US"] = "en_us";
    PumpkinLocale["FR_FR"] = "fr_fr";
    PumpkinLocale["IT_IT"] = "it_it";
    PumpkinLocale["DE_DE"] = "de_de";
    PumpkinLocale["ES_ES"] = "es_es";
})(PumpkinLocale || (PumpkinLocale = {}));
/**
 * Map from BCP-47 locale code (see dictation.cc) to directory name in
 * dictation/parse/pumpkin/ for supported Pumpkin locales.
 * TODO(crbug.com/1264544): Determine if all en* languages can be mapped to
 * en_us. Possible locales are listed in dictation.cc,
 * kWebSpeechSupportedLocales.
 */
const SUPPORTED_LOCALES = {
    // English.
    'en-US': PumpkinLocale.EN_US,
    'en-AU': PumpkinLocale.EN_US,
    'en-CA': PumpkinLocale.EN_US,
    'en-GB': PumpkinLocale.EN_US,
    'en-GH': PumpkinLocale.EN_US,
    'en-HK': PumpkinLocale.EN_US,
    'en-IN': PumpkinLocale.EN_US,
    'en-KE': PumpkinLocale.EN_US,
    'en-NG': PumpkinLocale.EN_US,
    'en-NZ': PumpkinLocale.EN_US,
    'en-PH': PumpkinLocale.EN_US,
    'en-PK': PumpkinLocale.EN_US,
    'en-SG': PumpkinLocale.EN_US,
    'en-TZ': PumpkinLocale.EN_US,
    'en-ZA': PumpkinLocale.EN_US,
    // French.
    'fr-BE': PumpkinLocale.FR_FR,
    'fr-CA': PumpkinLocale.FR_FR,
    'fr-CH': PumpkinLocale.FR_FR,
    'fr-FR': PumpkinLocale.FR_FR,
    // Italian.
    'it-CH': PumpkinLocale.IT_IT,
    'it-IT': PumpkinLocale.IT_IT,
    // German.
    'de-AT': PumpkinLocale.DE_DE,
    'de-CH': PumpkinLocale.DE_DE,
    'de-DE': PumpkinLocale.DE_DE,
    // Spanish.
    'es-AR': PumpkinLocale.ES_ES,
    'es-BO': PumpkinLocale.ES_ES,
    'es-CL': PumpkinLocale.ES_ES,
    'es-CO': PumpkinLocale.ES_ES,
    'es-CR': PumpkinLocale.ES_ES,
    'es-DO': PumpkinLocale.ES_ES,
    'es-EC': PumpkinLocale.ES_ES,
    'es-ES': PumpkinLocale.ES_ES,
    'es-GT': PumpkinLocale.ES_ES,
    'es-HN': PumpkinLocale.ES_ES,
    'es-MX': PumpkinLocale.ES_ES,
    'es-NI': PumpkinLocale.ES_ES,
    'es-PA': PumpkinLocale.ES_ES,
    'es-PE': PumpkinLocale.ES_ES,
    'es-PR': PumpkinLocale.ES_ES,
    'es-PY': PumpkinLocale.ES_ES,
    'es-SV': PumpkinLocale.ES_ES,
    'es-US': PumpkinLocale.ES_ES,
    'es-UY': PumpkinLocale.ES_ES,
    'es-VE': PumpkinLocale.ES_ES,
};
/**
 * PumpkinTagger Hypothesis argument names. These should match the variable
 * argument placeholders in voiceaccess.patterns_template and the static strings
 * defined in voiceaccess/utils/PumpkinUtils.java in google3.
 */
var HypothesisArgumentName;
(function (HypothesisArgumentName) {
    HypothesisArgumentName["SEM_TAG"] = "SEM_TAG";
    HypothesisArgumentName["NUM_ARG"] = "NUM_ARG";
    HypothesisArgumentName["OPEN_ENDED_TEXT"] = "OPEN_ENDED_TEXT";
    HypothesisArgumentName["BEGIN_PHRASE"] = "BEGIN_PHRASE";
    HypothesisArgumentName["END_PHRASE"] = "END_PHRASE";
})(HypothesisArgumentName || (HypothesisArgumentName = {}));
const SANDBOXED_PUMPKIN_TAGGER_JS_FILE = 'dictation/parse/sandboxed_pumpkin_tagger.rollup.js';
TestImportManager.exportForTesting(['SUPPORTED_LOCALES', SUPPORTED_LOCALES]);

// Copyright 2021 The Chromium Authors
// Use of this source code is governed by a BSD-style license that can be
// found in the LICENSE file.
/** A parsing strategy that utilizes the Pumpkin semantic parser. */
class PumpkinParseStrategy extends ParseStrategy {
    /** @param {!InputController} inputController */
    constructor(inputController) {
        super(inputController);
        this.init_();
    }
    pumpkinData_ = null;
    pumpkinTaggerReady_ = false;
    tagResolver_ = null;
    worker_ = null;
    locale_ = null;
    requestedPumpkinInstall_ = false;
    onPumpkinTaggerReadyChangedForTesting_ = null;
    init_() {
        this.refreshLocale_();
        if (!this.locale_) {
            return;
        }
        this.requestedPumpkinInstall_ = true;
        chrome.accessibilityPrivate.installPumpkinForDictation(data => {
            // TODO(crbug.com/259352407): Consider retrying installation at a later
            // time if it failed.
            this.onPumpkinInstalled_(data);
        });
    }
    onPumpkinInstalled_(data) {
        if (!data) {
            console.warn('Pumpkin installed, but data is empty');
            return;
        }
        for (const [key, value] of Object.entries(data)) {
            if (!value || value.byteLength === 0) {
                throw new Error(`Pumpkin data incomplete, missing data for ${key}`);
            }
        }
        this.refreshLocale_();
        if (!this.locale_ || !this.isEnabled()) {
            return;
        }
        // Create SandboxedPumpkinTagger.
        this.setPumpkinTaggerReady_(false);
        this.pumpkinData_ = data;
        this.worker_ = new Worker(new URL(SANDBOXED_PUMPKIN_TAGGER_JS_FILE, import.meta.url), { type: 'module' });
        this.worker_.onmessage = (message) => this.onMessage_(message);
    }
    /**
     * Called when the SandboxedPumpkinTagger posts a message to the background
     * context.
     */
    onMessage_(message) {
        const command = message.data;
        switch (command.type) {
            case FromPumpkinTaggerCommand.READY:
                this.refreshLocale_();
                if (!this.locale_) {
                    throw new Error(`Can't load SandboxedPumpkinTagger in an unsupported locale ${LocaleInfo.locale}`);
                }
                this.sendToSandboxedPumpkinTagger_({
                    type: ToPumpkinTaggerCommand.LOAD,
                    locale: this.locale_,
                    pumpkinData: this.pumpkinData_,
                });
                this.pumpkinData_ = null;
                return;
            case FromPumpkinTaggerCommand.FULLY_INITIALIZED:
                this.setPumpkinTaggerReady_(true);
                this.maybeRefresh_();
                return;
            case FromPumpkinTaggerCommand.TAG_RESULTS:
                // TODO(crbug.com/314203187): Not null asserted, check that this is
                // correct.
                this.tagResolver_(command.results);
                return;
            case FromPumpkinTaggerCommand.REFRESHED:
                this.setPumpkinTaggerReady_(true);
                this.maybeRefresh_();
                return;
            default:
                throw new Error(`Unrecognized message received from SandboxedPumpkinTagger: ${command.type}`);
        }
    }
    sendToSandboxedPumpkinTagger_(command) {
        if (!this.worker_) {
            throw new Error(`Worker not ready, cannot send command to SandboxedPumpkinTagger: ${command.type}`);
        }
        this.worker_.postMessage(command);
    }
    /**
     * In Android Voice Access, Pumpkin Hypotheses will be converted to UserIntent
     * protos before being passed to Macros.
     * @return The macro matching the hypothesis if one can be found.
     */
    macroFromPumpkinHypothesis_(hypothesis) {
        const numArgs = hypothesis.actionArgumentList.length;
        if (!numArgs) {
            return null;
        }
        let repeat = 1;
        let text = '';
        let tag = MacroName.UNSPECIFIED;
        let beginPhrase = '';
        let endPhrase = '';
        for (let i = 0; i < numArgs; i++) {
            const argument = hypothesis.actionArgumentList[i];
            // See Variable Argument Placeholders in voiceaccess.patterns_template.
            if (argument.name === HypothesisArgumentName.SEM_TAG) {
                // Map Pumpkin's STOP_LISTENING to generic TOGGLE_DICTATION macro.
                // When this is run by Dictation, it always stops.
                if (argument.value === 'STOP_LISTENING') {
                    tag = MacroName.TOGGLE_DICTATION;
                }
                else {
                    tag = MacroName[argument.value];
                }
            }
            else if (argument.name === HypothesisArgumentName.NUM_ARG) {
                repeat = argument.value;
            }
            else if (argument.name === HypothesisArgumentName.OPEN_ENDED_TEXT) {
                text = argument.value;
            }
            else if (argument.name === HypothesisArgumentName.BEGIN_PHRASE) {
                beginPhrase = argument.value;
            }
            else if (argument.name === HypothesisArgumentName.END_PHRASE) {
                endPhrase = argument.value;
            }
        }
        switch (tag) {
            case MacroName.INPUT_TEXT_VIEW:
                return new InputTextViewMacro(text, this.getInputController());
            case MacroName.DELETE_PREV_CHAR:
                return new RepeatableKeyPressMacro.DeletePreviousCharacterMacro(this.getInputController(), repeat);
            case MacroName.NAV_PREV_CHAR:
                return new RepeatableKeyPressMacro.NavPreviousCharMacro(this.getInputController(), LocaleInfo.isRTLLocale(), repeat);
            case MacroName.NAV_NEXT_CHAR:
                return new RepeatableKeyPressMacro.NavNextCharMacro(this.getInputController(), LocaleInfo.isRTLLocale(), repeat);
            case MacroName.NAV_PREV_LINE:
                return new RepeatableKeyPressMacro.NavPreviousLineMacro(this.getInputController(), repeat);
            case MacroName.NAV_NEXT_LINE:
                return new RepeatableKeyPressMacro.NavNextLineMacro(this.getInputController(), repeat);
            case MacroName.COPY_SELECTED_TEXT:
                return new RepeatableKeyPressMacro.CopySelectedTextMacro(this.getInputController());
            case MacroName.PASTE_TEXT:
                return new RepeatableKeyPressMacro.PasteTextMacro();
            case MacroName.CUT_SELECTED_TEXT:
                return new RepeatableKeyPressMacro.CutSelectedTextMacro(this.getInputController());
            case MacroName.UNDO_TEXT_EDIT:
                return new RepeatableKeyPressMacro.UndoTextEditMacro();
            case MacroName.REDO_ACTION:
                return new RepeatableKeyPressMacro.RedoActionMacro();
            case MacroName.SELECT_ALL_TEXT:
                return new RepeatableKeyPressMacro.SelectAllTextMacro(this.getInputController());
            case MacroName.UNSELECT_TEXT:
                return new RepeatableKeyPressMacro.UnselectTextMacro(this.getInputController(), LocaleInfo.isRTLLocale());
            case MacroName.LIST_COMMANDS:
                return new ListCommandsMacro();
            case MacroName.TOGGLE_DICTATION:
                return new ToggleDictationMacro(this.getInputController().isActive());
            case MacroName.DELETE_PREV_WORD:
                return new RepeatableKeyPressMacro.DeletePrevWordMacro(this.getInputController(), repeat);
            case MacroName.DELETE_PREV_SENT:
                return new DeletePrevSentMacro(this.getInputController());
            case MacroName.NAV_NEXT_WORD:
                return new RepeatableKeyPressMacro.NavNextWordMacro(this.getInputController(), LocaleInfo.isRTLLocale(), repeat);
            case MacroName.NAV_PREV_WORD:
                return new RepeatableKeyPressMacro.NavPrevWordMacro(this.getInputController(), LocaleInfo.isRTLLocale(), repeat);
            case MacroName.SMART_DELETE_PHRASE:
                return new SmartDeletePhraseMacro(this.getInputController(), text);
            case MacroName.SMART_REPLACE_PHRASE:
                return new SmartReplacePhraseMacro(this.getInputController(), beginPhrase, text);
            case MacroName.SMART_INSERT_BEFORE:
                return new SmartInsertBeforeMacro(this.getInputController(), text, endPhrase);
            case MacroName.SMART_SELECT_BTWN_INCL:
                return new SmartSelectBetweenMacro(this.getInputController(), beginPhrase, endPhrase);
            case MacroName.NAV_NEXT_SENT:
                return new NavNextSentMacro(this.getInputController());
            case MacroName.NAV_PREV_SENT:
                return new NavPrevSentMacro(this.getInputController());
            case MacroName.DELETE_ALL_TEXT:
                return new RepeatableKeyPressMacro.DeleteAllText(this.getInputController());
            case MacroName.NAV_START_TEXT:
                return new RepeatableKeyPressMacro.NavStartText(this.getInputController());
            case MacroName.NAV_END_TEXT:
                return new RepeatableKeyPressMacro.NavEndText(this.getInputController());
            case MacroName.SELECT_PREV_WORD:
                return new RepeatableKeyPressMacro.SelectPrevWord(this.getInputController(), repeat);
            case MacroName.SELECT_NEXT_WORD:
                return new RepeatableKeyPressMacro.SelectNextWord(this.getInputController(), repeat);
            case MacroName.SELECT_NEXT_CHAR:
                return new RepeatableKeyPressMacro.SelectNextChar(this.getInputController(), repeat);
            case MacroName.SELECT_PREV_CHAR:
                return new RepeatableKeyPressMacro.SelectPrevChar(this.getInputController(), repeat);
            case MacroName.REPEAT:
                return new RepeatMacro();
            default:
                // Every hypothesis is guaranteed to include a semantic tag due to the
                // way Voice Access set up its grammars. Not all tags are supported in
                // Dictation yet.
                console.log('Unsupported Pumpkin action: ', tag);
                return null;
        }
    }
    refreshLocale_() {
        this.locale_ = SUPPORTED_LOCALES[LocaleInfo.locale] || null;
    }
    /**
     * Refreshes SandboxedPumpkinTagger if the Dictation locale differs from
     * the pumpkin locale.
     */
    maybeRefresh_() {
        const dictationLocale = SUPPORTED_LOCALES[LocaleInfo.locale] || null;
        if (dictationLocale !== this.locale_) {
            this.refresh();
        }
    }
    refresh() {
        this.refreshLocale_();
        this.enabled = Boolean(this.locale_) && LocaleInfo.areCommandsSupported();
        if (!this.requestedPumpkinInstall_) {
            this.init_();
            return;
        }
        if (!this.isEnabled() || !this.locale_ || !this.pumpkinTaggerReady_) {
            return;
        }
        this.setPumpkinTaggerReady_(false);
        this.sendToSandboxedPumpkinTagger_({
            type: ToPumpkinTaggerCommand.REFRESH,
            locale: this.locale_,
        });
    }
    async parse(text) {
        if (!this.isEnabled() || !this.pumpkinTaggerReady_) {
            return null;
        }
        this.tagResolver_ = null;
        // Get results from Pumpkin.
        // TODO(crbug.com/1264544): Could increase the hypotheses count from 1
        // when we are ready to implement disambiguation.
        this.sendToSandboxedPumpkinTagger_({
            type: ToPumpkinTaggerCommand.TAG,
            text,
            numResults: 1,
        });
        const taggerResults = await new Promise((resolve) => {
            this.tagResolver_ = resolve;
        });
        if (!taggerResults || taggerResults.hypothesisList.length === 0) {
            return null;
        }
        return this.macroFromPumpkinHypothesis_(taggerResults.hypothesisList[0]);
    }
    isEnabled() {
        return this.enabled;
    }
    setPumpkinTaggerReady_(ready) {
        this.pumpkinTaggerReady_ = ready;
        if (this.onPumpkinTaggerReadyChangedForTesting_) {
            this.onPumpkinTaggerReadyChangedForTesting_();
        }
    }
}

// Copyright 2021 The Chromium Authors
// Use of this source code is governed by a BSD-style license that can be
// found in the LICENSE file.
/**
 * SimpleMacroFactory converts spoken strings into Macros using string matching.
 */
class SimpleMacroFactory {
    macroName_;
    inputController_;
    commandRegex_ = null;
    constructor(macroName, inputController) {
        if (!SimpleMacroFactory.getData_()[macroName]) {
            throw new Error('Macro is not supported by SimpleMacroFactory: ' + macroName);
        }
        this.macroName_ = macroName;
        this.inputController_ = inputController;
        this.initializeCommandRegex_(this.macroName_);
    }
    /**
     * Builds a RegExp that can be used to parse a command. For example, the
     * SmartReplacePhraseMacro can be parsed with the pattern:
     * /replace (.*) with (.*)/i.
     */
    initializeCommandRegex_(macroName) {
        const matchAnythingPattern = '(.*)';
        const args = [];
        switch (macroName) {
            case MacroName.INPUT_TEXT_VIEW:
            case MacroName.SMART_DELETE_PHRASE:
                args.push(matchAnythingPattern);
                break;
            case MacroName.SMART_REPLACE_PHRASE:
            case MacroName.SMART_INSERT_BEFORE:
            case MacroName.SMART_SELECT_BTWN_INCL:
                args.push(matchAnythingPattern, matchAnythingPattern);
                break;
        }
        // TODO(crbug.com/314203187): Not null asserted, check that this is correct.
        const message = chrome.i18n.getMessage(SimpleMacroFactory.getData_()[macroName].messageId, args);
        const pattern = `^${message}$`;
        if (LocaleInfo.considerSpaces()) {
            this.commandRegex_ = new RegExp(pattern, 'i');
        }
        else {
            // A regex to be used if the Dictation language doesn't use spaces e.g.
            // Japanese.
            this.commandRegex_ = new RegExp(pattern.replace(/\s+/g, ''), 'i');
        }
    }
    createMacro(text) {
        // Check whether `text` matches `this.commandRegex_`, ignoring case and
        // whitespace.
        text = text.trim().toLowerCase();
        // TODO(crbug.com/314203187): Not null asserted, check that this is correct.
        if (!this.commandRegex_.test(text)) {
            return null;
        }
        const initialArgs = [];
        switch (this.macroName_) {
            case MacroName.COPY_SELECTED_TEXT:
            case MacroName.CUT_SELECTED_TEXT:
            case MacroName.DELETE_ALL_TEXT:
            case MacroName.DELETE_PREV_CHAR:
            case MacroName.DELETE_PREV_SENT:
            case MacroName.DELETE_PREV_WORD:
            case MacroName.NAV_END_TEXT:
            case MacroName.NAV_NEXT_CHAR:
            case MacroName.NAV_NEXT_LINE:
            case MacroName.NAV_NEXT_SENT:
            case MacroName.NAV_NEXT_WORD:
            case MacroName.NAV_START_TEXT:
            case MacroName.NAV_PREV_CHAR:
            case MacroName.NAV_PREV_LINE:
            case MacroName.NAV_PREV_SENT:
            case MacroName.NAV_PREV_WORD:
            case MacroName.NEW_LINE:
            case MacroName.SELECT_ALL_TEXT:
            case MacroName.SELECT_NEXT_CHAR:
            case MacroName.SELECT_NEXT_WORD:
            case MacroName.SELECT_PREV_CHAR:
            case MacroName.SELECT_PREV_WORD:
            case MacroName.SMART_DELETE_PHRASE:
            case MacroName.SMART_INSERT_BEFORE:
            case MacroName.SMART_REPLACE_PHRASE:
            case MacroName.SMART_SELECT_BTWN_INCL:
            case MacroName.UNSELECT_TEXT:
                initialArgs.push(this.inputController_);
                break;
        }
        // TODO(crbug.com/314203187): Not null asserted, check that this is correct.
        const result = this.commandRegex_.exec(text);
        if (!result) {
            return null;
        }
        // `result[0]` contains the entire matched text, while all subsequent
        // indices contain text matched by each /(.*)/. We're only interested in
        // text matched by /(.*)/, so ignore `result[0]`.
        const extractedArgs = result.slice(1);
        const finalArgs = initialArgs.concat(extractedArgs);
        const data = SimpleMacroFactory.getData_();
        // TODO(crbug.com/314203187): Not null asserted, check that this is correct.
        const macro = new data[this.macroName_].build(...finalArgs);
        if (macro.isSmart() && !LocaleInfo.allowSmartEditing()) {
            return null;
        }
        return macro;
    }
    /**
     * Returns data that is used to create a macro. `messageId` is used to
     * retrieve the macro's command string and `build` is used to construct the
     * macro.
     */
    static getData_() {
        return {
            [MacroName.DELETE_PREV_CHAR]: {
                messageId: 'dictation_command_delete_prev_char',
                build: RepeatableKeyPressMacro.DeletePreviousCharacterMacro,
            },
            [MacroName.NAV_PREV_CHAR]: {
                messageId: 'dictation_command_nav_prev_char',
                build: RepeatableKeyPressMacro.NavPreviousCharMacro,
            },
            [MacroName.NAV_NEXT_CHAR]: {
                messageId: 'dictation_command_nav_next_char',
                build: RepeatableKeyPressMacro.NavNextCharMacro,
            },
            [MacroName.NAV_PREV_LINE]: {
                messageId: 'dictation_command_nav_prev_line',
                build: RepeatableKeyPressMacro.NavPreviousLineMacro,
            },
            [MacroName.NAV_NEXT_LINE]: {
                messageId: 'dictation_command_nav_next_line',
                build: RepeatableKeyPressMacro.NavNextLineMacro,
            },
            [MacroName.COPY_SELECTED_TEXT]: {
                messageId: 'dictation_command_copy_selected_text',
                build: RepeatableKeyPressMacro.CopySelectedTextMacro,
            },
            [MacroName.PASTE_TEXT]: {
                messageId: 'dictation_command_paste_text',
                build: RepeatableKeyPressMacro.PasteTextMacro,
            },
            [MacroName.CUT_SELECTED_TEXT]: {
                messageId: 'dictation_command_cut_selected_text',
                build: RepeatableKeyPressMacro.CutSelectedTextMacro,
            },
            [MacroName.UNDO_TEXT_EDIT]: {
                messageId: 'dictation_command_undo_text_edit',
                build: RepeatableKeyPressMacro.UndoTextEditMacro,
            },
            [MacroName.REDO_ACTION]: {
                messageId: 'dictation_command_redo_action',
                build: RepeatableKeyPressMacro.RedoActionMacro,
            },
            [MacroName.SELECT_ALL_TEXT]: {
                messageId: 'dictation_command_select_all_text',
                build: RepeatableKeyPressMacro.SelectAllTextMacro,
            },
            [MacroName.UNSELECT_TEXT]: {
                messageId: 'dictation_command_unselect_text',
                build: RepeatableKeyPressMacro.UnselectTextMacro,
            },
            [MacroName.LIST_COMMANDS]: {
                messageId: 'dictation_command_list_commands',
                build: ListCommandsMacro,
            },
            [MacroName.NEW_LINE]: { messageId: 'dictation_command_new_line', build: NewLineMacro },
            [MacroName.TOGGLE_DICTATION]: {
                messageId: 'dictation_command_stop_listening',
                build: ToggleDictationMacro,
            },
            [MacroName.DELETE_PREV_WORD]: {
                messageId: 'dictation_command_delete_prev_word',
                build: RepeatableKeyPressMacro.DeletePrevWordMacro,
            },
            [MacroName.DELETE_PREV_SENT]: {
                messageId: 'dictation_command_delete_prev_sent',
                build: DeletePrevSentMacro,
            },
            [MacroName.NAV_NEXT_WORD]: {
                messageId: 'dictation_command_nav_next_word',
                build: RepeatableKeyPressMacro.NavNextWordMacro,
            },
            [MacroName.NAV_PREV_WORD]: {
                messageId: 'dictation_command_nav_prev_word',
                build: RepeatableKeyPressMacro.NavPrevWordMacro,
            },
            [MacroName.SMART_DELETE_PHRASE]: {
                messageId: 'dictation_command_smart_delete_phrase',
                build: SmartDeletePhraseMacro,
            },
            [MacroName.SMART_REPLACE_PHRASE]: {
                messageId: 'dictation_command_smart_replace_phrase',
                build: SmartReplacePhraseMacro,
            },
            [MacroName.SMART_INSERT_BEFORE]: {
                messageId: 'dictation_command_smart_insert_before',
                build: SmartInsertBeforeMacro,
            },
            [MacroName.SMART_SELECT_BTWN_INCL]: {
                messageId: 'dictation_command_smart_select_btwn_incl',
                build: SmartSelectBetweenMacro,
            },
            [MacroName.NAV_NEXT_SENT]: {
                messageId: 'dictation_command_nav_next_sent',
                build: NavNextSentMacro,
            },
            [MacroName.NAV_PREV_SENT]: {
                messageId: 'dictation_command_nav_prev_sent',
                build: NavPrevSentMacro,
            },
        };
    }
}
/** A parsing strategy that utilizes SimpleMacroFactory. */
class SimpleParseStrategy extends ParseStrategy {
    /**
     * Map of macro names to a factory for that macro.
     */
    macroFactoryMap_ = new Map();
    supportedMacros_ = new Set();
    constructor(inputController) {
        super(inputController);
        this.initialize_();
    }
    initialize_() {
        // Adds all macros that are supported by regular expressions. If a macro
        // has a string associated with it in dictation_strings.grd, then it belongs
        // in this set. Don't add macros that require arguments in their utterances
        // e.g. "select <phrase_or_word>" - these macros are better handled by
        // Pumpkin.
        this.supportedMacros_.add(MacroName.DELETE_PREV_CHAR)
            .add(MacroName.NAV_PREV_CHAR)
            .add(MacroName.NAV_NEXT_CHAR)
            .add(MacroName.NAV_PREV_LINE)
            .add(MacroName.NAV_NEXT_LINE)
            .add(MacroName.COPY_SELECTED_TEXT)
            .add(MacroName.PASTE_TEXT)
            .add(MacroName.CUT_SELECTED_TEXT)
            .add(MacroName.UNDO_TEXT_EDIT)
            .add(MacroName.REDO_ACTION)
            .add(MacroName.SELECT_ALL_TEXT)
            .add(MacroName.UNSELECT_TEXT)
            .add(MacroName.LIST_COMMANDS)
            .add(MacroName.NEW_LINE)
            .add(MacroName.TOGGLE_DICTATION)
            .add(MacroName.DELETE_PREV_WORD)
            .add(MacroName.DELETE_PREV_SENT)
            .add(MacroName.NAV_NEXT_WORD)
            .add(MacroName.NAV_PREV_WORD)
            .add(MacroName.SMART_DELETE_PHRASE)
            .add(MacroName.SMART_REPLACE_PHRASE)
            .add(MacroName.SMART_INSERT_BEFORE)
            .add(MacroName.SMART_SELECT_BTWN_INCL)
            .add(MacroName.NAV_NEXT_SENT)
            .add(MacroName.NAV_PREV_SENT);
        this.supportedMacros_.forEach((name) => {
            this.macroFactoryMap_.set(name, new SimpleMacroFactory(name, this.getInputController()));
        });
    }
    refresh() {
        this.enabled = LocaleInfo.areCommandsSupported();
        if (!this.enabled) {
            return;
        }
        this.macroFactoryMap_ = new Map();
        this.initialize_();
    }
    async parse(text) {
        const macros = [];
        for (const [_, factory] of this.macroFactoryMap_) {
            const macro = factory.createMacro(text);
            if (macro) {
                macros.push(macro);
            }
        }
        if (macros.length === 1) {
            return macros[0];
        }
        else if (macros.length === 2) {
            // Pick which macro to use from the list of matched macros.
            // TODO(crbug.com/1288965): Turn this into a disambiguation class as we
            // add more commands. Currently the only ambiguous macro is DELETE_PHRASE
            // which conflicts with other deletion macros. For example, the phrase
            // "Delete the previous word" should be parsed as a DELETE_PREV_WORD
            // instead of SMART_DELETE_PHRASE with phrase "the previous word".
            // Prioritize other deletion macros over SMART_DELETE_PHRASE.
            return macros[0].getName() === MacroName.SMART_DELETE_PHRASE ? macros[1] :
                macros[0];
        }
        else if (macros.length > 2) {
            console.warn(`Unexpected ambiguous macros found for text: ${text}.`);
            return macros[0];
        }
        // The command is simply to input the given text.
        // If `text` starts with `type`, then automatically remove it e.g. convert
        // 'Type testing 123' to 'testing 123'.
        const typePrefix = chrome.i18n.getMessage('dictation_command_input_text_view');
        if (text.trim().toLowerCase().startsWith(typePrefix)) {
            text = text.toLowerCase().replace(typePrefix, '').trimStart();
        }
        return new InputTextViewMacro(text, this.getInputController());
    }
}

// Copyright 2021 The Chromium Authors
// Use of this source code is governed by a BSD-style license that can be
// found in the LICENSE file.
/** SpeechParser handles parsing spoken transcripts into Macros. */
class SpeechParser {
    inputController_;
    inputTextStrategy_;
    simpleParseStrategy_;
    pumpkinParseStrategy_;
    /** @param inputController to interact with the IME. */
    constructor(inputController) {
        this.inputController_ = inputController;
        this.inputTextStrategy_ = new InputTextStrategy(this.inputController_);
        this.simpleParseStrategy_ = new SimpleParseStrategy(this.inputController_);
        this.pumpkinParseStrategy_ =
            new PumpkinParseStrategy(this.inputController_);
    }
    /** Refreshes the speech parser when the locale changes. */
    refresh() {
        // Pumpkin has its own strings for command parsing, but we disable it when
        // commands aren't supported for consistency.
        this.simpleParseStrategy_.refresh();
        this.pumpkinParseStrategy_.refresh();
    }
    /**
     * Parses user text to produce a macro command.
     * @param text The text to parse.
     */
    async parse(text) {
        if (this.pumpkinParseStrategy_.isEnabled()) {
            MetricsUtils$1.recordPumpkinUsed(true);
            const macro = await this.pumpkinParseStrategy_.parse(text);
            MetricsUtils$1.recordPumpkinSucceeded(Boolean(macro));
            if (macro) {
                return macro;
            }
        }
        // If we get here, then Pumpkin failed to parse `text`. There are cases
        // where this can happen e.g. if Pumpkin failed to initialize properly.
        // Try using `simpleParseStrategy_` as a fall-back.
        if (this.simpleParseStrategy_.isEnabled()) {
            MetricsUtils$1.recordPumpkinUsed(false);
            return await this.simpleParseStrategy_.parse(text);
        }
        // Input text as-is as a catch-all.
        MetricsUtils$1.recordPumpkinUsed(false);
        return await this.inputTextStrategy_.parse(text);
    }
    /** For testing purposes only. */
    disablePumpkinForTesting() {
        this.pumpkinParseStrategy_.setEnabled(false);
    }
}
TestImportManager.exportForTesting(SpeechParser);

// Copyright 2022 The Chromium Authors
// Use of this source code is governed by a BSD-style license that can be
// found in the LICENSE file.
var HintType = chrome.accessibilityPrivate.DictationBubbleHintType;
var IconType = chrome.accessibilityPrivate.DictationBubbleIconType;
/** States the Dictation bubble UI can be in. */
/* eslint-disable @typescript-eslint/naming-convention */
var UIState;
(function (UIState) {
    UIState["STANDBY"] = "standby";
    UIState["RECOGNIZING_TEXT"] = "recognizing_text";
    UIState["MACRO_SUCCESS"] = "macro_success";
    UIState["MACRO_FAIL"] = "macro_fail";
    UIState["HIDDEN"] = "hidden";
})(UIState || (UIState = {}));
/** Contexts in which hints can be shown. */
var HintContext;
(function (HintContext) {
    HintContext["STANDBY"] = "standby";
    HintContext["TEXT_COMMITTED"] = "text_committed";
    HintContext["TEXT_SELECTED"] = "text_selected";
    HintContext["MACRO_SUCCESS"] = "macro_success";
})(HintContext || (HintContext = {}));
/**
 * Handles interaction with the Dictation UI. All changes to the UI should go
 * this class.
 */
/* eslint-disable @typescript-eslint/naming-convention */
class UIController {
    showHintsTimeoutId_ = null;
    showHintsTimeoutMs_ = UIController.HintsTimeouts.STANDARD_HINT_TIMEOUT_MS_;
    /**
     * Sets the new state of the Dictation bubble UI. If a HintContext is
     * specified, additional hints will appear in the UI after a short timeout.
     */
    setState(state, properties) {
        const props = properties || {};
        const { text, context } = props;
        // Whenever the UI state changes, we should clear the hint timeout.
        this.clearHintsTimeout_();
        switch (state) {
            case UIState.STANDBY:
                chrome.accessibilityPrivate.updateDictationBubble({ visible: true, icon: IconType.STANDBY });
                break;
            case UIState.RECOGNIZING_TEXT:
                chrome.accessibilityPrivate.updateDictationBubble({ visible: true, icon: IconType.HIDDEN, text });
                break;
            case UIState.MACRO_SUCCESS:
                chrome.accessibilityPrivate.updateDictationBubble({ visible: true, icon: IconType.MACRO_SUCCESS, text });
                break;
            case UIState.MACRO_FAIL:
                chrome.accessibilityPrivate.updateDictationBubble({ visible: true, icon: IconType.MACRO_FAIL, text });
                break;
            case UIState.HIDDEN:
                chrome.accessibilityPrivate.updateDictationBubble({ visible: false, icon: IconType.HIDDEN });
                break;
        }
        if (!context || !LocaleInfo.areCommandsSupported()) {
            // Do not show hints if commands are not supported.
            return;
        }
        // If a HintContext was provided, set a timeout to show hints.
        const hints = UIController.getHintsForContext_(context);
        this.showHintsTimeoutId_ =
            setTimeout(() => this.showHints_(hints), this.showHintsTimeoutMs_);
    }
    clearHintsTimeout_() {
        if (this.showHintsTimeoutId_ !== null) {
            clearTimeout(this.showHintsTimeoutId_);
            this.showHintsTimeoutId_ = null;
        }
    }
    /** Shows hints in the UI bubble. */
    showHints_(hints) {
        chrome.accessibilityPrivate.updateDictationBubble({ visible: true, icon: IconType.STANDBY, hints });
    }
    /**
     * In some circumstances we shouldn't show the hints too quickly because
     * it is distracting to the user.
     * @param longerDuration Whether to wait for a longer time before showing
     *     hints.
     */
    setHintsTimeoutDuration(longerDuration) {
        this.showHintsTimeoutMs_ = longerDuration ?
            UIController.HintsTimeouts.LONGER_HINT_TIMEOUT_MS_ :
            UIController.HintsTimeouts.STANDARD_HINT_TIMEOUT_MS_;
    }
    static getHintsForContext_(context) {
        return UIController.CONTEXT_TO_HINTS_MAP_[context];
    }
}
(function (UIController) {
    /** The amount of time to wait before showing hints. */
    UIController.HintsTimeouts = {
        STANDARD_HINT_TIMEOUT_MS_: 2 * 1000,
        LONGER_HINT_TIMEOUT_MS_: 6 * 1000,
    };
    /** Maps HintContexts to hints that should be shown for that context. */
    UIController.CONTEXT_TO_HINTS_MAP_ = {
        [HintContext.STANDBY]: [HintType.TRY_SAYING, HintType.TYPE, HintType.HELP],
        [HintContext.TEXT_COMMITTED]: [
            HintType.TRY_SAYING,
            HintType.UNDO,
            HintType.DELETE,
            HintType.SELECT_ALL,
            HintType.HELP,
        ],
        [HintContext.TEXT_SELECTED]: [
            HintType.TRY_SAYING,
            HintType.UNSELECT,
            HintType.COPY,
            HintType.DELETE,
            HintType.HELP,
        ],
        [HintContext.MACRO_SUCCESS]: [HintType.TRY_SAYING, HintType.UNDO, HintType.HELP],
    };
})(UIController || (UIController = {}));
TestImportManager.exportForTesting(UIController);

// Copyright 2021 The Chromium Authors
// Use of this source code is governed by a BSD-style license that can be
// found in the LICENSE file.
var StreamType = chrome.audio.StreamType;
var SpeechRecognitionType = chrome.speechRecognitionPrivate.SpeechRecognitionType;
var ToastType = chrome.accessibilityPrivate.ToastType;
/**
 * Main class for the Chrome OS dictation feature.
 * TODO(b/314204374): Eliminate instances of null.
 */
class Dictation {
    inputController_ = null;
    uiController_ = null;
    speechParser_ = null;
    /** Whether or not Dictation is active. */
    active_ = false;
    cancelTone_ = new Audio('dictation/earcons/null_selection.wav');
    startTone_ = new Audio('dictation/earcons/audio_initiate.wav');
    endTone_ = new Audio('dictation/earcons/audio_end.wav');
    noSpeechTimeoutMs_ = Dictation.Timeouts.NO_SPEECH_NETWORK_MS;
    stopTimeoutId_ = null;
    interimText_ = '';
    chromeVoxEnabled_ = false;
    speechRecognitionOptions_ = null;
    metricsUtils_ = null;
    focusHandler_ = null;
    // API Listeners //
    speechRecognitionStopListener_ = null;
    speechRecognitionResultListener_ = null;
    speechRecognitionErrorListener_ = null;
    prefsListener_ = null;
    onToggleDictationListener_ = null;
    isContextCheckingFeatureEnabled_ = false;
    prevMacro_ = null;
    constructor() {
        this.initialize_();
    }
    /** Sets up Dictation's speech recognizer and various listeners. */
    initialize_() {
        this.focusHandler_ = new FocusHandler();
        this.inputController_ = new InputControllerImpl(() => this.stopDictation_(/*notify=*/ true), this.focusHandler_);
        this.uiController_ = new UIController();
        this.speechParser_ = new SpeechParser(this.inputController_);
        this.speechParser_.refresh();
        // Set default speech recognition properties. Locale will be updated when
        // `updateFromPrefs_` is called.
        this.speechRecognitionOptions_ = {
            locale: 'en-US',
            interimResults: true,
        };
        this.speechRecognitionStopListener_ = () => this.onSpeechRecognitionStopped_();
        this.speechRecognitionResultListener_ = event => this.onSpeechRecognitionResult_(event);
        this.speechRecognitionErrorListener_ = () => this.onSpeechRecognitionError_();
        this.prefsListener_ = prefs => this.updateFromPrefs_(prefs);
        this.onToggleDictationListener_ = activated => this.onToggleDictation_(activated);
        // Setup speechRecognitionPrivate API listeners.
        chrome.speechRecognitionPrivate.onStop.addListener(this.speechRecognitionStopListener_);
        chrome.speechRecognitionPrivate.onResult.addListener(this.speechRecognitionResultListener_);
        chrome.speechRecognitionPrivate.onError.addListener(this.speechRecognitionErrorListener_);
        chrome.settingsPrivate.getAllPrefs(prefs => this.updateFromPrefs_(prefs));
        chrome.settingsPrivate.onPrefsChanged.addListener(this.prefsListener_);
        // Listen for Dictation toggles (activated / deactivated) from the Ash
        // Browser process.
        chrome.accessibilityPrivate.onToggleDictation.addListener(this.onToggleDictationListener_);
        const contextCheckingFeature = chrome.accessibilityPrivate.AccessibilityFeature
            .DICTATION_CONTEXT_CHECKING;
        chrome.accessibilityPrivate.isFeatureEnabled(contextCheckingFeature, enabled => {
            this.isContextCheckingFeatureEnabled_ = enabled;
        });
    }
    /** Performs any destruction before dictation object is destroyed. */
    onDictationDisabled() {
        if (this.speechRecognitionStopListener_) {
            chrome.speechRecognitionPrivate.onStop.removeListener(this.speechRecognitionStopListener_);
        }
        if (this.speechRecognitionResultListener_) {
            chrome.speechRecognitionPrivate.onResult.removeListener(this.speechRecognitionResultListener_);
        }
        if (this.speechRecognitionErrorListener_) {
            chrome.speechRecognitionPrivate.onError.removeListener(this.speechRecognitionErrorListener_);
        }
        if (this.prefsListener_) {
            chrome.settingsPrivate.onPrefsChanged.removeListener(this.prefsListener_);
        }
        if (this.onToggleDictationListener_) {
            chrome.accessibilityPrivate.onToggleDictation.removeListener(this.onToggleDictationListener_);
        }
        if (this.inputController_) {
            this.inputController_.removeListeners();
        }
    }
    isActive() {
        return this.active_;
    }
    /**
     * Called when Dictation is toggled.
     * @param activated Whether Dictation was just activated.
     */
    onToggleDictation_(activated) {
        if (activated && !this.active_) {
            this.startDictation_();
        }
        else {
            this.stopDictation_(/*notify=*/ false);
        }
    }
    startDictation_() {
        this.active_ = true;
        if (this.chromeVoxEnabled_) {
            // Silence ChromeVox in case it was speaking. It can speak over the start
            // tone and also cause a feedback loop if the user is not using
            // headphones. This does not stop ChromeVox from speaking additional
            // utterances added to the queue later.
            chrome.accessibilityPrivate.silenceSpokenFeedback();
        }
        this.setStopTimeout_(Dictation.Timeouts.NO_FOCUSED_IME_MS, Dictation.StopReason.NO_FOCUSED_IME);
        // TODO(b/314203187): Determine if not null assertion is acceptable.
        this.inputController_.connect(() => this.verifyMicrophoneNotMuted_());
    }
    /**
     * Checks if the microphone is muted. If it is, then we stop Dictation and
     * show a notification to the user. If the microphone isn't muted, then we
     * proceed to start speech recognition. Because this is async, this method
     * checks that startup state is still correct before proceeding.
     */
    verifyMicrophoneNotMuted_() {
        if (!this.active_) {
            this.stopDictation_(/*notify=*/ true);
            return;
        }
        // TODO(b:299677121): Determine if it's possible for no mics to be
        // available. If that scenario is possible, we may have to use
        // `chrome.audio.getDevices` and verify that there's at least one input
        // device.
        chrome.audio.getMute(StreamType.INPUT, (muted) => {
            if (muted) {
                this.stopDictation_(/*notify=*/ true);
                chrome.accessibilityPrivate.showToast(ToastType.DICTATION_MIC_MUTED);
                return;
            }
            this.maybeStartSpeechRecognition_();
        });
    }
    /**
     * Called when Dictation has set itself as the IME during start-up. Because
     * this is async, checks that startup state is still correct before starting
     * speech recognition.
     */
    maybeStartSpeechRecognition_() {
        if (this.active_) {
            // TODO(b/314203187): Determine if not null assertion is acceptable.
            chrome.speechRecognitionPrivate.start(this.speechRecognitionOptions_, (type) => this.onSpeechRecognitionStarted_(type));
        }
        else {
            // We are no longer starting up - perhaps a stop came
            // through during the async callbacks. Ensure cleanup
            // by calling stopDictation_().
            this.stopDictation_(/*notify=*/ true);
        }
    }
    /**
     * Stops Dictation and notifies the browser.
     * @param notify True if we should notify the browser that Dictation
     * stopped.
     */
    stopDictation_(notify) {
        if (!this.active_) {
            return;
        }
        this.active_ = false;
        // Stop speech recognition.
        chrome.speechRecognitionPrivate.stop({}, () => { });
        if (this.interimText_) {
            // TODO(b/314203187): Determine if not null assertion is acceptable.
            this.endTone_.play();
        }
        else {
            // TODO(b/314203187): Determine if not null assertion is acceptable.
            this.cancelTone_.play();
        }
        // Clear any timeouts.
        this.clearStopTimeout_();
        // TODO(b/314203187): Determine if not null assertion is acceptable.
        this.inputController_.commitText(this.interimText_);
        this.hideCommandsUI_();
        this.inputController_.disconnect();
        Dictation.removeAsInputMethod();
        // Notify the browser that Dictation turned off.
        if (notify) {
            chrome.accessibilityPrivate.toggleDictation();
        }
    }
    /**
     * Sets the timeout to stop Dictation.
     * @param reason Optional reason for why Dictation
     *     stopped automatically.
     */
    setStopTimeout_(durationMs, reason) {
        if (this.stopTimeoutId_ !== null) {
            clearTimeout(this.stopTimeoutId_);
        }
        this.stopTimeoutId_ = setTimeout(() => {
            this.stopDictation_(/*notify=*/ true);
            if (reason === Dictation.StopReason.NO_FOCUSED_IME) {
                chrome.accessibilityPrivate.showToast(ToastType.DICTATION_NO_FOCUSED_TEXT_FIELD);
            }
        }, durationMs);
    }
    /** Called when the Speech Recognition engine receives a recognition event. */
    async onSpeechRecognitionResult_(event) {
        if (!this.active_) {
            return;
        }
        const transcript = event.transcript;
        const isFinal = event.isFinal;
        this.setStopTimeout_(isFinal ? this.noSpeechTimeoutMs_ :
            Dictation.Timeouts.NO_NEW_SPEECH_MS);
        await this.processSpeechRecognitionResult_(transcript, isFinal);
    }
    /**
     * Processes a speech recognition result.
     * @param isFinal Whether this is a finalized transcript or an
     *     interim result.
     */
    async processSpeechRecognitionResult_(transcript, isFinal) {
        if (!isFinal) {
            this.showInterimText_(transcript);
            return;
        }
        // TODO(b/314203187): Determine if not null assertion is acceptable.
        let macro = await this.speechParser_.parse(transcript);
        MetricsUtils$1.recordMacroRecognized(macro);
        macro = this.handleRepeat_(macro);
        // Check if the macro can execute.
        // TODO(crbug.com/1264544): Deal with ambiguous results here.
        const checkContextResult = macro.checkContext();
        if (!checkContextResult.canTryAction &&
            this.isContextCheckingFeatureEnabled_) {
            this.showMacroExecutionFailed_(macro, transcript, checkContextResult.failedContext);
            return;
        }
        // Try to run the macro.
        const runMacroResult = macro.run();
        if (!runMacroResult.isSuccess) {
            this.showMacroExecutionFailed_(macro, transcript);
            return;
        }
        if (macro.getName() === MacroName.LIST_COMMANDS) {
            // ListCommandsMacro opens a new tab, thereby changing the cursor focus
            // and ending the Dictation session.
            return;
        }
        // Provide feedback to the user that the macro executed successfully.
        this.showMacroExecuted_(macro, transcript);
    }
    /**
     * Called when Speech Recognition starts up and begins listening. Passed as
     * a callback to speechRecognitionPrivate.start().
     * @param type The type of speech recognition used.
     */
    onSpeechRecognitionStarted_(type) {
        if (chrome.runtime.lastError) {
            // chrome.runtime.lastError will be set if the call to
            // speechRecognitionPrivate.start() caused an error. When this happens,
            // the speech recognition private API will turn the associated recognizer
            // off. To align with this, we should call `stopDictation_`.
            this.stopDictation_(/*notify=*/ true);
            return;
        }
        if (!this.active_) {
            return;
        }
        this.noSpeechTimeoutMs_ = type === SpeechRecognitionType.NETWORK ?
            Dictation.Timeouts.NO_SPEECH_NETWORK_MS :
            Dictation.Timeouts.NO_SPEECH_ONDEVICE_MS;
        this.setStopTimeout_(this.noSpeechTimeoutMs_);
        // TODO(b/314203187): Determine if not null assertion is acceptable.
        this.startTone_.play();
        this.clearInterimText_();
        // Record metrics.
        this.metricsUtils_ = new MetricsUtils$1(type, LocaleInfo.locale);
        this.metricsUtils_.recordSpeechRecognitionStarted();
        // TODO(b/314203187): Determine if not null assertion is acceptable.
        this.uiController_.setState(UIState.STANDBY, { context: HintContext.STANDBY });
        this.focusHandler_.refresh();
    }
    /**
     * Called when speech recognition stops or when speech recognition encounters
     * an error.
     */
    onSpeechRecognitionStopped_() {
        if (this.metricsUtils_ !== null) {
            this.metricsUtils_.recordSpeechRecognitionStopped();
        }
        this.metricsUtils_ = null;
        // Stop dictation if it wasn't already stopped.
        this.stopDictation_(/*notify=*/ true);
    }
    onSpeechRecognitionError_() {
        // TODO: Dictation does not surface speech recognition errors to the user.
        // Informing the user of errors, for example lack of network connection or a
        // missing microphone, would be a useful feature.
        this.stopDictation_(/*notify=*/ true);
    }
    updateFromPrefs_(prefs) {
        prefs.forEach(pref => {
            switch (pref.key) {
                case Dictation.DICTATION_LOCALE_PREF:
                    if (pref.value) {
                        const locale = pref.value;
                        // TODO(b/314203187): Determine if not null assertion is acceptable.
                        this.speechRecognitionOptions_.locale = locale;
                        LocaleInfo.locale = locale;
                        this.speechParser_.refresh();
                    }
                    break;
                case Dictation.SPOKEN_FEEDBACK_PREF:
                    if (pref.value) {
                        this.chromeVoxEnabled_ = true;
                    }
                    else {
                        this.chromeVoxEnabled_ = false;
                    }
                    // Use a longer hints timeout when ChromeVox is enabled.
                    // TODO(b/314203187): Determine if not null assertion is acceptable.
                    this.uiController_.setHintsTimeoutDuration(this.chromeVoxEnabled_);
                    break;
                default:
                    return;
            }
        });
    }
    /** Shows the interim result in the UI. */
    showInterimText_(text) {
        // TODO(crbug.com/40792919): Need to find a way to show interim text that is
        // only whitespace. Google Cloud Speech can return a newline character
        // although SODA does not seem to do that. The newline character looks wrong
        // here.
        this.interimText_ = text;
        // TODO(b/314203187): Determine if not null assertion is acceptable.
        this.uiController_.setState(UIState.RECOGNIZING_TEXT, { text });
    }
    /** Clears the interim result in the UI. */
    clearInterimText_() {
        this.interimText_ = '';
        // TODO(b/314203187): Determine if not null assertion is acceptable.
        this.uiController_.setState(UIState.STANDBY);
    }
    /**
     * Shows that a macro was executed in the UI by putting a checkmark next to
     * the transcript.
     */
    showMacroExecuted_(macro, transcript) {
        MetricsUtils$1.recordMacroSucceeded(macro);
        if (macro.getName() === MacroName.INPUT_TEXT_VIEW ||
            macro.getName() === MacroName.NEW_LINE) {
            this.clearInterimText_();
            // TODO(b/314203187): Determine if not null assertion is acceptable.
            this.uiController_.setState(UIState.STANDBY, { context: HintContext.TEXT_COMMITTED });
            return;
        }
        this.interimText_ = '';
        const context = macro.getName() === MacroName.SELECT_ALL_TEXT ?
            HintContext.TEXT_SELECTED :
            HintContext.MACRO_SUCCESS;
        // TODO(b/314203187): Determine if not null assertion is acceptable.
        this.uiController_.setState(UIState.MACRO_SUCCESS, { text: transcript, context });
    }
    /**
     * Shows a message in the UI that a command failed to execute.
     * TODO(crbug.com/40792919): Optionally use the MacroError to provide
     * additional context.
     * @param transcript The user's spoken transcript, shown so they
     *     understand the final speech recognized which might be helpful in
     *     understanding why the command failed.
     */
    showMacroExecutionFailed_(macro, transcript, failedContext) {
        MetricsUtils$1.recordMacroFailed(macro);
        this.interimText_ = '';
        let text = '';
        if (!failedContext) {
            text = chrome.i18n.getMessage('dictation_command_failed_generic', [transcript]);
        }
        else {
            const reason = Dictation.getFailedContextReason(failedContext);
            text = chrome.i18n.getMessage('dictation_command_failed_with_reason', [transcript, reason]);
        }
        // TODO(b/314203187): Determine if not null assertion is acceptable.
        this.uiController_.setState(UIState.MACRO_FAIL, {
            text,
            context: HintContext.STANDBY,
        });
    }
    /**
     * Hides the commands UI bubble.
     */
    /* eslint-disable-next-line @typescript-eslint/naming-convention */
    hideCommandsUI_() {
        this.interimText_ = '';
        // TODO(b/314203187): Determine if not null assertion is acceptable.
        this.uiController_.setState(UIState.HIDDEN);
    }
    clearStopTimeout_() {
        if (this.stopTimeoutId_ !== null) {
            clearTimeout(this.stopTimeoutId_);
            this.stopTimeoutId_ = null;
        }
    }
    /**
     * Removes AccessibilityCommon as an input method so it doesn't show up in
     * the shelf input method picker UI.
     */
    static removeAsInputMethod() {
        chrome.languageSettingsPrivate.removeInputMethod(InputControllerImpl.IME_ENGINE_ID);
    }
    /** Used to set the NO_FOCUSED_IME_MS timeout for testing purposes only. */
    setNoFocusedImeTimeoutForTesting(duration) {
        Dictation.Timeouts.NO_FOCUSED_IME_MS = duration;
    }
    handleRepeat_(macro) {
        if (macro.getName() !== MacroName.REPEAT) {
            // If this macro is not the RepeatMacro, save it and return the existing
            // macro.
            this.prevMacro_ = macro;
            return macro;
        }
        // Handle cases where `macro` is the RepeatMacro.
        if (!this.prevMacro_) {
            // If there is no previous macro, return the RepeatMacro.
            return macro;
        }
        // Otherwise, return the previous macro.
        return this.prevMacro_;
    }
    /** Disables Pumpkin for tests that use regex-based command parsing. */
    disablePumpkinForTesting() {
        // TODO(b/314203187): Determine if not null assertion is acceptable.
        this.speechParser_.disablePumpkinForTesting();
    }
    static getFailedContextReason(context) {
        switch (context) {
            case Context.INACTIVE_INPUT_CONTROLLER:
                return chrome.i18n.getMessage('dictation_context_error_reason_inactive_input_controller');
            case Context.EMPTY_EDITABLE:
                return chrome.i18n.getMessage('dictation_context_error_reason_empty_editable');
            case Context.NO_SELECTION:
                return chrome.i18n.getMessage('dictation_context_error_reason_no_selection');
            case Context.INVALID_INPUT:
                return chrome.i18n.getMessage('dictation_context_error_reason_invalid_input');
            case Context.NO_PREVIOUS_MACRO:
                return chrome.i18n.getMessage('dictation_context_error_reason_no_previous_macro');
        }
        throw new Error('Cannot get error message for unsupported context: ' + context);
    }
}
(function (Dictation) {
    /** Dictation locale pref. */
    Dictation.DICTATION_LOCALE_PREF = 'settings.a11y.dictation_locale';
    /** ChromeVox enabled pref. */
    Dictation.SPOKEN_FEEDBACK_PREF = 'settings.accessibility';
    /** Timeout durations. */
    Dictation.Timeouts = {
        NO_SPEECH_NETWORK_MS: 10 * 1000,
        NO_SPEECH_ONDEVICE_MS: 20 * 1000,
        NO_NEW_SPEECH_MS: 5 * 1000,
        NO_FOCUSED_IME_MS: 1000,
    };
    (function (StopReason) {
        StopReason["NO_FOCUSED_IME"] = "Dictation stopped automatically: No focused IME";
    })(Dictation.StopReason || (Dictation.StopReason = {}));
})(Dictation || (Dictation = {}));
TestImportManager.exportForTesting(Dictation);

// Copyright 2024 The Chromium Authors
// Use of this source code is governed by a BSD-style license that can be
// found in the LICENSE file.
/**
 * The facial gestures that are supported by FaceGaze. New values should also
 * be added to FacialGesturesToMediapipeGestures in
 * facegaze/gesture_detector.ts, FacialGesture definition in
 * accessibility_private.json, and ConvertFacialGestureType in
 * accessibility_extension_api_ash.cc. Please keep alphabetical.
 */
var FacialGesture;
(function (FacialGesture) {
    FacialGesture["BROW_INNER_UP"] = "browInnerUp";
    FacialGesture["BROWS_DOWN"] = "browsDown";
    FacialGesture["EYE_SQUINT_LEFT"] = "eyeSquintLeft";
    FacialGesture["EYE_SQUINT_RIGHT"] = "eyeSquintRight";
    FacialGesture["EYES_BLINK"] = "eyesBlink";
    FacialGesture["EYES_LOOK_DOWN"] = "eyesLookDown";
    FacialGesture["EYES_LOOK_LEFT"] = "eyesLookLeft";
    FacialGesture["EYES_LOOK_RIGHT"] = "eyesLookRight";
    FacialGesture["EYES_LOOK_UP"] = "eyesLookUp";
    FacialGesture["JAW_LEFT"] = "jawLeft";
    FacialGesture["JAW_OPEN"] = "jawOpen";
    FacialGesture["JAW_RIGHT"] = "jawRight";
    FacialGesture["MOUTH_FUNNEL"] = "mouthFunnel";
    FacialGesture["MOUTH_LEFT"] = "mouthLeft";
    FacialGesture["MOUTH_PUCKER"] = "mouthPucker";
    FacialGesture["MOUTH_RIGHT"] = "mouthRight";
    FacialGesture["MOUTH_SMILE"] = "mouthSmile";
    FacialGesture["MOUTH_UPPER_UP"] = "mouthUpperUp";
})(FacialGesture || (FacialGesture = {}));

// Copyright 2024 The Chromium Authors
// Use of this source code is governed by a BSD-style license that can be
// found in the LICENSE file.
/** Handles setting the text content of the FaceGaze bubble UI. */
class BubbleController {
    resetBubbleTimeoutId_;
    baseText_ = [];
    getState_;
    constructor(getState) {
        this.getState_ = getState;
    }
    updateBubble(text) {
        chrome.accessibilityPrivate.updateFaceGazeBubble(text, /*isWarning=*/ false);
        this.setResetBubbleTimeout_();
    }
    setResetBubbleTimeout_() {
        this.clearTimeout_();
        this.resetBubbleTimeoutId_ = setTimeout(() => this.resetBubble(), BubbleController.RESET_BUBBLE_TIMEOUT_MS);
    }
    clearTimeout_() {
        if (this.resetBubbleTimeoutId_ !== undefined) {
            clearTimeout(this.resetBubbleTimeoutId_);
            this.resetBubbleTimeoutId_ = undefined;
        }
    }
    resetBubble() {
        this.baseText_ = [];
        const { paused, scrollMode, longClick, dictation, heldMacros, precision, isFaceLandmarkerResultValid, isCameraMuted, } = this.getState_();
        if (isFaceLandmarkerResultValid !== undefined &&
            !isFaceLandmarkerResultValid) {
            this.baseText_.push(chrome.i18n.getMessage('facegaze_invalid_result'));
        }
        if (isCameraMuted) {
            this.baseText_.push(chrome.i18n.getMessage('facegaze_camera_muted'));
        }
        if (heldMacros) {
            heldMacros.forEach((displayText) => {
                this.baseText_.push(displayText);
            });
        }
        if (precision) {
            this.baseText_.push(chrome.i18n.getMessage('facegaze_state_precision_active', BubbleController.getDisplayTextForGesture_(precision)));
        }
        if (paused) {
            this.baseText_.push(chrome.i18n.getMessage('facegaze_state_paused', BubbleController.getDisplayTextForGesture_(paused)));
        }
        else if (scrollMode) {
            this.baseText_.push(chrome.i18n.getMessage('facegaze_state_scroll_active', BubbleController.getDisplayTextForGesture_(scrollMode)));
        }
        else if (longClick) {
            this.baseText_.push(chrome.i18n.getMessage('facegaze_state_long_click_active', BubbleController.getDisplayTextForGesture_(longClick)));
        }
        else if (dictation) {
            this.baseText_.push(chrome.i18n.getMessage('facegaze_state_dictation_active', BubbleController.getDisplayTextForGesture_(dictation)));
        }
        const isWarning = this.baseText_.length > 0;
        if (this.baseText_.length === 0) {
            // If there is no other text to show in the bubble, then show the
            // 'Face control active' message. This is gives users, who may be
            // unfamiliar with FaceGaze, an understanding of why their computer
            // can be controlled without touching the device.
            this.baseText_.push(chrome.i18n.getMessage('facegaze_active'));
        }
        chrome.accessibilityPrivate.updateFaceGazeBubble(this.baseText_.join(', '), isWarning);
    }
    static getDisplayText(gesture, macro) {
        if (macro.getName() === MacroName.TOGGLE_PRECISION_CLICK &&
            macro.getToggleDirection() === ToggleDirection.OFF) {
            return undefined;
        }
        return chrome.i18n.getMessage('facegaze_display_text', [
            BubbleController.getDisplayTextForMacro_(macro),
            BubbleController.getDisplayTextForGesture_(gesture),
        ]);
    }
    static getDisplayTextForMacro_(macro) {
        const macroName = macro.getName();
        switch (macroName) {
            case MacroName.CUSTOM_KEY_COMBINATION:
                return chrome.i18n.getMessage('facegaze_macro_text_custom_key_combo', this.getDisplayTextForKeyCombo_(macro));
            case MacroName.KEY_PRESS_DOWN:
                return chrome.i18n.getMessage('facegaze_macro_text_key_press_down');
            case MacroName.KEY_PRESS_LEFT:
                return chrome.i18n.getMessage('facegaze_macro_text_key_press_left');
            case MacroName.KEY_PRESS_MEDIA_PLAY_PAUSE:
                return chrome.i18n.getMessage('facegaze_macro_text_media_play_pause');
            case MacroName.KEY_PRESS_RIGHT:
                return chrome.i18n.getMessage('facegaze_macro_text_key_press_right');
            case MacroName.KEY_PRESS_SCREENSHOT:
                return chrome.i18n.getMessage('facegaze_macro_text_screenshot');
            case MacroName.KEY_PRESS_SPACE:
                return chrome.i18n.getMessage('facegaze_macro_text_key_press_space');
            case MacroName.KEY_PRESS_TOGGLE_OVERVIEW:
                return chrome.i18n.getMessage('facegaze_macro_text_toggle_overview');
            case MacroName.KEY_PRESS_UP:
                return chrome.i18n.getMessage('facegaze_macro_text_key_press_up');
            case MacroName.MOUSE_CLICK_LEFT:
                return chrome.i18n.getMessage('facegaze_macro_text_mouse_click_left');
            case MacroName.MOUSE_CLICK_LEFT_DOUBLE:
                return chrome.i18n.getMessage('facegaze_macro_text_mouse_click_left_double');
            case MacroName.MOUSE_CLICK_LEFT_TRIPLE:
                return chrome.i18n.getMessage('facegaze_macro_text_mouse_click_left_triple');
            case MacroName.MOUSE_CLICK_RIGHT:
                return chrome.i18n.getMessage('facegaze_macro_text_mouse_click_right');
            case MacroName.MOUSE_LONG_CLICK_LEFT:
                return macro.getToggleDirection() === ToggleDirection.ON ?
                    chrome.i18n.getMessage('facegaze_macro_text_mouse_long_click_left_on') :
                    chrome.i18n.getMessage('facegaze_macro_text_mouse_long_click_left_off');
            case MacroName.RESET_CURSOR:
                return chrome.i18n.getMessage('facegaze_macro_text_reset_cursor');
            case MacroName.TOGGLE_DICTATION:
                return macro.getToggleDirection() === ToggleDirection.ON ?
                    chrome.i18n.getMessage('facegaze_macro_text_toggle_dictation_on') :
                    chrome.i18n.getMessage('facegaze_macro_text_toggle_dictation_off');
            case MacroName.TOGGLE_FACEGAZE:
                return macro.getToggleDirection() === ToggleDirection.ON ?
                    chrome.i18n.getMessage('facegaze_macro_text_toggle_facegaze_on') :
                    chrome.i18n.getMessage('facegaze_macro_text_toggle_facegaze_off');
            case MacroName.TOGGLE_PRECISION_CLICK:
                // This method is only called on TOGGLE_PRECISION_CLICK if the toggle
                // direction is `ToggleDirection.ON`.
                return chrome.i18n.getMessage('facegaze_macro_text_toggle_precision_on');
            case MacroName.TOGGLE_SCROLL_MODE:
                return macro.getToggleDirection() === ToggleDirection.ON ?
                    chrome.i18n.getMessage('facegaze_macro_text_toggle_scroll_mode_on') :
                    chrome.i18n.getMessage('facegaze_macro_text_toggle_scroll_mode_off');
            case MacroName.TOGGLE_VIRTUAL_KEYBOARD:
                return chrome.i18n.getMessage('facegaze_macro_text_toggle_virtual_keyboard');
            default:
                console.error(`Display text requested for unsupported macro ${macroName}`);
                return '';
        }
    }
    static getDisplayTextForKeyCombo_(macro) {
        const keyCombo = macro.getKeyCombination();
        // Pre-defined key press macros, like for MEDIA_PLAY_PAUSE and SNAPSHOT,
        // should not request display text for their key combinations.
        if (!keyCombo || !keyCombo.keyDisplay) {
            console.error(`Key combo text requested for unsupported macro ${macro.getName()}`);
            return '';
        }
        const keys = [];
        if (keyCombo.modifiers?.ctrl) {
            keys.push(chrome.i18n.getMessage('facegaze_macro_text_key_ctrl'));
        }
        if (keyCombo.modifiers?.alt) {
            keys.push(chrome.i18n.getMessage('facegaze_macro_text_key_alt'));
        }
        if (keyCombo.modifiers?.shift) {
            keys.push(chrome.i18n.getMessage('facegaze_macro_text_key_shift'));
        }
        if (keyCombo.modifiers?.search) {
            keys.push(chrome.i18n.getMessage('facegaze_macro_text_key_search'));
        }
        keys.push(keyCombo.keyDisplay);
        switch (keys.length) {
            case 2:
                return chrome.i18n.getMessage('facegaze_macro_text_keyboard_combo_one_modifier', keys);
            case 3:
                return chrome.i18n.getMessage('facegaze_macro_text_keyboard_combo_two_modifiers', keys);
            case 4:
                return chrome.i18n.getMessage('facegaze_macro_text_keyboard_combo_three_modifiers', keys);
            case 5:
                return chrome.i18n.getMessage('facegaze_macro_text_keyboard_combo_four_modifiers', keys);
            default:
                // keyDisplay comes directly from the original KeyEvent and should be
                // preserved as-is since keys may appear differently on keyboards
                // depending on locale and layout.
                return keyCombo.keyDisplay;
        }
    }
    static getDisplayTextForGesture_(gesture) {
        switch (gesture) {
            case FacialGesture.BROW_INNER_UP:
                return chrome.i18n.getMessage('facegaze_gesture_text_brow_inner_up');
            case FacialGesture.BROWS_DOWN:
                return chrome.i18n.getMessage('facegaze_gesture_text_brows_down');
            case FacialGesture.EYE_SQUINT_LEFT:
                return chrome.i18n.getMessage('facegaze_gesture_text_eye_squint_left');
            case FacialGesture.EYE_SQUINT_RIGHT:
                return chrome.i18n.getMessage('facegaze_gesture_text_eye_squint_right');
            case FacialGesture.EYES_BLINK:
                return chrome.i18n.getMessage('facegaze_gesture_text_eyes_blink');
            case FacialGesture.EYES_LOOK_DOWN:
                return chrome.i18n.getMessage('facegaze_gesture_text_eyes_look_down');
            case FacialGesture.EYES_LOOK_LEFT:
                return chrome.i18n.getMessage('facegaze_gesture_text_eyes_look_left');
            case FacialGesture.EYES_LOOK_RIGHT:
                return chrome.i18n.getMessage('facegaze_gesture_text_eyes_look_right');
            case FacialGesture.EYES_LOOK_UP:
                return chrome.i18n.getMessage('facegaze_gesture_text_eyes_look_up');
            case FacialGesture.JAW_LEFT:
                return chrome.i18n.getMessage('facegaze_gesture_text_jaw_left');
            case FacialGesture.JAW_OPEN:
                return chrome.i18n.getMessage('facegaze_gesture_text_jaw_open');
            case FacialGesture.JAW_RIGHT:
                return chrome.i18n.getMessage('facegaze_gesture_text_jaw_right');
            case FacialGesture.MOUTH_FUNNEL:
                return chrome.i18n.getMessage('facegaze_gesture_text_mouth_funnel');
            case FacialGesture.MOUTH_LEFT:
                return chrome.i18n.getMessage('facegaze_gesture_text_mouth_left');
            case FacialGesture.MOUTH_PUCKER:
                return chrome.i18n.getMessage('facegaze_gesture_text_mouth_pucker');
            case FacialGesture.MOUTH_RIGHT:
                return chrome.i18n.getMessage('facegaze_gesture_text_mouth_right');
            case FacialGesture.MOUTH_SMILE:
                return chrome.i18n.getMessage('facegaze_gesture_text_mouth_smile');
            case FacialGesture.MOUTH_UPPER_UP:
                return chrome.i18n.getMessage('facegaze_gesture_text_mouth_upper_up');
            default:
                console.error('Display text requested for unsupported FacialGesture ' + gesture);
                return '';
        }
    }
}
(function (BubbleController) {
    BubbleController.RESET_BUBBLE_TIMEOUT_MS = 2500;
})(BubbleController || (BubbleController = {}));
TestImportManager.exportForTesting(BubbleController);

// Copyright 2024 The Chromium Authors
// Use of this source code is governed by a BSD-style license that can be
// found in the LICENSE file.
const SettingsPath = 'manageAccessibility/faceGaze';
/** Keep in sync with with values at ash_pref_names.h. */
var PrefNames;
(function (PrefNames) {
    PrefNames["ACCELERATOR_DIALOG_HAS_BEEN_ACCEPTED"] = "settings.a11y.face_gaze.accelerator_dialog_has_been_accepted";
    PrefNames["ACTIONS_ENABLED"] = "settings.a11y.face_gaze.actions_enabled";
    PrefNames["ACTIONS_ENABLED_SENTINEL"] = "settings.a11y.face_gaze.actions_enabled_sentinel";
    PrefNames["CURSOR_CONTROL_ENABLED"] = "settings.a11y.face_gaze.cursor_control_enabled";
    PrefNames["CURSOR_CONTROL_ENABLED_SENTINEL"] = "settings.a11y.face_gaze.cursor_control_enabled_sentinel";
    PrefNames["CURSOR_USE_ACCELERATION"] = "settings.a11y.face_gaze.cursor_use_acceleration";
    PrefNames["FACE_GAZE_ENABLED"] = "settings.a11y.face_gaze.enabled";
    PrefNames["FACE_GAZE_ENABLED_SENTINEL"] = "settings.a11y.face_gaze.enabled_sentinel";
    PrefNames["FACE_GAZE_ENABLED_SENTINEL_SHOW_DIALOG"] = "settings.a11y.face_gaze.enabled_sentinel_show_dialog";
    PrefNames["GESTURE_TO_CONFIDENCE"] = "settings.a11y.face_gaze.gestures_to_confidence";
    PrefNames["GESTURE_TO_KEY_COMBO"] = "settings.a11y.face_gaze.gestures_to_key_combos";
    PrefNames["GESTURE_TO_MACRO"] = "settings.a11y.face_gaze.gestures_to_macros";
    PrefNames["PRECISION_CLICK"] = "settings.a11y.face_gaze.precision_click";
    PrefNames["PRECISION_CLICK_SPEED_FACTOR"] = "settings.a11y.face_gaze.precision_click_speed_factor";
    PrefNames["SPD_DOWN"] = "settings.a11y.face_gaze.cursor_speed_down";
    PrefNames["SPD_LEFT"] = "settings.a11y.face_gaze.cursor_speed_left";
    PrefNames["SPD_RIGHT"] = "settings.a11y.face_gaze.cursor_speed_right";
    PrefNames["SPD_UP"] = "settings.a11y.face_gaze.cursor_speed_up";
    PrefNames["VELOCITY_THRESHOLD"] = "settings.a11y.face_gaze.velocity_threshold";
})(PrefNames || (PrefNames = {}));
TestImportManager.exportForTesting(['PrefNames', PrefNames]);

// Copyright 2024 The Chromium Authors
// Use of this source code is governed by a BSD-style license that can be
// found in the LICENSE file.
/** Facial gestures recognized by Mediapipe. */
var MediapipeFacialGesture;
(function (MediapipeFacialGesture) {
    MediapipeFacialGesture["BROW_DOWN_LEFT"] = "browDownLeft";
    MediapipeFacialGesture["BROW_DOWN_RIGHT"] = "browDownRight";
    MediapipeFacialGesture["BROW_INNER_UP"] = "browInnerUp";
    MediapipeFacialGesture["EYE_BLINK_LEFT"] = "eyeBlinkLeft";
    MediapipeFacialGesture["EYE_BLINK_RIGHT"] = "eyeBlinkRight";
    MediapipeFacialGesture["EYE_LOOK_DOWN_LEFT"] = "eyeLookDownLeft";
    MediapipeFacialGesture["EYE_LOOK_DOWN_RIGHT"] = "eyeLookDownRight";
    MediapipeFacialGesture["EYE_LOOK_IN_LEFT"] = "eyeLookInLeft";
    MediapipeFacialGesture["EYE_LOOK_IN_RIGHT"] = "eyeLookInRight";
    MediapipeFacialGesture["EYE_LOOK_OUT_LEFT"] = "eyeLookOutLeft";
    MediapipeFacialGesture["EYE_LOOK_OUT_RIGHT"] = "eyeLookOutRight";
    MediapipeFacialGesture["EYE_LOOK_UP_LEFT"] = "eyeLookUpLeft";
    MediapipeFacialGesture["EYE_LOOK_UP_RIGHT"] = "eyeLookUpRight";
    MediapipeFacialGesture["EYE_SQUINT_LEFT"] = "eyeSquintLeft";
    MediapipeFacialGesture["EYE_SQUINT_RIGHT"] = "eyeSquintRight";
    MediapipeFacialGesture["JAW_LEFT"] = "jawLeft";
    MediapipeFacialGesture["JAW_OPEN"] = "jawOpen";
    MediapipeFacialGesture["JAW_RIGHT"] = "jawRight";
    MediapipeFacialGesture["MOUTH_FUNNEL"] = "mouthFunnel";
    MediapipeFacialGesture["MOUTH_LEFT"] = "mouthLeft";
    MediapipeFacialGesture["MOUTH_PUCKER"] = "mouthPucker";
    MediapipeFacialGesture["MOUTH_RIGHT"] = "mouthRight";
    MediapipeFacialGesture["MOUTH_SMILE_LEFT"] = "mouthSmileLeft";
    MediapipeFacialGesture["MOUTH_SMILE_RIGHT"] = "mouthSmileRight";
    MediapipeFacialGesture["MOUTH_UPPER_UP_LEFT"] = "mouthUpperUpLeft";
    MediapipeFacialGesture["MOUTH_UPPER_UP_RIGHT"] = "mouthUpperUpRight";
})(MediapipeFacialGesture || (MediapipeFacialGesture = {}));
/**
 * Mapping of gestures supported by FaceGaze to mediapipe gestures; allows for
 * compound gestures.
 */
const FacialGesturesToMediapipeGestures = new Map([
    [FacialGesture.BROW_INNER_UP, [MediapipeFacialGesture.BROW_INNER_UP]],
    [
        FacialGesture.BROWS_DOWN,
        [
            MediapipeFacialGesture.BROW_DOWN_LEFT,
            MediapipeFacialGesture.BROW_DOWN_RIGHT,
        ],
    ],
    [FacialGesture.EYE_SQUINT_LEFT, [MediapipeFacialGesture.EYE_SQUINT_LEFT]],
    [FacialGesture.EYE_SQUINT_RIGHT, [MediapipeFacialGesture.EYE_SQUINT_RIGHT]],
    [
        FacialGesture.EYES_BLINK,
        [
            MediapipeFacialGesture.EYE_BLINK_LEFT,
            MediapipeFacialGesture.EYE_BLINK_RIGHT,
        ],
    ],
    [
        FacialGesture.EYES_LOOK_DOWN,
        [
            MediapipeFacialGesture.EYE_LOOK_DOWN_LEFT,
            MediapipeFacialGesture.EYE_LOOK_DOWN_RIGHT,
        ],
    ],
    [
        FacialGesture.EYES_LOOK_LEFT,
        [
            MediapipeFacialGesture.EYE_LOOK_OUT_LEFT,
            MediapipeFacialGesture.EYE_LOOK_IN_RIGHT,
        ],
    ],
    [
        FacialGesture.EYES_LOOK_RIGHT,
        [
            MediapipeFacialGesture.EYE_LOOK_OUT_RIGHT,
            MediapipeFacialGesture.EYE_LOOK_IN_LEFT,
        ],
    ],
    [
        FacialGesture.EYES_LOOK_UP,
        [
            MediapipeFacialGesture.EYE_LOOK_UP_LEFT,
            MediapipeFacialGesture.EYE_LOOK_UP_RIGHT,
        ],
    ],
    [FacialGesture.JAW_LEFT, [MediapipeFacialGesture.JAW_LEFT]],
    [FacialGesture.JAW_OPEN, [MediapipeFacialGesture.JAW_OPEN]],
    [FacialGesture.JAW_RIGHT, [MediapipeFacialGesture.JAW_RIGHT]],
    [FacialGesture.MOUTH_FUNNEL, [MediapipeFacialGesture.MOUTH_FUNNEL]],
    [FacialGesture.MOUTH_LEFT, [MediapipeFacialGesture.MOUTH_LEFT]],
    [FacialGesture.MOUTH_PUCKER, [MediapipeFacialGesture.MOUTH_PUCKER]],
    [FacialGesture.MOUTH_RIGHT, [MediapipeFacialGesture.MOUTH_RIGHT]],
    [
        FacialGesture.MOUTH_SMILE,
        [
            MediapipeFacialGesture.MOUTH_SMILE_LEFT,
            MediapipeFacialGesture.MOUTH_SMILE_RIGHT,
        ],
    ],
    [
        FacialGesture.MOUTH_UPPER_UP,
        [
            MediapipeFacialGesture.MOUTH_UPPER_UP_LEFT,
            MediapipeFacialGesture.MOUTH_UPPER_UP_RIGHT,
        ],
    ],
]);
/**
 * Mapping of gestures supported by FaceGaze to mediapipe gestures that should
 * not be present in order to confidently trigger the supported gesture, so as
 * to reduce triggering gestures with involuntary facial movement like blinking.
 */
const FacialGesturesToExcludedMediapipeGestures = new Map([
    [
        FacialGesture.EYE_SQUINT_LEFT,
        [
            MediapipeFacialGesture.EYE_BLINK_RIGHT,
            MediapipeFacialGesture.EYE_SQUINT_RIGHT,
        ],
    ],
    [
        FacialGesture.EYE_SQUINT_RIGHT,
        [
            MediapipeFacialGesture.EYE_BLINK_LEFT,
            MediapipeFacialGesture.EYE_SQUINT_LEFT,
        ],
    ],
]);
/**
 * The confidence level at which a detected gesture B should disqualify a
 * different gesture A from being triggered if A relies on B not being
 * present. For example, if EYE_SQUINT_LEFT is detected but EYE_SQUINT_RIGHT
 * is also detected at confidence > EXCLUDED_GESTURE_THRESHOLD, then the
 * gesture was likely involuntary blinking and EYE_SQUINT_LEFT should not be
 * triggered. A confidence level of 0.5 seems to be the approximate confidence
 * that occurs for most involuntary blinking.
 */
const EXCLUDED_GESTURE_THRESHOLD = 0.5;
class GestureDetector {
    static mediapipeFacialGestureSet_ = new Set(Object.values(MediapipeFacialGesture));
    static toggleSendGestureDetectionInfo(enabled) {
        this.shouldSendGestureDetectionInfo_ = enabled;
    }
    /**
     * Computes which FacialGestures were detected. Note that this will only
     * return a gesture if it is specified in `confidenceMap`, as this function
     * uses the confidence to decide whether or not to include the gesture in
     * the final result.
     */
    static detect(result, confidenceMap) {
        // Look through the blendshapes to find the gestures from mediapipe that we
        // care about.
        const recognizedGestures = new Map();
        for (const classification of result.faceBlendshapes) {
            for (const category of classification.categories) {
                const gesture = category.categoryName;
                if (GestureDetector.mediapipeFacialGestureSet_.has(gesture)) {
                    recognizedGestures.set(gesture, category.score);
                }
            }
        }
        if (recognizedGestures.size === 0) {
            return [];
        }
        // Look through the facial gestures to see which were detected by mediapipe.
        const gestures = [];
        const gestureInfoForSettings = [];
        for (const [faceGazeGesture, mediapipeGestures] of FacialGesturesToMediapipeGestures) {
            const confidence = confidenceMap.get(faceGazeGesture);
            if (!this.shouldSendGestureDetectionInfo_ && !confidence) {
                // Settings is not requesting gesture detection information and
                // this gesture is not currently used by FaceGaze.
                continue;
            }
            // Score will be the minimum from among the compound gestures.
            let score = -1;
            let hasCompoundGesture = true;
            for (const mediapipeGesture of mediapipeGestures) {
                if (!recognizedGestures.has(mediapipeGesture)) {
                    hasCompoundGesture = false;
                    break;
                }
                // The score of a compound gesture is the maximum of its component
                // parts. This is max instead of min in case people have uneven
                // facial strength or dexterity.
                score = Math.max(score, recognizedGestures.get(mediapipeGesture));
            }
            if (!hasCompoundGesture) {
                continue;
            }
            let hasExcludedGesture = false;
            const excludedGestures = FacialGesturesToExcludedMediapipeGestures.get(faceGazeGesture);
            // Check for gestures that should not be present in order to confidently
            // register this gesture.
            if (excludedGestures) {
                for (const excludedGesture of excludedGestures) {
                    if (recognizedGestures.has(excludedGesture) &&
                        recognizedGestures.get(excludedGesture) >
                            EXCLUDED_GESTURE_THRESHOLD) {
                        hasExcludedGesture = true;
                    }
                }
            }
            if (hasExcludedGesture) {
                continue;
            }
            // For gestures detected with a confidence value over a threshold value of
            // 1, add the gesture and confidence value to the array of information
            // that will be sent to settings.
            if (this.shouldSendGestureDetectionInfo_ && score >= 0.01) {
                gestureInfoForSettings.push({ gesture: faceGazeGesture, confidence: score * 100 });
            }
            if (confidence && score < confidence) {
                continue;
            }
            gestures.push(faceGazeGesture);
        }
        if (this.shouldSendGestureDetectionInfo_ &&
            gestureInfoForSettings.length > 0) {
            chrome.accessibilityPrivate.sendGestureInfoToSettings(gestureInfoForSettings);
        }
        return gestures;
    }
}
TestImportManager.exportForTesting(GestureDetector, ['FacialGesture', FacialGesture], ['MediapipeFacialGesture', MediapipeFacialGesture], ['FacialGesturesToMediapipeGestures', FacialGesturesToMediapipeGestures]);

// Copyright 2024 The Chromium Authors
// Use of this source code is governed by a BSD-style license that can be
// found in the LICENSE file.
/** Minimum time duration for a gesture to be recognized. */
const DEFAULT_MINIMUM_DURATION_MS = 150;
/** Minimum repeat rate of a gesture. */
const DEFAULT_REPEAT_DELAY_MS = 1000;
class GestureTimer {
    repeatDelayMs_ = DEFAULT_REPEAT_DELAY_MS;
    minDurationMs_ = DEFAULT_MINIMUM_DURATION_MS;
    gestureStart_ = new Map();
    gestureLastRecognized_ = new Map();
    useGestureDuration_ = true;
    /**
     * Mark that the gesture has been detected at the given time stamp. If the
     * gesture has already been marked as started then this timestamp is ignored.
     */
    mark(gesture, timestamp) {
        if (!this.gestureStart_.has(gesture)) {
            this.gestureStart_.set(gesture, timestamp);
        }
    }
    /** Get the timestamp of the last time this gesture was recognized. */
    getLastRecognized(gesture) {
        return this.gestureLastRecognized_.get(gesture);
    }
    /** Set the timestamp of the last time this gesture was recognized. */
    setLastRecognized(gesture, timestamp) {
        this.gestureLastRecognized_.set(gesture, timestamp);
    }
    /** Reset the timer for the given gesture. */
    resetTimer(gesture) {
        this.gestureStart_.delete(gesture);
    }
    /** Reset the timer for all gestures. */
    resetAll() {
        this.gestureStart_.clear();
        this.gestureLastRecognized_.clear();
    }
    /**
     * Return true if the minimum repeat delay has elapsed since the given gesture
     * was last recognized.
     */
    isRepeatDelayValid(gesture, timestamp) {
        const lastRecognized = this.gestureLastRecognized_.get(gesture);
        if (!lastRecognized) {
            return true;
        }
        return timestamp.getTime() - lastRecognized.getTime() >=
            this.repeatDelayMs_;
    }
    /**
     * Return true if this gesture has been held for a valid duration in relation
     * to the given timestamp.
     */
    isDurationValid(gesture, timestamp) {
        if (!this.useGestureDuration_) {
            return true;
        }
        const startTime = this.gestureStart_.get(gesture);
        // If there is no start timestamp, then the duration is not valid.
        if (!startTime) {
            return false;
        }
        return timestamp.getTime() - startTime.getTime() > this.minDurationMs_;
    }
    // For testing purposes, we want to allow gestures to be recognized instantly
    // without requiring a valid duration.
    setGestureDurationForTesting(useDuration) {
        this.useGestureDuration_ = useDuration;
    }
}

// Copyright 2024 The Chromium Authors
// Use of this source code is governed by a BSD-style license that can be
// found in the LICENSE file.
/** Class that implements a macro to toggle a long click action. */
class MouseLongClickMacro extends Macro {
    mouseController_;
    constructor(mouseController) {
        super(MacroName.MOUSE_LONG_CLICK_LEFT);
        this.mouseController_ = mouseController;
    }
    isToggle() {
        return true;
    }
    getToggleDirection() {
        return this.mouseController_.isLongClickActive() ? ToggleDirection.OFF :
            ToggleDirection.ON;
    }
    checkContext() {
        if (!this.mouseController_.mouseLocation()) {
            return this.createFailureCheckContextResult_(MacroError.BAD_CONTEXT);
        }
        return this.createSuccessCheckContextResult_();
    }
    run() {
        if (!this.mouseController_.mouseLocation()) {
            return this.createRunMacroResult_(/*isSuccess=*/ false);
        }
        this.mouseController_.toggleLongClick();
        return this.createRunMacroResult_(/*isSuccess=*/ true);
    }
}

// Copyright 2024 The Chromium Authors
// Use of this source code is governed by a BSD-style license that can be
// found in the LICENSE file.
/** Class that implements a macro to toggle scroll mode. */
class MouseScrollMacro extends Macro {
    mouseController_;
    constructor(mouseController) {
        super(MacroName.TOGGLE_SCROLL_MODE);
        this.mouseController_ = mouseController;
    }
    isToggle() {
        return true;
    }
    getToggleDirection() {
        return this.mouseController_.isScrollModeActive() ? ToggleDirection.OFF :
            ToggleDirection.ON;
    }
    run() {
        this.mouseController_.toggleScrollMode();
        return this.createRunMacroResult_(/*isSuccess=*/ true);
    }
}

// Copyright 2024 The Chromium Authors
// Use of this source code is governed by a BSD-style license that can be
// found in the LICENSE file.
/**
 * Class that implements a macro to reset the cursor position.
 */
class ResetCursorMacro extends Macro {
    mouseController_;
    constructor(mouseController) {
        super(MacroName.RESET_CURSOR);
        this.mouseController_ = mouseController;
    }
    run() {
        this.mouseController_.resetLocation();
        return this.createRunMacroResult_(/*isSuccess=*/ true);
    }
}

// Copyright 2024 The Chromium Authors
// Use of this source code is governed by a BSD-style license that can be
// found in the LICENSE file.
var RoleType$1 = chrome.automation.RoleType;
var StateType = chrome.automation.StateType;
/** The default confidence threshold for facial gestures. */
const DEFAULT_CONFIDENCE_THRESHOLD = 0.5;
/** Handles converting facial gestures to Macros. */
class GestureHandler {
    // References to core classes.
    bubbleController_;
    gestureTimer_;
    mouseController_;
    // Other variables, such as state and callbacks.
    gesturesToKeyCombos_ = new Map();
    gestureToMacroName_ = new Map();
    gestureToConfidence_ = new Map();
    isDictationActive_;
    toggleInfoListener_;
    macrosToCompleteLater_ = new Map();
    paused_ = false;
    // Tracks the last gesture used to activate precision click. We need to track
    // this because TOGGLE_PRECISION_MODE isn't stored in `gestureToMacroName_`
    // and there are multiple ways to activate a precision click.
    lastPrecisionGesture_ = undefined;
    // The most recently detected gestures. We track this to know when a gesture
    // has ended.
    previousGestures_ = [];
    constructor(mouseController, bubbleController, isDictationActive) {
        this.mouseController_ = mouseController;
        this.bubbleController_ = bubbleController;
        this.isDictationActive_ = isDictationActive;
        this.toggleInfoListener_ = enabled => GestureDetector.toggleSendGestureDetectionInfo(enabled);
        this.gestureTimer_ = new GestureTimer();
    }
    start() {
        this.paused_ = false;
        chrome.accessibilityPrivate.onToggleGestureInfoForSettings.addListener(this.toggleInfoListener_);
    }
    stop() {
        this.paused_ = false;
        chrome.accessibilityPrivate.onToggleGestureInfoForSettings.removeListener(this.toggleInfoListener_);
        this.previousGestures_ = [];
        this.gestureTimer_.resetAll();
        // Executing these macros clears their state, so that we aren't left in a
        // mouse down or key down state.
        this.macrosToCompleteLater_.forEach((entry) => {
            entry.macro.run();
        });
        this.macrosToCompleteLater_.clear();
        this.lastPrecisionGesture_ = undefined;
    }
    isPaused() {
        return this.paused_;
    }
    getHeldMacroDisplayStrings() {
        const displayStrings = [];
        for (const entry of this.macrosToCompleteLater_.values()) {
            displayStrings.push(entry.displayText);
        }
        return displayStrings;
    }
    detectMacros(result) {
        const gestures = GestureDetector.detect(result, this.gestureToConfidence_);
        const { macros, displayText } = this.gesturesToMacros_(gestures);
        const macrosOnGestureEnd = this.popMacrosOnGestureEnd(gestures, this.previousGestures_);
        // Because these macros are finished when the gesture is released rather
        // than when the gesture is triggered for the second time, the bubble needs
        // to be manually reset here to ensure the corresponding macro description
        // is cleared rather than waiting for another FaceLandmarkerResult.
        if (macrosOnGestureEnd.length > 0) {
            this.bubbleController_.resetBubble();
        }
        macros.push(...macrosOnGestureEnd);
        this.previousGestures_ = gestures;
        return { macros, displayText };
    }
    togglePaused(gesture) {
        const newPaused = !this.paused_;
        const lastToggledTime = this.gestureTimer_.getLastRecognized(gesture);
        // Run start/stop before assigning the new pause value and gesture last
        // recognized time, since start/stop will modify these values.
        newPaused ? this.stop() : this.start();
        if (lastToggledTime) {
            this.gestureTimer_.setLastRecognized(gesture, lastToggledTime);
        }
        this.paused_ = newPaused;
    }
    getGestureForPause() {
        return this.getGestureFor_(MacroName.TOGGLE_FACEGAZE);
    }
    getGestureForScroll() {
        return this.getGestureFor_(MacroName.TOGGLE_SCROLL_MODE);
    }
    getGestureForLongClick() {
        return this.getGestureFor_(MacroName.MOUSE_LONG_CLICK_LEFT);
    }
    getGestureForDictation() {
        return this.getGestureFor_(MacroName.TOGGLE_DICTATION);
    }
    getGestureForPrecision() {
        return this.lastPrecisionGesture_;
    }
    getGestureFor_(macroName) {
        // Return the first found gesture assigned to the given macro.
        for (const [gesture, macro] of this.gestureToMacroName_.entries()) {
            if (macro === macroName) {
                return gesture;
            }
        }
        return undefined;
    }
    gesturesToMacros_(gestures) {
        const macroNames = new Map();
        for (const gesture of gestures) {
            const currentTime = new Date();
            // Check if this duration is valid before marking this gesture, otherwise
            // the first gesture frame will instantly trigger the gesture.
            const isDurationValid = this.gestureTimer_.isDurationValid(gesture, currentTime);
            this.gestureTimer_.mark(gesture, currentTime);
            if (!isDurationValid) {
                continue;
            }
            if (!this.gestureTimer_.isRepeatDelayValid(gesture, currentTime)) {
                continue;
            }
            // Do not respond if we are still waiting to complete this macro later as
            // it shouldn't be repeated until completed.
            if (this.macrosToCompleteLater_.has(gesture)) {
                continue;
            }
            this.gestureTimer_.setLastRecognized(gesture, currentTime);
            const name = this.gestureToMacroName_.get(gesture);
            if (name) {
                macroNames.set(name, gesture);
            }
        }
        // Construct display text.
        const displayStrings = [];
        // Construct macros from all the macro names.
        const result = [];
        for (const [macroName, gesture] of macroNames) {
            const initialMacro = this.macroFromName_(macroName, gesture);
            if (!initialMacro) {
                continue;
            }
            const macros = this.handlePrecisionClick_(initialMacro, gesture);
            for (const macro of macros) {
                result.push(macro);
                const displayText = BubbleController.getDisplayText(gesture, macro);
                if (displayText) {
                    displayStrings.push(displayText);
                }
                if (macro.triggersAtActionStartAndEnd()) {
                    // Cache this macro to be run a second time later,
                    // e.g. for key release.
                    this.macrosToCompleteLater_.set(gesture, { macro: macro, displayText: displayText });
                }
            }
        }
        const displayText = displayStrings.join(', ');
        return { macros: result, displayText };
    }
    /**
     * Gets the cached macros that are run again when a gesture ends. For example,
     * for a key press macro, the key press starts when the gesture is first
     * detected and the macro is run a second time when the gesture is no longer
     * detected, thus the key press will be held as long as the gesture is still
     * detected.
     */
    popMacrosOnGestureEnd(gestures, previousGestures) {
        const macrosForLater = [];
        previousGestures.forEach(previousGesture => {
            if (!gestures.includes(previousGesture)) {
                // Reset timer for gesture when it is stopped.
                this.gestureTimer_.resetTimer(previousGesture);
                // The gesture has stopped being recognized. Run the second half of this
                // macro, and stop saving it.
                const entry = this.macrosToCompleteLater_.get(previousGesture);
                if (!entry || !entry.macro) {
                    return;
                }
                macrosForLater.push(entry.macro);
                this.macrosToCompleteLater_.delete(previousGesture);
            }
        });
        return macrosForLater;
    }
    macroFromName_(name, gesture) {
        if (!this.isMacroAllowed_(name)) {
            return;
        }
        switch (name) {
            case MacroName.TOGGLE_DICTATION:
                return new ToggleDictationMacro(
                /*dictationActive=*/ this.isDictationActive_());
            case MacroName.MOUSE_CLICK_LEFT:
                return new MouseClickMacro(this.mouseController_.mouseLocation());
            case MacroName.MOUSE_CLICK_RIGHT:
                return new MouseClickMacro(this.mouseController_.mouseLocation(), /*leftClick=*/ false);
            case MacroName.MOUSE_LONG_CLICK_LEFT:
                return new MouseLongClickMacro(this.mouseController_);
            case MacroName.MOUSE_CLICK_LEFT_DOUBLE:
                return new MouseClickLeftDoubleMacro(this.mouseController_.mouseLocation());
            case MacroName.MOUSE_CLICK_LEFT_TRIPLE:
                return new MouseClickLeftTripleMacro(this.mouseController_.mouseLocation());
            case MacroName.RESET_CURSOR:
                return new ResetCursorMacro(this.mouseController_);
            case MacroName.KEY_PRESS_SPACE:
                return new KeyPressMacro(name, { key: KeyCode.SPACE });
            case MacroName.KEY_PRESS_DOWN:
                return new KeyPressMacro(name, { key: KeyCode.DOWN });
            case MacroName.KEY_PRESS_LEFT:
                return new KeyPressMacro(name, { key: KeyCode.LEFT });
            case MacroName.KEY_PRESS_RIGHT:
                return new KeyPressMacro(name, { key: KeyCode.RIGHT });
            case MacroName.KEY_PRESS_UP:
                return new KeyPressMacro(name, { key: KeyCode.UP });
            case MacroName.KEY_PRESS_TOGGLE_OVERVIEW:
                // The MEDIA_LAUNCH_APP1 key is bound to the kToggleOverview accelerator
                // action in accelerators.cc.
                return new KeyPressMacro(name, { key: KeyCode.MEDIA_LAUNCH_APP1 });
            case MacroName.KEY_PRESS_MEDIA_PLAY_PAUSE:
                return new KeyPressMacro(name, { key: KeyCode.MEDIA_PLAY_PAUSE });
            case MacroName.KEY_PRESS_SCREENSHOT:
                return new KeyPressMacro(name, { key: KeyCode.SNAPSHOT });
            case MacroName.OPEN_FACEGAZE_SETTINGS:
                return new CustomCallbackMacro(MacroName.OPEN_FACEGAZE_SETTINGS, () => {
                    chrome.accessibilityPrivate.openSettingsSubpage(SettingsPath);
                });
            case MacroName.TOGGLE_FACEGAZE:
                return new CustomCallbackMacro(MacroName.TOGGLE_FACEGAZE, () => {
                    this.mouseController_.togglePaused();
                    this.togglePaused(gesture);
                }, 
                /*toggleDirection=*/ this.paused_ ? ToggleDirection.ON :
                    ToggleDirection.OFF);
            case MacroName.TOGGLE_SCROLL_MODE:
                return new MouseScrollMacro(this.mouseController_);
            case MacroName.TOGGLE_VIRTUAL_KEYBOARD:
                return new CustomCallbackMacro(MacroName.TOGGLE_VIRTUAL_KEYBOARD, async () => {
                    // TODO(b/355662617): Unify with SwitchAccessPredicate.
                    const isVisible = (node) => {
                        return Boolean(!node.state[StateType.OFFSCREEN] && node.location &&
                            node.location.top >= 0 && node.location.left >= 0 &&
                            !node.state[StateType.INVISIBLE]);
                    };
                    const desktop = await AsyncUtil.getDesktop();
                    const keyboard = desktop.find({ role: RoleType$1.KEYBOARD });
                    const currentlyVisible = Boolean(keyboard && isVisible(keyboard) &&
                        keyboard.find({ role: RoleType$1.ROOT_WEB_AREA }));
                    // Toggle the visibility of the virtual keyboard.
                    chrome.accessibilityPrivate.setVirtualKeyboardVisible(!currentlyVisible);
                });
            case MacroName.CUSTOM_KEY_COMBINATION:
                const keyCombination = this.gesturesToKeyCombos_.get(gesture);
                if (!keyCombination) {
                    throw new Error(`Expected a custom key combination for gesture: ${gesture}`);
                }
                return new KeyPressMacro(name, keyCombination);
            default:
                return;
        }
    }
    isMacroAllowed_(name) {
        if (this.isDictationActive_() && name !== MacroName.TOGGLE_DICTATION) {
            return false;
        }
        if (this.mouseController_.isScrollModeActive() &&
            name !== MacroName.TOGGLE_SCROLL_MODE) {
            return false;
        }
        if (this.paused_ && name !== MacroName.TOGGLE_FACEGAZE) {
            return false;
        }
        if (this.mouseController_.isLongClickActive() &&
            name !== MacroName.MOUSE_LONG_CLICK_LEFT) {
            return false;
        }
        return true;
    }
    gesturesToMacrosChanged(bindings) {
        if (!bindings) {
            return;
        }
        // Update the whole map from this preference.
        this.gestureToMacroName_.clear();
        let hasScrollModeAction = false;
        let hasLongClickAction = false;
        for (const [gesture, assignedMacro] of Object.entries(bindings)) {
            if (assignedMacro === MacroName.UNSPECIFIED) {
                continue;
            }
            if (assignedMacro === MacroName.TOGGLE_SCROLL_MODE) {
                hasScrollModeAction = true;
            }
            if (assignedMacro === MacroName.MOUSE_LONG_CLICK_LEFT) {
                hasLongClickAction = true;
            }
            this.gestureToMacroName_.set(gesture, Number(assignedMacro));
            // Ensure the confidence for this gesture is set to the default,
            // if it wasn't set yet. This might happen if the user hasn't
            // opened the settings subpage yet.
            if (!this.gestureToConfidence_.has(gesture)) {
                this.gestureToConfidence_.set(gesture, DEFAULT_CONFIDENCE_THRESHOLD);
            }
        }
        // If a "toggle" action is removed while the relevant action
        // is active, then we should toggle out of the action. Otherwise,
        // the user will be stuck in the action with no way to exit.
        if (this.mouseController_.isScrollModeActive() && !hasScrollModeAction) {
            this.mouseController_.toggleScrollMode();
        }
        if (this.mouseController_.isLongClickActive() && !hasLongClickAction) {
            this.mouseController_.toggleLongClick();
        }
    }
    gesturesToConfidencesChanged(confidences) {
        if (!confidences) {
            return;
        }
        for (const [gesture, confidence] of Object.entries(confidences)) {
            this.gestureToConfidence_.set(gesture, Number(confidence) / 100.);
        }
    }
    gesturesToKeyCombosChanged(keyCombos) {
        if (!keyCombos) {
            return;
        }
        // Update the whole map from this preference.
        this.gesturesToKeyCombos_.clear();
        for (const [gesture, keyCombinationAsString] of Object.entries(keyCombos)) {
            const keyCombination = JSON.parse(keyCombinationAsString);
            this.gesturesToKeyCombos_.set(gesture, keyCombination);
        }
    }
    // Handles precision click. If precision click is enabled, three things can
    // happen:
    // 1. If precision mode is inactive and the original macro is anything other
    // than a click type, then this should return the original macro.
    // 2. If precision mode is inactive and the original macro is a click type,
    // then this should return a TOGGLE_PRECISION_CLICK so that precision click is
    // started.
    // 3. If precision mode is active, then this should return both the original
    // macro and a TOGGLE_PRECISION_CLICK macro so that the macro is performed and
    // precision click is ended.
    handlePrecisionClick_(macro, gesture) {
        if (!this.mouseController_.usePrecision()) {
            return [macro];
        }
        // This method excludes MOUSE_CLICK_LEFT_LONG because that is a two-step
        // click, whereas all other clicks are instantaneous.
        const isClickMacro = () => {
            const name = macro.getName();
            if (name === MacroName.MOUSE_CLICK_LEFT ||
                name === MacroName.MOUSE_CLICK_LEFT_DOUBLE ||
                name === MacroName.MOUSE_CLICK_LEFT_TRIPLE ||
                name === MacroName.MOUSE_CLICK_RIGHT) {
                return true;
            }
            return false;
        };
        const result = [];
        if (!this.mouseController_.isPrecisionActive()) {
            if (!isClickMacro()) {
                result.push(macro);
            }
            else {
                // If we're toggling precision click on, we need to save the gesture
                // that was used so that we can display it in the bubble UI.
                this.lastPrecisionGesture_ = gesture;
                result.push(new CustomCallbackMacro(MacroName.TOGGLE_PRECISION_CLICK, () => this.mouseController_.togglePrecision(), 
                /*toggleDirection=*/ ToggleDirection.ON));
            }
        }
        else {
            result.push(macro, new CustomCallbackMacro(MacroName.TOGGLE_PRECISION_CLICK, () => this.mouseController_.togglePrecision(), 
            /*toggleDirection=*/ ToggleDirection.OFF));
        }
        return result;
    }
}
TestImportManager.exportForTesting(GestureHandler);

// Copyright 2024 The Chromium Authors
// Use of this source code is governed by a BSD-style license that can be
// found in the LICENSE file.
/**
 * The metric used to record the average FaceLandmarker performance time on a
 * single video frame (in milliseconds).
 */
const FACELANDMARKER_PERFORMANCE_METRIC = 'Accessibility.FaceGaze.AverageFaceLandmarkerLatency';
/** A class used to record metrics for FaceGaze. */
class MetricsUtils {
    latencies_ = [];
    addFaceLandmarkerResultLatency(val) {
        this.latencies_.push(val);
        if (this.latencies_.length >= 100) {
            this.writeHistogram_();
            this.latencies_ = [];
        }
    }
    writeHistogram_() {
        let sum = 0;
        for (const val of this.latencies_) {
            sum += val;
        }
        const average = Math.ceil(sum / this.latencies_.length);
        chrome.metricsPrivate.recordMediumTime(FACELANDMARKER_PERFORMANCE_METRIC, average);
    }
}
TestImportManager.exportForTesting(MetricsUtils);

// Copyright 2024 The Chromium Authors
// Use of this source code is governed by a BSD-style license that can be
// found in the LICENSE file.
var ScrollDirection = chrome.accessibilityPrivate.ScrollDirection;
/**
 * The amount of cushion provided at the top and bottom of the screen during
 * scroll mode.
 */
const VERTICAL_CUSHION_FACTOR = 0.1;
/** Handles all scroll interaction. */
class ScrollModeController {
    active_ = false;
    scrollLocation_;
    lastScrollTime_ = 0;
    screenBounds_;
    originalCursorControlPref_;
    active() {
        return this.active_;
    }
    toggle(mouseLocation, screenBounds) {
        if (!mouseLocation || !screenBounds) {
            return;
        }
        this.active_ ? this.stop_() : this.start_(mouseLocation, screenBounds);
    }
    updateScrollLocation(mouseLocation) {
        if (!mouseLocation) {
            return;
        }
        this.scrollLocation_ = mouseLocation;
    }
    async start_(mouseLocation, screenBounds) {
        this.active_ = true;
        this.scrollLocation_ = mouseLocation;
        this.screenBounds_ = screenBounds;
        chrome.settingsPrivate.getPref(PrefNames.CURSOR_CONTROL_ENABLED, pref => {
            // Save the original cursor control setting and ensure cursor control
            // is enabled.
            this.originalCursorControlPref_ = pref.value;
            chrome.settingsPrivate.setPref(PrefNames.CURSOR_CONTROL_ENABLED, true);
        });
    }
    stop_() {
        this.active_ = false;
        this.scrollLocation_ = undefined;
        this.screenBounds_ = undefined;
        this.lastScrollTime_ = 0;
        // Set cursor control back to its original setting.
        chrome.settingsPrivate.setPref(PrefNames.CURSOR_CONTROL_ENABLED, Boolean(this.originalCursorControlPref_));
        this.originalCursorControlPref_ = undefined;
    }
    /** Scrolls based on the new mouse location. */
    scroll(mouseLocation) {
        if (!this.active_ || !this.scrollLocation_ || !this.screenBounds_ ||
            (new Date().getTime() - this.lastScrollTime_ <
                ScrollModeController.RATE_LIMIT)) {
            return;
        }
        // To scroll, the user must move the mouse to one of the four edges of the
        // screen. We prioritize up and down scrolling because it's more common to
        // scroll in these directions. In the up and down directions, we provide
        // a cushion so that the mouse doesn't have to be exactly at the top or
        // bottom of the screen. This makes it easier to scroll up/down.
        const verticalCushion = this.screenBounds_.height * VERTICAL_CUSHION_FACTOR;
        let direction;
        if (mouseLocation.y <= this.screenBounds_.top + verticalCushion) {
            direction = ScrollDirection.UP;
        }
        else if (mouseLocation.y >=
            this.screenBounds_.height + this.screenBounds_.top - verticalCushion) {
            direction = ScrollDirection.DOWN;
        }
        else if (mouseLocation.x <= this.screenBounds_.left) {
            direction = ScrollDirection.LEFT;
        }
        else if (mouseLocation.x >= this.screenBounds_.width + this.screenBounds_.left) {
            direction = ScrollDirection.RIGHT;
        }
        if (!direction) {
            return;
        }
        this.lastScrollTime_ = new Date().getTime();
        chrome.accessibilityPrivate.scrollAtPoint(this.scrollLocation_, direction);
    }
}
(function (ScrollModeController) {
    /**
     * The time in milliseconds that needs to be exceeded before sending another
     * scroll.
     */
    ScrollModeController.RATE_LIMIT = 50;
})(ScrollModeController || (ScrollModeController = {}));
TestImportManager.exportForTesting(ScrollModeController);

// Copyright 2024 The Chromium Authors
// Use of this source code is governed by a BSD-style license that can be
// found in the LICENSE file.
var SyntheticMouseEventButton = chrome.accessibilityPrivate.SyntheticMouseEventButton;
var LandmarkType;
(function (LandmarkType) {
    LandmarkType["FOREHEAD"] = "forehead";
    LandmarkType["FOREHEAD_TOP"] = "foreheadTop";
    LandmarkType["LEFT_TEMPLE"] = "leftTemple";
    LandmarkType["NOSE_TIP"] = "noseTip";
    LandmarkType["RIGHT_TEMPLE"] = "rightTemple";
    LandmarkType["ROTATION"] = "rotation";
})(LandmarkType || (LandmarkType = {}));
/** Default values for members that track preferences. */
const DEFAULT_MOUSE_SPEED = 10;
const DEFAULT_USE_MOUSE_ACCELERATION = true;
const DEFAULT_BUFFER_SIZE = 7;
const DEFAULT_PRECISION_SPEED_FACTOR = 50;
const DEFAULT_VELOCITY_FACTOR = 0.45;
/**
 * How long to wait after the user moves the mouse with a physical device
 * before moving the mouse with facegaze.
 */
const IGNORE_UPDATES_AFTER_MOUSE_MOVE_MS = 500;
/**
 * The indices of the tracked landmarks in a FaceLandmarkerResult.
 * See all landmarks at
 * https://storage.googleapis.com/mediapipe-assets/documentation/mediapipe_face_landmark_fullsize.png.
 */
const LANDMARK_INDICES = [
    { name: LandmarkType.FOREHEAD, index: 8 },
    { name: LandmarkType.FOREHEAD_TOP, index: 10 },
    { name: LandmarkType.NOSE_TIP, index: 4 },
    { name: LandmarkType.RIGHT_TEMPLE, index: 127 },
    { name: LandmarkType.LEFT_TEMPLE, index: 356 },
    // Rotation does not have a landmark index, but is included in this list
    // because it can be used as a landmark.
    { name: LandmarkType.ROTATION, index: -1 },
];
/**
 * The maximum value for the velocity threshold pref. We use this to ensure
 * this.velocityThresholdFactor_ is a decimal.
 */
const MAX_VELOCITY_THRESHOLD_PREF_VALUE = 20;
/** How frequently to run the mouse movement logic. */
const MOUSE_INTERVAL_MS = 16;
/** Handles all interaction with the mouse. */
class MouseController {
    // References to core classes.
    scrollModeController_;
    bubbleController_;
    /** Last seen mouse location (cached from event in onMouseMovedOrDragged_). */
    mouseLocation_;
    onMouseMovedHandler_;
    onMouseDraggedHandler_;
    screenBounds_;
    // Members that track preferences.
    targetBufferSize_ = DEFAULT_BUFFER_SIZE;
    useMouseAcceleration_ = DEFAULT_USE_MOUSE_ACCELERATION;
    spdRight_ = DEFAULT_MOUSE_SPEED;
    spdLeft_ = DEFAULT_MOUSE_SPEED;
    spdUp_ = DEFAULT_MOUSE_SPEED;
    spdDown_ = DEFAULT_MOUSE_SPEED;
    velocityThreshold_ = 0;
    velocityThresholdFactor_ = DEFAULT_VELOCITY_FACTOR;
    useVelocityThreshold_ = true;
    /** The most recent raw face landmark mouse locations. */
    buffer_ = [];
    /** Used for smoothing the recent points in the buffer. */
    smoothKernel_ = [];
    /** The most recent smoothed mouse location. */
    previousSmoothedLocation_;
    /** The last location in screen coordinates of the tracked landmark. */
    lastLandmarkLocation_;
    mouseInterval_ = -1;
    lastMouseMovedTime_ = 0;
    landmarkWeights_;
    paused_ = false;
    longClickActive_ = false;
    // Precision-related members.
    usePrecision_ = false;
    precisionActive_ = false;
    precisionSpeedFactor_ = DEFAULT_PRECISION_SPEED_FACTOR;
    constructor(bubbleController) {
        this.bubbleController_ = bubbleController;
        this.onMouseMovedHandler_ = new EventHandler([], chrome.automation.EventType.MOUSE_MOVED, event => this.onMouseMovedOrDragged_(event));
        this.onMouseDraggedHandler_ = new EventHandler([], chrome.automation.EventType.MOUSE_DRAGGED, event => this.onMouseMovedOrDragged_(event));
        this.scrollModeController_ = new ScrollModeController();
        this.calcSmoothKernel_();
        this.calcVelocityThreshold_();
        this.landmarkWeights_ = new Map();
        this.landmarkWeights_.set(LandmarkType.FOREHEAD, 0.1275);
        this.landmarkWeights_.set(LandmarkType.FOREHEAD_TOP, 0.0738);
        this.landmarkWeights_.set(LandmarkType.NOSE_TIP, 0.3355);
        this.landmarkWeights_.set(LandmarkType.LEFT_TEMPLE, 0.0336);
        this.landmarkWeights_.set(LandmarkType.RIGHT_TEMPLE, 0.0336);
        this.landmarkWeights_.set(LandmarkType.ROTATION, 0.3960);
        this.init();
    }
    async init() {
        chrome.accessibilityPrivate.enableMouseEvents(true);
        const desktop = await AsyncUtil.getDesktop();
        this.onMouseMovedHandler_.setNodes(desktop);
        this.onMouseMovedHandler_.start();
        this.onMouseDraggedHandler_.setNodes(desktop);
        this.onMouseDraggedHandler_.start();
    }
    isScrollModeActive() {
        return this.scrollModeController_.active();
    }
    isLongClickActive() {
        return this.longClickActive_;
    }
    usePrecision() {
        return this.usePrecision_;
    }
    isPrecisionActive() {
        return this.precisionActive_;
    }
    togglePrecision() {
        if (!this.usePrecision_) {
            return;
        }
        this.precisionActive_ = !this.precisionActive_;
        if (!this.precisionActive_) {
            this.bubbleController_.resetBubble();
        }
    }
    async start() {
        this.paused_ = false;
        // TODO(b/309121742): Handle display bounds changed.
        const screens = await new Promise((resolve) => {
            chrome.accessibilityPrivate.getDisplayBounds((screens) => {
                resolve(screens);
            });
        });
        if (!screens.length) {
            // TODO(b/309121742): Error handling for no detected screens.
            return;
        }
        this.screenBounds_ = screens[0];
        // Ensure the mouse location is set.
        // The user might not be touching the mouse because they only
        // have FaceGaze input, in which case we need to make the
        // mouse move to a known location in order to proceed.
        if (!this.mouseLocation_) {
            this.resetLocation();
        }
        // Start the logic to move the mouse.
        this.mouseInterval_ =
            setInterval(() => this.updateMouseLocation_(), MOUSE_INTERVAL_MS);
    }
    /** Update the current location of the tracked face landmark. */
    onFaceLandmarkerResult(result) {
        if (this.paused_ || !this.screenBounds_ || !result.faceLandmarks ||
            !result.faceLandmarks[0]) {
            return;
        }
        // These scale from 0 to 1.
        const avgLandmarkLocation = { x: 0, y: 0 };
        let hasLandmarks = false;
        for (const landmark of LANDMARK_INDICES) {
            let landmarkLocation;
            if (landmark.name === 'rotation' && result.facialTransformationMatrixes &&
                result.facialTransformationMatrixes.length) {
                landmarkLocation =
                    MouseController.calculateRotationFromFacialTransformationMatrix(result.facialTransformationMatrixes[0]);
            }
            else if (result.faceLandmarks[0][landmark.index] !== undefined) {
                landmarkLocation = result.faceLandmarks[0][landmark.index];
            }
            if (!landmarkLocation) {
                continue;
            }
            const x = landmarkLocation.x;
            const y = landmarkLocation.y;
            let weight = this.landmarkWeights_.get(landmark.name);
            if (!weight) {
                weight = 0;
            }
            avgLandmarkLocation.x += (x * weight);
            avgLandmarkLocation.y += (y * weight);
            hasLandmarks = true;
        }
        if (!hasLandmarks) {
            return;
        }
        // Calculate the absolute position on the screen, where the top left
        // corner represents (0,0) and the bottom right corner represents
        // (this.screenBounds_.width, this.screenBounds_.height).
        // TODO(b/309121742): Handle multiple displays.
        const absoluteY = Math.round(avgLandmarkLocation.y * this.screenBounds_.height +
            this.screenBounds_.top);
        // Reflect the x coordinate since the webcam doesn't mirror in the
        // horizontal direction.
        const scaledX = Math.round(avgLandmarkLocation.x * this.screenBounds_.width);
        const absoluteX = this.screenBounds_.width - scaledX + this.screenBounds_.left;
        this.lastLandmarkLocation_ = { x: absoluteX, y: absoluteY };
    }
    /**
     * Called every MOUSE_INTERVAL_MS, this function uses the most recent
     * landmark location to update the current mouse position within the
     * screen, applying appropriate scaling and smoothing.
     * This function doesn't simply set the absolute position of the tracked
     * landmark. Instead, it calculates deltas to be applied to the
     * current mouse location based on the landmark's location relative
     * to its previous location.
     */
    updateMouseLocation_() {
        if (this.paused_ || !this.lastLandmarkLocation_ || !this.mouseLocation_ ||
            !this.screenBounds_) {
            return;
        }
        // Add the most recent landmark point to the buffer.
        this.addPointToBuffer_(this.lastLandmarkLocation_);
        // Smooth the buffer to get the latest target point.
        const smoothed = this.applySmoothing_();
        // Compute the velocity: how position has changed compared to the previous
        // point. Note that we are assuming points come in at a regular interval,
        // but we could also run this regularly in a timeout to reduce the rate at
        // which points must be seen.
        if (!this.previousSmoothedLocation_) {
            // Initialize previous location to the current to avoid a jump at
            // start-up.
            this.previousSmoothedLocation_ = smoothed;
        }
        const velocityX = smoothed.x - this.previousSmoothedLocation_.x;
        const velocityY = smoothed.y - this.previousSmoothedLocation_.y;
        const scaledVel = this.asymmetryScale_({ x: velocityX, y: velocityY });
        this.previousSmoothedLocation_ = smoothed;
        if (this.useMouseAcceleration_) {
            scaledVel.x *= this.applySigmoidAcceleration_(scaledVel.x);
            scaledVel.y *= this.applySigmoidAcceleration_(scaledVel.y);
        }
        if (!this.scrollModeController_.active() &&
            !this.exceedsVelocityThreshold_(scaledVel.x) &&
            !this.exceedsVelocityThreshold_(scaledVel.y)) {
            // The velocity threshold wasn't exceeded, so we shouldn't update the
            // mouse location. We do this to avoid unintended jitteriness of the
            // mouse. When we're in scroll mode, we don't want to apply the velocity
            // threshold because we're not visibly moving the mouse.
            return;
        }
        scaledVel.x = this.applyVelocityThreshold_(scaledVel.x);
        scaledVel.y = this.applyVelocityThreshold_(scaledVel.y);
        // The mouse location is the previous location plus the velocity.
        const newX = this.mouseLocation_.x + scaledVel.x;
        const newY = this.mouseLocation_.y + scaledVel.y;
        // Update mouse location: onMouseMovedOrChanged_ is async and may not
        // be called again until after another point is received from the
        // face tracking, so better to keep a fresh copy.
        // Clamp to screen bounds.
        // TODO(b/309121742): Handle multiple displays.
        this.mouseLocation_ = {
            x: Math.max(Math.min(this.screenBounds_.width, Math.round(newX)), this.screenBounds_.left),
            y: Math.max(Math.min(this.screenBounds_.height, Math.round(newY)), this.screenBounds_.top),
        };
        if (this.scrollModeController_.active()) {
            this.scrollModeController_.scroll(this.mouseLocation_);
            return;
        }
        // Only update if it's been long enough since the last time the user
        // touched their physical mouse or trackpad.
        if (new Date().getTime() - this.lastMouseMovedTime_ >
            IGNORE_UPDATES_AFTER_MOUSE_MOVE_MS) {
            EventGenerator.sendMouseMove(this.mouseLocation_.x, this.mouseLocation_.y, { useRewriters: true, forceNotSynthetic: true });
            chrome.accessibilityPrivate.setCursorPosition(this.mouseLocation_);
        }
    }
    addPointToBuffer_(point) {
        // Add this latest point to the buffer.
        if (this.buffer_.length === this.targetBufferSize_) {
            this.buffer_.shift();
        }
        // Fill the buffer with this point until we reach buffer size.
        while (this.buffer_.length < this.targetBufferSize_) {
            this.buffer_.push(point);
        }
    }
    mouseLocation() {
        return this.mouseLocation_;
    }
    resetLocation() {
        if (this.paused_ || !this.screenBounds_ ||
            this.scrollModeController_.active()) {
            return;
        }
        const x = Math.round(this.screenBounds_.width / 2) + this.screenBounds_.left;
        const y = Math.round(this.screenBounds_.height / 2) + this.screenBounds_.top;
        this.mouseLocation_ = { x, y };
        chrome.accessibilityPrivate.setCursorPosition({ x, y });
    }
    reset() {
        this.stop();
        this.onMouseMovedHandler_.stop();
        this.onMouseDraggedHandler_.stop();
    }
    stop() {
        if (this.longClickActive_ && this.mouseLocation_) {
            // Release the existing long click action when the mouse controller is
            // stopped to ensure we do not leave the user in a permanent "drag" state.
            EventGenerator.sendMouseRelease(this.mouseLocation_.x, this.mouseLocation_.y, { forceNotSynthetic: true });
            this.longClickActive_ = false;
        }
        if (this.precisionActive_) {
            this.togglePrecision();
        }
        if (this.mouseInterval_ !== -1) {
            clearInterval(this.mouseInterval_);
            this.mouseInterval_ = -1;
        }
        this.lastLandmarkLocation_ = undefined;
        this.previousSmoothedLocation_ = undefined;
        this.lastMouseMovedTime_ = 0;
        this.buffer_ = [];
        this.paused_ = false;
    }
    togglePaused() {
        const newPaused = !this.paused_;
        // Run start/stop before assigning the new pause value, since start/stop
        // will modify the pause value.
        newPaused ? this.stop() : this.start();
        this.paused_ = newPaused;
    }
    toggleScrollMode() {
        this.scrollModeController_.toggle(this.mouseLocation_, this.screenBounds_);
        if (!this.isScrollModeActive()) {
            this.bubbleController_.resetBubble();
        }
    }
    toggleLongClick() {
        if (!this.mouseLocation_) {
            return;
        }
        this.longClickActive_ = !this.longClickActive_;
        if (this.longClickActive_) {
            EventGenerator.sendMousePress(this.mouseLocation_.x, this.mouseLocation_.y, SyntheticMouseEventButton.LEFT, { forceNotSynthetic: true });
            // Enable the DragEventRewriter so that mouse moved events get rewritten
            // into mouse dragged events.
            chrome.accessibilityPrivate.enableDragEventRewriter(true);
        }
        else {
            EventGenerator.sendMouseRelease(this.mouseLocation_.x, this.mouseLocation_.y, { forceNotSynthetic: true });
            chrome.accessibilityPrivate.enableDragEventRewriter(false);
        }
        if (!this.isLongClickActive()) {
            this.bubbleController_.resetBubble();
        }
    }
    /** Listener for when the mouse position changes. */
    onMouseMovedOrDragged_(event) {
        if (event.eventFrom === 'user') {
            // Mouse changes that aren't synthesized should actually move the mouse.
            // Assume all synthesized mouse movements come from within FaceGaze.
            this.mouseLocation_ = { x: event.mouseX, y: event.mouseY };
            this.lastMouseMovedTime_ = new Date().getTime();
            if (this.scrollModeController_.active()) {
                // Scroll mode honors physical mouse movements.
                this.scrollModeController_.updateScrollLocation(this.mouseLocation_);
            }
            if (this.longClickActive_) {
                // Send a drag event from the user's mouse move event.
                // FaceGaze cursor control should already have sent a drag
                // event, so this only needs to occur on user mouse movements.
                EventGenerator.sendMouseMove(event.mouseX, event.mouseY, { useRewriters: true, forceNotSynthetic: true });
            }
        }
    }
    /**
     * Construct a kernel for smoothing the recent facegaze points.
     * Specifically, this is an exponential curve with amplitude of 0.92 and
     * y-intercept of 0.08. This ensures that the curve hits the points (0, 0.08)
     * and (1, 1).
     * Note: Whenever the buffer size is updated, we must reconstruct
     * the smoothing kernel so that it is the right length.
     */
    calcSmoothKernel_() {
        this.smoothKernel_ = [];
        let sum = 0;
        const step = 1 / this.targetBufferSize_;
        // We use values step <= i <= 1 to determine the weight of each point.
        for (let i = step; i <= 1; i += step) {
            const smoothFactor = Math.E;
            const numerator = (Math.E ** (smoothFactor * i)) - 1;
            const denominator = (Math.E ** smoothFactor) - 1;
            const value = 0.92 * (numerator / denominator) + 0.08;
            this.smoothKernel_.push(value);
            sum += value;
        }
        for (let i = 0; i < this.targetBufferSize_; i++) {
            this.smoothKernel_[i] /= sum;
        }
    }
    /**
     * Applies the `smoothKernel_` to the `buffer_` of recent points to generate
     * a single point.
     */
    applySmoothing_() {
        const result = { x: 0, y: 0 };
        for (let i = 0; i < this.targetBufferSize_; i++) {
            const kernelPart = this.smoothKernel_[i];
            result.x += this.buffer_[i].x * kernelPart;
            result.y += this.buffer_[i].y * kernelPart;
        }
        return result;
    }
    /**
     * Magnifies velocities. This means the user has to move their head less far
     * to get to the edges of the screens.
     */
    asymmetryScale_(vel) {
        // If precision mode is active, reduce the mouse speed.
        const precisionClickMultiplier = (100 - this.precisionSpeedFactor_) / 100;
        const spdRight = this.usePrecision_ && this.precisionActive_ ?
            this.spdRight_ * precisionClickMultiplier :
            this.spdRight_;
        const spdLeft = this.usePrecision_ && this.precisionActive_ ?
            this.spdLeft_ * precisionClickMultiplier :
            this.spdLeft_;
        const spdDown = this.usePrecision_ && this.precisionActive_ ?
            this.spdDown_ * precisionClickMultiplier :
            this.spdDown_;
        const spdUp = this.usePrecision_ && this.precisionActive_ ?
            this.spdUp_ * precisionClickMultiplier :
            this.spdUp_;
        if (vel.x > 0) {
            vel.x *= spdRight;
        }
        else {
            vel.x *= spdLeft;
        }
        if (vel.y > 0) {
            vel.y *= spdDown;
        }
        else {
            vel.y *= spdUp;
        }
        return vel;
    }
    /**
     * Calculate a sigmoid function that creates an S curve with
     * a y intercept around ~.2 for velocity === 0 and
     * approaches 1.2 around velocity of 22. Change is near-linear
     * around velocities 0 to 9, centered at velocity of five.
     */
    applySigmoidAcceleration_(velocity) {
        const shift = 5;
        const slope = 0.3;
        const multiply = 1.2;
        velocity = Math.abs(velocity);
        const sig = 1 / (1 + Math.exp(-slope * (velocity - shift)));
        return multiply * sig;
    }
    calcVelocityThreshold_() {
        // Threshold is a function of speed. Threshold increases as speed increases
        // because it's easier to move the mouse accidentally at high mouse speeds.
        // The velocity threshold factor can be tuned by the user.
        const averageSpeed = (this.spdUp_ + this.spdDown_ + this.spdLeft_ + this.spdRight_) / 4;
        this.velocityThreshold_ = averageSpeed * this.velocityThresholdFactor_;
    }
    exceedsVelocityThreshold_(velocity) {
        if (!this.useVelocityThreshold_ ||
            (this.usePrecision_ && this.precisionActive_)) {
            // Do not use velocity threshold during a precision click.
            return true;
        }
        return Math.abs(velocity) > this.velocityThreshold_;
    }
    applyVelocityThreshold_(velocity) {
        if (!this.useVelocityThreshold_ ||
            (this.usePrecision_ && this.precisionActive_)) {
            // Do not apply velocity threshold during a precision click.
            return velocity;
        }
        if (Math.abs(velocity) < this.velocityThreshold_) {
            return 0;
        }
        return (velocity > 0) ? velocity - this.velocityThreshold_ :
            velocity + this.velocityThreshold_;
    }
    setLandmarkWeightsForTesting(useWeights) {
        if (!useWeights) {
            // If we don't want to use landmark weights, we should default to the
            // forehead location.
            this.landmarkWeights_ = new Map();
            this.landmarkWeights_.set('forehead', 1);
        }
    }
    setVelocityThresholdForTesting(useThreshold) {
        this.useVelocityThreshold_ = useThreshold;
    }
    setBufferSizeForTesting(size) {
        this.targetBufferSize_ = size;
        this.calcSmoothKernel_();
        while (this.buffer_.length > this.targetBufferSize_) {
            this.buffer_.shift();
        }
    }
    speedUpChanged(speed) {
        this.spdUp_ = speed;
        this.calcVelocityThreshold_();
    }
    speedDownChanged(speed) {
        this.spdDown_ = speed;
        this.calcVelocityThreshold_();
    }
    speedLeftChanged(speed) {
        this.spdLeft_ = speed;
        this.calcVelocityThreshold_();
    }
    speedRightChanged(speed) {
        this.spdRight_ = speed;
        this.calcVelocityThreshold_();
    }
    useCursorAccelerationChanged(useAcceleration) {
        this.useMouseAcceleration_ = useAcceleration;
    }
    velocityThresholdChanged(threshold) {
        // Ensure threshold factor is a decimal value.
        this.velocityThresholdFactor_ =
            threshold / MAX_VELOCITY_THRESHOLD_PREF_VALUE;
        this.calcVelocityThreshold_();
    }
    precisionClickChanged(usePrecision) {
        this.usePrecision_ = usePrecision;
    }
    precisionSpeedFactorChanged(speedFactor) {
        this.precisionSpeedFactor_ = speedFactor;
    }
    static calculateRotationFromFacialTransformationMatrix(facialTransformationMatrix) {
        const mat = facialTransformationMatrix.data;
        const m11 = mat[0];
        const m12 = mat[1];
        const m13 = mat[2];
        const m21 = mat[4];
        const m22 = mat[5];
        const m23 = mat[6];
        const m31 = mat[8];
        const m32 = mat[9];
        const m33 = mat[10];
        if (m31 === 1) {
            // cos(theta) is 0, so theta is pi/2 or -pi/2.
            // This seems like the head would have to be pretty rotated so we can
            // probably safely ignore it for now.
            console.log('cannot process matrix with m[3][1] == 1 yet.');
            return;
        }
        // First compute scaling and rotation from the facial transformation matrix.
        // Taken from glmatrix, https://glmatrix.net/docs/mat4.js.html.
        const scaling = [
            Math.hypot(m11, m12, m13),
            Math.hypot(m21, m22, m23),
            Math.hypot(m31, m32, m33),
        ];
        // Translation is unused but could be used in the future. Leaving it here
        // so we don't have to re-compute the math later.
        // const translation = [mat[12], mat[13], mat[14]];
        // Scale the m values to create sm values; used for x and y axis rotation
        // computation. On Brya, scaling is basically all 1s, so we could ignore it.
        // TODO(b:309121742): Determine if we can remove scaling from computation,
        // and use the matrix values directly.
        const sm31 = m31 / scaling[0];
        const sm32 = m32 / scaling[1];
        const sm33 = m33 / scaling[2];
        // Convert rotation matrix to Euler angles. Refer to math in
        // https://eecs.qmul.ac.uk/~gslabaugh/publications/euler.pdf.
        // This has units in radians.
        const xRotation = -1 * Math.asin(sm31);
        const yRotation = Math.atan2(sm32 / Math.cos(xRotation), sm33 / Math.cos(xRotation));
        // z-axis rotation is head tilt, and not used at the moment. Later, it could
        // be used during calibration. Leaving it here so we don't need to
        // re-compute the math later. const sm11 = m11 * is1; const sm21 = m21 *
        // is1; const zRotation =
        // Math.atan2(sm21 / Math.cos(xRotation), sm11 / Math.cos(xRotation));
        const x = 0.5 - xRotation / (Math.PI * 2);
        const y = 0.5 - yRotation / (Math.PI * 2);
        return { x, y };
    }
}
TestImportManager.exportForTesting(MouseController);

// Copyright 2023 The Chromium Authors
// Use of this source code is governed by a BSD-style license that can be
// found in the LICENSE file.
const CONNECT_TO_WEBCAM_TIMEOUT = 1000;
/**
 * The default number of times we should try to connect to the webcam. If we
 * cannot establish a connection after trying this many times, then we should
 * notify the user and turn off FaceGaze.
 */
const DEFAULT_CONNECT_TO_WEBCAM_RETRIES = 10;
/**
 * The interval, in milliseconds, for which we request results from the
 * FaceLandmarker API. This should be frequent enough to give a real-time
 * feeling.
 */
const DETECT_FACE_LANDMARKS_INTERVAL_MS = 60;
/**
 * The dimensions used for the camera stream. 192 x 192 are the dimensions
 * used by the FaceLandmarker, so frames that are larger than this must go
 * through a downsampling process, which takes extra work.
 */
const VIDEO_FRAME_DIMENSIONS = 192;
/** The wasm loader JS is checked in under this path. */
const WASM_LOADER_PATH = 'accessibility_common/mv2/third_party/mediapipe_task_vision/' +
    'vision_wasm_internal.js';
/** Handles interaction with the webcam and FaceLandmarker. */
class WebCamFaceLandmarker {
    // Core objects that power face landmark recognition.
    faceLandmarker_ = null;
    imageCapture_;
    bubbleController_;
    // Callbacks.
    onFaceLandmarkerResult_;
    onTrackMuted_;
    onTrackUnmuted_;
    // Event handlers that route to either private member functions or callbacks.
    onTrackEndedHandler_;
    onTrackMutedHandler_;
    onTrackUnmutedHandler_;
    // State-related members.
    stopped_ = true;
    // Members to track the connection to the webcam.
    connectToWebCamRetriesRemaining_ = DEFAULT_CONNECT_TO_WEBCAM_RETRIES;
    setWebCamConnected_;
    setReadyForTesting_;
    constructor(bubbleController, onFaceLandmarkerResult, onTrackMuted, onTrackUnmuted) {
        this.bubbleController_ = bubbleController;
        // Save callbacks.
        this.onFaceLandmarkerResult_ = onFaceLandmarkerResult;
        this.onTrackMuted_ = onTrackMuted;
        this.onTrackUnmuted_ = onTrackUnmuted;
        // Create handlers that run the above callbacks.
        this.onTrackEndedHandler_ = () => this.onTrackEnded_();
        this.onTrackMutedHandler_ = () => {
            this.onTrackMuted_();
        };
        this.onTrackUnmutedHandler_ = () => this.onTrackUnmuted_();
        this.intervalID_ = null;
        this.webCamConnected_ = new Promise(resolve => {
            this.setWebCamConnected_ = resolve;
        });
        this.readyForTesting_ = new Promise(resolve => {
            this.setReadyForTesting_ = resolve;
        });
    }
    /**
     * Initializes the FaceLandmarker, connects to the webcam, and starts
     * detecting face landmarks.
     */
    async init() {
        this.stopped_ = false;
        await this.createFaceLandmarker_();
        this.connectToWebCam_();
        await this.webCamConnected_;
        this.startDetectingFaceLandmarks_();
    }
    async createFaceLandmarker_() {
        let proceed;
        chrome.accessibilityPrivate.installFaceGazeAssets(async (assets) => {
            if (!assets) {
                // FaceGaze will not work unless the FaceGaze assets are successfully
                // installed. When the assets fail to install, AccessibilityManager
                // shows a notification to the user informing them of the failure and to
                // try installing again later. As a result, we should turn FaceGaze off
                // here and allow them to toggle the feature back on to retry the
                // download.
                console.error(`Couldn't create FaceLandmarker because FaceGaze assets couldn't be
              installed.`);
                chrome.settingsPrivate.setPref(PrefNames.FACE_GAZE_ENABLED, false);
                return;
            }
            // Create a blob to hold the wasm contents.
            const blob = new Blob([assets.wasm]);
            const customFileset = {
                // The wasm loader JS is checked in, so specify the path.
                wasmLoaderPath: chrome.runtime.getURL(WASM_LOADER_PATH),
                // The wasm is stored in a blob, so pass a URL to the blob.
                wasmBinaryPath: URL.createObjectURL(blob),
            };
            // Create the FaceLandmarker and set options.
            this.faceLandmarker_ = await FaceLandmarker.createFromModelBuffer(customFileset, new Uint8Array(assets.model));
            const options = {
                outputFaceBlendshapes: true,
                outputFacialTransformationMatrixes: true,
                runningMode: 'IMAGE',
                numFaces: 1,
            };
            this.faceLandmarker_.setOptions(options);
            if (this.setReadyForTesting_) {
                this.setReadyForTesting_();
            }
            proceed();
        });
        return new Promise(resolve => {
            proceed = resolve;
        });
    }
    async connectToWebCam_() {
        const constraints = {
            video: {
                height: VIDEO_FRAME_DIMENSIONS,
                width: VIDEO_FRAME_DIMENSIONS,
                facingMode: 'user',
            },
        };
        let stream;
        try {
            stream = await navigator.mediaDevices.getUserMedia(constraints);
        }
        catch (error) {
            if (this.connectToWebCamRetriesRemaining_ > 0) {
                const message = chrome.i18n.getMessage('facegaze_connect_to_camera', [this.connectToWebCamRetriesRemaining_]);
                this.bubbleController_.updateBubble(message);
                this.connectToWebCamRetriesRemaining_ -= 1;
                setTimeout(() => this.connectToWebCam_(), CONNECT_TO_WEBCAM_TIMEOUT);
            }
            else {
                chrome.settingsPrivate.setPref(PrefNames.FACE_GAZE_ENABLED, false);
            }
            return;
        }
        const tracks = stream.getVideoTracks();
        // It is possible for FaceGaze to be turned off before getUserMedia()
        // completes. If FaceGaze has stopped when we finish this promise, then
        // clean up the webcam resources so the webcam does not stay on.
        if (this.stopped_) {
            tracks[0].stop();
            return;
        }
        this.imageCapture_ = new ImageCapture(tracks[0]);
        this.imageCapture_.track.addEventListener('ended', this.onTrackEndedHandler_);
        this.imageCapture_.track.addEventListener('mute', this.onTrackMutedHandler_);
        this.imageCapture_.track.addEventListener('unmute', this.onTrackUnmutedHandler_);
        // Once we make it here, we know that the webcam is connected.
        this.connectToWebCamRetriesRemaining_ = DEFAULT_CONNECT_TO_WEBCAM_RETRIES;
        this.setWebCamConnected_();
    }
    onTrackEnded_() {
        if (this.imageCapture_) {
            // Tell MediaStreamTrack that we are no longer using this ended track.
            this.imageCapture_.track.stop();
            this.removeEventListeners_();
        }
        this.imageCapture_ = undefined;
        this.connectToWebCam_();
    }
    startDetectingFaceLandmarks_() {
        this.intervalID_ = setInterval(() => this.detectFaceLandmarks_(), DETECT_FACE_LANDMARKS_INTERVAL_MS);
    }
    async detectFaceLandmarks_() {
        if (!this.faceLandmarker_) {
            return;
        }
        let frame;
        try {
            frame = await this.imageCapture_.grabFrame();
        }
        catch (error) {
            // grabFrame() can occasionally return an error, so in these cases, we
            // should handle the error and simply return instead of trying to process
            // the frame.
            return;
        }
        const startTime = performance.now();
        const result = this.faceLandmarker_.detect(/*image=*/ frame);
        const latency = performance.now() - startTime;
        // Use a callback to send the result to the main FaceGaze object.
        this.onFaceLandmarkerResult_({ result, latency });
    }
    removeEventListeners_() {
        if (this.imageCapture_) {
            this.imageCapture_.track.removeEventListener('ended', this.onTrackEndedHandler_);
            this.imageCapture_.track.removeEventListener('mute', this.onTrackMutedHandler_);
            this.imageCapture_.track.removeEventListener('unmute', this.onTrackUnmutedHandler_);
        }
    }
    stop() {
        this.stopped_ = true;
        if (this.imageCapture_) {
            this.removeEventListeners_();
            this.imageCapture_.track.stop();
            this.imageCapture_ = undefined;
        }
        this.faceLandmarker_ = null;
        if (this.intervalID_ !== null) {
            clearInterval(this.intervalID_);
            this.intervalID_ = null;
        }
    }
}
TestImportManager.exportForTesting(WebCamFaceLandmarker);

// Copyright 2023 The Chromium Authors
// Use of this source code is governed by a BSD-style license that can be
// found in the LICENSE file.
/** Main class for FaceGaze. */
class FaceGaze {
    // References to core classes.
    bubbleController_;
    gestureHandler_;
    metricsUtils_;
    mouseController_;
    webCamFaceLandmarker_;
    // Other variables, such as state and callbacks.
    actionsEnabled_ = false;
    cursorControlEnabled_ = false;
    initialized_ = false;
    // isFaceLandmarkerResultValid_ is initialized to true to ensure the correct
    // UI messages are shown on startup.
    isFaceLandmarkerResultValid_ = true;
    isCameraMuted_ = false;
    onInitCallbackForTest_;
    prefsListener_;
    constructor(isDictationActive) {
        this.bubbleController_ = new BubbleController(() => {
            return {
                paused: this.gestureHandler_.isPaused() ?
                    this.gestureHandler_.getGestureForPause() :
                    undefined,
                scrollMode: this.mouseController_.isScrollModeActive() ?
                    this.gestureHandler_.getGestureForScroll() :
                    undefined,
                longClick: this.mouseController_.isLongClickActive() ?
                    this.gestureHandler_.getGestureForLongClick() :
                    undefined,
                dictation: isDictationActive() ?
                    this.gestureHandler_.getGestureForDictation() :
                    undefined,
                heldMacros: this.gestureHandler_.getHeldMacroDisplayStrings(),
                precision: this.mouseController_.isPrecisionActive() ?
                    this.gestureHandler_.getGestureForPrecision() :
                    undefined,
                isFaceLandmarkerResultValid: this.isFaceLandmarkerResultValid_,
                isCameraMuted: this.isCameraMuted_,
            };
        });
        this.webCamFaceLandmarker_ = new WebCamFaceLandmarker(this.bubbleController_, (resultWithLatency) => {
            const { result, latency } = resultWithLatency;
            this.processFaceLandmarkerResult_(result, latency);
        }, () => this.onCameraFeedMuted_(), () => this.onCameraFeedUnmuted_());
        this.mouseController_ = new MouseController(this.bubbleController_);
        this.gestureHandler_ = new GestureHandler(this.mouseController_, this.bubbleController_, isDictationActive);
        this.metricsUtils_ = new MetricsUtils();
        this.prefsListener_ = prefs => this.updateFromPrefs_(prefs);
        this.init_();
    }
    /** Initializes FaceGaze. */
    init_() {
        // TODO(b/309121742): Listen to magnifier bounds changed so as to update
        // cursor relative position logic when magnifier is running.
        chrome.settingsPrivate.getAllPrefs(prefs => this.updateFromPrefs_(prefs));
        chrome.settingsPrivate.onPrefsChanged.addListener(this.prefsListener_);
        if (this.onInitCallbackForTest_) {
            this.onInitCallbackForTest_();
            this.onInitCallbackForTest_ = undefined;
        }
        this.initialized_ = true;
        this.maybeShowConfirmationDialog_();
    }
    maybeShowConfirmationDialog_() {
        chrome.settingsPrivate.getPref(PrefNames.ACCELERATOR_DIALOG_HAS_BEEN_ACCEPTED, pref => {
            if (pref.value === undefined || pref.value === null) {
                return;
            }
            if (pref.value) {
                // If the confirmation dialog has already been accepted, there is no
                // need to show it again. We can proceed as if it's been accepted.
                this.onConfirmationDialog_(true);
                return;
            }
            // If the confirmation dialog has not been accepted yet, display it to
            // the user.
            const title = chrome.i18n.getMessage('facegaze_confirmation_dialog_title');
            const description = chrome.i18n.getMessage('facegaze_confirmation_dialog_desc');
            chrome.accessibilityPrivate.showConfirmationDialog(title, description, /*cancelName=*/ undefined, (accepted) => {
                this.onConfirmationDialog_(accepted);
            });
        });
    }
    /** Runs when the confirmation dialog has either been accepted or rejected. */
    onConfirmationDialog_(accepted) {
        chrome.settingsPrivate.setPref(PrefNames.ACCELERATOR_DIALOG_HAS_BEEN_ACCEPTED, accepted);
        if (!accepted) {
            // If the dialog was rejected, then disable the FaceGaze feature and do
            // not show the confirmation dialog for disabling.
            chrome.settingsPrivate.setPref(PrefNames.FACE_GAZE_ENABLED, false);
            return;
        }
        // If the dialog was accepted, then initialize FaceGaze.
        chrome.accessibilityPrivate.openSettingsSubpage(SettingsPath);
        this.bubbleController_.updateBubble('');
        this.webCamFaceLandmarker_.init();
    }
    updateFromPrefs_(prefs) {
        prefs.forEach(pref => {
            if (pref.value === undefined || pref.value === null) {
                return;
            }
            switch (pref.key) {
                case PrefNames.ACTIONS_ENABLED:
                    this.actionsEnabledChanged_(pref.value);
                    break;
                case PrefNames.CURSOR_CONTROL_ENABLED:
                    this.cursorControlEnabledChanged_(pref.value);
                    break;
                case PrefNames.CURSOR_USE_ACCELERATION:
                    this.mouseController_.useCursorAccelerationChanged(pref.value);
                    break;
                case PrefNames.GESTURE_TO_CONFIDENCE:
                    this.gestureHandler_.gesturesToConfidencesChanged(pref.value);
                    break;
                case PrefNames.GESTURE_TO_KEY_COMBO:
                    this.gestureHandler_.gesturesToKeyCombosChanged(pref.value);
                    break;
                case PrefNames.GESTURE_TO_MACRO:
                    this.gestureHandler_.gesturesToMacrosChanged(pref.value);
                    break;
                case PrefNames.PRECISION_CLICK:
                    this.mouseController_.precisionClickChanged(pref.value);
                    break;
                case PrefNames.PRECISION_CLICK_SPEED_FACTOR:
                    this.mouseController_.precisionSpeedFactorChanged(pref.value);
                    break;
                case PrefNames.SPD_UP:
                    this.mouseController_.speedUpChanged(pref.value);
                    break;
                case PrefNames.SPD_DOWN:
                    this.mouseController_.speedDownChanged(pref.value);
                    break;
                case PrefNames.SPD_LEFT:
                    this.mouseController_.speedLeftChanged(pref.value);
                    break;
                case PrefNames.SPD_RIGHT:
                    this.mouseController_.speedRightChanged(pref.value);
                    break;
                case PrefNames.VELOCITY_THRESHOLD:
                    this.mouseController_.velocityThresholdChanged(pref.value);
                    break;
                default:
                    return;
            }
        });
    }
    cursorControlEnabledChanged_(value) {
        if (this.cursorControlEnabled_ === value) {
            return;
        }
        this.cursorControlEnabled_ = value;
        if (this.cursorControlEnabled_) {
            this.mouseController_.start();
        }
        else {
            this.mouseController_.stop();
        }
    }
    actionsEnabledChanged_(value) {
        if (this.actionsEnabled_ === value) {
            return;
        }
        this.actionsEnabled_ = value;
        if (this.actionsEnabled_) {
            this.gestureHandler_.start();
        }
        else {
            this.gestureHandler_.stop();
            // If actions are turned off while a toggled action is active, then we
            // should toggle out of the relevant action. Otherwise, the user will be
            // stuck in the action with no way to exit.
            if (this.mouseController_.isScrollModeActive()) {
                this.mouseController_.toggleScrollMode();
            }
            if (this.mouseController_.isLongClickActive()) {
                this.mouseController_.toggleLongClick();
            }
            if (this.mouseController_.isPrecisionActive()) {
                this.mouseController_.togglePrecision();
            }
        }
    }
    processFaceLandmarkerResult_(result, latency) {
        if (!result) {
            return;
        }
        if (result.faceBlendshapes.length === 0 &&
            result.faceLandmarks.length === 0 &&
            result.facialTransformationMatrixes.length === 0) {
            // In practice, we can get results that are empty. Typically this happens
            // when the camera is obstructed, blocked by permissions, if the user is
            // out of frame, or if the user's face can't be detected for any other
            // reason. In all of these cases, the camera feed is active but either
            // gives us empty frames or frames where a face cannot be recognized,
            // which causes the FaceLandmarker to return this type of result.
            this.updateIsFaceLandmarkerResultValid_(/*valid=*/ false);
            return;
        }
        this.updateIsFaceLandmarkerResultValid_(/*valid=*/ true);
        if (latency !== undefined) {
            this.metricsUtils_.addFaceLandmarkerResultLatency(latency);
        }
        if (this.cursorControlEnabled_) {
            this.mouseController_.onFaceLandmarkerResult(result);
        }
        if (this.actionsEnabled_) {
            const { macros, displayText } = this.gestureHandler_.detectMacros(result);
            for (const macro of macros) {
                const checkContextResult = macro.checkContext();
                if (!checkContextResult.canTryAction) {
                    console.warn('Cannot execute macro in this context', macro.getName(), checkContextResult.error, checkContextResult.failedContext);
                    continue;
                }
                const runMacroResult = macro.run();
                if (!runMacroResult.isSuccess) {
                    console.warn('Failed to execute macro ', macro.getName(), runMacroResult.error);
                }
            }
            if (displayText) {
                this.bubbleController_.updateBubble(displayText);
            }
        }
    }
    updateIsFaceLandmarkerResultValid_(valid) {
        if (valid === this.isFaceLandmarkerResultValid_) {
            return;
        }
        this.isFaceLandmarkerResultValid_ = valid;
        this.bubbleController_.resetBubble();
    }
    onCameraFeedMuted_() {
        if (this.isCameraMuted_) {
            return;
        }
        this.isCameraMuted_ = true;
        this.bubbleController_.resetBubble();
    }
    onCameraFeedUnmuted_() {
        if (!this.isCameraMuted_) {
            return;
        }
        this.isCameraMuted_ = false;
        this.bubbleController_.resetBubble();
    }
    /** Destructor to remove any listeners. */
    onFaceGazeDisabled() {
        this.mouseController_.reset();
        this.gestureHandler_.stop();
        this.webCamFaceLandmarker_.stop();
        chrome.settingsPrivate.onPrefsChanged.removeListener(this.prefsListener_);
    }
    /** Allows tests to wait for FaceGaze to be fully initialized. */
    setOnInitCallbackForTest(callback) {
        if (!this.initialized_) {
            this.onInitCallbackForTest_ = callback;
            return;
        }
        callback();
    }
}
TestImportManager.exportForTesting(FaceGaze);

// Copyright 2020 The Chromium Authors
// Use of this source code is governed by a BSD-style license that can be
// found in the LICENSE file.
var EventType = chrome.automation.EventType;
var RoleType = chrome.automation.RoleType;
/** Main class for the Chrome OS magnifier. */
class Magnifier {
    type;
    /**
     * Whether focus following is enabled or not, based on
     * settings.a11y.screen_magnifier_focus_following preference.
     */
    screenMagnifierFocusFollowing_;
    /**
     * Whether ChromeVox focus following is enabled or not.
     * settings.a11y.screen_magnifier_chromevox_focus_following preference.
     */
    screenMagnifierFollowsChromeVox_ = true;
    /**
     * Whether Select to Speak focus following is enabled or not.
     * settings.a11y.screen_magnifier_select_to_speak_focus_following preference.
     */
    screenMagnifierFollowsSts_ = true;
    /**
     * Whether magnifier is currently initializing, and so should ignore
     * focus updates.
     */
    isInitializing_ = true;
    /** Last time mouse has moved (from last onMouseMovedOrDragged). */
    lastMouseMovedTime_;
    lastFocusSelectionOrCaretMove_;
    focusHandler_;
    activeDescendantHandler_;
    selectionHandler_;
    onCaretBoundsChangedHandler;
    onMagnifierBoundsChangedHandler_;
    onChromeVoxFocusChangedHandler_;
    onSelectToSpeakFocusChangedHandler_;
    updateFromPrefsHandler_;
    onMouseMovedHandler_;
    onMouseDraggedHandler_;
    lastChromeVoxBounds_;
    lastSelectToSpeakBounds_;
    onLoadDesktopCallbackForTest_;
    constructor(type) {
        this.type = type;
        this.focusHandler_ = new EventHandler([], EventType.FOCUS, event => this.onFocusOrSelectionChanged_(event));
        this.activeDescendantHandler_ = new EventHandler([], EventType.ACTIVE_DESCENDANT_CHANGED, event => this.onActiveDescendantChanged_(event));
        this.selectionHandler_ = new EventHandler([], EventType.SELECTION, event => this.onFocusOrSelectionChanged_(event));
        this.onCaretBoundsChangedHandler = new EventHandler([], EventType.CARET_BOUNDS_CHANGED, event => this.onCaretBoundsChanged(event));
        this.onMagnifierBoundsChangedHandler_ = new ChromeEventHandler(chrome.accessibilityPrivate.onMagnifierBoundsChanged, bounds => this.onMagnifierBoundsChanged_(bounds));
        this.onChromeVoxFocusChangedHandler_ = new ChromeEventHandler(chrome.accessibilityPrivate.onChromeVoxFocusChanged, bounds => this.onChromeVoxFocusChanged_(bounds));
        this.onSelectToSpeakFocusChangedHandler_ = new ChromeEventHandler(chrome.accessibilityPrivate.onSelectToSpeakFocusChanged, bounds => this.onSelectToSpeakFocusChanged_(bounds));
        this.updateFromPrefsHandler_ = new ChromeEventHandler(chrome.settingsPrivate.onPrefsChanged, prefs => this.updateFromPrefs_(prefs));
        this.onMouseMovedHandler_ = new EventHandler([], chrome.automation.EventType.MOUSE_MOVED, () => this.onMouseMovedOrDragged_());
        this.onMouseDraggedHandler_ = new EventHandler([], chrome.automation.EventType.MOUSE_DRAGGED, () => this.onMouseMovedOrDragged_());
        this.onLoadDesktopCallbackForTest_ = null;
        this.init_();
    }
    /** Destructor to remove listeners. */
    onMagnifierDisabled() {
        this.focusHandler_.stop();
        this.activeDescendantHandler_.stop();
        this.selectionHandler_.stop();
        this.onCaretBoundsChangedHandler.stop();
        this.onMagnifierBoundsChangedHandler_.stop();
        this.onChromeVoxFocusChangedHandler_.stop();
        this.onSelectToSpeakFocusChangedHandler_.stop();
        this.updateFromPrefsHandler_.stop();
        this.onMouseMovedHandler_.stop();
        this.onMouseDraggedHandler_.stop();
        this.lastMouseMovedTime_ = undefined;
        this.lastChromeVoxBounds_ = undefined;
        this.lastSelectToSpeakBounds_ = undefined;
        this.lastFocusSelectionOrCaretMove_ = undefined;
    }
    /** Initializes Magnifier. */
    init_() {
        chrome.settingsPrivate.getAllPrefs(prefs => this.updateFromPrefs_(prefs));
        this.updateFromPrefsHandler_.start();
        chrome.automation.getDesktop(desktop => {
            this.focusHandler_.setNodes(desktop);
            this.focusHandler_.start();
            this.activeDescendantHandler_.setNodes(desktop);
            this.activeDescendantHandler_.start();
            this.selectionHandler_.setNodes(desktop);
            this.selectionHandler_.start();
            this.onCaretBoundsChangedHandler.setNodes(desktop);
            this.onCaretBoundsChangedHandler.start();
            this.onMouseMovedHandler_.setNodes(desktop);
            this.onMouseMovedHandler_.start();
            this.onMouseDraggedHandler_.setNodes(desktop);
            this.onMouseDraggedHandler_.start();
            if (this.onLoadDesktopCallbackForTest_) {
                this.onLoadDesktopCallbackForTest_();
                this.onLoadDesktopCallbackForTest_ = null;
            }
        });
        this.onMagnifierBoundsChangedHandler_.start();
        this.onChromeVoxFocusChangedHandler_.start();
        this.onSelectToSpeakFocusChangedHandler_.start();
        chrome.accessibilityPrivate.enableMouseEvents(true);
        this.isInitializing_ = true;
        setTimeout(() => {
            this.isInitializing_ = false;
        }, Magnifier.IGNORE_FOCUS_UPDATES_INITIALIZATION_MS);
    }
    drawDebugRect_() {
        return Boolean(Flags.isEnabled(FlagName.MAGNIFIER_DEBUG_DRAW_RECT));
    }
    onMagnifierBoundsChanged_(bounds) {
        if (this.drawDebugRect_()) {
            chrome.accessibilityPrivate.setFocusRings([{
                    rects: [bounds],
                    type: chrome.accessibilityPrivate.FocusType.GLOW,
                    color: '#22d',
                }], chrome.accessibilityPrivate.AssistiveTechnologyType.MAGNIFIER);
        }
    }
    onChromeVoxFocusChanged_(bounds) {
        // Don't follow ChromeVox if focus following is off.
        if (!this.shouldFollowChromeVoxFocus()) {
            return;
        }
        // Don't follow ChromeVox focus if the mouse, keyboard focus or caret
        // has moved too recently.
        // TODO(b/259363112): Add a test for this.
        const now = new Date().getTime();
        if ((this.lastMouseMovedTime_ !== undefined &&
            now - this.lastMouseMovedTime_.getTime() <
                Magnifier.IGNORE_AT_UPDATES_AFTER_OTHER_MOVE_MS) ||
            (this.lastFocusSelectionOrCaretMove_ !== undefined &&
                now - this.lastFocusSelectionOrCaretMove_.getTime() <
                    Magnifier.IGNORE_AT_UPDATES_AFTER_OTHER_MOVE_MS)) {
            return;
        }
        // Ignore repeated updates from ChromeVox.
        if (bounds !== this.lastChromeVoxBounds_) {
            this.lastChromeVoxBounds_ = bounds;
            chrome.accessibilityPrivate.moveMagnifierToRect(bounds);
        }
    }
    onSelectToSpeakFocusChanged_(bounds) {
        // Don't follow select to speak if focus following is off.
        if (!this.shouldFollowStsFocus()) {
            return;
        }
        // Don't follow select to speak focus if the mouse, keyboard focus or caret
        // has moved too recently.
        // TODO(b/259363112): Add a test for this.
        const now = new Date().getTime();
        if ((this.lastMouseMovedTime_ !== undefined &&
            now - this.lastMouseMovedTime_.getTime() <
                Magnifier.IGNORE_AT_UPDATES_AFTER_OTHER_MOVE_MS) ||
            (this.lastFocusSelectionOrCaretMove_ !== undefined &&
                now - this.lastFocusSelectionOrCaretMove_.getTime() <
                    Magnifier.IGNORE_AT_UPDATES_AFTER_OTHER_MOVE_MS)) {
            return;
        }
        // Select to Speak refreshes the UI occasionally. We can
        // ignore repeated updates.
        if (bounds !== this.lastSelectToSpeakBounds_) {
            this.lastSelectToSpeakBounds_ = bounds;
            chrome.accessibilityPrivate.moveMagnifierToRect(bounds);
        }
    }
    /**
     * Sets |isInitializing_| inside tests to skip ignoring initial focus updates.
     */
    setIsInitializingForTest(isInitializing) {
        this.isInitializing_ = isInitializing;
    }
    /**
     * Sets |IGNORE_AT_UPDATES_AFTER_OTHER_MOVE_MS| inside tests to ensure all
     * automated input is received.
     */
    setIgnoreAssistiveTechnologyUpdatesAfterOtherMoveDurationForTest(duration) {
        Magnifier.IGNORE_AT_UPDATES_AFTER_OTHER_MOVE_MS = duration;
    }
    updateFromPrefs_(prefs) {
        prefs.forEach(pref => {
            switch (pref.key) {
                case Magnifier.Prefs.SCREEN_MAGNIFIER_CHROMEVOX_FOCUS_FOLLOWING:
                    this.screenMagnifierFollowsChromeVox_ = Boolean(pref.value);
                    break;
                case Magnifier.Prefs.SCREEN_MAGNIFIER_FOCUS_FOLLOWING:
                    this.screenMagnifierFocusFollowing_ = Boolean(pref.value);
                    break;
                case Magnifier.Prefs.SCREEN_MAGNIFIER_SELECT_TO_SPEAK_FOCUS_FOLLOWING:
                    this.screenMagnifierFollowsSts_ = Boolean(pref.value);
                    break;
                default:
                    return;
            }
        });
    }
    /**
     * Returns whether magnifier viewport should follow focus. Exposed for
     * testing.
     *
     * TODO(crbug.com/40730171): Add Chrome OS preference to allow disabling focus
     * following for docked magnifier.
     */
    shouldFollowFocus() {
        return Boolean(!this.isInitializing_ &&
            (this.type === Magnifier.Type.DOCKED ||
                this.type === Magnifier.Type.FULL_SCREEN &&
                    this.screenMagnifierFocusFollowing_));
    }
    shouldFollowChromeVoxFocus() {
        return !this.isInitializing_ && this.screenMagnifierFollowsChromeVox_;
    }
    shouldFollowStsFocus() {
        return !this.isInitializing_ && this.screenMagnifierFollowsSts_;
    }
    /**
     * Listener for when focus is updated. Moves magnifier to include focused
     * element in viewport.
     *
     * TODO(accessibility): There is a bit of magnifier shakiness on arrow down in
     * omnibox - probably focus following fighting with caret following - maybe
     * add timer for last focus event so that fast-following caret updates don't
     * shake screen.
     * TODO(accessibility): On page load, sometimes viewport moves to center of
     * webpage instead of spotlighting first focusable page element.
     */
    onFocusOrSelectionChanged_(event) {
        const node = event.target;
        if (!node.location || !this.shouldFollowFocus()) {
            return;
        }
        // TODO(b/267329383): Clean this up, since Number(undefined) is NaN, and
        // NaN should be avoided if possible.
        if (Number(new Date()) - Number(this.lastMouseMovedTime_) <
            Magnifier.IGNORE_FOCUS_UPDATES_AFTER_MOUSE_MOVE_MS) {
            return;
        }
        // Skip trying to move magnifier to encompass whole webpage or pdf. It's too
        // big, and magnifier usually ends up in middle at left edge of page.
        const isTooBig = AutomationPredicate.roles([RoleType.WEB_VIEW, RoleType.EMBEDDED_OBJECT]);
        if (node.isRootNode || isTooBig(node)) {
            return;
        }
        this.lastFocusSelectionOrCaretMove_ = new Date();
        chrome.accessibilityPrivate.moveMagnifierToRect(node.location);
    }
    /**
     * Listener for when active descendant is changed. Moves magnifier to include
     * active descendant in viewport.
     */
    onActiveDescendantChanged_(event) {
        const { activeDescendant } = event.target;
        if (!activeDescendant || !this.shouldFollowFocus()) {
            return;
        }
        const { location } = activeDescendant;
        if (!location) {
            return;
        }
        chrome.accessibilityPrivate.moveMagnifierToRect(location);
    }
    /**
     * Listener for when caret bounds have changed. Moves magnifier to include
     * caret in viewport.
     */
    onCaretBoundsChanged(event) {
        const { target } = event;
        if (!target || !target.caretBounds) {
            return;
        }
        // TODO(b/267329383): Clean this up, since Number(undefined) is NaN, and
        // NaN should be avoided if possible.
        if (Number(new Date()) - Number(this.lastMouseMovedTime_) <
            Magnifier.IGNORE_FOCUS_UPDATES_AFTER_MOUSE_MOVE_MS) {
            return;
        }
        // Note: onCaretBoundsChanged can get called when TextInputType is changed,
        // during which the caret bounds are set to an empty rect (0x0), and we
        // don't need to adjust the viewport position based on this bogus caret
        // position. This is only a transition period; the caret position will be
        // fixed upon focusing directly afterward.
        if (target.caretBounds.width === 0 && target.caretBounds.height === 0) {
            return;
        }
        this.lastFocusSelectionOrCaretMove_ = new Date();
        const caretBoundsCenter = RectUtil.center(target.caretBounds);
        chrome.accessibilityPrivate.magnifierCenterOnPoint(caretBoundsCenter);
    }
    /** Listener for when mouse moves or drags. */
    onMouseMovedOrDragged_() {
        this.lastMouseMovedTime_ = new Date();
    }
    /**
     * Used by C++ tests to ensure Magnifier load is competed.
     * @param callback Callback for when desktop is loaded from automation.
     */
    setOnLoadDesktopCallbackForTest(callback) {
        if (!this.focusHandler_.listening()) {
            this.onLoadDesktopCallbackForTest_ = callback;
            return;
        }
        // Desktop already loaded.
        callback();
    }
}
(function (Magnifier) {
    (function (Type) {
        Type["FULL_SCREEN"] = "fullScreen";
        Type["DOCKED"] = "docked";
    })(Magnifier.Type || (Magnifier.Type = {}));
    (function (Prefs) {
        Prefs["SCREEN_MAGNIFIER_FOCUS_FOLLOWING"] = "settings.a11y.screen_magnifier_focus_following";
        Prefs["SCREEN_MAGNIFIER_CHROMEVOX_FOCUS_FOLLOWING"] = "settings.a11y.screen_magnifier_chromevox_focus_following";
        Prefs["SCREEN_MAGNIFIER_SELECT_TO_SPEAK_FOCUS_FOLLOWING"] = "settings.a11y.screen_magnifier_select_to_speak_focus_following";
    })(Magnifier.Prefs || (Magnifier.Prefs = {}));
    /**
     * Duration of time directly after startup of magnifier to ignore focus
     * updates, to prevent the magnified region from jumping.
     */
    Magnifier.IGNORE_FOCUS_UPDATES_INITIALIZATION_MS = 500;
    /**
     * Duration of time directly after a mouse move or drag to ignore focus
     * updates, to prevent the magnified region from jumping.
     */
    Magnifier.IGNORE_FOCUS_UPDATES_AFTER_MOUSE_MOVE_MS = 250;
    /**
     * Duration of time directly after a mouse move or drag to ignore focus
     * updates from assistive technologies like Select to Speak and ChromeVox, to
     * prevent the magnified region from jumping.
     */
    Magnifier.IGNORE_AT_UPDATES_AFTER_OTHER_MOVE_MS = 1500;
})(Magnifier || (Magnifier = {}));
TestImportManager.exportForTesting(Magnifier);

// Copyright 2020 The Chromium Authors
// Use of this source code is governed by a BSD-style license that can be
// found in the LICENSE file.
/**
 * Class to manage loading resources depending on which Accessibility features
 * are enabled.
 */
class AccessibilityCommon {
    autoclick_ = null;
    magnifier_ = null;
    dictation_ = null;
    faceGaze_ = null;
    // For tests.
    autoclickLoadCallbackForTest_ = null;
    // TODO(b:315990318): Migrate these callbacks to Function after
    // setOnLoadDesktopCallbackForTest() is migrated to typescript.
    magnifierLoadCallbackForTest_ = null;
    dictationLoadCallbackForTest_ = null;
    facegazeLoadCallbackForTest_ = null;
    static FACEGAZE_PREF_NAME = 'settings.a11y.face_gaze.enabled';
    constructor() {
        this.init_();
    }
    static async init() {
        await Flags.init();
        globalThis.accessibilityCommon = new AccessibilityCommon();
    }
    getAutoclickForTest() {
        return this.autoclick_;
    }
    getFaceGazeForTest() {
        return this.faceGaze_;
    }
    getMagnifierForTest() {
        return this.magnifier_;
    }
    /**
     * Initializes the AccessibilityCommon extension.
     */
    init_() {
        chrome.accessibilityFeatures.autoclick.get({}, details => this.onAutoclickUpdated_(details));
        chrome.accessibilityFeatures.autoclick.onChange.addListener(details => this.onAutoclickUpdated_(details));
        chrome.accessibilityFeatures.screenMagnifier.get({}, details => this.onMagnifierUpdated_(Magnifier.Type.FULL_SCREEN, details));
        chrome.accessibilityFeatures.screenMagnifier.onChange.addListener(details => this.onMagnifierUpdated_(Magnifier.Type.FULL_SCREEN, details));
        chrome.accessibilityFeatures.dockedMagnifier.get({}, details => this.onMagnifierUpdated_(Magnifier.Type.DOCKED, details));
        chrome.accessibilityFeatures.dockedMagnifier.onChange.addListener(details => this.onMagnifierUpdated_(Magnifier.Type.DOCKED, details));
        chrome.accessibilityFeatures.dictation.get({}, details => this.onDictationUpdated_(details));
        chrome.accessibilityFeatures.dictation.onChange.addListener(details => this.onDictationUpdated_(details));
        // TODO(b/309121742): Add FaceGaze pref to the accessibilityFeatures
        // extension API.
        chrome.settingsPrivate.getPref(AccessibilityCommon.FACEGAZE_PREF_NAME, pref => this.onFaceGazeUpdated_(pref));
        chrome.settingsPrivate.onPrefsChanged.addListener(prefs => {
            for (const pref of prefs) {
                if (pref.key === AccessibilityCommon.FACEGAZE_PREF_NAME) {
                    this.onFaceGazeUpdated_(pref);
                    break;
                }
            }
        });
        // AccessibilityCommon is an IME so it shows in the input methods list
        // when it starts up. Remove from this list, Dictation will add it back
        // whenever needed.
        Dictation.removeAsInputMethod();
    }
    /**
     * Called when the autoclick feature is enabled or disabled.
     */
    onAutoclickUpdated_(details) {
        if (details.value && !this.autoclick_) {
            // Initialize the Autoclick extension.
            this.autoclick_ = new Autoclick();
            if (this.autoclickLoadCallbackForTest_) {
                this.autoclick_.setOnLoadDesktopCallbackForTest(this.autoclickLoadCallbackForTest_);
                this.autoclickLoadCallbackForTest_ = null;
            }
        }
        else if (!details.value && this.autoclick_) {
            // TODO(crbug.com/1096759): Consider using XHR to load/unload autoclick
            // rather than relying on a destructor to clean up state.
            this.autoclick_.onAutoclickDisabled();
            this.autoclick_ = null;
        }
    }
    /**
     * Called when the FaceGaze feature is fetched enabled or disabled.
     */
    onFaceGazeUpdated_(details) {
        if (details.value && !this.faceGaze_) {
            // Initialize the FaceGaze extension.
            this.faceGaze_ = new FaceGaze(() => {
                if (!this.dictation_) {
                    return false;
                }
                return this.dictation_.isActive();
            });
            if (this.facegazeLoadCallbackForTest_) {
                this.facegazeLoadCallbackForTest_();
                this.facegazeLoadCallbackForTest_ = null;
            }
        }
        else if (!details.value && this.faceGaze_) {
            this.faceGaze_.onFaceGazeDisabled();
            this.faceGaze_ = null;
        }
    }
    /**
     * Called when the magnifier feature is fetched enabled or disabled.
     */
    onMagnifierUpdated_(type, details) {
        if (details.value && !this.magnifier_) {
            this.magnifier_ = new Magnifier(type);
            if (this.magnifierLoadCallbackForTest_) {
                this.magnifier_.setOnLoadDesktopCallbackForTest(this.magnifierLoadCallbackForTest_);
                this.magnifierLoadCallbackForTest_ = null;
            }
        }
        else if (!details.value && this.magnifier_ && this.magnifier_.type === type) {
            this.magnifier_.onMagnifierDisabled();
            this.magnifier_ = null;
        }
    }
    /**
     * Called when the dictation feature is enabled or disabled.
     */
    onDictationUpdated_(details) {
        if (details.value && !this.dictation_) {
            this.dictation_ = new Dictation();
            if (this.dictationLoadCallbackForTest_) {
                this.dictationLoadCallbackForTest_();
                this.dictationLoadCallbackForTest_ = null;
            }
        }
        else if (!details.value && this.dictation_) {
            this.dictation_.onDictationDisabled();
            this.dictation_ = null;
        }
    }
    /**
     * Used by C++ tests to ensure a feature load is completed.
     * Set on AccessibilityCommon in case the feature has not started up yet.
     */
    setFeatureLoadCallbackForTest(feature, callback) {
        if (feature === 'autoclick') {
            if (!this.autoclick_) {
                this.autoclickLoadCallbackForTest_ = callback;
                return;
            }
            // Autoclick already loaded.
            this.autoclick_.setOnLoadDesktopCallbackForTest(callback);
        }
        else if (feature === 'dictation') {
            if (!this.dictation_) {
                this.dictationLoadCallbackForTest_ = callback;
                return;
            }
            // Dictation already loaded.
            callback();
        }
        else if (feature === 'magnifier') {
            if (!this.magnifier_) {
                this.magnifierLoadCallbackForTest_ = callback;
                return;
            }
            // Magnifier already loaded.
            this.magnifier_.setOnLoadDesktopCallbackForTest(callback);
        }
        else if (feature === 'facegaze') {
            if (!this.faceGaze_) {
                this.facegazeLoadCallbackForTest_ = callback;
                return;
            }
            // Facegaze already loaded.
            callback();
        }
    }
}
InstanceChecker.closeExtraInstances();
// Initialize the AccessibilityCommon extension.
AccessibilityCommon.init();
TestImportManager.exportForTesting(['AccessibilityCommon', AccessibilityCommon]);

export { AccessibilityCommon };
//# sourceMappingURL=accessibility_common_loader.rollup.js.map
