// Copyright 2025 The Chromium Authors
// Use of this source code is governed by a BSD-style license that can be
// found in the LICENSE file.
import { TestImportManager } from '/common/testing/test_import_manager.js';
import { OffscreenCommandType } from './offscreen_command_type.js';
/*
 * Helper to wrap messaging api between offscreen doc and service worker.
 */
export class Messenger {
    static OFFSCREEN_DOCUMENT_PATH = 'accessibility_common/mv3/offscreen.html';
    static instance;
    context_;
    // A promise that tracks when offscreen doc is ready.
    offscreenDocumentPromise_ = null;
    // Resolves the above promise when received 'READY' from offscreen doc.
    setOffscreenDocumentReady_;
    // Whether a `chrome.offscreen.createDocument` is in progress.
    offscreenDocumentCreating_ = false;
    // Tracks registered message handlers.
    registry_;
    // Tracks resolvers for waitForHandled.
    handlerResolvers_ = new Map();
    constructor(context) {
        this.context_ = context;
        this.registry_ = new Map();
        chrome.runtime.onMessage.addListener((message, _sender, sendResponse) => this.handleMessage_(message, sendResponse));
    }
    static async init(context) {
        if (Messenger.instance) {
            throw 'Error: trying to create two instances of singleton Messenger.';
        }
        Messenger.instance = new Messenger(context);
        if (context === Messenger.Context.OFFSCREEN) {
            Messenger.send(OffscreenCommandType.MESSENGER_SW_READY);
            return;
        }
        if (context === Messenger.Context.SERVICE_WORKER) {
            Messenger.registerHandler(OffscreenCommandType.MESSENGER_SW_READY, () => {
                Messenger.instance.onOffscreenDocumentReady_();
            });
            return Messenger.instance.ensureOffscreenDocument_();
        }
    }
    /*
     * Ensures offscreen document is created. Returns a Promise that resolves when
     * offscreen document is created. This method should handle cases of service
     * worker restart and offscreen doc re-creation.
     */
    async ensureOffscreenDocument_() {
        const offscreenUrl = chrome.runtime.getURL(Messenger.OFFSCREEN_DOCUMENT_PATH);
        const existingContexts = await chrome.runtime.getContexts({
            contextTypes: [chrome.runtime.ContextType.OFFSCREEN_DOCUMENT],
            documentUrls: [offscreenUrl]
        });
        if (existingContexts.length > 0) {
            // Offscreen document is created in previous service worker runs.
            if (!this.offscreenDocumentPromise_) {
                this.offscreenDocumentPromise_ = Promise.resolve();
            }
        }
        else if (!this.offscreenDocumentCreating_) {
            // Otherwise, create one if there is no pending creation.
            this.offscreenDocumentCreating_ = true;
            this.offscreenDocumentPromise_ = new Promise(resolve => {
                this.setOffscreenDocumentReady_ = resolve;
            });
            chrome.offscreen
                .createDocument({
                url: offscreenUrl,
                reasons: [chrome.offscreen.Reason.WORKERS],
                justification: 'Audio web API and web assembly execution',
            })
                .catch(error => {
                console.error('Failed to create offscreen document: ', error);
                this.offscreenDocumentCreating_ = false;
            });
        }
        return this.offscreenDocumentPromise_;
    }
    /**
     * Handles `MESSENGER_SW_READY` message from offscreen document.
     */
    onOffscreenDocumentReady_() {
        this.offscreenDocumentCreating_ = false;
        this.setOffscreenDocumentReady_();
    }
    doSend_(command, data) {
        return chrome.runtime.sendMessage(
        /*extensionId=*/ undefined, 
        /*message=*/ Object.assign({ command }, data));
    }
    // Sends a command message to the other side. Returns a promise that resolves
    // if the other side sends back a reply.
    static send(command, data = {}) {
        if (Messenger.instance.context_ == Messenger.Context.OFFSCREEN) {
            return Messenger.instance.doSend_(command, data);
        }
        return Messenger.instance.ensureOffscreenDocument_().then(() => {
            return Messenger.instance.doSend_(command, data);
        });
    }
    // Registers a command handler.
    static registerHandler(command, handler) {
        Messenger.instance.registry_.set(command, handler);
    }
    /**
     * Test-only helper that returns a promise that resolves after the given
     * command handler has run.
     */
    static waitForHandled(command) {
        return new Promise(resolve => {
            const resolvers = Messenger.instance.handlerResolvers_.get(command);
            if (resolvers) {
                resolvers.push(resolve);
            }
            else {
                Messenger.instance.handlerResolvers_.set(command, [resolve]);
            }
        });
    }
    // Gets the handler for a given command.
    static getHandler(command) {
        return Messenger.instance.registry_.get(command);
    }
    // Handles the command message received from the other parts of the extension.
    // For example, when running in the service worker, this handles messages from
    // the offscreen document. And when running in the offscreen document, it
    // handles messages from the service worker.
    handleMessage_(message, sendResponse) {
        const command = message['command'];
        const result = Messenger.getHandler(command)?.(message);
        const resolvers = this.handlerResolvers_.get(command);
        if (resolvers) {
            for (const resolver of resolvers) {
                resolver();
            }
            this.handlerResolvers_.delete(command);
        }
        // If handler is async, return true to allow async sendResponse.
        if (result instanceof Promise) {
            result.then(sendResponse).catch(sendResponse);
            return true;
        }
        // Otherwise return false.
        return false;
    }
}
(function (Messenger) {
    // The context of where `Messenger` runs.
    let Context;
    (function (Context) {
        Context["SERVICE_WORKER"] = "serviceWorker";
        Context["OFFSCREEN"] = "offscreen";
    })(Context = Messenger.Context || (Messenger.Context = {}));
    // Compression format to use. Only `gzip` and `deflate` are supported.
    const compressionFormat = 'gzip';
    async function compress(buffer) {
        const cs = new CompressionStream(compressionFormat);
        const readableStream = new ReadableStream({
            start(controller) {
                controller.enqueue(new Uint8Array(buffer));
                controller.close();
            }
        });
        const stream = readableStream.pipeThrough(cs);
        const response = new Response(stream);
        const blob = await response.blob();
        return await blob.arrayBuffer();
    }
    async function decompress(buffer) {
        const ds = new DecompressionStream(compressionFormat);
        const readableStream = new ReadableStream({
            start(controller) {
                controller.enqueue(new Uint8Array(buffer));
                controller.close();
            }
        });
        const stream = readableStream.pipeThrough(ds);
        const response = new Response(stream);
        const blob = await response.blob();
        return await blob.arrayBuffer();
    }
    // Function to convert ArrayBuffer to Base64 string
    async function arrayBufferToBase64(inBuffer) {
        const buffer = await compress(inBuffer);
        let binary = '';
        const bytes = new Uint8Array(buffer);
        const len = bytes.byteLength;
        for (let i = 0; i < len; i++) {
            binary += String.fromCharCode(bytes[i]);
        }
        return btoa(binary); // btoa is a global function for Base64 encoding
    }
    Messenger.arrayBufferToBase64 = arrayBufferToBase64;
    // Function to convert Base64 string back to ArrayBuffer
    async function base64ToArrayBuffer(base64) {
        const binaryString = atob(base64); // atob is a global function for Base64 decoding
        const len = binaryString.length;
        const bytes = new Uint8Array(len);
        for (let i = 0; i < len; i++) {
            bytes[i] = binaryString.charCodeAt(i);
        }
        return await decompress(bytes.buffer);
    }
    Messenger.base64ToArrayBuffer = base64ToArrayBuffer;
})(Messenger || (Messenger = {}));
TestImportManager.exportForTesting(Messenger);
