import { BridgeHelper } from '../../../../../../../../../../../../../../../../../../../common/bridge_helper.js';
import { TestImportManager } from '../../../../../../../../../../../../../../../../../../../common/testing/test_import_manager.js';
import { Features } from '../../../../../../../../../../../../../../../../../../../common/features.js';
import { Flags } from '../../../../../../../../../../../../../../../../../../../common/flags.js';
import { LocalStorage } from '../../../../../../../../../../../../../../../../../../../common/local_storage.js';
import { InstanceChecker } from '../../../../../../../../../../../../../../../../../../../common/mv2/instance_checker.js';
import { MessageFormat } from '../../../../../../../../../../../../../../../../../../../chromevox/mv2/third_party/messageformat/messageformat.rollup.js';
import { KeyCode } from '../../../../../../../../../../../../../../../../../../../common/key_code.js';
import { Settings } from '../../../../../../../../../../../../../../../../../../../common/settings.js';
import { StringUtil } from '../../../../../../../../../../../../../../../../../../../common/string_util.js';
import { AutomationPredicate } from '../../../../../../../../../../../../../../../../../../../common/automation_predicate.js';
import { AutomationUtil } from '../../../../../../../../../../../../../../../../../../../common/automation_util.js';
import { constants } from '../../../../../../../../../../../../../../../../../../../common/constants.js';
import { Cursor, CURSOR_NODE_INDEX, CursorUnit, CursorMovement, WrappingCursor } from '../../../../../../../../../../../../../../../../../../../common/cursors/cursor.js';
import { CursorRange } from '../../../../../../../../../../../../../../../../../../../common/cursors/range.js';
import { EventHandler } from '../../../../../../../../../../../../../../../../../../../common/event_handler.js';
import { EventGenerator } from '../../../../../../../../../../../../../../../../../../../common/event_generator.js';
import { AsyncUtil } from '../../../../../../../../../../../../../../../../../../../common/async_util.js';
import { SRE } from '../../../../../../../../../../../../../../../../../../../chromevox/mv2/third_party/sre/sre_browser.js';
import { PhoneticDictionaries } from '../../../../../../../../../../../../../../../../../../../chromevox/mv2/phonetic_dictionaries.js';
import { AutomationTreeWalker } from '../../../../../../../../../../../../../../../../../../../common/tree_walker.js';
import { TreePathRecoveryStrategy } from '../../../../../../../../../../../../../../../../../../../common/cursors/recovery_strategy.js';
import { BrowserUtil } from '../../../../../../../../../../../../../../../../../../../common/browser_util.js';
import { RectUtil } from '../../../../../../../../../../../../../../../../../../../common/rect_util.js';
import { BridgeCallbackId } from '../../../../../../../../../../../../../../../../../../../common/bridge_callback_manager.js';

// Copyright 2022 The Chromium Authors
// Use of this source code is governed by a BSD-style license that can be
// found in the LICENSE file.
/**
 * @fileoverview Enums for BridgeHelper functions.
 */
/**
 * Specifies one of the renderer contexts for the ChromeVox extension. Code
 * specific to each of these contexts is contained in the corresponding
 * directory, while code used by two or more contexts is found in common/.
 * @enum {string}
 */
var BridgeContext;
(function (BridgeContext) {
    BridgeContext["BACKGROUND"] = "background";
    BridgeContext["LEARN_MODE"] = "learnMode";
    BridgeContext["LOG_PAGE"] = "logPage";
    BridgeContext["OPTIONS"] = "options";
    BridgeContext["PANEL"] = "panel";
})(BridgeContext || (BridgeContext = {}));
const BridgeConstants = {
    Braille: {
        TARGET: 'Braille',
        Action: {
            BACK_TRANSLATE: 'backTranslate',
            PAN_LEFT: 'panLeft',
            PAN_RIGHT: 'panRight',
            SET_BYPASS: 'setBypass',
            WRITE: 'write',
        },
    },
    ChromeVoxPrefs: {
        TARGET: 'ChromeVoxPrefs',
        Action: {
            GET_PREFS: 'getPrefs',
            GET_STICKY_PREF: 'getStickyPref',
            SET_LOGGING_PREFS: 'setLoggingPrefs',
            SET_PREF: 'setPref',
        },
    },
    ChromeVoxRange: {
        TARGET: 'ChromeVoxRange',
        Action: {
            CLEAR_CURRENT_RANGE: 'clearCurrentRange',
        },
    },
    CommandHandler: {
        TARGET: 'CommandHandler',
        Action: {
            ON_COMMAND: 'onCommand',
        },
    },
    Earcons: {
        TARGET: 'Earcons',
        Action: {
            CANCEL_EARCON: 'cancelEarcon',
            PLAY_EARCON: 'playEarcon',
        },
    },
    EventSource: {
        TARGET: 'EventSource',
        Action: {
            GET: 'get',
        },
    },
    EventStreamLogger: {
        TARGET: 'EventStreamLogger',
        Action: {
            NOTIFY_EVENT_STREAM_FILTER_CHANGED: 'notifyEventStreamFilterChanged',
        },
    },
    ForcedActionPath: {
        TARGET: 'ForcedActionPath',
        Action: {
            LISTEN_FOR: 'listenFor',
            ON_KEY_DOWN: 'onKeyDown',
            STOP_LISTENING: 'stopListening',
        },
    },
    GestureCommandHandler: {
        TARGET: 'GestureCommandHandler',
        Action: {
            SET_BYPASS: 'setBypass',
        },
    },
    LearnMode: {
        TARGET: 'LearnMode',
        Action: {
            CLEAR_TOUCH_EXPLORE_OUTPUT_TIME: 'clearTouchExploreOutputTime',
            ON_ACCESSIBILITY_GESTURE: 'onAccessibilityGesture',
            ON_BRAILLE_KEY_EVENT: 'onBrailleKeyEvent',
            ON_KEY_DOWN: 'onKeyDown',
            ON_KEY_UP: 'onKeyUp',
            READY: 'ready',
        },
    },
    LogStore: {
        TARGET: 'LogStore',
        Action: {
            CLEAR_LOG: 'clearLog',
            GET_LOGS: 'getLogs',
        },
    },
    Panel: {
        TARGET: 'Panel',
        Action: {
            ADD_MENU_ITEM: 'addMenuItem',
            ON_CURRENT_RANGE_CHANGED: 'onCurrentRangeChanged',
        },
    },
    PanelBackground: {
        TARGET: 'PanelBackground',
        Action: {
            CLEAR_SAVED_NODE: 'clearSavedNode',
            CREATE_ALL_NODE_MENU_BACKGROUNDS: 'createAllNodeMenuBackgrounds',
            CREATE_NEW_I_SEARCH: 'createNewISearch',
            DESTROY_I_SEARCH: 'destroyISearch',
            GET_ACTIONS_FOR_CURRENT_NODE: 'getActionsForCurrentNode',
            INCREMENTAL_SEARCH: 'incrementalSearch',
            NODE_MENU_CALLBACK: 'nodeMenuCallback',
            ON_TUTORIAL_READY: 'onTutorialReady',
            PERFORM_CUSTOM_ACTION_ON_CURRENT_NODE: 'performCustomActionOnCurrentNode',
            PERFORM_STANDARD_ACTION_ON_CURRENT_NODE: 'performStandardActionOnCurrentNode',
            SAVE_CURRENT_NODE: 'saveCurrentNode',
            SET_PANEL_COLLAPSE_WATCHER: 'setPanelCollapseWatcher',
            SET_RANGE_TO_I_SEARCH_NODE: 'setRangeToISearchNode',
            WAIT_FOR_PANEL_COLLAPSE: 'waitForPanelCollapse',
        },
    },
    TtsBackground: {
        TARGET: 'TtsBackground',
        Action: {
            GET_CURRENT_VOICE: 'getCurrentVoice',
            SPEAK: 'speak',
            UPDATE_PUNCTUATION_ECHO: 'updatePunctuationEcho',
        },
    },
};

// Copyright 2023 The Chromium Authors
// Use of this source code is governed by a BSD-style license that can be
// found in the LICENSE file.
/**
 * @fileoverview Provides an interface for other renderers to communicate with
 * the ChromeVox learn mode page.
 */
const TARGET$6 = BridgeConstants.LearnMode.TARGET;
const Action$7 = BridgeConstants.LearnMode.Action;
class LearnModeBridge {
    static clearTouchExploreOutputTime() {
        return BridgeHelper.sendMessage(TARGET$6, Action$7.CLEAR_TOUCH_EXPLORE_OUTPUT_TIME);
    }
    static onAccessibilityGesture(gesture) {
        return BridgeHelper.sendMessage(TARGET$6, Action$7.ON_ACCESSIBILITY_GESTURE, gesture);
    }
    static onBrailleKeyEvent(event) {
        return BridgeHelper.sendMessage(TARGET$6, Action$7.ON_BRAILLE_KEY_EVENT, event);
    }
    static onKeyDown(event) {
        return BridgeHelper.sendMessage(TARGET$6, Action$7.ON_KEY_DOWN, event);
    }
    static onKeyUp(event) {
        return BridgeHelper.sendMessage(TARGET$6, Action$7.ON_KEY_UP, event);
    }
    static ready() {
        return BridgeHelper.sendMessage(TARGET$6, Action$7.READY);
    }
}
TestImportManager.exportForTesting(LearnModeBridge);

// Copyright 2014 The Chromium Authors
// Use of this source code is governed by a BSD-style license that can be
// found in the LICENSE file.
/**
 * @fileoverview Class which allows construction of annotated strings.
 */
class Spannable {
    /** Underlying string. */
    string_;
    /** Spans (annotations). */
    spans_ = [];
    /**
     * @param stringValue Initial value of the spannable.
     * @param annotation Initial annotation for the entire string.
     */
    constructor(stringValue, annotation) {
        this.string_ = stringValue instanceof Spannable ? '' : stringValue || '';
        // Append the initial spannable.
        if (stringValue instanceof Spannable) {
            this.append(stringValue);
        }
        // Optionally annotate the entire string.
        if (annotation !== undefined) {
            const len = this.string_.length;
            this.spans_.push({ value: annotation, start: 0, end: len });
        }
    }
    toString() {
        return this.string_;
    }
    /** @return The length of the string */
    get length() {
        return this.string_.length;
    }
    /**
     * Adds a span to some region of the string.
     * @param value Annotation.
     * @param start Starting index (inclusive).
     * @param end Ending index (exclusive).
     */
    setSpan(value, start, end) {
        this.removeSpan(value);
        this.setSpanInternal(value, start, end);
    }
    /**
     * @param value Annotation.
     * @param start Starting index (inclusive).
     * @param end Ending index (exclusive).
     */
    setSpanInternal(value, start, end) {
        if (0 <= start && start <= end && end <= this.string_.length) {
            // Zero-length spans are explicitly allowed, because it is possible to
            // query for position by annotation as well as the reverse.
            this.spans_.push({ value, start, end });
            this.spans_.sort(function (a, b) {
                let ret = a.start - b.start;
                if (ret === 0) {
                    ret = a.end - b.end;
                }
                return ret;
            });
        }
        else {
            throw new RangeError('span out of range (start=' + start + ', end=' + end +
                ', len=' + this.string_.length + ')');
        }
    }
    /**
     * Removes a span.
     * @param value Annotation.
     */
    removeSpan(value) {
        for (let i = this.spans_.length - 1; i >= 0; i--) {
            if (this.spans_[i].value === value) {
                this.spans_.splice(i, 1);
            }
        }
    }
    /**
     * Appends another Spannable or string to this one.
     * @param other String or spannable to concatenate.
     */
    append(other) {
        if (other instanceof Spannable) {
            const otherSpannable = other;
            const originalLength = this.length;
            this.string_ += otherSpannable.string_;
            other.spans_.forEach(span => this.setSpan(span.value, span.start + originalLength, span.end + originalLength));
        }
        else if (typeof other === 'string') {
            this.string_ += other;
        }
    }
    /**
     * Returns the first value matching a position.
     * @param position Position to query.
     * @return Value annotating that position, or undefined if none is
     *     found.
     */
    getSpan(position) {
        return valueOfSpan(this.spans_.find(spanCoversPosition(position)));
    }
    /**
     * Returns the first span value which is an instance of a given constructor.
     * @param constructor Constructor.
     * @return Object if found; undefined otherwise.
     */
    getSpanInstanceOf(constructor) {
        return valueOfSpan(this.spans_.find(spanInstanceOf(constructor)));
    }
    /**
     * Returns all span values which are an instance of a given constructor.
     * Spans are returned in the order of their starting index and ending index
     * for spans with equals tarting indices.
     * @param constructor Constructor.
     * @return Array of object.
     */
    getSpansInstanceOf(constructor) {
        return (this.spans_.filter(spanInstanceOf(constructor)).map(valueOfSpan));
    }
    /**
     * Returns all spans matching a position.
     * @param position Position to query.
     * @return Values annotating that position.
     */
    getSpans(position) {
        return (this.spans_.filter(spanCoversPosition(position)).map(valueOfSpan));
    }
    /**
     * Returns whether a span is contained in this object.
     * @param value Annotation.
     */
    hasSpan(value) {
        return this.spans_.some(spanValueIs(value));
    }
    /**
     * Returns the start of the requested span. Throws if the span doesn't exist
     * in this object.
     * @param value Annotation.
     */
    getSpanStart(value) {
        return this.getSpanByValueOrThrow_(value).start;
    }
    /**
     * Returns the end of the requested span. Throws if the span doesn't exist
     * in this object.
     * @param value Annotation.
     */
    getSpanEnd(value) {
        return this.getSpanByValueOrThrow_(value).end;
    }
    /**
     * @param value Annotation.
     */
    getSpanIntervals(value) {
        return this.spans_.filter(span => span.value === value).map(span => {
            return { start: span.start, end: span.end };
        });
    }
    /**
     * Returns the number of characters covered by the given span. Throws if
     * the span is not in this object.
     */
    getSpanLength(value) {
        const span = this.getSpanByValueOrThrow_(value);
        return span.end - span.start;
    }
    /**
     * Gets the internal object for a span or throws if the span doesn't exist.
     * @param value The annotation.
     */
    getSpanByValueOrThrow_(value) {
        const span = this.spans_.find(spanValueIs(value));
        if (span) {
            return span;
        }
        throw new Error('Span ' + value + ' doesn\'t exist in spannable');
    }
    /**
     * Returns a substring of this spannable.
     * Note that while similar to String#substring, this function is much less
     * permissive about its arguments. It does not accept arguments in the wrong
     * order or out of bounds.
     *
     * @param start Start index, inclusive.
     * @param end End index, exclusive.
     *     If excluded, the length of the string is used instead.
     * @return Substring requested.
     */
    substring(start, end) {
        end = end !== undefined ? end : this.string_.length;
        if (start < 0 || end > this.string_.length || start > end) {
            throw new RangeError('substring indices out of range');
        }
        const result = new Spannable(this.string_.substring(start, end));
        this.spans_.forEach(span => {
            if (span.start <= end && span.end >= start) {
                const newStart = Math.max(0, span.start - start);
                const newEnd = Math.min(end - start, span.end - start);
                result.spans_.push({ value: span.value, start: newStart, end: newEnd });
            }
        });
        return result;
    }
    /**
     * Trims whitespace from the beginning.
     * @return String with whitespace removed.
     */
    trimLeft() {
        return this.trim_(true, false);
    }
    /**
     * Trims whitespace from the end.
     * @return String with whitespace removed.
     */
    trimRight() {
        return this.trim_(false, true);
    }
    /**
     * Trims whitespace from the beginning and end.
     * @return String with whitespace removed.
     */
    trim() {
        return this.trim_(true, true);
    }
    /**
     * Trims whitespace from either the beginning and end or both.
     * @param trimStart Trims whitespace from the start of a string.
     * @param trimEnd Trims whitespace from the end of a string.
     * @return String with whitespace removed.
     */
    trim_(trimStart, trimEnd) {
        if (!trimStart && !trimEnd) {
            return this;
        }
        // Special-case whitespace-only strings, including the empty string.
        // As an arbitrary decision, we treat this as trimming the whitespace off
        // the end, rather than the beginning, of the string.
        // This choice affects which spans are kept.
        if (/^\s*$/.test(this.string_)) {
            return this.substring(0, 0);
        }
        // Otherwise, we have at least one non-whitespace character to use as an
        // anchor when trimming.
        // TODO(b/314203187): Not null asserted, check that this is correct.
        const trimmedStart = trimStart ? this.string_.match(/^\s*/)[0].length : 0;
        const trimmedEnd = trimEnd ? this.string_.match(/\s*$/).index : this.string_.length;
        return this.substring(trimmedStart, trimmedEnd);
    }
    /**
     * Returns this spannable to a json serializable form, including the text
     * and span objects whose types have been registered with
     * registerSerializableSpan or registerStatelessSerializableSpan.
     * @return the json serializable form.
     */
    toJson() {
        const spans = [];
        this.spans_.forEach(span => {
            const serializeInfo = serializableSpansByConstructor.get(span.value.constructor);
            if (serializeInfo) {
                const spanObj = {
                    type: serializeInfo.name,
                    start: span.start,
                    end: span.end,
                    value: undefined,
                };
                if (serializeInfo.toJson) {
                    spanObj.value = serializeInfo.toJson.apply(span.value);
                }
                spans.push(spanObj);
            }
        });
        return { string: this.string_, spans };
    }
    /**
     * Creates a spannable from a json serializable representation.
     * @param obj object containing the serializable representation.
     */
    static fromJson(obj) {
        if (typeof obj.string !== 'string') {
            throw new Error('Invalid spannable json object: string field not a string');
        }
        if (!(obj.spans instanceof Array)) {
            throw new Error('Invalid spannable json object: no spans array');
        }
        const result = new Spannable(obj.string);
        result.spans_ = obj.spans.map(span => {
            if (typeof span.type !== 'string') {
                throw new Error('Invalid span in spannable json object: type not a string');
            }
            if (typeof span.start !== 'number' || typeof span.end !== 'number') {
                throw new Error('Invalid span in spannable json object: start or end not a number');
            }
            // TODO(b/314203187): Not null asserted, check that this is correct.
            const serializeInfo = serializableSpansByName.get(span.type);
            const value = serializeInfo.fromJson(span.value);
            return { value, start: span.start, end: span.end };
        });
        return result;
    }
    /**
     * Registers a type that can be converted to a json serializable format.
     * @param constructor The type of object that can be converted.
     * @param name String identifier used in the serializable format.
     * @param fromJson A function that converts the serializable object to an
     *     actual object of this type.
     * @param toJson A function that converts this object to a json serializable
     *     object. The function will be called with |this| set to the object to
     *     convert.
     */
    static registerSerializableSpan(constructor, name, fromJson, toJson) {
        const obj = { name, fromJson, toJson };
        serializableSpansByName.set(name, obj);
        serializableSpansByConstructor.set(constructor, obj);
    }
    /**
     * Registers an object type that can be converted to/from a json
     * serializable form. Objects of this type carry no state that will be
     * preserved when serialized.
     * @param constructor The type of the object that can be converted. This
     *     constructor will be called with no arguments to construct new objects.
     * @param name Name of the type used in the serializable object.
     */
    static registerStatelessSerializableSpan(constructor, name) {
        const fromJson = function (_obj) {
            return new constructor();
        };
        const obj = { name, toJson: undefined, fromJson };
        serializableSpansByName.set(name, obj);
        serializableSpansByConstructor.set(constructor, obj);
    }
}
/**
 * A spannable that allows a span value to annotate discontinuous regions of the
 * string. In effect, a span value can be set multiple times.
 * Note that most methods that assume a span value is unique such as
 * |getSpanStart| will use the first span value.
 */
class MultiSpannable extends Spannable {
    /**
     * @param string Initial value of the spannable.
     * @param annotation Initial annotation for the entire string.
     */
    constructor(string, annotation) {
        super(string, annotation);
    }
    setSpan(value, start, end) {
        this.setSpanInternal(value, start, end);
    }
    substring(start, end) {
        const ret = Spannable.prototype.substring.call(this, start, end);
        return new MultiSpannable(ret);
    }
}
/** Maps type names to serialization info objects. */
const serializableSpansByName = new Map();
/** Maps constructors to serialization info objects. */
const serializableSpansByConstructor = new Map();
// Helpers for implementing the various |get*| methods of |Spannable|.
function spanInstanceOf(constructor) {
    return function (span) {
        return span.value instanceof constructor;
    };
}
function spanCoversPosition(position) {
    return function (span) {
        return span.start <= position && position < span.end;
    };
}
function spanValueIs(value) {
    return function (span) {
        return span.value === value;
    };
}
function valueOfSpan(span) {
    return span ? span.value : undefined;
}
TestImportManager.exportForTesting(Spannable, MultiSpannable);

// Copyright 2014 The Chromium Authors
// Use of this source code is governed by a BSD-style license that can be
// found in the LICENSE file.
/**
 * @fileoverview A simple container object for the brailling of a
 * navigation from one object to another.
 *
 */
/**
 * A class capturing the braille for navigation from one object to
 * another.
 */
class NavBraille {
    /** Text, annotated with DOM nodes. */
    text;
    /** Selection start index. */
    startIndex;
    /** Selection end index. */
    endIndex;
    constructor(kwargs) {
        this.text = (kwargs.text instanceof Spannable) ? kwargs.text :
            new Spannable(kwargs.text);
        this.startIndex =
            (kwargs.startIndex !== undefined) ? kwargs.startIndex : -1;
        this.endIndex =
            (kwargs.endIndex !== undefined) ? kwargs.endIndex : this.startIndex;
    }
    /**
     * Convenience for creating simple braille output.
     * @param text Text to represent in braille.
     * @return Braille output without a cursor.
     */
    static fromText(text) {
        return new NavBraille({ text });
    }
    /**
     * Creates a NavBraille from its serialized json form as created
     * by toJson().
     * @param json the serialized json object.
     */
    static fromJson(json) {
        if (typeof json.startIndex !== 'number' ||
            typeof json.endIndex !== 'number') {
            throw 'Invalid start or end index in serialized NavBraille: ' + json;
        }
        return new NavBraille({
            text: Spannable.fromJson(json.spannable),
            startIndex: json.startIndex,
            endIndex: json.endIndex,
        });
    }
    /** @return true if this braille description is empty. */
    isEmpty() {
        return this.text.length === 0;
    }
    /** @return A string representation of this object. */
    toString() {
        return 'NavBraille(text="' + this.text.toString() + '" ' +
            ' startIndex="' + this.startIndex + '" ' +
            ' endIndex="' + this.endIndex + '")';
    }
    /**
     * Returns a plain old data object with the same data.
     * Suitable for JSON encoding.
     */
    toJson() {
        return {
            spannable: this.text.toJson(),
            startIndex: this.startIndex,
            endIndex: this.endIndex,
        };
    }
}
TestImportManager.exportForTesting(NavBraille);

// Copyright 2022 The Chromium Authors
// Use of this source code is governed by a BSD-style license that can be
// found in the LICENSE file.
/**
 * @fileoverview Data relating to the earcons used in ChromeVox.
 */
/** Earcon names. */
var EarconId;
(function (EarconId) {
    EarconId["ALERT_MODAL"] = "alert_modal";
    EarconId["ALERT_NONMODAL"] = "alert_nonmodal";
    EarconId["BUTTON"] = "button";
    EarconId["CHECK_OFF"] = "check_off";
    EarconId["CHECK_ON"] = "check_on";
    EarconId["CHROMEVOX_LOADING"] = "chromevox_loading";
    EarconId["CHROMEVOX_LOADED"] = "chromevox_loaded";
    EarconId["EDITABLE_TEXT"] = "editable_text";
    EarconId["INVALID_KEYPRESS"] = "invalid_keypress";
    EarconId["LINK"] = "link";
    EarconId["LISTBOX"] = "listbox";
    EarconId["LIST_ITEM"] = "list_item";
    EarconId["LONG_DESC"] = "long_desc";
    EarconId["MATH"] = "math";
    EarconId["NO_POINTER_ANCHOR"] = "no_pointer_anchor";
    EarconId["OBJECT_CLOSE"] = "object_close";
    EarconId["OBJECT_ENTER"] = "object_enter";
    EarconId["OBJECT_EXIT"] = "object_exit";
    EarconId["OBJECT_OPEN"] = "object_open";
    EarconId["OBJECT_SELECT"] = "object_select";
    EarconId["PAGE_FINISH_LOADING"] = "page_finish_loading";
    EarconId["PAGE_START_LOADING"] = "page_start_loading";
    EarconId["POP_UP_BUTTON"] = "pop_up_button";
    EarconId["RECOVER_FOCUS"] = "recover_focus";
    EarconId["SELECTION"] = "selection";
    EarconId["SELECTION_REVERSE"] = "selection_reverse";
    EarconId["SKIP"] = "skip";
    EarconId["SLIDER"] = "slider";
    EarconId["SMART_STICKY_MODE_OFF"] = "smart_sticky_mode_off";
    EarconId["SMART_STICKY_MODE_ON"] = "smart_sticky_mode_on";
    EarconId["WRAP"] = "wrap";
    EarconId["WRAP_EDGE"] = "wrap_edge";
})(EarconId || (EarconId = {}));
(function (EarconId) {
    function fromName(name) {
        return EarconId[name];
    }
    EarconId.fromName = fromName;
})(EarconId || (EarconId = {}));
/**
 * Maps a earcon id to a message id description.
 * Only add mappings for earcons used in ChromeVox Next. This map gets
 * used to generate tutorial content.
 */
({
    [EarconId.ALERT_MODAL]: 'alert_modal_earcon_description',
    [EarconId.ALERT_NONMODAL]: 'alert_nonmodal_earcon_description',
    [EarconId.BUTTON]: 'button_earcon_description',
    [EarconId.CHECK_OFF]: 'check_off_earcon_description',
    [EarconId.CHECK_ON]: 'check_on_earcon_description',
    [EarconId.CHROMEVOX_LOADING]: 'chromevox_loading_earcon_description',
    [EarconId.EDITABLE_TEXT]: 'editable_text_earcon_description',
    [EarconId.INVALID_KEYPRESS]: 'invalid_keypress_earcon_description',
    [EarconId.LINK]: 'link_earcon_description',
    [EarconId.LISTBOX]: 'listbox_earcon_description',
    [EarconId.NO_POINTER_ANCHOR]: 'no_pointer_anchor_earcon_description',
    [EarconId.PAGE_START_LOADING]: 'page_start_loading_earcon_description',
    [EarconId.POP_UP_BUTTON]: 'pop_up_button_earcon_description',
    [EarconId.SLIDER]: 'slider_earcon_description',
    [EarconId.SMART_STICKY_MODE_OFF]: 'smart_sticky_mode_off_earcon_description',
    [EarconId.SMART_STICKY_MODE_ON]: 'smart_sticky_mode_on_earcon_description',
    [EarconId.WRAP]: 'wrap_earcon_description',
});
TestImportManager.exportForTesting(['EarconId', EarconId]);

// Copyright 2014 The Chromium Authors
// Use of this source code is governed by a BSD-style license that can be
// found in the LICENSE file.
/**
 * @fileoverview Defines methods related to retrieving translated messages.
 */
class Msgs {
    /** Return the current locale. */
    static getLocale() {
        return chrome.i18n.getMessage('locale');
    }
    /**
     * Returns the message with the given message id from the ChromeVox namespace.
     *
     * If we can't find a message, throw an exception.  This allows us to catch
     * typos early.
     *
     * @param messageId The id.
     * @param subs Substitution strings.
     * @return The localized message.
     */
    static getMsg(messageId, subs) {
        let message = Msgs.Untranslated[messageId.toUpperCase()];
        if (message !== undefined) {
            return Msgs.applySubstitutions_(message, subs);
        }
        message = chrome.i18n.getMessage(Msgs.NAMESPACE_ + messageId, subs);
        if ((message === undefined || message === '') &&
            messageId.endsWith('_brl')) {
            // Braille string entries are optional. If we couldn't find a braille-
            // specific string, try again without the '_brl' suffix.
            message = chrome.i18n.getMessage(Msgs.NAMESPACE_ + messageId.replace('_brl', ''), subs);
        }
        if (message === undefined || message === '') {
            throw new Error('Invalid ChromeVox message id: ' + messageId);
        }
        return message;
    }
    /**
     * Returns the message with the given message ID, formatted for the given
     * count.
     * @param subs Substitution strings.
     * @return The localized and formatted message.
     */
    static getMsgWithCount(messageId, count) {
        const msg = Msgs.getMsg(messageId);
        const formatter = new MessageFormat(chrome.i18n.getUILanguage(), msg);
        return formatter.format({ COUNT: count }, () => { });
    }
    /**
     * Processes an HTML DOM, replacing text content with translated text messages
     * on elements marked up for translation.  Elements whose class attributes
     * contain the 'i18n' class name are expected to also have an msgid
     * attribute. The value of the msgid attributes are looked up as message
     * IDs and the resulting text is used as the text content of the elements.
     *
     * @param root The root node where the translation should be performed.
     */
    static addTranslatedMessagesToDom(root) {
        const elts = root.querySelectorAll('.i18n');
        for (let i = 0; i < elts.length; i++) {
            const msgid = elts[i].getAttribute('msgid');
            if (!msgid) {
                throw new Error('Element has no msgid attribute: ' + elts[i]);
            }
            const val = Msgs.getMsg(msgid);
            if (elts[i].tagName === 'INPUT') {
                elts[i].setAttribute('placeholder', val);
            }
            else {
                elts[i].textContent = val;
            }
            elts[i].classList.add('i18n-processed');
        }
    }
    /**
     * Returns a number formatted correctly.
     * @return The number in the correct locale.
     */
    static getNumber(num) {
        return '' + num;
    }
    /**
     * Applies substitutions of the form $N, where N is a number from 1 to 9, to a
     * string. The numbers are one-based indices into |opt_subs|.
     */
    static applySubstitutions_(message, subs) {
        if (subs) {
            for (let i = 0; i < subs.length; i++) {
                message = message.replace('$' + (i + 1), subs[i]);
            }
        }
        return message;
    }
}
(function (Msgs) {
    /** The namespace for all Chromevox messages. */
    Msgs.NAMESPACE_ = 'chromevox_';
    /**
     * Strings that are displayed in the user interface but don't need
     * be translated.
     */
    Msgs.Untranslated = {
        /** The unchecked state for a checkbox in braille. */
        CHECKBOX_UNCHECKED_STATE_BRL: '( )',
        /** The checked state for a checkbox in braille. */
        CHECKBOX_CHECKED_STATE_BRL: '(x)',
        /** The unselected state for a radio button in braille. */
        RADIO_UNSELECTED_STATE_BRL: '( )',
        /** The selected state for a radio button in braille. */
        RADIO_SELECTED_STATE_BRL: '(x)',
        /** Brailled after a menu if the menu has a submenu. */
        ARIA_HAS_SUBMENU_BRL: '->',
        /** Describes an element with the ARIA role option. */
        ROLE_OPTION: ' ',
        /** Braille of element with the ARIA role option. */
        ROLE_OPTION_BRL: ' ',
        /** Braille of element that is checked. */
        CHECKED_TRUE_BRL: '(x)',
        /** Braille of element that is unchecked. */
        CHECKED_FALSE_BRL: '( )',
        /** Braille of element where the checked state is mixed or indeterminate. */
        CHECKED_MIXED_BRL: '(-)',
        /** Braille of element with the ARIA attribute aria-disabled=true. */
        ARIA_DISABLED_TRUE_BRL: 'xx',
        /** Braille of element with the ARIA attribute aria-expanded=true. */
        ARIA_EXPANDED_TRUE_BRL: '-',
        /** Braille of element with the ARIA attribute aria-expanded=false. */
        ARIA_EXPANDED_FALSE_BRL: '+',
        /** Braille of element with the ARIA attribute aria-invalid=true. */
        ARIA_INVALID_TRUE_BRL: '!',
        /** Braille of element with the ARIA attribute aria-pressed=true. */
        ARIA_PRESSED_TRUE_BRL: '=',
        /** Braille of element with the ARIA attribute aria-pressed=false. */
        ARIA_PRESSED_FALSE_BRL: ' ',
        /** Braille of element with the ARIA attribute aria-pressed=mixed. */
        ARIA_PRESSED_MIXED_BRL: '-',
        /** Braille of element with the ARIA attribute aria-selected=true. */
        ARIA_SELECTED_TRUE_BRL: '(x)',
        /** Braille of element with the ARIA attribute aria-selected=false. */
        ARIA_SELECTED_FALSE_BRL: '( )',
        /** Brailled after a menu if it has a submenu. */
        HAS_SUBMENU_BRL: '->',
        /** Brailled to describe a <time> tag. */
        TAG_TIME_BRL: ' ',
        /** Spoken when describing an ARIA value. */
        ARIA_VALUE_NOW: '$1',
        /** Brailled when describing an ARIA value. */
        ARIA_VALUE_NOW_BRL: '$1',
        /** Spoken when describing an ARIA value text. */
        ARIA_VALUE_TEXT: '$1',
        /** Brailled when describing an ARIA value text. */
        ARIA_VALUE_TEXT_BRL: '$1',
    };
})(Msgs || (Msgs = {}));
TestImportManager.exportForTesting(Msgs);

// Copyright 2019 The Chromium Authors
// Use of this source code is governed by a BSD-style license that can be
// found in the LICENSE file.
/**
 * @fileoverview Provides locale output services for ChromeVox, which
 * uses language information to automatically change the TTS voice.
 * Please note: we use the term 'locale' to refer to language codes e.g.
 * 'en-US'. For more information on locales:
 * https://en.wikipedia.org/wiki/Locale_(computer_software)
 */
class LocaleOutputHelper {
    availableVoices_;
    currentLocale_;
    lastSpokenLocale_;
    static instance;
    constructor() {
        this.currentLocale_ = BROWSER_UI_LOCALE || '';
        this.lastSpokenLocale_ = this.currentLocale_;
        this.availableVoices_ = [];
        const setAvailableVoices = () => {
            chrome.tts.getVoices(voices => {
                this.availableVoices_ = voices || [];
            });
        };
        setAvailableVoices();
        if (window.speechSynthesis) {
            window.speechSynthesis.addEventListener('voiceschanged', setAvailableVoices, /* useCapture */ false);
        }
    }
    /**
     * Computes |this.currentLocale_| and |outputString|, and returns them.
     * @param contextNode The AutomationNode that owns |text|.
     */
    computeTextAndLocale(text, contextNode) {
        if (!text || !contextNode) {
            return { text, locale: BROWSER_UI_LOCALE };
        }
        // Prefer the node's detected locale and fall back on the author-assigned
        // locale.
        const nodeLocale = contextNode.detectedLanguage || contextNode.language || '';
        const newLocale = this.computeNewLocale_(nodeLocale);
        let outputString = text;
        const shouldAnnounce = this.shouldAnnounceLocale_(newLocale);
        if (this.hasVoiceForLocale_(newLocale)) {
            this.setCurrentLocale_(newLocale);
            if (shouldAnnounce) {
                this.lastSpokenLocale_ = newLocale;
                // Prepend the human-readable locale to |outputString|.
                const displayLanguage = chrome.accessibilityPrivate.getDisplayNameForLocale(newLocale /* Locale to translate */, newLocale /* Target locale */);
                outputString =
                    Msgs.getMsg('language_switch', [displayLanguage, outputString]);
            }
        }
        else {
            // Alert the user that no voice is available for |newLocale|.
            this.setCurrentLocale_(BROWSER_UI_LOCALE);
            const displayLanguage = chrome.accessibilityPrivate.getDisplayNameForLocale(newLocale /* Locale to translate */, BROWSER_UI_LOCALE /* Target locale */);
            outputString =
                Msgs.getMsg('voice_unavailable_for_language', [displayLanguage]);
        }
        return { text: outputString, locale: this.currentLocale_ };
    }
    computeNewLocale_(nodeLocale) {
        nodeLocale = nodeLocale.toLowerCase();
        if (LocaleOutputHelper.isValidLocale_(nodeLocale)) {
            return nodeLocale;
        }
        return BROWSER_UI_LOCALE;
    }
    hasVoiceForLocale_(targetLocale) {
        const components = targetLocale.split('-');
        if (!components || components.length === 0) {
            return false;
        }
        const targetLanguage = components[0];
        for (const voice of this.availableVoices_) {
            if (!voice.lang) {
                continue;
            }
            const candidateLanguage = voice.lang.toLowerCase().split('-')[0];
            if (candidateLanguage === targetLanguage) {
                return true;
            }
        }
        return false;
    }
    setCurrentLocale_(locale) {
        if (LocaleOutputHelper.isValidLocale_(locale)) {
            this.currentLocale_ = locale;
        }
    }
    shouldAnnounceLocale_(newLocale) {
        const [lastSpokenLanguage, lastSpokenCountry] = this.lastSpokenLocale_.split('-');
        const [newLanguage, newCountry] = newLocale.split('-');
        if (lastSpokenLanguage !== newLanguage) {
            return true;
        }
        if (!newCountry) {
            // If |newCountry| is undefined, then we don't want to announce the
            // locale. For example, we don't want to announce 'en-us' -> 'en'.
            return false;
        }
        return lastSpokenCountry !== newCountry;
    }
    // =============== Static Methods ==============
    /** Creates a singleton instance of LocaleOutputHelper. */
    static init() {
        if (LocaleOutputHelper.instance !== undefined) {
            throw new Error('LocaleOutputHelper is a singleton, can only initialize once');
        }
        LocaleOutputHelper.instance = new LocaleOutputHelper();
    }
    static isValidLocale_(locale) {
        return chrome.accessibilityPrivate.getDisplayNameForLocale(locale, locale) !== '';
    }
}
// Local to module.
const BROWSER_UI_LOCALE = chrome.i18n.getUILanguage().toLowerCase();
TestImportManager.exportForTesting(LocaleOutputHelper);

// Copyright 2015 The Chromium Authors
// Use of this source code is governed by a BSD-style license that can be
// found in the LICENSE file.
/**
 * @fileoverview Commands to pass from the ChromeVox background page context
 * to the ChromeVox Panel.
 */
/**
 * Create one command to pass to the ChromeVox Panel.
 */
class PanelCommand {
    type;
    data;
    constructor(type, data) {
        this.type = type;
        this.data = data;
    }
    getPanelWindow() {
        const views = chrome.extension.getViews();
        for (let i = 0; i < views.length; i++) {
            if (views[i]['location'].href.indexOf('panel/panel.html') > 0) {
                return views[i];
            }
        }
        throw new Error('Could not find the panel window');
    }
    waitForPanel() {
        return new Promise(resolve => {
            const panelWindow = this.getPanelWindow();
            if (panelWindow.document.readyState === 'complete') {
                // The panel may already have loaded. In this case, resolve() and
                // do not wait for a load event that has already fired.
                resolve();
            }
            panelWindow.addEventListener('load', () => {
                resolve();
            });
        });
    }
    /** Send this command to the ChromeVox Panel window. */
    async send() {
        // Do not send commands to the ChromeVox Panel window until it has finished
        // loading and is ready to receive commands.
        await this.waitForPanel();
        const panelWindow = this.getPanelWindow();
        panelWindow.postMessage(JSON.stringify(this), window.location.origin);
    }
}
var PanelCommandType;
(function (PanelCommandType) {
    PanelCommandType["CLEAR_SPEECH"] = "clear_speech";
    PanelCommandType["ADD_NORMAL_SPEECH"] = "add_normal_speech";
    PanelCommandType["ADD_ANNOTATION_SPEECH"] = "add_annotation_speech";
    PanelCommandType["CLOSE_CHROMEVOX"] = "close_chromevox";
    PanelCommandType["UPDATE_BRAILLE"] = "update_braille";
    PanelCommandType["OPEN_MENUS"] = "open_menus";
    PanelCommandType["OPEN_MENUS_MOST_RECENT"] = "open_menus_most_recent";
    PanelCommandType["SEARCH"] = "search";
    PanelCommandType["TUTORIAL"] = "tutorial";
    PanelCommandType["ENABLE_TEST_HOOKS"] = "enable_test_hooks";
})(PanelCommandType || (PanelCommandType = {}));
TestImportManager.exportForTesting(PanelCommand, ['PanelCommandType', PanelCommandType]);

// Copyright 2023 The Chromium Authors
// Use of this source code is governed by a BSD-style license that can be
// found in the LICENSE file.
/**
 * List of commands. A new command needs to be added to the end of this list.
 * These values and indices are persisted to logs. Entries should not be
 * reordered and their indices should never be reused.
 */
var Command;
(function (Command) {
    // LINT.IfChange
    Command["ANNOUNCE_BATTERY_DESCRIPTION"] = "announceBatteryDescription";
    Command["ANNOUNCE_HEADERS"] = "announceHeaders";
    Command["ANNOUNCE_RICH_TEXT_DESCRIPTION"] = "announceRichTextDescription";
    Command["AUTORUNNER"] = "autorunner";
    Command["BACKWARD"] = "backward";
    Command["BOTTOM"] = "bottom";
    Command["CONTEXT_MENU"] = "contextMenu";
    Command["COPY"] = "copy";
    Command["CYCLE_PUNCTUATION_ECHO"] = "cyclePunctuationEcho";
    Command["CYCLE_TYPING_ECHO"] = "cycleTypingEcho";
    Command["DEBUG"] = "debug";
    Command["DECREASE_TTS_PITCH"] = "decreaseTtsPitch";
    Command["DECREASE_TTS_RATE"] = "decreaseTtsRate";
    Command["DECREASE_TTS_VOLUME"] = "decreaseTtsVolume";
    Command["DISABLE_CHROMEVOX_ARC_SUPPORT_FOR_CURRENT_APP"] = "disableChromeVoxArcSupportForCurrentApp";
    Command["DISABLE_LOGGING"] = "disableLogging";
    Command["DUMP_TREE"] = "dumpTree";
    Command["ENABLE_CHROMEVOX_ARC_SUPPORT_FOR_CURRENT_APP"] = "enableChromeVoxArcSupportForCurrentApp";
    Command["ENABLE_CONSOLE_TTS"] = "enableConsoleTts";
    Command["ENABLE_LOGGING"] = "enableLogging";
    Command["ENTER_SHIFTER"] = "enterShifter";
    Command["EXIT_SHIFTER"] = "exitShifter";
    Command["EXIT_SHIFTER_CONTENT"] = "exitShifterContent";
    Command["FORCE_CLICK_ON_CURRENT_ITEM"] = "forceClickOnCurrentItem";
    Command["FORCE_DOUBLE_CLICK_ON_CURRENT_ITEM"] = "forceDoubleClickOnCurrentItem";
    Command["FORCE_LONG_CLICK_ON_CURRENT_ITEM"] = "forceLongClickOnCurrentItem";
    Command["FORWARD"] = "forward";
    Command["FULLY_DESCRIBE"] = "fullyDescribe";
    Command["GO_TO_COL_FIRST_CELL"] = "goToColFirstCell";
    Command["GO_TO_COL_LAST_CELL"] = "goToColLastCell";
    Command["GO_TO_FIRST_CELL"] = "goToFirstCell";
    Command["GO_TO_LAST_CELL"] = "goToLastCell";
    Command["GO_TO_ROW_FIRST_CELL"] = "goToRowFirstCell";
    Command["GO_TO_ROW_LAST_CELL"] = "goToRowLastCell";
    Command["HANDLE_TAB"] = "handleTab";
    Command["HANDLE_TAB_PREV"] = "handleTabPrev";
    Command["HELP"] = "help";
    Command["INCREASE_TTS_PITCH"] = "increaseTtsPitch";
    Command["INCREASE_TTS_RATE"] = "increaseTtsRate";
    Command["INCREASE_TTS_VOLUME"] = "increaseTtsVolume";
    Command["JUMP_TO_BOTTOM"] = "jumpToBottom";
    Command["JUMP_TO_DETAILS"] = "jumpToDetails";
    Command["JUMP_TO_TOP"] = "jumpToTop";
    Command["LEFT"] = "left";
    Command["LINE_DOWN"] = "lineDown";
    Command["LINE_UP"] = "lineUp";
    Command["MOVE_TO_END_OF_LINE"] = "moveToEndOfLine";
    Command["MOVE_TO_START_OF_LINE"] = "moveToStartOfLine";
    Command["NOP"] = "nop";
    Command["NATIVE_NEXT_CHARACTER"] = "nativeNextCharacter";
    Command["NATIVE_NEXT_WORD"] = "nativeNextWord";
    Command["NATIVE_PREVIOUS_CHARACTER"] = "nativePreviousCharacter";
    Command["NATIVE_PREVIOUS_WORD"] = "nativePreviousWord";
    Command["NEXT_ARTICLE"] = "nextArticle";
    Command["NEXT_AT_GRANULARITY"] = "nextAtGranularity";
    Command["NEXT_BUTTON"] = "nextButton";
    Command["NEXT_CHARACTER"] = "nextCharacter";
    Command["NEXT_CHECKBOX"] = "nextCheckbox";
    Command["NEXT_COL"] = "nextCol";
    Command["NEXT_COMBO_BOX"] = "nextComboBox";
    Command["NEXT_CONTROL"] = "nextControl";
    Command["NEXT_EDIT_TEXT"] = "nextEditText";
    Command["NEXT_FORM_FIELD"] = "nextFormField";
    Command["NEXT_GRANULARITY"] = "nextGranularity";
    Command["NEXT_GRAPHIC"] = "nextGraphic";
    Command["NEXT_GROUP"] = "nextGroup";
    Command["NEXT_HEADING"] = "nextHeading";
    Command["NEXT_HEADING_1"] = "nextHeading1";
    Command["NEXT_HEADING_2"] = "nextHeading2";
    Command["NEXT_HEADING_3"] = "nextHeading3";
    Command["NEXT_HEADING_4"] = "nextHeading4";
    Command["NEXT_HEADING_5"] = "nextHeading5";
    Command["NEXT_HEADING_6"] = "nextHeading6";
    Command["NEXT_INVALID_ITEM"] = "nextInvalidItem";
    Command["NEXT_LANDMARK"] = "nextLandmark";
    Command["NEXT_LINE"] = "nextLine";
    Command["NEXT_LINK"] = "nextLink";
    Command["NEXT_LIST"] = "nextList";
    Command["NEXT_LIST_ITEM"] = "nextListItem";
    Command["NEXT_MATH"] = "nextMath";
    Command["NEXT_MEDIA"] = "nextMedia";
    Command["NEXT_OBJECT"] = "nextObject";
    Command["NEXT_PAGE"] = "nextPage";
    Command["NEXT_RADIO"] = "nextRadio";
    Command["NEXT_ROW"] = "nextRow";
    Command["NEXT_SECTION"] = "nextSection";
    Command["NEXT_SENTENCE"] = "nextSentence";
    Command["NEXT_SIMILAR_ITEM"] = "nextSimilarItem";
    Command["NEXT_SLIDER"] = "nextSlider";
    Command["NEXT_TABLE"] = "nextTable";
    Command["NEXT_VISITED_LINK"] = "nextVisitedLink";
    Command["NEXT_WORD"] = "nextWord";
    Command["OPEN_CHROMEVOX_MENUS"] = "openChromeVoxMenus";
    Command["OPEN_KEYBOARD_SHORTCUTS"] = "openKeyboardShortcuts";
    Command["OPEN_LONG_DESC"] = "openLongDesc";
    Command["PAN_LEFT"] = "panLeft";
    Command["PAN_RIGHT"] = "panRight";
    Command["PASS_THROUGH_MODE"] = "passThroughMode";
    Command["PAUSE_ALL_MEDIA"] = "pauseAllMedia";
    Command["PREVIOUS_ARTICLE"] = "previousArticle";
    Command["PREVIOUS_AT_GRANULARITY"] = "previousAtGranularity";
    Command["PREVIOUS_BUTTON"] = "previousButton";
    Command["PREVIOUS_CHARACTER"] = "previousCharacter";
    Command["PREVIOUS_CHECKBOX"] = "previousCheckbox";
    Command["PREVIOUS_COMBO_BOX"] = "previousComboBox";
    Command["PREVIOUS_COL"] = "previousCol";
    Command["PREVIOUS_CONTROL"] = "previousControl";
    Command["PREVIOUS_EDIT_TEXT"] = "previousEditText";
    Command["PREVIOUS_FORM_FIELD"] = "previousFormField";
    Command["PREVIOUS_GRANULARITY"] = "previousGranularity";
    Command["PREVIOUS_GRAPHIC"] = "previousGraphic";
    Command["PREVIOUS_GROUP"] = "previousGroup";
    Command["PREVIOUS_HEADING"] = "previousHeading";
    Command["PREVIOUS_HEADING_1"] = "previousHeading1";
    Command["PREVIOUS_HEADING_2"] = "previousHeading2";
    Command["PREVIOUS_HEADING_3"] = "previousHeading3";
    Command["PREVIOUS_HEADING_4"] = "previousHeading4";
    Command["PREVIOUS_HEADING_5"] = "previousHeading5";
    Command["PREVIOUS_HEADING_6"] = "previousHeading6";
    Command["PREVIOUS_INVALID_ITEM"] = "previousInvalidItem";
    Command["PREVIOUS_LANDMARK"] = "previousLandmark";
    Command["PREVIOUS_LINE"] = "previousLine";
    Command["PREVIOUS_LINK"] = "previousLink";
    Command["PREVIOUS_LIST"] = "previousList";
    Command["PREVIOUS_LIST_ITEM"] = "previousListItem";
    Command["PREVIOUS_MATH"] = "previousMath";
    Command["PREVIOUS_MEDIA"] = "previousMedia";
    Command["PREVIOUS_OBJECT"] = "previousObject";
    Command["PREVIOUS_PAGE"] = "previousPage";
    Command["PREVIOUS_RADIO"] = "previousRadio";
    Command["PREVIOUS_ROW"] = "previousRow";
    Command["PREVIOUS_SECTION"] = "previousSection";
    Command["PREVIOUS_SENTENCE"] = "previousSentence";
    Command["PREVIOUS_SIMILAR_ITEM"] = "previousSimilarItem";
    Command["PREVIOUS_SLIDER"] = "previousSlider";
    Command["PREVIOUS_TABLE"] = "previousTable";
    Command["PREVIOUS_VISITED_LINK"] = "previousVisitedLink";
    Command["PREVIOUS_WORD"] = "previousWord";
    Command["READ_CURRENT_TITLE"] = "readCurrentTitle";
    Command["READ_CURRENT_URL"] = "readCurrentURL";
    Command["READ_FROM_HERE"] = "readFromHere";
    Command["READ_LINK_URL"] = "readLinkURL";
    Command["READ_PHONETIC_PRONUNCIATION"] = "readPhoneticPronunciation";
    Command["REPORT_ISSUE"] = "reportIssue";
    Command["RESET_TEXT_TO_SPEECH_SETTINGS"] = "resetTextToSpeechSettings";
    Command["RIGHT"] = "right";
    Command["ROUTING"] = "routing";
    Command["SCROLL_BACKWARD"] = "scrollBackward";
    Command["SCROLL_FORWARD"] = "scrollForward";
    Command["SHOW_ACTIONS_MENU"] = "showActionsMenu";
    Command["SHOW_FORMS_LIST"] = "showFormsList";
    Command["SHOW_HEADINGS_LIST"] = "showHeadingsList";
    Command["SHOW_LANDMARKS_LIST"] = "showLandmarksList";
    Command["SHOW_LEARN_MODE_PAGE"] = "showLearnModePage";
    Command["SHOW_LINKS_LIST"] = "showLinksList";
    Command["SHOW_LOG_PAGE"] = "showLogPage";
    Command["SHOW_OPTIONS_PAGE"] = "showOptionsPage";
    Command["SHOW_PANEL_MENU_MOST_RECENT"] = "showPanelMenuMostRecent";
    Command["SHOW_TABLES_LIST"] = "showTablesList";
    Command["SHOW_TALKBACK_KEYBOARD_SHORTCUTS"] = "showTalkBackKeyboardShortcuts";
    Command["SHOW_TTS_SETTINGS"] = "showTtsSettings";
    Command["SPEAK_TABLE_LOCATION"] = "speakTableLocation";
    Command["SPEAK_TIME_AND_DATE"] = "speakTimeAndDate";
    Command["START_HISTORY_RECORDING"] = "startHistoryRecording";
    Command["STOP_HISTORY_RECORDING"] = "stopHistoryRecording";
    Command["STOP_SPEECH"] = "stopSpeech";
    Command["TOGGLE_BRAILLE_CAPTIONS"] = "toggleBrailleCaptions";
    Command["TOGGLE_BRAILLE_TABLE"] = "toggleBrailleTable";
    Command["TOGGLE_DICTATION"] = "toggleDictation";
    Command["TOGGLE_EARCONS"] = "toggleEarcons";
    Command["TOGGLE_KEYBOARD_HELP"] = "toggleKeyboardHelp";
    Command["TOGGLE_SCREEN"] = "toggleScreen";
    Command["TOGGLE_SEARCH_WIDGET"] = "toggleSearchWidget";
    Command["TOGGLE_SELECTION"] = "toggleSelection";
    Command["TOGGLE_SEMANTICS"] = "toggleSemantics";
    Command["TOGGLE_SPEECH_ON_OR_OFF"] = "toggleSpeechOnOrOff";
    Command["TOGGLE_STICKY_MODE"] = "toggleStickyMode";
    Command["TOP"] = "top";
    Command["VIEW_GRAPHIC_AS_BRAILLE"] = "viewGraphicAsBraille";
    Command["TOGGLE_CAPTIONS"] = "toggleCaptions";
    // Add a new command to the end of this list.
    // LINT.ThenChange(//tools/metrics/histograms/metadata/accessibility/enums.xml)
})(Command || (Command = {}));
/**
 * List of categories for the commands.
 * Note that the values here must correspond to the message resource tag for the
 * category.
 */
var CommandCategory;
(function (CommandCategory) {
    CommandCategory["ACTIONS"] = "actions";
    CommandCategory["CONTROLLING_SPEECH"] = "controlling_speech";
    CommandCategory["HELP_COMMANDS"] = "help_commands";
    CommandCategory["INFORMATION"] = "information";
    CommandCategory["JUMP_COMMANDS"] = "jump_commands";
    CommandCategory["MODIFIER_KEYS"] = "modifier_keys";
    CommandCategory["NAVIGATION"] = "navigation";
    CommandCategory["OVERVIEW"] = "overview";
    CommandCategory["TABLES"] = "tables";
    // The following categories are not displayed in the ChromeVox menus:
    CommandCategory["BRAILLE"] = "braille";
    CommandCategory["DEVELOPER"] = "developer";
    CommandCategory["NO_CATEGORY"] = "no_category";
})(CommandCategory || (CommandCategory = {}));

// Copyright 2014 The Chromium Authors
// Use of this source code is governed by a BSD-style license that can be
// found in the LICENSE file.
/**
 * @fileoverview A JavaScript class that represents a sequence of keys entered
 * by the user.
 */
/**
 * A class to represent a sequence of keys entered by a user or affiliated with
 * a ChromeVox command.
 * This class can represent the data from both types of key sequences:
 * COMMAND KEYS SPECIFIED IN A KEYMAP:
 * - Two discrete keys (at most): [Down arrow], [A, A] or [O, W] etc. Can
 *   specify one or both.
 * - Modifiers (like ctrl, alt, meta, etc)
 * - Whether or not the ChromeVox modifier key is required with the command.
 * USER INPUT:
 * - Two discrete keys (at most): [Down arrow], [A, A] or [O, W] etc.
 * - Modifiers (like ctlr, alt, meta, etc)
 * - Whether or not the ChromeVox modifier key was active when the keys were
 *   entered.
 * - Whether or not a prefix key was entered before the discrete keys.
 * - Whether sticky mode was active.
 */
class KeySequence {
    cvoxModifier;
    doubleTap;
    requireStickyMode;
    prefixKey;
    skipStripping;
    stickyMode;
    /**
     * Stores the key codes and modifiers for the keys in the key sequence.
     * TODO(rshearer): Consider making this structure an array of minimal
     * keyEvent-like objects instead so we don't have to worry about what
     * happens when ctrlKey.length is different from altKey.length.
     *
     * NOTE: If a modifier key is pressed by itself, we will store the keyCode
     * *and* set the appropriate modKey to be true. This mirrors the way key
     * events are created on Mac and Windows. For example, if the Meta key was
     * pressed by itself, the keys object will have:
     * {metaKey: [true], keyCode:[91]}
     */
    keys = {
        ctrlKey: [],
        searchKeyHeld: [],
        altKey: [],
        altGraphKey: [],
        shiftKey: [],
        metaKey: [],
        keyCode: [],
    };
    /**
     * @param originalEvent The original key event entered by a user.
     * The originalEvent may or may not have parameters stickyMode and keyPrefix
     * specified. We will also accept an event-shaped object.
     * @param cvoxModifier Whether or not the ChromeVox modifier key is active.
     * If not specified, we will try to determine whether the modifier was active
     * by looking at the originalEvent from key events when the cvox modifiers
     * are set. Defaults to false.
     * @param doubleTap Whether this is triggered via double tap.
     * @param skipStripping Whether to strip cvox modifiers.
     * @param requireStickyMode Whether to require sticky mode.
     */
    constructor(originalEvent, cvoxModifier, doubleTap, skipStripping, requireStickyMode) {
        this.doubleTap = Boolean(doubleTap);
        this.requireStickyMode = Boolean(requireStickyMode);
        this.skipStripping = Boolean(skipStripping);
        this.cvoxModifier =
            cvoxModifier ?? this.isCVoxModifierActive(originalEvent);
        this.stickyMode = Boolean(originalEvent.stickyMode);
        this.prefixKey = Boolean(originalEvent.keyPrefix);
        if (this.stickyMode && this.prefixKey) {
            throw 'Prefix key and sticky mode cannot both be enabled: ' +
                originalEvent;
        }
        // TODO (rshearer): We should take the user out of sticky mode if they
        // try to use the CVox modifier or prefix key.
        this.extractKey_(originalEvent);
    }
    /**
     * Adds an additional key onto the original sequence, for use when the user
     * is entering two shortcut keys. This happens when the user presses a key,
     * releases it, and then presses a second key. Those two keys together are
     * considered part of the sequence.
     * @param additionalKeyEvent The additional key to be added to
     * the original event. Should be an event or an event-shaped object.
     * @return Whether or not we were able to add a key. Returns false
     * if there are already two keys attached to this event.
     */
    addKeyEvent(additionalKeyEvent) {
        if (this.keys.keyCode.length > 1) {
            return false;
        }
        this.extractKey_(additionalKeyEvent);
        return true;
    }
    /**
     * Check for equality. Commands are matched based on the actual key codes
     * involved and on whether or not they both require a ChromeVox modifier key.
     *
     * If sticky mode or a prefix is active on one of the commands but not on
     * the other, then we try and match based on key code first.
     * - If both commands have the same key code and neither of them have the
     * ChromeVox modifier active then we have a match.
     * - Next we try and match with the ChromeVox modifier. If both commands have
     * the same key code, and one of them has the ChromeVox modifier and the other
     * has sticky mode or an active prefix, then we also have a match.
     */
    equals(rhs) {
        // Check to make sure the same keys with the same modifiers were pressed.
        if (!this.checkKeyEquality_(rhs)) {
            return false;
        }
        if (this.doubleTap !== rhs.doubleTap) {
            return false;
        }
        // So now we know the actual keys are the same.
        // If one key sequence requires sticky mode, return early the strict
        // sticky mode state.
        if (this.requireStickyMode || rhs.requireStickyMode) {
            return (this.stickyMode || rhs.stickyMode) && !this.cvoxModifier &&
                !rhs.cvoxModifier;
        }
        // If they both have the ChromeVox modifier, or they both don't have the
        // ChromeVox modifier, then they are considered equal.
        if (this.cvoxModifier === rhs.cvoxModifier) {
            return true;
        }
        // So only one of them has the ChromeVox modifier. If the one that doesn't
        // have the ChromeVox modifier has sticky mode or the prefix key then the
        // keys are still considered equal.
        const unmodified = this.cvoxModifier ? rhs : this;
        return unmodified.stickyMode || unmodified.prefixKey;
    }
    /**
     * Utility method that extracts the key code and any modifiers from a given
     * event and adds them to the object map.
     * @param keyEvent The keyEvent or event-shaped object to extract from.
     */
    extractKey_(keyEvent) {
        let keyCode;
        // TODO (rshearer): This is temporary until we find a library that can
        // convert between ASCII charcodes and keycodes.
        if (keyEvent.type === 'keypress' && keyEvent.keyCode >= 97 &&
            keyEvent.keyCode <= 122) {
            // Alphabetic keypress. Convert to the upper case ASCII code.
            keyCode = keyEvent.keyCode - 32;
        }
        else if (keyEvent.type === 'keypress') {
            keyCode = KEY_PRESS_CODE[keyEvent.keyCode];
        }
        this.keys.keyCode.push(keyCode || keyEvent.keyCode);
        for (const prop in this.keys) {
            if (prop !== 'keyCode') {
                if (this.isKeyModifierActive(keyEvent, prop)) {
                    this.keys[prop].push(true);
                }
                else {
                    this.keys[prop].push(false);
                }
            }
        }
        if (this.cvoxModifier) {
            this.rationalizeKeys_();
        }
    }
    /**
     * Rationalizes the key codes and the ChromeVox modifier for this keySequence.
     * This means we strip out the key codes and key modifiers stored for this
     * KeySequence that are also present in the ChromeVox modifier. For example,
     * if the ChromeVox modifier keys are Ctrl+Alt, and we've determined that the
     * ChromeVox modifier is active (meaning the user has pressed Ctrl+Alt), we
     * don't want this.keys.ctrlKey = true also because that implies that this
     * KeySequence involves the ChromeVox modifier and the ctrl key being held
     * down together, which doesn't make any sense.
     */
    rationalizeKeys_() {
        if (this.skipStripping) {
            return;
        }
        // TODO (rshearer): This is a hack. When the modifier key becomes
        // customizable then we will not have to deal with strings here.
        const modifierKeyCombo = KeySequence.modKeyStr.split(/\+/g);
        const index = this.keys.keyCode.length - 1;
        // For each modifier that is part of the CVox modifier, remove it from keys.
        if (modifierKeyCombo.indexOf('Ctrl') !== -1) {
            // TODO(b/314203187): Not null asserted, check these to make sure this is
            // correct.
            this.keys.ctrlKey[index] = false;
        }
        if (modifierKeyCombo.indexOf('Alt') !== -1) {
            // TODO(b/314203187): Not null asserted, check these to make sure this is
            // correct.
            this.keys.altKey[index] = false;
        }
        if (modifierKeyCombo.indexOf('Shift') !== -1) {
            // TODO(b/314203187): Not null asserted, check these to make sure this is
            // correct.
            this.keys.shiftKey[index] = false;
        }
        const metaKeyName = this.getMetaKeyName_();
        if (modifierKeyCombo.indexOf(metaKeyName) !== -1) {
            if (metaKeyName === 'Search') {
                // TODO(b/314203187): Not null asserted, check these to make sure this
                // is correct.
                this.keys.searchKeyHeld[index] = false;
                this.keys.metaKey[index] = false;
            }
            else if (metaKeyName === 'Cmd' || metaKeyName === 'Win') {
                this.keys.metaKey[index] = false;
            }
        }
    }
    /**
     * Get the user-facing name for the meta key (keyCode = 91), which varies
     * depending on the platform.
     * @return The user-facing string name for the meta key.
     */
    getMetaKeyName_() {
        return 'Search';
    }
    /**
     * Utility method that checks for equality of the modifiers (like shift and
     * alt) and the equality of key codes.
     * @return True if the modifiers and key codes in the key sequence are the
     * same.
     */
    checkKeyEquality_(rhs) {
        for (const i in this.keys) {
            // TODO(b/314203187): Not null asserted, check these to make sure this is
            // correct.
            for (let j = this.keys[i].length; j--;) {
                if (this.keys[i][j] !== rhs.keys[i][j]) {
                    return false;
                }
            }
        }
        return true;
    }
    getFirstKeyCode() {
        return this.keys.keyCode[0];
    }
    /**
     * Gets the number of keys in the sequence. Should be 1 or 2.
     * @return The number of keys in the sequence.
     */
    length() {
        return this.keys.keyCode.length;
    }
    /**
     * Checks if the specified key code represents a modifier key, i.e. Ctrl, Alt,
     * Shift, Search (on ChromeOS) or Meta.
     */
    isModifierKey(keyCode) {
        // Shift, Ctrl, Alt, Search/LWin
        return keyCode === KeyCode.SHIFT || keyCode === KeyCode.CONTROL ||
            keyCode === KeyCode.ALT || keyCode === KeyCode.SEARCH ||
            keyCode === KeyCode.APPS;
    }
    /**
     * Determines whether the Cvox modifier key is active during the keyEvent.
     * @param keyEvent The keyEvent or event-shaped object to check.
     * @return Whether or not the modifier key was active during the keyEvent.
     */
    isCVoxModifierActive(keyEvent) {
        // TODO (rshearer): Update this when the modifier key becomes customizable
        let modifierKeyCombo = KeySequence.modKeyStr.split(/\+/g);
        // For each modifier that is held down, remove it from the combo.
        // If the combo string becomes empty, then the user has activated the combo.
        if (this.isKeyModifierActive(keyEvent, 'ctrlKey')) {
            modifierKeyCombo =
                modifierKeyCombo.filter(modifier => modifier !== 'Ctrl');
        }
        if (this.isKeyModifierActive(keyEvent, 'altKey')) {
            modifierKeyCombo =
                modifierKeyCombo.filter(modifier => modifier !== 'Alt');
        }
        if (this.isKeyModifierActive(keyEvent, 'shiftKey')) {
            modifierKeyCombo =
                modifierKeyCombo.filter(modifier => modifier !== 'Shift');
        }
        if (this.isKeyModifierActive(keyEvent, 'metaKey') ||
            this.isKeyModifierActive(keyEvent, 'searchKeyHeld')) {
            const metaKeyName = this.getMetaKeyName_();
            modifierKeyCombo =
                modifierKeyCombo.filter(modifier => modifier !== metaKeyName);
        }
        return (modifierKeyCombo.length === 0);
    }
    /**
     * Determines whether a particular key modifier (for example, ctrl or alt) is
     * active during the keyEvent.
     * @param keyEvent The keyEvent or Event-shaped object to check.
     * @param modifier The modifier to check.
     * @return Whether or not the modifier key was active during the keyEvent.
     */
    isKeyModifierActive(keyEvent, modifier) {
        // We need to check the key event modifier and the keyCode because Linux
        // will not set the keyEvent.modKey property if it is the modKey by itself.
        // This bug filed as crbug.com/74044
        switch (modifier) {
            case 'ctrlKey':
                return (keyEvent.ctrlKey || keyEvent.keyCode === KeyCode.CONTROL);
            case 'altKey':
                return (keyEvent.altKey || (keyEvent.keyCode === KeyCode.ALT));
            case 'shiftKey':
                return (keyEvent.shiftKey || (keyEvent.keyCode === KeyCode.SHIFT));
            case 'metaKey':
                return (keyEvent.metaKey || (keyEvent.keyCode === KeyCode.SEARCH));
            case 'searchKeyHeld':
                // TODO(b/314203187): Not null asserted, check that this is correct.
                return keyEvent.keyCode === KeyCode.SEARCH ||
                    keyEvent['searchKeyHeld'];
        }
        return false;
    }
    isAnyModifierActive() {
        for (const modifierType in this.keys) {
            for (let i = 0; i < this.length(); i++) {
                // TODO(b/314203187): Not null asserted, check that this is correct.
                if (this.keys[modifierType][i] && modifierType !== 'keyCode') {
                    return true;
                }
            }
        }
        return false;
    }
    /** Creates a KeySequence event from a generic object. */
    static deserialize(sequenceObject) {
        const firstSequenceEvent = newEventLikeObject();
        firstSequenceEvent['stickyMode'] =
            (sequenceObject.stickyMode === undefined) ? false :
                sequenceObject.stickyMode;
        firstSequenceEvent['prefixKey'] = (sequenceObject.prefixKey === undefined) ?
            false :
            sequenceObject.prefixKey;
        const secondKeyPressed = sequenceObject.keys.keyCode.length > 1;
        const secondSequenceEvent = newEventLikeObject();
        for (const keyPressed in sequenceObject.keys) {
            // TODO(b/314203187): Not null asserted, check that this is correct.
            firstSequenceEvent[keyPressed] = sequenceObject.keys[keyPressed][0];
            if (secondKeyPressed) {
                secondSequenceEvent[keyPressed] = sequenceObject.keys[keyPressed][1];
            }
        }
        const skipStripping = sequenceObject.skipStripping !== undefined ?
            sequenceObject.skipStripping :
            true;
        const keySeq = new KeySequence(firstSequenceEvent, sequenceObject.cvoxModifier, sequenceObject.doubleTap, skipStripping, sequenceObject.requireStickyMode);
        if (secondKeyPressed) {
            KeySequence.sequenceSwitchKeyCodes.push(new KeySequence(firstSequenceEvent, sequenceObject.cvoxModifier));
            keySeq.addKeyEvent(secondSequenceEvent);
        }
        if (sequenceObject.doubleTap) {
            KeySequence.doubleTapCache.push(keySeq);
        }
        return keySeq;
    }
    /**
     * Creates a KeySequence event from a given string. The string should be in
     * the standard key sequence format described in keyUtil.keySequenceToString
     * and used in the key map JSON files.
     * @param keyStr The string representation of a key sequence.
     * @return The created KeySequence object.
     */
    static fromStr(keyStr) {
        const sequenceEvent = newEventLikeObject();
        const secondSequenceEvent = newEventLikeObject();
        let secondKeyPressed;
        if (keyStr.indexOf('>') === -1) {
            secondKeyPressed = false;
        }
        else {
            secondKeyPressed = true;
        }
        let cvoxPressed = false;
        sequenceEvent['stickyMode'] = false;
        sequenceEvent['prefixKey'] = false;
        const tokens = keyStr.split('+');
        for (let i = 0; i < tokens.length; i++) {
            const seqs = tokens[i].split('>');
            for (let j = 0; j < seqs.length; j++) {
                if (seqs[j].charAt(0) === '#') {
                    const keyCode = parseInt(seqs[j].substr(1), 10);
                    if (j > 0) {
                        secondSequenceEvent['keyCode'] = keyCode;
                    }
                    else {
                        sequenceEvent['keyCode'] = keyCode;
                    }
                }
                const keyName = seqs[j];
                if (seqs[j].length === 1) {
                    // Key is A/B/C...1/2/3 and we don't need to worry about setting
                    // modifiers.
                    if (j > 0) {
                        secondSequenceEvent['keyCode'] = seqs[j].charCodeAt(0);
                    }
                    else {
                        sequenceEvent['keyCode'] = seqs[j].charCodeAt(0);
                    }
                }
                else {
                    // Key is a modifier key
                    if (j > 0) {
                        KeySequence.setModifiersOnEvent_(keyName, secondSequenceEvent);
                        if (keyName === 'Cvox') {
                            cvoxPressed = true;
                        }
                    }
                    else {
                        KeySequence.setModifiersOnEvent_(keyName, sequenceEvent);
                        if (keyName === 'Cvox') {
                            cvoxPressed = true;
                        }
                    }
                }
            }
        }
        const keySeq = new KeySequence(sequenceEvent, cvoxPressed);
        if (secondKeyPressed) {
            keySeq.addKeyEvent(secondSequenceEvent);
        }
        return keySeq;
    }
    /**
     * Utility method for populating the modifiers on an event object that will be
     * used to create a KeySequence.
     * @param keyName A particular modifier key name (such as 'Ctrl').
     * @param seqEvent The event to populate.
     */
    static setModifiersOnEvent_(keyName, seqEvent) {
        if (keyName === 'Ctrl') {
            seqEvent['ctrlKey'] = true;
            seqEvent['keyCode'] = KeyCode.CONTROL;
        }
        else if (keyName === 'Alt') {
            seqEvent['altKey'] = true;
            seqEvent['keyCode'] = KeyCode.ALT;
        }
        else if (keyName === 'Shift') {
            seqEvent['shiftKey'] = true;
            seqEvent['keyCode'] = KeyCode.SHIFT;
        }
        else if (keyName === 'Search') {
            seqEvent['searchKeyHeld'] = true;
            seqEvent['keyCode'] = KeyCode.SEARCH;
        }
        else if (keyName === 'Cmd') {
            seqEvent['metaKey'] = true;
            seqEvent['keyCode'] = KeyCode.SEARCH;
        }
        else if (keyName === 'Win') {
            seqEvent['metaKey'] = true;
            seqEvent['keyCode'] = KeyCode.SEARCH;
        }
        else if (keyName === 'Insert') {
            seqEvent['keyCode'] = KeyCode.INSERT;
        }
    }
    /**
     * A cache of all key sequences that have been set as double-tappable. We need
     * this cache because repeated key down computations causes ChromeVox to
     * become less responsive. This list is small so we currently use an array.
     */
    static doubleTapCache = [];
    /**
     * If any of these keys is pressed with the modifier key, we go in sequence
     * mode where the subsequent independent key downs (while modifier keys are
     * down) are a part of the same shortcut.
     */
    static sequenceSwitchKeyCodes = [];
    static modKeyStr = 'Search';
}
// Private to module.
function newEventLikeObject() {
    return Object.assign({}, { type: '', keyCode: 0 });
}
// TODO(dtseng): This is incomplete; pull once we have appropriate libs.
/** Maps a keypress keycode to a keydown or keyup keycode. */
const KEY_PRESS_CODE = {
    39: 222,
    44: 188,
    45: 189,
    46: 190,
    47: 191,
    59: 186,
    91: 219,
    92: 220,
    93: 221,
};
TestImportManager.exportForTesting(KeySequence);

// Copyright 2014 The Chromium Authors
// Use of this source code is governed by a BSD-style license that can be
// found in the LICENSE file.
/**
 * @fileoverview This class acts as the persistent store for all static data
 * about commands.
 *
 * This store can safely be used within either a content or background script
 * context.
 *
 * If you are looking to add a user command, follow the below steps for best
 * integration with existing components:
 * 1. Add the command to the |Command| enum in command.js.
 * 2. Add a command below in COMMAND_DATA. Fill in each of the relevant JSON
 * keys.
 * Be sure to add a msg id and define it in chromevox/mv2/messages/messages.js which
 * describes the command. Please also add a category msg id so that the command
 * will show up in the options page.
 * 2. Add the command's logic to CommandHandler inside of our switch-based
 * dispatch method (onCommand).
 * 3. Add a key binding to KeySequence.
 */
class CommandStore {
    /**
     * Gets a message given a command.
     * @param command The command to query.
     * @return The message id, if any.
     */
    static messageForCommand(command) {
        return COMMAND_DATA[command]?.msgId;
    }
    /**
     * Gets a category given a command.
     * @param command The command to query.
     * @return The category, if any.
     */
    static categoryForCommand(command) {
        return COMMAND_DATA[command]?.category;
    }
    /**
     * Gets the first command associated with the message id
     * @return The command, if any.
     */
    static commandForMessage(msgId) {
        for (const commandName in COMMAND_DATA) {
            const command = COMMAND_DATA[commandName];
            if (command.msgId === msgId) {
                return commandName;
            }
        }
    }
    /**
     * Gets all commands for a category.
     * @param category The category to query.
     * @return The commands, if any.
     */
    static commandsForCategory(category) {
        const ret = [];
        for (const cmd in COMMAND_DATA) {
            const struct = COMMAND_DATA[cmd];
            if (category === struct.category) {
                ret.push(cmd);
            }
        }
        return ret;
    }
    /**
     * @param command The command to query.
     * @return Whether this command is denied in signed out contexts.
     */
    static denySignedOut(command) {
        return Boolean(COMMAND_DATA[command]?.denySignedOut);
    }
    static getKeyBindings() {
        const primaryKeyBindings = Object.entries(COMMAND_DATA)
            .filter(([_command, data]) => data.sequence)
            .map(([command, data]) => {
            // Always true, but closure compiler doesn't know that.
            if (data.sequence) {
                return {
                    command,
                    sequence: KeySequence.deserialize(data.sequence),
                };
            }
            return undefined;
        });
        const secondaryKeyBindings = Object.entries(COMMAND_DATA)
            .filter(([_command, data]) => data.altSequence)
            .map(([command, data]) => {
            // Always true, but closure compiler doesn't know that.
            if (data.altSequence) {
                return {
                    command,
                    sequence: KeySequence.deserialize(data.altSequence),
                };
            }
            return undefined;
        });
        return primaryKeyBindings.concat(secondaryKeyBindings);
    }
}
/**
 * Collection of command properties.
 * NOTE: sorted in lookup order when matching against commands.
 */
const COMMAND_DATA = {
    [Command.ANNOUNCE_BATTERY_DESCRIPTION]: {
        category: CommandCategory.INFORMATION,
        msgId: 'announce_battery_description',
        sequence: { cvoxModifier: true, keys: { keyCode: [KeyCode.O, KeyCode.B] } },
    },
    [Command.ANNOUNCE_HEADERS]: {
        category: CommandCategory.TABLES,
        msgId: 'announce_headers',
    },
    [Command.ANNOUNCE_RICH_TEXT_DESCRIPTION]: {
        category: CommandCategory.INFORMATION,
        msgId: 'announce_rich_text_description',
        sequence: { cvoxModifier: true, keys: { keyCode: [KeyCode.A, KeyCode.F] } },
    },
    [Command.AUTORUNNER]: { category: CommandCategory.NO_CATEGORY },
    [Command.BACKWARD]: {
        category: CommandCategory.NAVIGATION,
        msgId: 'backward',
    },
    [Command.BOTTOM]: {
        category: CommandCategory.BRAILLE,
        msgId: 'braille_bottom',
    },
    [Command.CONTEXT_MENU]: {
        category: CommandCategory.INFORMATION,
        msgId: 'show_context_menu',
        sequence: { cvoxModifier: true, keys: { keyCode: [KeyCode.M] } },
    },
    [Command.COPY]: {
        category: CommandCategory.NO_CATEGORY,
        sequence: { cvoxModifier: true, keys: { keyCode: [KeyCode.C], ctrlKey: [true] } },
    },
    [Command.CYCLE_PUNCTUATION_ECHO]: {
        category: CommandCategory.CONTROLLING_SPEECH,
        msgId: 'cycle_punctuation_echo',
        sequence: { cvoxModifier: true, keys: { keyCode: [KeyCode.A, KeyCode.P] } },
    },
    [Command.CYCLE_TYPING_ECHO]: {
        category: CommandCategory.CONTROLLING_SPEECH,
        msgId: 'cycle_typing_echo',
        sequence: { cvoxModifier: true, keys: { keyCode: [KeyCode.A, KeyCode.T] } },
    },
    [Command.DEBUG]: { category: CommandCategory.NO_CATEGORY },
    [Command.DECREASE_TTS_PITCH]: {
        category: CommandCategory.CONTROLLING_SPEECH,
        msgId: 'decrease_tts_pitch',
        sequence: {
            cvoxModifier: true,
            keys: { keyCode: [KeyCode.OEM_6], shiftKey: [true] },
        },
    },
    [Command.DECREASE_TTS_RATE]: {
        category: CommandCategory.CONTROLLING_SPEECH,
        msgId: 'decrease_tts_rate',
        sequence: {
            cvoxModifier: true,
            keys: { keyCode: [KeyCode.OEM_4], shiftKey: [true] },
        },
    },
    [Command.DECREASE_TTS_VOLUME]: {
        category: CommandCategory.CONTROLLING_SPEECH,
        msgId: 'decrease_tts_volume',
    },
    [Command.DISABLE_CHROMEVOX_ARC_SUPPORT_FOR_CURRENT_APP]: {
        category: CommandCategory.NO_CATEGORY,
        sequence: { cvoxModifier: true, keys: { keyCode: [KeyCode.A, KeyCode.OEM_6] } },
    },
    [Command.DISABLE_LOGGING]: {
        category: CommandCategory.NO_CATEGORY,
        sequence: { cvoxModifier: true, keys: { keyCode: [KeyCode.O, KeyCode.D] } },
    },
    [Command.DUMP_TREE]: {
        category: CommandCategory.NO_CATEGORY,
        sequence: {
            cvoxModifier: true,
            keys: { keyCode: [KeyCode.D, KeyCode.T], ctrlKey: [true] },
        },
    },
    [Command.ENABLE_CHROMEVOX_ARC_SUPPORT_FOR_CURRENT_APP]: {
        category: CommandCategory.NO_CATEGORY,
        sequence: { cvoxModifier: true, keys: { keyCode: [KeyCode.A, KeyCode.OEM_4] } },
    },
    [Command.ENABLE_CONSOLE_TTS]: {
        category: CommandCategory.DEVELOPER,
        msgId: 'enable_tts_log',
    },
    [Command.ENABLE_LOGGING]: {
        category: CommandCategory.NO_CATEGORY,
        sequence: { cvoxModifier: true, keys: { keyCode: [KeyCode.O, KeyCode.E] } },
    },
    [Command.ENTER_SHIFTER]: {
        category: CommandCategory.NAVIGATION,
        msgId: 'enter_content',
    },
    [Command.EXIT_SHIFTER]: {
        category: CommandCategory.NAVIGATION,
        msgId: 'exit_content',
    },
    [Command.EXIT_SHIFTER_CONTENT]: { category: CommandCategory.NO_CATEGORY },
    [Command.FORCE_CLICK_ON_CURRENT_ITEM]: {
        category: CommandCategory.ACTIONS,
        msgId: 'force_click_on_current_item',
        sequence: { cvoxModifier: true, keys: { keyCode: [KeyCode.SPACE] } },
    },
    [Command.FORCE_DOUBLE_CLICK_ON_CURRENT_ITEM]: {
        category: CommandCategory.NO_CATEGORY,
        sequence: { cvoxModifier: true, keys: { keyCode: [KeyCode.SPACE] }, doubleTap: true },
    },
    [Command.FORCE_LONG_CLICK_ON_CURRENT_ITEM]: {
        category: CommandCategory.NO_CATEGORY,
        msgId: 'force_long_click_on_current_item',
        sequence: {
            cvoxModifier: true,
            keys: { keyCode: [KeyCode.SPACE], shiftKey: [true] },
        },
    },
    [Command.FORWARD]: {
        category: CommandCategory.NAVIGATION,
        msgId: 'forward',
    },
    [Command.FULLY_DESCRIBE]: {
        category: CommandCategory.INFORMATION,
        msgId: 'fully_describe',
        sequence: { cvoxModifier: true, keys: { keyCode: [KeyCode.K] } },
    },
    [Command.GO_TO_COL_FIRST_CELL]: {
        category: CommandCategory.TABLES,
        msgId: 'skip_to_col_beginning',
        sequence: {
            cvoxModifier: true,
            keys: {
                keyCode: [KeyCode.UP],
                ctrlKey: [true],
                altKey: [true],
                shiftKey: [true],
            },
        },
    },
    [Command.GO_TO_COL_LAST_CELL]: {
        category: CommandCategory.TABLES,
        msgId: 'skip_to_col_end',
        sequence: {
            cvoxModifier: true,
            keys: {
                keyCode: [KeyCode.DOWN],
                ctrlKey: [true],
                altKey: [true],
                shiftKey: [true],
            },
        },
    },
    [Command.GO_TO_FIRST_CELL]: {
        category: CommandCategory.TABLES,
        msgId: 'skip_to_beginning',
        sequence: {
            cvoxModifier: true,
            keys: { keyCode: [KeyCode.LEFT], altKey: [true], shiftKey: [true] },
        },
    },
    [Command.GO_TO_LAST_CELL]: {
        category: CommandCategory.TABLES,
        msgId: 'skip_to_end',
        sequence: {
            cvoxModifier: true,
            keys: { keyCode: [KeyCode.RIGHT], altKey: [true], shiftKey: [true] },
        },
    },
    [Command.GO_TO_ROW_FIRST_CELL]: {
        category: CommandCategory.TABLES,
        msgId: 'skip_to_row_beginning',
        sequence: {
            cvoxModifier: true,
            keys: {
                keyCode: [KeyCode.LEFT],
                ctrlKey: [true],
                altKey: [true],
                shiftKey: [true],
            },
        },
    },
    [Command.GO_TO_ROW_LAST_CELL]: {
        category: CommandCategory.TABLES,
        msgId: 'skip_to_row_end',
        sequence: {
            cvoxModifier: true,
            keys: {
                keyCode: [KeyCode.RIGHT],
                ctrlKey: [true],
                altKey: [true],
                shiftKey: [true],
            },
        },
    },
    [Command.HANDLE_TAB]: {
        category: CommandCategory.NAVIGATION,
        msgId: 'handle_tab_next',
    },
    [Command.HANDLE_TAB_PREV]: {
        category: CommandCategory.NAVIGATION,
        msgId: 'handle_tab_prev',
    },
    [Command.HELP]: {
        category: CommandCategory.HELP_COMMANDS,
        msgId: 'help',
        sequence: { cvoxModifier: true, keys: { keyCode: [KeyCode.O, KeyCode.T] } },
    },
    [Command.INCREASE_TTS_PITCH]: {
        category: CommandCategory.CONTROLLING_SPEECH,
        msgId: 'increase_tts_pitch',
        sequence: { cvoxModifier: true, keys: { keyCode: [KeyCode.OEM_6] } },
    },
    [Command.INCREASE_TTS_RATE]: {
        category: CommandCategory.CONTROLLING_SPEECH,
        msgId: 'increase_tts_rate',
        sequence: { cvoxModifier: true, keys: { keyCode: [KeyCode.OEM_4] } },
    },
    [Command.INCREASE_TTS_VOLUME]: {
        category: CommandCategory.CONTROLLING_SPEECH,
        msgId: 'increase_tts_volume',
    },
    [Command.JUMP_TO_BOTTOM]: {
        category: CommandCategory.NAVIGATION,
        msgId: 'jump_to_bottom',
        sequence: { cvoxModifier: true, keys: { keyCode: [KeyCode.RIGHT], ctrlKey: [true] } },
    },
    [Command.JUMP_TO_DETAILS]: {
        category: CommandCategory.NAVIGATION,
        msgId: 'jump_to_details',
        sequence: { cvoxModifier: true, keys: { keyCode: [KeyCode.A, KeyCode.J] } },
    },
    [Command.JUMP_TO_TOP]: {
        category: CommandCategory.NAVIGATION,
        msgId: 'jump_to_top',
        sequence: { cvoxModifier: true, keys: { keyCode: [KeyCode.LEFT], ctrlKey: [true] } },
    },
    [Command.LEFT]: {
        category: CommandCategory.NAVIGATION,
        msgId: 'left',
    },
    [Command.LINE_DOWN]: {
        category: CommandCategory.BRAILLE,
        msgId: 'braille_line_down',
    },
    [Command.LINE_UP]: {
        category: CommandCategory.BRAILLE,
        msgId: 'braille_line_up',
    },
    [Command.MOVE_TO_START_OF_LINE]: { category: CommandCategory.NO_CATEGORY },
    [Command.MOVE_TO_END_OF_LINE]: { category: CommandCategory.NO_CATEGORY },
    [Command.NEXT_ARTICLE]: { category: CommandCategory.NO_CATEGORY },
    [Command.NEXT_AT_GRANULARITY]: {
        category: CommandCategory.NAVIGATION,
        msgId: 'next_at_granularity',
    },
    [Command.NEXT_BUTTON]: {
        category: CommandCategory.JUMP_COMMANDS,
        msgId: 'next_button',
        sequence: { cvoxModifier: true, keys: { keyCode: [KeyCode.B] } },
    },
    [Command.NEXT_CHARACTER]: {
        category: CommandCategory.NAVIGATION,
        msgId: 'next_character',
        sequence: {
            cvoxModifier: true,
            keys: { keyCode: [KeyCode.RIGHT], shiftKey: [true] },
        },
    },
    [Command.NEXT_CHECKBOX]: {
        category: CommandCategory.JUMP_COMMANDS,
        msgId: 'next_checkbox',
        sequence: { cvoxModifier: true, keys: { keyCode: [KeyCode.X] } },
    },
    [Command.NEXT_COL]: {
        category: CommandCategory.TABLES,
        msgId: 'skip_to_next_col',
        sequence: {
            cvoxModifier: true,
            keys: { keyCode: [KeyCode.RIGHT], ctrlKey: [true], altKey: [true] },
        },
    },
    [Command.NEXT_COMBO_BOX]: {
        category: CommandCategory.JUMP_COMMANDS,
        msgId: 'next_combo_box',
        sequence: { cvoxModifier: true, keys: { keyCode: [KeyCode.C] } },
    },
    [Command.NEXT_CONTROL]: { category: CommandCategory.NO_CATEGORY },
    [Command.NEXT_EDIT_TEXT]: {
        category: CommandCategory.JUMP_COMMANDS,
        msgId: 'next_edit_text',
        sequence: { cvoxModifier: true, keys: { keyCode: [KeyCode.E] } },
    },
    [Command.NEXT_FORM_FIELD]: {
        category: CommandCategory.JUMP_COMMANDS,
        msgId: 'next_form_field',
        sequence: { cvoxModifier: true, keys: { keyCode: [KeyCode.F] } },
    },
    [Command.NEXT_GRANULARITY]: {
        category: CommandCategory.NAVIGATION,
        msgId: 'next_granularity',
    },
    [Command.NEXT_GRAPHIC]: {
        category: CommandCategory.JUMP_COMMANDS,
        msgId: 'next_graphic',
        sequence: { cvoxModifier: true, keys: { keyCode: [KeyCode.G] } },
    },
    [Command.NEXT_GROUP]: {
        category: CommandCategory.NAVIGATION,
        msgId: 'next_group',
        sequence: { cvoxModifier: true, keys: { keyCode: [KeyCode.DOWN], ctrlKey: [true] } },
    },
    [Command.NEXT_HEADING]: {
        category: CommandCategory.JUMP_COMMANDS,
        msgId: 'next_heading',
        sequence: { cvoxModifier: true, keys: { keyCode: [KeyCode.H] } },
    },
    [Command.NEXT_HEADING_1]: {
        category: CommandCategory.JUMP_COMMANDS,
        msgId: 'next_heading1',
        sequence: { cvoxModifier: true, keys: { keyCode: [KeyCode.ONE] } },
    },
    [Command.NEXT_HEADING_2]: {
        category: CommandCategory.JUMP_COMMANDS,
        msgId: 'next_heading2',
        sequence: { cvoxModifier: true, keys: { keyCode: [KeyCode.TWO] } },
    },
    [Command.NEXT_HEADING_3]: {
        category: CommandCategory.JUMP_COMMANDS,
        msgId: 'next_heading3',
        sequence: { cvoxModifier: true, keys: { keyCode: [KeyCode.THREE] } },
    },
    [Command.NEXT_HEADING_4]: {
        category: CommandCategory.JUMP_COMMANDS,
        msgId: 'next_heading4',
        sequence: { cvoxModifier: true, keys: { keyCode: [KeyCode.FOUR] } },
    },
    [Command.NEXT_HEADING_5]: {
        category: CommandCategory.JUMP_COMMANDS,
        msgId: 'next_heading5',
        sequence: { cvoxModifier: true, keys: { keyCode: [KeyCode.FIVE] } },
    },
    [Command.NEXT_HEADING_6]: {
        category: CommandCategory.JUMP_COMMANDS,
        msgId: 'next_heading6',
        sequence: { cvoxModifier: true, keys: { keyCode: [KeyCode.SIX] } },
    },
    [Command.NEXT_INVALID_ITEM]: {
        category: CommandCategory.NAVIGATION,
        msgId: 'next_invalid_item',
        sequence: { cvoxModifier: true, keys: { keyCode: [KeyCode.N, KeyCode.I] } },
    },
    [Command.NEXT_LANDMARK]: {
        category: CommandCategory.JUMP_COMMANDS,
        msgId: 'next_landmark',
        sequence: { cvoxModifier: true, keys: { keyCode: [KeyCode.OEM_1] } },
    },
    [Command.NEXT_LINE]: {
        category: CommandCategory.NAVIGATION,
        msgId: 'next_line',
        sequence: { cvoxModifier: true, keys: { keyCode: [KeyCode.DOWN] } },
    },
    [Command.NEXT_LINK]: {
        category: CommandCategory.JUMP_COMMANDS,
        msgId: 'next_link',
        sequence: { cvoxModifier: true, keys: { keyCode: [KeyCode.L] } },
    },
    [Command.NEXT_LIST]: {
        category: CommandCategory.JUMP_COMMANDS,
        msgId: 'next_list',
        sequence: { cvoxModifier: true, keys: { keyCode: [KeyCode.J, KeyCode.L] } },
    },
    [Command.NEXT_LIST_ITEM]: {
        category: CommandCategory.JUMP_COMMANDS,
        msgId: 'next_list_item',
    },
    [Command.NEXT_MATH]: {
        category: CommandCategory.JUMP_COMMANDS,
        msgId: 'next_math',
    },
    [Command.NEXT_MEDIA]: {
        category: CommandCategory.JUMP_COMMANDS,
        msgId: 'next_media',
    },
    [Command.NEXT_OBJECT]: {
        category: CommandCategory.NAVIGATION,
        msgId: 'next_object',
        sequence: { cvoxModifier: true, keys: { keyCode: [KeyCode.RIGHT] } },
    },
    [Command.NEXT_PAGE]: { category: CommandCategory.NO_CATEGORY },
    [Command.NEXT_RADIO]: {
        category: CommandCategory.JUMP_COMMANDS,
        msgId: 'next_radio',
    },
    [Command.NEXT_ROW]: {
        category: CommandCategory.TABLES,
        msgId: 'skip_to_next_row',
        sequence: {
            cvoxModifier: true,
            keys: { keyCode: [KeyCode.DOWN], ctrlKey: [true], altKey: [true] },
        },
    },
    [Command.NEXT_SECTION]: { category: CommandCategory.NO_CATEGORY },
    [Command.NEXT_SENTENCE]: {
        category: CommandCategory.NAVIGATION,
        msgId: 'next_sentence',
    },
    [Command.NEXT_SIMILAR_ITEM]: {
        category: CommandCategory.NAVIGATION,
        msgId: 'next_similar_item',
        sequence: { cvoxModifier: true, keys: { keyCode: [KeyCode.I] } },
    },
    [Command.NEXT_SLIDER]: { category: CommandCategory.NO_CATEGORY },
    [Command.NEXT_TABLE]: {
        category: CommandCategory.JUMP_COMMANDS,
        msgId: 'next_table',
        sequence: { cvoxModifier: true, keys: { keyCode: [KeyCode.T] } },
    },
    [Command.NEXT_VISITED_LINK]: {
        category: CommandCategory.JUMP_COMMANDS,
        msgId: 'next_visited_link',
        sequence: { cvoxModifier: true, keys: { keyCode: [KeyCode.V] } },
    },
    [Command.NEXT_WORD]: {
        category: CommandCategory.NAVIGATION,
        msgId: 'next_word',
        sequence: {
            cvoxModifier: true,
            keys: { keyCode: [KeyCode.RIGHT], ctrlKey: [true], shiftKey: [true] },
        },
    },
    [Command.NOP]: { category: CommandCategory.NO_CATEGORY },
    [Command.OPEN_CHROMEVOX_MENUS]: {
        category: CommandCategory.NO_CATEGORY,
        msgId: 'menus_title',
    },
    [Command.OPEN_KEYBOARD_SHORTCUTS]: {
        category: CommandCategory.HELP_COMMANDS,
        denySignedOut: true,
        msgId: 'open_keyboard_shortcuts_menu',
        sequence: { cvoxModifier: true, keys: { keyCode: [KeyCode.S], ctrlKey: [true] } },
    },
    [Command.OPEN_LONG_DESC]: {
        category: CommandCategory.INFORMATION,
        msgId: 'open_long_desc',
    },
    [Command.PAN_LEFT]: {
        category: CommandCategory.BRAILLE,
        msgId: 'braille_pan_left',
    },
    [Command.PAN_RIGHT]: {
        category: CommandCategory.BRAILLE,
        msgId: 'braille_pan_right',
    },
    [Command.PASS_THROUGH_MODE]: {
        category: CommandCategory.MODIFIER_KEYS,
        msgId: 'pass_through_key_description',
        sequence: {
            cvoxModifier: true,
            keys: { keyCode: [KeyCode.ESCAPE], shiftKey: [true] },
        },
    },
    [Command.PAUSE_ALL_MEDIA]: {
        category: CommandCategory.INFORMATION,
        msgId: 'pause_all_media',
    },
    [Command.PREVIOUS_ARTICLE]: { category: CommandCategory.NO_CATEGORY },
    [Command.PREVIOUS_AT_GRANULARITY]: {
        category: CommandCategory.NAVIGATION,
        msgId: 'previous_at_granularity',
    },
    [Command.PREVIOUS_BUTTON]: {
        category: CommandCategory.JUMP_COMMANDS,
        msgId: 'previous_button',
        sequence: { cvoxModifier: true, keys: { keyCode: [KeyCode.B], shiftKey: [true] } },
    },
    [Command.PREVIOUS_CHARACTER]: {
        category: CommandCategory.NAVIGATION,
        msgId: 'previous_character',
        sequence: { cvoxModifier: true, keys: { keyCode: [KeyCode.LEFT], shiftKey: [true] } },
    },
    [Command.PREVIOUS_CHECKBOX]: {
        category: CommandCategory.JUMP_COMMANDS,
        msgId: 'previous_checkbox',
        sequence: { cvoxModifier: true, keys: { keyCode: [KeyCode.X], shiftKey: [true] } },
    },
    [Command.PREVIOUS_COL]: {
        category: CommandCategory.TABLES,
        msgId: 'skip_to_prev_col',
        sequence: {
            cvoxModifier: true,
            keys: { keyCode: [KeyCode.LEFT], ctrlKey: [true], altKey: [true] },
        },
    },
    [Command.PREVIOUS_COMBO_BOX]: {
        category: CommandCategory.JUMP_COMMANDS,
        msgId: 'previous_combo_box',
        sequence: { cvoxModifier: true, keys: { keyCode: [KeyCode.C], shiftKey: [true] } },
    },
    [Command.PREVIOUS_CONTROL]: { category: CommandCategory.NO_CATEGORY },
    [Command.PREVIOUS_EDIT_TEXT]: {
        category: CommandCategory.JUMP_COMMANDS,
        msgId: 'previous_edit_text',
        sequence: { cvoxModifier: true, keys: { keyCode: [KeyCode.E], shiftKey: [true] } },
    },
    [Command.PREVIOUS_FORM_FIELD]: {
        category: CommandCategory.JUMP_COMMANDS,
        msgId: 'previous_form_field',
        sequence: { cvoxModifier: true, keys: { keyCode: [KeyCode.F], shiftKey: [true] } },
    },
    [Command.PREVIOUS_GRANULARITY]: {
        category: CommandCategory.NAVIGATION,
        msgId: 'previous_granularity',
    },
    [Command.PREVIOUS_GRAPHIC]: {
        category: CommandCategory.JUMP_COMMANDS,
        msgId: 'previous_graphic',
        sequence: { cvoxModifier: true, keys: { keyCode: [KeyCode.G], shiftKey: [true] } },
    },
    [Command.PREVIOUS_GROUP]: {
        category: CommandCategory.NAVIGATION,
        msgId: 'previous_group',
        sequence: { cvoxModifier: true, keys: { keyCode: [KeyCode.UP], ctrlKey: [true] } },
    },
    [Command.PREVIOUS_HEADING]: {
        category: CommandCategory.JUMP_COMMANDS,
        msgId: 'previous_heading',
        sequence: { cvoxModifier: true, keys: { keyCode: [KeyCode.H], shiftKey: [true] } },
    },
    [Command.PREVIOUS_HEADING_1]: {
        category: CommandCategory.JUMP_COMMANDS,
        msgId: 'previous_heading1',
        sequence: { cvoxModifier: true, keys: { keyCode: [KeyCode.ONE], shiftKey: [true] } },
    },
    [Command.PREVIOUS_HEADING_2]: {
        category: CommandCategory.JUMP_COMMANDS,
        msgId: 'previous_heading2',
        sequence: { cvoxModifier: true, keys: { keyCode: [KeyCode.TWO], shiftKey: [true] } },
    },
    [Command.PREVIOUS_HEADING_3]: {
        category: CommandCategory.JUMP_COMMANDS,
        msgId: 'previous_heading3',
        sequence: {
            cvoxModifier: true,
            keys: { keyCode: [KeyCode.THREE], shiftKey: [true] },
        },
    },
    [Command.PREVIOUS_HEADING_4]: {
        category: CommandCategory.JUMP_COMMANDS,
        msgId: 'previous_heading4',
        sequence: { cvoxModifier: true, keys: { keyCode: [KeyCode.FOUR], shiftKey: [true] } },
    },
    [Command.PREVIOUS_HEADING_5]: {
        category: CommandCategory.JUMP_COMMANDS,
        msgId: 'previous_heading5',
        sequence: { cvoxModifier: true, keys: { keyCode: [KeyCode.FIVE], shiftKey: [true] } },
    },
    [Command.PREVIOUS_HEADING_6]: {
        category: CommandCategory.JUMP_COMMANDS,
        msgId: 'previous_heading6',
        sequence: { cvoxModifier: true, keys: { keyCode: [KeyCode.SIX], shiftKey: [true] } },
    },
    [Command.PREVIOUS_INVALID_ITEM]: {
        category: CommandCategory.NAVIGATION,
        msgId: 'previous_invalid_item',
        sequence: { cvoxModifier: true, keys: { keyCode: [KeyCode.P, KeyCode.I] } },
    },
    [Command.PREVIOUS_LANDMARK]: {
        category: CommandCategory.JUMP_COMMANDS,
        msgId: 'previous_landmark',
        sequence: {
            cvoxModifier: true,
            keys: { keyCode: [KeyCode.OEM_1], shiftKey: [true] },
        },
    },
    [Command.PREVIOUS_LINE]: {
        category: CommandCategory.NAVIGATION,
        msgId: 'previous_line',
        sequence: { cvoxModifier: true, keys: { keyCode: [KeyCode.UP] } },
    },
    [Command.PREVIOUS_LINK]: {
        category: CommandCategory.JUMP_COMMANDS,
        msgId: 'previous_link',
        sequence: { cvoxModifier: true, keys: { keyCode: [KeyCode.L], shiftKey: [true] } },
    },
    [Command.PREVIOUS_LIST]: {
        category: CommandCategory.JUMP_COMMANDS,
        msgId: 'previous_list',
        sequence: {
            cvoxModifier: true,
            keys: { keyCode: [KeyCode.J, KeyCode.L], shiftKey: [true] },
        },
    },
    [Command.PREVIOUS_LIST_ITEM]: {
        category: CommandCategory.JUMP_COMMANDS,
        msgId: 'previous_list_item',
    },
    [Command.PREVIOUS_MATH]: {
        category: CommandCategory.JUMP_COMMANDS,
        msgId: 'previous_math',
    },
    [Command.PREVIOUS_MEDIA]: {
        category: CommandCategory.JUMP_COMMANDS,
        msgId: 'previous_media',
    },
    [Command.PREVIOUS_OBJECT]: {
        category: CommandCategory.NAVIGATION,
        msgId: 'previous_object',
        sequence: { cvoxModifier: true, keys: { keyCode: [KeyCode.LEFT] } },
    },
    [Command.PREVIOUS_PAGE]: { category: CommandCategory.NO_CATEGORY },
    [Command.PREVIOUS_RADIO]: {
        category: CommandCategory.JUMP_COMMANDS,
        msgId: 'previous_radio',
    },
    [Command.PREVIOUS_ROW]: {
        category: CommandCategory.TABLES,
        msgId: 'skip_to_prev_row',
        sequence: {
            cvoxModifier: true,
            keys: { keyCode: [KeyCode.UP], ctrlKey: [true], altKey: [true] },
        },
    },
    [Command.PREVIOUS_SECTION]: { category: CommandCategory.NO_CATEGORY },
    [Command.PREVIOUS_SENTENCE]: {
        category: CommandCategory.NAVIGATION,
        msgId: 'previous_sentence',
    },
    [Command.PREVIOUS_SIMILAR_ITEM]: {
        category: CommandCategory.NAVIGATION,
        msgId: 'previous_similar_item',
        sequence: { cvoxModifier: true, keys: { keyCode: [KeyCode.I], shiftKey: [true] } },
    },
    [Command.PREVIOUS_SLIDER]: { category: CommandCategory.NO_CATEGORY },
    [Command.PREVIOUS_TABLE]: {
        category: CommandCategory.JUMP_COMMANDS,
        msgId: 'previous_table',
        sequence: { cvoxModifier: true, keys: { keyCode: [KeyCode.T], shiftKey: [true] } },
    },
    [Command.PREVIOUS_VISITED_LINK]: {
        category: CommandCategory.JUMP_COMMANDS,
        msgId: 'previous_visited_link',
        sequence: { cvoxModifier: true, keys: { keyCode: [KeyCode.V], shiftKey: [true] } },
    },
    [Command.PREVIOUS_WORD]: {
        category: CommandCategory.NAVIGATION,
        msgId: 'previous_word',
        sequence: {
            cvoxModifier: true,
            keys: { keyCode: [KeyCode.LEFT], ctrlKey: [true], shiftKey: [true] },
        },
    },
    [Command.READ_CURRENT_TITLE]: {
        category: CommandCategory.INFORMATION,
        msgId: 'read_current_title',
        sequence: { cvoxModifier: true, keys: { keyCode: [KeyCode.A, KeyCode.W] } },
    },
    [Command.READ_CURRENT_URL]: {
        category: CommandCategory.INFORMATION,
        msgId: 'read_current_url',
        sequence: { cvoxModifier: true, keys: { keyCode: [KeyCode.A, KeyCode.U] } },
    },
    [Command.READ_FROM_HERE]: {
        category: CommandCategory.NAVIGATION,
        msgId: 'read_from_here',
        sequence: { cvoxModifier: true, keys: { keyCode: [KeyCode.R] } },
    },
    [Command.READ_LINK_URL]: {
        category: CommandCategory.INFORMATION,
        msgId: 'read_link_url',
        sequence: { cvoxModifier: true, keys: { keyCode: [KeyCode.A, KeyCode.L] } },
    },
    [Command.READ_PHONETIC_PRONUNCIATION]: {
        category: CommandCategory.INFORMATION,
        msgId: 'read_phonetic_pronunciation',
        sequence: { cvoxModifier: true, keys: { keyCode: [KeyCode.A, KeyCode.C] } },
    },
    [Command.REPORT_ISSUE]: {
        category: CommandCategory.HELP_COMMANDS,
        denySignedOut: true,
        msgId: 'panel_menu_item_report_issue',
        sequence: { cvoxModifier: true, keys: { keyCode: [KeyCode.A, KeyCode.I] } },
    },
    [Command.RESET_TEXT_TO_SPEECH_SETTINGS]: {
        category: CommandCategory.CONTROLLING_SPEECH,
        msgId: 'reset_tts_settings',
        sequence: {
            cvoxModifier: true,
            keys: { keyCode: [KeyCode.OEM_5], ctrlKey: [true], shiftKey: [true] },
        },
    },
    [Command.RIGHT]: {
        category: CommandCategory.NAVIGATION,
        msgId: 'right',
    },
    [Command.ROUTING]: {
        category: CommandCategory.BRAILLE,
        msgId: 'braille_routing',
    },
    [Command.SCROLL_BACKWARD]: {
        category: CommandCategory.NO_CATEGORY,
        msgId: 'action_scroll_backward_description',
    },
    [Command.SCROLL_FORWARD]: {
        category: CommandCategory.NO_CATEGORY,
        msgId: 'action_scroll_forward_description',
    },
    [Command.SHOW_ACTIONS_MENU]: {
        category: CommandCategory.NO_CATEGORY,
        sequence: { cvoxModifier: true, keys: { keyCode: [KeyCode.A], ctrlKey: [true] } },
    },
    [Command.SHOW_FORMS_LIST]: {
        category: CommandCategory.OVERVIEW,
        msgId: 'show_forms_list',
        sequence: { cvoxModifier: true, keys: { keyCode: [KeyCode.F], ctrlKey: [true] } },
    },
    [Command.SHOW_HEADINGS_LIST]: {
        category: CommandCategory.OVERVIEW,
        msgId: 'show_headings_list',
        sequence: { cvoxModifier: true, keys: { keyCode: [KeyCode.H], ctrlKey: [true] } },
    },
    [Command.SHOW_LANDMARKS_LIST]: {
        category: CommandCategory.OVERVIEW,
        msgId: 'show_landmarks_list',
        sequence: { cvoxModifier: true, keys: { keyCode: [KeyCode.OEM_1], ctrlKey: [true] } },
    },
    [Command.SHOW_LEARN_MODE_PAGE]: {
        category: CommandCategory.HELP_COMMANDS,
        denySignedOut: true,
        msgId: 'show_kb_explorer_page',
        sequence: { cvoxModifier: true, keys: { keyCode: [KeyCode.O, KeyCode.K] } },
    },
    [Command.SHOW_LINKS_LIST]: {
        category: CommandCategory.OVERVIEW,
        msgId: 'show_links_list',
        sequence: { cvoxModifier: true, keys: { keyCode: [KeyCode.L], ctrlKey: [true] } },
    },
    [Command.SHOW_LOG_PAGE]: {
        category: CommandCategory.HELP_COMMANDS,
        denySignedOut: true,
        msgId: 'show_log_page',
        sequence: { cvoxModifier: true, keys: { keyCode: [KeyCode.O, KeyCode.W] } },
    },
    [Command.SHOW_OPTIONS_PAGE]: {
        category: CommandCategory.HELP_COMMANDS,
        denySignedOut: true,
        msgId: 'show_options_page',
        sequence: { cvoxModifier: true, keys: { keyCode: [KeyCode.O, KeyCode.O] } },
    },
    [Command.SHOW_PANEL_MENU_MOST_RECENT]: {
        category: CommandCategory.HELP_COMMANDS,
        msgId: 'show_panel_menu',
    },
    [Command.SHOW_TABLES_LIST]: {
        category: CommandCategory.OVERVIEW,
        msgId: 'show_tables_list',
        sequence: { cvoxModifier: true, keys: { keyCode: [KeyCode.T], ctrlKey: [true] } },
    },
    [Command.SHOW_TALKBACK_KEYBOARD_SHORTCUTS]: {
        category: CommandCategory.NO_CATEGORY,
        sequence: { cvoxModifier: true, keys: { keyCode: [KeyCode.A, KeyCode.K] } },
    },
    [Command.SHOW_TTS_SETTINGS]: {
        category: CommandCategory.HELP_COMMANDS,
        denySignedOut: true,
        msgId: 'show_tts_settings',
        sequence: { cvoxModifier: true, keys: { keyCode: [KeyCode.O, KeyCode.S] } },
    },
    [Command.SPEAK_TABLE_LOCATION]: {
        category: CommandCategory.TABLES,
        msgId: 'speak_table_location',
    },
    [Command.SPEAK_TIME_AND_DATE]: {
        category: CommandCategory.INFORMATION,
        msgId: 'speak_time_and_date',
        sequence: { cvoxModifier: true, keys: { keyCode: [KeyCode.A, KeyCode.D] } },
    },
    [Command.START_HISTORY_RECORDING]: { category: CommandCategory.NO_CATEGORY },
    [Command.STOP_HISTORY_RECORDING]: { category: CommandCategory.NO_CATEGORY },
    [Command.STOP_SPEECH]: {
        category: CommandCategory.CONTROLLING_SPEECH,
        msgId: 'stop_speech_key',
        sequence: {
            cvoxModifier: false,
            keys: { ctrlKey: [true], keyCode: [KeyCode.CONTROL] },
        },
        altSequence: { keys: { ctrlKey: [true], keyCode: [KeyCode.CONTROL] } },
    },
    [Command.TOGGLE_BRAILLE_CAPTIONS]: {
        category: CommandCategory.HELP_COMMANDS,
        msgId: 'braille_captions',
        sequence: { cvoxModifier: true, keys: { keyCode: [KeyCode.A, KeyCode.B] } },
    },
    [Command.TOGGLE_BRAILLE_TABLE]: {
        category: CommandCategory.HELP_COMMANDS,
        msgId: 'toggle_braille_table',
        sequence: { cvoxModifier: true, keys: { keyCode: [KeyCode.A, KeyCode.G] } },
    },
    [Command.TOGGLE_CAPTIONS]: {
        // TODO(crbug.com/383403295): Set an appropriate category when feature is
        // launched.
        category: CommandCategory.NO_CATEGORY,
        msgId: 'toggle_captions',
        sequence: { cvoxModifier: true, keys: { keyCode: [KeyCode.O, KeyCode.C] } },
    },
    [Command.TOGGLE_DICTATION]: {
        category: CommandCategory.ACTIONS,
        msgId: 'toggle_dictation',
        sequence: { cvoxModifier: true, keys: { keyCode: [KeyCode.D] } },
    },
    [Command.TOGGLE_EARCONS]: {
        category: CommandCategory.CONTROLLING_SPEECH,
        msgId: 'toggle_earcons',
        sequence: { cvoxModifier: true, keys: { keyCode: [KeyCode.A, KeyCode.E] } },
    },
    [Command.TOGGLE_KEYBOARD_HELP]: {
        category: CommandCategory.HELP_COMMANDS,
        msgId: 'show_panel_menu',
        sequence: { cvoxModifier: true, keys: { keyCode: [KeyCode.OEM_PERIOD] } },
    },
    [Command.TOGGLE_SCREEN]: {
        category: CommandCategory.MODIFIER_KEYS,
        msgId: 'toggle_screen',
        sequence: { cvoxModifier: true, keys: { keyCode: [KeyCode.BRIGHTNESS_UP] } },
    },
    [Command.TOGGLE_SEARCH_WIDGET]: {
        category: CommandCategory.INFORMATION,
        msgId: 'toggle_search_widget',
        sequence: { cvoxModifier: true, keys: { keyCode: [KeyCode.OEM_2] } },
    },
    [Command.TOGGLE_SELECTION]: {
        category: CommandCategory.ACTIONS,
        msgId: 'toggle_selection',
        sequence: { cvoxModifier: true, keys: { keyCode: [KeyCode.S] } },
    },
    [Command.TOGGLE_SEMANTICS]: {
        category: CommandCategory.INFORMATION,
        msgId: 'toggle_semantics',
    },
    [Command.TOGGLE_SPEECH_ON_OR_OFF]: {
        category: CommandCategory.CONTROLLING_SPEECH,
        msgId: 'speech_on_off_description',
        sequence: { cvoxModifier: true, keys: { keyCode: [KeyCode.VOLUME_MUTE] } },
    },
    [Command.TOGGLE_STICKY_MODE]: {
        category: CommandCategory.MODIFIER_KEYS,
        msgId: 'toggle_sticky_mode',
        sequence: {
            skipStripping: false,
            doubleTap: true,
            keys: { keyCode: [KeyCode.SEARCH] },
        },
    },
    [Command.TOP]: {
        category: CommandCategory.BRAILLE,
        msgId: 'braille_top',
    },
    [Command.VIEW_GRAPHIC_AS_BRAILLE]: {
        category: CommandCategory.BRAILLE,
        msgId: 'view_graphic_as_braille',
        sequence: { cvoxModifier: true, keys: { keyCode: [KeyCode.G], altKey: [true] } },
    },
    // Keep these commands last since they potentially conflict with above
    // commands when sticky mode is enabled.
    [Command.NATIVE_NEXT_CHARACTER]: {
        category: CommandCategory.NO_CATEGORY,
        sequence: { cvoxModifier: false, keys: { keyCode: [KeyCode.RIGHT] } },
    },
    [Command.NATIVE_NEXT_WORD]: {
        category: CommandCategory.NO_CATEGORY,
        sequence: {
            cvoxModifier: false,
            keys: { keyCode: [KeyCode.RIGHT], ctrlKey: [true] },
        },
    },
    [Command.NATIVE_PREVIOUS_CHARACTER]: {
        category: CommandCategory.NO_CATEGORY,
        sequence: { cvoxModifier: false, keys: { keyCode: [KeyCode.LEFT] } },
    },
    [Command.NATIVE_PREVIOUS_WORD]: {
        category: CommandCategory.NO_CATEGORY,
        sequence: { cvoxModifier: false, keys: { keyCode: [KeyCode.LEFT], ctrlKey: [true] } },
    },
};

// Copyright 2023 The Chromium Authors
// Use of this source code is governed by a BSD-style license that can be
// found in the LICENSE file.
var SessionType = chrome.chromeosInfoPrivate.SessionType;
class PermissionChecker {
    isIncognito_;
    isKioskSession_;
    static instance;
    constructor() {
        this.isIncognito_ = Boolean(chrome.runtime.getManifest()['incognito']);
        this.isKioskSession_ = false;
    }
    static async init() {
        PermissionChecker.instance = new PermissionChecker();
        await PermissionChecker.instance.fetchState_();
    }
    static isAllowed(command) {
        return PermissionChecker.instance.isAllowed_(command);
    }
    isAllowed_(command) {
        if (!this.isIncognito_ && !this.isKioskSession_) {
            return true;
        }
        return !CommandStore.denySignedOut(command);
    }
    async fetchState_() {
        const result = await new Promise(resolve => chrome.chromeosInfoPrivate.get(['sessionType'], resolve));
        this.isKioskSession_ = result['sessionType'] === SessionType.KIOSK;
    }
}

// Copyright 2023 The Chromium Authors
// Use of this source code is governed by a BSD-style license that can be
// found in the LICENSE file.
/**
 * @fileoverview Class to manage Chrome settings prefs for ChromeVox. Wraps
 * around Settings for the underlying gets and sets. Acts as a translation layer
 * for key names, and for migrating. Will automatically migrate the ChromeVox
 * prefs that are listed in the |CHROMEVOX_PREFS| constant at the bottom of the
 * file. The prefs will move out of chrome.storage.local (used by the
 * LocalStorage class) into the Chrome settings prefs system (used by other
 * Accessibility services).
 *
 */
class SettingsManager {
    static instance;
    static CHROMEVOX_PREFS;
    static LIVE_CAPTION_PREF;
    static EVENT_STREAM_FILTERS;
    static async init() {
        if (SettingsManager.instance) {
            throw new Error('SettingsManager.init() should be called at most once in each ' +
                'browser context.');
        }
        SettingsManager.instance = new SettingsManager();
        await Settings.init(SettingsManager.getAllPrefNames());
        SettingsManager.migrateFromChromeStorage_();
    }
    /**
     * Gets Chrome settings pref name from local storage key name.
     * E.g. 'eventStreamFilters' -> 'settings.a11y.chromevox.event_stream_filters'
     * @param localStorageKey Name of a key used in local storage.
     * @return Corresponding name for the Chrome settings pref.
     */
    static getPrefName_(localStorageKey) {
        return 'settings.a11y.chromevox.' +
            StringUtil.camelToSnake(localStorageKey);
    }
    /**
     * Gets all Chrome settings pref names.
     */
    static getAllPrefNames() {
        return SettingsManager.CHROMEVOX_PREFS.map(SettingsManager.getPrefName_)
            .concat([SettingsManager.LIVE_CAPTION_PREF]);
    }
    /**
     * Migrates prefs from chrome.storage.local to Chrome settings prefs.
     */
    static migrateFromChromeStorage_() {
        for (const key of SettingsManager.CHROMEVOX_PREFS) {
            let value = LocalStorage.get(key);
            if (value === undefined) {
                continue;
            }
            // Convert virtualBrailleColumns and virtualBrailleRows to numbers.
            if (['virtualBrailleColumns', 'virtualBrailleRows'].includes(key)) {
                value = parseInt(value, 10);
            }
            const prefName = SettingsManager.getPrefName_(key);
            try {
                Settings.set(prefName, value);
                LocalStorage.remove(key);
            }
            catch {
                console.error('Invalid settings pref name:', prefName);
            }
        }
        // This object starts empty so that we can know if there are any event
        // stream filters left to migrate from LocalStorage.
        let eventStreamFilters = {};
        for (const key of SettingsManager.EVENT_STREAM_FILTERS) {
            const value = LocalStorage.get(key);
            if (value === undefined) {
                continue;
            }
            eventStreamFilters[key] = value;
            LocalStorage.remove(key);
        }
        if (Object.keys(eventStreamFilters).length) {
            // Combine with any event stream filters already in settings prefs.
            eventStreamFilters = {
                ...Settings.get(this.getPrefName_('eventStreamFilters')),
                ...eventStreamFilters,
            };
            Settings.set(this.getPrefName_('eventStreamFilters'), eventStreamFilters);
        }
    }
    static addListenerForKey(key, callback) {
        const pref = SettingsManager.getPrefName_(key);
        Settings.addListener(pref, callback);
    }
    static get(key, isChromeVox = true) {
        let pref = key;
        if (isChromeVox) {
            pref = SettingsManager.getPrefName_(key);
        }
        return Settings.get(pref);
    }
    /**
     * @param key
     * @param type A string (for primitives) or type constructor
     *     (for classes) corresponding to the expected type
     */
    static getTypeChecked(key, type, isChromeVox = true) {
        const value = SettingsManager.get(key, isChromeVox);
        if ((typeof type === 'string') && (typeof value === type)) {
            return value;
        }
        throw new Error('Value in SettingsManager for key "' + key + '" is not a ' + type);
    }
    static getBoolean(key, isChromeVox = true) {
        const value = SettingsManager.getTypeChecked(key, 'boolean', isChromeVox);
        return Boolean(value);
    }
    static getNumber(key) {
        const value = SettingsManager.getTypeChecked(key, 'number');
        if (isNaN(value)) {
            throw new Error('Value in SettingsManager for key "' + key + '" is NaN');
        }
        return Number(value);
    }
    static getString(key) {
        const value = SettingsManager.getTypeChecked(key, 'string');
        return String(value);
    }
    static set(key, value) {
        const pref = SettingsManager.getPrefName_(key);
        Settings.set(pref, value);
    }
    /**
     * Get event stream filters from the event_stream_filter dictionary pref.
     */
    static getEventStreamFilters() {
        return Settings.get(this.getPrefName_('eventStreamFilters'));
    }
    /**
     * Set an event stream filter on the event_stream_filter dictionary pref.
     */
    static setEventStreamFilter(key, value) {
        if (!SettingsManager.EVENT_STREAM_FILTERS.includes(key)) {
            throw new Error('Cannot set unknown event stream filter: ' + key);
        }
        const eventStreamFilters = Settings.get(this.getPrefName_('eventStreamFilters'));
        eventStreamFilters[key] = value;
        Settings.set(this.getPrefName_('eventStreamFilters'), eventStreamFilters);
    }
}
/**
 * List of the prefs used in ChromeVox, including in options page, each stored
 * as a Chrome settings pref.
 * @const {!Array<string>}
 */
SettingsManager.CHROMEVOX_PREFS = [
    'announceDownloadNotifications',
    'announceRichTextAttributes',
    'audioStrategy',
    'autoRead',
    'brailleSideBySide',
    'brailleTable',
    'brailleTable6',
    'brailleTable8',
    'brailleTableType',
    'brailleWordWrap',
    'capitalStrategy',
    'capitalStrategyBackup',
    'enableBrailleLogging',
    'enableEarconLogging',
    'enableEventStreamLogging',
    'enableSpeechLogging',
    'eventStreamFilters',
    'languageSwitching',
    'menuBrailleCommands',
    'numberReadingStyle',
    'preferredBrailleDisplayAddress',
    'punctuationEcho',
    'smartStickyMode',
    'speakTextUnderMouse',
    'usePitchChanges',
    'useVerboseMode',
    'virtualBrailleColumns',
    'virtualBrailleRows',
    'voiceName',
];
/**
 * The preference for when live caption is enabled.
 */
SettingsManager.LIVE_CAPTION_PREF =
    'accessibility.captions.live_caption_enabled';
/**
 * List of event stream filters used on the ChromeVox options page to indicate
 * which events to log, stored together in a Chrome settings dictionary pref.
 * @const {!Array<string>}
 */
SettingsManager.EVENT_STREAM_FILTERS = [
    'activedescendantchanged',
    'alert',
    // TODO(crbug.com/1464633) Fully remove ariaAttributeChangedDeprecated
    // starting in 122, because although it was removed in 118, it is still
    // present in earlier versions of LaCros.
    'ariaAttributeChangedDeprecated',
    'autocorrectionOccured',
    'blur',
    'checkedStateChanged',
    'childrenChanged',
    'clicked',
    'documentSelectionChanged',
    'documentTitleChanged',
    'expandedChanged',
    'focus',
    'focusContext',
    'hide',
    'hitTestResult',
    'hover',
    'imageFrameUpdated',
    'invalidStatusChanged',
    'layoutComplete',
    'liveRegionChanged',
    'liveRegionCreated',
    'loadComplete',
    'locationChanged',
    'mediaStartedPlaying',
    'mediaStoppedPlaying',
    'menuEnd',
    'menuItemSelected',
    'menuListValueChangedDeprecated',
    'menuPopupEnd',
    'menuPopupStart',
    'menuStart',
    'mouseCanceled',
    'mouseDragged',
    'mouseMoved',
    'mousePressed',
    'mouseReleased',
    'rowCollapsed',
    'rowCountChanged',
    'rowExpanded',
    'scrollPositionChanged',
    'scrolledToAnchor',
    'selectedChildrenChanged',
    'selection',
    'selectionAdd',
    'selectionRemove',
    'show',
    'stateChanged',
    'textChanged',
    'textSelectionChanged',
    'treeChanged',
    'valueInTextFieldChanged',
];
TestImportManager.exportForTesting(SettingsManager);

// Copyright 2022 The Chromium Authors
// Use of this source code is governed by a BSD-style license that can be
// found in the LICENSE file.
/**
 * @fileoverview Contains types related to speech generation.
 */
/**
 * Categories for a speech utterance. This can be used with the
 * CATEGORY_FLUSH queue mode, which flushes all utterances from a given
 * category but not other utterances.
 *
 * NAV: speech related to explicit navigation, or focus changing.
 * LIVE: speech coming from changes to live regions.
 */
var TtsCategory;
(function (TtsCategory) {
    TtsCategory["LIVE"] = "live";
    TtsCategory["NAV"] = "nav";
})(TtsCategory || (TtsCategory = {}));
/**
 * Queue modes for calls to {@code TtsInterface.speak}. The modes are listed in
 * descending order of priority.
 */
var QueueMode;
(function (QueueMode) {
    /**
     * Prepend the current utterance (if any) to the queue, stop speech, and
     * speak this utterance.
     */
    QueueMode[QueueMode["INTERJECT"] = 0] = "INTERJECT";
    /** Stop speech, clear everything, then speak this utterance. */
    QueueMode[QueueMode["FLUSH"] = 1] = "FLUSH";
    /**
     * Clear any utterances of the same category (as set by
     * properties['category']) from the queue, then enqueue this utterance.
     */
    QueueMode[QueueMode["CATEGORY_FLUSH"] = 2] = "CATEGORY_FLUSH";
    /** Append this utterance to the end of the queue. */
    QueueMode[QueueMode["QUEUE"] = 3] = "QUEUE";
})(QueueMode || (QueueMode = {}));
/**
 * Numeric properties of TtsSpeechProperties that will be set.
 */
var TtsAudioProperty;
(function (TtsAudioProperty) {
    TtsAudioProperty["PITCH"] = "pitch";
    TtsAudioProperty["RATE"] = "rate";
    TtsAudioProperty["VOLUME"] = "volume";
})(TtsAudioProperty || (TtsAudioProperty = {}));
/** Structure to store properties around TTS speech production. */
class TtsSpeechProperties {
    [TtsAudioProperty.PITCH];
    [TtsAudioProperty.RATE];
    [TtsAudioProperty.VOLUME];
    category;
    color;
    delay;
    doNotInterrupt;
    fontWeight;
    lang;
    math;
    pause;
    phoneticCharacters;
    punctuationEcho;
    token;
    voiceName;
    relativePitch;
    relativeRate;
    relativeVolume;
    startCallback;
    endCallback;
    onEvent;
    constructor(initialValues) {
        this.init_(initialValues);
    }
    toJSON() {
        return Object.assign({}, this);
    }
    init_(initialValues) {
        if (!initialValues) {
            return;
        }
        Object.assign(this, initialValues);
    }
}
var Personality;
(function (Personality) {
    // TTS personality for annotations - text spoken by ChromeVox that
    // elaborates on a user interface element but isn't displayed on-screen.
    Personality.ANNOTATION = new TtsSpeechProperties({
        'relativePitch': -0.25,
        // TODO:(rshearer) Added this color change for I/O presentation.
        'color': 'yellow',
        'punctuationEcho': 'none',
    });
    // TTS personality for announcements - text spoken by ChromeVox that
    // isn't tied to any user interface elements.
    Personality.ANNOUNCEMENT = new TtsSpeechProperties({
        'punctuationEcho': 'none',
    });
    // TTS personality for an aside - text in parentheses.
    Personality.ASIDE = new TtsSpeechProperties({
        'relativePitch': -0.1,
        'color': '#669',
    });
    // TTS personality for capital letters.
    Personality.CAPITAL = new TtsSpeechProperties({
        'relativePitch': 0.2,
    });
    // TTS personality for deleted text.
    Personality.DELETED = new TtsSpeechProperties({
        'punctuationEcho': 'none',
        'relativePitch': -0.6,
    });
    // TTS personality for dictation hints.
    Personality.DICTATION_HINT = new TtsSpeechProperties({
        'punctuationEcho': 'none',
        'relativePitch': 0.3,
    });
    // TTS personality for emphasis or italicized text.
    Personality.EMPHASIS = new TtsSpeechProperties({
        'relativeVolume': 0.1,
        'relativeRate': -0.1,
        'color': '#6bb',
        'fontWeight': 'bold',
    });
    // TTS personality for quoted text.
    Personality.QUOTE = new TtsSpeechProperties({
        'relativePitch': 0.1,
        'color': '#b6b',
        'fontWeight': 'bold',
    });
    // TTS personality for strong or bold text.
    Personality.STRONG = new TtsSpeechProperties({
        'relativePitch': 0.1,
        'color': '#b66',
        'fontWeight': 'bold',
    });
    // TTS personality for alerts from the system, such as battery level
    // warnings.
    Personality.SYSTEM_ALERT = new TtsSpeechProperties({
        'punctuationEcho': 'none',
        'doNotInterrupt': true,
    });
})(Personality || (Personality = {}));
/** Various TTS-related settings keys. */
var TtsSettings;
(function (TtsSettings) {
    // Color is for the lens display.
    TtsSettings["COLOR"] = "color";
    TtsSettings["FONT_WEIGHT"] = "fontWeight";
    TtsSettings["LANG"] = "lang";
    TtsSettings["PAUSE"] = "pause";
    TtsSettings["PHONETIC_CHARACTERS"] = "phoneticCharacters";
    TtsSettings["PITCH"] = "pitch";
    TtsSettings["PUNCTUATION_ECHO"] = "punctuationEcho";
    TtsSettings["RATE"] = "rate";
    TtsSettings["RELATIVE_PITCH"] = "relativePitch";
    TtsSettings["RELATIVE_RATE"] = "relativeRate";
    TtsSettings["RELATIVE_VOLUME"] = "relativeVolume";
    TtsSettings["VOLUME"] = "volume";
})(TtsSettings || (TtsSettings = {}));
/** List of punctuation echoes that the user can cycle through. */
const PunctuationEchoes = [
    // Punctuation echoed for the 'none' option.
    {
        name: 'none',
        msg: 'no_punctuation',
        regexp: /[-$#"()*;:<>\n\\\/+='~`@_]/g,
        clear: true,
    },
    // Punctuation echoed for the 'some' option.
    {
        name: 'some',
        msg: 'some_punctuation',
        regexp: /[$#"*<>\\\/\{\}+=~`%\u2022\u25e6\u25a0]/g,
        clear: false,
    },
    // Punctuation echoed for the 'all' option.
    {
        name: 'all',
        msg: 'all_punctuation',
        regexp: /[-$#"()*;:<>\n\\\/\{\}\[\]+='~`!@_.,?%\u2022\u25e6\u25a0]/g,
        clear: false,
    },
];
/**
 * Character dictionary. These symbols are replaced with their human readable
 * equivalents. This replacement only occurs for single character utterances.
 */
const CharacterDictionary = {
    ' ': 'space',
    '\u00a0': 'space',
    '`': 'backtick',
    '~': 'tilde',
    '!': 'exclamation',
    '@': 'at',
    '#': 'pound',
    '$': 'dollar',
    '%': 'percent',
    '^': 'caret',
    '&': 'ampersand',
    '*': 'asterisk',
    '(': 'open_paren',
    ')': 'close_paren',
    '-': 'dash',
    '_': 'underscore',
    '=': 'equals',
    '+': 'plus',
    '[': 'left_bracket',
    ']': 'right_bracket',
    '{': 'left_brace',
    '}': 'right_brace',
    '|': 'pipe',
    ';': 'semicolon',
    ':': 'colon',
    ',': 'comma',
    '.': 'dot',
    '<': 'less_than',
    '>': 'greater_than',
    '/': 'slash',
    '?': 'question_mark',
    '"': 'quote',
    '\'': 'apostrophe',
    '\t': 'tab',
    '\r': 'return',
    '\n': 'new_line',
    '\\': 'backslash',
    '\u2022': 'bullet',
    '\u25e6': 'white_bullet',
    '\u25a0': 'square_bullet',
};
/**
 * Substitution dictionary. These symbols or patterns are ALWAYS substituted
 * whenever they occur, so this should be reserved only for unicode characters
 * and characters that never have any different meaning in context.
 *
 * For example, do not include '$' here because $2 should be read as
 * "two dollars".
 */
const SubstitutionDictionary = {
    '://': 'colon slash slash',
    '\u00bc': 'one fourth',
    '\u00bd': 'one half',
    '\u2190': 'left arrow',
    '\u2191': 'up arrow',
    '\u2192': 'right arrow',
    '\u2193': 'down arrow',
    '\u21d0': 'left double arrow',
    '\u21d1': 'up double arrow',
    '\u21d2': 'right double  arrow',
    '\u21d3': 'down double arrow',
    '\u21e6': 'left arrow',
    '\u21e7': 'up arrow',
    '\u21e8': 'right arrow',
    '\u21e9': 'down arrow',
    '\u2303': 'control',
    '\u2318': 'command',
    '\u2325': 'option',
    '\u25b2': 'up triangle',
    '\u25b3': 'up triangle',
    '\u25b4': 'up triangle',
    '\u25b5': 'up triangle',
    '\u25b6': 'right triangle',
    '\u25b7': 'right triangle',
    '\u25b8': 'right triangle',
    '\u25b9': 'right triangle',
    '\u25ba': 'right pointer',
    '\u25bb': 'right pointer',
    '\u25bc': 'down triangle',
    '\u25bd': 'down triangle',
    '\u25be': 'down triangle',
    '\u25bf': 'down triangle',
    '\u25c0': 'left triangle',
    '\u25c1': 'left triangle',
    '\u25c2': 'left triangle',
    '\u25c3': 'left triangle',
    '\u25c4': 'left pointer',
    '\u25c5': 'left pointer',
    '\uf8ff': 'apple',
    '£': 'pound sterling',
};
TestImportManager.exportForTesting(['QueueMode', QueueMode], ['TtsSettings', TtsSettings], TtsSpeechProperties, ['TtsCategory', TtsCategory]);

// Copyright 2018 The Chromium Authors
// Use of this source code is governed by a BSD-style license that can be
// found in the LICENSE file.
/**
 * @fileoverview ChromeVox braille command data.
 */
var BrailleCommandData;
(function (BrailleCommandData) {
    /** Maps a dot pattern to a command. */
    BrailleCommandData.DOT_PATTERN_TO_COMMAND = {};
    /**
     * Makes a dot pattern given a list of dots numbered from 1 to 8 arranged in a
     * braille cell (a 2 x 4 dot grid).
     * @param dots The dots to be set in the returned pattern.
     */
    function makeDotPattern(dots) {
        return dots.reduce((pattern, cell) => pattern | (1 << cell - 1), 0);
    }
    BrailleCommandData.makeDotPattern = makeDotPattern;
    /** Gets a braille command based on a dot pattern from a chord. */
    function getCommand(dots) {
        return BrailleCommandData.DOT_PATTERN_TO_COMMAND[dots];
    }
    BrailleCommandData.getCommand = getCommand;
    /**
     * Gets a dot shortcut for a command.
     * @param chord True if the pattern comes from a chord.
     * @return The shortcut.
     */
    function getDotShortcut(command, chord) {
        const commandDots = getDots(command);
        return makeShortcutText(commandDots, chord);
    }
    BrailleCommandData.getDotShortcut = getDotShortcut;
    function makeShortcutText(pattern, chord) {
        const dots = [];
        for (let shifter = 0; shifter <= 7; shifter++) {
            if ((1 << shifter) & pattern) {
                dots.push(shifter + 1);
            }
        }
        let msgid;
        if (dots.length > 1) {
            msgid = 'braille_dots';
        }
        else if (dots.length === 1) {
            msgid = 'braille_dot';
        }
        if (msgid) {
            let dotText = Msgs.getMsg(msgid, [dots.join('-')]);
            if (chord) {
                dotText = Msgs.getMsg('braille_chord', [dotText]);
            }
            return dotText;
        }
        return '';
    }
    BrailleCommandData.makeShortcutText = makeShortcutText;
    /**
     * @return The dot pattern for |command|.
     */
    function getDots(command) {
        for (const keyStr in BrailleCommandData.DOT_PATTERN_TO_COMMAND) {
            const key = parseInt(keyStr, 10);
            if (command === BrailleCommandData.DOT_PATTERN_TO_COMMAND[key]) {
                return key;
            }
        }
        return 0;
    }
    BrailleCommandData.getDots = getDots;
    function init() {
        const map = function (dots, command) {
            const pattern = makeDotPattern(dots);
            const existingCommand = BrailleCommandData.DOT_PATTERN_TO_COMMAND[pattern];
            if (existingCommand) {
                throw 'Braille command pattern already exists: ' + dots + ' ' +
                    existingCommand + '. Trying to map ' + command;
            }
            BrailleCommandData.DOT_PATTERN_TO_COMMAND[pattern] = command;
        };
        map([2, 3], Command.PREVIOUS_GROUP);
        map([5, 6], Command.NEXT_GROUP);
        map([1], Command.PREVIOUS_OBJECT);
        map([4], Command.NEXT_OBJECT);
        map([2], Command.PREVIOUS_WORD);
        map([5], Command.NEXT_WORD);
        map([3], Command.PREVIOUS_CHARACTER);
        map([6], Command.NEXT_CHARACTER);
        map([1, 2, 3], Command.JUMP_TO_TOP);
        map([4, 5, 6], Command.JUMP_TO_BOTTOM);
        map([1, 4], Command.FULLY_DESCRIBE);
        map([1, 3, 4], Command.CONTEXT_MENU);
        map([1, 2, 3, 5], Command.READ_FROM_HERE);
        map([2, 3, 4], Command.TOGGLE_SELECTION);
        // Forward jump.
        map([1, 2], Command.NEXT_BUTTON);
        map([1, 5], Command.NEXT_EDIT_TEXT);
        map([1, 2, 4], Command.NEXT_FORM_FIELD);
        map([1, 2, 5], Command.NEXT_HEADING);
        map([4, 5], Command.NEXT_LINK);
        map([2, 3, 4, 5], Command.NEXT_TABLE);
        // Backward jump.
        map([1, 2, 7], Command.PREVIOUS_BUTTON);
        map([1, 5, 7], Command.PREVIOUS_EDIT_TEXT);
        map([1, 2, 4, 7], Command.PREVIOUS_FORM_FIELD);
        map([1, 2, 5, 7], Command.PREVIOUS_HEADING);
        map([4, 5, 7], Command.PREVIOUS_LINK);
        map([2, 3, 4, 5, 7], Command.PREVIOUS_TABLE);
        map([8], Command.FORCE_CLICK_ON_CURRENT_ITEM);
        map([3, 4], Command.TOGGLE_SEARCH_WIDGET);
        map([1, 4, 7], Command.TOGGLE_CAPTIONS);
        // Question.
        map([1, 4, 5, 6], Command.TOGGLE_KEYBOARD_HELP);
        // All cells.
        map([1, 2, 3, 4, 5, 6], Command.TOGGLE_SCREEN);
        // s.
        map([1, 2, 3, 4, 5], Command.TOGGLE_SPEECH_ON_OR_OFF);
        // g.
        map([1, 2, 4, 5], Command.TOGGLE_BRAILLE_TABLE);
        // Stop speech.
        map([5, 6, 7], Command.STOP_SPEECH);
    }
    BrailleCommandData.init = init;
    init();
})(BrailleCommandData || (BrailleCommandData = {}));
TestImportManager.exportForTesting(['BrailleCommandData', BrailleCommandData]);

// Copyright 2014 The Chromium Authors
// Use of this source code is governed by a BSD-style license that can be
// found in the LICENSE file.
/**
 * @fileoverview Braille command definitions.
 * These types are adapted from Chrome's private braille API.
 * They can be found in the Chrome source repo at:
 * src/chrome/common/extensions/api/braille_display_private.idl
 * We define them here since they don't actually exist as bindings under
 * chrome.brailleDisplayPrivate.*.
 */
// TODO(b/326623426): Move DomKeyCode to common/.
var DomKeyCode;
(function (DomKeyCode) {
    DomKeyCode["ARROW_DOWN"] = "ArrowDown";
    DomKeyCode["ARROW_LEFT"] = "ArrowLeft";
    DomKeyCode["ARROW_RIGHT"] = "ArrowRight";
    DomKeyCode["ARROW_UP"] = "ArrowUp";
    DomKeyCode["AUDIO_VOLUME_DOWN"] = "AudioVolumeDown";
    DomKeyCode["AUDIO_VOLUME_UP"] = "AudioVolumeUp";
    DomKeyCode["BACKSPACE"] = "Backspace";
    DomKeyCode["DELETE"] = "Delete";
    DomKeyCode["END"] = "End";
    DomKeyCode["ENTER"] = "Enter";
    DomKeyCode["ESCAPE"] = "Escape";
    DomKeyCode["HOME"] = "Home";
    DomKeyCode["INSERT"] = "Insert";
    DomKeyCode["PAGE_DOWN"] = "PageDown";
    DomKeyCode["PAGE_UP"] = "PageUp";
    DomKeyCode["TAB"] = "Tab";
})(DomKeyCode || (DomKeyCode = {}));
/** The set of commands sent from a braille display. */
var BrailleKeyCommand = chrome.brailleDisplayPrivate.KeyCommand;
var BrailleKeyEvent;
(function (BrailleKeyEvent) {
    /**
     * Returns the numeric key code for a DOM level 4 key code string.
     * NOTE: Only the key codes produced by the brailleDisplayPrivate API are
     * supported.
     * @param code DOM level 4 key code.
     * @return The numeric key code, or {@code undefined} if unknown.
     */
    function keyCodeToLegacyCode(code) {
        return BrailleKeyEvent.legacyKeyCodeMap[code];
    }
    BrailleKeyEvent.keyCodeToLegacyCode = keyCodeToLegacyCode;
    /**
     * Returns a char value appropriate for a synthezised key event for a given
     * key code.
     * @param keyCode The DOM level 4 key code.
     * @return Integer character code.
     */
    function keyCodeToCharValue(keyCode) {
        const SPECIAL_CODES = {
            [DomKeyCode.BACKSPACE]: 0x08,
            [DomKeyCode.TAB]: 0x09,
            [DomKeyCode.ENTER]: 0x0A,
        };
        // Note, the Chrome virtual keyboard falls back on the first character of
        // the key code if the key is not one of the above.  Do the same here.
        return SPECIAL_CODES[keyCode] || keyCode.charCodeAt(0);
    }
    BrailleKeyEvent.keyCodeToCharValue = keyCodeToCharValue;
    /*
     * Note: Some of the below mappings contain raw braille dot
     * patterns. These are written out in binary form to make clear
     * exactly what dots in the braille cell make up the pattern. The
     * braille cell is arranged in a 2 by 4 dot grid with each dot
     * assigned a number from 1-8.
     * 1 4
     * 2 5
     * 3 6
     * 7 8
     *
     * In binary form, the dot number minus 1 maps to the bit position
     * (from right to left).
     * For example, dots 1-6-7 would be
     * 0b1100001
     */
    /** Maps a braille pattern to a standard key code. */
    BrailleKeyEvent.brailleDotsToStandardKeyCode = {
        0b1: 'A',
        0b11: 'B',
        0b1001: 'C',
        0b11001: 'D',
        0b10001: 'E',
        0b1011: 'F',
        0b11011: 'G',
        0b10011: 'H',
        0b1010: 'I',
        0b11010: 'J',
        0b101: 'K',
        0b111: 'L',
        0b1101: 'M',
        0b11101: 'N',
        0b10101: 'O',
        0b1111: 'P',
        0b11111: 'Q',
        0b10111: 'R',
        0b1110: 'S',
        0b11110: 'T',
        0b100101: 'U',
        0b100111: 'V',
        0b111010: 'W',
        0b101101: 'X',
        0b111101: 'Y',
        0b110101: 'Z',
        0b110100: '0',
        0b10: '1',
        0b110: '2',
        0b10010: '3',
        0b110010: '4',
        0b100010: '5',
        0b10110: '6',
        0b110110: '7',
        0b100110: '8',
        0b10100: '9',
    };
    /** Maps a braille chord pattern to a standard key code. */
    BrailleKeyEvent.brailleChordsToStandardKeyCode = {
        0b1000000: DomKeyCode.BACKSPACE,
        0b10100: DomKeyCode.TAB,
        0b110101: DomKeyCode.ESCAPE,
        0b101000: DomKeyCode.ENTER,
    };
    /** Maps a braille dot chord pattern to standard key modifiers. */
    BrailleKeyEvent.brailleDotsToModifiers = {
        0b010010: { ctrlKey: true },
        0b100100: { altKey: true },
        0b1000100: { shiftKey: true },
        0b1010010: { ctrlKey: true, shiftKey: true },
        0b1100100: { altKey: true, shiftKey: true },
    };
    /** Map from DOM level 4 key codes to legacy numeric key codes. */
    BrailleKeyEvent.legacyKeyCodeMap = {
        [DomKeyCode.BACKSPACE]: KeyCode.BACK,
        [DomKeyCode.TAB]: KeyCode.TAB,
        [DomKeyCode.ENTER]: KeyCode.RETURN,
        [DomKeyCode.ESCAPE]: KeyCode.ESCAPE,
        [DomKeyCode.HOME]: KeyCode.HOME,
        [DomKeyCode.ARROW_LEFT]: KeyCode.LEFT,
        [DomKeyCode.ARROW_UP]: KeyCode.UP,
        [DomKeyCode.ARROW_RIGHT]: KeyCode.RIGHT,
        [DomKeyCode.ARROW_DOWN]: KeyCode.DOWN,
        [DomKeyCode.PAGE_UP]: KeyCode.PRIOR,
        [DomKeyCode.PAGE_DOWN]: KeyCode.NEXT,
        [DomKeyCode.END]: KeyCode.END,
        [DomKeyCode.INSERT]: KeyCode.INSERT,
        [DomKeyCode.DELETE]: KeyCode.DELETE,
        [DomKeyCode.AUDIO_VOLUME_DOWN]: KeyCode.VOLUME_DOWN,
        [DomKeyCode.AUDIO_VOLUME_UP]: KeyCode.VOLUME_UP,
    };
})(BrailleKeyEvent || (BrailleKeyEvent = {}));
// Add 0-9.
for (let i = '0'.charCodeAt(0); i < '9'.charCodeAt(0); ++i) {
    BrailleKeyEvent.legacyKeyCodeMap[String.fromCharCode(i)] =
        /** @type {Key.Code} */ (i);
}
// Add A-Z.
for (let i = 'A'.charCodeAt(0); i < 'Z'.charCodeAt(0); ++i) {
    BrailleKeyEvent.legacyKeyCodeMap[String.fromCharCode(i)] =
        /** @type {Key.Code} */ (i);
}
// Add the F1 to F12 keys.
for (let i = 0; i < 12; ++i) {
    BrailleKeyEvent.legacyKeyCodeMap['F' + (i + 1)] =
        /** @type {Key.Code} */ (112 + i);
}
TestImportManager.exportForTesting(['BrailleKeyCommand', BrailleKeyCommand], ['BrailleKeyEvent', BrailleKeyEvent]);

// Copyright 2022 The Chromium Authors
// Use of this source code is governed by a BSD-style license that can be
// found in the LICENSE file.
/**
 * @fileoverview Types available for tracking the current event source.
 */
var EventSourceType;
(function (EventSourceType) {
    EventSourceType["NONE"] = "none";
    EventSourceType["BRAILLE_KEYBOARD"] = "brailleKeyboard";
    EventSourceType["STANDARD_KEYBOARD"] = "standardKeyboard";
    EventSourceType["TOUCH_GESTURE"] = "touchGesture";
})(EventSourceType || (EventSourceType = {}));
TestImportManager.exportForTesting(['EventSourceType', EventSourceType]);

// Copyright 2018 The Chromium Authors
// Use of this source code is governed by a BSD-style license that can be
// found in the LICENSE file.
/**
 * @fileoverview Tracks event sources.
 */
class EventSource {
    state_;
    static instance;
    constructor() {
        this.state_ = chrome.accessibilityPrivate.IS_DEFAULT_EVENT_SOURCE_TOUCH ?
            EventSourceType.TOUCH_GESTURE :
            EventSourceType.NONE;
    }
    static init() {
        EventSource.instance = new EventSource();
        BridgeHelper.registerHandler(BridgeConstants.EventSource.TARGET, BridgeConstants.EventSource.Action.GET, () => EventSource.get());
    }
    static set(source) {
        EventSource.instance.state_ = source;
    }
    static get() {
        return EventSource.instance.state_;
    }
}
TestImportManager.exportForTesting(EventSource);

// Copyright 2019 The Chromium Authors
// Use of this source code is governed by a BSD-style license that can be
// found in the LICENSE file.
/**
 * @fileoverview Class definitions of log that are stored in LogStore
 */
/**
 * Supported log types.
 * Note that filter type checkboxes are shown in this order at the log page.
 */
var LogType;
(function (LogType) {
    LogType["SPEECH"] = "speech";
    LogType["SPEECH_RULE"] = "speechRule";
    LogType["BRAILLE"] = "braille";
    LogType["BRAILLE_RULE"] = "brailleRule";
    LogType["EARCON"] = "earcon";
    LogType["EVENT"] = "event";
    LogType["TEXT"] = "text";
    LogType["TREE"] = "tree";
})(LogType || (LogType = {}));
class BaseLog {
    logType;
    date;
    constructor(logType) {
        this.logType = logType;
        this.date = new Date();
    }
    serialize() {
        return {
            logType: this.logType,
            date: this.date.toString(), // Explicit toString(); converts either way.
            value: this.toString(),
        };
    }
    toString() {
        return '';
    }
}
class EventLog extends BaseLog {
    docUrl_;
    rootName_;
    targetName_;
    type_;
    constructor(event) {
        super(LogType.EVENT);
        this.type_ = event.type;
        this.targetName_ = event.target.name;
        // TODO(b/314203187): Not null asserted, check that this is correct.
        this.rootName_ = event.target.root.name;
        this.docUrl_ = event.target.docUrl;
    }
    toString() {
        return `EventType = ${this.type_}, TargetName = ${this.targetName_}, ` +
            `RootName = ${this.rootName_}, DocumentURL = ${this.docUrl_}`;
    }
}
class SpeechLog extends BaseLog {
    category_;
    queueMode_;
    textString_;
    constructor(textString, queueMode, category) {
        super(LogType.SPEECH);
        this.textString_ = textString;
        this.queueMode_ = queueMode;
        this.category_ = category;
    }
    toString() {
        let logStr = 'Speak';
        if (this.queueMode_ === QueueMode.FLUSH) {
            logStr += ' (F)';
        }
        else if (this.queueMode_ === QueueMode.CATEGORY_FLUSH) {
            logStr += ' (C)';
        }
        else if (this.queueMode_ === QueueMode.INTERJECT) {
            logStr += ' (I)';
        }
        else {
            logStr += ' (Q)';
        }
        if (this.category_) {
            logStr += ' category=' + this.category_;
        }
        logStr += ' "' + this.textString_ + '"';
        return logStr;
    }
}
class TextLog extends BaseLog {
    logStr_;
    constructor(logStr, logType) {
        super(logType);
        this.logStr_ = logStr;
    }
    toString() {
        return this.logStr_;
    }
}
class TreeLog extends BaseLog {
    tree_;
    constructor(tree) {
        super(LogType.TREE);
        this.tree_ = tree;
    }
    toString() {
        return this.tree_.treeToString();
    }
}
TestImportManager.exportForTesting(['LogType', LogType]);

// Copyright 2015 The Chromium Authors
// Use of this source code is governed by a BSD-style license that can be
// found in the LICENSE file.
/**
 * @fileoverview Objects used in spannables as annotations for ARIA values
 * and selections.
 */
/** Attached to the value region of a braille spannable. */
class ValueSpan {
    /** The offset of the span into the value. */
    offset;
    constructor(offset) {
        this.offset = offset;
    }
    /**
     * Creates a value span from a json serializable object.
     * @param obj The json serializable object to convert.
     * @return The value span.
     */
    static fromJson(obj) {
        return new ValueSpan(obj.offset);
    }
    /**
     * Converts this object to a json serializable object.
     * @return The JSON representation.
     */
    toJson() {
        return this;
    }
}
Spannable.registerSerializableSpan(ValueSpan, 'ValueSpan', ValueSpan.fromJson, ValueSpan.prototype.toJson);
/** Attached to the selected text within a value. */
class ValueSelectionSpan {
}
Spannable.registerStatelessSerializableSpan(ValueSelectionSpan, 'ValueSelectionSpan');
/**
 * Causes raw cells to be added when translating from text to braille.
 * This is supported by the {@code ExpandingBrailleTranslator}
 * class.
 */
class ExtraCellsSpan {
    cells;
    constructor() {
        this.cells = new Uint8Array(0).buffer;
    }
}
/** Indicates a text form during translation in Liblouis. */
class BrailleTextStyleSpan {
    formType;
    constructor(formType) {
        this.formType = formType;
    }
}
TestImportManager.exportForTesting(BrailleTextStyleSpan, ExtraCellsSpan, ValueSelectionSpan, ValueSpan);

// Copyright 2022 The Chromium Authors
// Use of this source code is governed by a BSD-style license that can be
// found in the LICENSE file.
/**
 * @fileoverview Stores the current focus bounds and manages setting the focus
 * ring location.
 */
class FocusBounds {
    static current_ = [];
    static get() {
        return FocusBounds.current_;
    }
    static set(bounds) {
        FocusBounds.current_ = bounds;
        chrome.accessibilityPrivate.setFocusRings([{
                rects: bounds,
                type: chrome.accessibilityPrivate.FocusType.GLOW,
                color: constants.FOCUS_COLOR,
            }], chrome.accessibilityPrivate.AssistiveTechnologyType.CHROME_VOX);
    }
}
TestImportManager.exportForTesting(['FocusBounds', FocusBounds]);

// Copyright 2018 The Chromium Authors
// Use of this source code is governed by a BSD-style license that can be
// found in the LICENSE file.
/**
 * @fileoverview Creates event stream logger.
 */
var EventType$d = chrome.automation.EventType;
const Action$6 = BridgeConstants.EventStreamLogger.Action;
const TARGET$5 = BridgeConstants.EventStreamLogger.TARGET;
class EventStreamLogger {
    listener_ = (event) => this.onEvent_(event);
    node_;
    static instance;
    constructor(node) {
        this.node_ = node;
    }
    /** Initializes global state for EventStreamLogger. */
    static async init() {
        const desktop = await AsyncUtil.getDesktop();
        EventStreamLogger.instance = new EventStreamLogger(desktop);
        EventStreamLogger.instance.updateAllFilters(SettingsManager.getBoolean('enableEventStreamLogging'));
        BridgeHelper.registerHandler(TARGET$5, Action$6.NOTIFY_EVENT_STREAM_FILTER_CHANGED, (name, enabled) => EventStreamLogger.instance.onFilterChanged_(name, enabled));
    }
    updateAllFilters(checked) {
        for (const type of Object.values(EventType$d)) {
            if (LocalStorage.get(type)) {
                this.onFilterChanged_(type, checked);
            }
        }
    }
    // ============ Private methods =============
    /** Adds onEvent_ to this handler. */
    addListener_(eventType) {
        this.node_.addEventListener(eventType, this.listener_, false);
    }
    /** Removes onEvent_ from this handler. */
    removeListener_(eventType) {
        this.node_.removeEventListener(eventType, this.listener_, false);
    }
    onEvent_(evt) {
        const eventLog = new EventLog(evt);
        LogStore.instance.writeLog(eventLog);
        console.log(eventLog.toString());
    }
    onFilterChanged_(eventType, checked) {
        if (checked) {
            this.addListener_(eventType);
        }
        else {
            this.removeListener_(eventType);
        }
    }
}

// Copyright 2022 The Chromium Authors
// Use of this source code is governed by a BSD-style license that can be
// found in the LICENSE file.
class LogUrlWatcher {
    static instance;
    static init() {
        // TODO(b/314203187): Not null asserted, check that this is correct.
        ChromeVoxPrefs.instance.enableOrDisableLogUrlWatcher();
    }
    static create() {
        if (LogUrlWatcher.instance) {
            return;
        }
        LogUrlWatcher.instance = new LogUrlWatcher();
        ChromeVoxRange.addObserver(LogUrlWatcher.instance);
        // Initialize using the current range.
        LogUrlWatcher.instance.onCurrentRangeChanged(ChromeVoxRange.current);
    }
    static destroy() {
        if (!LogUrlWatcher.instance) {
            return;
        }
        ChromeVoxRange.removeObserver(LogUrlWatcher.instance);
        LogUrlWatcher.instance = null;
    }
    /** ChromeVoxRangeObserver implementation. */
    onCurrentRangeChanged(range, _fromEditing) {
        if (range && range.start && range.start.node && range.start.node.root) {
            // TODO(b/314203187): Not null asserted, check that this is correct.
            LogStore.instance.shouldSkipOutput =
                range.start.node.root.docUrl.indexOf(chrome.extension.getURL('chromevox/mv2/log_page/log.html')) === 0;
        }
        else {
            LogStore.instance.shouldSkipOutput = false;
        }
    }
}

// Copyright 2018 The Chromium Authors
// Use of this source code is governed by a BSD-style license that can be
// found in the LICENSE file.
/**
 * @fileoverview Handles math output and exploration.
 */
/**
 * Handles specialized code to navigate, announce, and interact with math
 * content (encoded in MathML).
 */
class MathHandler {
    node_;
    capturing_ = false;
    static instance = undefined;
    constructor(node) {
        this.node_ = node;
    }
    /**
     * Speaks the current node.
     * @return Boolean indicating whether any math was spoken.
     */
    speak() {
        let mathml = this.node_.mathContent;
        if (!mathml) {
            return false;
        }
        // Ensure it has a `math` root node.
        if (!/^<math>${mathml}<\/math>$/.test(mathml)) {
            mathml = '<math>' + mathml + '</math>';
        }
        let text = null;
        try {
            text = SRE.walk(mathml);
        }
        catch (e) {
            // Swallow exceptions from third party library.
        }
        if (!text) {
            return false;
        }
        // Ensure we are capturing key events once we reach the math node, so that
        // arrow keys will be captured. Note that if sticky mode is on, we are
        // already capturing these events, and this will have no effect. If sticky
        // mode is off, we now capture.
        this.capturing_ = true;
        chrome.accessibilityPrivate.setKeyboardListener(true, true);
        ChromeVox.tts.speak(text, QueueMode.FLUSH);
        ChromeVox.tts.speak(Msgs.getMsg('hint_math_keyboard'), QueueMode.QUEUE);
        return true;
    }
    isCapturing() {
        return this.capturing_;
    }
    node() {
        return this.node_;
    }
    /**
     * Initializes the global instance based on the current cursor range,
     * if it is a Math node.
     * @return Boolean indicating whether an instance was created.
     */
    static init(range) {
        const node = range.start.node;
        if (node && AutomationPredicate.math(node)) {
            MathHandler.instance = new MathHandler(node);
        }
        else if (MathHandler.instance !== undefined) {
            MathHandler.instance = undefined;
        }
        return Boolean(MathHandler.instance);
    }
    /**
     * Ensures the MathHandler instance is still valid after moving to the current
     * cursor range. If not, ensures that the keyboard listener is cleared.
     * @param range The current cursor range
     */
    static checkInstance(range) {
        if (!MathHandler.instance) {
            return;
        }
        if (!range || MathHandler.instance.node() !== range.start.node ||
            range.start.node !== range.end.node) {
            MathHandler.instance = undefined;
            // Ensure we are no longer capturing key events unless sticky mode is on.
            chrome.accessibilityPrivate.setKeyboardListener(true, ChromeVoxPrefs.isStickyPrefOn);
        }
    }
    /**
     * Handles key events.
     * @return Boolean indicating whether an event should propagate.
     */
    static onKeyDown(evt) {
        if (!MathHandler.instance) {
            return true;
        }
        if (evt.ctrlKey || evt.altKey || evt.metaKey || evt.shiftKey ||
            evt.stickyMode) {
            return true;
        }
        const output = SRE.move(evt.keyCode);
        if (output) {
            ChromeVox.tts.speak(output, QueueMode.FLUSH);
        }
        return false;
    }
}
TestImportManager.exportForTesting(MathHandler);

// Copyright 2014 The Chromium Authors
// Use of this source code is governed by a BSD-style license that can be
// found in the LICENSE file.
/**
 * @fileoverview Defines a Tts interface.
 * All TTS engines in ChromeVox conform to the this interface.
 */
/** @interface */
class TtsInterface {
}
TestImportManager.exportForTesting(TtsInterface);

// Copyright 2014 The Chromium Authors
// Use of this source code is governed by a BSD-style license that can be
// found in the LICENSE file.
/**
 * @fileoverview A composite TTS allows ChromeVox to use multiple TTS engines at
 * the same time.
 */
/**
 * A Composite Tts
 * @implements {TtsInterface}
 */
class CompositeTts {
    constructor() {
        /**
         * @type {Array<TtsInterface>}
         * @private
         */
        this.ttsEngines_ = [];
    }
    /**
     * Adds a TTS engine to the composite TTS
     * @param {TtsInterface} tts The TTS to add.
     * @return {!CompositeTts} this.
     */
    add(tts) {
        this.ttsEngines_.push(tts);
        return this;
    }
    /**
     * @param {string} textString
     * @param {QueueMode} queueMode
     * @param {TtsSpeechProperties=} properties
     * @return {TtsInterface}
     * @override
     */
    speak(textString, queueMode, properties) {
        this.ttsEngines_.forEach(engine => engine.speak(textString, queueMode, properties));
        return this;
    }
    /**
     * Returns true if any of the TTSes are still speaking.
     * @override
     */
    isSpeaking() {
        return this.ttsEngines_.some(engine => engine.isSpeaking());
    }
    /**
     * @override
     */
    stop() {
        this.ttsEngines_.forEach(engine => engine.stop());
    }
    /** @override */
    addCapturingEventListener(listener) {
        this.ttsEngines_.forEach(engine => engine.addCapturingEventListener(listener));
    }
    /** @override */
    removeCapturingEventListener(listener) {
        this.ttsEngines_.forEach(engine => engine.removeCapturingEventListener(listener));
    }
    /** @override */
    increaseOrDecreaseProperty(propertyName, increase) {
        this.ttsEngines_.forEach(engine => engine.increaseOrDecreaseProperty(propertyName, increase));
    }
    /** @override */
    setProperty(propertyName, value) {
        this.ttsEngines_.forEach(engine => engine.setProperty(propertyName, value));
    }
    /** @override */
    propertyToPercentage(property) {
        const percentages = this.ttsEngines_.map(engine => engine.propertyToPercentage(property));
        // Return the first non-null percent, or null if all values are null.
        return percentages.find(percent => percent !== undefined) ?? null;
    }
    /** @override */
    getDefaultProperty(property) {
        const defaultValues = this.ttsEngines_.map(engine => engine.getDefaultProperty(property));
        // Return the first non-null value, or null if all values are null.
        return defaultValues.find(value => value !== null) ?? null;
    }
    /** @override */
    toggleSpeechOnOrOff() {
        let value = false;
        this.ttsEngines_.forEach(engine => value = value || engine.toggleSpeechOnOrOff());
        return value;
    }
}

// Copyright 2014 The Chromium Authors
// Use of this source code is governed by a BSD-style license that can be
// found in the LICENSE file.
/**
 * @fileoverview A TTS engine that writes to globalThis.console.
 */
class ConsoleTts {
    /**
     * True if the console TTS is enabled by the user.
     */
    enabled_;
    constructor() {
        this.enabled_ = ChromeVoxPrefs.instance?.getPrefs()['enableSpeechLogging'];
    }
    speak(textString, queueMode, properties) {
        if (this.enabled_ && globalThis.console) {
            const category = properties?.category ?? TtsCategory.NAV;
            const speechLog = new SpeechLog(textString, queueMode, category);
            LogStore.instance.writeLog(speechLog);
            console.log(speechLog.toString());
        }
        return this;
    }
    isSpeaking() {
        return false;
    }
    stop() {
        if (this.enabled_) {
            console.log('Stop');
        }
    }
    // @ts-ignore Unread value.
    addCapturingEventListener(_listener) { }
    // @ts-ignore Unread value.
    removeCapturingEventListener(_listener) { }
    // @ts-ignore Unread value.
    increaseOrDecreaseProperty(_propertyName, _increase) { }
    // @ts-ignore Unread value.
    setProperty(_propertyName, _value) { }
    // @ts-ignore Unread value.
    propertyToPercentage(_property) {
        return null;
    }
    /**
     * Sets the enabled bit.
     * @param enabled The new enabled bit.
     */
    setEnabled(enabled) {
        this.enabled_ = enabled;
    }
    // @ts-ignore Unread value.
    getDefaultProperty(_property) {
        return 0;
    }
    // @ts-ignore Unread value.
    toggleSpeechOnOrOff() {
        return true;
    }
}

// Copyright 2014 The Chromium Authors
// Use of this source code is governed by a BSD-style license that can be
// found in the LICENSE file.
/**
 * Base class for Text-to-Speech engines that actually transform
 * text to speech (as opposed to logging or other behaviors).
 */
class AbstractTts {
    /**
     * Default value for TTS properties.
     * Note that these as well as the subsequent properties might be different
     * on different host platforms (like Chrome, Android, etc.).
     */
    propertyDefault;
    /** Min value for TTS properties. */
    propertyMin;
    /** Max value for TTS properties. */
    propertyMax;
    /** Step value for TTS properties. */
    propertyStep;
    /** Default TTS properties for this TTS engine. */
    ttsProperties = new TtsSpeechProperties();
    /** Substitution dictionary regexp. */
    static substitutionDictionaryRegexp_;
    /** Repetition filter regexp. */
    static repetitionRegexp_ = /([-\/\\|!@#$%^&*\(\)=_+\[\]\{\}.?;'":<>\u2022\u25e6\u25a0])\1{2,}/g;
    /** Regexp filter for negative dollar and pound amounts. */
    static negativeCurrencyAmountRegexp_ = /-[£\$](\d{1,3})(\d+|(,\d{3})*)(\.\d{1,})?/g;
    constructor() {
        const pitchDefault = 1;
        const pitchMin = 0.2;
        const pitchMax = 2.0;
        const pitchStep = 0.1;
        const rateDefault = 1;
        const rateMin = 0.2;
        const rateMax = 5.0;
        const rateStep = 0.1;
        const volumeDefault = 1;
        const volumeMin = 0.2;
        const volumeMax = 1.0;
        const volumeStep = 0.1;
        this.propertyDefault = {
            pitch: pitchDefault,
            rate: rateDefault,
            volume: volumeDefault,
        };
        this.propertyMin = {
            pitch: pitchMin,
            rate: rateMin,
            volume: volumeMin,
        };
        this.propertyMax = {
            pitch: pitchMax,
            rate: rateMax,
            volume: volumeMax,
        };
        this.propertyStep = { rate: rateStep, pitch: pitchStep, volume: volumeStep };
        if (AbstractTts.substitutionDictionaryRegexp_ === undefined) {
            // Create an expression that matches all words in the substitution
            // dictionary.
            const symbols = [];
            for (const symbol in SubstitutionDictionary) {
                symbols.push(symbol);
            }
            const expr = '(' + symbols.join('|') + ')';
            AbstractTts.substitutionDictionaryRegexp_ = new RegExp(expr, 'ig');
        }
    }
    /** TtsInterface implementation. */
    speak(_textString, _queueMode, _properties) {
        return this;
    }
    /** TtsInterface implementation. */
    isSpeaking() {
        return false;
    }
    /** TtsInterface implementation. */
    stop() { }
    /** TtsInterface implementation. */
    addCapturingEventListener(_listener) { }
    /** TtsInterface implementation. */
    removeCapturingEventListener(_listener) { }
    /** TtsInterface implementation. */
    increaseOrDecreaseProperty(propertyName, increase) {
        // TODO(b/314203187): Not null asserted, check that this is correct.
        const step = this.propertyStep[propertyName];
        let current = this.ttsProperties[propertyName];
        current = increase ? current + step : current - step;
        this.setProperty(propertyName, current);
    }
    /** TtsInterface implementation. */
    setProperty(propertyName, value) {
        // TODO(b/314203187): Not null asserted, check that this is correct.
        const min = this.propertyMin[propertyName];
        const max = this.propertyMax[propertyName];
        this.ttsProperties[propertyName] = Math.max(Math.min(value, max), min);
    }
    /**
     * Converts an engine property value to a percentage from 0.00 to 1.00.
     * @param property The property to convert.
     * @return The percentage of the property.
     */
    propertyToPercentage(property) {
        // TODO(b/314203187): Not null asserted, check that this is correct.
        return ((this.ttsProperties[property]) - this.propertyMin[property]) /
            Math.abs(this.propertyMax[property] - this.propertyMin[property]);
    }
    /**
     * Merges the given properties with the default ones. Always returns a
     * new object, so that you can safely modify the result of mergeProperties
     * without worrying that you're modifying an object used elsewhere.
     * @param properties The properties to merge with the current ones.
     * @return The merged properties.
     */
    mergeProperties(properties) {
        const mergedProperties = new TtsSpeechProperties();
        if (this.ttsProperties) {
            for (const [key, value] of Object.entries(this.ttsProperties)) {
                mergedProperties[key] = value;
            }
        }
        if (properties) {
            // const tts = TtsSettings;
            if (typeof (properties[TtsSettings.VOLUME]) === 'number') {
                mergedProperties[TtsSettings.VOLUME] = properties[TtsSettings.VOLUME];
            }
            if (typeof (properties[TtsSettings.PITCH]) === 'number') {
                mergedProperties[TtsSettings.PITCH] = properties[TtsSettings.PITCH];
            }
            if (typeof (properties[TtsSettings.RATE]) === 'number') {
                mergedProperties[TtsSettings.RATE] = properties[TtsSettings.RATE];
            }
            if (typeof (properties[TtsSettings.LANG]) === 'string') {
                mergedProperties[TtsSettings.LANG] = properties[TtsSettings.LANG];
            }
            const context = this;
            const mergeRelativeProperty = function (abs, rel) {
                if (typeof (properties[rel]) === 'number' &&
                    typeof (mergedProperties[abs]) === 'number') {
                    mergedProperties[abs] += properties[rel];
                    // TODO(b/314203187): Not null asserted, check that this is correct.
                    const min = context.propertyMin[abs];
                    const max = context.propertyMax[abs];
                    if (mergedProperties[abs] > max) {
                        mergedProperties[abs] = max;
                    }
                    else if (mergedProperties[abs] < min) {
                        mergedProperties[abs] = min;
                    }
                }
            };
            mergeRelativeProperty(TtsAudioProperty.VOLUME, TtsSettings.RELATIVE_VOLUME);
            mergeRelativeProperty(TtsAudioProperty.PITCH, TtsSettings.RELATIVE_PITCH);
            mergeRelativeProperty(TtsAudioProperty.RATE, TtsSettings.RELATIVE_RATE);
        }
        for (const [key, value] of Object.entries(properties)) {
            if (mergedProperties[key] === undefined &&
                properties[key] !== undefined) {
                mergedProperties[key] = value;
            }
        }
        return mergedProperties;
    }
    /**
     * Method to preprocess text to be spoken properly by a speech
     * engine.
     *
     * 1. Replace any single character with a description of that character.
     *
     * 2. Convert all-caps words to lowercase if they don't look like an
     *    acronym / abbreviation.
     *
     * @param text A text string to be spoken.
     * @param properties Out parameter populated with how to speak the string.
     * @return The text formatted in a way that will sound better by most speech
     *     engines.
     */
    preprocess(text, properties = new TtsSpeechProperties()) {
        if (text.length === 1 && text.toLowerCase() !== text) {
            // Describe capital letters according to user's setting.
            if (SettingsManager.getString('capitalStrategy') === 'increasePitch') {
                // Closure doesn't allow the use of for..in or [] with structs, so
                // convert to a pure JSON object.
                const CAPITAL = Personality.CAPITAL.toJSON();
                for (const prop in CAPITAL) {
                    if (properties[prop] === undefined) {
                        properties[prop] = CAPITAL[prop];
                    }
                }
            }
            else if (SettingsManager.getString('capitalStrategy') === 'announceCapitals') {
                text = Msgs.getMsg('announce_capital_letter', [text]);
            }
        }
        if (!SettingsManager.getBoolean('usePitchChanges')) {
            delete properties['relativePitch'];
        }
        // Since dollar and sterling pound signs will be replaced with text, move
        // them to after the number if they stay between a negative sign and a
        // number.
        text = text.replace(AbstractTts.negativeCurrencyAmountRegexp_, match => {
            const minus = match[0];
            const number = match.substring(2);
            const currency = match[1];
            return minus + number + currency;
        });
        // Substitute all symbols in the substitution dictionary. This is pretty
        // efficient because we use a single regexp that matches all symbols
        // simultaneously.
        text = text.replace(AbstractTts.substitutionDictionaryRegexp_, function (symbol) {
            return ' ' + SubstitutionDictionary[symbol] + ' ';
        });
        // Handle single characters that we want to make sure we pronounce.
        if (text.length === 1) {
            return CharacterDictionary[text] ?
                Msgs.getMsgWithCount(CharacterDictionary[text], 1) :
                text.toUpperCase();
        }
        // Expand all repeated characters.
        text = text.replace(AbstractTts.repetitionRegexp_, AbstractTts.repetitionReplace_);
        return text;
    }
    /**
     * Constructs a description of a repeated character. Use as a param to
     * string.replace.
     * @param match The matching string.
     * @return The description.
     */
    static repetitionReplace_(match) {
        const count = match.length;
        return ' ' + Msgs.getMsgWithCount(CharacterDictionary[match[0]], count) +
            ' ';
    }
    /** TtsInterface implementation. */
    getDefaultProperty(property) {
        // TODO(b/314203187): Not null asserted, check that this is correct.
        return this.propertyDefault[property];
    }
    /** TtsInterface implementation. */
    toggleSpeechOnOrOff() {
        return true;
    }
}

// Use of this source code is governed by a MIT license that can be
// found in the LICENSE file.
var JaPhoneticMap;
(function (JaPhoneticMap) {
    /** A map containing phonetic disambiguation data for Japanese. */
    JaPhoneticMap.MAP = new Map([
        [' ', ' スペース'],
        ['團', 'ダン，ダンタイコウドウノダンノキュウジ'],
        ['土', 'ドヨウビ ノ ド，ツチ'],
        ['圜', 'カン，クニガマエニシゼンカンキョウノカンノツクリ'],
        ['園', 'ドウブツエン ノ エン'],
        ['圓', 'エン，ヒャクエンダマノエンノキュウジ'],
        ['圖', 'ズ，トショカンノトノキュウジ'],
        ['國', 'クニ，コクゴノコクノキュウジ'],
        ['圈', 'ケン，シュトケンノケンノキュウジ'],
        ['圉', 'ギョ，クニガマエニシアワセ'],
        ['圏', 'シュトケン ノ ケン'],
        ['圍', 'カコム，イゴノイノキュウジ'],
        ['圃', 'ハタケ ノ イミ ノ ホジョウ ノ ホ'],
        ['圀', 'クニノイタイジ，トクガワミツクニノクニ'],
        ['圄', 'ゴ，クニガマエニタゴサクノゴ'],
        ['圻', 'キ，ツチヘンニショクパンイッキンノキン'],
        ['圸', 'ママ，ツチヘンニヤマ'],
        ['地', 'チキュウ ノ チ'],
        ['圷', 'アクツ，ツチヘンニジョウゲノゲ'],
        ['在', 'ソンザイ ノ ザイ'],
        ['圭', 'ツチ フタツ ノ ケイ'],
        ['圦', 'イリ，ツチヘンニニュウガクスルノニュウ'],
        ['圧', 'アツリョク ノ アツ'],
        ['亀', 'ドウブツ ノ カメ，キ'],
        ['棲', 'ドウセイスル ノ セイ，スム'],
        ['亂', 'コンランスルノラン，ミダレル'],
        ['亅', 'ケツ，カンジノブシュノハネボウ'],
        ['了', 'リョウショウスル ノ リョウ'],
        ['争', 'キョウソウ ノ ソウ，アラソウ'],
        ['予', 'ヨテイ ノ ヨ'],
        ['事', 'ジケン ノ ジ，コト'],
        ['亊', 'ジケンノジ，コトノイタイジ'],
        ['二', 'カンスウジ ノ ニ'],
        ['于', 'ウ，ウチュウノウノウカンムリヲトッタモノ'],
        ['云', 'ウンヌン ノ ウン，イウ'],
        ['棡', 'コウ，キヘンニオカヤマケンノオカ'],
        ['互', 'ゴカク ノ ゴ，タガイ'],
        ['井', 'イドミズ ノ イ'],
        ['五', 'カンスウジ ノ ゴ'],
        ['棧', 'サンバシノサンノキュウジ'],
        ['亙', 'ジンメイヨウワタル ノ キョヨウジタイ'],
        ['亘', 'ジンメイ ノ ワタル，カキネ ノ カキ ノ ツクリ'],
        ['些', 'ササイ ノ サ，イササカ'],
        ['亜', 'アネッタイ ノ ア'],
        ['亟', 'キ，ホッキョクノキョクノツクリ'],
        ['亞', 'アネッタイノアノキュウジ'],
        ['亡', 'ボウメイスル ノ ボウ'],
        ['亠', 'トウ，ブシュノナベブタ'],
        ['棒', 'テツボウ ノ ボウ'],
        ['亢', 'コウ，テイコウスルノコウノツクリ'],
        ['亥', 'イ，ジュウニシ ノ イノシシ'],
        ['交', 'コウツウ ノ コウ，マジワル'],
        ['亦', 'セツゾクシ ノ マタ，ヘンカスル ノ ヘン ノ ウエガワ'],
        ['棘', 'トゲ，キョクヒドウブツノキョク'],
        ['亨', 'キョウ，ナベブタニ クチニ リョウ'],
        ['享', 'キョウラクテキ ノ キョウ'],
        ['亭', 'テイシュ ノ テイ'],
        ['京', 'トウキョウ ノ キョウ'],
        ['亮', 'ジンメイ ノ リョウ，ショカツリョウ ノ リョウ'],
        ['亰', 'トウキョウノキョウノイタイジ'],
        ['亳', 'ハク，コダイチュウゴクノインノミヤコノヒトツ'],
        ['棄', 'ハキスル ノ キ，ステル'],
        ['棆', 'リン，キヘンニリンリテキナノリン'],
        ['亶', 'タン，ブツダンノダンノツクリ'],
        ['棉', 'ショクブツ ノ ワタ，キヘン'],
        ['棊', 'キ，ショウギヲサスノギノイタイジ'],
        ['人', 'ヒト，ニンゲン ノ ニン'],
        ['棍', 'コンボウノコン，キヘンニコンチュウノコン'],
        ['酳', 'イン，ジュウニシノトリノミギニイトガシラニツキ'],
        ['酵', 'コウボキン ノ コウ'],
        ['k', 'ケイ キッチン'],
        ['酪', 'ラクノウ ノ ラク'],
        ['霊', 'レイエン ノ レイ，タマ'],
        ['酘', 'トウ，ジュウニシノトリノミギニルマタ'],
        ['‡', 'ダブルダガー'],
        ['酖', 'タン，ジュウニシノトリノミギニシズムノツクリ'],
        ['酉', 'トリ，ジュウニシ ノ トリ'],
        ['酋', 'シュウチョウ ノ シュウ'],
        ['棹', 'トウ，サオ，キヘンニタッキュウノタク'],
        ['棺', 'カンオケ ノ カン，ヒツギ'],
        ['酌', 'バンシャク ノ シャク'],
        ['A', 'エイ アニマル'],
        ['蓁', 'シン，クサカンムリニシンノシコウテイノシン'],
        ['蓆', 'セキ，クサカンムリニザセキノセキ'],
        ['蓄', 'チョチク ノ チク'],
        ['蓊', 'オウ，クサカンムリニオキナ'],
        ['蓋', 'ズガイコツ ノ ガイ，フタ'],
        ['蓉', 'ショクブツ ノ フヨウ ノ ヨウ'],
        ['蓍', 'シ，クサカンムリニヒレノツクリ'],
        ['蓐', 'ジョク，クサカンムリニブジョクスルノジョク'],
        ['蓑', 'ミノムシ ノ ミノ'],
        ['蓖', 'ヒマシユノヒ'],
        ['蓚', 'シュウ，クサカンムリニシュウリスルノシュウ'],
        ['蓙', 'ゴザノザ，クサカンムリニギンザノザ'],
        ['棠', 'トウ，コッカイギジドウノドウノツチノカワリニモクザイノモク'],
        ['蓮', 'レンコン ノ レン，ハス'],
        ['蓬', 'ホウライサン ノ ホウ，ヨモギ'],
        ['棣', 'テイ，キヘンニレイヅクリ'],
        ['蓴', 'ジュンサイノジュン'],
        ['蓿', 'シュク，クサカンムリニシュクダイノシュク'],
        ['蓼', 'タデクウムシモスキズキノタデ'],
        ['有', 'ユウメイジン ノ ユウ，アル'],
        ['月', 'ゲツヨウビ ノ ゲツ，ツキ'],
        ['朋', 'ホウユウ ノ ホウ，トモ'],
        ['服', 'ヨウフク ノ フク'],
        ['朏', 'ヒ、ミカヅキ、ツキヘンニデル'],
        ['最', 'サイショ ノ サイ，モットモ'],
        ['會', 'カイシャノカイ，アウノキュウジ'],
        ['望', 'キボウ ノ ボウ，ノゾム'],
        ['朝', 'アサヒ ノ アサ，チョウ'],
        ['期', 'キタイスル ノ キ'],
        ['朞', 'キ、キホンノキノツチノカワリニツキ'],
        ['朕', 'チンオモウニ ノ チン'],
        ['朔', 'サクジツ ノ サク，ツイタチ'],
        ['朗', 'ロウドク ノ ロウ'],
        ['朖', 'ロウドクノロウノイタイジ、ツキヘンニカイリョウスルノリョウ'],
        ['木', 'モクヨウビ ノ モク，キ'],
        ['末', 'ネンマツ ノ マツ，スエ'],
        ['未', 'ミライ ノ ミ'],
        ['札', 'ナフダ ノ フダ，サツ'],
        ['本', 'ホンモノ ノ ホン，モト'],
        ['朮', 'ジュツ、ジュツゴノジュツノツクリ'],
        ['朧', 'モウロウトスルノロウ'],
        ['朦', 'モウロウトスルノモウ'],
        ['朸', 'ロク、キヘンニチカラ'],
        ['机', 'ツクエ'],
        ['朽', 'ロウキュウカスル ノ キュウ，クチル'],
        ['森', 'シンリン ノ シン，モリ'],
        ['朿', 'シ、トゲ、シゲキスルノシノヒダリガワ'],
        ['朱', 'シュイロ ノ シュ'],
        ['棯', 'ジン，キヘンニザンネンノネン'],
        ['朴', 'ソボクナ ノ ボク'],
        ['朷', 'トウ、キヘンニカタナ'],
        ['朶', 'ミミタブヲイミスルジダノダ'],
        ['棔', 'コン，キヘンニコンスイジョウタイノコン'],
        ['棕', 'シュロノキノシュ'],
        ['棗', 'ショクブツノナツメ'],
        ['棚', 'タナオロシ ノ タナ'],
        ['砿', 'イシヘン ニ ヒロイ ノ コウ'],
        ['砺', 'レイ，イシヘン ニ ガンダレ ニ マン'],
        ['破', 'ハカイスル ノ ハ，ヤブル'],
        ['砲', 'ホウガンナゲ ノ ホウ'],
        ['砥', 'トイシ ノ ト'],
        ['砦', 'トリデ，コレ ノ シタニ イシ'],
        ['棟', 'ビョウトウ ノ トウ，ムネ'],
        ['砠', 'ショ，イシヘンニクミノツクリ'],
        ['研', 'ケンキュウ ノ ケン'],
        ['砕', 'クダク，フンサイスル ノ サイ'],
        ['砒', 'ゲンソノヒソノヒ'],
        ['砌', 'セイ，ミギリ，イシヘンニセツジツナノセツ'],
        ['砂', 'サトウ ノ サ，スナ'],
        ['髱', 'ホウ，カミガシラニツツム'],
        ['髷', 'キョク，マゲ，カミガシラニマゲル'],
        ['髴', 'ホウフツノフツ，カミガシラニフットウスルノフツノツクリ'],
        ['髻', 'ケイ，カミガシラニダイキチノキチ'],
        ['髣', 'ホウフツノホウ，カミガシラニセイホウケイノホウ'],
        ['髢', 'テイ，カミガシラニナリ，カモジ'],
        ['髦', 'ボウ，カミガシラニケイトノケ'],
        ['髫', 'チョウ，カミガシラニマネクノツクリ'],
        ['髪', 'カミノケ ノ カミ，ハツ'],
        ['髯', 'ゼン，ホウヒゲ'],
        ['髮', 'カミノケノカミノキュウジ'],
        ['髭', 'クチヒゲ ノ ヒゲ'],
        ['髓', 'セキズイノズイノキュウジ'],
        ['髑', 'ドクロノドク'],
        ['棋', 'キ，ショウギ ノ ギ'],
        ['體', 'カラダ，タイイクノタイノキュウジ'],
        ['高', 'タカイ，サイコウ ノ コウ'],
        ['髟', 'ヒョウ，カミガシラ'],
        ['髞', 'ソウ，タカイヒクイノタカイニアヤツルノツクリ'],
        ['髀', 'ヒ，ホネヘンニイヤシイ'],
        ['髄', 'セキズイ ノ ズイ'],
        ['髏', 'ドクロノロ'],
        ['巉', 'ザン，ヤマヘンニザンゲンスルノザンノツクリ'],
        ['巍', 'ギ，ヤマノシタニギシワジンデンノギ'],
        ['巌', 'ガンリュウジマ ノ ガン，イワオ'],
        ['巛', 'セン，ブシュノマガリガワ'],
        ['州', 'キュウシュウチホウ ノ シュウ'],
        ['川', 'カセン ノ セン，カワ'],
        ['巓', 'ヤマノイタダキヲイミスルサンテンノテン'],
        ['巒', 'ラン，コイビトノコイノキュウジノココロノカワリニヤマ'],
        ['巖', 'ゲンジュウナノゲン，キビシイノキュウジ'],
        ['巫', 'ジンジャノミコノフ'],
        ['巨', 'キョダイナ ノ キョ'],
        ['差', 'サベツカスル ノ サ'],
        ['巣', 'ハチノス ノ ス'],
        ['巡', 'ジュンカイ ノ ジュン，メグル'],
        ['巧', 'ギコウテキナ ノ コウ，タクミ'],
        ['左', 'ヒダリ，サユウ ノ サ'],
        ['工', 'ズコウ ノ コウ'],
        ['巻', 'ノリマキ ノ マキ'],
        ['巾', 'ゾウキン ノ キン，ハバ'],
        ['巽', 'ホウガク ノ タツミ'],
        ['巳', 'ミ，ジュウニシ ノ ミ'],
        ['已', 'ブンポウノイゼンケイノイ'],
        ['己', 'ジコショウカイ ノ コ，オノレ'],
        ['樒', 'ミツ，キヘンニヒミツノミツ'],
        ['巷', 'チマタ ノ コエ ノ チマタ，コウ'],
        ['巵', 'シ，サカズキノイタイジ'],
        ['巴', 'トモエナゲ ノ トモエ'],
        ['燕', 'ツバメ'],
        ['燔', 'ハン，ヒヘンニバンゴウノバン'],
        ['燗', 'アツカンノカン'],
        ['燐', 'リンサンヒリョウ ノ リン'],
        ['燒', 'ニクヲヤクノヤクノキュウジ'],
        ['營', 'エイギョウノエイ，イトナムノキュウジ'],
        ['燃', 'ネンリョウ ノ ネン，モエル'],
        ['燎', 'リョウゲンノヒノリョウ'],
        ['燉', 'トン，ヒヘンニチュウゴクノトンコウノトン'],
        ['燈', 'トウダイ ノ トウ，ヒヘン ニ ノボル'],
        ['燵', 'コタツノタツ'],
        ['燼', 'カイジンニキスノジン'],
        ['燿', 'ヨウ，ヒヘンニニチヨウビノヨウノツクリ'],
        ['燹', 'セン，ブタノツクリフタツノシタニカヨウビノカ'],
        ['燻', 'クンセイノクン，クスベルノイタイジ'],
        ['燥', 'カンソウキ ノ ソウ'],
        ['燧', 'スイ，ヒヘンニトゲルノキュウジ'],
        ['燦', 'サンゼントカガヤク ノ サン'],
        ['燠', 'イク，ヒヘンニオクバノオクノキュウジ'],
        ['燭', 'ショクダイ ノ ショク，ロウソク ノ ソク'],
        ['燬', 'キ，ヒヘンニメイヨキソンノキ'],
        ['燮', 'チョウワサセルノイミノショウ'],
        ['齒', 'ハブラシノハノキュウジ'],
        ['鰥', 'カン，オトコヤモメノヤモオ'],
        ['麸', 'オフノフ，ムギニオット'],
        ['樵', 'キコリ'],
        ['麺', 'メンルイ ノ メン'],
        ['齟', 'ソゴヲキタスノソ'],
        ['齎', 'モタラス，セイ'],
        ['鰯', 'イワシ，サカナヘン ニ ヨワイ'],
        ['齲', 'ウシノウ，ムシバ'],
        ['X', 'エックス エックスセン'],
        ['齷', 'アクセクスルノアク，ハノキュウジノミギニオクジョウノオク'],
        ['齶', 'アゴ，ガク，ハノキュウジノミギニガクゼントスルノガクノツクリ'],
        ['訳', 'ツウヤク ノ ヤク，ワケ'],
        ['許', 'キョカスル ノ キョ，ユルス'],
        ['閇', 'ヘイテンノヘイ，トジルノイタイジ'],
        ['門', 'センモン ノ モン'],
        ['訶', 'マカフシギノカ'],
        ['閂', 'カンヌキ'],
        ['訴', 'ウッタエル，ソショウ ノ ソ'],
        ['註', 'チュウシャク ノ チュウ，ゴンベン ニ シュジンコウ ノ シュ'],
        ['診', 'シンサツスル ノ シン，ミル'],
        ['閏', 'ウルウドシ ノ ウルウ'],
        ['閉', 'ヘイテン ノ ヘイ，トジル'],
        ['閊', 'ムネガツカエルノツカエル，モンガマエニヤマ'],
        ['証', 'ショウメイショ ノ ショウ'],
        ['訣', 'ケツベツ ノ ケツ，ワカレル'],
        ['閖', 'ロウ，モンガマエニミズ'],
        ['閑', 'カンコドリ ノ カン，ヒマ'],
        ['訥', 'ボクトツノトツ'],
        ['間', 'ジカン ノ カン，アイダ'],
        ['訪', 'ホウモンスル ノ ホウ，オトズレル'],
        ['閘', 'コウ，モンガマエニコウオツノコウ'],
        ['閙', 'ドウ，モンガマエニイチバノイチ'],
        ['設', 'セツビ ノ セツ'],
        ['訓', 'クンレン ノ クン'],
        ['閥', 'ハバツ ノ バツ'],
        ['訐', 'ケツ，ゴンベンニホス'],
        ['託', 'タクジショ ノ タク'],
        ['訖', 'キツ，ゴンベンニアマゴイノコウ'],
        ['関', 'カンケイ ノ カン，セキ'],
        ['閣', 'テンシュカク ノ カク'],
        ['訛', 'オクニナマリノナマリ、カ'],
        ['閭', 'リョ，モンガマエニオフロノロ'],
        ['記', 'キロク ノ キ，シルス'],
        ['訟', 'ソショウ ノ ショウ'],
        ['訝', 'ケゲンナカオノゲン、イブカル'],
        ['訃', 'フホウヲキクノフ'],
        ['訂', 'テイセイスル ノ テイ'],
        ['r', 'アール ルーム'],
        ['閲', 'エツランスル ノ エツ'],
        ['閼', 'フサグヲイミスルモンガマエノアツ'],
        ['訊', 'ジンモン ノ ジン，トウ'],
        ['閾', 'イキ，モンガマエニアルイハノアル'],
        ['計', 'トケイ ノ ケイ，ハカル'],
        ['討', 'トウロンスル ノ トウ'],
        ['訌', 'コウ、ゴンベンニズガコウサクノコウ'],
        ['齧', 'ゲッシルイノゲツ，カジル'],
        ['送', 'ホウソウキョク ノ ソウ，オクル'],
        ['.', 'ピリオド'],
        ['蕀', 'キョク，クサカンムリニキョクヒドウブツノキョク'],
        ['蕁', 'ジンマシンノジン'],
        ['蕃', 'バン，クサカンムリ ニ バンゴウ ノ バン'],
        ['蕎', 'ショクブツ ノ ソバ，キョウ'],
        ['蕈', 'シン，クサカンムリニニシノシタニハヤオキノハヤイ'],
        ['蕉', 'ミズバショウ ノ ショウ'],
        ['蕊', 'メシベ ノ シベ，ズイ'],
        ['蕋', 'メシベノシベ，ズイノイタイジ，クサカンムリニシケツスルノシミッツ'],
        ['蕕', 'ユウ，クサカンムリニユウヨスルノユウ'],
        ['蕗', 'ショクブツ ノ フキ'],
        ['蕘', 'ジョウ，クサカンムリニギョウシュンノギョウノキュウジ'],
        ['蕚', 'ショクブツノハナノガクノイタイジ'],
        ['蕣', 'シュン，クサカンムリニギョウシュンノシュン'],
        ['蕭', 'ショウ，クサカンムリニゲンシュクナノシュクノキュウジ'],
        ['蕨', 'ショクブツ ノ ワラビ'],
        ['蕩', 'ホウトウムスコ ノ トウ'],
        ['蕪', 'ヤサイ ノ カブ'],
        ['蕷', 'ヨ，クサカンムリニアズケル'],
        ['蕾', 'ツボミ，ライ，クサカンムリニカミナリ'],
        ['駁', 'ハンバクスル ノ バク'],
        ['嚔', 'テイ，クサメ'],
        ['嚏', 'テイ，クサメノイタイジ'],
        ['嚊', 'カカアデンカノカカ，クチヘンニハナカゼノハナ'],
        ['嚆', 'ハジマリヲイミスルコウシノコウ'],
        ['嚇', 'イカクスル ノ カク'],
        ['嚀', 'ネイ，クチヘンニテイネイノネイ'],
        ['嚼', 'ソシャクスルノシャク，カム'],
        ['嚴', 'ゲンジュウナノゲン，キビシイノキュウジ'],
        ['嚶', 'オウ，クチヘンニサクラノキュウジノツクリ'],
        ['嚮', 'キョウ，ウマレコキョウノキョウノシタニムカウ'],
        ['嚥', 'エンゲスルノエン'],
        ['嚠', 'リュウ，クチヘンニリュウホウノリュウ'],
        ['嚢', 'ヒョウノウ ノ ノウ，フクロ'],
        ['桾', 'クン，キヘンニショクンノクン'],
        ['桿', 'カン、キヘンニホスノイタイジ'],
        ['桶', 'オケ，フロオケ ノ オケ'],
        ['桷', 'カク，キヘンニツノ'],
        ['桴', 'フ，キヘンニツメノシタニコドモノコ'],
        ['档', 'トウ，キヘンニベントウノトウ'],
        ['桧', 'ショクブツ ノ ヒノキ，キヘン ニ カイシャ ノ カイ'],
        ['桙', 'ウ，ホコ，キヘンニシャカムニノム'],
        ['桟', 'サンバシ ノ サン'],
        ['桜', 'サクラ'],
        ['桝', 'キヘン ノ マス ノ イタイジ'],
        ['桓', 'カンムテンノウ ノ カン'],
        ['桐', 'キリ ノ タンス ノ キリ'],
        ['桑', 'クワノキ ノ クワ，ソウ'],
        ['桔', 'キヘン ニ キチ，キキョウ ノ イチモジメ'],
        ['案', 'アンナイ ノ アイ'],
        ['桎', 'シッコクトナルノシツ，アシカセ，キヘンニゲシノシ'],
        ['桍', 'コ，キヘンニハカマノツクリ'],
        ['桂', 'ゲッケイジュ ノ ケイ，カツラ'],
        ['桃', 'モモタロウ ノ モモ'],
        ['桀', 'ケツ，ゴウケツノケツノツクリ'],
        ['桁', 'ケタハズレ ノ ケタ'],
        ['框', 'アガリカマチノカマチ'],
        ['碆', 'ハ，ハチョウガアウノハノシタニイシバシノイシ'],
        ['碇', 'テイハク ノ テイ，イカリ'],
        ['碁', 'ゴイシ ノ ゴ'],
        ['碎', 'クダク，フンサイスルノサイノキュウジ'],
        ['碌', 'ロク，イシヘンニミドリノキュウジノツクリ'],
        ['碍', 'ユウズウムゲ ノ ゲ，ガイ'],
        ['颪', 'オロシ，ジョウゲノシタノシタニカゼ'],
        ['驥', 'キビニフスノキ'],
        ['碗', 'イシヘン ノ チャワン ノ ワン'],
        ['碕', 'イシヘン ニ キミョウナ ノ キ'],
        ['碓', 'ウス，イシヘン ニ フルトリ'],
        ['碑', 'キネンノヒ ノ ヒ，イシブミ'],
        ['碚', 'ハイ，イシヘンニバイリツノバイノツクリ'],
        ['驫', 'ヒョウ，ウマガミッツ'],
        ['驪', 'レイ，ウマヘンニウルワシイ'],
        ['驕', 'キョウマンナノキョウ，オゴル'],
        ['碧', 'ヘキガン ノ ヘキ'],
        ['驗', 'シケンヲウケルノケンノキュウジ'],
        ['碣', 'ケツ，イシヘンニカッショクノカツノツクリ'],
        ['碯', 'ノウ，イシヘンニズノウノノウノキュウジノツクリ'],
        ['驟', 'シュウウノシュウ，ウマヘンニジュラクダイノジュ'],
        ['碪', 'キヌタ，イシヘンニハナハダシイ'],
        ['驛', 'エキビルノエキノキュウジ'],
        ['碩', 'セキガク ノ セキ，イシヘン ニ ページ'],
        ['驅', 'ヨンリンクドウノク，カケルノキュウジ'],
        ['碵', 'セキ，イシヘンニテイシュクノテイ'],
        ['驀', 'バクシンスルノバク'],
        ['驃', 'ヒョウ，ウマヘンニトウヒョウスルノヒョウ'],
        ['驂', 'サン，ウマヘンニサンカスルノサン，マイルノキュウジ'],
        ['碾', 'テン，イシヘンニハッテンノテン'],
        ['碼', 'ナガサノタンイノヤード'],
        ['確', 'カクニン ノ カク，タシカ'],
        ['駈', 'カケル，ウマヘン ニ オカ'],
        ['颱', 'タイフウヲイミスルタイ，カゼノミギニダイドコロノダイ'],
        ['o', 'オー オープン'],
        ['颶', 'グ，カゼノミギニドウグノグ'],
        ['悒', 'ユウ，リッシンベンニトユウノユウ'],
        ['悗', 'バン，リッシンベンニメンキョノメン'],
        ['悖', 'ボツ，リッシンベンニボッパツスルノボツノヒダリガワ'],
        ['悔', 'コウカイスル ノ カイ，クヤム'],
        ['悛', 'カイシュンノジョウノシュン'],
        ['悚', 'ショウ，リッシンベンニタバ'],
        ['悟', 'サトル，カクゴ ノ ゴ'],
        ['悃', 'コン，リッシンベンニコマル'],
        ['悁', 'エン，リッシンベンニキヌイトノキヌノツクリ'],
        ['悄', 'ショウゼンタルスガタノショウ，リッシンベンニショウゾウガノショウ'],
        ['悋', 'シットヲイミスルリンキノリン'],
        ['闃', 'ケキ，モンガマエニモクテキノモクノシタニイヌ'],
        ['悉', 'シッカイ ノ シツ，コトゴトク'],
        ['悍', 'セイカンナカオツキノカン'],
        ['悌', 'テイ，リッシンベン ニ オトウト'],
        ['悳', 'トク，ドウトクノトクノイタイジ'],
        ['悲', 'ヒゲキ ノ ヒ，カナシイ'],
        ['悶', 'クモンスル ノ モン，モダエル'],
        ['悵', 'チョウ，リッシンベンニチョウタンノチョウ'],
        ['悴', 'ショウスイスルノスイ'],
        ['悸', 'ドウキガスルノキ'],
        ['悽', 'セイ，リッシンベンニフサイノサイ，ツマ'],
        ['悼', 'アイトウ ノ トウ，イタム'],
        ['患', 'カンジャ ノ カン，ワズラウ'],
        ['悠', 'ユウユウジテキ ノ ユウ'],
        ['悧', 'レイリノリ，リッシンベンニケンリノリ'],
        ['悦', 'エツラク ノ エツ，ヨロコブ'],
        ['悪', 'ゼンアク ノ アク，ワルイ'],
        ['悩', 'ナヤム，クノウスル ノ ノウ'],
        ['E', 'イー エッグ'],
        ['闊', 'ウカツナノカツ'],
        ['闖', 'チンニュウスルノチン，モンガマエニウマ'],
        ['闕', 'ケイバツノケッショノケツ'],
        ['誰', 'ダレカサン ノ ダレ'],
        ['課', 'ホウカゴ ノ カ'],
        ['誼', 'ヨシミヲツウズル ノ ヨシミ，ギ'],
        ['調', 'チョウサ ノ チョウ，シラベル'],
        ['誹', 'ヒボウスル ノ ヒ，ソシル'],
        ['誥', 'コウ，ゴンベンニツゲル'],
        ['誤', 'ゴカイスル ノ ゴ，アヤマリ'],
        ['誦', 'シヲアンショウスルノショウ'],
        ['誡', 'カイ，ゴンベンニイマシメル'],
        ['誠', 'セイジツナ ノ セイ，マコト'],
        ['誣', 'ブコクザイノブ'],
        ['読', 'ヨム，ドクショ ノ ドク'],
        ['説', 'セツメイ ノ セツ，トク'],
        ['誨', 'カイ，ゴンベンニマイニチノマイ'],
        ['誕', 'タンジョウビ ノ タン'],
        ['關', 'カンケイスルノカン，セキノキュウジ'],
        ['誑', 'タブラカス，キョウ'],
        ['誓', 'チカウ，セイヤクショ ノ セイ'],
        ['語', 'コクゴ ノ ゴ，カタル'],
        ['誘', 'サソウ，ユウワク ノ ユウ'],
        ['誚', 'ショウ，ゴンベンニショウゾウガノショウ'],
        ['誅', 'テンチュウヲクワエルノチュウ'],
        ['誄', 'ルイ，ゴンベンニライスキ'],
        ['誇', 'ホコリ，コチョウスル ノ コ'],
        ['誂', 'アツラエル，チョウ'],
        ['認', 'カクニン ノ ニン，ミトメル'],
        ['誌', 'ザッシ ノ シ'],
        ['誉', 'メイヨ ノ ヨ，ホマレ'],
        ['闥', 'タツ，モンガマエニハイタツノタツ'],
        ['闢', 'テンチカイビャクノビャク'],
        ['嵌', 'ハメキザイクノハメル，カン'],
        ['嵎', 'グウ，ヤマヘンニグウゼンノグウノツクリ'],
        ['嵋', 'ビ，ヤマヘンニマユゲノマユ'],
        ['嵜', 'ナガサキケンノサキノイタイジ'],
        ['℡', 'テレフォン'],
        ['嵐', 'アラシヤマ ノ アラシ'],
        ['嵒', 'ガン，シナモノノシナノシタニヤマ'],
        ['嵬', 'カイ，ヤマノシタニオニ'],
        ['嵯', 'サガテンノウ ノ サ'],
        ['嵩', 'カサアゲスル ノ カサ'],
        ['嵶', 'タワ，ヤマヘンニヨワイ'],
        ['嵳', 'サ，ヤマノシタニサベツカスルノサ'],
        ['灘', 'ナダ，サンズイ ニ ムズカシイ'],
        ['灑', 'サイ，サンズイニチュウカナベノカ'],
        ['灌', 'カンガイヨウスイノカン'],
        ['嫁', 'ハナヨメ ノ ヨメ'],
        ['災', 'サイナン ノ サイ，ワザワイ'],
        ['灼', 'シャクネツ ノ シャク'],
        ['灸', 'オキュウヲスエル ノ キュウ'],
        ['灰', 'ハイイロ ノ ハイ'],
        ['灯', 'ケイコウトウ ノ トウ'],
        ['火', 'カヨウビ ノ カ，ヒ'],
        ['灣', 'ワンガンノワンノキュウジ'],
        ['嘯', 'ショウ，ウソブク，クチヘンニシシュウノシュウノツクリ'],
        ['嘩', 'クチゲンカ ノ カ'],
        ['嘶', 'イナナク，セイ'],
        ['嘴', 'クチバシ，シ'],
        ['嘲', 'チョウショウスルノチョウ，アザケル'],
        ['嘱', 'ショクタクスル ノ ショク'],
        ['嘸', 'サゾカシノサゾ，クチヘンニムリノム'],
        ['嘆', 'ナゲク，タンガンスル ノ タン'],
        ['嘉', 'カ，ヨロコブ ノ シタガ クワエル'],
        ['嘖', 'サク，クチヘンニセキニンノセキ'],
        ['嘗', 'ガシンショウタン ノ ショウ，ナメル'],
        ['嘔', 'オウトスルノオウ，クチヘンニクベツスルノクノキュウジ'],
        ['嘛', 'ラマキョウノマ，クチヘンニマスイノマ'],
        ['嘘', 'ウソツキ ノ ウソ'],
        ['槽', 'ヨクソウ ノ ソウ'],
        ['槿', 'キン，ショクブツノムクゲ'],
        ['槹', 'コウ，キヘンニサツキノイチモジメ'],
        ['槻', 'ショクブツ ノ ツキ，キヘン ニ キソクテキ ノ キ'],
        ['槲', 'コク，キヘンニツノニホクトシチセイノト'],
        ['槭', 'シュク，キヘンニトオイシンセキノセキ'],
        ['槨', 'カク，キヘンニリンカクノカク'],
        ['槫', 'セン，キヘンニセンモンノセンノキュウジ'],
        ['槧', 'ザン，ザンシンナノザンノシタニモクザイノモク'],
        ['槝', 'カシノキノカシ，キヘンニシマグニノシマ'],
        ['槞', 'ロウ，キヘンニキョウリュウノリュウ'],
        ['様', 'カミサマ ノ サマ，ヨウ'],
        ['℃', 'ドシー'],
        ['槐', 'カイ，ショクブツノエンジュ'],
        ['槓', 'コウ，キヘンニミツグ'],
        ['槌', 'テッツイ ノ ツイ，ツチ'],
        ['槍', 'ヤリナゲ ノ ヤリ'],
        ['槎', 'サ，キヘンニサベツカスルノサ'],
        ['槊', 'サク，ハッサクノサクノシタニモクザイノモク'],
        ['構', 'コウゾウ ノ コウ，カマエル'],
        ['槇', 'ショクブツノマキノキュウジ'],
        ['槁', 'コウ，キヘンニタカイヒクイノタカイ'],
        ['槃', 'ネハンノハン'],
        ['\\', 'エンマーク'],
        ['藏', 'レイゾウコノゾウノイタイジ'],
        ['藍', 'アイイロ ノ アイ，ラン'],
        ['藉', 'セキ，ロウゼキモノノセキ，'],
        ['藁', 'ムギワラ ノ ワラ'],
        ['藜', 'アカザ，レイ'],
        ['藝', 'ゲイジュツノゲイノキュウジ'],
        ['藕', 'グウ，クサカンムリニスキヘンニグウゼンノグウノツクリ，'],
        ['藐', 'バク，クサカンムリニワカサトビボウノボウ'],
        ['藪', 'ヤブ，ソウ'],
        ['鑷', 'ジョウ、カネヘンニミミミッツ'],
        ['藩', 'ハイハンチケン ノ ハン'],
        ['藤', 'フジ ノ ハナ ノ フジ，トウ'],
        ['藥', 'クスリノイタイジ'],
        ['藾', 'ライ，クサカンムリニイライスルノライノキュウジ'],
        ['藺', 'イグサノイ，リン'],
        ['藻', 'ウミ ノ モクズ ノ モ，ソウ'],
        ['藹', 'ワキアイアイノアイ'],
        ['藷', 'クサカンムリ ニ ショクン ノ ショ'],
        ['怕', 'ハ，オソレル，リッシンベンニシロイ'],
        ['怖', 'キョウフ ノ フ，コワイ'],
        ['怐', 'コウ，リッシンベンニハイクノク'],
        ['怒', 'ゲキドスル ノ ド，イカル'],
        ['思', 'フシギ ノ シ，オモウ'],
        ['怜', 'リッシンベン ニ メイレイ ノ レイ'],
        ['怙', 'エコヒイキノコ，リッシンベンニチュウコシャノコ'],
        ['怛', 'ダツ，リッシンベンニガンタンノタン'],
        ['怏', 'オウ，リッシンベンニチュウオウノオウ'],
        ['怎', 'ソウ，サクヤノサクノツクリノシタニココロ'],
        ['怱', 'ソウ，ネギノクサカンムリノシタガワ'],
        ['怺', 'エイ，リッシンベンニエイキュウノエイ'],
        ['急', 'トッキュウ ノ キュウ，イソグ'],
        ['性', 'コセイテキ ノ セイ'],
        ['怦', 'ヒョウ，リッシンベンニヘイワノヘイ'],
        ['怡', 'イ，リッシンベンニダイドコロノダイ'],
        ['怠', 'タイマン ノ タイ，ナマケル'],
        ['怯', 'ヒキョウモノ ノ キョウ，オビエル'],
        ['怩', 'ジクジタルノジ，ハジル'],
        ['怨', 'オンネン ノ オン，ウラミ'],
        ['怫', 'フツ，リッシンベンニドル'],
        ['怪', 'カイブツ ノ カイ，アヤシイ'],
        ['笈', 'オイ，タケカンムリ ニ オヨブ'],
        ['笊', 'ザル'],
        ['笋', 'タケノコ，ジュンノイタイジ'],
        ['笏', 'コツ，クゲノモツシャク'],
        ['笂', 'ウツボ，タケカンムリニヒノマルノマル'],
        ['笄', 'コウガイ，ケイ，タケカンムリニトリイ'],
        ['鯲', 'ドジョウ，ウオヘンニナニナニニオイテノオ'],
        ['笆', 'ハ，タケカンムリニトモエ'],
        ['鯰', 'ナマズ，ウオヘンニザンネンノネン'],
        ['笘', 'チョウ，タケカンムリニドクセンスルノセン'],
        ['笙', 'フエノショウ'],
        ['笛', 'フエヲフク ノ フエ'],
        ['笞', 'ムチ，タケカンムリニダイドコロノダイ'],
        ['鯨', 'クジラ，ホゲイ ノ ゲイ'],
        ['笑', 'エガオ ノ エ，ワラウ'],
        ['鯤', 'コン，ウオヘンニコンチュウノコン'],
        ['鯣', 'スルメ，ウオヘンニボウエキフウノエキ'],
        ['鯢', 'ゲイ，ウオヘンニニラムノツクリ'],
        ['鯡', 'ヒ，ウオヘンニヒジョウグチノヒ'],
        ['笨', 'ホン，タケカンムリニホンモノノホン'],
        ['第', 'ダイイチインショウ ノ ダイ'],
        ['笠', 'ハナガサ ノ カサ'],
        ['鯖', 'サバ，サカナヘン ニ アオ'],
        ['鯔', 'サカナノボラ，ウオヘンニマガリガワノシタニタンボノタ'],
        ['笥', 'タンス ノ ス'],
        ['符', 'ギモンフ ノ フ'],
        ['鯏', 'アサリ，ウオヘンニケンリノリ'],
        ['笹', 'ショクブツ ノ ササ'],
        ['厳', 'ゲンジュウナ ノ ゲン，キビシイ'],
        ['鯊', 'サカナノハゼ，サンズイニスクナイノシタニウオ'],
        ['鯉', 'コイノボリ ノ コイ'],
        ['鯆', 'ホ，ウオヘンニシジンノトホノホ'],
        ['笳', 'カ，アシブエ，タケカンムリニカソクドノカ'],
        ['笵', 'ハン，タケカンムリニカワガハンランスルノハン'],
        ['笶', 'シ，タケカンムリニユミヤノヤ'],
        ['鯀', 'コン，ウオヘンニケイレツノケイ'],
        ['峙', 'タイジスルノジ，ヤマヘンニテラ'],
        ['峇', 'コウ，ヤマノシタニゴウカクノゴウ'],
        ['峅', 'クラ，ヤマヘンニベントウノベン'],
        ['島', 'シマグニ ノ シマ，トウ'],
        ['峰', 'サイコウホウ ノ ホウ，ミネ'],
        ['峽', 'ツガルカイキョウノキョウノキュウジ'],
        ['峻', 'キュウシュンナ ノ シュン'],
        ['峺', 'コウ，ヤマヘンニサラシナノサラ'],
        ['峡', 'ツガルカイキョウ ノ キョウ'],
        ['峠', 'トウゲ ノ チャヤ ノ トウゲ'],
        ['峯', 'ヤマカンムリ ノ ミネ，ホウ'],
        ['峭', 'ショウ，ヤマヘンニショウゾウガノショウ'],
        ['峪', 'ヨク，ヤマヘンニタニ'],
        ['峩', 'ヤマヘンニワレノガノイタイジ'],
        ['峨', 'ヤマヘン ニ ワレ ノ ガ'],
        ['烱', 'ヒヘンケイガンノシノケイノイタイジ'],
        ['烹', 'カッポウギ ノ ホウ'],
        ['烽', 'ノロシヲイミスルホウ，ヒヘンニミネノツクリ'],
        ['烈', 'キョウレツナ ノ レツ'],
        ['烋', 'コウ，キュウジツノキュウニレンガ'],
        ['烏', 'ウゴウノシュウ ノ ウ，カラス'],
        ['烙', 'ラクインヲオスノラク'],
        ['烝', 'ジョウ，スイジョウキノジョウからクサカンムリヲトッタカタチ'],
        ['烟', 'エントツノエンノイタイジ，ヒヘンニインガオウホウのイン'],
        ['㈱', 'カッコカブ'],
        ['㈲', 'カッコユウゲン'],
        ['㈹', 'カッコダイヒョウ'],
        ['s', 'エス スポーツ'],
        ['謾', 'マン、ゴンベンニウナギノツクリ'],
        ['銓', 'セン、カネヘンニゼンコクノゼン'],
        ['銕', 'テツノイタイジ、カネヘンニソンノウジョウイノイ'],
        ['謹', 'キンガシンネン ノ キン'],
        ['銘', 'メイガラ ノ メイ'],
        ['銚', 'オチョウシ ノ チョウ'],
        ['銛', 'モリデツクノモリ'],
        ['謳', 'セイシュンヲオウカスルノオウ'],
        ['銀', 'ギンイロ ノ ギン'],
        ['謬', 'ゴビュウ ノ ビュウ'],
        ['謫', 'タク、ゴンベンニテキハツスルノテキノツクリ'],
        ['銅', 'ドウメダル ノ ドウ'],
        ['謨', 'ボ、ゴンベンニバクダイナノバク'],
        ['謦', 'ケイガイニセッスルノケイ'],
        ['謡', 'ミンヨウ ノ ヨウ'],
        ['謠', 'ミンヨウノヨウノキュウジ'],
        ['謝', 'カンシャスル ノ シャ'],
        ['講', 'コウシュウカイ ノ コウ'],
        ['謚', 'オクリナ，シゴウノシノイタイジ'],
        ['謙', 'ケンソンスル ノ ケン'],
        ['銷', 'ショウ、カネヘンニショウゾウガノショウ'],
        ['謗', 'ヒボウスルノボウ，ソシル'],
        ['咨', 'シ，ハカル，ツギノシタニクチ'],
        ['謔', 'カイギャクノギャク'],
        ['謐', 'セイヒツナノヒツ'],
        ['謎', 'ナゾヲトク ノ ナゾ'],
        ['謌', 'ウタ，カ，ゴンベンニカノウセイノカヲカサネル'],
        ['謇', 'ケン，サムイノシタニテンノカワリニゴンベン'],
        ['謄', 'コセキトウホン ノ トウ'],
        ['謂', 'イワレ，ゴンベン ニ イブクロ ノ イ'],
        ['謁', 'ハイエツスル ノ エツ'],
        ['謀', 'インボウ ノ ボウ，ハカル'],
        ['I', 'アイ インク'],
        ['楊', 'ツマヨウジ ノ ヨウ，ヤナギ'],
        ['楔', 'クサビガタモジノクサビ，セツ'],
        ['楕', 'ダエンケイ ノ ダ'],
        ['楓', 'ショクブツ ノ カエデ，キヘン ニ カゼ'],
        ['楞', 'ロウ，キヘンニアミガシラニオヤカタノカタ'],
        ['楜', 'コ，キヘンニクロコショウノコ'],
        ['楝', 'レン，キヘンニイサメルノツクリ'],
        ['楚', 'シメンソカ ノ ソ'],
        ['楙', 'ボウ，ハヤシノアイダニムジュンノム'],
        ['楢', 'ナラノキ ノ ナラ'],
        ['楠', 'クスノキ ノ クス，キヘン ニ ミナミ'],
        ['楡', 'ニレノキノニレ'],
        ['楮', 'ショクブツノコウゾ'],
        ['楯', 'キヘン ニ ムジュン ノ ジュン，タテ'],
        ['業', 'ジュギョウ ノ ギョウ'],
        ['楪', 'チョウ，キヘンニシャベルノツクリ'],
        ['楫', 'シュウ，キヘンニクチノシタニミミ'],
        ['楷', 'カイショタイノカイ'],
        ['楴', 'テイ，キヘンニテイオウノテイ'],
        ['極', 'ホッキョク ノ キョク'],
        ['楳', 'キヘン ニ ナニガシ ノ ボウ，ウメ'],
        ['楾', 'ハンゾウ，キヘンニイズミ'],
        ['楼', 'マテンロウ ノ ロウ'],
        ['楽', 'オンガク ノ ガク，タノシイ'],
        ['楸', 'シュウ，キヘンニキセツノアキ'],
        ['楹', 'エイ，キヘンニミチル，エイ'],
        ['艘', 'フネヲカゾエルタンイノソウ'],
        ['艙', 'ソウ，フネヘンニカマクラバクフノクラ'],
        ['艚', 'ソウ，フネヘンニソウトウシュウノソウ'],
        ['艝', 'ソリ，フネヘンニユキグニノユキ'],
        ['艟', 'トウ，フネヘンニグリムドウワノドウ'],
        ['艀', 'ハシケ'],
        ['艇', 'ヒコウテイ ノ テイ'],
        ['艱', 'カンナンシンクノカン'],
        ['色', 'イロエンピツ ノ イロ，ショク'],
        ['艶', 'ヨウエンナ ノ エン，ツヤ'],
        ['艷', 'ヨウエンナノエン，ツヤノキュウジ'],
        ['艸', 'ソウ，ブシュノクサカンムリ'],
        ['艾', 'モグサ，ガイ'],
        ['艢', 'ショウ，フネヘンニリンショクノショク'],
        ['艤', 'ギ，フネヘンニギムノギ'],
        ['艦', 'センスイカン ノ カン'],
        ['艨', 'モウ，フネヘンニケイモウスルノモウ'],
        ['艪', 'フナコギニツカウロ，フネヘンニロドンナノロ'],
        ['艫', 'フネノトモ，フネヘンニハゼノキノハゼノツクリ'],
        ['艮', 'ウシトラ ノ コン'],
        ['良', 'カイリョウ ノ リョウ，ヨイ'],
        ['喨', 'リョウ，クチヘンニジンメイ，ショカツリョウノリョウ'],
        ['喩', 'ヒユノユ，タトエル'],
        ['喪', 'モチュウ ノ モ'],
        ['喫', 'キツエンスル ノ キツ'],
        ['喬', 'キョウボク ノ キョウ，タカシ'],
        ['單', 'タンジュンナノタンノキュウジ'],
        ['喧', 'ケンカリョウセイバイ ノ ケン'],
        ['喰', 'クチヘン ニ タベル ノ クウ'],
        ['営', 'エイギョウ ノ エイ，イトナム'],
        ['喉', 'ノドボトケ ノ ノド，コウ'],
        ['喊', 'カンセイヲアゲルノカン，サケブ'],
        ['喋', 'シャベル'],
        ['喀', 'カッケツスルノカク，クチヘンニオキャクノキャク'],
        ['喃', 'ナン，クチヘンニナンボクノナン'],
        ['善', 'ゼンアク ノ ゼン，ヨイ'],
        ['喇', 'ガッキノラッパノラツ'],
        ['喘', 'ゼンソクノゼン'],
        ['喙', 'カイ，クチバシ，タニンゴトニヨウカイスルノカイ'],
        ['喚', 'ショウニンカンモン ノ カン'],
        ['喜', 'キゲキ ノ キ，ヨロコブ'],
        ['喝', 'キョウカツスル ノ カツ'],
        ['喞', 'ショク，カコツ，クチヘンニソク'],
        ['喟', 'キ，クチヘンニイブクロノイ'],
        ['中', 'チュウガク ノ チュウ，ナカ'],
        ['宀', 'ベン，ブシュノウカンムリ'],
        ['它', 'タ，ヘビカラムシヘンヲトッタカタチ'],
        ['宅', 'ジュウタク ノ タク'],
        ['宇', 'ウチュウ ノ ウ'],
        ['守', 'シュビ ノ シュ，マモル'],
        ['安', 'アンシン ノ アン，ヤスイ'],
        ['宋', 'チュウゴク ノ コクメイ ノ ソウ'],
        ['完', 'カンリョウスル ノ カン'],
        ['宍', 'シシ，ウカンムリ ニ カンスウジ ノ ロク'],
        ['宏', 'ウカンムリ ニ ナム ノ コウ'],
        ['宕', 'トウ，アタゴ ノ ニモジメ，ウカンムリ ニ イシ'],
        ['宗', 'シュウキョウカイカク ノ シュウ'],
        ['官', 'ケイサツカン ノ カン'],
        ['宙', 'ウチュウセン ノ チュウ'],
        ['定', 'ヨテイ ノ テイ，サダメル'],
        ['宛', 'アテナ ノ アテ'],
        ['宜', 'ベンギ ノ ギ，ヨロシク'],
        ['宝', 'ホウセキ ノ ホウ，タカラ'],
        ['実', 'ジツリョク ノ ジツ，ミ'],
        ['客', 'オキャク ノ キャク'],
        ['宣', 'センデン ノ セン'],
        ['室', 'キョウシツ ノ シツ'],
        ['宥', 'ナダメル，ユウジョ ノ ユウ'],
        ['宦', 'カンガンノカン，ウカンムリニソウリダイジンノジン'],
        ['宮', 'オミヤマイリ ノ ミヤ，キュウ'],
        ['宰', 'ソウリヲ イミスル サイショウ ノ サイ'],
        ['害', 'サイガイ ノ ガイ'],
        ['宴', 'エンカイ ノ エン，ウタゲ'],
        ['宵', 'ヨイノクチ ノ ヨイ'],
        ['家', 'イエ，カゾク ノ カ'],
        ['宸', 'シン，ウカンムリニジュウニシノタツ'],
        ['容', 'ヨウセキ ノ ヨウ'],
        ['宿', 'シュクハク ノ シュク，ヤド'],
        ['箒', 'ソウ，ホウキ，タケカンムリニソウジキノソウノツクリ'],
        ['算', 'サンスウ ノ サン'],
        ['箔', 'アルミハク ノ ハク'],
        ['箕', 'ノウグ ノ ミ，タケカンムリ ニ ソ'],
        ['箚', 'トウ，タケカンムリニゴウカカクノゴウニリットウ'],
        ['箘', 'キン，タケカンムリニクニガマエノナカニノギヘンノノギ'],
        ['箙', 'フク，エビラ，タケカンムリニヨウフクノフク'],
        ['箟', 'キン，タケカンムリニコンチュウノコン'],
        ['箜', 'ク，タケカンムリニソラ'],
        ['箝', 'カンコウレイノカン'],
        ['箆', 'ヘラ'],
        ['箇', 'カジョウガキ ノ カ'],
        ['箋', 'ショホウセンノセン'],
        ['箏', 'ソウキョクノソウ，コト'],
        ['箍', 'タガガハズレルノタガ'],
        ['箱', 'ハコニツメル ノ ハコ'],
        ['箴', 'シン，タケカンムリニハリキュウノハリノツクリ'],
        ['箸', 'ハシオキ ノ ハシ'],
        ['管', 'カンリスル ノ カン'],
        ['箪', 'タンス ノ タン'],
        ['箭', 'セン，タケカンムリ ニ マエ'],
        ['砧', 'キヌタ，イシヘン ニ ウラナウ'],
        ['乃', 'ノギタイショウ ノ ノ，スナワチ'],
        ['㊨', 'マルミギ'],
        ['㊦', 'マルシタ'],
        ['㊧', 'マルヒダリ'],
        ['㊤', 'マルウエ'],
        ['㊥', 'マルナカ'],
        ['`', 'アクサングラーブ'],
        ['魁', 'キョカイ ノ カキ，サキガケ'],
        ['魃', 'ヒデリヲイミスルカンバツノバツ'],
        ['魂', 'タマシイ，シンコン ノ コン'],
        ['魅', 'ミリョク ノ ミ'],
        ['憚', 'キタンノナイノタン，ハバカル'],
        ['憙', 'キ，キドアイラクノキノシタニココロ'],
        ['憖', 'ギン，ライネンノライノキュウジトイヌノシタニココロ'],
        ['憔', 'ショウスイスルノショウ'],
        ['魍', 'チミモウリョウノモウ'],
        ['憑', 'ヒョウイスルノヒョウ，キツネツキノツク'],
        ['憐', 'カレンナ ノ レン，アワレム'],
        ['魑', 'チミモウリョウリョウノチ'],
        ['憎', 'ニクム，ゾウオ ノ ゾウ'],
        ['憊', 'ヒロウコンパイノハイ'],
        ['憇', 'キュウケイノケイ，イコウノイタイジ'],
        ['魘', 'エンセイカンノエンノシタニオニ'],
        ['魚', 'サカナ，キンギョ ノ ギョ'],
        ['憂', 'ユウウツ ノ ユウ，ウレウ'],
        ['憾', 'イカンナク ノ カン'],
        [
            '憺',
            'クシンサンタンノタン，リッシンベンニタンニンノタンノキュウジノツクリ',
        ],
        ['憶', 'キオク ノ オク，オボエル'],
        ['憲', 'ケンポウキネンビ ノ ケン'],
        ['魯', 'ロ，サカナ ノ シタニ ニチヨウビ ノ ニチ'],
        ['憮', 'ブゼントスルノブ'],
        ['憬', 'ドウケイノマトノケイ，リッシンベンニフウケイノケイ'],
        ['憫', 'レンビンノビン，リッシンベンニモンガマエニブンショウヲカクノブン'],
        ['魴', 'サカナノホウボウノホウ，ウオヘンニオヤカタノカタ'],
        ['憩', 'キュウケイ ノ ケイ，イコウ'],
        ['憧', 'ドウケイ ノ ドウ，アコガレル'],
        ['憤', 'フンガイスル ノ フン，イキドオル'],
        ['鈞', 'キン、カネヘンニヘイキンノキンノツクリ'],
        ['鈑', 'ハン、カネヘンニハンタイスルノハン'],
        ['鈔', 'ショウ、カネヘンニオオイスクナイノスクナイ'],
        ['鈕', 'チュウ、カネヘンニジュウニシノウシ'],
        ['鈎', 'カギ，カネヘン ニ ツツミガマエ ニ ム'],
        ['鈍', 'ドンカンナ ノ ドン，ニブイ'],
        ['睿', 'エイ，ヒエイザンノエイノヒダリガワ'],
        ['⊂', 'シンブブンシュウゴウ'],
        ['鈿', 'ラデンノデン'],
        ['鼾', 'イビキ'],
        ['鈷', 'カネヘン ニ フルイ ノ コ'],
        ['鈴', 'スズムシ ノ スズ'],
        ['鈩', 'ロ、カネヘンニトジマリノト'],
        ['鈬', 'ドウタクノタクノイタイジ'],
        ['6', 'ロク'],
        ['讀', 'ドクショノドク、ヨムノキュウジ'],
        ['讃', 'ゴンベン ノ ツイタ サンビカ ノ サン'],
        ['變', 'ヘンカスルノヘン，カエルノキュウジ'],
        ['讌', 'エン、ゴンベンニツバメ'],
        ['讎', 'フクシュウスルノシュウ、アダノイタイジ'],
        ['讐', 'フクシュウスル ノ シュウ，アダ'],
        ['讓', 'ジョウホスルノジョウ、ユズルノキュウジ'],
        ['讒', 'ザンゲンンスルノザン、ソシル'],
        ['讖', 'ヨゲンヲイミスルゴンベンノシン'],
        ['讙', 'カン，ゴンベンニカンガイヨウスイノカンノツクリ'],
        ['讚', 'ゴンベンノツイタサンビカノサンノイタイジ'],
        ['属', 'フゾク ノ ゾク'],
        ['屑', 'クズイレ ノ クズ'],
        ['屐', 'ゲキ，キグツ，シカバネニギョウニンベンニシエンスルノシ'],
        ['屓', 'ヒイキノキ，シカバネニカイガラノカイ'],
        ['展', 'ハッテン ノ テン'],
        ['屈', 'リクツ ノ クツ'],
        ['屋', 'オクジョウ ノ オク，ヤ'],
        ['届', 'トドケル'],
        ['屍', 'シカバネ'],
        ['屏', 'ビョウブノビョウ'],
        ['屎', 'シニョウノシ，クソ'],
        ['屁', 'ヒ，ヘリクツノヘ'],
        ['局', 'ユウビンキョク ノ キョク'],
        ['居', 'ジュウキョ ノ キョ，イル'],
        ['屆', 'トドケルノキュウジ'],
        ['屹', 'キツリツスルノキツ'],
        ['山', 'ヤマ，サンミャク ノ サン'],
        ['屶', 'カイ，ナタ，ヤマノシタニカタナ'],
        ['屬', 'フゾクノゾクノキュウジ'],
        ['屯', 'チュウトンスル ノ トン'],
        ['屮', 'ヒダリヲイミスルサ，ヤマノナガイタテボウガシタニヌケル'],
        ['屡', 'シバシバ'],
        ['屠', 'トサツスル ノ ト，ホフル'],
        ['履', 'リレキショ ノ リ'],
        ['層', 'コウソウビル ノ ソウ'],
        ['獪', 'ロウカイナノカイ'],
        ['獨', 'ドクリツノドクノキュウジ'],
        ['獣', 'ヤジュウ ノ ジュウ，ケモノ'],
        ['獻', 'ケンケツスルノケンノキュウジ'],
        ['獺', 'ドウブツノカワウソ'],
        ['獸', 'ヤジュウノジュウ，ケモノノキュウジ'],
        ['獲', 'カクトク ノ カク，エル'],
        ['獰', 'ドウモウナノドウ'],
        ['獵', 'リョウジュウノリョウノキュウジ'],
        ['獏', 'ケモノヘンノドウブツノバク'],
        ['獎', 'ショウガクキンノショウノキュウジノイタイジ'],
        ['獅', 'シシマイ ノ イチモジメ ノ シ'],
        ['獄', 'ジゴク ノ ゴク'],
        ['獗', 'ショウケツヲキワメルノケツ'],
        ['唳', 'レイ，クチヘンニモドル'],
        ['唱', 'ガッショウ ノ ショウ，トナエル'],
        ['唸', 'ウナル，クチヘンニザンネンノネン'],
        ['唹', 'オ，クチヘンニナニナニニオイテノオ'],
        ['唾', 'ダエキ ノ ダ，ツバ'],
        ['售', 'シュウ，ウル，フルトリノシタニクチ'],
        ['唯', 'ユイイツ ノ ユイ'],
        ['唐', 'ケントウシ ノ トウ'],
        ['唖', 'アゼントスル ノ ア'],
        ['唔', 'ゴ，クチヘンニタゴサクノゴ'],
        ['唆', 'シサスル ノ サ'],
        ['唇', 'クチビル，シン'],
        ['唄', 'ウタ，クチヘン ニ カイガラ ノ カイ'],
        ['唏', 'キ，クチヘンニキボウノキ'],
        ['絃', 'ゲン，イトヘン ニ ゲンマイ ノ ゲン'],
        ['終', 'シュウギョウシキ ノ シュウ，オワル'],
        ['絆', 'バンソウコウノバン，キズナ'],
        ['絅', 'ケイ，イトヘンニコウジョウシンノコウカラテンヲトッタモノ'],
        ['組', 'クミ，ソシキ ノ ソ'],
        ['絋', 'イトヘンニヒロバノヒロ'],
        ['絏', 'セツ，イトヘンニエイコウダンノエイ'],
        ['絎', 'コウ，イトヘンニリョコウスルノコウ'],
        ['経', 'ケイケンスル ノ ケイ'],
        ['結', 'ケツロン ノ ケツ'],
        ['絖', 'コウ，イトヘンニヒカリ'],
        ['絛', 'ジョウ，ジョウケンノジョウノキュウジノキノカワリニイト'],
        ['絞', 'シボル，コウシュケイ ノ コウ'],
        ['絣', 'カスリノキモノノカスリ'],
        ['絢', 'ケンランゴウカ ノ ケン'],
        ['絡', 'レンラクスル ノ ラク，カラム'],
        ['給', 'キュウショクシツ ノ キュウ'],
        ['絨', 'ソラトブジュウタンノジュウ'],
        ['絮', 'リュウジョノジョ，ワタ'],
        ['絳', 'コウ，イトヘンニコウスイリョウノコウノツクリ'],
        ['絲', 'ケイトノイトノキュウジ，イトフタツ'],
        ['統', 'トウイツ ノ トウ'],
        ['絶', 'ゼッタイテキ ノ ゼツ'],
        ['絵', 'エホン ノ エ'],
        ['絹', 'キヌイト ノ キヌ'],
        ['絽', 'ロノキモノノロ'],
        ['鼡', 'ネズミノイタイジ'],
        ['w', 'ダブリュー ウィンドウ'],
        ['櫁', 'ミツ，キヘンニハチミツノミツ'],
        ['櫂', 'トウ，フネヲコグカイ'],
        ['櫃', 'オヒツノヒツ'],
        ['櫚', 'シュロノキノロ'],
        ['櫛', 'カミヲ トカス クシ'],
        ['櫞', 'クエンサンノエン，キヘンニエンガワノエンノキュウジ'],
        ['櫟', 'クヌギ，レキ，キヘンニオンガクカノガクノキュウジ'],
        ['櫑', 'ライ，キヘンニスイデンノデンガミッツ'],
        ['櫓', 'ヒノミヤグラ ノ ヤグラ'],
        ['櫨', 'ハゼノキ ノ ハゼ'],
        ['櫪', 'レキ，キヘンニレキシカノレキノキュウジタイ'],
        ['櫺', 'レイ，キヘンニアメカンムリニクチミッツ'],
        ['櫻', 'サクラノキュウジ'],
        ['苛', 'イラダツ，カコク ノ カ'],
        ['苙', 'リュウ、クサカンムリニドクリツノリツ'],
        ['苞', 'ホウ、クサカンムリニツツム'],
        ['苟', 'イヤシクモ、クサカンムリニハイクノク'],
        ['苜', 'モク、クサカンムリニモクテキノモク'],
        ['苒', 'ゼン、フタタビノイッカクメノカワリニクサカンムリ'],
        ['苓', 'レイ，クサカンムリ ニ メイレイ ノ レイ'],
        ['苑', 'ジングウガイエン ノ エン'],
        ['M', 'エム マイク'],
        ['苗', 'ナエギ ノ ナエ，ビョウ'],
        ['苔', 'ショクブツ ノ コケ，タイ'],
        ['苅', 'カル，クサカンムリ ノ カル'],
        ['苺', 'クダモノノイチゴ、クサカンムリニハハ'],
        ['苻', 'フ、クサカンムリニフロクノフ'],
        ['苹', 'ヘイ、クサカンムリニヘイボンナノヘイ'],
        ['苳', 'トウ、クサカンムリニフユヤスミノフユ'],
        ['英', 'エイユウ ノ エイ'],
        ['苴', 'ショ、クサカンムリニクミノツクリ'],
        ['苫', 'トマコマイシ ノ トマ'],
        ['苣', 'キョ、クサカンムリニキョダイナノキョ'],
        ['苡', 'イ、クサカンムリニイジョウイカノイ'],
        ['苦', 'クロウ ノ ク，クルシイ'],
        ['苧', 'ショクブツ ノ カラムシ，チョ'],
        ['若', 'ワカモノ ノ ワカ'],
        ['愡', 'ソウ，リッシンベンニネギノクサカンムリノシタガワ'],
        ['愧', 'ザンキニタエナイノキ'],
        [
            '飃',
            'ヒョウヒョウトシタノヒョウノイタイジ，カゼノミギニトウヒョウスルノヒョウ',
        ],
        ['愨', 'マコトヲイミスルカク'],
        ['愬', 'ソ，ハッサクノサクノシタニココロ'],
        ['飄', 'ヒョウヒョウトシタノヒョウ，カゼノヒダリニトウヒョウスルノヒョウ'],
        ['飛', 'ヒコウキ ノ ヒ，トブ'],
        ['食', 'タベル，キュウショク ノ ショク'],
        ['愴', 'ヒソウナノソウ，リッシンベンニカマクラバクフノクラ'],
        ['飜', 'ホンヤクスルノホンノイタイジ'],
        ['愽', 'ハクブツカンノハクノイタイジ'],
        ['愼', 'シンチョウナノシン，ツツシムノキュウジ'],
        ['愿', 'ゲン，ノハラノハラノシタニココロ'],
        ['愾', 'キ，リッシンベンニクウキノキノキュウジ'],
        ['愁', 'アイシュウ ノ シュウ，ウレイ'],
        ['愀', 'ショウ，リッシンベンニキセツノアキ'],
        ['愃', 'セン，リッシンベンニセンデンスルノセン'],
        ['飯', 'ゴハン ノ ハン，メシ'],
        ['飮', 'インリョウスイノイン，ノムノキュウジ'],
        ['飭', 'チョク，ショクヘンニカタカナノノ，ヨコボウニチカラ'],
        ['愆', 'ケン，フエンスルノエンノシタニココロ'],
        ['愉', 'ユカイナ ノ ユ'],
        ['愈', 'ユ，イヨイヨ'],
        ['愍', 'ビン，タミトノブンノシタニココロ'],
        ['意', 'イミ ノ イ'],
        ['愎', 'フク，リッシンベンニオウフクノフクノツクリ'],
        ['愕', 'キョウガクスルノガク'],
        ['飾', 'ソウショクヒン ノ ショク，カザリ'],
        ['飽', 'アキル，ホウワ ノ ホウ'],
        ['飼', 'シイク ノ シ，カウ'],
        ['飲', 'ミズヲノム ノ ノム，イン'],
        ['愛', 'アイスル ノ アイ'],
        ['愚', 'グチ ノ グ，オロカ'],
        ['感', 'カンソウブン ノ カン'],
        ['飴', 'ミズアメ ノ アメ'],
        ['嬋', 'セン，オンナヘンニタンジュンノタンノキュウジ'],
        ['嬉', 'ウレシイ，キ'],
        ['嬌', 'アイキョウガアルノキョウ'],
        ['嬖', 'ヘイ，ヘキエキスルノヘキノシタニオンナ'],
        ['嬪', 'ベッピンノヒン'],
        ['#', 'シャープ'],
        ['嬬', 'ツマ，オンナヘン ニ ヒツジュヒン ノ ジュ'],
        ['嬢', 'オジョウサマ ノ ジョウ'],
        ['⇒', 'ナラバ'],
        ['⇔', 'ドウチ'],
        ['嬲', 'ジョウ，ナブル，オトコ，オンナ，オトコ'],
        ['嬰', 'アカンボウヲ イミスル エイジ ノ エイ'],
        ['嬶', 'カカア，オンナヘンニハナカゼノハナ'],
        ['稔', 'ミノル，ノギヘン ニ ザンネン ノ ネン'],
        ['稗', 'ショクブツ ノ ヒエ'],
        ['稜', 'ヤマ ノ リョウセン ノ リョウ'],
        ['稟', 'リンギショノリン'],
        ['稘', 'キ，ノギヘンニソノ'],
        ['稙', 'チョク，ノギヘンニチョクゼンノチョク'],
        ['稚', 'ヨウチエン ノ チ'],
        ['稀', 'マレ，ノギヘン ニ キボウ ノ キ'],
        ['稍', 'ショウ，ノギヘンニショウゾウガノショウ'],
        ['税', 'ゼイキン ノ ゼイ'],
        ['稈', 'ムギワラヲイミスルバッカンノカン'],
        ['程', 'ニッテイ ノ テイ，ホド'],
        ['稷', 'シャショクノシンノショク，ノギヘン'],
        ['稱', 'イチニンショウノショウノキュウジ'],
        ['稲', 'イネ，イナサク ノ イナ'],
        ['稼', 'カセグ，カ'],
        ['稽', 'ケイコスル'],
        ['稾', 'ゲンコウヨウシノコウノイタイジ'],
        ['稿', 'ゲンコウヨウシ ノ コウ'],
        ['稻', 'イネノキュウジ'],
        ['稠', 'チュウミツナノチュウ'],
        ['種', 'シュルイ ノ シュ，タネ'],
        ['口', 'クチブエ ノ クチ，コウ'],
        ['叢', 'ソウショ ノ ソウ，クサムラ'],
        ['叡', 'ヒエイザン ノ エイ'],
        ['句', 'ハイク ノ ク'],
        ['栴', 'ショクブツ ノ センダン ノ セン'],
        ['叫', 'サケブ，ゼッキョウ ノ キョウ'],
        ['只', 'タダイマ ノ タダ'],
        ['叩', 'カタタキ ノ タタク'],
        ['叨', 'トウ，クチヘンニカタナ'],
        ['可', 'カノウセイ ノ カ'],
        ['叮', 'テイ，クチヘンニイッチョウメノチョウ'],
        ['叭', 'ガッキノラッパノハ'],
        ['召', 'ショウシュウスル ノ ショウ，メス'],
        ['右', 'ミギ，サユウ ノ ユウ'],
        ['史', 'レキシテキ ノ シ'],
        ['叱', 'シカル，シッセキスル ノ シツ'],
        ['台', 'ダイドコロ ノ ダイ'],
        ['号', 'バンゴウ ノ ゴウ'],
        ['叶', 'カナウ，クチヘン ニ ジュウ'],
        ['叺', 'フクロノカマス，クチヘンニハイル'],
        ['司', 'シカイシャ ノ シ'],
        ['參', 'サンカスルノサン，マイルノキュウジ'],
        ['参', 'サンカスル ノ サン，マイル'],
        ['友', 'ユウジン ノ ユウ，トモ'],
        ['及', 'フキュウスル ノ キュウ，オヨブ'],
        ['叉', 'ヤシャ ノ シャ'],
        ['仁', 'ジンギ ノ ジン，ニオウ ノ ニ'],
        ['収', 'シュウニュウ ノ シュウ，オサメル'],
        ['反', 'ハンタイスル ノ ハン'],
        ['双', 'ソウガンキョウ ノ ソウ'],
        ['从', 'シタガウノイタイジ，ヒトフタツ'],
        ['受', 'ジュケン ノ ジュ，ウケル'],
        ['取', 'シュザイ ノ シュ，トル'],
        ['格', 'ゴウカク ノ カク'],
        ['叛', 'ハンラン ノ ハン，ソムク'],
        ['叙', 'ジジョデン ノ ジョ'],
        ['★', 'クロボシ'],
        ['☆', 'ホシジルシ'],
        ['班', 'ハンチョウ ノ ハン'],
        ['珮', 'ハイ，ハイヨウスルノハイノニンベンノカワリニオウヘン'],
        ['珪', 'オウヘン ニ ツチ フタツ ノ ケイ，ケイソウド ノ ケイ'],
        ['珥', 'ジ，オウヘンニミミ'],
        ['今', 'コンゲツ ノ コン，イマ'],
        ['珠', 'シュザン ノ シュ，タマ'],
        ['核', 'カクカゾク ノ カク'],
        ['現', 'ゲンジツ ノ ゲン，アラワレル'],
        ['珸', 'ゴ，オウヘンニコクゴノゴノツクリ'],
        ['珱', 'ヨウ，オウヘンニサクラノツクリ'],
        ['珍', 'メズラシイ，チンミ ノ チン'],
        ['珎', 'メズラシイ，チンミノチンノイタイジ'],
        ['珈', 'カ，コーヒーノアテジノイチモジメ'],
        ['珊', 'サンゴショウ ノ サン'],
        ['珀', 'コハクイロノハク'],
        ['珂', 'オウヘン ニ カノウセイ ノ カ'],
        ['珞', 'ラク，オウヘンニカクジノカク'],
        ['栫', 'ソン，キヘンニソンザイスルノソン'],
        ['仙', 'センニン ノ セン，ニンベン ニ ヤマ'],
        ['㌫', 'カナパーセント'],
        ['㌦', 'カナドル'],
        ['㌧', 'カナトン'],
        ['栗', 'クリノキ ノ クリ'],
        ['㌢', 'カナセンチ'],
        ['㌣', 'カナセント'],
        ['栖', 'スム，セイ，キヘン ニ ニシ'],
        ['㌻', 'カナページ'],
        ['㌶', 'カナヘクタール'],
        ['㌍', 'カナカロリー'],
        ['㌃', 'カナアール'],
        ['㌘', 'カナグラム'],
        ['㌔', 'カナキリ'],
        ['仭', 'フカサノタンイノジンノイタイジ，ニンベンニヤイバノイタイジ'],
        ['价', 'カイ，ニンベンニチュウカイスルノカイ'],
        ['栃', 'トチギケン ノ トチ'],
        ['d', 'ディー デスク'],
        ['驢', 'ドウブツノロバノロ'],
        ['鎚', 'カネヘン ノ ツチ'],
        ['鎔', 'ヨウ、カネヘンニヨウセキノヨウ'],
        ['鎖', 'サコク ノ サ，クサリ'],
        ['鎗', 'ヤリ，カネヘン ニ クラ，ソウ'],
        ['鎌', 'カマクラ ノ カマ'],
        ['鎹', 'コハカスガイノカスガイ'],
        ['鎰', 'イツ、カネヘンニリエキノエキ'],
        ['鎬', 'シノギヲケズルノシノギ'],
        ['鎭', 'チンアツスルノチン，シズメルノキュウジ'],
        ['鎮', 'チンアツスル ノ チン，シズメル'],
        ['鎧', 'ヨロイ，ガイシュウイッショク ノ ガイ'],
        ['蠏', 'カニノイタイジ，ムシヘンニカイケツスルノカイ'],
        ['蠎', 'ウワバミヲイミスルボウ，クサカンムリニホンソウスルノホン'],
        ['蠍', 'サソリ'],
        ['蠅', 'ムシノハエ'],
        ['蠑', 'エイ，ムシヘンニサカエルノキュウジ'],
        ['蠖', 'カク，ムシヘンニカクトクスルノカクノツクリ'],
        ['蠕', 'ゼンドウウンンドウノゼン'],
        ['蠣', 'カキ，ムシヘンニゲキレイスルノレイノキュウジノヒダリガワ'],
        ['蠢', 'ウゴメク，シュン'],
        ['蠡', 'レイ，ケイガシラニイノコ，ソノシタニムシフタツ'],
        ['蠧', 'ホンニツクムシノトギョノト'],
        ['蠻', 'ナンバンボウエキノバンノキュウジ'],
        ['蠹', 'ホンニツクムシノトギョノトノイタイジ'],
        ['蠱', 'コワクスルノコ，ムシミッツノシタニサラ'],
        ['蠶', 'ヨウサンノサン，カイコノキュウジ'],
        ['橘', 'タチバナ，カンキツ ノ キツ'],
        ['橙', 'トウ，ショクブツノダイダイ'],
        ['機', 'ヒコウキ ノ キ'],
        ['橇', 'セイ，ノリモノノソリ'],
        ['橄', 'カンランサンノカン，キヘンニユウカンナノカン'],
        ['橋', 'ホドウキョウ ノ キョウ，ハシ'],
        ['橈', 'ドウ，タワム，トウコツノトウ'],
        ['鼓', 'タイコ ノ コ，ツヅミ'],
        ['橲', 'ジサ，キヘンニキゲキノキ'],
        ['橸', 'マサ，キヘンニスイショウダマノショウ'],
        ['橿', 'カシハラジングウ ノ カシ'],
        ['橢', 'ダエンケイノダノキュウジ'],
        ['橡', 'キヘン ニ ドウブツ ノ ゾウ，トチ'],
        ['橦', 'トウ，キヘンニワラベ'],
        [':', 'コロン'],
        ['荘', 'ベッソウチ ノ ソウ'],
        ['荐', 'セン，クサカンムリニソンザイスルノソン'],
        ['荒', 'アレル，コウヤ ノ コウ'],
        ['荏', 'エゴマ ノ エ'],
        ['草', 'クサ，ソウゲン ノ ソウ'],
        ['荊', 'クサカンムリ ニ ケイムショ ノ ケイ，イバラ'],
        ['荅', 'トウ，クサカンムリニゴウカクノゴウ'],
        ['荀', 'ジュン，クサカンムリニゲジュンノジュン'],
        ['作', 'サクブン ノ サク，ツクル'],
        ['荼', 'ダビニフスノダ'],
        ['荻', 'オギ'],
        ['荵', 'ニン、クサカンムリニニンジャノニン'],
        ['荷', 'ニモツ ノ ニ'],
        ['荳', 'トウ、マメ、クサカンムリニマメモチノマメ'],
        ['咲', 'ハナガサク ノ サク'],
        ['咳', 'セキバライ ノ セキ'],
        ['咼', 'カ，ナベノツクリ'],
        ['咽', 'ジビインコウカ ノ イン，ノド'],
        ['咾', 'ロウ，クチヘンニロウジンノロウ'],
        ['咸', 'シンノミヤコ，カンヨウノカン'],
        ['咤', 'シッタゲキレイノタ，クチヘンニジュウタクノタク'],
        ['咥', 'テツ，クチヘンニゲシノシ'],
        ['咢', 'ガク，キョウガクスルノガクノツクリ'],
        ['咬', 'カム，コウ，クチヘンニマジワル'],
        ['咯', 'ラク，クチヘンニオノオノ'],
        ['茆', 'ボウ、クサカンムリニジュウニシノウサギ'],
        ['咫', 'ナガサノタンイノシ，シャクハチノシャクノミギニタダイマノタダ'],
        ['咐', 'ホ，クチヘンニフロクノフ，ツケル'],
        ['咒', 'ノロウ，ジュモンノジュノイタイジ'],
        ['咄', 'トッサノトツ，クチヘンニデル'],
        ['咆', 'トラノホウコウノホウ，ホエル'],
        ['咀', 'カミクダクイミノソシャクノソ'],
        ['和', 'ワフク ノ ワ'],
        ['咎', 'トガメル'],
        ['咏', 'エイ，クチヘンニエイキュウノエイ'],
        ['咋', 'サクネン ノ サク ノ ヒヘン ノ カワリニ クチヘン'],
        ['緇', 'シ，イトヘンニシチョウヘイノシノツクリ'],
        ['総', 'ソウリダイジン ノ ソウ'],
        ['緋', 'ヒイロ ノ ヒ'],
        ['緊', 'キンキュウ ノ キン'],
        ['緕', 'シ，イトヘンニイッセイシャゲキノセイ'],
        ['緑', 'ミドリ，リョクチャ ノ リョク'],
        ['緒', 'ナイショ ノ ショ'],
        ['緝', 'シュウ，イトヘンニクチノシタニミミ'],
        ['緜', 'モメンノメンノイタイジ'],
        ['緞', 'ドンスノドン，イトヘンニネダンノダン'],
        ['緘', 'カンモクノカン，イトヘンニシンノミヤコカンヨウノカン，'],
        ['線', 'センロ ノ セン'],
        ['緤', 'セツ，イトヘンニシャベルノツクリ'],
        ['緡', 'ビン，イトヘンニタミノシタニマイニチノニチ'],
        ['締', 'テイケツスル ノ テイ，シメル'],
        ['緬', 'チリメン ノ メン'],
        ['緯', 'イドケイド ノ イ'],
        ['緩', 'ユルヤカ，カンワスル ノ カン'],
        ['編', 'ヘンシュウスル ノ ヘン，アム'],
        ['練', 'レンシュウ ノ レン'],
        ['緲', 'ビョウ，イトヘンニモクテキノモクニスクナイ'],
        ['緻', 'チミツナノチ'],
        ['骰', 'トウ，ホネヘンニナゲルノツクリ'],
        ['Ⅸ', 'ローマスウジ キュウ'],
        ['Ⅹ', 'ローマスウジ ジュウ'],
        ['Ⅲ', 'ローマスウジ サン'],
        ['Ⅳ', 'ローマスウジ ヨン'],
        ['Ⅰ', 'ローマスウジ イチ'],
        ['Ⅱ', 'ローマスウジ ニ'],
        ['Ⅶ', 'ローマスウジ ナナ'],
        ['Ⅷ', 'ローマスウジ ハチ'],
        ['Ⅴ', 'ローマスウジ ゴ'],
        ['Ⅵ', 'ローマスウジ ロク'],
        ['媚', 'ビヤクノビ，コビル'],
        ['媛', 'エヒメケン ノ ヒメ'],
        ['媒', 'バイカイスル ノ バイ'],
        ['媼', 'オウ，オンナヘンニオンセンノオンノキュウジノツクリ'],
        ['媽', 'ボ，オンナヘンニウマ'],
        ['媾', 'コウ，オンナヘンニミゾノツクリ'],
        ['窟', 'ドウクツ ノ クツ'],
        ['窘', 'タシナメル，キン'],
        ['窖', 'コウ，アナカンムリニツゲル'],
        ['窗', 'マドノイタイジ'],
        [
            '窕',
            'ヨウチョウタルビジンノチョウ，アナカンムリニイッチョウエンノチョウ',
        ],
        ['窒', 'チッソ ノ チツ'],
        ['窓', 'マドグチ ノ マド，ソウ'],
        ['窈', 'ヨウチョウタルビジンノヨウ，アナカンムリニオサナイ'],
        ['窄', 'シヤキョウサク ノ サク，スボマル'],
        ['窃', 'セットウハン ノ セツ'],
        ['突', 'ショウトツスル ノ トツ，ツク'],
        ['窿', 'リュウ，アナカンムリニリュウキスルノリュウノキュウジ'],
        ['窺', 'アナカンムリ ノ ウカガウ'],
        ['窶', 'ヤツレル，ク'],
        ['窰', 'カマモトノカマノイタイジ'],
        ['窮', 'キュウクツ ノ キュウ，キワマル'],
        ['窯', 'カマモト ノ カマ，ヨウ'],
        ['窪', 'クボム，クボチ ノ クボ'],
        ['窩', 'カ，アナカンムリニナベノツクリ'],
        ['{', 'チュウカッコ'],
        ['額', 'キンガク ノ ガク，ヒタイ'],
        ['㎡', 'ヘイホウメートル'],
        ['㎞', 'キロメートル'],
        ['㎜', 'ミリメートル'],
        ['㎝', 'センチメートル'],
        ['㎎', 'ミリグラム'],
        ['㎏', 'キログラム'],
        ['披', 'ヒロウエン ノ ヒ'],
        ['題', 'シュクジュダイ ノ ダイ'],
        ['顏', 'カオ，ガンメンノガンノキュウジ'],
        ['顎', 'アゴ，ガク'],
        ['顛', 'コト ノ テンマツ ノ テン'],
        ['顋', 'エラ，フシギノシノミギニページ'],
        ['抬', 'タイ，テヘンニダイドコロノダイ'],
        ['顆', 'カリュウノカ，ツブ'],
        ['抻', 'シン，テヘンニシンコクスルノシン'],
        ['抹', 'マッチャ ノ マツ'],
        ['類', 'ジンルイ ノ ルイ'],
        ['願', 'ガンボウ ノ ガン，ネガウ'],
        ['抽', 'チュウセン ノ チュウ'],
        ['押', 'オウシュウスル ノ オウ，オス'],
        ['顕', 'ケンビキョウ ノ ケン，アキラカ'],
        ['顔', 'カオイロ ノ カオ，ガン'],
        ['抱', 'ホウフヲカタル ノ ホウ，ダク'],
        ['抵', 'テイコウスル ノ テイ'],
        ['把', 'ハアクスル ノ ハ'],
        ['抉', 'ケツ，テヘンニタスウケツノケツノツクリ'],
        ['顫', 'セン，ブツダンノダンノツクリノミギニページ'],
        ['抃', 'ベンブスルノベン，テヘンニナベブタノシタニカタカナノト'],
        ['抂', 'キョウ，テヘンニオウサマノオウ'],
        ['顧', 'コモン ノ コ，カエリミル'],
        ['技', 'ギジュツ ノ ギ，ワザ'],
        ['抄', 'コセキショウホン ノ ショウ'],
        ['抛', 'ホウテキスルノホウ，テヘンニカンスウジノキュウニチカラ'],
        ['折', 'コッセツ ノ セツ，オル'],
        ['択', 'センタクシ ノ タク'],
        ['抜', 'バツグン ノ バツ，ヌケル'],
        ['抓', 'ソウ，ツネル，テヘンニツメ'],
        ['抒', 'ジョジョウシノジョ，テヘンニヨテイノヨ'],
        ['抑', 'ヨクセイスル ノ ヨク，オサエル'],
        ['抗', 'テイコウスル ノ コウ'],
        ['抖', 'トウ，テヘンニホクトシチセイノト'],
        ['投', 'トウシュ ノ トウ，ナゲル'],
        ['抔', 'ホウ，テヘンニフシギノフ'],
        ['顯', 'ケンビキョウノケンノキュウジ'],
        ['Q', 'キュー クエスチョン'],
        ['沃', 'ヒヨクナトチ ノ ヨク'],
        ['沂', 'ギ，サンズイニショクパンイッキンノキン'],
        ['沁', 'シン，シミル，サンズイニココロ'],
        ['錢', 'ツリセンノセンノキュウジ'],
        ['錣', 'テツ、カネヘンニツヅルノツクリ'],
        ['錠', 'テジョウ ノ ジョウ'],
        ['錮', 'キンコケイノコ、カネヘンニカタメル'],
        ['錯', 'サクランスル ノ サク'],
        ['錬', 'レンキンジュツ ノ レン'],
        ['沈', 'チンボツスル ノ チン，シズム'],
        ['錫', 'キンゾク ノ スズ，シャク'],
        ['沍', 'ゴ，サンズイニタガイ'],
        ['沌', 'コントントシタ ノ トン'],
        ['沓', 'クツ，ミズ ノ シタニ ヒ'],
        ['沒', 'チンボツスルノボツノキュウジ'],
        ['沐', 'モクヨクスルノモク'],
        ['録', 'ロクオン ノ ロク'],
        ['沖', 'オキ，オキナワケン ノ オキ'],
        ['沛', 'ハイゼンタルアメのハイ'],
        ['沚', 'シ，サンズイニキンシスルノシ'],
        ['沙', 'ゴブサタ ノ サ，サンズイ ニ スクナイ'],
        ['錺', 'カザリ、カネヘンニホウコウザイノホウ'],
        ['錻', 'ブリキノブ、カネヘンニブキノブ'],
        ['錆', 'サビドメ ノ サビ'],
        ['沢', 'コウタク ノ タク，サワ'],
        ['没', 'チンボツスル ノ ボツ'],
        ['沫', 'ホウマツ ノ マツ，サンズイ ニ スエ'],
        ['錏', 'ア、カネヘンニアジアノアノキュウジ'],
        ['沮', 'イキソソウノソ，サンズイニソシスルノソノツクリ'],
        ['河', 'サンズイ ノ カワ'],
        ['沱', 'ダ，サンズイニブッダノダノツクリ'],
        ['錐', 'シカクスイ ノ スイ'],
        ['治', 'セイジ ノ ジオサメル'],
        ['沺', 'テン，サンズイニスイデンノデン'],
        ['油', 'セイキユ ノ ユ，アブラ'],
        ['沸', 'フットウスル ノ フツ，ワク'],
        ['沿', 'エンガン ノ エン，ソウ'],
        ['沾', 'テン，サンズイニドクセンスルノセン'],
        ['沽', 'コケンニカカワルノコ，サンズイニチュウコヒンノコ'],
        ['沼', 'ヌマチ ノ ヌマ，ショウ'],
        ['袍', 'ホウ、コロモヘンニツツム'],
        ['袈', 'オボウサン ノ ケサ ノ ケ'],
        ['袋', 'フクロ，ユウタイルイ ノ タイ'],
        ['袁', 'エン，ルイジンエンノエンノツクリ'],
        ['袂', 'タモト'],
        ['袞', 'コンリョウノソデノコン'],
        ['袙', 'バク、コロモヘンニクロシロノシロ'],
        ['顴', 'ケン，カンヅメノカンノキュウジノツクリノミギニページ'],
        ['袗', 'シン、コロモヘンニシンサツシツノシンノツクリ'],
        ['袖', 'フリソデ ノ ソデ'],
        ['袒', 'サタンスルノタン、ハダヌグ'],
        ['袮', 'ネ、コロモヘンニイチニンショウノショウノツクリ'],
        ['被', 'ヒガイシャ ノ ヒ，コウムル'],
        ['袤', 'ボウ，コロモノアイダニムジュンノム'],
        ['袢', 'ジュバンノバン'],
        ['袿', 'ケイ、コロモヘンニツチフタツ'],
        ['顳', 'ジョウ，ミミミッツノミギニページ'],
        ['袵', 'ジン、コロモヘンニセキニンノニン'],
        ['袴', 'ハオリハカマ ノ ハカマ'],
        ['袷', 'アワセ，コロモヘン ニ ゴウカク ノ ゴウ'],
        ['袱', 'フク、チャブダイノニモジメ'],
        ['袰', 'ホロ、ハハノシタニコロモ'],
        ['卦', 'ハッケ ノ ケ'],
        ['占', 'ドクセンスル ノ セン，シメル'],
        ['卯', 'ウ，ジュウニシ ノ ウサギ'],
        ['卮', 'シ，サカズキ'],
        ['卩', 'セツ，ブシュノワリフ'],
        ['卵', 'タマゴ'],
        ['却', 'ヘンキャクスル ノ キャク'],
        ['卷', 'カン，ノリマキノマキノキュウジ'],
        ['危', 'キケン ノ キ，アブナイ'],
        ['印', 'インサツ ノ イン'],
        ['即', 'ソクザニ ノ ソク，スナワチ'],
        ['卿', 'キョウ，スウキケイ ノ ケイ'],
        ['卸', 'オロシウリ ノ オロシ'],
        ['卻', 'ヘンキャクスルノキャクノイタイジ'],
        ['卅', 'ソウ，カンスウジノサンジュウ'],
        ['升', 'イッショウビン ノ ショウ，マス'],
        ['卆', 'ソツギョウノソツノイタイジ'],
        ['十', 'カンスウジ ノ ジュウ'],
        ['區', 'クベツスルノクノキュウジ'],
        ['\'', 'アポストロフィ'],
        ['卍', 'マンジ'],
        ['卉', 'カキエンゲイノキ'],
        ['午', 'ショウゴ ノ ゴ'],
        ['半', 'ハンブン ノ ハン，ナカバ'],
        ['協', 'キョウリョクスル ノ キョウ'],
        ['南', 'ナンボク ノ ナン，ミナミ'],
        ['卑', 'ヒレツナ ノ ヒ，イヤシイ'],
        ['卓', 'タッキュウ ノ タク'],
        ['卒', 'ソツギョウ ノ ソツ'],
        ['卜', 'ウラナウ，ボク'],
        ['卞', 'ベン，ナベブタノシタニカタカナノト'],
        ['単', 'タンジュン ノ タン'],
        ['博', 'ハクブツカン ノ ハク'],
        ['牴', 'テイ，ウシヘンニヒクイノツクリ'],
        ['牲', 'ギセイシャ ノ セイ'],
        ['牾', 'ゴ，ウシヘンニコクゴノゴノツクリ'],
        ['牽', 'ケンインスル ノ ケン'],
        ['特', 'トクベツ ノ トク'],
        ['牧', 'ボクジョウ ノ ボク'],
        ['牢', 'ロウヤ ノ ロウ'],
        ['牡', 'ハナ ノ ボタン ノ ボ，オス'],
        ['物', 'ドウブツ ノ ブツ，モノ'],
        ['牒', 'サイゴツウチョウ ノ チョウ，フダ'],
        ['牟', 'シャカムニ ノ ム，ム ノ シタニ ウシ'],
        ['牝', 'ヒンバ ノ ヒン，メス'],
        ['牛', 'ウシ，ギュウニュウ ノ ギュウ'],
        ['牙', 'ゾウゲ ノ ゲ，キバ'],
        ['牘', 'トク，カタヘンニハンバイスルノバイノキュウジ'],
        ['片', 'カタオモイ ノ カタ，ヘン'],
        ['牆', 'ショウ，カキ，ショウヘンニリンショクノショク'],
        ['牀', 'コシカケノショウギノショウ，ショウヘンニクサキノキ'],
        ['牌', 'イハイ ノ ハイ，フダ'],
        ['牋', 'ショホウセンノセンノイタイジ'],
        ['版', 'モクハンガ ノ ハン'],
        ['吾', 'ワレ，ゴ，カンスウジ ノ ゴ ノ シタニ クチ'],
        ['吼', 'シシクノク，クチヘンニコウシモウシノコウ'],
        ['吽', 'アウンノコキュウノウン，クチヘンニドウブツノウシ'],
        ['吻', 'セップンスル ノ フン'],
        ['吸', 'コキュウスル ノ キュウ，スウ'],
        ['吹', 'スイソウガク ノ スイ，フク'],
        ['吶', 'トツベンノトツ，クチヘンニコクナイノナイ'],
        ['吮', 'セン，クチヘンニカタカナノムノシタニヒトアシ'],
        ['听', 'オモサノタンイノポンド'],
        ['吭', 'コウ，クチヘンニテイコウスルノコウノツクリ'],
        ['含', 'フクム，ガンチク ノ ガン'],
        ['吩', 'フン，クチヘンニハンブンノブン'],
        ['否', 'ヒテイスル ノ ヒ'],
        ['吠', 'ホエル，クチヘン ニ イヌ'],
        ['吟', 'ギンミスル ノ ギン'],
        ['吝', 'リンショクカノリン'],
        ['君', 'ショクン ノ クン，キミ'],
        ['閃', 'ヒラメキ，センコウ ノ セン'],
        ['吐', 'トイキ ノ ト，ハク'],
        ['向', 'コウジョウスル ノ コウ，ムカウ'],
        ['后', 'コウゴウ ノ ゴウ，キサキ'],
        ['吏', 'ヤクニンヲ イミスル カンリ ノ リ'],
        ['同', 'オナジ，ドウジ ノ ドウ'],
        ['名', 'ナマエ ノ ナ，メイ'],
        ['吊', 'ツルス，クチ ノ シタニ ハバ'],
        ['吋', 'ナガサ ノ タンイ ノ インチ'],
        ['合', 'ゴウカク ノ ゴウ，アウ'],
        ['吉', 'キチトキョウ ノ キチ，ヨシ'],
        ['各', 'カクジ ノ カク，オノオノ'],
        ['吃', 'ドモル，キツオン ノ キツ'],
        ['吁', 'ク，クチヘンニウチュウノウノウカンムリヲトッタモノ'],
        ['籏', 'ハタ，タケカンムリニハタヲフルノハタ'],
        ['籍', 'コクセキ ノ セキ'],
        ['籌', 'チュウ，タケカンムリニコトブキノキュウジ'],
        ['籃', 'ヨウランキノラン'],
        ['籀', 'チュウ，タケカンムリニテヘントリュウガクセイノシュウ'],
        ['籟', 'ライ，タケカンムリニイライスルノライノキュウジ'],
        ['籘', 'トウイスノトウノイタイジ'],
        ['籖', 'クジビキノクジ，センノイタイジ'],
        ['籔', 'ス，タケカンムリニカズノキュウジ'],
        ['籐', 'トウイスノトウ'],
        ['閔', 'ビン，モンガマエニブンショウヲカクノブン'],
        ['籬', 'リ，マガキ'],
        ['籥', 'ヤク，タケカンムリニブシュノヤク'],
        ['籤', 'クジビキノクジ，セン'],
        ['籠', 'ロウジョウスルノロウ，カゴ'],
        ['籾', 'モミガラ ノ モミ'],
        ['籵', 'ナガサノタンイノデカメートル'],
        ['米', 'コメ，シンマイ ノ マイ'],
        ['閤', 'タイコウキ ノ コウ'],
        ['閧', 'コウ，モンガマエニキョウツウテンノキョウ'],
        ['閠', 'ウルウドシノウルウノイタイジ'],
        ['錦', 'ニシキ，キン'],
        ['閨', 'ケイボウノケイ，ネヤ'],
        ['言', 'ゲンゴ ノ ゲン，イウ'],
        ['錨', 'イカリ，カネヘン ニ ナエ'],
        ['h', 'エイチ ホテル'],
        ['閹', 'エン，モンガマエニキソクエンエンノエン'],
        ['錵', 'ニエ、カネヘンニハナビノハナ'],
        ['閻', 'エンマサマノエン'],
        ['比', 'ヒカクスル ノ ヒ，クラベル'],
        ['毒', 'ショウドク ノ ドク'],
        ['毓', 'イク，マイニチノマイニナガレルノツクリ'],
        ['毟', 'ムシル，スクナイノシタニウモウノモウ'],
        ['毘', 'ビシャモンテン ノ ビ'],
        ['毛', 'ケイト ノ ケ，ウモウ ノ モウ'],
        ['毅', 'キゼントシタ ノ キ'],
        ['毆', 'オウダスルノオウ，ナグルノキュウジ'],
        ['毀', 'メイヨキソンノキ'],
        ['母', 'フボ ノ ボ，ハハ'],
        ['毎', 'マイニチ ノ マイ'],
        ['毋', 'ブ，ナカレ，マイニチノマイノシタガワ'],
        ['毳', 'セイ，ウモウノモウガミッツ'],
        ['毬', 'マリ，ウモウノモウノミギニモトメル'],
        ['毯', 'ジュウタンノタン，ウモウノモウニホノオ'],
        ['毫', 'キゴウスルノゴウ'],
        ['菠', 'ハ、クサカンムリニハチョウガアウノハ'],
        ['華', 'ゴウカ ノ カ，ハナヤカ'],
        ['菫', 'ショクブツノスミレ'],
        ['菩', 'ボダイジュ ノ ボ'],
        ['菷', 'ソウ、ホウキ，クサカンムリニソウジキノソウノツクリ'],
        ['菴', 'アン，イオリ，クサカンムリニキソクエンエンノエン'],
        ['菲', 'ヒ、クサカンムリニヒジョウグチノヒ'],
        ['菰', 'コモカブリ ノ コモ，クサカンムリ ニ コドク ノ コ'],
        ['菱', 'ヒシガタ ノ ヒシ'],
        ['菽', 'シュク、クサカンムリニシュクジョノシュクノツクリ'],
        ['菻', 'リン、クサカンムリニゾウキバヤシノハヤシ'],
        ['菅', 'スゲガサ ノ スゲ，カン'],
        ['菁', 'セイ、カブ、クサカンムリニセイシュンノセイノキュウジタイ'],
        ['菎', 'コン、クサカンムリニコンチュウノコン'],
        ['菌', 'サッキンスル ノ キン'],
        ['菊', 'キクニンギョウ ノ キク'],
        ['菖', 'ハナショウブ ノ ショウ'],
        ['菓', 'オカシ ノ カ'],
        ['菟', 'ト，クサカンムリ ニ ウサギ'],
        ['菜', 'ヤサイ ノ サイ，ナ'],
        ['菘', 'スズナ、クサカンムリニマツノキノマツ'],
        ['戯', 'ギキョク ノ ギ'],
        ['戮', 'サツリクスルノリク'],
        ['駒', 'コマ，ウマヘン ニ ハイク ノ ク'],
        ['駑', 'ドバノド，ヤッコノシタニウマ'],
        ['截', 'セツゼンタルノセツ，サイバイスルノサイノキノカワリニフルトリ'],
        ['駟', 'シ，ウマヘンニカンスウジノヨン'],
        ['駝', 'ドウブツノラクダノダ'],
        ['戦', 'センソウ ノ セン，タタカウ'],
        ['戡', 'カン，ハナハダシイノミギニホコヅクリ'],
        ['駘', 'シュンプタイトウノタイ'],
        ['駆', 'ヨンリンクドウ ノ ク，カケル'],
        ['房', 'レイダンボウ ノ ボウ'],
        ['駄', 'ダガシ ノ ダ'],
        ['戸', 'トジマリ ノ ト，コ'],
        ['戻', 'モドル，ヘンレイキン ノ レイ'],
        ['戴', 'タイカンシキ ノ タイ，イタダク'],
        ['戰', 'センソウノセン，タタカウノキュウジ'],
        ['戳', 'タク，センタクモノノタクノツクリニホコヅクリ'],
        ['戲', 'ギキョクノギノキュウジ'],
        ['戍', 'ジュ，マモル，ボシンセンソウノボノナカニテン'],
        ['戌', 'ジュツ，ジュウニシノイヌ'],
        ['戎', 'エビス，ホコガマエ ノ ジュウ'],
        ['戉', 'マサカリ，エツ，カネヘンナシ'],
        ['戈', 'カ，ブシュノホコヅクリ'],
        ['駱', 'ドウブツノラクダノラク'],
        ['戊', 'ボシンセンソウ ノ ボ'],
        ['駿', 'シュンメ ノ シュン'],
        ['駻', 'カンバノカン，アラウマ'],
        ['戀', 'コイビトノコイノキュウジ'],
        ['駸', 'シン，ウマヘンニシンリャクスルノシンノツクリ'],
        ['戝', 'サンゾクノゾクノイタイジ'],
        ['戟', 'シゲキスル ノ ゲキ，ホコ'],
        ['戞', 'カツ，ユウウツノユウノウエガワニホコヅクリ'],
        ['駢', 'ヘン，ウマヘンニヘイヨウスルノヘイノツクリ'],
        ['戛', 'カツ，ページノハノカワリニホコヅクリ'],
        ['戚', 'トオイシンセキ ノ セキ'],
        ['戔', 'サン，アサイノキュウジノツクリ'],
        ['駭', 'ガイ，オドロク，ウマヘンニジュウニシノイノシシ'],
        ['或', 'アルイハ ノ アル'],
        ['我', 'ガマン ノ ガ，ワレ'],
        ['成', 'セイチョウスル ノ セイ，ナル'],
        ['戒', 'ケイカイスル ノ カイ，イマシメル'],
        ['錚', 'ソウ、カネヘンニアラソウノキュウジ'],
        ['錘', 'ボウスイケイ ノ スイ，オモリ'],
        ['錙', 'シ、カネヘンニシチョウヘイノシノツクリ'],
        ['娑', 'シャバニデルノシャ'],
        ['娟', 'ケン，オンナヘンニキヌイトノキヌノツクリ'],
        ['娜', 'アダッポイノダ'],
        ['娚', 'ナン，オンナヘンニオトコ'],
        ['娘', 'ヒトリムスメ ノ ムスメ'],
        ['娃', 'アイ，オンナヘン ニ ツチ フタツ'],
        ['威', 'イゲンガアル ノ イ'],
        ['娉', 'ヘイ，オンナヘンニショウヘイスルノヘイ'],
        ['娶', 'メトル，シュ'],
        ['娵', 'シュ，オンナヘンニシュザイノシュ'],
        ['娼', 'ショウフ ノ ショウ，オンナヘン ニ ヒガ フタツ'],
        ['娥', 'ガ，オンナヘンニガマンノガ'],
        ['娠', 'ニンシンスル ノ シン'],
        ['娯', 'ゴラク ノ ゴ'],
        ['娩', 'ブンベンスル ノ ベン'],
        ['産', 'ザイサン ノ サン'],
        ['甥', 'オイメイ ノ オイ'],
        ['甦', 'ソ，ヨミガエル，サラシナノサラニジドウセイトノセイ'],
        ['用', 'リヨウスル ノ ヨウ，モチイル'],
        ['甫', 'シジン ノ トホ ノ ホ'],
        ['甬', 'ヨウ，ニホンブヨウノヨウノツクリ'],
        ['田', 'スイデン ノ デン，タ'],
        ['由', 'リユウ ノ ユウ'],
        ['甲', 'コウオツ ノ コウ'],
        ['申', 'シンコクスル ノ シン，モウス'],
        ['男', 'オトコ，ダンシ ノ ダン'],
        ['甸', 'デン，ツツミガマエニタンボノタ'],
        ['町', 'マチ，シチョウソン ノ チョウ'],
        ['画', 'ガヨウシ ノ ガ'],
        ['甼', 'マチ，シチョウソンノチョウノイタイジ'],
        ['甃', 'シュウ，カワラノウエニキセツノアキ'],
        ['甄', 'ケン，ケムリノツクリノミギニカワラ'],
        ['甅', 'センチグラム，カワラノミギニタンイノリン'],
        ['甌', 'オウ，クベツノクノキュウジノミギニカワラ'],
        ['甍', 'ボウ，イラカ'],
        ['甎', 'セン，センモンカノセンノキュウジノミギニカワラ'],
        ['甑', 'モノヲムス コシキ'],
        ['甓', 'ヘキ，カベノツチノカワリニカワラ'],
        ['甕', 'オウ，カメ，ヨウリツスルノヨウノツクリノシタニカワラ'],
        ['甘', 'アマイ，カンミリョウ ノ カン'],
        ['甚', 'ジンダイナ ノ ジン，ハナハダシイ'],
        ['甜', 'テンサイトウ ノ テン'],
        ['甞', 'ガシンショウタンノショウノイタイジ'],
        ['生', 'イキル，セイカツ ノ セイ'],
        ['勧', 'カンユウスル ノ カン，ススメル'],
        ['勦', 'ソウ，ハチノスノキュウジニチカラ'],
        ['勤', 'キンム ノ キン'],
        ['勣', 'セキ，セキニンノセキニチカラ'],
        ['勢', 'セイリョク ノ セイ，イキオイ'],
        ['勠', 'リク，ゴビュウノビュウノツクリニチカラ'],
        ['勿', 'モチロン ノ モチ'],
        ['勾', 'キュウコウバイ ノ コウ'],
        ['勺', 'ヨウセキ ノ タンイ ノ シャク，ツツミガマエ ニ テン'],
        ['勹', 'ホウ，ブシュノツツミガマエ'],
        ['勸', 'カンユウスルノカン，ススメルノキュウジ'],
        ['勵', 'ゲキレイスルノレイノキュウジ'],
        ['勳', 'クンショウノクン，イサオノキュウジ'],
        ['勲', 'クンショウ ノ クン，イサオ'],
        ['勍', 'ケイ，トウキョウノキョウニチカラ'],
        ['勉', 'ベンキョウ ノ ベン'],
        ['勇', 'ユウキ ノ ユウ，イサマシイ'],
        ['勅', 'チョクメイ ノ チョク'],
        ['勃', 'ボッパツスル ノ ボツ'],
        ['勁', 'ケイ，ツヨイ，ケイドウミャクノケイノヒダリガワニチカラ'],
        ['募', 'ボシュウスル ノ ボ，ツノル'],
        ['勞', 'ロウドウノロウノキュウジ'],
        ['勝', 'カチマケ ノ カチ，ショウ'],
        ['務', 'ギム ノ ム，ツトメル'],
        ['勘', 'オカンジョウ ノ カン'],
        ['勗', 'キョク，ニチョウビノニチノシタニキュウジョスルノジョ'],
        ['5', 'ゴ'],
        ['動', 'ドウブツ ノ ドウ，ウゴク'],
        ['勒', 'ミロクボサツノロク'],
        ['ゃ', 'チイサイ ヤカン ノ ヤ'],
        ['狸', 'タヌキ'],
        ['狽', 'ロウバイスル ノ バイ'],
        ['狼', 'オオカミ，ロウバイスル ノ ロウ'],
        ['狷', 'ケンカイナノケン，ケモノヘンニキヌイトノキヌノツクリ'],
        ['狩', 'シュリョウ ノ シュ，カリ'],
        ['狭', 'セマイ，キョウ'],
        ['独', 'ドクリツ ノ ドク'],
        ['狡', 'コウカツナノコウ，ズルイ'],
        ['狠', 'コン，ケモノヘンニダイコンノコンノツクリ'],
        ['狢', 'ケモノヘンノドウブツノムジナ'],
        ['狙', 'ソゲキ ノ ソ，ネラウ'],
        ['狛', 'コマイヌ ノ コマ，ケモノヘン ニ シロ'],
        ['狐', 'キツネ'],
        ['狒', 'ドウブツノヒヒノヒ'],
        ['狗', 'テング ノ ク，イヌ'],
        ['狎', 'コウ，ナレル，ケモノヘンニコウオツノコウ'],
        ['狃', 'ジュウ，ケモノヘンニジュウニシノウシ'],
        ['狂', 'クルウ，キョウゲン ノ キョウ'],
        ['狄', 'ナンバンホクテキノテキ'],
        ['狆', 'イヌノチン'],
        ['〝', 'ヒゲヒラキ'],
        ['〟', 'ヒゲトジ'],
        ['【', 'クロカッコ'],
        ['】', 'トジクロカッコ'],
        ['〒', 'ユウビン'],
        ['〓', 'ゲタジルシ'],
        ['〔', 'キッコウ'],
        ['〕', 'トジキッコウ'],
        ['〈', 'ヤマカッコ'],
        ['〉', 'トジヤマカッコ'],
        ['《', 'ニジュウヤマカッコ'],
        ['》', 'トジニジュウヤマカッコ'],
        ['「', 'カギ'],
        ['」', 'トジカギ'],
        ['『', 'ニジュウカギ'],
        ['』', 'トジニジュウカギ'],
        ['　', 'スペース'],
        ['、', 'テン'],
        ['。', 'マル'],
        ['〃', 'オナジク'],
        ['々', 'オドリジ'],
        ['〆', 'シメ'],
        ['〇', 'ゼロ'],
        ['§', 'セクション'],
        ['¨', 'ウムラウト'],
        ['´', 'アクサンテギュ'],
        ['¶', 'ダンラクキゴウ'],
        ['°', 'ド'],
        ['±', 'プラスマイナス'],
        ['邨', 'ムラ，シチョウソンノソンノイタイジ'],
        ['邪', 'ジャマ ノ ジャ，ヨコシマ'],
        ['邯', 'カンタンノユメノカン'],
        ['氏', 'シメイ ノ シ，ウジ'],
        ['那', 'ワカダンナ ノ ナ'],
        ['氈', 'モウセンゴケノセン'],
        ['邦', 'イホウジン ノ ホウ'],
        ['邸', 'テイタク ノ テイ，ヤシキ'],
        ['气', 'キ，ブシュノキガマエ'],
        ['気', 'クウキ ノ キ'],
        ['嶋', 'トウ，ヤマヘン ニ トリ ノ シマ'],
        ['民', 'コクミン ノ ミン，タミ'],
        ['氓', 'ボウ，ボウメイスルノボウノミギニコクミンノミン'],
        ['嶌', 'トウ，シマグニノシマノイタイジ，ヤマノシタニトリ'],
        ['邵', 'ショウ，マネクノツクリノミギニオオザト'],
        ['氛', 'フン，キガマエニハンブンノブン'],
        ['氤', 'イン，キガマエニインガオウホウノイン'],
        ['邊', 'シュウヘンノヘン，アタリノキュウジ'],
        ['氣', 'クウキノキノキュウジ'],
        ['邏', 'ケイラスルノラ'],
        ['邀', 'ヨウ，シンニョウニゲキレイスルノゲキノツクリ'],
        ['邁', 'マイシンスルノマイ'],
        ['邂', 'メグリアウイミノカイコウノカイ'],
        ['邃', 'スイ，シンニュウニアナカンムリニグンタイノタイノツクリ'],
        ['還', 'カンレキ ノ カン，カエル'],
        ['邇', 'ジ，シンニョウニソツジナガラノジ'],
        ['水', 'スイヨウビ ノ スイ，ミズ'],
        ['氷', 'ヒョウガ ノ ヒョウ，コオリ'],
        ['邑', 'トユウ ノ ユウ，ムラ'],
        ['氾', 'カワガハンランスル ノ ハン'],
        ['永', 'エイキュウ ノ エイ，ナガイ'],
        ['褓', 'ホウ、ムツキ、コロモヘンニホケンシツノホ'],
        ['褒', 'ホメル，ホウビ ノ ホウ'],
        ['褐', 'カッショク ノ カツ'],
        ['嶄', 'ザン，ヤマノシタニザンシンナノザン'],
        ['褝', 'タン、コロモヘンニタンジュンノタン'],
        ['複', 'フクザツ ノ フク'],
        ['褄', 'ツジツマガアウノツマ'],
        ['U', 'ユー ユージュアリー'],
        ['褂', 'カイ、コロモヘンニカケブトンノカケノツクリ'],
        ['嶇', 'ク，ヤマヘンニクベツスルノクノキュウジ'],
        ['褊', 'ヘン、コロモヘンニヘンペイソクノヘン'],
        ['褶', 'シュウキョクサンミャクノシュウ、ヒダ'],
        ['千', 'カンスウジ ノ セン'],
        ['褻', 'ワイセツナノセツ'],
        ['褸', 'ランルヲマトウノル'],
        ['褥', 'サンジョクノジョク、コロモヘンニブジョクノジョク'],
        [
            '褫',
            'チ，テイシンショウノテイノキュウジノシンニョウノカワリニコロモヘン',
        ],
        ['褪', 'イロアセルノアセル、コロモヘンニタイクツノタイ'],
        ['嶝', 'トウ，ヤマヘンニトザンノト'],
        ['歟', 'ヨ，アタエルノキュウジニフカケツノケツ'],
        ['歛', 'カン，ケンサスルノケンノキュウジノツクリニフカケツノケツ'],
        [
            '歙',
            'キュウ，ゴウカクノゴウノシタニウモウノウ，ソノミギニフカケツノケツ',
        ],
        ['麈', 'シュ，シカノシタニシュジンコウノシュ'],
        ['歔', 'キョ，キョエイシンノキョノキュウジニフカケツノケツ'],
        ['麋', 'ビ，シカノシタニコメ'],
        ['麌', 'オジカ，ゴ，シカノシタニゴフクヤノゴノキュウジケイ'],
        ['歓', 'カンゲイ ノ カン，ヨロコブ'],
        ['歐', 'オウベイノオウノキュウジ'],
        ['歎', 'ナゲク，タンニショウ ノ タン'],
        ['麑', 'カノコ，ゲイ，シカノシタニジドウセイトノジノキュウジ'],
        ['歌', 'ウタ，カシュ ノ カ'],
        ['麓', 'サンロク ノ ロク，フモト'],
        ['麕', 'ノロジカノノロ，シカノシタニクニガマエニノギヘンノノギ'],
        ['歉', 'ケン，カネルニフカケツノケツ'],
        ['歇', 'カンケツセンノケツ，カッショクノカツノツクリニフカケツノケツ'],
        ['歃', 'ソウ，ソウニュウクノソウノキュウジノツクリニフカケツノケツ'],
        ['麟', 'キリン ノ リン'],
        ['歿', 'ボツ，カバネヘンニチンボツノボツノキュウジノツクリ'],
        ['死', 'シボウ ノ シ，シヌ'],
        ['歸', 'キタクスルノキ，カエルノキュウジ'],
        ['歹', 'ガツ，ブシュノカバネヘン'],
        ['麩', 'オフノフ，ムギノキュウジニオット'],
        ['歴', 'レキシ ノ レキ'],
        ['歳', 'サイゲツ ノ サイ，トシ'],
        ['歯', 'ハブラシ ノ ハ，シ'],
        ['歪', 'ヒズム，ワイキョクスル ノ ワイ'],
        ['歩', 'アルク，ホドウ ノ ホ'],
        ['武', 'ブキ ノ ブ'],
        ['麹', 'コウジキン ノ コウジ'],
        ['此', 'ヒガンシガン ノ シ，コノ'],
        ['麻', 'マスイ ノ マ，アサ'],
        ['止', 'キンシスル ノ シ，トメル'],
        ['正', 'タダシイ，ショウガツ ノ ショウ'],
        ['麾', 'イエヤスキカノキ，マスイノマノシタニウモウノモウ'],
        ['歡', 'カンゲイスルノカンノキュウジ'],
        ['聨', 'ミミヘンノレンゴウノレンノイタイジ'],
        ['聯', 'ミミヘン ノ レンゴウ ノ レン'],
        ['聡', 'ソウメイナ ノ ソウ，サトイ'],
        ['聢', 'シカト，ミミヘンニサダメル'],
        ['聹', 'ネイ，ミミヘンニテイネイノネイ'],
        ['聽', 'ホチョウキノチョウノキュウジ'],
        ['聾', 'ロウガッコウ ノ ロウ'],
        ['聿', 'イツ，ブシュノフデヅクリ'],
        ['聰', 'ソウメイナノソウノキュウジ'],
        ['聲', 'オンセイノセイ，コエノキュウジ'],
        ['聳', 'ソビエル'],
        ['聴', 'チョウカク ノ チョウ，キク'],
        ['聶', 'ジョウ，ミミミッツ'],
        ['職', 'ショクギョウ ノ ショク'],
        ['聊', 'ブリョウヲカコツノリョウ'],
        ['聆', 'レイ，ミミヘンニメイレイスルノレイ'],
        ['聘', 'ショウヘイスルノヘイ'],
        ['聚', 'ジュラクダイノジュ'],
        ['聞', 'シンブン ノ ブン，キコエル'],
        ['聟', 'ムコノイタイジ，ツウチヒョウノチノシタニミミ'],
        ['聒', 'カツ，ミミヘンニシタヲカムノシタ'],
        ['聖', 'セイカリレー ノ セイ'],
        ['+', 'プラス'],
        ['糒', 'ヒ，ホシイイ，コメヘンニジュンビスルノビノツクリ'],
        ['糖', 'ブドウトウ ノ トウ'],
        ['糘', 'スクモ，コメヘンニイエ'],
        ['糜', 'ビランノビ'],
        ['糟', 'カス，ソウコウ ノ ツマ ノ ソウ'],
        ['糞', 'フンニョウ ノ フン，クソ'],
        ['糀', 'コウジ，コメヘンニハナビノハナ'],
        ['糂', 'サン，コメヘンニハナハダシイ'],
        ['糅', 'ジュウ，コメヘンニジュウドウノジュウ'],
        ['糊', 'ノリヅケ ノ ノリ，コ'],
        ['糎', 'ナガサ ノ タンイ ノ センチメートル'],
        ['糲', 'レイ，コメヘンニハゲムノキュウジノヒダリガワ'],
        ['糴', 'テキ，カイヨネ'],
        ['糶', 'チョウ，ウリヨネ'],
        ['糸', 'ケイト ノ イト，シ'],
        ['系', 'タイヨウケイ ノ ケイ'],
        ['糺', 'キュウダンスルノキュウ，イトヘンニフダノツクリ'],
        ['糾', 'フンキュウスル ノ キュウ'],
        ['糠', 'ヌカヅケ ノ ヌカ'],
        ['糢', 'アイマイモコノモ，コメヘンニバクダイナノバク'],
        ['糧', 'ショクリョウ ノ リョウ，カテ'],
        ['糯', 'モチゴメ，コメヘンニヒツジュヒンノジュ'],
        ['妙', 'キミョウナ ノ ミョウ'],
        ['妛', 'ヤマノシタニカンスウジノイチ，ソノシタニオンナ'],
        ['妝', 'ショウ，ショウグンノショウノキュウジノヒダリガワニオンナ'],
        ['妓', 'ゲイギ ノ ギ，オンナヘン ニ ササエル'],
        ['妖', 'モリ ノ ヨウセイ ノ ヨウ'],
        ['妊', 'ニンシンスル ノ ニン'],
        ['妍', 'ケン，オンナヘンニケンキュウノケンノツクリ'],
        ['妁', 'シャク，オンナヘンニヨウセキノタンイノシャク'],
        ['如', 'トツジョ ノ ジョ'],
        ['妃', 'オウヒ ノ ヒ，キサキ'],
        ['妄', 'コダイモウソウ ノ モウ'],
        ['妹', 'イモウト，シマイ ノ マイ'],
        ['妻', 'フサイ ノ サイ，ツマ'],
        ['妾', 'メカケ，ショウ'],
        ['妲', 'ダツ，オンナヘンニガンタンノタン'],
        ['妨', 'ボウガイスル ノ ボウ，サマタゲル'],
        ['妬', 'シットスル ノ ト，ネタム'],
        ['妣', 'ヒ，オンナヘンニヒカクスルノヒ'],
        ['妥', 'ダキョウスル ノ ダ'],
        ['疫', 'メンエキ ノ エキ'],
        ['疣', 'イボ，ユウ，ヤマイダレニモットモ'],
        ['疥', 'ヒフビョウノカイセンノカイ，ヤマイダレニカイゴノカイ'],
        ['疸', 'オウダンノダン，ヤマイダレニガンタンノタン'],
        ['疹', 'フウシン ノ シン'],
        ['疾', 'シッペイ ノ シツ'],
        ['疼', 'トウツウノトウ，ウズク'],
        ['疽', 'エソヲオコスノソ'],
        ['疲', 'ツカレル，ヒロウスル ノ ヒ'],
        ['疳', 'カン，ヤマイダレニアマイ'],
        ['疱', 'ミズボウソウノホウ'],
        ['疵', 'キズ，ヤマイダレニヒガンシガンノシ'],
        ['疊', 'タタミノキュウジ'],
        ['疋', 'オリモノ ノ タンイ ノ ヒキ'],
        ['疉', 'タタミノキュウジノシタガヘンカシタイタイジ'],
        ['疎', 'カソチ ノ ソ'],
        ['疏', 'ソスイ ノ ソ，ベンソ ノ ソ'],
        ['疂', 'タタミノキュウジノチュウオウガヘンカシタイタイジ'],
        ['疆', 'シンキョウウイグルジチクノキョウ'],
        ['疇', 'ハンチュウノハン，タヘンニコトブキノキュウジ'],
        ['疚', 'キュウ，ヤマイダレニエイキュウノキュウ'],
        ['疝', 'センキモチノセン，ヤマイダレニヤマ'],
        ['疑', 'ギモン ノ ギ，ウタガウ'],
        ['疔', 'メンチョウノチョウ'],
        ['l', 'エル ラブ'],
        ['掵', 'ハバ，テヘンニイノチ'],
        ['掴', 'ツカム，テヘン ニ クニ'],
        ['掲', 'ケイジバン ノ ケイ，カカゲル'],
        ['饑', 'キキンニミマワレルノキ，ヒダルイ'],
        ['掾', 'エン，テヘンニエンガワノエンノキュウジノツクリ'],
        ['饒', 'ジョウゼツノジョウ，ユタカ'],
        ['掻', 'ヒッカク ノ カク，カッカソウヨウ ノ ソウ'],
        ['饗', 'キョウエン ノ キョウ，モテナス'],
        ['控', 'コウジョスル ノ コウ，ヒカエル'],
        ['接', 'セツゾクスル ノ セツ'],
        ['掣', 'セイ，セイゲンスルノセイノシタニテ'],
        ['探', 'タンケン ノ タン，サガス'],
        ['採', 'サイヨウ ノ サイ，トル'],
        ['掠', 'カスメル，リャクダツ ノ リャク'],
        ['掬', 'スクイナゲ ノ スクウ'],
        ['掫', 'ソウ，テヘンニシュザイスルノシュ'],
        ['措', 'ソチスル ノ ソ'],
        ['掩', 'エンゴシャゲキ ノ エン'],
        ['推', 'スイセンスル ノ スイ，オス'],
        ['掖', 'エキ，テヘンニヒルヨルノヨル'],
        ['排', 'ハイジョスル ノ ハイ'],
        ['掟', 'オキテ，テヘンニサダメル'],
        ['掛', 'カケブトン ノ カケ'],
        ['掘', 'ホル，ハックツ ノ クツ'],
        ['掃', 'ソウジスル ノ ソウ，ハク'],
        ['掀', 'キン，テヘンニキンキジャクヤクノキン'],
        ['掏', 'トウ，テヘンニトウゲイカノトウノツクリ'],
        ['掎', 'キ，テヘンニキミョウナノキ'],
        ['掌', 'ショウアクスル ノ ショウ，テノヒラ'],
        ['掉', 'チョウ，テヘンニタッキュウノタク'],
        ['授', 'ジュギョウ ノ ジュ，サズケル'],
        ['蹂', 'ジュウリンスルノジュウ'],
        ['蹄', 'テイテツ ノ テイ，ヒヅメ'],
        ['蹇', 'ケン，サムイノシタニテンノカワリニアシ'],
        ['蹈', 'トウ，アシヘンニイネノキュウジノツクリ'],
        ['蹉', 'ジンセイノサテツノサ'],
        ['蹊', 'ケイ，アシヘンニケイリュウヅリノケイノキュウジノツクリ'],
        ['蹌', 'ヨロメク，ソウロウノソウ'],
        ['蹐', 'セキ，アシヘンニセキズイノセキ'],
        ['蹕', 'ヒツ，アシヘンニヒッキョウノヒツ'],
        ['蹙', 'ヒンシュクヲカウノシュク'],
        ['蹟', 'キセキ ノ セキ，アシヘン ニ セキニン ノ セキ'],
        ['蹠', 'セキ，アシヘンニショミンノショ'],
        ['蹣', 'スイホマンサンノマン，アシヘンニマンゾクノマンノキュウジノツクリ'],
        ['蹤', 'ショウ，アシヘンニジュウジスルノジュウノキュウジ'],
        ['蹲', 'ソンキョスルノソン，アシヘンニソンケイスルノソン'],
        ['蹴', 'ケル，シュウキュウ ノ シュウ'],
        ['蹶', 'ケツ，アシヘンニユウボクミンゾクノトッケツノケツ'],
        ['蹼', 'ホク，アシヘンニダボクノボクノツクリ'],
        ['進', 'シンポ ノ シン，ススム'],
        ['涎', 'スイゼンノマトノゼン，ヨダレ'],
        ['涌', 'ワク，コワクダニ ノ ワク'],
        ['逶', 'イ，シンニョウニガッキュウイインノイ'],
        ['消', 'ショウボウシャ ノ ショウ，ケス'],
        ['涅', 'ネハンノネ'],
        ['逹', 'ハイタツノタツノイタイジ'],
        ['逾', 'ユ，シンニョウニニレノキノニレノツクリ'],
        ['逼', 'ヒッパクスル ノ ヒツ'],
        ['逢', 'ダンジョ ノ アイビキ ノ アイ'],
        ['連', 'レンラク ノ レン'],
        ['造', 'カイゾウ ノ ゾウ，ツクル'],
        ['涜', 'ボウトクスル ノ トク，ケガス'],
        ['涛', 'トウ，サンズイ ニ コトブキ'],
        ['逧', 'サコ，シンニョウニタニ'],
        ['涙', 'ナミダ，ルイセン ノ ルイ'],
        ['涕', 'テイ，ナミダ，サンズイニオトウト'],
        ['涓', 'ケン，サンズイニキヌイトノキヌノツクリ'],
        ['涯', 'イッショウガイ ノ ガイ'],
        ['逓', 'テイシンショウ ノ テイ'],
        ['逐', 'チクジ ノ チク'],
        ['逑', 'キュウ、シンニョウニモトメル'],
        ['逖', 'テキ、シンニョウニイテキノテキ'],
        ['逗', 'トウリュウスル ノ トウ，シンニョウ ニ マメ'],
        ['途', 'トチュウ ノ ト'],
        ['逕', 'ケイ、シンニョウニケイドウミャクノケイノヒダリガワ'],
        ['通', 'コウツウ ノ ツウ，トオル'],
        ['這', 'ハウ'],
        ['逞', 'タクマシイ，テイ'],
        ['速', 'ソクド ノ ソク，ハヤイ'],
        ['逝', 'セイキョスル ノ セイ，ユク'],
        ['逃', 'ニゲル，トウボウスル ノ トウ'],
        ['退', 'タイクツ ノ タイ'],
        ['涼', 'スズシイ，ノウリョウ ノ リョウ'],
        ['逆', 'ギャクテン ノ ギャク，サカサ'],
        ['涸', 'カレル，サンズイニカタマル'],
        ['逋', 'ホ、シンニョウニシジンノトホノホ'],
        ['涵', 'トクヲカンヨウスルノカン'],
        ['逎', 'シュウ，シンニョウニジュウニシノトリ'],
        ['液', 'エキタイ ノ エキ'],
        ['逍', 'サンポヲイミスルショウヨウノショウ'],
        ['覘', 'テン，ドクセンスルノセンノミギニケンブツスルノケン'],
        ['B', 'ビー ボーイ'],
        ['覓', 'ベキ，ツメノシタニケンブツスルノケン'],
        ['覗', 'ノゾキミル ノ ノゾク'],
        ['視', 'シリョクケンサ ノ シ'],
        ['覈', 'ガク，オオイカンムリニゲキレイスルノゲキノツクリ'],
        ['見', 'ミル，ケンブツ ノ ケン'],
        ['覊', 'キビセイサクノキノイタイジ'],
        ['規', 'キソクテキ ノ キ'],
        ['要', 'ヨウキュウスル ノ ヨウ'],
        ['覃', 'タン，オオイカンムリノシタニハヤオキノハヤイ'],
        ['覇', 'セイハスル ノ ハ'],
        ['覆', 'フクメン ノ フク，オオウ'],
        ['覺', 'カンカクノカク，オボエルノキュウジ'],
        ['覽', 'ハクランカイノランノキュウジ'],
        ['覿', 'コウカテキメンノテキ'],
        ['観', 'カンキャク ノ カン'],
        [
            '覲',
            'キン，キンムノキンノキュウジタイノチカラノカワリニケンブツスルノケン',
        ],
        ['覩', 'ト，カガクシャノシャノキュウジタイノミギニケンブツスルノケン'],
        ['親', 'オヤ，リョウシン ノ シン'],
        ['覬', 'キ，ガイセンモンノガイノミギニケンブツスルノケン'],
        ['覯', 'コウ，ミゾノツクリノミギニケンブツスルノケン'],
        ['覡', 'ケキ，ジンジャノミコノフノミギニケンブツスルノケン'],
        ['覧', 'ハクランカイ ノ ラン'],
        ['覦', 'ユ，ニレノキノニレノツクリノミギニケンブツスルノケン'],
        ['も', 'モミジ ノ モ'],
        ['剰', 'ジシンカジョウ ノ ジョウ'],
        ['剳', 'ゴジュウノトウノトウノツクリニリットウ'],
        ['割', 'ワリアイ ノ ワリ'],
        ['創', 'ソウリツスル ノ ソウ'],
        ['ょ', 'チイサイ ヨット ノ ヨ'],
        ['や', 'ヤカン ノ ヤ'],
        ['ゅ', 'チイサイ ユカタ ノ ユ'],
        ['り', 'リンゴ ノ リ'],
        ['る', 'ルスバン ノ ル'],
        ['よ', 'ヨット ノ ヨ'],
        ['ら', 'ラジオ ノ ラ'],
        ['剽', 'ヒョウセツスルノヒョウ'],
        ['わ', 'ワカメ ノ ワ'],
        ['れ', 'レモン ノ レ'],
        ['ろ', 'ロウカ ノ ロ'],
        ['を', 'ヲワリ ノ オ'],
        ['ん', 'オシマイ ノ ン'],
        ['ゐ', 'ムカシ ノ イ'],
        ['ゑ', 'ムカシ ノ エ'],
        ['剥', 'ハクリスル ノ ハク，ハガレル'],
        ['剤', 'ヤクザイシ ノ ザイ'],
        ['剩', 'ジョウ，ジシンカジョウノジョウノキュウジ'],
        ['゛', 'ダクテン'],
        ['剪', 'ニワキヲセンテイスルノセン，キル'],
        ['ゞ', 'クリカエシダクテン'],
        ['゜', 'ハンダクテン'],
        ['ゝ', 'クリカエシ'],
        ['ア', 'アサヒ ノ ア'],
        ['ィ', 'チイサイ イチゴ ノ イ'],
        ['ァ', 'チイサイ アサヒ ノ ア'],
        ['ウ', 'ウサギ ノ ウ'],
        ['ェ', 'チイサイ エイゴ ノ エ'],
        ['イ', 'イチゴ ノ イ'],
        ['ゥ', 'チイサイ ウサギ ノ ウ'],
        ['オ', 'オオサカ ノ オ'],
        ['カ', 'カゾク ノ カ'],
        ['エ', 'エイゴ ノ エ'],
        ['ォ', 'チイサイ オオサカ ノ オ'],
        ['ギ', 'ギンコウ ノ ギ'],
        ['ク', 'クスリ ノ ク'],
        ['ガ', 'ガッコウ ノ ガ'],
        ['キ', 'キッテ ノ キ'],
        ['ゲ', 'ゲーム ノ ゲ'],
        ['コ', 'コドモ ノ コ'],
        ['グ', 'グランド ノ グ'],
        ['ケ', 'ケシキ ノ ケ'],
        ['ザ', 'ザブトン ノ ザ'],
        ['シ', 'シンブン ノ シ'],
        ['則', 'ホウソク ノ ソク'],
        ['サ', 'サクラ ノ サ'],
        ['ズ', 'ズボン ノ ズ'],
        ['セ', 'セカイ ノ セ'],
        ['ジ', 'ジカン ノ ジ'],
        ['ス', 'スズメ ノ ス'],
        ['前', 'マエ，ゼンゴ ノ ゼン'],
        ['剌', 'ハツラツノラツ'],
        ['剏', 'ソウ，ヘイヨウスルノヘイノツクリニヤイバノイタイジ'],
        ['ソ', 'ソロバン ノ ソ'],
        ['罘', 'フウ，アミガシラニフシギノフ'],
        ['罟', 'コ，アミガシラニチュウコシャノコ'],
        ['网', 'ボウ，ブシュノアミガシラ'],
        ['罐', 'カンヅメノカンノキュウジ'],
        ['罕', 'トリアミヲイミスルカン'],
        ['罔', 'モウ，モウマクノモウノツクリ'],
        ['罎', 'アキビンノビン，ホトギヘンニクモリ'],
        ['罍', 'ライ，タハタノタミッツノシタニホトギ'],
        ['罌', 'オウ，カイガラノカイフタツノシタニホトギ'],
        ['罅', 'ヒビワレルノヒビ，カ'],
        ['罹', 'リカンスルノリ，カカル'],
        ['罸', 'ツミトバツノバツノイタイジ'],
        ['署', 'ショメイスル ノ ショ'],
        ['罰', 'ツミトバツ ノ バツ'],
        ['罷', 'ヒメンスル ノ ヒ'],
        ['罵', 'バセイ ノ バ，ノノシル'],
        ['罫', 'ケイセンヲヒク ノ ケイ'],
        ['罪', 'ハンザイ ノ ザイ，ツミ'],
        ['罩', 'トウ，アミガシラニタッキュウノタク'],
        ['罨', 'アンポウノアン'],
        ['置', 'ソウチ ノ チ，オク'],
        ['ゴ', 'ゴリラ ノ ゴ'],
        ['罠', 'ワナ，アミガシラニタミ'],
        ['罧', 'シノヅケ，アミガシラニミツリンノリン'],
        ['嶂', 'ショウ，ヤマヘンニブンショウヲカクノショウ'],
        ['邱', 'キュウ，サキュウノキュウノミギニオオザト'],
        ['ゼ', 'ゼンブ ノ ゼ'],
        ['擠', 'セイ，テヘンニヘソノツクリ'],
        ['擡', 'タイトウスルノタイ，テヘンニダイドコロノダイノキュウジ'],
        ['擢', 'バッテキスル ノ テキ'],
        ['擣', 'トウ，テヘンニコトブキノキュウジ'],
        ['擦', 'マサツ ノ サツ，スル'],
        ['擧', 'センキョノキョ，アゲルノキュウジ'],
        ['擬', 'モギテン ノ ギ'],
        ['擯', 'ヒン，テヘンニライヒンノヒン'],
        ['鸚', 'トリノオウムノオウ'],
        ['擱', 'カクザスルノカク，テヘンニテンシュカクノカク'],
        ['擲', 'トウテキキョウギノテキ，ナゲウツ'],
        ['擴', 'カクダイスルノカクノキュウジ'],
        ['擶', 'セン，テヘンニタケカンムリニマエ'],
        ['擺', 'ハイ，テヘンニヒメンスルノヒ'],
        ['擽', 'リャク，テヘンニオンガクノガクノキュウジ'],
        ['擾', 'ジョウラン ノ ジョウ，テヘン ニ ユウウツ ノ ユウ'],
        ['擁', 'ヨウリツスル ノ ヨウ'],
        ['擂', 'ライ，テヘンニカミナリ'],
        ['擅', 'ドクセンジョウノセン，テヘンニダンカノダンノツクリ'],
        ['擇', 'センタクシノタクノキュウジ'],
        ['操', 'ソウジュウスル ノ ソウ，アヤツル'],
        ['擒', 'キン，テヘンニモウキンルイノキン'],
        ['擔', 'タンニンノタンノキュウジ'],
        ['擘', 'ヘキ，カベノツチノカワリニテ'],
        ['據', 'キョテンノキョノキュウジ'],
        ['胱', 'ボウコウエンノコウ'],
        ['胴', 'ドウアゲ ノ ドウ'],
        ['胸', 'キョウイ ノ キョウ，ムネ'],
        ['胼', 'ヘイ，ニクヅキニヘイゴウスルノヘイノツクリ'],
        ['能', 'サイノウ ノ ノウ'],
        ['胡', 'クロコショウ ノ コ'],
        ['胤', 'ゴラクイン ノ イン，タネ'],
        ['胥', 'ショ，オリモノノタンイノヒキノシタニニクヅキ'],
        ['Y', 'ワイ ヤング'],
        ['胯', 'コ，ニクヅキニハカマノツクリ'],
        ['胖', 'ハン，ニクヅキニハンブンノハン'],
        ['驤', 'ジョウ，ウマヘンニユズルノツクリノキュウジ'],
        ['胚', 'ハイガマイノハイ'],
        ['胛', 'コウ，ニクヅキニコウオツノコウ'],
        ['胙', 'ソ，ニクヅキニムカシナガラノナガラ'],
        ['胞', 'サイボウ ノ ボウ'],
        ['胝', 'チ，ニクヅキニテイコウスルノテイノツクリ'],
        ['胃', 'イブクロ ノ イ'],
        ['胆', 'ダイタンナ ノ タン'],
        ['胄', 'チュウ，リユウノユウノシタニニクヅキ'],
        ['胎', 'ジュタイスル ノ ジュ'],
        ['背', 'セボネ ノ セ'],
        ['難', 'コンナン ノ ナン，ムズカシイ'],
        ['離', 'ハナレル，リリク ノ リ'],
        ['挺', 'クウテイブタイ ノ テイ'],
        ['挽', 'バンカイスル ノ バン'],
        ['挿', 'ソウニュウスル ノ ソウ，サス'],
        ['挾', 'ハサム，キョウノキュウジ'],
        ['雫', 'シズク，アメカンムリ ニ シタ'],
        ['雪', 'ユキダルマ ノ ユキ，セツ'],
        ['雨', 'アメ，ウテン ノ ウ'],
        ['挨', 'アイサツスル ノ アイ'],
        ['挫', 'ネンザ ノ ザ，クジク'],
        ['雰', 'フンイキ ノ フン'],
        ['雷', 'カミナリ，ライウ ノ ライ'],
        ['零', 'レイサイキギョウ ノ レイ'],
        ['振', 'サンシンスル ノ シン，フル'],
        ['電', 'デンワ ノ デン'],
        ['雹', 'ヒョウアラレノヒョウ，アメカンムリニツツム'],
        ['挧', 'ウ，テヘンニウモウノウ'],
        ['挙', 'センキョ ノ キョ，アゲル'],
        ['雁', 'ワタリドリ ノ ガン，カリ'],
        ['雀', 'スズメ，ジャク'],
        ['雇', 'ヤトウ，コヨウスル ノ コ'],
        ['集', 'シュウゴウスル ノ シュウ，アツマル'],
        ['挟', 'ハサム，キョウ'],
        ['雄', 'オスメス ノ オス，ユウ'],
        ['挑', 'チョウセンスル ノ チョウ，イドム'],
        ['驩', 'カン，ウマヘンニカンボクノカン'],
        ['雉', 'トリノキジ、チ'],
        ['雎', 'ショ、ナオカツノカツノミギニフルトリ'],
        ['雍', 'シンノヨウセイテイノヨウ'],
        ['雌', 'オスメス ノ メス，シ'],
        ['按', 'アンマ ノ アン'],
        ['挈', 'ケツ，ケイヤクスルノケイノダイノカワリニテ'],
        ['雑', 'ザツオン ノ ザツ'],
        ['挌', 'カク，テヘンニカクジノカク'],
        ['雕', 'チョウ、シュウヘンノシュウノミギニフルトリ'],
        ['持', 'ジゾクスル ノ ジ，モツ'],
        ['雙', 'ソウガンキョウノソウノキュウジ'],
        ['挂', 'ケイ，テヘンニツチフタツ'],
        ['指', 'シドウスル ノ シ，ユビ'],
        ['雜', 'ザツオンノザツノイキュウジ'],
        ['車', 'クルマ，デンシャ ノ シャ'],
        ['軋', 'アツレキノアツ'],
        ['軈', 'ヤガテ、ミヘンニオウエンスルノオウノキュウジ'],
        ['軌', 'キドウニノル ノ キ'],
        ['軍', 'グンタイ ノ グン'],
        ['軆', 'タイ、ミヘンニユタカ'],
        ['軅', 'ヤガテ、ミヘンニトリノカリ'],
        ['軛', 'クビキ、クルマヘンニヤクドシノヤク'],
        ['∇', 'ナブラ'],
        ['軒', 'イッケンヤ ノ ケン，ノキ'],
        ['軫', 'シン、クルマヘンニシンサツスルノシンノツクリ'],
        ['転', 'ウンテン ノ テン，コロガス'],
        ['軣', 'ゴウ、トドロクノイタイジ'],
        ['軻', 'カ、クルマヘンニカノウセイノカ'],
        ['軸', 'カケジク ノ ジク'],
        ['√', 'ルート'],
        ['軾', 'ショク、クルマヘンニニュウガクシキノシキ'],
        ['軼', 'イツ、クルマヘンニウシナウ'],
        ['軽', 'カルイオモイ ノ カルイ，ケイ'],
        ['∝', 'ヒレイ'],
        ['夢', 'ユメ，ムチュウ ノ ム'],
        ['大', 'ダイショウ ノ ダイ，オオキイ'],
        ['夥', 'カ，オビタダシイ'],
        ['太', 'フトイホソイ ノ フトイ，タイ'],
        ['夫', 'オット，フサイ ノ フ'],
        ['天', 'テンキヨホウ ノ テン'],
        ['央', 'チュウオウ ノ オウ'],
        ['夬', 'カイ，ワケル，タスウケツノケツノツクリ'],
        ['夭', 'ヨウセツスルノヨウ，ワカイ'],
        ['夲', 'トウ，オオキイノシタニスウジノジュウ'],
        ['知', 'チシキ ノ チ，シル'],
        ['失', 'シツレン ノ シツウシナウ'],
        ['夷', 'ジョウイ ノ イ，エビス'],
        ['夸', 'カ，コチョウスルノコノツクリ'],
        ['驚', 'オドロク，キョウタン ノ キョウ'],
        ['夾', 'キョウ，ハサムノキュウジノツクリ'],
        ['頗', 'スコブル'],
        ['夂', 'チ，ブシュノフユガシラ'],
        ['夊', 'スイ，ブシュノスイニョウ'],
        ['変', 'ヘンカ ノ ヘン，カエル'],
        ['夏', 'ナツヤスミ ノ ナツ，カ'],
        ['/', 'スラッシュ'],
        ['頚', 'クビ，ケイ'],
        ['夐', 'ケイ，スイニョウノハルカ'],
        ['外', 'ガイコク ノ ガイ，ソト'],
        ['夕', 'ユウガタ ノ ユウ'],
        ['多', 'オオイスクナイ ノ オオイ，タ'],
        ['夛', 'オオイスクナイノオオイノイタイジ'],
        ['夘', 'ジュウニシノウノイタイジ'],
        ['夙', 'ツトニ，シュク'],
        ['夜', 'コンヤ ノ ヤ，ヨル'],
        ['∨', 'マタハ'],
        ['琥', 'コハクイロノコ'],
        ['琢', 'セッサタクマ ノ タク，ミガク'],
        ['驍', 'ウマヘンニギョウシュンノギョウノキュウジ'],
        ['琿', 'コン，オウヘンニグンタイノグン'],
        ['琺', 'ホウロウナベノホウ，オウヘンニホウリツノホウ'],
        ['琴', 'モッキン ノ キン，コト'],
        ['琵', 'ビワコ ノ ビ'],
        ['琶', 'ビワコ ノ ニモジメ ノ ハ'],
        ['琲', 'ハイ，コーヒーノアテジノニモジメ'],
        ['琳', 'オウヘン ニ ハヤシ ノ リン'],
        ['琉', 'リュウキュウ ノ リュウ'],
        ['琅', 'ロウ，オウヘンニカイリョウスルノリョウ'],
        ['理', 'リユウ ノ リ'],
        ['∮', 'シュウカイセキブン'],
        ['球', 'チキュウ ノ キュウ，タマ'],
        ['出', 'デル，ガイシュツ ノ シュツ'],
        ['凹', 'オウトツ ノ オウ，クボム'],
        ['凸', 'オウトツ ノ トツ'],
        ['凾', 'ハコダテノハコ，カンノイタイジ'],
        ['函', 'ハコダテ ノ ハコ，カン'],
        ['凱', 'ガイセンモン ノ ガイ'],
        ['凰', 'トリノホウオウノオウ'],
        ['凶', 'キチトキョウ ノ キョウ'],
        ['凵', 'ケン，ブシュノウケバコ'],
        ['凪', 'ユウナギ ノ ナギ'],
        ['凩', 'コガラシ'],
        ['凭', 'ヒョウ，モタレル'],
        ['凡', 'ヘイボン ノ ボン'],
        ['几', 'キチョウメンノキ，ツクエ'],
        ['凧', 'タコアゲ ノ タコ'],
        ['処', 'ショブン ノ ショ'],
        ['凛', 'リンレツタルノリンノイタイジ'],
        ['凝', 'コル，ギョウコスル ノ ギョウ'],
        ['凜', 'リンレツタルノリン'],
        ['凖', 'ジュンビスルノジュンノイタイジ'],
        ['頏', 'コウ，テイコウスルノコウノツクリノミギニページ'],
        ['凋', 'チョウラク ノ チョウ，シボム'],
        ['凉', 'スズシイノイタイジ，ニスイ'],
        ['頌', 'ショウシュンノショウ，オオヤケニページ'],
        ['凍', 'レイトウスル ノ トウ，コオル'],
        ['凌', 'リョウガスル ノ リョウ，シノグ'],
        ['准', 'ヒジュンスル ノ ジュン'],
        ['凅', 'コ，コオル，ニスイニカタマル'],
        ['凄', 'セイサンナ ノ セイ，スゴイ'],
        ['頷', 'ウナズク，ガン'],
        ['頸', 'ケイツイノケイ，クビ'],
        ['頼', 'タノム，イライ ノ ライ'],
        ['頽', 'タイハイテキナニタイ，ハゲヤマノハゲノミギニページ'],
        ['頡', 'ケツ，ダイキチノキチニページ'],
        ['鞣', 'ナメス，ジュウ'],
        ['}', 'トジチュウカッコ'],
        ['頭', 'アタマ，ズツウ ノ ズ'],
        ['ﾇ', 'ヌリエ ノ ヌ'],
        ['p', 'ピー パパ'],
        ['醴', 'アマザケ，レイ'],
        [
            '醵',
            'キョキンノキョ，ジュウニシノトリノミギニゲキダンノゲキノヒダリガワ',
        ],
        ['洒', 'オシャレノシャ，サンズイニニシ'],
        ['洗', 'センタクモノ ノ セン，アラウ'],
        ['洙', 'シュ，サンズイニシュイロノシュ'],
        ['洛', 'ジョウラクスル ノ ラク'],
        ['醸', 'ジョウゾウスル ノ ジョウ，カモス'],
        ['洟', 'ハナミズ，イ，サンズイニジョウイノイ'],
        ['洞', 'ドウクツ ノ ドウ，ホラ'],
        ['醤', 'ショウユ ノ ショウ，ヒシオ'],
        ['醢', 'カイ，ジュウニシノトリノミギニミギノシタニサラ'],
        ['洋', 'ヨウフク ノ ヨウ'],
        ['醯', 'ケイ，ジュウニシノトリノミギニナガレルノツクリノシタニサラ'],
        ['洌', 'セイレツナイズミノレツ，サンズイニギョウレツノレツ'],
        ['醪', 'モロミ，ロウ'],
        ['醫', 'オイシャサンノイノキュウジ'],
        ['洳', 'ジョ，サンズイニトツジョノジョ'],
        ['洲', 'サンカクス ノ ス，サンズイツキ'],
        ['洵', 'シュン，サンズイニゲジュンノジュン'],
        ['醒', 'カクセイザイ ノ セイ，サメル'],
        ['洶', 'キョウ，サンズイニムネガイタムノムネノツクリ'],
        ['醜', 'シュウタイ ノ シュウ，ミニクイ'],
        ['洸', 'コウ，サンズイニヒカリ'],
        ['活', 'セイカツ ノ カツ'],
        ['洽', 'コウ，サンズイニゴウカクノゴウ'],
        ['派', 'ハバツ ノ ハ'],
        ['醇', 'ホウジュンナカオリ ノ ジュン'],
        ['津', 'ツナミ ノ ツ，シン'],
        ['醂', 'ミリンノリン，ジュウニシノトリノミギニハヤシ'],
        ['洩', 'キミツロウエイ ノ エイ，モラス'],
        ['醍', 'ダイゴミ ノ ダイ'],
        ['洫', 'キョク，サンズイニケツエキノケツ'],
        ['洪', 'コウズイ ノ コウ'],
        ['醉', 'デイスイスルノスイノキュウジ'],
        ['醋', 'サク，ジュウニシノトリノミギニムカシ'],
        ['蘢', 'ロウ，クサカンムリニキョウリュウノリュウノキュウジ'],
        ['蘯', 'ホウトウムスコノトウノイタイジ'],
        ['蘭', 'ランガク ノ ラン'],
        ['蘰', 'カズラ，クサカンムリニイトヘンニウナギノツクリ'],
        ['蘿', 'ラ，クサカンムリニモウラスルノラ'],
        [
            '蘂',
            'メシベノシベ，ズイノイタイジ，クサカンムリニココロミッツニジュモクノモク',
        ],
        ['蘇', 'ソセイスル ノ ソ，ヨミガエル'],
        ['蘆', 'アシ，クサカンムリニハゼノキノハゼノツクリ'],
        ['蘋', 'ヒン，クサカンムリニヒンパンナノヒンノキュウジ'],
        ['蘊', 'ウンチクヲカタムケルノウン，イトヘン'],
        ['蘓', 'ソセイノソノイタイジ，クサカンムリニノギヘンニサカナ'],
        ['蘗', 'オウバクシュウノバクノイタイジ'],
        ['蘖', 'ゲツ，ヒコバエ'],
        ['蘚', 'セン，コケ，クサカンムリニシンセンナノセン'],
        ['龍', 'タツ，リュウ ノ キュウジ'],
        ['摩', 'マサツ ノ マ'],
        ['摯', 'シンシナタイドノシ，シツジノシツノシタニテ'],
        ['摧', 'サイ，テヘンニモヨオシノツクリ'],
        ['摺', 'テスリアシスリ ノ スリ'],
        ['龝', 'アキ，ノギヘンニカメノキュウジ'],
        ['摸', 'モシャスル ノ モ，テヘン'],
        ['龕', 'ガンドウノガン，ゴウカクノゴウノシタニリュウ'],
        ['摶', 'タン，テヘンニセンモンノセンノキュウジ'],
        ['摎', 'コウ，テヘンニニカワノツクリ'],
        ['摂', 'セッセイスル ノ セイ，トル'],
        ['龠', 'ブシュノヤク，フエ'],
        ['摘', 'テキハツスル ノ テキ，ツム'],
        ['腴', 'ユ，ニクヅキニシュユノマノユ'],
        ['腰', 'ヨウツウ ノ ヨウ，コシ'],
        ['腱', 'アキレスケンノケン'],
        ['F', 'エフ フレンド'],
        ['腿', 'ダイタイコツ ノ タイ，モモ'],
        ['腸', 'ダイチョウ，ショウチョウ ノ チョウ'],
        ['腹', 'ハラペコ ノ ハラ，フク'],
        ['腺', 'コウジョウセン ノ セン'],
        ['腥', 'セイ，ニクヅキニワクセイノセイ'],
        ['腦', 'ズノウメイセキノノウノキュウジ'],
        ['腮', 'エラ，ニクヅキニシアンガオノシ'],
        ['腫', 'シュヨウ ノ シュ，ハレル'],
        ['腔', 'コウクウゲカ ノ クウ，ニクヅキニソラ'],
        ['腕', 'ウデ，ワンリョク ノ ワン'],
        ['腐', 'クサル，フハイスル ノ フ'],
        ['腑', 'フヌケノフ'],
        ['腓', 'ヒコツノヒ，コムラ'],
        ['腟', 'チツ，ニクヅキニキョウシツノシツ'],
        ['腆', 'テン，ニクヅキニコクゴジテンノテン'],
        ['腎', 'ジンゾウビョウ ノ ジン'],
        ['腋', 'エキ，ワキ，ニクヅキニヨルヒルノヨル'],
        ['郢', 'エイ，ゾウテイスルノテイノミギニオオザト'],
        ['郡', 'チメイ ノ グン，コオリ'],
        ['翕', 'キュウ，ゴウカクスルノゴウニウモウノウ'],
        ['翔', 'ヒショウスルノショウ，カケル，ヒツジニウモウノウ'],
        ['習', 'レンシュウ ノ シュウ，ナラウ'],
        ['翌', 'ヨクジツ ノ ヨク'],
        ['翊', 'ヨク，ドクリツノリツノミギニウモウノウ'],
        ['翅', 'シ，ササエルノミギニウモウノウ'],
        ['翆', 'ヒスイノスイノイタイジ'],
        ['翁', 'オキナ，オウ'],
        ['翼', 'ツバサ，ビヨク ノ ヨク'],
        ['翹', 'ギョウ，ギョウシュンノギョウノキュウジニウモウノウ'],
        ['翻', 'ホンヤクスル ノ ホン'],
        [
            '翰',
            'ショカンヲ オクル ノ カン，シンカンセン ノ カン ノ ホス ノ カワリニ ウモウ ノ ウ',
        ],
        ['翳', 'カゲリ，エイ，オイシャサンノイニルマタ，ソノシタニウモウノウ'],
        ['翩', 'ヘンポントヒルガエルノヘン'],
        ['翫', 'ガン，レンシュウ ノ シュウ ニ ゲンキ ノ ゲン'],
        ['翦', 'セン，マエウシロノマエノシタニウモウノウ'],
        ['翡', 'ヒスイノヒ，ヒジョウグチノヒノシタニウモウノウ'],
        ['翠', 'ヒスイ ノ スイ'],
        ['墮', 'ダラクスルノダノキュウジ'],
        ['墨', 'ボクジュウ ノ ボク，スミ'],
        ['墫', 'タルノイタイジ'],
        ['墳', 'コフン ノ フン'],
        ['鐡', 'テツドウノテツノキュウジノテイノカワリニマメノシタガワ'],
        ['墾', 'カイコンチ ノ コン'],
        ['墸', 'ツチヘンニチョシャノチョノキュウジタイ'],
        ['墹', 'ママ，ツチヘンニアイダ'],
        ['墺', 'オウ，ツチヘンニオクバノオクノキュウジ'],
        ['墻', 'ショウ，ツチヘンニリンショクノショク'],
        ['墅', 'ショ，ノハラノノノシタニツチ'],
        ['境', 'コッキョウ ノ キョウ，サカイ'],
        ['増', 'ゾウカスル ノ ゾウ，フエル'],
        ['墓', 'ボチ ノ ボ，ハカ'],
        ['墜', 'ツイラクスル ノ ツイ'],
        ['墟', 'ハイキョノキョ'],
        ['環', 'シゼンカンキョウ ノ カン'],
        ['郊', 'トウキョウキンコウ ノ コウ'],
        ['璽', 'ギョメイギョジ ノ ジ'],
        ['璧', 'ヘキ，カンペキナノヘキ'],
        ['璢', 'ルリイロノルノイタイジ'],
        ['郎', 'モモタロウ ノ ロウ'],
        ['璞', 'ハク，アラタマ，オウヘンニシモベノツクリ'],
        ['璃', 'ルリイロ ノ リ'],
        ['璋', 'ショウ，オウヘンニブンショウヲカクノショウ'],
        ['郛', 'フ，ツメノシタニコドモノコノミギニオオザト'],
        ['鰭', 'サカナ ノ ヒレ'],
        ['陥', 'オチイル，カンラク ノ カン'],
        ['除', 'ジョガイ ノ ジョ，ノゾク'],
        ['陦', 'トウ、コザトヘンニコトブキ'],
        ['陣', 'シュツジンスル ノ ジン'],
        ['院', 'ニュウインスル ノ イン'],
        ['陬', 'ソウ、コザトヘンニシュトクスルノシュ'],
        ['陪', 'バイシンイン ノ バイ'],
        ['陵', 'キュウリョウチ ノ リョウ，ミササギ'],
        ['陷', 'オチイル，カンラクノカンノキュウジ'],
        ['陶', 'トウゲイカ ノ トウ'],
        ['陰', 'インキナ ノ イン，カゲ'],
        ['陳', 'チンレツスル ノ チン'],
        ['陲', 'スイ、コザトヘンニスイチョクノスイ'],
        ['陽', 'タイヨウケイ ノ ヨウ'],
        ['陸', 'リクジョウ ノ リク'],
        ['険', 'ボウケン ノ ケン，ケワシイ'],
        ['附', 'コザトヘンノ ツイタ フゾク ノ フ'],
        ['陀', 'アミダブツ ノ ダ'],
        ['陂', 'ハ，コザトヘンニケガワノカワ'],
        ['降', 'ユキガフル ノ フル，コウ'],
        ['陌', 'ハク，コザトヘンニカンスウジノヒャク'],
        ['陏', 'ダ，コザトヘンニユウメイジンノユウ'],
        ['陋', 'ガンメイコロウノロウ'],
        ['限', 'ゲンカイ ノ ゲン，カギル'],
        ['陝', 'チュウゴクノセンセイショウノセン'],
        ['陜', 'セン，コザトヘンニセマイノキュウジノツクリ'],
        ['陟', 'チョク、コザトヘンニアルクノキュウジ'],
        ['陞', 'ショウ、コザトヘンニイッショウビンノショウノシタニドヨウビノド'],
        ['陛', 'ヘイカ ノ ヘイ'],
        ['轌', 'ソリ、クルマヘンニユキ'],
        ['轍', 'ワダチ'],
        ['轎', 'キョウ、クルマヘンニホドウキョウノキョウノツクリ'],
        ['轉', 'ウンテンスルノテンノキュウジ'],
        ['轄', 'カンカツスル ノ カツ'],
        ['轅', 'エン、ナガエ、クルマヘンニルイジンエンノエンノツクリ'],
        ['轆', 'ロクロノロク'],
        ['轂', 'コシキ、コクモツノコクノノギノカワリニクルマ'],
        ['轜', 'ジ、クルマヘンニヒツジュヒンノジュ'],
        ['轟', 'ゴウオン ノ ゴウ，トドロク'],
        ['轗', 'カン、クルマヘンニカンソウブンノカン'],
        [']', 'トジカクカッコ'],
        ['轤', 'ロクロノロ'],
        ['轡', 'クツワ'],
        ['轢', 'アツレキノレキ'],
        ['轣', 'レキ、クルマヘンニレキシカノレキ'],
        ['鼻', 'ハナカゼ ノ ハナ'],
        ['溘', 'コウ，サンズイニキョネンノキョニサラ'],
        ['溟', 'メイ，サンズイニメイフクヲイノルノメイ'],
        ['溝', 'ミゾ，ハイスイコウ ノ コウ'],
        ['溜', 'リュウインガサガル ノ リュウ，タマリ'],
        ['源', 'スイゲン ノ ゲン，ミナモト'],
        ['鄲', 'カンタンノユメノタン'],
        ['準', 'ジュンビスル ノ ジュン'],
        ['鄰', 'トナリ，リンジンノリンノタイジ'],
        ['鄭', 'ジンメイ ノ テイセイコウ ノ テイ'],
        ['溏', 'トウ，サンズイニケントウシノトウ'],
        ['溌', 'ゲンキハツラツ ノ ハツ'],
        ['溂', 'ハツラツノラツ，サンズイニシゲキスルノシ'],
        ['満', 'マンゾク ノ マン'],
        ['溺', 'デキアイスル ノ デキ，オボレル'],
        ['溽', 'ジョク，サンズイニブジョクノジョク'],
        ['鄙', 'ヘンピナノヒ，ヒナ'],
        ['溲', 'ソウ，サンズイニサンバソウノソウ'],
        ['溷', 'コン，サンズイニクニガマエノナカニブタノツクリ'],
        ['溶', 'ヨウエキ ノ ヨウ，トケル'],
        ['溪', 'ケイリュウヅリノケイノキュウジ'],
        ['溯', 'ソジョウスルノソ，サンズイニハッサクノサク'],
        ['溢', 'アフレル，ノウイッケツ ノ イツ'],
        ['鄂', 'ガク，ワニノツクリノミギニオオザト'],
        ['溥', 'シンコクコウテイノフギノフ'],
        ['蚤', 'ムシ ノ ノミ'],
        ['蚣', 'コウ，ムシヘンニオオヤケ'],
        ['蚯', 'キュウ，ムシヘンニサキュウノキュウ'],
        ['蚩', 'オロカヲイミスルシ'],
        ['蚫', 'アワビ，ムシヘンニツツム'],
        ['蚪', 'トウ，ムシヘンニホクトシチセイノト'],
        ['蚶', 'カン，ムシヘンニアマイ'],
        ['蚰', 'ユウ，ムシヘンニリユウノユウ'],
        ['蚌', 'イツボウノアラソイノボウ'],
        ['蚋', 'ゼイ，ムシヘンニコクナイノナイ'],
        ['蚊', 'カトリセンコウ ノ カ'],
        ['蚕', 'ヨウサン ノ サン，カイコ'],
        ['蚓', 'イン，ミミズ，ムシヘンニインリョクノイン'],
        ['兼', 'カネル，ケンギョウ ノ ケン'],
        ['典', 'コクゴジテン ノ テン'],
        ['兵', 'ヘイタイ ノ ヘイ'],
        ['具', 'ドウグ ノ グ'],
        ['其', 'ソノタ ノ ソ'],
        ['共', 'キョウツウ ノ キョウ，トモ'],
        ['六', 'カンスウジ ノ ロク'],
        ['公', 'シュジンコウ ノ コウ，オオヤケ'],
        ['兮', 'ケイ，カンスウジノハチノシタニゴウキュウスルノゴウノシタガワ'],
        ['兩', 'リョウテノリョウノキュウジ'],
        ['全', 'ゼンコク ノ ゼン，マッタク'],
        ['八', 'カンスウジ ノ ハチ'],
        ['兪', 'ユ，ニレノキノニレノツクリ'],
        ['入', 'ハイル，ニュウガク ノ ニュウ'],
        ['兢', 'センセンキョウキョウノキョウ，コクフクスルノコクフタツ'],
        ['兜', 'テツカブト ノ カブト'],
        ['党', 'ヨトウヤトウ ノ トウ'],
        ['兔', 'ドウブツノウサギノイタイジ'],
        ['児', 'ジドウセイト ノ ジ'],
        ['兒', 'ジドウセイトノジノキュウジ'],
        ['免', 'メンキョ ノ メン'],
        ['兌', 'ダカンシヘイノダ'],
        ['3', 'サン'],
        ['兎', 'ウサギ'],
        ['光', 'ヒカリ，ニッコウ ノ コウ'],
        ['先', 'センセイ ノ セン，サキ'],
        ['克', 'コクフクスル ノ コク，カツ'],
        ['充', 'ジュウジツ ノ ジュウ'],
        ['兄', 'アニ オトウト ノ アニ'],
        ['兇', 'キョウダンニタオレル ノ キョウ，ヒトアシ ノ キョウ'],
        ['兆', 'イッチョウエン ノ チョウ'],
        ['允', 'インカ ノ イン，ユルス'],
        ['兀', 'ゴツ，カンスウジノイチノシタニヒトアシ'],
        ['元', 'ゲンキ ノ ゲン，モト'],
        ['顱', 'ロ，ハゼノキノハゼノツクリノミギニページ'],
        ['匸', 'ケイ，ブシュノカクシガマエ'],
        ['顰', 'ヒンシュクヲカウノヒン'],
        ['繧', 'ウン，イトヘンニクモユキノクモ'],
        ['繦', 'スキ，キョウ，イトヘンニユミニイエドモノヒダリガワ'],
        ['繭', 'マユ，カイコ ノ マユ'],
        ['繪', 'カイガノカイ，エホンノエノキュウジ'],
        ['繩', 'ジョウモンドキノジョウノキュウジ'],
        ['繰', 'クル，クリカエシ ノ クリ'],
        ['繿', 'ラン，イトヘンニエイガカントクノカン'],
        ['繽', 'ヒン，イトヘンニライヒンノヒン'],
        ['繼', 'ナマチュウケイノケイノキュウジ'],
        ['繻', 'オリモノノシュスノシュ'],
        ['繹', 'エンエキホウノエキ'],
        ['繆', 'ビュウ，イトヘンニニカワノツクリ'],
        ['繃', 'ホウタイヲマクノホウ，イトヘンニクズレル'],
        ['繁', 'ハンエイスル ノ ハン，シゲル'],
        ['繍', 'シシュウスル ノ シュウ'],
        ['繋', 'ケイリュウスル ノ ケイ，ツナグ'],
        ['繊', 'カガクセンイ ノ セン'],
        ['繖', 'サン，イトヘンニサンポミチノサン'],
        ['繕', 'シュウゼンスル ノ ゼン，ツクロウ'],
        ['織', 'ソシキ ノ シキ'],
        ['繞', 'カンジブシュノニョウ，イトヘンニギョウシュンノギョウノキュウジ'],
        ['繝', 'ゲン，イトヘンニアイダノキュウジ'],
        ['繚', 'リョウランノリョウ，イトヘンニドウリョウノリョウノツクリ'],
        ['繙', 'ハン，ヒモトク'],
        ['駕', 'ガ，リョウガスル ノ ガ'],
        ['駐', 'チュウシャジョウ ノ チュウ'],
        ['駛', 'シ，ハセル，ウマヘンニレキシノシ'],
        ['駅', 'エキビル ノ エキ'],
        ['t', 'ティー タイム'],
        ['旬', 'ゲジュン ノ ジュン'],
        ['旭', 'キョクジツ ノ キョク，アサヒ'],
        ['鼕', 'トウ，ツヅミノシタニフユ'],
        ['旨', 'ロンシヲ ノベル ノ シ，ムネ'],
        ['早', 'ソウチョウ ノ ソウ，ハヤイ'],
        ['日', 'ニチヨウビ ノ ニチ，ヒ'],
        ['旦', 'ガンタン ノ タン'],
        ['旧', 'キュウレキ ノ キュウ'],
        ['无', 'ム，ナイ，テンキヨホウノテンノヨンカクメヲオツニスル'],
        ['旡', 'キ，ツマル，キトクケンノキ，スデニノツクリ'],
        ['既', 'キトクケン ノ キ，スデニ'],
        ['鼇', 'オオガメヲイミスルゴウ'],
        ['旺', 'ショクヨクオウセイナ ノ オウ'],
        ['旻', 'ビン，アキゾラ，ニチヨウビノニチノシタニサクブンノブン'],
        ['鼎', 'テイダン ノ テイ，カナエ'],
        ['旱', 'カンバツノカン，ヒデリ'],
        ['鼈', 'ベッコウアメノベツ'],
        ['旌', 'ハタジルシヲイミスルセイ'],
        ['族', 'スイゾクカン ノ ゾク'],
        ['旋', 'センカイスル ノ セン'],
        ['旄', 'ケデツクッタハタカザリノボウ'],
        ['旅', 'リョコウ ノ リョ，タビ'],
        ['旆', 'オオキナハタヲイミスルタイハイノハイ'],
        ['旁', 'ホウ，カンジノブシュノツクリ'],
        ['旃', 'セン，ホウヘンニセンダンノキノセンノツクリ'],
        ['旙', 'ハタガヒルガエルサマノハンノイタイジ'],
        ['鼠', 'ネズミ'],
        ['旛', 'ハタガヒルガエルサマノハン'],
        ['鼬', 'ドウブツノイタチ'],
        ['旗', 'コッキ ノ キ，ハタ'],
        ['旒', 'ハタヲカゾエルタンイノリュウ'],
        ['臾', 'シュユノマノユ'],
        ['臼', 'ダッキュウスル ノ キュウ，ウス'],
        ['臺', 'ダイドコロノダイノキュウジ'],
        ['臻', 'シン，ブシュノイタルニシンノシコウテイノシン'],
        ['致', 'イッチスル ノ チ，イタス'],
        ['至', 'シキュウ ノ シ，イタル'],
        ['臭', 'クサイ，アクシュウ'],
        ['自', 'ジテンシャ ノ ジ，ミズカラ'],
        ['臨', 'リンジ ノ リン'],
        ['臧', 'ゾウ，レイゾウコノゾウノイタイジカラクサカンムリヲトッタカタチ'],
        ['臥', 'フス，ガシンショウタン ノ ガ'],
        ['臣', 'ソウリダイジン ノ ジン'],
        ['臠', 'レン，コイビトノコイノキュウジノココロノカワリニニク'],
        ['駲', 'ウマヘンニキュウシュウチホウノシュウ'],
        ['臟', 'ゴゾウロップノゾウノイタイジ'],
        ['臚', 'ロ，ニクヅキニハゼノキノハゼノツクリ'],
        ['臘', 'ロウ，ニクヅキニリョウジュウノリョウノキュウジノツクリ'],
        ['臙', 'エン，ニクヅキニツバメ'],
        ['臓', 'シンゾウカンゾウ ノ ゾウ'],
        ['臑', 'ジュ，ニクヅキニジュヨウトキョウキュウノジュ'],
        ['臍', 'セイカタンデンノセイ，ヘソ'],
        ['臈', 'ロウ，ニクヅキニクズモチノクズ'],
        ['臉', 'セン，ニクヅキニマブタノツクリ'],
        ['臆', 'オクビョウナ ノ オク'],
        ['臂', 'ハチメンロッピノヒ，ヒジ'],
        ['臀', 'シリヲイミスルデンブノデン'],
        ['韮', 'ショクブツ ノ ニラ'],
        ['韭', 'ショクブツノニラノセイジ'],
        ['響', 'オンキョウ ノ キョウ，ヒビク'],
        ['韻', 'ヨインガノコル ノ イン'],
        ['韶', 'ショウ、オンガクノオンニマネクノツクリ'],
        ['韵', 'ヨインガノコルノインノイタイジ'],
        ['音', 'オト，オンガク ノ オン'],
        ['韲', 'ナマスヲイミスルセイノイタイジ'],
        ['韋', 'イダテンノイ'],
        ['韈', 'ベツ，カワヘンニケイベツスルノベツ'],
        ['韆', 'セン，カワヘンニサセンスルノセン'],
        ['韃', 'ダッタンジンノタツ'],
        ['韜', 'リクトウサンリャクノトウ'],
        ['韓', 'カンコクジン ノ カン'],
        ['返', 'ヘンジ ノ ヘン，カエス'],
        ['近', 'キンジョ ノ キン，チカイ'],
        ['J', 'ジェイ ジャパン'],
        ['迚', 'トテモ、シンニョウニチュウガクノチュウ'],
        ['迄', 'コレマデ ノ マデ'],
        ['迅', 'ジンソクナ ノ ジン'],
        ['迂', 'ウカイスル ノ ウ'],
        ['駮', 'ハク，ウマヘンニマジワル'],
        ['迎', 'カンゲイスル ノ ゲイ，ムカエル'],
        ['迷', 'メイワク ノ メイ，マヨウ'],
        ['迴', 'カイ、シンニョウニカイスウケンノカイ'],
        ['述', 'ジュツゴ ノ ジュツ，ノベル'],
        ['追', 'ツイセキスル ノ ツイ，オウ'],
        ['迺', 'ナイ，シンニョウニニシヒガシノニシ'],
        ['迸', 'ホトバシル，ホウ'],
        ['迹', 'セキ，シンニョウニソクセキヲノコスノセキノツクリ'],
        ['迦', 'オシャカサマ ノ カ'],
        ['迥', 'ケイ，シンニョウニケイガンノシノケイノツクリ'],
        ['迢', 'チョウ、シンニョウニチョウノウリョクノチョウノツクリ'],
        ['迯', 'ニゲル，トウボウスルノトウノイタイジ'],
        ['>', 'ダイナリ'],
        ['迪', 'テキ，シンニョウニリユウノユウ'],
        ['迫', 'ハクリョク ノ ハク，セマル'],
        ['迩', 'シンニョウ ニ ナンジ ノ イタイジ ノ ニ'],
        ['堯', 'ギョウシュンノギョウノキュウジ'],
        ['堪', 'カンニンブクロ ノ カン，タエル'],
        ['堤', 'テイボウ ノ テイ，ツツミ'],
        ['堡', 'キョウトウホのホ，ホケンシツノホノシタニツチ'],
        ['堽', 'コウ，ツチヘンニアミガシラニタダシイ'],
        ['堺', 'オオサカフサカイシ ノ サカイ'],
        ['場', 'バショ ノ バ，ジョウ'],
        ['堵', 'アンドスル ノ ト'],
        ['堰', 'エンテイ ノ エン，セキ'],
        ['報', 'ホウドウ ノ ホウ'],
        ['堊', 'アク，アネッタイノアノキュウジタイノシタニツチ'],
        ['堋', 'ホウ，ツチヘンニホウユウノホウ'],
        ['堆', 'タイセキブツ ノ タイ'],
        ['堅', 'ケンジツナ ノ ケン，カタイ'],
        ['堂', 'コッカイギジドウ ノ ドウ'],
        ['堀', 'ホリバタ ノ ホリ'],
        ['堝', 'カ，ツチヘンニナベノツクリ'],
        ['堙', 'イン，ツチヘンニエントツノエンノツクリ'],
        ['堕', 'ダラクスル ノ ダ'],
        ['眸', 'メイボウコウシノボウ'],
        ['眺', 'ナガメル，チョウボウ ノ チョウ'],
        ['眼', 'ソウガンキョウ ノ ガン'],
        ['眷', 'サイシケンゾクノケン'],
        ['眩', 'ゲンワクスルノゲン，マブシイ'],
        ['眠', 'ネムル，スイミン ノ ミン'],
        ['眤', 'ジツ，メヘンニアマデラノアマ'],
        ['眥', 'シ，ヒガンシガンノシノシタニモクテキノモク'],
        ['眦', 'シ，メヘンニヒガンシガンノシ'],
        ['眛', 'マイ，メヘンニミライノミ'],
        ['眞', 'マゴコロノマノキュウジ'],
        ['真', 'マゴコロ ノ マ，シン'],
        ['眈', 'コシタンタンノタン'],
        ['眉', 'マユ，ハクビ ノ ビ'],
        ['看', 'カンビョウ ノ カン'],
        ['県', 'トドウフケン ノ ケン'],
        ['省', 'ショウリャクスル ノ ショウ，ハブク'],
        ['眄', 'ウコサベンノベン'],
        ['眇', 'タメツスガメツノスガメ'],
        ['僅', 'ワズカ，キンサ ノ キン'],
        ['僂', 'ロウ，ニンベンニロウカクノロウノキュウジノツクリ'],
        ['像', 'ブツゾウ ノ ゾウ'],
        ['働', 'ロウドウ ノ ドウ，ハタラク'],
        ['僊', 'セン，ニンベンニサセンスルノセンノツクリ'],
        ['僉', 'シケンヲウケルノケンノキュウジノツクリ'],
        ['僖', 'キ，ニンベンニキゲキノキ，ヨロコブ'],
        ['僕', 'シモベ，コウボク ノ ボク'],
        ['僑', 'チュウゴク ノ カキョウ ノ キョウ'],
        ['僞', 'ギゾウノギ，ニセノキュウジ'],
        ['僚', 'カイシャ ノ ドウリョウ ノ リョウ'],
        ['僧', 'ソウリョ ノ ソウ'],
        ['僥', 'ギョウコウノギョウ，ニンベンニギョウシュンギョウノキュウジ'],
        ['僣', 'センエツナガラノセンノイタイジ'],
        ['僮', 'ドウ，ニンベンニワラベ'],
        ['僭', 'センエツナガラノセン'],
        ['僵', 'キョウ，ニンベンニカシハラジングウノカシノツクリ'],
        ['僻', 'ヘキチ ノ ヘキ'],
        ['價', 'カチノカ，アタイノキュウジ'],
        ['魏', 'ギシワジンデンノギ'],
        ['渝', 'ユ，サンズイニニレノキノニレノツクリ'],
        ['渟', 'テイ，サンズイニテイシュカンパクノテイ'],
        ['渙', 'タイショウカンパツノカン，サンズイニカンキセンノカンノツクリ'],
        ['減', 'ゲンショウスル ノ ゲン，ヘル'],
        ['渚', 'ナギサ，ショ'],
        ['渕', 'フチノイタイジ，サンズイニハナガサクノサクノツクリニリットウ'],
        ['渓', 'ケイリュウヅリ ノ ケイ'],
        ['渉', 'ダンタイコウショウ ノ ショウ'],
        ['済', 'キュウサイスル ノ サイ，スマス'],
        ['渋', 'ジュウタイスル ノ ジュウ，シブイ'],
        ['渊', 'フチノイタイジ，タテボウニホンノナカニコメ'],
        ['清', 'セイケツ ノ セイ，キヨイ'],
        ['渇', 'カツボウスル ノ カツ，カワク'],
        ['渾', 'コンシンノチカラノコン，サンズイニグンタイノグン'],
        ['游', 'ユウ，アソブノシンニョウノカワリニサンズイ'],
        ['渺', 'ビョウ，サンズイニモクテキノモクニスクナイ'],
        ['渭', 'イスイノイ，サンズイニイブクロノイ'],
        ['測', 'ソクテイ ノ ソク，ハカル'],
        ['港', 'クウコウ ノ コウ，ミナト'],
        ['渮', 'カ，サンズイニイラツク'],
        ['温', 'オンセン ノ オン，アタタマル'],
        ['渫', 'シュンセツスルノセツ'],
        ['渥', 'アツミハントウ ノ アツ'],
        ['渤', 'ボッカイワンノボツ'],
        ['渦', 'ウズマキ ノ ウズ，カ'],
        ['渡', 'ハシヲワタル ノ ワタル'],
        ['渠', 'アンキョ ノ キョ，ミゾ'],
        ['渣', 'サ，サンズイニケンサノサ'],
        ['蜩', 'ヒグラシ、ムシヘンニシュウヘンノシュウ'],
        ['蜥', 'セキ，ムシヘンニブンセキスルノセキ'],
        ['蜿', 'エン、ムシヘンニアテナノアテ'],
        ['蜻', 'トンボノイチモジメ，セイ、ムシヘンニアオノキュウジ'],
        ['蜷', 'ニナ，ムシヘンニノリマキノマキノキュウジ'],
        ['蜴', 'エキ、ムシヘンニボウエキフウノエキ'],
        ['a', 'エイ アニマル'],
        ['蜍', 'ジョ、ムシヘンニアマリ'],
        ['蜊', 'カイノアサリ、ムシヘンニケンリノリ'],
        ['蜉', 'フ，ムシヘンニツメノシタニコドモノコ'],
        ['蜈', 'ゴ，ムシヘンニゴフクヤノゴノキュウジケイ'],
        ['蜆', 'シジミ，ケン'],
        ['蜃', 'シンキロウノシン'],
        ['蜂', 'ハチミツ ノ ハチ，ホウ'],
        ['蜀', 'サンゴクシノショク'],
        ['蜜', 'ハチミツ ノ ミツ'],
        ['蜚', 'ゴキブリノイチモジメ，ヒジョウグチノヒノシタニムシ'],
        ['蜘', 'クモ，ムシヘン ニ チシキ ノ チ'],
        ['蜒', 'エン，ムシヘンニエンチョウスルノエン'],
        ['蜑', 'タン，エンチョウスルノエンノシタニムシ'],
        ['魔', 'マホウ ノ マ'],
        ['敷', 'シク，ザシキ ノ シキ'],
        ['整', 'セイリスル ノ セイ，トトノエル'],
        ['敵', 'テキミカタ ノ テキ'],
        ['敲', 'スイコウヲカサネルノコウ，タタク'],
        ['数', 'サンスウ ノ スウ，カズ'],
        ['數', 'サンスウノスウノキュウジ'],
        ['敦', 'チュウゴク ノ トンコウ ノ トン'],
        ['敢', 'ユウカンナ ノ カン，アエテ'],
        ['散', 'サンポ ノ サン，チル'],
        ['敬', 'ソンケイ ノ ケイ，ウヤマウ'],
        ['敖', 'ゴウ，ゴウマンナノゴウノツクリ'],
        ['敗', 'ハイボク ノ ハイ，ヤブレル'],
        ['敕', 'チョクメイノチョクノキュウジ'],
        ['救', 'キュウジョ ノ キュウ，スクウ'],
        ['敞', 'カイグンコウショウノショウタレナシ'],
        ['敝', 'ヘイイハボウノヘイ，ニジュウアシナシ'],
        ['敘', 'ジジョデンノジョノイタイジ'],
        ['教', 'キョウカショ ノ キョウ，オシエル'],
        ['故', 'コキョウ ノ コ，ユエ'],
        ['敏', 'ビンカンナ ノ ビン'],
        ['敍', 'ジジョデンノジョノキュウジ'],
        ['效', 'コウカテキナノコウノキュウジ'],
        ['7', 'ナナ'],
        ['徳', 'ドウトク ノ トク'],
        ['徴', 'トクチョウテキナ ノ チョウ'],
        ['徹', 'テツヤスル ノ テツ'],
        ['徼', 'キョウ，ギョウニンベンニゲキレイスルノゲキノツクリ'],
        ['徽', 'キショウヲツケル ノ キ，シルシ'],
        ['徠', 'ジュガクシャノオギュウソライノライ'],
        ['御', 'オンチュウ ノ オン，ゴハン ノ ゴ'],
        ['徨', 'ホウコウスルノコウ，サマヨウ'],
        ['復', 'オウフク ノ フク'],
        ['循', 'アクジュンカン ノ ジュン'],
        ['徭', 'ヨウ，ギョウニンベンニユレルノキュウジノツクリ'],
        ['微', 'ビセイブツ ノ ビ'],
        ['徐', 'ジョコウスル ノ ジョ'],
        ['徑', 'エンノハンケイノケイノキュウジ'],
        ['徒', 'トホ ノ ト'],
        ['従', 'ジュウジスル ノ ジュウ，シタガウ'],
        ['得', 'トクイ ノ トク，エル'],
        ['徘', 'ハイカイスルノハイ，サマヨウ'],
        ['徙', 'シ，ギョウニンベンニトメルニハシルノシタガワ'],
        ['從', 'ジュウジスルノジュウ，シタガウノキュウジ'],
        ['往', 'オウフク ノ オウ'],
        ['征', 'エンセイグン ノ セイ'],
        ['徂', 'ジュガクシャノオギュウソライノソ'],
        ['徃', 'オウフクスルノオウノイタイジ'],
        ['径', 'エン ノ ハンケイ ノ ケイ'],
        ['待', 'キタイスル ノ タイ，マツ'],
        ['徇', 'ジュン，ギョウニンベンニゲジュンノジュン'],
        ['很', 'コン，ギョウニンベンニダイコンノコンノツクリ'],
        ['徊', 'ハイカイスルノカイ，ギョウニンベンニマワル'],
        ['律', 'ホウリツ ノ リツ'],
        ['後', 'ウシロ，ゼンゴ ノ ゴ'],
        ['ｫ', 'チイサイ オオサカ ノ オ'],
        ['≒', '二アリーイコール'],
        ['≪', 'ヒジョウニチイサイ'],
        ['≫', 'ヒジョウニオオキイ'],
        ['≦', 'ショウナリイコール'],
        ['≧', 'ダイナリイコール'],
        ['≠', 'ノットイコール'],
        ['≡', 'ゴウドウ'],
        ['ω', 'ギリシャ オメガ'],
        ['ψ', 'ギリシャ プサイ'],
        ['χ', 'ギリシャ カイ'],
        ['φ', 'ギリシャ ファイ'],
        ['υ', 'ギリシャ ウプシロン'],
        ['τ', 'ギリシャ タウ'],
        ['σ', 'ギリシャ シグマ'],
        ['ρ', 'ギリシャ ロー'],
        ['π', 'ギリシャ パイ'],
        ['x', 'エックス エックスセン'],
        ['靱', 'キョウジンナノジンノイタイジ'],
        ['靴', 'クツシタ ノ クツ，カ'],
        ['靹', 'ドウ，カワヘンニコクナイノナイ'],
        ['瞋', 'シン，メヘンニマゴコロノマノキュウジ'],
        ['靺', 'マツ，カワヘンニネンマツノマツ'],
        ['瞎', 'カツ，メヘンニサイガイノガイノキュウジタイ'],
        ['靼', 'ダッタンジンノタン'],
        ['靡', 'フウビスルノビ，ナビク'],
        ['靠', 'コウ，ツゲルノシタニヒジョウグチノヒ'],
        ['瞑', 'メイモクスルノメイ'],
        ['靤', 'ホウ，ジメンノメンノミギニツツム'],
        ['靦', 'テン、ジメンノメンノミギニケンブツスルノケン'],
        ['革', 'カクメイ ノ カク'],
        ['靨', 'エクボ、エンセイカンノエンノシタニジメンノメン'],
        ['靫', 'ウツボカズラノウツボ'],
        ['瞞', 'ギマンニミチタノマン'],
        ['瞠', 'ドウ，メヘンニドウモクスルノドウ'],
        ['青', 'アオイ，セイシュン ノ セイ'],
        ['瞥', 'イチベツスル ノ ベツ'],
        ['静', 'レイセイ ノ セイ，シズカ'],
        ['靜', 'レイセイノセイ、シズカノキュウジ'],
        ['瞬', 'シュンカン ノ シュン，マタタク'],
        ['瞭', 'メイリョウナ ノ リョウ'],
        ['瞳', 'ヒトミ，ドウコウ ノ ドウ'],
        ['瞰', 'チョウカンズノカン，ミオロス'],
        ['靂', 'セイテンノヘキレキノレキ'],
        ['瞶', 'キ，メヘンニキチョウヒンノキ'],
        ['靄', 'アサモヤノモヤ'],
        ['靆', 'タイ、クモノミギニタイホジョウノタイ'],
        ['靉', 'アイ、クモノミギニアイスルノアイ'],
        ['瞻', 'セン，メヘンニタンニンノタンノキュウジノツクリ'],
        ['瞹', 'アイ，メヘンニアイスルノアイ'],
        ['瞿', 'キョウクスルノクノツクリ，メガフタツノシタニフルトリ'],
        ['瞼', 'マブタ，ガンケンノケン'],
        ['瞽', 'コ，ツヅミノシタニモクテキノモク'],
        ['豚', 'ブタ，トンジル ノ トン'],
        ['豐', 'ユタカナノキュウジ'],
        ['豕', 'シ，ブシュノイノコ'],
        ['豈', 'アニハカランヤノアニ'],
        ['豊', 'ホウサク ノ ホウ，ユタカ'],
        ['豌', 'エンドウマメノエン'],
        ['豎', 'ジュ、ケンジツナノケンノツチノカワリニマメ'],
        ['豁', 'カツゼントサトルノカツ'],
        ['豆', 'トウフ ノ トウ，マメ'],
        ['豸', 'チ，ブシュノムジナ'],
        ['豹', 'ドウブツ ノ ヒョウ'],
        ['豺', 'サイロウノムレノサイ、ヤマイヌ'],
        ['豼', 'ヒ，ムジナヘンニヒカクスルノヒ'],
        ['豪', 'ゴウカイナ ノ ゴウ'],
        ['豫', 'ヨテイノヨノキュウジ'],
        ['豬', 'チョ、イノコノミギニカガクシャノシャノキュウジタイ'],
        ['象', 'インショウテキ ノ ショウ，ゾウ'],
        ['豢', 'コブシノテノカワリニイノコ'],
        ['澤', 'コウタクノタク，サワノキュウジ'],
        ['澣', 'カン，サンズイニシンカンセンノカン'],
        ['澡', 'ソウ，サンズイニソウジュウスルノソウノツクリ'],
        ['澪', 'ミオツクシノミオ，レイ'],
        ['澳', 'オウ，サンズイニオクバノオクノキュウジ'],
        ['澱', 'チンデンスル ノ デン，ヨドム'],
        ['澹', 'アンタンタルクモユキノタン'],
        ['澆', 'ギョウ，サンズイニギョウシュンノギョウノキュウジ'],
        ['澄', 'ミズガスム ノ スム'],
        ['澂', 'ミズガスムノスムノイタイジ'],
        ['澁', 'ジュウタイスルノジュウ，シブイノキュウジ'],
        ['澀', 'ジュウタイスルノジュウ，シブイノノキュウジノイタイジ'],
        ['澎', 'ホウ，サンズイニツヅミノヒダリガワニサンヅクリ'],
        ['澗', 'カン，サンズイ ニ アイダ'],
        ['澑', 'リュウインガサガルノリュウ，タマリノイタイジ'],
        ['伺', 'シンタイウカガイ ノ ウカガイ'],
        ['螳', 'カマキリヲイミスルトウロウノトウ，ムシヘンニコウカイドウノドウ'],
        ['螻', 'ロウ，ムシノケラノイチモジメ'],
        ['螺', 'ラセン ノ ラ'],
        ['螽', 'シュウ，キセツノフユノシタニムシフタツ'],
        ['N', 'エヌ ノベンバー'],
        ['螢', 'ホタル，ケイコウトウノケイノキュウジ'],
        ['螫', 'セキ，オンシャヲウアタエルノシャノシタニムシ'],
        ['螯', 'ゴウ，ゴウマンナノゴウノツクリノシタニムシ'],
        ['螟', 'メイ，ムシヘンニメイフクヲイノルノメイ'],
        ['螂', 'カマキリヲイミスルトウロウノロウ，ムシヘンニモモタロウノロウ'],
        ['融', 'キンユウ ノ ユウ'],
        ['狹', 'キョウ，セマイノキュウジ'],
        ['伯', 'ハクシャク ノ ハク'],
        ['佰', 'ハク，ニンベンニカンスウジノヒャク'],
        ['偈', 'ゲ，ニンベンニカッショクノカツノツクリ'],
        ['佳', 'カサク ノ カ，ニンベン ニ ツチ フタツ ノ カ'],
        ['併', 'ヘイヨウスル ノ ヘイ，アワセル'],
        ['佶', 'キツ，ニンベンニダイキチノキチ'],
        ['偃', 'エン，ニンベンニエンテイノエン，セキノツクリ'],
        ['佻', 'ケイチョウフハクノチョウ'],
        ['佼', 'ニンベン ニ マジワル ノ コウ'],
        ['假', 'カメンノカ，カリノキュウジ'],
        ['使', 'シヨウスル ノ シ，ツカウ'],
        ['做', 'サ，ニンベンニコキョウノコ'],
        ['停', 'テイシャ ノ テイ'],
        ['佩', 'ハイヨウスルノハイ，オビル'],
        ['偕', 'カイ，トモニ，ニンベンニミナサマノミナ'],
        ['佯', 'ヨウ，ニンベンニドウブツノヒツジ'],
        ['佐', 'ホサスル ノ サ'],
        ['佑', 'テンユウ ノ ユウ，ニンベン ニ ミギ'],
        ['体', 'カラダ，タイイク ノ タイ'],
        ['何', 'ナニモノ ノ ナニ'],
        ['佗', 'タ，ワビ，ニンベンニウカンムリニカタカナノヒ'],
        ['余', 'ヨユウ ノ ヨ，アマル'],
        ['佚', 'アンイツノイツ，ニンベンニウシナウ'],
        ['佛', 'ブツゾウノブツノキュウジ'],
        ['健', 'ケンコウ ノ ケン，スコヤカ'],
        ['佝', 'ク，ニンベンニハイクノク'],
        ['佞', 'カンネイノネイ'],
        ['偸', 'トウ，ニンベンニユカイナノユノツクリ'],
        ['佃', 'ツクダニ ノ ツクダ'],
        ['偽', 'ギゾウ ノ ギ，ニセ'],
        ['但', 'タダシガキ ノ タダシ'],
        ['佇', 'チョリツスルノチョ，タタズム'],
        ['偲', 'コジンヲ シノブ ノ シノブ'],
        ['偵', 'テイサツスル ノ テイ'],
        ['位', 'タンイ ノ イ'],
        ['低', 'ヒクイ，テイ'],
        ['住', 'ジュウショ ノ ジュウ，スム'],
        ['$', 'ドル'],
        ['祷', 'モクトウスル ノ トウ，イノル'],
        ['祺', 'キ，シメスヘンニソレ'],
        ['祿', 'ゲンロクブンカノロクノキュウジ'],
        ['祢', 'ネ，シメスヘン ニ ナンジ ノ イタイジ'],
        ['祠', 'シ，ホコラ，シメスヘンニシカイシャノシ'],
        ['祥', 'フショウジ ノ ショウ'],
        ['票', 'トウヒョウスル ノ ヒョウ'],
        ['傾', 'カタムク，ケイコウ ノ ケイ'],
        ['祭', 'サイジツ ノ サイ，マツリ'],
        ['祓', 'ハライキヨメルノハライ，フツ'],
        ['祐', 'テンユウ ノ ユウ，シメスヘン ニ ミギ'],
        ['祗', 'シ，シメスヘンニヒクイノツクリ'],
        ['祖', 'ソセン ノ ソ'],
        ['祕', 'ヒミツノヒノキュウジ'],
        ['祚', 'クライヲウケツグセンソノソ'],
        ['祟', 'タタル，スイ'],
        ['神', 'カミサマ ノ カミ，シン'],
        ['祝', 'シュクジツ ノ シュク，イワウ'],
        ['祁', 'シメスヘン ニ オオザト ノ キ'],
        ['祀', 'サイシノシ，マツル'],
        ['祇', 'ギオンマツリ ノ ギ'],
        ['祉', 'シャカイフクシ ノ シ'],
        ['祈', 'イノル，キガンスル ノ キ'],
        ['齡', 'ネンレイノレイノキュウジ'],
        ['齢', 'ネンレイ ノ レイ'],
        ['書', 'キョウカショ ノ ショ，カク'],
        ['曹', 'ソウトウシュウ ノ ソウ'],
        ['鰡', 'ウオヘンニリュウガクノリュウ，ボラ'],
        ['曼', 'マンダラノマン'],
        ['曽', 'キソジ ノ ソ'],
        ['曾', 'キソジ ノ ソ ノ キュウジ'],
        ['替', 'フリカエコウザ ノ カエル'],
        ['曰', 'イワクツキノイワク'],
        ['曲', 'キョクセン ノ キョク，マガル'],
        ['曳', 'エイコウスル ノ エイ，ヒク'],
        ['更', 'ヘンコウ ノ コウ，サラニ'],
        ['曵', 'フネヲエイコウスルノエイノイタイジ'],
        ['曷', 'カツ、カッショクノカツノツクリ'],
        ['鰲', 'ゴウ，ゴウマンノゴウノツクリノシタニサカナ'],
        ['曩', 'ノウ，ヨウビノニチノシタニユズルノキュウジノツクリ'],
        ['鰰', 'サカナノハタハタ，ウオヘンニカミサマノカミ'],
        ['曠', 'コウヤノコウ，ニチヘンニヒロイノキュウジ'],
        ['鰻', 'ウナギ'],
        ['鰹', 'カツオ，ウオヘン ニ カタイ'],
        ['鰾', 'ウキブクロ，ウオヘンニトウヒョウスルノヒョウ'],
        ['曦', 'ギ，ウツクシイタイヨウヲイミスルニチヘンノギ'],
        ['曙', 'アケボノ，ショ'],
        ['曚', 'ボウ，ニチヘンニケイモウスルノモウ'],
        ['曜', 'ニチヨウビ ノ ヨウ'],
        ['曝', 'サラス，ニチヘン ニ ボウリョク ノ ボウ'],
        ['鰄', 'イ，ウオヘンニイゲンガアルノイ'],
        ['鰊', 'サカナノニシン，レン'],
        ['鰈', 'サカナノカレイ'],
        ['鰉', 'コウ，ウオヘンニコウタイシノコウ'],
        ['銃', 'ケンジュウウ ノ ジュウ'],
        ['曖', 'アイマイナノアイ'],
        ['鰍', 'カジカ，ウオヘン ニ アキ'],
        ['鰒', 'フグ，ウオヘンニハラノツクリ'],
        ['曉', 'アカツキノキュウジ'],
        ['鰐', 'ドウブツ ノ ワニ'],
        ['鰔', 'カン，ウオヘンニハリキュウノハリノツクリ'],
        ['鰕', 'ウオヘンノエビ，カ'],
        ['曁', 'キ，キトクケンノキノシタニガンタンノタン'],
        ['曄', 'ヨウ，ニチヘンニチュウカナベノカ'],
        ['曇', 'クモリ，ドンテン ノ ドン'],
        ['０', 'ゼロ'],
        ['１', 'イチ'],
        ['２', 'ニ'],
        ['３', 'サン'],
        ['４', 'ヨン'],
        ['５', 'ゴ'],
        ['６', 'ロク'],
        ['７', 'ナナ'],
        ['８', 'ハチ'],
        ['９', 'キュウ'],
        ['：', 'コロン'],
        ['；', 'セミコロン'],
        ['＜', 'ショウナリ'],
        ['＝', 'イコール'],
        ['＞', 'ダイナリ'],
        ['？', 'クエスチョン'],
        ['！', 'カンタンフ'],
        ['＃', 'シャープ'],
        ['＄', 'ドル'],
        ['％', 'パーセント'],
        ['＆', 'アンド'],
        ['（', 'カッコ'],
        ['）', 'トジカッコ'],
        ['＊', 'アスタリスク'],
        ['＋', 'プラス'],
        ['，', 'コンマ'],
        ['－', 'マイナス'],
        ['．', 'ピリオド'],
        ['／', 'スラッシュ'],
        ['Ｐ', 'ピー パパ'],
        ['Ｑ', 'キュー クエスチョン'],
        ['Ｒ', 'アール ルーム'],
        ['Ｓ', 'エス スポーツ'],
        ['e', 'イー エッグ'],
        ['Ｕ', 'ユー ユージュアリー'],
        ['Ｖ', 'ブイ ビデオ'],
        ['Ｗ', 'ダブリュー ウィンドウ'],
        ['Ｘ', 'エックス エックスセン'],
        ['Ｙ', 'ワイ ヤング'],
        ['Ｚ', 'ゼット ズー'],
        ['［', 'カクカッコ'],
        ['＼', 'バックスラッシュ'],
        ['］', 'トジカクカッコ'],
        ['＾', 'ベキジョウ'],
        ['＿', 'アンダーライン'],
        ['＠', 'アットマーク'],
        ['Ａ', 'エイ アニマル'],
        ['Ｂ', 'ビー ボーイ'],
        ['Ｃ', 'シー キャット'],
        ['Ｄ', 'ディー デスク'],
        ['Ｅ', 'イー エッグ'],
        ['Ｆ', 'エフ フレンド'],
        ['Ｇ', 'ジー ゴルフ'],
        ['Ｈ', 'エイチ ホテル'],
        ['Ｉ', 'アイ インク'],
        ['Ｊ', 'ジェイ ジャパン'],
        ['Ｋ', 'ケイ キッチン'],
        ['Ｌ', 'エル ラブ'],
        ['Ｍ', 'エム マイク'],
        ['Ｎ', 'エヌ ノベンバー'],
        ['Ｏ', 'オー オープン'],
        ['瘁', 'スイ，ヤマイダレニソツギョウノソツ'],
        ['瘍', 'イカイヨウノヨウ'],
        ['瘉', 'ユ，ヤマイダレニニレノキノツクリ'],
        ['瘋', 'フウテンノフウ，ヤマイダレニカゼ'],
        ['瘟', 'オン，ヤマイダレニオンセンノオンノキュウジノツクリ'],
        ['瘤', 'ドウミャクリュウノリュウ，コブ'],
        ['瘧', 'オコリ，ヤマイダレニギャクタイノギャク'],
        ['瘠', 'セキ，ヤマイダレニセキズイノセキ'],
        ['瘡', 'ミズボウソウノソウ'],
        ['瘢', 'ハンコンノハン，ヤマイダレニイッパンテキノハン'],
        ['瘴', 'ショウ，ヤマイダレニブンショウヲカクノショウ'],
        ['瘰', 'ルイレキノルイ，ヤマイダレニルイセキスルノルイ'],
        ['瘻', 'ジロウノロウ，ヤマイダレニカズノキュウジノヒダリガワ'],
        ['賢', 'ケンジャ ノ ケン，カシコイ'],
        ['賣', 'ショウバイノバイ、ウルノキュウジ'],
        ['賠', 'バイショウキン ノ バイ'],
        ['賦', 'ゲップバライ ノ フ'],
        ['賤', 'ゲセンナノセン，イヤシイ'],
        ['質', 'シツモン ノ シツ'],
        ['賭', 'トバク ノ ト，カケル'],
        ['賺', 'ナダメスカスノスカス'],
        ['賻', 'フ、カイヘンニハクブツカンノハクノツクリ'],
        ['購', 'コウニュウスル ノ コウ'],
        ['槙', 'ショクブツ ノ マキ，キヘン ニ マゴコロ ノ マ'],
        ['賂', 'ワイロ ノ ロ'],
        ['賃', 'チンギン ノ チン'],
        ['賀', 'ネンガジョウ ノ ガ'],
        ['賁', 'ヒ，フンカスルノフンノツクリ'],
        ['資', 'シゲン ノ シ'],
        ['賄', 'ワイロ ノ ワイ，マカナウ'],
        ['賊', 'サンゾク ノ ゾク'],
        ['賈', 'コ、オオイガシラニカイガラノカイ'],
        ['賎', 'イヤシイ，セン'],
        ['賍', 'ゾウブツザイノゾウノタイジ'],
        ['賓', 'ライヒン ノ ヒン'],
        ['賑', 'ニギワウ，インシン ノ シン'],
        ['賚', 'ライ，ライネンノライノキュウジノシタニカイガラノカイ'],
        ['賛', 'サンセイスル ノ サン'],
        ['賞', 'ノーベルショウ ノ ショウ'],
        ['賜', 'オンシ ノ シ，タマワル'],
        ['弸', 'ホウ，ユミヘンニホウユウヲイミスルホウ'],
        ['弾', 'ダンリョク ノ ダン，ハズム'],
        ['弼', 'ホヒツ ノ ヒツ，スケ'],
        ['弱', 'キョウジャク ノ ジャク，ヨワイ'],
        ['強', 'ベンキョウ ノ キョウ，ツヨイ'],
        ['張', 'キンチョウスル ノ チョウ，ハル'],
        ['弩', 'チョウドキュウノド，ヤッコノシタニユミ'],
        ['弯', 'ワンガンノワンノツクリ'],
        ['弭', 'ビ，ユミヘンニミミ'],
        ['弦', 'ゲンガッキ ノ ゲン，ツル'],
        ['弧', 'コヲエガク ノ コ'],
        ['弥', 'アミダブツ ノ ミ'],
        ['弛', 'ユルム，シカンスル ノ シ'],
        ['弘', 'コウボウタイシ ノ コウ，ヒロイ'],
        ['弟', 'アニオトオト ノ オトオト'],
        ['弓', 'ユミヤ ノ ユミ'],
        ['弐', 'ケイヤクショニ ツカウ スウジ ノ ニ'],
        ['弑', 'シギャクスルノシ，コロスノヒダリガワニニュウガクシキノシキ'],
        ['弖', 'テニヲハノテ，ユミノシタニカンスウジノイチ'],
        ['弗', 'フッソ ノ フツ，ドル'],
        ['弔', 'トムラウ，チョウジ ノ チョウ'],
        ['引', 'インヨウスル ノ イン，ヒク'],
        ['弊', 'ヘイガイ ノ ヘイ'],
        ['弋', 'ヨク，ブシュノシキガマエ'],
        ['弉', 'サンゾウホウシゲンジョウノジョウノイタイジ'],
        ['式', 'ニュウガクシキ ノ シキ'],
        ['弌', 'イチ，カンスウジノイチノイタイジ，シキガマエニイチ'],
        ['弍', 'カンスウジノニノイタイジ，シキガマエニニ'],
        ['弃', 'ハイキスルノキ，ステルノイタイジ'],
        [';', 'セミコロン'],
        ['弁', 'ベントウ ノ ベン'],
        ['弄', 'ホンロウスル ノ ロウ，モテアソブ'],
        ['◆', 'クロヒシガタ'],
        ['◇', 'ヒシガタ'],
        ['○', 'マルジルシ'],
        ['◎', 'ニジュウマル'],
        ['●', 'クロマル'],
        ['黐', 'トリモチノモチ，キビヘンニハナレルノツクリ'],
        ['◯', 'オオキナマル'],
        ['騾', 'ドウブツノラバノラ'],
        ['點', 'テンスウノテンノキュウジ'],
        ['埓', 'ラチガアカナイノラチノイタイジ'],
        ['埒', 'ラチガアカナイノラチ'],
        ['俸', 'ホウキュウ ノ ホウ'],
        ['俾', 'ヘイ，ニンベンニヒクツナノヒ'],
        ['埖', 'ゴミ，ツチヘンニハナビノハナ'],
        ['埔', 'ホ，ツチヘンニシジンノトホノホ'],
        ['俳', 'ハイク ノ ハイ'],
        ['域', 'チイキ ノ イキ'],
        ['埜', 'ヤ，ハヤシ ノ シタニ ツチ'],
        ['埃', 'ホコリマミレノホコリ，ジンアイノアイ'],
        ['埀', 'スイチョクノスイ，タレルノイタイジ'],
        ['修', 'シュウリスル ノ シュウ'],
        ['埆', 'カク，ツチヘンニツノ'],
        ['埋', 'ウメル，マイボツスル ノ マイ'],
        ['俣', 'ミナマタビョウ ノ マタ'],
        ['信', 'シンゴウキ ノ シン'],
        ['城', 'ジョウカマチ ノ ジョウ，シロ'],
        ['俤', 'オモカゲ，ニンベンニオトウト'],
        ['俥', 'クルマ，ニンベンニデンシャノシャ'],
        ['俚', 'リゲンノリ，ニンベンニサトガエリノサト'],
        ['俛', 'ベン，ニンベンニメンキョノメン'],
        ['俘', 'トリコ，フリョノフ'],
        ['執', 'シツジ ノ シツ，トル'],
        ['俟', 'シ，マツ，ニンベンニカタカナノムノシタニユミヤノヤ'],
        ['埴', 'ハニワ ノ ハニ'],
        ['基', 'キホン ノ キ，モトヅク'],
        ['培', 'サイバイスル ノ バイ，ツチカウ'],
        ['俑', 'ヘイバヨウノヨウ'],
        ['俗', 'フウゾク ノ ゾク'],
        ['俔', 'ケン，ニンベンニケンブツスルノケン'],
        ['埼', 'サイタマケン ノ イチモジメ ノ サキ'],
        ['埣', 'サイ，ツチヘンニソツギョウノソツ'],
        ['埠', 'ミナト ノ フトウ ノ フ'],
        ['俎', 'マナイタ，ソジョウノソ'],
        ['係', 'カンケイ ノ ケイ，カカリ'],
        ['促', 'サイソクスル ノ ソク，ウナガス'],
        ['俄', 'ニワカ，ガゼン ノ ガ'],
        ['黨', 'ヨトウヤトウノトウノキュウジ'],
        ['鱚', 'サカナノキス，ウオヘンニヨロコブ'],
        ['黻', 'ヒザカケヲイミスルフツ'],
        ['鱒', 'マス，サカナヘン ニ ソンケイスル ノ ソン'],
        ['黹', 'チ，ブシュノフツ'],
        ['迭', 'コウテツスル ノ テツ'],
        ['黽', 'ビン，ナワノキュウジノツクリ'],
        ['鱈', 'タラ，サカナヘン ニ ユキ'],
        ['銹', 'シュウ、カネヘンニヒイデル'],
        ['騁', 'テイ，ハセル，ウマヘンニショウヘイスルノヘイノツクリ'],
        ['騅', 'スイ，ウマヘンニフルトリ'],
        ['|', 'タテボウ'],
        ['漫', 'マンガ ノ マン'],
        ['漬', 'ツケモノ ノ ツケル'],
        ['漠', 'サハラサバク ノ バク'],
        ['漣', 'レン，サザナミ'],
        ['漢', 'カンジテスト ノ カン'],
        ['漸', 'ゼンゾウスル ノ ゼン，ヨウヤク'],
        ['漿', 'ショウ，ショウグンノショウノキュウジノシタニミズ'],
        ['漾', 'ヨウ，サンズイニカミサマノカミノキュウジノツクリ'],
        ['漱', 'ナツメソウセキノソウ，ススグ'],
        ['漲', 'チョウ，ミナギル'],
        ['漉', 'チャコシ ノ コス'],
        ['漏', 'モラス，ロウデン ノ ロウ'],
        ['漁', 'ギョコウ ノ ギョ'],
        ['漂', 'ヒョウリュウスル ノ ヒョウ，タダヨウ'],
        ['漆', 'ウルシヌリ ノ ウルシ，シツ'],
        ['漑', 'カンガイヨウスイノガイ'],
        ['漓', 'ボッコンリンリノリ，サンズイニリリクノリノヒダリガワ'],
        ['漕', 'コグ，ソウテイ ノ ソウ'],
        ['演', 'エンソウスル ノ エン'],
        ['萸', 'ユ、クサカンムリニシュユノマノユ'],
        ['落', 'ラクセンスル ノ ラク，オチル'],
        ['萼', 'ショクブツノハナノガク'],
        ['萱', 'クサカンムリ ニ センデン ノ セン'],
        ['萵', 'ワ，クサカンムリニナベノツクリ'],
        ['萪', 'カ、クサカンムリニキョウカショノカ'],
        ['萩', 'ショクブツ ノ ハギ'],
        ['萬', 'マンネンヒツノマン、ヨロズノキュウジ'],
        ['萢', 'ヤチ、クサカンムリニアワダテルノアワ'],
        ['萠', 'ホウガノホウ，モエルノイタイジ'],
        ['鱧', 'サカナノハモ，ウオヘンニユタカ'],
        ['萓', 'ギ、クサカンムリニベンギノギ'],
        ['萋', 'セイ、クサカンムリニアイサイカノサイ'],
        ['萎', 'イシュクスル ノ イ，ナエル'],
        ['萍', 'ヘイ，クサカンムリニサンズイニヘイワノヘイ'],
        ['萌', 'ホウガ ノ ホウ，モエル'],
        ['萃', 'スイ、クサカンムリニソツギョウノソツ'],
        ['萇', 'チョウ、クサカンムリニチョウナンノチョウ'],
        ['萄', 'ブドウシュ ノ ドウ'],
        ['鶤', 'コン，グンタイノグンノミギニトリ'],
        ['鶯', 'ウグイス，アタマニカヨウビノカガフタツ'],
        ['鶩', 'ボク，ムジュンノムトノブンノシタニトリ'],
        ['鶫', 'ツグミ，ニシンノツクリノミギニトリ'],
        ['鶴', 'トリ ノ ツル'],
        ['鶲', 'オウ，オキナノミギニトリ'],
        ['鶸', 'ジャク，ヨワイノミギニトリ，ヒワ'],
        ['鶺', 'セキレイノセキ'],
        ['鶻', 'コツ，ホネノミギニトリ'],
        ['鶇', 'ツグミ，ヒガシノミギニトリ'],
        ['鶏', 'ニワトリ，ヨウケイ ノ ケイ'],
        ['鶉', 'トリノウズラ'],
        ['鶚', 'ガク，ワニノツクリノミギニトリ，ミサゴ'],
        ['時', 'ジカン ノ ジ，トキ'],
        ['晃', 'ニチヨウビ ノ ニチ ノ シタニ ヒカリ，アキラ'],
        ['ﾘ', 'リンゴ ノ リ'],
        ['晁', 'チョウ，ニチヨウビノニチノシタニイッチョウエンノチョウ'],
        ['ﾞ', 'ダクテン'],
        ['ﾟ', 'ハンダクテン'],
        ['晄', 'コウ，ニチヘンニヒカリ'],
        ['ﾝ', 'オシマイ ノ ン'],
        ['ﾒ', 'メガネ ノ メ'],
        ['晋', 'チュウゴク ノ コクメイ ノ シン，ススム'],
        ['ﾐ', 'ミカン ノ ミ'],
        ['晉', 'チュウゴクノコクメイノシン，ススムノキュウジ'],
        ['ﾖ', 'ヨット ノ ヨ'],
        ['晏', 'アン，ニチヨウビノニチノシタニアンシンノアン'],
        ['ﾔ', 'ヤカン ノ ヤ'],
        ['ﾕ', 'ユカタ ノ ユ'],
        ['晒', 'サラシモメン ノ サラシ'],
        ['ﾋ', 'ヒカリ ノ ヒ'],
        ['ﾈ', 'ネズミ ノ ネ'],
        ['R', 'アール ルーム'],
        ['ﾎ', 'ホケン ノ ホ'],
        ['ﾏ', 'マッチ ノ マ'],
        ['ﾌ', 'フトン ノ フ'],
        ['ﾍ', 'ヘイワ ノ ヘ'],
        ['ﾂ', 'ツバメ ノ ツ'],
        ['ﾃ', 'テガミ ノ テ'],
        ['ﾀ', 'タバコ ノ タ'],
        ['ﾁ', 'チキュウ ノ チ'],
        ['晞', 'キ，ニチヘンニキボウノキ'],
        ['晟', 'セイ，ニチヨウビノニチノシタニセイチョウスルノセイ'],
        ['ﾄ', 'トウキョウ ノ ト'],
        ['晝', 'ヒルヤスミノヒル，チュウノキュウジ'],
        ['晢', 'セツ，コッセツスルノセツノシタニニチヨウビノニチ'],
        ['晦', 'ミソカ，ニチヘン ニ マイ'],
        ['晧', 'コウ，ヒヘンニツゲル'],
        ['晤', 'ゴ，ニチヘンニタゴサクノゴ'],
        ['晨', 'シン，ニチヨウビノニチノシタニジュウニシノタツ'],
        ['晩', 'バンゴハン ノ バン'],
        ['普', 'フキュウスル ノ フ'],
        ['景', 'フウケイ ノ ケイ'],
        ['晰', 'ズノウメイセキノセキ'],
        ['面', 'ジメン ノ メン'],
        ['晶', 'スイショウダマ ノ ショウ'],
        ['晴', 'ハレ，セイテン ノ セイ'],
        ['智', 'チ，シル ノ シタニ ニチヨウビ ノ ニチ'],
        ['馗', 'ショウキサマノキ'],
        ['惠', 'メグム，チエノエノキュウジ'],
        ['靭', 'キョウジンナ ノ ジン'],
        ['称', 'イチニンショウ ノ ショウ'],
        ['馘', 'カクシュスルノカク，クビヘンニアル'],
        ['移', 'イドウ ノ イ，ウツル'],
        ['香', 'コウシンリョウ ノ コウ，カオリ'],
        ['秤', 'テンビン ノ ビン，ハカリ'],
        ['秧', 'オウ，ノギヘンニチュウオウシュウケンノオウ'],
        ['秦', 'シン ノ シコウテイ ノ シン'],
        ['秡', 'ハツ、ノギヘンニハライキヨメルノハライノツクリ'],
        ['秣', 'リョウマツノマツ'],
        ['秬', 'キョ、ノギヘンニキョダイナノキョ'],
        ['秩', 'チツジョ ノ チツ'],
        ['秕', 'シイナ，ノギヘンニヒカクスルノヒ'],
        ['科', 'キョウカショ ノ カ'],
        ['秒', 'ビョウヨミ ノ ビョウ'],
        ['租', 'ソゼイ ノ ソ'],
        ['靖', 'ヤスクニジンジャ ノ ヤス'],
        ['秘', 'ヒミツ ノ ヒ'],
        ['私', 'シテツ ノ シ，ワタシ'],
        ['秀', 'シュウサイ ノ シュウ，ヒイデル'],
        ['秉', 'コクモツノタンイヘイ'],
        ['秋', 'アキマツリ ノ アキ，シュウ'],
        ['庸', 'ボンヨウナ ノ ヨウ'],
        ['庵', 'イオリ，タクアン ノ アン'],
        ['庶', 'ショミン ノ ショ'],
        ['康', 'ケンコウ ノ コウ，イエヤス ノ ヤス'],
        ['庭', 'テイエン ノ テイ，ニワ'],
        ['庫', 'レイゾウコ ノ コ'],
        ['度', 'オンドケイ ノ ド'],
        ['座', 'ザセキ ノ ザ，スワル'],
        ['庠', 'ショウ，マダレニヒツジ'],
        ['府', 'トドウフケン ノ フ'],
        ['庚', 'コウシンヅカ ノ コウ，カノエ'],
        ['底', 'ソコヂカラ ノ ソコ，テイ'],
        ['庖', 'マダレ ノ ホウチョウ ノ ホウ'],
        ['店', 'ミセ，ショウテン ノ テン'],
        ['序', 'ジュンジョ ノ ジョ'],
        ['床', 'トコヤ ノ トコ，ショウ'],
        ['庄', 'ショウヤ ノ ショウ'],
        ['庇', 'ヒゴスル ノ ヒ，ヒサシ'],
        ['庁', 'キショウチョウ ノ チョウ'],
        ['広', 'ヒロバ ノ ヒロ，コウ'],
        ['╋', 'フトクロス'],
        ['靈', 'レイエンノレイ、タマノキュウジ'],
        ['╂', 'フトタテセンクロス'],
        ['馳', 'ゴチソウ ノ チ，ハセル'],
        ['馭', 'ギョシャノギョ，ウマヘンニマタ'],
        ['餮', 'テツ，ムサボル'],
        ['馮', 'ヒョウ，タノム，ニスイニウマ'],
        ['馨', 'ジンメイ ノ カオル，ケイ，カグワシイ'],
        ['皎', 'コウ，シロヘンニマジワル'],
        ['鑼', 'ドラヤキノラ'],
        ['鑿', 'コウグノノミ'],
        ['鑾', 'ラン，コイビトノコイノキュウジノココロノカワリニキンヨウビノキン'],
        ['皋', 'ショクブツノサツキノイタイジ'],
        ['皈', 'キタクスルノキ，カエルノイタイジ'],
        ['皆', 'ミナサマ ノ ミナ，カイ'],
        ['皇', 'コウタイシ ノ コウ'],
        ['的', 'モクテキ ノ テキ'],
        ['皃', 'ビボウヲタモツノボウノイタイジ'],
        ['皀', 'キョウ，ハクチョウノハクノシタニカタカナノヒ'],
        ['皚', 'ガイ，シロヘンニヨロイノツクリ'],
        ['皙', 'セキ，ブンセキスルノセキノシタニハクチョウノハク'],
        ['皖', 'カン，シロヘンニカンリョウスルノカン'],
        ['皓', 'メイボウコウシノコウ，シロヘンニツゲル'],
        ['皐', 'ショクブツ ノ サツキ'],
        ['鑢', 'コウグノヤスリ'],
        ['皮', 'カワヲムク ノ カワ，ヒ'],
        ['鑞', 'ロウヅケザイクノロウ'],
        ['鑛', 'テッコウセキノコウノキュウジ'],
        ['鑚', 'ケンサンヲツムノサンノイタイジ'],
        ['鑑', 'ズカン ノ カン'],
        ['鑓', 'ヤリ，カネヘン ニ ハケンスル ノ ケン'],
        ['鑒', 'ズカンノカンノイタイジ'],
        ['皿', 'サラウドン ノ サラ'],
        ['皺', 'シワヲノバスノシワ'],
        ['皸', 'クン，ショウグンノグンノミギニケガワノカワ'],
        ['皹', 'クン，ショウグンノグンノヒダリニケガワノカワ'],
        ['皷', 'ツヅミノイタイジ，ツクリガケガワノカワ'],
        ['皴', 'シュン，シュンビンサノシュンノツクリニケガワノカワ'],
        ['鑁', 'バンナジノバン'],
        ['皰', 'ニキビ，ホウ，ケガワノカワニツツム'],
        ['赤', 'アカンボウ ノ アカ，セキ'],
        ['赦', 'オンシャ ノ シャ，ユルス'],
        ['赧', 'カオヲアカラメルイミノタンガンノタン'],
        ['邉', 'シュウヘンノヘン，アタリノキュウジノイタイジ'],
        ['赭', 'シャ，アカイロノアカノミギニカガクシャノシャノキュウジタイ'],
        ['赫', 'アカヲ フタツ ナラベタ カク'],
        ['赴', 'フニンスル ノ フ，オモムク'],
        ['起', 'キリツスル ノ キ，オキル'],
        ['走', 'ハシル，キョウソウ ノ ソウ'],
        ['赱', 'ハシル，キョウソウノソウノイタイジ'],
        ['赳', 'キュウ、ソウニョウニサケブノツクリ'],
        ['i', 'アイ インク'],
        ['釼', 'ケン、カネヘンニヤイバ'],
        ['梳', 'ソ，クシケズル，キヘンニナガレルノツクリ'],
        ['梱', 'コンポウスル ノ コン'],
        ['械', 'キカイテキ ノ カイ'],
        ['梶', 'カジノキ ノ カジ，キヘン ニ オ'],
        ['梵', 'ボンジノボン'],
        ['梺', 'フモト，ハヤシノシタニウエシタノシタ'],
        ['梹', 'ショクブツノビンロウジュノビンノイタイジ，キヘンニヘイタイノヘイ'],
        ['梼', 'トウ，キヘン ニ コトブキ'],
        ['梢', 'マッショウシンケイ ノ ショウ，コズエ'],
        ['梠', 'リョ，キヘンニオフロノロ'],
        ['梧', 'ゴ，キヘン ニ ワレ，アオギリ'],
        ['梦', 'ユメ，ムチュウノムノイタイジ'],
        ['梨', 'クダモノ ノ ナシ'],
        ['梯', 'ハシゴ，キヘン ニ オトウト'],
        ['梭', 'サ，キヘンニシュンビンサノシュンノツクリ'],
        ['梓', 'アズサ，ジョウシスル ノ シ'],
        ['梗', 'ノウコウソク ノ コウ'],
        ['梔', 'ショクブツノクチナシ'],
        ['梛', 'ダ，キヘンニワカダンナノナ'],
        ['鋏', 'ドウグノハサミ'],
        ['梟', 'フクロウ'],
        ['條', 'ジョウケンノジョウノキュウジ'],
        ['梃', 'テコイレノテコ，テイ'],
        ['梁', 'リョウザンパク ノ リョウ，ハリ'],
        ['梅', 'ウメボシ ノ ウメ'],
        ['梏', 'シッコクトナルノコク，テカセ'],
        ['梍', 'ソウ，キヘンニシロクロノシロニカンスウジノナナ'],
        ['蒼', 'ガンメンソウハク ノ ソウ，アオイ'],
        ['蒿', 'コウ，クサカンムリニタカイヒクイノタカイ'],
        ['蒹', 'ケン，クサカンムリニカネル'],
        ['蒸', 'ジョウキ ノ ジョウ，ムス'],
        ['蒻', 'コンニャクノニャク'],
        ['鋲', 'ガビョウ ノ ビョウ'],
        ['蒲', 'カバヤキ ノ カバ'],
        ['蒭', 'スウ，クサカンムリニハンスウスルノスウ'],
        ['蒡', 'キンピラゴボウノボウ'],
        ['蒜', 'ショクブツ ノ ヒル'],
        [
            '蒟',
            'コンニャクノイチモジメ，クサカンムリニドクリツノリツ，ソノミギニハイクノク',
        ],
        ['蒙', 'ケイモウスル ノ モウ'],
        ['蒔', 'タネヲマク ノ マク'],
        ['蒐', 'クサカンムリ ニ オニ ノ シュウ'],
        ['蒋', 'クサカンムリ ニ ショウグン ノ ショウ'],
        ['蒄', 'カン、クサカンムリニゲッケイカンノカン'],
        ['蒂', 'タイ，クサカンムリニテイオウノテイ'],
        ['‰', 'パーミル'],
        ['″', 'ビョウ'],
        ['褞', 'オン、コロモヘンニオンセンノオンノキュウジノツクリ'],
        ['※', 'コメジルシ'],
        ['?', 'クエスチョン'],
        ['乾', 'カワク，カンソウキ ノ カン'],
        ['坑', 'タンコウ ノ コウ，アナ'],
        ['坐', 'ザセキ ノ ザカラマダレヲトッタザ，スワル'],
        ['鋤', 'ノウグ ノ スキ，カネヘン ニ タスケル'],
        ['乱', 'コンラン ノ ラン，ミダレル'],
        ['乳', 'ギュウニュウ ノ ニュウ，チチ'],
        ['均', 'ヘイキン ノ キン'],
        ['址', 'シ，シロアトヲイミスルジョウシノシ'],
        ['坂', 'サカミチ ノ サカ'],
        ['坏', 'タカツキノツキ'],
        ['坎', 'カン，アナ，ツチヘンニホケツノケツ'],
        ['乢', 'カイ，ヤマヘンニレイギノレイノツクリ'],
        ['坊', 'ボウズ ノ ボウ'],
        ['九', 'カンスウジ ノ キュウ'],
        ['乞', 'コウ，アマゴイスル ノ コウ'],
        ['也', 'クウヤネンブツ ノ ヤ，ナリ'],
        ['乘', 'ジョウシャスルノジョウ，ノルノキュウジ'],
        ['乙', 'コウオツ ノ オツ'],
        ['乕', 'ドウブツノトラノイタイジ'],
        ['坿', 'フ，ツチヘンニフロクノフ'],
        ['乗', 'ジョウシャスル ノ ジョウ，ノル'],
        ['坤', 'ケンコン ノ コン，ツチヘン ニ モウス'],
        ['乎', 'コ，ヨブ ノ ツクリ'],
        ['坦', 'ヘイタンナミチ ノ タン'],
        ['坡', 'ハ，ツチヘンニケガワノカワ'],
        ['鋭', 'エイリナ ノ エイ，スルドイ'],
        ['之', 'コレ，アクタガワリュウノスケ ノ ノ'],
        ['久', 'エイキュウ ノ キュウ，ヒサシイ'],
        ['坩', 'カン，ルツボ，ツチヘンニアマイ'],
        ['乂', 'ガイ，クサカリノカリノヒダリガワ'],
        ['坪', 'タテツボ ノ ツボ'],
        ['褌', 'フンドシ、コロモヘンニグンタイノグン'],
        ['硼', 'ゲンソノホウソノホウ'],
        ['硴', 'カキ，イシヘンニハナビノハナ'],
        ['硲', 'ハザマ，イシヘン ニ タニ'],
        ['硯', 'スズリ'],
        ['硬', 'ヒャクエンコウカ ノ コウ，カタイ'],
        ['硫', 'リュウサン ノ リュウ'],
        ['―', 'バー'],
        ['硝', 'ショウサンエン ノ ショウ'],
        ['硅', 'ゲンソノケイソノケイ，イシヘンニツチフタツ'],
        ['“', 'クォーテーション'],
        ['麁', 'ソ，アライ，カタカナノクノシタニシカ'],
        ['麒', 'キリンジノキ，シカノミギニソノ'],
        ['鴬', 'ウグイス'],
        ['鴪', 'イツ，アナヲホルノアナノミギニトリ'],
        ['鴫', 'シギヤキ ノ シギ'],
        ['鴨', 'トリ ノ カモ'],
        ['鴦', 'エンオウノオウ，オシドリ'],
        ['麗', 'カレイナ ノ レイ，ウルワシイ'],
        ['鴣', 'トリノシャコノコ，チュウコシャノコノミギニトリ'],
        ['鴾', 'ボウ，シャカムニノムノミギニトリ'],
        ['鴿', 'コウ，ゴウカクノゴウノミギニトリ'],
        ['鴻', 'コウコク ノ ココロザシ ノ コウ，オオトリ'],
        ['鴎', 'カモメ'],
        ['G', 'ジー ゴルフ'],
        ['鴈', 'ガン，ガンダレニニンベンニトリ'],
        ['鴉', 'キバニトリ，カラス'],
        ['鴆', 'チン，シズムノツクリノミギニトリ'],
        ['鴇', 'トリ ノ トキ'],
        ['鴃', 'トリヘンニカイテキノカイ，モズ'],
        ['麝', 'ジャコウジカノジャ，シカノシタニハンシャノシャ'],
        ['鴟', 'シ，テイコウスルノテイノツクリノミギニトリ'],
        ['鴛', 'エンオウ ノ エン，オシドリ'],
        ['鴕', 'ダ，トリヘンニブッダノダノツクリ'],
        ['鴒', 'セキレイノレイ'],
        ['柄', 'ヒトガラ ノ ガラ'],
        ['柆', 'ロウ，キヘンニドクリツノリツ'],
        ['柁', 'カジ，キヘン ニ ウカンムリ ニ カタカナ ノ ヒ'],
        ['柎', 'フ，キヘンニフロクノフ'],
        ['柏', 'カシワモチ ノ カシワ'],
        ['柊', 'ショクブツ ノ ヒイラギ，キヘン ニ フユ'],
        ['柔', 'ジュウドウ ノ ジュウ，ヤワラカイ'],
        ['某', 'ナニガシ ノ ボウ'],
        ['柑', 'カンキツルイ ノ カン'],
        ['染', 'センショク ノ セン，ソメル'],
        ['柝', 'タク，キヘンニハイセキウンドウノセキ'],
        ['柞', 'サク，キヘンニサクブンノサクノツクリ'],
        ['柘', 'ザクロ ノ イチモジメ，キヘン ニ イシ'],
        ['柚', 'ショクブツ ノ ユズ，ユウ'],
        ['麥', 'ムギノキュウジ'],
        ['柤', 'サ，キヘンニナオカツノカツ'],
        ['柧', 'コ，キヘンニウリ'],
        ['麦', 'ムギチャ ノ ムギ'],
        ['柢', 'テイ，キヘンニヒクイノツクリ'],
        ['柬', 'カン，イサメルノツクリ'],
        ['柮', 'トツ，キヘンニガイシュツスルノトツ'],
        ['柯', 'カ，キヘンニカノウセイノカ'],
        ['開', 'カイハツ ノ カイ，ヒラク'],
        ['柩', 'ヒツギ，レイキュウシャノキュウ'],
        ['柴', 'シバイヌ ノ シバ，タキギ ノ シバ'],
        ['柵', 'サクデカコム ノ サク'],
        ['柱', 'ハシラ，チュウ'],
        ['柳', 'ヤナギ，センリュウ ノ リュウ'],
        ['柾', 'ショクブツ ノ マサキ，キヘン ニ タダシイ'],
        ['柿', 'クダモノ ノ カキ'],
        ['麪', 'メンルイノメンノイタイジ'],
        ['査', 'ケンサ ノ サ'],
        ['焔', 'ホノオ，ヒヘン ノ ホノオ'],
        ['焙', 'バイセンスルノバイ，アブル'],
        ['焚', 'フンショコウジュ ノ フン，タキビ ノ タキ'],
        ['焜', 'コン，ヒヘンニコンチュウノコン'],
        ['麭', 'ホウ，ムギノキュウジニツツム'],
        ['焉', 'シュウエンノチノエン'],
        ['金', 'キンヨウビ ノ キン，カネ'],
        ['然', 'ダイシゼン ノ ゼン'],
        ['焼', 'ニクヲヤク ノ ヤク'],
        ['無', 'ムリ ノ ム，ナイ'],
        ['焦', 'ショウテンキョリ ノ ショウ'],
        ['闇', 'クラヤミ ノ ヤミ'],
        ['路', 'ドウロ ノ ロ'],
        ['釐', 'リ，ミライノミニノブンノシタニタンイノリン'],
        ['跪', 'ヒザマズク，キ'],
        ['跫', 'キョウ、キョウリュウノキョウノココロノカワリニテアシノアシ'],
        ['跨', 'コセンキョウ ノ コ，マタグ'],
        ['闍', 'アジャリノジャ'],
        ['闌', 'ラン，ハランバンジョウノランノツクリ'],
        ['跣', 'セン，アシヘンニイキサキノサキ'],
        ['跡', 'ツイセキスル ノ セキ，アト'],
        ['跿', 'ト，アシヘンニダッソウスルノソウ'],
        ['跼', 'キョク，アシヘンニヤッキョクノキョク'],
        ['闔', 'コウ，モンガマエニキョネンノキョニサラ'],
        ['V', 'ブイ ビデオ'],
        ['践', 'ジッセンスル ノ セン'],
        ['跳', 'チョウヤクスル ノ チョウ，トブ'],
        ['闘', 'トウギュウシ ノ トウ，タタカウ'],
        ['跏', 'カ、アシヘンニカソクドノカ'],
        ['跌', 'ジンセイノサテツノテツ'],
        ['跋', 'チョウリョウバッコノバツ'],
        ['闡', 'セン，モンガマエニタンジュンナノタンノキュウジ'],
        ['麼', 'モ，マスイノマノシタニイトガシラ'],
        ['跂', 'キ，アシヘンニササエル'],
        ['跟', 'コン，アシヘンニダイコンノコンノツクリ'],
        ['距', 'キョリ ノ キョ'],
        ['跚', 'スイホマンサンノサン、アシヘンニイッサツニサツノサツ'],
        ['跛', 'ハ，アシヘンニケガワノカワ'],
        ['跖', 'セキ、アシヘンニイシ'],
        ['麿', 'ウタマロ ノ マロ'],
        ['劭', 'ショウ，マネクノツクリニチカラ'],
        ['針', 'ホウシン ノ シン，ハリ'],
        ['帆', 'ハンセン ノ ハ，ホバシラ ノ ホ'],
        ['市', 'シヤクショ ノ シ，イチ'],
        ['布', 'ヌノ，モウフ ノ フ'],
        ['希', 'キボウ ノ キ，マレ'],
        ['帋', 'ガヨウシノシ，カミノイタイジ'],
        ['帖', 'テチョウ ノ チョウ，ハバニウラナウ'],
        ['帑', 'ド，ヤッコノシタニハバ'],
        ['帝', 'テイオウ ノ テイ，ミカド'],
        ['帚', 'ホウキ，ソウジキノソウノツクリ'],
        ['帛', 'ハク，ハクチョウノハクノシタニハバ'],
        ['帙', 'チツ，ハバヘンニウシナウ'],
        ['帥', 'ダイゲンスイ ノ スイ'],
        ['帯', 'ケイタイ ノ タイ，オビ'],
        ['席', 'ザセキ ノ セキ'],
        ['師', 'ボクシ ノ シ'],
        ['非', 'ヒジョウグチ ノ ヒ'],
        ['帶', 'ケイタイノタイ，オビノキュウジ'],
        ['帷', 'カタビラ，ハバヘンニフルトリ'],
        ['帳', 'レンラクチョウ ノ チョウ'],
        ['帰', 'キタクスル ノ キ，カエル'],
        ['帽', 'ダツボウスル ノ ボウ'],
        ['常', 'ヒジョウグチ ノ ジョウ'],
        [',', 'コンマ'],
        ['回', 'マワル，カイテン ノ カイ'],
        ['四', 'カンスウジ ノ ヨン'],
        ['囚', 'シュウジン ノ シュウ，トラワレル'],
        ['囘', 'マワル，カイテンノカイノイタイジ，ドウガマエニジュウニシノミ'],
        ['囗', 'イ，ブシュノクニガマエ'],
        ['囓', 'ゲツ，クチヘンニゲッシルイノゲツ'],
        ['囑', 'ショク，ショクタクスルノショクノキュウジ'],
        ['囎', 'ソ，クチヘンニカイガラノカイニキソジノソノキュウジ'],
        ['囈', 'ゲイ，クチヘンニゲイジュツノゲイノキュウジ'],
        ['囃', 'ゴニンバヤシノハヤシ'],
        ['囂', 'ケンケンゴウゴウノゴウ'],
        ['囁', 'ササヤク'],
        ['囀', 'サエズル，クチヘンニウンテンノテンノキュウジ'],
        ['囿', 'ユウ，クニガマエニユウメイジンノユウ'],
        ['国', 'コクゴ ノ コク，クニ'],
        ['固', 'コタイ ノ コ，カタイ'],
        ['囹', 'レイ，クニガマエニメイレイスルノレイ'],
        ['図', 'トショカン ノ ト，ズ'],
        ['囲', 'イゴ ノ イ，カコム'],
        ['困', 'コンナン ノ コン，コマル'],
        ['囮', 'オトリソウサノオトリ'],
        ['団', 'ダンタイコウドウ ノ ダン'],
        ['因', 'ゲンイン ノ イン，チナミニ'],
        ['偉', 'イダイ ノ イ，エライ'],
        ['ё', 'キリル ィヨー'],
        ['偏', 'ヘンセイフウ ノ ヘン，カタヨル'],
        ['с', 'キリル エス'],
        ['р', 'キリル エル'],
        ['у', 'キリル ウー'],
        ['т', 'キリル テー'],
        ['х', 'キリル ハー'],
        ['ф', 'キリル エフ'],
        ['ч', 'キリル チェー'],
        ['ц', 'キリル ツェー'],
        ['щ', 'キリル シシャー'],
        ['ш', 'キリル シャー'],
        ['ы', 'キリル ウィ'],
        ['ъ', 'キリル コウオンキゴウ'],
        ['э', 'キリル エー'],
        ['ь', 'キリル ナンオンキゴウ'],
        ['я', 'キリル ヤー'],
        ['ю', 'キリル ユー'],
        ['覚', 'カンカク ノ カク，オボエル'],
        ['蔆', 'ヒシガタノヒシノイタイジ'],
        ['蔀', 'シトミ，クサカンムリ ニ ブチョウ ノ ブ'],
        ['蔗', 'ショトウノショ，クサカンムリニショミンノショ'],
        ['蔕', 'タイ，クサカンムリニオビノキュウジ'],
        ['蔔', 'ホク，クサカンムリニホフクゼンシンノフク'],
        ['蔓', 'マンエンスル ノ マン，ツル'],
        ['蔑', 'ケイベツスル ノ ベツ，サゲスム'],
        ['偐', 'ガン，ニンベンニヒコボシノヒコノキュウジ'],
        ['蔟', 'ソク，クサカンムリニスイゾクカンノゾク'],
        ['蔚', 'クサカンムリ ニ リクグンタイイ ノ イ'],
        ['蔘', 'シン，クサカンムリニサンギインノサンノキュウジ'],
        ['蔦', 'ショクブツ ノ ツタ'],
        ['蔡', 'サイ，クサカンムリニシュクサイジツノサイ'],
        ['蔭', 'クサカンムリ ノ カゲ'],
        ['蔬', 'ソサイルイノソ'],
        ['蔵', 'レイゾウコ ノ ゾウ'],
        ['2', 'ニ'],
        ['蔽', 'シャヘイスル ノ ヘイ'],
        ['偖', 'シャ，ニンベンニカガクシャノシャノキュウジタイ'],
        ['偬', 'ソウ，ニンベンニネギノクサカンムリヲトッタモノ'],
        ['仆', 'ボク，ニンベンニウラナウノボク'],
        ['仇', 'アダ，キュウテキ ノ キュウ'],
        ['仄', 'ソクブンスルノソク，ホノカ'],
        ['仂', 'ロク，ニンベンニチカラ'],
        ['什', 'ジュウキ ノ ジュウ，ニンベン ニ カンスウジ ノ ジュウ'],
        ['栲', 'コウ，キヘンニサンコウショノコウ'],
        ['栽', 'サイバイスル ノ サイ'],
        ['仏', 'ブツゾウ ノ ブツ，ホトケ'],
        ['仍', 'ジョウ，ニンベンニノギタイショウノノ'],
        ['根', 'ダイコン ノ コン，ネ'],
        ['介', 'カイゴ ノ カイ，スケ'],
        ['他', 'タニン ノ タ'],
        ['仗', 'ギジョウヘイノジョウ'],
        ['仔', 'シサイナ ノ シ，ニンベン ニ コドモ ノ コ'],
        ['仕', 'シゴト ノ シ，ツカエル'],
        ['校', 'ガッコウ ノ コウ'],
        ['栢', 'カヤ，キヘン ニ ヒャク'],
        ['仞', 'フカサノタンイノジン，ニンベンニカタナニテン'],
        ['仟', 'セン，ニンベンニカンスウジノセン'],
        ['仝', 'ドウジョウ'],
        ['栩', 'ク，キヘンニウモウノウ'],
        ['付', 'フロク ノ フ，ツケル'],
        ['株', 'カブシキ ノ カブ'],
        ['令', 'メイレイ ノ レイ'],
        ['以', 'イジョウ，イカ ノ イ'],
        ['代', 'ダイヒョウ ノ ダイ，カワル'],
        ['栓', 'センヌキ ノ セン'],
        ['仮', 'カメン ノ カ，カリ'],
        ['栞', 'シオリヲハサムノシオリ'],
        ['件', 'ジョウケン ノ ケン'],
        ['栄', 'エイヨウ ノ エイ，サカエル'],
        ['仲', 'ナカマ ノ ナカ'],
        ['仰', 'ギョウテンスル ノ ギョウ，アオグ'],
        ['栂', 'ショクブツ ノ ツガ，キヘン ニ ハハ'],
        ['任', 'セキニン ノ ニン'],
        ['m', 'エム マイク'],
        ['側', 'リョウガワ ノ ガワ，ソク'],
        ['偶', 'グウゼン ノ グウ'],
        ['磁', 'ジシャク ノ ジ'],
        ['磅', 'オモサノタンイノポンド'],
        ['磆', 'カツ，イシヘンニホネ'],
        ['骸', 'ガイコツ ノ ガイ，ムクロ'],
        ['磋', 'セッサタクマノサ，イシヘンニサ'],
        ['磊', 'ゴウホウライラクノライ'],
        ['骼', 'カク，ホネヘンニゴウカクノカクノツクリ'],
        ['磑', 'ガイ，イシヘンニヨロイノツクリ'],
        ['磐', 'ジョウバンセン ノ バン'],
        ['磔', 'タッケイノタク，ハリツケ'],
        ['骨', 'ガイコツ ノ コツ，ホネ'],
        ['磚', 'セン，カワラ，イシヘンニセンモンノセンノキュウジ'],
        ['骭', 'カン，ホネヘンニアセノツクリ'],
        ['磧', 'セキ，カワラ，イシヘンニセキニンノセキ'],
        ['磨', 'ケンマスル ノ マ，ミガク'],
        ['磬', 'ケイ，コエノキュウジノミミノカワリニイシ'],
        ['磯', 'イソヅリ ノ イソ'],
        ['磴', 'トウ，イシヘンニトザンノト'],
        ['磽', 'コウ，イシヘンニギョウシュンノギョウノキュウジ'],
        ['李', 'リカニ カンムリヲ タダサズ ノ リ，スモモ'],
        ['杏', 'アンズ，キョウリン ノ キョウ'],
        ['杉', 'スギノキ ノ スギ'],
        ['杆', 'カン、キヘンニホス'],
        ['杁', 'イリ、キヘンニハイル'],
        ['杞', 'キユウニオワルノキ'],
        ['束', 'ヤクソク ノ ソク'],
        ['杜', 'モリ，キヘン ニ ツチ'],
        ['杙', 'ヨク、キヘンニダイヒョウノダイノツクリ'],
        ['杖', 'ハクジョウ ノ ジョウ，ツエ'],
        ['杓', 'シャクシジョウギ ノ シャク'],
        ['材', 'ザイリョウ ノ ザイ'],
        ['村', 'ムラ，シチョウソン ノ ソン'],
        ['杯', 'カンパイ ノ ハイ，サカズキ'],
        ['杭', 'クイヲウツ ノ クイ'],
        ['杪', 'ビョウ、コズエ、キヘンニスクナイ'],
        ['杤', 'トチギケンノトチノイタイジ、キヘンニマンネンヒツノマン'],
        ['来', 'ライネン ノ ライ，クル'],
        ['杢', 'モク，キ ノ シタニ コウ'],
        ['杣', 'ソマ、キヘンニヤマ'],
        ['杠', 'コウ、キヘンニズガコウサクノコウ'],
        ['条', 'ジョウケン ノ ジョウ'],
        ['松', 'マツノキ ノ マツ，ショウ'],
        ['板', 'コクバン ノ バン，イタ'],
        ['杼', 'ジョ、キヘンニヨテイノヨ'],
        ['杷', 'ショクブツ ノ ビワ ノ ハ，キヘン ニ トモエ'],
        ['杵', 'ウスキネ ノ キネ'],
        ['杲', 'コウ，アキラカ，ニチヨウビノニチノシタニジュモクノモク'],
        ['杳', 'ヨウトシテシレナイノヨウ，ヨウビノモクノシタニニチ'],
        ['杰', 'ゴウケツノケツノイタイジ，モクザイノモクノシタニレンガ'],
        ['東', 'トウザイ ノ トウ，ヒガシ'],
        ['貅', 'キュウ、ムジナヘンニヤスム'],
        ['C', 'シー キャット'],
        ['⑦', 'マルナナ'],
        ['⑧', 'マルハチ'],
        ['⑤', 'マルゴ'],
        ['⑥', 'マルロク'],
        ['③', 'マルサン'],
        ['④', 'マルヨン'],
        ['①', 'マルイチ'],
        ['②', 'マルニ'],
        ['⑮', 'マルジュウゴ'],
        ['⑯', 'マルジュウロク'],
        ['⑬', 'マルジュウサン'],
        ['⑭', 'マルジュウヨン'],
        ['⑪', 'マルジュウイチ'],
        ['⑫', 'マルジュウニ'],
        ['⑨', 'マルキュウ'],
        ['⑩', 'マルジュウ'],
        ['⑲', 'マルジュウキュウ'],
        ['⑳', 'マルニジュウ'],
        ['⑰', 'マルジュウナナ'],
        ['⑱', 'マルジュウハチ'],
        ['嶐', 'リュウ，ヤマノシタニリュウキスルノリュウニイッカククワエタカタチ'],
        ['嶬', 'ギ，ヤマヘンニギムノギ'],
        ['嶮', 'ケン，ヤマヘンニシケンヲウケルノケンノキュウジノツクリ'],
        ['嶢', 'ギョウ，ヤマヘンニギョウシュンノギョウノキュウジ'],
        ['嶺', 'サンレイ ノ レイ，ミネ'],
        ['嶼', 'ショ，ヤマヘンニアタエルノキュウジ'],
        ['嶽', 'サンガクブノガクノキュウジ'],
        ['嶷', 'ギ，ヤマノシタニギモンノギ'],
        ['黌', 'ショウヘイコウノコウ'],
        ['熙', 'キ，モトシュショウノホソカワモリヒロノヒロ'],
        ['熟', 'ジュクゴ ノ ジュク，ジュクスル'],
        ['熔', 'ヨウガン ノ ヨウ，ヒヘン ニ ナイヨウ ノ ヨウ'],
        ['熕', 'コウ，ヒヘンニシャカイコウケンノコウ'],
        ['熊', 'ドウブツ ノ クマ'],
        ['熈', 'コウキジテンノキノイタイジ'],
        ['熏', 'クンセイノクン，クスベル'],
        ['熄', 'ソク，ヒヘンニイキヲスルノイキ'],
        ['熹', 'キ，キゲキノキニレンガ'],
        ['熾', 'シレツナノシ'],
        ['熱', 'ネツ，アツイ'],
        ['熨', 'ノシ，リクグンタイイノイノシタニカヨウビノカ'],
        ['熬', 'ゴウ，ゴウマンナノゴウノツクリニレンガ'],
        ['鯵', 'アジ，サカナヘン ニ マイル'],
        ['詰', 'ツメル，キツモン ノ キツ'],
        ['話', 'ハナシ，デンワ ノ ワ'],
        ['該', 'ガイトウスル ノ ガイ'],
        ['詳', 'ショウサイ ノ ショウ，クワシイ'],
        ['詼', 'カイ，ゴンベンニハイイロノハイノキュウジタイ'],
        ['詠', 'エイタン ノ エイ，ヨム'],
        ['詢', 'ジュン，ゴンベンニゲジュンノジュン'],
        ['詣', 'モウデル，サンケイスル ノ ケイ'],
        ['鯱', 'サカナノシャチ'],
        ['試', 'シアイ ノ シ，ココロミル'],
        ['詩', 'シヲツクル ノ シ'],
        ['詫', 'オワビスル ノ ワビ'],
        ['詬', 'コウ，ゴンベンニコウゴウヘイカノゴウ'],
        ['詭', 'キベンヲロウスルノキ'],
        ['詮', 'センサクスル ノ セン'],
        ['詐', 'サギシ ノ サ'],
        ['詑', 'タ，ゴンベン ニ ウカンムリ ニ カタカナ ノ ヒ'],
        ['詒', 'タイ，ゴンベンニダイドコロノダイ'],
        ['詔', 'ショウチョク ノ ショウ，ミコトノリ'],
        ['評', 'ヒョウカスル ノ ヒョウ'],
        ['詛', 'ジュソノソ、ゴンベンニナオカツノカツ'],
        ['詞', 'サクシサッキョク ノ シ'],
        ['詁', 'クンコガクノコ、ゴンベンニチュウコシャノコ'],
        ['詆', 'テイ，ゴンベンニテイコウスルノテイノツクリ'],
        ['詈', 'バリゾウゴンノリ'],
        ['長', 'ナガイ，チョウタン ノ チョウ'],
        ['榿', 'キ，キヘンニヨロイノツクリ'],
        ['榾', 'コツ，キヘンニホネ'],
        ['榻', 'トウ，キヘンニヨウビノニチノシタニハネ'],
        ['榴', 'リュウ，ザクロ，キヘンニリュウガクスルノリュウ'],
        ['榲', 'オツ，キヘンニオンセンノオンノキュウジタイノツクリ'],
        ['榱', 'スイ，キヘンニオトロエル'],
        ['榮', 'エイヨウノエイ，サカエルノキュウジ'],
        ['榧', 'カヤノキノカヤ'],
        ['榠', 'メイ，キヘンニメイフクヲイノルノメイ'],
        ['榜', 'ヒョウボウスルノボウ'],
        ['榛', 'ショクブツ ノ ハシバミ'],
        ['榕', 'ヨウ，キヘンニヨウセキノヨウ'],
        ['榔', 'ショクブツ ノ ビンロウジュ ノ ロウ'],
        ['榑', 'フ，キヘンニハクブツカンノハクノキュウジノツクリ'],
        ['榎', 'エノキ，キヘン ニ ナツ'],
        ['榊', 'ショクブツ ノ サカキ，キヘン ニ カミサマ ノ カミ'],
        ['概', 'ガイネン ノ ガイ，オオムネ'],
        ['榁', 'ムロノキノムロ，キヘンニムロマチノムロ'],
        ['鯛', 'サカナ ノ タイ'],
        ['Z', 'ゼット ズー'],
        ['ﾚ', 'レモン ノ レ'],
        ['ﾛ', 'ロウカ ノ ロ'],
        ['鯒', 'サカナノコチ，ウオヘンニオケノツクリ'],
        ['鯑', 'カズノコ，ウオヘンニマレ'],
        ['鷙', 'シ，シツジノシツノシタニトリ'],
        ['噤', 'キン，ツグム，クチヘンニキンシスルノキン'],
        ['器', 'ガッキ ノ キ，ウツワ'],
        ['噫', 'アア，クチヘンニイミノイ'],
        ['噪', 'ソウ，クチヘンニソウジュウスルノソウノツクリ'],
        ['噬', 'ホゾヲカムノカム，クチヘンニゼイチクノゼイ'],
        ['噴', 'フンカスル ノ フン，フク'],
        ['噸', 'オモサ ノ タンイ ノ トン'],
        ['噺', 'ハナシカ ノ ハナシ，クチヘン ニ アタラシイ'],
        ['噂', 'ウワサバナシ ノ ウワサ'],
        ['噌', 'ミソ ノ ソ'],
        ['噎', 'エツ，ムセブ，クチヘンニケイヤクショニツカウイチノキュウジ'],
        ['噐', 'ウツワ，ガッキノキノイタイジ'],
        ['噛', 'カム，クチヘン ニ ハ'],
        ['0', 'ゼロ'],
        ['ﾓ', 'モミジ ノ モ'],
        ['薈', 'ワイ，クサカンムリニカイシャインノカイノキュウジ'],
        ['薊', 'ショクブツノアザミ'],
        ['薀', 'ウンチクヲカタムケルノウン，サンズイ'],
        ['薄', 'ウスイ，ケイハク ノ ハク'],
        ['薇', 'ビ，バライロノバラノニモジメ'],
        ['薙', 'クサナギ ノ ナギ'],
        ['薛', 'カワラヨモギヲイミスルセツ'],
        ['薜', 'ヘイ，クサカンムリニカベノウエガワ'],
        ['薑', 'キョウ，クサカンムリニカシハラジングウノカシノツクリ'],
        ['薐', 'リョウ，クサカンムリニヤマノリョウセンノリョウ'],
        ['薔', 'ショウ，バライロノバラノイチモジメ'],
        ['薗', 'ソノ，クサカンムリ ニ ガクエン ノ エン'],
        ['薩', 'サツマハン ノ サツ'],
        ['薨', 'コウキョスルノコウ，ミマカル'],
        ['薫', 'クンプウ ノ クン，カオル'],
        ['薪', 'タキギ，マキワリ ノ マキ'],
        ['薬', 'ヤクヒン ノ ヤク，クスリ'],
        ['薯', 'バレイショ ノ ショ，イモ'],
        ['薮', 'ヤブ，クサカンムリ ニ サンスウ ノ スウ'],
        ['薤', 'ラッキョウ'],
        ['薦', 'スイセンジョウ ノ セン，ススメル'],
        ['薹', 'ダイ，トウガタツノトウ'],
        ['薺', 'ナナクサノナズナ，セイ'],
        ['霽', 'セイ、アメカンムリニイッセイシャゲキノセイノキュウジ？'],
        ['霹', 'セイテンノヘキレキノヘキ'],
        ['霧', 'ノウム ノ ム，キリ'],
        ['癧', 'ルイレキノレキ，ヤマイダレニレキシカノレキノキュウジ'],
        ['霓', 'ゲイ、アメカンムリニジドウセイトノジノキュウジ'],
        ['筋', 'キンニク ノ キン，スジ'],
        ['等', 'イットウショウ ノ トウ，ヒトシイ'],
        ['筈', 'テハズ ノ ハズ'],
        ['筏', 'イカダ'],
        ['筍', 'タケノコ，ジュン'],
        ['筌', 'チャセンノセン，タケカンムリニゼンコクノゼン'],
        ['騰', 'フットウスル ノ トウ'],
        ['筆', 'フデバコ ノ フデ，ヒツ'],
        ['筅', 'チャセンノセン，タケカンムリニイキサキノサキ'],
        ['騫', 'ケン，サムイノシタニテンノカワリニウマ'],
        ['騨', 'ヒダタカヤマ ノ ダ'],
        ['筝', 'ソウキョクノソウ，コトノイタイジ'],
        ['筒', 'ツツ，スイトウ ノ トウ'],
        ['筑', 'チクゼンニ ノ チク'],
        ['筐', 'キョウタイノキョウ，ハコ'],
        ['策', 'サクリャク ノ サク'],
        ['答', 'コタエ，トウアン ノ トウ'],
        ['騙', 'ダマス'],
        ['筮', 'ゼイチクノゼイ'],
        ['筬', 'セイ，オサ，タケカンムリニセイリツスルノセイ'],
        ['騒', 'サワグ，ソウオン ノ ソウ'],
        ['験', 'ニュウガクシケン ノ ケン'],
        ['筧', 'カケヒ，タケカンムリニケンブツスルノケン'],
        ['筥', 'ロ，タケカンムリニオフロノロ'],
        ['筺', 'キョウタイノキョウ，ハコノイタイジ'],
        ['騎', 'キヘイタイ ノ キ'],
        ['騏', 'キリンノキ，ウマヘンニソノ'],
        ['筱', 'ジョウ，シノダケノシノノイタイジ'],
        ['筰', 'サク，タケカンムリニサクブンノサク'],
        ['筵', 'エン，ムシロ，タケカンムリニエンキスルノエン'],
        ['筴', 'キョウ，タケカンムリニセマイノキュウジノツクリ'],
        ['惑', 'メイワク ノ ワク'],
        ['惓', 'ケン，リッシンベンニノリマキノマキ'],
        ['惘', 'ボウ，リッシンベンニモウマクノモウノツクリ'],
        ['惚', 'ホレル，コウコツ ノ コツ'],
        ['惜', 'オシム，セキハイ ノ セキ'],
        ['惟', 'シイ ノ イ，リッシンベン ニ フルトリ'],
        ['情', 'ユウジョウ ノ ジョウ，ナサケ'],
        ['惆', 'チュウ，リッシンベンニシュウヘンノシュウ'],
        ['惇', 'リッシンベン ニ キョウラクテキ ノ キョウ，トン'],
        ['惰', 'タイダ ノ ダ'],
        ['惱', 'ナヤム，クノウスルノノウノキュウジ'],
        ['想', 'カンソウブン ノ ソウ'],
        ['惴', 'ズイ，リッシンベンニゼンソクノゼンノツクリ'],
        ['惶', 'コウ，リッシンベンニコウタイシノコウ'],
        ['惷', 'シュン，キセツノハルノシタニココロ'],
        ['惹', 'ジャッキスル ノ ジャク，ヒク'],
        ['惺', 'セイ，リッシンベンニホシ'],
        ['惻', 'ソクインノジョウノソク，リッシンベンニホウソクノソク'],
        ['q', 'キュー クエスチョン'],
        ['惡', 'ゼンアクノアク，ワルイノキュウジ'],
        ['惣', 'オソウザイ ノ ソウ'],
        ['惧', 'キグスルノグ'],
        ['惨', 'ザンパイスル ノ ザン，ミジメ'],
        ['霖', 'リンウノリン、アメエカンムリニサンリンノリン'],
        ['瀝', 'ヒレキスルノレキ'],
        ['瀞', 'ナガトロ ノ トロ，サンズイ ニ シズカ'],
        ['瀟', 'ショウシャナタテモノノショウ'],
        ['瀘', 'ロ，サンズイニハゼノキノハゼノツクリ'],
        ['瀚', 'コウカンナゾウショノカン'],
        ['瀛', 'ウミヲイミスルサンズイノエイ'],
        ['瀕', 'ヒンシ ノ ジュウショウ ノ ヒン'],
        ['瀑', 'ナイアガラバクフノバク'],
        ['瀏', 'リュウ，サンズイニリュウホウノリュウ'],
        ['瀉', 'イッシャセンリノシャ，サンズイニシャシンノシャノキュウジ'],
        ['瀋', 'チュウゴクノシンヨウノシン，サンズイニシンサスルノシン'],
        ['瀁', 'ヨウ，サンズイニヤシナウ'],
        ['瀾', 'ハランバンジョウノラン'],
        ['瀰', 'ビ，サンズイニアミダブツノミノキュウジ'],
        ['瀲', 'レン，サンズイニイッテンニシュウレンスルノレン'],
        ['瀬', 'セトモノ ノ セ'],
        ['瀦', 'ミズタマリヲ イミスル チョ'],
        ['瀧', 'タキ ノ キュウジ'],
        ['諺', 'コトワザ，ゲン'],
        ['鋒', 'キュウセンポウ ノ ホウ，ホコサキ'],
        ['諸', 'ショクン ノ ショ'],
        ['諾', 'ショウダクスル ノ ダク'],
        ['諳', 'ソランズル，アン'],
        ['諱', 'イミナ，ゴンベンニイダイナノイノツクリ'],
        ['諷', 'フウシノフウ，ゴンベンニカゼ'],
        ['諫', 'イサメル，カン'],
        ['諮', 'シモンスル ノ シ，ハカル'],
        ['諭', 'キョウユ ノ ユ，サトス'],
        ['諢', 'コン，ゴンベンニグンタイノグン'],
        ['諠', 'ケン，ゴンベンニセンデンスルノセン'],
        ['諡', 'オクリナ，シゴウノシ'],
        ['諦', 'アキラメル，テイカンスル ノ テイ'],
        ['諧', 'カイギャクノカイ'],
        ['諤', 'カンカンガクガクノガク'],
        ['諚', 'ジョウ，ゴンベンニサダメル'],
        ['諛', 'ユ，ゴンベンニシュユノマノユ'],
        ['諞', 'ヘン，ゴンベンニヘンペイソクノヘン'],
        ['諜', 'チョウホウキカン ノ チョウ'],
        ['諒', 'リョウカイスル ノ リョウ，ゴンベン ニ キョウト ノ キョウ'],
        ['鋺', 'カナマリ、カネヘンニウデノツクリ'],
        ['鋸', 'ノコギリ，キョ'],
        ['論', 'ケツロン ノ ロン'],
        ['鋼', 'コウテツ ノ コウ，ハガネ'],
        ['請', 'セイキュウショ ノ セイ，コウ'],
        ['諏', 'スワコ ノ ス'],
        ['諌', 'イサメル，カンゲンスル ノ カン'],
        ['諍', 'イサカイ，ソウ'],
        ['諂', 'ヘツラウ，テン'],
        ['倚', 'イ，ニンベンニキミョウナノキ'],
        ['鋩', 'ボウ、カネヘンニクサカンムリニボウメイスルノボウ'],
        ['談', 'ジョウダンヲイウ ノ ダン'],
        ['諄', 'ジュン，ゴンベンニキョウラクテキノキョウ'],
        ['倖', 'ギョウコウ ノ コウ，ニンベン ニ サチ'],
        ['適', 'テキトウ ノ テキ'],
        ['倔', 'クツ，ニンベンニリクツノクツ'],
        ['ﾆ', 'ニホン ノ ニ'],
        ['遥', 'ヨウハイスル ノ ヨウ，ハルカ'],
        ['崑', 'コンロンサンミャクノコン，ヤマノシタニコンチュウノコン'],
        ['崖', 'ガケ，ダンガイ ノ ガイ'],
        ['崗', 'カコウガンノコウ，オカヤマケンノオカノイタイジ'],
        ['崔', 'サイ，ヤマノシタニフルトリ'],
        ['崕', 'ガケ，ダンガイノガイノイタイジ'],
        ['崚', 'リョウ，ヤマヘンニリョウセンノリョウノツクリ'],
        ['崛', 'クツ，ヤマヘンニリクツノクツ'],
        ['崘', 'コンロンサンミャクノロンノイタイジ'],
        ['崙', 'コンロンサンミャクノロン'],
        ['崟', 'ギン，ヤマノシタニキンヨウビノキン'],
        ['倭', 'ギシワジンデン ノ ワ'],
        ['崇', 'スウハイスル ノ スウ，アガメル'],
        ['崋', 'ワタナベカザンノカ'],
        ['崎', 'ナガサキ ノ サキ'],
        ['遽', 'キュウキョキタクスルノキョ'],
        ['崢', 'ソウ，ヤマヘンニアラソウノキュウジ'],
        ['倦', 'ケンタイカン ノ ケン，ウム'],
        ['崩', 'ホウカイスル ノ ホウ，クズレル'],
        ['選', 'センキョ ノ セン，エラブ'],
        ['ﾅ', 'ナマエ ノ ナ'],
        ['遺', 'イデン ノ イ'],
        ['侘', 'タ，ワビシイノワビ，ニンベンニジュウタクノタク'],
        ['遷', 'サセンスル ノ セン'],
        ['遲', 'チコクスルノチ，オソイノキュウジ'],
        ['遍', 'フヘンテキナ ノ ヘン'],
        ['嗣', 'ヨツギ ノ ツギ，セイシ ノ シ'],
        ['遏', 'アツ，シンニョウニカッショクノカツノツクリ'],
        ['嗤', 'シ，ワラウ，アザワラウ'],
        ['倶', 'クラブ ノ ク'],
        ['嗹', 'レン，クチヘンニレンラクノレン'],
        ['嗾', 'ソウ，ソソノカス，クチヘンニスイゾクカンノゾク'],
        ['嗽', 'ソウ，クチススグ，クチヘンニナツメソウセキノソウノツクリ'],
        ['嗷', 'ゴウ，クチヘンニゴウマンノゴウノツクリ'],
        ['嗇', 'リンショクカノショク'],
        ['嗅', 'キュウカクノキュウ，カグ'],
        ['嗄', 'コエガカレルノカレル，クチヘンニナツ'],
        ['嗚', 'オエツノオ，クチヘンニカラス'],
        ['嗟', 'トッサノサ，クチヘンニサベツカスルノサ'],
        ['嗜', 'シコウヒンノシ，タシナム'],
        ['遇', 'ユウグウスル ノ グウ'],
        ['嗔', 'シン，イカル，クチヘンニマゴコロノマノキュウジ'],
        ['遙', 'ヨウハイスルノヨウ，ハルカノキュウジ'],
        ['霙', 'ミゾレ、アメカンムリニエイユウノエイ'],
        ['違', 'イハン ノ イ，チガウ'],
        ['達', 'ハイタツ ノ タツ'],
        ['震', 'ジシン ノ シン，フルエル'],
        ['舒', 'ジョ，シュクシャノシャノキュウジノミギニヨテイノヨ'],
        ['舐', 'ナメル，シタヘンニウジガミノウジ'],
        ['舗', 'ホソウスル ノ ホ'],
        ['舖', 'ホソウドウロノホノキュウジ'],
        ['舛', 'ソムクヲ イミスル セン，マス'],
        ['舘', 'ヤカタ ノ イタイジ，カン'],
        ['舟', 'ゴエツドウシュウ ノ シュウ，フネ'],
        ['舞', 'カブキ ノ ブ，マウ'],
        ['舜', 'ギョウシュン ノ シュン'],
        ['舂', 'ウスヅク，ショウ'],
        ['舁', 'カゴヲカクノカク'],
        ['與', 'アタエル，キュウヨノヨノキュウジ'],
        ['舅', 'シュウト'],
        ['舊', 'キュウレキノキュウノキュウジ'],
        ['舉', 'センキョノキョ，アゲルノキュウジノイタイジ'],
        ['興', 'キョウミ ノ キョウ'],
        ['舎', 'ガッコウ ノ コウシャ ノ シャ'],
        ['舍', 'ガッコウノコウシャノシャノキュウジ'],
        ['舌', 'シタヲカム ノ シタ'],
        ['舳', 'フネノヘサキ，フネヘンニリユウノユウ'],
        ['舷', 'ウゲンサゲン ノ ゲン，フナバタ'],
        ['舶', 'ハクライヒン ノ ハク'],
        ['舵', 'カジ，ソウダシュ ノ ダ'],
        ['船', 'フネ，センチョウ ノ セン'],
        ['舸', 'カ，フネヘンニカノウセイノカ'],
        ['舫', 'モヤイムスビノモヤイ'],
        ['航', 'コウクウキ ノ コウ'],
        ['舩', 'フネ，フネヘンニオオヤケ'],
        ['舮', 'フネノトモ，フネヘンニトジマリノト'],
        ['般', 'イッパンテキ ノ ハン'],
        ['椁', 'カク，キヘンニキョウラクテキナノキョウ'],
        ['椀', 'キヘン ノ シルワン ノ ワン'],
        ['椅', 'イス ノ イ'],
        ['椄', 'ショウ，キヘンニメカケ'],
        ['椈', 'キク，キヘンニキクノハナノキクノクサカンムリナシ'],
        ['椋', 'ムクドリ ノ ムク，キヘン ニ キョウト ノ キョウ'],
        ['植', 'ショクブツ ノ ショク，ウエル'],
        ['椌', 'コウ，キヘンニクウキノクウ'],
        ['椏', 'ア，キヘンニアネッタイノアノキュウジ'],
        ['椎', 'セキツイ ノ ツイ，シイ'],
        ['椒', 'サンショウウオノショウ'],
        ['椙', 'スギ，キヘン ニ ニチガ フタツ'],
        ['椛', 'カバ，キヘン ニ ハナ'],
        ['椚', 'ショクブツノクヌギ，キヘンニセンモンノモン'],
        ['検', 'ケンサ ノ ケン'],
        ['椡', 'ショクブツノクヌギ，キヘンニトウチャクノトウ'],
        ['椣', 'ショクブツノシデ，キヘンニコクゴジテンノテン'],
        ['椢', 'カイ，キヘンニコクゴノコク'],
        ['椥', 'チ，キヘンニチシキノチ'],
        ['椦', 'ケン，キヘンニジョウシャケンノケンノカタナノカワリニチカラ'],
        ['椨', 'タブノキ，キヘンニトドウフケンノフ'],
        ['椪', 'ショクブツノポンカンノポン，キヘンニナラブ'],
        ['椰', 'ヤシノキノヤ'],
        ['椴', 'トドマツ ノ トド，キヘン ニ シュダン ノ ダン'],
        ['椶', 'シュロノキノシュノイタイジ'],
        ['椹', 'チン，キヘンニハナハダシイ'],
        ['椽', 'テン，キヘンニエンガワノエンノキュウジタイノツクリ'],
        ['椿', 'ショクブツ ノ ツバキ，キヘン ニ ハル'],
        ['篌', 'コウ，タケカンムリニオウコウキゾクノコウ'],
        ['篏', 'ハメキザイクノハメル，カンノイタイジ'],
        ['築', 'ケンチク ノ チク，キズク'],
        ['鮹', 'タコ，ウオヘンニショウゾウガノショウ'],
        ['篋', 'キョウ，タケカンムリニハコガマエノナカニセマイノキュウジノツクリ'],
        ['鮴', 'サカナノゴリ，ウオヘンニヤスム'],
        ['範', 'モハン ノ ハン'],
        ['篇', 'タケカンムリ ノ ヘンシュウスル ノ ヘン'],
        ['篆', 'テンコクノテン'],
        ['篁', 'コウ，タカムラ，タケカンムリニコウタイシノコウ'],
        ['節', 'キセツ ノ セツ，フシ'],
        ['篝', 'コウ，カガリビノカガリ'],
        ['鮭', 'サケ，サカナヘン ニ ツチ フタツ'],
        ['鮮', 'シンセンナ ノ セン，アザヤカ'],
        ['鮨', 'スシ，ウオヘンニウマイ'],
        ['鮪', 'マグロ，サカナヘン ニ アル'],
        ['鮫', 'サメ，サカナヘン ニ マジワル'],
        ['鮠', 'サカナノハヤ，ウオヘンニアブナイ'],
        ['篭', 'カゴ ノ イタイジ，タケカンムリ ニ キョウリュウ ノ リュウ'],
        ['鮟', 'サカナノアンコウノアン，ウオヘンニヤスイ'],
        ['篩', 'フルイニカケルノフルイ'],
        ['鰤', 'サカナノブリ'],
        ['霍', 'カク，アメカンムリニフルトリ'],
        ['篥', 'ヒチリキノリキ，タケカンムリニクリノキノクリ'],
        ['篤', 'キトクジョウタイ ノ トク'],
        ['鮖', 'カジカ，ウオヘンニイシ'],
        ['篦', 'ヘラ'],
        ['篠', 'シノダケ ノ シノ'],
        ['鮒', 'フナ，サカナヘン ニ ツケル'],
        ['鮓', 'スシ，ウオヘンニサクブンノサクノツクリ'],
        ['鮎', 'アユ，サカナヘン ニ ウラナウ'],
        ['篷', 'ホウ，トマ，タケカンムリニダンジョノアイビキノアイ'],
        ['篶', 'エン，タケカンムリニシュウエンノチノエン'],
        ['篳', 'ヒチリキノヒチ，タケカンムリニヒッキョウノヒツ'],
        ['鮃', 'ヒラメ，ウオヘンニタイラ'],
        ['恚', 'イ，ツチフタツノシタニココロ'],
        ['恙', 'ヨウ，ツツガナククラスノツツガ'],
        ['恟', 'キョウ，リッシンベンニカラダノムネノツクリ'],
        ['恒', 'コウオンドウブツ ノ コウ，ツネニ'],
        ['鰮', 'オン，ウオヘンニトラワレビトノシュウノシタニサラ'],
        ['恐', 'キョウフ ノ キョウ，オソロシイ'],
        ['恕', 'カンジョ ノ ジョ，ユルス'],
        ['恊', 'キョウリョクスルノキョウノイタイジ'],
        ['恋', 'コイビト ノ コイ，レン'],
        ['恍', 'コウコツノヒトノコウ'],
        ['恂', 'ジュン，リッシンベンニゲジュンノジュン'],
        ['恃', 'キョウジヲタモツノジ'],
        ['恁', 'ジン，セキニンノニンノシタニココロ'],
        ['恆', 'コウオンドウブツノコウ，ツネニノキュウジ'],
        ['^', 'ベキジョウ'],
        ['恰', 'カップクガヨイ ノ カツ，アタカモ'],
        ['恷', 'キュウ，キュウジツノキュウノシタニココロ'],
        ['恵', 'メグム，チエ ノ エ'],
        ['恪', 'カク，リッシンベンニカクジノカク'],
        ['恫', 'ドウカツスルノドウ'],
        ['恨', 'ウラミ，ツウコン ノ コン'],
        ['恩', 'オンガエシ ノ オン'],
        ['息', 'イキヲスル ノ イキ，ソク'],
        ['恬', 'ムヨクテンタンノテン'],
        ['恭', 'キョウジュン ノ キョウ，ウヤウヤシイ'],
        ['恢', 'テンモウカイカイ ノ カイ'],
        ['恣', 'シイテキナノシ，ホシイママ'],
        ['恠', 'カイブツノカイノイタイジ，リッシンベンニソンザイノザイ'],
        ['恤', 'ジュツ，リッシンベンニケツエキノケツ'],
        ['恥', 'ハジヲカク ノ ハジ'],
        ['鰺', 'サカナノアジノイタイジ，カタカナノムガミッツ'],
        ['寃', 'エンザイヲハラスノエンノイタイジ'],
        ['寂', 'セイジャク ノ ジャク，サビシイ'],
        ['寇', 'モウコシュウライノゲンコウノコウ'],
        ['密', 'ヒミツ ノ ミツ'],
        ['寅', 'トラ，ジュウニシ ノ トラ'],
        ['寄', 'キフスル ノ キ，ヨル'],
        ['寉', 'トリノツルノイタイジ，ウカンムリニフルトリ'],
        ['富', 'ホウフ ノ フ，トム'],
        ['寓', 'グウワ ノ グウ'],
        ['寒', 'アツイサムイ ノ サムイ，カン'],
        ['寐', 'ムビノビ，ネムル'],
        ['寔', 'ショク，ウカンムリニゼヒノゼ'],
        ['寛', 'カンダイナ ノ カン，ヒロイ'],
        ['鰆', 'サワラ，ウオヘンニハル'],
        ['察', 'ケイサツ ノ サツ'],
        ['寞', 'セキバクノバク'],
        ['寝', 'シンシツ ノ シン，ネル'],
        ['寢', 'シンシツノシン，ネルノキュウジ'],
        ['寡', 'カモクナ ノ カ，スクナイ'],
        ['寧', 'テイネイ ノ ネイ'],
        ['實', 'ジツリョクノジツ，ミノキュウジ'],
        ['寥', 'セキリョウカンノリョウ'],
        ['寤', 'サメルヲイミスルゴ'],
        ['寫', 'シャシンノシャ，ウツスノキュウジ'],
        ['審', 'シンサスル ノ シン'],
        ['寨', 'サイ，サムイノシタニテンノカワリニモクザイノモク'],
        ['寮', 'ガクセイリョウ ノ リョウ'],
        ['寳', 'ホウセキノホウ，タカラノキュウジノイタイジ'],
        ['寰', 'カン，ウカンムリニシゼンカンキョウノカンノツクリ'],
        ['寶', 'ホウセキノホウ，タカラノキュウジ'],
        ['寵', 'チョウアイヲウケル ノ チョウ'],
        ['寺', 'テラ，ジイン ノ ジ'],
        ['寸', 'スンポウ ノ スン'],
        ['寿', 'コトブキ，ジュミョウ ノ ジュ'],
        ['対', 'ハンタイ ノ タイ'],
        ['鰌', 'ドジョウ，ウオヘンニシュウチョウサンノシュウ'],
        ['鰓', 'エラ，ウオヘンニオモウ'],
        ['4', 'ヨン'],
        ['岔', 'タ，ハンブンノブンノシタニヤマ'],
        ['岐', 'ブンキテン ノ キ'],
        ['岑', 'シン，ヤマノシタニコンゲツノコン'],
        ['鰛', 'オン，ウオヘンニオンセンノオンノツクリ'],
        ['岌', 'キュウ，ヤマノシタニフキュウスルノキュウ'],
        ['岶', 'ハク，ヤマヘンニシロイ'],
        ['岷', 'ビン，ヤマヘンニコクミンノミン'],
        ['岱', 'ダイヒョウ ノ ダイ ノ シタニ ヤマ，タイ'],
        ['岳', 'サンガク ノ ガク，タケ'],
        ['岼', 'ユリ，ヤマヘンニタイラ'],
        ['岾', 'ハケ，ヤマヘンニドクセンスルノセン'],
        ['岸', 'カイガン ノ ガン，キシ'],
        ['岻', 'チ，ヤマヘンニヒクイノツクリ'],
        ['岡', 'オカヤマケン ノ オカ'],
        ['岬', 'エリモミサキ ノ ミサキ'],
        ['岨', 'ケンソ ノ ソ，ヤマヘン ニ カツ'],
        ['岩', 'イワ，ガンセキ'],
        ['岫', 'シュウ，イワアナ，ヤマヘンニリユウノユウ'],
        ['鹵', 'ロカクノロ，シオ'],
        ['鹸', 'コナセッケン ノ ケン'],
        ['鮗', 'コノシロ，ウオヘンニフユ'],
        ['鮑', 'アワビ，ウオヘンニツツム'],
        ['饐', 'イ，スエル'],
        ['饕', 'トウ，ムサボル'],
        ['鳶', 'トリ ノ トビ'],
        ['Ｔ', 'ティー タイム'],
        ['饉', 'キキンニミマワレルノキン'],
        ['饋', 'キ，オクル，ショクヘンニキチョウヒンノキ'],
        ['饌', 'セン，ソナエル，ショクヘンニエラブノツクリ'],
        ['炮', 'ホウ，ヒヘンニツツム'],
        ['炯', 'ケイガンノシノケイ，ヒヘンニドウガマエニクチ'],
        ['炬', 'キョ，カガリビ，ヒヘンニキョダイナノキョ'],
        ['炭', 'セキタン ノ タン，スミ'],
        ['炳', 'ヘイ，ヒヘンニコウオツヘイノヘイ'],
        ['為', 'コウイ ノ イ，タメ'],
        ['炸', 'バクダンガサクレツスルノサク'],
        ['点', 'テンスウ ノ テン'],
        ['饂', 'ドン，ウドンノイチモジメ'],
        ['炎', 'エンジョウスル ノ エン，ホノオ'],
        ['饅', 'マンジュウノマン'],
        ['炊', 'スイハンキ ノ スイ，タク'],
        ['炉', 'ダンロ ノ ロ'],
        ['炒', 'イタメル，チャーハンノイチモジメ'],
        ['炙', 'ジンコウニカイシャスルノシャ，アブル'],
        ['譽', 'メイヨノヨ、ホマレノキュウジ'],
        ['鉗', 'カンシブンベンノカン'],
        ['鉐', 'セキ、カネヘンニイシ'],
        ['譴', 'ケンセキショブンノケン、セメル'],
        ['護', 'ホゴシャ ノ ゴ'],
        ['議', 'カイギシツ ノ ギ'],
        ['譱', 'ゼンアクノゼン，ヨイノイタイジ'],
        ['譲', 'ジョウホスル ノ ジョウ，ユズル'],
        ['鉚', 'リュウ、カネヘンニジュウニシノウ'],
        ['譬', 'タトエバナシノタトエ、ヒ'],
        ['鉄', 'テツドウ ノ テツ'],
        ['譯', 'ツウヤクスルノヤク、ワケノキュウジ'],
        ['譫', 'センモウジョウタイノセン'],
        ['警', 'ケイサツ ノ ケイ'],
        ['鉉', 'ゲン、カネヘンニゲンマイノゲン'],
        ['鉈', 'ナタ'],
        ['鉋', 'ダイクドウグノカンナ'],
        ['譜', 'フメン ノ フ'],
        ['譟', 'ソウ、ゴンベンニソウジュウスルノソウノツクリ'],
        ['識', 'チシキ ノ シキ'],
        ['譚', 'タン，ゴンベンニニシノシタニハヤオキノハヤイ'],
        ['譛', 'シン，ゴンベンニフリカエノカエ'],
        ['譖', 'シン，ゴンベンニセンエツナガラノセンノツクリ'],
        ['鉾', 'カマボコ ノ ホコ'],
        ['譌', 'オクニナマリノナマリノイタイジ'],
        ['鉤', 'コウ、カギ、カネヘンニハイクノク'],
        ['譎', 'ケツ、ゴンベンニタチバナノツクリ'],
        ['譏', 'キ、ゴンベンニキカガクノキ'],
        ['u', 'ユー ユージュアリー'],
        ['證', 'ショウ、ショウメイショノショウノキュウジ'],
        ['鉢', 'ハチウエ ノ ハチ'],
        ['譁', 'カ、ゴンベンニチュウカナベノカ'],
        ['z', 'ゼット ズー'],
        ['檎', 'リンゴ ノ ゴ'],
        ['檍', 'ヨク，キヘンニイミガナイノイ'],
        ['檀', 'ダンカ ノ ダン'],
        ['檄', 'ゲキヲトバスノゲキ'],
        ['檜', 'ショクブツノヒノキ，キヘンニカイシャノカイノキュウジ'],
        ['檐', 'エン，サンタンタルノタンノツクリ'],
        ['檗', 'オウバクシュウノバク，カベノツチノカワリニショクブツノキ'],
        ['檪', 'クヌギ，レキ，キヘンニオンガクカノガク'],
        ['檮', 'トウ，キヘンニコトブキノキュウジ'],
        ['檬', 'モウ，レモンノニモジメ'],
        ['檣', 'ショウ，ホバシラ'],
        ['檢', 'ケンサスルノケンノキュウジ'],
        ['檠', 'ケイ，ソンケイスルノケイノシタニモクザイノモク'],
        ['檻', 'セッカンスルノカン，オリ'],
        ['檸', 'ネイ，レモンノイチモジメ'],
        ['檳', 'ショクブツノビンロウジュノビン'],
        ['K', 'ケイ キッチン'],
        ['啣', 'クツワノイタイジ，クチヘンニオロシウリノオロシ'],
        ['啼', 'テイ，ナク，クチヘンニテイオウノテイ'],
        ['啾', 'シュウ，ナク，クチヘンニアキマツリノアキ'],
        ['啻', 'タダ，テイオウノテイノシタニクチ'],
        ['啌', 'コウ，クチヘンニクウキノクウ'],
        ['問', 'モンダイ ノ モン，トウ'],
        ['啅', 'タク，クチヘンニタッキュウノタク'],
        ['啄', 'タクボク ノ タク，ツイバム'],
        ['商', 'ショウバイ ノ ショウ'],
        ['啀', 'イガミアウノイガム，ガイ'],
        ['啝', 'カ，クチヘンニワフクノワ'],
        ['啜', 'ススル，セツ'],
        ['啗', 'タン，クラウ，クチヘンニアンコノアンノツクリ'],
        ['啖', 'タンカヲキルノタン'],
        ['啓', 'ケイハツ ノ ケイ'],
        ['銑', 'センテツ ノ セン，カネヘン ニ サキ'],
        ['銖', 'シュ、カネヘンニシュイロノシュ'],
        ['孅', 'セン，オンナヘンニクジビキノクジノシタガワ'],
        ['孀', 'ソウ，ヤモメ，オンナヘンニシモバシラノシモ'],
        ['孃', 'オジョウサマノジョウノキュウジ'],
        ['銜', 'カン、ギョウガマエニキンヨウビノキン'],
        ['孕', 'ハラム，ヨウ'],
        ['孔', 'コウシモウシ ノ コウ'],
        ['字', 'ローマジ ノ ジ'],
        ['孑', 'ノコリヲイミスルケツ，コドモノコノミギウデガカケタカタチ'],
        ['子', 'コドモ ノ コ'],
        ['孝', 'オヤフコウ ノ コウ'],
        ['孜', 'ツトム，コヘン ニ ノブン'],
        ['孟', 'コウシモウシ ノ モウ'],
        ['存', 'ソンザイスル ノ ソン'],
        ['孛', 'ボツ，ボッパツスルノヒダリガワ'],
        ['孚', 'フ，ハグクム，ツメノシタニコドモノコ'],
        ['孥', 'ド，ヤッコノシタニコドモノコ'],
        ['孤', 'コドク ノ コ'],
        ['学', 'マナブ，ガクシュウ ノ ガク'],
        ['季', 'キセツフウ ノ キ'],
        ['孩', 'ガイ，コヘンニジュウニシノイノシシ'],
        ['孫', 'マゴノテ ノ マゴ，ソン'],
        ['孵', 'タマゴガフカスルノフ'],
        ['→', 'ミギヤジルシ'],
        ['↑', 'ウエヤジルシ'],
        ['←', 'ヒダリヤジルシ'],
        ['孱', 'サン，シカバネニコドモノコガミッツ'],
        ['孰', 'シュク，ガクシュウジュクノジュクカラツチヲトッタカタチ'],
        ['孳', 'ジ，ジヒブカイノジノココロノカワリニコドモノコ'],
        ['學', 'マナブ，ガクシュウノガクノキュウジ'],
        ['孺', 'ジュ，コヘンニヒツジュヒンノジュ'],
        ['芒', 'ボウ、クサカンムリニボウメイスルノボウ'],
        ['芝', 'シバフ ノ シバ'],
        ['芟', 'サン、クサカンムリニルマタ'],
        ['芙', 'ショクブツ ノ フヨウ ノ フ'],
        ['芍', 'ショクブツノシャクヤクノシャク'],
        ['芋', 'ヤキイモ ノ イモ'],
        ['!', 'カンタンフ'],
        ['花', 'ハナ，カビン ノ カ'],
        ['芳', 'カンバシイ，ホウコウザイ ノ ホウ'],
        ['芽', 'ハツガ ノ ガ，キノメ ノ メ'],
        ['芹', 'ショクブツ ノ セリ'],
        ['芸', 'ゲイジュツ ノ ゲイ'],
        ['芻', 'スウ、ハンスウドウブツノスウ'],
        ['芥', 'アクタガワショウ ノ アクタ'],
        ['芦', 'アシ，クサカンムリ ニ トジマリ ノ ト'],
        ['芭', 'バショウ ノ バ'],
        ['芬', 'アクシュウフンプンノフン'],
        ['芯', 'エンピツ ノ シン ノ シン'],
        ['芫', 'ゲン、クサカンムリニゲンキナノゲン'],
        ['謖', 'ナイテバショクヲキルノショク'],
        ['週', 'イッシュウカン ノ シュウ'],
        ['逵', 'キ、シンニョウニリクジョウノリクノツクリ'],
        ['賽', 'サイセンバコノサイ'],
        ['逸', 'イツダツスル'],
        ['銭', 'ツリセン ノ セン'],
        ['逡', 'シュンジュンスルノシュン'],
        ['穗', 'イナホノホノキュウジ'],
        ['穐', 'アキ，ノギヘン ニ カメ'],
        ['穆', 'ノギヘン ノ ボク'],
        ['穃', 'ヨウ，ノギヘンニヨウセキノヨウ'],
        ['穂', 'イナホ ノ ホ'],
        ['穀', 'コクモツ ノ コク'],
        ['穏', 'ヘイオン ノ オン，オダヤカ'],
        ['穎', 'スグレルヲ イミスル エイ，ホサキ，ノギヘン'],
        ['積', 'メンセキ ノ セキ，ツモル'],
        ['穉', 'ヨウチエンノチノイタイジ'],
        ['究', 'ケンキュウ ノ キュウ'],
        ['穴', 'アナヲホル ノ アナ'],
        ['穰', 'ゴコクホウジョウノジョウノキュウジ'],
        ['穿', 'ウガツ，センコウスル ノ セン'],
        ['逮', 'タイホジョウ ノ タイ'],
        ['穽', 'オトシアナヲイミスルカンセイノセイ'],
        ['空', 'クウキ ノ クウ，ソラ'],
        ['穹', 'ソウキュウノキュウ，アナカンムリニユミ'],
        ['穣', 'ゴコクホウジョウ ノ ジョウ'],
        ['穢', 'ケガレ，ワイ'],
        ['穡', 'ショク，ノギヘンニリンショクノショク'],
        ['穫', 'シュウカクスル ノ カク，トリイレル'],
        ['穩', 'ヘイオンノオン，オダヤカノキュウジ'],
        ['鬆', 'ショウ，カミガシラニマツノキノマツ'],
        ['懐', 'カイチュウデントウ ノ カイ，フトコロ'],
        ['懌', 'エキ，リッシンベンニサワノキュウジノツクリ'],
        ['懍', 'リン，リッシンベンニリンギスルリン'],
        ['懈', 'ケタイスルノケ，リッシンベンニカイケツスルノカイ'],
        ['應', 'オウエンスルノオウノキュウジ'],
        ['懊', 'オウノウスルノオウ，ナヤム'],
        ['懋', 'ボウ，ハヤシノアイダニムジュンスルノム，ソノシタニココロ'],
        ['鬟', 'カン，カミガシラニジュンカンスルノカンノツクリ'],
        ['懆', 'ソウ，リッシンベンニソウジュウスルノソウノツクリ'],
        ['懇', 'コンシンカイ ノ コン，ネンゴロ'],
        ['鬚', 'シュ，ヒゲ，カミガシラニヒッスノス'],
        ['鬘', 'マン，カミガシラニマンダラノマン'],
        ['懃', 'インギンブレイノギン'],
        ['懼', 'キョウクスルノク，リッシンベンニメガフタツノシタニフルトリ'],
        ['懽', 'カン，リッシンベンニカンガイヨウスイノカンノツクリ'],
        ['懾', 'ショウ，リッシンベンニミミミッツ'],
        ['懿', 'ダイヨンダイイトクテンノウノイ'],
        ['懸', 'イッショウケンメイ ノ ケン，カケル'],
        ['鬣', 'リョウ，カミガシラニリョウジュウノリョウノキュウジノツクリ'],
        ['懺', 'ザンゲスルノザン，クイル'],
        ['懴', 'ザンゲスルノザン，クイルノイタイジ'],
        ['鬯', 'チョウ，ブシュノニオイザケ'],
        ['懶', 'ランダナノラン，リッシンベンニイライスルノライノキュウジ'],
        ['懷', 'カイチュウデントウノカイ，フトコロノキュウジ'],
        ['鬪', 'トウギュウシノトウノキュウジ，タタカウ'],
        ['懲', 'コラシメル，チョウバツ ノ チョウ'],
        ['鬩', 'ゲキ，トウガマエニニラムノツクリ'],
        ['鬲', 'カク，ヘダタリノツクリ'],
        ['鬱', 'ユウウツナノウツ'],
        ['懦', 'キョウダノダ，ヨワイ，リッシンベンニヒツジュヒンノジュ'],
        ['鬻', 'シュク，カユノシタニヘダテルノツクリ'],
        ['懣', 'フンマンヤルカカタナシノマン'],
        ['猩', 'ショウコウネツノショウ，ケモノヘンニホシ'],
        ['猪', 'イノシシ，チョトツモウシン ノ チョ'],
        ['猫', 'ヤマネコ ノ ネコ'],
        ['献', 'ケンケツスル ノ ケン'],
        ['猯', 'タン，ケモノヘンニマッタンノタンノツクリ'],
        ['猥', 'ヒワイナノワイ'],
        ['猾', 'コウカツナジンブツノカツ，ケモノヘンニホネ'],
        ['猿', 'ルイジンエン ノ エン，サル'],
        ['猴', 'コウ，ケモノヘンニオウコウキゾクノコウ'],
        ['猶', 'ユウヨスル ノ ユウ'],
        ['猷', 'ハカリゴトヲ イミスル ユウ'],
        ['猊', 'ゲイカノゲイ'],
        ['猛', 'モウショ ノ モウ'],
        ['猜', 'サイギシンノサイ'],
        ['猝', 'ソツ，ケモノヘンニソツギョウノソツ'],
        ['猟', 'リョウジュウ ノ リョウ'],
        ['猖', 'ショウケツヲキワメルノショウ'],
        ['猗', 'イ，ケモノヘンニキミョウナノキ'],
        ['鏝', 'ハンダゴテノコテ'],
        ['鏘', 'ショウ、カネヘンニショウグンノショウノキュウジ'],
        ['鏗', 'コウ、カネヘンニケンジツナノケン'],
        ['鏖', 'オウ、ドウブツノシカノシタニキンヨウビノキン'],
        ['逅', 'メグリアイノカイコウノコウ'],
        ['鏑', 'カブラヤ ノ カブラ'],
        ['鏐', 'リョウ、カネヘンニニカワノツクリ'],
        ['鏈', 'レン、カネヘンニレンラクノレン'],
        ['鏃', 'ユミヤノヤジリ'],
        ['b', 'ビー ボーイ'],
        ['透', 'トウメイナ ノ トウ，スケル'],
        ['鏨', 'コウグノタガネ'],
        ['鏥', 'シュウ，カネヘンニヤド'],
        ['鏤', 'ルキンノル，チリバメル'],
        ['鏡', 'カガミ，ソウガンキョウ ノ キョウ'],
        ['㍾', 'メイジ'],
        ['㍽', 'タイショウ'],
        ['㍼', 'ショウワ'],
        ['㍻', 'ヘイセイ'],
        ['㍍', 'カナメートル'],
        ['㍊', 'カナミリバール'],
        ['㍉', 'カナミリ'],
        ['㍗', 'カナワット'],
        ['㍑', 'カナリットル'],
        ['尚', 'オショウサン ノ ショウ'],
        ['尖', 'トガル，センエイ ノ セン'],
        ['尓', 'ジ，アミダブツノミノツクリ'],
        ['少', 'オオイスクナイ ノ ショウ，ショウ'],
        ['導', 'ユウドウ ノ ドウ，ミチビク'],
        ['小', 'ダイショウ ノ ショウ，チイサイ'],
        ['對', 'ハンタイノタイノキュウジ'],
        ['尊', 'ソンケイスル ノ ソン，トウトブ'],
        ['尋', 'ジンモンスル ノ ジン，タズネル'],
        ['專', 'センモンノセンノキュウジ'],
        ['尉', 'リクグンタイイ ノ イ'],
        ['将', 'ショウグン ノ ショウ'],
        ['將', 'ショウグンノショウノキュウジ'],
        ['射', 'ハンシャ ノ シャ'],
        ['尅', 'ゲコクジョウノコクノイタイジ，コクフクスルノコクニスン'],
        ['専', 'センモン ノ セン'],
        ['封', 'フウトウ ノ フウ'],
        ['尾', 'ビコウスル ノ ビ，オッポ ノ オ'],
        ['尿', 'トウニョウビョウ ノ ニョウ'],
        ['尼', 'アマデラ ノ アマ，ニ'],
        ['尽', 'ジンリョクスル ノ ジン，ツクス'],
        ['尺', 'シャクハチ ノ シャク'],
        ['尻', 'シリゴミスル ノ シリ'],
        ['尸', 'シ，ブシュノシカバネ'],
        ['尹', 'イン，イセエビノイノツクリ'],
        ['就', 'シュウショク ノ シュウ，ツク'],
        ['尭', 'ギョウシュン ノ ギョウ'],
        ['尨', 'ボウ，ムクイヌ'],
        ['尤', 'セツゾクシ ノ モットモ'],
        ['尢', 'オウ，ブシュノマゲアシ'],
        ['尠', 'セン，ハナハダシイニスクナイ'],
        ['8', 'ハチ'],
        ['哲', 'テツガク ノ テツ'],
        ['哽', 'コウ，クチヘンニサラシナノサラ'],
        ['哺', 'ホニュウルイノホ'],
        ['哦', 'ガ，クチヘンニガマンノガ'],
        ['哥', 'ウタ，カ，カノウセイノカヲカサネル'],
        ['哢', 'ロウ，クチヘンニホンロウスルノロウ'],
        ['員', 'ガッキュウイイン ノ イン'],
        ['哮', 'ホエル，ホウコウスルノコウ'],
        ['哭', 'ドウコクスルノコク，ナク'],
        ['哩', 'タンイ ノ マイル，クチヘン ニ サト'],
        ['哨', 'ゼンショウセン ノ ショウ'],
        ['哘', 'サソウ，クチヘンニリョコウノコウ'],
        ['哇', 'アイ，クチヘンニツチフタツ'],
        ['哄', 'コウ，クチヘンニキョウツウノキョウ'],
        ['哂', 'シン，ワラウ，クチヘンニニシヒガシノニシ'],
        ['品', 'シナモノ ノ シナ，ヒン'],
        ['哀', 'キドアイラク ノ アイ，アワレム'],
        ['哉', 'ゼンザイ ノ サイ，カナ'],
        ['哈', 'ゴウ，クチヘンニゴウカクノゴウ'],
        [
            '剱',
            'ケン，ツルギノイタイジ，ジョウヨウカンジノケンノツクリガヤイバノイタイジ',
        ],
        ['茜', 'アカネイロ ノ アカネ'],
        ['茘', 'クダモノノレイシノレイ，クサカンムリニカタナガミッツ'],
        ['茗', 'ミョウガノミョウ，クサカンムリニナマエノナ'],
        ['茖', 'カク、クサカンムリニオノオノ'],
        ['む', 'ムシ ノ ム'],
        ['茎', 'クキ，チカケイ ノ ケイ'],
        ['め', 'メガネ ノ メ'],
        ['茉', 'マツ、クサカンムリニネンマツノマツ'],
        ['ゆ', 'ユカタ ノ ユ'],
        ['茅', 'カヤブキ ノ カヤ'],
        ['茄', 'クサカンムリ ニ クワエル ノ カ，ナス'],
        ['范', 'ハン、クサカンムリニカワガハンランスルノハン'],
        ['茂', 'ハンモ ノ モ，シゲル'],
        ['剴', 'ガイ，ヨロイノツクリニリットウ'],
        ['茹', 'ジョ，ユデル，クサカンムリニトツジョノジョ'],
        ['茸', 'マツタケ ノ タケ，キノコ'],
        ['茶', 'チャイロ ノ チャ'],
        ['茵', 'イン、クサカンムリニインガオウホウノイン'],
        ['茴', 'ウイキョウノウイ、クサカンムリニカイスウケンノカイ'],
        ['茲', 'シ、クサカンムリニイトガシラフタツ'],
        ['茱', 'シュ，クサカンムリニシュイロノシュ'],
        ['茯', 'ブクリョウノブク，クサカンムリニキフクニトムノフク'],
        ['茫', 'ボウゼンジシツノボウ，クサカンムリニサンズイニボウメイスルノボウ'],
        ['茨', 'イバラキケン ノ イバラ'],
        ['茣', 'ゴザノゴ，クサカンムリニゴフクヤノゴノキュウジケイ'],
        ['ゎ', 'チイサイ ワカメ ノ ワ'],
        ['剿', 'ハチノスノスニリットウ'],
        ['剣', 'ケンドウ ノ ケン，ツルギ'],
        ['矼', 'コウ，イシヘンニズガコウサクノコウ'],
        ['紀', 'キゲンゼン ノ キ'],
        ['紂', 'インノチュウオウノチュウ，イトヘンニスンポウノスン'],
        ['約', 'ヤクソク ノ ヤク'],
        ['紅', 'コウチャ ノ コウ，ベニ'],
        ['紆', 'ウヨキョクセツノウ'],
        ['紊', 'ブン，フウキビンランノビン'],
        ['紋', 'モンショウ ノ モン'],
        ['納', 'ノウゼイ ノ ノウ，オサメル'],
        ['紐', 'コシヒモ ノ ヒモ'],
        ['純', 'ジュンスイ ノ ジュン'],
        ['紕', 'ヒ，イトヘンニヒカクスルノヒ'],
        ['紗', 'オリモノ ノ シャ，イトヘン ニ スクナイ'],
        ['紘', 'ハッコウイチウ ノ コウ'],
        ['紙', 'ガヨウシ ノ シ，カミ'],
        ['級', 'サイコウキュウ ノ キュウ'],
        ['紛', 'フンシツスル ノ フン，マギレル'],
        ['紜', 'ウン，イトヘンニデンセツノデンノツクリ'],
        ['素', 'ソシツ ノ ソ'],
        ['紡', 'ボウセキ ノ ボウ，ツムグ'],
        ['索', 'サクイン ノ サク'],
        ['紫', 'シガイセン ノ シ，ムラサキ'],
        ['紬', 'ツムギ，イトヘン ニ リユウ ノ ユウ'],
        ['紮', 'ケッサツノサツ，ナフダノフダノシタニイト'],
        ['累', 'ルイセキスル ノ ルイ'],
        ['細', 'ホソイフトイ ノ ホソイ，サイ'],
        ['紲', 'セツ，イトヘンニセカイノセ'],
        ['紳', 'シンシテキナ ノ シン'],
        ['紵', 'チョ，イトヘンニチョキンノチョノツクリ'],
        ['紹', 'ジコショウカイ ノ ショウ'],
        ['紺', 'コンイロ ノ コン'],
        ['紿', 'タイ，イトヘンニダイドコロノダイ'],
        ['樌', 'カン，キヘンニイッカンスルノカン'],
        ['樋', 'ヒグチイチヨウ ノ ヒ，トイ'],
        ['樊', 'トリカゴノイミノハンロウノハン'],
        ['樅', 'モミノキノモキ'],
        ['樂', 'オンガクノガク，タノシイノキュウジ'],
        ['樟', 'ショウノウ ノ ショウ，クス'],
        ['樞', 'チュウスウノスウノキュウジ'],
        ['標', 'モクヒョウ ノ ヒョウ'],
        ['樛', 'キュウ，キヘンニニカワノツクリ'],
        ['樔', 'ソウ，キヘンニハチノスノス'],
        ['樗', 'ショクブツ ノ オウチ，チョ'],
        ['樓', 'マテンロウノロウノキュウジ'],
        ['剔', 'テキシュツスルノテキ，ボウエキノエキ，ヤサシイニリットウ'],
        ['樮', 'ホクソ，キヘンニニシノシタニカヨウビノカ'],
        ['権', 'ケンリ ノ ケン'],
        ['樫', 'カシノキ ノ カシ'],
        ['横', 'タテヨコ ノ ヨコ，オウ'],
        ['剖', 'カイボウスル ノ ボウ'],
        ['模', 'モケイ ノ モ'],
        ['樣', 'カミサマノサマ，ヨウノキュウジ'],
        ['樢', 'チョウ，キヘンニヤキトリノトリ'],
        ['樽', 'タルザケ ノ タル'],
        ['樹', 'ジュモク ノ ジュ'],
        ['樸', 'ボク，キヘンニイチニンショウノボクノツクリ'],
        ['樺', 'シラカバ ノ カバ'],
        ['y', 'ワイ ヤング'],
        ['樶', 'サイ，キヘンニサイダイサイショウノサイ'],
        ['剛', 'シツジツゴウケン ノ ゴウ'],
        ['站', 'ヘイタンセンノタン'],
        ['竚', 'チョ，タツヘンニウカンムリニイッチョウメノチョウ'],
        ['竝', 'ナラブノキュウジ'],
        ['竜', 'キョウリュウ ノ リュウ，タツ'],
        ['竟', 'ヒッキョウノキョウ'],
        ['竓', 'リョウノタンイノミリリットル'],
        ['竒', 'キミョウナノキノイタイジ'],
        ['竕', 'リョウノタンイノデシリットル'],
        ['竈', 'カマド'],
        ['立', 'ドクリツ ノ リツ，タツ'],
        ['竊', 'セットウハンノセツノキュウジ'],
        ['竍', 'リョウノタンイノデカリットル'],
        ['竏', 'リョウノタンイノキロリットル'],
        ['竃', 'カマド'],
        ['剞', 'キ，キミョウナノキニリットウ'],
        ['竅', 'キョウ，アナカンムリニシゲキノゲキノツクリ'],
        ['竄', 'カイザンスルノザン，アナカンムリニネズミ'],
        ['竇', 'トウ，アナカンムリニハンバイノバイノキュウジ'],
        ['竹', 'タケ，ショウチクバイ ノ チク'],
        ['竸', 'キョウソウスルノキョウ，キソウノイタイジ'],
        ['竺', 'テンジク ノ ジク'],
        ['竿', 'サオダケ ノ サオ，カン'],
        ['竰', 'リョウノタンイノセンチリットル'],
        ['剃', 'ソル，テイハツ ノ テイ'],
        ['競', 'キョウソウスル ノ キョウ，キソウ'],
        ['竪', 'タテゴト ノ タテ'],
        ['竭', 'ケツ，タツヘンニカッショクノカツノツクリ'],
        ['端', 'タンマツ ノ タン，ハシ'],
        ['竡', 'リョウノタンイノヘクトリットル'],
        ['章', 'ブンショウヲヨム ノ ショウ'],
        ['竣', 'シュンコウシキ ノ シュン'],
        ['竢', 'シ，タツヘンニアイサツノアイノツクリ'],
        ['童', 'グリムドウワ ノ ドウ'],
        ['竦', 'スクム，ショウ'],
        ['慧', 'ケイガン ノ ケイ，サトイ'],
        ['慥', 'ソウ，リッシンベンニカイゾウスルノゾウ'],
        ['慢', 'ジマンスル ノ マン'],
        ['慣', 'シュウカン ノ カン，ナレル'],
        ['慮', 'エンリョスル ノ リョ'],
        ['慯', 'ショウ，リッシンベンニキリキズノキズ'],
        ['慫', 'ショウ，フクジュウスルノジュウノキュウジノシタニココロ'],
        ['慨', 'カンガイムリョウ ノ ガイ'],
        ['慶', 'ケイチョウキュウカ ノ ケイ，ヨロコブ'],
        ['慷', 'ヒフンコウガイノコウ，リッシンベンニイエヤスノヤス'],
        ['慴', 'ショウ，リッシンベンニレンシュウスルノシュウ，ナラウ'],
        ['慵', 'ヨウ，リッシンベンニボンヨウナノヨウ'],
        ['慳', 'ケン，リッシンベンニケンジツナノケン'],
        ['慰', 'イシャリョウ ノ イ，ナグサメル'],
        ['慱', 'タン，リッシンベンニセンモンノセンノキュウジ'],
        ['慾', 'ヨクボウ ノ ヨク，シタゴコロアリ'],
        ['剋', 'ゲコクジョウノコク，コクフクスルノコクニリットウ'],
        ['風', 'カゼ，タイフウ ノ フウ'],
        ['慇', 'インギンナノイン'],
        ['慄', 'センリツスルノリツ'],
        ['削', 'サクゲンスル ノ サク，ケズル'],
        ['慂', 'ヨウ，コワクダニノワクノシタニココロ'],
        ['颯', 'サッソウトシタノサツ'],
        ['慎', 'シンチョウナ ノ シン，ツツシム'],
        ['ゾ', 'ゾウキン ノ ゾ'],
        ['慌', 'アワテル，キョウコウ ノ コウ'],
        ['慍', 'ウン，リッシンベンニオンセンノオンノキュウジタイノツクリ'],
        ['慊', 'ケン，リッシンベンニカネル'],
        ['態', 'タイド ノ タイ'],
        ['慈', 'ジヒブカイ ノ ジ，イツクシム'],
        ['タ', 'タバコ ノ タ'],
        ['慕', 'ボジョウ ノ ボ，シタウ'],
        ['慓', 'ヒョウ，リッシンベンニトウヒョウスルノヒョウ'],
        ['慟', 'ドウコクスルノドウ'],
        ['慝', 'トク，トクメイノトクノシタニココロ'],
        ['慚', 'ザンキニタエナイノザン，リッシンベン'],
        ['慘', 'ザンパイスルノザン，ミジメノキュウジ'],
        ['慙', 'ザンキニタエナイノザン，シタゴコロ'],
        ['嫌', 'キラウ，ケンオ ノ ケン'],
        ['嫋', 'ジョウ，タオヤカ'],
        ['嫉', 'シットスル ノ シツ，ソネム'],
        ['嫂', 'ソウ，アニヨメ'],
        ['Å', 'オングストローム'],
        ['嫗', 'オウ，オンナヘンニクベツノクノキュウジ'],
        ['嫖', 'ヒョウ，オンナヘンニトウヒョウノヒョウ'],
        ['嫐', 'ドウ，ナヤマシイ，オンナ，オトコ，オンナ'],
        ['嫩', 'ドン，オンナヘンニセイリスルノセイノウエガワ'],
        ['嫦', 'ジョウ，オンナヘンニヒジョウグチノジョウ'],
        ['嫣', 'エン，オンナヘンニシュウエンノチノエン'],
        ['嫡', 'チャクナン ノ チャク'],
        ['№', 'ナンバー'],
        ['嫻', 'カン，オンナヘンニモンガマエニモクヨウビノモク'],
        ['嫺', 'カン，オンナヘンニモンガマエニツキ'],
        ['㏄', 'シーシー'],
        ['㏍', 'ケーケー'],
        ['厠', 'カワヤ，ガンダレニホウソクノソク'],
        ['厥', 'ユウボクミンゾクノトッケツノケツ'],
        ['厦', 'カ，ガンダレニナツ'],
        ['厨', 'チュウボウ ノ チュウ，クリヤ'],
        ['厩', 'キュウシャ ノ キュウ，ウマヤ'],
        ['厭', 'エンセイカン ノ エン，イトウ'],
        ['厮', 'シ，ガンダレニシカイノケンイノシ'],
        ['厰', 'カイグンコウショウノショウノイタイジ，ガンダレ'],
        ['♯', 'シャープ'],
        ['♭', 'フラット'],
        ['♪', 'オンプ'],
        ['厶', 'シ，ワタクシ，カタカナノム'],
        ['去', 'キョネン ノ キョ，サル'],
        ['厂', 'カン，ブシュノガンダレ'],
        ['厄', 'ヤクドシ ノ ヤク'],
        ['厖', 'ボウダイナノボウ，ガンダレニムクイヌ'],
        ['厘', 'クブクリン ノ リン'],
        ['厚', 'アツガミ ノ アツ'],
        ['♂', 'オス'],
        ['♀', 'メス'],
        ['原', 'ゲンイン ノ ゲン，ハラ'],
        ['俺', 'オレサマ ノ オレ'],
        ['%', 'パーセント'],
        ['俶', 'シュク，ニンベンニシュクジョノシュクノツクリ'],
        ['俵', 'ドヒョウ ノ ヒョウ，タワラ'],
        ['俯', 'ウツムク，フカンスルノフ'],
        ['行', 'リョコウ ノ コウ，イク'],
        ['衍', 'フエンスルノエン'],
        ['血', 'ケツエキ ノ ケツ，チ'],
        ['衂', 'ジク，チヘンニヤイバノイタイジ'],
        ['衄', 'ジク，チヘンニジュウニシノウシ'],
        ['衆', 'シュウギイン ノ シュウ'],
        ['衙', 'ヤクショヲオイミスルカンガノガ'],
        ['衛', 'エイセイテキ ノ エイ'],
        ['衝', 'ショウトツスル ノ ショウ'],
        ['衞', 'エイセイテキノエイノキュウジ'],
        ['衒', 'キヲテラウノテラウ，ゲン'],
        ['術', 'ゲイジュツ ノ ジュツ'],
        ['街', 'ガイロジュ ノ ガイ'],
        ['表', 'ウラオモテ ノ オモテ，ヒョウ'],
        ['衫', 'サン，コロモヘンニサンヅクリ'],
        ['衡', 'ヘイコウカンカク ノ コウ'],
        ['衢', 'チマタヲイミスルガイクノク'],
        ['衣', 'イフク ノ イ，コロモ'],
        ['衽', 'ジン、コロモヘンニジンシンノランノジン'],
        ['衾', 'ドウキンスルノキン'],
        ['衿', 'エリ，コロモヘン ニ イマ'],
        ['衰', 'スイジャクスル ノ スイ，オトロエル'],
        ['衲', 'ノウ、コロモヘンニコクナイノナイ'],
        ['衵', 'ジツ、コロモヘンニニチヨウビノニチ'],
        ['衷', 'ワヨウセッチュウ ノ チュウ'],
        ['保', 'ホケンシツ ノ ホ，タモツ'],
        ['俐', 'レイリノリ，ニンベンニケンリノリ'],
        ['玲', 'レイロウナ ノ レイ，オウヘン ニ メイレイ ノ レイ'],
        ['玳', 'ドウブツノタイマイノタイ，オウヘンニダイヒョウノダイ'],
        ['玻', 'ハ，オウヘンニケガワノカワ'],
        ['玩', 'ガング ノ ガン，モテアソブ'],
        ['俊', 'シュンビンサ ノ シュン'],
        ['玖', 'オウヘン ニ ヒサシイ ノ ク'],
        ['率', 'ヒリツ ノ リツ，ヒキイル'],
        ['玄', 'ゲンマイ ノ ゲン'],
        ['王', 'オウサマ ノ オウ'],
        ['玉', 'ミズタマ ノ タマ，ギョク'],
        ['鍠', 'コウ、カネヘンニコウタイシノコウ'],
        ['鍬', 'ノウグ ノ クワ，カネヘン ニ アキ'],
        ['鍮', 'キンゾクノシンチュウノチュウ'],
        ['鍵', 'カギ，ケンバン ノ ケン'],
        ['鍼', 'シン，ハリキュウノハリ'],
        ['鍾', 'ショウニュウドウ ノ ショウ'],
        ['鍄', 'リョウ、カネヘンニトウキョウノキョウ'],
        ['鍋', 'ナベモノ ノ ナベ'],
        ['鍍', 'トキン ノ ト，カネヘン ニ オンドケイ ノ ド'],
        ['鍔', 'カタナ ノ ツバ，ガク'],
        ['鍖', 'チン、カネヘンニハナハダシイ'],
        ['鍛', 'キタエル，タンレン ノ タン'],
        ['鍜', 'カ，カネヘンニヒマツブシノヒマノツクリ'],
        ['綉', 'シシュウスルノシュウノイタイジ'],
        ['綏', 'スイゼイテンノウノスイ'],
        ['続', 'レンゾク ノ ゾク，ツヅク'],
        ['綛', 'カスリ，イトヘンニシノビ'],
        ['継', 'ケイゾクスル ノ ケイ，ツグ'],
        ['綟', 'レイ，イトヘンニモドル'],
        ['綜', 'サクソウスル ノ ソウ，イトヘン ニ シュウキョウ ノ シュウ'],
        ['經', 'ケイケンスルノケイノキュウジ'],
        ['綫', 'シンカンセンノセンノイタイジ'],
        ['綮', 'ケイ，ケイハツノケイノクチノカワリニイト'],
        ['綯', 'ナワヲナウノナウ'],
        ['綬', 'シジュホウショウ ノ ジュ，イトヘン ニ ウケル'],
        ['維', 'イジスル ノ イ'],
        ['綢', 'チュウ，イトヘンニシュウヘンノシュウノキュウジタイ'],
        ['綣', 'ケン，イトヘンニノリマキノマキノキュウジ'],
        ['綺', 'キレイナノキ，イトヘンニキミョウナノキ'],
        ['綻', 'ハタンスル ノ タン，ホコロビル'],
        ['綸', 'リンゲンアセノゴトシノリン'],
        ['綾', 'アヤオリ ノ アヤ'],
        ['綿', 'ワタ，モメン ノ メン'],
        ['綽', 'ヨユウシャクシャクノシャク'],
        ['網', 'モウマク ノ モウ，アミ'],
        ['綰', 'ワン，イトヘンニケイサツカンノカン'],
        ['綱', 'ツナヒキ ノ ツナ，コウ'],
        ['綴', 'ツヅル'],
        ['綵', 'サイ，イトヘンニハクシュカッサイノサイノキュウジタイ'],
        ['殖', 'ヨウショクスル ノ ショク，フヤス'],
        ['殕', 'ホウ，カバネヘンニバイリツノバイノツクリ'],
        ['殞', 'イン，カバネヘンニガッキュウイインノイン'],
        ['殘', 'ザンネンノザン，ノコルノキュウジ'],
        ['殆', 'ホトンド'],
        ['殄', 'テン，カバネヘンニシンサツスルノシンノツクリ'],
        ['殃', 'オウ，カバネヘンニチュウオウノオウ'],
        ['殀', 'ヨウ，カバネヘンニヨウカイヘンゲノヨウノツクリ'],
        ['殍', 'ヒョウ，カバネヘンニツメノシタニコドモノコ'],
        ['残', 'ザンネン ノ ザン，ノコル'],
        ['殊', 'トクシュ ノ シュ，コトニ'],
        ['殉', 'ジュンキョウシャ ノ ジュン'],
        ['殷', 'チュウゴクノオウチョウノイン'],
        ['段', 'シュダン ノ ダン'],
        ['殴', 'オウダスル ノ オウ，ナグル'],
        ['殳', 'シュ，ブシュノルマタ'],
        ['殲', 'テキヲセンメツスルノセン'],
        ['殱', 'テキヲセンメツスルノセンノイタイジ'],
        ['f', 'エフ フレンド'],
        ['殿', 'トノサマ ノ トノ'],
        ['殼', 'カイガラノカラ，カクノキュウジ'],
        ['殻', 'カイガラ ノ ガラ，カク'],
        ['殺', 'サツジン ノ サツ，コロス'],
        ['会', 'カイシャ ノ カイ，アウ'],
        ['殤', 'ショウ，カバネヘンニキリキズノツクリ'],
        ['殯', 'ヒン，モガリ，カバネヘンニライヒンノヒン'],
        ['殫', 'タン，カバネヘンニタンジュンノタンノキュウジ'],
        ['殪', 'エイ，カバネヘンニケイヤクショニツカウスウジイチノキュウジ'],
        ['餞', 'ハナムケ，センベツノセン'],
        ['呻', 'シンギンスルノシン，ウメク'],
        ['命', 'ウンメイ ノ メイ，イノチ'],
        ['呼', 'コキュウスル ノ コ，ヨブ'],
        ['呱', 'ココノコエヲアゲルノコ'],
        ['呰', 'シ，ヒガンシガンノシノシタニクチ'],
        ['味', 'イミ ノ ミ，アジ'],
        ['呵', 'タンカヲキルノカ，クチヘンニカノウセイノカ'],
        ['呷', 'コウ，サケヲアオルノアオル，クチヘンニコウオツノコウ'],
        ['呶', 'ド，クチヘンニヤッコ'],
        ['周', 'シュウヘン ノ シュウ，マワリ'],
        ['呪', 'ノロウ，ジュモン ノ ジュ'],
        ['呟', 'ツブヤク'],
        ['呑', 'ヘイドンスル ノ ドン，ノム'],
        ['呉', 'ゴフクヤ ノ ゴ，クレ'],
        ['呈', 'ゾウテイスル ノ テイ'],
        ['告', 'コウコク ノ コク，ツゲル'],
        ['呎', 'ナガサノタンイノフィート'],
        ['呀', 'カ，クチヘンニキバ'],
        ['呂', 'オフロ ノ ロ'],
        ['呆', 'アキレル，ボウゼントスル ノ ボウ'],
        ['<', 'ショウナリ'],
        ['婚', 'ケッコンスル ノ コン'],
        ['婁', 'ロウカク ノ ロウ ノ キュウジ ノ ツクリ'],
        ['婀', 'アダッポイノア'],
        ['婆', 'ロウバシン ノ バ'],
        ['婉', 'エンキョクヒョウゲンノエン'],
        ['婿', 'ハナムコ ノ ムコ'],
        ['婢', 'ヒ，オンナヘンニヒクツナノヒ'],
        ['婦', 'フウフ ノ フ'],
        ['婪', 'ドンランナノラン，ハヤシノシタニオンナ'],
        ['婬', 'イン，オンナヘンニインランノインノツクリ'],
        ['莠', 'ユウ，クサカンムリニヒイデル'],
        ['莢', 'キョウ，マメノサヤ'],
        ['鷦', 'ショウ，コゲルノミギニトリ'],
        ['莨', 'ロウ、タバコ、クサカンムリニカイリョウスルノリョウ'],
        ['莫', 'バクダイナ ノ バク'],
        ['莪', 'ガ，クサカンムリニガマンノガ'],
        ['莱', 'ホウライサン ノ ライ'],
        ['莵', 'ト，クサカンムリニウサギノイタイジ'],
        ['莽', 'ソウモウノシンノモウ'],
        ['莅', 'リ，クサカンムリニタンイノイ'],
        ['莇', 'ショ、クサカンムリニキュウジョスルノジョ'],
        ['莉', 'リ、クサカンムリニリエキノリ'],
        ['莊', 'ベッソウチノソウノイタイジ'],
        ['莎', 'サ、クサカンムリニゴブサタノサ'],
        ['鷯', 'リョウ，ドウリョウノリョウノツクリノミギニトリ'],
        ['莓', 'クダモノノイチゴ、クサカンムリニマイニチノマイ'],
        ['莖', 'チカケイノケイノキュウジ'],
        ['莚', 'エン，ムシロ，クサカンムリニエンキスルノエン'],
        ['莟', 'カン，クサカンムリニフクメル'],
        ['莞', 'クサカンムリ ニ カンリョウスル ノ カン'],
        ['鷲', 'トリ ノ ワシ'],
        ['鷽', 'ガク，ガクカンムリノキュウジタイノシタニトリ'],
        ['鷺', 'シラサギ ノ サギ'],
        ['鷆', 'テン，マゴコロノマノキュウジノミギニトリ'],
        ['餐', 'バンサン ノ サン'],
        ['畢', 'ヒッキョウ ノ ヒツ，オワル'],
        ['畠', 'ハタケ，シロ ノ シタニ タ'],
        ['畧', 'リャクゴノリャクノイタイジ'],
        ['畦', 'アゼミチ ノ アゼ，タ ニ ツチ フタツ'],
        ['略', 'リャクゴ ノ リャク'],
        ['畤', 'ジ，タヘンニテラ'],
        ['畫', 'ガヨウシノガノキュウジ'],
        ['番', 'バンゴウ ノ バン'],
        ['畩', 'ケサ，タヘンニコロモガエノコロモ'],
        ['畭', 'ヨ，タヘンニアマリ'],
        ['畳', 'タタミ，ヨジョウハン ノ ジョウ'],
        ['異', 'イコク ノ イ，コトナル'],
        ['畷', 'シジョウナワテシ ノ ナワテ'],
        ['當', 'トウバンノトウ，アタルノキュウジ'],
        ['畴', 'ハン，タヘンニコトブキ'],
        ['畸', 'キ，タヘンニキミョウナノキ'],
        ['畿', 'キンキチホウ ノ キ'],
        ['畆', 'メンセキノタンイノセ，ウネノイタイジ'],
        ['畄', 'リュウガクノリュウノイタイジ'],
        ['畋', 'テン，スイデンノデンニノブン'],
        ['畊', 'コウ，タガヤスノイタイジ'],
        ['畉', 'フ，タガヤス，タヘンニオット'],
        ['畏', 'イフスル ノ イ，カシコマル'],
        ['畍', 'セカイノカイノイタイジ'],
        ['界', 'セカイ ノ カイ'],
        ['畑', 'タハタ ノ ハタ，ハタケ'],
        ['畔', 'コハン ノ ハン，ホトリ'],
        ['畛', 'シン，アゼ，タヘンニシンサツスルノシンノツクリ'],
        ['畚', 'ホン，オマイリノマイノサンズクリノカワリニスイデンノデン'],
        ['留', 'リュウガク ノ リュウ'],
        ['畝', 'ハタケ ノ ウネ，メンセキ ノ タンイ ノ セ'],
        ['畜', 'カチク ノ チク'],
        ['頒', 'ハンプスル ノ ハン'],
        ['頓', 'セイトンスル ノ トン'],
        ['預', 'ヨキン ノ ヨ，アズケル'],
        ['頑', 'ガンバル ノ ガン'],
        ['括', 'ソウカツ ノ カツ，ククル'],
        ['拭', 'フッショク ノ ショク，ヌグウ'],
        ['拮', 'キッコウスルノキツ'],
        ['拯', 'ジョウ，テヘンニダイジンヲイミスルジョウショウノジョウ'],
        ['拠', 'キョテン ノ キョ'],
        ['拡', 'カクダイ ノ カク'],
        ['領', 'ダイトウリョウ ノ リョウ'],
        ['昊', 'コウ，ニチヨウビノニチノシタニテンキヨホウノテン'],
        ['頂', 'チョウテン ノ チョウ，イタダキ'],
        ['頃', 'コロアイ ノ コロ'],
        ['頁', 'ページ'],
        ['順', 'ジュンバン ノ ジュン'],
        ['拾', 'オカネヲ ヒロウ ノ ヒロウ'],
        ['拿', 'ダホスルノダ，ゴウカクノゴウノシタニテ'],
        ['拱', 'コマヌク，キョウ，テヘンノキョウツウノキョウ'],
        ['須', 'ヒッス ノ ス，スベカラク'],
        ['拳', 'ケンジュウ ノ ケン，コブシ'],
        ['拵', 'コシラエル，ソン'],
        ['拶', 'アイサツ ノ サツ'],
        ['拷', 'ゴウモンスル ノ ゴウ'],
        ['拈', 'ネン，テヘンニドクセンスルノセン'],
        ['拉', 'ラチスルノラ'],
        ['拊', 'フ，テヘンニフロクノフ'],
        ['拌', 'カクハンスルノハン，カキマゼル'],
        ['拍', 'ハクシュスル ノ ハク'],
        ['頴', 'スグレルヲ イミスル エイ，ホサキ，シメス'],
        ['拏', 'ダホスルノダ，ヤッコノシタニテ'],
        ['頻', 'ヒンパンナ ノ ヒン'],
        ['拂', 'カネヲハラウノハラウノキュウジ'],
        ['担', 'タンニン ノ タン'],
        ['拆', 'タク，テヘンニハイセキウンドウノセキ'],
        ['拇', 'ボインヲオスノボ'],
        ['拘', 'コウソクジカン ノ コウ'],
        ['拙', 'チセツ ノ セツ，ツタナイ'],
        ['招', 'マネク，ショウタイスル ノ ショウ'],
        ['拜', 'オガムノキュウジ'],
        ['拝', 'オガム，ハイケイ ノ ハイ'],
        ['頤', 'オトガイ'],
        ['拐', 'ユウカイジケン ノ カイ'],
        ['拑', 'カン，テヘンニアマイ'],
        ['拒', 'キョヒスル ノ キョ，コバム'],
        ['拓', 'カイタクスル ノ タク'],
        ['拔', 'バツグンノバツ，ヌケルノキュウジ'],
        ['頬', 'ホオヅエ ノ ホオ'],
        ['拗', 'ヨウオンノヨウ，テヘンニオオサナイ'],
        ['爵', 'ハクシャク ノ シャク'],
        ['父', 'フボ ノ フ，チチ'],
        ['爰', 'オウエンスルノエンノキュウジノツクリ'],
        ['爲', 'コウイノイ，タメノキュウジ'],
        ['爼', 'マナイタ，ソジョウノソノイタイジ'],
        ['爽', 'サワヤカ，ソウカイナ ノ ソウ'],
        ['爾', 'ソツジナガラ ノ ジ，ナンジ'],
        ['爿', 'ショウヘン，ショウグンノショウノキュウジノヘン'],
        ['爺', 'コウコウヤ ノ ヤ，ジイ'],
        ['爻', 'コウ，ハンバクスルノバクノツクリ'],
        ['爬', 'ハチュウルイノハ'],
        ['爭', 'アラソウノキュウジ'],
        ['爨', 'ハンゴウスイサンノサン'],
        ['爪', 'ツメキリ ノ ツメ'],
        ['爐', 'ダンロノロノキュウジ'],
        ['爛', 'テンシンランマンノラン'],
        ['爆', 'バクハツスル ノ バク'],
        ['爍', 'シャク，ヒヘンニタノシイノキュウジ'],
        ['況', 'ジッキョウチュウケイ ノ キョウ'],
        ['部', 'ブチョウ ノ ブ'],
        ['泄', 'ハイセツブツノセツ'],
        ['泅', 'シュウ，サンズイニシュウジンフクノシュウ'],
        ['郭', 'リンカク ノ カク，クルワ'],
        ['泉', 'オンセン ノ セン，イズミ'],
        ['泊', 'シュクハクスル ノ ハク，トマル'],
        ['泌', 'ヒニョウキ ノ ヒ'],
        ['郤', 'ケキ，タニノミギニオオザト'],
        ['泓', 'オウ，サンズイニコウボウタイシノコウ'],
        ['法', 'ホウリツ ノ ホウ'],
        ['都', 'トカイ ノ ト，ミヤコ'],
        ['泗', 'シ，サンズイニカンスウジノヨン'],
        ['泙', 'ヘイ，サンズイニヘイワノヘイ'],
        ['泛', 'ハン，サンズイニビンボウノボウ'],
        ['郷', 'ウマレコキョウ ノ キョウ'],
        ['泝', 'ソ，サンズイニハイセキウンドウノセキ'],
        ['郵', 'ユウビン ノ ユウ'],
        ['泡', 'アワ，スイホウ ノ ホウ'],
        ['波', 'ツナミ ノ ナミ，ハ'],
        ['泣', 'ナク，ゴウキュウスル ノ キュウ'],
        ['泥', 'デイスイ ノ デイ，ドロ'],
        ['注', 'チュウモクスル ノ チュウ，ソソグ'],
        ['泪', 'ナミダノイタイジ，サンズイニモクテキノモク'],
        ['泯', 'ビン，サンズイニコクミンノミン'],
        ['泰', 'アンタイ ノ タイ'],
        ['泱', 'オウ，サンズイニチュウオウノオウ'],
        ['泳', 'スイエイ ノ エイ，オヨグ'],
        ['ほ', 'ホケン ノ ホ'],
        ['ぺ', 'ペット ノ ペ'],
        ['べ', 'ベルト ノ ベ'],
        ['へ', 'ヘイワ ノ ヘ'],
        ['み', 'ミカン ノ ミ'],
        ['ま', 'マッチ ノ マ'],
        ['ぽ', 'ポスト ノ ポ'],
        ['ぼ', 'ボタン ノ ボ'],
        ['び', 'ビール ノ ビ'],
        ['ひ', 'ヒカリ ノ ヒ'],
        ['ぱ', 'パジャマ ノ パ'],
        ['ば', 'バナナ ノ バ'],
        ['ぷ', 'プリント ノ プ'],
        ['ぶ', 'ブランコ ノ ブ'],
        ['ふ', 'フトン ノ フ'],
        ['ぴ', 'ピアノ ノ ピ'],
        ['に', 'ニホン ノ ニ'],
        ['な', 'ナマエ ノ ナ'],
        ['ど', 'ドレミ ノ ド'],
        ['と', 'トウキョウ ノ ト'],
        ['は', 'ハガキ ノ ハ'],
        ['の', 'ノハラ ノ ノ'],
        ['ね', 'ネズミ ノ ネ'],
        ['ぬ', 'ヌリエ ノ ヌ'],
        ['っ', 'チイサイ ツバメ ノ ツ'],
        ['ぢ', 'ハナヂ ノ ヂ'],
        ['ち', 'チキュウ ノ チ'],
        ['だ', 'ダルマ ノ ダ'],
        ['で', 'デンワ ノ デ'],
        ['て', 'テガミ ノ テ'],
        ['づ', 'ツヅミ ノ ヅ'],
        ['つ', 'ツバメ ツ'],
        ['せ', 'セカイ ノ セ'],
        ['ず', 'ズボン ノ ズ'],
        ['す', 'スズメ ノ ス'],
        ['じ', 'ジカン ノ ジ'],
        ['た', 'タバコ ノ タ'],
        ['ぞ', 'ゾウキン ノ ゾ'],
        ['そ', 'ソロバン ノ ソ'],
        ['ぜ', 'ゼンブ ノ ゼ'],
        ['こ', 'コドモ ノ コ'],
        ['げ', 'ゲーム ノ ゲ'],
        ['け', 'ケシキ ノ ケ'],
        ['ぐ', 'グランド ノ グ'],
        ['し', 'シンブン ノ シ'],
        ['ざ', 'ザブトン ノ ザ'],
        ['さ', 'サクラ ノ サ'],
        ['ご', 'ゴリラ ノ ゴ'],
        ['か', 'カゾク ノ カ'],
        ['お', 'オオサカ ノ オ'],
        ['ぉ', 'チイサイ オオサカ ノ オ'],
        ['え', 'エイゴ ノ エ'],
        ['く', 'クスリ ノ ク'],
        ['ぎ', 'ギンコウ ノ ギ'],
        ['き', 'キッテ ノ キ'],
        ['が', 'ガッコウ ノ ガ'],
        ['ぃ', 'チイサイ イチゴ ノ イ'],
        ['あ', 'アサヒ ノ ア'],
        ['ぁ', 'チイサイ アサヒ ノ ア'],
        ['ぇ', 'チイサイ エイゴ ノ エ'],
        ['う', 'ウサギ ノ ウ'],
        ['ぅ', 'チイサイ ウサギ ノ ウ'],
        ['い', 'イチゴ ノ イ'],
        ['匪', 'ヒゾク ノ ヒ，ハコガマエ ニ ヒジョウグチ ノ ヒ'],
        ['匯', 'カイ，ハコガマエニサンズイトフルトリ'],
        ['匣', 'コウ，ハコガマエニコウオツノコウ'],
        ['匠', 'キョショウ ノ ショウ，タクミ'],
        ['匡', 'タダス，キョウ，ハコガマエ ニ オウ'],
        ['区', 'クベツ ノ ク'],
        ['医', 'オイシャサン ノ イ'],
        ['S', 'エス スポーツ'],
        ['匹', 'イッピキニヒキ ノ ヒキ'],
        ['匿', 'トクメイ ノ トク，カクス'],
        ['匳', 'ハコガマエニシケンヲウケルノケンノキュウジノツクリ'],
        ['匱', 'キ，ハコガマエニキチョウヒンノキ'],
        ['匈', 'ユウボクミンゾクノキョウドノキョウ'],
        ['匏', 'ホウ，ヒサゴ，ハカマノツクリニ，ヒョウガフルノヒョウノシタガワ'],
        ['匍', 'ホフクゼンシンノホ'],
        ['匂', 'イイニオイ ノ ニオイ'],
        ['匁', 'オモサ ノ タンイ ノ モンメ'],
        ['匆', 'ソウ，ネギカラクサカンムリトココロヲトッタカタチ'],
        ['包', 'ホウタイ ノ ホウ，ツツム'],
        ['匚', 'ホウ，ブシュノハコガマエ'],
        ['匙', 'サジカゲン ノ サジ'],
        ['匝', 'ソウ，ハコガマエ ニ ハバ'],
        ['匐', 'ホフクゼンシンノフク'],
        ['化', 'カセキ ノ カ，バケル'],
        ['北', 'キタミナミ ノ キタ，ホク'],
        ['匕', 'サジヲイミスルヒ'],
        ['餉', 'ショウ，ショクヘンニムカウ'],
        ['餃', 'ギョーザノイチモジメ'],
        ['裔', 'マツエイノエイ'],
        ['裕', 'ヨユウ ノ ユウ'],
        ['裘', 'キュウ、カワゴロモ'],
        ['裙', 'クン、コロモヘンニショクンノクン'],
        ['裟', 'オボウサン ノ ケサ ノ サ'],
        ['補', 'ホジョ ノ ホ，オギナウ'],
        ['裝', 'フクソウノソウノキュウジ'],
        ['裂', 'ブンレツスル ノ レツ，サケル'],
        ['裃', 'レイフクノカミシモ'],
        ['裁', 'サイバン ノ サイ，サバク'],
        ['裄', 'ユキタケノユキ、コロモヘンニリョコウキノコウ'],
        ['装', 'フクソウ ノ ソウ'],
        ['鋳', 'チュウゾウスル ノ チュウ，イル'],
        ['裏', 'ウラオモテ ノ ウラ'],
        ['裲', 'リョウ、コロモヘンニリョウテノリョウノキュウジ'],
        ['裳', 'イショウ ノ ショウ，モスソ'],
        ['裴', 'ハイ、ヒジョウグチノヒノシタニコロモ'],
        [')', 'トジカッコ'],
        ['裸', 'ハダカ，ラタイ ノ ラ'],
        ['裹', 'カ、コロモノアイダニコウカテキノカ'],
        ['裾', 'スソノ ノ スソ'],
        ['餬', 'コ，カユ，ショクヘンニクロコショウノコ'],
        ['裼', 'セキ、コロモヘンニボウエキフウノエキ'],
        ['製', 'セイゾウスル ノ セイ'],
        ['裡', 'キンリ ノ リ，コロモヘン ニ サト'],
        ['裨', 'ヒ、コロモヘンニヒクツナノヒ'],
        ['耨', 'ドウ，スキヘンニブジョクスルノジョク'],
        ['耡', 'ジョ，スキヘンニキュウジョスルノジョ'],
        ['耻', 'ハジヲカクノハジノイタイジ'],
        ['耿', 'コウ，ミミヘンニカヨウビノカ'],
        ['耽', 'タンデキスル ノ タン，フケル'],
        ['耳', 'ミミ，ジビカ ノ ジ'],
        ['耶', 'ウヤムヤ ノ ヤ'],
        ['耋', 'テツ，ロウジンノロウノシタニダイシキュウノシ，'],
        ['而', 'シコウシテ，ケイジジョウ ノ ジ'],
        ['考', 'カンガエル，サンコウショ ノ コウ'],
        ['老', 'ロウジン ノ ロウ'],
        ['耀', 'エイヨウエイガ ノ ヨウ，カガヤク'],
        ['耆', 'キ，ロウジンノロウノシタニマイニチノニチ'],
        ['者', 'ガクシャ ノ シャ，モノ'],
        ['耄', 'モウロクスルノモウ，ロウジンノロウノシタニケイトノケ'],
        ['耙', 'ハ，スキヘンニトモエ'],
        ['耘', 'コウウンキノウン，ブシュノスキヘンニウンヌンノウン'],
        ['耜', 'シ，スキヘンニケイサツカンノカンノシタガワ'],
        ['耒', 'ライ，ブシュノスキヘン'],
        ['耐', 'ニンタイ ノ タイ，タエル'],
        ['耗', 'ショウモウヒン ノ モウ'],
        ['耕', 'タガヤス，ノウコウ ノ コウ'],
        ['簍', 'ロウ，タケカンムリニロウカクノロウノキュウジノツクリ'],
        ['簇', 'ソウ，ムラガル，タケカンムリニスイゾクカンノゾク'],
        ['簀', 'スノコノス'],
        ['簟', 'テン，タカムシロ，タケカンムリニニシノシタニハヤオキノハヤイ'],
        ['簔', 'ミノノイタイジ，タケカンムリニオトロエルノヘンケイ'],
        ['簗', 'サカナヲトルヤナ，タケカンムリニリョウザンパクノリョウ'],
        ['簑', 'ミノノイタイジ，タケカンムリニオトロエル'],
        ['簒', 'サンダツスルノサン，ウバウ'],
        ['簓', 'ササラ'],
        ['簪', 'カンザシ，シン'],
        ['簫', 'ショウ，タケカンムリニゲンシュクナノシュクノキュウジ'],
        ['簧', 'コウ，タケカンムリニキイロノキノキュウジ'],
        ['簡', 'カンタンナ ノ カン'],
        ['簣', 'キ，タケカンムリニキチョウヒンノキ'],
        ['簽', 'セン，タケカンムリニシケンヲウケルノケンノキュウジノツクリ'],
        ['簾', 'スダレ，レン'],
        ['簿', 'メイボ ノ ボ'],
        ['簸', 'ハ，タケカンムリ ニ ソノタ ノ ソニ カワ'],
        ['簷', 'エン、タケカンムリニタンニンノタンノキュウジノツクリ'],
        ['鉞', 'フエツノエツ、マサカリ'],
        ['欝', 'ユウウツナ ノ ウツ'],
        ['欟', 'カン，キヘンニカンキャクノカンノキュウジ'],
        ['黄', 'キイロ ノ キ，オウ'],
        ['欒', 'イッカダンランノラン'],
        ['黏', 'ネバル，ネン，キビヘンニドクセンスルノセン'],
        ['黎', 'レイメイキノレイ'],
        ['黍', 'コクモツ ノ キビ，ショ'],
        ['欖', 'カンランサンノラン，キヘンニハクランカイノランノキュウジ'],
        ['黒', 'コクバン ノ コク，クロ'],
        ['權', 'ケンリノケンノキュウジ'],
        ['鉛', 'エンピツ ノ エン，ナマリ'],
        ['黔', 'クロイ，ケン，クロイノミギニイマ'],
        ['黛', 'マユズミ'],
        ['黙', 'チンモク ノ モク，ダマル'],
        ['默', 'チンモクスルノモク，ダマルノキュウジ'],
        ['欅', 'ケヤキ，キヘンニセンキョケンノキョノキュウジ'],
        ['欄', 'クウラン ノ ラン'],
        ['黝', 'アオグロイ，ユウ，クロノミギニオサナイ'],
        ['黜', 'シリゾケル，チュツ，クロノミギニデル'],
        ['欹', 'キ，キミョウナノキニフカケツノケツ'],
        ['欸', 'アイ，アイサツノアイノツクリニフカケツノケツ'],
        ['欺', 'サギ ノ ギ，アザムク'],
        ['欽', 'キンテイケンポウ ノ キン'],
        ['黥', 'ゲイ，イレズミ，クロノミギニトウキョウノキョウ'],
        ['款', 'エンシャッカン ノ カン'],
        ['欲', 'ヨクボウ ノ ヨク，ホシイ'],
        ['黯', 'クライ，アン，クロノミギニオンガクノオン'],
        ['欷', 'キ，キボウノキニフカケツノケツ'],
        ['黷', 'ケガス，トク，クロノミギニハンバイノバイノキュウジ'],
        ['黶', 'ホクロ，エン'],
        ['黴', 'カビ，バイキンノバイ'],
        ['次', 'モクジ ノ ジ，ツギ'],
        ['欠', 'ホケツ ノ ケツ，カケル'],
        ['欣', 'キンキジャクヤク ノ キン，ヨロコブ'],
        ['欧', 'オウベイ ノ オウ'],
        ['黼', 'ヌイトリノモヨウヲイミスルフ'],
        ['痣', 'アザ，ヤマイダレニココロザシ'],
        ['痢', 'ゲリヲスル ノ リ'],
        ['痩', 'ヤセル，ソウシン ノ ソウ'],
        ['痴', 'グチ ノ チ'],
        ['痰', 'タンガカラムノタン'],
        ['痳', 'リン，ヤマイダレニハヤシ'],
        ['痲', 'ヤマイダレノマヒスルノマ'],
        ['痼', 'ジビョウヲイミスルコシツノコ'],
        ['痿', 'イ，ヤマイダレニガッキュウイインノイ'],
        ['痾', 'ア，ヤマイダレニアミダブツノア'],
        ['痺', 'マヒスルノヒ，シビレル'],
        ['病', 'ビョウキ ノ ビョウ，ヤマイ'],
        ['症', 'カフンショウ ノ ショウ'],
        ['痃', 'ケン，ヤマイダレニゲンマイノゲン'],
        ['痂', 'カ，カサブタ，ヤマイダレニカソクドノカ'],
        ['痍', 'ショウイグンジンノイ'],
        ['痊', 'セン，ヤマイダレニゼンコクノゼン'],
        ['痕', 'コンセキヲ ノコス ノ コン'],
        ['痔', 'イボジ ノ ジ'],
        ['痒', 'カッカソウヨウノヨウ，カユイ'],
        ['痞', 'ヒ，ヤマイダレニヒテイスルノヒ'],
        ['痙', 'イケイレンノケイ'],
        ['痘', 'テンネントウ ノ トウ'],
        ['痛', 'ズツウ ノ ツウ，イタイ'],
        ['首', 'シュト ノ シュ，クビ'],
        ['扱', 'アツカウ'],
        ['扶', 'フヨウカゾク ノ フ'],
        ['批', 'ヒハンスル ノ ヒ'],
        ['找', 'カ，テヘンニホコヅクリ'],
        ['承', 'ショウダクスル ノ ショウ'],
        ['扼', 'セッシヤクワンノヤク'],
        ['扣', 'コウ，テヘンニクチ'],
        ['扠', 'サ，テヘンニヤシャノシャ'],
        ['鉱', 'テッコウセキ ノ コウ'],
        ['扨', 'サテ，テヘンニヤイバノイタイジ'],
        ['扮', 'フンソウヲコラス ノ フン'],
        ['馴', 'カイナラス ノ ナラス，ジュン'],
        ['打', 'タゲキ ノ ダ，ウツ'],
        ['払', 'カネヲハラウ ノ ハラウ'],
        ['馼', 'ブン，ウマヘンニサクブンノブン'],
        ['扛', 'コウ，テヘンニズガコウサクノコウ'],
        ['托', 'タクハツ ノ タク'],
        ['j', 'ジェイ ジャパン'],
        ['扞', 'カン，テヘンニホス'],
        ['馥', 'フクイクタルノフク，カオリ'],
        ['所', 'ジュウショ ノ ショ，トコロ'],
        ['扁', 'ヘン，ヘンシュウスル，アムノツクリ'],
        ['扇', 'センプウキ ノ セン，オウギ'],
        ['馬', 'ウマ，バシャ ノ バ'],
        ['手', 'テアシ ノ テ，シュ'],
        ['扈', 'チョウリョウバッコノコ'],
        ['扉', 'トビラ，モンピ ノ ヒ'],
        ['扎', 'サツ，テヘンニレイギノレイノツクリ'],
        ['才', 'サイノウ ノ サイ'],
        ['鉦', 'ショウコ ノ ショウ，カネヘン ニ タダシイ'],
        ['姚', 'ヨウ，オンナヘンニイッチョウエンノチョウ'],
        ['姙', 'ニン，ニンシンスルノニンノイタイジ'],
        ['′', 'フン'],
        ['姜', 'ショウガノニモジメ，ヒツジノシタニオンナ'],
        ['姓', 'セイメイハンダン ノ セイ'],
        ['姑', 'コソク ノ コ，シュウトメ'],
        ['姐', 'アネゴハダ ノ アネ'],
        ['委', 'ガッキュウイイン ノ イ'],
        ['始', 'カイシスル ノ シ，ハジメル'],
        ['†', 'ダガー'],
        ['姉', 'アネ，シマイ ノ シ'],
        ['‥', 'テンテン'],
        ['…', 'テンテンテン'],
        ['姆', 'ボ，オンナヘンニハハ'],
        ['姻', 'コンイン ノ イン'],
        ['‐', 'ハイフン'],
        ['姿', 'スガタ，シセイ ノ シ'],
        ['’', 'トジシングル'],
        ['‘', 'シングル'],
        ['”', 'トジクォーテーション'],
        ['姶', 'オンナヘン ニ ゴウカク ノ ゴウ'],
        ['姫', 'オヒメサマ ノ ヒメ'],
        ['姪', 'オイメイ ノ メイ'],
        ['姨', 'イ，オンナヘンニソンノウジョウイノイ'],
        ['釣', 'サカナツリ ノ ツリ'],
        ['姦', 'カシマシイ，オンナミッツ ノ カン'],
        ['姥', 'ウバザクラ ノ ウバ'],
        ['ヽ', 'クリカエシ'],
        ['ー', 'チョウオン'],
        ['ヾ', 'クリカエシダクテン'],
        ['・', 'ナカテン'],
        ['ヵ', 'チイサイ カゾク ノ カ'],
        ['ヴ', 'ウサギ ノ ウニ ダクテン'],
        ['ヶ', 'チイサイ ケシキ ノ ケ'],
        ['ヱ', 'ムカシ ノ エ'],
        ['ヰ', 'ムカシ ノ イ'],
        ['ン', 'オシマイ ノ ン'],
        ['ヲ', 'ヲワリ ノ オ'],
        ['ロ', 'ロウカ ノ ロ'],
        ['レ', 'レモン ノ レ'],
        ['ワ', 'ワカメ ノ ワ'],
        ['ヮ', 'チイサイ ワカメ ノ ワ'],
        ['ラ', 'ラジオ ノ ラ'],
        ['ヨ', 'ヨット ノ ヨ'],
        ['ル', 'ルスバン ノ ル'],
        ['リ', 'リンゴ ノ リ'],
        ['ュ', 'チイサイ ユカタ ノ ユ'],
        ['ヤ', 'ヤカン ノ ヤ'],
        ['ョ', 'チイサイ ヨット ノ ヨ'],
        ['ユ', 'ユカタ ノ ユ'],
        ['メ', 'メガネ ノ メ'],
        ['ム', 'ムシ ノ ム'],
        ['ャ', 'チイサイ ヤカン ノ ヤ'],
        ['モ', 'モミジ ノ モ'],
        ['ポ', 'ポスト ノ ポ'],
        ['ボ', 'ボタン ノ ボ'],
        ['ミ', 'ミカン ノ ミ'],
        ['マ', 'マッチ ノ マ'],
        ['ベ', 'ベルト ノ ベ'],
        ['ヘ', 'ヘイワ ノ ヘ'],
        ['ホ', 'ホケン ノ ホ'],
        ['ペ', 'ペット ノ ペ'],
        ['フ', 'フトン ノ フ'],
        ['ピ', 'ピアノ ノ ピ'],
        ['プ', 'プリント ノ プ'],
        ['ブ', 'ブランコ ノ ブ'],
        ['パ', 'パジャマ ノ パ'],
        ['バ', 'バナナ ノ バ'],
        ['ビ', 'ビール ノ ビ'],
        ['ヒ', 'ヒカリ ノ ヒ'],
        ['ネ', 'ネズミ ノ ネ'],
        ['ヌ', 'ヌリエ ノ ヌ'],
        ['ハ', 'ハガキ ノ ハ'],
        ['ノ', 'ノハラ ノ ノ'],
        ['ド', 'ドレミ ノ ド'],
        ['ト', 'トウキョウ ノ ト'],
        ['ニ', 'ニホン ノ ニ'],
        ['ナ', 'ナマエ ノ ナ'],
        ['ヅ', 'ツヅミ ノ ヅ'],
        ['ツ', 'ツバメ ノ ツ'],
        ['デ', 'デンワ ノ デ'],
        ['テ', 'テガミ ノ テ'],
        ['チ', 'チキュウ ノ チ'],
        ['ダ', 'ダルマ ノ ダ'],
        ['ッ', 'チイサイ ツバメ ノ ツ'],
        ['ヂ', 'ハナヂ ノ ヂ'],
        ['劬', 'ク，ハイクノクニチカラ'],
        ['@', 'アットマーク'],
        ['助', 'キュウジョ ノ ジョ，タスケル'],
        ['努', 'ドリョク ノ ド'],
        ['劫', 'ミライエイゴウ ノ ゴウ'],
        ['加', 'ツイカ ノ カ，クワエル'],
        ['劣', 'オトル，ユウレツ ノ レツ'],
        ['劼', 'カツ，ダイキチノキチニチカラ'],
        ['劾', 'ダンガイスル ノ ガイ'],
        ['効', 'コウカテキ ノ コウ'],
        ['労', 'ロウドウ ノ ロウ'],
        ['劵', 'ケン，ジョウシャケンノケンノカタナノカワリニチカラ'],
        ['励', 'ゲキレイスル ノ レイ，ハゲマス'],
        ['劍', 'ケン，ツルギノキュウジ'],
        ['劈', 'ヘキ，カベノツチノカワリニカタナ'],
        ['劉', 'リュウホウ ノ リュウ'],
        ['劇', 'ニンギョウゲキ ノ ゲキ'],
        ['劃', 'ワカツ，カクゼン ノ カク'],
        ['釟', 'ハツ、カネヘンニカンスウジノハチ'],
        ['功', 'フセイコウ ノ コウ'],
        ['力', 'チカラ，タイリョク ノ リョク'],
        ['劔', 'ケン，ツルギノイタイジ，ツクリガヤイバノイタイジ'],
        ['劑', 'ヤクザイシノザイノキュウジ'],
        ['劒', 'ケン，ツルギノイタイジ，ツクリガカタナニテン'],
        ['鋪', 'カネヘン ノ ホ'],
        ['踊', 'オドル，ブヨウ ノ ヨウ'],
        ['踉', 'ヨロメク，ソウロウノロウ'],
        ['踈', 'カソチノソノイタイジ，アシヘンニヤクソクノソク'],
        ['踏', 'フム，ザットウ ノ トウ'],
        ['踐', 'ジッセンスルノセンノキュウジ'],
        ['釘', 'クギ'],
        ['踟', 'チ，アシヘンニケンチジノチ'],
        ['踞', 'ソンキョスルノキョ，アシヘンニジュウキョノキョ'],
        ['踝', 'クルブシ，カ'],
        ['踪', 'シッソウジケンノソウ'],
        ['踰', 'ユ，アシヘンニニレノキノニレノツクリ'],
        ['踵', 'カカト，ショウ'],
        ['踴', 'オドル，ブヨウノヨウノイタイジ，アシヘンニイサマシイ'],
        ['釁', 'スキマヲイミスルキンゲキノキン'],
        ['釀', 'ジョウゾウスルノジョウ，カモスノキュウジ'],
        ['野', 'ヤサイ ノ ヤ，ノハラ ノ ノ'],
        ['里', 'サトガエリ ノ サト，リ'],
        ['釈', 'シャクホウスル ノ シャク'],
        ['襖', 'ショウジフスマ ノ フスマ'],
        ['襞', 'ヒダ，カベノツチノカワリニコロモ'],
        ['襟', 'エリマキ ノ エリ'],
        ['襄', 'ジョウ，ユズルノキュウジノツクリ'],
        ['襁', 'キョウ、コロモヘンニベンキョウノキョウ'],
        ['襃', 'キヨホウヘンノホウノイタイジ'],
        ['襌', 'タン、コロモヘンニタンジュンノタンノキュウジ'],
        ['襍', 'ザツオンノザツノイタイジ'],
        ['襴', 'キンランドンスノラン'],
        ['襷', 'タスキ'],
        ['襲', 'シュウゲキ ノ シュウ，オソウ'],
        ['襾', 'エ，ブシュノオオイカンムリ'],
        ['西', 'ニシヒガシ ノ ニシ，セイ'],
        ['襤', 'ランルヲマトウノラン'],
        ['襦', 'ジュバンノジュ'],
        ['襠', 'トウ、コロモヘンニベントウノトウノキュウジ'],
        ['襭', 'ケツ，コロモヘンニダイキチノキチニページ'],
        ['襯', 'シン、コロモヘンニオヤコノオヤ'],
        ['襪', 'ベツ，コロモヘンニケイベツスルノベツ'],
        ['鸛', 'トリノコウノトリ，カン'],
        ['鸞', 'シンランショウニンノラン'],
        ['犹', 'ユウ，ケモノヘンニセツゾクシノモットモ'],
        ['状', 'ネンガジョウ ノ ジョウ'],
        ['犲', 'サイ，ケモノヘンニカタカナノオ'],
        ['犯', 'ハンザイ ノ ハン'],
        ['犬', 'イヌ，モウドウケン ノ ケン'],
        ['黠', 'カツ，クロノミギニダイキチノキチ'],
        ['犧', 'ギセイシャノギノキュウジ'],
        ['犢', 'トク，コウシ，ウシヘンニハンバイスルノバイノキュウジ'],
        ['犠', 'ギセイシャ ノ ギ'],
        ['犖', 'ラク，ホタルノキュウジノムシノカワリニドウブツノウシ'],
        ['犒', 'コウ，ウシヘンニタカイヒクイノタカイ'],
        ['犇', 'ホン，ヒシメク，ドウブツノウシミッツ'],
        ['犂', 'レイ，カラスキ'],
        ['犀', 'ドウブツ ノ サイ'],
        ['犁', 'レイ，カラスキノイタイジ'],
        ['遭', 'ソウナンスル ノ ソウ'],
        ['遯', 'トン，シンニョウニトンジルノトン'],
        ['遮', 'シャダンスル ノ シャ，サエギル'],
        ['汎', 'ハンヨウセイ ノ ハン'],
        ['遨', 'ゴウ，シンニョウニゴウマンナノゴウノツクリ'],
        ['求', 'ヨウキュウ ノ キュウ，モトメル'],
        ['汀', 'ミギワ，サンズイ ニ イッチョウメ ノ チョウ'],
        ['汁', 'ミソシル ノ シル，カジュウ ノ ジュウ'],
        ['遡', 'サカノボル，ソジョウスル ノ ソ'],
        ['遠', 'エンソク ノ エン，トオイ'],
        ['遣', 'ハケンスル ノ ケン，ツカワス'],
        ['汚', 'オセン ノ オ，キタナイ'],
        ['遼', 'ゼントリョウエン ノ リョウ，ハルカ'],
        ['避', 'ヒショチ ノ ヒ，サケル'],
        ['汞', 'コウ，ズガコウサクノコウノシタニミズ'],
        ['江', 'エドジダイ ノ エ'],
        ['汝', 'ナンジ，サンズイ ニ オンナ'],
        ['遵', 'ジュンシュスル ノ ジュン，シンニョウ ニ ソンケイ ノ ソン'],
        ['汐', 'チョウセキ ノ セキ，シオ'],
        ['遶', 'ジョウ，シンニョウニギョウシュンノギョウノキュウジ'],
        ['汗', 'アセ，ハッカン ノ カン'],
        ['汕', 'サン，サンズイニヤマ'],
        ['汪', 'オウ，サンズイニオウサマノオウ'],
        ['汨', 'ベキラノベキ，サンズイニヨウビノニチ'],
        ['過', 'カコ ノ カ，スギル'],
        ['遉', 'テイ，シンニョウニテイシュクナノテイ'],
        ['運', 'ウンドウ ノ ウン，ハコブ'],
        ['遊', 'ユウエンチ ノ ユウ，アソブ'],
        ['汢', 'ヌタ，サンズイニツチ'],
        ['池', 'チョスイチ ノ チ，イケ'],
        ['遁', 'トンソウスル ノ トン'],
        ['遂', 'スイコウスル ノ スイ，トゲル'],
        ['決', 'タスウケツ ノ ケツ，キメル'],
        ['遜', 'ケンソンスル ノ ソン'],
        ['遞', 'テイシンショウノテイノキュウジ'],
        ['汾', 'フン，サンズイニハンブンノブン'],
        ['遘', 'コウ，シンニョウニミゾノツクリ'],
        ['汽', 'キテキ ノ キ'],
        ['汲', 'クム，キュウキュウトスル ノ キュウ'],
        ['汳', 'ハン，サンズイニハンタイスルノハン'],
        ['汰', 'シゼントウタ ノ タ'],
        ['遖', 'アッパレ，シンニョウニミナミ'],
        ['遑', 'コウ、シンニョウニコウタイシノコウ'],
        ['遐', 'カ，シンニョウニヒマツブシノヒマノツクリ'],
        ['道', 'ミチ，ドウロ ノ ドウ'],
        ['遒', 'シュウ，シンニョウニシュウチョウサンノシュウ'],
        ['粗', 'ソマツナ ノ ソ，アライ'],
        ['粕', 'サケカス ノ カス，コメヘン ニ シロ'],
        ['粒', 'ツブ，リュウシ ノ リュウ'],
        ['粐', 'ヌカ，コメヘンニトジマリノト'],
        ['粟', 'キビアワ ノ アワ'],
        ['粛', 'ゲンシュクナ ノ シュク'],
        ['粘', 'ネバル，ネンド ノ ネン'],
        ['粂', 'クメ，ヒサシイ ノ シタニ コメ'],
        ['粃', 'シイナ，コメヘンニヒカクスルノヒ'],
        ['粁', 'キョリ ノ タンイ ノ キロメートル'],
        ['粍', 'ナガサ ノ タンイ ノ ミリメートル'],
        ['粋', 'ジュンスイナ ノ スイ，イキ'],
        ['粉', 'フンマツ ノ フン，コナ'],
        ['粲', 'サン，サンゼントカガヤクノサンノツクリ'],
        ['粳', 'ウルチマイノウルチ'],
        ['粱', 'リョウ，リョウザンパクノリョウノキノカワリニコメ'],
        ['精', 'セイシンテキ ノ セイ'],
        ['粽', 'ソウ，チマキ'],
        ['粹', 'ジュンスイナノスイノキュウジ'],
        ['粧', 'オケショウ ノ ショウ'],
        ['粤', 'チュウゴクカントンショウヲイミスルエツ'],
        ['粥', 'オカユ ノ カユ'],
        ['粢', 'シトギ，モクジノジノシタニコメ'],
        ['粡', 'トウ，コメヘンニオナジ'],
        ['粮', 'ショクリョウノリョウ，カテノイタイジ'],
        ['粭', 'スクモ，コメヘンニゴウカクノゴウ'],
        ['粫', 'ウルチ，コメヘンニケイジジョウノジ'],
        ['粨', 'ナガサノタンイノヘクトメートル'],
        ['撥', 'ハネル，ハツ，テヘンニハツメイノハツノキュウジ'],
        ['撤', 'テッタイスル ノ テツ'],
        ['撫', 'アイブスル ノ ブ，ナデル'],
        ['撩', 'リョウランノリョウ，テヘンニドウリョウノリョウノツクリ'],
        ['撮', 'サツエイスル ノ サツ，トル'],
        ['播', 'デンパスル ノ ハ，マク'],
        ['撲', 'ダボク ノ ボク'],
        ['撰', 'チョクセンワカシュウ ノ セン，エラブ'],
        ['撻', 'ベンタツスルノタツ'],
        ['撹', 'カクランスル ノ カク'],
        ['撼', 'シンカンサセルノカン，テヘンニカンソウブンノカン'],
        ['撃', 'コウゲキ ノ ゲキ，ウツ'],
        ['撈', 'ギョロウノロウ，テヘンニロウドウノロウノキュウジ'],
        ['撓', 'フトウフクツノトウ，タワム'],
        ['撒', 'ミズヲマク ノ マク，サン'],
        ['撕', 'シ，テヘンニシカイノケンイノシ'],
        ['鹽', 'シオノキュウジ，エン'],
        ['撚', 'ヒネル，テヘン ニ シゼン ノ ゼン'],
        ['鹿', 'ドウブツ ノ シカ'],
        [
            '鹹',
            'カンスイコノカン，カライ，ロカクノロノミギニハリキュウノハリノツクリ',
        ],
        ['撞', 'ジカドウチャク ノ ドウ，ツク'],
        ['W', 'ダブリュー ウィンドウ'],
        ['×', 'カケル'],
        ['魄', 'ハク，シロヘンニオニ'],
        ['÷', 'ワル'],
        ['奘', 'サンゾウホウシゲンジョウノジョウ'],
        ['奚', 'ケイ，ケイリュウヅリノケイノキュウジノツクリ'],
        ['奕', 'イゴヲイミスルエキ'],
        ['奔', 'ホンソウスル ノ ホン'],
        ['套', 'ジョウトウシュダン ノ トウ'],
        ['契', 'ケイヤクスル ノ ケイ'],
        ['奐', 'カン，カンキセンノカンノツクリ'],
        ['奏', 'エンソウスル ノ ソウ，カナデル'],
        ['奎', 'ケイ，オオキイノシタニツチフタツ'],
        ['奉', 'ホウシカツドウ ノ ホウ，タテマツル'],
        ['奈', 'ナラケン ノ ナ'],
        ['奄', 'アマミオオシマ ノ イチモジメ ノ エン'],
        ['奇', 'キミョウナ ノ キ'],
        ['魎', 'チミモウリョウノリョウ'],
        ['好', 'ダイスキ ノ スキ，コウ'],
        ['奸', 'カン，オンナヘンニホス'],
        ['奴', 'ドレイ ノ ド，ヤツ'],
        ['女', 'オンナ，ジョシ ノ ジョ'],
        ['奬', 'ショウガクキンノショウノキュウジ'],
        ['奮', 'コウフン ノ フン，フルウ'],
        ['奩', 'レン，ハコ，オオキイノシタニクベツノクノキュウジ'],
        ['奨', 'ショウガクキン ノ ショウ'],
        ['奪', 'ウバウ，リャクダツ ノ ダツ'],
        ['奥', 'オクバ ノ オク'],
        ['奧', 'オクバノオクノキュウジ'],
        ['奠', 'オコウデンノデン，シュウチョウノシュウノシタニオオキイ'],
        ['奢', 'ゴウシャノシャ，オゴリ'],
        ['肭', 'ドツ，ニクヅキニコクナイノナイ'],
        ['肬', 'イボ，ユウ，ニクヅキニモットモ'],
        ['肯', 'コウテイヒテイ ノ コウ'],
        ['肩', 'カタコリ ノ カタ，ケン'],
        ['肪', 'シボウブン ノ ボウ'],
        ['肥', 'ヒマン ノ ヒ，コエル'],
        ['股', 'コカンセツ ノ コ，マタ'],
        ['肢', 'センタクシ ノ シ'],
        ['ﾙ', 'ルスバン ノ ル'],
        ['肺', 'ハイカツリョウ ノ ハイ'],
        ['肴', 'シュコウ ノ コウ，サカナ'],
        ['肱', 'ココウノシン ノ コウ，ヒジ'],
        ['育', 'キョウイク ノ イク，ソダテル'],
        ['肌', 'ハダイロ ノ ハダ'],
        ['肉', 'ギュウニク ノ ニク'],
        ['ﾜ', 'ワカメ ノ ワ'],
        ['肋', 'ロクマクエン ノ ロク'],
        ['肅', 'ゲンシュクナノシュクノキュウジ'],
        ['肄', 'イ，ギモンノギノヒダリガワニフデヅクリ'],
        ['肇', 'クニヲハジメル ノ ハジメ，チョウ'],
        ['肆', 'シ，ケイヤクショニツカウスウジノヨン'],
        ['-', 'マイナス'],
        ['肝', 'カンゾウ ノ カン，キモ'],
        ['肘', 'ヒジ，ニクヅキニスンポウ ノ スン'],
        ['肛', 'オシリノコウモンノコウ'],
        ['肚', 'ズ，ニクヅキニドヨウビノド'],
        ['肖', 'ショウゾウガ ノ ショウ'],
        ['肓', 'ヤマイコウコウノウシロノコウ'],
        ['ﾑ', 'ムシ ノ ム'],
        ['ﾗ', 'ラジオ ノ ラ'],
        ['ﾊ', 'ハガキ ノ ハ'],
        ['ﾉ', 'ノハラ ノ ノ'],
        ['躅', 'チョク，アシヘンニサンゴクシノショク'],
        ['躄', 'ヘキ，カベノツチノカワリニテアシノアシ'],
        ['躇', 'チュウチョスルノチョ'],
        ['躁', 'ソウウツビョウノソウ'],
        ['躍', 'カツヤクスル ノ ヤク'],
        ['躋', 'セイ，アシヘンニヘソノツクリ'],
        ['躊', 'チュウチョスルノチュウ'],
        ['躔', 'テン，アシヘンニマトウノツクリ'],
        ['躑', 'テキ，アシヘンニトウテキキョウギノテキノツクリ'],
        ['躓', 'ツマズク，チ'],
        ['躙', 'ジュウリンスルノリン'],
        ['躡', 'ジョウ、アシヘンニミミミッツ'],
        ['躬', 'キュウ、ミヘンニユミ'],
        ['躯', 'タイク ノ ク，ミ'],
        ['身', 'シンチョウタイジュウ ノ シン，ミ'],
        ['躪', 'ジュウリンスルノリンノイタイジ'],
        ['躱', 'タ、ミヘンニジダノジ'],
        ['躰', 'タイ，ミヘンニホンモノノホン'],
        ['躾', 'シツケ、ミヘンニウツクシイ'],
        ['遅', 'オソイ，チコク ノ チ'],
        ['雲', 'クモノウエ ノ クモ，ウン'],
        ['瑯', 'ホウロウナベノロウ，オウヘンニモモタロウノロウ'],
        ['瑪', 'メノウノメ'],
        ['瑩', 'ホタルノキュウジノムシノカワリニタマ'],
        ['瑤', 'ヨウ，オウヘンニユレルノツクリノキュウジ'],
        ['瑣', 'ハンサナテツヅキノサ'],
        ['瑠', 'ルリイロ ノ ル'],
        ['瑾', 'キン，キズヲイミスルカキンノキン'],
        ['瑶', 'ヨウ，オウヘンニユレルノツクリ'],
        ['瑳', 'セッサタクマ ノ サ，オウヘン ニ サ'],
        ['瑰', 'オウヘンニオニ'],
        ['瑁', 'ドウブツノタイマイノマイ'],
        ['瑟', 'キンシツアイワスノシツ'],
        ['瑞', 'ズイウン ノ ズイ，シルシ'],
        ['瑜', 'ユ，オウヘンニニレノキノニレノツクリ'],
        ['瑛', 'オウヘン ニ エイゴ ノ エイ'],
        ['瑚', 'サンゴショウ ノ ゴ'],
        ['瑙', 'メノウノノウ'],
        ['瑕', 'キズヲイミスルカシノカ'],
        ['援', 'オウエンスル ノ エン'],
        ['揶', 'ヤユスルノヤ'],
        ['餝', 'ショク，カザルノイタイジ，ショクヘンニホウコウザイノホウ'],
        ['餘', 'ヨユウノヨ，アマルノキュウジ'],
        ['餔', 'ホ，クラウ，ショクヘンニシジンノトホノホ'],
        ['餒', 'ダイ，ショクヘンニダキョウスルノダ，ウエル'],
        ['餓', 'ガシスル ノ ガ，ウエル'],
        ['揺', 'ユレル，ドウヨウスル ノ ヨウ'],
        ['餌', 'エサ，エヅケ ノ エ'],
        ['養', 'エイヨウ ノ ヨウ，ヤシナウ'],
        ['握', 'アクリョク ノ アク，ニギル'],
        ['揣', 'シマオクソクノシ，テヘンニタンマツノタンノツクリ'],
        ['揮', 'シキスル ノ キ'],
        ['餅', 'モチツキ ノ モチ'],
        ['揩', 'カイ，テヘンニミナサマノミナ'],
        ['餾', 'リュウ，ムス，ショクヘンニリュウガクノリュウ'],
        ['揖', 'イチユウ ノ ユウ，テヘン ニ クチ ノ シタニ ミミ'],
        ['餽', 'キ，マツル，ショクヘンニオニ'],
        ['提', 'テイアン ノ テイ'],
        ['雅', 'ユウガ ノ ガ，ミヤビ'],
        ['插', 'ソウニュウスルノソウ，サスノキュウジ'],
        ['揚', 'アゲル，カラアゲ'],
        ['換', 'カンキセン ノ カン，カエル'],
        ['揄', 'ヤユスルノユ'],
        ['雋', 'フルトリノシタニオウトツノオウノシタ１カクヲトッタモノ'],
        ['揆', 'ヒャクショウイッキノキ'],
        ['揀', 'カン，テヘンニサカナノニシンノツクリ'],
        ['館', 'トショカン ノ カン'],
        ['揃', 'ソロエル'],
        ['餤', 'タン，クラウ，ショクヘンニホノオ'],
        ['描', 'エガク，ビョウシャスル ノ ビョウ'],
        ['揉', 'モム，テヘンニジュウドウノジュウ'],
        ['餠', 'モチツキノモチノキュウジ'],
        ['餡', 'クズアンノアン'],
        ['雖', 'イエドモ'],
        ['淌', 'ショウ，サンズイニオショウサンノショウ'],
        ['釶', 'シ、カネヘンニクウヤネンブツノヤ'],
        ['釵', 'サイ、カネヘンニヤシャノシャ'],
        ['淋', 'サビシイ，サンズイ ニ ハヤシ'],
        ['釿', 'ギン、カネヘンニパンイッキンノキン'],
        ['淅', 'セキ，サンズイニブンセキスルノセキ'],
        ['淆', 'ギョクセキコンコウノコウ'],
        ['淇', 'キ，サンズイニキホンノキカラツチヲトッタカタチ'],
        ['淀', 'ヨドガワ ノ ヨド'],
        ['釧', 'クシロ，コンセンゲンヤ ノ セン'],
        ['釦', 'イフク ノ ボタン'],
        ['淞', 'ショウ，サンズイニマツノキノマツ'],
        ['淘', 'シゼントウタ ノ トウ'],
        ['淙', 'ソウ，サンズイニシュウキョウカイカクノシュウ'],
        ['釡', 'カマメシノカマノイタイジ'],
        ['淕', 'リク，サンズイニリクジョウノリクノツクリ'],
        ['淑', 'シュクジョ ノ シュク'],
        ['淒', 'セイ，サンズイニツマオットノツマ'],
        ['淬', 'サイ，サンズイニソツギョウノソツ'],
        ['釖', 'トウ、カネヘンニカタナ'],
        ['淮', 'ワイガノワイ，サンズイニフルトリ'],
        ['淨', 'ジョウスイキノジョウノキュウジ'],
        ['淪', 'リン，サンズイニケツロンノロンノツクリ'],
        ['淫', 'インラン ノ イン，ミダラ'],
        ['淤', 'オケツノオ，トドコオル'],
        ['淦', 'カン，サンズイニキンヨウビノキン'],
        ['釜', 'カマメシ ノ カマ'],
        ['釛', 'コク，カネヘンニチカラ'],
        ['淡', 'タンスイギョ ノ タン，アワイ'],
        ['n', 'エヌ ノベンバー'],
        ['采', 'ハクシュカッサイ ノ サイ'],
        ['釆', 'ハン，バンゴウ ノ バン カラ タ ヲ トッタ ジ'],
        ['淹', 'エン，サンズイニアマミオオシマノイチモジメ'],
        ['淺', 'アサイノキュウジ'],
        ['添', 'ソエル，テンプスル ノ テン'],
        ['量', 'コウスイリョウ ノ リョウ，ハカル'],
        ['淵', 'ミズガ ヨドンデ フカイ フチ'],
        ['重', 'タイジュウ ノ ジュウ，オモイ'],
        ['混', 'コンザツ ノ コン'],
        ['釋', 'シャクホウスルノシャクノキュウジ'],
        ['深', 'シンコクナ ノ シン，フカイ'],
        ['釉', 'ユウヤクノユウ，ウワグスリ'],
        ['淳', 'ジュン，サンズイ ニ キョウラクテキナ ノ キョウ'],
        ['副', 'フクカイチョウ ノ フク'],
        ['制', 'セイゲンスル ノ セイ'],
        ['刷', 'インサツ ノ サツ，スル'],
        ['刳', 'クリヌクノクル，コ'],
        ['到', 'トウチャクスル ノ トウ，イタル'],
        ['刺', 'シゲキスル ノ シ，サス'],
        ['刻', 'ジコク ノ コク，キザム'],
        ['券', 'ジョウシャケン ノ ケン'],
        ['刹', 'セツナテキナノナ'],
        ['刧', 'ミライエイゴウノゴウノイタイジ'],
        ['判', 'ハンダン ノ ハン'],
        ['別', 'トクベツ ノ ベツ，ワカレル'],
        ['刮', 'カツモクスルノカツ'],
        ['刪', 'サン，ケズル，イッサツニサツノサツノミギニリットウ'],
        ['利', 'ケンリ ノ リ'],
        ['列', 'ギョウレツ ノ レツ'],
        ['刔', 'ケツ，エグル，タスウケツノケツノツクリノミギニリットウ'],
        ['刑', 'ケイムショ ノ ケイ'],
        ['初', 'サイショ ノ ショ，ハジメテ'],
        ['分', 'ハンブン ノ ブン，ワケル'],
        ['切', 'タイセツ ノ セツ，キル'],
        ['刄', 'ハモノノハ，ヤイバノイタイジ'],
        ['刃', 'ハモノ ノ ハ，ヤイバ'],
        ['刀', 'カタナ，ニホントウ ノ トウ'],
        ['刎', 'フンケイノマジワリノフン'],
        ['刊', 'シュウカンシ ノ カン'],
        ['刋', 'セン，カタカナノチノミギニリットウ'],
        ['刈', 'クサカリ ノ カリ，カル'],
        ['軟', 'ジュウナン ノ ナン，ヤワラカ'],
        ['項', 'コウモク ノ コウ，キョウツウコウ ノ コウ'],
        ['D', 'ディー デスク'],
        ['觜', 'クチバシ，シ，ヒガンシガンノシノシタニツノ'],
        ['觝', 'テイ，ツノヘンニテイコウスルノテイノツクリ'],
        ['觚', 'コ，ツノヘンニウリ'],
        ['角', 'カクド ノ カク，ツノ'],
        ['觀', 'カンキャクノカン，ミルノキュウジ'],
        ['觸', 'セッショクスルノショク，フレルノキュウジ'],
        ['觴', 'ショウ、ツノヘンニキリキズノキズノツクリ'],
        ['触', 'セッショクスル ノ ショク，フレル'],
        ['觧', 'カイ，ツノヘンニヒツジ'],
        ['解', 'カイケツスル ノ カイ'],
        ['脳', 'ズノウ ノ ノウ'],
        ['脱', 'ヌグ，ダツボウ ノ ダツ'],
        ['脾', 'ゾウキノヒゾウノヒ'],
        ['脹', 'ボウチョウスル ノ チョウ，ニクヅキニナガイ'],
        ['脣', 'シン，クチビル，ジュウニシノタツノシタニニクヅキ'],
        ['脯', 'ホ，ニクヅキニシジンノトホノホ'],
        ['脩', 'ソクシュウヲオサメルノシュウ'],
        ['脛', 'ケイ，スネ，ニクヅキニケイツイノケイノヒダリガワ'],
        ['脚', 'キャクホン ノ キャク，アシ'],
        ['脇', 'ワキヤク ノ ワキ'],
        ['脆', 'ゼイジャクナ ノ ゼイ，モロイ'],
        ['脅', 'キョウハク ノ キョウ，オドス'],
        ['脂', 'シボウ ノ シ，アブラ'],
        ['脊', 'セキズイ ノ セキ'],
        ['脉', 'サンミャクノミャクノイタイジ，ニクヅキニエイキュウノエイ'],
        ['脈', 'サンミャク ノ ミャク'],
        ['缸', 'コウ，ホトギヘンニズガコウサクノコウ'],
        ['缺', 'ホケツノケツ，カケルノキュウジ'],
        ['缶', 'カンヅメ ノ カン'],
        ['搦', 'カラメテノカラメ，テヘンニキョウジャクノジャク'],
        ['搭', 'トウジョウテツヅキ ノ トウ'],
        ['搬', 'ハンニュウスル ノ ハン'],
        ['搨', 'トウ，テヘンニニチヨウビノニチノシタニハネ'],
        ['搴', 'ケン，サムイノシタニテンノカワリニテ'],
        ['搶', 'ソウ，テヘンニカマクラバクフノクラ'],
        ['搾', 'サクシュ ノ サク，シボル'],
        ['携', 'ケイタイ ノ ケイ，タズサエル'],
        ['搆', 'コウ，テヘンニミゾノツクリ'],
        ['損', 'ソンガイ ノ ソン'],
        ['搏', 'ミャクハクノハク，テヘンニハクブツカンノハクノツクリ'],
        ['搗', 'モチツキノツキ'],
        ['搖', 'ユレル，ドウヨウスルノヨウノキュウジ'],
        ['搓', 'サ，テヘンニサベツカスルノサ'],
        ['搜', 'ソウサジョウノソウ，サガスノキュウジ'],
        ['瓱', 'ミリグラム，カワラニウモウノモウ'],
        ['瓰', 'デシグラム，カワラニハンブンニスルノブン'],
        ['瓲', 'トン，カワラニチュウトンスルノトン'],
        ['瓷', 'シ，カワラノウエニモクジノジ'],
        ['瓶', 'アキビン ノ ビン'],
        ['瓸', 'ヘクトグラム，カワラニカンスウジノヒャク'],
        ['瓠', 'コ，ヒサゴ，ハカマノツクリニウリ'],
        ['瓣', 'ハナビラノベンノキュウジ'],
        ['瓢', 'ヒョウタン ノ ヒョウ，ヒサゴ'],
        ['瓧', 'デカグラム，カワラニカンスウジノジュウ'],
        ['瓦', 'レンガ ノ ガ，カワラ'],
        ['瓩', 'キログラム，カワラニカンスウジノセン'],
        ['瓮', 'オウ，オオヤケノシタニカワラ'],
        ['瓔', 'ヨウ，オウヘンニサクラノキュウジノツクリ'],
        ['瓜', 'ウリ，スイカ ノ カ'],
        ['瓊', 'ケイ，ヤサカニノマガタマノニ'],
        ['瓏', 'レイロウタルノロウ，オウヘンニキョウリュウノリュウノキュウジ'],
        ['隠', 'インキョスル ノ イン，カクレル'],
        ['隣', 'トナリ，リンジン ノ リン'],
        ['捺', 'ナツインスル ノ ナツ'],
        ['捻', 'ネンザ ノ ネン，ヒネル'],
        ['隧', 'トンネルヲイミスルスイドウノスイ'],
        ['捶', 'スイ，テヘンニスイチョクノスイ'],
        ['捷', 'ビンショウナ ノ ショウ'],
        ['險', 'ボウケンカノケンノキュウジ'],
        ['捲', 'マクル，ケンドジュウライ ノ ケン'],
        ['据', 'スエル，スエオキ ノ スエ'],
        ['隱', 'イン、カクレルノキュウジ'],
        ['隲', 'シツ、コザトヘンニスコシノシタニバシャノバ'],
        ['隴', 'リョウ、コザトヘンニキョウリュウノリュウノキュウジ'],
        ['捫', 'モン，テヘンニセンモンノモン'],
        ['捨', 'シシャゴニュウ ノ シャ，ステル'],
        ['捩', 'レイ，ネジル，テヘンニモドル'],
        ['隸', 'レイ、ドレイカイホウノレイノキュウジ'],
        ['捧', 'ササゲル，ホウ'],
        ['隻', 'フネイッセキ ノ セキ'],
        ['隼', 'トリ ノ ハヤブサ'],
        ['捜', 'ソウサジョウ ノ ソウ，サガス'],
        ['隅', 'ヨスミ ノ スミ，グウ'],
        ['隆', 'リュウキスル ノ リュウ'],
        ['隈', 'クマ，カイワイ ノ ワイ'],
        ['捗', 'シンチョクジョウキョウ ノ チョク，ハカドル'],
        ['隊', 'グンタイ ノ タイ'],
        ['捕', 'タイホスル ノ ホ，トラエル'],
        ['隍', 'コウ、コザトヘンニコウタイシノコウ'],
        ['捐', 'ギエンキンノエン，テヘンニキヌイトノキヌノツクリ'],
        ['随', 'ズイヒツ ノ ズイ'],
        ['捏', 'ネツゾウスルノネツ，コネル'],
        ['捌', 'ウリサバク ノ サバク，テヘン ニ ワカレル'],
        ['捍', 'カン，テヘンニヒデリヲイミスルカンバツノカン'],
        ['隔', 'カンカク ノ カク，ヘダタリ'],
        ['隕', 'インセキノイン、オチル'],
        ['捉', 'トラエル，テヘン ニ アシ'],
        ['隘', 'アイロノアイ、セマイ'],
        ['隙', 'スキマ ノ スキ，ゲキ'],
        ['際', 'ジッサイ ノ サイ'],
        ['障', 'ショウガイブツ ノ ショウ'],
        ['↓', 'シタヤジルシ'],
        ['壥', 'ツチヘンニガンダレニボクジュウノボクノキュウジ'],
        ['壤', 'ドジョウカイリョウノジョウノキュウジ'],
        ['壯', 'ソウダイナノソウノキュウジ'],
        ['壮', 'ソウダイナ ノ ソウ'],
        ['壬', 'ジンシンノラン ノ ジン'],
        ['士', 'ベンゴシ ノ シ'],
        ['壷', 'オモウツボ ノ ツボ'],
        ['売', 'ショウバイ ノ バイ，ウル'],
        ['壱', 'ケイヤクショニ ツカウ スウジ ノ イチ'],
        ['声', 'コエ，オンセイ ノ セイ'],
        ['[', 'カクカッコ'],
        ['壼', 'コン，サムライノシタニワカンムリ，アネッタイノアノキュウジ'],
        ['壻', 'ハナムコノムコノイタイジ，ツチヘン'],
        ['壺', 'オモウツボノツボノセイジ'],
        ['壹', 'ケイヤクショノツカウスウジイチノキュウジ'],
        ['壇', 'ブツダン ノ ダン'],
        ['壅', 'ヨウ，ヨウリツスルノヨウノツクリノシタニツチ'],
        ['壁', 'ヘキガ ノ ヘキ，カベ'],
        ['壌', 'ドジョウカイリョウ ノ ジョウ'],
        ['壊', 'ハカイ ノ カイ，コワス'],
        ['⌒', 'コキゴウ'],
        ['壗', 'ママ，ツチヘンニワガママノママノツクリ'],
        ['壕', 'ボウクウゴウ ノ ゴウ'],
        ['壓', 'アツリョクノアツノキュウジ'],
        ['壑', 'タニヲイミスルガク'],
        ['(', 'カッコ'],
        ['壟', 'ロウ，キョウリュウノリュウノキュウジノシタニツチ'],
        ['壞', 'コワス，ハカイノカイノキュウジ'],
        ['壜', 'ビン，ツチヘンニクモリ'],
        ['壙', 'コウ，ツチヘンニヒロバノヒロノキュウジ'],
        ['壘', 'ルイセキスルノルイノキュウジ'],
        ['冽', 'リンレツノレツ，ニスイニギョウレツノレツ'],
        ['冰', 'ヒョウガノヒョウ，コオリノイタイジ，ニスイ'],
        ['冱', 'コ，ニスイニゴカクノゴ'],
        ['冲', 'チュウ，ニスイニチュウガッコウノチュウ'],
        ['决', 'タスウケツノケツ，キメルノイタイジ，ニスイ'],
        ['冴', 'メガサエル ノ サエル'],
        ['况', 'ジッキョウチュウケイノキョウノイタイジ，ニスイ'],
        ['冶', 'ヤキンガク ノ ヤ'],
        ['冷', 'レイトウ ノ レイ，ツメタイ'],
        ['冨', 'フジサン ノ フ ノ イタイジ，ワカンムリ'],
        ['冩', 'シャシンノシャ，ウツスノキュウジノイタイジ，ワカンムリ'],
        ['冪', 'ベキ，ワカンムリニジマクノマク'],
        ['冫', 'ヒョウ，ブシュノニスイ'],
        ['冬', 'フユヤスミ ノ フユ，トウ'],
        ['冠', 'カンムリ，エイカン ノ カン'],
        ['冢', 'チョウ，カイヅカノツカノモトノジ'],
        ['冤', 'エンザイヲハラスノエン'],
        ['冥', 'メイフクヲ イノル ノ メイ'],
        ['冦', 'モウコシュウライノゲンコウノコウノイタイジ'],
        ['写', 'シャシン ノ シャ，ウツス'],
        ['冐', 'ボウケンカノボウノイタイジ'],
        ['冑', 'カッチュウノチュウ，カブト'],
        ['冒', 'ボウケンカ ノ ボウ'],
        ['冓', 'コウ，ミゾノツクリ'],
        ['冕', 'ベン，カンムリ'],
        ['冖', 'ベキ，ブシュノワカンムリ'],
        ['冗', 'ジョウダンヲイウ ノ ジョウ'],
        ['冉', 'ゼン，フタタビカライッカクメヲトッタカタチ'],
        ['冊', 'イッサツニサツ ノ サツ'],
        ['册', 'イッサツニサツノサツノキュウジ'],
        ['再', 'フタタビ，サイ'],
        ['冏', 'ケイ，ドウガマエニカタカナノルトロ'],
        ['冀', 'キ，キタノシタニコトナル'],
        ['冂', 'ケイ，ブシュノドウガマエ'],
        ['内', 'コクナイ ノ ナイ，ウチ'],
        ['円', 'ヒャクエンダマ ノ エン'],
        ['輌', 'リョウ、クルマヘンニリョウテノリョウ'],
        ['輊', 'チ、クルマヘンニゲシトウジノシ'],
        ['載', 'レンサイ ノ サイ，ノセル'],
        ['輅', 'ロ、クルマヘンニカクジノカク'],
        ['較', 'ヒカクテキ ノ カク'],
        ['輟', 'テツ、クルマヘンニツヅルノツクリ'],
        ['輝', 'カガヤク，キ'],
        ['輜', 'シチョウヘイノシ'],
        ['輛', 'リョウ、クルマヘンニリョウテノリョウノキュウジ'],
        ['輙', 'チョウ，クルマヘンニウケトルノトル'],
        ['輕', 'カルイオモイノカルイノキュウジ'],
        ['輔', 'クルマヘン ノ ホサ ノ ホ'],
        ['輓', 'バン，クルマヘンニメンキョノメン'],
        ['輒', 'チョウ，クルマヘンニミミニオツ'],
        ['輯', 'ヘンシュウ ノ シュウ，クルマヘン ニ クチニ ミミ'],
        ['輪', 'シャリン ノ リン，ワ'],
        ['輩', 'センパイ コウハイ ノ ハイ'],
        ['輦', 'ホウレンノレン、オットフタツノシタニクルマ'],
        ['1', 'イチ'],
        ['輾', 'テン、クルマヘンニハッテンノテン'],
        ['輻', 'フク、クルマヘンニシュクフクノフクノツクリ'],
        ['輹', 'フク、クルマヘンニオウフクスルノフクノツクリ'],
        ['輸', 'ユニュウ ノ ユ'],
        ['輳', 'ソウ、クルマヘンニカナデル'],
        ['虧', 'キ，トラガシラニフルトリミギニオセンスルノオノツクリ'],
        ['虫', 'ムシ，コンチュウ ノ チュウ'],
        ['虱', 'シラミ，カゼノイッカクメヲトッタモノ'],
        ['虹', 'ニジ，ナナイロ ノ ニジ'],
        ['虻', 'アブハチトラズ ノ アブ'],
        ['虍', 'コ，ブシュノトラガシラ'],
        ['虎', 'トラ，モウコ ノ コ'],
        ['虐', 'ギャクタイ ノ ギャク，シイタゲル'],
        ['虔', 'ケイケンナイノリノケン'],
        ['處', 'ショブンスルノショノキュウジ'],
        ['虚', 'キョエイシン ノ キョ'],
        ['虜', 'ホリョ ノ リョ，トリコ'],
        ['虞', 'グビジンソウ ノ グ'],
        ['號', 'バンゴウノゴウノキュウジ'],
        ['酸', 'サンソ ノ サン'],
        ['浜', 'スナハマ ノ ハマ'],
        ['酲', 'テイ，ジュウニシノトリノミギニゾウテイスルノテイ'],
        ['浚', 'シュセツスルノシュン'],
        ['酷', 'ザンコク ノ コク，ヒドイ'],
        ['浙', 'セッコウショウノセツ，サンズイニコッセツスルノセツノツクリ'],
        ['酩', 'メイテイスルノメイ'],
        ['浄', 'ジョウスイキ ノ ジョウ'],
        ['浅', 'アサイ フカイ ノ アサイ'],
        ['酬', 'ホウシュウ ノ シュウ'],
        ['流', 'リュウコウ ノ リュウ，ナガス'],
        ['酣', 'カン，ジュウニシノトリノミギニアマイ'],
        ['酢', 'スノモノ ノ ス'],
        ['酥', 'ソ，ジュウニシノトリノミギニノギヘンノノギ'],
        ['海', 'ウミ，カイガイ ノ カイ'],
        ['浴', 'ニュウヨクスル ノ ヨク，アビル'],
        ['酒', 'サケ，ニホンシュ ノ シュ'],
        ['酔', 'ヨウ，デイスイ ノ スイ'],
        ['浸', 'シントウスル ノ シン，ヒタス'],
        ['浹', 'ショウ，サンズイニセマイノキュウジノツクリ'],
        ['浦', 'ツツウラウラ ノ ウラ'],
        ['浤', 'コウ，サンズイニウカンムリニカタカナノナトム'],
        ['酊', 'メイテイスルノテイ'],
        ['配', 'ハイタツ ノ ハイ，クバル'],
        ['浣', 'カンチョウスルノカン，サンズイニカンリョウスルノカン'],
        ['酎', 'イモジョウチュウ ノ チュウ'],
        ['浮', 'フリョク ノ フ，ウク'],
        ['浬', 'タンイ ノ カイリ，サンズイ ニ サト'],
        ['浪', 'ロウヒスル ノ ロウ'],
        ['浩', 'コウゼン ノ コウ，サンズイ ニ ツゲル'],
        ['羣', 'グンシュウノグン，ムレルノイタイジ'],
        ['群', 'グンシュウ ノ グン，ムレ'],
        ['羨', 'センボウスル ノ セン，ウラヤム'],
        ['義', 'ギム ノ ギ'],
        ['羮', 'ミズヨウカンノカン，アツモノノイタイジ'],
        ['羯', 'ガッキノカッコノカツ'],
        ['羲', 'チュウゴクノデンセツジョウノテンシ，フクギノギ'],
        ['丼', 'ドンブリ'],
        ['羶', 'セン，ヒツジヘンニブツダンノダンノツクリ'],
        ['羸', 'ルイジャクノルイ'],
        ['羹', 'ミズヨウカンノカン，アツモノ'],
        ['羽', 'ハネ，ウモウ ノ ウ'],
        ['羂', 'ケン，アミガシラニキヌイトノキヌ'],
        ['羃', 'ベキ，アミガシラニジマクノマク'],
        ['羆', 'ヒグマ，アミガシラニクマ'],
        ['羇', 'キ，アミガシラニカクメイノカクニキミョウナノキ'],
        ['羅', 'モウラスル ノ ラ'],
        ['羊', 'ヨウモウ ノ ヨウ，ヒツジ'],
        ['羈', 'キビセイサクノキ'],
        ['美', 'ビジン ノ ビ，ウツクシイ'],
        [
            '羌',
            'チュウゴクホクセイブミンゾクノヒトツ，キョウ，ヒツジノシタニヒトアシ',
        ],
        ['羔', 'コウ，コヒツジ，ヒツジノシタニレンガ'],
        ['羚', 'レイ，ヒツジヘンニメイレイスルノレイ'],
        ['羞', 'シュウチシンノシュウ'],
        ['羝', 'テイ，オヒツジ，ヒツジヘンニテイコウスルノテイノツクリ'],
        ['斯', 'シカイ ノ ケンイ ノ シ，カク'],
        ['断', 'ハンダン ノ ダン'],
        ['斬', 'ザンシンナ ノ ザン，キル'],
        ['斫', 'シャク，イシヘンニパンイッキンノキン'],
        ['齔', 'シン，ハノキュウジノミギニカタカナノヒ'],
        ['斧', 'ドウグ ノ オノ'],
        ['斥', 'ハイセキスル ノ セキ，シリゾケル'],
        ['斤', 'ショクパンイッキン ノ キン'],
        ['斡', 'アッセンスル ノ アツ'],
        ['施', 'シセツ ノ シ，ホドコス'],
        ['於', 'ナニナニニオイテ ノ オ'],
        ['方', 'オヤカタ ノ カタ，ホウガク ノ ホウ'],
        ['斷', 'ダンネンスルノダンノキュウジ'],
        ['齋', 'ショサイノサイノキュウジ'],
        ['齊', 'イッセイシャゲキノセイノキュウジ'],
        ['齏', 'ナマスヲイミスルセイ'],
        ['新', 'アタラシイ，シンブン ノ シン'],
        ['斎', 'ショサイニコモル ノ サイ'],
        ['斌', 'ヒン，ブンブリョウドウ ノ ブント ブ'],
        ['斉', 'イッセイシャゲキ ノ セイ'],
        ['斈', 'マナブ，ガクシュウノガクノイタイジ'],
        ['文', 'サクブン ノ ブン'],
        ['斃', 'ノタレジニヲイミスルヘイシノヘイ'],
        ['斂', 'イッテンニシュウレンスルノレン'],
        ['斟', 'シンシャクスルノシン，クム'],
        ['齠', 'チョウ，ハノキュウジノミギニショウシュウレイジョウノショウ'],
        ['齣', 'コマ，ハノキュウジノミギニハイクノク'],
        ['斜', 'ナナメ，シャメン ノ シャ'],
        ['斛', 'ヨウリョウノタンイノコク，ツノヘンニタンイノト'],
        ['料', 'リョウリスル ノ リョウ'],
        ['齦', 'ギン，ハグキ'],
        ['斗', 'ホクトシチセイ ノ ト'],
        ['齪', 'アクセクスルノセク，ハノキュウジノミギニアシ'],
        ['齬', 'ソゴヲキタスノゴ'],
        ['斑', 'マダラ，ハンテン ノ ハン'],
        ['斐', 'ヒ，カイノクニ ノ ニモジメ'],
        ['剄', 'ケイドウミャクノケイノヒダリガワニリットウ'],
        ['並', 'ナミキ ノ ナミ，ナラブ'],
        ['不', 'フシギ ノ フ'],
        ['下', 'ジョウゲ ノ ゲ，シタ'],
        ['塩', 'ショクエン ノ エン，シオ'],
        ['填', 'ジュウテンスル ノ テン，ハメル'],
        ['H', 'エイチ ホテル'],
        ['塢', 'オ，ツチヘンニウゴウノシュウノウ'],
        ['塹', 'ザンゴウノザン'],
        ['塾', 'ガクシュウジュク ノ ジュク'],
        ['塰', 'アマ，ウミノシタニツチ'],
        ['塲', 'バショノバノイタイジ'],
        ['塵', 'フンジン ノ ジン，チリ'],
        ['塋', 'エイ，ハカ，エイギョウノエイノキュウジノロノカワリニツチ'],
        ['塊', 'ダンカイ ノ カイ，カタマリ'],
        ['塁', 'マンルイ ノ ルイ'],
        ['塀', 'ヘイ，イタベイ ノ ヘイ'],
        ['塙', 'ハナワ，ツチヘン ニ タカイ'],
        ['塘', 'トウ，ツチヘン ニ ケントウシ ノ トウ'],
        ['塚', 'ツカ，カイヅカ ノ ツカ'],
        ['塞', 'ノウコウソク ノ ソク，フサガル'],
        ['塑', 'カソセイ ノ ソ'],
        ['塒', 'ジ，ネグラ，ツチヘンニニチジノジ'],
        ['塔', 'ゴジュウ ノ トウ'],
        ['塗', 'トソウ ノ ト，ヌル'],
        ['膸', 'セキズイノズイノイタイジ'],
        ['膺', 'ケンケンフクヨウノヨウ'],
        ['膽', 'ダイタンナノタンノキュウジ'],
        ['膿', 'カノウスル ノ ノウ，ウミ'],
        ['膾', 'ジンコウニカイシャスルノカイ，ナマス'],
        ['膰', 'ハン，ニクヅキニバンゴウノバン'],
        ['膳', 'アゲゼンスエゼン ノ ゼン'],
        ['膵', 'スイゾウノスイ'],
        ['膩', 'ジ，ニクヅキニケイヤクショニツカウカンスウジノニノキュウジ'],
        ['膨', 'ボウチョウスル ノ ボウ，フクラム'],
        ['膠', 'コウゲンビョウノコウ，ニカワ'],
        ['膣', 'チツ，ニクヅキニチッソノチツ'],
        ['膤', 'ユキ，ニクヅキニユキグニノユキ'],
        ['膚', 'ヒフ ノ フ，ハダ'],
        ['膝', 'ヒザマクラ ノ ヒザ'],
        ['膜', 'モウマク ノ マク'],
        ['膓', 'チョクチョウノチョウノイタイジ，ニクヅキニキリキズノキズノツクリ'],
        ['膕', 'カク，ニクヅキニユキグニノクニノキュウジ'],
        ['膈', 'カク，ニクヅキニヘダテルノツクリ'],
        ['膊', 'ジョウハクコツノハク'],
        ['膏', 'ナンコウヲヌル ノ コウ'],
        ['膀', 'ボウコウエンノボウ'],
        ['膃', 'オットセイノオツ'],
        ['膂', 'リョ，リョヒノリョノシタニニクヅキ'],
        ['辛', 'コウシンリョウ ノ シン，カライ'],
        ['辜', 'ムコノコ，チュウコシャノコノシタニカライ'],
        ['辟', 'ヘキエキスルノヘキ'],
        ['辞', 'コクゴジテン ノ ジ'],
        ['辱', 'ブジョクスル ノ ジョク，ハズカシメル'],
        ['辰', 'タツ，ジュウニシ ノ タツ'],
        ['農', 'ノウギョウ ノ ノウ'],
        ['辷', 'シンニョウニカンスウジノイチ，スベル'],
        ['辻', 'ヨツツジ ノ ツジ'],
        ['辺', 'シュウヘン ノ ヘン，アタリ'],
        ['込', 'モウシコム ノ コム'],
        ['辿', 'タドル'],
        ['辣', 'ラツワンノラツ'],
        ['辧', 'ベンベツスルノベンノキュウジ，カライ，シンフタツノアイダニカタナ'],
        [
            '辨',
            'ベンベツスルノベンノキュウジ，カライ，シンフタツノアイダニリットウ',
        ],
        ['辭', 'コクゴジテンノジノキュウジ'],
        ['辯', 'ベンゴシノベンノキュウジ，カライフタツノアイダニゲンゴノゲン'],
        ['辮', 'ベンパツノベン，カライ，シンフタツノアイダニイト'],
        ['⊇', 'ブブンシュウゴウギャク'],
        ['⊆', 'ブブンシュウゴウ'],
        ['睹', 'ト，メヘンニカガクシャノシャノキュウジタイ'],
        ['⊃', 'シンブブンシュウゴウギャク'],
        ['睾', 'ダンセイノコウガンノコウ'],
        ['睫', 'マツゲ，ショウ'],
        ['睨', 'ヘイゲイスルノゲイ，ニラム'],
        ['督', 'エイガカントク ノ トク'],
        ['睡', 'スイミン ノ スイ'],
        ['睦', 'シンボクカイ ノ ボク，ムツマジイイ'],
        ['睥', 'ヘイゲイスルノヘイ，ニラム'],
        ['睛', 'ガリョウテンセイヲカクノセイ，ヒトミ'],
        ['睚', 'ガイ，メヘンニイッショウガイノガイノツクリ'],
        ['⊥', 'スイチョク'],
        ['⊿', 'サンカクケイ'],
        ['着', 'キモノ ノ キ，チャク'],
        ['睇', 'テイ，メヘンニアニオトウトノオトウト'],
        ['阪', 'オオサカ ノ サカ，ハン'],
        ['阨', 'アイ，コザトヘンニヤクドシノヤク'],
        ['阮', 'ゲン，コザトヘンニガンタンノガン'],
        ['阯', 'シ，コザトヘンニキンシスルノシ'],
        ['阡', 'セン，コザトヘンニカンスウジノセン'],
        ['阻', 'ソシスル ノ ソ，ハバム'],
        ['阿', 'アミダブツ ノ ア'],
        ['防', 'ヨボウスル ノ ボウ，フセグ'],
        ['阜', 'ギフケン ノ フ，オカ'],
        ['儂', 'ノウ，ニンベンニノウギョウノノウ'],
        ['儀', 'ギシキ ノ ギ'],
        ['儁', 'シュン，ニンベンニフルトリニユミ'],
        ['億', 'オクマンチョウジャ ノ オク'],
        ['儉', 'ケンヤクスルノケンノキュウジ'],
        ['儒', 'ジュガク ノ ジュ'],
        ['儖', 'ラン，ニンベンニカントクスルノカン'],
        ['儔', 'チュウ，ニンベンニコトブキノキュウジ'],
        ['儕', 'セイ，ニンベンニイッセイシャゲキノセイノキュウジ'],
        ['儚', 'ボウ，ニンベンニユメ，ハカナイ'],
        ['儘', 'ワガママノママ，ジン'],
        ['償', 'バイショウキン ノ ショウ，ツグナウ'],
        ['儡', 'カイライセイケンノライ'],
        ['優', 'ユウショウスル ノ ユウ，ヤサシイ'],
        ['儲', 'モウカル，ニンベン ニ ショクン ノ ショ'],
        ['儷', 'レイ，ニンベンニウルワシイ'],
        ['儺', 'ダ，ニンベンニコンナンノナン'],
        ['儻', 'トウ，ニンベンニヨトウヤトウノトウノキュウジ'],
        ['儿', 'ジン，ブシュノヒトアシ'],
        ['儼', 'ゲン，ニンベンニゲンジュウナノゲン'],
        ['滝', 'タキツボ ノ タキ'],
        ['滞', 'ジュウタイスル ノ タイ，トドコオル'],
        ['滑', 'エンカツ ノ カツ，ナメラカ'],
        ['滓', 'ザンシノシ，サンズイニウカンムリニカライ'],
        ['滔', 'ミズガトウトウトナガレルノトウ'],
        ['滕', 'トウ，フジノハナノフジノクサカンムリナシ'],
        ['滉', 'コウ，サンズイニホロバシャノホロノツクリ'],
        ['滋', 'ジヨウキョウソウ ノ ジ'],
        ['滌', 'テキ，サンズイニジョウケンノジョウノキュウジ'],
        ['滂', 'ボウ，サンズイニロボウノボウノツクリ'],
        ['滄', 'ソウ，サンズイニカマクラバクフノクラ'],
        ['滅', 'ハメツスル ノ メツ，ホロビル'],
        ['滸', 'スイコデンノコ'],
        ['滾', 'タギル，コン'],
        ['滿', 'マンゾクノマンノキュウジ'],
        ['滲', 'シン，ニジム，サンズイニサンカスルノサンノキュウジ'],
        ['滴', 'スイテキ ノ テキ，シズク'],
        ['滷', 'ロ，サンズイニロカクヒンノロ'],
        ['滬', 'コ，サンズイニバッコスルノコ'],
        ['滯', 'ジュウタイスルノタイ，トドコオルノキュウジ'],
        ['_', 'アンダーライン'],
        ['蛩', 'キョウ、キョウリュウノキョウノココロノカワリニムシ'],
        ['蛮', 'ナンバンボウエキ ノ バン'],
        ['蛯', 'エビ，ムシヘンニロウジンノロウ'],
        ['蛬', 'キョウ、キョウツウスルノキョウノシタニムシ'],
        ['蛭', 'チヲスウドウブツ ノ ヒル'],
        ['蛤', 'ハマグリ，ムシヘン ニ アウ'],
        ['蛻', 'モヌケノカラノモヌケ'],
        ['蛸', 'ギョカイルイ ノ タコ'],
        ['蛹', 'ムシノサナギ、ヨウ'],
        ['蛾', 'ガ，ムシ ノ ガ'],
        ['蛋', 'タンパクシツ ノ タン'],
        ['蛉', 'トンボノニモジメ，レイ，ムシヘンニメイレイスルノレイ'],
        ['蛎', 'ウミ ノ カキ，レイ'],
        ['蛍', 'ホタル，ケイコウトウ ノ ケイ'],
        ['蛆', 'ウジムシノウジ，ショ'],
        ['蛇', 'ヘビ，ダイジャ ノ ジャ'],
        ['蛄', 'スシネタノシャコノコ'],
        ['蛛', 'クモ，ムシヘンニシュイロノシュ'],
        ['蛙', 'ドウブツ ノ カエル'],
        ['蛞', 'ナメクジノイチモジメ，ムシヘンニシタヲカムノシタ'],
        ['蛟', 'コウ、ミズチ、ムシヘンニマジワル'],
        ['蛔', 'カイ、ムシヘンニカイスウケンノカイ'],
        ['Ψ', 'ギリシャ プサイ'],
        ['Ω', 'ギリシャ オメガ'],
        ['Π', 'ギリシャ パイ'],
        ['Ρ', 'ギリシャ ロー'],
        ['Σ', 'ギリシャ シグマ'],
        ['Τ', 'ギリシャ タウ'],
        ['Υ', 'ギリシャ ウプシロン'],
        ['Φ', 'ギリシャ ファイ'],
        ['Χ', 'ギリシャ カイ'],
        ['θ', 'ギリシャ シータ'],
        ['ι', 'ギリシャ イオタ'],
        ['κ', 'ギリシャ カッパ'],
        ['λ', 'ギリシャ ラムダ'],
        ['μ', 'ギリシャ ミュー'],
        ['ν', 'ギリシャ ニュー'],
        ['ξ', 'ギリシャ グザイ'],
        ['ο', 'ギリシャ オミクロン'],
        ['α', 'ギリシャ アルファ'],
        ['β', 'ギリシャ ベータ'],
        ['γ', 'ギリシャ ガンマ'],
        ['δ', 'ギリシャ デルタ'],
        ['ε', 'ギリシャ イプシロン'],
        ['ζ', 'ギリシャ ゼータ'],
        ['η', 'ギリシャ イータ'],
        ['Θ', 'ギリシャ シータ'],
        ['Ι', 'ギリシャ イオタ'],
        ['Κ', 'ギリシャ カッパ'],
        ['Λ', 'ギリシャ ラムダ'],
        ['Μ', 'ギリシャ ミュー'],
        ['Ν', 'ギリシャ ニュー'],
        ['Ξ', 'ギリシャ グザイ'],
        ['Ο', 'ギリシャ オミクロン'],
        ['Α', 'ギリシャ アルファ'],
        ['Β', 'ギリシャ ベータ'],
        ['Γ', 'ギリシャ ガンマ'],
        ['Δ', 'ギリシャ デルタ'],
        ['Ε', 'ギリシャ イプシロン'],
        ['Ζ', 'ギリシャ ゼータ'],
        ['Η', 'ギリシャ イータ'],
        ['縦', 'タテヨコ ノ タテ，ジュウ'],
        ['縡', 'サイ，イトヘンニテッケツサイショウノサイ'],
        ['縢', 'イトカガリノカガル，トウ'],
        ['縣', 'トドウフケンノケンノキュウジ'],
        ['縮', 'タンシュク ノ シュク，チヂム'],
        ['縫', 'サイホウスル ノ ホウ，ヌウ'],
        ['縵', 'マン，イトヘンニマンダラノマン'],
        ['縷', 'イチルノノゾミノル'],
        ['縱', 'ジュウ，タテヨコノタテノキュウジ'],
        ['縲', 'ルイ，イトヘンニルイセキスルノルイ'],
        ['總', 'ソウリダイジンノソウノキュウジ'],
        ['績', 'セイセキヒョウ ノ セキ'],
        ['縹', 'ヒョウ，ハナダ，イトヘンニトウヒョウスルノヒョウ'],
        ['縺', 'レン，モツレル'],
        ['縻', 'キビセイサクノビ'],
        ['縄', 'ナワ，ジョウモンドキ ノ ジョウ'],
        ['縅', 'ヨロイノオドシ，イトヘンニイカクスルノイ'],
        ['縁', 'エンガワ ノ エン，フチ'],
        ['縉', 'シン，イトヘンニチュウゴクノコクメイノシン，ススムノキュウジ'],
        ['縊', 'イシスルノイ，クビル'],
        ['縋', 'スガリツクノスガル，ツイ，イトヘンニツイセキスルノツイ'],
        ['縒', 'ヨリヲモドスノヨリ，イトヘンニサベツカノサ'],
        ['縞', 'シマモヨウ ノ シマ'],
        ['縟', 'ジョク，イトヘンニブジョクスルノジョク'],
        ['縛', 'シバル，ソクバク ノ バク'],
        ['攵', 'ボク，ブシュノノブン，ハイボクスルノハイノツクリ'],
        ['攴', 'ボク，ブシュノトマタ'],
        ['攷', 'コウ，ギコウテキナノコウノツクリノミギニノブン'],
        ['收', 'シュウニュウノシュウノキュウジ'],
        ['改', 'カイリョウ ノ カイ，アラタメル'],
        ['攸', 'ユウ，ニンベンニタテボウニノブン'],
        ['攻', 'コウゲキ ノ コウ，セメル'],
        ['政', 'セイジカ ノ セイ'],
        ['放', 'ホウソウキョク ノ ホウ，ハナツ'],
        ['攣', 'ケイレンヲオコスノレン'],
        ['攤', 'タン，テヘンニコンナンナノナンノキュウジ'],
        ['騷', 'サワグ，ソウオンノソウノキュウジ'],
        ['攫', 'イッカクセンキンノカク，ツカム'],
        ['攪', 'コウランスルノコウ，ミダス'],
        ['攬', 'ラン，テヘンニハクランカイノランノキュウジ'],
        ['支', 'シエンスル ノ シ，ササエル'],
        ['鳳', 'ホウオウ ノ オウ，オオトリ'],
        ['攘', 'ソンノウジョウイノジョウ，ハラウ'],
        ['鳰', 'ミズトリノニオ，ニュウガクノニュウニトリ'],
        ['攝', 'セッセイスルノセイ，トルノキュウジ'],
        ['攜', 'ケイタイスルノケイ，タズサエルノイタイジ'],
        ['鳴', 'ヒメイヲアゲル ノ メイ，ナク'],
        ['鳫', 'ガン，ガンダレニトリ'],
        ['攀', 'ガンペキヲトウハンスルノハン'],
        ['鳩', 'トリ ノ ハト'],
        ['攅', 'サン，アツメル，テヘンニサンセイスルノサン'],
        ['鳬', 'ミズドリノケリ，トリノヨツテンノカワリニツクエノツクリ'],
        ['鬧', 'ドウ，トウガマエニイチバノイチ'],
        ['鳧', 'ミズドリノケリ，トリノシタニツクエノツクリ'],
        ['鳥', 'トリ，ヤチョウ ノ チョウ'],
        ['∈', 'ゾクスル'],
        ['∋', 'ゲントシテフクム'],
        ['∀', 'フツウゲンテイシ'],
        ['∃', 'ソンザイゲンテイシ'],
        ['∂', 'ラウンドディー'],
        ['石', 'ホウセキ ノ セキ，イシ'],
        ['短', 'チョウタン ノ タン，ミジカイ'],
        ['矯', 'キョウセイシリョク ノ キョウ'],
        ['矮', 'ワイショウナノワイ'],
        ['矩', 'クケイ ノ ク'],
        ['∟', 'チョッカク'],
        ['∞', 'ムゲンダイ'],
        ['∑', 'シグマ'],
        ['矧', 'ヤヘン ニ インヨウスル ノ イン，ハグ'],
        ['矣', 'イ，アイサツノアイノツクリ'],
        ['矢', 'ユミヤ ノ ヤ'],
        ['∩', 'キャップ'],
        ['矜', 'キョウジヲタモツノキョウ'],
        ['∫', 'インテグラル'],
        ['∪', 'カップ'],
        ['∬', 'ダブルインテグラル'],
        ['矛', 'ムジュン ノ ム，ホコ'],
        ['矚', 'ショク，メヘンニフゾクノゾクノキュウジ'],
        ['∠', 'カク'],
        ['矗', 'チク，チョクゼンノチョクガミッツ'],
        ['∥', 'タテヘイコウ'],
        ['∧', 'オヨビ'],
        ['矍', 'ゲンキカクシャクノカク'],
        ['鑵', 'カン、カネヘンニカンヅメノカンノキュウジノツクリ'],
        ['∽', 'ソウジ'],
        ['矇', 'モウ，メヘンニケイモウスルノモウ'],
        ['∵', 'ナゼナラバ'],
        ['∴', 'ユエニ'],
        ['鞭', 'キョウベンヲトル ノ ベン，ムチ'],
        ['鞨', 'カツ、カワヘンニカッショクノカツノツクリ'],
        ['鞫', 'キク，カワヘンニツツミガマエノナカニゴンベン'],
        ['鞦', 'シュウ，カワヘンニキセツノアキ'],
        ['鞠', 'ケマリ ノ マリ，キク'],
        ['鑰', 'ヤク，カネヘンニブシュノヤク'],
        ['鬥', 'トウ，ブシュノトウガマエ'],
        ['鞴', 'フイゴ，カワヘンニジュンビスルノビノツクリ'],
        ['鬢', 'ビン，カミガシラニライヒンノヒン'],
        ['鞳', 'トウ、カワヘンニゴジュウノトウノトウノツクリ'],
        ['鞍', 'アンバ ノ アン，クラ'],
        ['鞏', 'キョウ、キョウノココロノカワリニカクメイノカク'],
        ['鞋', 'ワラジノニモジメ，カイ'],
        ['鞄', 'カバン'],
        ['鞅', 'オウ，カワヘンニチュウオウノオウ'],
        ['鞆', 'トモ、カワヘンニコウオツヘイノヘイ'],
        ['鞁', 'ヒ，カワヘンニケガワノカワ'],
        ['鞜', 'トウ，カワヘンニミズノシタニニチヨウビノニチ'],
        ['鬮', 'キュウ，トウガマエニドウブツノカメノキュウジ'],
        ['鞘', 'カタナ ノ サヤ，ショウ'],
        ['v', 'ブイ ビデオ'],
        ['鞐', 'コハゼ、カワヘンニウエシタ'],
        ['鑪', 'ロ、カネヘンニハゼノキノハゼノツクリ'],
        ['鬨', 'トキノコエヲアゲルノトキ'],
        ['鑠', 'ゲンキカクシャクノシャク'],
        ['隨', 'ズイヒツノズイノキュウジ'],
        ['忱', 'シン，リッシンベンニチンボツスルノチンノツクリ'],
        ['忰', 'ショウスイスルノスイノイタイジ'],
        ['念', 'ザンネン ノ ネン'],
        ['忻', 'キン，リッシンベンニパンイッキンノキン'],
        ['忸', 'ジクジタルノジク，ハジル'],
        ['忿', 'フン，フンベツノフンノシタニココロ'],
        ['忽', 'コツゼン ノ コツ，タチマチ'],
        ['忠', 'チュウコクスル ノ チュウ'],
        ['忤', 'ゴ，リッシンベンニショウゴノゴ'],
        ['快', 'カイテキ ノ カイ，ココロヨイ'],
        ['鬼', 'オニ，キュウケツキ ノ キ'],
        ['隰', 'シツ、コザトヘンニシメッポイノキュジノツクリ'],
        ['志', 'ココロザス，イシ ノ シ'],
        ['忖', 'ソンタクスルノソン，リッシンベンニスンポウノスン'],
        ['忙', 'イソガシイ，タボウ ノ ボウ'],
        ['忘', 'ボウネンカイ ノ ボウ，ワスレル'],
        ['忝', 'テン，カタジケナイ，テンプスルノテンノツクリ'],
        ['応', 'オウエン ノ オウ'],
        ['心', 'ココロ，アンシン ノ シン'],
        ['必', 'ヒツヨウ ノ ヒツ，カナラズ'],
        ['忍', 'ニンジャ ノ ニン，シ ノ ブ'],
        ['忌', 'イッシュウキ ノ キ，イム'],
        ['隶', 'イ、ドレイカイホウノレイノツクリ'],
        ['隷', 'ドレイカイホウ ノ レイ'],
        ['隹', 'スイ、ブシュノフルトリ'],
        ['L', 'エル ラブ'],
        ['鑄', 'チュウゾウスルノチュウ、イルノキュウジ'],
        ['隋', 'ケンズイシノズイ'],
        ['階', 'カイダンヲ ノボル ノ カイ'],
        ['谺', 'カ、コダマ、タニヘンニキバ'],
        ['谿', 'ケイ，ケイリュウヅリノケイノキュウジノツクリノミギニタニ'],
        ['谷', 'タニガワ ノ タニ，コク'],
        ['隗', 'カイ、コザトヘンニオニ'],
        ['蝮', 'ヘビノマムシ、フク'],
        ['蝨', 'シラミ'],
        ['蝪', 'トウ，ムシヘンニバショノバノツクリ'],
        ['蝦', 'ムシヘン ノ エビ，カ'],
        ['蝠', 'コウモリノニモジメ，ムシヘンニフクカイチョウノフクノヒダリガワ'],
        ['蝣', 'ユウ、アソブノシンリョウノカワリニムシヘン'],
        ['蝿', 'ムシ ノ ハエ'],
        ['蝸', 'カタツムリノカギュウノカ'],
        ['蝴', 'コ、ムシヘンニクロコショウノコ'],
        ['蝶', 'アゲハチョウ ノ チョウ'],
        ['蝌', 'カ、ムシヘンニキョウカショノカ'],
        ['蝎', 'ダカツノゴトクキラウノカツ、ムシヘンニカッショクノカツノツクリ'],
        ['"', 'クォーテーション'],
        ['蝉', 'セミシグレ ノ セミ'],
        ['蝋', 'ロウソクタテ ノ ロウ'],
        ['蝟', 'イ、ムシヘンニイブクロノイ'],
        ['蝙', 'コウモリノイチモジメ、ムシヘンニヘンペイソクノヘン'],
        ['蝕', 'ニッショク ノ ショク，ムシバム'],
        ['蝗', 'コウ、イナゴ、ムシヘンニコウタイシノコウ'],
        ['蝓', 'ナメクジノニモジメ，ムシヘンニニレノキノニレノツクリ'],
        ['醺', 'ビクンヲオビルノクン'],
        ['傅', 'フイクスルノフ，メノト'],
        ['估', 'コ，ニンベンニチュウコシャノコ'],
        ['傀', 'カイライセイケンノカイ'],
        ['伶', 'レイジン ノ レイ，ニンベン ニ メイレイ ノ レイ'],
        ['伴', 'ドウハンスル ノ ハン，トモナウ'],
        ['傍', 'ボウカンシャ ノ ボウ，カタワラ'],
        ['伸', 'シンシュク ノ シン，ノビル'],
        ['伽', 'オトギバナシ ノ トギ，カ'],
        ['似', 'ルイジ ノ ジ，ニル'],
        ['傑', 'ゴウケツ ノ ケツ'],
        ['傘', 'ラッカサン ノ サン，カサ'],
        ['備', 'ジュンビスル ノ ビ，ソナエル'],
        ['傚', 'コウ，ニンベンニコウカテキノコウノキュウジ'],
        ['休', 'ヤスム，キュウジツ ノ キュウ'],
        ['伐', 'バッサイスル ノ バツ'],
        ['催', 'モヨオス，カイサイ ノ サイ'],
        ['傭', 'ヨウヘイ ノ ヨウ，ヤトウ'],
        ['伝', 'デンセツ ノ デン，ツタエル'],
        ['伜', 'セガレノイタイジ，ニンベンニカンスウジノキュウトジュウ'],
        ['傴', 'ウ，ニンベンニクベツスルノクノキュウジ'],
        ['債', 'サイムヲオウ ノ サイ'],
        ['企', 'キカク ノ キ，クワダテル'],
        ['傷', 'キズ，ジュウショウ ノ ショウ'],
        ['傲', 'ゴウマンナノゴウ'],
        ['傳', 'デンセツノデンノキュウジ'],
        ['伊', 'イセエビ ノ イ'],
        ['伉', 'コウ，ニンベンニテイコウスルノコウノツクリ'],
        ['伏', 'キフクニトム ノ フク，フセル'],
        ['伎', 'カブキ ノ キ'],
        ['伍', 'ラクゴスル ノ ゴ，ニンベン ニ カンスウジ ノ ゴ'],
        ['湧', 'ユウシュツスル ノ ユウ，ワク'],
        ['湫', 'シュウ，サンズイニアキマツリノアキ'],
        ['湮', 'ショウコインメツノイン，サンズイニケムリノツクリ'],
        ['湯', 'ネットウ ノ トウ，ユ'],
        ['湲', 'エン，サンズイニオウエンスルノエンノツクリ'],
        ['醗', 'サケヲハッコウサセル ノ ハツ，カモス'],
        ['湶', 'セン，サンズイニオンセンノセン'],
        ['醐', 'ダイゴミ ノ ゴ'],
        ['湾', 'ワンガン ノ ワン'],
        ['湿', 'シツド ノ シツ，シメル'],
        ['湃', 'ハイ，サンズイニオガムノキュウジ'],
        ['湊', 'ミナト，サンズイ ニ カナデル'],
        ['湎', 'チンメンスルノメン，サンズイニジメンノメン'],
        ['湍', 'タン，サンズイニタンマツノタンノツクリ'],
        ['湖', 'コハン ノ コ，ミズウミ'],
        ['湛', 'ミズヲタタエル ノ タン'],
        ['湘', 'ショウナンカイガン ノ ショウ'],
        ['湟', 'コウ，サンズイニコウタイシノコウ'],
        ['纎', 'カガクセンイノセンノキュウジノイタイジ'],
        ['纏', 'ハンテン ノ テン，マトウ'],
        ['續', 'レンゾクノゾク，ツヅクノキュウジ'],
        ['纈', 'コウケチゾメノケチ'],
        ['纉', 'サン，イトヘンニサンセイスルノサン'],
        ['纂', 'ジショヲヘンサンスル ノ サン'],
        ['纃', 'シ，イトヘンニイッセイシャゲキノセイノキュウジ'],
        ['纜', 'ラン，イトヘンニハクランカイノランノキュウジ'],
        ['纛', 'トク，ドクヤクノドクノシタニトドウフケンノケンノキュウジ'],
        ['纖', 'カガクセンイノセンノキュウジ'],
        ['纔', 'ワズカヲイミスルイトヘンノサイ'],
        ['纒', 'ハンテンノテン，マトウノイタイジ'],
        ['纓', 'エイ，イトヘンニサクラノキュウジノツクリ'],
        ['纐', 'コウケチゾメノコウ'],
        ['暹', 'セン，シンポノシンノウエニニチヨウビノニチ'],
        ['暸', 'リョウ，メイリョウナノリョウノメノカワリニヨウビノニチ'],
        ['暾', 'トン，ニチヘンニチュウゴクノトンコウノトン'],
        ['暼', 'ヘツ，イチベツスルノベツノメノカワリニヨウビノニチ'],
        ['鱗', 'ウロコ，ヘンリン ノ リン'],
        ['暴', 'ボウリョク ノ ボウ，アバレル'],
        ['暫', 'ザンテイテキ ノ ザン，シバラク'],
        ['暮', 'オセイボ ノ ボ，クラス'],
        ['暢', 'リュウチョウナ ノ チョウ'],
        ['鱇', 'サカナノアンコウノコウ，ウオヘンニイエヤスノヤス'],
        ['鱆', 'ショウ，ウオヘンニブンショウヲカクノショウ'],
        ['暦', 'セイレキ ノ レキ，コヨミ'],
        ['暘', 'ヨウ，ニチヘンニタイヨウケイノヨウノツクリ'],
        ['鱸', 'サカナノスズキ'],
        ['暝', 'メイ，ニチヘンニメイフクヲイノルノメイ'],
        ['暑', 'アツイサムイ ノ アツイ，ショ'],
        ['鱶', 'サカナノフカ，ウオヘンニヤシナウ'],
        ['暗', 'アンキスル ノ アン，クライ'],
        ['暖', 'ダンボウ ノ ダン，アタタカイ'],
        ['暉', 'キ，ニチヘンニグンタイノグン'],
        ['暈', 'メマイヲイミスルゲンウンノウン'],
        ['暎', 'エイ，ニチヘンニエイユウノエイ'],
        ['暃', 'ニチヨウビノニチノシタニヒジョウグチノヒ'],
        ['暁', 'アカツキ'],
        ['暇', 'キュウカ ノ カ，ヒマ'],
        ['鱠', 'ウオヘンノナマス'],
        ['暄', 'ケン，ニチヘンニセンデンノセン'],
        ['c', 'シー キャット'],
        ['鉅', 'キョ、カネヘンニキョダイナノキョ'],
        ['壽', 'コトブキ，ジュミョウノジュノキュウジ'],
        ['彷', 'ホウコウスルノホウ，ギョウニンベンニオヤカタノカタ'],
        ['影', 'エイキョウ ノ エイ，カゲ'],
        ['彰', 'ヒョウショウジョウ ノ ショウ'],
        ['彳', 'テキ，ブシュノギョウニンベン'],
        ['彼', 'ヒガンバナ ノ ヒ，カレ'],
        ['彿', 'フツ，ギョウニンベンニドル'],
        ['役', 'ヤクワリ ノ ヤク'],
        ['彦', 'ヒコボシ ノ ヒコ'],
        ['彡', 'サン，ブシュノサンヅクリ'],
        ['形', 'チョウホウケイ ノ ケイ，カタチ'],
        ['彭', 'ホウ，ボウチョウリツノボウノツクリ'],
        ['彬', 'ヒン，ハヤシニ サンヅクリ，アキラ'],
        ['彩', 'シキサイ ノ サイ，イロドル'],
        ['彫', 'チョウコクカ ノ チョウ，ホル'],
        ['彪', 'トラニサンヅクリ ノ ヒョウ'],
        ['彗', 'ハレースイセイノスイ'],
        ['彖', 'タン，ケイガシラニイノコ'],
        ['彑', 'ケイ，ブシュノケイガシラ'],
        ['当', 'トウバン ノ トウ，アタル'],
        ['彝', 'イ，ゴイノイノウエガワニコメ，イト，ニジュウアシ'],
        ['彜', 'イ，ゴイノイノウエガワニコナ，ニジュウアシ'],
        ['彙', 'キホンゴイノイ'],
        ['彁', 'ユミヘンニカノウセイノカヲカサネル'],
        ['彌', 'アミダブツノミノキュウジ'],
        ['彎', 'ワンガンノワンノキュウジノツクリ'],
        ['彈', 'ダンリョクノダン，ハズムノキュウジ'],
        ['彊', 'ジキョウジュツ ノ キョウ，ツヨイ'],
        ['ｓ', 'エス スポーツ'],
        ['ｒ', 'アール ルーム'],
        ['ｑ', 'キュー クエスチョン'],
        ['ｐ', 'ピー パパ'],
        ['ｗ', 'ダブリュー ウィンドウ'],
        ['ｖ', 'ブイ ビデオ'],
        ['ｕ', 'ユー ユージュアリー'],
        ['ｔ', 'ティー タイム'],
        ['｛', 'チュウカッコ'],
        ['ｚ', 'ゼット ズー'],
        ['ｙ', 'ワイ ヤング'],
        ['ｘ', 'エックス エックスセン'],
        ['～', 'ナミセン'],
        ['｝', 'トジチュウカッコ'],
        ['｜', 'タテセン'],
        ['ｃ', 'シー キャット'],
        ['ｂ', 'ビー ボーイ'],
        ['ａ', 'エイ アニマル'],
        ['｀', 'アクサングラーブ'],
        ['ｇ', 'ジー ゴルフ'],
        ['ｆ', 'エフ フレンド'],
        ['ｅ', 'イー エッグ'],
        ['ｄ', 'ディー デスク'],
        ['ｋ', 'ケイ キッチン'],
        ['ｊ', 'ジェイ ジャパン'],
        ['ｉ', 'アイ インク'],
        ['ｈ', 'エイチ ホテル'],
        ['ｏ', 'オー オープン'],
        ['ｎ', 'エヌ ノベンバー'],
        ['ｍ', 'エム マイク'],
        ['ｌ', 'エル ラブ'],
        ['ｳ', 'ウサギ ノ ウ'],
        ['ｲ', 'イチゴ ノ イ'],
        ['ｱ', 'アサヒ ノ ア'],
        ['ｰ', 'チョウオン'],
        ['ｷ', 'キッテ ノ キ'],
        ['ｶ', 'カゾク ノ カ'],
        ['ｵ', 'オオサカ ノ オ'],
        ['ｴ', 'エイゴ ノ エ'],
        ['ｻ', 'サクラ ノ サ'],
        ['ｺ', 'コドモ ノ コ'],
        ['ｹ', 'ケシキ ノ ケ'],
        ['ｸ', 'クスリ ノ ク'],
        ['ｿ', 'ソロバン ノ ソ'],
        ['ｾ', 'セカイ ノ セ'],
        ['ｽ', 'スズメ ノ ス'],
        ['ｼ', 'シンブン ノ シ'],
        ['｣', 'トジカギ'],
        ['｢', 'カギ'],
        ['｡', 'マル'],
        ['ｧ', 'チイサイ アサヒ ノ ア'],
        ['ｦ', 'ヲワリ ノ ヲ'],
        ['･', 'ナカテン'],
        ['､', 'テン'],
        ['9', 'キュウ'],
        ['ｪ', 'チイサイ エイゴ ノ エ'],
        ['ｩ', 'チイサイ ウサギ ノ ウ'],
        ['ｨ', 'チイサイ イチゴ ノ イ'],
        ['ｯ', 'チイサイ ツバメ ノ ツ'],
        ['ｮ', 'チイサイ ヨット ノ ヨ'],
        ['ｭ', 'チイサイ ユカタ ノ ユ'],
        ['ｬ', 'チイサイ ヤカン ノ ヤ'],
        ['貝', 'カイガラ ノ カイ'],
        ['貞', 'テイシュク ノ テイ'],
        ['貘', 'ムジナヘンノドウブツノバク'],
        ['貔', 'ユウモウナグンタイヲイミスルヒキュウノヒ'],
        ['貍', 'リ、ムジナヘンニサトガエリノサト'],
        ['貌', 'ビボウ ノ ボウ，カタチ'],
        ['貎', 'ゲイ、ムジナヘンニジドウセイトノジノキュウジ'],
        ['貉', 'ムジナヘンノドウブツノムジナ'],
        ['貊', 'ハク、ムジナヘンニカンスウジノヒャク'],
        ['倍', 'バイリツ ノ バイ'],
        ['貂', 'ドウブツノテン'],
        ['貽', 'イズシノイ、カイヘンニダイドコロノダイ'],
        ['貼', 'ハル，カイヘン ニ ウラナウ'],
        ['貿', 'ボウエキコウ ノ ボウ'],
        ['貸', 'ホンヲカス ノ カ'],
        ['費', 'ショウヒ ノ ヒ'],
        ['貴', 'キチョウヒン ノ キ'],
        ['買', 'カウ，バイシュウ ノ バイ'],
        ['貶', 'キヨホウヘンノヘン，オトシメル'],
        ['貰', 'モラウ'],
        ['貳', 'カンスウジノニ，シキガマエニ，ニノシタニカイガラノカイ'],
        ['貲', 'シ、ヒガンシガンノシノシタニカイガラノカイ'],
        ['貭', 'シツモンノシツノイタイジ'],
        ['責', 'セキニン ノ セキ'],
        ['貯', 'チョキン ノ チョ，タクワエル'],
        [
            '貮',
            'カンスウジノニノイタイジ，ブキノブノトメルノカワリニカイガラノカイ',
        ],
        ['販', 'ハンバイスル ノ ハン'],
        ['貨', 'カモツ ノ カ'],
        ['貫', 'イッカンスル ノ カン，ツラヌク'],
        ['貪', 'ドンヨクノドン、ムサボル'],
        ['貧', 'ヒンジャク ノ ヒン，マズシイ'],
        ['財', 'ザイサン ノ ザイ'],
        ['負', 'カチマケ ノ マケ，フ'],
        ['貢', 'シャカイコウケンスル ノ コウ，ミツグ'],
        ['龜', 'ドウブツノカメノキュウジ'],
        ['△', 'サンカク'],
        ['▲', 'クロサンカク'],
        ['▽', 'ギャクサンカク'],
        ['▼', 'クロギャクサンカク'],
        ['□', 'シカク'],
        ['■', 'クロシカク'],
        ['癇', 'カンシャクヲオコスノカン'],
        ['癆', 'ロウガイノロウ，ヤマイダレニロウドウノロウノキュウジ'],
        ['露', 'ロシュツ ノ ロ，ツユ'],
        ['療', 'チリョウスル ノ リョウ'],
        ['霰', 'アラレ、アメカンムリニサンンポスルノサン'],
        ['霾', 'バイ、アメカンムリニムジナヘンニサトガエリノサト'],
        ['癌', 'ガンサイボウ ノ ガン'],
        ['霸', 'セイハスルノハノキュウジ'],
        ['癈', 'ハイジンノハイ，ヤマイダレニハツメイノハツノキュウジ'],
        ['癖', 'アクヘキ ノ ヘキ，クセ'],
        ['霤', 'リュウ、アメカンムリニリュガクノリュウ'],
        ['癒', 'チユスル ノ ユ，イヤス'],
        ['癜', 'ヒフビョウノナマズ，テン，ヤマイダレニトノサマノトノ'],
        ['霪', 'イン、アメカンムリニインランノイン'],
        ['癘', 'ショウレイノレイ，ヤマイダレニスウジノマンノキュウジ'],
        ['O', 'オー オープン'],
        ['癢', 'ヨウ，ヤマイダレニヤシナウ'],
        ['癡', 'グチノチノキュウジ'],
        ['霑', 'テン、アメカンムリニサンズイニドクセンスルノセン'],
        ['霞', 'ハルガスミ ノ カスミ'],
        ['霜', 'シモバシラ ノ シモ'],
        ['癬', 'カイセンキンノセン'],
        ['癪', 'カンシャクヲオコスノシャク'],
        ['癩', 'ライビョウノライ'],
        ['癨', 'カク，ヤマイダレニオニノカクランノカク'],
        ['霆', 'テイ、アメカンムリニキュウテイブンガクノテイ'],
        ['癶', 'ハツ，ブシュノハツガシラ'],
        ['霄', 'ショウ、アメカンムリニショウゾウガノショウ'],
        ['癲', 'テンカンヲオコスノテン'],
        ['需', 'ヒツジュヒン ノ ジュ'],
        ['癰', 'ヒフビョウノヨウ'],
        ['霎', 'ソウ、アメカンムリニメカケ'],
        ['百', 'カンスウジ ノ ヒャク'],
        ['白', 'シロイ，ハクチョウ ノ ハク'],
        ['發', 'ハツメイノハツノキュウジ'],
        ['登', 'トザン ノ ト，ノボル'],
        ['発', 'ハツメイ ノ ハツ'],
        ['霈', 'ハイ、アメカンムリニハイゼンタルノハイ'],
        ['癸', 'キ，ジッカンノミズノト'],
        ['侵', 'シンリャク ノ シン，オカス'],
        ['倏', 'シュク，ジョウケンノジョウノキュウジノキノカワリニドウブツノイヌ'],
        ['侶', 'ソウリョ ノ リョ'],
        ['個', 'コセイテキ ノ コ'],
        ['倉', 'ソウコ ノ ソウ，クラ'],
        ['倆', 'リョウ，ニンベンニリョウテノリョウノキュウジ'],
        ['便', 'ユウビン ノ ビン，タヨリ'],
        ['倅', 'セガレ，ニンベンニソツギョウノソツ'],
        ['借', 'シャクヨウ ノ シャクカリル'],
        ['価', 'カチ ノ カ，アタイ'],
        ['侠', 'ニンキョウ ノ キョウ'],
        ['候', 'リッコウホ ノ コウ'],
        ['侭', 'ワガママ ノ ママ，ニンベン ニ ツクス'],
        ['侯', 'オウコウキゾク ノ コウ'],
        ['侮', 'ブジョクスル ノ ブ，アナドル'],
        ['倒', 'トウサンスル ノ トウ，タオレル'],
        ['侫', 'カンネイノネイノイタイジ'],
        ['們', 'モン，ニンベンニセンモンノモン'],
        ['倬', 'タク，ニンベンニタッキュウノタク'],
        ['侖', 'リン，シャリンノリンノツクリ'],
        ['侑', 'ユウ，ニンベンニユウメイジンノユウ'],
        ['倫', 'リンリテキナ ノ リン'],
        ['倨', 'キョゴウノキョ，ニンベンニジュウキョノキョ'],
        ['倩', 'セン，ニンベンニアオノキュウジタイ'],
        ['依', 'イライスル ノ イ'],
        ['値', 'ネダン ノ ネ，アタイ'],
        ['倥', 'コウ，ニンベンニクウキノクウ'],
        ['倣', 'モホウスル ノ ホウ，ナラウ'],
        ['供', 'テイキョウスル ノ キョウ，ソナエル'],
        ['倡', 'ショウ，ニンベンニガッショウキョクノショウノツクリ'],
        ['來', 'ライネンノライ，クルノキュウジ'],
        ['侃', 'カンカンガクガク ノ カン'],
        ['倹', 'ケンヤクスル ノ ケン'],
        ['侍', 'ジジュウ ノ ジ，サムライ'],
        ['侏', 'シュジュノシュ，ニンベンニシュイロノシュ'],
        ['侈', 'シャシニナガレルノシ，ニンベンニオオイスクナイノオオイ'],
        ['例', 'レイダイ ノ レイ，タトエル'],
        ['濤', 'ドトウノトウ，ナミ'],
        ['濠', 'ゴウ，サンズイ ニ ゴウカイナ ノ ゴウ'],
        ['濡', 'ヌレル'],
        ['濬', 'シュン，サンズイニヒエイザンノエイノヒダリガワ'],
        ['濮', 'ボク，サンズイニイチニンショウノボク'],
        ['濯', 'センタクモノ ノ タク'],
        ['濫', 'カセン ノ ハンラン ノ ラン'],
        ['濶', 'ウカツナノカツノイタイジ'],
        ['濱', 'スナハマノハマノキュウジ'],
        ['濳', 'センスイカンノセン，モグルノイタイジ'],
        ['濾', 'ロカスルノロ，コス'],
        ['濺', 'セン，サンズイニゲセンナノセン'],
        ['濆', 'フン，サンズイニフンカスルノフンノツクリ'],
        ['激', 'カゲキ ノ ゲキ，ハゲシイ'],
        ['濁', 'ニゴル，ダクリュウ ノ ダク'],
        ['濂', 'レン，サンズイニレンカバンノレン'],
        ['濃', 'ノウタン ノ ノウ，コイ'],
        ['濔', 'ビ，サンズイニソツジナガラノジ'],
        ['濕', 'シツドノシツ，シメルノキュウジ'],
        ['濟', 'キュウサイスルノサイ，スマスノキュウジ'],
        ['濘', 'デイネイノネイ，サンズイニテイネイノネイ'],
        ['濛', 'モウ，サンズイニケイモウスルノモウ'],
        ['郁', 'フクイク ノ イク'],
        ['飆', 'ツムジカゼ，ヒョウ，カゼノヒダリニイヌミッツ'],
        ['P', 'ピー パパ'],
        ['蟶', 'マテガイノマテ，ムシヘンニヒジリ'],
        [
            '蟷',
            'カマキリヲイミスルトウロウノトウ，ムシヘンニベントウノトウノキュウジ',
        ],
        ['蟲', 'ムシ，コンチュウノチュウノキュウジ'],
        ['蟾', 'セン，ムシヘンニタンニンノタンノキュウジノツクリ'],
        ['蟻', 'ムシ ノ アリ，ギ'],
        ['蟹', 'カニ'],
        ['蟠', 'ワダカマル，ハン，ムシヘンニバンゴウノバン'],
        ['蟯', 'ギョウチュウノギョウ'],
        ['蟒', 'ウワバミヲイミスルボウ，クサカンムリニソウモウノシンノモウ'],
        ['蟐', 'アカガエル，ムシヘンニヒジョウグチノジョウ'],
        ['蟆', 'ガマガエル，バクノイタイジ'],
        ['蟇', 'ガマガエル，バク'],
        ['蟄', 'ケイチツノチツ'],
        ['蟀', 'コオロギノニモジメ，ムシヘンニソツギョウノソツ'],
        ['蟋', 'コオロギノイチモジメ，ムシヘンニノゴメノシタニココロ'],
        ['飫', 'ヨ，アキル，ショクヘンニヒヨクナトチノヨクノツクリ'],
        ['飩', 'ドン，ウドンノニモジメ'],
        ['飢', 'キガ ノ キ，ウエル'],
        ['&', 'アンド'],
        ['￥', 'エンマーク'],
        ['￡', 'ポンド'],
        ['￠', 'セント'],
        ['￣', 'オーバーライン'],
        ['￢', 'ヒテイ'],
        ['乖', 'カイリスルノカイ，ソムクノイミ'],
        ['示', 'テンジスル ノ ジ，シメス'],
        ['礼', 'レイギ ノ レイ'],
        ['社', 'シャカイ ノ シャ，ヤシロ'],
        ['礦', 'コウ，イシヘンニヒロバノヒロノキュウジ'],
        ['礪', 'トナミヘイヤノト'],
        ['礫', 'ガレキノレキ'],
        ['礬', 'コウブツノミョウバンノバン'],
        ['礑', 'トウ，イシヘンニベントウノトウノキュウジ'],
        ['礒', 'ギ，イシヘンニギムノギ'],
        ['乍', 'ムカシナガラ ノ ナガラ'],
        ['礙', 'ガイ，イシヘンニウタガウ'],
        ['礁', 'サンゴショウ ノ ショウ'],
        ['乏', 'ビンボウ ノ ボウ，トボシイ'],
        ['礇', 'イキ，イシヘンニオクバノオクノキュウジ'],
        ['礎', 'キソテキナ ノ ソ，イシズエ'],
        ['昼', 'ヒルヤスミ ノ ヒル，チュウ'],
        ['昿', 'コウ，ニチヘンニヒロイ'],
        ['昵', 'ジッコンノアイダガラノジツ，ニチヘンニアマデラノアマ'],
        ['昴', 'スバル'],
        ['鷭', 'バン，バンゴウノバンノミギニトリ'],
        ['昶', 'チョウ，エイキュウノエイノミギニニチヨウビノニチ'],
        ['昭', 'ショウワウマレ ノ ショウ'],
        ['是', 'ゼヒ ノ ゼ'],
        ['昨', 'サクネン ノ サク'],
        ['春', 'ハルヤスミ ノ ハル，シュン'],
        ['昧', 'アイマイナ ノ マイ'],
        ['映', 'エイガ ノ エイ，ウツス'],
        ['鷹', 'トリ ノ タカ'],
        ['鷸', 'イツ，タチバナノツクリノミギニトリ，シギ'],
        ['昜', 'ヨウ，タイヨウケイノヨウノツクリ'],
        ['星', 'ホシ，セイザ ノ セイ'],
        ['鷄', 'ニワトリノキュウジ'],
        ['鷂', 'テン，マゴコロノマノミギニトリ'],
        ['鷁', 'ゲキ，リエキノエキノミギニトリ，アオサギ'],
        ['鷏', 'テン，マゴコロノマノミギニトリ'],
        ['昔', 'ムカシバナシ ノ ムカシ'],
        ['易', 'ボウエキ ノ エキ，ヤサシイ'],
        ['昌', 'ショウ，ニチ ノ シタニ ニチ'],
        ['昏', 'コンスイスル ノ コン'],
        ['明', 'セツメイ ノ メイ，アカルイ'],
        ['鷓', 'トリノシャコノシャ，ショミンノショノミギニトリ'],
        ['倪', 'タンゲイノゲイ，ニンベンニヌラムノツクリ'],
        ['昇', 'ショウカクスル ノ ショウ，ノボル'],
        ['昆', 'コンチュウ ノ コン'],
        ['昃', 'ソク，ニチヨウビノニチノシタニソクブンスルノソク'],
        ['昂', 'ゲキコウスル ノ コウ'],
        ['└', 'ヒダリシタ'],
        ['┗', 'フトヒダリシタ'],
        ['┐', 'ミギウエ'],
        ['┓', 'フトミギウエ'],
        ['鐃', 'ドウ、カネヘンニギョウシュンノギョウノキュウジ'],
        ['┝', 'タテセンフトミギ'],
        ['├', 'タテセンミギ'],
        ['┘', 'ミギシタ'],
        ['┛', 'フトミギシタ'],
        ['━', 'フトヨコセン'],
        ['─', 'ヨコセン'],
        ['┃', 'フトタテセン'],
        ['│', 'タテセン'],
        ['┌', 'ヒダリウエ'],
        ['┏', 'フトヒダリウエ'],
        ['┴', 'ヨコセンウエ'],
        ['┷', 'フトヨコセンホソウエ'],
        ['┰', 'ヨコセンフトシタ'],
        ['┳', 'フトヨコセンシタ'],
        ['┼', 'クロス'],
        ['┿', 'フトヨコセンクロス'],
        ['┸', 'ヨコセンフトウエ'],
        ['┻', 'フトヨコセンウエ'],
        ['┥', 'タテセンフトヒダリ'],
        ['┤', 'タテセンヒダリ'],
        ['┠', 'フトタテセンホソミギ'],
        ['┣', 'フトタテセンミギ'],
        ['┬', 'ヨコセンシタ'],
        ['┯', 'フトヨコセンホソシタ'],
        ['┨', 'フトタテセンホソヒダリ'],
        ['┫', 'フトタテセンヒダリ'],
        ['盈', 'ミチル，エイ'],
        ['益', 'リエキ ノ エキ'],
        ['盍', 'コウ，フタカラクサカンムリヲトッタカタチ'],
        ['盃', 'サカズキ，フシギ ノ フ ノ シタニ サラ'],
        ['盂', 'ウラボンノウ'],
        ['盆', 'ボンオドリ ノ ボン'],
        ['盛', 'オオモリ ノ モリ，セイ'],
        ['盜', 'トウナンノトウ，ヌスムノキュウジ'],
        ['盟', 'ドウメイスル ノ メイ'],
        ['盞', 'サン，サラノウエニアサイノキュウジノツクリ'],
        ['盒', 'ハンゴウスイサンノゴウ'],
        ['盗', 'トウナン ノ トウ，ヌスム'],
        ['盖', 'フタノイタイジ，サラノウエニヒツジ'],
        ['盪', 'ノウシントウノトウ'],
        ['目', 'モクテキ ノ モク，メ'],
        ['盡', 'ジンリョクスルノジン，ツクスノキュウジ'],
        ['監', 'カントクスル ノ カン'],
        ['盥', 'タライ，カン'],
        ['盤', 'エンバンナゲ ノ バン'],
        ['盧', 'ロ，ロカスルノロノツクリ'],
        ['相', 'ソウダンスル ノ ソウ，アイ'],
        ['盻', 'ケイ，メヘンニカンスウジノハチノシタニバンゴウノゴウノシタガワ'],
        ['盾', 'ムジュン ノ ジュン，タテ'],
        ['钁', 'カク，カネヘンニゲンキカクシャクノカク'],
        ['盲', 'モウガッコウ ノ モウ'],
        ['直', 'チョクセン ノ チョク，ナオル'],
        ['鄒', 'スウ，スウセイノスウノツクリノミギニオオザト'],
        ['鐘', 'ケイショウヲナラス ノ ショウ，カネ'],
        ['廿', 'カンスウジ ノ ニジュウ'],
        ['廾', 'キョウ，ブシュノニジュウアシ'],
        ['廼', 'エンニョウ ニ ニシ，ナイ'],
        ['廻', 'エンニョウ ニ マワル ノ カイ'],
        ['建', 'ケンチク ノ ケン，タテル'],
        ['廸', 'テキ，エンニョウニリユウノユウ'],
        ['廷', 'キュウテイブンガク ノ テイ'],
        ['延', 'エンキスル ノ エン，ノビル'],
        ['廴', 'イン，ブシュノエンニョウ'],
        ['廳', 'チョウ，キショウチョウノチョウノキュウジ'],
        ['廱', 'ヨウ，マダレニカワ，クチ，トモエ，フルトリ'],
        ['廰', 'チョウ，キショウチョウノチョウノキュウジノイタイジ'],
        ['廬', 'ロザンノロ，マダレニハゼノキノハゼノツクリ'],
        ['廩', 'リン，マダレニリンギスルノリン'],
        ['廨', 'カイ，マダレニカイケツスルノカイ'],
        ['廣', 'ヒロバノヒロノキュウジ'],
        ['廢', 'ハイシスルノハイ，スタレルノキュウジ'],
        ['廡', 'ブ，マダレニムリノム'],
        ['廠', 'カイグンコウショウ ノ ショウ'],
        ['廟', 'レイビョウ ノ ビョウ'],
        ['廝', 'シ，マダレニシカイノケンイノシ'],
        ['廛', 'テン，マトイノツクリ'],
        ['廚', 'チュウボウノチュウ，クリヤノキュウジ'],
        ['g', 'ジー ゴルフ'],
        ['廖', 'リョウ，マダレニゴビュウノビュウノツクリ'],
        ['廓', 'ユウカク ノ カク，クルワ'],
        ['廐', 'キュウシャノキュウ，ウマヤノイタイジ，キバ'],
        ['廏', 'キュウシャノキュウ，ウマヤノイタイジ，ルマタ'],
        ['廊', 'ワタリロウカ ノ ロウ'],
        ['廉', 'レンカバン ノ レン'],
        ['廈', 'カ，マダレニナツ'],
        ['廃', 'ハイシスル ノ ハイ，スタレル'],
        ['廂', 'ショウ，マダレニソウダンスルノソウ'],
        ['廁', 'カワヤ，マダレニホウソクノソク'],
        ['贇', 'ゲンピンヤキノイン'],
        ['贅', 'ゼイタクノゼイ'],
        ['贄', 'イケニエノニエ'],
        ['贏', 'カチマケヲイミスルエイシュノエイ'],
        ['贍', 'セン，カイヘンニクシンサンタンノタンノツクリ'],
        ['贋', 'ガンサク ノ ガン，ニセ'],
        ['贊', 'サンセイハンタイノサンノキュウジ'],
        ['贈', 'ゾウトウヒン ノ ゾウ，オクル'],
        ['贖', 'ショクザイノショク，アガナウ'],
        ['贔', 'ヒイキノヒ、カイミッツ'],
        ['贓', 'ゾウブツザイノゾウ'],
        ['贐', 'ハナムケ、ジン'],
        ['=', 'イコール'],
        ['葹', 'シ，クサカンムリニホドコス'],
        ['葺', 'ヤネヲフク ノ フク'],
        ['葱', 'ショクブツ ノ ネギ'],
        ['葵', 'アオイマツリ ノ アオイ'],
        ['葷', 'クンシュサンモンニイルヲユルサズノクン'],
        ['葩', 'ハ，クサカンムリニシロイノミギニトモエ'],
        ['葫', 'コ、クサカンムリニミズウミノツクリ'],
        ['葬', 'ソウシキ ノ ソウ，ホウムル'],
        ['葭', 'カ、アシ、クサカンムリニヒマツブシノヒマノツクリ'],
        ['葮', 'タン、クサカンムリニシュダンノダン'],
        ['葯', 'ヤク，クサカンムリニヤクソクノヤク'],
        ['葡', 'ブドウシュ ノ ブ'],
        ['葢', 'フタノイタイジ，クサカンムリニフトイニサラ'],
        ['董', 'コットウヒン ノ トウ'],
        ['葦', 'アシブエ ノ アシ'],
        ['葛', 'カットウ ノ カツ，クズ'],
        ['著', 'チョシャ ノ チョ，イチジルシイ'],
        ['葉', 'キノハ ノ ハ，ヨウ'],
        ['葎', 'ショクブツ ノ ムグラ，クサカンムリ ニ ホウリツ ノ リツ'],
        ['嬾', 'ラン，オンナヘンニタヨルノキュウジ'],
        ['葆', 'ホウ、クサカンムリニホアンノホ'],
        ['丿', 'ヘツ，カタカナノノノカタチ'],
        ['垓', 'カズノタンイノガイ'],
        ['主', 'シュジンコウ ノ シュ，ヌシ'],
        ['丹', 'タンネンナ ノ タン'],
        ['丸', 'ヒノマル ノ マル，ガン'],
        ['丶', 'チュウ，ブンノクギリフゴウ，カタチハテン'],
        ['串', 'クシカツノ クシ'],
        ['丱', 'カン，アゲマキ'],
        ['垂', 'スイチョク ノ スイ，タレル'],
        ['个', 'カ，モノノタンイ，ヤネノシタニタテボウ'],
        ['垈', 'タイ，ダイヒョウシャノダイノシタニツチ'],
        ['垉', 'ホウ，ツチヘンニツツム'],
        ['型', 'モケイ ノ ケイ，カタ'],
        ['両', 'リョウテ ノ リョウ'],
        ['垰', 'タオ，ツチヘンニウエトシタ'],
        ['丞', 'ダイジンヲ イミスル ジョウショウ ノ ジョウ'],
        ['垳', 'ガケ，ツチヘンニリョコウスルノコウ'],
        ['丙', 'コウオツヘイ ノ ヘイ'],
        ['丘', 'サキュウ ノ キュウ，オカ'],
        ['丗', 'セカイノセノキュウジ'],
        ['世', 'セカイ ノ セ，ヨ'],
        ['丕', 'ヒ，フシギノフノシタニカンスウジノイチ'],
        ['且', 'ナオカツ ノ カツ'],
        ['丑', 'ウシ，ジュウニシ ノ ウシ'],
        ['丐', 'カイ，コウ，アタエルノイミ'],
        ['垠', 'ギン，ツチヘンニダイコンノコンノツクリ'],
        ['与', 'アタエル，キュウヨ ノ ヨ'],
        ['垢', 'テアカ ノ アカ，コウ'],
        ['垣', 'カキネ ノ カキ'],
        ['垤', 'テツ，アリヅカ，ツチヘンニゲシノシ'],
        ['上', 'ジョウゲ ノ ジョウ，ウエ'],
        ['三', 'カンスウジ ノ サン'],
        ['丈', 'ダイジョウブ ノ ジョウ，タケ'],
        ['万', 'マンネンヒツ ノ マン'],
        ['垪', 'ハ，ツチヘンニヘイゴウノヘイノツクリ'],
        ['七', 'カンスウジ ノ ナナ'],
        ['丁', 'イッチョウメ ノ チョウ'],
        ['一', 'カンスウジ ノ イチ'],
        ['潮', 'マンチョウ ノ チョウ，シオ'],
        ['潯', 'ジン，サンズイニジンジョウショウガッコウノジン'],
        ['潭', 'タン，サンズイニニシノシタニハヤオキノハヤイ'],
        ['潦', 'ロウ，サンズイニメイリョウナノリョウノツクリ'],
        ['潤', 'リジュン ノ ジュン，ウルオウ'],
        ['潼', 'トウ，サンズイニグリムドウワノドウ'],
        ['潺', 'サン，サンズイニシカバネノシタニコドモノコミッツ'],
        ['潸', 'サン，サンズイニハヤシノシタニツキ'],
        ['潴', 'チョ，サンズイニイノシシ'],
        ['潰', 'ツブレル，イカイヨウ ノ カイ'],
        ['潅', 'カンボク ノ カン'],
        ['潁', 'エイ，カタカナノヒノシタニミズ，ソノミギニページ'],
        ['潟', 'ニイガタケン ノ カタ'],
        ['潜', 'センスイカン ノ セン，モグル'],
        ['潛', 'センスイカンノセン，モグルノキュウジ'],
        ['潘', 'ハン，サンズイニバンゴウノバン'],
        ['潔', 'セイケツ ノ ケツ'],
        ['禺', 'グウ，グウゼンノグウノツクリ'],
        ['禹', 'チュウゴクノデンセツノオウノウ'],
        ['禾', 'ノギヘン ノ ノギ，カ'],
        ['禿', 'ハゲヤマ ノ ハゲ'],
        ['禽', 'モウキンルイ ノ キン'],
        ['禳', 'ジョウ，シメスヘンニユズルノキュウジノツクリ'],
        ['禰', 'スクネ ノ ネ，シメスヘン ニ ナンジ'],
        ['禪', 'ザゼンノゼンノキュウジ'],
        ['禮', 'レイギノレイノキュウジ'],
        ['禦', 'ボウギョ ノ ギョ ノ キュウジ'],
        ['禧', 'キ，シメスヘンニヨロコブ'],
        ['禝', 'シャショクノシンノショク，シメスヘン'],
        ['禊', 'ミソギ，ケイ'],
        ['禎', 'テイ，シメスヘン ニ テイシュク ノ テイ'],
        ['福', 'フクビキ ノ フク'],
        ['禍', 'ワザワイ，カ'],
        ['禀', 'リンギショノリンノイタイジ'],
        ['禁', 'キンシスル ノ キン'],
        ['禄', 'ゲンロクブンカ ノ ロク'],
        ['禅', 'ザゼン ノ ゼン'],
        ['鵬', 'オオトリ，ホウユウ ノ ホウニトリ'],
        ['鵯', 'ヒヨドリ，ヒクツナノヒノミギニトリ'],
        ['鵡', 'トリ ノ オウム ノ ム'],
        ['鵠', 'コウコク ノ ココロザシ ノ コク'],
        ['鵤', 'ツノヘンニトリ，イカルガ'],
        ['鵺', 'ヨゾラノヨノミギニトリ，ヌエ'],
        ['鵲', 'カササギ，ムカシノミギニトリ'],
        ['鵈', 'ミミヘンニトリ，トビ'],
        ['鵁', 'コウ，コウツウノコウノミギニトリ'],
        ['鵄', 'シ，ゲシノシノミギニトリ'],
        ['鵆', 'ギョウガマエノナカニトリ，チドリ'],
        ['鵙', 'ケキ，カイヘンニトリ'],
        ['鵝', 'ガ，ガマンノガノミギニトリ，ガチョウノガ'],
        ['鵜', 'ウノミニスル ノ ウ'],
        ['鵞', 'ガ，ガマンノガノシタニトリ，ガチョウノガ'],
        ['鵑', 'ケン，キヌイトノキヌノツクリノミギニトリ'],
        ['鵐', 'ブ，ジンジャノミコノフノミギニトリ'],
        ['~', 'チルダ'],
        ['古', 'フルイ，コダイ ノ コ'],
        ['T', 'ティー タイム'],
        ['枇', 'ショクブツ ノ ビワ ノ ビ，キヘン ニ クラベル'],
        ['枅', 'ケイ，キヘンニセイホウケイノケイノヒダリガワ'],
        ['枌', 'フン、キヘンニハンブンノブン'],
        ['枋', 'ホウ、キヘンニセイホウケイノホウ'],
        ['枉', 'オウ、マガル、キヘンニオウサマノオウ'],
        ['林', 'シンリン ノ リン，ハヤシ'],
        ['枕', 'マクラ'],
        ['析', 'ブンセキスル ノ セキ'],
        ['枝', 'キノエダ ノ エダ'],
        ['果', 'ケッカ ノ カ，ハタス'],
        ['枚', 'マイスウ ノ マイ'],
        ['枦', 'ハゼノキノハゼノイタイジ，キヘンニトジマリノト'],
        ['枢', 'チュウスウ ノ スウ'],
        ['枡', 'マス、キヘンニイッショウビンノショウ'],
        ['枠', 'ソトワク ノ ワク'],
        ['枯', 'カレル，コカツスル ノ コ'],
        ['枩', 'マツノキノマツノイタイジ'],
        ['枷', 'カ，キヘンニカソクドノカ'],
        ['架', 'ジュウジカ ノ カ，カケル'],
        ['枴', 'カイ，キヘンニユウカイハンノカイノツクリ'],
        ['枳', 'カラタチ，キヘンニタダイマノタダ'],
        ['枹', 'フ，キヘンニヒョウガフルノヒョウノシタガワ'],
        ['枸', 'ク，キヘンニハイクノク'],
        ['в', 'キリル ヴェー'],
        ['г', 'キリル ゲー'],
        ['а', 'キリル アー'],
        ['б', 'キリル ベー'],
        ['ж', 'キリル ジェー'],
        ['з', 'キリル ゼー'],
        ['д', 'キリル デー'],
        ['е', 'キリル ィエー'],
        ['к', 'キリル カー'],
        ['л', 'キリル エル'],
        ['и', 'キリル イー'],
        ['й', 'キリル イークラトコエ'],
        ['о', 'キリル オー'],
        ['п', 'キリル ペー'],
        ['м', 'キリル エム'],
        ['н', 'キリル エヌ'],
        ['Т', 'キリル テー'],
        ['У', 'キリル ウー'],
        ['Р', 'キリル エル'],
        ['С', 'キリル エス'],
        ['Ц', 'キリル ツェー'],
        ['Ч', 'キリル チェー'],
        ['Ф', 'キリル エフ'],
        ['Х', 'キリル ハー'],
        ['Ъ', 'キリル コウオンキゴウ'],
        ['Ы', 'キリル ウィ'],
        ['Ш', 'キリル シャー'],
        ['Щ', 'キリル シシャー'],
        ['Ю', 'キリル ユー'],
        ['Я', 'キリル ヤー'],
        ['Ь', 'キリル ナンオンキゴウ'],
        ['Э', 'キリル エー'],
        ['В', 'キリル ヴェー'],
        ['Г', 'キリル ゲー'],
        ['А', 'キリル アー'],
        ['Б', 'キリル ベー'],
        ['Ж', 'キリル ジェー'],
        ['З', 'キリル ゼー'],
        ['Д', 'キリル デー'],
        ['Е', 'キリル ィエー'],
        ['К', 'キリル カー'],
        ['Л', 'キリル エル'],
        ['И', 'キリル イー'],
        ['Й', 'キリル イークラトコエ'],
        ['О', 'キリル オー'],
        ['П', 'キリル ペー'],
        ['М', 'キリル エム'],
        ['Н', 'キリル エヌ'],
        ['鑽', 'ケンサンヲツムノサン'],
        ['Ё', 'キリル ィヨー'],
        ['趨', 'スウセイ ノ スウ'],
        ['又', 'マタガシ ノ マタ'],
        ['鐇', 'ハン、カネヘンニバンゴウノバン'],
        ['趣', 'シュミ ノ シュ，オモムキ'],
        ['鐓', 'タイ、カネヘンニチュウゴクノトンコウノトン'],
        ['鐐', 'リョウ、カネヘンニドウリョウノリョウノツクリ'],
        ['趺', 'フ，アシヘンニオット'],
        ['鐔', 'タン，カネヘンニニシノシタニハヤオキノハヤイ'],
        ['趾', 'ジョウシコウエンノシ'],
        ['鐚', 'ビタイチモンノビタ'],
        ['足', 'アシ，エンソク ノ ソク'],
        ['鐙', 'アブミ'],
        ['越', 'ユウエツカン ノ エツ，コス'],
        ['趁', 'チン，ソウニョウニメズラシイノツクリ'],
        [
            '鐫',
            'セン、カネヘンニフルトリノシタニオウトツノオウノシタ１カクヲトッタモノ',
        ],
        ['*', 'アスタリスク'],
        ['超', 'チョウノウリョク ノ チョウ，コエル'],
        ['趙', 'チュウゴクノコクメイノチョウ'],
        ['鐶', 'カン、カネヘンニジュンカンスルノカンノツクリ'],
        ['鐵', 'テツドウノテツノキュウジ'],
        ['鐺', 'トウ、カネヘンニベントウノトウノキュウジ'],
        ['鐸', 'ドウタク ノ タク'],
        ['叔', 'オジオバ ノ シュク'],
        ['叟', 'サンバソウノソウ，オキナ'],
        ['霏', 'ヒ、アメカンムリニヒジョウグチノヒ'],
        ['幀', 'テイ，ハバヘンニテイシュクナノテイ'],
        ['幃', 'イ，ハバヘンニイダイナノイノツクリ'],
        ['幅', 'ゾウフクスル ノ フク，ハバ'],
        ['幄', 'アク，ハバヘンニオクジョウノオク'],
        ['幇', 'ホウジョスルノホウ'],
        ['幌', 'ホロバシャ ノ ホロ'],
        ['幎', 'ベイ，ハバヘンニメイフクヲイノルノメイ'],
        ['幕', 'ジマク ノ マク'],
        ['幔', 'マンマクノマン，ハバヘンニウナギノツクリ'],
        ['幗', 'カク，ハバヘンニクニノキュウジ'],
        ['輿', 'タマノコシ ノ コシ'],
        ['幟', 'キシヲセンメイニスルノシ，ノボリ'],
        ['幡', 'ヤハタ ノ ハタ'],
        ['幣', 'ゾウヘイキョク ノ ヘイ'],
        ['幢', 'トウ，ハバヘンニグリムドウワノドウ'],
        ['幤', 'ゾウヘイキョクノヘイノイタイジ'],
        ['平', 'ヘイワ ノ ヘイ，タイラ'],
        ['干', 'カンチョウ ノ カン，ホス'],
        ['幵', 'ケン，ホスガフタツ'],
        ['年', 'ガクネン ノ ネン，トシ'],
        ['并', 'ヘイ，ヘイヨウスルノヘイノツクリ'],
        ['幹', 'シンカンセン ノ カン，ミキ'],
        ['幸', 'コウフク ノ コウ，シアワセ'],
        ['幻', 'マボロシ，ゲンソウ ノ ゲン'],
        ['幺', 'ヨウ，ブシュノイトガシラ'],
        ['幽', 'ユウレイセン ノ ユウ'],
        ['幼', 'ヨウチエン ノ ヨウ，オサナイ'],
        ['广', 'ゲン，ブシュノマダレ'],
        ['幾', 'キカガク ノ キ'],
        ['煖', 'ダン，ヒヘンニダンボウノダンノツクリ'],
        ['煕', 'コウキジテンノキ'],
        ['雛', 'ヒナマツリ ノ ヒナ'],
        ['煙', 'エントツ ノ エン，ケムリ'],
        ['煉', 'レンガ ノ レン，ネル'],
        ['煎', 'センチャ ノ セン，イル'],
        ['煌', 'コウ，ヒヘンニコウタイシノコウ'],
        ['煽', 'アオル，センドウスル ノ セン'],
        ['煢', 'ケイ，ホタルノキュウジノカンムリニジンソクナノジンノナカミ'],
        ['照', 'ショウメイ ノ ショウ，テラス'],
        ['煦', 'ク，ヨウビノニチノミギニハイクノクニレンガ'],
        ['煥', 'サイキカンパツノカン'],
        ['煤', 'スス，バイエン ノ バイ'],
        ['煩', 'ボンノウ ノ ボン，ワズラワシイ'],
        ['煮', 'ニモノ ノ ニ，シャフツ ノ シャ'],
        ['煬', 'ヨウ，ヒヘンニタイヨウケイノヨウノツクリ'],
    ]);
})(JaPhoneticMap || (JaPhoneticMap = {}));

// Use of this source code is governed by a BSD-style license that can be
// found in the LICENSE file.
/**
 * @fileoverview Provides Japanese phonetic disambiguation data for ChromeVox.
 */
class JaPhoneticData {
    /** A map containing phonetic disambiguation data for Japanese. */
    static phoneticMap_;
    /** Initialize phoneticMap_ by |map|. */
    static init() {
        this.phoneticMap_ = JaPhoneticMap.MAP;
    }
    /** Returns a phonetic reading for |char|. */
    static forCharacter(char) {
        const characterSet = JaPhoneticData.getCharacterSet(char, JaPhoneticData.CharacterSet.NONE);
        let resultChar = JaPhoneticData.maybeGetLargeLetterKana(char);
        resultChar = JaPhoneticData.phoneticMap_.get(resultChar) || resultChar;
        const prefix = JaPhoneticData.getPrefixForCharacter(characterSet);
        if (prefix) {
            return prefix + ' ' + resultChar;
        }
        return resultChar;
    }
    /** Returns a phonetic reading for |text|. */
    static forText(text) {
        const result = [];
        const chars = [...text];
        let lastCharacterSet = JaPhoneticData.CharacterSet.NONE;
        for (const char of chars) {
            const currentCharacterSet = JaPhoneticData.getCharacterSet(char, lastCharacterSet);
            const info = JaPhoneticData.getPrefixInfo(lastCharacterSet, currentCharacterSet);
            if (info.prefix) {
                // Need to announce the new character set explicitly.
                result.push(info.prefix);
            }
            if (info.delimiter === false && result.length > 0) {
                // Does not convert small Kana if it is not the beginning of the
                // element.
                result[result.length - 1] += char;
            }
            else if (JaPhoneticData.alwaysReadPhonetically(currentCharacterSet)) {
                result.push(JaPhoneticData.phoneticMap_.get(char) || char);
            }
            else {
                result.push(JaPhoneticData.maybeGetLargeLetterKana(char));
            }
            lastCharacterSet = currentCharacterSet;
        }
        return result.join(' ');
    }
    static getCharacterSet(character, lastCharacterSet) {
        // See https://www.unicode.org/charts/PDF/U3040.pdf
        if (character >= 'ぁ' && character <= 'ゖ') {
            if (JaPhoneticData.isSmallLetter(character)) {
                return JaPhoneticData.CharacterSet.HIRAGANA_SMALL_LETTER;
            }
            return JaPhoneticData.CharacterSet.HIRAGANA;
        }
        // See https://www.unicode.org/charts/PDF/U30A0.pdf
        if (character >= 'ァ' && character <= 'ヺ') {
            if (JaPhoneticData.isSmallLetter(character)) {
                return JaPhoneticData.CharacterSet.KATAKANA_SMALL_LETTER;
            }
            return JaPhoneticData.CharacterSet.KATAKANA;
        }
        // See https://unicode.org/charts/PDF/UFF00.pdf
        if (character >= 'ｦ' && character <= 'ﾟ') {
            if (JaPhoneticData.isSmallLetter(character)) {
                return JaPhoneticData.CharacterSet.HALF_WIDTH_KATAKANA_SMALL_LETTER;
            }
            return JaPhoneticData.CharacterSet.HALF_WIDTH_KATAKANA;
        }
        if (character >= 'A' && character <= 'Z') {
            return JaPhoneticData.CharacterSet.HALF_WIDTH_ALPHABET_UPPER;
        }
        if (character >= 'a' && character <= 'z') {
            return JaPhoneticData.CharacterSet.HALF_WIDTH_ALPHABET_LOWER;
        }
        if (character >= '0' && character <= '9') {
            return JaPhoneticData.CharacterSet.HALF_WIDTH_NUMERIC;
        }
        // See https://unicode.org/charts/PDF/U0000.pdf
        if (character >= '!' && character <= '~') {
            return JaPhoneticData.CharacterSet.HALF_WIDTH_SYMBOL;
        }
        if (character >= 'Ａ' && character <= 'Ｚ') {
            return JaPhoneticData.CharacterSet.FULL_WIDTH_ALPHABET_UPPER;
        }
        if (character >= 'ａ' && character <= 'ｚ') {
            return JaPhoneticData.CharacterSet.FULL_WIDTH_ALPHABET_LOWER;
        }
        if (character >= '０' && character <= '９') {
            return JaPhoneticData.CharacterSet.FULL_WIDTH_NUMERIC;
        }
        // See https://unicode.org/charts/PDF/UFF00.pdf
        if (character >= '！' && character <= '～') {
            return JaPhoneticData.CharacterSet.FULL_WIDTH_SYMBOL;
        }
        if (character === 'ー') {
            switch (lastCharacterSet) {
                case JaPhoneticData.CharacterSet.HIRAGANA:
                case JaPhoneticData.CharacterSet.KATAKANA:
                case JaPhoneticData.CharacterSet.HIRAGANA_SMALL_LETTER:
                case JaPhoneticData.CharacterSet.KATAKANA_SMALL_LETTER:
                    return lastCharacterSet;
            }
        }
        // See https://www.unicode.org/charts/PDF/U0400.pdf and
        // https://www.unicode.org/charts/PDF/U0370.pdf
        if ((character >= 'А' && character <= 'Я') ||
            (character >= 'Α' && character <= 'Ω')) {
            return JaPhoneticData.CharacterSet.FULL_WIDTH_CYRILLIC_OR_GREEK_UPPER;
        }
        if ((character >= 'а' && character <= 'я') ||
            (character >= 'α' && character <= 'ω')) {
            return JaPhoneticData.CharacterSet.FULL_WIDTH_CYRILLIC_OR_GREEK_LOWER;
        }
        // Returns OTHER for all other characters, including Kanji.
        return JaPhoneticData.CharacterSet.OTHER;
    }
    /**
     * Returns true if the character is part of the small letter character set.
     */
    static isSmallLetter(character) {
        return JaPhoneticData.SMALL_TO_LARGE.has(character);
    }
    /** Returns a large equivalent if the character is a small letter of Kana. */
    static maybeGetLargeLetterKana(character) {
        return JaPhoneticData.SMALL_TO_LARGE.get(character) || character;
    }
    static getDefaultPrefix(characterSet) {
        // TODO(b/314203187): Not null asserted, check that this is correct.
        return JaPhoneticData.DEFAULT_PREFIX.get(characterSet);
    }
    static getPrefixForCharacter(characterSet) {
        // Removing an annoucement of capital because users can distinguish
        // uppercase and lowercase by capiatalStrategy options.
        switch (characterSet) {
            case JaPhoneticData.CharacterSet.HALF_WIDTH_ALPHABET_UPPER:
            case JaPhoneticData.CharacterSet.HALF_WIDTH_ALPHABET_LOWER:
            case JaPhoneticData.CharacterSet.FULL_WIDTH_CYRILLIC_OR_GREEK_UPPER:
            case JaPhoneticData.CharacterSet.FULL_WIDTH_CYRILLIC_OR_GREEK_LOWER:
            case JaPhoneticData.CharacterSet.OTHER:
                return null;
            case JaPhoneticData.CharacterSet.FULL_WIDTH_ALPHABET_UPPER:
                return 'ゼンカク';
        }
        return JaPhoneticData.getDefaultPrefix(characterSet);
    }
    /**
     * Returns an object containing the relationship between the preceding
     * character set and the current character set.
     * @return Object containing prefixes.
     * delimiter: true if a space between preceding character and current
     * character is necessary. A space leaves a pause so users can recognize that
     * the type of characters has changed.
     * prefix: a string that represents the character set. Null if unncessary.
     */
    static getPrefixInfo(lastCharacterSet, currentCharacterSet) {
        // Don't add prefixes for the same character set except for the sets always
        // read phonetically.
        if (lastCharacterSet === currentCharacterSet) {
            return JaPhoneticData.alwaysReadPhonetically(currentCharacterSet) ?
                { delimiter: true, prefix: null } :
                { delimiter: false, prefix: null };
        }
        // Exceptional cases:
        switch (currentCharacterSet) {
            case JaPhoneticData.CharacterSet.HIRAGANA:
                switch (lastCharacterSet) {
                    case JaPhoneticData.CharacterSet.NONE:
                    case JaPhoneticData.CharacterSet.HIRAGANA_SMALL_LETTER:
                        return { delimiter: false, prefix: null };
                    case JaPhoneticData.CharacterSet.HALF_WIDTH_ALPHABET_UPPER:
                    case JaPhoneticData.CharacterSet.HALF_WIDTH_ALPHABET_LOWER:
                    case JaPhoneticData.CharacterSet.HALF_WIDTH_NUMERIC:
                    case JaPhoneticData.CharacterSet.FULL_WIDTH_ALPHABET_UPPER:
                    case JaPhoneticData.CharacterSet.FULL_WIDTH_ALPHABET_LOWER:
                    case JaPhoneticData.CharacterSet.FULL_WIDTH_NUMERIC:
                    case JaPhoneticData.CharacterSet.OTHER:
                        return { delimiter: true, prefix: null };
                }
                break;
            case JaPhoneticData.CharacterSet.KATAKANA:
                switch (lastCharacterSet) {
                    case JaPhoneticData.CharacterSet.KATAKANA_SMALL_LETTER:
                        return { delimiter: false, prefix: null };
                }
                break;
            case JaPhoneticData.CharacterSet.HIRAGANA_SMALL_LETTER:
                switch (lastCharacterSet) {
                    case JaPhoneticData.CharacterSet.NONE:
                    case JaPhoneticData.CharacterSet.HIRAGANA:
                        return { delimiter: false, prefix: null };
                    case JaPhoneticData.CharacterSet.OTHER:
                        return { delimiter: true, prefix: null };
                }
                break;
            case JaPhoneticData.CharacterSet.KATAKANA_SMALL_LETTER:
                switch (lastCharacterSet) {
                    case JaPhoneticData.CharacterSet.KATAKANA:
                        return { delimiter: false, prefix: null };
                }
                break;
            case JaPhoneticData.CharacterSet.HALF_WIDTH_KATAKANA:
                switch (lastCharacterSet) {
                    case JaPhoneticData.CharacterSet.HALF_WIDTH_KATAKANA_SMALL_LETTER:
                        return { delimiter: false, prefix: null };
                }
                break;
            case JaPhoneticData.CharacterSet.HALF_WIDTH_KATAKANA_SMALL_LETTER:
                switch (lastCharacterSet) {
                    case JaPhoneticData.CharacterSet.HALF_WIDTH_KATAKANA:
                        return { delimiter: false, prefix: null };
                }
                break;
            case JaPhoneticData.CharacterSet.HALF_WIDTH_ALPHABET_UPPER:
                switch (lastCharacterSet) {
                    case JaPhoneticData.CharacterSet.FULL_WIDTH_ALPHABET_UPPER:
                    case JaPhoneticData.CharacterSet.FULL_WIDTH_ALPHABET_LOWER:
                    case JaPhoneticData.CharacterSet.FULL_WIDTH_NUMERIC:
                    case JaPhoneticData.CharacterSet.FULL_WIDTH_SYMBOL:
                        return { delimiter: true, prefix: 'ハンカクオオモジ' };
                }
                break;
            case JaPhoneticData.CharacterSet.HALF_WIDTH_ALPHABET_LOWER:
            case JaPhoneticData.CharacterSet.HALF_WIDTH_NUMERIC:
            case JaPhoneticData.CharacterSet.HALF_WIDTH_SYMBOL:
                switch (lastCharacterSet) {
                    case JaPhoneticData.CharacterSet.NONE:
                        return { delimiter: false, prefix: null };
                    case JaPhoneticData.CharacterSet.HIRAGANA:
                    case JaPhoneticData.CharacterSet.KATAKANA:
                    case JaPhoneticData.CharacterSet.HIRAGANA_SMALL_LETTER:
                    case JaPhoneticData.CharacterSet.KATAKANA_SMALL_LETTER:
                    case JaPhoneticData.CharacterSet.HALF_WIDTH_KATAKANA:
                    case JaPhoneticData.CharacterSet.HALF_WIDTH_KATAKANA_SMALL_LETTER:
                    case JaPhoneticData.CharacterSet.HALF_WIDTH_ALPHABET_UPPER:
                    case JaPhoneticData.CharacterSet.HALF_WIDTH_ALPHABET_LOWER:
                    case JaPhoneticData.CharacterSet.HALF_WIDTH_NUMERIC:
                    case JaPhoneticData.CharacterSet.HALF_WIDTH_SYMBOL:
                    case JaPhoneticData.CharacterSet.OTHER:
                        return { delimiter: true, prefix: null };
                }
                break;
            case JaPhoneticData.CharacterSet.FULL_WIDTH_ALPHABET_UPPER:
                switch (lastCharacterSet) {
                    case JaPhoneticData.CharacterSet.FULL_WIDTH_ALPHABET_LOWER:
                    case JaPhoneticData.CharacterSet.FULL_WIDTH_NUMERIC:
                    case JaPhoneticData.CharacterSet.FULL_WIDTH_SYMBOL:
                        return { delimiter: true, prefix: 'オオモジ' };
                }
                break;
            case JaPhoneticData.CharacterSet.FULL_WIDTH_ALPHABET_LOWER:
            case JaPhoneticData.CharacterSet.FULL_WIDTH_NUMERIC:
            case JaPhoneticData.CharacterSet.FULL_WIDTH_SYMBOL:
                switch (lastCharacterSet) {
                    case JaPhoneticData.CharacterSet.FULL_WIDTH_ALPHABET_UPPER:
                    case JaPhoneticData.CharacterSet.FULL_WIDTH_ALPHABET_LOWER:
                    case JaPhoneticData.CharacterSet.FULL_WIDTH_NUMERIC:
                    case JaPhoneticData.CharacterSet.FULL_WIDTH_SYMBOL:
                        return { delimiter: true, prefix: null };
                }
                break;
            case JaPhoneticData.CharacterSet.FULL_WIDTH_CYRILLIC_OR_GREEK_LOWER:
                switch (lastCharacterSet) {
                    case JaPhoneticData.CharacterSet.FULL_WIDTH_CYRILLIC_OR_GREEK_UPPER:
                        return {
                            delimiter: true,
                            prefix: JaPhoneticData.getDefaultPrefix(currentCharacterSet),
                        };
                }
                return { delimiter: true, prefix: null };
            case JaPhoneticData.CharacterSet.OTHER:
                return { delimiter: true, prefix: null };
        }
        // Returns the default prefix.
        return {
            delimiter: true,
            prefix: JaPhoneticData.getDefaultPrefix(currentCharacterSet),
        };
    }
    /**
     * @param {JaPhoneticData.CharacterSet} characterSet
     * @return {boolean}
     */
    static alwaysReadPhonetically(characterSet) {
        switch (characterSet) {
            case JaPhoneticData.CharacterSet.HALF_WIDTH_SYMBOL:
            case JaPhoneticData.CharacterSet.FULL_WIDTH_SYMBOL:
            case JaPhoneticData.CharacterSet.FULL_WIDTH_CYRILLIC_OR_GREEK_UPPER:
            case JaPhoneticData.CharacterSet.FULL_WIDTH_CYRILLIC_OR_GREEK_LOWER:
            case JaPhoneticData.CharacterSet.OTHER:
                return true;
        }
        return false;
    }
}
(function (JaPhoneticData) {
    let CharacterSet;
    (function (CharacterSet) {
        CharacterSet[CharacterSet["NONE"] = 0] = "NONE";
        CharacterSet[CharacterSet["HIRAGANA"] = 1] = "HIRAGANA";
        CharacterSet[CharacterSet["KATAKANA"] = 2] = "KATAKANA";
        CharacterSet[CharacterSet["HIRAGANA_SMALL_LETTER"] = 3] = "HIRAGANA_SMALL_LETTER";
        CharacterSet[CharacterSet["KATAKANA_SMALL_LETTER"] = 4] = "KATAKANA_SMALL_LETTER";
        CharacterSet[CharacterSet["HALF_WIDTH_KATAKANA"] = 5] = "HALF_WIDTH_KATAKANA";
        CharacterSet[CharacterSet["HALF_WIDTH_KATAKANA_SMALL_LETTER"] = 6] = "HALF_WIDTH_KATAKANA_SMALL_LETTER";
        CharacterSet[CharacterSet["HALF_WIDTH_ALPHABET_UPPER"] = 7] = "HALF_WIDTH_ALPHABET_UPPER";
        CharacterSet[CharacterSet["HALF_WIDTH_ALPHABET_LOWER"] = 8] = "HALF_WIDTH_ALPHABET_LOWER";
        CharacterSet[CharacterSet["HALF_WIDTH_NUMERIC"] = 9] = "HALF_WIDTH_NUMERIC";
        CharacterSet[CharacterSet["HALF_WIDTH_SYMBOL"] = 10] = "HALF_WIDTH_SYMBOL";
        CharacterSet[CharacterSet["FULL_WIDTH_ALPHABET_UPPER"] = 11] = "FULL_WIDTH_ALPHABET_UPPER";
        CharacterSet[CharacterSet["FULL_WIDTH_ALPHABET_LOWER"] = 12] = "FULL_WIDTH_ALPHABET_LOWER";
        CharacterSet[CharacterSet["FULL_WIDTH_NUMERIC"] = 13] = "FULL_WIDTH_NUMERIC";
        CharacterSet[CharacterSet["FULL_WIDTH_SYMBOL"] = 14] = "FULL_WIDTH_SYMBOL";
        CharacterSet[CharacterSet["FULL_WIDTH_CYRILLIC_OR_GREEK_UPPER"] = 15] = "FULL_WIDTH_CYRILLIC_OR_GREEK_UPPER";
        CharacterSet[CharacterSet["FULL_WIDTH_CYRILLIC_OR_GREEK_LOWER"] = 16] = "FULL_WIDTH_CYRILLIC_OR_GREEK_LOWER";
        CharacterSet[CharacterSet["OTHER"] = 17] = "OTHER";
    })(CharacterSet = JaPhoneticData.CharacterSet || (JaPhoneticData.CharacterSet = {}));
    JaPhoneticData.DEFAULT_PREFIX = new Map([
        // 'あ'
        [CharacterSet.HIRAGANA, 'ヒラガナ'],
        // 'ア'
        [CharacterSet.KATAKANA, 'カタカナ'],
        // 'ぁ'
        [CharacterSet.HIRAGANA_SMALL_LETTER, 'ヒラガナ チイサイ'],
        // 'ァ'
        [CharacterSet.KATAKANA_SMALL_LETTER, 'カタカナ チイサイ'],
        // 'ｱ'
        [CharacterSet.HALF_WIDTH_KATAKANA, 'ハンカク'],
        // 'ｧ'
        [CharacterSet.HALF_WIDTH_KATAKANA_SMALL_LETTER, 'ハンカク チイサイ'],
        // 'A'
        [CharacterSet.HALF_WIDTH_ALPHABET_UPPER, 'オオモジ'],
        // 'a'
        [CharacterSet.HALF_WIDTH_ALPHABET_LOWER, 'ハンカク'],
        // '1'
        [CharacterSet.HALF_WIDTH_NUMERIC, 'ハンカク'],
        // '@'
        [CharacterSet.HALF_WIDTH_SYMBOL, 'ハンカク'],
        // 'Ａ'
        [CharacterSet.FULL_WIDTH_ALPHABET_UPPER, 'ゼンカクオオモジ'],
        // 'ａ'
        [CharacterSet.FULL_WIDTH_ALPHABET_LOWER, 'ゼンカク'],
        // '１'
        [CharacterSet.FULL_WIDTH_NUMERIC, 'ゼンカク'],
        // '＠'
        [CharacterSet.FULL_WIDTH_SYMBOL, 'ゼンカク'],
        // 'Α'
        [CharacterSet.FULL_WIDTH_CYRILLIC_OR_GREEK_UPPER, 'オオモジ'],
        // 'α'
        [CharacterSet.FULL_WIDTH_CYRILLIC_OR_GREEK_LOWER, 'コモジ'],
    ]);
    /** This object maps small letters of Kana to their large equivalents. */
    JaPhoneticData.SMALL_TO_LARGE = new Map([
        // Hiragana
        ['ぁ', 'あ'],
        ['ぃ', 'い'],
        ['ぅ', 'う'],
        ['ぇ', 'え'],
        ['ぉ', 'お'],
        ['っ', 'つ'],
        ['ゃ', 'や'],
        ['ゅ', 'ゆ'],
        ['ょ', 'よ'],
        ['ゎ', 'わ'],
        ['ゕ', 'か'],
        ['ゖ', 'け'],
        // Katakana
        ['ァ', 'ア'],
        ['ィ', 'イ'],
        ['ゥ', 'ウ'],
        ['ェ', 'エ'],
        ['ォ', 'オ'],
        ['ッ', 'ツ'],
        ['ャ', 'ヤ'],
        ['ュ', 'ユ'],
        ['ョ', 'ヨ'],
        ['ヮ', 'ワ'],
        ['ヵ', 'カ'],
        ['ヶ', 'ケ'],
        // HalfWidthKatakana
        ['ｧ', 'ｱ'],
        ['ｨ', 'ｲ'],
        ['ｩ', 'ｳ'],
        ['ｪ', 'ｴ'],
        ['ｫ', 'ｵ'],
        ['ｬ', 'ﾔ'],
        ['ｭ', 'ﾕ'],
        ['ｮ', 'ﾖ'],
        ['ｯ', 'ﾂ'],
    ]);
})(JaPhoneticData || (JaPhoneticData = {}));

// Copyright 2019 The Chromium Authors
// Use of this source code is governed by a BSD-style license that can be
// found in the LICENSE file.
/**
 * @fileoverview Provides phonetic disambiguation functionality across multiple
 * languages for ChromeVox.
 */
class PhoneticData {
    /**
     * Returns the phonetic disambiguation for |char| in |locale|.
     * Returns empty string if disambiguation can't be found.
     * @param {string} char
     * @param {string} locale
     * @return {string}
     */
    static forCharacter(char, locale) {
        if (!char || !locale) {
            throw Error('PhoneticData api requires non-empty arguments.');
        }
        if (locale === 'ja') {
            return JaPhoneticData.forCharacter(char);
        }
        char = char.toLowerCase();
        locale = locale.toLowerCase();
        let map = null;
        // Try a lookup using |locale|, but use only the language component if the
        // lookup fails, e.g. "en-us" -> "en" or "zh-hant-hk" -> "zh".
        map = PhoneticDictionaries.phoneticMap_[locale] ||
            PhoneticDictionaries.phoneticMap_[locale.split('-')[0]];
        if (!map) {
            return '';
        }
        return map[char] || '';
    }
    /**
     * @param {string} text
     * @param {string} locale
     * @return {string}
     */
    static forText(text, locale) {
        if (locale === 'ja') {
            // Japanese phonetic readings require specialized logic.
            return JaPhoneticData.forText(text);
        }
        const result = [];
        const chars = [...text];
        for (const char of chars) {
            const phoneticText = PhoneticData.forCharacter(char, locale);
            result.push(char + ': ' + phoneticText);
        }
        return result.join(', ');
    }
}
TestImportManager.exportForTesting(PhoneticData, JaPhoneticData);

// Copyright 2023 The Chromium Authors
// Use of this source code is governed by a BSD-style license that can be
// found in the LICENSE file.
/**
 * @fileoverview Default implementation for TTS in the background context.
 */
class Utterance {
    static nextUtteranceId_ = 1;
    textString;
    properties;
    queueMode;
    id;
    /**
     * @param textString The string of text to be spoken.
     * @param properties Speech properties to use for this utterance.
     */
    constructor(textString, properties, queueMode) {
        this.textString = textString;
        this.properties = properties;
        this.queueMode = queueMode;
        this.id = Utterance.nextUtteranceId_++;
    }
}
/**
 * This class is the default implementation for TTS in the background context.
 */
class PrimaryTts extends AbstractTts {
    // The amount of time, in milliseconds, to wait before speaking a hint.
    static hint_delay_ms_ = 1000;
    /**
     * The list of properties allowed to be passed to the chrome.tts.speak API.
     * Anything outside this list will be stripped.
     */
    static ALLOWED_PROPERTIES_ = [
        'desiredEventTypes',
        'enqueue',
        'extensionId',
        'gender',
        'lang',
        'onEvent',
        'pitch',
        'rate',
        'requiredEventTypes',
        'voiceName',
        'volume',
    ];
    static SKIP_WHITESPACE_ = /^[\s\u00a0]*$/;
    // The current voice name.
    currentVoice;
    lastEventType = chrome.tts.EventType.END;
    currentPunctuationEcho_;
    /**
     * A list of punctuation characters that should always be spliced into
     * output even with literal word substitutions. This is important for tts
     * prosity.
     */
    retainPunctuation_ = [';', '?', '!', '\''];
    // The id of a callback returned by setTimeout.
    timeoutId_;
    // Capturing tts event listeners.
    capturingTtsEventListeners_ = [];
    // The current utterance.
    currentUtterance_ = null;
    // The utterance queue.
    utteranceQueue_ = [];
    // Queue of utterances interrupted by interjected utterances.
    utteranceQueueInterruptedByInterjection_ = [];
    constructor() {
        super();
        this.currentPunctuationEcho_ =
            SettingsManager.getNumber(TtsSettings.PUNCTUATION_ECHO);
        if (window.speechSynthesis) {
            window.speechSynthesis.onvoiceschanged = () => this.updateVoice(SettingsManager.getString('voiceName'));
        }
        else {
            // SpeechSynthesis API is not available on chromecast. Call
            // updateVoice to set the one and only voice as the current
            // voice.
            this.updateVoice('');
        }
        SettingsManager.addListenerForKey('voiceName', voiceName => this.updateVoice(voiceName));
        // Migration: local LocalStorage tts properties -> Chrome pref settings.
        if (LocalStorage.get('rate')) {
            chrome.settingsPrivate.setPref('settings.tts.speech_rate', LocalStorage.get('rate'));
            LocalStorage.remove('rate');
        }
        if (LocalStorage.get('pitch')) {
            chrome.settingsPrivate.setPref('settings.tts.speech_pitch', LocalStorage.get('pitch'));
            LocalStorage.remove('pitch');
        }
        if (LocalStorage.get('volume')) {
            chrome.settingsPrivate.setPref('settings.tts.speech_volume', LocalStorage.get('volume'));
            LocalStorage.remove('volume');
        }
        // At startup.
        chrome.settingsPrivate.getAllPrefs(prefs => this.updateFromPrefs_(false, prefs));
        // At runtime.
        chrome.settingsPrivate.onPrefsChanged.addListener(prefs => this.updateFromPrefs_(true, prefs));
    }
    /**
     * @param textString The string of text to be spoken.
     * @param queueMode The queue mode to use for speaking.
     * @param properties Speech properties to use
     *     for this utterance.
     * @return A tts object useful for chaining speak calls.
     */
    speak(textString, queueMode, properties) {
        super.speak(textString, queueMode, properties);
        // |textString| gets manipulated throughout this function. Save the
        // original value for functions that may need it.
        const originalTextString = textString;
        if (this.ttsProperties[TtsSettings.VOLUME] === 0) {
            return this;
        }
        if (!properties) {
            properties = new TtsSpeechProperties();
        }
        if (textString.length > constants.OBJECT_MAX_CHARCOUNT) {
            // The text is too long. Try to split the text into multiple chunks
            // based on line breaks.
            this.speakSplittingText_(textString, queueMode, properties);
            return this;
        }
        textString = this.preprocess(textString, properties);
        // This pref on SettingsManager gets set by the options page.
        if (SettingsManager.get('numberReadingStyle') === 'asDigits') {
            textString = this.getNumberAsDigits_(textString);
        }
        // TODO(dtseng): some TTS engines don't handle strings that don't produce
        // any speech very well. Handle empty and whitespace only strings
        // (including non-breaking space) here to mitigate the issue somewhat.
        if (PrimaryTts.SKIP_WHITESPACE_.test(textString)) {
            // Explicitly call start and end callbacks before skipping this text.
            if (properties.startCallback) {
                try {
                    properties.startCallback();
                }
                catch (e) {
                }
            }
            if (properties.endCallback) {
                try {
                    properties.endCallback();
                }
                catch (e) {
                }
            }
            if (queueMode === QueueMode.FLUSH) {
                this.stop();
            }
            return this;
        }
        const mergedProperties = this.mergeProperties(properties);
        if (this.currentVoice && this.currentVoice !== constants.SYSTEM_VOICE) {
            mergedProperties['voiceName'] = this.currentVoice;
        }
        const utterance = new Utterance(textString, mergedProperties, queueMode);
        this.speakUsingQueue_(utterance);
        // Attempt to queue phonetic speech with property['delay']. This ensures
        // that phonetic hints are delayed when we process them.
        this.pronouncePhonetically_(originalTextString, properties);
        return this;
    }
    getUtteranceQueueForTest() {
        return this.utteranceQueue_;
    }
    /**
     * Split the given textString into smaller chunks and call this.speak() for
     * each chunks.
     * @param textString The string of text to be spoken.
     * @param queueMode The queue mode to use for speaking.
     * @param properties Speech properties to use for this utterance.
     */
    speakSplittingText_(textString, queueMode, properties) {
        const chunks = PrimaryTts.splitUntilSmall(textString, '\n\r ');
        for (const chunk of chunks) {
            this.speak(chunk, queueMode, properties);
            queueMode = QueueMode.QUEUE;
        }
    }
    /**
     * Splits |text| until each substring's length is smaller than or equal to
     * constants.OBJECT_MAX_CHARCOUNT.
     */
    static splitUntilSmall(text, delimiters) {
        if (text.length === 0) {
            return [];
        }
        if (text.length <= constants.OBJECT_MAX_CHARCOUNT) {
            return [text];
        }
        const midIndex = text.length / 2;
        if (!delimiters) {
            return PrimaryTts.splitUntilSmall(text.substring(0, midIndex), delimiters)
                .concat(PrimaryTts.splitUntilSmall(text.substring(midIndex, text.length), delimiters));
        }
        const delimiter = delimiters[0];
        let splitIndex = text.lastIndexOf(delimiter, midIndex);
        if (splitIndex === -1) {
            splitIndex = text.indexOf(delimiter, midIndex);
        }
        if (splitIndex === -1) {
            delimiters = delimiters.slice(1);
            return PrimaryTts.splitUntilSmall(text, delimiters);
        }
        return PrimaryTts.splitUntilSmall(text.substring(0, splitIndex), delimiters)
            .concat(PrimaryTts.splitUntilSmall(text.substring(splitIndex + 1, text.length), delimiters));
    }
    /**
     * Use the speech queue to handle the given speech request.
     * @param utterance The utterance to speak.
     */
    speakUsingQueue_(utterance) {
        const queueMode = utterance.queueMode;
        // First, take care of removing the current utterance and flushing
        // anything from the queue we need to. If we remove the current utterance,
        // make a note that we're going to stop speech.
        if (queueMode === QueueMode.FLUSH ||
            queueMode === QueueMode.CATEGORY_FLUSH ||
            queueMode === QueueMode.INTERJECT) {
            (new PanelCommand(PanelCommandType.CLEAR_SPEECH)).send();
            if (this.shouldCancel_(this.currentUtterance_, utterance)) {
                // Clear timeout in case currentUtterance_ is a delayed utterance.
                this.clearTimeout_();
                this.cancelUtterance_(this.currentUtterance_);
                this.currentUtterance_ = null;
            }
            let i = 0;
            while (i < this.utteranceQueue_.length) {
                if (this.shouldCancel_(this.utteranceQueue_[i], utterance)) {
                    this.cancelUtterance_(this.utteranceQueue_[i]);
                    this.utteranceQueue_.splice(i, 1);
                }
                else {
                    i++;
                }
            }
        }
        // Now, some special handling for interjections.
        if (queueMode === QueueMode.INTERJECT) {
            // Move all utterances to a secondary queue to be restored later.
            this.utteranceQueueInterruptedByInterjection_ = this.utteranceQueue_;
            // The interjection is the only utterance.
            this.utteranceQueue_ = [utterance];
            // Ensure to clear the current utterance and prepend it for it to repeat
            // later.
            if (this.currentUtterance_) {
                this.utteranceQueueInterruptedByInterjection_.unshift(this.currentUtterance_);
                this.currentUtterance_ = null;
            }
            // Restore the interrupted utterances after allowing all other
            // utterances in this callstack to process.
            setTimeout(() => {
                // Utterances on the current queue are now also interjections.
                for (let i = 0; i < this.utteranceQueue_.length; i++) {
                    this.utteranceQueue_[i].queueMode = QueueMode.INTERJECT;
                }
                this.utteranceQueue_ = this.utteranceQueue_.concat(this.utteranceQueueInterruptedByInterjection_);
            }, 0);
        }
        else {
            // Next, add the new utterance to the queue.
            this.utteranceQueue_.push(utterance);
        }
        // Now start speaking the next item in the queue.
        this.startSpeakingNextItemInQueue_();
    }
    /**
     * If nothing is speaking, pop the first item off the speech queue and
     * start speaking it. This is called when a speech request is made and
     * when the current utterance finishes speaking.
     */
    startSpeakingNextItemInQueue_() {
        if (this.currentUtterance_) {
            return;
        }
        if (this.utteranceQueue_.length === 0) {
            return;
        }
        // There is no voice to speak with (e.g. the tts system has not fully
        // initialized).
        if (!this.currentVoice) {
            return;
        }
        // Clear timeout for delayed utterances (hints and phonetic speech).
        this.clearTimeout_();
        // Check top of utteranceQueue for delayed utterance.
        if (this.utteranceQueue_[0].properties['delay']) {
            // Remove 'delay' property and set a timeout to process this utterance
            // after the delay has passed.
            delete this.utteranceQueue_[0].properties['delay'];
            this.timeoutId_ = setTimeout(() => this.startSpeakingNextItemInQueue_(), PrimaryTts.hint_delay_ms_);
            return;
        }
        this.currentUtterance_ = this.utteranceQueue_.shift();
        const utterance = this.currentUtterance_;
        const utteranceId = utterance.id;
        utterance.properties['onEvent'] = event => {
            this.onTtsEvent_(event, utteranceId);
        };
        const validatedProperties = {};
        for (let i = 0; i < PrimaryTts.ALLOWED_PROPERTIES_.length; i++) {
            const p = PrimaryTts.ALLOWED_PROPERTIES_[i];
            if (utterance.properties[p]) {
                validatedProperties[p] = utterance.properties[p];
            }
        }
        // Update the caption panel.
        if (utterance.properties && utterance.properties['pitch'] &&
            utterance.properties['pitch'] < this.ttsProperties['pitch']) {
            (new PanelCommand(PanelCommandType.ADD_ANNOTATION_SPEECH, utterance.textString))
                .send();
        }
        else {
            (new PanelCommand(PanelCommandType.ADD_NORMAL_SPEECH, utterance.textString))
                .send();
        }
        chrome.tts.speak(utterance.textString, validatedProperties);
    }
    /**
     * Called when we get a speech event from Chrome. We ignore any event
     * that doesn't pertain to the current utterance, but when speech starts
     * or ends we optionally call callback functions, and start speaking the
     * next utterance if there's another one enqueued.
     * @param event The TTS event from chrome.
     * @param utteranceId The id of the associated utterance.
     */
    onTtsEvent_(event, utteranceId) {
        this.lastEventType = event['type'];
        // Ignore events sent on utterances other than the current one.
        if (!this.currentUtterance_ || utteranceId !== this.currentUtterance_.id) {
            return;
        }
        const utterance = this.currentUtterance_;
        switch (event['type']) {
            case chrome.tts.EventType.START:
                this.capturingTtsEventListeners_.forEach(listener => listener.onTtsStart());
                if (utterance.properties['startCallback']) {
                    try {
                        utterance.properties['startCallback']();
                    }
                    catch (e) {
                    }
                }
                break;
            case chrome.tts.EventType.END:
                // End callbacks could cause additional speech to queue up.
                this.currentUtterance_ = null;
                this.capturingTtsEventListeners_.forEach(listener => listener.onTtsEnd());
                if (utterance.properties['endCallback']) {
                    try {
                        utterance.properties['endCallback']();
                    }
                    catch (e) {
                    }
                }
                this.startSpeakingNextItemInQueue_();
                break;
            case chrome.tts.EventType.INTERRUPTED:
                this.cancelUtterance_(utterance);
                this.currentUtterance_ = null;
                for (let i = 0; i < this.utteranceQueue_.length; i++) {
                    this.cancelUtterance_(this.utteranceQueue_[i]);
                }
                this.utteranceQueue_.length = 0;
                this.capturingTtsEventListeners_.forEach(listener => listener.onTtsInterrupted());
                break;
            case chrome.tts.EventType.ERROR:
                this.onError_(event['errorMessage']);
                this.currentUtterance_ = null;
                this.startSpeakingNextItemInQueue_();
                break;
        }
    }
    /**
     * Determines if |utteranceToCancel| should be canceled (interrupted if
     * currently speaking, or removed from the queue if not), given the new
     * utterance we want to speak and the queue mode. If the queue mode is
     * QUEUE or FLUSH, the logic is straightforward. If the queue mode is
     * CATEGORY_FLUSH, we only flush utterances with the same category.
     *
     * @param utteranceToCancel The utterance in question.
     * @param newUtterance The new utterance we're enqueueing.
     * @return True if this utterance should be canceled.
     */
    shouldCancel_(utteranceToCancel, newUtterance) {
        if (!utteranceToCancel) {
            return false;
        }
        if (utteranceToCancel.properties['doNotInterrupt']) {
            return false;
        }
        switch (newUtterance.queueMode) {
            case QueueMode.QUEUE:
                return false;
            case QueueMode.INTERJECT:
                return utteranceToCancel.queueMode === QueueMode.INTERJECT;
            case QueueMode.FLUSH:
                return true;
            case QueueMode.CATEGORY_FLUSH:
                return (utteranceToCancel.properties['category'] ===
                    newUtterance.properties['category']);
        }
    }
    /**
     * Do any cleanup necessary to cancel an utterance, like callings its
     * callback function if any.
     * @param utterance The utterance to cancel.
     */
    cancelUtterance_(utterance) {
        if (utterance && utterance.properties['endCallback']) {
            try {
                utterance.properties['endCallback'](true);
            }
            catch (e) {
            }
        }
    }
    increaseOrDecreaseProperty(propertyName, increase) {
        super.increaseOrDecreaseProperty(propertyName, increase);
        const value = this.ttsProperties[propertyName];
        this.setProperty(propertyName, value);
    }
    setProperty(propertyName, value) {
        super.setProperty(propertyName, value);
        let pref;
        switch (propertyName) {
            case TtsAudioProperty.RATE:
                pref = 'settings.tts.speech_rate';
                break;
            case TtsAudioProperty.PITCH:
                pref = 'settings.tts.speech_pitch';
                break;
            case TtsAudioProperty.VOLUME:
                pref = 'settings.tts.speech_volume';
                break;
            default:
                return;
        }
        chrome.settingsPrivate.setPref(pref, this.ttsProperties[propertyName]);
    }
    isSpeaking() {
        super.isSpeaking();
        return Boolean(this.currentUtterance_);
    }
    stop() {
        super.stop();
        this.cancelUtterance_(this.currentUtterance_);
        this.currentUtterance_ = null;
        for (let i = 0; i < this.utteranceQueue_.length; i++) {
            this.cancelUtterance_(this.utteranceQueue_[i]);
        }
        for (let i = 0; i < this.utteranceQueueInterruptedByInterjection_.length; i++) {
            this.cancelUtterance_(this.utteranceQueueInterruptedByInterjection_[i]);
        }
        this.utteranceQueue_.length = 0;
        this.utteranceQueueInterruptedByInterjection_.length = 0;
        (new PanelCommand(PanelCommandType.CLEAR_SPEECH)).send();
        chrome.tts.stop();
        this.capturingTtsEventListeners_.forEach(listener => listener.onTtsInterrupted());
    }
    addCapturingEventListener(listener) {
        this.capturingTtsEventListeners_.push(listener);
    }
    removeCapturingEventListener(listener) {
        this.capturingTtsEventListeners_ =
            this.capturingTtsEventListeners_.filter(item => {
                return item !== listener;
            });
    }
    /**
     * An error handler passed as a callback to chrome.tts.speak.
     * @param errorMessage Describes the error (set by onEvent).
     */
    onError_(_errorMessage) {
        this.updateVoice(this.currentVoice);
    }
    preprocess(text, properties = new TtsSpeechProperties()) {
        // Perform generic processing.
        text = super.preprocess(text, properties);
        // Perform any remaining processing such as punctuation expansion.
        let punctEcho;
        if (properties[TtsSettings.PUNCTUATION_ECHO]) {
            for (let i = 0; punctEcho = PunctuationEchoes[i]; i++) {
                if (properties[TtsSettings.PUNCTUATION_ECHO] === punctEcho.name) {
                    break;
                }
            }
        }
        else {
            punctEcho = PunctuationEchoes[this.currentPunctuationEcho_];
        }
        text = text.replace(punctEcho.regexp, this.createPunctuationReplace_(punctEcho.clear));
        // Remove all whitespace from the beginning and end, and collapse all
        // inner strings of whitespace to a single space.
        text = text.replace(/\s+/g, ' ').replace(/^\s+|\s+$/g, '');
        // Look for the pattern [number + lowercase letter], such as in "5g
        // network". We want to capitalize the letter to prevent it from being
        // substituted with a unit in the TTS engine; in the above case, the
        // string would get spoken as "5 grams network", which we want to avoid.
        // We do not match against the letter "a" in the regular expression
        // because it is a word and we do not want to capitalize it just because
        // it comes after a number.
        text = text.replace(/(\d)(\s*)([b-z])\b/g, (_unused, num, whitespace, letter) => num + whitespace + letter.toUpperCase());
        return text;
    }
    toggleSpeechOnOrOff() {
        const previousValue = this.ttsProperties[TtsSettings.VOLUME];
        const toggle = () => {
            if (previousValue === 0) {
                this.ttsProperties[TtsSettings.VOLUME] = 1;
            }
            else {
                this.ttsProperties[TtsSettings.VOLUME] = 0;
                this.stop();
            }
        };
        if (previousValue === 0) {
            toggle();
        }
        else {
            // Let the caller make any last minute announcements in the current call
            // stack.
            setTimeout(toggle, 0);
        }
        return previousValue === 0;
    }
    /**
     * Method that updates the punctuation echo level, and also persists setting
     * to settings prefs.
     * @param punctuationEcho The index of the desired punctuation echo
     * level in PunctuationEchoes.
     */
    updatePunctuationEcho(punctuationEcho) {
        this.currentPunctuationEcho_ = punctuationEcho;
        SettingsManager.set(TtsSettings.PUNCTUATION_ECHO, punctuationEcho);
    }
    /**
     * Method that cycles among the available punctuation echo levels.
     * @return The resulting punctuation level message id.
     */
    cyclePunctuationEcho() {
        this.updatePunctuationEcho((this.currentPunctuationEcho_ + 1) % PunctuationEchoes.length);
        return PunctuationEchoes[this.currentPunctuationEcho_].msg;
    }
    /**
     * Converts a number into space-separated digits.
     * For numbers containing 4 or fewer digits, we return the original number.
     * This ensures that numbers like 123,456 or 2011 are not "digitized" while
     * 123456 is.
     * @param text The text to process.
     * @return A string with all numbers converted.
     * @private
     */
    getNumberAsDigits_(text) {
        return text.replace(/[0-9０-９]+/g, function (num) {
            return num.split('').join(' ');
        });
    }
    /**
     * Constructs a function for string.replace that handles description of a
     *  punctuation character.
     * @param clear Whether we want to use whitespace in place of
     *     match.
     * @return The replacement function.
     */
    createPunctuationReplace_(clear) {
        return match => {
            const retain = this.retainPunctuation_.indexOf(match) !== -1 ? match : ' ';
            return clear ? retain :
                ' ' + Msgs.getMsgWithCount(CharacterDictionary[match], 1) +
                    retain + ' ';
        };
    }
    /**
     * Queues phonetic disambiguation for characters if disambiguation is found.
     * @param text The text for which we want to get phonetic data.
     * @param properties Speech properties to use for this utterance.
     * @private
     */
    pronouncePhonetically_(text, properties) {
        // Math should never be pronounced phonetically.
        if (properties.math) {
            return;
        }
        // Only pronounce phonetic hints when explicitly requested.
        if (!properties.phoneticCharacters) {
            return;
        }
        // Remove this property so we don't trap ourselves in a loop.
        delete properties.phoneticCharacters;
        // If undefined language, use the UI language of the browser as a best
        // guess.
        if (!properties.lang) {
            properties.lang = chrome.i18n.getUILanguage();
        }
        const phoneticText = text.length === 1 ?
            PhoneticData.forCharacter(text, properties.lang) :
            PhoneticData.forText(text, properties.lang);
        if (phoneticText) {
            properties.delay = true;
            this.speak(phoneticText, QueueMode.QUEUE, properties);
        }
    }
    /**
     * Clears the last timeout set via setTimeout.
     */
    clearTimeout_() {
        if (this.timeoutId_ !== undefined) {
            clearTimeout(this.timeoutId_);
            this.timeoutId_ = undefined;
        }
    }
    /**
     * Update the current voice used to speak based upon values in storage. If
     * that does not succeed, fallback to use system locale when picking a
     * voice.
     * @param voiceName Voice name to set.
     * @param opt_callback Called when the voice is determined.
     */
    updateVoice(voiceName, opt_callback) {
        chrome.tts.getVoices((voices) => {
            const systemVoice = { voiceName: constants.SYSTEM_VOICE };
            voices.unshift(systemVoice);
            const newVoice = voices.find(v => {
                return v.voiceName === voiceName;
            }) ||
                systemVoice;
            if (newVoice && newVoice.voiceName) {
                this.currentVoice = newVoice.voiceName;
                this.startSpeakingNextItemInQueue_();
            }
            if (opt_callback) {
                opt_callback(this.currentVoice);
            }
        });
    }
    updateFromPrefs_(announce, prefs) {
        prefs.forEach(pref => {
            let msg;
            let propertyName;
            switch (pref.key) {
                case 'settings.tts.speech_rate':
                    propertyName = TtsAudioProperty.RATE;
                    msg = 'announce_rate';
                    this.setHintDelayMS(/** @type {number} */ (pref.value));
                    break;
                case 'settings.tts.speech_pitch':
                    propertyName = TtsAudioProperty.PITCH;
                    msg = 'announce_pitch';
                    break;
                case 'settings.tts.speech_volume':
                    propertyName = TtsAudioProperty.VOLUME;
                    msg = 'announce_volume';
                    break;
                default:
                    return;
            }
            this.ttsProperties[propertyName] = pref.value;
            if (!announce) {
                return;
            }
            const valueAsPercent = Math.round((this.propertyToPercentage(propertyName) || 0) * 100);
            const announcement = Msgs.getMsg(msg, [valueAsPercent.toString()]);
            ChromeVox.tts.speak(announcement, QueueMode.FLUSH, Personality.ANNOTATION);
        });
    }
    /**
     * Sets |hint_delay_ms_| given the speech rate.
     * We want an inverse relationship between the speech rate and the hint
     * delay; the faster the speech rate, the shorter the delay should be.
     * Default speech rate (value of 1) should map to a delay of 1000 MS.
     */
    setHintDelayMS(rate) {
        PrimaryTts.hint_delay_ms_ = 1000 / rate;
    }
}
TestImportManager.exportForTesting(PrimaryTts);

// Copyright 2014 The Chromium Authors
// Use of this source code is governed by a BSD-style license that can be
// found in the LICENSE file.
/**
 * @fileoverview Sends Text-To-Speech commands to Chrome's native TTS
 * extension API.
 */
const Action$5 = BridgeConstants.TtsBackground.Action;
const TARGET$4 = BridgeConstants.TtsBackground.TARGET;
/** This class broadly handles TTS within the background context. */
class TtsBackground {
    static instance;
    compositeTts_;
    consoleTts_;
    primaryTts_;
    constructor() {
        this.consoleTts_ = new ConsoleTts();
        this.primaryTts_ = new PrimaryTts();
        this.compositeTts_ =
            new CompositeTts().add(this.primaryTts_).add(this.consoleTts_);
    }
    static init() {
        TtsBackground.instance = new TtsBackground();
        ChromeVox.tts = TtsBackground.composite;
        BridgeHelper.registerHandler(TARGET$4, Action$5.UPDATE_PUNCTUATION_ECHO, (echo) => TtsBackground.primary.updatePunctuationEcho(echo));
        BridgeHelper.registerHandler(TARGET$4, Action$5.GET_CURRENT_VOICE, () => TtsBackground.primary.currentVoice);
    }
    static get composite() {
        if (!TtsBackground.instance) {
            throw new Error('Cannot access composite TTS before TtsBackground has been ' +
                'initialized.');
        }
        return TtsBackground.instance.compositeTts_;
    }
    static get console() {
        if (!TtsBackground.instance) {
            throw new Error('Cannot access console TTS before TtsBackground has been ' +
                'initialized.');
        }
        return TtsBackground.instance.consoleTts_;
    }
    static get primary() {
        if (!TtsBackground.instance) {
            throw new Error('Cannot access primary TTS before TtsBackground has been ' +
                'initialized.');
        }
        return TtsBackground.instance.primaryTts_;
    }
    static resetTextToSpeechSettings() {
        const rate = ChromeVox.tts.getDefaultProperty('rate');
        const pitch = ChromeVox.tts.getDefaultProperty('pitch');
        const volume = ChromeVox.tts.getDefaultProperty('volume');
        ChromeVox.tts.setProperty(TtsAudioProperty.RATE, rate ? rate : 1);
        ChromeVox.tts.setProperty(TtsAudioProperty.PITCH, pitch ? pitch : 1);
        ChromeVox.tts.setProperty(TtsAudioProperty.VOLUME, volume ? volume : 1);
        SettingsManager.set('voiceName', constants.SYSTEM_VOICE);
        TtsBackground.primary.updateVoice('', () => {
            // Ensure this announcement doesn't get cut off by speech triggered by
            // updateFromPrefs_().
            const speechProperties = { ...Personality.ANNOTATION };
            speechProperties.doNotInterrupt = true;
            ChromeVox.tts.speak(Msgs.getMsg('announce_tts_default_settings'), QueueMode.FLUSH, new TtsSpeechProperties(speechProperties));
        });
    }
    /** Toggles speech on or off and announces the change. */
    static toggleSpeechWithAnnouncement() {
        const state = ChromeVox.tts.toggleSpeechOnOrOff();
        new Output().format(state ? '@speech_on' : '@speech_off').go();
    }
}
TestImportManager.exportForTesting(TtsBackground);

// Copyright 2014 The Chromium Authors
// Use of this source code is governed by a BSD-style license that can be
// found in the LICENSE file.
/**
 * @fileoverview Common page for reading and writing preferences from
 * the background context (background page or options page).
 */
const Action$4 = BridgeConstants.ChromeVoxPrefs.Action;
const TARGET$3 = BridgeConstants.ChromeVoxPrefs.TARGET;
/**
 * Handles access and setting of preferences, regardless of whether they live in
 * system Settings or the ChromeVox Options page.
 */
class ChromeVoxPrefs {
    static instance;
    static darkScreen = false;
    /**
     * This indicates whether or not the sticky mode pref is toggled on.
     * Use ChromeVoxPrefs.isStickyModeOn() to test if sticky mode is enabled
     * either through the pref or due to being temporarily toggled on.
     */
    static isStickyPrefOn = false;
    /**
     * If set to true or false, this value overrides ChromeVoxPrefs.isStickyPrefOn
     * temporarily - in order to temporarily enable sticky mode while doing
     * 'read from here' or to temporarily disable it while using a widget.
     */
    static stickyOverride = null;
    /**
     * Whether the screen is darkened.
     *
     * Starts each session as false, since the display will be on whenever
     * ChromeVox starts.
     */
    static darkScreen_ = false;
    constructor() {
        // Default per session sticky to off.
        LocalStorage.set('sticky', false);
    }
    /**
     * Merge the default values of all known prefs with what's found in
     * LocalStorage.
     */
    static init() {
        ChromeVoxPrefs.instance = new ChromeVoxPrefs();
        ChromeVoxPrefs.isStickyPrefOn = LocalStorage.getBoolean('sticky');
        // Set the default value of any pref that isn't already in LocalStorage.
        for (const pref in DEFAULT_PREFS) {
            if (LocalStorage.get(pref) === undefined) {
                LocalStorage.set(pref, DEFAULT_PREFS[pref]);
            }
        }
        // TODO(b/314203187): Not null asserted, check that this is correct.
        BridgeHelper.registerHandler(TARGET$3, Action$4.GET_PREFS, () => ChromeVoxPrefs.instance.getPrefs());
        BridgeHelper.registerHandler(TARGET$3, Action$4.GET_STICKY_PREF, () => ChromeVoxPrefs.isStickyPrefOn);
        BridgeHelper.registerHandler(TARGET$3, Action$4.SET_LOGGING_PREFS, (key, value) => ChromeVoxPrefs.instance.setLoggingPrefs(key, value));
        BridgeHelper.registerHandler(TARGET$3, Action$4.SET_PREF, (key, value) => ChromeVoxPrefs.instance.setPref(key, value));
    }
    /**
     * Get the prefs (not including keys).
     * @return A map of all prefs except the key map from LocalStorage and
     *     SettingsManager.
     */
    getPrefs() {
        let prefs = {};
        for (const pref in DEFAULT_PREFS) {
            prefs[pref] = LocalStorage.get(pref);
        }
        for (const pref of SettingsManager.CHROMEVOX_PREFS) {
            prefs[pref] = SettingsManager.get(pref);
        }
        prefs = { ...prefs, ...SettingsManager.getEventStreamFilters() };
        return prefs;
    }
    /** Set the value of a pref. */
    setPref(key, value) {
        if (SettingsManager.EVENT_STREAM_FILTERS.includes(key)) {
            SettingsManager.setEventStreamFilter(key, Boolean(value));
            return;
        }
        if (SettingsManager.CHROMEVOX_PREFS.includes(key)) {
            SettingsManager.set(key, value);
            return;
        }
        if (LocalStorage.get(key) !== value) {
            LocalStorage.set(key, value);
        }
    }
    /** Set the value of a pref of logging options. */
    setLoggingPrefs(key, value) {
        SettingsManager.set(key, value);
        if (key === 'enableSpeechLogging') {
            TtsBackground.console.setEnabled(value);
        }
        else if (key === 'enableEventStreamLogging') {
            EventStreamLogger.instance.updateAllFilters(value);
        }
        this.enableOrDisableLogUrlWatcher();
    }
    /**
     * Returns whether sticky mode is on, taking both the global sticky mode
     * pref and the temporary sticky mode override into account.
     */
    static isStickyModeOn() {
        if (ChromeVoxPrefs.stickyOverride !== null) {
            return ChromeVoxPrefs.stickyOverride;
        }
        else {
            return ChromeVoxPrefs.isStickyPrefOn;
        }
    }
    /**
     * Sets the value of the sticky mode pref, as well as updating the listeners
     * and announcing.
     */
    setAndAnnounceStickyPref(value) {
        if (!MathHandler.instance?.isCapturing()) {
            // Do not overwrite the keyboard listener if the MathHandler is using it
            // right now. This ensures that turning off sticky mode while on a math
            // node will not change consumption of events by the MathHandler. Note if
            // we were turning on sticky mode then this has no effect, because
            // MathHandler has already begun capturing.
            chrome.accessibilityPrivate.setKeyboardListener(true, value);
        }
        new Output()
            .withInitialSpeechProperties(Personality.ANNOTATION)
            .withString(value ? Msgs.getMsg('sticky_mode_enabled') :
            Msgs.getMsg('sticky_mode_disabled'))
            .go();
        this.setPref('sticky', value);
        ChromeVoxPrefs.isStickyPrefOn = value;
    }
    get darkScreen() {
        return ChromeVoxPrefs.darkScreen_;
    }
    set darkScreen(newVal) {
        ChromeVoxPrefs.darkScreen_ = newVal;
    }
    enableOrDisableLogUrlWatcher() {
        for (const pref of Object.values(LoggingPrefs)) {
            if (SettingsManager.getBoolean(pref)) {
                LogUrlWatcher.create();
                return;
            }
        }
        LogUrlWatcher.destroy();
    }
}
var LoggingPrefs;
(function (LoggingPrefs) {
    LoggingPrefs["SPEECH"] = "enableSpeechLogging";
    LoggingPrefs["BRAILLE"] = "enableBrailleLogging";
    LoggingPrefs["EARCON"] = "enableEarconLogging";
    LoggingPrefs["EVENT"] = "enableEventStreamLogging";
})(LoggingPrefs || (LoggingPrefs = {}));
// Local to module.
/**
 * The default value of all preferences in LocalStorage except the key map.
 *
 * TODO(b/262786141): Move each of these to SettingsManager.
 */
const DEFAULT_PREFS = {
    'brailleCaptions': false,
    'earcons': true,
    'sticky': false,
    'typingEcho': 0,
};
TestImportManager.exportForTesting(ChromeVoxPrefs);

// Copyright 2018 The Chromium Authors
// Use of this source code is governed by a BSD-style license that can be
// found in the LICENSE file.
/**
 * @fileoverview Store ChromeVox log.
 */
const Action$3 = BridgeConstants.LogStore.Action;
const TARGET$2 = BridgeConstants.LogStore.TARGET;
/** Exported for testing. */
const LOG_LIMIT = 3000;
class LogStore {
    static instance;
    /** Ring buffer of size LOG_LIMIT. */
    logs_;
    shouldSkipOutput_ = false;
    /**
     * |this.logs_| is implemented as a ring buffer which starts
     * from |this.startIndex_| and ends at |this.startIndex_ - 1|.
     * In the initial state, this array is filled by undefined.
     */
    startIndex_ = 0;
    constructor() {
        this.logs_ = Array(LOG_LIMIT);
        this.startIndex_ = 0;
    }
    /**
     * Creates logs of type |type| in order.
     * This is not the best way to create logs fast but
     * getLogsOfType() is not called often.
     */
    getLogsOfType(logType) {
        const returnLogs = [];
        for (let i = 0; i < LOG_LIMIT; i++) {
            const index = (this.startIndex_ + i) % LOG_LIMIT;
            if (!this.logs_[index]) {
                continue;
            }
            if (this.logs_[index].logType === logType) {
                returnLogs.push(this.logs_[index]);
            }
        }
        return returnLogs;
    }
    /**
     * Create logs in order.
     * This is not the best way to create logs fast but
     * getLogs() is not called often.
     */
    getLogs() {
        const returnLogs = [];
        for (let i = 0; i < LOG_LIMIT; i++) {
            const index = (this.startIndex_ + i) % LOG_LIMIT;
            if (!this.logs_[index]) {
                continue;
            }
            returnLogs.push(this.logs_[index]);
        }
        return returnLogs;
    }
    /** @param text The text string written to the braille display. */
    writeBrailleLog(text) {
        if (SettingsManager.getBoolean(LoggingPrefs.BRAILLE)) {
            const logStr = `Braille "${text}"`;
            this.writeTextLog(logStr, LogType.BRAILLE);
        }
    }
    /**
     * Write a text log to |this.logs_|.
     * To add a message to logs, this function should be called.
     */
    writeTextLog(logContent, logType) {
        if (this.shouldSkipOutput_) {
            return;
        }
        this.writeLog(new TextLog(logContent, logType));
    }
    /**
     * Write a tree log to this.logs_.
     * To add a message to logs, this function should be called.
     */
    writeTreeLog(logContent) {
        if (this.shouldSkipOutput_) {
            return;
        }
        this.writeLog(new TreeLog(logContent));
    }
    /**
     * Write a log to this.logs_.
     * To add a message to logs, this function should be called.
     */
    writeLog(log) {
        if (this.shouldSkipOutput_) {
            return;
        }
        this.logs_[this.startIndex_] = log;
        this.startIndex_ += 1;
        if (this.startIndex_ === LOG_LIMIT) {
            this.startIndex_ = 0;
        }
    }
    /** Clear this.logs_ and set to initial states. */
    clearLog() {
        this.logs_ = Array(LOG_LIMIT);
        this.startIndex_ = 0;
    }
    set shouldSkipOutput(newValue) {
        this.shouldSkipOutput_ = newValue;
    }
    static init() {
        LogStore.instance = new LogStore();
        BridgeHelper.registerHandler(TARGET$2, Action$3.CLEAR_LOG, () => LogStore.instance.clearLog());
        BridgeHelper.registerHandler(TARGET$2, Action$3.GET_LOGS, () => LogStore.instance.getLogs().map(log => log.serialize()));
    }
}
TestImportManager.exportForTesting(LogStore, ['LOG_LIMIT', LOG_LIMIT]);

// Copyright 2018 The Chromium Authors
// Use of this source code is governed by a BSD-style license that can be
// found in the LICENSE file.
/**
 * @fileoverview Provides output logger.
 */
class OutputFormatLogger {
    str_ = '';
    storageEnabledKey_;
    logType_;
    /** @param enableKey The key to enable logging in LocalStorage. */
    constructor(enableKey, type) {
        this.storageEnabledKey_ = enableKey;
        this.logType_ = type;
    }
    get loggingDisabled() {
        return !LocalStorage.get(this.storageEnabledKey_);
    }
    /** Sends the queued logs to the LogStore. */
    commitLogs() {
        if (this.str_) {
            LogStore.instance.writeTextLog(this.str_, this.logType_);
        }
    }
    write(str) {
        if (this.loggingDisabled) {
            return;
        }
        this.str_ += str;
    }
    writeTokenWithValue(token, value) {
        if (this.loggingDisabled) {
            return;
        }
        this.writeToken(token);
        if (value) {
            this.str_ += value;
        }
        else {
            this.str_ += 'EMPTY';
        }
        this.str_ += '\n';
    }
    writeToken(token) {
        if (this.loggingDisabled) {
            return;
        }
        this.str_ += '$' + token + ': ';
    }
    writeRule(rule) {
        if (this.loggingDisabled) {
            return;
        }
        this.str_ += 'RULE: ';
        this.str_ += rule.event + ' ' + rule.role;
        if (rule.navigation) {
            this.str_ += ' ' + rule.navigation;
        }
        if (rule.output) {
            this.str_ += ' ' + rule.output;
        }
        this.str_ += '\n';
    }
    bufferClear() {
        if (this.loggingDisabled) {
            return;
        }
        this.str_ += '\nBuffer is cleared.\n';
    }
    writeError(errorMsg) {
        if (this.loggingDisabled) {
            return;
        }
        this.str_ += 'ERROR with message: ';
        this.str_ += errorMsg;
        this.str_ += '\n';
    }
}

// Copyright 2021 The Chromium Authors
// Use of this source code is governed by a BSD-style license that can be
// found in the LICENSE file.
/**
 * @fileoverview Definitions of all types related to output.
 */
const AriaCurrentState = chrome.automation.AriaCurrentState;
const Restriction$1 = chrome.automation.Restriction;
/** The ordering of contextual output. */
var OutputContextOrder;
(function (OutputContextOrder) {
    // The (ancestor) context comes before the node output.
    OutputContextOrder["FIRST"] = "first";
    // The (ancestor) context comes before the node output when moving forward,
    // after when moving backward.
    OutputContextOrder["DIRECTED"] = "directed";
    // The (ancestor) context comes after the node output.
    OutputContextOrder["LAST"] = "last";
    // Ancestor context is placed both before and after node output.
    OutputContextOrder["FIRST_AND_LAST"] = "firstAndLast";
})(OutputContextOrder || (OutputContextOrder = {}));
/** Used to annotate utterances with speech properties. */
class OutputSpeechProperties {
    properties_ = {};
    get properties() {
        return this.properties_;
    }
    toJSON() {
        // Make a copy of our properties since the caller really shouldn't be
        // modifying our local state.
        const clone = {};
        for (const key in this.properties_) {
            clone[key] = this.properties_[key];
        }
        return clone;
    }
}
/** Custom actions performed while rendering an output string. */
class OutputAction {
}
/** Action to play an earcon. */
class OutputEarconAction extends OutputAction {
    earcon;
    location;
    constructor(earcon, location) {
        super();
        this.earcon = earcon;
        this.location = location;
    }
    run() {
        ChromeVox.earcons.playEarcon(this.earcon, this.location);
    }
    toJSON() {
        return { earcon: this.earcon };
    }
}
/**
 * Annotation for text with a selection inside it.
 */
class OutputSelectionSpan {
    startIndex;
    endIndex;
    constructor(startIndex, endIndex) {
        // TODO(dtseng): Direction lost below; should preserve for braille panning.
        this.startIndex = startIndex < endIndex ? startIndex : endIndex;
        this.endIndex = endIndex > startIndex ? endIndex : startIndex;
    }
}
/**
 * Wrapper for automation nodes as annotations.  Since the
 * {@code chrome.automation.AutomationNode} constructor isn't exposed in the
 * API, this class is used to allow instanceof checks on these annotations.
 */
class OutputNodeSpan {
    node;
    offset;
    /** @param offset Offsets into the node's text. Defaults to 0. */
    constructor(node, offset) {
        this.node = node;
        this.offset = offset ? offset : 0;
    }
}
/** Possible events handled by ChromeVox internally. */
var OutputCustomEvent;
(function (OutputCustomEvent) {
    OutputCustomEvent["NAVIGATE"] = "navigate";
})(OutputCustomEvent || (OutputCustomEvent = {}));
/** Rules for mapping properties to a msg id. */
const OutputPropertyMap = {
    CHECKED: {
        'true': 'checked_true',
        'false': 'checked_false',
        'mixed': 'checked_mixed',
    },
    PRESSED: {
        'true': 'aria_pressed_true',
        'false': 'aria_pressed_false',
        'mixed': 'aria_pressed_mixed',
    },
    RESTRICTION: {
        [Restriction$1.DISABLED]: 'aria_disabled_true',
        [Restriction$1.READ_ONLY]: 'aria_readonly_true',
    },
    STATE: {
        [AriaCurrentState.TRUE]: 'aria_current_true',
        [AriaCurrentState.PAGE]: 'aria_current_page',
        [AriaCurrentState.STEP]: 'aria_current_step',
        [AriaCurrentState.LOCATION]: 'aria_current_location',
        [AriaCurrentState.DATE]: 'aria_current_date',
        [AriaCurrentState.TIME]: 'aria_current_time',
    },
};
/** Metadata about supported automation states. */
const OUTPUT_STATE_INFO = {
    collapsed: { on: { msgId: 'aria_expanded_false' } },
    default: { on: { msgId: 'default_state' } },
    expanded: { on: { msgId: 'aria_expanded_true' } },
    multiselectable: { on: { msgId: 'aria_multiselectable_true' } },
    required: { on: { msgId: 'aria_required_true' } },
    visited: { on: { msgId: 'visited_state' } },
};
/** Maps input types to message IDs. */
const INPUT_TYPE_MESSAGE_IDS = {
    'email': 'input_type_email',
    'number': 'input_type_number',
    'password': 'input_type_password',
    'search': 'input_type_search',
    'tel': 'input_type_number',
    'text': 'input_type_text',
    'url': 'input_type_url',
};
var OutputFormatType;
(function (OutputFormatType) {
    OutputFormatType["BRAILLE"] = "braille";
    OutputFormatType["SPEAK"] = "speak";
})(OutputFormatType || (OutputFormatType = {}));
var OutputNavigationType;
(function (OutputNavigationType) {
    OutputNavigationType["END_OF"] = "endOf";
    OutputNavigationType["ENTER"] = "enter";
    OutputNavigationType["LEAVE"] = "leave";
    OutputNavigationType["START_OF"] = "startOf";
})(OutputNavigationType || (OutputNavigationType = {}));
/** Delimiter to use between output values. */
const SPACE = ' ';
TestImportManager.exportForTesting(OutputAction, OutputEarconAction, OutputNodeSpan, OutputSelectionSpan, ['OutputCustomEvent', OutputCustomEvent], ['OutputContextOrder', OutputContextOrder]);

// Copyright 2023 The Chromium Authors
// Use of this source code is governed by a BSD-style license that can be
// found in the LICENSE file.
var RoleType$i = chrome.automation.RoleType;
class BrailleOutput {
    buffer = [];
    formatLog = new OutputFormatLogger('enableBrailleLogging', LogType.BRAILLE_RULE);
    equals(rhs) {
        if (this.buffer.length !== rhs.buffer.length) {
            return false;
        }
        for (let i = 0; i < this.buffer.length; i++) {
            if (this.buffer[i].toString() !== rhs.buffer[i].toString()) {
                return false;
            }
        }
        return true;
    }
    subNode(range, options) {
        const node = range.start.node;
        const rangeStart = range.start.index;
        const rangeEnd = range.end.index;
        options.annotation.push(new OutputNodeSpan(node));
        // TODO(b/314203187): Not null asserted, check that this is correct.
        const selStart = node.textSelStart;
        const selEnd = node.textSelEnd;
        if (selStart !== undefined && selEnd >= rangeStart &&
            selStart <= rangeEnd) {
            // Editable text selection.
            // |rangeStart| and |rangeEnd| are indices set by the caller and are
            // assumed to be inside of the range. In braille, we only ever expect
            // to get ranges surrounding a line as anything smaller doesn't make
            // sense.
            // |selStart| and |selEnd| reflect the editable selection. The
            // relative selStart and relative selEnd for the current line are then
            // just the difference between |selStart|, |selEnd| with |rangeStart|.
            // See editing_test.js for examples.
            options.annotation.push(new OutputSelectionSpan(selStart - rangeStart, selEnd - rangeStart));
        }
        else if (rangeStart !== 0 || rangeEnd !== range.start.getText().length) {
            // Non-editable text selection over less than the full contents
            // covered by the range. We exclude full content underlines because it
            // is distracting to read braille with all cells underlined with a
            // cursor.
            options.annotation.push(new OutputSelectionSpan(rangeStart, rangeEnd));
        }
        return options;
    }
    /** Converts the braille |spans| buffer to a single spannable. */
    static mergeSpans(spans) {
        let separator = ''; // Changes to space as appropriate.
        let prevHasInlineNode = false;
        let prevIsName = false;
        return spans.reduce((result, cur) => {
            // Ignore empty spans except when they contain a selection.
            const hasSelection = cur.getSpanInstanceOf(OutputSelectionSpan);
            if (cur.length === 0 && !hasSelection) {
                return result;
            }
            // For empty selections, we just add the space separator to account for
            // showing the braille cursor.
            if (cur.length === 0 && hasSelection) {
                result.append(cur);
                result.append(SPACE);
                separator = '';
                return result;
            }
            // Keep track of if there's an inline node associated with
            // |cur|.
            const hasInlineNode = cur.getSpansInstanceOf(OutputNodeSpan)
                .some((spannableObj) => {
                const spannable = spannableObj;
                if (!spannable.node) {
                    return false;
                }
                return spannable.node.display === 'inline' ||
                    spannable.node.role === RoleType$i.INLINE_TEXT_BOX;
            });
            const isName = cur.hasSpan('name');
            // Now, decide whether we should include separators between the previous
            // span and |cur|.
            // Never separate chunks without something already there at this point.
            // The only case where we know for certain that a separator is not
            // needed is when the previous and current values are in-lined and part
            // of the node's name. In all other cases, use the surrounding
            // whitespace to ensure we only have one separator between the node
            // text.
            if (result.length === 0 ||
                (hasInlineNode && prevHasInlineNode && isName && prevIsName)) {
                separator = '';
            }
            else if (result.toString()[result.length - 1] === SPACE ||
                cur.toString()[0] === SPACE) {
                separator = '';
            }
            else {
                separator = SPACE;
            }
            prevHasInlineNode = hasInlineNode;
            prevIsName = isName;
            result.append(separator);
            result.append(cur);
            return result;
        }, new Spannable());
    }
}

// Copyright 2022 The Chromium Authors
// Use of this source code is governed by a BSD-style license that can be
// found in the LICENSE file.
/**
 * @fileoverview Augments chrome.automation.RoleType with abstract types for
 * ChromeVox.
 */
var AbstractRole;
(function (AbstractRole) {
    AbstractRole["CONTAINER"] = "abstractContainer";
    AbstractRole["FORM_FIELD_CONTAINER"] = "abstractFormFieldContainer";
    AbstractRole["ITEM"] = "abstractItem";
    AbstractRole["LIST"] = "abstractList";
    AbstractRole["NAME_FROM_CONTENTS"] = "abstractNameFromContents";
    AbstractRole["RANGE"] = "abstractRange";
    AbstractRole["SPAN"] = "abstractSpan";
})(AbstractRole || (AbstractRole = {}));
var CustomRole;
(function (CustomRole) {
    CustomRole["DEFAULT"] = "default";
    CustomRole["NO_ROLE"] = "noRole";
})(CustomRole || (CustomRole = {}));

// Copyright 2021 The Chromium Authors
// Use of this source code is governed by a BSD-style license that can be
// found in the LICENSE file.
/**
 * @fileoverview Roel information for the Output module.
 */
var RoleType$h = chrome.automation.RoleType;
const OutputRoleInfo = {
    [RoleType$h.ABBR]: { msgId: 'tag_abbr', inherits: AbstractRole.CONTAINER },
    [RoleType$h.ALERT]: { msgId: 'role_alert' },
    [RoleType$h.ALERT_DIALOG]: { msgId: 'role_alertdialog', contextOrder: OutputContextOrder.FIRST },
    [RoleType$h.ARTICLE]: { msgId: 'role_article', inherits: AbstractRole.ITEM },
    [RoleType$h.APPLICATION]: { msgId: 'role_application', inherits: AbstractRole.CONTAINER },
    [RoleType$h.AUDIO]: { msgId: 'tag_audio', inherits: AbstractRole.FORM_FIELD_CONTAINER },
    [RoleType$h.BANNER]: { msgId: 'role_banner', inherits: AbstractRole.CONTAINER },
    [RoleType$h.BUTTON]: { msgId: 'role_button', earcon: EarconId.BUTTON },
    [RoleType$h.CHECK_BOX]: { msgId: 'role_checkbox' },
    [RoleType$h.COLUMN_HEADER]: { msgId: 'role_columnheader', inherits: RoleType$h.CELL },
    [RoleType$h.COMBO_BOX_GROUPING]: { msgId: 'role_combobox', inherits: AbstractRole.FORM_FIELD_CONTAINER },
    [RoleType$h.COMBO_BOX_MENU_BUTTON]: { msgId: 'role_combobox', earcon: EarconId.LISTBOX },
    [RoleType$h.COMBO_BOX_SELECT]: {
        msgId: 'role_button',
        earcon: EarconId.POP_UP_BUTTON,
        inherits: RoleType$h.COMBO_BOX_MENU_BUTTON,
    },
    [RoleType$h.COMMENT]: {
        msgId: 'role_comment',
        contextOrder: OutputContextOrder.FIRST_AND_LAST,
        verboseAncestry: true,
        inherits: AbstractRole.SPAN,
    },
    [RoleType$h.COMPLEMENTARY]: { msgId: 'role_complementary', inherits: AbstractRole.CONTAINER },
    [RoleType$h.CONTENT_DELETION]: {
        msgId: 'role_content_deletion',
        contextOrder: OutputContextOrder.FIRST_AND_LAST,
        verboseAncestry: true,
        inherits: AbstractRole.SPAN,
    },
    [RoleType$h.CONTENT_INSERTION]: {
        msgId: 'role_content_insertion',
        contextOrder: OutputContextOrder.FIRST_AND_LAST,
        verboseAncestry: true,
        inherits: AbstractRole.SPAN,
    },
    [RoleType$h.CONTENT_INFO]: { msgId: 'role_contentinfo', inherits: AbstractRole.CONTAINER },
    [RoleType$h.DATE]: { msgId: 'input_type_date', inherits: AbstractRole.FORM_FIELD_CONTAINER },
    [RoleType$h.DEFINITION]: { msgId: 'role_definition', inherits: AbstractRole.CONTAINER },
    [RoleType$h.DESCRIPTION_LIST]: { msgId: 'role_description_list', inherits: AbstractRole.LIST },
    [RoleType$h.DESCRIPTION_LIST_DETAIL_DEPRECATED]: { msgId: 'role_description_list_detail', inherits: AbstractRole.ITEM },
    [RoleType$h.DIALOG]: {
        msgId: 'role_dialog',
        contextOrder: OutputContextOrder.DIRECTED,
        ignoreAncestry: true,
    },
    [RoleType$h.DIRECTORY_DEPRECATED]: { msgId: 'role_directory', inherits: AbstractRole.CONTAINER },
    [RoleType$h.DOC_ABSTRACT]: { msgId: 'role_doc_abstract', inherits: AbstractRole.SPAN },
    [RoleType$h.DOC_ACKNOWLEDGMENTS]: { msgId: 'role_doc_acknowledgments', inherits: AbstractRole.SPAN },
    [RoleType$h.DOC_AFTERWORD]: { msgId: 'role_doc_afterword', inherits: AbstractRole.CONTAINER },
    [RoleType$h.DOC_APPENDIX]: { msgId: 'role_doc_appendix', inherits: AbstractRole.SPAN },
    [RoleType$h.DOC_BACK_LINK]: {
        msgId: 'role_doc_back_link',
        earcon: EarconId.LINK,
        inherits: RoleType$h.LINK,
    },
    [RoleType$h.DOC_BIBLIO_ENTRY]: {
        msgId: 'role_doc_biblio_entry',
        earcon: EarconId.LIST_ITEM,
        inherits: AbstractRole.ITEM,
    },
    [RoleType$h.DOC_BIBLIOGRAPHY]: { msgId: 'role_doc_bibliography', inherits: AbstractRole.SPAN },
    [RoleType$h.DOC_BIBLIO_REF]: {
        msgId: 'role_doc_biblio_ref',
        earcon: EarconId.LINK,
        inherits: RoleType$h.LINK,
    },
    [RoleType$h.DOC_CHAPTER]: { msgId: 'role_doc_chapter', inherits: AbstractRole.SPAN },
    [RoleType$h.DOC_COLOPHON]: { msgId: 'role_doc_colophon', inherits: AbstractRole.SPAN },
    [RoleType$h.DOC_CONCLUSION]: { msgId: 'role_doc_conclusion', inherits: AbstractRole.SPAN },
    [RoleType$h.DOC_COVER]: { msgId: 'role_doc_cover', inherits: RoleType$h.IMAGE },
    [RoleType$h.DOC_CREDIT]: { msgId: 'role_doc_credit', inherits: AbstractRole.SPAN },
    [RoleType$h.DOC_CREDITS]: { msgId: 'role_doc_credits', inherits: AbstractRole.SPAN },
    [RoleType$h.DOC_DEDICATION]: { msgId: 'role_doc_dedication', inherits: AbstractRole.SPAN },
    [RoleType$h.DOC_ENDNOTE]: {
        msgId: 'role_doc_endnote',
        earcon: EarconId.LIST_ITEM,
        inherits: AbstractRole.ITEM,
    },
    [RoleType$h.DOC_ENDNOTES]: {
        msgId: 'role_doc_endnotes',
        earcon: EarconId.LISTBOX,
        inherits: RoleType$h.LIST,
    },
    [RoleType$h.DOC_EPIGRAPH]: { msgId: 'role_doc_epigraph', inherits: AbstractRole.SPAN },
    [RoleType$h.DOC_EPILOGUE]: { msgId: 'role_doc_epilogue', inherits: AbstractRole.SPAN },
    [RoleType$h.DOC_ERRATA]: { msgId: 'role_doc_errata', inherits: AbstractRole.SPAN },
    [RoleType$h.DOC_EXAMPLE]: { msgId: 'role_doc_example', inherits: AbstractRole.SPAN },
    [RoleType$h.DOC_FOOTNOTE]: {
        msgId: 'role_doc_footnote',
        earcon: EarconId.LIST_ITEM,
        inherits: AbstractRole.ITEM,
    },
    [RoleType$h.DOC_FOREWORD]: { msgId: 'role_doc_foreword', inherits: AbstractRole.SPAN },
    [RoleType$h.DOC_GLOSSARY]: { msgId: 'role_doc_glossary', inherits: AbstractRole.SPAN },
    [RoleType$h.DOC_GLOSS_REF]: {
        msgId: 'role_doc_gloss_ref',
        earcon: EarconId.LINK,
        inherits: RoleType$h.LINK,
    },
    [RoleType$h.DOC_INDEX]: { msgId: 'role_doc_index', inherits: AbstractRole.SPAN },
    [RoleType$h.DOC_INTRODUCTION]: { msgId: 'role_doc_introduction', inherits: AbstractRole.SPAN },
    [RoleType$h.DOC_NOTE_REF]: {
        msgId: 'role_doc_note_ref',
        earcon: EarconId.LINK,
        inherits: RoleType$h.LINK,
    },
    [RoleType$h.DOC_NOTICE]: { msgId: 'role_doc_notice', inherits: AbstractRole.SPAN },
    [RoleType$h.DOC_PAGE_BREAK]: { msgId: 'role_doc_page_break', inherits: AbstractRole.SPAN },
    [RoleType$h.DOC_PAGE_FOOTER]: { msgId: 'role_doc_page_footer', inherits: AbstractRole.SPAN },
    [RoleType$h.DOC_PAGE_HEADER]: { msgId: 'role_doc_page_header', inherits: AbstractRole.SPAN },
    [RoleType$h.DOC_PAGE_LIST]: { msgId: 'role_doc_page_list', inherits: AbstractRole.SPAN },
    [RoleType$h.DOC_PART]: { msgId: 'role_doc_part', inherits: AbstractRole.SPAN },
    [RoleType$h.DOC_PREFACE]: { msgId: 'role_doc_preface', inherits: AbstractRole.SPAN },
    [RoleType$h.DOC_PROLOGUE]: { msgId: 'role_doc_prologue', inherits: AbstractRole.SPAN },
    [RoleType$h.DOC_PULLQUOTE]: { msgId: 'role_doc_pullquote', inherits: AbstractRole.SPAN },
    [RoleType$h.DOC_QNA]: { msgId: 'role_doc_qna', inherits: AbstractRole.SPAN },
    [RoleType$h.DOC_SUBTITLE]: { msgId: 'role_doc_subtitle', inherits: RoleType$h.HEADING },
    [RoleType$h.DOC_TIP]: { msgId: 'role_doc_tip', inherits: AbstractRole.SPAN },
    [RoleType$h.DOC_TOC]: { msgId: 'role_doc_toc', inherits: AbstractRole.SPAN },
    [RoleType$h.DOCUMENT]: { msgId: 'role_document', inherits: AbstractRole.CONTAINER },
    [RoleType$h.FORM]: { msgId: 'role_form', inherits: AbstractRole.CONTAINER },
    [RoleType$h.GRAPHICS_DOCUMENT]: { msgId: 'role_graphics_document', inherits: AbstractRole.CONTAINER },
    [RoleType$h.GRAPHICS_OBJECT]: { msgId: 'role_graphics_object', inherits: AbstractRole.CONTAINER },
    [RoleType$h.GRAPHICS_SYMBOL]: { msgId: 'role_graphics_symbol', inherits: RoleType$h.IMAGE },
    [RoleType$h.GRID]: { msgId: 'role_grid', inherits: RoleType$h.TABLE },
    [RoleType$h.GROUP]: { msgId: 'role_group', inherits: AbstractRole.CONTAINER },
    [RoleType$h.HEADING]: {
        msgId: 'role_heading',
    },
    [RoleType$h.IMAGE]: {
        msgId: 'role_img',
    },
    [RoleType$h.IME_CANDIDATE]: { msgId: 'ime_candidate', ignoreAncestry: true },
    [RoleType$h.INPUT_TIME]: { msgId: 'input_type_time', inherits: AbstractRole.FORM_FIELD_CONTAINER },
    [RoleType$h.LINK]: { msgId: 'role_link', earcon: EarconId.LINK },
    [RoleType$h.LIST]: { msgId: 'role_list', inherits: AbstractRole.LIST },
    [RoleType$h.LIST_BOX]: { msgId: 'role_listbox', earcon: EarconId.LISTBOX },
    [RoleType$h.LIST_BOX_OPTION]: { msgId: 'role_listitem', earcon: EarconId.LIST_ITEM },
    [RoleType$h.LIST_GRID]: { msgId: 'role_list_grid', inherits: RoleType$h.TABLE },
    [RoleType$h.LIST_ITEM]: {
        msgId: 'role_listitem',
        earcon: EarconId.LIST_ITEM,
        inherits: AbstractRole.ITEM,
    },
    [RoleType$h.LOG]: { msgId: 'role_log', inherits: AbstractRole.NAME_FROM_CONTENTS },
    [RoleType$h.MAIN]: { msgId: 'role_main', inherits: AbstractRole.CONTAINER },
    [RoleType$h.MARK]: {
        msgId: 'role_mark',
        contextOrder: OutputContextOrder.FIRST_AND_LAST,
        verboseAncestry: true,
        inherits: AbstractRole.CONTAINER,
    },
    [RoleType$h.MARQUEE]: { msgId: 'role_marquee', inherits: AbstractRole.NAME_FROM_CONTENTS },
    [RoleType$h.MATH]: { msgId: 'role_math', inherits: AbstractRole.CONTAINER },
    [RoleType$h.MENU]: {
        msgId: 'role_menu',
        contextOrder: OutputContextOrder.FIRST,
        ignoreAncestry: true,
    },
    [RoleType$h.MENU_BAR]: {
        msgId: 'role_menubar',
    },
    [RoleType$h.MENU_ITEM]: { msgId: 'role_menuitem' },
    [RoleType$h.MENU_ITEM_CHECK_BOX]: { msgId: 'role_menuitemcheckbox' },
    [RoleType$h.MENU_ITEM_RADIO]: { msgId: 'role_menuitemradio' },
    [RoleType$h.MENU_LIST_OPTION]: { msgId: 'role_listitem' },
    [RoleType$h.MENU_LIST_POPUP]: { msgId: 'role_listbox' },
    [RoleType$h.METER]: { msgId: 'role_meter', inherits: AbstractRole.RANGE },
    [RoleType$h.NAVIGATION]: { msgId: 'role_navigation', inherits: AbstractRole.CONTAINER },
    [RoleType$h.NOTE]: { msgId: 'role_note', inherits: AbstractRole.CONTAINER },
    [RoleType$h.PROGRESS_INDICATOR]: { msgId: 'role_progress_indicator', inherits: AbstractRole.RANGE },
    [RoleType$h.POP_UP_BUTTON]: {
        msgId: 'role_button',
        earcon: EarconId.POP_UP_BUTTON,
        inherits: RoleType$h.COMBO_BOX_MENU_BUTTON,
    },
    [RoleType$h.RADIO_BUTTON]: { msgId: 'role_radio' },
    [RoleType$h.RADIO_GROUP]: { msgId: 'role_radiogroup', inherits: AbstractRole.FORM_FIELD_CONTAINER },
    [RoleType$h.REGION]: { msgId: 'role_region', inherits: AbstractRole.CONTAINER },
    [RoleType$h.ROW]: { msgId: 'role_row' },
    [RoleType$h.ROW_HEADER]: { msgId: 'role_rowheader', inherits: RoleType$h.CELL },
    [RoleType$h.SCROLL_BAR]: { msgId: 'role_scrollbar', inherits: AbstractRole.RANGE },
    [RoleType$h.SECTION]: { msgId: 'role_region', inherits: AbstractRole.CONTAINER },
    [RoleType$h.SEARCH]: { msgId: 'role_search', inherits: AbstractRole.CONTAINER },
    [RoleType$h.SEARCH_BOX]: { msgId: 'role_search', earcon: EarconId.EDITABLE_TEXT },
    [RoleType$h.SLIDER]: {
        msgId: 'role_slider',
        inherits: AbstractRole.RANGE,
        earcon: EarconId.SLIDER,
    },
    [RoleType$h.SPIN_BUTTON]: {
        msgId: 'role_spinbutton',
        inherits: AbstractRole.RANGE,
        earcon: EarconId.LISTBOX,
    },
    [RoleType$h.SPLITTER]: { msgId: 'role_separator', inherits: AbstractRole.SPAN },
    [RoleType$h.STATUS]: { msgId: 'role_status', inherits: AbstractRole.NAME_FROM_CONTENTS },
    [RoleType$h.SUBSCRIPT]: { msgId: 'role_subscript', inherits: AbstractRole.SPAN },
    [RoleType$h.SUGGESTION]: {
        msgId: 'role_suggestion',
        contextOrder: OutputContextOrder.FIRST_AND_LAST,
        verboseAncestry: true,
        inherits: AbstractRole.SPAN,
    },
    [RoleType$h.SUPERSCRIPT]: { msgId: 'role_superscript', inherits: AbstractRole.SPAN },
    [RoleType$h.TAB]: { msgId: 'role_tab', inherits: AbstractRole.CONTAINER },
    [RoleType$h.TAB_LIST]: { msgId: 'role_tablist', inherits: AbstractRole.FORM_FIELD_CONTAINER },
    [RoleType$h.TAB_PANEL]: { msgId: 'role_tabpanel', inherits: AbstractRole.CONTAINER },
    [RoleType$h.TEXT_FIELD]: { msgId: 'input_type_text', earcon: EarconId.EDITABLE_TEXT },
    [RoleType$h.TEXT_FIELD_WITH_COMBO_BOX]: { msgId: 'role_combobox', earcon: EarconId.EDITABLE_TEXT },
    [RoleType$h.TIME]: { msgId: 'tag_time', inherits: AbstractRole.FORM_FIELD_CONTAINER },
    [RoleType$h.TIMER]: { msgId: 'role_timer', inherits: AbstractRole.NAME_FROM_CONTENTS },
    [RoleType$h.TOGGLE_BUTTON]: { msgId: 'role_toggle_button', inherits: RoleType$h.CHECK_BOX },
    [RoleType$h.TOOLBAR]: { msgId: 'role_toolbar', ignoreAncestry: true },
    [RoleType$h.TREE]: { msgId: 'role_tree' },
    [RoleType$h.TREE_ITEM]: { msgId: 'role_treeitem' },
    [RoleType$h.VIDEO]: { msgId: 'tag_video', inherits: AbstractRole.FORM_FIELD_CONTAINER },
    [RoleType$h.WINDOW]: { ignoreAncestry: true },
};
TestImportManager.exportForTesting(['OutputRoleInfo', OutputRoleInfo]);

// Copyright 2021 The Chromium Authors
// Use of this source code is governed by a BSD-style license that can be
// found in the LICENSE file.
/**
 * @fileoverview Provides a class which computes various types of ancestor
 * chains given the current node.
 */
const Dir$a = constants.Dir;
const RoleType$g = chrome.automation.RoleType;
/**
 * This class computes four types of ancestry chains (see below for specific
 * definition). By default, enter and leave are computed, with an option to
 * compute start and end ancestors on construction. These ancestors are cached,
 * so are generally valid for the current call stack, wherein ancestry data is
 * stable.
 */
class OutputAncestryInfo {
    /**
     * Enter ancestors are all ancestors of |node| which are not ancestors of
     * |prevNode|. This list of nodes is useful when trying to understand what
     * nodes to consider for a newly positioned node e.g. for speaking focus. One
     * can think of this as a leave ancestry list if we swap |ndoe| and
     * |prevNode|.
     */
    enterAncestors;
    /**
     * Leave ancestors are all ancestors of |prevNode| which are not ancestors of
     * |node|. This list of nodes is useful when trying to understand what nodes
     * to consider for a newly unpositioned node e.g. for speaking blur. One can
     * think of this as an enter ancestry list if we swap |ndoe| and |prevNode|.
     */
    leaveAncestors;
    /**
     * Start ancestors are all ancestors of |node| when which are not ancestors of
     * the node immediately before |node| in linear object navigation. This list
     * of nodes is useful when trying to understand what nodes to consider for
     * describing |node|'s ancestry context limited to ancestors of most
     * relevance.
     */
    startAncestors = [];
    /**
     * End ancestors are all ancestors of |node| which are not ancestors of the
     * node immediately after |node| in linear object navigation. This list of
     * nodes is useful when trying to understand what nodes to consider for
     * describing |node|'s ancestry context limited to ancestors of most
     * relevance.
     */
    endAncestors = [];
    /**
     * @param node The primary node to consider for ancestry computation.
     * @param prevNode The previous node (in user-initiated navigation).
     * @param suppressStartAndEndAncestors Whether to compute |node|'s start and
     *     end ancestors (see below for definitions).
     */
    constructor(node, prevNode, suppressStartAndEndAncestors = true) {
        this.enterAncestors = OutputAncestryInfo.byContextFirst_(AutomationUtil.getUniqueAncestors(prevNode, node));
        this.leaveAncestors =
            OutputAncestryInfo
                .byContextFirst_(AutomationUtil.getUniqueAncestors(node, prevNode))
                .reverse();
        if (suppressStartAndEndAncestors) {
            return;
        }
        let afterEndNode = AutomationUtil.findNextNode(node, Dir$a.FORWARD, AutomationPredicate.leafOrStaticText, { root: r => r === node.root, skipInitialSubtree: true });
        if (!afterEndNode) {
            afterEndNode = AutomationUtil.getTopLevelRoot(node) || node.root;
        }
        if (afterEndNode) {
            this.endAncestors = OutputAncestryInfo.byContextFirst_(AutomationUtil.getUniqueAncestors(afterEndNode, node), true /* discardRootOrEditableRoot */);
        }
        let beforeStartNode = AutomationUtil.findNextNode(node, Dir$a.BACKWARD, AutomationPredicate.leafOrStaticText, { root: r => r === node.root, skipInitialAncestry: true });
        if (!beforeStartNode) {
            beforeStartNode = AutomationUtil.getTopLevelRoot(node) || node.root;
        }
        if (beforeStartNode) {
            this.startAncestors =
                OutputAncestryInfo
                    .byContextFirst_(AutomationUtil.getUniqueAncestors(beforeStartNode, node), true /* discardRootOrEditableRoot */)
                    .reverse();
        }
    }
    /**
     * @param discardRootOrEditableRoot Whether to stop ancestry computation at a
     *     root or editable root.
     */
    static byContextFirst_(ancestors, discardRootOrEditableRoot = false) {
        let contextFirst = [];
        let rest = [];
        for (let i = 0; i < ancestors.length - 1; i++) {
            const node = ancestors[i];
            // Discard ancestors of deepest window or if requested.
            if (node.role === RoleType$g.WINDOW ||
                (discardRootOrEditableRoot &&
                    AutomationPredicate.rootOrEditableRoot(node))) {
                contextFirst = [];
                rest = [];
            }
            // TODO(b/314203187): Not null asserted, check that this is correct.
            if ((OutputRoleInfo[node.role] || {}).contextOrder ===
                OutputContextOrder.FIRST) {
                contextFirst.push(node);
            }
            else {
                rest.push(node);
            }
        }
        return rest.concat(contextFirst.reverse());
    }
}

// Copyright 2021 The Chromium Authors
// Use of this source code is governed by a BSD-style license that can be
// found in the LICENSE file.
/**
 * @fileoverview Parses the output format.
 */
class OutputFormatTree {
    value = '';
    firstChild;
    nextSibling;
    parent;
    constructor() { }
    static parseFormat(format) {
        let formatTrees = [];
        // Hacky way to support args.
        if (typeof (format) === 'string') {
            format = format.replace(/([,:])\s+/gm, '$1');
            const words = format.split(' ');
            // Ignore empty strings.
            words.filter(word => Boolean(word));
            formatTrees = words.map(word => OutputFormatTree.buildFromString_(word));
        }
        else if (format) {
            formatTrees = [format];
        }
        return formatTrees;
    }
    /** Parses the token containing a custom function and returns a tree. */
    static buildFromString_(inputStr) {
        const root = new OutputFormatTree();
        let currentNode = root;
        let index = 0;
        let braceNesting = 0;
        while (index < inputStr.length) {
            if (inputStr[index] === '(') {
                currentNode.firstChild = new OutputFormatTree();
                currentNode.firstChild.parent = currentNode;
                currentNode = currentNode.firstChild;
            }
            else if (inputStr[index] === ')') {
                // TODO(b/314203187): Not null asserted, check that this is correct.
                currentNode = currentNode.parent;
            }
            else if (inputStr[index] === '{') {
                braceNesting++;
                currentNode.value += inputStr[index];
            }
            else if (inputStr[index] === '}') {
                braceNesting--;
                currentNode.value += inputStr[index];
            }
            else if (inputStr[index] === ',' && braceNesting === 0) {
                currentNode.nextSibling = new OutputFormatTree();
                currentNode.nextSibling.parent = currentNode.parent;
                currentNode = currentNode.nextSibling;
            }
            else if (inputStr[index] === ' ' || inputStr[index] === '\n') ;
            else {
                currentNode.value += inputStr[index];
            }
            index++;
        }
        if (currentNode !== root) {
            throw 'Unbalanced parenthesis: ' + inputStr;
        }
        return root;
    }
}

// Copyright 2021 The Chromium Authors
// Use of this source code is governed by a BSD-style license that can be
// found in the LICENSE file.
/**
 * @fileoverview Provides a push parser for Output format rules.
 */
class OutputFormatParser {
    observer_;
    constructor(observer) {
        this.observer_ = observer;
    }
    /** Starts parsing the given output format. */
    parse(format) {
        const formatTrees = OutputFormatTree.parseFormat(format);
        formatTrees.forEach((tree) => {
            // Obtain the operator token.
            let token = tree.value;
            // Set suffix options.
            const options = { annotation: [], isUnique: false };
            options.isUnique = token[token.length - 1] === '=';
            if (options.isUnique) {
                token = token.substring(0, token.length - 1);
            }
            // Process token based on prefix.
            const prefix = token[0];
            token = token.slice(1);
            if (this.observer_.onTokenStart(token)) {
                return;
            }
            // All possible tokens based on prefix.
            let skipToNextToken = false;
            // TODO(b/314203187): Not null asserted, check that this is correct.
            if (prefix === '$') {
                skipToNextToken =
                    this.observer_.onNodeAttributeOrSpecialToken(token, tree, options);
            }
            else if (prefix === '@') {
                skipToNextToken = this.observer_.onMessageToken(token, tree, options);
            }
            else if (prefix === '!') {
                skipToNextToken =
                    this.observer_.onSpeechPropertyToken(token, tree, options);
            }
            if (skipToNextToken) {
                return;
            }
            this.observer_.onTokenEnd();
        });
    }
}

// Copyright 2022 The Chromium Authors
// Use of this source code is governed by a BSD-style license that can be
// found in the LICENSE file.
/**
 * @fileoverview Class that formats the parsed output tree.
 */
const Dir$9 = constants.Dir;
const NameFromType = chrome.automation.NameFromType;
const RoleType$f = chrome.automation.RoleType;
var StateType$f = chrome.automation.StateType;
// TODO(anastasi): Move formatting logic to this class.
class OutputFormatter {
    speechProps_;
    output_;
    params_;
    constructor(output, params) {
        this.speechProps_ = params.opt_speechProps;
        this.output_ = output;
        this.params_ = params;
    }
    /**
     * Format the node given the format specifier.
     * @param params All the required and optional parameters for formatting.
     */
    static format(output, params) {
        const formatter = new OutputFormatter(output, params);
        new OutputFormatParser(formatter).parse(params.outputFormat);
    }
    /** OutputFormatParserObserver implementation */
    onTokenStart() { }
    /** OutputFormatParserObserver implementation */
    onNodeAttributeOrSpecialToken(token, tree, options) {
        if (this.output_.shouldSuppress(token)) {
            return true;
        }
        if (token === 'cellIndexText') {
            this.formatCellIndexText_(this.params_, token, options);
        }
        else if (token === 'checked') {
            this.formatChecked_(this.params_, token);
        }
        else if (token === 'descendants') {
            this.formatDescendants_(this.params_, token);
        }
        else if (token === 'description') {
            this.formatDescription_(this.params_, token, options);
        }
        else if (token === 'find') {
            this.formatFind_(this.params_, token, tree);
        }
        else if (token === 'inputType') {
            this.formatInputType_(this.params_, token, options);
        }
        else if (token === 'indexInParent') {
            this.formatIndexInParent_(this.params_, token, tree, options);
        }
        else if (token === 'joinedDescendants') {
            this.formatJoinedDescendants_(this.params_, token, options);
        }
        else if (token === 'listNestedLevel') {
            this.formatListNestedLevel_(this.params_);
        }
        else if (token === 'name') {
            this.formatName_(this.params_, token, options);
        }
        else if (token === 'nameFromNode') {
            this.formatNameFromNode_(this.params_, token, options);
        }
        else if (token === 'nameOrDescendants') {
            // This token is similar to nameOrTextContent except it gathers
            // rich output for descendants. It also lets name from contents
            // override the descendants text if |node| has only static text
            // children.
            this.formatNameOrDescendants_(this.params_, token, options);
        }
        else if (token === 'nameOrTextContent' || token === 'textContent') {
            this.formatTextContent_(this.params_, token, options);
        }
        else if (token === 'node') {
            this.formatNode_(this.params_, token, tree, options);
        }
        else if (token === 'phoneticReading') {
            this.formatPhoneticReading_(this.params_);
        }
        else if (token === 'precedingBullet') {
            this.formatPrecedingBullet_(this.params_);
        }
        else if (token === 'pressed') {
            this.formatPressed_(this.params_, token);
        }
        else if (token === 'restriction') {
            this.formatRestriction_(this.params_, token);
        }
        else if (token === 'role') {
            if (!SettingsManager.get('useVerboseMode')) {
                return true;
            }
            if (this.output_.useAuralStyle) {
                this.speechProps_ = new OutputSpeechProperties();
                this.speechProps_.properties['relativePitch'] = -0.3;
            }
            this.formatRole_(this.params_, token, options);
        }
        else if (token === 'state') {
            this.formatState_(this.params_, token);
        }
        else if (token === 'tableCellRowIndex' || token === 'tableCellColumnIndex') {
            this.formatTableCellIndex_(this.params_, token, options);
        }
        else if (token === 'urlFilename') {
            this.formatUrlFilename_(this.params_, token, options);
        }
        else if (token === 'value') {
            this.formatValue_(this.params_, token, options);
            // TODO(b/314203187): Not null asserted, check that this is correct.
        }
        else if (this.params_.node[token] !== undefined) {
            this.formatAsFieldAccessor_(this.params_, token, options);
        }
        else if (OUTPUT_STATE_INFO[token]) {
            this.formatAsStateValue_(this.params_, token, options);
        }
        else if (tree.firstChild) {
            this.formatCustomFunction_(this.params_, token, tree, options);
        }
        return undefined;
    }
    /** OutputFormatParserObserver implementation */
    onMessageToken(token, tree, options) {
        this.params_.outputFormatLogger.write(' @');
        if (this.output_.useAuralStyle) {
            if (!this.speechProps_) {
                this.speechProps_ = new OutputSpeechProperties();
            }
            this.speechProps_.properties['relativePitch'] = -0.2;
        }
        this.formatMessage_(this.params_, token, tree, options);
    }
    /** OutputFormatParserObserver implementation */
    onSpeechPropertyToken(token, tree, _options) {
        this.params_.outputFormatLogger.write(' ! ' + token + '\n');
        this.speechProps_ = new OutputSpeechProperties();
        this.speechProps_.properties[token] = true;
        if (tree.firstChild) {
            if (!this.output_.useAuralStyle) {
                this.speechProps_ = undefined;
                return true;
            }
            let value = tree.firstChild.value;
            // Currently, speech params take either attributes or floats.
            let float = 0;
            if (float = parseFloat(value)) {
                value = float;
            }
            else {
                // TODO(b/314203187): Not null asserted, check that this is correct.
                value = parseFloat(this.params_.node[value]) / -10.0;
            }
            this.speechProps_.properties[token] = value;
            return true;
        }
        return undefined;
    }
    /** OutputFormatParserObserver implementation */
    onTokenEnd() {
        const buff = this.params_.outputBuffer;
        // Post processing.
        if (this.speechProps_) {
            if (buff.length > 0) {
                buff[buff.length - 1].setSpan(this.speechProps_, 0, 0);
                this.speechProps_ = null;
            }
        }
    }
    createRoles_(tree) {
        const roles = new Set();
        for (let currentNode = tree.firstChild; currentNode; currentNode = currentNode.nextSibling) {
            roles.add(currentNode.value);
        }
        return roles;
    }
    error_(formatLog, errorMsg) {
        formatLog.writeError(errorMsg);
        console.error(errorMsg);
    }
    formatAsFieldAccessor_(data, token, options) {
        const buff = data.outputBuffer;
        // TODO(b/314203187): Not null asserted, check that this is correct.
        const node = data.node;
        const formatLog = data.outputFormatLogger;
        options.annotation.push(token);
        let value = node[token];
        if (typeof value === 'number') {
            value = String(value);
        }
        this.output_.append(buff, value, options);
        formatLog.writeTokenWithValue(token, value);
    }
    formatAsStateValue_(data, token, options) {
        const buff = data.outputBuffer;
        // TODO(b/314203187): Not null asserted, check that this is correct.
        const node = data.node;
        const formatLog = data.outputFormatLogger;
        options.annotation.push('state');
        const stateInfo = OUTPUT_STATE_INFO[token];
        const resolvedInfo = node.state[token] ? stateInfo.on : stateInfo.off;
        if (!resolvedInfo) {
            return;
        }
        if (this.output_.formatAsSpeech && resolvedInfo.earcon) {
            options.annotation.push(new OutputEarconAction(resolvedInfo.earcon), node.location || undefined);
        }
        const msgId = this.output_.formatAsBraille ? resolvedInfo.msgId + '_brl' :
            resolvedInfo.msgId;
        // TODO(b/314203187): Not null asserted, check that this is correct.
        const msg = Msgs.getMsg(msgId);
        this.output_.append(buff, msg, options);
        formatLog.writeTokenWithValue(token, msg);
    }
    formatCellIndexText_(data, token, options) {
        const buff = data.outputBuffer;
        // TODO(b/314203187): Not null asserted, check that this is correct.
        const node = data.node;
        const formatLog = data.outputFormatLogger;
        // TODO(b/314203187): Not null asserted, check that this is correct.
        if (node.ariaCellColumnIndexText) {
            let value = node.ariaCellColumnIndexText;
            let row = node;
            while (row && row.role !== RoleType$f.ROW) {
                row = row.parent;
            }
            // TODO(b/314203187): Not null asserted, check that this is correct.
            if (!row || !row.ariaCellRowIndexText) {
                return;
            }
            // TODO(b/314203187): Not null asserted, check that this is correct.
            value += row.ariaCellRowIndexText;
            this.output_.append(buff, value, options);
            formatLog.writeTokenWithValue(token, value);
        }
        else {
            formatLog.write(token);
            OutputFormatter.format(this.output_, {
                node,
                outputFormat: ` @cell_summary($if($tableCellAriaRowIndex,
          $tableCellAriaRowIndex, $tableCellRowIndex),
        $if($tableCellAriaColumnIndex, $tableCellAriaColumnIndex,
          $tableCellColumnIndex))`,
                outputBuffer: buff,
                outputFormatLogger: formatLog,
            });
        }
    }
    formatChecked_(data, token) {
        const buff = data.outputBuffer;
        // TODO(b/314203187): Not null asserted, check that this is correct.
        const node = data.node;
        const formatLog = data.outputFormatLogger;
        // TODO(b/314203187): Not null asserted, check that this is correct.
        const msg = OutputPropertyMap['CHECKED'][node.checked];
        if (msg) {
            formatLog.writeToken(token);
            OutputFormatter.format(this.output_, {
                node,
                outputFormat: '@' + msg,
                outputBuffer: buff,
                outputFormatLogger: formatLog,
            });
        }
    }
    formatCustomFunction_(data, token, tree, options) {
        const buff = data.outputBuffer;
        // TODO(b/314203187): Not null asserted, check that this is correct.
        const node = data.node;
        const formatLog = data.outputFormatLogger;
        // Custom functions.
        if (token === 'if') {
            formatLog.writeToken(token);
            // TODO(b/314203187): Not null asserted, check that this is correct.
            const cond = tree.firstChild;
            const attrib = cond.value.slice(1);
            if (AutomationUtil.isTruthy(node, attrib)) {
                formatLog.write(attrib + '==true => ');
                OutputFormatter.format(this.output_, {
                    node,
                    outputFormat: cond.nextSibling || '',
                    outputBuffer: buff,
                    outputFormatLogger: formatLog,
                });
            }
            else if (AutomationUtil.isFalsey(node, attrib)) {
                formatLog.write(attrib + '==false => ');
                OutputFormatter.format(this.output_, {
                    node,
                    outputFormat: cond.nextSibling.nextSibling || '',
                    outputBuffer: buff,
                    outputFormatLogger: formatLog,
                });
            }
        }
        else if (token === 'nif') {
            formatLog.writeToken(token);
            // TODO(b/314203187): Not null asserted, check that this is correct.
            const cond = tree.firstChild;
            const attrib = cond.value.slice(1);
            if (AutomationUtil.isFalsey(node, attrib)) {
                formatLog.write(attrib + '==false => ');
                OutputFormatter.format(this.output_, {
                    node,
                    outputFormat: cond.nextSibling || '',
                    outputBuffer: buff,
                    outputFormatLogger: formatLog,
                });
            }
            else if (AutomationUtil.isTruthy(node, attrib)) {
                formatLog.write(attrib + '==true => ');
                OutputFormatter.format(this.output_, {
                    node,
                    outputFormat: cond.nextSibling.nextSibling || '',
                    outputBuffer: buff,
                    outputFormatLogger: formatLog,
                });
            }
        }
        else if (token === 'earcon') {
            // Ignore unless we're generating speech output.
            if (!this.output_.formatAsSpeech) {
                return;
            }
            // TODO(b/314203187): Not null asserted, check that this is correct.
            options.annotation.push(new OutputEarconAction(EarconId.fromName(tree.firstChild.value), node.location || undefined));
            this.output_.append(buff, '', options);
            formatLog.writeTokenWithValue(token, tree.firstChild.value);
        }
    }
    formatDescendants_(data, token) {
        const buff = data.outputBuffer;
        const node = data.node;
        const formatLog = data.outputFormatLogger;
        if (!node) {
            return;
        }
        let leftmost = node;
        let rightmost = node;
        if (AutomationPredicate.leafOrStaticText(node)) {
            // Find any deeper leaves, if any, by starting from one level
            // down.
            leftmost = node.firstChild;
            rightmost = node.lastChild;
            if (!leftmost || !rightmost) {
                return;
            }
        }
        // Construct a range to the leftmost and rightmost leaves. This
        // range gets rendered below which results in output that is the
        // same as if a user navigated through the entire subtree of |node|.
        leftmost = AutomationUtil.findNodePre(leftmost, Dir$9.FORWARD, AutomationPredicate.leafOrStaticText);
        rightmost = AutomationUtil.findNodePre(rightmost, Dir$9.BACKWARD, AutomationPredicate.leafOrStaticText);
        if (!leftmost || !rightmost) {
            return;
        }
        const subrange = new CursorRange(new Cursor(leftmost, CURSOR_NODE_INDEX), new Cursor(rightmost, CURSOR_NODE_INDEX));
        let prev = undefined;
        if (node) {
            prev = CursorRange.fromNode(node);
        }
        formatLog.writeToken(token);
        this.output_.render(subrange, prev, OutputCustomEvent.NAVIGATE, buff, formatLog, { suppressStartEndAncestry: true });
    }
    formatDescription_(data, token, options) {
        const buff = data.outputBuffer;
        // TODO(b/314203187): Not null asserted, check that this is correct.
        const node = data.node;
        const formatLog = data.outputFormatLogger;
        if (node.name === node.description) {
            return;
        }
        options.annotation.push(token);
        this.output_.append(buff, node.description || '', options);
        formatLog.writeTokenWithValue(token, node.description);
    }
    formatFind_(data, token, tree) {
        const buff = data.outputBuffer;
        const formatLog = data.outputFormatLogger;
        // TODO(b/314203187): Not null asserted, check that this is correct.
        let node = data.node;
        // Find takes two arguments: JSON query string and format string.
        if (tree.firstChild) {
            const jsonQuery = tree.firstChild.value;
            node = node.find(JSON.parse(jsonQuery));
            const formatString = tree.firstChild.nextSibling || '';
            if (node) {
                formatLog.writeToken(token);
                OutputFormatter.format(this.output_, {
                    node,
                    outputFormat: formatString,
                    outputBuffer: buff,
                    outputFormatLogger: formatLog,
                });
            }
        }
    }
    formatIndexInParent_(data, token, tree, options) {
        const buff = data.outputBuffer;
        // TODO(b/314203187): Not null asserted, check that this is correct.
        const node = data.node;
        const formatLog = data.outputFormatLogger;
        if (node.parent) {
            options.annotation.push(token);
            let roles;
            if (tree.firstChild) {
                roles = this.createRoles_(tree);
            }
            else {
                roles = new Set();
                roles.add(node.role);
            }
            let count = 0;
            for (let i = 0, child; child = node.parent.children[i]; i++) {
                if (roles.has(child.role)) {
                    count++;
                }
                if (node === child) {
                    break;
                }
            }
            this.output_.append(buff, String(count));
            formatLog.writeTokenWithValue(token, String(count));
        }
    }
    formatInputType_(data, token, options) {
        const buff = data.outputBuffer;
        // TODO(b/314203187): Not null asserted, check that this is correct.
        const node = data.node;
        const formatLog = data.outputFormatLogger;
        if (!node.inputType) {
            return;
        }
        options.annotation.push(token);
        let msgId = INPUT_TYPE_MESSAGE_IDS[node.inputType] || 'input_type_text';
        if (this.output_.formatAsBraille) {
            msgId = msgId + '_brl';
        }
        this.output_.append(buff, Msgs.getMsg(msgId), options);
        formatLog.writeTokenWithValue(token, Msgs.getMsg(msgId));
    }
    formatJoinedDescendants_(data, _token, options) {
        const buff = data.outputBuffer;
        const node = data.node;
        const formatLog = data.outputFormatLogger;
        const unjoined = [];
        formatLog.write('joinedDescendants {');
        OutputFormatter.format(this.output_, {
            node,
            outputFormat: '$descendants',
            outputBuffer: unjoined,
            outputFormatLogger: formatLog,
        });
        this.output_.append(buff, unjoined.join(' '), options);
        formatLog.write('}: ' + (unjoined.length ? unjoined.join(' ') : 'EMPTY') + '\n');
    }
    formatListNestedLevel_(data) {
        const buff = data.outputBuffer;
        const node = data.node;
        let level = 0;
        let current = node;
        while (current) {
            if (current.role === RoleType$f.LIST) {
                level += 1;
            }
            current = current.parent;
        }
        this.output_.append(buff, level.toString());
    }
    formatMessage_(data, token, tree, options) {
        const buff = data.outputBuffer;
        // TODO(b/314203187): Not null asserted, check that this is correct.
        const node = data.node;
        const formatLog = data.outputFormatLogger;
        const isPluralized = (token[0] === '@');
        if (isPluralized) {
            token = token.slice(1);
        }
        // Tokens can have substitutions.
        const pieces = token.split('+');
        token = pieces.reduce((prev, cur) => {
            let lookup = cur;
            if (cur[0] === '$') {
                lookup = node[cur.slice(1)];
            }
            return prev + lookup;
        }, '');
        const msgId = token;
        let msgArgs = [];
        formatLog.write(token + '{');
        if (!isPluralized) {
            msgArgs = this.getNonPluralizedArguments_(tree, node, formatLog, msgArgs);
        }
        let msg = Msgs.getMsg(msgId, msgArgs);
        try {
            if (this.output_.formatAsBraille) {
                msg = Msgs.getMsg(msgId + '_brl', msgArgs) || msg;
            }
        }
        catch (e) {
            // TODO(accessibility): Handle whatever error this is.
        }
        if (!msg) {
            this.error_(formatLog, 'Could not get message ' + msgId);
            return;
        }
        if (isPluralized) {
            msg = this.getPluralizedMessage_(tree, node, formatLog, msg);
        }
        formatLog.write('}');
        this.output_.append(buff, msg, options);
        formatLog.write(': ' + msg + '\n');
    }
    formatName_(data, token, options) {
        const buff = data.outputBuffer;
        // TODO(b/314203187): Not null asserted, check that this is correct.
        const node = data.node;
        const prevNode = data.opt_prevNode;
        const formatLog = data.outputFormatLogger;
        options.annotation.push(token);
        const earcon = node ? this.output_.findEarcon(node, prevNode) : null;
        if (earcon) {
            options.annotation.push(earcon);
        }
        // Place the selection on the first character of the name if the
        // node is the active descendant. This ensures the braille window is
        // panned appropriately.
        if (node.activeDescendantFor && node.activeDescendantFor.length > 0) {
            options.annotation.push(new OutputSelectionSpan(0, 0));
        }
        if (SettingsManager.get('languageSwitching')) {
            this.output_.assignLocaleAndAppend(node.name || '', node, buff, options);
        }
        else {
            this.output_.append(buff, node.name || '', options);
        }
        formatLog.writeTokenWithValue(token, node.name);
    }
    formatNameFromNode_(data, token, options) {
        const buff = data.outputBuffer;
        // TODO(b/314203187): Not null asserted, check that this is correct.
        const node = data.node;
        const formatLog = data.outputFormatLogger;
        if (node.nameFrom === NameFromType.CONTENTS) {
            return;
        }
        options.annotation.push('name');
        this.output_.append(buff, node.name || '', options);
        formatLog.writeTokenWithValue(token, node.name);
    }
    formatNameOrDescendants_(data, token, options) {
        const buff = data.outputBuffer;
        // TODO(b/314203187): Not null asserted, check that this is correct.
        const node = data.node;
        const formatLog = data.outputFormatLogger;
        options.annotation.push(token);
        if (node.name &&
            (node.nameFrom !== NameFromType.CONTENTS ||
                node.children.every(child => child.role === RoleType$f.STATIC_TEXT))) {
            this.output_.append(buff, node.name || '', options);
            formatLog.writeTokenWithValue(token, node.name);
        }
        else {
            formatLog.writeToken(token);
            OutputFormatter.format(this.output_, {
                node,
                outputFormat: '$descendants',
                outputBuffer: buff,
                outputFormatLogger: formatLog,
            });
        }
    }
    formatNode_(data, token, tree, options) {
        const buff = data.outputBuffer;
        // TODO(b/314203187): Not null asserted, check that this is correct.
        const node = data.node;
        const formatLog = data.outputFormatLogger;
        let prevNode = data.opt_prevNode;
        if (!tree.firstChild) {
            return;
        }
        const relationName = tree.firstChild.value;
        if (relationName === 'tableCellColumnHeaders') {
            // Skip output when previous position falls on the same column.
            while (prevNode && !AutomationPredicate.cellLike(prevNode)) {
                prevNode = prevNode.parent;
            }
            if (prevNode &&
                prevNode.tableCellColumnIndex === node.tableCellColumnIndex) {
                return;
            }
            const headers = node.tableCellColumnHeaders;
            if (headers) {
                for (let i = 0; i < headers.length; i++) {
                    const header = headers[i].name;
                    if (header) {
                        this.output_.append(buff, header, options);
                        formatLog.writeTokenWithValue(token, header);
                    }
                }
            }
        }
        else if (relationName === 'tableCellRowHeaders') {
            const headers = node.tableCellRowHeaders;
            if (headers) {
                for (let i = 0; i < headers.length; i++) {
                    const header = headers[i].name;
                    if (header) {
                        this.output_.append(buff, header, options);
                        formatLog.writeTokenWithValue(token, header);
                    }
                }
            }
        }
        else if (node[relationName]) {
            let related;
            // TODO(https://crbug.com/1460020): Support multiple IDs in
            // aria-errormessage.
            if (relationName === 'errorMessage') {
                related = node[relationName][0];
            }
            else {
                related = node[relationName];
            }
            this.output_.formatNode(related, related, OutputCustomEvent.NAVIGATE, buff, formatLog);
        }
    }
    formatPhoneticReading_(data) {
        const buff = data.outputBuffer;
        // TODO(b/314203187): Not null asserted, check that this is correct.
        const node = data.node;
        const text = PhoneticData.forText(node.name || '', chrome.i18n.getUILanguage());
        this.output_.append(buff, text);
    }
    formatPrecedingBullet_(data) {
        const buff = data.outputBuffer;
        // TODO(b/314203187): Not null asserted, check that this is correct.
        const node = data.node;
        let current = node;
        if (current.role === RoleType$f.INLINE_TEXT_BOX) {
            current = current.parent;
        }
        if (!current || current.role !== RoleType$f.STATIC_TEXT) {
            return;
        }
        current = current.previousSibling;
        if (current && current.role === RoleType$f.LIST_MARKER) {
            this.output_.append(buff, current.name || '');
        }
    }
    formatPressed_(data, token) {
        const buff = data.outputBuffer;
        // TODO(b/314203187): Not null asserted, check that this is correct.
        const node = data.node;
        const formatLog = data.outputFormatLogger;
        // TODO(b/314203187): Not null asserted, check that this is correct.
        const msg = OutputPropertyMap['PRESSED'][node.checked];
        if (msg) {
            formatLog.writeToken(token);
            OutputFormatter.format(this.output_, {
                node,
                outputFormat: '@' + msg,
                outputBuffer: buff,
                outputFormatLogger: formatLog,
            });
        }
    }
    formatRestriction_(data, token) {
        const buff = data.outputBuffer;
        // TODO(b/314203187): Not null asserted, check that this is correct.
        const node = data.node;
        const formatLog = data.outputFormatLogger;
        // TODO(b/314203187): Not null asserted, check that this is correct.
        const msg = OutputPropertyMap['RESTRICTION'][node.restriction];
        if (msg) {
            formatLog.writeToken(token);
            OutputFormatter.format(this.output_, {
                node,
                outputFormat: '@' + msg,
                outputBuffer: buff,
                outputFormatLogger: formatLog,
            });
        }
    }
    formatRole_(data, token, options) {
        const buff = data.outputBuffer;
        // TODO(b/314203187): Not null asserted, check that this is correct.
        const node = data.node;
        const formatLog = data.outputFormatLogger;
        // TODO(b/314203187): Not null asserted, check that this is correct.
        options.annotation.push(token);
        let msg = node.role;
        const info = OutputRoleInfo[node.role];
        if (node.roleDescription) {
            msg = node.roleDescription;
        }
        else if (info) {
            if (this.output_.formatAsBraille) {
                msg = Msgs.getMsg(info.msgId + '_brl');
            }
            else if (info.msgId) {
                msg = Msgs.getMsg(info.msgId);
            }
        }
        else {
            // We can safely ignore this role. ChromeVox output tests cover
            // message id validity.
            return;
        }
        this.output_.append(buff, msg || '', options);
        formatLog.writeTokenWithValue(token, msg);
    }
    formatState_(data, token) {
        const buff = data.outputBuffer;
        // TODO(b/314203187): Not null asserted, check that this is correct.
        const node = data.node;
        const formatLog = data.outputFormatLogger;
        if (node.state) {
            Object.getOwnPropertyNames(node.state).forEach(state => {
                const stateInfo = OUTPUT_STATE_INFO[state];
                if (stateInfo && !stateInfo.isRoleSpecific && stateInfo.on) {
                    formatLog.writeToken(token);
                    OutputFormatter.format(this.output_, {
                        node,
                        outputFormat: '$' + state,
                        outputBuffer: buff,
                        outputFormatLogger: formatLog,
                    });
                }
            });
        }
    }
    formatTableCellIndex_(data, token, options) {
        const buff = data.outputBuffer;
        // TODO(b/314203187): Not null asserted, check that this is correct.
        const node = data.node;
        const formatLog = data.outputFormatLogger;
        let value = node[token];
        if (value === undefined) {
            return;
        }
        value = String(value + 1);
        options.annotation.push(token);
        this.output_.append(buff, value, options);
        formatLog.writeTokenWithValue(token, value);
    }
    formatTextContent_(data, token, options) {
        const buff = data.outputBuffer;
        // TODO(b/314203187): Not null asserted, check that this is correct.
        const node = data.node;
        const formatLog = data.outputFormatLogger;
        if (node.name && token === 'nameOrTextContent') {
            formatLog.writeToken(token);
            OutputFormatter.format(this.output_, {
                node,
                outputFormat: '$name',
                outputBuffer: buff,
                outputFormatLogger: formatLog,
            });
            return;
        }
        if (!node.firstChild) {
            return;
        }
        const root = node;
        const walker = new AutomationTreeWalker(node, Dir$9.FORWARD, {
            visit: AutomationPredicate.leafOrStaticText,
            leaf: n => {
                // The root might be a leaf itself, but we still want to descend
                // into it.
                return n !== root && AutomationPredicate.leafOrStaticText(n);
            },
            root: r => r === root,
        });
        const outputStrings = [];
        while (walker.next().node) {
            // TODO(b/314203187): Not null asserted, check that this is correct.
            if (walker.node.name) {
                outputStrings.push(walker.node.name.trim());
            }
        }
        const finalOutput = outputStrings.join(' ');
        this.output_.append(buff, finalOutput, options);
        formatLog.writeTokenWithValue(token, finalOutput);
    }
    formatUrlFilename_(data, token, options) {
        const buff = data.outputBuffer;
        // TODO(b/314203187): Not null asserted, check that this is correct.
        const node = data.node;
        const formatLog = data.outputFormatLogger;
        options.annotation.push('name');
        const url = node.url || '';
        let filename = '';
        if (url.substring(0, 4) !== 'data') {
            filename = url.substring(url.lastIndexOf('/') + 1, url.lastIndexOf('.'));
            // Hack to not speak the filename if it's ridiculously long.
            if (filename.length >= 30) {
                filename = filename.substring(0, 16) + '...';
            }
        }
        this.output_.append(buff, filename, options);
        formatLog.writeTokenWithValue(token, filename);
    }
    formatValue_(data, token, options) {
        const buff = data.outputBuffer;
        // TODO(b/314203187): Not null asserted, check that this is correct.
        const node = data.node;
        const formatLog = data.outputFormatLogger;
        const text = node.value || '';
        // TODO(b/314203187): Not null asserted, check that this is correct.
        if (!node.state[StateType$f.EDITABLE] && node.name === text) {
            return;
        }
        let selectedText = '';
        if (node.textSelStart !== undefined) {
            options.annotation.push(new OutputSelectionSpan(node.textSelStart || 0, node.textSelEnd || 0));
            if (node.value) {
                selectedText =
                    node.value.substring(node.textSelStart || 0, node.textSelEnd || 0);
            }
        }
        options.annotation.push(token);
        // TODO(b/314203187): Not null asserted, check that this is correct.
        if (selectedText && !this.output_.formatAsBraille &&
            node.state[StateType$f.FOCUSED]) {
            this.output_.append(buff, selectedText, options);
            this.output_.append(buff, Msgs.getMsg('selected'));
            formatLog.writeTokenWithValue(token, selectedText);
            formatLog.write('selected\n');
        }
        else {
            this.output_.append(buff, text, options);
            formatLog.writeTokenWithValue(token, text);
        }
    }
    getNonPluralizedArguments_(tree, node, formatLog, msgArgs) {
        let curArg = tree.firstChild;
        while (curArg) {
            if (curArg.value[0] !== '$') {
                this.unexpectedValue_(formatLog, curArg.value);
                return msgArgs;
            }
            const msgBuff = [];
            OutputFormatter.format(this.output_, {
                node,
                outputFormat: curArg,
                outputBuffer: msgBuff,
                outputFormatLogger: formatLog,
            });
            let extraArgs = msgBuff.map(spannable => spannable.toString());
            // Fill in empty string if nothing was formatted.
            if (!msgBuff.length) {
                extraArgs = [''];
            }
            msgArgs = msgArgs.concat(extraArgs);
            curArg = curArg.nextSibling;
        }
        return msgArgs;
    }
    getPluralizedMessage_(tree, node, formatLog, msg) {
        const arg = tree.firstChild;
        if (!arg || arg.nextSibling) {
            this.error_(formatLog, 'Pluralized messages take exactly one argument');
            return msg;
        }
        if (arg.value[0] !== '$') {
            this.unexpectedValue_(formatLog, arg.value);
            return msg;
        }
        const argBuff = [];
        OutputFormatter.format(this.output_, {
            node,
            outputFormat: arg,
            outputBuffer: argBuff,
            outputFormatLogger: formatLog,
        });
        const namedArgs = { COUNT: Number(argBuff[0]) };
        const formatter = new MessageFormat(chrome.i18n.getUILanguage(), msg);
        return formatter.format(namedArgs, () => { });
    }
    unexpectedValue_(formatLog, value) {
        this.error_(formatLog, 'Unexpected value: ' + value);
    }
}

// Copyright 2022 The Chromium Authors
// Use of this source code is governed by a BSD-style license that can be
// found in the LICENSE file.
// These functions are public only to output/ classes.
class OutputInterface {
}

// Copyright 2022 The Chromium Authors
// Use of this source code is governed by a BSD-style license that can be
// found in the LICENSE file.
/**
 * @fileoverview Contains the rules for output based on type information.
 */
const EventType$c = chrome.automation.EventType;
const RoleType$e = chrome.automation.RoleType;
class OutputRule {
    event_;
    role_ = CustomRole.DEFAULT;
    navigation_;
    output_;
    constructor(event) {
        this.event_ = this.getEvent_(event);
    }
    getEvent_(event) {
        if (OutputRule.RULES[event]) {
            return event;
        }
        return OutputCustomEvent.NAVIGATE;
    }
    get specifier() {
        return {
            event: this.event_,
            role: this.role_,
            navigation: this.navigation_,
            output: this.output_,
        };
    }
    get formatString() {
        // TODO(crbug.com/314203187): Not null asserted, check that this is correct.
        return OutputRule.RULES[this.event_][this.role_][this.output_];
    }
    /**
     * @return true if the role was set, false otherwise.
     */
    populateRole(role, formatName) {
        if (this.hasRule_(role, formatName) && role) {
            this.role_ = role;
            return true;
        }
        if (this.hasRule_(parent(role), formatName) &&
            parent(role) !== CustomRole.NO_ROLE) {
            this.role_ = parent(role);
            return true;
        }
        return false;
    }
    // The following setter functions are a temporary measure.
    // TODO(anastasi): move the logic for determining the below properties into
    // this class.
    set role(role) {
        this.role_ = role;
    }
    set output(output) {
        this.output_ = output;
    }
    get event() {
        return this.event_;
    }
    get role() {
        return this.role_;
    }
    get navigation() {
        return this.navigation_;
    }
    get output() {
        return this.output_;
    }
    // ========= Private methods =========
    /**
     * Returns true if there is a rule for this role/format combo.
     */
    hasRule_(role, format) {
        // TODO(crbug.com/314203187): Not null asserted, check that this is correct.
        const eventBlock = OutputRule.RULES[this.event_];
        return Boolean(role && eventBlock[role] && format && eventBlock[role][format]);
    }
}
class AncestryOutputRule extends OutputRule {
    constructor(eventType, role, navigationType, tryBraille) {
        super(eventType);
        this.populateRole(role, navigationType);
        this.populateNavigation(navigationType);
        this.populateOutput(tryBraille);
    }
    populateNavigation(navigationType) {
        // TODO(crbug.com/314203187): Not null asserted, check that this is correct.
        if (navigationType && OutputRule.RULES[this.event_][this.role_] &&
            OutputRule.RULES[this.event_][this.role_][navigationType]) {
            this.navigation_ = navigationType;
        }
    }
    populateOutput(tryBraille) {
        // TODO(crbug.com/314203187): Not null asserted, check that this is correct.
        if (!OutputRule.RULES[this.event_][this.role_]) {
            // Invalid rule case.
            return;
        }
        const rule = OutputRule.RULES[this.event_][this.role_][this.navigation_];
        if (!rule || !isEnterRule(rule)) {
            return;
        }
        const enter = rule;
        if (enter.speak) {
            this.output_ = OutputFormatType.SPEAK;
        }
        if (tryBraille && enter.braille) {
            this.output_ = OutputFormatType.BRAILLE;
        }
    }
    get defined() {
        // TODO(crbug.com/314203187): Not null asserted, check that this is correct.
        return Boolean(OutputRule.RULES[this.event_][this.role_] &&
            OutputRule.RULES[this.event_][this.role_][this.navigation_]);
    }
    get enterFormat() {
        // TODO(crbug.com/314203187): Not null asserted, check that this is correct.
        const rule = OutputRule.RULES[this.event_][this.role_][this.navigation_];
        if (this.output_ && isEnterRule(rule)) {
            return rule[this.output_];
        }
        return String(rule) || '';
    }
}
function parent(role) {
    if (!role) {
        return CustomRole.NO_ROLE;
    }
    return OutputRoleInfo[role]?.inherits ??
        CustomRole.NO_ROLE;
}
function isEnterRule(rule) {
    if (!rule || typeof rule !== 'object') {
        return false;
    }
    const keys = Object.keys(rule);
    if (keys.length > 2) {
        return false;
    }
    for (const key of keys) {
        if (key === 'speak' || key === 'braille') {
            continue;
        }
        return false;
    }
    return true;
}
(function (OutputRule) {
    /**
     * Rules specifying format of AutomationNodes for output.
     * Please see above for more information on properties.
     */
    OutputRule.RULES = {
        navigate: {
            [CustomRole.DEFAULT]: {
                speak: `$name $node(activeDescendant) $value $state $restriction $role
          $description`,
                braille: ``,
            },
            [AbstractRole.CONTAINER]: {
                startOf: `$nameFromNode $role $state $description`,
                endOf: `@end_of_container($role)`,
            },
            [AbstractRole.FORM_FIELD_CONTAINER]: {
                enter: `$nameFromNode $role $state $description`,
                leave: `@exited_container($role)`,
            },
            [AbstractRole.ITEM]: {
                // Note that ChromeVox generally does not output position/count. Only
                // for some roles (see sub-output rules) or when explicitly provided by
                // an author (via posInSet), do we include them in the output.
                enter: `$nameFromNode $role $state $restriction $description
          $if($posInSet, @describe_index($posInSet, $setSize))`,
                speak: `$state $nameOrTextContent= $role
          $if($posInSet, @describe_index($posInSet, $setSize))
          $description $restriction`,
            },
            [AbstractRole.LIST]: {
                startOf: `$nameFromNode $role $if($setSize, @@list_with_items($setSize))
          $restriction $description`,
                endOf: `@end_of_container($role) @@list_nested_level($listNestedLevel)`,
            },
            [AbstractRole.NAME_FROM_CONTENTS]: {
                speak: `$nameOrDescendants $node(activeDescendant) $value $state
          $restriction $role $description`,
            },
            [AbstractRole.RANGE]: {
                speak: `$name $node(activeDescendant) $description $role
          $if($value, $value, $if($valueForRange, $valueForRange))
          $state $restriction
          $if($minValueForRange, @aria_value_min($minValueForRange))
          $if($maxValueForRange, @aria_value_max($maxValueForRange))`,
            },
            [AbstractRole.SPAN]: {
                startOf: `$nameFromNode $role $state $description`,
                endOf: `@end_of_container($role)`,
            },
            [RoleType$e.ALERT]: {
                enter: `$name $role $state`,
                speak: `$earcon(ALERT_NONMODAL) $role $nameOrTextContent $description
          $state`,
            },
            [RoleType$e.ALERT_DIALOG]: {
                enter: `$earcon(ALERT_MODAL) $name $state $description $roleDescription
          $textContent`,
                speak: `$earcon(ALERT_MODAL) $name $nameOrTextContent $description $state
          $role`,
            },
            [RoleType$e.BUTTON]: {
                speak: `$name $node(activeDescendant) $state $restriction $role
          $description`,
            },
            [RoleType$e.CELL]: {
                enter: {
                    speak: `$cellIndexText $node(tableCellColumnHeaders) $nameFromNode
            $roleDescription $state`,
                    braille: `$state $cellIndexText $node(tableCellColumnHeaders)
            $nameFromNode $roleDescription`,
                },
                speak: `$nameFromNode $descendants $cellIndexText
          $node(tableCellColumnHeaders) $roleDescription $state $description`,
                braille: `$state
          $name $cellIndexText $node(tableCellColumnHeaders) $roleDescription
          $description`,
            },
            [RoleType$e.CHECK_BOX]: {
                speak: `$if($checked, $earcon(CHECK_ON), $earcon(CHECK_OFF))
          $name $role $if($checkedStateDescription, $checkedStateDescription, $checked)
          $description $state $restriction`,
            },
            [RoleType$e.CLIENT]: { speak: `$name` },
            [RoleType$e.COMBO_BOX_MENU_BUTTON]: {
                speak: `$name $value $role @aria_has_popup
          $if($setSize, @@list_with_items($setSize))
          $state $restriction $description`,
            },
            [RoleType$e.DATE]: { enter: `$nameFromNode $role $state $restriction $description` },
            [RoleType$e.DIALOG]: { enter: `$nameFromNode $role $description` },
            [RoleType$e.GENERIC_CONTAINER]: {
                enter: `$nameFromNode $description $state`,
                speak: `$nameOrTextContent $description $state`,
            },
            [RoleType$e.EMBEDDED_OBJECT]: { speak: `$name` },
            [RoleType$e.GRID]: {
                speak: `$name $node(activeDescendant) $role $state $restriction
          $description`,
            },
            [RoleType$e.GRID_CELL]: {
                enter: {
                    speak: `$cellIndexText $node(tableCellColumnHeaders) $nameFromNode
            $roleDescription $state`,
                    braille: `$state $cellIndexText $node(tableCellColumnHeaders)
            $nameFromNode $roleDescription`,
                },
                speak: `$nameFromNode $descendants $cellIndexText
          $node(tableCellColumnHeaders) $roleDescription $state $description`,
                braille: `$state
          $name $cellIndexText $node(tableCellColumnHeaders) $roleDescription
          $description
          $if($selected, @aria_selected_true)`,
            },
            [RoleType$e.GROUP]: {
                enter: `$nameFromNode $roleDescription $state $restriction $description`,
                speak: `$nameOrDescendants $value $state $restriction $roleDescription
          $description`,
                leave: ``,
            },
            [RoleType$e.HEADING]: {
                enter: `!relativePitch(hierarchicalLevel)
          $nameFromNode=
          $if($hierarchicalLevel, @tag_h+$hierarchicalLevel, $role) $state
          $description`,
                speak: `!relativePitch(hierarchicalLevel)
          $nameOrDescendants=
          $if($hierarchicalLevel, @tag_h+$hierarchicalLevel, $role) $state
          $restriction $description`,
            },
            [RoleType$e.IMAGE]: {
                speak: `$if($name, $name,
          $if($imageAnnotation, $imageAnnotation, $urlFilename))
          $value $state $role $description`,
            },
            [RoleType$e.IME_CANDIDATE]: {
                speak: `$name $phoneticReading @describe_index($posInSet, $setSize)`
            },
            [RoleType$e.INLINE_TEXT_BOX]: { speak: `$precedingBullet $name=` },
            [RoleType$e.INPUT_TIME]: { enter: `$nameFromNode $role $state $restriction $description` },
            [RoleType$e.LABEL_TEXT]: {
                speak: `$name $value $state $restriction $roleDescription $description`,
            },
            [RoleType$e.LINE_BREAK]: { speak: `$name=` },
            [RoleType$e.LINK]: {
                enter: `$nameFromNode= $role $state $restriction`,
                speak: `$name $value $state $restriction
          $if($inPageLinkTarget, @internal_link, $role) $description`,
            },
            [RoleType$e.LIST]: {
                speak: `$nameFromNode $descendants $role
          @@list_with_items($setSize) $description $state`,
            },
            [RoleType$e.LIST_BOX]: {
                enter: `$nameFromNode $role @@list_with_items($setSize)
          $restriction $description`,
            },
            [RoleType$e.LIST_BOX_OPTION]: {
                speak: `$state $name $role @describe_index($posInSet, $setSize)
          $description $restriction
          $nif($selected, @aria_selected_false)`,
                braille: `$state $name $role @describe_index($posInSet, $setSize)
          $description $restriction
          $if($selected, @aria_selected_true, @aria_selected_false)`,
            },
            [RoleType$e.LIST_MARKER]: { speak: `$name` },
            [RoleType$e.MENU]: {
                enter: `$name $role `,
                speak: `$name $node(activeDescendant)
          $role @@list_with_items($setSize) $description $state $restriction`,
            },
            [RoleType$e.MENU_ITEM]: {
                speak: `$name $role $if($hasPopup, @has_submenu)
          @describe_index($posInSet, $setSize) $description $state $restriction`,
            },
            [RoleType$e.MENU_ITEM_CHECK_BOX]: {
                speak: `$if($checked, $earcon(CHECK_ON), $earcon(CHECK_OFF))
          $name $role $checked $state $restriction $description
          @describe_index($posInSet, $setSize)`,
            },
            [RoleType$e.MENU_ITEM_RADIO]: {
                speak: `$if($checked, $earcon(CHECK_ON), $earcon(CHECK_OFF))
          $if($checked, @describe_menu_item_radio_selected($name),
          @describe_menu_item_radio_unselected($name)) $state $roleDescription
          $restriction $description
          @describe_index($posInSet, $setSize)`,
            },
            [RoleType$e.MENU_LIST_OPTION]: {
                speak: `$state $name $role @describe_index($posInSet, $setSize)
          $description $restriction
          $nif($selected, @aria_selected_false)`,
                braille: `$state $name $role @describe_index($posInSet, $setSize)
          $description $restriction
          $if($selected, @aria_selected_true, @aria_selected_false)`,
            },
            [RoleType$e.PARAGRAPH]: { speak: `$nameOrDescendants $roleDescription` },
            [RoleType$e.RADIO_BUTTON]: {
                speak: `$if($checked, $earcon(CHECK_ON), $earcon(CHECK_OFF))
          $if($checked, @describe_radio_selected($name),
          @describe_radio_unselected($name))
          @describe_index($posInSet, $setSize)
          $roleDescription $description $state $restriction`,
            },
            [RoleType$e.ROOT_WEB_AREA]: { enter: `$name`, speak: `$if($name, $name, @web_content)` },
            [RoleType$e.REGION]: { speak: `$state $nameOrTextContent $description $roleDescription` },
            [RoleType$e.ROW]: {
                startOf: `$node(tableRowHeader) $roleDescription
          $if($hierarchicalLevel, @describe_depth($hierarchicalLevel))`,
                speak: ` $if($hierarchicalLevel, @describe_depth($hierarchicalLevel))
          $name $node(activeDescendant) $value $state $restriction $role
          $if($selected, @aria_selected_true) $description`,
            },
            [RoleType$e.STATIC_TEXT]: { speak: `$precedingBullet $name= $description` },
            [RoleType$e.SWITCH]: {
                speak: `$if($checked, $earcon(CHECK_ON), $earcon(CHECK_OFF))
          $if($checked, @describe_switch_on($name),
          @describe_switch_off($name)) $roleDescription
          $description $state $restriction`,
            },
            [RoleType$e.TAB]: {
                speak: `@describe_tab($name) $roleDescription $description
          @describe_index($posInSet, $setSize) $state $restriction
          $if($selected, @aria_selected_true)`,
            },
            [RoleType$e.TABLE]: {
                enter: `$roleDescription @table_summary($name,
          $if($ariaRowCount, $ariaRowCount, $tableRowCount),
          $if($ariaColumnCount, $ariaColumnCount, $tableColumnCount))
          $node(tableHeader)`,
            },
            [RoleType$e.TAB_LIST]: {
                speak: `$name $node(activeDescendant) $state $restriction $role
          $description`,
            },
            [RoleType$e.TEXT_FIELD]: {
                speak: `$name $value
          $if($roleDescription, $roleDescription,
              $if($multiline, @tag_textarea,
                  $if($inputType, $inputType, $role)))
          $description $state $restriction`,
            },
            [RoleType$e.TIMER]: {
                speak: `$nameFromNode $descendants $value $state $role
        $description`,
            },
            [RoleType$e.TOGGLE_BUTTON]: {
                speak: `$if($checked, $earcon(CHECK_ON), $earcon(CHECK_OFF))
          $name $role $pressed $description $state $restriction`,
            },
            [RoleType$e.TOOLBAR]: { enter: `$name $role $description $restriction` },
            [RoleType$e.TREE]: { enter: `$name $role @@list_with_items($setSize) $restriction` },
            [RoleType$e.TREE_ITEM]: {
                enter: `$role $expanded $collapsed $restriction
          @describe_index($posInSet, $setSize)
          @describe_depth($hierarchicalLevel)`,
                speak: `$name
          $role $description $state $restriction
          $nif($selected, @aria_selected_false)
          @describe_index($posInSet, $setSize)
          @describe_depth($hierarchicalLevel)`,
            },
            [RoleType$e.UNKNOWN]: { speak: `` },
            [RoleType$e.WINDOW]: {
                enter: `@describe_window($name) $description`,
                speak: `@describe_window($name) $description $earcon(OBJECT_OPEN)`,
            },
        },
        [EventType$c.CONTROLS_CHANGED]: {
            [RoleType$e.TAB]: {
                speak: `@describe_tab($name) @describe_index($posInSet, $setSize)
          @aria_selected_true`,
            },
        },
        [EventType$c.MENU_START]: {
            [CustomRole.DEFAULT]: { speak: `@chrome_menu_opened($name)  $earcon(OBJECT_OPEN)` },
        },
        [EventType$c.MENU_END]: {
            [CustomRole.DEFAULT]: { speak: `@chrome_menu_closed $earcon(OBJECT_CLOSE)` },
        },
        [EventType$c.ALERT]: {
            [CustomRole.DEFAULT]: { speak: `$earcon(ALERT_NONMODAL) $nameOrTextContent $description` },
        },
    };
})(OutputRule || (OutputRule = {}));
TestImportManager.exportForTesting(OutputRule);

// Copyright 2014 The Chromium Authors
// Use of this source code is governed by a BSD-style license that can be
// found in the LICENSE file.
/**
 * @fileoverview Provides output services for ChromeVox.
 */
var EventType$b = chrome.automation.EventType;
var RoleType$d = chrome.automation.RoleType;
var StateType$e = chrome.automation.StateType;
var Dir$8 = constants.Dir;
/**
 * An Output object formats a CursorRange into speech, braille, or both
 * representations. This is typically a |Spannable|.
 * The translation from Range to these output representations rely upon format
 * rules which specify how to convert AutomationNode objects into annotated
 * strings.
 * The format of these rules is as follows.
 * $ prefix: used to substitute either an attribute or a specialized value from
 *     an AutomationNode. Specialized values include role and state.
 *     For example, $value $role $enabled
 *
 * Note: (@) means @ to avoid Closure mistaking it for an annotation.
 *
 * (@) prefix: used to substitute a message. Note the ability to specify params
 * to the message.  For example, '@tag_html' '@selected_index($text_sel_start,
 *     $text_sel_end').
 *
 * (@@) prefix: similar to @, used to substitute a message, but also pulls the
 *     localized string through MessageFormat to support locale
 *     aware plural handling.  The first argument should be a number which will
 *     be passed as a COUNT named parameter to MessageFormat.
 *     TODO(plundblad): Make subsequent arguments normal placeholder arguments
 *     when needed.
 * = suffix: used to specify substitution only if not previously appended.
 *     For example, $name= would insert the name attribute only if no name
 * attribute had been inserted previously.
 *
 * TODO(b/314204374): Eliminate instances of null.
 * @implements {OutputInterface}
 */
class Output extends OutputInterface {
    /**
     * If set, the next speech utterance will use this value instead of the normal
     * queueing mode.
     */
    static forceModeForNextSpeechUtterance_;
    speechBuffer_ = [];
    brailleOutput_ = new BrailleOutput();
    locations_ = [];
    speechEndCallback_;
    // Store output rules.
    speechFormatLog_;
    formatOptions_;
    // The speech category for the generated speech utterance.
    speechCategory_;
    // The speech queue mode for the generated speech utterance.
    queueMode_;
    contextOrder_;
    suppressions_ = {};
    enableHints_ = true;
    initialSpeechProps_;
    drawFocusRing_ = true;
    /**
     * Tracks all ancestors which have received primary formatting in
     * |ancestryHelper_|.
     */
    formattedAncestors_;
    replacements_ = {};
    constructor() {
        super();
        // TODO(dtseng): Include braille specific rules.
        this.speechEndCallback_ = (() => { });
        this.speechFormatLog_ =
            new OutputFormatLogger('enableSpeechLogging', LogType.SPEECH_RULE);
        // Current global options.
        this.formatOptions_ = { speech: true, braille: false, auralStyle: false };
        this.speechCategory_ = TtsCategory.NAV;
        this.contextOrder_ = OutputContextOrder.LAST;
        this.initialSpeechProps_ = new TtsSpeechProperties();
        this.formattedAncestors_ = new WeakSet();
    }
    /**
     * Calling this will make the next speech utterance use |mode| even if it
     * would normally queue or do a category flush. This differs from the
     * |withQueueMode| instance method as it can apply to future output.
     */
    static forceModeForNextSpeechUtterance(mode) {
        if (Output.forceModeForNextSpeechUtterance_ === undefined ||
            mode === undefined ||
            // Only allow setting to higher queue modes.
            mode < Output.forceModeForNextSpeechUtterance_) {
            Output.forceModeForNextSpeechUtterance_ = mode;
        }
    }
    /** @return True if there is any speech that will be output. */
    get hasSpeech() {
        return this.speechBuffer_.some(speech => speech.length);
    }
    /** @return True if there is only whitespace in this output. */
    get isOnlyWhitespace() {
        return this.speechBuffer_.every(buff => !/\S+/.test(buff.toString()));
    }
    /** @return Spannable representing the braille output. */
    get braille() {
        return BrailleOutput.mergeSpans(this.brailleOutput_.buffer);
    }
    /**
     * Specify ranges for speech.
     * @return |this| instance of Output for chaining.
     */
    withSpeech(range, prevRange, type) {
        this.formatOptions_ = { speech: true, braille: false, auralStyle: false };
        this.formattedAncestors_ = new WeakSet();
        this.render(range, prevRange, type, this.speechBuffer_, this.speechFormatLog_);
        return this;
    }
    /**
     * Specify ranges for aurally styled speech.
     * @return |this| instance of Output for chaining.
     */
    withRichSpeech(range, prevRange, type) {
        this.formatOptions_ = { speech: true, braille: false, auralStyle: true };
        this.formattedAncestors_ = new WeakSet();
        this.render(range, prevRange, type, this.speechBuffer_, this.speechFormatLog_);
        return this;
    }
    /**
     * Specify ranges for braille.
     * @return |this| instance of Output for chaining.
     */
    withBraille(range, prevRange, type) {
        this.formatOptions_ = { speech: false, braille: true, auralStyle: false };
        this.formattedAncestors_ = new WeakSet();
        // Braille sometimes shows contextual information depending on role.
        if (range.start.equals(range.end) && range.start.node &&
            AutomationPredicate.contextualBraille(range.start.node) &&
            range.start.node.parent) {
            let start = range.start.node.parent;
            while (start.firstChild) {
                start = start.firstChild;
            }
            let end = range.start.node.parent;
            while (end.lastChild) {
                end = end.lastChild;
            }
            prevRange = CursorRange.fromNode(range.start.node.parent);
            range = new CursorRange(Cursor.fromNode(start), Cursor.fromNode(end));
        }
        this.render(range, prevRange, type, this.brailleOutput_.buffer, this.brailleOutput_.formatLog);
        return this;
    }
    /**
     * Specify ranges for location.
     *  @return |this| instance of Output for chaining.
     */
    withLocation(range, prevRange, type) {
        this.formatOptions_ = { speech: false, braille: false, auralStyle: false };
        this.formattedAncestors_ = new WeakSet();
        this.render(range, prevRange, type, [] /*unused output*/, new OutputFormatLogger('', LogType.SPEECH_RULE) /*unused log*/);
        return this;
    }
    /**
     * Specify the same ranges for speech and braille.
     * @return |this| instance of Output for chaining.
     */
    withSpeechAndBraille(range, prevRange, type) {
        this.withSpeech(range, prevRange, type);
        this.withBraille(range, prevRange, type);
        return this;
    }
    /**
     * Specify the same ranges for aurally styled speech and braille.
     * @return |this| instance of Output for chaining.
     */
    withRichSpeechAndBraille(range, prevRange, type) {
        this.withRichSpeech(range, prevRange, type);
        this.withBraille(range, prevRange, type);
        return this;
    }
    /**
     * Applies the given speech category to the output.
     * @return |this| instance of Output for chaining.
     */
    withSpeechCategory(category) {
        this.speechCategory_ = category;
        return this;
    }
    /**
     * Applies the given speech queue mode to the output.
     * @return |this| instance of Output for chaining.
     */
    withQueueMode(queueMode) {
        this.queueMode_ = queueMode;
        return this;
    }
    /**
     * Outputs a string literal.
     * @return |this| instance of Output for chaining.
     */
    withString(value) {
        this.append(this.speechBuffer_, value);
        this.append(this.brailleOutput_.buffer, value);
        this.speechFormatLog_.write('withString: ' + value + '\n');
        this.brailleOutput_.formatLog.write('withString: ' + value + '\n');
        return this;
    }
    /**
     * Outputs formatting nodes after this will contain context first.
     * @return |this| instance of Output for chaining.
     */
    withContextFirst() {
        this.contextOrder_ = OutputContextOrder.FIRST;
        return this;
    }
    /**
     * Don't include hints in subsequent output.
     * @return |this| instance of Output for chaining.
     */
    withoutHints() {
        this.enableHints_ = false;
        return this;
    }
    /**
     * Don't draw a focus ring based on this output.
     * @return |this| instance of Output for chaining.
     */
    withoutFocusRing() {
        this.drawFocusRing_ = false;
        return this;
    }
    /**
     * Applies given initial speech properties to all output.
     * @return |this| instance of Output for chaining.
     */
    withInitialSpeechProperties(speechProps) {
        this.initialSpeechProps_ = speechProps;
        return this;
    }
    /**
     * Given a string of text and a string to replace |text| with,
     * causes any speech output to apply the replacement.
     * @param text The text to be replaced.
     * @param replace The string to replace |text| with.
     */
    withSpeechTextReplacement(text, replace) {
        this.replacements_[text] = replace;
    }
    /**
     * Suppresses processing of a token for subsequent formatting commands.
     * @return |this| instance of Output for chaining.
     */
    suppress(token) {
        this.suppressions_[token] = true;
        return this;
    }
    /**
     * Apply a format string directly to the output buffer. This lets you
     * output a message directly to the buffer using the format syntax.
     * @param formatStr The format string to apply.
     * @param optNode Optional node to apply the formatting to.
     * @return |this| instance of Output for chaining.
     */
    format(formatStr, optNode) {
        return this.formatForSpeech(formatStr, optNode)
            .formatForBraille(formatStr, optNode);
    }
    /**
     * Apply a format string directly to the speech output buffer. This lets you
     * output a message directly to the buffer using the format syntax.
     * @param formatStr The format string to apply.
     * @param optNode Optional node to apply the formatting to.
     * @return |this| instance of Output for chaining.
     */
    formatForSpeech(formatStr, optNode) {
        const node = optNode || undefined;
        this.formatOptions_ = { speech: true, braille: false, auralStyle: false };
        this.formattedAncestors_ = new WeakSet();
        OutputFormatter.format(this, {
            node,
            outputFormat: formatStr,
            outputBuffer: this.speechBuffer_,
            outputFormatLogger: this.speechFormatLog_,
        });
        return this;
    }
    /**
     * Apply a format string directly to the braille output buffer. This lets you
     * output a message directly to the buffer using the format syntax.
     * @param formatStr The format string to apply.
     * @param optNode Optional node to apply the formatting to.
     * @return |this| instance of Output for chaining.
     */
    formatForBraille(formatStr, optNode) {
        const node = optNode || undefined;
        this.formatOptions_ = { speech: false, braille: true, auralStyle: false };
        this.formattedAncestors_ = new WeakSet();
        OutputFormatter.format(this, {
            node,
            outputFormat: formatStr,
            outputBuffer: this.brailleOutput_.buffer,
            outputFormatLogger: this.brailleOutput_.formatLog,
        });
        return this;
    }
    /**
     * Triggers callback for a speech event.
     * @param callback Callback function that takes in an optional boolean and has
     *     a void return.
     * @return |this| instance of Output for chaining.
     */
    onSpeechEnd(callback) {
        this.speechEndCallback_ = (optCleanupOnly => {
            if (!optCleanupOnly) {
                callback();
            }
        });
        return this;
    }
    /** Executes all specified output. */
    go() {
        // Speech.
        this.sendSpeech_();
        // Braille.
        if (this.brailleOutput_.buffer.length) {
            this.sendBraille_();
        }
        // Display.
        if (this.speechCategory_ !== TtsCategory.LIVE && this.drawFocusRing_) {
            FocusBounds.set(this.locations_);
            if (this.locations_ !== undefined && this.locations_.length !== 0) {
                chrome.accessibilityPrivate.setChromeVoxFocus(this.locations_[0]);
            }
        }
    }
    determineQueueMode_() {
        if (Output.forceModeForNextSpeechUtterance_ !== undefined) {
            const result = Output.forceModeForNextSpeechUtterance_;
            if (this.speechBuffer_.length > 0) {
                Output.forceModeForNextSpeechUtterance_ = undefined;
            }
            return result;
        }
        if (this.queueMode_ !== undefined) {
            return this.queueMode_;
        }
        return QueueMode.QUEUE;
    }
    /**
     * @return speech properties for given buffer.
     */
    getSpeechPropsForBuff_(buff) {
        let speechProps;
        const speechPropsInstance = (buff.getSpanInstanceOf(OutputSpeechProperties));
        if (!speechPropsInstance) {
            speechProps = this.initialSpeechProps_;
        }
        else {
            for (const [key, value] of Object.entries(this.initialSpeechProps_)) {
                if (speechPropsInstance.properties[key] === undefined) {
                    speechPropsInstance.properties[key] = value;
                }
            }
            speechProps = new TtsSpeechProperties(speechPropsInstance.properties);
        }
        speechProps.category = this.speechCategory_;
        speechProps.startCallback = () => {
            const actions = buff.getSpansInstanceOf(OutputAction);
            if (actions) {
                actions.forEach(action => action.run());
            }
        };
        return speechProps;
    }
    sendBraille_() {
        const buff = BrailleOutput.mergeSpans(this.brailleOutput_.buffer);
        const selSpan = buff.getSpanInstanceOf(OutputSelectionSpan);
        let startIndex = -1;
        let endIndex = -1;
        if (selSpan) {
            const valueStart = buff.getSpanStart(selSpan);
            const valueEnd = buff.getSpanEnd(selSpan);
            startIndex = valueStart + selSpan.startIndex;
            endIndex = valueStart + selSpan.endIndex;
            try {
                buff.setSpan(new ValueSpan(0), valueStart, valueEnd);
                buff.setSpan(new ValueSelectionSpan(), startIndex, endIndex);
            }
            catch (e) {
                console.log(e);
            }
        }
        const output = new NavBraille({ text: buff, startIndex, endIndex });
        ChromeVox.braille.write(output);
        this.brailleOutput_.formatLog.commitLogs();
    }
    sendSpeech_() {
        let queueMode = this.determineQueueMode_();
        let encounteredNonWhitespace = false;
        for (let i = 0; i < this.speechBuffer_.length; i++) {
            const buff = this.speechBuffer_[i];
            const text = buff.toString();
            // Consider empty strings as non-whitespace; they often have earcons
            // associated with them, so need to be "spoken".
            const isNonWhitespace = text === '' || /\S+/.test(text);
            encounteredNonWhitespace = isNonWhitespace || encounteredNonWhitespace;
            // Skip whitespace if we've already encountered non-whitespace. This
            // prevents output like 'foo', 'space', 'bar'.
            if (!isNonWhitespace && encounteredNonWhitespace) {
                continue;
            }
            const speechProps = this.getSpeechPropsForBuff_(buff);
            if (i === this.speechBuffer_.length - 1) {
                speechProps.endCallback = this.speechEndCallback_;
            }
            let finalSpeech = buff.toString();
            for (const text in this.replacements_) {
                finalSpeech = finalSpeech.replace(text, this.replacements_[text]);
            }
            ChromeVox.tts.speak(finalSpeech, queueMode, speechProps);
            // Skip resetting |queueMode| if the |text| is empty. If we don't do this,
            // and the tts engine doesn't generate a callback, we might not properly
            // flush.
            if (text !== '') {
                queueMode = QueueMode.QUEUE;
            }
        }
        this.speechFormatLog_.commitLogs();
    }
    /**
     * @param rhs Object to compare.
     * @return True if this object is equal to |rhs|.
     */
    equals(rhs) {
        if (this.speechBuffer_.length !== rhs.speechBuffer_.length) {
            return false;
        }
        for (let i = 0; i < this.speechBuffer_.length; i++) {
            if (this.speechBuffer_[i].toString() !==
                rhs.speechBuffer_[i].toString()) {
                return false;
            }
        }
        return this.brailleOutput_.equals(rhs.brailleOutput_);
    }
    render(range, prevRange, type, buff, formatLog, _optionalArgs) {
        if (prevRange && !prevRange.isValid()) {
            prevRange = undefined;
        }
        // Scan all ancestors to get the value of |contextOrder|.
        let parent = range.start.node;
        const prevParent = prevRange ? prevRange.start.node : parent;
        if (!parent || !prevParent) {
            return;
        }
        while (parent) {
            // TODO(b/314203187): Determine if not null assertion is acceptable.
            if (parent.role === RoleType$d.WINDOW) {
                break;
            }
            // TODO(b/314203187): Determine if not null assertion is acceptable.
            if (OutputRoleInfo[parent.role] &&
                OutputRoleInfo[parent.role]?.contextOrder) {
                // TODO(b/314203187): Determine if not null assertion is acceptable.
                this.contextOrder_ =
                    OutputRoleInfo[parent.role]?.contextOrder || this.contextOrder_;
                break;
            }
            parent = parent.parent;
        }
        if (range.isSubNode()) {
            this.subNode_(range, prevRange, type, buff, formatLog);
        }
        else {
            this.range_(range, prevRange, type, buff, formatLog, _optionalArgs);
        }
        this.hint_(range, AutomationUtil.getUniqueAncestors(prevParent, range.start.node), type, buff, formatLog);
    }
    range_(range, prevRange, type, rangeBuff, formatLog, _optionalArgs) {
        if (!range.start.node || !range.end.node) {
            return;
        }
        if (!prevRange && range.start.node.root) {
            prevRange = CursorRange.fromNode(range.start.node.root);
        }
        else if (!prevRange) {
            return;
        }
        const isForward = prevRange.compare(range) === Dir$8.FORWARD;
        const addContextBefore = this.contextOrder_ === OutputContextOrder.FIRST ||
            this.contextOrder_ === OutputContextOrder.FIRST_AND_LAST ||
            (this.contextOrder_ === OutputContextOrder.DIRECTED &&
                isForward);
        const addContextAfter = this.contextOrder_ === OutputContextOrder.LAST ||
            this.contextOrder_ === OutputContextOrder.FIRST_AND_LAST ||
            (this.contextOrder_ === OutputContextOrder.DIRECTED &&
                !isForward);
        const preferStartOrEndAncestry = this.contextOrder_ === OutputContextOrder.FIRST_AND_LAST;
        let prevNode = prevRange.start.node;
        let node = range.start.node;
        const formatNodeAndAncestors = (node, prevNode) => {
            const buff = [];
            if (addContextBefore) {
                this.ancestry_(node, prevNode, type, buff, formatLog, { preferStart: preferStartOrEndAncestry });
            }
            this.formatNode(node, prevNode, type, buff, formatLog);
            if (addContextAfter) {
                this.ancestry_(node, prevNode, type, buff, formatLog, { preferEnd: preferStartOrEndAncestry });
            }
            if (node.activeDescendant?.location) {
                this.locations_.push(node.activeDescendant.location);
            }
            else if (node.location) {
                this.locations_.push(node.location);
            }
            return buff;
        };
        let lca = null;
        if (range.start.node !== range.end.node) {
            lca = AutomationUtil.getLeastCommonAncestor(range.end.node, range.start.node);
        }
        if (addContextAfter) {
            prevNode = lca || prevNode;
        }
        // Do some bookkeeping to see whether this range partially covers node(s) at
        // its endpoints.
        let hasPartialNodeStart = false;
        let hasPartialNodeEnd = false;
        if (AutomationPredicate.selectableText(range.start.node) &&
            range.start.index > 0) {
            hasPartialNodeStart = true;
        }
        // TODO(b/314203187): Determine if not null assertion is acceptable.
        if (AutomationPredicate.selectableText(range.end.node) &&
            range.end.index >= 0 && range.end.index < range.end.node.name.length) {
            hasPartialNodeEnd = true;
        }
        let pred;
        if (range.isInlineText()) {
            pred = AutomationPredicate.leaf;
        }
        else if (hasPartialNodeStart || hasPartialNodeEnd) {
            pred = AutomationPredicate.selectableText;
        }
        else {
            pred = AutomationPredicate.object;
        }
        // Computes output for nodes (including partial subnodes) between endpoints
        // of |range|.
        while (node && range.end.node &&
            AutomationUtil.getDirection(node, range.end.node) === Dir$8.FORWARD) {
            if (hasPartialNodeStart && node === range.start.node) {
                const rangeStartNodeName = range.start.node.name;
                const nodeName = node.name;
                if (range.start.index !== rangeStartNodeName?.length) {
                    const partialRange = new CursorRange(new Cursor(node, range.start.index), 
                    // TODO(b/314203187): Determine if not null assertion is
                    // acceptable.
                    new Cursor(node, nodeName?.length, { preferNodeStartEquivalent: true }));
                    this.subNode_(partialRange, prevRange, type, rangeBuff, formatLog);
                }
            }
            else if (hasPartialNodeEnd && node === range.end.node) {
                if (range.end.index !== 0) {
                    const partialRange = new CursorRange(new Cursor(node, 0), new Cursor(node, range.end.index));
                    this.subNode_(partialRange, prevRange, type, rangeBuff, formatLog);
                }
            }
            else {
                rangeBuff.push.apply(rangeBuff, formatNodeAndAncestors(node, prevNode));
            }
            // End early if the range is just a single node.
            if (range.start.node === range.end.node) {
                break;
            }
            prevNode = node;
            node = AutomationUtil.findNextNode(node, Dir$8.FORWARD, pred, { root: r => r === lca }) ||
                prevNode;
            // Reached a boundary.
            if (node === prevNode) {
                break;
            }
        }
        // Finally, add on ancestry announcements, if needed.
        if (addContextAfter) {
            // No lca; the range was already fully described.
            if (lca == null || !prevRange.start.node) {
                return;
            }
            // Since the lca itself needs to be part of the ancestry output, use its
            // first child as a target.
            const target = lca.firstChild || lca;
            this.ancestry_(target, prevRange.start.node, type, rangeBuff, formatLog);
        }
    }
    ancestry_(node, prevNode, type, buff, formatLog, optionalArgs) {
        if (!SettingsManager.get('useVerboseMode')) {
            return;
        }
        if (node.role && OutputRoleInfo[node.role] &&
            OutputRoleInfo[node.role]?.ignoreAncestry) {
            return;
        }
        const info = new OutputAncestryInfo(node, prevNode, Boolean(optionalArgs?.suppressStartEndAncestry));
        // Enter, leave ancestry.
        this.ancestryHelper_({
            node,
            prevNode,
            buff,
            formatLog,
            type,
            ancestors: info.leaveAncestors,
            navigationType: OutputNavigationType.LEAVE,
            exclude: [...info.enterAncestors, node],
        });
        this.ancestryHelper_({
            node,
            prevNode,
            buff,
            formatLog,
            type,
            ancestors: info.enterAncestors,
            navigationType: OutputNavigationType.ENTER,
            excludePreviousAncestors: true,
        });
        if (optionalArgs?.suppressStartEndAncestry) {
            return;
        }
        // Start of, end of ancestry.
        if (!optionalArgs?.preferEnd) {
            this.ancestryHelper_({
                node,
                prevNode,
                buff,
                formatLog,
                type,
                ancestors: info.startAncestors,
                navigationType: OutputNavigationType.START_OF,
                excludePreviousAncestors: true,
            });
        }
        if (!optionalArgs?.preferStart) {
            this.ancestryHelper_({
                node,
                prevNode,
                buff,
                formatLog,
                type,
                ancestors: info.endAncestors,
                navigationType: OutputNavigationType.END_OF,
                exclude: [...info.startAncestors].concat(node),
            });
        }
    }
    ancestryHelper_(args) {
        let { prevNode, buff, formatLog, type, ancestors, navigationType } = args;
        const excludeRoles = args.exclude ? new Set(args.exclude.map(node => node.role)) : new Set();
        // Customize for braille node annotations.
        const originalBuff = buff;
        for (let j = ancestors.length - 1, formatNode; (formatNode = ancestors[j]); j--) {
            // TODO(b/314203187): Determine if not null assertion is acceptable.
            const roleInfo = OutputRoleInfo[formatNode.role] || {};
            if (!roleInfo?.verboseAncestry &&
                (excludeRoles.has(formatNode.role) ||
                    (args.excludePreviousAncestors &&
                        this.formattedAncestors_.has(formatNode)))) {
                continue;
            }
            const rule = new AncestryOutputRule(type, formatNode.role, navigationType, this.formatAsBraille);
            if (!rule.defined) {
                continue;
            }
            if (this.formatAsBraille) {
                buff = [];
                formatLog.bufferClear();
            }
            excludeRoles.add(formatNode.role);
            formatLog.writeRule(rule.specifier);
            this.formattedAncestors_.add(formatNode);
            OutputFormatter.format(this, {
                node: formatNode,
                outputFormat: rule.enterFormat,
                outputBuffer: buff,
                outputFormatLogger: formatLog,
                opt_prevNode: prevNode,
            });
            if (this.formatAsBraille && buff.length) {
                const nodeSpan = BrailleOutput.mergeSpans(buff);
                nodeSpan.setSpan(new OutputNodeSpan(formatNode), 0, nodeSpan.length);
                originalBuff.push(nodeSpan);
            }
        }
    }
    formatNode(node, prevNode, type, buff, formatLog) {
        const originalBuff = buff;
        if (this.formatOptions_.braille) {
            buff = [];
            formatLog.bufferClear();
        }
        const rule = new OutputRule(type);
        rule.output = OutputFormatType.SPEAK;
        rule.populateRole(node.role, rule.output);
        if (this.formatOptions_.braille) {
            // Overwrite rule by braille rule if exists.
            if (rule.populateRole(node.role, OutputFormatType.BRAILLE)) {
                rule.output = OutputFormatType.BRAILLE;
            }
        }
        formatLog.writeRule(rule.specifier);
        OutputFormatter.format(this, {
            node,
            outputFormat: rule.formatString,
            outputBuffer: buff,
            outputFormatLogger: formatLog,
            opt_prevNode: prevNode,
        });
        // Restore braille and add an annotation for this node.
        if (this.formatOptions_.braille) {
            const nodeSpan = BrailleOutput.mergeSpans(buff);
            nodeSpan.setSpan(new OutputNodeSpan(node), 0, nodeSpan.length);
            originalBuff.push(nodeSpan);
        }
    }
    subNode_(range, prevRange, type, buff, formatLog) {
        if (!prevRange) {
            prevRange = range;
        }
        const dir = CursorRange.getDirection(prevRange, range);
        const node = range.start.node;
        const prevNode = prevRange.getBound(dir).node;
        if (!node || !prevNode) {
            return;
        }
        let options = { annotation: ['name'], isUnique: true };
        const rangeStart = range.start.index;
        const rangeEnd = range.end.index;
        if (this.formatOptions_.braille) {
            options = this.brailleOutput_.subNode(range, options);
        }
        // Intentionally skip subnode output for
        // outputTypes.OutputContextOrder.DIRECTED.
        if (this.contextOrder_ === OutputContextOrder.FIRST ||
            (this.contextOrder_ === OutputContextOrder.FIRST_AND_LAST &&
                range.start.index === 0)) {
            this.ancestry_(node, prevNode, type, buff, formatLog, { preferStart: true });
        }
        const earcon = this.findEarcon(node, prevNode);
        if (earcon) {
            options.annotation.push(earcon);
        }
        let text = '';
        // TODO(b/314203187): Determine if not null assertion is acceptable.
        if (this.formatOptions_.braille && !node.state[StateType$e.EDITABLE]) {
            // In braille, we almost always want to show the entire contents and
            // simply place the cursor under the SelectionSpan we set above.
            text = range.start.getText();
        }
        else {
            // This is output for speech or editable braille.
            text = range.start.getText().substring(rangeStart, rangeEnd);
        }
        if (SettingsManager.get('languageSwitching')) {
            this.assignLocaleAndAppend(text, node, buff, options);
        }
        else {
            this.append(buff, text, options);
        }
        formatLog.write('subNode_: ' + text + '\n');
        if (this.contextOrder_ === OutputContextOrder.LAST ||
            (this.contextOrder_ === OutputContextOrder.FIRST_AND_LAST &&
                range.end.index === range.end.getText().length)) {
            this.ancestry_(node, prevNode, type, buff, formatLog, { preferEnd: true });
        }
        range.start.node.boundsForRange(rangeStart, rangeEnd, loc => {
            if (loc) {
                this.locations_.push(loc);
            }
        });
    }
    /**
     * Renders all hints for the given |range|.
     *
     * Add new hints to either method computeHints_ or computeDelayedHints_. Hints
     * are outputted in order, so consider the relative priority of any new
     * additions. Rendering processes these two methods in order. The only
     * distinction is a small delay gets introduced before the first hint in
     * |computeDelayedHints_|.
     *
     * @param buff receives the rendered output.
     */
    hint_(range, uniqueAncestors, type, buff, formatLog) {
        if (!this.enableHints_ || !SettingsManager.get('useVerboseMode')) {
            return;
        }
        // No hints for alerts, which can be targeted at controls.
        if (type === EventType$b.ALERT) {
            return;
        }
        // Hints are not yet specialized for braille.
        if (this.formatOptions_.braille) {
            return;
        }
        const node = range.start.node;
        if (node.restriction === chrome.automation.Restriction.DISABLED) {
            return;
        }
        const msgs = Output.computeHints_(node, uniqueAncestors);
        const delayedMsgs = Output.computeDelayedHints_(node, uniqueAncestors);
        if (delayedMsgs.length > 0) {
            delayedMsgs[0].props = new OutputSpeechProperties();
            delayedMsgs[0].props.properties['delay'] = true;
        }
        const allMsgs = msgs.concat(delayedMsgs);
        for (const msg of allMsgs) {
            if (msg.msgId) {
                const text = Msgs.getMsg(msg.msgId, msg.subs);
                this.append(buff, text, { annotation: [msg.props] });
                formatLog.write('hint_: ' + text + '\n');
            }
            else if (msg.text) {
                this.append(buff, msg.text, { annotation: [msg.props] });
                formatLog.write('hint_: ' + msg.text + '\n');
            }
            else if (msg.outputFormat) {
                formatLog.write('hint_: ...');
                OutputFormatter.format(this, {
                    node,
                    outputFormat: msg.outputFormat,
                    outputBuffer: buff,
                    outputFormatLogger: formatLog,
                    opt_speechProps: msg.props,
                });
            }
            else {
                throw new Error('Unexpected hint: ' + msg);
            }
        }
    }
    /**
     * Internal helper to |hint_|. Returns a list of message hints.
     * Note that the above caller
     * expects one and only one key be set.
     * @return a list of message hints.
     */
    static computeHints_(node, uniqueAncestors) {
        const ret = [];
        if (node['errorMessage']) {
            ret.push({ outputFormat: '$node(errorMessage)' });
        }
        // Provide a hint for sort direction.
        let sortDirectionNode = node;
        while (sortDirectionNode && sortDirectionNode !== sortDirectionNode.root) {
            if (!sortDirectionNode.sortDirection) {
                sortDirectionNode = sortDirectionNode.parent;
                continue;
            }
            if (sortDirectionNode.sortDirection ===
                chrome.automation.SortDirectionType.ASCENDING) {
                ret.push({ msgId: 'sort_ascending' });
            }
            else if (sortDirectionNode.sortDirection ===
                chrome.automation.SortDirectionType.DESCENDING) {
                ret.push({ msgId: 'sort_descending' });
            }
            break;
        }
        let currentNode = node;
        let ancestorIndex = 0;
        do {
            if (currentNode.ariaCurrentState &&
                OutputPropertyMap['STATE'][currentNode.ariaCurrentState]) {
                ret.push({
                    msgId: OutputPropertyMap['STATE'][currentNode.ariaCurrentState],
                });
                break;
            }
            currentNode = uniqueAncestors[ancestorIndex++];
        } while (currentNode);
        return ret;
    }
    /**
     * Internal helper to |hint_|. Returns a list of message hints.
     * Note that the above caller
     * expects one and only one key be set.
     * @return a list of message hints.
     */
    static computeDelayedHints_(node, uniqueAncestors) {
        const ret = [];
        // TODO(b/314203187): Determine if not null assertion is acceptable.
        const nodeState = node.state;
        if (EventSource.get() === EventSourceType.TOUCH_GESTURE) {
            if (nodeState[StateType$e.EDITABLE]) {
                ret.push({
                    msgId: nodeState[StateType$e.FOCUSED] ? 'hint_is_editing' :
                        'hint_double_tap_to_edit',
                });
                return ret;
            }
            const isWithinVirtualKeyboard = AutomationUtil.getAncestors(node).find(n => n.role === RoleType$d.KEYBOARD);
            if (AutomationPredicate.clickable(node) && !isWithinVirtualKeyboard) {
                ret.push({
                    msgId: 'hint_actionable',
                    subs: [
                        Msgs.getMsg('action_double_tap', []),
                        node.doDefaultLabel ? node.doDefaultLabel :
                            Msgs.getMsg('label_activate', []),
                    ],
                });
            }
            const enteredVirtualKeyboard = uniqueAncestors.find(n => n.role === RoleType$d.KEYBOARD);
            if (enteredVirtualKeyboard) {
                ret.push({ msgId: 'hint_touch_type' });
            }
            return ret;
        }
        if (nodeState[StateType$e.EDITABLE] && nodeState[StateType$e.FOCUSED] &&
            (nodeState[StateType$e.MULTILINE] ||
                nodeState[StateType$e.RICHLY_EDITABLE])) {
            ret.push({ msgId: 'hint_search_within_text_field' });
        }
        if (node.placeholder) {
            ret.push({ text: node.placeholder });
        }
        // Invalid Grammar text.
        if (uniqueAncestors.find(AutomationPredicate.hasInvalidGrammarMarker)) {
            ret.push({ msgId: 'hint_invalid_grammar' });
        }
        // Invalid Spelling text.
        if (uniqueAncestors.find(AutomationPredicate.hasInvalidSpellingMarker)) {
            ret.push({ msgId: 'hint_invalid_spelling' });
        }
        // Only include tooltip as a hint as a last alternative. It may have been
        // included as the name or description previously. As a rule of thumb,
        // only include it if there's no name and no description.
        if (node.tooltip && !node.name && !node.description) {
            ret.push({ text: node.tooltip });
        }
        if (AutomationPredicate.checkable(node)) {
            ret.push({ msgId: 'hint_checkable' });
        }
        else if (AutomationPredicate.clickable(node)) {
            ret.push({
                msgId: 'hint_actionable',
                subs: [
                    Msgs.getMsg('action_search_plus_space', []),
                    node.doDefaultLabel ? node.doDefaultLabel :
                        Msgs.getMsg('label_activate', []),
                ],
            });
        }
        if (AutomationPredicate.longClickable(node)) {
            ret.push({
                msgId: 'hint_actionable',
                subs: [
                    Msgs.getMsg('action_search_plus_shift_plus_space', []),
                    node.longClickLabel ? node.longClickLabel :
                        Msgs.getMsg('label_long_click', []),
                ],
            });
        }
        if (node.autoComplete === 'list' || node.autoComplete === 'both' ||
            nodeState[StateType$e.AUTOFILL_AVAILABLE]) {
            ret.push({ msgId: 'hint_autocomplete_list' });
        }
        if (node.autoComplete === 'inline' || node.autoComplete === 'both') {
            ret.push({ msgId: 'hint_autocomplete_inline' });
        }
        if (node.customActions && node.customActions.length > 0) {
            ret.push({ msgId: 'hint_action' });
        }
        if (node.accessKey) {
            ret.push({ text: Msgs.getMsg('access_key', [node.accessKey]) });
        }
        // Ancestry based hints.
        let foundAncestor;
        if (uniqueAncestors.find(AutomationPredicate.table)) {
            ret.push({ msgId: 'hint_table' });
        }
        // This hint is not based on the role (it uses state), so we need to care
        // about ordering; prefer deepest ancestor first.
        if (foundAncestor = uniqueAncestors.reverse().find((AutomationPredicate.roles([RoleType$d.MENU, RoleType$d.MENU_BAR])))) {
            if (foundAncestor.state && foundAncestor.state['horizontal']) {
                ret.push({ msgId: 'hint_menu_horizontal' });
            }
            else {
                ret.push({ msgId: 'hint_menu' });
            }
        }
        if (uniqueAncestors.find(function (n) {
            return Boolean(n.details);
        })) {
            ret.push({ msgId: 'hint_details' });
        }
        return ret;
    }
    append(buff, value, optOptions) {
        optOptions = optOptions || { isUnique: false, annotation: [] };
        // Reject empty values without meaningful annotations.
        if ((!value || value.length === 0) &&
            optOptions.annotation.every(annotation => !(annotation instanceof OutputAction) &&
                !(annotation instanceof OutputSelectionSpan))) {
            return;
        }
        const spannableToAdd = new Spannable(value);
        optOptions.annotation.forEach(annotation => spannableToAdd.setSpan(annotation, 0, spannableToAdd.length));
        // |isUnique| specifies an annotation that cannot be duplicated.
        if (optOptions.isUnique) {
            const annotationSansNodes = optOptions.annotation.filter(annotation => !(annotation instanceof OutputNodeSpan));
            const alreadyAnnotated = buff.some((spannable) => {
                annotationSansNodes.some(annotation => {
                    if (!spannable.hasSpan(annotation)) {
                        return false;
                    }
                    const start = spannable.getSpanStart(annotation);
                    const end = spannable.getSpanEnd(annotation);
                    const substr = spannable.substring(start, end);
                    if (substr && value) {
                        return substr.toString() === value.toString();
                    }
                    else {
                        return false;
                    }
                });
            });
            if (alreadyAnnotated) {
                return;
            }
        }
        buff.push(spannableToAdd);
    }
    /**
     * @return found OutputAction or if not found, undefined/null.
     */
    findEarcon(node, optPrevNode) {
        if (node === optPrevNode) {
            // TODO(b/314203187): Determine if not null assertion is acceptable.
            return null;
        }
        if (this.formatOptions_.speech) {
            let earconFinder = node;
            let ancestors;
            if (optPrevNode) {
                ancestors = AutomationUtil.getUniqueAncestors(optPrevNode, node);
            }
            else {
                ancestors = AutomationUtil.getAncestors(node);
            }
            while (earconFinder = ancestors.pop()) {
                // TODO(b/314203187): Determine if not null assertion is acceptable.
                const role = earconFinder.role;
                const info = OutputRoleInfo[role];
                if (info && info.earcon) {
                    return new OutputEarconAction(info.earcon, node.location || undefined);
                }
                earconFinder = earconFinder.parent;
            }
        }
        // TODO(b/314203187): Determine if not null assertion is acceptable.
        return null;
    }
    /**
     * @returns a human friendly string with the contents of output.
     */
    toString() {
        return this.speechBuffer_.map(v => v.toString()).join(' ');
    }
    /**
     * Gets the spoken output with separator '|'.
     * @return Spannable containing spoken output with separator '|'.
     */
    get speechOutputForTest() {
        return this.speechBuffer_.reduce((prev, cur) => {
            if (prev === null) {
                return cur;
            }
            prev.append('|');
            prev.append(cur);
            return prev;
            // TODO(b/314203187): Determine if not null assertion is acceptable.
        }, null);
    }
    assignLocaleAndAppend(text, contextNode, buff, options) {
        const data = LocaleOutputHelper.instance.computeTextAndLocale(text, contextNode);
        const speechProps = new OutputSpeechProperties();
        speechProps.properties['lang'] = data.locale;
        this.append(buff, data.text, options);
        // Attach associated SpeechProperties if the buffer is
        // non-empty.
        if (buff.length > 0) {
            buff[buff.length - 1].setSpan(speechProps, 0, 0);
        }
    }
    shouldSuppress(token) {
        return this.suppressions_[token];
    }
    get useAuralStyle() {
        return this.formatOptions_.auralStyle;
    }
    get formatAsBraille() {
        return this.formatOptions_.braille;
    }
    get formatAsSpeech() {
        return this.formatOptions_.speech;
    }
}
TestImportManager.exportForTesting(Output);

// Copyright 2015 The Chromium Authors
// Use of this source code is governed by a BSD-style license that can be
// found in the LICENSE file.
/**
 * @fileoverview Basic facilities to handle events from a single automation
 * node.
 */
const ActionType$2 = chrome.automation.ActionType;
class BaseAutomationHandler {
    listeners_ = {};
    node_;
    /** Controls announcement of non-user-initiated events. */
    static announceActions = false;
    constructor(node) {
        this.node_ = node;
    }
    /** Adds an event listener to this handler. */
    addListener_(eventType, eventCallback) {
        if (this.listeners_[eventType]) {
            throw 'Listener already added: ' + eventType;
        }
        // Note: Keeping this bind lets us keep the addListener_ callsites simpler.
        const listener = this.makeListener_(eventCallback.bind(this));
        this.node_.addEventListener(eventType, listener, true);
        this.listeners_[eventType] = listener;
    }
    /** Removes all listeners from this handler. */
    removeAllListeners() {
        for (const type in this.listeners_) {
            const eventType = type;
            // TODO(b/314203187): Not null asserted, check that this is correct.
            this.node_.removeEventListener(eventType, this.listeners_[eventType], true);
        }
        this.listeners_ = {};
    }
    makeListener_(callback) {
        return (evt) => {
            if (this.willHandleEvent_(evt)) {
                return;
            }
            callback(evt);
            this.didHandleEvent_(evt);
        };
    }
    /**
     * Called before the event |evt| is handled.
     * @return True to skip processing this event.
     */
    willHandleEvent_(_evt) {
        return false;
    }
    /** Called after the event |evt| is handled. */
    didHandleEvent_(_evt) { }
    /** Provides all feedback once ChromeVox's focus changes. */
    onEventDefault(evt) {
        const node = evt.target;
        if (!node) {
            return;
        }
        // Decide whether to announce and sync this event.
        const prevRange = ChromeVoxRange.getCurrentRangeWithoutRecovery();
        if ((prevRange && !prevRange.requiresRecovery()) &&
            BaseAutomationHandler.disallowEventFromAction(evt)) {
            return;
        }
        ChromeVoxRange.set(CursorRange.fromNode(node));
        // Because Closure doesn't know this is non-null.
        if (!ChromeVoxRange.current) {
            return;
        }
        // Don't output if focused node hasn't changed. Allow focus announcements
        // when interacting via touch. Touch never sets focus without a double tap.
        if (prevRange && evt.type === 'focus' &&
            ChromeVoxRange.current.equalsWithoutRecovery(prevRange) &&
            EventSource.get() !== EventSourceType.TOUCH_GESTURE) {
            return;
        }
        const output = new Output();
        output.withRichSpeechAndBraille(ChromeVoxRange.current, prevRange ?? undefined, evt.type);
        output.go();
    }
    /**
     * Returns true if the event contains an action that should not be processed.
     */
    static disallowEventFromAction(evt) {
        return !BaseAutomationHandler.announceActions &&
            evt.eventFrom === 'action' &&
            evt.eventFromAction !==
                ActionType$2.DO_DEFAULT &&
            evt.eventFromAction !==
                ActionType$2.SHOW_CONTEXT_MENU;
    }
}
TestImportManager.exportForTesting(BaseAutomationHandler);

// Copyright 2022 The Chromium Authors
// Use of this source code is governed by a BSD-style license that can be
// found in the LICENSE file.
/**
 * @fileoverview Interface to prevent circular dependencies.
 */
class DesktopAutomationInterface extends BaseAutomationHandler {
}
(function (DesktopAutomationInterface) {
})(DesktopAutomationInterface || (DesktopAutomationInterface = {}));
TestImportManager.exportForTesting(DesktopAutomationInterface);

// Copyright 2022 The Chromium Authors
// Use of this source code is governed by a BSD-style license that can be
// found in the LICENSE file.
/**
 * Interface defining the methods of performing or modifying the performance of
 * commands.
 */
class CommandHandlerInterface {
}
(function (CommandHandlerInterface) {
})(CommandHandlerInterface || (CommandHandlerInterface = {}));
TestImportManager.exportForTesting(CommandHandlerInterface);

// Copyright 2016 The Chromium Authors
// Use of this source code is governed by a BSD-style license that can be
// found in the LICENSE file.
/**
 * @fileoverview ChromeVox braille commands.
 */
const IntentCommandType$3 = chrome.automation.IntentCommandType;
const IntentMoveDirectionType = chrome.automation.IntentMoveDirectionType;
const IntentTextBoundaryType$2 = chrome.automation.IntentTextBoundaryType;
const RoleType$c = chrome.automation.RoleType;
const StateType$d = chrome.automation.StateType;
class BrailleCommandHandler {
    static instance;
    bypassed_ = false;
    constructor() { }
    static init() {
        if (BrailleCommandHandler.instance) {
            throw new Error('BrailleCommandHandler cannot be instantiated more than once.');
        }
        BrailleCommandHandler.instance = new BrailleCommandHandler();
    }
    /**
     * Global setting for bypassing this handler.
     *
     * Used by LearnMode to capture the events and prevent the standard behavior,
     * in favor of reporting what that behavior would be.
     */
    static setBypass(state) {
        BrailleCommandHandler.instance.bypassed_ = state;
    }
    /**
     * Handles a braille command.
     * @return True if evt was processed.
     */
    static onBrailleKeyEvent(evt, content) {
        if (BrailleCommandHandler.instance.bypassed_) {
            // Prevent any handling of the event when bypassed.
            return true;
        }
        EventSource.set(EventSourceType.BRAILLE_KEYBOARD);
        // Try to restore to the last valid range.
        ChromeVoxRange.restoreLastValidRangeIfNeeded();
        // Note: panning within content occurs earlier in event dispatch.
        Output.forceModeForNextSpeechUtterance(QueueMode.FLUSH);
        switch (evt.command) {
            case BrailleKeyCommand.PAN_LEFT:
                CommandHandlerInterface.instance.onCommand(Command.PREVIOUS_OBJECT);
                break;
            case BrailleKeyCommand.PAN_RIGHT:
                CommandHandlerInterface.instance.onCommand(Command.NEXT_OBJECT);
                break;
            case BrailleKeyCommand.LINE_UP:
                CommandHandlerInterface.instance.onCommand(Command.PREVIOUS_LINE);
                break;
            case BrailleKeyCommand.LINE_DOWN:
                CommandHandlerInterface.instance.onCommand(Command.NEXT_LINE);
                break;
            case BrailleKeyCommand.TOP:
                CommandHandlerInterface.instance.onCommand(Command.JUMP_TO_TOP);
                break;
            case BrailleKeyCommand.BOTTOM:
                CommandHandlerInterface.instance.onCommand(Command.JUMP_TO_BOTTOM);
                break;
            case BrailleKeyCommand.ROUTING:
                // TODO(b/314203187): Not null asserted, check that this is correct.
                const textEditHandler = DesktopAutomationInterface.instance.textEditHandler;
                textEditHandler?.injectInferredIntents([{
                        command: IntentCommandType$3.MOVE_SELECTION,
                        textBoundary: IntentTextBoundaryType$2.CHARACTER,
                        moveDirection: IntentMoveDirectionType.FORWARD,
                    }]);
                BrailleCommandHandler.onRoutingCommand_(content.text, 
                // Cast ok since displayPosition is always defined in this case.
                evt.displayPosition);
                break;
            case BrailleKeyCommand.CHORD:
                if (!evt.brailleDots) {
                    return false;
                }
                const command = BrailleCommandData.getCommand(evt.brailleDots);
                if (command) {
                    if (BrailleCommandHandler.onEditCommand_(command)) {
                        CommandHandlerInterface.instance.onCommand(command);
                    }
                }
                break;
            default:
                return false;
        }
        return true;
    }
    static onRoutingCommand_(text, position) {
        let actionNodeSpan = null;
        let selectionSpan = null;
        const selSpans = text.getSpansInstanceOf(OutputSelectionSpan);
        const nodeSpans = text.getSpansInstanceOf(OutputNodeSpan);
        for (const selSpan of selSpans) {
            if (text.getSpanStart(selSpan) <= position &&
                position < text.getSpanEnd(selSpan)) {
                selectionSpan = selSpan;
                break;
            }
        }
        let interval;
        for (const nodeSpan of nodeSpans) {
            const intervals = text.getSpanIntervals(nodeSpan);
            const tempInterval = intervals.find(inner => inner.start <= position && position <= inner.end);
            if (tempInterval) {
                actionNodeSpan = nodeSpan;
                interval = tempInterval;
            }
        }
        if (!actionNodeSpan) {
            return;
        }
        let actionNode = actionNodeSpan.node;
        const offset = actionNodeSpan.offset;
        if (actionNode.role === RoleType$c.INLINE_TEXT_BOX) {
            actionNode = actionNode.parent;
        }
        actionNode.doDefault();
        if (actionNode.role !== RoleType$c.STATIC_TEXT &&
            !actionNode.state[StateType$d.EDITABLE]) {
            return;
        }
        if (!selectionSpan) {
            selectionSpan = actionNodeSpan;
        }
        if (actionNode.state.richlyEditable) {
            const start = interval?.start ?? text.getSpanStart(selectionSpan);
            const targetPosition = position - start + offset;
            chrome.automation.setDocumentSelection({
                anchorObject: actionNode,
                anchorOffset: targetPosition,
                focusObject: actionNode,
                focusOffset: targetPosition,
            });
        }
        else {
            const start = text.getSpanStart(selectionSpan);
            const targetPosition = position - start + offset;
            actionNode.setSelection(targetPosition, targetPosition);
        }
    }
    /**
     * Customizes ChromeVox commands when issued from a braille display while
     * within editable text.
     * @return True if the command should propagate.
     */
    static onEditCommand_(command) {
        const current = ChromeVoxRange.current;
        const state = current?.start?.node?.state ?? {};
        if (ChromeVoxPrefs.isStickyModeOn() || !state[StateType$d.EDITABLE]) {
            return true;
        }
        // TODO(b/314203187): Not null asserted, check that this is correct.
        const textEditHandler = DesktopAutomationInterface.instance.textEditHandler;
        const editable = AutomationUtil.getEditableRoot(current.start.node);
        if (!editable || editable !== textEditHandler?.node) {
            return true;
        }
        const isMultiline = AutomationPredicate.multiline(editable);
        switch (command) {
            case 'forceClickOnCurrentItem':
                EventGenerator.sendKeyPress(KeyCode.RETURN);
                break;
            case 'previousCharacter':
                EventGenerator.sendKeyPress(KeyCode.LEFT);
                break;
            case 'nextCharacter':
                EventGenerator.sendKeyPress(KeyCode.RIGHT);
                break;
            case 'previousWord':
                EventGenerator.sendKeyPress(KeyCode.LEFT, { ctrl: true });
                break;
            case 'nextWord':
                EventGenerator.sendKeyPress(KeyCode.RIGHT, { ctrl: true });
                break;
            case 'previousObject':
            case 'previousLine':
                if (!isMultiline || textEditHandler.isSelectionOnFirstLine()) {
                    return true;
                }
                EventGenerator.sendKeyPress(KeyCode.UP);
                break;
            case 'nextObject':
            case 'nextLine':
                if (!isMultiline) {
                    return true;
                }
                if (textEditHandler.isSelectionOnLastLine()) {
                    textEditHandler.moveToAfterEditText();
                    return false;
                }
                EventGenerator.sendKeyPress(KeyCode.DOWN);
                break;
            case 'previousGroup':
                EventGenerator.sendKeyPress(KeyCode.UP, { ctrl: true });
                break;
            case 'nextGroup':
                EventGenerator.sendKeyPress(KeyCode.DOWN, { ctrl: true });
                break;
            default:
                return true;
        }
        return false;
    }
}
TestImportManager.exportForTesting(BrailleCommandHandler);

// Copyright 2015 The Chromium Authors
// Use of this source code is governed by a BSD-style license that can be
// found in the LICENSE file.
class ChromeVoxState {
    static instance;
    static position = {};
    static resolveReadyPromise_;
    static readyPromise_ = new Promise(resolve => ChromeVoxState.resolveReadyPromise_ = resolve);
    static ready() {
        return ChromeVoxState.readyPromise_;
    }
}
TestImportManager.exportForTesting(ChromeVoxState);

// Copyright 2014 The Chromium Authors
// Use of this source code is governed by a BSD-style license that can be
// found in the LICENSE file.
/**
 * @fileoverview Defines a global object that holds references to the three
 * different output engines.
 */
/** A central access point for the different modes of output. */
class ChromeVox {
    static braille;
    static earcons;
    static tts;
}
// Braille bridge functions.
BridgeHelper.registerHandler(BridgeConstants.Braille.TARGET, BridgeConstants.Braille.Action.BACK_TRANSLATE, (cells) => Promise.resolve(ChromeVox.braille?.backTranslate(cells)));
BridgeHelper.registerHandler(BridgeConstants.Braille.TARGET, BridgeConstants.Braille.Action.SET_BYPASS, async (bypass) => {
    await ChromeVoxState.ready();
    BrailleCommandHandler.setBypass(bypass);
});
BridgeHelper.registerHandler(BridgeConstants.Braille.TARGET, BridgeConstants.Braille.Action.PAN_LEFT, () => ChromeVox.braille?.panLeft());
BridgeHelper.registerHandler(BridgeConstants.Braille.TARGET, BridgeConstants.Braille.Action.PAN_RIGHT, () => ChromeVox.braille?.panRight());
BridgeHelper.registerHandler(BridgeConstants.Braille.TARGET, BridgeConstants.Braille.Action.WRITE, (text) => ChromeVox.braille?.write(new NavBraille({ text: new Spannable(text) })));
// Earcon bridge functions.
BridgeHelper.registerHandler(BridgeConstants.Earcons.TARGET, BridgeConstants.Earcons.Action.CANCEL_EARCON, (earconId) => ChromeVox.earcons?.cancelEarcon(earconId));
BridgeHelper.registerHandler(BridgeConstants.Earcons.TARGET, BridgeConstants.Earcons.Action.PLAY_EARCON, (earconId) => ChromeVox.earcons?.playEarcon(earconId));
// TTS bridge functions.
BridgeHelper.registerHandler(BridgeConstants.TtsBackground.TARGET, BridgeConstants.TtsBackground.Action.SPEAK, (text, queueMode, properties) => ChromeVox.tts?.speak(text, queueMode, properties));
TestImportManager.exportForTesting(['ChromeVox', ChromeVox]);

// Copyright 2023 The Chromium Authors
// Use of this source code is governed by a BSD-style license that can be
// found in the LICENSE file.
/**
 * @fileoverview Classes that handle the ChromeVox range.
 */
const Dir$7 = constants.Dir;
const RoleType$b = chrome.automation.RoleType;
const StateType$c = chrome.automation.StateType;
const Action$2 = BridgeConstants.ChromeVoxRange.Action;
const TARGET$1 = BridgeConstants.ChromeVoxRange.TARGET;
/**
 * An interface implemented by objects to observe ChromeVox range changes.
 * TODO(b/346347267): Convert to an interface post-TypeScript migration.
 */
class ChromeVoxRangeObserver {
}
/** Handles tracking of and changes to the ChromeVox range. */
class ChromeVoxRange {
    current_ = null;
    pageSel_ = null;
    previous_ = null;
    static observers_ = [];
    static instance;
    constructor() { }
    static init() {
        if (ChromeVoxRange.instance) {
            throw new Error('Cannot create more than one ChromeVoxRange');
        }
        ChromeVoxRange.instance = new ChromeVoxRange();
        BridgeHelper.registerHandler(TARGET$1, Action$2.CLEAR_CURRENT_RANGE, () => ChromeVoxRange.set(null));
    }
    static get current() {
        if (ChromeVoxRange.instance.current_?.isValid()) {
            return ChromeVoxRange.instance.current_;
        }
        return null;
    }
    static clearSelection() {
        ChromeVoxRange.instance.pageSel_ = null;
    }
    /** Return the current range, but focus recovery is not applied to it. */
    static getCurrentRangeWithoutRecovery() {
        return ChromeVoxRange.instance.current_;
    }
    /**
     * Check for loss of focus which results in us invalidating our current range.
     */
    static maybeResetFromFocus() {
        ChromeVoxRange.instance.maybeResetFromFocus_();
    }
    /**
     * Navigate to the given range - it both sets the range and outputs it.
     * @param {!CursorRange} range The new range.
     * @param {boolean=} opt_focus Focus the range; defaults to true.
     * @param {TtsSpeechProperties=} opt_speechProps Speech properties.
     * @param {boolean=} opt_skipSettingSelection If true, does not set
     *     the selection, otherwise it does by default.
     * @param {boolean=} opt_skipOutput If true, does not send outputs,
     *     otherwise it does by default.
     */
    static navigateTo(range, focus, speechProps, skipSettingSelection, skipOutput) {
        ChromeVoxRange.instance.navigateTo_(range, focus, speechProps, skipSettingSelection, skipOutput);
    }
    /** Restores the last valid ChromeVox range. */
    static restoreLastValidRangeIfNeeded() {
        ChromeVoxRange.instance.restoreLastValidRangeIfNeeded_();
    }
    static set(newRange, fromEditing) {
        ChromeVoxRange.instance.set_(newRange, fromEditing);
    }
    /**
     * @return true if the selection is toggled on, false if it is toggled off.
     */
    static toggleSelection() {
        return ChromeVoxRange.instance.toggleSelection_();
    }
    // ================= Observer Functions =================
    static addObserver(observer) {
        ChromeVoxRange.observers_.push(observer);
    }
    static removeObserver(observer) {
        const index = ChromeVoxRange.observers_.indexOf(observer);
        if (index > -1) {
            ChromeVoxRange.observers_.splice(index, 1);
        }
    }
    // ================= Private Methods =================
    /**
     * Check for loss of focus which results in us invalidating our current
     * range. Note the getFocus() callback is synchronous, so the focus will be
     * updated when this function returns (despite being technicallly a separate
     * function call). Note: do not convert this method to async, as it would
     * change the execution order described above.
     */
    maybeResetFromFocus_() {
        chrome.automation.getFocus((focus) => {
            const cur = ChromeVoxRange.current;
            // If the current node is not valid and there's a current focus:
            if (cur && !cur.isValid() && focus) {
                ChromeVoxRange.set(CursorRange.fromNode(focus));
            }
            // If there's no focused node:
            if (!focus) {
                ChromeVoxRange.set(null);
                return;
            }
            // This case detects when TalkBack (in ARC++) is enabled (which also
            // covers when the ARC++ window is active). Clear the ChromeVox range
            // so keys get passed through for ChromeVox commands.
            // TODO(b/314203187): Not null asserted, check that this is correct.
            if (ChromeVoxState.instance.talkBackEnabled &&
                // This additional check is not strictly necessary, but we use it to
                // ensure we are never inadvertently losing focus. ARC++ windows set
                // "focus" on a root view.
                focus.role === RoleType$b.CLIENT) {
                ChromeVoxRange.set(null);
            }
        });
    }
    /**
     * Navigate to the given range - it both sets the range and outputs it.
     * @param focus Focus the range; defaults to true.
     */
    navigateTo_(range, focus, speechProps, skipSettingSelection, skipOutput) {
        focus = focus ?? true;
        speechProps = speechProps ?? new TtsSpeechProperties();
        skipSettingSelection = skipSettingSelection ?? false;
        skipOutput = skipOutput ?? false;
        const prevRange = ChromeVoxRange.getCurrentRangeWithoutRecovery();
        // Specialization for math output.
        if (MathHandler.init(range)) {
            const mathWasSpoken = MathHandler.instance.speak();
            if (mathWasSpoken) {
                // Set the visual focus bounds onto the Math node.
                FocusBounds.set([MathHandler.instance.node().location]);
            }
            skipOutput ||= mathWasSpoken;
            focus = false;
        }
        if (focus) {
            this.setFocusToRange_(range, prevRange);
        }
        ChromeVoxRange.set(range);
        const o = new Output();
        let selectedRange;
        let msg;
        if (this.pageSel_?.isValid() && range.isValid()) {
            // Suppress hints.
            o.withoutHints();
            // Selection across roots isn't supported.
            const pageRootStart = this.pageSel_.start.node.root;
            const pageRootEnd = this.pageSel_.end.node.root;
            const curRootStart = range.start.node.root;
            const curRootEnd = range.end.node.root;
            // Deny crossing over the start of the page selection and roots.
            if (pageRootStart !== pageRootEnd || pageRootStart !== curRootStart ||
                pageRootEnd !== curRootEnd) {
                o.format('@end_selection');
                // TODO(b/314203187): Not null asserted, check that this is correct.
                DesktopAutomationInterface.instance.ignoreDocumentSelectionFromAction(false);
                this.pageSel_ = null;
            }
            else {
                // Expand or shrink requires different feedback.
                // Page sel is the only place in ChromeVox where we used directed
                // selections. It is important to keep track of the directedness in
                // places, but when comparing to other ranges, take the undirected
                // range.
                const dir = this.pageSel_.normalize().compare(range);
                if (dir) {
                    // Directed expansion.
                    msg = '@selected';
                }
                else {
                    // Directed shrink.
                    msg = '@unselected';
                    selectedRange = prevRange;
                }
                const wasBackwardSel = this.pageSel_.start.compare(this.pageSel_.end) === Dir$7.BACKWARD ||
                    dir === Dir$7.BACKWARD;
                this.pageSel_ = new CursorRange(this.pageSel_.start, wasBackwardSel ? range.start : range.end);
                this.pageSel_.select();
            }
        }
        else if (!skipSettingSelection) {
            // Ensure we don't select the editable when we first encounter it.
            let lca = null;
            if (range.start.node && prevRange?.start.node) {
                lca = AutomationUtil.getLeastCommonAncestor(prevRange.start.node, range.start.node);
            }
            // TODO(b/314203187): Not null asserted, check that this is correct.
            if (!lca || lca.state[StateType$c.EDITABLE] ||
                !range.start.node.state[StateType$c.EDITABLE]) {
                range.select();
            }
        }
        o.withRichSpeechAndBraille(selectedRange ?? range, prevRange ?? undefined, OutputCustomEvent.NAVIGATE)
            .withInitialSpeechProperties(speechProps);
        if (msg) {
            o.format(msg);
        }
        if (!skipOutput) {
            o.go();
        }
    }
    notifyObservers_(range, fromEditing) {
        for (const observer of ChromeVoxRange.observers_) {
            observer.onCurrentRangeChanged(range, fromEditing);
        }
    }
    restoreLastValidRangeIfNeeded_() {
        // Never restore range when TalkBack is enabled as commands such as
        // Search+Left, go directly to TalkBack.
        // TODO(b/314203187): Not null asserted, check that this is correct.
        if (ChromeVoxState.instance.talkBackEnabled) {
            return;
        }
        if (!this.current_?.isValid()) {
            this.current_ = this.previous_;
        }
    }
    set_(newRange, fromEditing) {
        // Clear anything that was frozen on the braille display whenever
        // the user navigates.
        ChromeVox.braille.thaw();
        // There's nothing to be updated in this case.
        if ((!newRange && !this.current_) || (newRange && !newRange.isValid())) {
            FocusBounds.set([]);
            return;
        }
        this.previous_ = this.current_;
        this.current_ = newRange;
        // Check math state in case we've moved away from a math node.
        MathHandler.checkInstance(newRange);
        this.notifyObservers_(newRange, fromEditing);
        if (!this.current_) {
            FocusBounds.set([]);
            return;
        }
        const start = this.current_.start.node;
        start.makeVisible();
        chrome.metricsPrivate.recordBoolean('Accessibility.ScreenReader.ScrollToImage', start.role === RoleType$b.IMAGE);
        start.setAccessibilityFocus();
        const root = AutomationUtil.getTopLevelRoot(start);
        if (!root || root.role === RoleType$b.DESKTOP || root === start) {
            return;
        }
        // TODO(b/314203187): Not null asserted, check that this is correct.
        const loc = start.unclippedLocation;
        const x = loc.left + loc.width / 2;
        const y = loc.top + loc.height / 2;
        const position = { x, y };
        let url = root.docUrl;
        url = url.substring(0, url.indexOf('#')) || url;
        ChromeVoxState.position[url] = position;
    }
    setFocusToRange_(range, prevRange) {
        const start = range.start.node;
        const end = range.end.node;
        // First, see if we've crossed a root. Remove once webview handles focus
        // correctly.
        if (prevRange && prevRange.start.node && start) {
            const entered = AutomationUtil.getUniqueAncestors(prevRange.start.node, start);
            const isPluginOrIframe = AutomationPredicate.roles([RoleType$b.PLUGIN_OBJECT, RoleType$b.IFRAME]);
            entered.filter(isPluginOrIframe).forEach((container) => {
                // TODO(b/314203187): Not null asserted, check that this is correct.
                if (!container.state[StateType$c.FOCUSED]) {
                    container.focus();
                }
            });
        }
        // TODO(b/314203187): Not null asserted, check that this is correct.
        if (start.state[StateType$c.FOCUSED] || end.state[StateType$c.FOCUSED]) {
            return;
        }
        // TODO(b/314203187): Not null asserted, check that this is correct.
        const isFocusableLinkOrControl = (node) => node.state[StateType$c.FOCUSABLE] &&
            AutomationPredicate.linkOrControl(node);
        // Next, try to focus the start or end node.
        // TODO(b/314203187): Not null asserted, check that this is correct.
        if (!AutomationPredicate.structuralContainer(start) &&
            start.state[StateType$c.FOCUSABLE]) {
            if (!start.state[StateType$c.FOCUSED]) {
                start.focus();
            }
            return;
        }
        else if (!AutomationPredicate.structuralContainer(end) &&
            end.state[StateType$c.FOCUSABLE]) {
            if (!end.state[StateType$c.FOCUSED]) {
                end.focus();
            }
            return;
        }
        // If a common ancestor of |start| and |end| is a link, focus that.
        let ancestor = AutomationUtil.getLeastCommonAncestor(start, end);
        while (ancestor && ancestor.root === start.root) {
            if (isFocusableLinkOrControl(ancestor)) {
                // TODO(b/314203187): Not null asserted, check that this is correct.
                if (!ancestor.state[StateType$c.FOCUSED]) {
                    ancestor.focus();
                }
                return;
            }
            ancestor = ancestor.parent;
        }
        // If nothing is focusable, set the sequential focus navigation starting
        // point, which ensures that the next time you press Tab, you'll reach
        // the next or previous focusable node from |start|.
        // TODO(b/314203187): Not null asserted, check that this is correct.
        if (!start.state[StateType$c.OFFSCREEN]) {
            start.setSequentialFocusNavigationStartingPoint();
        }
    }
    /** @return true if the selection is toggled on, false if toggled off. */
    toggleSelection_() {
        if (!this.pageSel_) {
            ChromeVox.earcons.playEarcon(EarconId.SELECTION);
            this.pageSel_ = ChromeVoxRange.current;
            // TODO(b/314203187): Not null asserted, check that this is correct.
            DesktopAutomationInterface.instance.ignoreDocumentSelectionFromAction(true);
            return true;
        }
        else {
            // TODO(b/314203187): Not null asserted, check that this is correct.
            const root = this.current_.start.node.root;
            if (root && root.selectionStartObject && root.selectionEndObject &&
                !isNaN(Number(root.selectionStartOffset)) &&
                !isNaN(Number(root.selectionEndOffset))) {
                ChromeVox.earcons.playEarcon(EarconId.SELECTION_REVERSE);
                // TODO(b/314203187): Not null asserted, check that this is correct.
                const sel = new CursorRange(new Cursor(root.selectionStartObject, root.selectionStartOffset), new Cursor(root.selectionEndObject, root.selectionEndOffset));
                new Output()
                    .format('@end_selection')
                    .withSpeechAndBraille(sel, sel, OutputCustomEvent.NAVIGATE)
                    .go();
                // TODO(b/314203187): Not null asserted, check that this is correct.
                DesktopAutomationInterface.instance.ignoreDocumentSelectionFromAction(false);
            }
            this.pageSel_ = null;
            return false;
        }
    }
}
TestImportManager.exportForTesting(ChromeVoxRange, ChromeVoxRangeObserver);

// Copyright 2021 The Chromium Authors
// Use of this source code is governed by a BSD-style license that can be
// found in the LICENSE file.
/**
 * @fileoverview Handles auto scrolling on navigation.
 */
var Dir$6 = constants.Dir;
var EventType$a = chrome.automation.EventType;
const RoleType$a = chrome.automation.RoleType;
/**
 * Handles scrolling, based either on a user command or an event firing.
 * Most of the logic is to support ARC++.
 */
class AutoScrollHandler {
    isScrolling_ = false;
    scrollingNode_ = null;
    rangeBeforeScroll_ = null;
    lastScrolledTime_ = new Date(0);
    relatedFocusEventHappened_ = false;
    allowWebContentsForTesting_ = false;
    static instance;
    static init() {
        AutoScrollHandler.instance = new AutoScrollHandler();
    }
    /**
     * This should be called before any command triggers ChromeVox navigation.
     *
     * @param target The range that is going to be navigated
     *     before scrolling.
     * @param dir The direction to navigate.
     * @param pred The predicate to match.
     * @param unit The unit to navigate by.
     * @param speechProps The optional speech properties
     *     given to |navigateTo| to provide feedback from the current command.
     * @param rootPred The predicate that expresses
     *     the current navigation root.
     * @param retryCommandFunc The callback used to retry the command
     *     with refreshed tree after scrolling.
     * @return True if the given navigation can be executed. False if the given
     *     navigation shouldn't happen, and AutoScrollHandler handles the command
     *     instead.
     */
    onCommandNavigation(target, dir, pred, unit, speechProps, rootPred, retryCommandFunc) {
        if (this.isScrolling_) {
            // Prevent interrupting the current scrolling.
            return false;
        }
        // TODO(b/314203187): Not null asserted, check that this is correct.
        if (!target.start?.node || !ChromeVoxRange.current.start.node) {
            return true;
        }
        const rangeBeforeScroll = ChromeVoxRange.current;
        let scrollable = this.findScrollableAncestor_(target);
        // At the beginning or the end of the document, there is a case where the
        // range stays there. It's worth trying scrolling the containing scrollable.
        // TODO(b/314203187): Not null asserted, check that this is correct.
        if (!scrollable && target.equals(rangeBeforeScroll) &&
            (unit === CursorUnit.WORD || unit === CursorUnit.CHARACTER)) {
            scrollable =
                this.tryFindingContainingScrollableIfAtEdge_(target, dir, scrollable ?? undefined);
        }
        if (!scrollable) {
            return true;
        }
        this.isScrolling_ = true;
        this.scrollingNode_ = scrollable;
        this.rangeBeforeScroll_ = rangeBeforeScroll;
        this.lastScrolledTime_ = new Date();
        this.relatedFocusEventHappened_ = false;
        this.scrollForCommandNavigation_(target, dir, pred, unit, speechProps ?? undefined, rootPred, retryCommandFunc)
            .catch(() => this.isScrolling_ = false);
        return false;
    }
    /**
     * This function will scroll to find nodes that are offscreen and not in the
     * tree.
     * @param bound The current cell node.
     * @param command the command handler command.
     * @param dir the direction of movement
     * @return True if the given navigation can be executed. False if
     *     the given navigation shouldn't happen, and AutoScrollHandler handles
     *     the command instead.
     */
    scrollToFindNodes(bound, command, target, dir, postScrollCallback) {
        if (bound.parent?.hasHiddenOffscreenNodes && target) {
            let pred = null;
            // Handle grids going over edge.
            if (bound.parent?.role === RoleType$a.GRID) {
                // TODO(b/314203187): Not null asserted, check that this is correct.
                const currentRow = bound.tableCellRowIndex;
                const totalRows = bound.parent.tableRowCount;
                const currentCol = bound.tableCellColumnIndex;
                const totalCols = bound.parent.tableColumnCount;
                if (command === Command.NEXT_ROW || command === Command.PREVIOUS_ROW) {
                    if (dir === Dir$6.BACKWARD && currentRow === 0) {
                        return true;
                    }
                    else if (dir === Dir$6.FORWARD && currentRow === (totalRows - 1)) {
                        return true;
                    }
                    // Create predicate
                    pred = AutomationPredicate.makeTableCellPredicate(bound, {
                        row: true,
                        dir,
                    });
                }
                else if (command === Command.NEXT_COL || command === Command.PREVIOUS_COL) {
                    if (dir === Dir$6.BACKWARD && currentCol === 0) {
                        return true;
                    }
                    else if (dir === Dir$6.FORWARD && currentCol === (totalCols - 1)) {
                        return true;
                    }
                    // Create predicate
                    pred = AutomationPredicate.makeTableCellPredicate(bound, {
                        col: true,
                        dir,
                    });
                }
            }
            return this.onCommandNavigation(target, dir, pred, null, null, AutomationPredicate.root, postScrollCallback);
        }
        return true;
    }
    /** @param target The range that is going to be navigated before scrolling. */
    findScrollableAncestor_(target) {
        let ancestors;
        if (ChromeVoxRange.current && target.equals(ChromeVoxRange.current)) {
            ancestors = AutomationUtil.getAncestors(target.start.node);
        }
        else {
            // TODO(b/314203187): Not null asserted, check that this is correct.
            ancestors = AutomationUtil.getUniqueAncestors(target.start.node, ChromeVoxRange.current.start.node);
        }
        // Check if we are in ARC++. Scrolling behavior should only happen there,
        // where additional nodes are not loaded until the user scrolls.
        if (!this.allowWebContentsForTesting_ &&
            !ancestors.find(node => node.role === RoleType$a.APPLICATION)) {
            return null;
        }
        const scrollable = ancestors.find(node => AutomationPredicate.autoScrollable(node));
        return scrollable ?? null;
    }
    tryFindingContainingScrollableIfAtEdge_(target, dir, scrollable) {
        let current = target.start.node;
        let parent = current.parent;
        while (parent?.root === current.root) {
            if (!(dir === Dir$6.BACKWARD && parent?.firstChild === current) &&
                !(dir === Dir$6.FORWARD && parent?.lastChild === current)) {
                // Currently on non-edge node. Don't try scrolling.
                return undefined;
            }
            if (AutomationPredicate.autoScrollable(current)) {
                scrollable = current;
            }
            // TODO(b/314203187): Not null asserted, check that this is correct.
            current = parent;
            parent = current.parent;
        }
        return scrollable;
    }
    /**
     * @param target The range that is going to be navigated before scrolling.
     * @param dir The direction to navigate.
     * @param pred The predicate to match.
     * @param unit The unit to navigate by.
     * @param speechProps The optional speech properties given to |navigateTo| to
     *     provide feedback for the current command.
     * @param rootPred The predicate that expresses the current navigation root.
     * @param retryCommandFunc The callback used to retry the command with
     *     refreshed tree after scrolling.
     */
    async scrollForCommandNavigation_(target, dir, pred, unit, speechProps, rootPred, retryCommandFunc) {
        const scrollResult = await this.scrollInDirection_(this.scrollingNode_ ?? undefined, dir);
        if (!scrollResult) {
            this.isScrolling_ = false;
            ChromeVoxRange.navigateTo(target, false, speechProps);
            return;
        }
        // Wait for a scrolled event or timeout.
        await this.waitForScrollEvent_(this.scrollingNode_ ?? undefined);
        // When a scrolling animation happens, SCROLL_POSITION_CHANGED event can
        // be dispatched multiple times, and there is a chance that the ui tree is
        // in an intermediate state. Just wait for a while so that the UI gets
        // stabilized.
        await new Promise(resolve => setTimeout(resolve, DELAY_HANDLE_SCROLLED_MS));
        this.isScrolling_ = false;
        if (!this.scrollingNode_) {
            throw Error('Illegal state in AutoScrollHandler.');
        }
        if (!this.scrollingNode_.root) {
            // Maybe scrollable node has disappeared. Do nothing.
            return;
        }
        if (this.relatedFocusEventHappened_) {
            const nextRange = this.handleScrollingInAndroidRecyclerView_(pred, unit, dir, rootPred, this.scrollingNode_);
            ChromeVoxRange.navigateTo(nextRange ?? target, false, speechProps);
            return;
        }
        // If the focus has been changed for some reason, do nothing to
        // prevent disturbing the latest navigation.
        // TODO(b/314203187): Not null asserted, check that this is correct.
        if (!ChromeVoxRange.current ||
            !ChromeVoxRange.current.equals(this.rangeBeforeScroll_)) {
            return;
        }
        // Usual case. Retry navigation with a refreshed tree.
        retryCommandFunc();
    }
    async scrollInDirection_(scrollable, dir) {
        return new Promise((resolve, reject) => {
            if (!scrollable) {
                reject('Scrollable cannot be null or undefined when scrolling');
                return;
            }
            if (dir === Dir$6.FORWARD) {
                scrollable.scrollForward(resolve);
            }
            else {
                scrollable.scrollBackward(resolve);
            }
        });
    }
    async waitForScrollEvent_(scrollable) {
        return new Promise((resolve, reject) => {
            if (!scrollable) {
                reject('Scrollable cannot be null when waiting for scroll event');
                return;
            }
            let cleanUp;
            let listener;
            let timeoutId;
            const onTimeout = () => {
                cleanUp();
                reject('timeout to wait for scrolled event.');
            };
            const onScrolled = () => {
                cleanUp();
                resolve();
            };
            cleanUp = () => {
                listener.stop();
                clearTimeout(timeoutId);
            };
            // TODO(b/314203187): Not null asserted, check that this is correct.
            listener = new EventHandler(scrollable, RELATED_SCROLL_EVENT_TYPES, onScrolled, { capture: true });
            listener.start();
            timeoutId = setTimeout(onTimeout, TIMEOUT_SCROLLED_EVENT_MS);
        });
    }
    /**
     * This block handles scrolling on Android RecyclerView.
     * When scrolling happens, RecyclerView can delete an invisible item, which
     * might be the focused node before scrolling, or can re-use the focused node
     * for a newly added node. When one of these happens, the focused node first
     * disappears from the ARC tree and the focus is moved up to the View that has
     * the ARC tree. In this case, navigate to the first or the last matching
     * range in the scrollable.
     *
     * @param pred The predicate to match.
     * @param unit The unit to navigate by.
     * @param dir The direction to navigate.
     * @param rootPred The predicate that expresses the current navigation root.
     */
    handleScrollingInAndroidRecyclerView_(pred, unit, dir, rootPred, scrollable) {
        let nextRange = null;
        if (!pred && unit) {
            nextRange = CursorRange.fromNode(scrollable).sync(unit, dir);
            if (unit === CursorUnit.NODE) {
                // TODO(b/314203187): Not null asserted, check that this is correct.
                nextRange = CommandHandlerInterface.instance.skipLabelOrDescriptionFor(nextRange, dir);
            }
        }
        else if (pred) {
            let node;
            if (dir === Dir$6.FORWARD) {
                node = AutomationUtil.findNextNode(scrollable, dir, pred, { root: rootPred, skipInitialSubtree: false });
            }
            else {
                // TODO(b/314203187): Not null asserted, check that this is correct.
                node = AutomationUtil.findNodePost(this.scrollingNode_, dir, pred);
            }
            if (node) {
                nextRange = CursorRange.fromNode(node);
            }
        }
        return nextRange;
    }
    /**
     * This should be called before the focus event triggers the navigation.
     *
     * On scrolling, sometimes previously focused node disappears.
     * There are two common problematic patterns in ARC tree here:
     * 1. No node has focus. The root ash window now get focus.
     * 2. Another node in the window gets focus.
     * Above two cases interrupt the auto scrolling. So when any of these are
     * happening, this returns false.
     *
     * @param node the node that is the focus event target.
     * @return True if others can handle the event. False if the event shouldn't
     *     be propagated.
     */
    onFocusEventNavigation(node) {
        if (!this.scrollingNode_ || !node) {
            return true;
        }
        const elapsedTime = new Date().valueOf() - this.lastScrolledTime_.valueOf();
        if (elapsedTime > TIMEOUT_FOCUS_EVENT_DROP_MS) {
            return true;
        }
        const isUnrelatedEvent = this.scrollingNode_.root !== node.root &&
            !AutomationUtil.isDescendantOf(this.scrollingNode_, node);
        if (isUnrelatedEvent) {
            return true;
        }
        this.relatedFocusEventHappened_ = true;
        return false;
    }
}
// Variables local to the module.
/**
 * An array of Automation event types that AutoScrollHandler observes when
 * performing a scroll action.
 */
const RELATED_SCROLL_EVENT_TYPES = [
    // This is sent by ARC++.
    EventType$a.SCROLL_POSITION_CHANGED,
    // These two events are sent by Web and Views via AXEventGenerator.
    EventType$a.SCROLL_HORIZONTAL_POSITION_CHANGED,
    EventType$a.SCROLL_VERTICAL_POSITION_CHANGED,
];
/**
 * The timeout that we wait for scroll event since scrolling action's callback
 * is executed.
 * TODO(hirokisato): Find an appropriate timeout in our case. TalkBack uses 500
 * milliseconds. See
 * https://github.com/google/talkback/blob/acd0bc7631a3dfbcf183789c7557596a45319e1f/talkback/src/main/java/ScrollEventInterpreter.java#L169
 */
const TIMEOUT_SCROLLED_EVENT_MS = 1500;
/**
 * The timeout that the focused event should be dropped. This is longer than
 * |TIMEOUT_CALLBACK_MS| because a focus event can happen after the scrolling.
 */
const TIMEOUT_FOCUS_EVENT_DROP_MS = 2000;
/**
 * The delay in milliseconds to wait to handle a scrolled event after the event
 * is first dispatched in order to wait for UI stabilized. See also
 * https://github.com/google/talkback/blob/6c0b475b7f52469e309e51bfcc13de58f18176ff/talkback/src/main/java/com/google/android/accessibility/talkback/interpreters/AutoScrollInterpreter.java#L42
 */
const DELAY_HANDLE_SCROLLED_MS = 150;
TestImportManager.exportForTesting(AutoScrollHandler);

// Copyright 2025 The Chromium Authors
// Use of this source code is governed by a BSD-style license that can be
// found in the LICENSE file.
/**
 * @fileoverview Manages opening and closing the live caption bubble, as well
 * as moving ChromeVox focus to the bubble when it appears and restoring focus
 * when it goes away.
 */
var EventType$9 = chrome.automation.EventType;
class CaptionsHandler {
    static instance;
    inCaptions_ = false;
    previousFocus_ = null;
    previousPrefState_;
    waitingForCaptions_ = false;
    captionsLabel_ = null;
    boundOnCaptionsLabelChanged_ = null;
    // Max lines of captions to keep around.
    static MAX_LINES = 100;
    // Captions line buffer.
    captionLines_ = [];
    // Tracks the last data sent to the Braille display.
    lastOutput_ = {
        lineIndex: -1,
        start: -1,
        end: -1,
    };
    static init() {
        CaptionsHandler.instance = new CaptionsHandler();
    }
    static open() {
        CaptionsHandler.instance.saveLiveCaptionValue_();
        if (CaptionsHandler.instance.isLiveCaptionEnabled_()) {
            CaptionsHandler.instance.tryJumpToCaptionBubble_();
        }
        else {
            CaptionsHandler.instance.enableLiveCaption_();
        }
    }
    static close() {
        CaptionsHandler.instance.resetLiveCaption_();
        CaptionsHandler.instance.onExitCaptions_();
    }
    static inCaptions() {
        return CaptionsHandler.instance.inCaptions_;
    }
    // Returns true if the alert was handled and the default handing should be
    // skipped, false otherwise.
    maybeHandleAlert(event) {
        if (!this.waitingForCaptions_) {
            // Filter out captions bubble dialog alert and announcement when this is
            // in the bubble. Otherwise, the alert overwrites the braille output.
            if (this.inCaptions_ && this.captionsLabel_ &&
                (this.captionsLabel_ === this.tryFindCaptions_(event.target) ||
                    (event.target.parent &&
                        this.captionsLabel_ ===
                            this.tryFindCaptions_(event.target.parent)))) {
                return true;
            }
            return false;
        }
        const captionsLabel = this.tryFindCaptions_(event.target);
        if (!captionsLabel) {
            return false;
        }
        this.waitingForCaptions_ = false;
        this.jumpToCaptionBubble_(captionsLabel);
        return true;
    }
    // ChromeVoxRangeObserver implementation.
    onCurrentRangeChanged(range) {
        if (!range) {
            return;
        }
        const currentNode = range.start.node;
        if (currentNode.className !== CAPTION_BUBBLE_LABEL) {
            this.onExitCaptions_();
        }
        else {
            this.startObserveCaptions_(currentNode);
        }
    }
    /** This function assumes the preference for live captions is false. */
    enableLiveCaption_() {
        this.waitingForCaptions_ = true;
        chrome.accessibilityPrivate.enableLiveCaption(true);
    }
    isLiveCaptionEnabled_() {
        return SettingsManager.getBoolean('accessibility.captions.live_caption_enabled', /*isChromeVox=*/ false);
    }
    jumpToCaptionBubble_(captionsLabel) {
        this.previousFocus_ = ChromeVoxRange.current;
        this.onEnterCaptions_();
        ChromeVoxRange.navigateTo(CursorRange.fromNode(captionsLabel), 
        /*focus=*/ true, /*speechProps=*/ undefined, 
        /*skipSettingSelection=*/ undefined, 
        /*skipOutput=*/ true);
    }
    onEnterCaptions_() {
        this.inCaptions_ = true;
        ChromeVoxRange.addObserver(this);
    }
    onExitCaptions_() {
        this.inCaptions_ = false;
        ChromeVoxRange.removeObserver(this);
        this.stopObserveCaptions_();
        this.restoreFocus_();
    }
    resetLiveCaption_() {
        // Don't disable live captions if they were enabled before we began.
        if (this.previousPrefState_) {
            return;
        }
        chrome.accessibilityPrivate.enableLiveCaption(false);
    }
    restoreFocus_() {
        if (!this.previousFocus_) {
            return;
        }
        ChromeVoxRange.navigateTo(this.previousFocus_);
    }
    saveLiveCaptionValue_() {
        this.previousPrefState_ = this.isLiveCaptionEnabled_();
    }
    tryFindCaptions_(node) {
        return node.find({ attributes: { className: CAPTION_BUBBLE_LABEL } });
    }
    /**
     * Attempts to locate the caption bubble in the accessibility tree. If found,
     * moves focus there.
     */
    async tryJumpToCaptionBubble_() {
        const desktop = await new Promise(resolve => chrome.automation.getDesktop(resolve));
        const captionsLabel = this.tryFindCaptions_(desktop);
        if (!captionsLabel) {
            return;
        }
        this.jumpToCaptionBubble_(captionsLabel);
    }
    /** Starts to observe live caption label. */
    startObserveCaptions_(captionsLabel) {
        this.captionsLabel_ = captionsLabel;
        if (!this.boundOnCaptionsLabelChanged_) {
            this.boundOnCaptionsLabelChanged_ = () => this.onCaptionsLabelChanged_();
        }
        this.captionsLabel_.addEventListener(EventType$9.TEXT_CHANGED, this.boundOnCaptionsLabelChanged_, true);
        this.updateCaptions_();
    }
    /** Stops observing live caption label. */
    stopObserveCaptions_() {
        if (!this.captionsLabel_) {
            return;
        }
        this.captionLines_ = [];
        this.resetLastOutput_();
        this.captionsLabel_.removeEventListener(EventType$9.TEXT_CHANGED, this.boundOnCaptionsLabelChanged_, true);
        this.captionsLabel_ = null;
    }
    /** Invoked when captions label is changed. */
    onCaptionsLabelChanged_() {
        this.updateCaptions_();
    }
    /**
     * Extracts the caption lines from captions bubble and merges with the line
     * buffer.
     */
    updateCaptions_() {
        if (!this.captionsLabel_) {
            return;
        }
        let currentLines = [];
        for (let i = 0, child; child = this.captionsLabel_.children[i]; ++i) {
            if (child.name) {
                currentLines.push(child.name);
            }
        }
        this.mergeLines_(currentLines);
    }
    /** Merge the current lines in caption bubble with the line buffer. */
    mergeLines_(currentLines) {
        // Finds an insertion point to update the line buffer.
        const fullMatchFirstLine = this.captionLines_.lastIndexOf(currentLines[0]);
        if (fullMatchFirstLine !== -1) {
            // If first line of `currentLines` is found, copy `currentLines` at the
            // index.
            this.captionLines_.splice(fullMatchFirstLine, this.captionLines_.length - fullMatchFirstLine, ...currentLines);
        }
        else if (currentLines.length === 1 || this.captionLines_.length === 1) {
            // If buffer is initiating, just copying over. Reset tracking if
            // `currentLines` is not relevant.
            if (this.lastOutput_.lineIndex !== -1) {
                let lastOutputText = this.captionLines_[this.lastOutput_.lineIndex].substring(this.lastOutput_.start, this.lastOutput_.end);
                if (currentLines[0].indexOf(lastOutputText) === -1) {
                    this.resetLastOutput_();
                }
            }
            this.captionLines_ = [...currentLines];
        }
        else {
            // Otherwise, restart tracking.
            this.captionLines_ = [...currentLines];
            this.resetLastOutput_();
        }
        // Trim when the line buffer is more than MAX_LINES.
        while (this.captionLines_.length > CaptionsHandler.MAX_LINES) {
            this.captionLines_.shift();
            this.lastOutput_.lineIndex--;
        }
        if (this.lastOutput_.lineIndex < 0) {
            this.resetLastOutput_();
        }
        if (this.lastOutput_.lineIndex === -1 && this.captionLines_.length > 0) {
            this.output_(0);
        }
    }
    /** Reset output tracking */
    resetLastOutput_() {
        this.lastOutput_ = {
            lineIndex: -1,
            start: -1,
            end: -1,
        };
    }
    /** Outputs the given line. */
    output_(index, start, end) {
        let text = this.captionLines_[index];
        this.lastOutput_.lineIndex = index;
        this.lastOutput_.start = start ?? 0;
        this.lastOutput_.end = end ?? text.length;
        new Output()
            .withString(text.substring(this.lastOutput_.start, this.lastOutput_.end))
            .go();
    }
    /** Output the previous line if there is one. */
    previous() {
        if (this.lastOutput_.lineIndex == -1) {
            return;
        }
        // If the last output is in middle of the line, restart from the beginning.
        if (this.lastOutput_.start != 0) {
            this.output_(this.lastOutput_.lineIndex, 0);
            return;
        }
        // No more previous lines.
        if (this.lastOutput_.lineIndex == 0) {
            return;
        }
        this.output_(this.lastOutput_.lineIndex - 1);
    }
    /** Output the next line if there is one. */
    next() {
        if (this.lastOutput_.lineIndex == -1) {
            return;
        }
        // If the last output is not at the end, start from where it stops.
        if (this.captionLines_[this.lastOutput_.lineIndex].length >
            this.lastOutput_.end) {
            this.output_(this.lastOutput_.lineIndex, this.lastOutput_.end);
            return;
        }
        // No more next lines.
        if (this.lastOutput_.lineIndex == this.captionLines_.length - 1) {
            return;
        }
        this.output_(this.lastOutput_.lineIndex + 1);
    }
}
const CAPTION_BUBBLE_LABEL = 'CaptionBubbleLabel';
TestImportManager.exportForTesting(CaptionsHandler);

// Copyright 2014 The Chromium Authors
// Use of this source code is governed by a BSD-style license that can be
// found in the LICENSE file.
/**
 * @fileoverview If the braille captions feature is enabled, sends
 * braille content to the Panel on Chrome OS, or a content script on
 * other platforms.
 */
class BrailleCaptionsBackground {
    static instance;
    listener_;
    constructor(listener) {
        this.listener_ = listener;
    }
    /** Called once to initialize the class. */
    static init(listener) {
        BrailleCaptionsBackground.instance =
            new BrailleCaptionsBackground(listener);
    }
    /** Returns whether the braille captions feature is enabled. */
    static isEnabled() {
        return Boolean(LocalStorage.get(PREF_KEY));
    }
    /**
     * @param text Text of the shown braille.
     * @param cells Braille cells shown on the display.
     * @param brailleToText Map of Braille letters to the first
     *     index of corresponding text letter.
     * @param offsetsForSlices
     *    Offsets to use for calculating indices. The element is the braille
     * offset, the second is the text offset.
     * @param rows Number of rows to display.
     * @param columns Number of columns to display.
     */
    static setContent(text, cells, brailleToText, offsetsForSlices, rows, columns) {
        // Convert the cells to Unicode braille pattern characters.
        const byteBuf = new Uint8Array(cells);
        let brailleChars = '';
        for (const byteVal of byteBuf) {
            brailleChars +=
                String.fromCharCode(BRAILLE_UNICODE_BLOCK_START | byteVal);
        }
        const groups = BrailleCaptionsBackground.groupBrailleAndText(brailleChars, text, brailleToText, offsetsForSlices);
        const data = { groups, rows, cols: columns };
        (new PanelCommand(PanelCommandType.UPDATE_BRAILLE, data)).send();
    }
    /**
     * @param cells Braille cells shown on the display.
     * @param rows Number of rows to display.
     * @param columns Number of columns to display.
     */
    static setImageContent(cells, rows, columns) {
        // Convert the cells to Unicode braille pattern characters.
        const byteBuf = new Uint8Array(cells);
        let brailleChars = '';
        for (const byteVal of byteBuf) {
            brailleChars +=
                String.fromCharCode(BRAILLE_UNICODE_BLOCK_START | byteVal);
        }
        const groups = [['Image', brailleChars]];
        const data = { groups, rows, cols: columns };
        (new PanelCommand(PanelCommandType.UPDATE_BRAILLE, data)).send();
    }
    /**
     * @param brailleChars Braille characters shown on the display.
     * @param text Text of the shown braille.
     * @param brailleToText Map of Braille cells to the first index of
     *     corresponding text letter.
     * @param offsets Offsets to use for calculating indices. The element is the
     *     braille offset, the second is the text offset.
     * @return The groups of braille and texts to be displayed.
     */
    static groupBrailleAndText(brailleChars, text, brailleToText, offsets) {
        let brailleBuf = '';
        const groups = [];
        let textIndex = 0;
        const brailleOffset = offsets.brailleOffset;
        const textOffset = offsets.textOffset;
        const calculateTextIndex = (index) => brailleToText[index + brailleOffset] - textOffset;
        for (let i = 0; i < brailleChars.length; ++i) {
            const textSliceIndex = calculateTextIndex(i);
            if (i !== 0 && textSliceIndex !== textIndex) {
                groups.push([text.substr(textIndex, textSliceIndex - textIndex), brailleBuf]);
                brailleBuf = '';
                textIndex = textSliceIndex;
            }
            brailleBuf += brailleChars.charAt(i);
        }
        // Puts the rest of the text into the last group.
        if (brailleBuf.length > 0 && text.charAt(textIndex) !== ' ') {
            groups.push([text.substr(textIndex), brailleBuf]);
        }
        return groups;
    }
    /**
     * Sets whether the overlay should be active.
     * @param newValue The new value of the active flag.
     */
    static setActive(newValue) {
        const oldValue = BrailleCaptionsBackground.isEnabled();
        // TODO(b/314203187): Not null asserted, check that this is correct.
        ChromeVoxPrefs.instance.setPref(PREF_KEY, newValue);
        if (oldValue !== newValue) {
            BrailleCaptionsBackground.instance.listener_
                .onBrailleCaptionsStateChanged();
            const msg = newValue ? Msgs.getMsg('braille_captions_enabled') :
                Msgs.getMsg('braille_captions_disabled');
            ChromeVox.tts.speak(msg, QueueMode.QUEUE);
            ChromeVox.braille.write(NavBraille.fromText(msg));
        }
    }
    /**
     * Asynchronously returns a display state representing the state of the
     * captions feature. This is used when no actual hardware display is
     * connected.
     */
    static getVirtualDisplayState() {
        if (BrailleCaptionsBackground.isEnabled()) {
            const rows = SettingsManager.getNumber('virtualBrailleRows');
            const columns = SettingsManager.getNumber('virtualBrailleColumns');
            // TODO(accessibility) make `cellSize` customizable.
            return {
                available: true,
                textRowCount: rows,
                textColumnCount: columns,
                cellSize: 8,
            };
        }
        else {
            return {
                available: false,
                textRowCount: 0,
                textColumnCount: 0,
                cellSize: 0,
            };
        }
    }
}
// Local to module.
/** Key set in local storage when this feature is enabled. */
const PREF_KEY = 'brailleCaptions';
/**
 * Unicode block of braille pattern characters.  A braille pattern is formed
 * from this value with the low order 8 bits set to the bits representing
 * the dots as per the ISO 11548-1 standard.
 */
const BRAILLE_UNICODE_BLOCK_START = 0x2800;
TestImportManager.exportForTesting(BrailleCaptionsBackground);

// Copyright 2014 The Chromium Authors
// Use of this source code is governed by a BSD-style license that can be
// found in the LICENSE file.
/**
 * @fileoverview Holds information about a braille table.
 */
var BrailleTable;
(function (BrailleTable) {
    BrailleTable.TABLE_PATH = 'chromevox/third_party/liblouis/tables.json';
    /**
     * Retrieves a list of all available braille tables.
     * @param callback Called asynchronously with an array of tables.
     */
    async function getAll(callback) {
        const needsDisambiguation = new Map();
        function preprocess(tables) {
            tables.forEach((table) => {
                // Append the common definitions to all table filenames.
                table.fileNames += (',' + COMMON_DEFS_FILENAME);
                // Save all tables which have a mirroring duplicate for locale + grade.
                const key = table.locale + table.grade;
                if (!needsDisambiguation.has(key)) {
                    needsDisambiguation.set(key, []);
                }
                const entry = needsDisambiguation.get(key);
                entry.push(table);
            });
            for (const entry of needsDisambiguation.values()) {
                if (entry.length > 1) {
                    entry.forEach((table) => table.alwaysUseEnDisplayName = true);
                }
            }
            return tables;
        }
        const url = chrome.extension.getURL(BrailleTable.TABLE_PATH);
        if (!url) {
            throw 'Invalid path: ' + BrailleTable.TABLE_PATH;
        }
        const response = await fetch(url);
        if (response.ok) {
            const tables = await response.json();
            callback(preprocess(tables));
        }
    }
    BrailleTable.getAll = getAll;
    /**
     * Finds a table in a list of tables by id.
     * @param tables tables to search in.
     * @param id id of table to find.
     * @return The found table, or null if not found.
     */
    function forId(tables, id) {
        return tables.filter(table => table.id === id)[0] || null;
    }
    BrailleTable.forId = forId;
    /**
     * Returns an uncontracted braille table corresponding to another, possibly
     * contracted, table.  If {@code table} is the lowest-grade table for its
     * locale and dot count, {@code table} itself is returned.
     * @param tables tables to search in.
     * @param table Table to match.
     * @return {!BrailleTable.Table} Corresponding uncontracted table,
     *     or {@code table} if it is uncontracted.
     */
    function getUncontracted(tables, table) {
        function mostUncontractedOf(current, candidate) {
            // An 8 dot table for the same language is preferred over a 6 dot table
            // even if the locales differ by region.
            if (current.dots === '6' && candidate.dots === '8' &&
                current.locale.lastIndexOf(candidate.locale, 0) === 0) {
                return candidate;
            }
            if (current.locale === candidate.locale &&
                current.dots === candidate.dots && (current.grade !== undefined) &&
                (candidate.grade !== undefined) && candidate.grade < current.grade) {
                return candidate;
            }
            return current;
        }
        return tables.reduce(mostUncontractedOf, table);
    }
    BrailleTable.getUncontracted = getUncontracted;
    /**
     * @param table Table to get name for.
     * @return Localized display name.
     */
    function getDisplayName(table) {
        const uiLanguage = chrome.i18n.getUILanguage().toLowerCase();
        const localeName = chrome.accessibilityPrivate.getDisplayNameForLocale(table.locale /* locale to be displayed */, uiLanguage /* locale to localize into */);
        const enDisplayName = table.enDisplayName;
        if (!localeName && !enDisplayName) {
            return;
        }
        let baseName;
        if (enDisplayName &&
            (table.alwaysUseEnDisplayName || uiLanguage.startsWith('en') ||
                !localeName)) {
            baseName = enDisplayName;
        }
        else {
            baseName = localeName;
        }
        if (!table.grade && !table.variant) {
            return baseName;
        }
        else if (table.grade && !table.variant) {
            return Msgs.getMsg('braille_table_name_with_grade', [baseName, table.grade]);
        }
        else if (!table.grade && table.variant) {
            return Msgs.getMsg('braille_table_name_with_variant', [baseName, table.variant]);
        }
        else {
            // TODO(b/314203187): Not null asserted, check that this is correct.
            return Msgs.getMsg('braille_table_name_with_variant_and_grade', [baseName, table.variant, table.grade]);
        }
    }
    BrailleTable.getDisplayName = getDisplayName;
})(BrailleTable || (BrailleTable = {}));
// Local to module.
const COMMON_DEFS_FILENAME = 'cvox-common.cti';
TestImportManager.exportForTesting(['BrailleTable', BrailleTable]);

// Copyright 2014 The Chromium Authors
// Use of this source code is governed by a BSD-style license that can be
// found in the LICENSE file.
/**
 * @fileoverview Translates text to braille, optionally with some parts
 * uncontracted.
 */
/**
 * A wrapper around one or two braille translators that uses contracted
 * braille or not based on the selection start- and end-points (if any) in the
 * translated text.  If only one translator is provided, then that translator
 * is used for all text regardless of selection.  If two translators
 * are provided, then the uncontracted translator is used for some text
 * around the selection end-points and the contracted translator is used
 * for all other text.  When determining what text to use uncontracted
 * translation for around a position, a region surrounding that position
 * containing either only whitespace characters or only non-whitespace
 * characters is used.
 */
class ExpandingBrailleTranslator {
    defaultTranslator_;
    uncontractedTranslator_;
    /**
     * @param defaultTranslator The translator for all text when the uncontracted
     *     translator is not used.
     * @param uncontractedTranslator Translator to use for uncontracted braille
     *     translation.
     */
    constructor(defaultTranslator_, uncontractedTranslator_) {
        this.defaultTranslator_ = defaultTranslator_;
        this.uncontractedTranslator_ = uncontractedTranslator_;
    }
    /**
     * Translates text to braille using the translator(s) provided to the
     * constructor.  See LibLouis.Translator for further details.
     * @param text Text to translate.
     * @param expansionType Indicates how the text marked by a value span,
     *    if any, is expanded.
     * @param callback Called when the translation is done.  It takes resulting
     *    braille cells and positional mappings as parameters.
     */
    translate(text, expansionType, callback) {
        const expandRanges = this.findExpandRanges_(text, expansionType);
        const extraCellsSpans = text.getSpansInstanceOf(ExtraCellsSpan)
            .filter(span => span.cells.byteLength > 0);
        const extraCellsPositions = extraCellsSpans.map(span => text.getSpanStart(span));
        const formTypeMap = new Array(text.length).fill(0);
        text.getSpansInstanceOf(BrailleTextStyleSpan).forEach((span) => {
            const start = text.getSpanStart(span);
            const end = text.getSpanEnd(span);
            for (let i = start; i < end; i++) {
                formTypeMap[i] |= span.formType;
            }
        });
        if (expandRanges.length === 0 && extraCellsSpans.length === 0) {
            this.defaultTranslator_.translate(text.toString(), formTypeMap, ExpandingBrailleTranslator.nullParamsToEmptyAdapter_(text.length, callback));
            return;
        }
        const chunks = [];
        function maybeAddChunkToTranslate(translator, start, end) {
            if (start < end) {
                chunks.push({ translator, start, end });
            }
        }
        function addExtraCellsChunk(pos, cells) {
            const chunk = {
                translator: null,
                start: pos,
                end: pos,
                cells,
                textToBraille: [],
                brailleToText: new Array(cells.byteLength),
            };
            for (let i = 0; i < cells.byteLength; ++i) {
                chunk.brailleToText[i] = 0;
            }
            chunks.push(chunk);
        }
        function addChunk(translator, start, end) {
            while (extraCellsSpans.length > 0 && extraCellsPositions[0] <= end) {
                maybeAddChunkToTranslate(translator, start, extraCellsPositions[0]);
                // TODO(b/314203187): Not null asserted, check that this is correct.
                start = extraCellsPositions.shift();
                addExtraCellsChunk(start, extraCellsSpans.shift().cells);
            }
            maybeAddChunkToTranslate(translator, start, end);
        }
        let lastEnd = 0;
        for (let i = 0; i < expandRanges.length; ++i) {
            const range = expandRanges[i];
            if (lastEnd < range.start) {
                addChunk(this.defaultTranslator_, lastEnd, range.start);
            }
            // TODO(b/314203187): Not null asserted, check that this is correct.
            addChunk(this.uncontractedTranslator_, range.start, range.end);
            lastEnd = range.end;
        }
        addChunk(this.defaultTranslator_, lastEnd, text.length);
        const chunksToTranslate = chunks.filter((chunk) => chunk.translator);
        let numPendingCallbacks = chunksToTranslate.length;
        function chunkTranslated(chunk, cells, textToBraille, brailleToText) {
            chunk.cells = cells;
            chunk.textToBraille = textToBraille;
            chunk.brailleToText = brailleToText;
            if (--numPendingCallbacks <= 0) {
                finish();
            }
        }
        function finish() {
            // TODO(b/314203187): Not null asserted, check that this is correct.
            const totalCells = chunks.reduce((accum, chunk) => accum + chunk.cells.byteLength, 0);
            const cells = new Uint8Array(totalCells);
            let cellPos = 0;
            const textToBraille = [];
            const brailleToText = [];
            function appendAdjusted(array, toAppend, adjustment) {
                array.push.apply(array, toAppend.map(elem => adjustment + elem));
            }
            for (let i = 0, chunk; chunk = chunks[i]; ++i) {
                // TODO(b/314203187): Not null asserted, check that this is correct.
                cells.set(new Uint8Array(chunk.cells), cellPos);
                appendAdjusted(textToBraille, chunk.textToBraille, cellPos);
                appendAdjusted(brailleToText, chunk.brailleToText, chunk.start);
                cellPos += chunk.cells.byteLength;
            }
            callback(cells.buffer, textToBraille, brailleToText);
        }
        if (chunksToTranslate.length > 0) {
            chunksToTranslate.forEach((chunk) => {
                // TODO(b/314203187): Not null asserted, check that this is correct.
                chunk.translator.translate(text.toString().substring(chunk.start, chunk.end), formTypeMap.slice(chunk.start, chunk.end), ExpandingBrailleTranslator.nullParamsToEmptyAdapter_(chunk.end - chunk.start, (cells, textToBraille, brailleToText) => chunkTranslated(chunk, cells, textToBraille, brailleToText)));
            });
        }
        else {
            finish();
        }
    }
    /**
     * Expands a position to a range that covers the consecutive range of
     * either whitespace or non whitespace characters around it.
     * @param str Text to look in.
     * @param pos Position to start looking at.
     * @param start Minimum value for the start position of the returned range.
     * @param end Maximum value for the end position of the returned range.
     * @return The calculated range.
     */
    static rangeForPosition_(str, pos, start, end) {
        if (start < 0 || end > str.length) {
            throw RangeError('End-points out of range looking for braille expansion range');
        }
        if (pos < start || pos >= end) {
            throw RangeError('Position out of range looking for braille expansion range');
        }
        // Find the last chunk of either whitespace or non-whitespace before and
        // including pos.
        start = str.substring(start, pos + 1).search(/(\s+|\S+)$/) + start;
        // Find the characters to include after pos, starting at pos so that
        // they are the same kind (either whitespace or not) as the
        // characters starting at start.
        // TODO(b/314203187): Not null asserted, check that this is correct.
        end = pos + /^(\s+|\S+)/.exec(str.substring(pos, end))[0].length;
        return { start, end };
    }
    /**
     * Finds the ranges in which contracted braille should not be used.
     * @param text Text to find expansion ranges in.
     * @param expansionType Indicates how the text marked up as the value is
     *     expanded.
     * @return The calculated ranges.
     */
    findExpandRanges_(text, expansionType) {
        const result = [];
        if (this.uncontractedTranslator_ &&
            expansionType !== ExpandingBrailleTranslator.ExpansionType.NONE) {
            const value = text.getSpanInstanceOf(ValueSpan);
            if (value) {
                const valueStart = text.getSpanStart(value);
                const valueEnd = text.getSpanEnd(value);
                switch (expansionType) {
                    case ExpandingBrailleTranslator.ExpansionType.SELECTION:
                        this.addRangesForSelection_(text, valueStart, valueEnd, result);
                        break;
                    case ExpandingBrailleTranslator.ExpansionType.ALL:
                        result.push({ start: valueStart, end: valueEnd });
                        break;
                }
            }
        }
        return result;
    }
    /**
     * Finds ranges to expand around selection end points inside the value of
     * a string.  If any ranges are found, adds them to outRanges.
     * @param text Text to find ranges in.
     * @param valueStart Start of the value in text.
     * @param valueEnd End of the value in text.
     * @param outRanges Destination for the expansion ranges. Untouched if no
     *     ranges are found. Note that ranges may be coalesced.
     */
    addRangesForSelection_(text, valueStart, valueEnd, outRanges) {
        const selection = text.getSpanInstanceOf(ValueSelectionSpan);
        if (!selection) {
            return;
        }
        const selectionStart = text.getSpanStart(selection);
        const selectionEnd = text.getSpanEnd(selection);
        if (selectionStart < valueStart || selectionEnd > valueEnd) {
            return;
        }
        const expandPositions = [];
        if (selectionStart === valueEnd) {
            if (selectionStart > valueStart) {
                expandPositions.push(selectionStart - 1);
            }
        }
        else {
            if (selectionStart === selectionEnd && selectionStart > valueStart) {
                expandPositions.push(selectionStart - 1);
            }
            expandPositions.push(selectionStart);
            // Include the selection end if the length of the selection is
            // greater than one (otherwise this position would be redundant).
            if (selectionEnd > selectionStart + 1) {
                // Look at the last actual character of the selection, not the
                // character at the (exclusive) end position.
                expandPositions.push(selectionEnd - 1);
            }
        }
        let lastRange = outRanges[outRanges.length - 1] || null;
        for (let i = 0; i < expandPositions.length; ++i) {
            const range = ExpandingBrailleTranslator.rangeForPosition_(text.toString(), expandPositions[i], valueStart, valueEnd);
            if (lastRange && lastRange.end >= range.start) {
                lastRange.end = range.end;
            }
            else {
                outRanges.push(range);
                lastRange = range;
            }
        }
    }
    /**
     * Adapts callback to accept null arguments and treat them as if the
     * translation result is empty.
     * @param inputLength Length of the input to the translation.
     *     Used for populating textToBraille if null.
     * @param callback The callback to adapt.
     * @return An adapted version of the callback.
     */
    static nullParamsToEmptyAdapter_(inputLength, callback) {
        return function (cells, textToBraille, brailleToText) {
            if (!textToBraille) {
                textToBraille = new Array(inputLength);
                for (let i = 0; i < inputLength; ++i) {
                    textToBraille[i] = 0;
                }
            }
            callback(cells || new ArrayBuffer(0), textToBraille, brailleToText || []);
        };
    }
}
(function (ExpandingBrailleTranslator) {
    (function (ExpansionType) {
        /**
         * Use the default translator all of the value, regardless of any selection.
         * This is typically used when the user is in the middle of typing and the
         * typing started outside of a word.
         */
        ExpansionType[ExpansionType["NONE"] = 0] = "NONE";
        /**
         * Expand text around the selection end-points if any.  If the selection is
         * a cursor, expand the text that occupies the positions right before and
         * after the cursor.  This is typically used when the user hasn't started
         * typing contracted braille or when editing inside a word.
         */
        ExpansionType[ExpansionType["SELECTION"] = 1] = "SELECTION";
        /**
         * Expand all text covered by the value span.  this is typically used when
         * the user is editing a text field where it doesn't make sense to use
         * contracted braille (such as a url or email address).
         */
        ExpansionType[ExpansionType["ALL"] = 2] = "ALL";
    })(ExpandingBrailleTranslator.ExpansionType || (ExpandingBrailleTranslator.ExpansionType = {}));
})(ExpandingBrailleTranslator || (ExpandingBrailleTranslator = {}));
TestImportManager.exportForTesting(ExpandingBrailleTranslator);

// Copyright 2014 The Chromium Authors
// Use of this source code is governed by a BSD-style license that can be
// found in the LICENSE file.
/**
 * @fileoverview JavaScript shim for the liblouis Web Assembly wrapper.
 */
/** Encapsulates a liblouis Web Assembly instance in the page. */
class LibLouis {
    /** Path to .wasm file for the module. */
    wasmPath_;
    /** Whether liblouis is loaded. */
    isLoaded_ = false;
    /** Pending RPC callbacks. Maps from message IDs to callbacks. */
    pendingRpcCallbacks_ = {};
    /** Next message ID to be used. Incremented with each sent message. */
    nextMessageId_ = 1;
    worker;
    /**
     * @param wasmPath Path to .wasm file for the module.
     * @param tablesDir Path to tables directory.
     */
    constructor(wasmPath, _tablesDir, loadCallback) {
        this.wasmPath_ = wasmPath;
        this.loadOrReload_(loadCallback);
    }
    /**
     * Convenience method to wait for the constructor to resolve its callback.
     * @param wasmPath Path to .wasm file for the module.
     * @param tablesDir Path to tables directory.
     */
    static async create(wasmPath, tablesDir) {
        return new Promise(resolve => new LibLouis(wasmPath, tablesDir, resolve));
    }
    isLoaded() {
        return this.isLoaded_;
    }
    /**
     * Returns a translator for the desired table, asynchronously.
     * This object must be attached to a document when requesting a translator.
     * @param {string} tableNames Comma separated list of braille table names for
     *     liblouis.
     * @return {!Promise<LibLouis.Translator>} the translator, or {@code null}
     *     on failure.
     */
    async getTranslator(tableNames) {
        return new Promise(resolve => {
            if (!this.isLoaded_) {
                // TODO: save last callback.
                resolve(null /* translator */);
                return;
            }
            this.rpc('CheckTable', { 'table_names': tableNames }, (reply) => {
                if (reply['success']) {
                    const translator = new LibLouis.Translator(this, tableNames);
                    resolve(translator);
                }
                else {
                    resolve(null /* translator */);
                }
            });
        });
    }
    /**
     * Dispatches a message to the remote end and returns the reply
     * asynchronously. A message ID will be automatically assigned (as a
     * side-effect).
     * @param command Command name to be sent.
     * @param message JSONable message to be sent.
     * @param callback Callback to receive the reply.
     */
    rpc(command, message, callback) {
        if (!this.worker) {
            throw Error('Cannot send RPC: liblouis instance not loaded');
        }
        const messageId = '' + this.nextMessageId_++;
        message['message_id'] = messageId;
        message['command'] = command;
        const json = JSON.stringify(message);
        if (LibLouis.DEBUG) {
            globalThis.console.debug('RPC -> ' + json);
        }
        this.worker.postMessage(json);
        this.pendingRpcCallbacks_[messageId] = callback;
    }
    /** Invoked when the Web Assembly instance successfully loads. */
    onInstanceLoad_() { }
    /** Invoked when the Web Assembly instance fails to load. */
    onInstanceError_(e) {
        globalThis.console.error('Error in liblouis ' + e.message);
        this.loadOrReload_();
    }
    /** Invoked when the Web Assembly instance posts a message. */
    onInstanceMessage_(e) {
        if (LibLouis.DEBUG) {
            globalThis.console.debug('RPC <- ' + e.data);
        }
        const message = /** @type {!Object} */ (JSON.parse(e.data));
        const messageId = message['in_reply_to'];
        if (messageId === undefined) {
            globalThis.console.warn('liblouis Web Assembly module sent message with no ID', message);
            return;
        }
        if (message['error'] !== undefined) {
            globalThis.console.error('liblouis Web Assembly error', message['error']);
        }
        const callback = this.pendingRpcCallbacks_[messageId];
        if (callback !== undefined) {
            delete this.pendingRpcCallbacks_[messageId];
            callback(message);
        }
    }
    loadOrReload_(loadCallback) {
        this.worker = new Worker(this.wasmPath_);
        this.worker.addEventListener('message', e => this.onInstanceMessage_(e), false /* useCapture */);
        this.worker.addEventListener('error', e => this.onInstanceError_(e), false /* useCapture */);
        this.rpc('load', {}, () => {
            this.isLoaded_ = true;
            loadCallback && loadCallback(this);
            this.onInstanceLoad_();
        });
    }
}
(function (LibLouis) {
    (function (FormType) {
        FormType[FormType["PLAIN_TEXT"] = 0] = "PLAIN_TEXT";
        FormType[FormType["ITALIC"] = 1] = "ITALIC";
        FormType[FormType["UNDERLINE"] = 2] = "UNDERLINE";
        FormType[FormType["BOLD"] = 4] = "BOLD";
        FormType[FormType["COMPUTER_BRAILLE"] = 8] = "COMPUTER_BRAILLE";
    })(LibLouis.FormType || (LibLouis.FormType = {}));
    /** Set to {@code true} to enable debug logging of RPC messages. */
    LibLouis.DEBUG = false;
    /** Braille translator which uses a Web Assembly instance of liblouis. */
    class Translator {
        instance_;
        tableNames_;
        /**
         * @param instance The instance wrapper.
         * @param tableNames Comma separated list of Table names to be passed to
         *     liblouis.
         */
        constructor(instance, tableNames) {
            this.instance_ = instance;
            this.tableNames_ = tableNames;
        }
        /**
         * Translates text into braille cells.
         * @param text Text to be translated.
         * @param callback Callback for result. Takes 3 parameters: the resulting
         *     cells, mapping from text to braille positions and mapping from
         *     braille to text positions. If translation fails for any reason, all
         *     parameters are null.
         */
        translate(text, formTypeMap, callback) {
            if (!this.instance_.worker) {
                callback(null /*cells*/, null /*textToBraille*/, null /*brailleToText*/);
                return;
            }
            // TODO(https://crbug.com/1340093): the upstream LibLouis translations for
            // form type output is broken.
            formTypeMap = 0;
            const message = {
                'table_names': this.tableNames_,
                text,
                form_type_map: formTypeMap,
            };
            this.instance_.rpc('Translate', message, (reply) => {
                let cells = null;
                let textToBraille = null;
                let brailleToText = null;
                if (reply['success'] && typeof reply['cells'] === 'string') {
                    cells = Translator.decodeHexString_(reply['cells']);
                    if (reply['text_to_braille'] !== undefined) {
                        textToBraille = reply['text_to_braille'];
                    }
                    if (reply['braille_to_text'] !== undefined) {
                        brailleToText = reply['braille_to_text'];
                    }
                }
                else if (text.length > 0) {
                    // TODO(plundblad): The nacl wrapper currently returns an error
                    // when translating an empty string.  Address that and always log
                    // here.
                    console.error('Braille translation error for ' + JSON.stringify(message));
                }
                callback(cells, textToBraille, brailleToText);
            });
        }
        /**
         * Translates braille cells into text.
         * @param cells Cells to be translated.
         * @param callback Callback for result.
         */
        backTranslate(cells, callback) {
            if (!this.instance_.worker) {
                callback(null /*text*/);
                return;
            }
            if (cells.byteLength === 0) {
                // liblouis doesn't handle empty input, so handle that trivially
                // here.
                callback('');
                return;
            }
            const message = {
                'table_names': this.tableNames_,
                'cells': Translator.encodeHexString_(cells),
            };
            this.instance_.rpc('BackTranslate', message, (reply) => {
                if (!reply['success'] || typeof reply['text'] !== 'string') {
                    callback(null /* text */);
                    return;
                }
                let text = reply['text'];
                // TODO(https://crbug.com/1340087): LibLouis has bugs in
                // backtranslation.
                const view = new Uint8Array(cells);
                if (view.length > 0 && view[view.length - 1] === 0 &&
                    !text.endsWith(' ')) {
                    // LibLouis omits spaces for some backtranslated contractions even
                    // though it is passed a blank cell. This is a workaround until
                    // LibLouis fixes this issue.
                    text += ' ';
                }
                callback(text);
            });
        }
        /**
         * Decodes a hexadecimal string to an {@code ArrayBuffer}.
         * @param hex Hexadecimal string.
         * @return Decoded binary data.
         */
        static decodeHexString_(hex) {
            if (!/^([0-9a-f]{2})*$/i.test(hex)) {
                throw Error('invalid hexadecimal string');
            }
            const array = new Uint8Array(hex.length / 2);
            let idx = 0;
            for (let i = 0; i < hex.length; i += 2) {
                array[idx++] = parseInt(hex.substring(i, i + 2), 16);
            }
            return array.buffer;
        }
        /**
         * Encodes an {@code ArrayBuffer} in hexadecimal.
         * @param arrayBuffer Binary data.
         * @return Hexadecimal string.
         */
        static encodeHexString_(arrayBuffer) {
            const array = new Uint8Array(arrayBuffer);
            let hex = '';
            for (const b of array) {
                hex += (b < 0x10 ? '0' : '') + b.toString(16);
            }
            return hex;
        }
    }
    LibLouis.Translator = Translator;
})(LibLouis || (LibLouis = {}));
TestImportManager.exportForTesting(LibLouis);

// Copyright 2014 The Chromium Authors
// Use of this source code is governed by a BSD-style license that can be
// found in the LICENSE file.
/**
 * @fileoverview Keeps track of the current braille translators.
 */
class BrailleTranslatorManager {
    liblouis_;
    changeListeners_ = [];
    tables_ = [];
    expandingTranslator_ = null;
    defaultTableId_ = null;
    defaultTranslator_ = null;
    uncontractedTableId_ = null;
    uncontractedTranslator_ = null;
    static instance;
    constructor(liblouisForTest) {
        this.liblouis_ =
            liblouisForTest ||
                new LibLouis(chrome.extension.getURL('chromevox/third_party/liblouis/liblouis_wrapper.js'), chrome.extension.getURL('chromevox/mv2/background/braille/tables'), () => this.loadLiblouis_());
    }
    static init() {
        if (BrailleTranslatorManager.instance) {
            throw new Error('\nCannot create two BrailleTranslatorManagers');
        }
        BrailleTranslatorManager.instance = new BrailleTranslatorManager();
        SettingsManager.addListenerForKey('brailleTable', (brailleTable) => BrailleTranslatorManager.instance.refresh(brailleTable));
    }
    static backTranslate(cells) {
        return new Promise(resolve => {
            const translator = BrailleTranslatorManager.instance.getDefaultTranslator();
            if (!translator) {
                console.error('Braille translator is null, cannot call backTranslate');
                resolve('');
                return;
            }
            translator.backTranslate(cells, resolve);
        });
    }
    /**
     * Adds a listener to be called whenever there is a change in the
     * translator(s) returned by other methods of this instance.
     */
    addChangeListener(listener) {
        this.changeListeners_.push(listener);
    }
    /**
     * Refreshes the braille translator(s) used for input and output.  This
     * should be called when something has changed (such as a preference) to
     * make sure that the correct translator is used.
     * @param brailleTable The table for this translator to use.
     * @param brailleTable8 Optionally specify an uncontracted table.
     * @param finishCallback Called when the refresh finishes.
     */
    async refresh(brailleTable, brailleTable8, finishCallback) {
        finishCallback = finishCallback ?? (() => { });
        if (brailleTable && brailleTable === this.defaultTableId_) {
            finishCallback();
            return;
        }
        const tables = this.tables_;
        if (tables.length === 0) {
            finishCallback();
            return;
        }
        // Look for the table requested.
        let table = BrailleTable.forId(tables, brailleTable);
        if (!table) {
            // Match table against current locale.
            const currentLocale = chrome.i18n.getMessage('@@ui_locale').split(/[_-]/);
            const major = currentLocale[0];
            const minor = currentLocale[1];
            const firstPass = tables.filter(table => table.locale.split(/[_-]/)[0] === major);
            if (firstPass.length > 0) {
                table = firstPass[0];
                if (minor) {
                    const secondPass = firstPass.filter(table => table.locale.split(/[_-]/)[1] === minor);
                    if (secondPass.length > 0) {
                        table = secondPass[0];
                    }
                }
            }
        }
        if (!table) {
            table = BrailleTable.forId(tables, 'en-nabcc');
        }
        // If the user explicitly set an 8 dot table, use that when looking
        // for an uncontracted table.  Otherwise, use the current table and let
        // getUncontracted find an appropriate corresponding table.
        const table8Dot = brailleTable8 ? BrailleTable.forId(tables, brailleTable8) : null;
        const uncontractedTable = BrailleTable.getUncontracted(tables, table8Dot ?? table);
        const newDefaultTableId = table.id;
        const newUncontractedTableId = table.id === uncontractedTable.id ? null : uncontractedTable.id;
        if (newDefaultTableId === this.defaultTableId_ &&
            newUncontractedTableId === this.uncontractedTableId_) {
            finishCallback();
            return;
        }
        const finishRefresh = (defaultTranslator, uncontractedTranslator) => {
            this.defaultTableId_ = newDefaultTableId;
            this.uncontractedTableId_ = newUncontractedTableId;
            // TODO(crbug.com/314203187): Not null asserted, check that this is
            // correct.
            this.expandingTranslator_ = defaultTranslator ?
                new ExpandingBrailleTranslator(defaultTranslator, uncontractedTranslator) :
                null;
            this.defaultTranslator_ = defaultTranslator;
            this.uncontractedTranslator_ = uncontractedTranslator;
            this.changeListeners_.forEach(listener => listener());
            finishCallback();
        };
        const translator = await this.liblouis_.getTranslator(table.fileNames);
        if (!newUncontractedTableId) {
            finishRefresh(translator, null);
        }
        else {
            const uncontractedTranslator = await this.liblouis_.getTranslator(uncontractedTable.fileNames);
            finishRefresh(translator, uncontractedTranslator);
        }
    }
    /**
     * @return The current expanding braille translator, or null if none is
     * available.
     */
    getExpandingTranslator() {
        return this.expandingTranslator_;
    }
    /**
     * @return The current braille translator to use by default, or null if none
     * is available.
     */
    getDefaultTranslator() {
        return this.defaultTranslator_;
    }
    /**
     * @return The current uncontracted braille translator, or null if it is the
     * same as the default translator.
     */
    getUncontractedTranslator() {
        return this.uncontractedTranslator_;
    }
    /** Toggles the braille table type. */
    toggleBrailleTable() {
        let brailleTableType = SettingsManager.getString('brailleTableType');
        let output = '';
        if (brailleTableType === 'brailleTable6') {
            brailleTableType = 'brailleTable8';
            // This label reads "switch to 8 dot braille".
            output = '@OPTIONS_BRAILLE_TABLE_TYPE_6';
        }
        else {
            brailleTableType = 'brailleTable6';
            // This label reads "switch to 6 dot braille".
            output = '@OPTIONS_BRAILLE_TABLE_TYPE_8';
        }
        const brailleTable = SettingsManager.getString(brailleTableType);
        SettingsManager.set('brailleTable', brailleTable);
        SettingsManager.set('brailleTableType', brailleTableType);
        this.refresh(brailleTable);
        new Output().format(output).go();
    }
    /**
     * Asynchronously fetches the list of braille tables and refreshes the
     * translators when done.
     * Resolves when tables are loaded.
     */
    async fetchTables_() {
        return new Promise((r) => {
            BrailleTable.getAll(tables => {
                this.tables_ = tables;
                // Initial refresh; set options from user preferences.
                this.refresh(SettingsManager.getString('brailleTable'), undefined, r);
            });
        });
    }
    /**
     * Loads the liblouis instance by attaching it to the document.
     */
    loadLiblouis_() {
        this.fetchTables_();
    }
    getLibLouisForTest() {
        return this.liblouis_;
    }
    /**
     * @return The currently loaded braille tables, or an empty array if they are
     * not yet loaded.
     */
    getTablesForTest() {
        return this.tables_;
    }
    /** Loads liblouis tables and returns a promise resolved when loaded. */
    async loadTablesForTest() {
        await this.fetchTables_();
    }
}
TestImportManager.exportForTesting(BrailleTranslatorManager);

// Copyright 2022 The Chromium Authors
// Use of this source code is governed by a BSD-style license that can be
// found in the LICENSE file.
/**
 * Dots representing a cursor.
 */
const CURSOR_DOTS = 1 << 6 | 1 << 7;
TestImportManager.exportForTesting(['CURSOR_DOTS', CURSOR_DOTS]);

// Copyright 2015 The Chromium Authors
// Use of this source code is governed by a BSD-style license that can be
// found in the LICENSE file.
/**
 * @fileoverview Logic for panning a braille display within a line of braille
 * content that might not fit on a single display.
 */
class PanStrategy {
    displaySize_ = { rows: 1, columns: 40 };
    /** Start and end are both inclusive. */
    viewPort_ = { firstRow: 0, lastRow: 0 };
    /**
     * The ArrayBuffer holding the braille cells after it's been processed to
     * wrap words that are cut off by the column boundaries.
     */
    wrappedBuffer_ = new ArrayBuffer(0);
    /**
     * The original text that corresponds with the braille buffers. There is
     * only one textBuffer that correlates with both fixed and wrapped buffers.
     */
    textBuffer_ = '';
    /**
     * The ArrayBuffer holding the original braille cells, without being
     * processed to wrap words.
     */
    fixedBuffer_ = new ArrayBuffer(0);
    /**
     * The updated mapping from braille cells to text characters for the wrapped
     * buffer.
     */
    wrappedBrailleToText_ = [];
    /** The original mapping from braille cells to text characters. */
    fixedBrailleToText_ = [];
    /**
     * Indicates whether the pan strategy is wrapped or fixed. It is wrapped
     * when true.
     */
    panStrategyWrapped_ = false;
    cursor_ = { start: -1, end: -1 };
    wrappedCursor_ = { start: -1, end: -1 };
    /**
     * Gets the current viewport which is never larger than the current
     * display size and whose end points are always within the limits of
     * the current content.
     */
    get viewPort() {
        return this.viewPort_;
    }
    /** Gets the current displaySize. */
    get displaySize() {
        return this.displaySize_;
    }
    /** @return The offset of braille and text indices of the current slice. */
    get offsetsForSlices() {
        return {
            brailleOffset: this.viewPort_.firstRow * this.displaySize_.columns,
            textOffset: this.brailleToText[this.viewPort_.firstRow * this.displaySize_.columns],
        };
    }
    /** @return The number of lines in the fixedBuffer. */
    get fixedLineCount() {
        return Math.ceil(this.fixedBuffer_.byteLength / this.displaySize_.columns);
    }
    /** @return The number of lines in the wrappedBuffer. */
    get wrappedLineCount() {
        return Math.ceil(this.wrappedBuffer_.byteLength / this.displaySize_.columns);
    }
    /**
     * @return The map of Braille cells to the first index of the corresponding
     *     text character.
     */
    get brailleToText() {
        if (this.panStrategyWrapped_) {
            return this.wrappedBrailleToText_;
        }
        else {
            return this.fixedBrailleToText_;
        }
    }
    /**
     * @return Buffer of the slice of braille cells within the bounds of the
     * viewport.
     */
    getCurrentBrailleViewportContents(showCursor = true) {
        const buf = this.panStrategyWrapped_ ? this.wrappedBuffer_ : this.fixedBuffer_;
        let startIndex;
        let endIndex;
        if (this.panStrategyWrapped_) {
            startIndex = this.wrappedCursor_.start;
            endIndex = this.wrappedCursor_.end;
        }
        else {
            startIndex = this.cursor_.start;
            endIndex = this.cursor_.end;
        }
        if (startIndex >= 0 && startIndex < buf.byteLength &&
            endIndex >= startIndex && endIndex <= buf.byteLength) {
            const dataView = new DataView(buf);
            while (startIndex < endIndex) {
                let value = dataView.getUint8(startIndex);
                if (showCursor) {
                    value |= CURSOR_DOTS;
                }
                else {
                    value &= ~CURSOR_DOTS;
                }
                dataView.setUint8(startIndex, value);
                startIndex++;
            }
        }
        return buf.slice(this.viewPort_.firstRow * this.displaySize_.columns, (this.viewPort_.lastRow + 1) * this.displaySize_.columns);
    }
    /**
     * @return String of the slice of text letters corresponding with the current
     * braille slice.
     */
    getCurrentTextViewportContents() {
        const brailleToText = this.brailleToText;
        // Index of last braille character in slice.
        let index = (this.viewPort_.lastRow + 1) * this.displaySize_.columns - 1;
        // Index of first text character that the last braille character points
        // to.
        const end = brailleToText[index];
        // Increment index until brailleToText[index] points to a different char.
        // This is the cutoff point, as substring cuts up to, but not including,
        // brailleToText[index].
        while (index < brailleToText.length && brailleToText[index] === end) {
            index++;
        }
        return this.textBuffer_.substring(brailleToText[this.viewPort_.firstRow * this.displaySize_.columns], brailleToText[index]);
    }
    /** Sets the current pan strategy and resets the viewport. */
    setPanStrategy(wordWrap) {
        this.panStrategyWrapped_ = wordWrap;
        this.panToPosition_(0);
    }
    /**
     * Sets the display size.  This call may update the viewport.
     * @param rowCount the new row size, or 0 if no display is present.
     * @param columnCount the new column size, or 0 if no display is
     * present.
     */
    setDisplaySize(rowCount, columnCount) {
        this.displaySize_ = { rows: rowCount, columns: columnCount };
        this.setContent(this.textBuffer_, this.fixedBuffer_, this.fixedBrailleToText_, 0);
    }
    /**
     * Sets the internal data structures that hold the fixed and wrapped buffers
     * and maps.
     * @param textBuffer Text of the shown braille.
     * @param translatedContent The new braille content.
     * @param fixedBrailleToText Map of Braille cells to the first index of
     *     corresponding text letter.
     * @param targetPosition Target position.  The viewport is changed to overlap
     *     this position.
     */
    setContent(textBuffer, translatedContent, fixedBrailleToText, targetPosition) {
        this.viewPort_.firstRow = 0;
        this.viewPort_.lastRow = this.displaySize_.rows - 1;
        this.fixedBrailleToText_ = fixedBrailleToText;
        this.wrappedBrailleToText_ = [];
        this.textBuffer_ = textBuffer;
        this.fixedBuffer_ = translatedContent;
        // Convert the cells to Unicode braille pattern characters.
        const view = new Uint8Array(translatedContent);
        // Check for a multi-line display and encode with horizontal and vertical
        // spacing if so.
        //
        // We currently insert one column of dots horizontally between cells and two
        // rows of dots vertically between lines.
        //
        // TODO(accessibility): extend this to work with word wrapping below.
        if (!this.panStrategyWrapped_ && this.displaySize_.rows > 1) {
            // All known displays have even number of rows and columns.
            // TODO(accessibility): this check should move elsewhere.
            if (this.displaySize_.rows % 2 !== 0 ||
                this.displaySize_.columns % 2 !== 0) {
                return;
            }
            // Iterate in two-byte groupings.
            const horizontalSpacedBraille = [];
            for (let index = 0; index < translatedContent.byteLength; index += 2) {
                const first = view[index];
                const second = view[index + 1];
                // The first cell is always written.
                horizontalSpacedBraille.push(first);
                // The second cell turns into two cells.
                // The initial cell has one vertical blank dot column on the left.
                horizontalSpacedBraille.push((second & 0b111) << 3 | (second & 0b1000000) << 1);
                // The next cell has one vertical blank dot column on the right.
                horizontalSpacedBraille.push((second & 0b111000) >> 3 | (second & 0b10000000) >> 1);
            }
            // Now, space the lines vertically by inserting two blank dot rows.
            let spacedBraille = [];
            // Iterate by two cell rows at once.
            for (let row = 0; row < this.displaySize_.rows; row += 2) {
                // The first row gets added verbatim.
                for (let index = 0; index < this.displaySize_.columns; index++) {
                    const rowIndex = row * this.displaySize_.columns + index;
                    spacedBraille.push(horizontalSpacedBraille[rowIndex]);
                }
                // The second cell row turns into two cell rows: an upper row with
                // spacing a blank dot row above, and a lower cell row with blank dot
                // row spacing below. The upper cell row can be pushed below; store the
                // lower cell row for after.
                const nextRow = row + 1;
                const lowerRow = [];
                for (let index = 0; index < this.displaySize_.columns; index++) {
                    const rowIndex = nextRow * this.displaySize_.columns + index;
                    const value = horizontalSpacedBraille[rowIndex];
                    // Downshift the top two dots by two positions.
                    // e.g. dot 1 goes to dot 3, dot 4 to dot 6.
                    let upperRowValue = (value & 0b1001) << 2;
                    // Downshift dot 2 to dot 7.
                    upperRowValue |= (value & 0b010) << 5;
                    // Downshift dot 5 to dot 8.
                    upperRowValue |= (value & 0b10000) << 3;
                    spacedBraille.push(upperRowValue);
                    // Lower row.
                    // Upshift the bottom two dots by two positions.
                    // e.g. dot 3 to dot 1, dot 6 to dot 4.
                    let lowerRowValue = (value & 0b100100) >> 2;
                    // Upshift dot 7 to dot 2.
                    lowerRowValue |= (value & 0b1000000) >> 5;
                    // Upshift dot 8 to dot 5.
                    lowerRowValue |= (value & 0b10000000) >> 3;
                    lowerRow.push(lowerRowValue);
                }
                spacedBraille = spacedBraille.concat(lowerRow);
            }
            const brailleUint8Array = new Uint8Array(spacedBraille);
            this.fixedBuffer_ = new ArrayBuffer(brailleUint8Array.length);
            new Uint8Array(this.fixedBuffer_).set(brailleUint8Array);
        }
        const wrappedBrailleArray = [];
        let lastBreak = 0;
        let cellsPadded = 0;
        let index;
        for (index = 0; index < translatedContent.byteLength + cellsPadded; index++) {
            // Is index at the beginning of a new line?
            if (index !== 0 && index % this.displaySize_.columns === 0) {
                if (view[index - cellsPadded] === 0) {
                    // Delete all empty cells at the beginning of this line.
                    while (index - cellsPadded < view.length &&
                        view[index - cellsPadded] === 0) {
                        cellsPadded--;
                    }
                    index--;
                    lastBreak = index;
                }
                else if (view[index - cellsPadded - 1] !== 0 &&
                    lastBreak % this.displaySize_.columns !== 0) {
                    // If first cell is not empty, we need to move the whole word down
                    // to this line and padd to previous line with 0's, from |lastBreak|
                    // to index. The braille to text map is also updated. If lastBreak
                    // is at the beginning of a line, that means the current word is
                    // bigger than |this.displaySize_.columns| so we shouldn't wrap.
                    for (let j = lastBreak + 1; j < index; j++) {
                        wrappedBrailleArray[j] = 0;
                        this.wrappedBrailleToText_[j] = this.wrappedBrailleToText_[j - 1];
                        cellsPadded++;
                    }
                    lastBreak = index;
                    index--;
                }
                else {
                    // |lastBreak| is at the beginning of a line, so current word is
                    // bigger than |this.displaySize_.columns| so we shouldn't wrap.
                    this.maybeSetWrappedCursor_(index - cellsPadded, wrappedBrailleArray.length);
                    wrappedBrailleArray.push(view[index - cellsPadded]);
                    this.wrappedBrailleToText_.push(fixedBrailleToText[index - cellsPadded]);
                }
            }
            else {
                if (view[index - cellsPadded] === 0) {
                    lastBreak = index;
                }
                this.maybeSetWrappedCursor_(index - cellsPadded, wrappedBrailleArray.length);
                wrappedBrailleArray.push(view[index - cellsPadded]);
                this.wrappedBrailleToText_.push(fixedBrailleToText[index - cellsPadded]);
            }
        }
        // It's possible the end of the wrapped cursor falls at the
        // |translatedContent.byteLength| exactly.
        this.maybeSetWrappedCursor_(index - cellsPadded, wrappedBrailleArray.length);
        // Convert the wrapped Braille Uint8 Array back to ArrayBuffer.
        const wrappedBrailleUint8Array = new Uint8Array(wrappedBrailleArray);
        this.wrappedBuffer_ = new ArrayBuffer(wrappedBrailleUint8Array.length);
        new Uint8Array(this.wrappedBuffer_).set(wrappedBrailleUint8Array);
        this.panToPosition_(targetPosition);
    }
    setCursor(startIndex, endIndex) {
        this.cursor_ = { start: startIndex, end: endIndex };
    }
    getCursor() {
        return this.cursor_;
    }
    /**
     * Refreshes the wrapped cursor given a mapping from an unwrapped index to a
     * wrapped index.
     */
    maybeSetWrappedCursor_(unwrappedIndex, wrappedIndex) {
        // We only care about the bounds of the index start/end.
        if (this.cursor_.start !== unwrappedIndex &&
            this.cursor_.end !== unwrappedIndex) {
            return;
        }
        if (this.cursor_.start === unwrappedIndex) {
            this.wrappedCursor_.start = wrappedIndex;
        }
        else if (this.cursor_.end === unwrappedIndex) {
            this.wrappedCursor_.end = wrappedIndex;
        }
    }
    /**
     * If possible, changes the viewport to a part of the line that follows
     * the current viewport.
     * @return true if the viewport was changed.
     */
    next() {
        const contentLength = this.panStrategyWrapped_ ? this.wrappedLineCount : this.fixedLineCount;
        const newStart = this.viewPort_.lastRow + 1;
        let newEnd;
        if (newStart + this.displaySize_.rows - 1 < contentLength) {
            newEnd = newStart + this.displaySize_.rows - 1;
        }
        else {
            newEnd = contentLength - 1;
        }
        if (newEnd >= newStart) {
            this.viewPort_ = { firstRow: newStart, lastRow: newEnd };
            return true;
        }
        return false;
    }
    /**
     * If possible, changes the viewport to a part of the line that precedes
     * the current viewport.
     * @return true if the viewport was changed.
     */
    previous() {
        const contentLength = this.panStrategyWrapped_ ? this.wrappedLineCount : this.fixedLineCount;
        if (this.viewPort_.firstRow > 0) {
            let newStart;
            let newEnd;
            if (this.viewPort_.firstRow < this.displaySize_.rows) {
                newStart = 0;
                newEnd = Math.min(this.displaySize_.rows, contentLength);
            }
            else {
                newEnd = this.viewPort_.firstRow - 1;
                newStart = newEnd - this.displaySize_.rows + 1;
            }
            if (newStart <= newEnd) {
                this.viewPort_ = { firstRow: newStart, lastRow: newEnd };
                return true;
            }
        }
        return false;
    }
    /**
     * Moves the viewport so that it overlaps a target position without taking
     * the current viewport position into consideration.
     * @param position Target position.
     */
    panToPosition_(position) {
        if (this.displaySize_.rows * this.displaySize_.columns > 0) {
            this.viewPort_ = { firstRow: -1, lastRow: -1 };
            while (this.next() &&
                (this.viewPort_.lastRow + 1) * this.displaySize_.columns <=
                    position) {
                // Nothing to do.
            }
        }
        else {
            this.viewPort_ = { firstRow: position, lastRow: position };
        }
    }
}
TestImportManager.exportForTesting(PanStrategy);

// Copyright 2014 The Chromium Authors
// Use of this source code is governed by a BSD-style license that can be
// found in the LICENSE file.
/**
 * @fileoverview Puts text on a braille display.
 */
class BrailleDisplayManager {
    blinkerId_;
    content_ = new NavBraille({});
    commandListener_ = () => { };
    /**
     * Current display state to show in the Virtual Braille Captions display.
     * This is different from realDisplayState_ if the braille captions feature
     * is enabled and there is no hardware display connected.  Otherwise, it is
     * the same object as realDisplayState_.
     */
    displayState_ = { available: false, textRowCount: 0, textColumnCount: 0, cellSize: 0 };
    expansionType_ = ExpandingBrailleTranslator.ExpansionType.SELECTION;
    panStrategy_ = new PanStrategy();
    /**
     * State reported from the chrome api, reflecting a real hardware
     * display.
     */
    realDisplayState_ = this.displayState_;
    static instance;
    constructor() {
        BrailleTranslatorManager.instance.addChangeListener(() => this.translateContent_(this.content_, this.expansionType_));
        SettingsManager.addListenerForKey('brailleWordWrap', (wrap) => this.updatePanStrategy_(wrap));
        SettingsManager.addListenerForKey('virtualBrailleRows', () => this.onBrailleCaptionsStateChanged());
        SettingsManager.addListenerForKey('virtualBrailleColumns', () => this.onBrailleCaptionsStateChanged());
        this.updatePanStrategy_(SettingsManager.getBoolean('brailleWordWrap'));
        BrailleCaptionsBackground.init(this);
        if (chrome.brailleDisplayPrivate !== undefined) {
            const onDisplayStateChanged = (newState) => this.refreshDisplayState_(newState);
            chrome.brailleDisplayPrivate.getDisplayState(onDisplayStateChanged);
            chrome.brailleDisplayPrivate.onDisplayStateChanged.addListener(onDisplayStateChanged);
            chrome.brailleDisplayPrivate.onKeyEvent.addListener((event) => this.onKeyEvent_(event));
        }
        else {
            // Get the initial captions state since we won't refresh the display
            // state in an API callback in this case.
            this.onBrailleCaptionsStateChanged();
        }
    }
    static init() {
        if (BrailleDisplayManager.instance) {
            throw new Error('Cannot create two BrailleDisplayManager instances');
        }
        BrailleDisplayManager.instance = new BrailleDisplayManager();
    }
    /**
     * Takes an image, in the form of a data url, and converts it to braille.
     * @param imageUrl The image, in the form of a data url.
     * @return The image, encoded in binary form, suitable for writing to a
     *     braille display.
     */
    static async convertImageDataUrlToBraille(imageUrl, displayState) {
        // The number of dots in a braille cell.
        // All known displays have a cell width of 2.
        const cellWidth = 2;
        // Derive the cell height from the display's cell size and the width above
        // e.g. 8 / 2 = 4.
        let cellHeight = displayState.cellSize / cellWidth;
        // Sanity check.
        if (cellHeight === 0) {
            // Set the height to a reasonable min.
            cellHeight = 3;
        }
        // All known displays don't exceed a cell height of 4.
        const maxCellHeight = 4;
        const rows = displayState.textRowCount;
        const columns = displayState.textColumnCount;
        const imageDataUrl = imageUrl;
        return new Promise((resolve) => {
            const imgElement = document.createElement('img');
            imgElement.src = imageDataUrl;
            imgElement.onload = () => {
                const canvas = document.createElement('canvas');
                // TODO(b/314203187): Not null asserted, check that this is correct.
                const context = canvas.getContext('2d');
                canvas.width = columns * cellWidth;
                canvas.height = rows * cellHeight;
                context.drawImage(imgElement, 0, 0, canvas.width, canvas.height);
                const imageData = context.getImageData(0, 0, canvas.width, canvas.height);
                const brailleBuf = BrailleDisplayManager.convertImageDataToBraille(imageData.data, { rows, columns, cellWidth, cellHeight, maxCellHeight });
                resolve(brailleBuf);
            };
        });
    }
    /**
     * @param data Encodes pixel data p_1, ..., p_n in groupings of RGBA. For
     *     example, for pixel 1, p_1_r, p_1_g, p_1_b, p_1_a. 1 ... n go from left
     *     to right, top to bottom.
     *
     *     The array looks like:
     *     [p_1_r, p_1_g, p_1_b, p_1_a, ... p_n_r, p_n_g, p_n_b, p_n_a].
     * @param state Dimensions of the braille display in cell units (number of
     *     rows, columns) and dot units (cell*).
     * @return a buffer encoding the braille dots according to that expected by
     *     BRLTTY.
     */
    static convertImageDataToBraille(data, state) {
        const { rows, columns, cellWidth, cellHeight, maxCellHeight } = state;
        const outputData = [];
        // The data should have groupings of 4 i.e. visible by 4.
        if (data.length % 4 !== 0) {
            return new ArrayBuffer(0);
        }
        // Convert image to black and white by thresholding the luminance for
        // all opaque (non-transparent) pixels.
        for (let i = 0; i < data.length; i += 4) {
            const red = data[i];
            const green = data[i + 1];
            const blue = data[i + 2];
            const alpha = data[i + 3];
            const luminance = 0.2126 * red + 0.7152 * green + 0.0722 * blue;
            // TODO(accessibility): this is a naive way to threshold. Consider
            // computing a global threshold based on an average of the top two most
            // frequent values using a histogram.
            // Show braille pin if the alpha is greater than the threshold and
            // the luminance is less than the threshold.
            const show = (alpha >= ALPHA_THRESHOLD &&
                luminance < LUMINANCE_THRESHOLD);
            outputData.push(show);
        }
        // Pick the mapping for the given cell height (default to 6-dot).
        const DOT_MAP = cellHeight === 4 ? COORDS_TO_BRAILLE_8DOT : COORDS_TO_BRAILLE_6DOT;
        // Convert black-and-white array to the proper encoding for Braille
        // cells.
        const brailleBuf = new ArrayBuffer(rows * columns);
        const view = new Uint8Array(brailleBuf);
        for (let i = 0; i < rows; i++) {
            for (let j = 0; j < columns; j++) {
                // Index in braille array.
                const brailleIndex = i * columns + j;
                for (let cellColumn = 0; cellColumn < cellWidth; cellColumn++) {
                    for (let cellRow = 0; cellRow < cellHeight; cellRow++) {
                        const bitmapIndex = (i * columns * cellHeight + j + cellRow * columns) * cellWidth +
                            cellColumn;
                        if (outputData[bitmapIndex]) {
                            const index = cellColumn * maxCellHeight + cellRow;
                            view[brailleIndex] += DOT_MAP[index];
                        }
                    }
                }
            }
        }
        return brailleBuf;
    }
    /**
     * @param content Content to send to the braille display.
     * @param expansionType If the text has a {@code ValueSpan}, this indicates
     *     how that part of the display content is expanded when translating to
     *     braille. (See {@code ExpandingBrailleTranslator}).
     */
    setContent(content, expansionType) {
        this.translateContent_(content, expansionType);
    }
    /**
     * Takes an image, in the form of a data url, and displays it in braille
     * onto the physical braille display and the virtual braille captions display.
     * @param imageUrl The image, in the form of a data url.
     */
    setImageContent(imageUrl) {
        if (!this.displayState_.available) {
            return;
        }
        BrailleDisplayManager
            .convertImageDataUrlToBraille(imageUrl, this.displayState_)
            .then((brailleBuf) => {
            if (this.realDisplayState_.available) {
                chrome.brailleDisplayPrivate.writeDots(brailleBuf, this.displayState_.textColumnCount, this.displayState_.textRowCount);
            }
            if (BrailleCaptionsBackground.isEnabled()) {
                BrailleCaptionsBackground.setImageContent(brailleBuf, this.displayState_.textRowCount, this.displayState_.textColumnCount);
            }
        });
    }
    /**
     * Sets the command listener.  When a command is invoked, the listener will be
     * called with the BrailleKeyEvent corresponding to the command and the
     * content that was present on the display when the command was invoked.  The
     * content is guaranteed to be identical to an object previously used as the
     * parameter to BrailleDisplayManager.setContent, or null if no content was
     * set.
     */
    setCommandListener(func) {
        this.commandListener_ = func;
    }
    /** @return The current display state. */
    getDisplayState() {
        return this.displayState_;
    }
    /**
     * @param newState Display state reported by the extension API. Note that the
     *     type is almost the same as BrailleDisplayState except that the
     *     extension API allows some fields to be undefined, while
     *     BrailleDisplayState does not.
     */
    refreshDisplayState_(newState) {
        const oldColumnCount = this.displayState_.textColumnCount || 0;
        const oldRowCount = this.displayState_.textRowCount || 0;
        const oldCellSize = this.displayState_.cellSize || 0;
        const processDisplayState = (displayState) => {
            this.displayState_ = displayState;
            const newColumnCount = displayState.textColumnCount || 0;
            const newRowCount = displayState.textRowCount || 0;
            const newCellSize = displayState.cellSize || 0;
            if (oldColumnCount !== newColumnCount || oldRowCount !== newRowCount ||
                oldCellSize !== newCellSize) {
                // TODO(accessibility): Audit whether changes in cellSize need to be
                // communicated to PanStrategy.
                this.panStrategy_.setDisplaySize(newRowCount, newColumnCount);
            }
            this.refresh_();
        };
        this.realDisplayState_ = {
            available: newState.available,
            textRowCount: newState.textRowCount || 0,
            textColumnCount: newState.textColumnCount || 0,
            cellSize: newState.cellSize || 0,
        };
        if (newState.available) {
            // Update the dimensions of the virtual braille captions display to those
            // of a real physical display when one is plugged in.
            processDisplayState(newState);
            SettingsManager.set('menuBrailleCommands', true);
        }
        else {
            processDisplayState(BrailleCaptionsBackground.getVirtualDisplayState());
            SettingsManager.set('menuBrailleCommands', false);
        }
    }
    /**
     * BrailleCaptionsListener implementation.
     * Called when the state of braille captions changes.
     */
    onBrailleCaptionsStateChanged() {
        // Force reevaluation of the display state based on our stored real
        // hardware display state, meaning that if a real display is connected,
        // that takes precedence over the state from the captions 'virtual' display.
        this.refreshDisplayState_(this.realDisplayState_);
    }
    /**
     * Refreshes what is shown on the physical braille display and the virtual
     * braille captions display.
     */
    refresh_() {
        if (this.blinkerId_ !== undefined) {
            clearInterval(this.blinkerId_);
        }
        // If there's no cursor, don't schedule blinking.
        const cursor = this.panStrategy_.getCursor();
        const hideCursor = cursor.start === -1 || cursor.end === -1;
        this.refreshInternal_(!hideCursor);
        if (hideCursor) {
            return;
        }
        let showCursor = false;
        this.blinkerId_ = setInterval(() => {
            this.refreshInternal_(showCursor);
            showCursor = !showCursor;
        }, BrailleDisplayManager.CURSOR_BLINK_TIME_MS);
    }
    /** @param showCursor Whether to show the cursor. */
    refreshInternal_(showCursor) {
        if (!this.displayState_.available) {
            return;
        }
        const brailleBuf = this.panStrategy_.getCurrentBrailleViewportContents(showCursor);
        const textBuf = this.panStrategy_.getCurrentTextViewportContents();
        if (this.realDisplayState_.available) {
            chrome.brailleDisplayPrivate.writeDots(brailleBuf, this.realDisplayState_.textColumnCount, this.realDisplayState_.textRowCount);
        }
        if (BrailleCaptionsBackground.isEnabled()) {
            BrailleCaptionsBackground.setContent(textBuf, brailleBuf, this.panStrategy_.brailleToText, this.panStrategy_.offsetsForSlices, this.displayState_.textRowCount, this.displayState_.textColumnCount);
        }
    }
    /**
     * @param newContent New display content.
     * @param newExpansionType How the value part of of the new content should be
     *     expanded with regards to contractions.
     */
    translateContent_(newContent, newExpansionType) {
        const writeTranslatedContent = (cells, textToBraille, brailleToText) => {
            this.content_ = newContent;
            this.expansionType_ = newExpansionType;
            const startIndex = this.content_.startIndex;
            const endIndex = this.content_.endIndex;
            let targetPosition;
            if (startIndex >= 0) {
                let translatedStartIndex;
                let translatedEndIndex;
                if (startIndex >= textToBraille.length) {
                    // Allow the cells to be extended with one extra cell for
                    // a carret after the last character.
                    const extCells = new ArrayBuffer(cells.byteLength + 1);
                    new Uint8Array(extCells).set(new Uint8Array(cells));
                    // Last byte is initialized to 0.
                    cells = extCells;
                    translatedStartIndex = cells.byteLength - 1;
                }
                else {
                    translatedStartIndex = textToBraille[startIndex];
                }
                if (endIndex >= textToBraille.length) {
                    // endIndex can't be past-the-end of the last cell unless
                    // startIndex is too, so we don't have to do another
                    // extension here.
                    translatedEndIndex = cells.byteLength;
                }
                else {
                    translatedEndIndex = textToBraille[endIndex];
                }
                // Add the cursor to cells.
                this.setCursor_(cells, translatedStartIndex, translatedEndIndex);
                targetPosition = translatedStartIndex;
            }
            else {
                this.setCursor_(cells, -1, -1);
                targetPosition = 0;
            }
            this.panStrategy_.setContent(this.content_.text.toString(), cells, brailleToText, targetPosition);
            this.refresh_();
        };
        const translator = BrailleTranslatorManager.instance.getExpandingTranslator();
        if (!translator) {
            writeTranslatedContent(new ArrayBuffer(0), [], []);
        }
        else {
            translator.translate(newContent.text, newExpansionType, writeTranslatedContent);
        }
    }
    onKeyEvent_(event) {
        switch (event.command) {
            case BrailleKeyCommand.PAN_LEFT:
                this.panLeft();
                break;
            case BrailleKeyCommand.PAN_RIGHT:
                this.panRight();
                break;
            case BrailleKeyCommand.ROUTING:
                this.route(event.displayPosition);
                break;
            default:
                this.commandListener_(event, this.content_);
                break;
        }
    }
    /**
     * Shift the display by one full display size and refresh the content.
     * Sends the appropriate command if the display is already at the leftmost
     * position.
     */
    panLeft() {
        if (this.panStrategy_.previous()) {
            this.refresh_();
        }
        else if (CaptionsHandler.inCaptions()) {
            CaptionsHandler.instance.previous();
        }
        else {
            this.commandListener_({ command: BrailleKeyCommand.PAN_LEFT }, this.content_);
        }
    }
    /**
     * Shifts the display position to the right by one full display size and
     * refreshes the content.  Sends the appropriate command if the display is
     * already at its rightmost position.
     */
    panRight() {
        if (this.panStrategy_.next()) {
            this.refresh_();
        }
        else if (CaptionsHandler.inCaptions()) {
            CaptionsHandler.instance.next();
        }
        else {
            this.commandListener_({ command: BrailleKeyCommand.PAN_RIGHT }, this.content_);
        }
    }
    /**
     * Moves the cursor to the given braille position.
     * @param braillePosition The 0-based position relative to the start of the
     *     currently displayed text. The position is given in braille cells, not
     *     text cells.
     */
    route(braillePosition) {
        if (braillePosition === undefined) {
            return;
        }
        const displayPosition = this.brailleToTextPosition_(braillePosition +
            this.panStrategy_.viewPort.firstRow *
                this.panStrategy_.displaySize.columns);
        this.commandListener_({ command: BrailleKeyCommand.ROUTING, displayPosition }, this.content_);
    }
    /**
     * Sets a cursor within translated content.
     * @param buffer Buffer to add cursor to.
     * @param startIndex The start index to place the cursor.
     * @param endIndex The end index to place the cursor (exclusive).
     */
    setCursor_(buffer, startIndex, endIndex) {
        if (startIndex < 0 || startIndex >= buffer.byteLength ||
            endIndex < startIndex || endIndex > buffer.byteLength) {
            this.panStrategy_.setCursor(-1, -1);
            return;
        }
        if (startIndex === endIndex) {
            endIndex = startIndex + 1;
        }
        this.panStrategy_.setCursor(startIndex, endIndex);
    }
    /**
     * Returns the text position corresponding to an absolute braille position,
     * that is not accounting for the current pan position.
     * @param braillePosition Braille position relative to the startof
     *        the translated content.
     * @return The mapped position in code units.
     */
    brailleToTextPosition_(braillePosition) {
        const mapping = this.panStrategy_.brailleToText;
        if (braillePosition < 0) {
            // This shouldn't happen.
            console.error('WARNING: Braille position < 0: ' + braillePosition);
            return 0;
        }
        else if (braillePosition >= mapping.length) {
            // This happens when the user clicks on the right part of the display
            // when it is not entirely filled with content.  Allow addressing the
            // position after the last character.
            return this.content_.text.length;
        }
        else {
            return mapping[braillePosition];
        }
    }
    updatePanStrategy_(wordWrap) {
        this.panStrategy_.setPanStrategy(wordWrap);
        this.refresh_();
    }
}
(function (BrailleDisplayManager) {
    /**
     * Time elapsed before a cursor changes state. This results in a blinking
     * effect.
     */
    BrailleDisplayManager.CURSOR_BLINK_TIME_MS = 1000;
})(BrailleDisplayManager || (BrailleDisplayManager = {}));
// Local to module.
/**
 * Alpha threshold for a pixel to be possibly displayed as a raised dot when
 * converting an image to braille, where 255 means only fully-opaque
 * pixels can be raised (if their luminance passes the luminance threshold),
 * and 0 means that alpha is effectively ignored and only luminance matters.
 */
const ALPHA_THRESHOLD = 255;
/**
 * Luminance threshold for a pixel to be displayed as a raised dot when
 * converting an image to braille, on a scale of 0 (black) to 255 (white).
 * A pixel whose luminance is less than the given threshold will be raised.
 */
const LUMINANCE_THRESHOLD = 128;
/**
 * Array mapping an index in a 6-dot braille cell, in column-first order,
 * to its corresponding bit mask in the standard braille cell encoding.
 */
const COORDS_TO_BRAILLE_6DOT = [0x1, 0x2, 0x4, 0x8, 0x10, 0x20];
/**
 * Array mapping an index in an 8-dot braille cell, in column-first order,
 * to its corresponding bit mask in the standard braille cell encoding.
 */
const COORDS_TO_BRAILLE_8DOT = [0x1, 0x2, 0x4, 0x40, 0x8, 0x10, 0x20, 0x80];
TestImportManager.exportForTesting(BrailleDisplayManager);

// Copyright 2014 The Chromium Authors
// Use of this source code is governed by a BSD-style license that can be
// found in the LICENSE file.
/**
 * @fileoverview Handles braille input keys when the user is typing or editing
 * text in an input field.  This class cooperates with the Braille IME
 * that is built into Chrome OS to do the actual text editing.
 */
/**
 * Regular expression that matches a string that starts with at least one
 * non-whitespace character.
 */
const STARTS_WITH_NON_WHITESPACE_RE = /^\S/;
/**
 * Regular expression that matches a string that ends with at least one
 * non-whitespace character.
 */
const ENDS_WITH_NON_WHITESPACE_RE = /\S$/;
/**
 * The entry state is the state related to entering a series of braille cells
 * without 'interruption', where interruption can be things like non braille
 * keyboard input or unexpected changes to the text surrounding the cursor.
 */
class EntryState {
    translator_;
    inputHandler;
    /** Braille cells that have been typed by the user so far. */
    cells = [];
    /** Text resulting from translating this.cells. */
    text = '';
    /**
     * List of strings that we expect to be set as preceding text of the
     * selection. This is populated when we send text changes to the IME so
     * that our own changes don't reset the pending cells.
     */
    pendingTextsBefore_ = [];
    constructor(inputHandler, translator_) {
        this.translator_ = translator_;
        this.inputHandler = inputHandler;
    }
    /**
     * @return The translator used by this entry state. This doesn't change for a
     * given object.
     */
    get translator() {
        return this.translator_;
    }
    /**
     * Appends a braille cell to the current input and updates the text if
     * necessary.
     * @param cell The braille cell to append.
     */
    appendCell(cell) {
        this.cells.push(cell);
        this.updateText_();
    }
    /**
     * Deletes the last cell of the input and updates the text if neccary.
     * If there's no more input in this object afterwards, clears the entry state
     * of the input handler.
     */
    deleteLastCell() {
        if (--this.cells.length <= 0) {
            this.sendTextChange_('');
            this.inputHandler?.clearEntryState();
            return;
        }
        this.updateText_();
    }
    /**
     * Called when the text before the cursor changes giving this object a
     * chance to clear the entry state of the input handler if the change
     * wasn't expected.
     * @param newText New text before the cursor.
     */
    onTextBeforeChanged(newText) {
        // See if we are expecting this change as a result of one of our own
        // edits. Allow changes to be coalesced by the input system in an attempt
        // to not be too brittle.
        for (let i = 0; i < this.pendingTextsBefore_.length; ++i) {
            if (newText === this.pendingTextsBefore_[i]) {
                // Delete all previous expected changes and ignore this one.
                this.pendingTextsBefore_.splice(0, i + 1);
                return;
            }
        }
        // There was an actual text change (or cursor movement) that we hadn't
        // caused ourselves, reset any pending input.
        this.inputHandler?.clearEntryState();
    }
    /**
     * Makes sure the current text is permanently added to the edit field.
     * After this call, this object should be abandoned.
     */
    commit() { }
    /**
     * @return true if the entry state uses uncommitted cells.
     */
    get usesUncommittedCells() {
        return false;
    }
    /**
     * Updates the translated text based on the current cells and sends the
     * delta to the IME.
     */
    updateText_() {
        const cellsBuffer = new Uint8Array(this.cells).buffer;
        const commit = this.lastCellIsBlank_;
        if (!commit && this.usesUncommittedCells) {
            this.inputHandler?.updateUncommittedCells(cellsBuffer);
        }
        this.translator_.backTranslate(cellsBuffer, result => {
            if (result === null) {
                console.error('Error when backtranslating braille cells');
                return;
            }
            if (!this.inputHandler) {
                return;
            }
            this.sendTextChange_(result);
            this.text = result;
            if (commit) {
                this.inputHandler.commitAndClearEntryState();
            }
        });
    }
    get lastCellIsBlank_() {
        return this.cells[this.cells.length - 1] === 0;
    }
    /**
     * Sends new text to the IME.  This should be overridden by subclasses.
     * The old text is still available in the text property.
     */
    sendTextChange_(_newText) { }
}
/**
 * Entry state that uses deleteSurroundingText and commitText calls to the IME
 * to update the currently entered text.
 */
class EditsEntryState extends EntryState {
    sendTextChange_(newText) {
        const oldText = this.text;
        // Find the common prefix of the old and new text.
        const commonPrefixLength = StringUtil.longestCommonPrefixLength(oldText, newText);
        // How many characters we need to delete from the existing text to replace
        // them with characters from the new text.
        const deleteLength = oldText.length - commonPrefixLength;
        // New text, if any, to insert after deleting the deleteLength characters
        // before the cursor.
        const toInsert = newText.substring(commonPrefixLength);
        if (deleteLength > 0 || toInsert.length > 0) {
            // After deleting, we expect this text to be present before the cursor.
            const textBeforeAfterDelete = this.inputHandler?.currentTextBefore.substring(0, this.inputHandler.currentTextBefore.length - deleteLength);
            if (deleteLength > 0 && textBeforeAfterDelete) {
                // Queue this text up to be ignored when the change comes in.
                this.pendingTextsBefore_.push(textBeforeAfterDelete);
            }
            if (toInsert.length > 0) {
                // Likewise, queue up what we expect to be before the cursor after
                // the replacement text is inserted.
                this.pendingTextsBefore_.push(textBeforeAfterDelete + toInsert);
            }
            // Send the replace operation to be performed asynchronously by the IME.
            this.inputHandler?.postImeMessage({
                type: 'replaceText',
                contextID: this.inputHandler.inputContext?.contextID,
                deleteBefore: deleteLength,
                newText: toInsert,
            });
        }
    }
}
/**
 * Entry state that only updates the edit field when a blank cell is entered.
 * During the input of a single 'word', the uncommitted text is stored by the
 * IME.
 */
class LateCommitEntryState extends EntryState {
    commit() {
        this.inputHandler?.postImeMessage({
            type: 'commitUncommitted',
            contextID: this.inputHandler.inputContext?.contextID,
        });
    }
    get usesUncommittedCells() {
        return true;
    }
    sendTextChange_(newText) {
        this.inputHandler?.postImeMessage({
            type: 'setUncommitted',
            contextID: this.inputHandler.inputContext?.contextID,
            text: newText,
        });
    }
}
class BrailleInputHandler {
    /** Port of the connected IME if any. */
    imePort_ = null;
    /**
     * True when the Braille IME is connected and has signaled that it is
     * active.
     */
    imeActive_ = false;
    /** Text that currently follows the last selection end-point. */
    currentTextAfter_ = '';
    /**
     * Cells that were entered while the IME wasn't active.  These will be
     * submitted once the IME becomes active and reports the current input
     * field. This is necessary because the IME is activated on the first
     * braille dots command, but we'll receive the command in parallel.  To work
     * around the race, we store the cell entered until we can submit it to the
     * IME.
     */
    pendingCells_ = [];
    entryState_ = null;
    uncommittedCellsSpan_ = null;
    uncommittedCellsChangedListener_ = null;
    /**
     * The input context of the current input field, as reported by the IME.
     * null if no input field has focus.
     */
    inputContext = null;
    /** Text that currently precedes the first selection end-point. */
    currentTextBefore = '';
    static instance;
    /** The ID of the Braille IME extension built into Chrome OS. */
    static IME_EXTENSION_ID_ = 'jddehjeebkoimngcbdkaahpobgicbffp';
    /** Name of the port to use for communicating with the Braille IME. */
    static IME_PORT_NAME_ = 'BrailleIme.Port';
    constructor() {
        BrailleTranslatorManager.instance.addChangeListener(() => this.commitAndClearEntryState());
        chrome.runtime.onConnectExternal.addListener((port) => this.onImeConnect_(port));
    }
    static init() {
        if (BrailleInputHandler.instance) {
            throw new Error('Cannot create two BrailleInputHandler instances');
        }
        BrailleInputHandler.instance = new BrailleInputHandler();
    }
    /**
     * Called when the content on the braille display is updated.  Modifies the
     * input state according to the new content.
     * @param text Text, optionally with value and selection spans.
     * @param listener Called when the uncommitted cells have changed.
     */
    onDisplayContentChanged(text, listener) {
        const valueSpan = text.getSpanInstanceOf(ValueSpan);
        const selectionSpan = text.getSpanInstanceOf(ValueSelectionSpan);
        if (!(valueSpan && selectionSpan)) {
            return;
        }
        // Don't call the old listener any further, since new content is being
        // set.  If the old listener is not cleared here, it could be called
        // spuriously if the entry state is cleared below.
        this.uncommittedCellsChangedListener_ = null;
        const valueStart = text.getSpanStart(valueSpan);
        const valueEnd = text.getSpanEnd(valueSpan);
        const selectionStart = text.getSpanStart(selectionSpan);
        const selectionEnd = text.getSpanEnd(selectionSpan);
        if (selectionStart < valueStart || selectionEnd > valueEnd) {
            console.error('Selection outside of value in braille content');
            this.clearEntryState();
            return;
        }
        const newTextBefore = text.toString().substring(valueStart, selectionStart);
        if (this.currentTextBefore !== newTextBefore && this.entryState_) {
            this.entryState_.onTextBeforeChanged(newTextBefore);
        }
        this.currentTextBefore = newTextBefore;
        this.currentTextAfter_ = text.toString().substring(selectionEnd, valueEnd);
        this.uncommittedCellsSpan_ = new ExtraCellsSpan();
        text.setSpan(this.uncommittedCellsSpan_, selectionStart, selectionStart);
        if (this.entryState_ && this.entryState_.usesUncommittedCells) {
            this.updateUncommittedCells(new Uint8Array(this.entryState_.cells).buffer);
        }
        this.uncommittedCellsChangedListener_ = listener;
    }
    /**
     * Handles braille key events used for input by editing the current input
     * field appropriately.
     * @return true if the event was handled, false if it should propagate
     *     further.
     */
    onBrailleKeyEvent(event) {
        if (event.command === BrailleKeyCommand.DOTS) {
            return this.onBrailleDots_(event.brailleDots);
        }
        // Any other braille command cancels the pending cells.
        this.pendingCells_.length = 0;
        if (event.command === BrailleKeyCommand.STANDARD_KEY) {
            if (event.standardKeyCode === 'Backspace' && !event.altKey &&
                !event.ctrlKey && !event.shiftKey && this.onBackspace_()) {
                return true;
            }
            else {
                this.commitAndClearEntryState();
                this.sendKeyEventPair_(event);
                return true;
            }
        }
        return false;
    }
    /**
     * Returns how the value of the currently displayed content should be
     * expanded given the current input state.
     */
    getExpansionType() {
        if (this.inAlwaysUncontractedContext_()) {
            return ExpandingBrailleTranslator.ExpansionType.ALL;
        }
        if (this.entryState_ &&
            this.entryState_.translator ===
                BrailleTranslatorManager.instance.getDefaultTranslator()) {
            return ExpandingBrailleTranslator.ExpansionType.NONE;
        }
        return ExpandingBrailleTranslator.ExpansionType.SELECTION;
    }
    /**
     * @return true if we have an input context and uncontracted braille should
     * always be used for that context.
     */
    inAlwaysUncontractedContext_() {
        const inputType = this.inputContext ? this.inputContext.type : '';
        return inputType === 'url' || inputType === 'email';
    }
    /**
     * Called when a user typed a braille cell.
     * @param dots The dot pattern of the cell.
     * @return Whether the event was handled or should be allowed to
     *    propagate further.
     */
    onBrailleDots_(dots) {
        if (!this.imeActive_) {
            this.pendingCells_.push(dots);
            return true;
        }
        if (!this.inputContext) {
            return false;
        }
        if (!this.entryState_) {
            if (!(this.entryState_ = this.createEntryState_())) {
                return false;
            }
        }
        this.entryState_.appendCell(dots);
        return true;
    }
    /**
     * Handles the backspace key by deleting the last typed cell if possible.
     * @return true if the event was handled, false if it wasn't and should
     * propagate further.
     */
    onBackspace_() {
        if (this.imeActive_ && this.entryState_) {
            this.entryState_.deleteLastCell();
            return true;
        }
        return false;
    }
    /**
     * Creates a new empty EntryState based on the current input
     * context and surrounding text.
     * @return The newly created state object, or null if it couldn't be created
     *     (e.g. if there's no braille translator available yet).
     */
    createEntryState_() {
        let translator = BrailleTranslatorManager.instance.getDefaultTranslator();
        if (!translator) {
            return null;
        }
        const uncontractedTranslator = BrailleTranslatorManager.instance.getUncontractedTranslator();
        let constructor = EditsEntryState;
        if (uncontractedTranslator) {
            const textBefore = this.currentTextBefore;
            const textAfter = this.currentTextAfter_;
            if (this.inAlwaysUncontractedContext_() ||
                (ENDS_WITH_NON_WHITESPACE_RE.test(textBefore)) ||
                (STARTS_WITH_NON_WHITESPACE_RE.test(textAfter))) {
                translator = uncontractedTranslator;
            }
            else {
                constructor = LateCommitEntryState;
            }
        }
        return new constructor(this, translator);
    }
    /** Commits the current entry state and clears it, if any. */
    commitAndClearEntryState() {
        if (this.entryState_) {
            this.entryState_.commit();
            this.clearEntryState();
        }
    }
    /** Clears the current entry state without committing it. */
    clearEntryState() {
        if (this.entryState_) {
            if (this.entryState_.usesUncommittedCells) {
                this.updateUncommittedCells(new ArrayBuffer(0));
            }
            this.entryState_.inputHandler = null;
            this.entryState_ = null;
        }
    }
    updateUncommittedCells(cells) {
        if (this.uncommittedCellsSpan_) {
            this.uncommittedCellsSpan_.cells = cells;
        }
        if (this.uncommittedCellsChangedListener_) {
            this.uncommittedCellsChangedListener_();
        }
    }
    /**
     * Called when another extension connects to this extension.  Accepts
     * connections from the ChromeOS builtin Braille IME and ignores connections
     * from other extensions.
     * @param port The port used to communicate with the other extension.
     */
    onImeConnect_(port) {
        if (port.name !== BrailleInputHandler.IME_PORT_NAME_ ||
            port.sender.id !== BrailleInputHandler.IME_EXTENSION_ID_) {
            return;
        }
        if (this.imePort_) {
            this.imePort_.disconnect();
        }
        port.onDisconnect.addListener(() => this.onImeDisconnect_(port));
        port.onMessage.addListener((message) => this.onImeMessage_(message));
        this.imePort_ = port;
    }
    /** Called when a message is received from the IME. */
    onImeMessage_(message) {
        if (typeof message !== 'object') {
            console.error('Unexpected message from Braille IME: ', JSON.stringify(message));
        }
        switch (message.type) {
            case 'activeState':
                this.imeActive_ = message.active;
                break;
            case 'inputContext':
                this.inputContext = message.context;
                this.clearEntryState();
                if (this.imeActive_ && this.inputContext) {
                    this.pendingCells_.forEach(this.onBrailleDots_, this);
                }
                this.pendingCells_.length = 0;
                break;
            case 'brailleDots':
                this.onBrailleDots_(message['dots']);
                break;
            case 'backspace':
                // Note that we can't send the backspace key through the
                // virtualKeyboardPrivate API in this case because it would then be
                // processed by the IME again, leading to an infinite loop.
                this.postImeMessage({
                    type: 'keyEventHandled',
                    requestId: message['requestId'],
                    result: this.onBackspace_(),
                });
                break;
            case 'reset':
                this.clearEntryState();
                break;
            default:
                console.error('Unexpected message from Braille IME: ', JSON.stringify(message));
                break;
        }
    }
    /**
     * Called when the IME port is disconnected.
     * @param port The port that was disconnected.
     */
    onImeDisconnect_(_port) {
        this.imePort_ = null;
        this.clearEntryState();
        this.imeActive_ = false;
        this.inputContext = null;
    }
    /**
     * Posts a message to the IME.
     * @param message The message.
     * @return true if the message was sent, false if there was no connection
     * open to the IME.
     */
    postImeMessage(message) {
        if (this.imePort_) {
            this.imePort_.postMessage(message);
            return true;
        }
        return false;
    }
    /**
     * Sends a keydown key event followed by a keyup event corresponding to an
     * event generated by the braille display.
     * @param event The braille key event to base the key events on.
     */
    sendKeyEventPair_(event) {
        // TODO(crbug.com/314203187): Not null asserted, check that this is correct.
        const keyName = event.standardKeyCode;
        const numericCode = BrailleKeyEvent.keyCodeToLegacyCode(keyName);
        if (!numericCode) {
            throw Error('Unknown key code in event: ' + JSON.stringify(event));
        }
        EventGenerator.sendKeyPress(numericCode, {
            shift: Boolean(event.shiftKey),
            ctrl: Boolean(event.ctrlKey),
            alt: Boolean(event.altKey),
        });
    }
}
TestImportManager.exportForTesting(BrailleInputHandler);

// Copyright 2017 The Chromium Authors
// Use of this source code is governed by a BSD-style license that can be
// found in the LICENSE file.
/**
 * @fileoverview Rewrites a braille key event.
 */
/**
 * A class that transforms a sequence of braille key events into a standard key
 * event.
 */
class BrailleKeyEventRewriter {
    static instance;
    incrementalKey_ = null;
    static init() {
        if (BrailleKeyEventRewriter.instance) {
            throw new Error('Cannot create two BrailleKeyEventRewriter instances');
        }
        BrailleKeyEventRewriter.instance = new BrailleKeyEventRewriter();
    }
    /**
     * Accumulates and optionally modifies in-coming braille key events.
     * @return False to continue event propagation.
     */
    onBrailleKeyEvent(evt) {
        let standardKeyCode;
        const dots = evt.brailleDots;
        if (!dots) {
            this.incrementalKey_ = null;
            return false;
        }
        if (evt.command === BrailleKeyCommand.CHORD) {
            Output.forceModeForNextSpeechUtterance(QueueMode.CATEGORY_FLUSH);
            const modifiers = BrailleKeyEvent.brailleDotsToModifiers[dots];
            // Check for a modifier mapping.
            if (modifiers) {
                this.incrementalKey_ = this.incrementalKey_ || {};
                for (const key in modifiers) {
                    this.incrementalKey_[key] = true;
                }
                return true;
            }
            // Check for a chord to standard key mapping.
            standardKeyCode = BrailleKeyEvent.brailleChordsToStandardKeyCode[dots];
        }
        // Check for a 'dots' command, which is typed on the keyboard with a
        // previous incremental key press.
        if (evt.command === BrailleKeyCommand.DOTS && this.incrementalKey_) {
            // Check if this braille pattern has a standard key mapping.
            standardKeyCode = BrailleKeyEvent.brailleDotsToStandardKeyCode[dots];
        }
        if (standardKeyCode) {
            evt.command = BrailleKeyCommand.STANDARD_KEY;
            evt.standardKeyCode = standardKeyCode;
            if (this.incrementalKey_) {
                // Apply all modifiers seen so far to the outgoing event as a standard
                // keyboard command.
                evt.altKey = this.incrementalKey_.altKey;
                evt.ctrlKey = this.incrementalKey_.ctrlKey;
                evt.shiftKey = this.incrementalKey_.shiftKey;
                this.incrementalKey_ = null;
            }
            return false;
        }
        this.incrementalKey_ = null;
        return false;
    }
}

// Copyright 2014 The Chromium Authors
// Use of this source code is governed by a BSD-style license that can be
// found in the LICENSE file.
/**
 * @fileoverview Sends Braille commands to the Braille API.
 */
class BrailleBackground {
    frozen_ = false;
    static instance;
    constructor() {
        BrailleDisplayManager.instance.setCommandListener((evt, content) => this.routeBrailleKeyEvent_(evt, content));
    }
    static init() {
        if (BrailleBackground.instance) {
            throw new Error('Cannot create two BrailleBackground instances');
        }
        // Must be called first.
        BrailleTranslatorManager.init();
        // Must be called before creating BrailleBackground.
        BrailleDisplayManager.init();
        BrailleInputHandler.init();
        BrailleKeyEventRewriter.init();
        BrailleBackground.instance = new BrailleBackground();
    }
    /** BrailleInterface implementation. */
    write(params) {
        if (this.frozen_) {
            return;
        }
        LogStore.instance.writeBrailleLog(params.text.toString());
        this.setContent_(params, null);
    }
    /** BrailleInterface implementation. */
    writeRawImage(imageDataUrl) {
        if (this.frozen_) {
            return;
        }
        BrailleDisplayManager.instance.setImageContent(imageDataUrl);
    }
    /** BrailleInterface implementation. */
    freeze() {
        this.frozen_ = true;
    }
    /** BrailleInterface implementation. */
    thaw() {
        this.frozen_ = false;
    }
    /** BrailleInterface implementation. */
    getDisplayState() {
        return BrailleDisplayManager.instance.getDisplayState();
    }
    /** BrailleInterface implementation. */
    panLeft() {
        BrailleDisplayManager.instance.panLeft();
    }
    /** BrailleInterface implementation. */
    panRight() {
        BrailleDisplayManager.instance.panRight();
    }
    /** BrailleInterface implementation. */
    route(displayPosition) {
        return BrailleDisplayManager.instance.route(displayPosition);
    }
    /** BrailleInterface implementation. */
    async backTranslate(cells) {
        return await BrailleTranslatorManager.backTranslate(cells);
    }
    setContent_(newContent, _newContentId) {
        const updateContent = () => BrailleDisplayManager.instance.setContent(newContent, BrailleInputHandler.instance.getExpansionType());
        BrailleInputHandler.instance.onDisplayContentChanged(newContent.text, updateContent);
        updateContent();
    }
    /**
     * Handles braille key events by dispatching either to the event rewriter,
     * input handler, or ChromeVox's background object.
     * @param content Content of display when event fired.
     */
    routeBrailleKeyEvent_(brailleEvt, content) {
        if (BrailleKeyEventRewriter.instance.onBrailleKeyEvent(brailleEvt)) {
            return;
        }
        if (BrailleInputHandler.instance.onBrailleKeyEvent(brailleEvt)) {
            return;
        }
        if (ChromeVoxState.instance) {
            ChromeVoxState.instance.onBrailleKeyEvent(brailleEvt, content);
        }
    }
}
TestImportManager.exportForTesting(BrailleBackground);

// Copyright 2019 The Chromium Authors
// Use of this source code is governed by a BSD-style license that can be
// found in the LICENSE file.
/**
 * @fileoverview Listens for download events and provides corresponding
 * notifications in ChromeVox.
 */
var DownloadState = chrome.downloads.State;
/** Handles all download events and notifications for ChromeVox. */
class DownloadHandler {
    /**
     * Maps download item ID to an object containing its file name and progress
     * update function.
     */
    downloadItemData_ = {};
    static instance;
    /**
     * Performs initialization. Populates downloadItemData_ object and registers
     * event listener for chrome.downloads.onChanged events.
     */
    static init() {
        DownloadHandler.instance = new DownloadHandler();
        // Populate downloadItemData_.
        // Retrieve 20 most recent downloads sorted by most recent start time.
        chrome.downloads.search({ orderBy: ['-startTime'], limit: FILE_LIMIT }, (results) => DownloadHandler.instance.populateDownloadItemData_(results));
        // Note: No event listener for chrome.downloads.onCreated because
        // onCreated does not actually correspond to when the download starts;
        // it corresponds to when the user clicks the download button, which
        // sometimes leads to a screen where the user can decide where to save the
        // download.
        // Fired when any of a DownloadItem's properties, except bytesReceived and
        // estimatedEndTime, change. Only contains properties that changed.
        chrome.downloads.onChanged.addListener((item) => DownloadHandler.instance.onChanged_(item));
    }
    /**
     * Notifies user of download progress for file.
     * @param id The ID of the file we are providing an update for.
     */
    notifyProgress_(id) {
        chrome.downloads.search({ id }, (results) => this.notifyProgressResults_(results));
    }
    notifyProgressResults_(results) {
        if (results?.length !== 1) {
            return;
        }
        // Results should have only one item because IDs are unique.
        const updatedItem = results[0];
        const storedItem = this.downloadItemData_[updatedItem.id];
        const percentComplete = Math.round((updatedItem.bytesReceived / updatedItem.totalBytes) * 100);
        const percentDelta = percentComplete - storedItem.percentComplete;
        // Convert time delta from milliseconds to seconds.
        const timeDelta = Math.round((Date.now() - storedItem.time) / 1000);
        // Calculate notification score for this download.
        // This equation was determined by targeting 30 seconds and 50% complete
        // as reasonable milestones before giving an update.
        const score = percentDelta + (5 / 3) * timeDelta;
        // Only report downloads that have scores above the threshold value.
        if (score > UPDATE_THRESHOLD) {
            // Update state.
            storedItem.time = Date.now();
            storedItem.percentComplete = percentComplete;
            // Determine time remaining and units.
            if (!updatedItem.estimatedEndTime) {
                return;
            }
            const endTime = new Date(updatedItem.estimatedEndTime);
            let timeRemaining = Math.round((endTime.getTime() - Date.now()) / 1000);
            let timeUnit = '';
            if (!timeRemaining || (timeRemaining < 0)) {
                return;
            }
            else if (timeRemaining < 60) {
                // Seconds. Use up until 1 minute remaining.
                timeUnit = Msgs.getMsgWithCount('seconds', timeRemaining);
            }
            else if (timeRemaining < 3600) {
                // Minutes. Use up until 1 hour remaining.
                timeRemaining = Math.floor(timeRemaining / 60);
                timeUnit = Msgs.getMsgWithCount('minutes', timeRemaining);
            }
            else if (timeRemaining < 36000) {
                // Hours. Use up until 10 hours remaining.
                timeRemaining = Math.floor(timeRemaining / 3600);
                timeUnit = Msgs.getMsgWithCount('hours', timeRemaining);
            }
            else {
                // If 10+ hours remaining, do not report progress.
                return;
            }
            const optSubs = [
                storedItem.percentComplete.toString(),
                storedItem.fileName,
                timeRemaining.toString(),
                timeUnit.toString(),
            ];
            this.speechAndBrailleOutput_('download_progress', QueueMode.FLUSH, optSubs);
        }
    }
    onChanged_(delta) {
        // The type of notification ChromeVox reports can be inferred based on the
        // available properties, as they have been observed to be mutually
        // exclusive.
        const name = delta.filename;
        const state = delta.state;
        const paused = delta.paused;
        // The ID is always set no matter what.
        const id = delta.id;
        const storedItem = this.downloadItemData_[id];
        // New download if we're not tracking the item and if the filename was
        // previously empty.
        if (!storedItem && name?.previous === '') {
            this.startTrackingDownloadDelta_(delta);
            // Speech and braille output.
            const optSub = this.downloadItemData_[id].fileName;
            this.speechAndBrailleOutput_('download_started', QueueMode.FLUSH, [optSub]);
        }
        else if (state) {
            const currentState = state.current;
            let msgId = '';
            // Only give notification for COMPLETE and INTERRUPTED.
            // IN_PROGRESS notifications are given by notifyProgress function.
            if (currentState === DownloadState.COMPLETE) {
                msgId = 'download_completed';
            }
            else if (currentState === DownloadState.INTERRUPTED) {
                msgId = 'download_stopped';
            }
            else {
                return;
            }
            const optSubs = [storedItem.fileName];
            clearInterval(storedItem.notifyProgressId);
            delete this.downloadItemData_[id];
            // Speech and braille output.
            this.speechAndBrailleOutput_(msgId, QueueMode.FLUSH, optSubs);
        }
        else if (paused) {
            // Will be either resumed or paused.
            let msgId = 'download_resumed';
            const optSubs = [storedItem.fileName];
            if (paused.current === true) {
                // Download paused.
                msgId = 'download_paused';
                clearInterval(storedItem.notifyProgressId);
            }
            else {
                // Download resumed.
                storedItem.notifyProgressId = setInterval(() => this.notifyProgress_(id), INTERVAL_TIME_MILLISECONDS);
                storedItem.time = Date.now();
            }
            // Speech and braille output.
            this.speechAndBrailleOutput_(msgId, QueueMode.FLUSH, optSubs);
        }
    }
    populateDownloadItemData_(results) {
        if (!results || results.length === 0) {
            return;
        }
        for (const item of results) {
            // If download is in progress, start tracking it.
            if (item.state === DownloadState.IN_PROGRESS) {
                this.startTrackingDownloadItem_(item);
            }
        }
    }
    /**
     * Output download notification as speech and braille.
     * @param msgId The msgId for Output.
     * @param queueMode The queue mode.
     * @param optSubs Substitution strings.
     */
    speechAndBrailleOutput_(msgId, queueMode, optSubs) {
        if (SettingsManager.get('announceDownloadNotifications')) {
            const msg = Msgs.getMsg(msgId, optSubs);
            new Output().withString(msg).withQueueMode(queueMode).go();
        }
    }
    /**
     * Store item data.
     * @param item The download item to track.
     */
    startTrackingDownloadItem_(item) {
        const id = item.id;
        // Don't add if we are already tracking file.
        if (this.downloadItemData_[id]) {
            return;
        }
        const fullPath = item.filename;
        const fileName = fullPath.substring(fullPath.lastIndexOf('/') + 1);
        const notifyProgressId = setInterval(() => this.notifyProgress_(id), INTERVAL_TIME_MILLISECONDS);
        let percentComplete = 0;
        if (item.bytesReceived && item.totalBytes) {
            percentComplete =
                Math.round((item.bytesReceived / item.totalBytes) * 100);
        }
        this.downloadItemData_[id] =
            { fileName, notifyProgressId, time: Date.now(), percentComplete };
    }
    /**
     * Store item data.
     * @param item The download item to track.
     */
    startTrackingDownloadDelta_(item) {
        const id = item.id;
        // Don't add if we are already tracking file.
        if (this.downloadItemData_[id]) {
            return;
        }
        const fullPath = item.filename?.current;
        const fileName = fullPath ? fullPath.substring(fullPath.lastIndexOf('/') + 1) : '';
        const notifyProgressId = setInterval(() => this.notifyProgress_(id), INTERVAL_TIME_MILLISECONDS);
        this.downloadItemData_[id] =
            { fileName, notifyProgressId, time: Date.now(), percentComplete: 0 };
    }
}
// Local to module.
/**
 * Threshold value used when determining whether to report an update to user.
 */
const UPDATE_THRESHOLD = 100;
/** The limit for the number of download results to receive when querying. */
const FILE_LIMIT = 20;
/** The time interval, in milliseconds, for calling notifyProgress. */
const INTERVAL_TIME_MILLISECONDS = 10000;
TestImportManager.exportForTesting(DownloadHandler);

// Copyright 2014 The Chromium Authors
// Use of this source code is governed by a BSD-style license that can be
// found in the LICENSE file.
/**
 * Base class for implementing earcons.
 * When adding earcons, please add them to chromevox/mv2/common/earcon_id.js.
 */
class AbstractEarcons {
    /**
     * Whether or not earcons are available.
     * @return True if earcons are available.
     */
    earconsAvailable() {
        return true;
    }
    /**
     * Whether or not earcons are enabled.
     * @return True if earcons are enabled.
     */
    get enabled() {
        return LocalStorage.getBoolean('earcons');
    }
    /**
     * Set whether or not earcons are enabled.
     * @param value True turns on earcons, false turns off earcons.
     */
    set enabled(value) {
        LocalStorage.set('earcons', value);
    }
}
TestImportManager.exportForTesting(AbstractEarcons);

// Copyright 2015 The Chromium Authors
// Use of this source code is governed by a BSD-style license that can be
// found in the LICENSE file.
/**
 * @fileoverview This is the low-level class that generates ChromeVox's
 * earcons. It's designed to be self-contained and not depend on the
 * rest of the code.
 */
/** EarconEngine generates ChromeVox's earcons using the web audio API. */
class EarconEngine {
    // Public control parameters. All of these are meant to be adjustable.
    /** The output volume, as an amplification factor. */
    outputVolume = 1.0;
    /**
     * As notated below, all pitches are in the key of C. This can be set to
     * transpose the key from C to another pitch.
     */
    transposeToKey = Note.B_FLAT3;
    /** The click volume, as an amplification factor. */
    clickVolume = 0.4;
    /**
     * The volume of the static sound, as an amplification factor.
     */
    staticVolume = 0.2;
    /** The base delay for repeated sounds, in seconds. */
    baseDelay = 0.045;
    /** The base stereo panning, from -1 to 1. */
    basePan = CENTER_PAN;
    /** The base reverb level as an amplification factor. */
    baseReverb = 0.4;
    /** The choice of the reverb impulse response to use. */
    reverbSound = Reverb.SMALL_ROOM;
    /** The base pitch for the 'wrap' sound. */
    wrapPitch = Note.G_FLAT3;
    /** The base pitch for the 'alert' sound. */
    alertPitch = Note.G_FLAT3;
    /** The default pitch. */
    defaultPitch = Note.G3;
    /** The choice of base sound for most controls. */
    controlSound = WavSoundFile.CONTROL;
    /**
     * The delay between sounds in the on/off sweep effect,
     * in seconds.
     */
    sweepDelay = 0.045;
    /**
     * The delay between echos in the on/off sweep, in seconds.
     */
    sweepEchoDelay = 0.15;
    /** The number of echos in the on/off sweep. */
    sweepEchoCount = 3;
    /** The pitch offset of the on/off sweep. */
    sweepPitch = Note.C3;
    /**
     * The final gain of the progress sound, as an
     * amplification factor.
     */
    progressFinalGain = 0.05;
    /** The multiplicative decay rate of the progress ticks. */
    // eslint-disable-next-line @typescript-eslint/naming-convention
    progressGain_Decay = 0.7;
    // Private variables.
    /** The audio context. */
    context_ = new AudioContext();
    /** The reverb node, lazily initialized. */
    reverbConvolver_ = null;
    /**
     * @private A map between the name of an audio data file and its loaded AudioBuffer.
     */
    buffers_ = {};
    loops_ = {};
    /**
     * The source audio nodes for queued tick / tocks for progress.
     * Kept around so they can be canceled.
     */
    progressSources_ = [];
    /** The current gain for progress sounds. */
    progressGain_ = 1.0;
    /** The current time for progress sounds. */
    progressTime_ = this.context_.currentTime;
    /** The setInterval ID for progress sounds. */
    progressIntervalID_ = null;
    /**
     * Maps a earcon name to the last source input audio for that
     * earcon.
     */
    lastEarconSources_ = {};
    currentTrackedEarcon_;
    constructor() {
        // Initialization: load the base sound data files asynchronously.
        Object.values(WavSoundFile)
            .concat(Object.values(Reverb))
            .forEach(sound => this.loadSound(sound, `${BASE_URL}${sound}.wav`));
        Object.values(OggSoundFile)
            .forEach(sound => this.loadSound(sound, `${BASE_URL}${sound}.ogg`));
    }
    /**
     * A high-level way to ask the engine to play a specific earcon.
     * @param earcon The earcon to play.
     */
    playEarcon(earcon) {
        // These earcons are not tracked by the engine via their audio sources.
        switch (earcon) {
            case EarconId.CHROMEVOX_LOADED:
                this.onChromeVoxLoaded();
                return;
            case EarconId.CHROMEVOX_LOADING:
                this.onChromeVoxLoading();
                return;
            case EarconId.PAGE_FINISH_LOADING:
                this.cancelProgress();
                return;
            case EarconId.PAGE_START_LOADING:
                this.startProgress();
                return;
            case EarconId.POP_UP_BUTTON:
                this.onPopUpButton();
                return;
            // These had earcons in previous versions of ChromeVox but
            // they're currently unused / unassigned.
            case EarconId.LIST_ITEM:
            case EarconId.LONG_DESC:
            case EarconId.MATH:
            case EarconId.OBJECT_CLOSE:
            case EarconId.OBJECT_ENTER:
            case EarconId.OBJECT_EXIT:
            case EarconId.OBJECT_OPEN:
            case EarconId.OBJECT_SELECT:
            case EarconId.RECOVER_FOCUS:
                return;
        }
        // These earcons are tracked by the engine via their audio sources.
        if (this.lastEarconSources_[earcon] !== undefined) {
            // Playback of |earcon| is in progress.
            return;
        }
        this.currentTrackedEarcon_ = earcon;
        switch (earcon) {
            case EarconId.ALERT_MODAL:
            case EarconId.ALERT_NONMODAL:
                this.onAlert();
                break;
            case EarconId.BUTTON:
                this.onButton();
                break;
            case EarconId.CHECK_OFF:
                this.onCheckOff();
                break;
            case EarconId.CHECK_ON:
                this.onCheckOn();
                break;
            case EarconId.EDITABLE_TEXT:
                this.onTextField();
                break;
            case EarconId.INVALID_KEYPRESS:
                this.onInvalidKeypress();
                break;
            case EarconId.LINK:
                this.onLink();
                break;
            case EarconId.LISTBOX:
                this.onSelect();
                break;
            case EarconId.SELECTION:
                this.onSelection();
                break;
            case EarconId.SELECTION_REVERSE:
                this.onSelectionReverse();
                break;
            case EarconId.SKIP:
                this.onSkim();
                break;
            case EarconId.SLIDER:
                this.onSlider();
                break;
            case EarconId.SMART_STICKY_MODE_OFF:
                this.onSmartStickyModeOff();
                break;
            case EarconId.SMART_STICKY_MODE_ON:
                this.onSmartStickyModeOn();
                break;
            case EarconId.NO_POINTER_ANCHOR:
                this.onNoPointerAnchor();
                break;
            case EarconId.WRAP:
            case EarconId.WRAP_EDGE:
                this.onWrap();
                break;
        }
        this.currentTrackedEarcon_ = undefined;
        // Clear source once it finishes playing.
        const source = this.lastEarconSources_[earcon];
        if (source !== undefined &&
            source instanceof AudioScheduledSourceNode) {
            source.onended = () => {
                delete this.lastEarconSources_[earcon];
            };
        }
    }
    /**
     * Fetches a sound asynchronously and loads its data into an AudioBuffer.
     *
     * @param name The name of the sound to load.
     * @param url The url where the sound should be fetched from.
     */
    async loadSound(name, url) {
        const response = await fetch(url);
        if (response.ok) {
            const arrayBuffer = await response.arrayBuffer();
            const decodedAudio = await this.context_.decodeAudioData(arrayBuffer);
            this.buffers_[name] = decodedAudio;
        }
    }
    /**
     * Return an AudioNode containing the final processing that all
     * sounds go through: output volume / gain, panning, and reverb.
     * The chain is hooked up to the destination automatically, so you
     * just need to connect your source to the return value from this
     * method.
     *
     * @param properties
     *     An object where you can override the default
     *     gain, pan, and reverb, otherwise these are taken from
     *     outputVolume, basePan, and baseReverb.
     * @return The filters to be applied to all sounds, connected
     *     to the destination node.
     */
    createCommonFilters(properties) {
        let gain = this.outputVolume;
        if (properties.gain) {
            gain *= properties.gain;
        }
        const gainNode = this.context_.createGain();
        gainNode.gain.value = gain;
        const first = gainNode;
        let last = gainNode;
        const pan = properties.pan ?? this.basePan;
        if (pan !== 0) {
            const panNode = this.context_.createPanner();
            panNode.setPosition(pan, 0, 0);
            panNode.setOrientation(0, 0, 1);
            last.connect(panNode);
            last = panNode;
        }
        const reverb = properties.reverb ?? this.baseReverb;
        if (reverb) {
            if (!this.reverbConvolver_) {
                this.reverbConvolver_ = this.context_.createConvolver();
                this.reverbConvolver_.buffer = this.buffers_[this.reverbSound];
                this.reverbConvolver_.connect(this.context_.destination);
            }
            // Dry
            last.connect(this.context_.destination);
            // Wet
            const reverbGainNode = this.context_.createGain();
            reverbGainNode.gain.value = reverb;
            last.connect(reverbGainNode);
            reverbGainNode.connect(this.reverbConvolver_);
        }
        else {
            last.connect(this.context_.destination);
        }
        return first;
    }
    /**
     * High-level interface to play a sound from a buffer source by name,
     * with some simple adjustments like pitch change (in half-steps),
     * a start time (relative to the current time, in seconds),
     * gain, panning, and reverb.
     *
     * The only required parameter is the name of the sound. The time, pitch,
     * gain, panning, and reverb are all optional and are passed in an
     * object of optional properties.
     *
     * @param sound The name of the sound to play. It must already
     *     be loaded in a buffer.
     * @param properties
     *     An object where you can override the default pitch, gain, pan,
     *     and reverb.
     * @return The source node, so you can stop it
     *     or set event handlers on it.
     */
    play(sound, properties = {}) {
        const source = this.context_.createBufferSource();
        source.buffer = this.buffers_[sound];
        if (properties.loop) {
            this.loops_[sound] = source;
        }
        const pitch = properties.pitch ?? this.defaultPitch;
        // Changes the playback rate of the sample – which also changes the pitch.
        source.playbackRate.value = this.multiplierFor_(pitch);
        source.loop = properties.loop ?? false;
        const destination = this.createCommonFilters(properties);
        source.connect(destination);
        if (this.currentTrackedEarcon_) {
            this.lastEarconSources_[this.currentTrackedEarcon_] = source;
        }
        if (properties.time) {
            source.start(this.context_.currentTime + properties.time);
        }
        else {
            source.start(this.context_.currentTime);
        }
        return source;
    }
    /**
     * Stops the loop of the specified sound file, if one exists.
     * @param sound The name of the sound file.
     */
    stopLoop(sound) {
        if (!this.loops_[sound]) {
            return;
        }
        this.loops_[sound].stop();
        delete this.loops_[sound];
    }
    /** Play the static sound. */
    onStatic() {
        this.play(WavSoundFile.STATIC, { gain: this.staticVolume });
    }
    /** Play the link sound. */
    onLink() {
        this.play(WavSoundFile.STATIC, { gain: this.clickVolume });
        this.play(this.controlSound, { pitch: Note.G4 });
    }
    /** Play the button sound. */
    onButton() {
        this.play(WavSoundFile.STATIC, { gain: this.clickVolume });
        this.play(this.controlSound);
    }
    /** Play the text field sound. */
    onTextField() {
        this.play(WavSoundFile.STATIC, { gain: this.clickVolume });
        this.play(WavSoundFile.STATIC, { time: this.baseDelay * 1.5, gain: this.clickVolume * 0.5 });
        this.play(this.controlSound, { pitch: Note.B3 });
        this.play(this.controlSound, { pitch: Note.B3, time: this.baseDelay * 1.5, gain: 0.5 });
    }
    /** Play the pop up button sound. */
    onPopUpButton() {
        this.play(WavSoundFile.STATIC, { gain: this.clickVolume });
        this.play(this.controlSound);
        this.play(this.controlSound, { time: this.baseDelay * 3, gain: 0.2, pitch: Note.G4 });
        this.play(this.controlSound, { time: this.baseDelay * 4.5, gain: 0.2, pitch: Note.G4 });
    }
    /** Play the check on sound. */
    onCheckOn() {
        this.play(WavSoundFile.STATIC, { gain: this.clickVolume });
        this.play(this.controlSound, { pitch: Note.D3 });
        this.play(this.controlSound, { pitch: Note.D4, time: this.baseDelay * 2 });
    }
    /** Play the check off sound. */
    onCheckOff() {
        this.play(WavSoundFile.STATIC, { gain: this.clickVolume });
        this.play(this.controlSound, { pitch: Note.D4 });
        this.play(this.controlSound, { pitch: Note.D3, time: this.baseDelay * 2 });
    }
    /** Play the smart sticky mode on sound. */
    onSmartStickyModeOn() {
        this.play(WavSoundFile.STATIC, { gain: this.clickVolume * 0.5 });
        this.play(this.controlSound, { pitch: Note.D4 });
    }
    /** Play the smart sticky mode off sound. */
    onSmartStickyModeOff() {
        this.play(WavSoundFile.STATIC, { gain: this.clickVolume * 0.5 });
        this.play(this.controlSound, { pitch: Note.D3 });
    }
    /** Play the select control sound. */
    onSelect() {
        this.play(WavSoundFile.STATIC, { gain: this.clickVolume });
        this.play(this.controlSound);
        this.play(this.controlSound, { time: this.baseDelay });
        this.play(this.controlSound, { time: this.baseDelay * 2 });
    }
    /** Play the slider sound. */
    onSlider() {
        this.play(WavSoundFile.STATIC, { gain: this.clickVolume });
        this.play(this.controlSound);
        this.play(this.controlSound, { time: this.baseDelay, gain: 0.5, pitch: Note.A3 });
        this.play(this.controlSound, { time: this.baseDelay * 2, gain: 0.25, pitch: Note.B3 });
        this.play(this.controlSound, { time: this.baseDelay * 3, gain: 0.125, pitch: Note.D_FLAT4 });
        this.play(this.controlSound, { time: this.baseDelay * 4, gain: 0.0625, pitch: Note.E_FLAT4 });
    }
    /** Play the skim sound. */
    onSkim() {
        this.play(WavSoundFile.SKIM);
    }
    /** Play the selection sound. */
    onSelection() {
        this.play(OggSoundFile.SELECTION);
    }
    /** Play the selection reverse sound. */
    onSelectionReverse() {
        this.play(OggSoundFile.SELECTION_REVERSE);
    }
    /** Play the invalid keypress sound. */
    onInvalidKeypress() {
        this.play(OggSoundFile.INVALID_KEYPRESS);
    }
    onNoPointerAnchor() {
        this.play(WavSoundFile.STATIC, { gain: this.clickVolume * 0.2 });
        const freq1 = this.frequencyFor_(Note.A_FLAT4);
        this.generateSinusoidal({
            attack: 0.00001,
            decay: 0.01,
            dur: 0.1,
            gain: 0.008,
            freq: freq1,
            overtones: 1,
            overtoneFactor: 0.1,
        });
    }
    /**
     * Generate a synthesized musical note based on a sum of sinusoidals shaped
     * by an envelope, controlled by a number of properties.
     *
     * The sound has a frequency of |freq|, or if |endFreq| is specified, does
     * an exponential ramp from |freq| to |endFreq|.
     *
     * If |overtones| is greater than 1, the sound will be mixed with additional
     * sinusoidals at multiples of |freq|, each one scaled by |overtoneFactor|.
     * This creates a rounder tone than a pure sine wave.
     *
     * The envelope is shaped by the duration |dur|, the attack time |attack|,
     * and the decay time |decay|, in seconds.
     *
     * As with other functions, |pan| and |reverb| can be used to override
     * basePan and baseReverb.
     *
     * @param properties
     *     An object containing the properties that can be used to
     *     control the sound, as described above.
     */
    generateSinusoidal(properties) {
        const envelopeNode = this.context_.createGain();
        envelopeNode.connect(this.context_.destination);
        const time = properties.time ?? 0;
        // Generate an oscillator for the frequency corresponding to the specified
        // frequency, and then additional overtones at multiples of that frequency
        // scaled by the overtoneFactor. Cue the oscillator to start and stop
        // based on the start time and specified duration.
        //
        // If an end frequency is specified, do an exponential ramp to that end
        // frequency.
        let gain = properties.gain;
        // TODO(b/314203187): Determine if not null assertion is acceptable.
        for (let i = 0; i < properties.overtones; i++) {
            const osc = this.context_.createOscillator();
            if (this.currentTrackedEarcon_) {
                this.lastEarconSources_[this.currentTrackedEarcon_] = osc;
            }
            osc.frequency.value = properties.freq * (i + 1);
            if (properties.endFreq) {
                osc.frequency.setValueAtTime(properties.freq * (i + 1), this.context_.currentTime + time);
                osc.frequency.exponentialRampToValueAtTime(properties.endFreq * (i + 1), this.context_.currentTime + properties.dur);
            }
            osc.start(this.context_.currentTime + time);
            osc.stop(this.context_.currentTime + time + properties.dur);
            const gainNode = this.context_.createGain();
            gainNode.gain.value = gain;
            osc.connect(gainNode);
            gainNode.connect(envelopeNode);
            gain *= properties.overtoneFactor;
        }
        // Shape the overall sound by an envelope based on the attack and
        // decay times.
        // TODO(b/314203187): Determine if not null assertion is acceptable.
        envelopeNode.gain.setValueAtTime(0, this.context_.currentTime + time);
        envelopeNode.gain.linearRampToValueAtTime(1, this.context_.currentTime + time + properties.attack);
        envelopeNode.gain.setValueAtTime(1, this.context_.currentTime + time + properties.dur - properties.decay);
        envelopeNode.gain.linearRampToValueAtTime(0, this.context_.currentTime + time + properties.dur);
        // Route everything through the common filters like reverb at the end.
        const destination = this.createCommonFilters({});
        envelopeNode.connect(destination);
    }
    /** Play an alert sound. */
    onAlert() {
        const freq1 = this.frequencyFor_(this.alertPitch - 2);
        const freq2 = this.frequencyFor_(this.alertPitch - 3);
        this.generateSinusoidal({
            attack: 0.02,
            decay: 0.07,
            dur: 0.15,
            gain: 0.3,
            freq: freq1,
            overtones: 3,
            overtoneFactor: 0.1,
        });
        this.generateSinusoidal({
            attack: 0.02,
            decay: 0.07,
            dur: 0.15,
            gain: 0.3,
            freq: freq2,
            overtones: 3,
            overtoneFactor: 0.1,
        });
        this.currentTrackedEarcon_ = undefined;
    }
    /** Play a wrap sound. */
    onWrap() {
        this.play(WavSoundFile.STATIC, { gain: this.clickVolume * 0.3 });
        const freq1 = this.frequencyFor_(this.wrapPitch - 8);
        const freq2 = this.frequencyFor_(this.wrapPitch + 8);
        this.generateSinusoidal({
            attack: 0.01,
            decay: 0.1,
            dur: 0.15,
            gain: 0.3,
            freq: freq1,
            endFreq: freq2,
            overtones: 1,
            overtoneFactor: 0.1,
        });
    }
    /**
     * Queue up a few tick/tock sounds for a progress bar. This is called
     * repeatedly by setInterval to keep the sounds going continuously.
     */
    generateProgressTickTocks_() {
        // TODO(b/314203187): Determine if not null assertion is acceptable.
        this.progressTime_ = this.progressTime_;
        while (this.progressTime_ < this.context_.currentTime + 3.0) {
            let t = this.progressTime_ - this.context_.currentTime;
            this.progressSources_.push([
                this.progressTime_,
                this.play(WavSoundFile.STATIC, { gain: 0.5 * this.progressGain_, time: t }),
            ]);
            this.progressSources_.push([
                this.progressTime_,
                this.play(this.controlSound, { pitch: Note.E_FLAT5, time: t, gain: this.progressGain_ }),
            ]);
            if (this.progressGain_ > this.progressFinalGain) {
                this.progressGain_ *= this.progressGain_Decay;
            }
            t += 0.5;
            this.progressSources_.push([
                this.progressTime_,
                this.play(WavSoundFile.STATIC, { gain: 0.5 * this.progressGain_, time: t }),
            ]);
            this.progressSources_.push([
                this.progressTime_,
                this.play(this.controlSound, { pitch: Note.E_FLAT4, time: t, gain: this.progressGain_ }),
            ]);
            if (this.progressGain_ > this.progressFinalGain) {
                this.progressGain_ *= this.progressGain_Decay;
            }
            this.progressTime_ += 1.0;
        }
        let removeCount = 0;
        while (removeCount < this.progressSources_.length &&
            this.progressSources_[removeCount][0] <
                this.context_.currentTime - 0.2) {
            removeCount++;
        }
        this.progressSources_.splice(0, removeCount);
    }
    /**
     * Start playing tick / tock progress sounds continuously until
     * explicitly canceled.
     */
    startProgress() {
        if (this.progressIntervalID_) {
            this.cancelProgress();
        }
        this.progressSources_ = [];
        this.progressGain_ = 0.5;
        this.progressTime_ = this.context_.currentTime;
        this.generateProgressTickTocks_();
        this.progressIntervalID_ =
            setInterval(() => this.generateProgressTickTocks_(), 1000);
    }
    /** Stop playing any tick / tock progress sounds. */
    cancelProgress() {
        if (!this.progressIntervalID_) {
            return;
        }
        for (let i = 0; i < this.progressSources_.length; i++) {
            this.progressSources_[i][1].stop();
        }
        this.progressSources_ = [];
        clearInterval(this.progressIntervalID_);
        this.progressIntervalID_ = null;
    }
    /** Plays sound indicating ChromeVox is loading. */
    onChromeVoxLoading() {
        this.play(OggSoundFile.CHROMEVOX_LOADING, { loop: true });
    }
    /**
     * Plays the sound indicating ChromeVox has loaded, and cancels the ChromeVox
     * loading sound.
     */
    onChromeVoxLoaded() {
        this.stopLoop(OggSoundFile.CHROMEVOX_LOADING);
        this.play(OggSoundFile.CHROMEVOX_LOADED);
    }
    /**
     * @param {chrome.automation.Rect} rect
     * @param {chrome.automation.Rect} container
     */
    setPositionForRect(rect, container) {
        // The horizontal position computed as a percentage relative to its
        // container.
        let x = (rect.left + rect.width / 2) / container.width;
        // Clamp.
        x = Math.min(Math.max(x, 0.0), 1.0);
        // Map to between the negative maximum pan x position and the positive max x
        // pan position.
        x = (2 * x - 1) * MAX_PAN_ABS_X_POSITION;
        this.basePan = x;
    }
    /** Resets panning to default (centered). */
    resetPan() {
        this.basePan = CENTER_PAN;
    }
    multiplierFor_(note) {
        const halfStepsFromA220 = note + HALF_STEPS_TO_C + this.transposeToKey;
        return Math.pow(HALF_STEP, halfStepsFromA220);
    }
    frequencyFor_(note) {
        return A3_HZ * this.multiplierFor_(note);
    }
}
// Local to module.
/* The list of sound data files to load. */
const WavSoundFile = {
    CONTROL: 'control',
    SKIM: 'skim',
    STATIC: 'static',
};
/* The list of sound data files to load. */
const OggSoundFile = {
    CHROMEVOX_LOADED: 'chromevox_loaded',
    CHROMEVOX_LOADING: 'chromevox_loading',
    INVALID_KEYPRESS: 'invalid_keypress',
    SELECTION: 'selection',
    SELECTION_REVERSE: 'selection_reverse',
};
/** The list of reverb data files to load. */
const Reverb = {
    SMALL_ROOM: 'small_room_2',
};
/** Pitch values for different notes. */
var Note;
(function (Note) {
    Note[Note["C2"] = -24] = "C2";
    Note[Note["D_FLAT2"] = -23] = "D_FLAT2";
    Note[Note["D2"] = -22] = "D2";
    Note[Note["E_FLAT2"] = -21] = "E_FLAT2";
    Note[Note["E2"] = -20] = "E2";
    Note[Note["F2"] = -19] = "F2";
    Note[Note["G_FLAT2"] = -18] = "G_FLAT2";
    Note[Note["G2"] = -17] = "G2";
    Note[Note["A_FLAT2"] = -16] = "A_FLAT2";
    Note[Note["A2"] = -15] = "A2";
    Note[Note["B_FLAT2"] = -14] = "B_FLAT2";
    Note[Note["B2"] = -13] = "B2";
    Note[Note["C3"] = -12] = "C3";
    Note[Note["D_FLAT3"] = -11] = "D_FLAT3";
    Note[Note["D3"] = -10] = "D3";
    Note[Note["E_FLAT3"] = -9] = "E_FLAT3";
    Note[Note["E3"] = -8] = "E3";
    Note[Note["F3"] = -7] = "F3";
    Note[Note["G_FLAT3"] = -6] = "G_FLAT3";
    Note[Note["G3"] = -5] = "G3";
    Note[Note["A_FLAT3"] = -4] = "A_FLAT3";
    Note[Note["A3"] = -3] = "A3";
    Note[Note["B_FLAT3"] = -2] = "B_FLAT3";
    Note[Note["B3"] = -1] = "B3";
    Note[Note["C4"] = 0] = "C4";
    Note[Note["D_FLAT4"] = 1] = "D_FLAT4";
    Note[Note["D4"] = 2] = "D4";
    Note[Note["E_FLAT4"] = 3] = "E_FLAT4";
    Note[Note["E4"] = 4] = "E4";
    Note[Note["F4"] = 5] = "F4";
    Note[Note["G_FLAT4"] = 6] = "G_FLAT4";
    Note[Note["G4"] = 7] = "G4";
    Note[Note["A_FLAT4"] = 8] = "A_FLAT4";
    Note[Note["A4"] = 9] = "A4";
    Note[Note["B_FLAT4"] = 10] = "B_FLAT4";
    Note[Note["B4"] = 11] = "B4";
    Note[Note["C5"] = 12] = "C5";
    Note[Note["D_FLAT5"] = 13] = "D_FLAT5";
    Note[Note["D5"] = 14] = "D5";
    Note[Note["E_FLAT5"] = 15] = "E_FLAT5";
    Note[Note["E5"] = 16] = "E5";
    Note[Note["F5"] = 17] = "F5";
    Note[Note["G_FLAT5"] = 18] = "G_FLAT5";
    Note[Note["G5"] = 19] = "G5";
    Note[Note["A_FLAT5"] = 20] = "A_FLAT5";
    Note[Note["A5"] = 21] = "A5";
    Note[Note["B_FLAT5"] = 22] = "B_FLAT5";
    Note[Note["B5"] = 23] = "B5";
    Note[Note["C6"] = 24] = "C6";
})(Note || (Note = {}));
/** The number of half-steps in an octave. */
const HALF_STEPS_PER_OCTAVE = 12;
/**
 * The number of half-steps from the base pitch (A220Hz) to C4 (middle C).
 */
const HALF_STEPS_TO_C = 3;
/**The scale factor for one half-step. */
const HALF_STEP = Math.pow(2.0, 1.0 / HALF_STEPS_PER_OCTAVE);
/**The frequency of the note A3, in Hertz. */
const A3_HZ = 220;
/** The base url for earcon sound resources. */
const BASE_URL = chrome.extension.getURL('common/earcons/');
/**The maximum value to pass to PannerNode.setPosition. */
const MAX_PAN_ABS_X_POSITION = 4;
/**Default (centered) pan position. */
const CENTER_PAN = 0;

// Copyright 2015 The Chromium Authors
// Use of this source code is governed by a BSD-style license that can be
// found in the LICENSE file.
/**
 * @fileoverview Earcons library that uses EarconEngine to play back
 * auditory cues.
 */
const DeviceType$1 = chrome.audio.DeviceType;
/**
 * High-level class that manages when each earcon should start (and when
 * relevant, stop) playing.
 */
class Earcons extends AbstractEarcons {
    engine_ = new EarconEngine();
    shouldPan_ = true;
    constructor() {
        super();
        if (chrome.audio) {
            chrome.audio.getDevices({ isActive: true, streamTypes: [chrome.audio.StreamType.OUTPUT] }, (devices) => this.updateShouldPanForDevices_(devices));
            chrome.audio.onDeviceListChanged.addListener((devices) => this.updateShouldPanForDevices_(devices));
        }
        else {
            this.shouldPan_ = false;
        }
    }
    /**
     * @return The human-readable name of the earcon set.
     */
    getName() {
        return 'ChromeVox earcons';
    }
    /**
     * Plays the specified earcon sound.
     * @param {EarconId} earcon An earcon identifier.
     * @param {chrome.automation.Rect=} opt_location A location associated with
     *     the earcon such as a control's bounding rectangle.
     */
    playEarcon(earcon, opt_location) {
        if (!this.enabled) {
            return;
        }
        if (SettingsManager.getBoolean('enableEarconLogging')) {
            LogStore.instance.writeTextLog(earcon, LogType.EARCON);
            console.log('Earcon ' + earcon);
        }
        if (ChromeVoxRange.current?.isValid()) {
            const node = ChromeVoxRange.current.start.node;
            const rect = opt_location ?? node.location;
            const container = node.root?.location;
            if (this.shouldPan_ && container) {
                this.engine_.setPositionForRect(rect, container);
            }
            else {
                this.engine_.resetPan();
            }
        }
        this.engine_.playEarcon(earcon);
    }
    cancelEarcon(earcon) {
        switch (earcon) {
            case EarconId.PAGE_START_LOADING:
                this.engine_.cancelProgress();
                break;
            case EarconId.CHROMEVOX_LOADING:
                // Cannot simply run engine_.chromevoxLoaded because that plays an
                // additional earcon.
                this.engine_.stopLoop(EarconId.CHROMEVOX_LOADING);
                break;
        }
    }
    toggle() {
        this.enabled = !this.enabled;
        const announce = this.enabled ? Msgs.getMsg('earcons_on') : Msgs.getMsg('earcons_off');
        ChromeVox.tts.speak(announce, QueueMode.FLUSH, Personality.ANNOTATION);
    }
    /**
     * Updates |this.shouldPan_| based on whether internal speakers are active
     * or not.
     * @param devices
     */
    updateShouldPanForDevices_(devices) {
        this.shouldPan_ = !devices.some((device) => device.isActive &&
            device.deviceType === DeviceType$1.INTERNAL_SPEAKER);
    }
}

// Copyright 2015 The Chromium Authors
// Use of this source code is governed by a BSD-style license that can be
// found in the LICENSE file.
/**
 * An observer that reacts to ChromeVox range changes that modifies braille
 * table output when over email or url text fields.
 */
class EditingRangeObserver {
    static instance;
    constructor() {
        ChromeVoxState.ready().then(() => ChromeVoxRange.addObserver(this));
    }
    static init() {
        if (EditingRangeObserver.instance) {
            throw new Error('Cannot call EditingRangeObserver.init more than once');
        }
        EditingRangeObserver.instance = new EditingRangeObserver();
    }
    onCurrentRangeChanged(range, _fromEditing) {
        const inputType = range && range.start.node.inputType;
        if (inputType === 'email' || inputType === 'url') {
            BrailleTranslatorManager.instance.refresh(SettingsManager.getString('brailleTable8'));
            return;
        }
        BrailleTranslatorManager.instance.refresh(SettingsManager.getString('brailleTable'));
    }
}

// Copyright 2017 The Chromium Authors
// Use of this source code is governed by a BSD-style license that can be
// found in the LICENSE file.
/**
 * @fileoverview Custom Automation Event.
 *
 * An object similar to a chrome.automation.AutomationEvent that we can
 * construct, unlike the object from the extension system.
 */
/**
 * An object we can use instead of a chrome.automation.AutomationEvent.
 */
class CustomAutomationEvent {
    type;
    target;
    eventFrom;
    eventFromAction;
    intents;
    constructor(type, target, params = {}) {
        this.type = type;
        this.target = target;
        this.eventFrom = params.eventFrom || '';
        this.eventFromAction = params.eventFromAction;
        this.intents = params.intents || [];
    }
    /**
     * Stops the propagation of this event.
     */
    stopPropagation() {
        throw Error('Can\'t call stopPropagation on a CustomAutomationEvent');
    }
}
TestImportManager.exportForTesting(CustomAutomationEvent);

// Copyright 2016 The Chromium Authors
// Use of this source code is governed by a BSD-style license that can be
// found in the LICENSE file.
/**
 * @fileoverview Provides bindings to instantiate objects in the automation API.
 *
 * Due to restrictions in the extension system, it is not ordinarily possible to
 * construct an object defined by the extension API. However, given an instance
 * of that object, we can save its constructor for future use.
 */
chrome.automation.AutomationNode;
chrome.automation.AutomationEvent;
const EventType$8 = chrome.automation.EventType;
const AutomationObjectConstructorInstaller = {
    /**
     * Installs the AutomationNode and AutomationEvent classes based on an
     * AutomationNode instance.
     * @param {AutomationNode} node
     */
    async init(node) {
        return new Promise(resolve => {
            chrome.automation.AutomationNode =
                /** @type {function (new:AutomationNode)} */ (node.constructor);
            node.addEventListener(EventType$8.CHILDREN_CHANGED, function installAutomationEvent(e) {
                chrome.automation.AutomationEvent =
                    /** @type {function (new:AutomationEvent)} */ (e.constructor);
                node.removeEventListener(chrome.automation.EventType.CHILDREN_CHANGED, installAutomationEvent, true);
                resolve();
            }, true);
        });
    },
};

// Copyright 2020 The Chromium Authors
// Use of this source code is governed by a BSD-style license that can be
// found in the LICENSE file.
/**
 * @fileoverview An EditableLine encapsulates all data concerning a line in the
 * automation tree necessary to provide output. Editable: an editable selection
 * (e.g. start/end offsets) get saved. Line: nodes/offsets at the beginning/end
 * of a line get saved.
 */
const Dir$5 = constants.Dir;
const RoleType$9 = chrome.automation.RoleType;
const StateType$b = chrome.automation.StateType;
const Movement = CursorMovement;
const Unit = CursorUnit;
class EditableLine {
    start_;
    end_;
    localContainerStartOffset_;
    localContainerEndOffset_;
    // Computed members.
    endContainer_;
    lineStart_;
    lineStartContainer_;
    lineStartContainerRecovery_;
    lineEnd_;
    lineEndContainer_;
    localLineStartContainerOffset_ = 0;
    localLineEndContainerOffset_ = 0;
    startContainer_;
    startContainerValue_ = '';
    value_;
    /**
     * Controls whether line computations include offscreen inline text boxes.
     * Note that a caller should have this set prior to creating a line.
     */
    static includeOffscreen = true;
    /**
     * @param baseLineOnStart  Controls whether to use |startNode| or |endNode|
     *     for Line computations. Selections are automatically truncated up to
     *     either the line start or end.
     */
    constructor(startNode, startIndex, endNode, endIndex, baseLineOnStart) {
        this.start_ = new Cursor(startNode, startIndex);
        this.start_ = this.start_.deepEquivalent ?? this.start_;
        this.end_ = new Cursor(endNode, endIndex);
        this.end_ = this.end_.deepEquivalent ?? this.end_;
        // Update |startIndex| and |endIndex| if the calls above to
        // Cursor.deepEquivalent results in cursors to different container
        // nodes. The cursors can point directly to inline text boxes, in which case
        // we should not adjust the container start or end index.
        if (!AutomationPredicate.text(startNode) ||
            (this.start_.node !== startNode &&
                this.start_.node.parent !== startNode)) {
            startIndex =
                (this.start_.index === CURSOR_NODE_INDEX && this.start_.node.name) ?
                    this.start_.node.name.length :
                    this.start_.index;
        }
        if (!AutomationPredicate.text(endNode) ||
            (this.end_.node !== endNode && this.end_.node.parent !== endNode)) {
            endIndex =
                (this.end_.index === CURSOR_NODE_INDEX && this.end_.node.name) ?
                    this.end_.node.name.length :
                    this.end_.index;
        }
        this.localContainerStartOffset_ = startIndex;
        this.localContainerEndOffset_ = endIndex;
        // Note that we calculate the line based only upon |start_| or
        // |end_| even if they do not fall on the same line. It is up to
        // the caller to specify which end to base this line upon since it requires
        // reasoning about two lines.
        let nameLen = 0;
        const lineBase = baseLineOnStart ? this.start_ : this.end_;
        const lineExtend = baseLineOnStart ? this.end_ : this.start_;
        if (lineBase.node.name) {
            nameLen = lineBase.node.name.length;
        }
        this.value_ = new Spannable(lineBase.node.name ?? '', lineBase);
        if (lineBase.node === lineExtend.node) {
            this.value_.setSpan(lineExtend, 0, nameLen);
        }
        this.startContainer_ = this.start_.node;
        if (this.startContainer_.role === RoleType$9.INLINE_TEXT_BOX) {
            this.startContainer_ = this.startContainer_.parent;
        }
        this.startContainerValue_ =
            this.startContainer_?.role === RoleType$9.TEXT_FIELD ?
                this.startContainer_?.value ?? '' :
                this.startContainer_?.name ?? '';
        this.endContainer_ = this.end_.node;
        if (this.endContainer_.role === RoleType$9.INLINE_TEXT_BOX) {
            this.endContainer_ = this.endContainer_.parent;
        }
        // Initialize defaults.
        this.lineStart_ = lineBase.node;
        this.lineEnd_ = this.lineStart_;
        this.lineStartContainer_ = this.lineStart_.parent;
        this.lineEndContainer_ = this.lineStart_.parent;
        // Annotate each chunk with its associated inline text box node.
        this.value_.setSpan(this.lineStart_, 0, nameLen);
        // Also, track the nodes necessary for selection (either their parents, in
        // the case of inline text boxes, or the node itself).
        // TODO(b/314203187): Not null asserted, check that this is correct.
        const parents = [this.startContainer_];
        // Keep track of visited nodes to ensure we don't visit the same node twice.
        // Workaround for crbug.com/1203840.
        const visited = new WeakSet();
        if (this.lineStart_) {
            visited.add(this.lineStart_);
        }
        const startData = this.computeLineStartMetadata_(this.lineStart_, this.value_, parents, visited);
        this.lineStart_ = startData.lineStart;
        // TODO(b/314203187): Not null asserted, check that this is correct.
        this.lineStartContainer_ = this.lineStart_.parent;
        this.value_ = startData.value;
        const textCountBeforeLineStart = startData.textCountBeforeLineStart;
        this.localLineStartContainerOffset_ = textCountBeforeLineStart;
        if (this.lineStartContainer_) {
            this.lineStartContainerRecovery_ =
                new TreePathRecoveryStrategy(this.lineStartContainer_);
        }
        const endData = this.computeLineEndMetadata_(this.lineEnd_, this.value_, parents, visited);
        this.lineEnd_ = endData.lineEnd;
        // TODO(b/314203187): Not null asserted, check that this is correct.
        this.lineEndContainer_ = this.lineEnd_.parent;
        this.value_ = endData.value;
        const textCountAfterLineEnd = endData.textCountAfterLineEnd;
        if (this.lineEndContainer_.name) {
            this.localLineEndContainerOffset_ =
                this.lineEndContainer_.name.length - textCountAfterLineEnd;
        }
        // Annotate with all parent static texts as NodeSpans so that braille
        // routing can key properly into the node with an offset.
        this.value_ = this.annotateWithParents_(this.value_, parents, textCountBeforeLineStart, textCountAfterLineEnd);
    }
    computeLineStartMetadata_(scanNode, value, parents, visited) {
        let lineStart = scanNode;
        if (scanNode) {
            scanNode = this.getPreviousOnLine_(scanNode);
        }
        // Compute |lineStart|.
        while (scanNode && !visited.has(scanNode)) {
            visited.add(scanNode);
            lineStart = scanNode;
            if (scanNode.role !== RoleType$9.INLINE_TEXT_BOX) {
                parents.unshift(scanNode);
            }
            else if (parents[0] !== scanNode.parent) {
                parents.unshift(scanNode.parent);
            }
            const prepend = new Spannable(scanNode.name, scanNode);
            prepend.append(value);
            value = prepend;
            scanNode = this.getPreviousOnLine_(scanNode);
        }
        // Note that we need to account for potential offsets into the static texts
        // as follows.
        let textCountBeforeLineStart = 0;
        let finder = lineStart;
        // TODO(b/314203187): Not null asserted, check that this is correct.
        while (finder.previousSibling &&
            (EditableLine.includeOffscreen ||
                !finder.previousSibling.state[StateType$b.OFFSCREEN])) {
            finder = finder.previousSibling;
            textCountBeforeLineStart += finder.name?.length ?? 0;
        }
        return { lineStart, value, textCountBeforeLineStart };
    }
    computeLineEndMetadata_(scanNode, value, parents, visited) {
        let lineEnd = scanNode;
        if (scanNode) {
            scanNode = this.getNextOnLine_(scanNode);
        }
        // Compute |lineEnd|.
        while (scanNode && !visited.has(scanNode)) {
            visited.add(scanNode);
            lineEnd = scanNode;
            if (scanNode.role !== RoleType$9.INLINE_TEXT_BOX) {
                parents.push(scanNode);
            }
            else if (parents[parents.length - 1] !== scanNode.parent) {
                // TODO(b/314203187): Not null asserted, check that this is correct.
                parents.push(scanNode.parent);
            }
            let annotation = scanNode;
            if (scanNode === this.end_.node) {
                annotation = this.end_;
            }
            value.append(new Spannable(scanNode.name, annotation));
            scanNode = this.getNextOnLine_(scanNode);
        }
        // Note that we need to account for potential offsets into the static texts
        // as follows.
        let textCountAfterLineEnd = 0;
        let finder = lineEnd;
        // TODO(b/314203187): Not null asserted, check that this is correct.
        while (finder.nextSibling &&
            (EditableLine.includeOffscreen ||
                !finder.nextSibling.state[StateType$b.OFFSCREEN])) {
            finder = finder.nextSibling;
            textCountAfterLineEnd += finder.name?.length ?? 0;
        }
        return { lineEnd, value, textCountAfterLineEnd };
    }
    annotateWithParents_(value, parents, textCountBeforeLineStart, textCountAfterLineEnd) {
        let len = 0;
        for (let i = 0; i < parents.length; i++) {
            const parent = parents[i];
            if (!parent.name) {
                continue;
            }
            const prevLen = len;
            let currentLen = parent.name.length;
            let offset = 0;
            // Subtract off the text count before when at the start of line.
            if (i === 0) {
                currentLen -= textCountBeforeLineStart;
                offset = textCountBeforeLineStart;
            }
            // Subtract text count after when at the end of the line.
            if (i === parents.length - 1) {
                currentLen -= textCountAfterLineEnd;
            }
            len += currentLen;
            try {
                value.setSpan(new OutputNodeSpan(parent, offset), prevLen, len);
                // Also, annotate this span if it is associated with line container.
                if (parent === this.startContainer_) {
                    value.setSpan(parent, prevLen, len);
                }
            }
            catch (e) {
                console.error(e);
            }
        }
        return value;
    }
    getNextOnLine_(node) {
        const nextOnLine = node.nextOnLine;
        const nextSibling = node.nextSibling;
        if (nextOnLine?.role) {
            // Ensure that there is a next-on-line node. The role can be undefined
            // for an object that has been destroyed since the object was first
            // cached.
            return nextOnLine;
        }
        if (nextSibling?.previousOnLine === node) {
            // Catches potential breaks in the chain of next-on-line nodes.
            return nextSibling.firstChild;
        }
        return undefined;
    }
    getPreviousOnLine_(node) {
        const previousLine = node.previousOnLine;
        const previousSibling = node.previousSibling;
        if (previousLine?.role) {
            // Ensure that there is a previous-on-line node. The role can be undefined
            // for an object that has been destroyed since the object was first
            // cached.
            return previousLine;
        }
        if (previousSibling?.lastChild?.nextOnLine === node) {
            // Catches potential breaks in the chain of previous-on-line nodes.
            return previousSibling.lastChild;
        }
        return undefined;
    }
    /** Gets the selection offset based on the text content of this line. */
    get startOffset() {
        // It is possible that the start cursor points to content before this line
        // (e.g. in a multi-line selection).
        try {
            return this.value_.getSpanStart(this.start_) +
                (this.start_.index === CURSOR_NODE_INDEX ? 0 : this.start_.index);
        }
        catch (e) {
            // When that happens, fall back to the start of this line.
            return 0;
        }
    }
    /** Gets the selection offset based on the text content of this line. */
    get endOffset() {
        try {
            return this.value_.getSpanStart(this.end_) +
                (this.end_.index === CURSOR_NODE_INDEX ? 0 : this.end_.index);
        }
        catch (e) {
            // When that happens, fall back to the end of this line.
            return this.value_.length;
        }
    }
    /**
     * Gets the selection offset based on the parent's text.
     * The parent is expected to be static text.
     */
    get localStartOffset() {
        return this.localContainerStartOffset_;
    }
    /**
     * Gets the selection offset based on the parent's text.
     * The parent is expected to be static text.
     */
    get localEndOffset() {
        return this.localContainerEndOffset_;
    }
    /**
     * Gets the start offset of the container, relative to the line text
     * content. The container refers to the static text parenting the inline
     * text box.
     */
    get containerStartOffset() {
        return this.value_.getSpanStart(this.startContainer_);
    }
    /**
     * Gets the end offset of the container, relative to the line text content.
     * The container refers to the static text parenting the inline text box.
     */
    get containerEndOffset() {
        return this.value_.getSpanEnd(this.startContainer_) - 1;
    }
    /** @return The text content of this line. */
    get text() {
        return this.value_.toString();
    }
    get selectedText() {
        return this.value_.toString().substring(this.startOffset, this.endOffset);
    }
    get startContainer() {
        return this.startContainer_;
    }
    get endContainer() {
        return this.endContainer_;
    }
    get value() {
        return this.value_;
    }
    get start() {
        return this.start_;
    }
    get end() {
        return this.end_;
    }
    get localContainerStartOffset() {
        return this.localContainerStartOffset_;
    }
    get localContainerEndOffset() {
        return this.localContainerEndOffset_;
    }
    get startContainerValue() {
        return this.startContainerValue_;
    }
    hasCollapsedSelection() {
        return this.start_.equals(this.end_);
    }
    /** @return Whether this line has selection over text nodes. */
    hasTextSelection() {
        if (this.start_.node && this.end_.node) {
            return AutomationPredicate.text(this.start_.node) &&
                AutomationPredicate.text(this.end_.node);
        }
        return false;
    }
    /**
     * Returns true if |otherLine| surrounds the same line as |this|. Note that
     * the contents of the line might be different.
     */
    isSameLine(otherLine) {
        // Equality is intentionally loose here as any of the state nodes can be
        // invalidated at any time. We rely upon the start/anchor of the line
        // staying the same.
        const startNodeAndOffsetMatch = otherLine.lineStartContainer_ === this.lineStartContainer_ &&
            otherLine.localLineStartContainerOffset_ ===
                this.localLineStartContainerOffset_;
        const endNodeAndOffsetMatch = otherLine.lineEndContainer_ === this.lineEndContainer_ &&
            otherLine.localLineEndContainerOffset_ ===
                this.localLineEndContainerOffset_;
        const recoveryNodeAndOffsetMatch = otherLine.lineStartContainerRecovery_?.node ===
            this.lineStartContainerRecovery_?.node &&
            otherLine.localLineStartContainerOffset_ ===
                this.localLineStartContainerOffset_;
        return startNodeAndOffsetMatch || endNodeAndOffsetMatch ||
            recoveryNodeAndOffsetMatch;
    }
    /**
     * Returns true if |otherLine| surrounds the same line as |this| and has the
     * same selection.
     */
    isSameLineAndSelection(otherLine) {
        return this.isSameLine(otherLine) &&
            this.startOffset === otherLine.startOffset &&
            this.endOffset === otherLine.endOffset;
    }
    /** Returns whether this line comes before |otherLine| in document order. */
    isBeforeLine(otherLine) {
        if (!this.lineStartContainer_ || !otherLine.lineStartContainer_) {
            return false;
        }
        if (this.isSameLine(otherLine)) {
            return this.endOffset <= otherLine.endOffset;
        }
        return AutomationUtil.getDirection(this.lineStartContainer_, otherLine.lineStartContainer_) ===
            Dir$5.FORWARD;
    }
    /**
     * Performs a validation that this line still refers to a line given its
     * internally tracked state.
     */
    isValidLine() {
        if (!this.lineStartContainer_ || !this.lineEndContainer_) {
            return false;
        }
        const start = new Cursor(this.lineStartContainer_, this.localLineStartContainerOffset_);
        const end = new Cursor(this.lineEndContainer_, this.localLineEndContainerOffset_ - 1);
        const localStart = start.deepEquivalent ?? start;
        const localEnd = end.deepEquivalent ?? end;
        const localStartNode = localStart.node;
        const localEndNode = localEnd.node;
        // Unfortunately, there are asymmetric errors in lines, so we need to
        // check in both directions.
        let testStartNode = localStartNode;
        do {
            if (testStartNode === localEndNode) {
                return true;
            }
            // Hack/workaround for broken *OnLine links.
            if (testStartNode.nextOnLine?.role) {
                testStartNode = testStartNode.nextOnLine;
            }
            else if (testStartNode.nextSibling?.previousOnLine === testStartNode) {
                testStartNode = testStartNode.nextSibling;
            }
            else {
                break;
            }
        } while (testStartNode);
        let testEndNode = localEndNode;
        do {
            if (testEndNode === localStartNode) {
                return true;
            }
            // Hack/workaround for broken *OnLine links.
            if (testEndNode.previousOnLine?.role) {
                testEndNode = testEndNode.previousOnLine;
            }
            else if (testEndNode.previousSibling?.nextOnLine === testEndNode) {
                testEndNode = testEndNode.previousSibling;
            }
            else {
                break;
            }
        } while (testEndNode);
        return false;
    }
    /** Speaks the line using text to speech. */
    speakLine(prevLine) {
        // Detect when the entire line is just a breaking space. This occurs on
        // Google Docs and requires that we speak it as a new line. However, we
        // still need to account for all of the possible rich output occurring from
        // ancestors of line nodes.
        const isLineBreakingSpace = this.text === '\u00a0';
        // TODO(b/314203187): Not null asserted, check that this is correct.
        const prev = prevLine?.startContainer_.role ? prevLine.startContainer_ : null;
        const lineNodes = this.value_.getSpansInstanceOf(this.startContainer_.constructor);
        const speakNodeAtIndex = (index, prev) => {
            const cur = lineNodes[index];
            if (!cur) {
                return;
            }
            if (cur.children.length) {
                speakNodeAtIndex(++index, cur);
                return;
            }
            const o = new Output();
            if (isLineBreakingSpace) {
                // Apply a replacement for \u00a0 to \n.
                o.withSpeechTextReplacement('\u00a0', '\n');
            }
            o.withRichSpeech(CursorRange.fromNode(cur), prev ? CursorRange.fromNode(prev) : CursorRange.fromNode(cur), OutputCustomEvent.NAVIGATE)
                .onSpeechEnd(() => speakNodeAtIndex(++index, cur));
            // Ignore whitespace only output except if it is leading content on the
            // line.
            if (!o.isOnlyWhitespace || index === 0) {
                o.go();
            }
            else {
                speakNodeAtIndex(++index, cur);
            }
        };
        // TODO(b/314203187): Not null asserted, check that this is correct.
        speakNodeAtIndex(0, prev);
    }
    /**
     * Creates a range around the character to the right of the line's starting
     * position.
     */
    createCharRange() {
        const start = this.start_;
        let end = start.move(Unit.CHARACTER, Movement.DIRECTIONAL, Dir$5.FORWARD);
        // The following conditions detect when|start|moves across a node boundary
        // to|end|.
        if (start.node !== end.node ||
            // When |start| and |end| are equal, that means we've reached
            // the end of the document. This is a node boundary as well.
            start.equals(end)) {
            end = new Cursor(start.node, start.index + 1);
        }
        return new CursorRange(start, end);
    }
    createWordRange(shouldMoveToPreviousWord) {
        const pos = this.start_;
        // When movement goes to the end of a word, we actually want to
        // describe the word itself; this is considered the previous word so
        // impacts the movement type below. We can give further context e.g.
        // by saying "end of word", if we chose to be more verbose.
        const start = pos.move(Unit.WORD, shouldMoveToPreviousWord ? Movement.DIRECTIONAL : Movement.BOUND, Dir$5.BACKWARD);
        const end = start.move(Unit.WORD, Movement.BOUND, Dir$5.FORWARD);
        return new CursorRange(start, end);
    }
}
TestImportManager.exportForTesting(EditableLine);

// Copyright 2023 The Chromium Authors
// Use of this source code is governed by a BSD-style license that can be
// found in the LICENSE file.
/**
 * A list of typing echo options.
 * This defines the way typed characters get spoken.
 * CHARACTER: echoes typed characters.
 * WORD: echoes a word once a breaking character is typed (i.e. spacebar).
 * CHARACTER_AND_WORD: combines CHARACTER and WORD behavior.
 * NONE: speaks nothing when typing.
 */
var TypingEchoState;
(function (TypingEchoState) {
    TypingEchoState[TypingEchoState["CHARACTER"] = 0] = "CHARACTER";
    TypingEchoState[TypingEchoState["WORD"] = 1] = "WORD";
    TypingEchoState[TypingEchoState["CHARACTER_AND_WORD"] = 2] = "CHARACTER_AND_WORD";
    TypingEchoState[TypingEchoState["NONE"] = 3] = "NONE";
})(TypingEchoState || (TypingEchoState = {}));
// STATE_COUNT is the number of possible echo levels.
const STATE_COUNT = 4;
class TypingEcho {
    /**
     * Stores the current choice of how ChromeVox should echo when entering text
     * into an editable text field.
     */
    static current = TypingEchoState.NONE;
    static init() {
        if (TypingEcho.current !== undefined) {
            throw new Error('TypingEcho should only be initialized once.');
        }
        TypingEcho.current = /** @type {TypingEchoState} */ (LocalStorage.get('typingEcho', TypingEchoState.CHARACTER));
        LocalStorage.addListenerForKey('typingEcho', newValue => TypingEcho.current = newValue);
    }
    /**
     * @param cur Current typing echo.
     * @return Next typing echo.
     */
    static cycle(cur) {
        return ((cur ?? TypingEcho.current) + 1) % STATE_COUNT;
    }
    static cycleWithAnnouncement() {
        LocalStorage.set('typingEcho', TypingEcho.cycle(LocalStorage.getNumber('typingEcho')));
        let announce = '';
        switch (LocalStorage.get('typingEcho')) {
            case TypingEchoState.CHARACTER:
                announce = Msgs.getMsg('character_echo');
                break;
            case TypingEchoState.WORD:
                announce = Msgs.getMsg('word_echo');
                break;
            case TypingEchoState.CHARACTER_AND_WORD:
                announce = Msgs.getMsg('character_and_word_echo');
                break;
            case TypingEchoState.NONE:
                announce = Msgs.getMsg('none_echo');
                break;
        }
        ChromeVox.tts.speak(announce, QueueMode.FLUSH, Personality.ANNOTATION);
    }
    /**
     * Return if characters should be spoken given the typing echo option.
     * @param typingEcho Typing echo option.
     * @return Whether the character should be spoken.
     */
    static shouldSpeakChar(typingEcho) {
        return typingEcho === TypingEchoState.CHARACTER_AND_WORD ||
            typingEcho === TypingEchoState.CHARACTER;
    }
}
TestImportManager.exportForTesting(['TypingEchoState', TypingEchoState]);

// Copyright 2015 The Chromium Authors
// Use of this source code is governed by a BSD-style license that can be
// found in the LICENSE file.
/**
 * @fileoverview Generalized logic for providing spoken feedback when editing
 * text fields, both single and multiline fields.
 *
 * {@code ChromeVoxEditableTextBase} is a generalized class that takes the
 * current state in the form of a text string, a cursor start location and a
 * cursor end location, and calls a speak method with the resulting text to
 * be spoken.  This class can be used directly for single line fields or
 * extended to override methods that extract lines for multiline fields
 * or to provide other customizations.
 */
/**
 * A class containing the information needed to speak
 * a text change event to the user.
 */
class TextChangeEvent {
    value_ = '';
    start;
    end;
    triggeredByUser;
    /**
     * @param newValue The new string value of the editable text control.
     * @param newStart The new 0-based start cursor/selection index.
     * @param newEnd The new 0-based end cursor/selection index.
     */
    constructor(newValue, newStart, newEnd, triggeredByUser) {
        this.value = newValue;
        this.start = newStart;
        this.end = newEnd;
        this.triggeredByUser = triggeredByUser;
        // Adjust offsets to be in left to right order.
        if (this.start > this.end) {
            const tempOffset = this.end;
            this.end = this.start;
            this.start = tempOffset;
        }
    }
    get value() {
        return this.value_;
    }
    set value(val) {
        this.value_ = val.replace(/\u00a0/g, ' ');
    }
}
/**
 * A class representing an abstracted editable text control.
 */
class ChromeVoxEditableTextBase {
    static shouldSpeakInsertions = false;
    static maxShortPhraseLen = 60;
    /** Current value of the text field. */
    value_ = '';
    /** 0-based selection start index. */
    start;
    /** 0-based selection end index. */
    end;
    /** True if this is a password field. */
    isPassword;
    /** Text-to-speech object implementing speak() and stop() methods. */
    tts;
    /** Whether or not the text field is multiline. */
    multiline = false;
    /**
     * Whether or not the last update to the text and selection was described.
     *
     * Some consumers of this flag like |ChromeVoxEventWatcher| depend on and
     * react to when this flag is false by generating alternative feedback.
     */
    lastChangeDescribed = false;
    /**
     * @param value The string value of the editable text control.
     * @param start The 0-based start cursor/selection index.
     * @param end The 0-based end cursor/selection index.
     * @param isPassword Whether the text control if a password field.
     */
    constructor(value, start, end, isPassword, tts) {
        this.value = value;
        this.start = start;
        this.end = end;
        this.isPassword = isPassword;
        this.tts = tts;
    }
    get value() {
        return this.value_;
    }
    set value(newValue) {
        this.value_ = newValue.replace('\u00a0', ' ');
    }
    getLineIndex(_charIndex) {
        return 0;
    }
    getLineStart(_lineIndex) {
        return 0;
    }
    getLineEnd(_lineIndex) {
        return this.value.length;
    }
    /**
     * Get the full text of the current line.
     * @param index The 0-based line index.
     * @return The text of the line.
     */
    getLine(index) {
        const lineStart = this.getLineStart(index);
        const lineEnd = this.getLineEnd(index);
        return this.value.substr(lineStart, lineEnd - lineStart);
    }
    /**
     * Speak text, but if it's a single character, describe the character.
     * @param str The string to speak.
     * @param triggeredByUser True if the speech was triggered by a user action.
     * @param personality Personality used to speak text.
     */
    speak(str, triggeredByUser, personality) {
        if (!str) {
            return;
        }
        let queueMode = QueueMode.QUEUE;
        if (triggeredByUser === true) {
            queueMode = QueueMode.CATEGORY_FLUSH;
        }
        const props = personality ?? new TtsSpeechProperties();
        props.category = TtsCategory.NAV;
        this.tts.speak(str, queueMode, props);
    }
    /**
     * The function is called by describeTextChanged and process if there's
     * some text changes likely made by IME.
     * @param {TextChangeEvent} prev The previous text change event.
     * @param {TextChangeEvent} evt The text change event.
     * @param {number} commonPrefixLen The number of characters in the common
     *     prefix of this.value and newValue.
     * @param {number} commonSuffixLen The number of characters in the common
     *     suffix of this.value and newValue.
     * @return {boolean} True if the event was processed.
     */
    describeTextChangedByIME(prev, evt, commonPrefixLen, commonSuffixLen) {
        // This supports typing Echo with IME.
        // - no selection range before and after.
        // - suffixes are common after both cursor end.
        // - prefixes are common at least max(0, "before length - 3").
        // Then, something changed in composition range. Announce the new
        // characters.
        const relaxedPrefixLen = Math.max(prev.start - MAX_CHANGE_CHARS_BY_SINGLE_TYPE, 0);
        let suffixLen = evt.value.length - evt.end;
        if (prev.start === prev.end && evt.start === evt.end &&
            prev.value.length - prev.end === suffixLen &&
            commonPrefixLen >= relaxedPrefixLen && commonPrefixLen < evt.start &&
            commonSuffixLen >= suffixLen) {
            if (LocalStorage.get('typingEcho') === TypingEchoState.CHARACTER ||
                LocalStorage.get('typingEcho') ===
                    TypingEchoState.CHARACTER_AND_WORD) {
                this.speak(evt.value.substring(commonPrefixLen, evt.start), evt.triggeredByUser);
            }
            return true;
        }
        // The followings happens when a user starts to select candidates.
        // - no selection range before and after.
        // - prefixes are common before "new cursor point".
        // - suffixes are common after "old cursor point".
        // Then, this suggests that pressing a space or a tab to start composition.
        // Let's announce the first suggested content.
        // Note that after announcing this, announcements will be made by candidate
        // window's selection event instead of ChromeVox's editable.
        const prefixLen = evt.start;
        suffixLen = prev.value.length - prev.end;
        if (prev.start === prev.end && evt.start === evt.end &&
            evt.start < prev.start && evt.value.length > prefixLen + suffixLen &&
            commonPrefixLen >= prefixLen && commonSuffixLen >= suffixLen) {
            this.speak(evt.value.substring(prefixLen, evt.value.length - suffixLen), evt.triggeredByUser, new TtsSpeechProperties({ 'phoneticCharacters': true }));
            return true;
        }
        return false;
    }
    /**
     * The function is called by describeTextChanged after it's figured out
     * what text was deleted, what text was inserted, and what additional
     * autocomplete text was added.
     * @param {TextChangeEvent} prev The previous text change event.
     * @param {TextChangeEvent} evt The text change event.
     * @param {number} prefixLen The number of characters in the common prefix
     *     of this.value and newValue.
     * @param {number} suffixLen The number of characters in the common suffix
     *     of this.value and newValue.
     * @param {string} autocompleteSuffix The autocomplete string that was added
     *     to the end, if any. It should be spoken at the end of the utterance
     *     describing the change.
     * @param {TtsSpeechProperties=} personality Personality to speak the
     *     text.
     */
    describeTextChangedHelper(prev, evt, prefixLen, suffixLen, autocompleteSuffix, personality) {
        const len = prev.value.length;
        const newLen = evt.value.length;
        const deletedLen = len - prefixLen - suffixLen;
        const deleted = prev.value.substr(prefixLen, deletedLen);
        const insertedLen = newLen - prefixLen - suffixLen;
        const inserted = evt.value.substr(prefixLen, insertedLen);
        let utterance = '';
        let triggeredByUser = evt.triggeredByUser;
        if (insertedLen > 1) {
            if (!ChromeVoxEditableTextBase.shouldSpeakInsertions) {
                return;
            }
            utterance = inserted;
        }
        else if (insertedLen === 1) {
            if ((LocalStorage.get('typingEcho') === TypingEchoState.WORD ||
                LocalStorage.get('typingEcho') ===
                    TypingEchoState.CHARACTER_AND_WORD) &&
                StringUtil.isWordBreakChar(inserted) && prefixLen > 0 &&
                !StringUtil.isWordBreakChar(evt.value.substr(prefixLen - 1, 1))) {
                // Speak previous word.
                let index = prefixLen;
                while (index > 0 &&
                    !StringUtil.isWordBreakChar(evt.value[index - 1])) {
                    index--;
                }
                if (index < prefixLen) {
                    utterance = evt.value.substr(index, prefixLen + 1 - index);
                }
                else {
                    utterance = inserted;
                    triggeredByUser = false; // Implies QUEUE_MODE_QUEUE.
                }
            }
            else if (LocalStorage.get('typingEcho') === TypingEchoState.CHARACTER ||
                LocalStorage.get('typingEcho') ===
                    TypingEchoState.CHARACTER_AND_WORD) {
                utterance = inserted;
            }
        }
        else if (deletedLen > 1 && !autocompleteSuffix) {
            utterance = deleted + ', deleted';
        }
        else if (deletedLen === 1) {
            utterance = deleted;
            // Single-deleted characters should also use Personality.DELETED.
            personality = Personality.DELETED;
        }
        if (autocompleteSuffix && utterance) {
            utterance += ', ' + autocompleteSuffix;
        }
        else if (autocompleteSuffix) {
            utterance = autocompleteSuffix;
        }
        if (utterance) {
            this.speak(utterance, triggeredByUser, personality);
        }
    }
}
// Private to module.
const MAX_CHANGE_CHARS_BY_SINGLE_TYPE = 3;
TestImportManager.exportForTesting(ChromeVoxEditableTextBase, TextChangeEvent);

// Copyright 2023 The Chromium Authors
// Use of this source code is governed by a BSD-style license that can be
// found in the LICENSE file.
const StateType$a = chrome.automation.StateType;
/**
 * A |ChromeVoxEditableTextBase| that implements text editing feedback
 * for automation tree text fields.
 */
class AutomationEditableText extends ChromeVoxEditableTextBase {
    lineBreaks_;
    node_;
    constructor(node) {
        // TODO(b/314203187): Not null asserted, check that this is correct.
        if (!node.state[StateType$a.EDITABLE]) {
            throw Error('Node must have editable state set to true.');
        }
        const value = AutomationEditableText.getProcessedValue_(node) ?? '';
        const lineBreaks = AutomationEditableText.getLineBreaks_(value);
        const start = node.textSelStart;
        const end = node.textSelEnd;
        super(value, Math.min(start, end, value.length), Math.min(Math.max(start, end), value.length), node.state[StateType$a.PROTECTED] /**password*/, ChromeVox.tts);
        this.lineBreaks_ = lineBreaks;
        this.multiline = node.state[StateType$a.MULTILINE] || false;
        this.node_ = node;
    }
    /**
     * Update the state of the text and selection and describe any changes as
     * appropriate.
     */
    changed(evt) {
        // Temporarily call via prototype during the migration.
        // Because the tests still use EditableTextBase, they invoke this method via
        // AutomationEditableText.prototype.changed.call(). The |this| object in that case does not
        // have a |shouldDescribeChange| method, so we have to reference it explicitly during the
        // migration.
        if (!AutomationEditableText.prototype.shouldDescribeChange.call(this, evt)) {
            this.lastChangeDescribed = false;
            return;
        }
        if (evt.value === this.value) {
            AutomationEditableText.prototype.describeSelectionChanged.call(this, evt);
        }
        else {
            AutomationEditableText.prototype.describeTextChanged.call(this, new TextChangeEvent(this.value, this.start, this.end, true), evt);
        }
        this.lastChangeDescribed = true;
        this.value = evt.value;
        this.start = evt.start;
        this.end = evt.end;
    }
    /**
     * Describe a change in the selection or cursor position when the text
     * stays the same.
     * @param evt The text change event.
     */
    describeSelectionChanged(evt) {
        // TODO(deboer): Factor this into two function:
        //   - one to determine the selection event
        //   - one to speak
        if (this.isPassword) {
            this.speak(Msgs.getMsg('password_char'), evt.triggeredByUser);
            return;
        }
        if (evt.start === evt.end) {
            // It's currently a cursor.
            if (this.start !== this.end) {
                // It was previously a selection.
                this.speak(this.value.substring(this.start, this.end), evt.triggeredByUser);
                this.speak(Msgs.getMsg('removed_from_selection'));
            }
            else if (this.getLineIndex(this.start) !== this.getLineIndex(evt.start)) {
                // Moved to a different line; read it.
                let lineValue = this.getLine(this.getLineIndex(evt.start));
                if (lineValue === '') {
                    lineValue = Msgs.getMsg('text_box_blank');
                }
                else if (lineValue === '\n') ;
                else if (/^\s+$/.test(lineValue)) {
                    lineValue = Msgs.getMsg('text_box_whitespace');
                }
                this.speak(lineValue, evt.triggeredByUser);
            }
            else if (this.start === evt.start + 1 || this.start === evt.start - 1) {
                // Moved by one character; read it.
                if (evt.start === this.value.length) {
                    this.speak(Msgs.getMsg('end_of_text_verbose'), evt.triggeredByUser);
                }
                else {
                    this.speak(this.value.substr(evt.start, 1), evt.triggeredByUser, new TtsSpeechProperties({ 'phoneticCharacters': evt.triggeredByUser }));
                }
            }
            else {
                // Moved by more than one character. Read all characters crossed.
                this.speak(this.value.substr(Math.min(this.start, evt.start), Math.abs(this.start - evt.start)), evt.triggeredByUser);
            }
        }
        else {
            // It's currently a selection.
            if (this.start + 1 === evt.start && this.end === this.value.length &&
                evt.end === this.value.length) {
                // Autocomplete: the user typed one character of autocompleted text.
                if (LocalStorage.get('typingEcho') === TypingEchoState.CHARACTER ||
                    LocalStorage.get('typingEcho') ===
                        TypingEchoState.CHARACTER_AND_WORD) {
                    this.speak(this.value.substr(this.start, 1), evt.triggeredByUser);
                }
                this.speak(this.value.substr(evt.start));
            }
            else if (this.start === this.end) {
                // It was previously a cursor.
                this.speak(this.value.substr(evt.start, evt.end - evt.start), evt.triggeredByUser);
                this.speak(Msgs.getMsg('selected'));
            }
            else if (this.start === evt.start && this.end < evt.end) {
                this.speak(this.value.substr(this.end, evt.end - this.end), evt.triggeredByUser);
                this.speak(Msgs.getMsg('added_to_selection'));
            }
            else if (this.start === evt.start && this.end > evt.end) {
                this.speak(this.value.substr(evt.end, this.end - evt.end), evt.triggeredByUser);
                this.speak(Msgs.getMsg('removed_from_selection'));
            }
            else if (this.end === evt.end && this.start > evt.start) {
                this.speak(this.value.substr(evt.start, this.start - evt.start), evt.triggeredByUser);
                this.speak(Msgs.getMsg('added_to_selection'));
            }
            else if (this.end === evt.end && this.start < evt.start) {
                this.speak(this.value.substr(this.start, evt.start - this.start), evt.triggeredByUser);
                this.speak(Msgs.getMsg('removed_from_selection'));
            }
            else {
                // The selection changed but it wasn't an obvious extension of
                // a previous selection. Just read the new selection.
                this.speak(this.value.substr(evt.start, evt.end - evt.start), evt.triggeredByUser);
                this.speak(Msgs.getMsg('selected'));
            }
        }
    }
    /** Describe a change where the text changes. */
    describeTextChanged(prev, evt) {
        let personality = new TtsSpeechProperties();
        if (evt.value.length < (prev.value.length - 1)) {
            personality = Personality.DELETED;
        }
        if (this.isPassword) {
            this.speak(Msgs.getMsg('password_char'), evt.triggeredByUser, personality);
            return;
        }
        // First, see if there's a selection at the end that might have been
        // added by autocomplete. If so, replace the event information with it.
        const origEvt = evt;
        let autocompleteSuffix = '';
        if (evt.start < evt.end && evt.end === evt.value.length) {
            autocompleteSuffix = evt.value.slice(evt.start);
            evt = new TextChangeEvent(evt.value.slice(0, evt.start), evt.start, evt.start, evt.triggeredByUser);
        }
        // Precompute the length of prefix and suffix of values.
        const commonPrefixLen = StringUtil.longestCommonPrefixLength(evt.value, prev.value);
        const commonSuffixLen = StringUtil.longestCommonSuffixLength(evt.value, prev.value);
        // Now see if the previous selection (if any) was deleted
        // and any new text was inserted at that character position.
        // This would handle pasting and entering text by typing, both from
        // a cursor and from a selection.
        let prefixLen = prev.start;
        let suffixLen = prev.value.length - prev.end;
        if (evt.value.length >= prefixLen + suffixLen + (evt.end - evt.start) &&
            commonPrefixLen >= prefixLen && commonSuffixLen >= suffixLen) {
            this.describeTextChangedHelper(prev, origEvt, prefixLen, suffixLen, autocompleteSuffix, personality);
            return;
        }
        // Next, see if one or more characters were deleted from the previous
        // cursor position and the new cursor is in the expected place. This
        // handles backspace, forward-delete, and similar shortcuts that delete
        // a word or line.
        prefixLen = evt.start;
        suffixLen = evt.value.length - evt.end;
        if (prev.start === prev.end && evt.start === evt.end &&
            commonPrefixLen >= prefixLen && commonSuffixLen >= suffixLen) {
            // Forward deletions causes reading of the character immediately to the
            // right of the caret.
            if (prev.start === evt.start && prev.end === evt.end) {
                this.speak(evt.value[evt.start], evt.triggeredByUser);
            }
            else {
                this.describeTextChangedHelper(prev, origEvt, prefixLen, suffixLen, autocompleteSuffix, personality);
            }
            return;
        }
        // See if the change is related to IME's complex operation.
        if (this.describeTextChangedByIME(prev, evt, commonPrefixLen, commonSuffixLen)) {
            return;
        }
        // If all above fails, we assume the change was not the result of a normal
        // user editing operation, so we'll have to speak feedback based only
        // on the changes to the text, not the cursor position / selection.
        // First, restore the event.
        evt = origEvt;
        // Try to do a diff between the new and the old text. If it is a one
        // character insertion/deletion at the start or at the end, just speak that
        // character.
        if ((evt.value.length === (prev.value.length + 1)) ||
            ((evt.value.length + 1) === prev.value.length)) {
            // The user added text either to the beginning or the end.
            if (evt.value.length > prev.value.length) {
                if (commonPrefixLen === prev.value.length) {
                    this.speak(evt.value[evt.value.length - 1], evt.triggeredByUser, personality);
                    return;
                }
                else if (commonSuffixLen === prev.value.length) {
                    this.speak(evt.value[0], evt.triggeredByUser, personality);
                    return;
                }
            }
            // The user deleted text either from the beginning or the end.
            if (evt.value.length < prev.value.length) {
                if (commonPrefixLen === evt.value.length) {
                    this.speak(prev.value[prev.value.length - 1], evt.triggeredByUser, personality);
                    return;
                }
                else if (commonSuffixLen === evt.value.length) {
                    this.speak(prev.value[0], evt.triggeredByUser, personality);
                    return;
                }
            }
        }
        if (this.multiline) {
            // The below is a somewhat loose way to deal with non-standard
            // insertions/deletions. Intentionally skip for multiline since deletion
            // announcements are covered above and insertions are non-standard
            // (possibly due to auto complete). Since content editable's often refresh
            // content by removing and inserting entire chunks of text, this type of
            // logic often results in unintended consequences such as reading all text
            // when only one character has been entered.
            return;
        }
        // If the text is short, just speak the whole thing.
        if (evt.value.length <= ChromeVoxEditableTextBase.maxShortPhraseLen) {
            this.describeTextChangedHelper(prev, evt, 0, 0, '', personality);
            return;
        }
        // Otherwise, look for the common prefix and suffix, but back up so
        // that we can speak complete words, to be minimally confusing.
        prefixLen = commonPrefixLen;
        while (prefixLen < prev.value.length && prefixLen < evt.value.length &&
            prev.value[prefixLen] === evt.value[prefixLen]) {
            prefixLen++;
        }
        while (prefixLen > 0 &&
            !StringUtil.isWordBreakChar(prev.value[prefixLen - 1])) {
            prefixLen--;
        }
        // For suffix, commonSuffixLen is not used because suffix here won't overlap
        // with prefix, and also we need to consider |autocompleteSuffix|.
        suffixLen = 0;
        while (suffixLen < (prev.value.length - prefixLen) &&
            suffixLen < (evt.value.length - prefixLen) &&
            prev.value[prev.value.length - suffixLen - 1] ===
                evt.value[evt.value.length - suffixLen - 1]) {
            suffixLen++;
        }
        while (suffixLen > 0 &&
            !StringUtil.isWordBreakChar(prev.value[prev.value.length - suffixLen])) {
            suffixLen--;
        }
        this.describeTextChangedHelper(prev, evt, prefixLen, suffixLen, '', personality);
    }
    /**
     * @param evt The new text changed event to test.
     * @return True if the event, when compared to the previous text, should
     *     trigger description.
     */
    shouldDescribeChange(evt) {
        if (evt.value === this.value && evt.start === this.start &&
            evt.end === this.end) {
            return false;
        }
        return true;
    }
    /** Called when the text field has been updated. */
    onUpdate(_intents) {
        const oldValue = this.value;
        const oldStart = this.start;
        const oldEnd = this.end;
        const newValue = AutomationEditableText.getProcessedValue_(this.node_) ?? '';
        if (oldValue !== newValue) {
            this.lineBreaks_ = AutomationEditableText.getLineBreaks_(newValue);
        }
        const textChangeEvent = new TextChangeEvent(newValue, Math.min(this.node_.textSelStart ?? 0, newValue.length), Math.min(this.node_.textSelEnd ?? 0, newValue.length), true /* triggered by user */);
        this.changed(textChangeEvent);
        this.outputBraille_(oldValue, oldStart, oldEnd);
    }
    /** Returns true if selection starts on the first line. */
    isSelectionOnFirstLine() {
        return this.getLineIndex(this.start) === 0;
    }
    /** Returns true if selection ends on the last line. */
    isSelectionOnLastLine() {
        return this.getLineIndex(this.end) >= this.lineBreaks_.length - 1;
    }
    getLineIndex(charIndex) {
        let lineIndex = 0;
        while (charIndex > this.lineBreaks_[lineIndex]) {
            lineIndex++;
        }
        return lineIndex;
    }
    getLineStart(lineIndex) {
        if (lineIndex === 0) {
            return 0;
        }
        // The start of this line is defined as the line break of the previous line
        // + 1 (the hard line break).
        return this.lineBreaks_[lineIndex - 1] + 1;
    }
    getLineEnd(lineIndex) {
        return this.lineBreaks_[lineIndex];
    }
    getLineIndexForBrailleOutput_(oldStart) {
        let lineIndex = this.getLineIndex(this.start);
        // Output braille at the end of the selection that changed, if start and end
        // differ.
        if (this.start !== this.end && this.start === oldStart) {
            lineIndex = this.getLineIndex(this.end);
        }
        return lineIndex;
    }
    getTextFromIndexAndStart_(lineIndex, lineStart) {
        const lineEnd = this.getLineEnd(lineIndex);
        let lineText = this.value.substr(lineStart, lineEnd - lineStart);
        if (lineIndex === 0) {
            const textFieldTypeMsg = Msgs.getMsg(this.multiline ? 'tag_textarea_brl' : 'role_textbox_brl');
            lineText += ' ' + textFieldTypeMsg;
        }
        return lineText;
    }
    outputBraille_(_oldValue, oldStart, _oldEnd) {
        const lineIndex = this.getLineIndexForBrailleOutput_(oldStart);
        const lineStart = this.getLineStart(lineIndex);
        let lineText = this.getTextFromIndexAndStart_(lineIndex, lineStart);
        const startIndex = this.start - lineStart;
        const endIndex = this.end - lineStart;
        // If the line is not the last line, and is empty, insert an explicit line
        // break so that braille output is correctly cleared and has a position for
        // a caret to be shown.
        if (lineText === '' && lineIndex < this.lineBreaks_.length - 1) {
            lineText = '\n';
        }
        const value = new Spannable(lineText, new OutputNodeSpan(this.node_));
        value.setSpan(new ValueSpan(0), 0, lineText.length);
        value.setSpan(new ValueSelectionSpan(), startIndex, endIndex);
        ChromeVox.braille.write(new NavBraille({ text: value, startIndex, endIndex }));
    }
    static getProcessedValue_(node) {
        let value = node.value;
        if (node.inputType === 'tel') {
            value = value?.trimEnd();
        }
        return value;
    }
    static getLineBreaks_(value) {
        const lineBreaks = [];
        const lines = value.split('\n');
        let total = 0;
        for (let i = 0; i < lines.length; i++) {
            total += lines[i].length;
            lineBreaks[i] = total;
            // Account for the line break itself.
            total++;
        }
        return lineBreaks;
    }
}
TestImportManager.exportForTesting(AutomationEditableText);

// Copyright 2019 The Chromium Authors
// Use of this source code is governed by a BSD-style license that can be
// found in the LICENSE file.
/**
 * @fileoverview Provides color matching services for ChromeVox.
 */
class Color {
    /**
     * Returns a string representation of a color.
     * @param color The argb value represented as an integer.
     */
    static getColorDescription(color) {
        if (!color) {
            return '';
        }
        // Convert to unsigned integer.
        color = color >>> 0;
        // The following 24 bits represent the rgb value. Filter out first 8 bits.
        const rgb = color & 0x00ffffff;
        const optSubs = [
            Color.findClosestMatchingColor(rgb),
            Color.getOpacityPercentage(color).toString(),
        ];
        return Msgs.getMsg('color_description', optSubs);
    }
    /**
     * Extracts the opacity of the color, which is encoded within the first 8
     * bits.
     * @param color An integer representation of a color.
     */
    static getOpacityPercentage(color) {
        return Math.round(((color >>> 24) / 256) * 100);
    }
    /**
     * Finds the most similar stored color given an rgb value.
     * @param target The rgb value as an integer.
     */
    static findClosestMatchingColor(target) {
        const bestMatch = ColorObjectArray.reduce((closest, color) => {
            const distance = Color.findDistance(target, color.value);
            if (distance < closest.distance) {
                return { color, distance };
            }
            return closest;
        }, { distance: Number.MAX_VALUE, color: ColorObjectArray[0] });
        // Do not report color if most similar color is too inaccurate.
        if (bestMatch.distance > DISTANCE_THRESHOLD) {
            return '';
        }
        return Msgs.getMsg(bestMatch.color.colorMessageId);
    }
    /**
     * Calculates the distance between two 3-D points, encoded as numbers,
     * that represent colors.
     * The first 8 bits are unused as they have either been shifted off or are
     * simply filled by zeros. The x component is designated by the second
     * 8 bits. The y component is designated by the third 8 bits.
     * The z component is designated by the last 8 bits.
     */
    static findDistance(firstColor, secondColor) {
        // Extract x, y, and z components.
        const firstColorX = (firstColor & 0xff0000) >> 16;
        const firstColorY = (firstColor & 0x00ff00) >> 8;
        const firstColorZ = (firstColor & 0x0000ff);
        const secondColorX = (secondColor & 0xff0000) >> 16;
        const secondColorY = (secondColor & 0x00ff00) >> 8;
        const secondColorZ = (secondColor & 0x0000ff);
        return Math.pow(secondColorX - firstColorX, 2) +
            Math.pow(secondColorY - firstColorY, 2) +
            Math.pow(secondColorZ - firstColorZ, 2);
    }
}
// Module-local variables.
/**
 * The distance between black and dark grey is the threshold.
 * 0x000000 = Black.
 * 0x282828 = Dark Grey. This value was chosen somewhat arbitrarily. It encodes
 * a shade of grey that could be visibly identified as black.
 * @const {number}
 */
const DISTANCE_THRESHOLD = Color.findDistance(0X000000, 0X282828);
/**
 * Holds objects that contain hexadecimal RGB values of colors and their
 * corresponding ChromeVox message IDs.
 * @private {!Array<{colorMessageId: string, value: number}>}
 * Obtained from url: https://www.w3schools.com/lib/w3color.js
 */
const ColorObjectArray = [
    { 'value': 0x0, 'colorMessageId': 'color_black' },
    { 'value': 0x6400, 'colorMessageId': 'color_dark_green' },
    { 'value': 0x8000, 'colorMessageId': 'color_green' },
    { 'value': 0x800080, 'colorMessageId': 'color_purple' },
    { 'value': 0xb8860b, 'colorMessageId': 'color_dark_golden_rod' },
    { 'value': 0xfffacd, 'colorMessageId': 'color_lemon_chiffon' },
    { 'value': 0xa0522d, 'colorMessageId': 'color_sienna' },
    { 'value': 0xffa500, 'colorMessageId': 'color_orange' },
    { 'value': 0x8b4513, 'colorMessageId': 'color_saddle_brown' },
    { 'value': 0xffff, 'colorMessageId': 'color_cyan' },
    { 'value': 0xadff2f, 'colorMessageId': 'color_green_yellow' },
    { 'value': 0xd2691e, 'colorMessageId': 'color_chocolate' },
    { 'value': 0x800000, 'colorMessageId': 'color_maroon' },
    { 'value': 0xdaa520, 'colorMessageId': 'color_golden_rod' },
    { 'value': 0x228b22, 'colorMessageId': 'color_forest_green' },
    { 'value': 0x6b8e23, 'colorMessageId': 'color_olive_drab' },
    { 'value': 0xfffff0, 'colorMessageId': 'color_ivory' },
    { 'value': 0xf5f5dc, 'colorMessageId': 'color_beige' },
    { 'value': 0xa52a2a, 'colorMessageId': 'color_brown' },
    { 'value': 0x9acd32, 'colorMessageId': 'color_yellow_green' },
    { 'value': 0xff4500, 'colorMessageId': 'color_orange_red' },
    { 'value': 0x556b2f, 'colorMessageId': 'color_dark_olive_green' },
    { 'value': 0x32cd32, 'colorMessageId': 'color_lime_green' },
    { 'value': 0xff00, 'colorMessageId': 'color_lime' },
    { 'value': 0xeee8aa, 'colorMessageId': 'color_pale_golden_rod' },
    { 'value': 0xff69b4, 'colorMessageId': 'color_hot_pink' },
    { 'value': 0xdc143c, 'colorMessageId': 'color_crimson' },
    { 'value': 0xb0e0e6, 'colorMessageId': 'color_powder_blue' },
    { 'value': 0x808000, 'colorMessageId': 'color_olive' },
    { 'value': 0xffffe0, 'colorMessageId': 'color_light_yellow' },
    { 'value': 0xfaf0e6, 'colorMessageId': 'color_linen' },
    { 'value': 0x8b, 'colorMessageId': 'color_dark_blue' },
    { 'value': 0xf8f8ff, 'colorMessageId': 'color_ghost_white' },
    { 'value': 0xff6347, 'colorMessageId': 'color_tomato' },
    { 'value': 0xf0e68c, 'colorMessageId': 'color_khaki' },
    { 'value': 0x2f4f4f, 'colorMessageId': 'color_dark_slate_grey' },
    { 'value': 0xff7f50, 'colorMessageId': 'color_coral' },
    { 'value': 0xf5fffa, 'colorMessageId': 'color_mint_cream' },
    { 'value': 0x8080, 'colorMessageId': 'color_teal' },
    { 'value': 0x8b008b, 'colorMessageId': 'color_dark_magenta' },
    { 'value': 0xffa07a, 'colorMessageId': 'color_light_salmon' },
    { 'value': 0x2e8b57, 'colorMessageId': 'color_sea_green' },
    { 'value': 0xff0000, 'colorMessageId': 'color_red' },
    { 'value': 0xbc8f8f, 'colorMessageId': 'color_rosy_brown' },
    { 'value': 0xcd5c5c, 'colorMessageId': 'color_indian_red' },
    { 'value': 0xd3d3d3, 'colorMessageId': 'color_light_grey' },
    { 'value': 0xf4a460, 'colorMessageId': 'color_sandy_brown' },
    { 'value': 0x90ee90, 'colorMessageId': 'color_light_green' },
    { 'value': 0xadd8e6, 'colorMessageId': 'color_light_blue' },
    { 'value': 0xff8c00, 'colorMessageId': 'color_dark_orange' },
    { 'value': 0x696969, 'colorMessageId': 'color_dim_grey' },
    { 'value': 0xffebcd, 'colorMessageId': 'color_blanched_almond' },
    { 'value': 0xbdb76b, 'colorMessageId': 'color_dark_khaki' },
    { 'value': 0xff00ff, 'colorMessageId': 'color_magenta' },
    { 'value': 0x191970, 'colorMessageId': 'color_midnight_blue' },
    { 'value': 0x3cb371, 'colorMessageId': 'color_medium_sea_green' },
    { 'value': 0xfa8072, 'colorMessageId': 'color_salmon' },
    { 'value': 0xff1493, 'colorMessageId': 'color_deep_pink' },
    { 'value': 0xe9967a, 'colorMessageId': 'color_dark_salmon' },
    { 'value': 0xcd853f, 'colorMessageId': 'color_peru' },
    { 'value': 0xff7f, 'colorMessageId': 'color_spring_green' },
    { 'value': 0x80, 'colorMessageId': 'color_navy' },
    { 'value': 0xf08080, 'colorMessageId': 'color_light_coral' },
    { 'value': 0x4b0082, 'colorMessageId': 'color_indigo' },
    { 'value': 0xffffff, 'colorMessageId': 'color_white' },
    { 'value': 0xc71585, 'colorMessageId': 'color_medium_violet_red' },
    { 'value': 0xdeb887, 'colorMessageId': 'color_burly_wood' },
    { 'value': 0xe6e6fa, 'colorMessageId': 'color_lavender' },
    { 'value': 0x483d8b, 'colorMessageId': 'color_dark_slate_blue' },
    { 'value': 0xd2b48c, 'colorMessageId': 'color_tan' },
    { 'value': 0x8fbc8f, 'colorMessageId': 'color_dark_sea_green' },
    { 'value': 0x708090, 'colorMessageId': 'color_slate_grey' },
    { 'value': 0xdb7093, 'colorMessageId': 'color_pale_violet_red' },
    { 'value': 0xfff8dc, 'colorMessageId': 'color_cornsilk' },
    { 'value': 0xafeeee, 'colorMessageId': 'color_pale_turquoise' },
    { 'value': 0x778899, 'colorMessageId': 'color_light_slate_grey' },
    { 'value': 0x98fb98, 'colorMessageId': 'color_pale_green' },
    { 'value': 0x663399, 'colorMessageId': 'color_rebecca_purple' },
    { 'value': 0xfa9a, 'colorMessageId': 'color_medium_spring_green' },
    { 'value': 0xffc0cb, 'colorMessageId': 'color_pink' },
    { 'value': 0x5f9ea0, 'colorMessageId': 'color_cadet_blue' },
    { 'value': 0x808080, 'colorMessageId': 'color_grey' },
    { 'value': 0xee82ee, 'colorMessageId': 'color_violet' },
    { 'value': 0xa9a9a9, 'colorMessageId': 'color_dark_grey' },
    { 'value': 0x20b2aa, 'colorMessageId': 'color_light_sea_green' },
    { 'value': 0x8b8b, 'colorMessageId': 'color_dark_cyan' },
    { 'value': 0xffdead, 'colorMessageId': 'color_navajo_white' },
    { 'value': 0xf0f8ff, 'colorMessageId': 'color_alice_blue' },
    { 'value': 0xfffaf0, 'colorMessageId': 'color_floral_white' },
    { 'value': 0xffe4e1, 'colorMessageId': 'color_misty_rose' },
    { 'value': 0xf5deb3, 'colorMessageId': 'color_wheat' },
    { 'value': 0x4682b4, 'colorMessageId': 'color_steel_blue' },
    { 'value': 0xffe4b5, 'colorMessageId': 'color_moccasin' },
    { 'value': 0xffdab9, 'colorMessageId': 'color_peach_puff' },
    { 'value': 0xffd700, 'colorMessageId': 'color_gold' },
    { 'value': 0xfff0f5, 'colorMessageId': 'color_lavender_blush' },
    { 'value': 0xc0c0c0, 'colorMessageId': 'color_silver' },
    { 'value': 0xffb6c1, 'colorMessageId': 'color_light_pink' },
    { 'value': 0xf0ffff, 'colorMessageId': 'color_azure' },
    { 'value': 0xffe4c4, 'colorMessageId': 'color_bisque' },
    { 'value': 0x9932cc, 'colorMessageId': 'color_dark_orchid' },
    { 'value': 0xfdf5e6, 'colorMessageId': 'color_old_lace' },
    { 'value': 0x48d1cc, 'colorMessageId': 'color_medium_turquoise' },
    { 'value': 0x6a5acd, 'colorMessageId': 'color_slate_blue' },
    { 'value': 0xcd, 'colorMessageId': 'color_medium_blue' },
    { 'value': 0x40e0d0, 'colorMessageId': 'color_turquoise' },
    { 'value': 0xced1, 'colorMessageId': 'color_dark_turquoise' },
    { 'value': 0xfafad2, 'colorMessageId': 'color_light_golden_rod_yellow' },
    { 'value': 0x9400d3, 'colorMessageId': 'color_dark_violet' },
    { 'value': 0x7fffd4, 'colorMessageId': 'color_aquamarine' },
    { 'value': 0xffefd5, 'colorMessageId': 'color_papaya_whip' },
    { 'value': 0xda70d6, 'colorMessageId': 'color_orchid' },
    { 'value': 0xfaebd7, 'colorMessageId': 'color_antique_white' },
    { 'value': 0xd8bfd8, 'colorMessageId': 'color_thistle' },
    { 'value': 0x9370db, 'colorMessageId': 'color_medium_purple' },
    { 'value': 0xdcdcdc, 'colorMessageId': 'color_gainsboro' },
    { 'value': 0xdda0dd, 'colorMessageId': 'color_plum' },
    { 'value': 0xb0c4de, 'colorMessageId': 'color_light_steel_blue' },
    { 'value': 0x8b0000, 'colorMessageId': 'color_dark_red' },
    { 'value': 0xfff5ee, 'colorMessageId': 'color_sea_shell' },
    { 'value': 0x4169e1, 'colorMessageId': 'color_royal_blue' },
    { 'value': 0x8a2be2, 'colorMessageId': 'color_blue_violet' },
    { 'value': 0x7cfc00, 'colorMessageId': 'color_lawn_green' },
    { 'value': 0xe0ffff, 'colorMessageId': 'color_light_cyan' },
    { 'value': 0xb22222, 'colorMessageId': 'color_fire_brick' },
    { 'value': 0x87ceeb, 'colorMessageId': 'color_sky_blue' },
    { 'value': 0x6495ed, 'colorMessageId': 'color_cornflower_blue' },
    { 'value': 0x7b68ee, 'colorMessageId': 'color_medium_slate_blue' },
    { 'value': 0xff, 'colorMessageId': 'color_blue' },
    { 'value': 0xf0fff0, 'colorMessageId': 'color_honeydew' },
    { 'value': 0xba55d3, 'colorMessageId': 'color_medium_orchid' },
    { 'value': 0xf5f5f5, 'colorMessageId': 'color_white_smoke' },
    { 'value': 0xffff00, 'colorMessageId': 'color_yellow' },
    { 'value': 0x87cefa, 'colorMessageId': 'color_light_sky_blue' },
    { 'value': 0xbfff, 'colorMessageId': 'color_deep_sky_blue' },
    { 'value': 0xfffafa, 'colorMessageId': 'color_snow' },
    { 'value': 0x66cdaa, 'colorMessageId': 'color_medium_aqua_marine' },
    { 'value': 0x7fff00, 'colorMessageId': 'color_chartreuse' },
    { 'value': 0x1e90ff, 'colorMessageId': 'color_dodger_blue' },
];
TestImportManager.exportForTesting(Color);

// Copyright 2020 The Chromium Authors
// Use of this source code is governed by a BSD-style license that can be
// found in the LICENSE file.
/**
 * @fileoverview Handles automation intents for speech feedback.
 * Braille is *not* handled in this module.
 */
const IntentCommandType$2 = chrome.automation.IntentCommandType;
const IntentTextBoundaryType$1 = chrome.automation.IntentTextBoundaryType;
const RoleType$8 = chrome.automation.RoleType;
/** A stateless class that turns intents into speech. */
class IntentHandler {
    /**
     * Called when intents are received from an AutomationEvent.
     * @return Whether intents are handled.
     */
    static onIntents(intents, cur, prev) {
        if (intents.length === 0) {
            return false;
        }
        // Currently, discard all other intents once one is handled.
        for (let i = 0; i < intents.length; i++) {
            if (IntentHandler.onIntent(intents[i], cur, prev)) {
                return true;
            }
        }
        return false;
    }
    /**
     * Called when an intent is received.
     * @return Whether the intent was handled.
     */
    static onIntent(intent, cur, prev) {
        switch (intent.command) {
            case IntentCommandType$2.MOVE_SELECTION:
                return IntentHandler.onMoveSelection(intent, cur, prev);
            // TODO: implement support.
            case IntentCommandType$2.CLEAR_SELECTION:
            case IntentCommandType$2.DELETE:
            case IntentCommandType$2.DICTATE:
            case IntentCommandType$2.EXTEND_SELECTION:
            case IntentCommandType$2.FORMAT:
            case IntentCommandType$2.HISTORY:
            case IntentCommandType$2.INSERT:
            case IntentCommandType$2.MARKER:
            case IntentCommandType$2.SET_SELECTION:
                break;
        }
        return false;
    }
    /**
     * Called when the text selection moves.
     * @return Whether the intent was handled.
     */
    static onMoveSelection(intent, cur, prev) {
        switch (intent.textBoundary) {
            case IntentTextBoundaryType$1.CHARACTER:
                return IntentHandler.onCharacterMoveSelection_(intent, cur, prev);
            case IntentTextBoundaryType$1.LINE_END:
            case IntentTextBoundaryType$1.LINE_START:
            case IntentTextBoundaryType$1.LINE_START_OR_END:
                // TODO(b/314203187): Not null asserted, check that this is correct.
                cur.speakLine(prev);
                return true;
            case IntentTextBoundaryType$1.PARAGRAPH_START:
                return IntentHandler.onParagraphStartMoveSelection_(intent, cur, prev);
            case IntentTextBoundaryType$1.WORD_END:
            case IntentTextBoundaryType$1.WORD_START:
                return IntentHandler.onWordMoveSelection_(intent, cur, prev);
            // TODO: implement support.
            case IntentTextBoundaryType$1.FORMAT_END:
            case IntentTextBoundaryType$1.FORMAT_START:
            case IntentTextBoundaryType$1.FORMAT_START_OR_END:
            case IntentTextBoundaryType$1.OBJECT:
            case IntentTextBoundaryType$1.PAGE_END:
            case IntentTextBoundaryType$1.PAGE_START:
            case IntentTextBoundaryType$1.PAGE_START_OR_END:
            case IntentTextBoundaryType$1.PARAGRAPH_END:
            case IntentTextBoundaryType$1.PARAGRAPH_START_OR_END:
            case IntentTextBoundaryType$1.SENTENCE_END:
            case IntentTextBoundaryType$1.SENTENCE_START:
            case IntentTextBoundaryType$1.SENTENCE_START_OR_END:
            case IntentTextBoundaryType$1.WEB_PAGE:
            case IntentTextBoundaryType$1.WORD_START_OR_END:
                break;
        }
        return false;
    }
    /**
     * Called when the text selection moves with a boundary type of CHARACTER.
     * @return Whether the intent was handled.
     */
    static onCharacterMoveSelection_(_intent, cur, prev) {
        // Read character to the right of the cursor by building a character
        // range.
        let prevRange = undefined;
        if (prev) {
            prevRange = prev.createCharRange();
        }
        const newRange = cur.createCharRange();
        // Use the Output module for feedback so that we get contextual
        // information e.g. if we've entered a suggestion, insertion, or
        // deletion.
        const output = new Output();
        const text = cur.text;
        if (text.substring(cur.startOffset, cur.startOffset + 1).length === 0) {
            // There isn't any text to the right of the cursor.
            if (prev) {
                // Detect cases where |cur| is immediately before an abstractSpan.
                const enteredAncestors = AutomationUtil.getUniqueAncestors(prev.end.node, cur.end.node);
                const exitedAncestors = AutomationUtil.getUniqueAncestors(cur.end.node, prev.end.node);
                // Scan up only to a root or the editable root.
                let ancestor;
                const ancestors = enteredAncestors.concat(exitedAncestors);
                while ((ancestor = ancestors.pop()) &&
                    !AutomationPredicate.rootOrEditableRoot(ancestor)) {
                    // TODO(b/314203187): Not null asserted, check that this is correct.
                    const roleInfo = OutputRoleInfo[ancestor.role];
                    if (roleInfo && roleInfo['inherits'] === 'abstractSpan') {
                        // Let the caller handle this case.
                        return false;
                    }
                }
            }
            // This block special cases readout of the cursor when it reaches the
            // end of a line.
            if (text === '\u00a0') {
                output.withString('\u00a0');
            }
            else {
                // It is assumed to be a new line otherwise.
                output.withString('\n');
            }
        }
        output.withRichSpeech(newRange, prevRange, OutputCustomEvent.NAVIGATE).go();
        // Handled.
        return true;
    }
    /**
     * Called when the text selection moves with a boundary type of
     * PARAGRAPH_START.
     * @return Whether the intent was handled.
     */
    static onParagraphStartMoveSelection_(_intent, cur, _prev) {
        let node = cur.startContainer;
        // TODO(b/314203187): Not null asserted, check that this is correct.
        if (node.role === RoleType$8.LINE_BREAK) {
            return false;
        }
        while (node && AutomationPredicate.text(node)) {
            node = node.parent;
        }
        if (!node || node.role === RoleType$8.TEXT_FIELD) {
            return false;
        }
        new Output()
            .withRichSpeech(CursorRange.fromNode(node), undefined, OutputCustomEvent.NAVIGATE)
            .go();
        return true;
    }
    /**
     * Called when the text selection moves with a boundary type of WORD_START or
     * WORD_END.
     * @return Whether the intent was handled.
     */
    static onWordMoveSelection_(intent, cur, prev) {
        let prevRange = undefined;
        if (prev) {
            prevRange = prev.createWordRange(false);
        }
        const newRange = cur.createWordRange(intent.textBoundary === IntentTextBoundaryType$1.WORD_END);
        new Output()
            .withSpeech(newRange, prevRange, OutputCustomEvent.NAVIGATE)
            .go();
        return true;
    }
}
TestImportManager.exportForTesting(IntentHandler);

// Copyright 2023 The Chromium Authors
// Use of this source code is governed by a BSD-style license that can be
// found in the LICENSE file.
/**
 * @fileoverview Handle processing for richly editable text.
 */
var Dir$4 = constants.Dir;
var FormType = LibLouis.FormType;
var MarkerType$1 = chrome.automation.MarkerType;
var RoleType$7 = chrome.automation.RoleType;
var StateType$9 = chrome.automation.StateType;
/**
 * A |ChromeVoxEditableTextBase| that implements text editing feedback
 * for automation tree text fields using anchor and focus selection.
 */
class RichEditableText extends AutomationEditableText {
    startLine_;
    endLine_;
    line_;
    misspelled = false;
    grammarError = false;
    bold_ = false;
    italic_ = false;
    underline_ = false;
    lineThrough_ = false;
    fontFamily_;
    fontSize_;
    fontColor_;
    linked_;
    subscript_;
    superscript_;
    constructor(node) {
        super(node);
        const root = this.node_.root;
        if (!root || !root.selectionStartObject || !root.selectionEndObject ||
            root.selectionStartOffset === undefined ||
            root.selectionEndOffset === undefined) {
            return;
        }
        this.startLine_ = new EditableLine(root.selectionStartObject, root.selectionStartOffset, root.selectionStartObject, root.selectionStartOffset);
        this.endLine_ = new EditableLine(root.selectionEndObject, root.selectionEndOffset, root.selectionEndObject, root.selectionEndOffset);
        this.line_ = new EditableLine(root.selectionStartObject, root.selectionStartOffset, root.selectionEndObject, root.selectionEndOffset);
        this.updateIntraLineState_(this.line_);
    }
    isSelectionOnFirstLine() {
        let deep = this.line_.end.node;
        while (deep.previousOnLine) {
            deep = deep.previousOnLine;
        }
        const next = AutomationUtil.findNextNode(deep, Dir$4.BACKWARD, AutomationPredicate.inlineTextBox, { root: r => r === this.node_ });
        if (!next) {
            return true;
        }
        const exited = AutomationUtil.getUniqueAncestors(next, deep);
        return exited.includes(this.node_);
    }
    isSelectionOnLastLine() {
        let deep = this.line_.end.node;
        while (deep.nextOnLine) {
            deep = deep.nextOnLine;
        }
        const next = AutomationUtil.findNextNode(deep, Dir$4.FORWARD, AutomationPredicate.inlineTextBox, { root: r => r === this.node_ });
        if (!next) {
            return true;
        }
        const exited = AutomationUtil.getUniqueAncestors(next, deep);
        return exited.includes(this.node_);
    }
    onUpdate(intents) {
        // TODO(b/314203187): Not null asserted, check that this is correct.
        const root = this.node_.root;
        if (!root.selectionStartObject || !root.selectionEndObject ||
            root.selectionStartOffset === undefined ||
            root.selectionEndOffset === undefined) {
            return;
        }
        const startLine = new EditableLine(root.selectionStartObject, root.selectionStartOffset, root.selectionStartObject, root.selectionStartOffset);
        const endLine = new EditableLine(root.selectionEndObject, root.selectionEndOffset, root.selectionEndObject, root.selectionEndOffset);
        const prevStartLine = this.startLine_;
        const prevEndLine = this.endLine_;
        this.startLine_ = startLine;
        this.endLine_ = endLine;
        const baseLineOnStart = prevEndLine.isSameLineAndSelection(endLine);
        const isSameSelection = baseLineOnStart && prevStartLine.isSameLineAndSelection(startLine);
        const cur = new EditableLine(root.selectionStartObject, root.selectionStartOffset, root.selectionEndObject, root.selectionEndOffset, baseLineOnStart);
        if (isSameSelection && this.line_ && this.line_.text === cur.text) {
            // Nothing changed, return.
            return;
        }
        const prev = this.line_;
        this.line_ = cur;
        this.handleSpeech_(cur, prev, startLine, endLine, prevStartLine, prevEndLine, baseLineOnStart, intents);
        this.handleBraille_(baseLineOnStart);
    }
    handleSpeech_(cur, prev, startLine, endLine, prevStartLine, prevEndLine, baseLineOnStart, intents) {
        // During continuous read, skip speech (which gets handled in
        // CommandHandler). We use the speech end callback to trigger additional
        // speech.
        // Also, skip speech based on the predicate.
        // TODO(b/314203187): Not null asserted, check that this is correct.
        if (ChromeVoxState.instance.isReadingContinuously ||
            AutomationPredicate.shouldOnlyOutputSelectionChangeInBraille(this.node_)) {
            this.updateIntraLineState_(cur);
            return;
        }
        // End of document announcements are special because it's the only situation
        // in which there's no more content to the right of the linecursor. This
        // condition has to detect a precise state change where a user moves (not
        // changes) within the last line.
        if (this.isSelectionOnLastLine() && cur.hasCollapsedSelection() &&
            cur.text.length === cur.endOffset && prev.isSameLine(cur) &&
            cur.text === prev.text) {
            // Omit announcements if the document is completely empty.
            if (!this.isSelectionOnFirstLine() || cur.text.length > 0) {
                ChromeVox.tts.speak(Msgs.getMsg('end_of_text_verbose'), QueueMode.CATEGORY_FLUSH);
            }
            this.updateIntraLineState_(cur);
            return;
        }
        // Before entering into our state machine below, use selected intents to
        // decipher ambiguous cases.
        if (this.maybeSpeakUsingIntents_(intents, cur, prev)) {
            return;
        }
        // We must validate the previous lines below as state changes in the
        // accessibility tree may have invalidated the lines.
        // Selection stayed within the same line(s) and didn't cross into new lines.
        // Handle speech output for collapsed selections and all selections on text
        // areas using EditableTextBase.
        // TODO(accessibility): eventually remove usage of the EditableTextBase
        // plain text state machine by migrating all cases to be handled by
        // EditableLine.
        if ((cur.hasCollapsedSelection() || this.node_.htmlTag === 'textarea') &&
            startLine.isSameLine(prevStartLine) &&
            endLine.isSameLine(prevEndLine)) {
            // Intra-line changes.
            if (cur.hasTextSelection()) {
                if (!prev.hasTextSelection() && cur.hasCollapsedSelection() &&
                    cur.startOffset > prev.startOffset) {
                    // EditableTextBase cannot handle this state transition (moving
                    // forward from rich text to a caret in plain text). Fall back to
                    // simply reading the character to the right of the caret. We achieve
                    // this by updating the indices first, then sending the new change.
                    // These members come from EditableTextBase.
                    this.start = cur.endOffset > 0 ? cur.endOffset - 1 : 0;
                    this.end = this.start;
                }
                // Delegate to EditableTextBase (via |changed|), which handles plain
                // text state output.
                let text = cur.text;
                if (text === '\n') {
                    text = '';
                }
                this.changed(new TextChangeEvent(text, cur.startOffset, cur.endOffset, true));
            }
            else {
                // Handle description of non-textual lines.
                new Output()
                    .withRichSpeech(new CursorRange(cur.start, cur.end), new CursorRange(prev.start, prev.end), OutputCustomEvent.NAVIGATE)
                    .go();
            }
            // Be careful to update state in EditableTextBase since we don't
            // explicitly call through to it here.
            this.updateIntraLineState_(cur);
            this.speakAllMarkers_(cur);
            return;
        }
        const curBase = baseLineOnStart ? endLine : startLine;
        // TODO(b/314203187): Not null asserted, check that this is correct.
        if ((cur.startContainer.role === RoleType$7.TEXT_FIELD ||
            (cur.startContainer === prev.startContainer &&
                cur.endContainer === prev.endContainer)) &&
            cur.startContainerValue !== prev.startContainerValue) {
            // This block catches text changes between |prev| and | cur|. Note that
            // we can end up here if |prevStartLine| or |prevEndLine| were invalid
            // above for intra-line changes. This block therefore catches all text
            // changes including those that occur within a single line and up to those
            // that occur within a static text. It also catches text changes that
            // result in an empty text field, so we handle the case where the
            // container is the text field itself.
            // Take the difference of the text at the paragraph level (i.e. the value
            // of the container) and speak that.
            this.describeTextChanged(new TextChangeEvent(prev.startContainerValue, prev.localContainerStartOffset, prev.localContainerEndOffset, true), new TextChangeEvent(cur.startContainerValue, cur.localContainerStartOffset, cur.localContainerEndOffset, true));
        }
        else if (cur.text === '') {
            // This line has no text content. Describe the DOM selection.
            new Output()
                .withRichSpeech(new CursorRange(cur.start, cur.end), new CursorRange(prev.start, prev.end), OutputCustomEvent.NAVIGATE)
                .go();
        }
        else if (!prev.hasCollapsedSelection() && !cur.hasCollapsedSelection() &&
            (curBase.isSameLineAndSelection(prevStartLine) ||
                curBase.isSameLineAndSelection(prevEndLine))) {
            // This is a selection that gets extended from the same anchor.
            // Speech requires many more states than braille.
            const curExtent = baseLineOnStart ? startLine : endLine;
            let suffixMsg = '';
            if (curBase.isBeforeLine(curExtent)) {
                // Forward selection.
                if (prev.isBeforeLine(curBase) && !prev.start.equals(curBase.start)) {
                    // Wrapped across the baseline. Read out the new selection.
                    suffixMsg = 'selected';
                    this.speakTextSelection_(curBase.startContainer, curBase.localStartOffset, curExtent.endContainer, curExtent.localEndOffset);
                }
                else {
                    if (prev.isBeforeLine(curExtent)) {
                        // Grew.
                        suffixMsg = 'selected';
                        this.speakTextSelection_(prev.endContainer, prev.localEndOffset, curExtent.endContainer, curExtent.localEndOffset);
                    }
                    else {
                        // Shrank.
                        suffixMsg = 'unselected';
                        this.speakTextSelection_(curExtent.endContainer, curExtent.localEndOffset, prev.endContainer, prev.localEndOffset);
                    }
                }
            }
            else {
                // Backward selection.
                if (curBase.isBeforeLine(prev)) {
                    // Wrapped across the baseline. Read out the new selection.
                    suffixMsg = 'selected';
                    this.speakTextSelection_(curExtent.startContainer, curExtent.localStartOffset, curBase.endContainer, curBase.localEndOffset);
                }
                else {
                    if (curExtent.isBeforeLine(prev)) {
                        // Grew.
                        suffixMsg = 'selected';
                        this.speakTextSelection_(curExtent.startContainer, curExtent.localStartOffset, prev.startContainer, prev.localStartOffset);
                    }
                    else {
                        // Shrank.
                        suffixMsg = 'unselected';
                        this.speakTextSelection_(prev.startContainer, prev.localStartOffset, curExtent.startContainer, curExtent.localStartOffset);
                    }
                }
            }
            ChromeVox.tts.speak(Msgs.getMsg(suffixMsg), QueueMode.QUEUE);
        }
        else if (!cur.hasCollapsedSelection()) {
            // Without any other information, try describing the selection. This state
            // catches things like select all.
            this.speakTextSelection_(cur.startContainer, cur.localStartOffset, cur.endContainer, cur.localEndOffset);
            ChromeVox.tts.speak(Msgs.getMsg('selected'), QueueMode.QUEUE);
        }
        else {
            // A catch-all for any other transitions.
            // Describe the current line. This accounts for previous/current
            // selections and picking the line edge boundary that changed (as computed
            // above). This is also the code path for describing paste. It also covers
            // jump commands which are non-overlapping selections from prev to cur.
            this.line_.speakLine(prev);
        }
        this.updateIntraLineState_(cur);
    }
    /**
     * @param baseLineOnStart When true, the brailled line will show
     *     ancestry context based on the start of the selection. When false, it
     *     will use the end of the selection.
     */
    handleBraille_(baseLineOnStart) {
        const isEmpty = !this.node_.find({ role: RoleType$7.STATIC_TEXT });
        const isFirstLine = this.isSelectionOnFirstLine();
        const cur = this.line_;
        if (cur.value === null) {
            return;
        }
        let value = new MultiSpannable(isEmpty ? '' : cur.value);
        if (!this.node_.constructor) {
            return;
        }
        value.getSpansInstanceOf(this.node_.constructor).forEach(spanObj => {
            const span = spanObj;
            const styleObj = span.role === RoleType$7.INLINE_TEXT_BOX ? span.parent : span;
            if (!styleObj) {
                return;
            }
            const style = styleObj;
            let formType = FormType.PLAIN_TEXT;
            // Currently no support for sub/superscript in 3rd party liblouis library.
            if (style.bold) {
                formType |= FormType.BOLD;
            }
            if (style.italic) {
                formType |= FormType.ITALIC;
            }
            if (style.underline) {
                formType |= FormType.UNDERLINE;
            }
            if (formType === FormType.PLAIN_TEXT) {
                return;
            }
            const start = value.getSpanStart(span);
            const end = value.getSpanEnd(span);
            value.setSpan(new BrailleTextStyleSpan(formType), start, end);
        });
        value.setSpan(new ValueSpan(0), 0, value.length);
        // Provide context for the current selection.
        const context = baseLineOnStart ? cur.startContainer : cur.endContainer;
        if (context && context.role !== RoleType$7.TEXT_FIELD) {
            const output = new Output().suppress('name').withBraille(CursorRange.fromNode(context), CursorRange.fromNode(this.node_), OutputCustomEvent.NAVIGATE);
            if (output.braille.length) {
                const end = cur.containerEndOffset + 1;
                const prefix = value.substring(0, end);
                const suffix = value.substring(end, value.length);
                value = prefix;
                value.append(SPACE);
                value.append(output.braille);
                if (suffix.length) {
                    if (suffix.toString()[0] !== SPACE) {
                        value.append(SPACE);
                    }
                    value.append(suffix);
                }
            }
        }
        let start = cur.startOffset;
        let end = cur.endOffset;
        if (isFirstLine) {
            if (!/\s/.test(value.toString()[value.length - 1])) {
                value.append(SPACE);
            }
            if (isEmpty) {
                // When the text field is empty, place the selection cursor immediately
                // after the space and before the 'ed' role msg indicator below.
                start = value.length - 1;
                end = start;
            }
            value.append(Msgs.getMsg('tag_textarea_brl'));
        }
        value.setSpan(new ValueSelectionSpan(), start, end);
        ChromeVox.braille.write(new NavBraille({ text: value, startIndex: start, endIndex: end }));
    }
    speakTextSelection_(startNode, startOffset, endNode, endOffset) {
        if (!startNode || !endNode) {
            return;
        }
        const selectedRange = new CursorRange(new Cursor(startNode, startOffset), new Cursor(endNode, endOffset));
        new Output()
            .withRichSpeech(selectedRange, CursorRange.fromNode(this.node_), OutputCustomEvent.NAVIGATE)
            .go();
    }
    speakTextMarker_(container, selStart, selEnd) {
        const markersWithinSelection = {};
        const markers = container.markers;
        if (markers) {
            for (const marker of markers) {
                // See if our selection intersects with this marker.
                if (marker.startOffset >= selStart || selEnd < marker.endOffset) {
                    for (const key in marker.flags) {
                        markersWithinSelection[key] = true;
                    }
                }
            }
        }
        const msgs = [];
        if (this.misspelled === !(markersWithinSelection[MarkerType$1.SPELLING])) {
            this.misspelled = !this.misspelled;
            msgs.push(this.misspelled ? 'misspelling_start' : 'misspelling_end');
        }
        if (this.grammarError === !(markersWithinSelection[MarkerType$1.GRAMMAR])) {
            this.grammarError = !this.grammarError;
            msgs.push(this.grammarError ? 'grammar_start' : 'grammar_end');
        }
        if (msgs.length) {
            msgs.forEach(msg => ChromeVox.tts.speak(Msgs.getMsg(msg), QueueMode.QUEUE, Personality.ANNOTATION));
        }
    }
    speakTextStyle_(style) {
        const msgs = [];
        const fontSize = style.fontSize;
        const fontColor = Color.getColorDescription(style.color);
        const linked = style.state[StateType$9.LINKED];
        const subscript = style.state['subscript'];
        const superscript = style.state['superscript'];
        const bold = style.bold;
        const italic = style.italic;
        const underline = style.underline;
        const lineThrough = style.lineThrough;
        const fontFamily = style.fontFamily;
        // Only report text style attributes if they change.
        if (fontSize && (fontSize !== this.fontSize_)) {
            this.fontSize_ = fontSize;
            msgs.push({ msg: 'font_size', opt_subs: [String(this.fontSize_)] });
        }
        if (fontColor && (fontColor !== this.fontColor_)) {
            this.fontColor_ = fontColor;
            msgs.push({ msg: 'font_color', opt_subs: [this.fontColor_] });
        }
        if (linked !== this.linked_) {
            this.linked_ = linked;
            msgs.push(this.linked_ ? { msg: 'link' } : { msg: 'not_link' });
        }
        if (style.subscript !== this.subscript_) {
            this.subscript_ = subscript;
            msgs.push(this.subscript_ ? { msg: 'subscript' } : { msg: 'not_subscript' });
        }
        if (style.superscript !== this.superscript_) {
            this.superscript_ = superscript;
            msgs.push(this.superscript_ ? { msg: 'superscript' } : { msg: 'not_superscript' });
        }
        if (bold !== this.bold_) {
            this.bold_ = bold;
            msgs.push(this.bold_ ? { msg: 'bold' } : { msg: 'not_bold' });
        }
        if (italic !== this.italic_) {
            this.italic_ = italic;
            msgs.push(this.italic_ ? { msg: 'italic' } : { msg: 'not_italic' });
        }
        if (underline !== this.underline_) {
            this.underline_ = underline;
            msgs.push(this.underline_ ? { msg: 'underline' } : { msg: 'not_underline' });
        }
        if (lineThrough !== this.lineThrough_) {
            this.lineThrough_ = lineThrough;
            msgs.push(this.lineThrough_ ? { msg: 'linethrough' } : { msg: 'not_linethrough' });
        }
        if (fontFamily && (fontFamily !== this.fontFamily_)) {
            this.fontFamily_ = fontFamily;
            msgs.push({ msg: 'font_family', opt_subs: [this.fontFamily_] });
        }
        if (msgs.length) {
            msgs.forEach(msgObject => ChromeVox.tts.speak(Msgs.getMsg(msgObject.msg, msgObject.opt_subs), QueueMode.QUEUE, Personality.ANNOTATION));
        }
    }
    describeSelectionChanged(evt) {
        // Note that since Chrome allows for selection to be placed immediately at
        // the end of a line (i.e. end === value.length) and since we try to describe
        // the character to the right, just describe it as a new line.
        if ((this.start + 1) === evt.start && evt.start === this.value.length) {
            this.speak('\n', evt.triggeredByUser);
            return;
        }
        AutomationEditableText.prototype.describeSelectionChanged.call(this, evt);
    }
    getLineIndex(_charIndex) {
        return 0;
    }
    getLineStart(_lineIndex) {
        return 0;
    }
    getLineEnd(_lineIndex) {
        return this.value.length;
    }
    changed(evt) {
        // This path does not use the Output module to synthesize speech.
        Output.forceModeForNextSpeechUtterance(undefined);
        AutomationEditableText.prototype.changed.call(this, evt);
    }
    updateIntraLineState_(cur) {
        let text = cur.text;
        if (text === '\n') {
            text = '';
        }
        this.value = text;
        this.start = cur.startOffset;
        this.end = cur.endOffset;
    }
    maybeSpeakUsingIntents_(intents, cur, prev) {
        if (IntentHandler.onIntents(intents, cur, prev)) {
            this.updateIntraLineState_(cur);
            this.speakAllMarkers_(cur);
            return true;
        }
        return false;
    }
    speakAllMarkers_(cur) {
        const container = cur.startContainer;
        if (!container) {
            return;
        }
        this.speakTextMarker_(container, cur.localStartOffset, cur.localEndOffset);
        if (SettingsManager.get('announceRichTextAttributes')) {
            this.speakTextStyle_(container);
        }
    }
}

// Copyright 2023 The Chromium Authors
// Use of this source code is governed by a BSD-style license that can be
// found in the LICENSE file.
var Dir$3 = constants.Dir;
var IntentCommandType$1 = chrome.automation.IntentCommandType;
var RoleType$6 = chrome.automation.RoleType;
var StateType$8 = chrome.automation.StateType;
/**
 * A handler for automation events in a focused text field or editable root
 * such as a |contenteditable| subtree.
 */
class TextEditHandler {
    node;
    editableText_;
    inferredIntents_ = [];
    constructor(node) {
        this.node = node;
        // TODO(b/314203187): Not null asserted, check to make sure it's correct.
        if (!node.state[StateType$8.EDITABLE]) {
            throw new Error('|node| must be editable.');
        }
        this.editableText_ = this.createEditableText_();
    }
    /**
     * ChromeVox handles two general groups of text fields:
     * A rich text field is one where selection gets placed on a DOM
     * descendant to a root text field. This is one of:
     * - content editables (detected via editable state and contenteditable
     * html attribute, or just richly editable state)
     * - text areas (<textarea>) detected via its html tag
     *
     * A non-rich text field is one where accessibility only provides a value,
     * and a pair of numbers for the selection start and end. ChromeVox places
     * single-lined text fields, including those from web content, and ARC++
     * in this group. In addition, multiline ARC++ text fields are treated
     * this way.
     *
     * Note that these definitions slightly differ from those in Blink, which
     * only considers text fields in web content.
     */
    useRichText_() {
        // TODO(b/314203187): Not null asserted, check to make sure it's correct.
        return this.node.state[StateType$8.RICHLY_EDITABLE] ||
            this.node.nonAtomicTextFieldRoot;
    }
    createEditableText_() {
        const isTextArea = this.node.htmlTag === 'textarea';
        const useRichText = this.useRichText_() || isTextArea;
        // Prior to creating the specific editable text handler, ensure that text
        // areas exclude offscreen elements in line computations. This is because
        // text areas from Blink expose a single large static text node which can
        // have thousands or more inline text boxes. This is a very specific check
        // because ignoring offscreen nodes can impact the way in which we convert
        // from a tree position to a deep equivalent on the inline text boxes.
        const firstStaticText = this.node.find({ role: RoleType$6.STATIC_TEXT });
        EditableLine.includeOffscreen = !isTextArea || !firstStaticText ||
            firstStaticText.children.length < MAX_INLINE_TEXT_BOXES;
        return useRichText ? new RichEditableText(this.node) :
            new AutomationEditableText(this.node);
    }
    /**
     * Receives the following kinds of events when the node provided to the
     * constructor is focused: |focus|, |textChanged|, |textSelectionChanged| and
     * |valueInTextFieldChanged|.
     * An implementation of this method should emit the appropriate braille and
     * spoken feedback for the event.
     */
    onEvent(evt) {
        // TODO(b/314203187): Not null asserted, check to make sure it's correct.
        if (!evt.target.state['focused'] || evt.target !== this.node) {
            return;
        }
        let intents = evt.intents;
        // Check for inferred intents applied by other modules e.g. CommandHandler.
        // Be strict about what's allowed and limit only to overriding set
        // selections.
        if (this.inferredIntents_.length > 0 &&
            (intents.length === 0 || intents.some(isSetOrClear))) {
            intents = this.inferredIntents_;
        }
        this.inferredIntents_ = [];
        this.editableText_.onUpdate(intents);
    }
    /** Returns true if selection starts at the first line. */
    isSelectionOnFirstLine() {
        return this.editableText_.isSelectionOnFirstLine();
    }
    /** Returns true if selection ends at the last line. */
    isSelectionOnLastLine() {
        return this.editableText_.isSelectionOnLastLine();
    }
    /** Moves range to after this text field. */
    moveToAfterEditText() {
        const after = AutomationUtil.findNextNode(this.node, Dir$3.FORWARD, AutomationPredicate.object, { skipInitialSubtree: true });
        ChromeVoxRange.navigateTo(CursorRange.fromNode(after ?? this.node));
    }
    /**
     * Injects intents into the stream of editing events. In particular, |intents|
     * will be applied to the next processed edfiting event.
     */
    injectInferredIntents(intents) {
        this.inferredIntents_ = intents;
    }
    /**
     * @param node The root editable node, i.e. the root of a
     *     contenteditable subtree or a text field.
     */
    static createForNode(node) {
        // TODO(b/314203187): Not null asserted, check to make sure it's correct.
        if (!node.state['editable']) {
            throw new Error('Expected editable node.');
        }
        return new TextEditHandler(node);
    }
}
// Local to module.
const MAX_INLINE_TEXT_BOXES = 500;
function isSetOrClear(intent) {
    return intent.command === IntentCommandType$1.SET_SELECTION ||
        intent.command === IntentCommandType$1.CLEAR_SELECTION;
}
TestImportManager.exportForTesting(TextEditHandler);

// Copyright 2015 The Chromium Authors
// Use of this source code is governed by a BSD-style license that can be
// found in the LICENSE file.
/**
 * @fileoverview Implements support for live regions in ChromeVox.
 */
const EventType$7 = chrome.automation.EventType;
const RoleType$5 = chrome.automation.RoleType;
const StateType$7 = chrome.automation.StateType;
const TreeChangeObserverFilter$1 = chrome.automation.TreeChangeObserverFilter;
const TreeChangeType = chrome.automation.TreeChangeType;
/**
 * Handles events and announcements associated with "live regions", which are
 * regions marked as likely to change and important to announce.
 */
class LiveRegions {
    lastDesktopLiveRegionChangedTime_ = new Date(0);
    lastDesktopLiveRegionChangedText_ = '';
    /**
     * Set of nodes that have been announced as part of a live region since
     * |this.lastLiveRegionTime_|, to prevent duplicate announcements.
     */
    liveRegionNodeSet_ = new WeakSet();
    /**
     * The time the last live region event was output.
     */
    lastLiveRegionTime_ = new Date(0);
    /**
     * A list of nodes that have changed as part of one atomic tree update.
     */
    changedNodes_ = [];
    static instance;
    /**
     * Live region events received in fewer than this many milliseconds will
     * queue, otherwise they'll be output with a category flush.
     */
    static LIVE_REGION_QUEUE_TIME_MS = 5000;
    /**
     * Live region events received on the same node in fewer than this many
     * milliseconds will be dropped to avoid a stream of constant chatter.
     */
    static LIVE_REGION_MIN_SAME_NODE_MS = 20;
    /**
     * Whether live regions from background tabs should be announced or not.
     */
    static announceLiveRegionsFromBackgroundTabs_ = false;
    constructor() {
        chrome.automation.addTreeChangeObserver(TreeChangeObserverFilter$1.LIVE_REGION_TREE_CHANGES, (treeChange) => this.onTreeChange(treeChange));
    }
    static init() {
        if (LiveRegions.instance) {
            throw 'Error: Trying to create two instances of singleton LiveRegions';
        }
        LiveRegions.instance = new LiveRegions();
    }
    static announceDesktopLiveRegionChanged(area) {
        const desktopOrApplication = AutomationPredicate.roles([RoleType$5.DESKTOP, RoleType$5.APPLICATION]);
        if (!area.root || !desktopOrApplication(area.root)) {
            return;
        }
        const output = new Output();
        if (area.containerLiveStatus === 'assertive') {
            output.withQueueMode(QueueMode.CATEGORY_FLUSH);
        }
        else if (area.containerLiveStatus === 'polite') {
            output.withQueueMode(QueueMode.QUEUE);
        }
        else {
            return;
        }
        const withinDelay = ((new Date()).getTime() -
            LiveRegions.instance.lastDesktopLiveRegionChangedTime_.getTime()) <
            DESKTOP_CHANGE_DELAY_MS;
        output
            .withRichSpeechAndBraille(CursorRange.fromNode(area), undefined, EventType$7.LIVE_REGION_CHANGED)
            .withSpeechCategory(TtsCategory.LIVE);
        if (withinDelay &&
            output.toString() ===
                LiveRegions.instance.lastDesktopLiveRegionChangedText_) {
            return;
        }
        LiveRegions.instance.lastDesktopLiveRegionChangedTime_ = new Date();
        LiveRegions.instance.lastDesktopLiveRegionChangedText_ = output.toString();
        output.go();
    }
    /**
     * Called when the automation tree is changed.
     */
    onTreeChange(treeChange) {
        const type = treeChange.type;
        const node = treeChange.target;
        if ((!node.containerLiveStatus || node.containerLiveStatus === 'off') &&
            type !== TreeChangeType.SUBTREE_UPDATE_END) {
            return;
        }
        if (this.shouldIgnoreLiveRegion_(node)) {
            return;
        }
        const relevant = node.containerLiveRelevant || '';
        const additions = relevant.indexOf('additions') >= 0;
        const text = relevant.indexOf('text') >= 0;
        const removals = relevant.indexOf('removals') >= 0;
        const all = relevant.indexOf('all') >= 0;
        if (all ||
            (additions &&
                (type === TreeChangeType.NODE_CREATED ||
                    type === TreeChangeType.SUBTREE_CREATED))) {
            this.queueLiveRegionChange_(node);
        }
        else if (all || (text && type === TreeChangeType.TEXT_CHANGED)) {
            this.queueLiveRegionChange_(node);
        }
        if ((all || removals) && type === TreeChangeType.NODE_REMOVED) {
            this.outputLiveRegionChange_(node, '@live_regions_removed');
        }
        if (type === TreeChangeType.SUBTREE_UPDATE_END) {
            this.processQueuedTreeChanges_();
        }
    }
    queueLiveRegionChange_(node) {
        this.changedNodes_.push(node);
    }
    processQueuedTreeChanges_() {
        // Schedule all live regions after all events in the native C++
        // EventBundle.
        this.liveRegionNodeSet_ = new WeakSet();
        for (let i = 0; i < this.changedNodes_.length; i++) {
            const node = this.changedNodes_[i];
            this.outputLiveRegionChange_(node, undefined);
        }
        this.changedNodes_ = [];
    }
    /**
     * Given a node that needs to be spoken as part of a live region
     * change and an additional optional format string, output the
     * live region description.
     * @param node The changed node.
     * @param opt_prependFormatStr If set, a format string for
     *     Output to prepend to the output.
     * @private
     */
    outputLiveRegionChange_(node, opt_prependFormatStr) {
        if (node.containerLiveBusy) {
            return;
        }
        while (node.containerLiveAtomic && !node.liveAtomic && node.parent) {
            node = node.parent;
        }
        if (this.liveRegionNodeSet_.has(node)) {
            this.lastLiveRegionTime_ = new Date();
            return;
        }
        this.outputLiveRegionChangeForNode_(node, opt_prependFormatStr);
    }
    /**
     * @param node The changed node.
     * @param opt_prependFormatStr If set, a format string for
     *     Output to prepend to the output.
     */
    outputLiveRegionChangeForNode_(node, opt_prependFormatStr) {
        const range = CursorRange.fromNode(node);
        const output = new Output();
        output.withSpeechCategory(TtsCategory.LIVE);
        // Queue live regions coming from background tabs.
        let hostView = AutomationUtil.getTopLevelRoot(node);
        hostView = hostView && hostView.parent ? hostView.parent : null;
        const currentRange = ChromeVoxRange.current;
        const forceQueue = !hostView || !hostView.state['focused'] ||
            (currentRange && currentRange.start.node.root !== node.root) ||
            node.containerLiveStatus === 'polite';
        // Enqueue live region updates that were received at approximately
        // the same time, otherwise flush previous live region updates.
        const queueTime = LiveRegions.LIVE_REGION_QUEUE_TIME_MS;
        const currentTime = new Date();
        const delta = currentTime.getTime() - this.lastLiveRegionTime_.getTime();
        if (delta > queueTime && !forceQueue) {
            output.withQueueMode(QueueMode.CATEGORY_FLUSH);
        }
        else {
            output.withQueueMode(QueueMode.QUEUE);
        }
        if (opt_prependFormatStr) {
            output.format(opt_prependFormatStr);
        }
        output.withSpeech(range, range, OutputCustomEvent.NAVIGATE);
        if (!output.hasSpeech && node.liveAtomic) {
            output.format('$joinedDescendants', node);
        }
        if (!output.hasSpeech) {
            return;
        }
        // We also have to add recursively the children of this live region node
        // since all children could potentially get described and we don't want to
        // describe their tree changes especially during page load within the
        // LiveRegions.LIVE_REGION_MIN_SAME_NODE_MS to prevent excessive chatter.
        this.addNodeToNodeSetRecursive_(node);
        output.go();
        this.lastLiveRegionTime_ = currentTime;
    }
    addNodeToNodeSetRecursive_(root) {
        this.liveRegionNodeSet_.add(root);
        for (let child = root.firstChild; child; child = child.nextSibling) {
            this.addNodeToNodeSetRecursive_(child);
        }
    }
    shouldIgnoreLiveRegion_(node) {
        if (LiveRegions.announceLiveRegionsFromBackgroundTabs_) {
            return false;
        }
        const currentRange = ChromeVoxRange.current;
        if (currentRange && currentRange.start.node.root === node.root) {
            return false;
        }
        let hostView = AutomationUtil.getTopLevelRoot(node);
        hostView = hostView && hostView.parent ? hostView.parent : null;
        if (!hostView) {
            return true;
        }
        if (hostView.role === RoleType$5.WINDOW &&
            !(hostView.state[StateType$7.INVISIBLE])) {
            return false;
        }
        if (hostView.state['focused']) {
            return false;
        }
        return true;
    }
}
/**
 * Time to wait until processing more live region change events on the same
 * text content.
 */
const DESKTOP_CHANGE_DELAY_MS = 100;
TestImportManager.exportForTesting(LiveRegions);

// Copyright 2015 The Chromium Authors
// Use of this source code is governed by a BSD-style license that can be
// found in the LICENSE file.
/**
 * @fileoverview Handles automation from a desktop automation node.
 */
const ActionType$1 = chrome.automation.ActionType;
const Dir$2 = constants.Dir;
const EventType$6 = chrome.automation.EventType;
const IntentCommandType = chrome.automation.IntentCommandType;
const IntentTextBoundaryType = chrome.automation.IntentTextBoundaryType;
const RoleType$4 = chrome.automation.RoleType;
const StateType$6 = chrome.automation.StateType;
class DesktopAutomationHandler extends DesktopAutomationInterface {
    /**
     * Time to wait until processing more value changed events.
     */
    static MIN_VALUE_CHANGE_DELAY_MS = 50;
    /**
     * Time to wait until processing more alert events with the same text content.
     */
    static MIN_ALERT_DELAY_MS = 50;
    /**
     * URLs employing NTP (New tap page) searchbox.
     */
    static NTP_SEARCHBOX_URLS = new Set(['chrome://new-tab-page/', 'chrome-untrusted://lens-overlay/']);
    /** The object that speaks changes to an editable text field. */
    textEditHandler_ = null;
    /** The last time we handled a value changed event. */
    lastValueChanged_ = new Date(0);
    /** The last node that triggered a value changed event. */
    lastValueTarget_ = null;
    /** The last time we handled an alert event. */
    lastAlertTime_ = new Date(0);
    /** The last alert text we processed. */
    lastAlertText_ = '';
    /** The last root URL encountered. */
    lastRootUrl_ = '';
    /** Whether a submenu is currently showing. */
    isSubMenuShowing_ = false;
    /** Whether document selection changes should be ignored. */
    shouldIgnoreDocumentSelectionFromAction_ = false;
    /** The current page number (for pagination tracking). */
    currentPage_ = -1;
    /** The total number of pages (for pagination tracking). */
    totalPages_ = -1;
    constructor(node) {
        super(node);
        this.init_(node);
    }
    async init_(node) {
        this.addListener_(EventType$6.ALERT, (event) => this.onAlert_(event));
        this.addListener_(EventType$6.BLUR, () => this.onBlur_());
        this.addListener_(EventType$6.DOCUMENT_SELECTION_CHANGED, (event) => this.onDocumentSelectionChanged_(event));
        this.addListener_(EventType$6.FOCUS, (event) => this.onFocus_(event));
        // Note that live region changes from views are really announcement
        // events. Their target nodes contain no live region semantics and have no
        // relation to live regions which are supported in |LiveRegions|.
        this.addListener_(EventType$6.LIVE_REGION_CHANGED, (event) => this.onLiveRegionChanged_(event));
        this.addListener_(EventType$6.LOAD_COMPLETE, (event) => this.onLoadComplete_(event));
        this.addListener_(EventType$6.FOCUS_AFTER_MENU_CLOSE, (event) => this.onMenuEnd_(event));
        this.addListener_(EventType$6.MENU_POPUP_START, (event) => this.onMenuPopupStart_(event));
        this.addListener_(EventType$6.MENU_START, (event) => this.onMenuStart_(event));
        this.addListener_(EventType$6.RANGE_VALUE_CHANGED, (event) => this.onValueChanged_(event));
        this.addListener_(EventType$6.SCROLL_POSITION_CHANGED, (event) => this.onScrollPositionChanged(event));
        this.addListener_(EventType$6.SCROLL_HORIZONTAL_POSITION_CHANGED, (event) => this.onScrollPositionChanged(event));
        this.addListener_(EventType$6.SCROLL_VERTICAL_POSITION_CHANGED, (event) => this.onScrollPositionChanged(event));
        // Called when a same-page link is followed or the url fragment changes.
        this.addListener_(EventType$6.SCROLLED_TO_ANCHOR, (event) => this.onScrolledToAnchor(event));
        this.addListener_(EventType$6.SELECTION, (event) => this.onSelection(event));
        this.addListener_(EventType$6.TEXT_SELECTION_CHANGED, (event) => this.onEditableChanged_(event));
        this.addListener_(EventType$6.VALUE_IN_TEXT_FIELD_CHANGED, (event) => this.onEditableChanged_(event));
        this.addListener_(EventType$6.VALUE_CHANGED, (event) => this.onValueChanged_(event));
        this.addListener_(EventType$6.AUTOFILL_AVAILABILITY_CHANGED, this.onAutofillAvailabilityChanged);
        this.addListener_(EventType$6.ORIENTATION_CHANGED, (event) => this.onOrientationChanged(event));
        // Called when a child MenuItem is collapsed.
        this.addListener_(EventType$6.COLLAPSED, (event) => this.onMenuItemCollapsed(event));
        await AutomationObjectConstructorInstaller.init(node);
        const focus = await AsyncUtil.getFocus();
        if (focus) {
            const event = new CustomAutomationEvent(EventType$6.FOCUS, focus, { eventFrom: 'page', eventFromAction: ActionType$1.FOCUS });
            this.onFocus_(event);
        }
    }
    get textEditHandler() {
        return this.textEditHandler_ ?? undefined;
    }
    /**
     * Handles the result of a hit test.
     */
    onHitTestResult(node) {
        // It's possible the |node| hit has lost focus (via its root).
        // TODO(crbug.com/314203187): Not null asserted, check that this is correct.
        const host = node.root.parent;
        if (node.parent && host && host.role === RoleType$4.WEB_VIEW &&
            !host.state[StateType$6.FOCUSED]) {
            return;
        }
        // It is possible that the user moved since we requested a hit test.  Bail
        // if the current range is valid and on the same page as the hit result
        // (but not the root).
        if (ChromeVoxRange.current && ChromeVoxRange.current.start &&
            ChromeVoxRange.current.start.node &&
            ChromeVoxRange.current.start.node.root) {
            const cur = ChromeVoxRange.current.start.node;
            if (cur.role !== RoleType$4.ROOT_WEB_AREA &&
                AutomationUtil.getTopLevelRoot(node) ===
                    AutomationUtil.getTopLevelRoot(cur)) {
                return;
            }
        }
        chrome.automation.getFocus(function (focus) {
            if (!focus && !node) {
                return;
            }
            focus = node || focus;
            const focusedRoot = AutomationUtil.getTopLevelRoot(focus);
            const output = new Output();
            if (focus !== focusedRoot && focusedRoot) {
                output.format('$name', focusedRoot);
            }
            // Even though we usually don't output events from actions, hit test
            // results should generate output.
            const range = CursorRange.fromNode(focus);
            ChromeVoxRange.set(range);
            output
                .withRichSpeechAndBraille(range, undefined, OutputCustomEvent.NAVIGATE)
                .go();
        });
    }
    /**
     * Makes an announcement without changing focus.
     */
    onAlert_(evt) {
        if (CaptionsHandler.instance.maybeHandleAlert(evt)) {
            return;
        }
        const node = evt.target;
        const range = CursorRange.fromNode(node);
        const output = new Output();
        // Whenever chromevox is running together with dictation, we want to
        // announce the hints provided by the Dictation feature in a different voice
        // to differentiate them from regular UI text.
        if (node.className === 'DictationHintView') {
            output.withInitialSpeechProperties(Personality.DICTATION_HINT);
        }
        output.withSpeechCategory(TtsCategory.LIVE)
            .withSpeechAndBraille(range, undefined, evt.type);
        const alertDelayMet = new Date().getTime() - this.lastAlertTime_.getTime() >
            DesktopAutomationHandler.MIN_ALERT_DELAY_MS;
        if (!alertDelayMet && output.toString() === this.lastAlertText_) {
            return;
        }
        this.lastAlertTime_ = new Date();
        this.lastAlertText_ = output.toString();
        // A workaround for alert nodes that contain no actual content.
        if (output.toString()) {
            output.go();
        }
    }
    onBlur_() {
        // Nullify focus if it no longer exists.
        chrome.automation.getFocus(function (focus) {
            if (!focus) {
                ChromeVoxRange.set(null);
            }
        });
    }
    onDocumentSelectionChanged_(evt) {
        let selectionStart = evt.target.selectionStartObject;
        // No selection.
        if (!selectionStart) {
            return;
        }
        // A caller requested this event be ignored.
        if (this.shouldIgnoreDocumentSelectionFromAction_ &&
            evt.eventFrom === 'action') {
            return;
        }
        // Editable selection.
        // TODO(crbug.com/314203187): Not null asserted, check that this is correct.
        if (selectionStart.state[StateType$6.EDITABLE]) {
            selectionStart =
                AutomationUtil.getEditableRoot(selectionStart) ?? selectionStart;
            this.onEditableChanged_(new CustomAutomationEvent(evt.type, selectionStart, {
                eventFrom: evt.eventFrom,
                eventFromAction: evt.eventFromAction,
                intents: evt.intents,
            }));
        }
        // Non-editable selections are handled in |Background|.
    }
    /**
     * Provides all feedback once a focus event fires.
     */
    onFocus_(evt) {
        let node = evt.target;
        const isRootWebArea = node.role === RoleType$4.ROOT_WEB_AREA;
        const isFrame = isRootWebArea && node.parent && node.parent.root &&
            node.parent.root.role === RoleType$4.ROOT_WEB_AREA;
        if (isRootWebArea && !isFrame && evt.eventFrom !== 'action') {
            chrome.automation.getFocus((focus) => this.maybeRecoverFocusAndOutput_(evt, focus));
            return;
        }
        // Invalidate any previous editable text handler state.
        if (!this.createTextEditHandlerIfNeeded_(node, true)) {
            this.textEditHandler_ = null;
        }
        // Discard focus events on embeddedObject and webView.
        const shouldDiscard = AutomationPredicate.roles([RoleType$4.EMBEDDED_OBJECT, RoleType$4.PLUGIN_OBJECT, RoleType$4.WEB_VIEW]);
        if (shouldDiscard(node)) {
            return;
        }
        if (node.role === RoleType$4.UNKNOWN) {
            // Ideally, we'd get something more meaningful than focus on an unknown
            // node, but this does sometimes occur. Sync downward to a more reasonable
            // target.
            node = AutomationUtil.findNodePre(node, Dir$2.FORWARD, AutomationPredicate.object);
            if (!node) {
                return;
            }
        }
        if (!node.root) {
            return;
        }
        if (!AutoScrollHandler.instance.onFocusEventNavigation(node)) {
            return;
        }
        // Update the focused root url, which gets used as part of focus recovery.
        this.lastRootUrl_ = node.root.docUrl ?? '';
        // Consider the case when a user presses tab rapidly. The key events may
        // come in far before the accessibility focus events. We therefore must
        // category flush here or the focus events will all queue up.
        Output.forceModeForNextSpeechUtterance(QueueMode.CATEGORY_FLUSH);
        const event = new CustomAutomationEvent(EventType$6.FOCUS, node, {
            eventFrom: evt.eventFrom,
            eventFromAction: evt.eventFromAction,
            intents: evt.intents,
        });
        this.onEventDefault(event);
        // Refresh the handler, if needed, now that ChromeVox focus is up to date.
        this.createTextEditHandlerIfNeeded_(node);
        // Reset `isSubMenuShowing_` when a focus changes because focus
        // changes should automatically close any menus.
        this.isSubMenuShowing_ = false;
    }
    onLiveRegionChanged_(evt) {
        LiveRegions.announceDesktopLiveRegionChanged(evt.target);
    }
    /**
     * Provides all feedback once a load complete event fires.
     */
    onLoadComplete_(evt) {
        // We are only interested in load completes on valid top level roots.
        const top = AutomationUtil.getTopLevelRoot(evt.target);
        if (!top || top !== evt.target.root || !top.docUrl) {
            return;
        }
        this.lastRootUrl_ = '';
        chrome.automation.getFocus((focus) => {
            // In some situations, ancestor windows get focused before a descendant
            // webView/rootWebArea. In particular, a window that gets opened but no
            // inner focus gets set. We catch this generically by re-targetting focus
            // if focus is the ancestor of the load complete target (below).
            if (!focus) {
                return;
            }
            const focusIsAncestor = AutomationUtil.isDescendantOf(evt.target, focus);
            const focusIsDescendant = AutomationUtil.isDescendantOf(focus, evt.target);
            if (!focusIsAncestor && !focusIsDescendant) {
                return;
            }
            if (focusIsAncestor) {
                focus = evt.target;
            }
            // Create text edit handler, if needed, now in order not to miss initial
            // value change if text field has already been focused when initializing
            // ChromeVox.
            this.createTextEditHandlerIfNeeded_(focus);
            // If auto read is set, skip focus recovery and start reading from the
            // top.
            if (LocalStorage.get('autoRead') &&
                AutomationUtil.getTopLevelRoot(evt.target) === evt.target) {
                ChromeVoxRange.set(CursorRange.fromNode(evt.target));
                ChromeVox.tts.stop();
                CommandHandlerInterface.instance.onCommand(Command.READ_FROM_HERE);
                return;
            }
            this.maybeRecoverFocusAndOutput_(evt, focus);
        });
    }
    /**
     * Sets whether document selections from actions should be ignored.
     */
    ignoreDocumentSelectionFromAction(val) {
        this.shouldIgnoreDocumentSelectionFromAction_ = val;
    }
    onNativeNextOrPreviousCharacter() {
        if (this.textEditHandler) {
            this.textEditHandler.injectInferredIntents([{
                    command: IntentCommandType.MOVE_SELECTION,
                    textBoundary: IntentTextBoundaryType.CHARACTER,
                }]);
        }
    }
    onNativeNextOrPreviousWord(isNext) {
        if (this.textEditHandler) {
            this.textEditHandler.injectInferredIntents([{
                    command: IntentCommandType.MOVE_SELECTION,
                    textBoundary: isNext ? IntentTextBoundaryType.WORD_END :
                        IntentTextBoundaryType.WORD_START,
                }]);
        }
    }
    /**
     * Provides all feedback once a change event in a text field fires.
     */
    onEditableChanged_(evt) {
        // TODO(crbug.com/314203187): Not null asserted, check that this is correct.
        if (!evt.target.state[StateType$6.EDITABLE]) {
            return;
        }
        // Skip all unfocused text fields.
        // TODO(crbug.com/314203187): Not null asserted, check that this is correct.
        if (!evt.target.state[StateType$6.FOCUSED] &&
            evt.target.state[StateType$6.EDITABLE]) {
            return;
        }
        const isInput = evt.target.htmlTag === 'input';
        const isTextArea = evt.target.htmlTag === 'textarea';
        // TODO(crbug.com/314203187): Not null asserted, check that this is correct.
        const isContentEditable = evt.target.state[StateType$6.RICHLY_EDITABLE];
        switch (evt.type) {
            case EventType$6.DOCUMENT_SELECTION_CHANGED:
                // Event type DOCUMENT_SELECTION_CHANGED is duplicated by
                // TEXT_SELECTION_CHANGED for <input> elements.
                if (isInput) {
                    return;
                }
                break;
            case EventType$6.FOCUS:
                // Allowed regardless of the role.
                break;
            case EventType$6.TEXT_SELECTION_CHANGED:
            // Event type TEXT_SELECTION_CHANGED is duplicated by
            // DOCUMENT_SELECTION_CHANGED for content editables and text areas.
            // Fall through.
            case EventType$6.VALUE_IN_TEXT_FIELD_CHANGED:
                // By design, generated only for simple inputs.
                if (isContentEditable || isTextArea) {
                    return;
                }
                break;
            case EventType$6.VALUE_CHANGED:
                // During a transition period, VALUE_CHANGED is duplicated by
                // VALUE_IN_TEXT_FIELD_CHANGED for text field roles.
                //
                // TOTO(NEKTAR): Deprecate and remove VALUE_CHANGED.
                if (isContentEditable || isInput || isTextArea) {
                    return;
                }
                break;
            default:
                return;
        }
        if (!this.createTextEditHandlerIfNeeded_(evt.target)) {
            return;
        }
        if (!ChromeVoxRange.current) {
            this.onEventDefault(evt);
            ChromeVoxRange.set(CursorRange.fromNode(evt.target));
        }
        // Sync the ChromeVox range to the editable, if a selection exists.
        // TODO(crbug.com/314203187): Not null asserted, check that this is correct.
        const selectionStartObject = evt.target.root.selectionStartObject;
        const selectionStartOffset = evt.target.root.selectionStartOffset || 0;
        const selectionEndObject = evt.target.root.selectionEndObject;
        const selectionEndOffset = evt.target.root.selectionEndOffset || 0;
        if (selectionStartObject && selectionEndObject) {
            // Sync to the selection's deep equivalent especially in editables, where
            // selection is often on the root text field with a child offset.
            const selectedRange = new CursorRange(new WrappingCursor(selectionStartObject, selectionStartOffset)
                .deepEquivalent, new WrappingCursor(selectionEndObject, selectionEndOffset)
                .deepEquivalent);
            // Sync ChromeVox range with selection.
            // TODO(crbug.com/314203187): Not null asserted, check that this is
            // correct.
            if (!ChromeVoxState.instance.isReadingContinuously) {
                ChromeVoxRange.set(selectedRange, true /* from editing */);
            }
        }
        // TODO(crbug.com/314203187): Not null asserted, check that this is correct.
        this.textEditHandler_.onEvent(evt);
    }
    /**
     * Provides all feedback once a rangeValueChanged or a valueInTextFieldChanged
     * event fires.
     */
    onValueChanged_(evt) {
        // Skip root web areas.
        if (evt.target.role === RoleType$4.ROOT_WEB_AREA) {
            return;
        }
        // Delegate to the edit text handler if this is an editable, with the
        // exception of spin buttons.
        // TODO(crbug.com/314203187): Not null asserted, check that this is correct.
        if (evt.target.state[StateType$6.EDITABLE] &&
            evt.target.role !== RoleType$4.SPIN_BUTTON) {
            // If a value changed event came from NTP Searchbox input, announce the
            // new value. This is a special behavior for NTP Searchbox to announce
            // its suggestions when users navigate them using the up/down arrow key.
            // `evt.intents` is empty when NTP Searchbox gets auto-completed by
            // navigating a list of suggestions; it won't be empty if users type,
            // delete, or paste text in NTP Searchbox.
            // TODO(crbug.com/328824322): Remove the special behavior and implement
            // the active-descendant-based approach in NTP Searchbox when
            // crbug.com/346835896 lands in the stable.
            // TODO(crbug.com/314203187): Not null asserted, check that this is
            // correct.
            const urlString = evt.target.root?.url ?? '';
            if (DesktopAutomationHandler.NTP_SEARCHBOX_URLS.has(urlString) &&
                evt.target.htmlTag === 'input' && !evt.intents?.length) {
                new Output()
                    .withString(evt.target.value)
                    .withSpeechCategory(TtsCategory.NAV)
                    .withQueueMode(QueueMode.CATEGORY_FLUSH)
                    .withoutFocusRing()
                    .go();
                return;
            }
            this.onEditableChanged_(evt);
            return;
        }
        const target = evt.target;
        const fromDesktop = target.root.role === RoleType$4.DESKTOP;
        const onDesktop = ChromeVoxRange.current?.start.node.root.role === RoleType$4.DESKTOP;
        const isSlider = target.role === RoleType$4.SLIDER;
        // TODO(accessibility): get rid of callers who use value changes on list
        // boxes.
        const isListBox = target.role === RoleType$4.LIST_BOX;
        if (fromDesktop && !onDesktop && !isSlider && !isListBox) {
            // Only respond to value changes from the desktop if it's coming from a
            // slider e.g. the volume slider. Do this to avoid responding to frequent
            // updates from UI e.g. download progress bars.
            return;
        }
        // TODO(crbug.com/314203187): Not null asserted, check that this is correct.
        if (!target.state[StateType$6.FOCUSED] &&
            (!fromDesktop || (!isSlider && !isListBox)) &&
            !AutomationUtil.isDescendantOf(ChromeVoxRange.current?.start.node, target)) {
            return;
        }
        if (new Date().getTime() - this.lastValueChanged_.getTime() <=
            DesktopAutomationHandler.MIN_VALUE_CHANGE_DELAY_MS) {
            return;
        }
        this.lastValueChanged_ = new Date();
        const output = new Output();
        output.withoutFocusRing();
        if (fromDesktop &&
            (!this.lastValueTarget_ || this.lastValueTarget_ !== target)) {
            const range = CursorRange.fromNode(target);
            output.withRichSpeechAndBraille(range, range, OutputCustomEvent.NAVIGATE);
            this.lastValueTarget_ = target;
        }
        else {
            output.format('$if($value, $value, $if($valueForRange, $valueForRange))', target);
        }
        Output.forceModeForNextSpeechUtterance(QueueMode.INTERJECT);
        output.go();
    }
    /**
     * Handle updating the active indicator when the document scrolls.
     */
    onScrollPositionChanged(evt) {
        const currentRange = ChromeVoxRange.current;
        if (currentRange && currentRange.isValid()) {
            new Output().withLocation(currentRange, undefined, evt.type).go();
            if (EventSource.get() !== EventSourceType.TOUCH_GESTURE) {
                return;
            }
            const root = AutomationUtil.getTopLevelRoot(currentRange.start.node);
            if (!root || root.scrollY === undefined) {
                return;
            }
            const currentPage = Math.ceil(root.scrollY / root.location.height) || 1;
            // TODO(crbug.com/314203187): Not null asserted, check that this is
            // correct.
            const totalPages = Math.ceil((root.scrollYMax - root.scrollYMin) / root.location.height) ||
                1;
            // Ignore announcements if we've already announced something for this page
            // change. Note that this need not care about the root if it changed as
            // well.
            if (this.currentPage_ === currentPage &&
                this.totalPages_ === totalPages) {
                return;
            }
            this.currentPage_ = currentPage;
            this.totalPages_ = totalPages;
            ChromeVox.tts.speak(Msgs.getMsg('describe_pos_by_page', [String(currentPage), String(totalPages)]), QueueMode.QUEUE);
        }
    }
    onSelection(evt) {
        // Invalidate any previous editable text handler state since some nodes,
        // like menuitems, can receive selection while focus remains on an
        // editable leading to braille output routing to the editable.
        this.textEditHandler_ = null;
        chrome.automation.getFocus((focus) => {
            const target = evt.target;
            // Desktop tabs get "selection" when there's a focused webview during
            // tab switching.
            // TODO(crbug.com/314203187): Not null asserted, check that this is
            // correct.
            if (target.role === RoleType$4.TAB &&
                target.root.role === RoleType$4.DESKTOP) {
                // Read it only if focus is on the
                // omnibox. We have to resort to this check to get tab switching read
                // out because on switching to a new tab, focus actually remains on the
                // *same* omnibox.
                const currentRange = ChromeVoxRange.current;
                if (currentRange && currentRange.start && currentRange.start.node &&
                    currentRange.start.node.className === 'OmniboxViewViews') {
                    const range = CursorRange.fromNode(target);
                    new Output()
                        .withRichSpeechAndBraille(range, range, OutputCustomEvent.NAVIGATE)
                        .go();
                }
                // This also suppresses tab selection output when ChromeVox is not on
                // the omnibox.
                return;
            }
            let override = false;
            // TODO(crbug.com/314203187): Not null asserted, check that this is
            // correct.
            const isDesktop = (focus && target.root === focus.root &&
                focus.root.role === RoleType$4.DESKTOP);
            // TableView fires selection events on rows/cells
            // and we want to ignore those because it also fires focus events.
            const skip = AutomationPredicate.roles([RoleType$4.CELL, RoleType$4.GRID_CELL, RoleType$4.ROW]);
            if (isDesktop && skip(target)) {
                return;
            }
            // IME candidates are announced, independent of focus.
            // This shouldn't move ChromeVoxRange to keep editing work.
            if (target.role === RoleType$4.IME_CANDIDATE) {
                const range = CursorRange.fromNode(target);
                new Output().withRichSpeech(range, undefined, evt.type).go();
                return;
            }
            // Menu items always announce on selection events, independent of focus.
            if (AutomationPredicate.menuItem(target)) {
                override = true;
            }
            // Overview mode should allow selections.
            if (isDesktop) {
                let walker = target;
                while (walker && walker.className !== 'OverviewDeskBarWidget' &&
                    walker.className !== 'OverviewModeLabel' &&
                    walker.className !== 'Desk_Container_A') {
                    walker = walker.parent;
                }
                override = Boolean(walker) || override;
            }
            // Autofill popup menu items are always announced on selection events,
            // independent of focus.
            // The `PopupSeparatorView` is intentionally omitted because it
            // cannot be focused.
            if (target.className === 'PopupSuggestionView' ||
                target.className === 'PopupPasswordSuggestionView' ||
                target.className === 'PopupFooterView' ||
                target.className === 'PopupWarningView' ||
                target.className === 'PopupBaseView' ||
                target.className ===
                    'PasswordGenerationPopupViewViews::GeneratedPasswordBox' ||
                target.className === 'PopupRowView' ||
                target.className === 'PopupRowWithButtonView' ||
                target.className === 'PopupRowContentView') {
                override = true;
            }
            // The popup view associated with a datalist element does not descend
            // from the input with which it is associated.
            if (focus?.role === RoleType$4.TEXT_FIELD_WITH_COMBO_BOX &&
                target.role === RoleType$4.LIST_BOX_OPTION) {
                override = true;
            }
            if (override || (focus && AutomationUtil.isDescendantOf(target, focus))) {
                this.onEventDefault(evt);
            }
        });
    }
    /**
     * Provides all feedback once a menu end event fires.
     */
    onMenuEnd_(_evt) {
        // This is a work around for Chrome context menus not firing a focus event
        // after you close them.
        chrome.automation.getFocus((focus) => {
            // Directly output the node here; do not go through |onFocus_| as it
            // contains a lot of logic that can move the selection (if in an
            // editable).
            if (!focus) {
                return;
            }
            const range = CursorRange.fromNode(focus);
            new Output()
                .withRichSpeechAndBraille(range, undefined, OutputCustomEvent.NAVIGATE)
                .go();
            ChromeVoxRange.set(range);
        });
        // Reset the state to stop handling a Collapsed event.
        this.isSubMenuShowing_ = false;
    }
    onMenuPopupStart_(event) {
        // Handles a MenuPopupStart event only if it's from a menu node. This event
        // will be fired from a menu node, instead of a menu item node, when its
        // sub-menu gets expanded.
        if (!event.target || !AutomationPredicate.menu(event.target)) {
            return;
        }
        // Set a state to start handling a Collapsed event.
        this.isSubMenuShowing_ = true;
    }
    onMenuStart_(event) {
        Output.forceModeForNextSpeechUtterance(QueueMode.CATEGORY_FLUSH);
        this.onEventDefault(event);
    }
    /**
     * Provides all feedback once a scrolled to anchor event fires.
     */
    onScrolledToAnchor(evt) {
        if (!evt.target) {
            return;
        }
        if (ChromeVoxRange.current) {
            const target = evt.target;
            const current = ChromeVoxRange.current.start.node;
            if (AutomationUtil.getTopLevelRoot(current) !==
                AutomationUtil.getTopLevelRoot(target)) {
                // Ignore this event if the root of the target differs from that of the
                // current range.
                return;
            }
        }
        this.onEventDefault(evt);
    }
    /**
     * Handles autofill availability changes.
     */
    onAutofillAvailabilityChanged(evt) {
        const node = evt.target;
        // TODO(crbug.com/314203187): Not null asserted, check that this is correct.
        const state = node.state;
        const currentRange = ChromeVoxRange.current;
        // Notify the user about available autofill options on focused element.
        if (currentRange && currentRange.isValid() && state[StateType$6.FOCUSED] &&
            state[StateType$6.AUTOFILL_AVAILABLE]) {
            new Output()
                .withString(Msgs.getMsg('hint_autocomplete_list'))
                .withLocation(currentRange, undefined, evt.type)
                .withQueueMode(QueueMode.QUEUE)
                .go();
        }
    }
    /**
     * Handles orientation changes on the desktop node.
     */
    onOrientationChanged(evt) {
        // Changes on display metrics result in the desktop node's
        // vertical/horizontal states changing.
        if (evt.target.role === RoleType$4.DESKTOP) {
            // TODO(crbug.com/314203187): Not null asserted, check that this is
            // correct.
            const msg = evt.target.state[StateType$6.HORIZONTAL] ? 'device_landscape' :
                'device_portrait';
            new Output().format('@' + msg).go();
        }
    }
    /**
     * Handles focus back to a parent MenuItem when its child is collapsed.
     */
    onMenuItemCollapsed(evt) {
        const target = evt.target;
        // TODO(crbug.com/314203187): Not null asserted, check that this is correct.
        if (!this.isSubMenuShowing_ || !AutomationPredicate.menuItem(target) ||
            !target.state[StateType$6.COLLAPSED] || !target.selected) {
            return;
        }
        this.onEventDefault(evt);
    }
    /**
     * Create an editable text handler for the given node if needed.
     * @param node
     * @param opt_onFocus True if called within a focus event
     *     handler. False by default.
     * @return True if the handler exists (created/already present).
     */
    createTextEditHandlerIfNeeded_(node, opt_onFocus) {
        // TODO(crbug.com/314203187): Not null asserted, check that this is correct.
        if (!node.state[StateType$6.EDITABLE]) {
            return false;
        }
        if (!ChromeVoxRange.current || !ChromeVoxRange.current.start ||
            !ChromeVoxRange.current.start.node) {
            return false;
        }
        const topRoot = AutomationUtil.getTopLevelRoot(node);
        if (!node.state[StateType$6.FOCUSED] ||
            (topRoot && topRoot.parent &&
                !topRoot.parent.state[StateType$6.FOCUSED])) {
            return false;
        }
        // Re-target the node to the root of the editable.
        const target = AutomationUtil.getEditableRoot(node);
        let voxTarget = ChromeVoxRange.current.start.node;
        voxTarget = AutomationUtil.getEditableRoot(voxTarget) || voxTarget;
        // It is possible that ChromeVox has range over some other node when a
        // text field is focused. Only allow this when focus is on a desktop node,
        // ChromeVox is over the keyboard, or during focus events.
        // TODO(crbug.com/314203187): Not null asserted, check that this is correct.
        if (!target || !voxTarget ||
            (!opt_onFocus && target !== voxTarget &&
                target.root.role !== RoleType$4.DESKTOP &&
                voxTarget.root.role !== RoleType$4.DESKTOP &&
                !AutomationUtil.isDescendantOf(target, voxTarget) &&
                !AutomationUtil.getAncestors(voxTarget.root)
                    .find((n) => n.role === RoleType$4.KEYBOARD))) {
            return false;
        }
        if (!this.textEditHandler_ || this.textEditHandler_.node !== target) {
            this.textEditHandler_ = TextEditHandler.createForNode(target);
        }
        return Boolean(this.textEditHandler_);
    }
    maybeRecoverFocusAndOutput_(evt, focus) {
        if (!focus) {
            return;
        }
        const focusedRoot = AutomationUtil.getTopLevelRoot(focus);
        if (!focusedRoot) {
            return;
        }
        let curRoot;
        if (ChromeVoxRange.current) {
            curRoot =
                AutomationUtil.getTopLevelRoot(ChromeVoxRange.current.start.node);
        }
        // If initial focus was already placed inside this page (e.g. if a user
        // starts tabbing before load complete), then don't move ChromeVox's
        // position on the page.
        if (curRoot && focusedRoot === curRoot &&
            this.lastRootUrl_ === focusedRoot.docUrl) {
            return;
        }
        this.lastRootUrl_ = focusedRoot.docUrl || '';
        const o = new Output();
        // Restore to previous position.
        // TODO(crbug.com/314203187): Not null asserted, check that this is correct.
        let url = focusedRoot.docUrl;
        url = url.substring(0, url.indexOf('#')) || url;
        const pos = ChromeVoxState.position[url];
        // Deny recovery for chrome urls.
        if (pos && url.indexOf('chrome://') !== 0) {
            focusedRoot.hitTestWithReply(pos.x, pos.y, node => this.onHitTestResult(node));
            return;
        }
        // If range is already on |focus|, exit early to prevent duplicating output.
        const currentRange = ChromeVoxRange.current;
        if (currentRange && currentRange.start && currentRange.start.node &&
            currentRange.start.node === focus) {
            return;
        }
        // This catches initial focus (i.e. on startup).
        if (!curRoot && focus !== focusedRoot) {
            o.format('$name', focusedRoot);
        }
        ChromeVoxRange.set(CursorRange.fromNode(focus));
        if (!ChromeVoxRange.current) {
            return;
        }
        o.withRichSpeechAndBraille(ChromeVoxRange.current, undefined, evt.type)
            .go();
        // Reset `isSubMenuShowing_` when a focus changes because focus
        // changes should automatically close any menus.
        this.isSubMenuShowing_ = false;
    }
    /** Initializes global state for DesktopAutomationHandler. */
    static async init() {
        if (DesktopAutomationInterface.instance) {
            throw new Error('DesktopAutomationInterface.instance already exists.');
        }
        const desktop = await AsyncUtil.getDesktop();
        DesktopAutomationInterface.instance = new DesktopAutomationHandler(desktop);
    }
}
TestImportManager.exportForTesting(DesktopAutomationHandler);

// Copyright 2019 The Chromium Authors
// Use of this source code is governed by a BSD-style license that can be
// found in the LICENSE file.
/**
 * @fileoverview Handles automation events on the currently focused node.
 */
const EventType$5 = chrome.automation.EventType;
const RoleType$3 = chrome.automation.RoleType;
const StateType$5 = chrome.automation.StateType;
class FocusAutomationHandler extends BaseAutomationHandler {
    previousActiveDescendant_;
    static instance;
    async initListener_() {
        const desktop = await AsyncUtil.getDesktop();
        desktop.addEventListener(EventType$5.FOCUS, (node) => this.onFocus(node), false);
    }
    static async init() {
        if (FocusAutomationHandler.instance) {
            throw new Error('Trying to create two instances of singleton FocusAutomationHandler');
        }
        FocusAutomationHandler.instance = new FocusAutomationHandler(undefined);
        await FocusAutomationHandler.instance.initListener_();
    }
    onFocus(evt) {
        this.removeAllListeners();
        // Events on roots and web views can be very noisy due to bubbling. Ignore
        // these.
        if (evt.target.root === evt.target ||
            evt.target.role === RoleType$3.WEB_VIEW) {
            return;
        }
        this.previousActiveDescendant_ = evt.target.activeDescendant;
        this.node_ = evt.target;
        this.addListener_(EventType$5.ACTIVE_DESCENDANT_CHANGED, this.onActiveDescendantChanged);
        this.addListener_(EventType$5.DETAILS_CHANGED, this.onDetailsChanged);
        this.addListener_(EventType$5.MENU_ITEM_SELECTED, this.onEventIfSelected);
        this.addListener_(EventType$5.SELECTED_VALUE_CHANGED, this.onSelectedValueChanged_);
    }
    /** Handles active descendant changes. */
    async onActiveDescendantChanged(evt) {
        if (!evt.target.activeDescendant) {
            if (ChromeVoxRange.current?.equals(CursorRange.fromNode(evt.target))) {
                new Output()
                    .withLocation(ChromeVoxRange.current, undefined, evt.type)
                    .go();
            }
            return;
        }
        let skipFocusCheck = false;
        const focus = await AsyncUtil.getFocus();
        if (focus !== null && AutomationPredicate.popUpButton(focus)) {
            skipFocusCheck = true;
        }
        // TODO(b/314203187): Not null asserted, check that this is correct.
        if (!skipFocusCheck && !evt.target.state[StateType$5.FOCUSED]) {
            return;
        }
        // Various events might come before a key press (which forces flushed
        // speech) and this handler. Force output to be at least category flushed.
        Output.forceModeForNextSpeechUtterance(QueueMode.CATEGORY_FLUSH);
        const prev = this.previousActiveDescendant_ ?
            CursorRange.fromNode(this.previousActiveDescendant_) :
            ChromeVoxRange.current;
        new Output()
            .withoutHints()
            .withRichSpeechAndBraille(CursorRange.fromNode(evt.target.activeDescendant), prev ?? undefined, OutputCustomEvent.NAVIGATE)
            .go();
        this.previousActiveDescendant_ = evt.target.activeDescendant;
    }
    /** Informs users that details are now available. */
    onDetailsChanged(_evt) {
        // TODO(b/314203187): Not null asserted, check that this is correct.
        const range = ChromeVoxRange.current;
        let node = range.start?.node;
        while (node && (!node.details || !node.details.length)) {
            node = node.parent;
        }
        if (!node) {
            return;
        }
        // Note that we only output speech. Braille output shows the entire line, so
        // details output should not be based on an announcement like this. Don't
        // allow interruption of this announcement which can occur in a slew of
        // events (e.g. typing).
        new Output()
            .withInitialSpeechProperties(new TtsSpeechProperties({ doNotInterrupt: true }))
            .formatForSpeech('@hint_details')
            .go();
    }
    onEventIfSelected(evt) {
        if (evt.target.selected) {
            this.onEventDefault(evt);
        }
    }
    onSelectedValueChanged_(evt) {
        // TODO(b/314203187): Not null asserted, check that this is correct.
        if (!AutomationPredicate.popUpButton(evt.target) ||
            evt.target.state[StateType$5.EDITABLE]) {
            return;
        }
        // Focus might be on a container above the popup button.
        if (this.node_ !== evt.target) {
            return;
        }
        // Return early if the menu is expanded, to avoid double speech, as active
        // descendant events will also be received and announced.
        // TODO(b/314203187): Not null asserted, check that this is correct.
        if (evt.target.state[StateType$5.EXPANDED]) {
            return;
        }
        if (evt.target.value) {
            const output = new Output();
            output.format('$value @describe_index($posInSet, $setSize)', evt.target);
            output.go();
            return;
        }
    }
}

// Copyright 2016 The Chromium Authors
// Use of this source code is governed by a BSD-style license that can be
// found in the LICENSE file.
/**
 * @fileoverview Handles media automation events.
 */
const EventType$4 = chrome.automation.EventType;
class MediaAutomationHandler extends BaseAutomationHandler {
    static MIN_WAITTIME_MS = 1000;
    static instance = null;
    mediaRoots_ = new Set();
    lastTtsEvent_ = new Date();
    async addListeners_() {
        ChromeVox.tts.addCapturingEventListener(this);
        this.node_ = await AsyncUtil.getDesktop();
        this.addListener_(EventType$4.MEDIA_STARTED_PLAYING, this.onMediaStartedPlaying);
        this.addListener_(EventType$4.MEDIA_STOPPED_PLAYING, this.onMediaStoppedPlaying);
    }
    static async init() {
        if (MediaAutomationHandler.instance) {
            throw 'Error: trying to create two instances of singleton MediaAutomationHandler';
        }
        MediaAutomationHandler.instance = new MediaAutomationHandler();
        await MediaAutomationHandler.instance.addListeners_();
    }
    onTtsStart() {
        this.lastTtsEvent_ = new Date();
        this.update_({ start: true });
    }
    onTtsEnd() {
        const now = new Date();
        setTimeout(() => {
            const then = this.lastTtsEvent_;
            if (now < then) {
                return;
            }
            this.lastTtsEvent_ = now;
            this.update_({ end: true });
        }, MediaAutomationHandler.MIN_WAITTIME_MS);
    }
    onTtsInterrupted() {
        this.onTtsEnd();
    }
    onMediaStartedPlaying(evt) {
        this.mediaRoots_.add(evt.target);
        const audioStrategy = SettingsManager.get('audioStrategy');
        if (ChromeVox.tts.isSpeaking() && audioStrategy === 'audioDuck') {
            this.update_({ start: true });
        }
    }
    onMediaStoppedPlaying() {
        // Intentionally does nothing (to cover resume).
    }
    /**
     * Updates the media state for all observed automation roots.
     */
    update_(options) {
        const it = this.mediaRoots_.values();
        let item = it.next();
        const audioStrategy = SettingsManager.get('audioStrategy');
        while (!item.done) {
            const root = item.value;
            if (options.start) {
                if (audioStrategy === 'audioDuck') {
                    root.startDuckingMedia();
                }
                else if (audioStrategy === 'audioSuspend') {
                    root.suspendMedia();
                }
            }
            else if (options.end) {
                if (audioStrategy === 'audioDuck') {
                    root.stopDuckingMedia();
                }
                else if (audioStrategy === 'audioSuspend') {
                    root.resumeMedia();
                }
            }
            item = it.next();
        }
    }
}

// Copyright 2021 The Chromium Authors
// Use of this source code is governed by a BSD-style license that can be
// found in the LICENSE file.
/**
 * @fileoverview Handles page loading sounds based on automation events.
 */
const EventType$3 = chrome.automation.EventType;
const StateType$4 = chrome.automation.StateType;
class PageLoadSoundHandler extends BaseAutomationHandler {
    didRequestLoadSound_ = false;
    static instance;
    constructor() {
        super(undefined);
    }
    async initListeners_() {
        this.node_ = await AsyncUtil.getDesktop();
        this.addListener_(EventType$3.LOAD_COMPLETE, this.onLoadComplete);
        this.addListener_(EventType$3.LOAD_START, this.onLoadStart);
        ChromeVoxRange.addObserver(this);
    }
    static async init() {
        if (PageLoadSoundHandler.instance) {
            throw 'Error: Trying to create two instances of singleton ' +
                'PageLoadSoundHandler';
        }
        PageLoadSoundHandler.instance = new PageLoadSoundHandler();
        await PageLoadSoundHandler.instance.initListeners_();
    }
    /** Stops page load sound on load complete. */
    onLoadComplete(evt) {
        // We are only interested in load completes on valid top level roots.
        const top = AutomationUtil.getTopLevelRoot(evt.target);
        if (!top || top !== evt.target.root || !top.docUrl) {
            return;
        }
        // TODO(b/314203187): Not null asserted, check that this is correct.
        if (this.didRequestLoadSound_ && top.parent &&
            top.parent.state[StateType$4.FOCUSED]) {
            ChromeVox.earcons.playEarcon(EarconId.PAGE_FINISH_LOADING);
            this.didRequestLoadSound_ = false;
        }
    }
    /** Starts page load sound on load start. */
    onLoadStart(evt) {
        // We are only interested in load starts on focused top level roots.
        const top = AutomationUtil.getTopLevelRoot(evt.target);
        // TODO(b/314203187): Not null asserted, check that this is correct.
        if (top && top === evt.target.root && top.docUrl && top.parent &&
            top.parent.state[StateType$4.FOCUSED]) {
            ChromeVox.earcons.playEarcon(EarconId.PAGE_START_LOADING);
            this.didRequestLoadSound_ = true;
        }
    }
    /** ChromeVoxRangeObserver implementation */
    onCurrentRangeChanged(range) {
        if (!range || !range.start || !range.start.node) {
            return;
        }
        const top = AutomationUtil.getTopLevelRoot(range.start.node);
        // |top| might be undefined e.g. if range is not in a root web area.
        if (this.didRequestLoadSound_ && (!top || top.docLoadingProgress === 1)) {
            ChromeVox.earcons.playEarcon(EarconId.PAGE_FINISH_LOADING);
            this.didRequestLoadSound_ = false;
        }
        // Note that we intentionally don't re-start progress playback here even if
        // the docLoadingProgress < 1.
    }
}
TestImportManager.exportForTesting(PageLoadSoundHandler);

// Copyright 2019 The Chromium Authors
// Use of this source code is governed by a BSD-style license that can be
// found in the LICENSE file.
/**
 * @fileoverview Handles automation from ChromeVox's current range.
 */
const EventType$2 = chrome.automation.EventType;
const RoleType$2 = chrome.automation.RoleType;
const StateType$3 = chrome.automation.StateType;
class RangeAutomationHandler extends BaseAutomationHandler {
    lastAttributeTarget_;
    lastAttributeOutput_;
    delayedAttributeOutputId_ = -1;
    static instance;
    constructor() {
        super();
        ChromeVoxRange.addObserver(this);
    }
    static init() {
        if (RangeAutomationHandler.instance) {
            throw new Error('Trying to create two copies of singleton RangeAutomationHandler');
        }
        RangeAutomationHandler.instance = new RangeAutomationHandler();
    }
    onCurrentRangeChanged(newRange, _fromEditing) {
        if (this.node_) {
            this.removeAllListeners();
            this.node_ = undefined;
        }
        if (!newRange || !newRange.start.node || !newRange.end.node) {
            return;
        }
        this.node_ = AutomationUtil.getLeastCommonAncestor(newRange.start.node, newRange.end.node) ||
            newRange.start.node;
        // Some re-targeting is needed for cases like tables.
        let retarget = this.node_;
        while (retarget && retarget !== retarget.root) {
            // Table headers require retargeting for events because they often have
            // event types we care about e.g. sort direction.
            if (AutomationPredicate.tableHeader(retarget)) {
                this.node_ = retarget;
                break;
            }
            retarget = retarget.parent;
        }
        // TODO: some of the events mapped to onAttributeChanged need to have
        // specific handlers that only output the specific attribute. There also
        // needs to be an audit of all attribute change events to ensure they get
        // outputted.
        // TODO(crbug.com/1464633) Fully remove ARIA_ATTRIBUTE_CHANGED_DEPRECATED
        // starting in 122, because although it was removed in 118, it is still
        // present in earlier versions of LaCros.
        this.addListener_(EventType$2.ARIA_ATTRIBUTE_CHANGED_DEPRECATED, this.onAttributeChanged);
        this.addListener_(EventType$2.AUTO_COMPLETE_CHANGED, this.onAttributeChanged);
        this.addListener_(EventType$2.IMAGE_ANNOTATION_CHANGED, this.onAttributeChanged);
        this.addListener_(EventType$2.NAME_CHANGED, this.onAttributeChanged);
        this.addListener_(EventType$2.DESCRIPTION_CHANGED, this.onAttributeChanged);
        this.addListener_(EventType$2.ROLE_CHANGED, this.onAttributeChanged);
        this.addListener_(EventType$2.AUTOCORRECTION_OCCURED, this.onEventIfInRange);
        this.addListener_(EventType$2.CHECKED_STATE_CHANGED, this.onCheckedStateChanged);
        this.addListener_(EventType$2.CHECKED_STATE_DESCRIPTION_CHANGED, this.onCheckedStateChanged);
        this.addListener_(EventType$2.COLLAPSED, this.onEventIfInRange);
        this.addListener_(EventType$2.CONTROLS_CHANGED, this.onControlsChanged);
        this.addListener_(EventType$2.EXPANDED, this.onEventIfInRange);
        this.addListener_(EventType$2.IMAGE_FRAME_UPDATED, this.onImageFrameUpdated_);
        this.addListener_(EventType$2.INVALID_STATUS_CHANGED, this.onEventIfInRange);
        this.addListener_(EventType$2.LOCATION_CHANGED, this.onLocationChanged);
        this.addListener_(EventType$2.RELATED_NODE_CHANGED, this.onAttributeChanged);
        this.addListener_(EventType$2.ROW_COLLAPSED, this.onEventIfInRange);
        this.addListener_(EventType$2.ROW_EXPANDED, this.onEventIfInRange);
        this.addListener_(EventType$2.STATE_CHANGED, this.onAttributeChanged);
        this.addListener_(EventType$2.SORT_CHANGED, this.onAttributeChanged);
    }
    onEventIfInRange(evt) {
        if (BaseAutomationHandler.disallowEventFromAction(evt)) {
            return;
        }
        const prev = ChromeVoxRange.current;
        if (!prev) {
            return;
        }
        // TODO: we need more fine grained filters for attribute changes.
        // TODO(b/314203187): Not null asserted, check that this is correct.
        if (prev.contentEquals(CursorRange.fromNode(evt.target)) ||
            evt.target.state[StateType$3.FOCUSED]) {
            const prevTarget = this.lastAttributeTarget_;
            // Re-target to active descendant if it exists.
            const prevOutput = this.lastAttributeOutput_;
            this.lastAttributeTarget_ = evt.target.activeDescendant || evt.target;
            this.lastAttributeOutput_ = new Output().withRichSpeechAndBraille(CursorRange.fromNode(this.lastAttributeTarget_), prev, OutputCustomEvent.NAVIGATE);
            if (this.lastAttributeTarget_ === prevTarget && prevOutput &&
                prevOutput.equals(this.lastAttributeOutput_)) {
                return;
            }
            // If the target or an ancestor is controlled by another control, we may
            // want to delay the output.
            let maybeControlledBy = evt.target;
            while (maybeControlledBy) {
                if (maybeControlledBy.controlledBy &&
                    maybeControlledBy.controlledBy.find(n => Boolean(n.autoComplete))) {
                    clearTimeout(this.delayedAttributeOutputId_);
                    this.delayedAttributeOutputId_ = setTimeout(() => this.lastAttributeOutput_.go(), ATTRIBUTE_DELAY_MS);
                    return;
                }
                maybeControlledBy = maybeControlledBy.parent;
            }
            this.lastAttributeOutput_.go();
        }
    }
    onAttributeChanged(evt) {
        // Don't report changes on editable nodes since they interfere with text
        // selection changes. Users can query via Search+k for the current state
        // of the text field (which would also report the entire value).
        // TODO(b/314203187): Not null asserted, check that this is correct.
        if (evt.target.state[StateType$3.EDITABLE]) {
            return;
        }
        // Don't report changes in static text nodes which can be extremely noisy.
        if (evt.target.role === RoleType$2.STATIC_TEXT) {
            return;
        }
        // To avoid output of stale information, don't report changes in IME
        // candidates. IME candidate output is handled during selection events.
        if (evt.target.role === RoleType$2.IME_CANDIDATE) {
            return;
        }
        // Don't report changes while captions are focused because CaptionsHandler
        // handles captions bubble text changes directly.
        if (CaptionsHandler.inCaptions()) {
            return;
        }
        // Report attribute changes for specific generated events.
        if (evt.type === chrome.automation.EventType.SORT_CHANGED) {
            let msgId;
            if (evt.target.sortDirection ===
                chrome.automation.SortDirectionType.ASCENDING) {
                msgId = 'sort_ascending';
            }
            else if (evt.target.sortDirection ===
                chrome.automation.SortDirectionType.DESCENDING) {
                msgId = 'sort_descending';
            }
            if (msgId) {
                new Output().withString(Msgs.getMsg(msgId)).go();
            }
            return;
        }
        // Only report attribute changes on some *Option roles if it is selected.
        if (AutomationPredicate.listOption(evt.target) && !evt.target.selected) {
            return;
        }
        this.onEventIfInRange(evt);
    }
    /** Provides all feedback once a checked state changed event fires. */
    onCheckedStateChanged(evt) {
        if (!AutomationPredicate.checkable(evt.target)) {
            return;
        }
        const event = new CustomAutomationEvent(EventType$2.CHECKED_STATE_CHANGED, evt.target, {
            eventFrom: evt.eventFrom,
            eventFromAction: evt.eventFromAction,
            intents: evt.intents,
        });
        this.onEventIfInRange(event);
    }
    onControlsChanged(event) {
        if (event.target.role === RoleType$2.TAB) {
            new Output()
                .withSpeech(CursorRange.fromNode(event.target), undefined, event.type)
                .go();
        }
    }
    /**
     * Updates the focus ring if the location of the current range, or
     * an descendant of the current range, changes.
     */
    onLocationChanged(evt) {
        const cur = ChromeVoxRange.current;
        if (!cur || !cur.isValid()) {
            if (FocusBounds.get().length) {
                FocusBounds.set([]);
            }
            return;
        }
        // Rather than trying to figure out if the current range falls somewhere
        // in |evt.target|, just update it if our cached bounds don't match.
        const oldFocusBounds = FocusBounds.get();
        let startRect = cur.start.node.location;
        let endRect = cur.end.node.location;
        if (cur.start.node.activeDescendant) {
            startRect = cur.start.node.activeDescendant.location;
        }
        if (cur.end.node.activeDescendant) {
            endRect = cur.end.node.activeDescendant.location;
        }
        const found = oldFocusBounds.some((rect) => this.areRectsEqual_(rect, startRect)) &&
            oldFocusBounds.some((rect) => this.areRectsEqual_(rect, endRect));
        if (found) {
            return;
        }
        // Currently only considers if there's an active descendant on the
        // start node.
        const activeDescendant = cur.start.node.activeDescendant;
        if (activeDescendant) {
            new Output()
                .withLocation(CursorRange.fromNode(activeDescendant), undefined, evt.type)
                .go();
        }
        else {
            new Output().withLocation(cur, undefined, evt.type).go();
        }
    }
    /** Called when an image frame is received on a node. */
    onImageFrameUpdated_(evt) {
        const target = evt.target;
        if (target.imageDataUrl) {
            ChromeVox.braille.writeRawImage(target.imageDataUrl);
        }
    }
    areRectsEqual_(rectA, rectB) {
        return rectA.left === rectB.left && rectA.top === rectB.top &&
            rectA.width === rectB.width && rectA.height === rectB.height;
    }
}
// Local to module.
/**
 * Time to wait before announcing attribute changes that are otherwise too
 * disruptive.
 */
const ATTRIBUTE_DELAY_MS = 1500;
TestImportManager.exportForTesting(RangeAutomationHandler);

// Copyright 2016 The Chromium Authors
// Use of this source code is governed by a BSD-style license that can be
// found in the LICENSE file.
/**
 * @fileoverview Handles output for Chrome's built-in find.
 */
const MarkerType = chrome.automation.MarkerType;
const TreeChangeObserverFilter = chrome.automation.TreeChangeObserverFilter;
/**
 * Handles navigation among the results when using the built-in find behavior
 * (i.e. Ctrl-F).
 */
class FindHandler {
    treeChangeObserver_ = (change) => this.onTextMatch_(change);
    // The last time a find marker was received.
    lastFindMarkerReceived = new Date();
    static instance;
    constructor() {
        chrome.automation.addTreeChangeObserver(TreeChangeObserverFilter.TEXT_MARKER_CHANGES, this.treeChangeObserver_);
    }
    /** Initializes this module. */
    static init() {
        if (FindHandler.instance) {
            throw 'Error: Trying to create two instances of singleton FindHandler';
        }
        FindHandler.instance = new FindHandler();
    }
    onTextMatch_(evt) {
        if (!evt.target.markers?.some((marker) => marker.flags[MarkerType.TEXT_MATCH])) {
            return;
        }
        // When a user types, a flurry of events gets sent from the tree updates
        // being applied. Drop all but the first. Note that when hitting enter,
        // there's only one marker changed ever sent.
        const delta = new Date().getTime() - this.lastFindMarkerReceived.getTime();
        this.lastFindMarkerReceived = new Date();
        if (delta < DROP_MATCH_WITHIN_TIME_MS) {
            return;
        }
        const range = CursorRange.fromNode(evt.target);
        ChromeVoxRange.set(range);
        new Output()
            .withRichSpeechAndBraille(range, undefined, OutputCustomEvent.NAVIGATE)
            .go();
    }
}
// Local to module.
/**
 * The amount of time where a subsequent find text marker is dropped from
 * output.
 */
const DROP_MATCH_WITHIN_TIME_MS = 50;

// Copyright 2014 The Chromium Authors
// Use of this source code is governed by a BSD-style license that can be
// found in the LICENSE file.
/**
 * @fileoverview Responsible for loading scripts into the inject context.
 */
class InjectedScriptLoader {
    code_ = {};
    constructor() { }
    static injectContentScriptForGoogleDocs() {
        // Build a regexp to match all allowed urls.
        let matches = [];
        try {
            matches = chrome.runtime.getManifest()['content_scripts'][0]['matches'];
        }
        catch (e) {
            throw new Error('Unable to find content script matches entry in manifest.');
        }
        // Build one large regexp.
        const docsRe = new RegExp(matches.join('|'));
        // Inject the content script into all running tabs allowed by the
        // manifest. This block is still necessary because the extension system
        // doesn't re-inject content scripts into already running tabs.
        chrome.windows.getAll({ 'populate': true }, (windows) => {
            for (let i = 0; i < windows.length; i++) {
                // TODO(b/314203187): Determine if not null assertion is acceptable.
                const tabs = windows[i].tabs.filter(tab => docsRe.test(tab.url));
                InjectedScriptLoader.injectContentScript_(tabs);
            }
        });
    }
    /**
     * Loads a dictionary of file contents for Javascript files.
     * @param files A list of file names.
     * @private
     */
    async fetchCode_(files) {
        Promise.all(files.map(file => this.loadScriptAsCode_(file)));
    }
    async loadScriptAsCode_(fileName) {
        if (this.code_[fileName]) {
            return;
        }
        // Load the script by fetching its source and running 'eval' on it
        // directly, with a magic comment that makes Chrome treat it like it
        // loaded normally. Wait until it's fetched before loading the
        // next script.
        const url = chrome.extension.getURL(fileName) + '?' + new Date().getTime();
        const response = await fetch(url);
        if (response.ok) {
            let scriptText = await response.text();
            // Add a magic comment to the bottom of the file so that
            // Chrome knows the name of the script in the JavaScript debugger.
            const debugSrc = fileName.replace('closure/../', '');
            // The 'chromevox' id is only used in the DevTools instead of a long
            // extension id.
            scriptText +=
                '\n//# sourceURL= chrome-extension://chromevox/' + debugSrc + '\n';
            this.code_[fileName] = scriptText;
        }
        else {
            // Cause the promise created by this async function to reject.
            throw new Error(`${response.status}: ${response.statusText}`);
        }
    }
    async executeCodeInAllTabs_(tabs) {
        for (const tab of tabs) {
            // Inject the ChromeVox content script code into the tab.
            await Promise.all(Object.values(this.code_).map(script => this.execute_(script, tab)));
        }
    }
    /**
     * Inject the content scripts into already existing tabs.
     * @param tabs The tabs where ChromeVox scripts should be
     *     injected.
     * @private
     */
    static async injectContentScript_(tabs) {
        if (!InjectedScriptLoader.instance) {
            InjectedScriptLoader.instance = new InjectedScriptLoader();
        }
        await InjectedScriptLoader.instance.fetchCode_(contentScriptFiles);
        await InjectedScriptLoader.instance.executeCodeInAllTabs_(tabs);
    }
    async execute_(code, tab) {
        await new Promise(resolve => chrome.tabs.executeScript(tab.id, { code, allFrames: true }, resolve));
        if (chrome.runtime.lastError) {
            console.error('Could not inject into tab', tab);
        }
    }
}
(function (InjectedScriptLoader) {
})(InjectedScriptLoader || (InjectedScriptLoader = {}));
// Local to module.
const contentScriptFiles = chrome.runtime.getManifest()['content_scripts'][0]['js'];

// Copyright 2014 The Chromium Authors
// Use of this source code is governed by a BSD-style license that can be
// found in the LICENSE file.
/**
 * @fileoverview This class provides a stable interface for initializing,
 * querying, and modifying a ChromeVox key map.
 *
 * An instance contains an object-based bi-directional mapping from key binding
 * to a function name of a user command (herein simply called a command).
 * A caller is responsible for providing a JSON keymap (a simple Object key
 * value structure), which has (key, command) key value pairs.
 *
 * To retrieve static data about user commands, see CommandStore.
 */
class KeyMap {
    /** An array of bindings - Commands and KeySequences. */
    bindings_;
    /**
     * Maps a command to a key. This optimizes the process of searching for a
     * key sequence when you already know the command.
     */
    commandToKey_ = {};
    static instance;
    constructor(keyBindings) {
        this.bindings_ = keyBindings;
        this.buildCommandToKey_();
    }
    /**
     * The number of mappings in the keymap.
     * @return The number of mappings.
     */
    length() {
        return this.bindings_.length;
    }
    /**
     * Returns a copy of all KeySequences in this map.
     * @return Array of all keys.
     */
    keys() {
        return this.bindings_.map(binding => binding.sequence);
    }
    /** Returns a shallow copy of the Command, KeySequence bindings. */
    bindings() {
        return this.bindings_.slice();
    }
    /** Checks if this key map has a given binding. */
    hasBinding(command, sequence) {
        if (this.commandToKey_ != null) {
            return this.commandToKey_[command] === sequence;
        }
        return this.bindings_.some(b => b.command === command && b.sequence.equals(sequence));
    }
    /** Checks if this key map has a given command. */
    hasCommand(command) {
        if (this.commandToKey_ != null) {
            return this.commandToKey_[command] !== undefined;
        }
        return this.bindings_.some(b => b.command === command);
    }
    /** Checks if this key map has a given key. */
    hasKey(key) {
        return this.bindings_.some(b => b.sequence.equals(key));
    }
    /** Gets a command given a key. */
    commandForKey(key) {
        return this.bindings_.find(b => b.sequence.equals(key))?.command;
    }
    /** Gets a key given a command. */
    keyForCommand(command) {
        if (this.commandToKey_ != null) {
            // TODO(b/314203187): Not null asserted, check that this is correct.
            return [this.commandToKey_[command]];
        }
        return this.bindings_.filter(b => b.command === command)
            .map(b => b.sequence);
    }
    /** Convenience method for getting the ChromeVox key map. */
    static get() {
        if (KeyMap.instance) {
            return KeyMap.instance;
        }
        const keyBindings = CommandStore.getKeyBindings();
        KeyMap.instance = new KeyMap(keyBindings);
        return KeyMap.instance;
    }
    /** Builds the map of commands to keys. */
    buildCommandToKey_() {
        // TODO (dtseng): What about more than one sequence mapped to the same
        // command?
        for (const binding of this.bindings_) {
            if (this.commandToKey_[binding.command] !== undefined) {
                // There's at least two key sequences mapped to the same
                // command. continue.
                continue;
            }
            this.commandToKey_[binding.command] = binding.sequence;
        }
    }
}
TestImportManager.exportForTesting(KeyMap);

// Copyright 2014 The Chromium Authors
// Use of this source code is governed by a BSD-style license that can be
// found in the LICENSE file.
/**
 * @fileoverview A collection of JavaScript utilities used to simplify working
 * with keyboard events.
 */
var KeyUtil;
(function (KeyUtil) {
    /**
     * Convert a key event into a Key Sequence representation.
     *
     * @param keyEvent The keyEvent to convert.
     * @return A key sequence representation of the key event.
     */
    function keyEventToKeySequence(keyEvent) {
        if (KeyUtil.prevKeySequence &&
            (KeyUtil.maxSeqLength === KeyUtil.prevKeySequence.length())) {
            // Reset the sequence buffer if max sequence length is reached.
            KeyUtil.sequencing = false;
            KeyUtil.prevKeySequence = null;
        }
        const hasKeyPrefix = keyEvent.keyPrefix;
        const stickyMode = keyEvent.stickyMode;
        // Either we are in the middle of a key sequence (N > H), or the key prefix
        // was pressed before (Ctrl+Z), or sticky mode is enabled
        const keyIsPrefixed = KeyUtil.sequencing || hasKeyPrefix || stickyMode;
        // Create key sequence.
        let keySequence = new KeySequence(keyEvent);
        // Check if the Cvox key should be considered as pressed because the
        // modifier key combination is active.
        const keyWasCvox = keySequence.cvoxModifier;
        if (keyIsPrefixed || keyWasCvox) {
            if (!KeyUtil.sequencing && KeyUtil.isSequenceSwitchKeyCode(keySequence)) {
                // If this is the beginning of a sequence.
                KeyUtil.sequencing = true;
                KeyUtil.prevKeySequence = keySequence;
                return keySequence;
            }
            else if (KeyUtil.sequencing) {
                // TODO(b/314203187): Not nulls asserted, check that this is correct.
                if (KeyUtil.prevKeySequence.addKeyEvent(keyEvent)) {
                    keySequence = KeyUtil.prevKeySequence;
                    KeyUtil.prevKeySequence = null;
                    KeyUtil.sequencing = false;
                    return keySequence;
                }
                else {
                    throw 'Think sequencing is enabled, yet KeyUtil.prevKeySequence' +
                        'already has two key codes (' + KeyUtil.prevKeySequence + ')';
                }
            }
        }
        else {
            KeyUtil.sequencing = false;
        }
        // Repeated keys pressed.
        const currTime = new Date().getTime();
        if (KeyUtil.isDoubleTapKey(keySequence) && KeyUtil.prevKeySequence &&
            keySequence.equals(KeyUtil.prevKeySequence)) {
            const prevTime = KeyUtil.modeKeyPressTime;
            const delta = currTime - prevTime;
            if (!keyEvent.repeat && prevTime > 0 && delta < 300) /* Double tap */ {
                keySequence = KeyUtil.prevKeySequence;
                keySequence.doubleTap = true;
                KeyUtil.prevKeySequence = null;
                KeyUtil.sequencing = false;
                return keySequence;
            }
            // The user double tapped the sticky key but didn't do it within the
            // required time. It's possible they will try again, so keep track of the
            // time the sticky key was pressed and keep track of the corresponding
            // key sequence.
        }
        KeyUtil.prevKeySequence = keySequence;
        KeyUtil.modeKeyPressTime = currTime;
        return keySequence;
    }
    KeyUtil.keyEventToKeySequence = keyEventToKeySequence;
    /**
     * Returns the string representation of the specified key code.
     * @return A string representation of the key event.
     */
    function keyCodeToString(keyCode) {
        if (keyCode === KeyCode.CONTROL) {
            return 'Ctrl';
        }
        if (KeyCode.name(keyCode)) {
            return KeyCode.name(keyCode);
        }
        // Anything else
        return '#' + keyCode;
    }
    KeyUtil.keyCodeToString = keyCodeToString;
    /**
     * Returns the keycode of a string representation of the specified modifier.
     *
     * @param keyString Modifier key.
     * @return Key code.
     */
    function modStringToKeyCode(keyString) {
        switch (keyString) {
            case 'Ctrl':
                return KeyCode.CONTROL;
            case 'Alt':
                return KeyCode.ALT;
            case 'Shift':
                return KeyCode.SHIFT;
            case 'Cmd':
            case 'Win':
                return KeyCode.SEARCH;
        }
        return -1;
    }
    KeyUtil.modStringToKeyCode = modStringToKeyCode;
    /**
     * Returns the key codes of a string representation of the ChromeVox
     * modifiers.
     *
     * @return Array of key codes.
     */
    function cvoxModKeyCodes() {
        const modKeyCombo = KeySequence.modKeyStr.split(/\+/g);
        const modKeyCodes = modKeyCombo.map(keyString => KeyUtil.modStringToKeyCode(keyString));
        return modKeyCodes;
    }
    KeyUtil.cvoxModKeyCodes = cvoxModKeyCodes;
    /**
     * Checks if the specified key code is a key used for switching into a
     * sequence mode. Sequence switch keys are specified in
     * KeySequence.sequenceSwitchKeyCodes
     *
     * @param rhKeySeq The key sequence to check.
     * @return true if it is a sequence switch keycode, false otherwise.
     */
    function isSequenceSwitchKeyCode(rhKeySeq) {
        for (let i = 0; i < KeySequence.sequenceSwitchKeyCodes.length; i++) {
            const lhKeySeq = KeySequence.sequenceSwitchKeyCodes[i];
            if (lhKeySeq.equals(rhKeySeq)) {
                return true;
            }
        }
        return false;
    }
    KeyUtil.isSequenceSwitchKeyCode = isSequenceSwitchKeyCode;
    /**
     * Get readable string description of the specified keycode.
     *
     * @param keyCode The key code.
     * @return Returns a string description.
     */
    function getReadableNameForKeyCode(keyCode) {
        const msg = Msgs.getMsg.bind(Msgs);
        switch (keyCode) {
            case KeyCode.BROWSER_BACK:
                return msg('back_key');
            case KeyCode.BROWSER_FORWARD:
                return msg('forward_key');
            case KeyCode.BROWSER_REFRESH:
                return msg('refresh_key');
            case KeyCode.ZOOM:
                return msg('toggle_full_screen_key');
            case KeyCode.MEDIA_LAUNCH_APP1:
                return msg('window_overview_key');
            case KeyCode.BRIGHTNESS_DOWN:
                return msg('brightness_down_key');
            case KeyCode.BRIGHTNESS_UP:
                return msg('brightness_up_key');
            case KeyCode.VOLUME_MUTE:
                return msg('volume_mute_key');
            case KeyCode.VOLUME_DOWN:
                return msg('volume_down_key');
            case KeyCode.VOLUME_UP:
                return msg('volume_up_key');
            case KeyCode.ASSISTANT:
                return msg('assistant_key');
            case KeyCode.MEDIA_PLAY_PAUSE:
                return msg('media_play_pause');
        }
        return KeyCode.name(keyCode);
    }
    KeyUtil.getReadableNameForKeyCode = getReadableNameForKeyCode;
    /**
     * Get the platform specific sticky key keycode.
     * @return The platform specific sticky key keycode.
     */
    function getStickyKeyCode() {
        return KeyCode.SEARCH;
    }
    KeyUtil.getStickyKeyCode = getStickyKeyCode;
    // TODO (clchen): Refactor this function away since it is no longer used.
    function getReadableNameForStr(_keyStr) {
        return null;
    }
    KeyUtil.getReadableNameForStr = getReadableNameForStr;
    /**
     * Creates a string representation of a KeySequence.
     * A KeySequence  with a keyCode of 76 ('L') and the control and alt keys down
     * would return the string 'Ctrl+Alt+L', for example. A key code that doesn't
     * correspond to a letter or number will typically return a string with a
     * pound and then its keyCode, like '#39' for Right Arrow. However,
     * if the opt_readableKeyCode option is specified, the key code will return a
     * readable string description like 'Right Arrow' instead of '#39'.
     *
     * The modifiers always come in this order:
     *
     *   Ctrl
     *   Alt
     *   Shift
     *   Meta
     *
     * @param keySequence The KeySequence object.
     * @param readableKeyCode Whether or not to return a readable
     * string description instead of a string with a pound symbol and a keycode.
     * Default is false.
     * @param modifiers Restrict printout to only modifiers. Defaults to false.
     */
    async function keySequenceToString(keySequence, readableKeyCode, modifiers) {
        // TODO(rshearer): Move this method and the getReadableNameForKeyCode and
        // the method to KeySequence after we refactor isModifierActive (when the
        // modifie key becomes customizable and isn't stored as a string). We can't
        // do it earlier because isModifierActive uses
        // KeyUtil.getReadableNameForKeyCode, and I don't want KeySequence to depend
        // on KeyUtil.
        let str = '';
        const numKeys = keySequence.length();
        for (let index = 0; index < numKeys; index++) {
            if (str !== '' && !modifiers) {
                str += ', then ';
            }
            else if (str !== '') {
                str += '+';
            }
            // This iterates through the sequence. Either we're on the first key
            // pressed or the second
            let tempStr = '';
            for (const keyPressed in keySequence.keys) {
                // This iterates through the actual key, taking into account any
                // modifiers.
                //@ts-expect-error Indexing with string not allowed
                if (!keySequence.keys[keyPressed][index]) {
                    continue;
                }
                let modifier = '';
                switch (keyPressed) {
                    case 'ctrlKey':
                        // TODO(rshearer): This is a hack to work around the special casing
                        // of the Ctrl key that used to happen in keyEventToString. We won't
                        // need it once we move away from strings completely.
                        modifier = 'Ctrl';
                        break;
                    case 'searchKeyHeld':
                        const searchKey = KeyUtil.getReadableNameForKeyCode(KeyCode.SEARCH);
                        modifier = searchKey;
                        break;
                    case 'altKey':
                        modifier = 'Alt';
                        break;
                    case 'altGraphKey':
                        modifier = 'AltGraph';
                        break;
                    case 'shiftKey':
                        modifier = 'Shift';
                        break;
                    case 'metaKey':
                        const metaKey = KeyUtil.getReadableNameForKeyCode(KeyCode.SEARCH);
                        modifier = metaKey;
                        break;
                    case 'keyCode':
                        const keyCode = keySequence.keys[keyPressed][index];
                        // We make sure the keyCode isn't for a modifier key. If it is, then
                        // we've already added that into the string above.
                        if (keySequence.isModifierKey(keyCode) || modifiers) {
                            break;
                        }
                        if (!readableKeyCode) {
                            tempStr += KeyUtil.keyCodeToString(keyCode);
                            break;
                        }
                        // First, try using Chrome OS's localized DOM key string conversion.
                        let domKeyString = await AsyncUtil.getLocalizedDomKeyStringForKeyCode(keyCode);
                        if (!domKeyString) {
                            tempStr += KeyUtil.getReadableNameForKeyCode(keyCode);
                            break;
                        }
                        // Upper case single-lettered key strings for better tts.
                        if (domKeyString.length === 1) {
                            domKeyString = domKeyString.toUpperCase();
                        }
                        tempStr += domKeyString;
                        break;
                }
                if (str.indexOf(modifier) === -1) {
                    tempStr += modifier + '+';
                }
            }
            str += tempStr;
            // Strip trailing +.
            if (str[str.length - 1] === '+') {
                str = str.slice(0, -1);
            }
        }
        if (keySequence.cvoxModifier || keySequence.prefixKey) {
            if (str !== '') {
                str = 'Search+' + str;
            }
            else {
                str = 'Search+Search';
            }
        }
        else if (keySequence.stickyMode) {
            // Strip trailing ', then '.
            const cut = str.slice(str.length - ', then '.length);
            if (cut === ', then ') {
                str = str.slice(0, str.length - cut.length);
            }
            str = str + '+' + str;
        }
        return str;
    }
    KeyUtil.keySequenceToString = keySequenceToString;
    /**
     * Looks up if the given key sequence is triggered via double tap.
     * @return True if key is triggered via double tap.
     */
    function isDoubleTapKey(key) {
        let isSet = false;
        const originalState = key.doubleTap;
        key.doubleTap = true;
        for (let i = 0, keySeq; keySeq = KeySequence.doubleTapCache[i]; i++) {
            if (keySeq.equals(key)) {
                isSet = true;
                break;
            }
        }
        key.doubleTap = originalState;
        return isSet;
    }
    KeyUtil.isDoubleTapKey = isDoubleTapKey;
    KeyUtil.modeKeyPressTime = 0;
    KeyUtil.sequencing = false;
    KeyUtil.prevKeySequence = null;
    KeyUtil.stickyKeySequence = null;
    /**
     * Maximum number of key codes the sequence buffer may hold. This is the max
     * length of a sequential keyboard shortcut, i.e. the number of key that can
     * be pressed one after the other while modifier keys (Cros+Shift) are held
     * down.
     */
    KeyUtil.maxSeqLength = 2;
})(KeyUtil || (KeyUtil = {}));

// Copyright 2014 The Chromium Authors
// Use of this source code is governed by a BSD-style license that can be
// found in the LICENSE file.
/**
 * @fileoverview Handles user keyboard input events.
 */
var ChromeVoxKbHandler;
(function (ChromeVoxKbHandler) {
    /** The key map */
    ChromeVoxKbHandler.handlerKeyMap = KeyMap.get();
    /**
     * Handles key down events.
     *
     * @param evt The key down event to process.
     * @return True if the default action should be performed.
     */
    ChromeVoxKbHandler.basicKeyDownActionsListener = function (evt) {
        const keySequence = KeyUtil.keyEventToKeySequence(evt);
        const functionName = ChromeVoxKbHandler.handlerKeyMap.commandForKey(keySequence);
        // TODO (clchen): Disambiguate why functions are null. If the user
        // pressed something that is not a valid combination, make an error
        // noise so there is some feedback.
        if (!functionName) {
            return !KeyUtil.sequencing;
        }
        // This is the key event handler return value - true if the event should
        // propagate and the default action should be performed, false if we eat
        // the key.
        let returnValue = true;
        const commandResult = ChromeVoxKbHandler.commandHandler(functionName);
        if (commandResult !== undefined) {
            returnValue = commandResult;
        }
        else if (keySequence.cvoxModifier) {
            // Modifier/prefix is active -- prevent default action
            returnValue = false;
        }
        return returnValue;
    };
})(ChromeVoxKbHandler || (ChromeVoxKbHandler = {}));
TestImportManager.exportForTesting(['ChromeVoxKbHandler', ChromeVoxKbHandler]);

// Copyright 2020 The Chromium Authors
// Use of this source code is governed by a BSD-style license that can be
// found in the LICENSE file.
/**
 * @fileoverview Forces user actions down a predetermined path.
 */
/** The types of actions we want to monitor. */
var ActionType;
(function (ActionType) {
    ActionType["BRAILLE"] = "braille";
    ActionType["GESTURE"] = "gesture";
    ActionType["KEY_SEQUENCE"] = "key_sequence";
    ActionType["MOUSE_EVENT"] = "mouse_event";
    ActionType["RANGE_CHANGE"] = "range_change";
})(ActionType || (ActionType = {}));
/**
 * Monitors user actions and forces them down a predetermined path. Receives a
 * queue of expected actions upon construction and blocks ChromeVox execution
 * until each action is matched. Hooks into various handlers to intercept user
 * actions before they are processed by the rest of ChromeVox.
 */
class ForcedActionPath {
    actionIndex_ = 0;
    actions_ = [];
    onFinishedCallback_;
    static instance;
    static postGestureCallbackForT