// Copyright 2018 The Chromium Authors
// Use of this source code is governed by a BSD-style license that can be
// found in the LICENSE file.
/**
 * @fileoverview Provides output logger.
 */
import { LocalStorage } from '/common/local_storage.js';
import { LogStore } from '../logging/log_store.js';
export class OutputFormatLogger {
    str_ = '';
    storageEnabledKey_;
    logType_;
    /** @param enableKey The key to enable logging in LocalStorage. */
    constructor(enableKey, type) {
        this.storageEnabledKey_ = enableKey;
        this.logType_ = type;
    }
    get loggingDisabled() {
        return !LocalStorage.get(this.storageEnabledKey_);
    }
    /** Sends the queued logs to the LogStore. */
    commitLogs() {
        if (this.str_) {
            LogStore.instance.writeTextLog(this.str_, this.logType_);
        }
    }
    write(str) {
        if (this.loggingDisabled) {
            return;
        }
        this.str_ += str;
    }
    writeTokenWithValue(token, value) {
        if (this.loggingDisabled) {
            return;
        }
        this.writeToken(token);
        if (value) {
            this.str_ += value;
        }
        else {
            this.str_ += 'EMPTY';
        }
        this.str_ += '\n';
    }
    writeToken(token) {
        if (this.loggingDisabled) {
            return;
        }
        this.str_ += '$' + token + ': ';
    }
    writeRule(rule) {
        if (this.loggingDisabled) {
            return;
        }
        this.str_ += 'RULE: ';
        this.str_ += rule.event + ' ' + rule.role;
        if (rule.navigation) {
            this.str_ += ' ' + rule.navigation;
        }
        if (rule.output) {
            this.str_ += ' ' + rule.output;
        }
        this.str_ += '\n';
    }
    bufferClear() {
        if (this.loggingDisabled) {
            return;
        }
        this.str_ += '\nBuffer is cleared.\n';
    }
    writeError(errorMsg) {
        if (this.loggingDisabled) {
            return;
        }
        this.str_ += 'ERROR with message: ';
        this.str_ += errorMsg;
        this.str_ += '\n';
    }
}
