import '../../../../../../../../../../../../../../../../../../common/async_util.js';
import '../../../../../../../../../../../../../../../../../../common/event_generator.js';
import { InstanceChecker } from '../../../../../../../../../../../../../../../../../../common/mv2/instance_checker.js';
import { TestImportManager } from '../../../../../../../../../../../../../../../../../../common/testing/test_import_manager.js';
import { AutomationPredicate } from '../../../../../../../../../../../../../../../../../../common/automation_predicate.js';
import { AutomationUtil } from '../../../../../../../../../../../../../../../../../../common/automation_util.js';
import { constants } from '../../../../../../../../../../../../../../../../../../common/constants.js';
import { Flags, FlagName } from '../../../../../../../../../../../../../../../../../../common/flags.js';
import { NodeNavigationUtils } from '../../../../../../../../../../../../../../../../../../common/node_navigation_utils.js';
import { NodeUtils } from '../../../../../../../../../../../../../../../../../../common/node_utils.js';
import { ParagraphUtils } from '../../../../../../../../../../../../../../../../../../common/paragraph_utils.js';
import { WordUtils } from '../../../../../../../../../../../../../../../../../../common/word_utils.js';
import { RectUtil } from '../../../../../../../../../../../../../../../../../../common/rect_util.js';
import { KeyCode } from '../../../../../../../../../../../../../../../../../../common/key_code.js';

// Copyright 2020 The Chromium Authors
// Use of this source code is governed by a BSD-style license that can be
// found in the LICENSE file.
var SelectToSpeakConstants;
(function (SelectToSpeakConstants) {
    SelectToSpeakConstants.SEARCH_KEY_CODE = KeyCode.SEARCH;
    SelectToSpeakConstants.CONTROL_KEY_CODE = KeyCode.CONTROL;
    SelectToSpeakConstants.READ_SELECTION_KEY_CODE = KeyCode.S;
    /**
     * How often (in ms) to check that the currently spoken node is
     * still valid and in the same position. Decreasing this will make
     * STS seem more reactive to page changes but decreasing it too much
     * could cause performance issues.
     */
    SelectToSpeakConstants.NODE_STATE_TEST_INTERVAL_MS = 500;
    /**
     * Max size in pixels for a region selection to be considered a paragraph
     * selection vs a selection of specific nodes. Generally paragraph
     * selection is a single click (size 0), though allow for a little
     * jitter.
     */
    SelectToSpeakConstants.PARAGRAPH_SELECTION_MAX_SIZE = 5;
})(SelectToSpeakConstants || (SelectToSpeakConstants = {}));
TestImportManager.exportForTesting(['SelectToSpeakConstants', SelectToSpeakConstants]);

// Copyright 2018 The Chromium Authors
// Use of this source code is governed by a BSD-style license that can be
// found in the LICENSE file.
/**
 * Class to handle user-input, from mouse, keyboard, and copy-paste events.
 */
class InputHandler {
    callbacks_;
    didTrackMouse_;
    isSearchKeyDown_;
    isSelectionKeyDown_;
    keysCurrentlyDown_;
    keysPressedTogether_;
    lastClearClipboardDataTime_;
    lastReadClipboardDataTime_;
    mouseStart_;
    mouseEnd_;
    trackingMouse_;
    static kClipboardClearMaxDelayMs;
    static kClipboardReadMaxDelayMs;
    /**
     * Please keep fields in alphabetical order.
     */
    constructor(callbacks) {
        this.callbacks_ = callbacks;
        this.didTrackMouse_ = false;
        this.isSearchKeyDown_ = false;
        this.isSelectionKeyDown_ = false;
        this.keysCurrentlyDown_ = new Set();
        /**
         * All of the keys pressed since the last time 0 keys were pressed.
         */
        this.keysPressedTogether_ = new Set();
        /**
         * The timestamp at which the last clipboard data clear was requested.
         * Used to make sure we don't clear the clipboard on a user's request,
         * but only after the clipboard was used to read selected text.
         */
        this.lastClearClipboardDataTime_ = new Date(0);
        /**
         * The timestamp at which clipboard data read was requested by the user
         * doing a "read selection" keystroke on a Google Docs app. If a
         * clipboard change event comes in within kClipboardReadMaxDelayMs,
         * Select-to-Speak will read that text out loud.
         */
        this.lastReadClipboardDataTime_ = new Date(0);
        this.mouseStart_ = { x: 0, y: 0 };
        this.mouseEnd_ = { x: 0, y: 0 };
        this.trackingMouse_ = false;
    }
    clearClipboard_() {
        this.lastClearClipboardDataTime_ = new Date();
        document.execCommand('copy');
    }
    onClipboardCopy_(evt) {
        if (new Date().getTime() - this.lastClearClipboardDataTime_.getTime() <
            InputHandler.kClipboardClearMaxDelayMs) {
            // onClipboardPaste has just completed reading the clipboard for speech.
            // This is used to clear the clipboard.
            // @ts-ignore: TODO(b/270623046): clipboardData can be null.
            evt.clipboardData.setData('text/plain', '');
            evt.preventDefault();
            this.lastClearClipboardDataTime_ = new Date(0);
        }
    }
    onClipboardDataChanged_() {
        if (new Date().getTime() - this.lastReadClipboardDataTime_.getTime() <
            InputHandler.kClipboardReadMaxDelayMs) {
            // The data has changed, and we are ready to read it.
            // Get it using a paste.
            document.execCommand('paste');
        }
    }
    onClipboardPaste_(evt) {
        if (new Date().getTime() - this.lastReadClipboardDataTime_.getTime() <
            InputHandler.kClipboardReadMaxDelayMs) {
            // Read the current clipboard data.
            evt.preventDefault();
            // @ts-ignore: TODO(b/270623046): clipboardData can be null.
            this.callbacks_.onTextReceived(evt.clipboardData.getData('text/plain'));
            this.lastReadClipboardDataTime_ = new Date(0);
            // Clear the clipboard data by copying nothing (the current document).
            // Do this in a timeout to avoid a recursive warning per
            // https://crbug.com/363288.
            setTimeout(() => this.clearClipboard_(), 0);
        }
    }
    /**
     * Set up event listeners for mouse and keyboard events. These are
     * forwarded to us from the SelectToSpeakEventHandler so they should
     * be interpreted as global events on the whole screen, not local to
     * any particular window.
     */
    setUpEventListeners() {
        chrome.clipboard.onClipboardDataChanged.addListener(() => this.onClipboardDataChanged_());
        document.addEventListener('paste', evt => this.onClipboardPaste_(evt));
        document.addEventListener('copy', evt => this.onClipboardCopy_(evt));
        chrome.accessibilityPrivate.onSelectToSpeakKeysPressedChanged.addListener((keysPressed) => {
            this.onKeysPressedChanged(new Set(keysPressed));
        });
        chrome.accessibilityPrivate.onSelectToSpeakMouseChanged.addListener((eventType, mouseX, mouseY) => {
            this.onMouseEvent(eventType, mouseX, mouseY);
        });
    }
    /**
     * Change whether or not we are tracking the mouse.
     * @param tracking True if we should start tracking the mouse, false
     *     otherwise.
     */
    setTrackingMouse(tracking) {
        this.trackingMouse_ = tracking;
    }
    /**
     * Gets the rect that has been drawn by clicking and dragging the mouse.
     */
    getMouseRect() {
        return RectUtil.rectFromPoints(this.mouseStart_.x, this.mouseStart_.y, this.mouseEnd_.x, this.mouseEnd_.y);
    }
    /**
     * Sets the date at which we last wanted the clipboard data to be read.
     */
    onRequestReadClipboardData() {
        this.lastReadClipboardDataTime_ = new Date();
    }
    /**
     * Called when the mouse is pressed, released or moved and the user is
     * in a mode where select-to-speak is capturing mouse events (for example
     * holding down Search).
     * Visible for testing.
     *
     * @param type The event type.
     * @param mouseX The mouse x coordinate in global screen
     *     coordinates.
     * @param mouseY The mouse y coordinate in global screen
     *     coordinates.
     * Visible for testing.
     */
    onMouseEvent(type, mouseX, mouseY) {
        if (type === chrome.accessibilityPrivate.SyntheticMouseEventType.PRESS) {
            this.onMouseDown_(mouseX, mouseY);
        }
        else if (type === chrome.accessibilityPrivate.SyntheticMouseEventType.RELEASE) {
            this.onMouseUp_(mouseX, mouseY);
        }
        else {
            this.onMouseMove_(mouseX, mouseY);
        }
    }
    /**
     * Called when the mouse is pressed and the user is in a
     * mode where select-to-speak is capturing mouse events (for example
     * holding down Search).
     * @param mouseX The mouse x coordinate in global screen
     *     coordinates.
     * @param mouseY The mouse y coordinate in global screen
     *     coordinates.
     */
    onMouseDown_(mouseX, mouseY) {
        // If the user hasn't clicked 'search', or if they are currently
        // trying to highlight a selection, don't track the mouse.
        if (this.callbacks_.canStartSelecting() &&
            (!this.isSearchKeyDown_ || this.isSelectionKeyDown_)) {
            return false;
        }
        this.callbacks_.onSelectingStateChanged(true /* is selecting */, mouseX, mouseY);
        this.trackingMouse_ = true;
        this.didTrackMouse_ = true;
        this.mouseStart_ = { x: mouseX, y: mouseY };
        this.onMouseMove_(mouseX, mouseY);
        return false;
    }
    /**
     * Called when the mouse is moved or dragged and the user is in a
     * mode where select-to-speak is capturing mouse events (for example
     * holding down Search).
     * @param mouseX The mouse x coordinate in global screen
     *     coordinates.
     * @param mouseY The mouse y coordinate in global screen
     *     coordinates.
     */
    onMouseMove_(mouseX, mouseY) {
        if (!this.trackingMouse_) {
            return;
        }
        const rect = RectUtil.rectFromPoints(this.mouseStart_.x, this.mouseStart_.y, mouseX, mouseY);
        this.callbacks_.onSelectionChanged(rect);
    }
    /**
     * Called when the mouse is released and the user is in a
     * mode where select-to-speak is capturing mouse events (for example
     * holding down Search).
     * @param mouseX The mouse x coordinate in global screen
     *     coordinates.
     * @param mouseY The mouse y coordinate in global screen
     *     coordinates.
     */
    onMouseUp_(mouseX, mouseY) {
        if (!this.trackingMouse_) {
            return;
        }
        this.onMouseMove_(mouseX, mouseY);
        this.trackingMouse_ = false;
        if (!this.keysCurrentlyDown_.has(SelectToSpeakConstants.SEARCH_KEY_CODE)) {
            // This is only needed to cancel something started with the search key.
            this.didTrackMouse_ = false;
        }
        this.mouseEnd_ = { x: mouseX, y: mouseY };
        const ctrX = Math.floor((this.mouseStart_.x + this.mouseEnd_.x) / 2);
        const ctrY = Math.floor((this.mouseStart_.y + this.mouseEnd_.y) / 2);
        this.callbacks_.onSelectingStateChanged(false /* is no longer selecting */, ctrX, ctrY);
    }
    /**
     * Visible for testing.
     */
    onKeysPressedChanged(keysCurrentlyPressed) {
        if (keysCurrentlyPressed.size > this.keysCurrentlyDown_.size) {
            // If a key was pressed.
            for (const key of keysCurrentlyPressed) {
                // Union with keysPressedTogether_ to track all the keys that have been
                // pressed.
                this.keysPressedTogether_.add(key);
            }
            if (this.keysPressedTogether_.size === 1 &&
                keysCurrentlyPressed.has(SelectToSpeakConstants.SEARCH_KEY_CODE)) {
                this.isSearchKeyDown_ = true;
            }
            else if (this.isSearchKeyDown_ && keysCurrentlyPressed.size === 2 &&
                keysCurrentlyPressed.has(SelectToSpeakConstants.READ_SELECTION_KEY_CODE) &&
                !this.trackingMouse_) {
                // Only go into selection mode if we aren't already tracking the mouse.
                this.isSelectionKeyDown_ = true;
            }
            else if (!this.trackingMouse_) {
                // Some other key was pressed.
                this.isSearchKeyDown_ = false;
            }
        }
        else {
            // If a key was released.
            const searchKeyReleased = this.keysCurrentlyDown_.has(SelectToSpeakConstants.SEARCH_KEY_CODE) &&
                !keysCurrentlyPressed.has(SelectToSpeakConstants.SEARCH_KEY_CODE);
            const ctrlKeyReleased = this.keysCurrentlyDown_.has(SelectToSpeakConstants.CONTROL_KEY_CODE) &&
                !keysCurrentlyPressed.has(SelectToSpeakConstants.CONTROL_KEY_CODE);
            const speakSelectionKeyReleased = this.keysCurrentlyDown_.has(SelectToSpeakConstants.READ_SELECTION_KEY_CODE) &&
                !keysCurrentlyPressed.has(SelectToSpeakConstants.READ_SELECTION_KEY_CODE);
            if (speakSelectionKeyReleased) {
                if (this.isSelectionKeyDown_ && this.keysPressedTogether_.size === 2 &&
                    this.keysPressedTogether_.has(SelectToSpeakConstants.SEARCH_KEY_CODE)) {
                    this.callbacks_.onKeystrokeSelection();
                }
                this.isSelectionKeyDown_ = false;
            }
            else if (searchKeyReleased) {
                // Search key released.
                this.isSearchKeyDown_ = false;
                // If we were in the middle of tracking the mouse, cancel it.
                if (this.trackingMouse_) {
                    this.trackingMouse_ = false;
                    this.callbacks_.onRequestCancel();
                }
            }
            // Stop speech when the user taps and releases Control or Search
            // without using the mouse or pressing any other keys along the way.
            if (!this.didTrackMouse_ && (ctrlKeyReleased || searchKeyReleased) &&
                this.keysPressedTogether_.size === 1) {
                this.trackingMouse_ = false;
                this.callbacks_.onRequestCancel();
            }
            // We don't remove from keysPressedTogether_ because it tracks all the
            // keys which were pressed since pressing started.
        }
        // Reset our state with the Chrome OS key state. This ensures that even if
        // we miss a key event (may happen during login/logout/screensaver?) we
        // quickly get back to the correct state.
        this.keysCurrentlyDown_ = keysCurrentlyPressed;
        if (this.keysCurrentlyDown_.size === 0) {
            this.didTrackMouse_ = false;
            this.keysPressedTogether_.clear();
        }
    }
}
// Number of milliseconds to wait after requesting a clipboard read
// before clipboard change and paste events are ignored.
InputHandler.kClipboardReadMaxDelayMs = 1000;
// Number of milliseconds to wait after requesting a clipboard copy
// before clipboard copy events are ignored, used to clear the clipboard
// after reading data in a paste event.
InputHandler.kClipboardClearMaxDelayMs = 500;

// Copyright 2018 The Chromium Authors
// Use of this source code is governed by a BSD-style license that can be
// found in the LICENSE file.
/**
 * Manages getting and storing user preferences.
 */
class PrefsManager {
    /** Please keep fields in alphabetical order. */
    backgroundShadingEnabled_ = false;
    color_ = '#da36e8';
    /**
     * Whether to allow enhanced network voices in Select-to-Speak. Unlike
     * |this.enhancedNetworkVoicesEnabled_|, which represents the user's
     * preference, |this.enhancedNetworkVoicesAllowed_| is set by admin via
     * policy. |this.enhancedNetworkVoicesAllowed_| does not override
     * |this.enhancedNetworkVoicesEnabled_| but changes
     * this.enhancedNetworkVoicesEnabled().
     */
    enhancedNetworkVoicesAllowed_ = true;
    /**
     * A pref indicating whether the user enables the network voices. The pref
     * is synced to local storage as "enhancedNetworkVoices". Use
     * this.enhancedNetworkVoicesEnabled() to refer whether to enable the
     * network voices instead of using this pref directly.
     */
    enhancedNetworkVoicesEnabled_ = false;
    enhancedVoiceName_ = PrefsManager.DEFAULT_NETWORK_VOICE;
    enhancedVoicesDialogShown_ = false;
    extensionForVoice_ = new Map();
    highlightColor_ = '#5e9bff';
    migrationInProgress_ = false;
    navigationControlsEnabled_ = true;
    speechRate_ = 1.0;
    validVoiceNames_ = new Set();
    voiceNameFromLocale_ = null;
    voiceNameFromPrefs_ = null;
    wordHighlight_ = true;
    /** TODO(crbug.com/950391): Ask UX about the default value here. */
    voiceSwitching_ = false;
    /**
     * Used by tests to wait for settings changes to be propagated.
     */
    updateSettingsPrefsCallbackForTest_ = null;
    constructor() { }
    /**
     * Get the list of TTS voices, and set the default voice if not already set.
     */
    updateDefaultVoice_() {
        let uiLocale = chrome.i18n.getMessage('@@ui_locale');
        uiLocale = uiLocale.replace('_', '-').toLowerCase();
        chrome.tts.getVoices(voices => {
            this.validVoiceNames_ = new Set();
            if (voices.length === 0) {
                return;
            }
            voices.forEach(voice => {
                // TODO(b/270623046): voice.eventTypes may be undefined.
                if (!voice.eventTypes.includes(chrome.tts.EventType.START) ||
                    !voice.eventTypes.includes(chrome.tts.EventType.END) ||
                    !voice.eventTypes.includes(chrome.tts.EventType.WORD) ||
                    !voice.eventTypes.includes(chrome.tts.EventType.CANCELLED)) {
                    return;
                }
                if (voice.voiceName) {
                    this.extensionForVoice_.set(voice.voiceName, voice.extensionId || '');
                    if ((voice.extensionId !== PrefsManager.ENHANCED_TTS_EXTENSION_ID) &&
                        !voice.remote) {
                        // Don't consider network voices when computing default.
                        this.validVoiceNames_.add(voice.voiceName);
                    }
                }
            });
            voices.sort(function (a, b) {
                function score(voice) {
                    if (voice.lang === undefined) {
                        return -1;
                    }
                    const lang = voice.lang.toLowerCase();
                    let s = 0;
                    if (lang === uiLocale) {
                        s += 2;
                    }
                    if (lang.substr(0, 2) === uiLocale.substr(0, 2)) {
                        s += 1;
                    }
                    return s;
                }
                return score(b) - score(a);
            });
            const firstVoiceName = voices[0].voiceName;
            if (firstVoiceName) {
                this.voiceNameFromLocale_ = firstVoiceName;
            }
        });
    }
    /**
     * Migrates Select-to-Speak rate and pitch settings to global Text-to-Speech
     * settings. This is a one-time migration that happens on upgrade to M70.
     * See http://crbug.com/866550.
     */
    migrateToGlobalTtsSettings_(rateStr, pitchStr) {
        if (this.migrationInProgress_) {
            return;
        }
        this.migrationInProgress_ = true;
        let stsRate = PrefsManager.DEFAULT_RATE;
        let stsPitch = PrefsManager.DEFAULT_PITCH;
        let globalRate = PrefsManager.DEFAULT_RATE;
        let globalPitch = PrefsManager.DEFAULT_PITCH;
        if (rateStr !== undefined) {
            stsRate = parseFloat(rateStr);
        }
        if (pitchStr !== undefined) {
            stsPitch = parseFloat(pitchStr);
        }
        // Get global prefs using promises so that we can receive both pitch and
        // rate before doing migration logic.
        const getPrefsPromises = [];
        getPrefsPromises.push(new Promise((resolve, reject) => {
            chrome.settingsPrivate.getPref('settings.tts.speech_rate', pref => {
                if (pref === undefined) {
                    reject();
                }
                globalRate = pref.value;
                resolve();
            });
        }));
        getPrefsPromises.push(new Promise((resolve, reject) => {
            chrome.settingsPrivate.getPref('settings.tts.speech_pitch', pref => {
                if (pref === undefined) {
                    reject();
                }
                globalPitch = pref.value;
                resolve();
            });
        }));
        Promise.all(getPrefsPromises)
            .then(() => {
            const stsOptionsModified = stsRate !== PrefsManager.DEFAULT_RATE ||
                stsPitch !== PrefsManager.DEFAULT_PITCH;
            const globalOptionsModified = globalRate !== PrefsManager.DEFAULT_RATE ||
                globalPitch !== PrefsManager.DEFAULT_PITCH;
            const optionsEqual = stsRate === globalRate && stsPitch === globalPitch;
            if (optionsEqual) {
                // No need to write global prefs if all the prefs are the same
                // as defaults. Just remove STS rate and pitch.
                this.onTtsSettingsMigrationSuccess_();
                return;
            }
            if (stsOptionsModified && !globalOptionsModified) {
                // Set global prefs using promises so we can set both rate and
                // pitch successfully before removing the preferences from
                // chrome.storage.sync.
                const setPrefsPromises = [];
                setPrefsPromises.push(new Promise((resolve, reject) => {
                    chrome.settingsPrivate.setPref('settings.tts.speech_rate', stsRate, '' /* unused, see crbug.com/866161 */, success => {
                        if (success) {
                            resolve();
                        }
                        else {
                            reject();
                        }
                    });
                }));
                setPrefsPromises.push(new Promise((resolve, reject) => {
                    chrome.settingsPrivate.setPref('settings.tts.speech_pitch', stsPitch, '' /* unused, see crbug.com/866161 */, success => {
                        if (success) {
                            resolve();
                        }
                        else {
                            reject();
                        }
                    });
                }));
                Promise.all(setPrefsPromises)
                    .then(() => this.onTtsSettingsMigrationSuccess_(), error => {
                    console.log(error);
                    this.migrationInProgress_ = false;
                });
            }
            else if (globalOptionsModified) {
                // Global options were already modified, so STS will use global
                // settings regardless of whether STS was modified yet or not.
                this.onTtsSettingsMigrationSuccess_();
            }
        }, error => {
            console.log(error);
            this.migrationInProgress_ = false;
        });
    }
    /**
     * When TTS settings are successfully migrated, removes rate and pitch from
     * chrome.storage.sync.
     */
    onTtsSettingsMigrationSuccess_() {
        chrome.storage.sync.remove('rate');
        chrome.storage.sync.remove('pitch');
        this.migrationInProgress_ = false;
    }
    /**
     * Loads prefs and policy from chrome.settingsPrivate.
     */
    updateSettingsPrefs_(prefs) {
        for (const pref of prefs) {
            switch (pref.key) {
                case PrefsManager.VOICE_NAME_KEY:
                    this.voiceNameFromPrefs_ = pref.value;
                    break;
                case PrefsManager.SPEECH_RATE_KEY:
                    this.speechRate_ = pref.value;
                    break;
                case PrefsManager.WORD_HIGHLIGHT_KEY:
                    this.wordHighlight_ = pref.value;
                    break;
                case PrefsManager.HIGHLIGHT_COLOR_KEY:
                    this.highlightColor_ = pref.value;
                    break;
                case PrefsManager.BACKGROUND_SHADING_KEY:
                    this.backgroundShadingEnabled_ = pref.value;
                    break;
                case PrefsManager.NAVIGATION_CONTROLS_KEY:
                    this.navigationControlsEnabled_ = pref.value;
                    break;
                case PrefsManager.ENHANCED_NETWORK_VOICES_KEY:
                    this.enhancedNetworkVoicesEnabled_ = pref.value;
                    break;
                case PrefsManager.ENHANCED_VOICES_DIALOG_SHOWN_KEY:
                    this.enhancedVoicesDialogShown_ = pref.value;
                    break;
                case PrefsManager.ENHANCED_VOICE_NAME_KEY:
                    this.enhancedVoiceName_ = pref.value;
                    break;
                case PrefsManager.ENHANCED_VOICES_POLICY_KEY:
                    this.enhancedNetworkVoicesAllowed_ = pref.value;
                    break;
                case PrefsManager.VOICE_SWITCHING_KEY:
                    this.voiceSwitching_ = pref.value;
                    break;
            }
        }
        if (this.updateSettingsPrefsCallbackForTest_) {
            this.updateSettingsPrefsCallbackForTest_();
        }
    }
    /**
     * Migrates prefs from chrome.storage to Chrome settings prefs. This will
     * enable us to move Select-to-speak options into the Chrome OS Settings app.
     * This should only occur once per pref, as we remove the chrome.storage pref
     * after we copy it over.
     */
    migrateStorageToSettingsPref_(storagePrefName, settingsPrefName, value) {
        chrome.settingsPrivate.setPref(settingsPrefName, value);
        chrome.storage.sync.remove(storagePrefName);
    }
    /**
     * Loads prefs from chrome.storage and sets values in settings prefs if
     * necessary.
     */
    async updateStoragePrefs_() {
        const prefs = await new Promise(resolve => chrome.storage.sync.get([
            'voice',
            'rate',
            'pitch',
            'wordHighlight',
            'highlightColor',
            'backgroundShading',
            'navigationControls',
            'enhancedNetworkVoices',
            'enhancedVoicesDialogShown',
            'enhancedVoiceName',
            'voiceSwitching',
        ], resolve));
        if (prefs['voice']) {
            this.voiceNameFromPrefs_ = prefs['voice'];
            this.migrateStorageToSettingsPref_('voice', PrefsManager.VOICE_NAME_KEY, this.voiceNameFromPrefs_);
        }
        if (prefs['wordHighlight'] !== undefined) {
            this.wordHighlight_ = prefs['wordHighlight'];
            this.migrateStorageToSettingsPref_('wordHighlight', PrefsManager.WORD_HIGHLIGHT_KEY, this.wordHighlight_);
        }
        if (prefs['highlightColor']) {
            this.highlightColor_ = prefs['highlightColor'];
            this.migrateStorageToSettingsPref_('highlightColor', PrefsManager.HIGHLIGHT_COLOR_KEY, this.highlightColor_);
        }
        if (prefs['backgroundShading'] !== undefined) {
            this.backgroundShadingEnabled_ = prefs['backgroundShading'];
            this.migrateStorageToSettingsPref_('backgroundShading', PrefsManager.BACKGROUND_SHADING_KEY, this.backgroundShadingEnabled_);
        }
        if (prefs['navigationControls'] !== undefined) {
            this.navigationControlsEnabled_ = prefs['navigationControls'];
            this.migrateStorageToSettingsPref_('navigationControls', PrefsManager.NAVIGATION_CONTROLS_KEY, this.navigationControlsEnabled_);
        }
        if (prefs['enhancedNetworkVoices'] !== undefined) {
            this.enhancedNetworkVoicesEnabled_ = prefs['enhancedNetworkVoices'];
            this.migrateStorageToSettingsPref_('enhancedNetworkVoices', PrefsManager.ENHANCED_NETWORK_VOICES_KEY, this.enhancedNetworkVoicesEnabled_);
        }
        if (prefs['enhancedVoicesDialogShown'] !== undefined) {
            this.enhancedVoicesDialogShown_ = prefs['enhancedVoicesDialogShown'];
            this.migrateStorageToSettingsPref_('enhancedVoicesDialogShown', PrefsManager.ENHANCED_VOICES_DIALOG_SHOWN_KEY, this.enhancedVoicesDialogShown_);
        }
        if (prefs['enhancedVoiceName'] !== undefined) {
            this.enhancedVoiceName_ = prefs['enhancedVoiceName'];
            this.migrateStorageToSettingsPref_('enhancedVoiceName', PrefsManager.ENHANCED_VOICE_NAME_KEY, this.enhancedVoiceName_);
        }
        if (prefs['voiceSwitching'] !== undefined) {
            this.voiceSwitching_ = prefs['voiceSwitching'];
            this.migrateStorageToSettingsPref_('voiceSwitching', PrefsManager.VOICE_SWITCHING_KEY, this.voiceSwitching_);
        }
        if (prefs['rate'] && prefs['pitch']) {
            // Removes 'rate' and 'pitch' prefs after migrating data to global
            // TTS settings if appropriate.
            this.migrateToGlobalTtsSettings_(prefs['rate'], prefs['pitch']);
        }
    }
    /**
     * Loads prefs and policy from chrome.storage and chrome.settingsPrivate,
     * sets default values if necessary, and registers a listener to update prefs
     * and policy when they change.
     */
    async initPreferences() {
        // Migrate from storage prefs if necessary.
        await this.updateStoragePrefs_();
        // Initialize prefs from settings.
        const settingsPrefs = await new Promise(resolve => chrome.settingsPrivate.getAllPrefs(resolve));
        this.updateSettingsPrefs_(settingsPrefs);
        chrome.settingsPrivate.onPrefsChanged.addListener(prefs => this.updateSettingsPrefs_(prefs));
        chrome.storage.onChanged.addListener(() => this.updateStoragePrefs_());
        this.updateDefaultVoice_();
        chrome.tts.onVoicesChanged.addListener(() => {
            this.updateDefaultVoice_();
        });
    }
    /**
     * Get the voice name of the user's preferred local voice.
     * @return Name of preferred local voice.
     */
    getLocalVoice() {
        // To use the default (system) voice: don't specify options['voiceName'].
        if (this.voiceNameFromPrefs_ === PrefsManager.SYSTEM_VOICE) {
            return undefined;
        }
        // Pick the voice name from prefs first, or the one that matches
        // the locale next, but don't pick a voice that isn't currently
        // loaded. If no voices are found, leave the voiceName option
        // unset to let the browser try to route the speech request
        // anyway if possible.
        if (this.voiceNameFromPrefs_ &&
            this.validVoiceNames_.has(this.voiceNameFromPrefs_)) {
            return this.voiceNameFromPrefs_;
        }
        else if (this.voiceNameFromLocale_ &&
            this.validVoiceNames_.has(this.voiceNameFromLocale_)) {
            return this.voiceNameFromLocale_;
        }
        return undefined;
    }
    /**
     * Generates the basic speech options for Select-to-Speak based on user
     * preferences. Call for each chrome.tts.speak.
     */
    getSpeechOptions(voiceSwitchingData) {
        const options = {};
        const data = voiceSwitchingData || {
            language: undefined,
            useVoiceSwitching: false,
        };
        const useEnhancedVoices = this.enhancedNetworkVoicesEnabled() && navigator.onLine;
        if (useEnhancedVoices) {
            options['voiceName'] = this.enhancedVoiceName_;
        }
        else {
            const useVoiceSwitching = data.useVoiceSwitching;
            const language = data.language;
            // If `useVoiceSwitching` is true, then we should omit `voiceName` from
            // options and let the TTS engine pick the right voice for the language.
            const localVoice = useVoiceSwitching ? undefined : this.getLocalVoice();
            if (localVoice !== undefined) {
                options['voiceName'] = localVoice;
            }
            if (language !== undefined) {
                options['lang'] = language;
            }
        }
        return options;
    }
    /**
     * Returns extension ID of the TTS engine for given voice name.
     * @param voiceName Voice name specified in TTS options
     * @return extension ID of TTS engine
     */
    ttsExtensionForVoice(voiceName) {
        return this.extensionForVoice_.get(voiceName) || '';
    }
    /**
     * Checks if the voice is an enhanced network TTS voice.
     * @returns {boolean} True if the voice is an enhanced network TTS voice.
     */
    isNetworkVoice(voiceName) {
        return this.ttsExtensionForVoice(voiceName) ===
            PrefsManager.ENHANCED_TTS_EXTENSION_ID;
    }
    /**
     * Gets the user's word highlighting enabled preference.
     * @return True if word highlighting is enabled.
     */
    wordHighlightingEnabled() {
        return this.wordHighlight_;
    }
    /**
     * Gets the user's word highlighting color preference.
     * @return Highlight color.
     */
    highlightColor() {
        return this.highlightColor_;
    }
    /**
     * Gets the focus ring color. This is not currently a user preference but it
     * could be in the future; stored here for similarity to highlight color.
     * @return Highlight color.
     */
    focusRingColor() {
        return this.color_;
    }
    /**
     * Gets the user's focus ring background color. If the user disabled greying
     * out the background, alpha will be set to fully transparent.
     * @return True if the background shade should be drawn.
     */
    backgroundShadingEnabled() {
        return this.backgroundShadingEnabled_;
    }
    /**
     * Gets the user's preference for showing navigation controls that allow them
     * to navigate to next/previous sentences, paragraphs, and more.
     * @return True if navigation controls should be shown when STS is
     *     active.
     */
    navigationControlsEnabled() {
        return this.navigationControlsEnabled_;
    }
    /**
     * Gets the user's preference for speech rate.
     * @return Current TTS speech rate.
     */
    speechRate() {
        return this.speechRate_;
    }
    /**
     * Gets the user's preference for whether enhanced network TTS voices are
     * enabled. Always returns false if the policy disallows the feature.
     * @return True if enhanced TTS voices are enabled.
     */
    enhancedNetworkVoicesEnabled() {
        return this.enhancedNetworkVoicesAllowed_ ?
            this.enhancedNetworkVoicesEnabled_ :
            false;
    }
    /**
     * Gets the admin's policy for whether enhanced network TTS voices are
     * allowed.
     * @return True if enhanced TTS voices are allowed.
     */
    enhancedNetworkVoicesAllowed() {
        return this.enhancedNetworkVoicesAllowed_;
    }
    /**
     * Gets whether the initial popup authorizing enhanced network voices has been
     * shown to the user or not.
     *
     * @returns True if the initial popup dialog has been shown already.
     */
    enhancedVoicesDialogShown() {
        return this.enhancedVoicesDialogShown_;
    }
    /**
     * Sets whether enhanced network voices are enabled or not from initial popup.
     * @param enabled Specifies if the user enabled enhanced voices in
     *     the popup.
     */
    setEnhancedNetworkVoicesFromDialog(enabled) {
        if (enabled === undefined) {
            return;
        }
        this.enhancedNetworkVoicesEnabled_ = enabled;
        chrome.settingsPrivate.setPref(PrefsManager.ENHANCED_NETWORK_VOICES_KEY, this.enhancedNetworkVoicesEnabled_);
        this.enhancedVoicesDialogShown_ = true;
        chrome.settingsPrivate.setPref(PrefsManager.ENHANCED_VOICES_DIALOG_SHOWN_KEY, this.enhancedVoicesDialogShown_);
        if (!this.enhancedNetworkVoicesAllowed_) {
            console.warn('Network voices dialog was shown when the policy disallows it.');
        }
    }
    /**
     * Gets the user's preference for whether automatic voice switching between
     * languages is enabled.
     */
    voiceSwitchingEnabled() {
        return this.voiceSwitching_;
    }
}
(function (PrefsManager) {
    /**
     * Constant used as the value for a menu option representing the current
     * device language.
     */
    PrefsManager.USE_DEVICE_LANGUAGE = 'select_to_speak_device_language';
    /**
     * Constant representing the system TTS voice.
     */
    PrefsManager.SYSTEM_VOICE = 'select_to_speak_system_voice';
    /**
     * Constant representing the voice name for the default (server-selected)
     * network TTS voice.
     */
    PrefsManager.DEFAULT_NETWORK_VOICE = 'default-wavenet';
    /**
     * Extension ID of the enhanced network TTS voices extension.
     */
    PrefsManager.ENHANCED_TTS_EXTENSION_ID = 'jacnkoglebceckolkoapelihnglgaicd';
    /**
     * Extension ID of the Google TTS voices extension.
     */
    PrefsManager.GOOGLE_TTS_EXTENSION_ID = 'gjjabgpgjpampikjhjpfhneeoapjbjaf';
    /**
     * Extension ID of the eSpeak TTS voices extension.
     */
    PrefsManager.ESPEAK_EXTENSION_ID = 'dakbfdmgjiabojdgbiljlhgjbokobjpg';
    /**
     * Default speech rate for both Select-to-Speak and global prefs.
     */
    PrefsManager.DEFAULT_RATE = 1.0;
    /**
     * Default speech pitch for both Select-to-Speak and global prefs.
     */
    PrefsManager.DEFAULT_PITCH = 1.0;
    /**
     * Settings key for the pref for whether to shade the background area of the
     * screen (where text isn't currently being spoken).
     */
    PrefsManager.BACKGROUND_SHADING_KEY = 'settings.a11y.select_to_speak_background_shading';
    /**
     * Settings key for the pref for whether enhanced network TTS voices are
     * enabled.
     */
    PrefsManager.ENHANCED_NETWORK_VOICES_KEY = 'settings.a11y.select_to_speak_enhanced_network_voices';
    /**
     * Settings key for the pref indicating the user's enhanced voice preference.
     */
    PrefsManager.ENHANCED_VOICE_NAME_KEY = 'settings.a11y.select_to_speak_enhanced_voice_name';
    /**
     * Settings key for the pref indicating whether initial popup authorizing
     * enhanced network voices has been shown to the user or not.
     */
    PrefsManager.ENHANCED_VOICES_DIALOG_SHOWN_KEY = 'settings.a11y.select_to_speak_enhanced_voices_dialog_shown';
    /**
     * Settings key for the policy indicating whether to allow enhanced network
     * voices.
     */
    PrefsManager.ENHANCED_VOICES_POLICY_KEY = 'settings.a11y.enhanced_network_voices_in_select_to_speak_allowed';
    /**
     * Settings key for the pref indicating the user's word highlighting color
     * preference.
     */
    PrefsManager.HIGHLIGHT_COLOR_KEY = 'settings.a11y.select_to_speak_highlight_color';
    /**
     * Settings key for the pref for showing navigation controls.
     */
    PrefsManager.NAVIGATION_CONTROLS_KEY = 'settings.a11y.select_to_speak_navigation_controls';
    /**
     * Settings key for the pref indicating the user's system-wide preference TTS
     * speech rate.
     */
    PrefsManager.SPEECH_RATE_KEY = 'settings.tts.speech_rate';
    /**
     * Settings key for the pref indicating the user's voice preference.
     */
    PrefsManager.VOICE_NAME_KEY = 'settings.a11y.select_to_speak_voice_name';
    /**
     * Settings key for the pref for enabling automatic voice switching between
     * languages.
     */
    PrefsManager.VOICE_SWITCHING_KEY = 'settings.a11y.select_to_speak_voice_switching';
    /**
     * Settings key for the pref indicating whether to enable word highlighting.
     */
    PrefsManager.WORD_HIGHLIGHT_KEY = 'settings.a11y.select_to_speak_word_highlight';
})(PrefsManager || (PrefsManager = {}));
TestImportManager.exportForTesting(PrefsManager);

// Copyright 2018 The Chromium Authors
// Use of this source code is governed by a BSD-style license that can be
// found in the LICENSE file.
// Utilities for UMA metrics.
class MetricsUtils {
    /**
     * Records a cancel event if speech was in progress.
     */
    static recordCancelIfSpeaking() {
        // TODO(b/1157214): Use select-to-speak's internal state instead of TTS
        // state.
        chrome.tts.isSpeaking(speaking => {
            if (speaking) {
                MetricsUtils.recordCancelEvent_();
            }
        });
    }
    /**
     * Records an event that Select-to-Speak has begun speaking.
     * @param method The CrosSelectToSpeakStartSpeechMethod enum
     *    that reflects how this event was triggered by the user.
     * @param prefsManager A PrefsManager with the users's current
     *    preferences.
     */
    static recordStartEvent(method, prefsManager) {
        chrome.metricsPrivate.recordUserAction(MetricsUtils.START_SPEECH_METRIC);
        chrome.metricsPrivate.recordEnumerationValue(MetricsUtils.START_SPEECH_METHOD_METRIC.METRIC_NAME, method, MetricsUtils.START_SPEECH_METHOD_METRIC.EVENT_COUNT);
        chrome.metricsPrivate.recordBoolean(MetricsUtils.BACKGROUND_SHADING_METRIC, prefsManager.backgroundShadingEnabled());
        chrome.metricsPrivate.recordBoolean(MetricsUtils.NAVIGATION_CONTROLS_METRIC, prefsManager.navigationControlsEnabled());
        chrome.metricsPrivate.recordBoolean(MetricsUtils.ENHANCED_NETWORK_VOICES_METRIC, prefsManager.enhancedNetworkVoicesEnabled());
    }
    /**
     * Records an event that Select-to-Speak speech has been canceled.
     */
    static recordCancelEvent_() {
        chrome.metricsPrivate.recordUserAction(MetricsUtils.CANCEL_SPEECH_METRIC);
    }
    /**
     * Records an event that Select-to-Speak speech has been paused.
     */
    static recordPauseEvent() {
        chrome.metricsPrivate.recordUserAction(MetricsUtils.PAUSE_SPEECH_METRIC);
    }
    /**
     * Records an event that Select-to-Speak speech has been resumed from pause.
     */
    static recordResumeEvent() {
        chrome.metricsPrivate.recordUserAction(MetricsUtils.RESUME_SPEECH_METRIC);
    }
    /**
     * Records a user-requested state change event from a given state.
     */
    static recordSelectToSpeakStateChangeEvent(changeType) {
        chrome.metricsPrivate.recordEnumerationValue(MetricsUtils.STATE_CHANGE_METRIC.METRIC_NAME, changeType, MetricsUtils.STATE_CHANGE_METRIC.EVENT_COUNT);
    }
    /**
     * Converts the speech multiplier into an enum based on
     * tools/metrics/histograms/enums.xml.
     * The value returned by this function is persisted to logs. Log entries
     * should not be renumbered and numeric values should never be reused, so this
     * function should not be changed.
     * @param speechRate The current speech rate.
     * @return The current speech rate as an int for metrics.
     */
    static speechMultiplierToSparseHistogramInt_(speechRate) {
        return Math.floor(speechRate * 100);
    }
    /**
     * Records the speed override chosen by the user.
     */
    static recordSpeechRateOverrideMultiplier(rate) {
        chrome.metricsPrivate.recordSparseValue(MetricsUtils.OVERRIDE_SPEECH_RATE_MULTIPLIER_METRIC, MetricsUtils.speechMultiplierToSparseHistogramInt_(rate));
    }
    /**
     * Records the TTS engine used for a single speech utterance.
     * @param voiceName voice in TTS
     * @param prefsManager A PrefsManager with the users's current preferences.
     */
    static recordTtsEngineUsed(voiceName, prefsManager) {
        let ttsEngine;
        if (voiceName === '') {
            // No voice name passed to TTS, default voice is used
            ttsEngine = MetricsUtils.TtsEngineUsed.SYSTEM_DEFAULT;
        }
        else {
            const extensionId = prefsManager.ttsExtensionForVoice(voiceName);
            ttsEngine = MetricsUtils.ttsEngineForExtensionId_(extensionId);
        }
        chrome.metricsPrivate.recordEnumerationValue(MetricsUtils.TTS_ENGINE_USED_METRIC.METRIC_NAME, ttsEngine, MetricsUtils.TTS_ENGINE_USED_METRIC.EVENT_COUNT);
    }
    /**
     * Converts extension id of TTS voice into metric for logging.
     * @param extensionId Extension ID of TTS engine
     * @returns Enum used in TtsEngineUsed histogram.
     */
    static ttsEngineForExtensionId_(extensionId) {
        switch (extensionId) {
            case PrefsManager.ENHANCED_TTS_EXTENSION_ID:
                return MetricsUtils.TtsEngineUsed.GOOGLE_NETWORK;
            case PrefsManager.ESPEAK_EXTENSION_ID:
                return MetricsUtils.TtsEngineUsed.ESPEAK;
            case PrefsManager.GOOGLE_TTS_EXTENSION_ID:
                return MetricsUtils.TtsEngineUsed.GOOGLE_LOCAL;
            default:
                return MetricsUtils.TtsEngineUsed.UNKNOWN;
        }
    }
    /**
     * Record the number of OCRed pages in the PDF file opened with STS in Chrome
     * PDF Viewer.
     * @param numOcredPages Number of OCRed pages in the PDF file
     */
    static recordNumPdfPagesOcred(numOcredPages) {
        chrome.metricsPrivate.recordMediumCount(MetricsUtils.PDF_OCR_PAGES_OCRED_METRIC, numOcredPages);
    }
}
(function (MetricsUtils) {
    /**
     * CrosSelectToSpeakStartSpeechMethod enums.
     * These values are persisted to logs and should not be renumbered or re-used.
     * See tools/metrics/histograms/enums.xml.
     */
    let StartSpeechMethod;
    (function (StartSpeechMethod) {
        StartSpeechMethod[StartSpeechMethod["MOUSE"] = 0] = "MOUSE";
        StartSpeechMethod[StartSpeechMethod["KEYSTROKE"] = 1] = "KEYSTROKE";
        StartSpeechMethod[StartSpeechMethod["CONTEXT_MENU"] = 2] = "CONTEXT_MENU";
    })(StartSpeechMethod = MetricsUtils.StartSpeechMethod || (MetricsUtils.StartSpeechMethod = {}));
    /**
     * Constants for the start speech method metric,
     * CrosSelectToSpeakStartSpeechMethod.
     */
    MetricsUtils.START_SPEECH_METHOD_METRIC = {
        EVENT_COUNT: Object.keys(StartSpeechMethod).length,
        METRIC_NAME: 'Accessibility.CrosSelectToSpeak.StartSpeechMethod',
    };
    /**
     * CrosSelectToSpeakStateChangeEvent enums.
     * These values are persisted to logs and should not be renumbered or re-used.
     * See tools/metrics/histograms/enums.xml.
     */
    let StateChangeEvent;
    (function (StateChangeEvent) {
        StateChangeEvent[StateChangeEvent["START_SELECTION"] = 0] = "START_SELECTION";
        StateChangeEvent[StateChangeEvent["CANCEL_SPEECH"] = 1] = "CANCEL_SPEECH";
        StateChangeEvent[StateChangeEvent["CANCEL_SELECTION"] = 2] = "CANCEL_SELECTION";
    })(StateChangeEvent = MetricsUtils.StateChangeEvent || (MetricsUtils.StateChangeEvent = {}));
    /**
     * Constants for the state change metric, CrosSelectToSpeakStateChangeEvent.
     */
    MetricsUtils.STATE_CHANGE_METRIC = {
        EVENT_COUNT: Object.keys(StateChangeEvent).length,
        METRIC_NAME: 'Accessibility.CrosSelectToSpeak.StateChangeEvent',
    };
    /**
     * CrosSelectToSpeakTtsEngineUsed enums.
     * These values are persisted to logs and should not be renumbered or re-used.
     * See tools/metrics/histograms/enums.xml.
     */
    let TtsEngineUsed;
    (function (TtsEngineUsed) {
        TtsEngineUsed[TtsEngineUsed["UNKNOWN"] = 0] = "UNKNOWN";
        TtsEngineUsed[TtsEngineUsed["SYSTEM_DEFAULT"] = 1] = "SYSTEM_DEFAULT";
        TtsEngineUsed[TtsEngineUsed["ESPEAK"] = 2] = "ESPEAK";
        TtsEngineUsed[TtsEngineUsed["GOOGLE_LOCAL"] = 3] = "GOOGLE_LOCAL";
        TtsEngineUsed[TtsEngineUsed["GOOGLE_NETWORK"] = 4] = "GOOGLE_NETWORK";
    })(TtsEngineUsed = MetricsUtils.TtsEngineUsed || (MetricsUtils.TtsEngineUsed = {}));
    /**
     * Constants for the TTS engine metric, CrosSelectToSpeak.TtsEngineUsed.
     */
    MetricsUtils.TTS_ENGINE_USED_METRIC = {
        EVENT_COUNT: Object.keys(TtsEngineUsed).length,
        METRIC_NAME: 'Accessibility.CrosSelectToSpeak.TtsEngineUsed',
    };
    /**
     * The start speech metric name.
     */
    MetricsUtils.START_SPEECH_METRIC = 'Accessibility.CrosSelectToSpeak.StartSpeech';
    /**
     * The cancel speech metric name.
     */
    MetricsUtils.CANCEL_SPEECH_METRIC = 'Accessibility.CrosSelectToSpeak.CancelSpeech';
    /**
     * The pause speech metric name.
     */
    MetricsUtils.PAUSE_SPEECH_METRIC = 'Accessibility.CrosSelectToSpeak.PauseSpeech';
    /**
     * The resume speech after pausing metric name.
     */
    MetricsUtils.RESUME_SPEECH_METRIC = 'Accessibility.CrosSelectToSpeak.ResumeSpeech';
    /**
     * The background shading metric name.
     */
    MetricsUtils.BACKGROUND_SHADING_METRIC = 'Accessibility.CrosSelectToSpeak.BackgroundShading';
    /**
     * The navigation controls metric name.
     */
    MetricsUtils.NAVIGATION_CONTROLS_METRIC = 'Accessibility.CrosSelectToSpeak.NavigationControls';
    /**
     * The metric name for enhanced network TTS voices.
     */
    MetricsUtils.ENHANCED_NETWORK_VOICES_METRIC = 'Accessibility.CrosSelectToSpeak.EnhancedNetworkVoices';
    /**
     * The speech rate override histogram metric name.
     */
    MetricsUtils.OVERRIDE_SPEECH_RATE_MULTIPLIER_METRIC = 'Accessibility.CrosSelectToSpeak.OverrideSpeechRateMultiplier';
    MetricsUtils.PDF_OCR_PAGES_OCRED_METRIC = 'Accessibility.PdfOcr.CrosSelectToSpeak.PagesOcred';
})(MetricsUtils || (MetricsUtils = {}));

// Copyright 2021 The Chromium Authors
// Use of this source code is governed by a BSD-style license that can be
// found in the LICENSE file.
/**
 * The wrapper for Select-to-speak's text-to-speech features.
 */
class TtsManager {
    clientTtsOptions_;
    currentCharIndex_;
    fallbackVoice_;
    isNetworkVoice_;
    isSpeaking_;
    pauseCompleteCallback_;
    text_;
    /** Please keep fields in alphabetical order. */
    constructor() {
        /**
         * The TTS options that the client passed in.
         */
        this.clientTtsOptions_ = {};
        /**
         * The current char index to the |this.text_| indicating the current spoken
         * word. For example, if |this.text_| is "hello world" and TTS is speaking
         * the second word, the |this.currentCharIndex_| should be 6.
         */
        this.currentCharIndex_ = 0;
        /**
         * The fallback voice to use if TTS fails.
         */
        this.fallbackVoice_ = undefined;
        /**
         * Whether the last TTS request was made with a network voice.
         */
        this.isNetworkVoice_ = false;
        /**
         * Whether TTS is speaking.
         */
        this.isSpeaking_ = false;
        /**
         * Function to be called when STS finishes a pausing request.
         */
        this.pauseCompleteCallback_ = null;
        /**
         * The text currently being spoken.
         */
        this.text_ = null;
    }
    /**
     * Whether TTS is speaking. If TTS is paused, this will return false.
     */
    isSpeaking() {
        return this.isSpeaking_;
    }
    /**
     * Sets TtsManager with the parameters and starts reading |text|.
     * @param text The text to read.
     * @param ttsOptions The options for TTS.
     * @param networkVoice Whether a network voice is specified for TTS.
     * @param fallbackVoice A voice to use if to retry if TTS
     *     fails.
     */
    speak(text, ttsOptions, networkVoice, fallbackVoice) {
        // @ts-ignore: TODO(b/): Change to `enqueue`.
        if (ttsOptions.enqueued) {
            console.warn('TtsManager does not support a queue of utterances.');
            return;
        }
        this.cleanTtsState_();
        this.text_ = text;
        this.isNetworkVoice_ = networkVoice;
        this.fallbackVoice_ = fallbackVoice;
        this.startSpeakingTextWithOffset_(0, false /* resume */, ttsOptions);
    }
    /**
     * Starts reading text with |offset|.
     * @param offset The character offset into the text at which to start
     *     speaking.
     * @param resume Whether it is a resume action.
     * @param ttsOptions The options for TTS.
     */
    startSpeakingTextWithOffset_(offset, resume, ttsOptions) {
        // @ts-ignore: TODO(b/270623046): this.text_ can be null.
        const text = this.text_.slice(offset);
        const modifiedOptions = Object.assign({}, ttsOptions);
        // Saves a copy of the ttsOptions for resume.
        Object.assign(this.clientTtsOptions_, ttsOptions);
        modifiedOptions.onEvent = event => {
            switch (event.type) {
                case chrome.tts.EventType.ERROR:
                    if (this.isNetworkVoice_) {
                        // Retry with local voice. Use modifiedOptions to preserve
                        // word and character indices.
                        console.warn('Network TTS error, retrying with local voice');
                        const localOptions = Object.assign({}, modifiedOptions);
                        localOptions.voiceName = this.fallbackVoice_;
                        if (this.text_) {
                            this.speak(this.text_, localOptions, /*networkVoice=*/ false, undefined);
                        }
                    }
                    break;
                case chrome.tts.EventType.START:
                    this.isSpeaking_ = true;
                    // Find the first non-space char index in text, or 0 if the text is
                    // null or the first char is non-space.
                    this.currentCharIndex_ = (text || '').search(/\S|$/) + offset;
                    if (resume) {
                        TtsManager.sendEventToOptions(ttsOptions, {
                            type: chrome.tts.EventType.RESUME,
                            charIndex: this.currentCharIndex_,
                        });
                        break;
                    }
                    TtsManager.sendEventToOptions(ttsOptions, {
                        type: chrome.tts.EventType.START,
                        charIndex: this.currentCharIndex_,
                    });
                    break;
                case chrome.tts.EventType.END:
                    this.isSpeaking_ = false;
                    this.currentCharIndex_ = text.length + offset;
                    TtsManager.sendEventToOptions(ttsOptions, {
                        type: chrome.tts.EventType.END,
                        charIndex: this.currentCharIndex_,
                    });
                    break;
                case chrome.tts.EventType.WORD:
                    this.isSpeaking_ = true;
                    // @ts-ignore: TODO(b/270623046): event.charIndex can be undefined.
                    this.currentCharIndex_ = event.charIndex + offset;
                    TtsManager.sendEventToOptions(ttsOptions, {
                        type: chrome.tts.EventType.WORD,
                        charIndex: this.currentCharIndex_,
                        length: event.length,
                    });
                    break;
                case chrome.tts.EventType.INTERRUPTED:
                case chrome.tts.EventType.CANCELLED:
                    this.isSpeaking_ = false;
                    // Checks |this.pauseCompleteCallback_| as a proxy to see if the
                    // interrupted events are from |this.pause()|.
                    if (this.pauseCompleteCallback_) {
                        TtsManager.sendEventToOptions(ttsOptions, {
                            type: chrome.tts.EventType.PAUSE,
                            charIndex: this.currentCharIndex_,
                        });
                        this.pauseCompleteCallback_();
                        break;
                    }
                    TtsManager.sendEventToOptions(ttsOptions, event);
                    break;
                // Passes other events directly.
                default:
                    TtsManager.sendEventToOptions(ttsOptions, event);
                    break;
            }
        };
        chrome.tts.speak(text, modifiedOptions);
    }
    /**
     * Pause the TTS. The chrome.tts.pause method is not fully supported by all
     * TTS engines so we mock the logic using chrome.tts.stop. This function also
     * sets the |this.pauseCompleteCallback_|, which will be executed at the end
     * of the pause process in TTS. This enables us to execute functions when the
     * pause request is finished. For example, to navigate the next sentence, we
     * trigger pause_ and start finding the next sentence when the pause function
     * is fulfilled.
     */
    pause() {
        return new Promise(resolve => {
            this.pauseCompleteCallback_ = () => {
                this.pauseCompleteCallback_ = null;
                resolve();
            };
            chrome.tts.stop();
        });
    }
    /**
     * Resumes the TTS.
     * @param ttsOptions The options for TTS. If this is not passed,
     *     the previous options will be used.
     */
    resume(ttsOptions) {
        ttsOptions = ttsOptions || this.clientTtsOptions_;
        // If TTS is speaking now, returns immediately.
        if (this.isSpeaking_) {
            return;
        }
        // If there is no content in the remaining text, sends an error message and
        // returns early. This avoids sending 'end' events to client.
        // @ts-ignore: TODO(b/270623046): this.text_ can be null.
        if (this.text_.slice(this.currentCharIndex_).trim().length === 0) {
            TtsManager.sendEventToOptions(ttsOptions, {
                type: chrome.tts.EventType.ERROR,
                errorMessage: TtsManager.ErrorMessage.RESUME_WITH_EMPTY_CONTENT,
            });
            return;
        }
        this.startSpeakingTextWithOffset_(this.currentCharIndex_, true /* resume */, ttsOptions);
    }
    /**
     * Stops the TTS.
     */
    stop() {
        chrome.tts.stop();
    }
    cleanTtsState_() {
        this.text_ = null;
        this.clientTtsOptions_ = {};
        this.currentCharIndex_ = 0;
        this.pauseCompleteCallback_ = null;
        this.isSpeaking_ = false;
        this.isNetworkVoice_ = false;
    }
    /**
     * Sends TtsEvent to TtsOptions.
     * @param options
     * @param event
     */
    static sendEventToOptions(options, event) {
        if (options.onEvent) {
            options.onEvent(event);
            return;
        }
        console.warn('onEvent is not defined in the TtsOptions');
    }
}
(function (TtsManager) {
    (function (ErrorMessage) {
        ErrorMessage["RESUME_WITH_EMPTY_CONTENT"] = "Cannot resume with empty content.";
    })(TtsManager.ErrorMessage || (TtsManager.ErrorMessage = {}));
})(TtsManager || (TtsManager = {}));
TestImportManager.exportForTesting(TtsManager);

// Copyright 2021 The Chromium Authors
// Use of this source code is governed by a BSD-style license that can be
// found in the LICENSE file.
const EventType$1 = chrome.automation.EventType;
const FocusRingStackingOrder = chrome.accessibilityPrivate.FocusRingStackingOrder;
// This must match the name of view class that implements the SelectToSpeakTray:
// ash/system/accessibility/select_to_speak/select_to_speak_tray.h
const SELECT_TO_SPEAK_TRAY_CLASS_NAME = 'SelectToSpeakTray';
// This must match the name of view class that implements the menu view:
// ash/system/accessibility/select_to_speak/select_to_speak_menu_view.h
const SELECT_TO_SPEAK_MENU_CLASS_NAME = 'SelectToSpeakMenuView';
// This must match the name of view class that implements the speed view:
// ash/system/accessibility/select_to_speak/select_to_speak_speed_view.h
const SELECT_TO_SPEAK_SPEED_CLASS_NAME = 'SelectToSpeakSpeedView';
// This must match the name of view class that implements the bubble views:
// ash/system/tray/tray_bubble_view.h
const TRAY_BUBBLE_VIEW_CLASS_NAME = 'TrayBubbleView';
// This must match the name of view class that implements the buttons used in
// the floating panel:
// ash/system/accessibility/floating_menu_button.h
const FLOATING_MENU_BUTTON_CLASS_NAME = 'FloatingMenuButton';
// A RGBA hex string for the default background shading color, which is black at
// 40% opacity (hex 66). This should be equivalent to using
// AshColorProvider::ShieldLayerType kShield40.
const DEFAULT_BACKGROUND_SHADING_COLOR = '#0006';
/**
 * Manages user interface elements controlled by Select-to-speak, such the
 * focus ring, floating control panel, tray button, and word highlight.
 */
class UiManager {
    // TODO(b/314204374): Convert from null to undefined.
    desktop_;
    listener_;
    // TODO(b/314204374): Convert from null to undefined.
    panelButton_;
    prefsManager_;
    /**
     * Please keep fields in alphabetical order.
     */
    constructor(prefsManager, listener) {
        this.desktop_ = null;
        this.listener_ = listener;
        /**
         * Button in the floating panel, useful for restoring focus to the panel.
         */
        this.panelButton_ = null;
        this.prefsManager_ = prefsManager;
        this.init_();
    }
    init_() {
        // Cache desktop and listen to focus changes.
        chrome.automation.getDesktop(desktop => {
            this.desktop_ = desktop;
            // Listen to focus changes so we can grab the floating panel when it
            // goes into focus, so it can be used later without having to search
            // through the entire tree.
            desktop.addEventListener(EventType$1.FOCUS, evt => {
                this.onFocusChange_(evt);
            }, true);
        });
        // Listen to panel events.
        chrome.accessibilityPrivate.onSelectToSpeakPanelAction.addListener((panelAction, value) => {
            this.onPanelAction_(panelAction, value);
        });
        // Listen to event from activating tray button.
        chrome.accessibilityPrivate.onSelectToSpeakStateChangeRequested.addListener(() => {
            this.listener_.onStateChangeRequested();
        });
    }
    /**
     * Handles Select-to-speak panel action.
     * @param panelAction Action to perform.
     * @param value Optional value associated with action.
     */
    onPanelAction_(panelAction, value) {
        switch (panelAction) {
            case chrome.accessibilityPrivate.SelectToSpeakPanelAction.NEXT_PARAGRAPH:
                this.listener_.onNextParagraphRequested();
                break;
            case chrome.accessibilityPrivate.SelectToSpeakPanelAction
                .PREVIOUS_PARAGRAPH:
                this.listener_.onPreviousParagraphRequested();
                break;
            case chrome.accessibilityPrivate.SelectToSpeakPanelAction.NEXT_SENTENCE:
                this.listener_.onNextSentenceRequested();
                break;
            case chrome.accessibilityPrivate.SelectToSpeakPanelAction
                .PREVIOUS_SENTENCE:
                this.listener_.onPreviousSentenceRequested();
                break;
            case chrome.accessibilityPrivate.SelectToSpeakPanelAction.EXIT:
                this.listener_.onExitRequested();
                break;
            case chrome.accessibilityPrivate.SelectToSpeakPanelAction.PAUSE:
                this.listener_.onPauseRequested();
                break;
            case chrome.accessibilityPrivate.SelectToSpeakPanelAction.RESUME:
                this.listener_.onResumeRequested();
                break;
            case chrome.accessibilityPrivate.SelectToSpeakPanelAction.CHANGE_SPEED:
                if (!value) {
                    console.warn('Change speed request receieved with invalid value', value);
                    return;
                }
                this.listener_.onChangeSpeedRequested(value);
                break;
            default:
                console.warn('Unknown panel action received', panelAction);
        }
    }
    /**
     * Handles desktop-wide focus changes.
     */
    onFocusChange_(evt) {
        const focusedNode = evt.target;
        // As an optimization, look for the STS floating panel and store in case
        // we need to access that node at a later point (such as focusing panel).
        if (focusedNode.className !== FLOATING_MENU_BUTTON_CLASS_NAME) {
            // When panel is focused, initial focus is always on one of the buttons.
            return;
        }
        const windowParent = AutomationUtil.getFirstAncestorWithRole(focusedNode, chrome.automation.RoleType.WINDOW);
        if (windowParent &&
            windowParent.className === TRAY_BUBBLE_VIEW_CLASS_NAME &&
            windowParent.children.length === 1 &&
            windowParent.children[0].className ===
                SELECT_TO_SPEAK_MENU_CLASS_NAME) {
            this.panelButton_ = focusedNode;
        }
    }
    /**
     * Sets focus to the floating control panel, if present.
     */
    setFocusToPanel() {
        // Used cached panel node if possible to avoid expensive desktop.find().
        // Note: Checking role attribute to see if node is still valid.
        if (this.panelButton_ && this.panelButton_.role) {
            // The panel itself isn't focusable, so set focus to most recently
            // focused panel button.
            this.panelButton_.focus();
            return;
        }
        this.panelButton_ = null;
        // Fallback to more expensive method of finding panel.
        if (!this.desktop_) {
            console.error('No cached desktop object, cannot focus panel');
            return;
        }
        const menuView = this.desktop_.find({ attributes: { className: SELECT_TO_SPEAK_MENU_CLASS_NAME } });
        if (menuView !== null && menuView.parent &&
            menuView.parent.className === TRAY_BUBBLE_VIEW_CLASS_NAME) {
            // The menu view's parent is the TrayBubbleView can can be assigned focus.
            this.panelButton_ =
                menuView.find({ role: chrome.automation.RoleType.TOGGLE_BUTTON });
            this.panelButton_.focus();
        }
    }
    /**
     * Sets the focus ring to |rects|. If |drawBackground|, draws the grey focus
     * background with the alpha set in prefs. |panelVisible| determines
     * the stacking order, so focus rings do not appear on top of panel.
     */
    setFocusRings_(rects, drawBackground, panelVisible) {
        let color = '#0000'; // Fully transparent.
        if (drawBackground && this.prefsManager_.backgroundShadingEnabled()) {
            color = DEFAULT_BACKGROUND_SHADING_COLOR;
        }
        // If we're also showing a floating panel, ensure the focus ring appears
        // below the panel UI.
        const stackingOrder = panelVisible ?
            FocusRingStackingOrder.BELOW_ACCESSIBILITY_BUBBLES :
            FocusRingStackingOrder.ABOVE_ACCESSIBILITY_BUBBLES;
        chrome.accessibilityPrivate.setFocusRings([{
                rects,
                type: chrome.accessibilityPrivate.FocusType.GLOW,
                stackingOrder,
                color: this.prefsManager_.focusRingColor(),
                backgroundColor: color,
            }], chrome.accessibilityPrivate.AssistiveTechnologyType.SELECT_TO_SPEAK);
    }
    /**
     * Updates the floating control panel.
     */
    updatePanel_(showPanel, anchorRect, paused, speechRateMultiplier) {
        if (showPanel) {
            if (anchorRect === undefined || paused === undefined ||
                speechRateMultiplier === undefined) {
                console.error('Cannot display panel: missing required parameters');
                return;
            }
            // If the feature is enabled and we have a valid focus ring, flip the
            // pause and resume button according to the current STS and TTS state.
            // Also, update the location of the panel according to the focus ring.
            chrome.accessibilityPrivate.updateSelectToSpeakPanel(
            /* show= */ true, /* anchor= */ anchorRect, 
            /* isPaused= */ paused, 
            /* speed= */ speechRateMultiplier);
        }
        else {
            // Dismiss the panel if either the feature is disabled or the focus ring
            // is not valid.
            chrome.accessibilityPrivate.updateSelectToSpeakPanel(/* show= */ false);
        }
    }
    /**
     * Updates word highlight.
     * @param node Current node being spoken.
     * @param currentWord Character offsets of
     *    current word spoken within node if word highlighting is enabled.
     */
    updateHighlight_(node, 
    // TODO(b/314204374): Convert null to undefined.
    currentWord, paused) {
        if (!currentWord) {
            chrome.accessibilityPrivate.setHighlights([], this.prefsManager_.highlightColor());
            return;
        }
        // getStartCharIndexInParent is only defined for nodes with role
        // INLINE_TEXT_BOX.
        const charIndexInParent = node.role === chrome.automation.RoleType.INLINE_TEXT_BOX ?
            ParagraphUtils.getStartCharIndexInParent(node) :
            0;
        node.boundsForRange(currentWord.start - charIndexInParent, currentWord.end - charIndexInParent, bounds => {
            const highlights = bounds ? [bounds] : [];
            chrome.accessibilityPrivate.setHighlights(highlights, this.prefsManager_.highlightColor());
            if (!paused) {
                // If speech is ongoing, update the bounds. (If it was paused,
                // reading focus hasn't actually changed, so there's no need for
                // this notification).
                chrome.accessibilityPrivate.setSelectToSpeakFocus(bounds ? bounds : node.location);
            }
        });
    }
    /**
     * Renders user selection rect, in the form of a focus ring.
     */
    setSelectionRect(rect) {
        // TODO(crbug.com/40753028): Support showing two focus rings at once, in case
        // a focus ring highlighting a node group is already present.
        this.setFocusRings_([rect], false /* don't draw background */, false /* panelVisible */);
    }
    /**
     * Updates overlay UI based on current node and panel state.
     * @param nodeGroup Current node group.
     * @param node Current node being spoken.
     * @param currentWord Character offsets of
     *    current word spoken within node if word highlighting is enabled.
     */
    update(nodeGroup, node, 
    // TODO(b/314204374): Convert null to undefined.
    currentWord, panelState) {
        const { showPanel, paused, speechRateMultiplier } = panelState;
        // Show the block parent of the currently verbalized node with the
        // focus ring. If the node has no siblings in the group, highlight just
        // the one node.
        let focusRingRect;
        const currentBlockParent = nodeGroup.blockParent;
        if (currentBlockParent !== null && nodeGroup.nodes.length > 1) {
            focusRingRect = currentBlockParent.location;
        }
        else {
            focusRingRect = node.location;
        }
        this.updateHighlight_(node, currentWord, paused);
        if (focusRingRect) {
            this.setFocusRings_([focusRingRect], true /* draw background */, showPanel);
            this.updatePanel_(showPanel, focusRingRect, paused, speechRateMultiplier);
        }
        else {
            console.warn('No node location; cannot render focus ring or panel');
        }
    }
    /**
     * Clears overlay UI, hiding focus rings, panel, and word highlight.
     */
    clear() {
        this.setFocusRings_([], false /* do not draw background */, false /* panel not visible */);
        chrome.accessibilityPrivate.setHighlights([], this.prefsManager_.highlightColor());
        this.updatePanel_(false /* hide panel */);
    }
    /**
     * @return Whether given node is the Select-to-speak floating panel.
     */
    static isPanel(node) {
        if (!node) {
            return false;
        }
        // Determine if the node is part of the floating panel or the reading speed
        // selection bubble.
        return (node.className === TRAY_BUBBLE_VIEW_CLASS_NAME &&
            node.children.length === 1 &&
            (node.children[0].className === SELECT_TO_SPEAK_MENU_CLASS_NAME ||
                node.children[0].className === SELECT_TO_SPEAK_SPEED_CLASS_NAME));
    }
    /**
     * @return Whether given node is the Select-to-speak tray button.
     */
    static isTrayButton(node) {
        if (!node) {
            return false;
        }
        return AutomationUtil.getAncestors(node).find(n => {
            return n.className === SELECT_TO_SPEAK_TRAY_CLASS_NAME;
        }) !== undefined;
    }
}
TestImportManager.exportForTesting(UiManager, ['SELECT_TO_SPEAK_TRAY_CLASS_NAME', SELECT_TO_SPEAK_TRAY_CLASS_NAME]);

// Copyright 2016 The Chromium Authors
// Use of this source code is governed by a BSD-style license that can be
// found in the LICENSE file.
var EventType = chrome.automation.EventType;
var RoleType = chrome.automation.RoleType;
var SelectToSpeakState = chrome.accessibilityPrivate.SelectToSpeakState;
// Matches one of the known GSuite apps which need the clipboard to find and
// read selected text. Includes sandbox and non-sandbox versions.
const GSUITE_APP_REGEXP = /^https:\/\/docs\.(?:sandbox\.)?google\.com\/(?:(?:presentation)|(?:document)|(?:spreadsheets)|(?:drawings)|(?:scenes)){1}\//;
/**
 * Determines if a node is in one of the known Google GSuite apps that needs
 * special case treatment for speaking selected text. Not all Google GSuite
 * pages are included, because some are not known to have a problem with
 * selection: Forms is not included since it's relatively similar to any HTML
 * page, for example.
 * @param node The node to check
 * @return The root node of the GSuite app, or null if none is found.
 */
function getGSuiteAppRoot(node) {
    while (node !== undefined && node.root !== undefined) {
        if (node.root.url !== undefined && GSUITE_APP_REGEXP.exec(node.root.url)) {
            return node.root;
        }
        node = node.root.parent;
    }
    return null;
}
/**
 * Select-to-speak component extension controller.
 */
class SelectToSpeak {
    currentCharIndex_;
    currentNodeGroupIndex_;
    // TODO(b/314203187): In many places we've added a currentNodeGroupItem_!,
    // determine if this is correct or if a check should be added.
    currentNodeGroupItem_;
    currentNodeGroupItemIndex_;
    currentNodeGroups_;
    currentNodeWord_;
    desktop_;
    inputHandler_;
    intervalId_;
    nullSelectionTone_;
    onStateChangeRequestedCallbackForTest_;
    prefsManager_;
    scrollToSpokenNode_;
    speechRateMultiplier_;
    state_;
    supportsNavigationPanel_;
    ttsManager_;
    uiManager_;
    onLoadDesktopCallbackForTest_;
    readyForTestingPromise = new Promise(resolve => this.readyForTestingCallback_ = resolve);
    /** Please keep fields in alphabetical order. */
    constructor() {
        /**
         * The start char index of the word to be spoken. The index is relative
         * to the text content of the current node group.
         */
        this.currentCharIndex_ = -1;
        /**
         * The index for the node group currently being spoken in
         * |this.currentNodeGroups_|.
         */
        this.currentNodeGroupIndex_ = -1;
        /**
         * The node group item currently being spoken. A node group item is a
         * representation of the original input nodes, but may not be the same. For
         * example, an input inline text node will be represented by its static text
         * node in the node group item.
         */
        this.currentNodeGroupItem_ = null;
        /**
         * The index for the current node group item within the current node group,
         * The current node group can be accessed from |this.currentNodeGroups_|
         * using |this.currentNodeGroupIndex_|. In most cases,
         * |this.currentNodeGroupItemIndex_| can be used to get
         * |this.currentNodeGroupItem_| from the current node group. However, in
         * Gsuite, we will have node group items outside of a node group.
         */
        this.currentNodeGroupItemIndex_ = -1;
        /**
         * The node groups to be spoken. We process content into node groups and
         * pass one node group at a time to the TTS engine. Note that we do not use
         * node groups for user-selected text in Gsuite. See more details in
         * readNodesBetweenPositions_.
         */
        this.currentNodeGroups_ = [];
        /**
         * The indexes within the current node group item representing the word
         * currently being spoken. Only updated if word highlighting is enabled.
         */
        this.currentNodeWord_ = null;
        this.desktop_;
        this.inputHandler_ = null;
        /**
         * The interval ID from a call to setInterval, which is set whenever
         * speech is in progress.
         */
        this.intervalId_;
        this.nullSelectionTone_ = new Audio('earcons/null_selection.ogg');
        /**
         * Function to be called when a state change request is received from the
         * accessibilityPrivate API.
         */
        this.onStateChangeRequestedCallbackForTest_ = null;
        this.prefsManager_ = new PrefsManager();
        this.scrollToSpokenNode_ = false;
        /** Speech rate multiplier. */
        this.speechRateMultiplier_ = 1.0;
        /**
         * The current state of the SelectToSpeak extension, from
         * SelectToSpeakState.
         */
        this.state_ = SelectToSpeakState.INACTIVE;
        /**
         * Whether the current nodes support use of the navigation panel.
         */
        this.supportsNavigationPanel_ = true;
        this.ttsManager_ = new TtsManager();
        this.uiManager_ = new UiManager(this.prefsManager_, /*listener=*/ this);
        this.onLoadDesktopCallbackForTest_ = null;
        this.init_();
    }
    async init_() {
        chrome.automation.getDesktop(desktop => {
            this.desktop_ = desktop;
            // After the user selects a region of the screen, we do a hit test at
            // the center of that box using the automation API. The result of the
            // hit test is a MOUSE_RELEASED accessibility event.
            desktop.addEventListener(EventType.MOUSE_RELEASED, evt => this.onAutomationHitTest_(evt), true);
            // Chrome PDF Viewer with PDF OCR sends a layout complete event when
            // finishing extracting text from inaccessible PDF pages. The same for
            // Backlight (AKA Gallery on ChromeOS).
            desktop.addEventListener(EventType.LAYOUT_COMPLETE, evt => this.onLayoutComplete_(evt), true);
            if (this.onLoadDesktopCallbackForTest_) {
                this.onLoadDesktopCallbackForTest_();
                this.onLoadDesktopCallbackForTest_ = null;
            }
        });
        this.prefsManager_.initPreferences();
        this.runContentScripts_();
        this.setUpEventListeners_();
        await Flags.init();
        const createArgs = {
            title: chrome.i18n.getMessage('select_to_speak_listen_context_menu_option_text'),
            contexts: [chrome.contextMenus.ContextType.SELECTION],
            id: 'select_to_speak',
        };
        if (Flags.isEnabled(FlagName.MANIFEST_V3)) {
            chrome.contextMenus.onClicked.addListener(() => {
                this.getFocusedNodeAndSpeakSelectedText_();
            });
        }
        else {
            createArgs['onclick'] = () => {
                this.getFocusedNodeAndSpeakSelectedText_();
            };
        }
        // Install the context menu in the Ash browser.
        await chrome.contextMenus.create(createArgs);
        // Listen for context menu clicks from other contexts.
        chrome.accessibilityPrivate.onSelectToSpeakContextMenuClicked.addListener(() => {
            this.getFocusedNodeAndSpeakSelectedText_();
        });
        this.readyForTestingCallback_();
    }
    /**
     * Gets the node group currently being spoken.
     */
    getCurrentNodeGroup_() {
        if (this.currentNodeGroups_.length === 0) {
            return undefined;
        }
        return this.currentNodeGroups_[this.currentNodeGroupIndex_];
    }
    /**
     * Determines if navigation controls should be shown (and other related
     * functionality, such as auto-dismiss and click-to-navigate to sentence,
     * should be activated) based on feature flag and user setting.
     */
    shouldShowNavigationControls_() {
        return this.prefsManager_.navigationControlsEnabled() &&
            this.supportsNavigationPanel_;
    }
    /**
     * Read the status message under the status node in a PDF accessibility tree
     * if PDF content is still being loaded. In the loading phase, the PDF a11y
     * tree will have one child node with the banner role, which contains the
     * loading status message as follows:
     * pdfRoot
     * - banner
     * -- status
     * --- staticText: "Loading PDF"
     */
    readPdfStatusNodeIfStillLoading_(pdfRoot) {
        if (pdfRoot.role === RoleType.PDF_ROOT && pdfRoot.children.length === 1 &&
            pdfRoot.firstChild.role === RoleType.BANNER &&
            pdfRoot.firstChild.children.length === 1 &&
            pdfRoot.firstChild.firstChild.role === RoleType.STATUS &&
            pdfRoot.firstChild.firstChild.children.length === 1 &&
            pdfRoot.firstChild.firstChild.firstChild.role ===
                RoleType.STATIC_TEXT) {
            this.startSpeechQueue_([pdfRoot.firstChild.firstChild.firstChild], {
                clearFocusRing: true,
            });
            return true;
        }
        return false;
    }
    onLayoutComplete_(evt) {
        const root = evt.target;
        if (!root.url || !root.url.endsWith('.pdf')) {
            return;
        }
        // Check if it's a PDF being viewed in Backlight (AKA the Gallery App on
        // ChromeOS), or the Chrome PDF Viewer with PDF OCR in the full-page view.
        const pdfRoot = root.find({ role: RoleType.PDF_ROOT });
        if (!pdfRoot) {
            return;
        }
        this.recordOcredPagesInPdf_(pdfRoot);
    }
    /**
     * Record the number of OCRed pages in the PDF accessibility tree.
     */
    recordOcredPagesInPdf_(pdfRoot) {
        // When PDF OCR successfully extracts text from inaccessible PDF pages, PDF
        // pages with OCRed content will have the "ocred_page" class name.
        const orcedPages = pdfRoot.findAll({ attributes: { className: 'ocred_page' } });
        MetricsUtils.recordNumPdfPagesOcred(orcedPages.length);
    }
    /**
     * Called in response to our hit test after the mouse is released,
     * when the user is in a mode where Select-to-speak is capturing
     * mouse events (for example holding down Search).
     * @param evt The automation event from the hit test.
     */
    onAutomationHitTest_(evt) {
        // Walk up to the nearest window, web area, document, graphics document,
        // toolbar, or dialog that the hit node is contained inside. Only speak
        // objects within that container. In the future we might include other
        // root-like roles here. (Consider harmonizing with the `ui::IsRootLike`
        // method.)
        let root = evt.target;
        // In Chrome PDF Viewer, PDF content for a large PDF might still be loading
        // into a PDF accessibility tree when the user selects text on a PDF page.
        // In this case, the PDF root has only one child node, which is the status
        // node that contains a loading status message. Read this status message if
        // the user tries selecting text during this loading phase. The same should
        // happen in the Gallery App (AKA Backlight). Backlight uses a different
        // role for its PDF container: `ax::mojom::Role::kGraphicsDocument`.
        if ((root.role === RoleType.EMBEDDED_OBJECT ||
            root.role === RoleType.GRAPHICS_DOCUMENT) &&
            root.children.length === 1 &&
            root.firstChild.role === RoleType.PDF_ROOT &&
            root.firstChild.children.length === 1 &&
            this.readPdfStatusNodeIfStillLoading_(root.firstChild)) {
            return;
        }
        // TODO: Use AutomationPredicate.root instead?
        while (root.parent && root.role !== RoleType.WINDOW &&
            root.role !== RoleType.ROOT_WEB_AREA &&
            root.role !== RoleType.DESKTOP && root.role !== RoleType.DIALOG &&
            root.role !== RoleType.ALERT_DIALOG &&
            root.role !== RoleType.TOOLBAR && root.role !== RoleType.DOCUMENT &&
            root.role !== RoleType.GRAPHICS_DOCUMENT) {
            root = root.parent;
        }
        const rect = this.inputHandler_.getMouseRect();
        let nodes = [];
        chrome.automation.getFocus(focusedNode => {
            // In some cases, e.g. ARC++, the window received in the hit test request,
            // which is computed based on which window is the event handler for the
            // hit point, isn't the part of the tree that contains the actual
            // content. In such cases, use focus to get the root.
            // TODO(katie): Determine if this work-around needs to be ARC++ only. If
            // so, look for classname exoshell on the root or root parent to confirm
            // that a node is in ARC++.
            if (!NodeUtils.findAllMatching(root, rect, nodes) && focusedNode &&
                focusedNode.root.role !== RoleType.DESKTOP) {
                // TODO(b/314203187): Determine if not null assertion is appropriate
                // here.
                NodeUtils.findAllMatching(focusedNode.root, rect, nodes);
            }
            if (nodes.length === 1 && UiManager.isTrayButton(nodes[0])) {
                // Don't read only the Select-to-Speak toggle button in the tray unless
                // more items are being read.
                return;
            }
            if (this.shouldShowNavigationControls_() && nodes.length > 0 &&
                (rect.width <= SelectToSpeakConstants.PARAGRAPH_SELECTION_MAX_SIZE ||
                    rect.height <=
                        SelectToSpeakConstants.PARAGRAPH_SELECTION_MAX_SIZE)) {
                // If this is a single click (zero sized selection) on a text node, then
                // expand to entire paragraph.
                nodes = NodeUtils.getAllNodesInParagraph(nodes[0]);
            }
            this.startSpeechQueue_(nodes, {
                clearFocusRing: true,
            });
            MetricsUtils.recordStartEvent(MetricsUtils.StartSpeechMethod.MOUSE, this.prefsManager_);
        });
    }
    getFocusedNodeAndSpeakSelectedText_() {
        chrome.automation.getFocus(focusedNode => this.requestSpeakSelectedText_(MetricsUtils.StartSpeechMethod.CONTEXT_MENU, focusedNode));
    }
    /**
     * Queues up selected text for reading by finding the Position objects
     * representing the selection.
     * @param method the method that
     *     caused the text to speak.
     */
    requestSpeakSelectedText_(method, focusedNode) {
        // If nothing is selected, return early. Check if the focused node has
        // textSelStart and textSelEnd. For native UI like the omnibox, the root
        // might not have a selectionStartObject and selectionEndObject. Therefore
        // we must check textSelStart and textSelEnd on the focused node.
        if (!focusedNode || !focusedNode.root) {
            this.onNullSelection_();
            return;
        }
        const hasSelectionObjects = focusedNode.root.selectionStartObject &&
            focusedNode.root.selectionEndObject;
        const hasTextSelection = focusedNode.textSelStart !== undefined &&
            focusedNode.textSelEnd !== undefined;
        if (!hasSelectionObjects && !hasTextSelection) {
            this.onNullSelection_();
            return;
        }
        let startObject;
        let startOffset = 0;
        let endObject;
        let endOffset = 0;
        // Use selectionStartObject/selectionEndObject if available. Otherwise,
        // use textSelStart/textSelEnd to get the selection offset.
        if (hasSelectionObjects) {
            startObject = focusedNode.root.selectionStartObject;
            startOffset = focusedNode.root.selectionStartOffset || 0;
            endObject = focusedNode.root.selectionEndObject;
            endOffset = focusedNode.root.selectionEndOffset || 0;
        }
        else if (hasTextSelection) {
            startObject = focusedNode;
            startOffset = focusedNode.textSelStart || 0;
            endObject = focusedNode;
            endOffset = focusedNode.textSelEnd || 0;
        }
        if (startObject === endObject && startOffset === endOffset) {
            this.onNullSelection_();
            return;
        }
        // First calculate the equivalent position for this selection.
        // Sometimes the automation selection returns an offset into a root
        // node rather than a child node, which may be a bug. This allows us to
        // work around that bug until it is fixed or redefined.
        // Note that this calculation is imperfect: it uses node name length
        // to index into child nodes. However, not all node names are
        // user-visible text, so this does not always work. Instead, we must
        // fix the Blink bug where focus offset is not specific enough to
        // say which node is selected and at what charOffset. See
        // https://crbug.com/803160 for more.
        const startPosition = NodeUtils.getDeepEquivalentForSelection(startObject, startOffset, true);
        const endPosition = NodeUtils.getDeepEquivalentForSelection(endObject, endOffset, false);
        // TODO(katie): We go into these blocks but they feel redundant. Can
        // there be another way to do this? (E.g. by using the `SelIsBackward` field
        // in `AXTreeData`?)
        let firstPosition;
        let lastPosition;
        if (startPosition.node === endPosition.node) {
            if (startPosition.offset < endPosition.offset) {
                firstPosition = startPosition;
                lastPosition = endPosition;
            }
            else {
                lastPosition = startPosition;
                firstPosition = endPosition;
            }
        }
        else {
            const dir = AutomationUtil.getDirection(startPosition.node, endPosition.node);
            // Highlighting may be forwards or backwards. Make sure we start at the
            // first node.
            if (dir === constants.Dir.FORWARD) {
                firstPosition = startPosition;
                lastPosition = endPosition;
            }
            else {
                lastPosition = startPosition;
                firstPosition = endPosition;
            }
        }
        this.cancelIfSpeaking_(true /* clear the focus ring */);
        this.readNodesBetweenPositions_(firstPosition, lastPosition, method, focusedNode);
    }
    /**
     * Reads nodes between positions.
     * @param firstPosition The first position at which to start reading.
     * @param lastPosition The last position at which to stop reading.
     * @param method the method used to
     *     activate the speech, null if not activated by user.
     * @param focusedNode The node with user focus.
     */
    readNodesBetweenPositions_(firstPosition, lastPosition, method, focusedNode) {
        const nodes = [];
        // TODO(b/314204374): AutomationUtil.findNextNode may return null.
        let selectedNode = firstPosition.node;
        // If the method is set, a user requested the speech.
        const userRequested = method !== null;
        const methodNumber = method !== null ? method : -1;
        // Certain nodes such as omnibox store text value in the value property,
        // instead of the name property. The getNodeName method in ParagraphUtils
        // does handle this case properly, so use this static method to get text
        // from either `name' or `value' of the node.
        const nodeName = ParagraphUtils.getNodeName(selectedNode);
        if (nodeName && firstPosition.offset < nodeName.length &&
            !NodeUtils.shouldIgnoreNode(selectedNode, /* include offscreen */ true) &&
            !NodeUtils.isNotSelectable(selectedNode)) {
            // Initialize to the first node in the list if it's valid and inside
            // of the offset bounds.
            nodes.push(selectedNode);
        }
        else {
            // The selectedNode actually has no content selected. Let the list
            // initialize itself to the next node in the loop below.
            // This can happen if you click-and-drag starting after the text in
            // a first line to highlight text in a second line.
            firstPosition.offset = 0;
        }
        while (selectedNode && selectedNode !== lastPosition.node &&
            AutomationUtil.getDirection(selectedNode, lastPosition.node) ===
                constants.Dir.FORWARD) {
            // TODO: Is there a way to optimize the directionality checking of
            // AutomationUtil.getDirection(selectedNode, finalNode)?
            // For example, by making a helper and storing partial computation?
            selectedNode = AutomationUtil.findNextNode(selectedNode, constants.Dir.FORWARD, AutomationPredicate.leafWithText);
            if (!selectedNode) {
                break;
            }
            else if (NodeUtils.isTextField(selectedNode)) {
                // Dive down into the next text node.
                // Why does leafWithText return text fields?
                selectedNode = AutomationUtil.findNextNode(selectedNode, constants.Dir.FORWARD, AutomationPredicate.leafWithText);
                if (!selectedNode) {
                    break;
                }
            }
            if (!NodeUtils.shouldIgnoreNode(selectedNode, /* include offscreen */ true) &&
                !NodeUtils.isNotSelectable(selectedNode)) {
                nodes.push(selectedNode);
            }
        }
        if (nodes.length > 0) {
            if (lastPosition.node !== nodes[nodes.length - 1]) {
                // The node at the last position was not added to the list, perhaps it
                // was whitespace or invisible. Clear the ending offset because it
                // relates to a node that doesn't exist.
                this.startSpeechQueue_(nodes, {
                    clearFocusRing: userRequested,
                    startCharIndex: firstPosition.offset,
                });
            }
            else {
                this.startSpeechQueue_(nodes, {
                    clearFocusRing: userRequested,
                    startCharIndex: firstPosition.offset,
                    endCharIndex: lastPosition.offset,
                });
            }
            if (focusedNode) {
                this.initializeScrollingToOffscreenNodes_(focusedNode.root);
            }
            if (userRequested) {
                MetricsUtils.recordStartEvent(methodNumber, this.prefsManager_);
            }
        }
        else {
            // Gsuite apps include webapps beyond Docs, see getGSuiteAppRoot and
            // GSUITE_APP_REGEXP.
            const gsuiteAppRootNode = getGSuiteAppRoot(focusedNode);
            if (!gsuiteAppRootNode) {
                return;
            }
            chrome.tabs.query({ active: true }, tabs => {
                // Closure doesn't realize that we did a !gsuiteAppRootNode earlier
                // so we check again here.
                if (!gsuiteAppRootNode || gsuiteAppRootNode.url === undefined) {
                    return;
                }
                this.inputHandler_.onRequestReadClipboardData();
                this.currentNodeGroupItem_ =
                    new ParagraphUtils.NodeGroupItem(gsuiteAppRootNode, 0, false);
                if (tabs.length === 0 || tabs[0].url !== gsuiteAppRootNode.url) {
                    return;
                }
                const tab = tabs[0];
                chrome.tabs.executeScript(tab.id, {
                    allFrames: true,
                    matchAboutBlank: true,
                    code: 'document.execCommand("copy");',
                });
                if (userRequested) {
                    MetricsUtils.recordStartEvent(methodNumber, this.prefsManager_);
                }
            });
        }
    }
    /**
     * Gets ready to cancel future scrolling to offscreen nodes as soon as
     * a user-initiated scroll is done.
     * @param root The root node to listen for events on.
     */
    initializeScrollingToOffscreenNodes_(root) {
        if (!root) {
            return;
        }
        this.scrollToSpokenNode_ = true;
        const listener = (event) => {
            if (event.eventFrom !== 'action') {
                // User initiated event. Cancel all future scrolling to spoken nodes.
                // If the user wants a certain scroll position we will respect that.
                this.scrollToSpokenNode_ = false;
                // Now remove these event listeners, we no longer need them.
                root.removeEventListener(EventType.SCROLL_POSITION_CHANGED, listener, false);
                root.removeEventListener(EventType.SCROLL_HORIZONTAL_POSITION_CHANGED, listener, false);
                root.removeEventListener(EventType.SCROLL_VERTICAL_POSITION_CHANGED, listener, false);
            }
        };
        // ARC++ fires the first event, Views/Web fire the horizontal/vertical
        // scroll position changed events via AXEventGenerator.
        root.addEventListener(EventType.SCROLL_POSITION_CHANGED, listener, false);
        root.addEventListener(EventType.SCROLL_HORIZONTAL_POSITION_CHANGED, listener, false);
        root.addEventListener(EventType.SCROLL_VERTICAL_POSITION_CHANGED, listener, false);
    }
    /**
     * Plays a tone to let the user know they did the correct
     * keystroke but nothing was selected.
     */
    onNullSelection_() {
        if (!this.shouldShowNavigationControls_()) {
            this.nullSelectionTone_.play();
            return;
        }
        this.uiManager_.setFocusToPanel();
    }
    /**
     * Whether the STS is on a pause state, where |this.ttsManager_.isSpeaking| is
     * false and |this.state_| is SPEAKING.
     * TODO(leileilei): use two SelectToSpeak states to differentiate speaking and
     * pausing with panel.
     */
    isPaused_() {
        return !this.ttsManager_.isSpeaking() &&
            this.state_ === SelectToSpeakState.SPEAKING;
    }
    /**
     * Pause the TTS.
     */
    pause_() {
        return this.ttsManager_.pause();
    }
    /**
     * Resume the TTS.
     */
    resume_() {
        // If TTS is not paused, return early.
        if (!this.isPaused_()) {
            return;
        }
        const currentNodeGroup = this.getCurrentNodeGroup_();
        // If there is no processed node group, that means the user has not selected
        // anything. Ignore the resume command.
        if (!currentNodeGroup) {
            return;
        }
        this.ttsManager_.resume(this.getTtsOptionsForCurrentNodeGroup_());
    }
    /**
     * If resume is successful, a resume event will be sent. We use this event to
     * update node state.
     */
    onTtsResumeSucceedEvent_(event) {
        // If the node group is invalid, ignore the resume event. This is not
        // expected.
        const currentNodeGroup = this.getCurrentNodeGroup_();
        if (!currentNodeGroup) {
            console.warn('Unexpected invalid node group on TTS resume event.');
            return;
        }
        this.onTtsWordEvent_(event, currentNodeGroup);
    }
    /**
     * When resuming with empty content, an error event will be sent. If there
     * is no remaining user-selected content, STS will read from the current
     * position to the end of the current paragraph. If there is no content left
     * in this paragraph, we navigate to the next paragraph.
     */
    onTtsResumeErrorEvent_(_event) {
        // If the node group is invalid, ignore the error event. This is not
        // expected.
        const currentNodeGroup = this.getCurrentNodeGroup_();
        if (!currentNodeGroup) {
            console.warn('Unexpected invalid node group on TTS error event when resuming.');
            return;
        }
        // STS should try to read from the current position to the end of the
        // current paragraph. First, we get the current position. If we do not find
        // a position based on the |this.currentCharIndex_|, that means we have
        // reached the end of current node group. We fallback to the end position.
        const currentPosition = NodeUtils.getPositionFromNodeGroup(currentNodeGroup, this.currentCharIndex_, true /* fallbackToEnd */);
        // If we have passed the user-selected content, STS should speak the content
        // from the current position to the end of the current node group.
        const { nodes: remainingNodes, offset } = NodeNavigationUtils.getNextNodesInParagraphFromPosition(currentPosition, constants.Dir.FORWARD);
        // If there is no remaining nodes in this paragraph, we navigate to the next
        // paragraph.
        if (remainingNodes.length === 0) {
            this.navigateToNextParagraph_(constants.Dir.FORWARD);
            return;
        }
        this.startSpeechQueue_(remainingNodes, {
            clearFocusRing: false,
            startCharIndex: offset,
        });
    }
    /**
     * Stop speech. If speech was in-progress, the interruption
     * event will be caught and clearFocusRingAndNode_ will be
     * called, stopping visual feedback as well.
     * If speech was not in progress, i.e. if the user was drawing
     * a focus ring on the screen, this still clears the visual
     * focus ring.
     */
    stopAll_() {
        this.ttsManager_.stop();
        this.uiManager_.clear();
        this.onStateChanged_(SelectToSpeakState.INACTIVE);
    }
    /**
     * Clears the current focus ring and node, but does
     * not stop the speech.
     */
    clearFocusRingAndNode_() {
        this.uiManager_.clear();
        // Clear the node and also stop the interval testing.
        this.resetNodes_();
        this.supportsNavigationPanel_ = true;
        if (this.intervalId_ !== undefined) {
            clearInterval(this.intervalId_);
            this.intervalId_ = undefined;
        }
        this.scrollToSpokenNode_ = false;
    }
    /**
     * Resets the instance variables for nodes and node groups.
     */
    resetNodes_() {
        this.currentNodeGroups_ = [];
        this.currentNodeGroupIndex_ = -1;
        this.currentNodeGroupItem_ = null;
        this.currentNodeGroupItemIndex_ = -1;
        this.currentNodeWord_ = null;
        this.currentCharIndex_ = -1;
    }
    /**
     * Runs content scripts that allow Select-to-Speak access to
     * Google Docs content without a11y mode enabled, in every open
     * tab. Should be run when Select-to-Speak starts up so that any
     * tabs already opened will be checked.
     * This should be kept in sync with the "content_scripts" section in
     * the Select-to-Speak manifest.
     */
    runContentScripts_() {
        const scripts = chrome.runtime.getManifest()['content_scripts'][0]['js'];
        // We only ever expect one content script.
        if (scripts.length !== 1) {
            throw new Error('Only expected one script; got ' + JSON.stringify(scripts));
        }
        const script = scripts[0];
        chrome.tabs.query({
            url: [
                'https://docs.google.com/document*',
                'https://docs.sandbox.google.com/*',
            ],
        }, tabs => {
            tabs.forEach(tab => {
                chrome.tabs.executeScript(tab.id, { file: script });
            });
        });
    }
    /**
     * Set up event listeners user input.
     */
    setUpEventListeners_() {
        this.inputHandler_ = new InputHandler({
            // canStartSelecting: Whether mouse selection can begin.
            canStartSelecting: () => {
                return this.state_ !== SelectToSpeakState.SELECTING;
            },
            // onSelectingStateChanged: Started or stopped mouse selection.
            onSelectingStateChanged: (isSelecting, x, y) => {
                if (isSelecting) {
                    this.onStateChanged_(SelectToSpeakState.SELECTING);
                    // Fire a hit test event on click to warm up the cache, and cancel
                    // if speaking.
                    this.cancelIfSpeaking_(false /* don't clear the focus ring */);
                    this.desktop_.hitTest(x, y, EventType.MOUSE_PRESSED);
                }
                else {
                    this.onStateChanged_(SelectToSpeakState.INACTIVE);
                    // Do a hit test at the center of the area the user dragged over.
                    // This will give us some context when searching the accessibility
                    // tree. The hit test will result in a EventType.MOUSE_RELEASED
                    // event being fired on the result of that hit test, which will
                    // trigger onAutomationHitTest_.
                    this.desktop_.hitTest(x, y, EventType.MOUSE_RELEASED);
                }
            },
            // onSelectionChanged: Mouse selection rect changed.
            onSelectionChanged: rect => {
                this.uiManager_.setSelectionRect(rect);
            },
            // onKeystrokeSelection: Keys pressed for reading highlighted text.
            onKeystrokeSelection: () => {
                chrome.automation.getFocus(focusedNode => this.requestSpeakSelectedText_(MetricsUtils.StartSpeechMethod.KEYSTROKE, focusedNode));
            },
            // onRequestCancel: User requested canceling input/speech.
            onRequestCancel: () => {
                // User manually requested cancel, so log cancel metric.
                MetricsUtils.recordCancelIfSpeaking();
                this.cancelIfSpeaking_(true /* clear the focus ring */);
            },
            // onTextReceived: Text received from a 'paste' event to read aloud.
            onTextReceived: text => this.startSpeech_(text),
        });
        this.inputHandler_.setUpEventListeners();
        // Initialize the state to SelectToSpeakState.INACTIVE.
        chrome.accessibilityPrivate.setSelectToSpeakState(this.state_);
    }
    /**
     * Called when Chrome OS is requesting Select-to-Speak to switch states.
     */
    onStateChangeRequested() {
        // Switch Select-to-Speak states on request.
        // We will need to track the current state and toggle from one state to
        // the next when this function is called, and then call
        // accessibilityPrivate.setSelectToSpeakState with the new state.
        switch (this.state_) {
            case SelectToSpeakState.INACTIVE:
                // Start selection.
                this.inputHandler_.setTrackingMouse(true);
                this.onStateChanged_(SelectToSpeakState.SELECTING);
                MetricsUtils.recordSelectToSpeakStateChangeEvent(MetricsUtils.StateChangeEvent.START_SELECTION);
                break;
            case SelectToSpeakState.SPEAKING:
                // Stop speaking. User manually requested, so log cancel metric.
                MetricsUtils.recordCancelIfSpeaking();
                this.cancelIfSpeaking_(true /* clear the focus ring */);
                MetricsUtils.recordSelectToSpeakStateChangeEvent(MetricsUtils.StateChangeEvent.CANCEL_SPEECH);
                break;
            case SelectToSpeakState.SELECTING:
                // Cancelled selection.
                this.inputHandler_.setTrackingMouse(false);
                this.onStateChanged_(SelectToSpeakState.INACTIVE);
                MetricsUtils.recordSelectToSpeakStateChangeEvent(MetricsUtils.StateChangeEvent.CANCEL_SELECTION);
        }
        this.onStateChangeRequestedCallbackForTest_ &&
            this.onStateChangeRequestedCallbackForTest_();
    }
    /** Handles user request to navigate to next paragraph. */
    onNextParagraphRequested() {
        this.navigateToNextParagraph_(constants.Dir.FORWARD);
    }
    /** Handles user request to navigate to previous paragraph. */
    onPreviousParagraphRequested() {
        this.navigateToNextParagraph_(constants.Dir.BACKWARD);
    }
    /** Handles user request to navigate to next sentence. */
    onNextSentenceRequested() {
        this.navigateToNextSentence_(constants.Dir.FORWARD);
    }
    /** Handles user request to navigate to previous sentence. */
    onPreviousSentenceRequested() {
        this.navigateToNextSentence_(constants.Dir.BACKWARD);
    }
    /** Handles user request to navigate to exit STS. */
    onExitRequested() {
        // User manually requested, so log cancel metric.
        MetricsUtils.recordCancelIfSpeaking();
        this.stopAll_();
    }
    /** Handles user request to pause TTS. */
    onPauseRequested() {
        MetricsUtils.recordPauseEvent();
        this.pause_();
    }
    /** Handles user request to resume TTS. */
    onResumeRequested() {
        if (this.isPaused_()) {
            MetricsUtils.recordResumeEvent();
            this.resume_();
        }
    }
    /**
     * Handles user request to adjust reading speed.
     */
    onChangeSpeedRequested(rateMultiplier) {
        this.speechRateMultiplier_ = rateMultiplier;
        // If currently playing, stop TTS, then resume from current spot.
        if (!this.isPaused_()) {
            this.pause_().then(() => {
                this.resume_();
            });
        }
    }
    /**
     * Navigates to the next sentence.
     * @param direction Direction to search for the next sentence.
     *     If set to forward, we look for the sentence start after the current
     *     position. Otherwise, we look for the sentence start before the current
     *     position.
     */
    async navigateToNextSentence_(direction) {
        if (!this.isPaused_()) {
            await this.pause_();
        }
        const { nodes, offset } = NodeNavigationUtils.getNodesForNextSentence(this.getCurrentNodeGroup_(), this.currentCharIndex_, direction, nodes => this.skipPanel_(nodes));
        if (nodes.length === 0) {
            return;
        }
        // Ensure the first node in the paragraph is visible.
        nodes[0].makeVisible();
        this.startSpeechQueue_(nodes, {
            startCharIndex: offset,
        });
    }
    /**
     * Navigates to the next text block in the given direction.
     */
    async navigateToNextParagraph_(direction) {
        if (!this.isPaused_()) {
            // Stop TTS if it is currently playing.
            await this.pause_();
        }
        const nodes = NodeNavigationUtils.getNodesForNextParagraph(this.getCurrentNodeGroup_(), direction, nodes => this.skipPanel_(nodes));
        // Return early if the nodes are empty.
        if (nodes.length === 0) {
            return;
        }
        // Ensure the first node in the paragraph is visible.
        nodes[0].makeVisible();
        this.startSpeechQueue_(nodes);
    }
    /**
     * A predicate for paragraph selection and navigation. The current
     * implementation filters out paragraph that belongs to the panel.
     * @return Whether the paragraph made of the |nodes| is valid
     */
    skipPanel_(nodes) {
        return !AutomationUtil.getAncestors(nodes[0]).find(n => UiManager.isPanel(n));
    }
    /**
     * Enqueue speech for the single given string. The string is not associated
     * with any particular nodes, so this does not do any work around drawing
     * focus rings, unlike startSpeechQueue_ below.
     * @param text The text to speak.
     */
    startSpeech_(text) {
        this.prepareForSpeech_(true /* clearFocusRing */);
        this.maybeShowEnhancedVoicesDialog_(() => {
            const options = this.prefsManager_.getSpeechOptions(null);
            const fallbackVoiceName = this.prefsManager_.getLocalVoice();
            // Without nodes to anchor on, navigate is not supported.
            this.supportsNavigationPanel_ = false;
            options.onEvent = event => {
                if (event.type === 'start') {
                    this.onStateChanged_(SelectToSpeakState.SPEAKING);
                    this.updateUi_();
                }
                else if (event.type === 'end' || event.type === 'interrupted' ||
                    event.type === 'cancelled') {
                    // Automatically dismiss when we're at the end.
                    this.onStateChanged_(SelectToSpeakState.INACTIVE);
                }
            };
            const voiceName = options['voiceName'] || '';
            MetricsUtils.recordTtsEngineUsed(voiceName || '', this.prefsManager_);
            this.ttsManager_.speak(text, options, this.prefsManager_.isNetworkVoice(voiceName), fallbackVoiceName);
        });
    }
    /**
     * Enqueue nodes to TTS queue and start TTS. This function can be used for
     * adding nodes, either from user selection (e.g., mouse selection) or
     * navigation control (e.g., next paragraph).
     * @param  nodes The nodes to speak.
     * @param optParams:
     *    clearFocusRing: Whether to clear the focus ring or not. For example, we
     * need to clear the focus ring when starting from scratch but we do not need
     * to clear the focus ring when resuming from a previous pause. If this is not
     * passed, will default to false.
     *    startCharIndex: The index into the first node's text at which to start
     * speaking. If this is not passed, will start at 0.
     *    endCharIndex: The index into the last node's text at which to end
     * speech. If this is not passed, will stop at the end.
     */
    startSpeechQueue_(nodes, optParams) {
        this.maybeShowEnhancedVoicesDialog_(() => {
            const params = optParams || {};
            const clearFocusRing = params.clearFocusRing || false;
            let startCharIndex = params.startCharIndex;
            let endCharIndex = params.endCharIndex;
            this.prepareForSpeech_(clearFocusRing /* clear the focus ring */);
            if (nodes.length === 0) {
                return;
            }
            // Remember the original first and last node in the given list, as
            // |startCharIndex| and |endCharIndex| pertain to them. If, after SVG
            // resorting, the first or last nodes are re-ordered, do not clip them.
            const originalFirstNode = nodes[0];
            const originalLastNode = nodes[nodes.length - 1];
            // Sort any SVG child nodes, if present, by visual reading order.
            NodeUtils.sortSvgNodesByReadingOrder(nodes);
            // Override start or end index if original nodes were sorted.
            if (originalFirstNode !== nodes[0]) {
                startCharIndex = undefined;
            }
            if (originalLastNode !== nodes[nodes.length - 1]) {
                endCharIndex = undefined;
            }
            this.supportsNavigationPanel_ = this.isNavigationPanelSupported_(nodes);
            this.updateNodeGroups_(nodes, startCharIndex, endCharIndex);
            // Play TTS according to the current state variables.
            this.startCurrentNodeGroup_();
        });
    }
    /**
     * Updates the node groups to be spoken. Converts |nodes|, |startCharIndex|,
     * and |endCharIndex| into node groups, and updates |this.currentNodeGroups_|
     * and |this.currentNodeGroupIndex_|.
     * @param nodes The nodes to speak.
     * @param startCharIndex The index into the first node's text at
     *     which to start speaking. If this is not passed, will start at 0.
     * @param endCharIndex The index into the last node's text at which
     *     to end speech. If this is not passed, will stop at the end.
     */
    updateNodeGroups_(nodes, startCharIndex, endCharIndex) {
        this.resetNodes_();
        for (let i = 0; i < nodes.length; i++) {
            // When navigation controls are enabled, disable the clipping of overflow
            // words. When overflow words are clipped, words scrolled out of view are
            // clipped, which is undesirable for our navigation features as we
            // generate node groups for next/previous paragraphs which may be fully or
            // partially scrolled out of view.
            const nodeGroup = ParagraphUtils.buildNodeGroup(nodes, i, {
                splitOnLanguage: this.shouldUseVoiceSwitching_(),
                clipOverflowWords: !this.shouldShowNavigationControls_(),
            });
            const isFirstNodeGroup = i === 0;
            const shouldApplyStartOffset = isFirstNodeGroup && startCharIndex !== undefined;
            const firstNodeHasInlineText = nodeGroup.nodes.length > 0 && nodeGroup.nodes[0].hasInlineText;
            if (shouldApplyStartOffset) {
                let startIndexInNodeGroup;
                if (firstNodeHasInlineText) {
                    // We assume that the start offset will only be applied to the first
                    // node in the first NodeGroup. The |startCharIndex| needs to be
                    // adjusted. The first node of the NodeGroup may not be at the
                    // beginning of the parent of the NodeGroup. (e.g., an inlineText in
                    // its staticText parent). Thus, we need to adjust the start index.
                    const startIndexInNodeParent = ParagraphUtils.getStartCharIndexInParent(nodes[0]);
                    startIndexInNodeGroup = startCharIndex + startIndexInNodeParent +
                        nodeGroup.nodes[0].startChar;
                }
                else {
                    // Text field such as omnibox doesn't have inline text, but text in
                    // the value property. In case the user selects some text within, we
                    // need to adjust |startCharIndex| accordingly.
                    startIndexInNodeGroup = startCharIndex + nodeGroup.nodes[0].startChar;
                }
                this.applyOffset(nodeGroup, startIndexInNodeGroup, true /* isStartOffset */);
            }
            // Advance i to the end of this group, to skip all nodes it contains.
            i = nodeGroup.endIndex;
            const isLastNodeGroup = (i === nodes.length - 1);
            const shouldApplyEndOffset = isLastNodeGroup && endCharIndex !== undefined;
            const lastNodeHasInlineText = nodeGroup.nodes.length > 0 &&
                nodeGroup.nodes[nodeGroup.nodes.length - 1].hasInlineText;
            if (shouldApplyEndOffset) {
                let endIndexInNodeGroup;
                if (lastNodeHasInlineText) {
                    // We assume that the end offset will only be applied to the last
                    // node in the last NodeGroup. Similarly, |endCharIndex| needs to be
                    // adjusted.
                    const startIndexInNodeParent = ParagraphUtils.getStartCharIndexInParent(nodes[i]);
                    endIndexInNodeGroup = endCharIndex + startIndexInNodeParent +
                        nodeGroup.nodes[nodeGroup.nodes.length - 1].startChar;
                }
                else {
                    // Text field such as omnibox doesn't have inline text, but text in
                    // the value property. In case the user selects some text within, we
                    // need to adjust |endCharIndex| accordingly.
                    endIndexInNodeGroup = endCharIndex +
                        nodeGroup.nodes[nodeGroup.nodes.length - 1].startChar;
                }
                this.applyOffset(nodeGroup, endIndexInNodeGroup, false /* isStartOffset */);
            }
            if (nodeGroup.nodes.length === 0 && !isLastNodeGroup) {
                continue;
            }
            this.currentNodeGroups_.push(nodeGroup);
        }
        // Sets the initial node group index to zero if this.currentNodeGroups_ has
        // items.
        if (this.currentNodeGroups_.length > 0) {
            this.currentNodeGroupIndex_ = 0;
        }
    }
    /**
     * Starts reading the current node group.
     */
    startCurrentNodeGroup_() {
        const nodeGroup = this.getCurrentNodeGroup_();
        if (!nodeGroup) {
            return;
        }
        if (!nodeGroup.text) {
            this.onNodeGroupSpeakingCompleted_();
            return;
        }
        const options = this.getTtsOptionsForCurrentNodeGroup_();
        const voiceName = (options && options['voiceName']) || '';
        const fallbackVoiceName = this.prefsManager_.getLocalVoice();
        MetricsUtils.recordTtsEngineUsed(voiceName, this.prefsManager_);
        this.ttsManager_.speak(
        // TODO(b/314203187): Options may be undefined.
        nodeGroup.text, options, this.prefsManager_.isNetworkVoice(voiceName), fallbackVoiceName);
    }
    getTtsOptionsForCurrentNodeGroup_() {
        const nodeGroup = this.getCurrentNodeGroup_();
        if (!nodeGroup) {
            return;
        }
        const options = {};
        let language;
        let useVoiceSwitching = false;
        if (this.shouldUseVoiceSwitching_() && nodeGroup.detectedLanguage) {
            language = nodeGroup.detectedLanguage;
            useVoiceSwitching = true;
        }
        Object.assign(options, this.prefsManager_.getSpeechOptions({ language, useVoiceSwitching }));
        if (this.shouldShowNavigationControls_()) {
            options.rate = this.getSpeechRate_();
            // Log speech rate multiple applied by Select-to-speak.
            MetricsUtils.recordSpeechRateOverrideMultiplier(this.speechRateMultiplier_);
        }
        const nodeGroupText = nodeGroup.text || '';
        options.onEvent = (event) => {
            switch (event.type) {
                case chrome.tts.EventType.START:
                    if (nodeGroup.nodes.length <= 0) {
                        break;
                    }
                    this.onStateChanged_(SelectToSpeakState.SPEAKING);
                    // Update |this.currentCharIndex_|. Find the first non-space char
                    // index in nodeGroup text, or 0 if the text is undefined or the first
                    // char is non-space.
                    this.currentCharIndex_ = nodeGroupText.search(/\S|$/);
                    this.syncCurrentNodeWithCharIndex_(nodeGroup, this.currentCharIndex_);
                    if (this.prefsManager_.wordHighlightingEnabled()) {
                        // At start, find the first word and highlight that. Clear the
                        // previous word in the node.
                        this.currentNodeWord_ = null;
                        // If |this.currentCharIndex_| is not 0, that means we have applied
                        // a start offset. Thus, we need to pass startIndexInNodeGroup to
                        // optStartIndex and overwrite the word boundaries in the original
                        // node.
                        this.updateNodeHighlight_(nodeGroupText, this.currentCharIndex_, this.currentCharIndex_ !== 0 ? this.currentCharIndex_ :
                            undefined);
                    }
                    else {
                        this.updateUi_();
                    }
                    break;
                case chrome.tts.EventType.RESUME:
                    this.onTtsResumeSucceedEvent_(event);
                    break;
                case chrome.tts.EventType.ERROR:
                    if (event.errorMessage ===
                        TtsManager.ErrorMessage.RESUME_WITH_EMPTY_CONTENT) {
                        this.onTtsResumeErrorEvent_(event);
                    }
                    break;
                // @ts-expect-error: Fallthrough on purpose.
                case chrome.tts.EventType.PAUSE:
                    // Updates the select to speak state to speaking to keep navigation
                    // panel visible, so that the user can click resume from the panel.
                    this.onStateChanged_(SelectToSpeakState.SPEAKING);
                // Fall through.
                case chrome.tts.EventType.INTERRUPTED:
                case chrome.tts.EventType.CANCELLED:
                    if (!this.shouldShowNavigationControls_()) {
                        this.onStateChanged_(SelectToSpeakState.INACTIVE);
                        break;
                    }
                    if (this.state_ === SelectToSpeakState.SELECTING) {
                        // Do not go into inactive state if navigation controls are enabled
                        // and we're currently making a new selection. This enables users
                        // to select new nodes while STS is active without first exiting.
                        break;
                    }
                    break;
                case chrome.tts.EventType.END:
                    this.onNodeGroupSpeakingCompleted_();
                    break;
                case chrome.tts.EventType.WORD:
                    this.onTtsWordEvent_(event, nodeGroup);
                    break;
            }
        };
        return options;
    }
    /**
     * When a node group is completed, we start speaking the next node group
     * indicated by the end index. If we have reached the last node group, this
     * function will update STS status depending whether the navigation feature is
     * enabled.
     */
    onNodeGroupSpeakingCompleted_() {
        const currentNodeGroup = this.getCurrentNodeGroup_();
        // Update the current char index to the end of the node group. If the
        // endOffset is undefined, we set the index to the length of the node
        // group's text.
        if (currentNodeGroup && currentNodeGroup.endOffset !== undefined) {
            this.currentCharIndex_ = currentNodeGroup.endOffset;
        }
        else {
            const nodeGroupText = (currentNodeGroup && currentNodeGroup.text) || '';
            this.currentCharIndex_ = nodeGroupText.length;
        }
        const isLastNodeGroup = (this.currentNodeGroupIndex_ === this.currentNodeGroups_.length - 1);
        if (isLastNodeGroup) {
            if (!this.shouldShowNavigationControls_()) {
                this.onStateChanged_(SelectToSpeakState.INACTIVE);
            }
            else {
                // If navigation features are enabled, we should keep STS state to
                // speaking so that the user can hit resume to continue.
                this.onStateChanged_(SelectToSpeakState.SPEAKING);
            }
            return;
        }
        // Start reading the next node group.
        this.currentNodeGroupIndex_++;
        this.startCurrentNodeGroup_();
    }
    /**
     * Update |this.currentNodeGroupItem_|, the current speaking or the node to be
     * spoken in the node group.
     * @param nodeGroup the current nodeGroup.
     * @param charIndex the start char index of the word to be spoken.
     *    The index is relative to the entire NodeGroup.
     * @param optStartFromNodeGroupIndex the NodeGroupIndex to start
     *    with. If undefined, search from 0.
     * @return If the found NodeGroupIndex is different from the
     *    |optStartFromNodeGroupIndex|.
     */
    syncCurrentNodeWithCharIndex_(nodeGroup, charIndex, optStartFromNodeGroupIndex) {
        if (optStartFromNodeGroupIndex === undefined) {
            optStartFromNodeGroupIndex = 0;
        }
        // There is no speaking word, set the NodeGroupItemIndex to 0.
        if (charIndex <= 0) {
            this.currentNodeGroupItemIndex_ = 0;
            this.currentNodeGroupItem_ =
                nodeGroup.nodes[this.currentNodeGroupItemIndex_];
            return this.currentNodeGroupItemIndex_ === optStartFromNodeGroupIndex;
        }
        // Sets the |this.currentNodeGroupItemIndex_| to
        // |optStartFromNodeGroupIndex|
        this.currentNodeGroupItemIndex_ = optStartFromNodeGroupIndex;
        this.currentNodeGroupItem_ =
            nodeGroup.nodes[this.currentNodeGroupItemIndex_];
        if (this.currentNodeGroupItemIndex_ + 1 < nodeGroup.nodes.length) {
            let next = nodeGroup.nodes[this.currentNodeGroupItemIndex_ + 1];
            let nodeUpdated = false;
            // TODO(katie): For something like a date, the start and end
            // node group nodes can actually be different. Example:
            // "<span>Tuesday,</span> December 18, 2018".
            // Check if we've reached this next node yet. Since charIndex is the
            // start char index of the target word, we just need to make sure the
            // next.startchar is bigger than it.
            while (next && charIndex >= next.startChar &&
                this.currentNodeGroupItemIndex_ + 1 < nodeGroup.nodes.length) {
                next = this.incrementCurrentNodeAndGetNext_(nodeGroup);
                nodeUpdated = true;
            }
            return nodeUpdated;
        }
        return false;
    }
    /**
     * Apply start or end offset to the text of the |nodeGroup|.
     * @param nodeGroup the input nodeGroup.
     * @param offset the size of offset.
     * @param isStartOffset whether to apply a startOffset or an
     *     endOffset.
     */
    applyOffset(nodeGroup, offset, isStartOffset) {
        if (isStartOffset) {
            // Applying start offset. Remove all text before the start index so that
            // it is not spoken. Backfill with spaces so that index counting
            // functions don't get confused.
            nodeGroup.text = ' '.repeat(offset) + nodeGroup.text.substr(offset);
        }
        else {
            // Remove all text after the end index so it is not spoken.
            nodeGroup.text = nodeGroup.text.substr(0, offset);
            nodeGroup.endOffset = offset;
        }
    }
    /**
     * Prepares for speech. Call once before this.ttsManager_.speak is called.
     * @param clearFocusRing Whether to clear the focus ring.
     */
    prepareForSpeech_(clearFocusRing) {
        this.cancelIfSpeaking_(clearFocusRing /* clear the focus ring */);
        // Update the UI on an interval, to adapt to automation tree changes.
        if (this.intervalId_ !== undefined) {
            clearInterval(this.intervalId_);
        }
        this.intervalId_ = setInterval(() => this.updateUi_(), SelectToSpeakConstants.NODE_STATE_TEST_INTERVAL_MS);
    }
    /**
     * Uses the 'word' speech event to determine which node is currently being
     * spoken, and prepares for highlight if enabled.
     * @param event The event to use for updates.
     * @param nodeGroup The node group for this
     *     utterance.
     */
    onTtsWordEvent_(event, nodeGroup) {
        if (event.charIndex === undefined) {
            return;
        }
        // Not all speech engines include length in the ttsEvent object.
        const hasLength = event.length !== undefined && event.length >= 0;
        const length = event.length || 0;
        // Only update the |this.currentCharIndex_| if event has a higher charIndex.
        // TTS sometimes will report an incorrect number at the end of an utterance.
        this.currentCharIndex_ = Math.max(event.charIndex, this.currentCharIndex_);
        console.debug(nodeGroup.text + ' (index ' + event.charIndex + ')');
        let debug = '-'.repeat(event.charIndex);
        if (hasLength) {
            debug += '^'.repeat(length);
        }
        else {
            debug += '^';
        }
        console.debug(debug);
        // First determine which node contains the word currently being spoken,
        // and update this.currentNodeGroupItem_, this.currentNodeWord_, and
        // this.currentNodeGroupItemIndex_ to match.
        const nodeUpdated = this.syncCurrentNodeWithCharIndex_(nodeGroup, event.charIndex, this.currentNodeGroupItemIndex_);
        if (nodeUpdated && !this.prefsManager_.wordHighlightingEnabled()) {
            // If we are doing a per-word highlight, we update the UI after figuring
            // out what the currently highlighted word is. Otherwise, update now.
            this.updateUi_();
        }
        // Finally update the word highlight if it is enabled.
        if (this.prefsManager_.wordHighlightingEnabled()) {
            if (hasLength) {
                this.currentNodeWord_ = {
                    'start': event.charIndex - this.currentNodeGroupItem_.startChar,
                    'end': event.charIndex + length - this.currentNodeGroupItem_.startChar,
                };
                this.updateUi_();
            }
            else {
                this.updateNodeHighlight_(nodeGroup.text, event.charIndex);
            }
        }
        else {
            this.currentNodeWord_ = null;
        }
    }
    /**
     * Updates the current node and relevant points to be the next node in the
     * group, then returns the next node in the group after that.
     */
    incrementCurrentNodeAndGetNext_(nodeGroup) {
        // Move to the next node.
        this.currentNodeGroupItemIndex_ += 1;
        this.currentNodeGroupItem_ =
            nodeGroup.nodes[this.currentNodeGroupItemIndex_];
        // Setting this.currentNodeWord_ to null signals it should be recalculated
        // later.
        this.currentNodeWord_ = null;
        if (this.currentNodeGroupItemIndex_ + 1 >= nodeGroup.nodes.length) {
            return null;
        }
        return nodeGroup.nodes[this.currentNodeGroupItemIndex_ + 1];
    }
    /**
     * Updates the state.
     */
    onStateChanged_(state) {
        if (this.state_ !== state) {
            if (state === SelectToSpeakState.INACTIVE) {
                this.clearFocusRingAndNode_();
            }
            // Send state change event to Chrome.
            chrome.accessibilityPrivate.setSelectToSpeakState(state);
            this.state_ = state;
        }
    }
    /**
     * Cancels the current speech queue.
     * @param clearFocusRing Whether to clear the focus ring as well.
     */
    cancelIfSpeaking_(clearFocusRing) {
        if (clearFocusRing) {
            this.stopAll_();
        }
        else {
            // Just stop speech
            this.ttsManager_.stop();
        }
    }
    /**
     * @return Promise that resolves to whether the given node
     *     should be considered in the foreground or not.
     */
    isNodeInForeground_(node) {
        return new Promise(resolve => {
            this.desktop_.hitTestWithReply(node.location.left, node.location.top, nodeAtLocation => {
                chrome.automation.getFocus(focusedNode => {
                    const window = NodeUtils.getNearestContainingWindow(nodeAtLocation);
                    const currentWindow = NodeUtils.getNearestContainingWindow(node);
                    if (currentWindow != null && window != null &&
                        currentWindow === window) {
                        resolve(true);
                        return;
                    }
                    if (UiManager.isPanel(window) ||
                        UiManager.isPanel(NodeUtils.getNearestContainingWindow(focusedNode))) {
                        // If the focus is on the Select-to-speak panel or the hit test
                        // landed on the panel, treat the current node as if it is in
                        // the foreground.
                        resolve(true);
                        return;
                    }
                    if (focusedNode && currentWindow) {
                        // See if the focused node window matches the currentWindow.
                        // This may happen in some cases, for example, ARC++, when the
                        // window which received the hit test request is not part of the
                        // tree that contains the actual content. In such cases, use
                        // focus to get the appropriate root.
                        const focusedWindow = NodeUtils.getNearestContainingWindow(focusedNode.root);
                        if (focusedWindow != null && currentWindow === focusedWindow) {
                            resolve(true);
                            return;
                        }
                    }
                    resolve(false);
                });
            });
        });
    }
    /**
     * @return Current node that is being spoken.
     */
    getCurrentSpokenNode_() {
        if (!this.currentNodeGroupItem_) {
            return null;
        }
        if (this.currentNodeGroupItem_.hasInlineText && this.currentNodeWord_) {
            return ParagraphUtils.findInlineTextNodeByCharacterIndex(this.currentNodeGroupItem_.node, this.currentNodeWord_.start);
        }
        else if (this.currentNodeGroupItem_.hasInlineText &&
            this.shouldShowNavigationControls_()) {
            // If navigation controls are enabled, but word highlighting is disabled
            // (currentNodeWord_ === null), still find the inline text node so the
            // focus ring will highlight the whole block.
            return ParagraphUtils.findInlineTextNodeByCharacterIndex(this.currentNodeGroupItem_.node, 0);
        }
        // No inline text or word highlighting and navigation controls are
        // disabled.
        return this.currentNodeGroupItem_.node;
    }
    /**
     * Updates the UI based on the current STS and node state.
     * @return Promise that resolves when operation is complete.
     */
    async updateUi_() {
        if (this.currentNodeGroupItem_ === null) {
            // Nothing to do.
            return;
        }
        // Determine whether current node is in the foreground. If node has no
        // location, assume it is not in the foreground.
        const node = this.currentNodeGroupItem_.node;
        const inForeground = node.location !== undefined ?
            await this.isNodeInForeground_(node) :
            false;
        // Verify that current node item is still pointing to the same node after
        // asynchronous |isNodeInForeground_| operation.
        if (this.currentNodeGroupItem_ === null ||
            this.currentNodeGroupItem_.node !== node) {
            return;
        }
        const nodeState = NodeUtils.getNodeState(node);
        if (nodeState === NodeUtils.NodeState.NODE_STATE_INVALID ||
            nodeState === NodeUtils.NodeState.NODE_STATE_INVISIBLE ||
            !inForeground) {
            // Current node is in background or node is invalid/invisible.
            this.uiManager_.clear();
            return;
        }
        const spokenNode = this.getCurrentSpokenNode_();
        const currentNodeGroup = this.getCurrentNodeGroup_();
        if (!currentNodeGroup || !spokenNode) {
            console.warn('Could not update UI; no node group or spoken node');
            return;
        }
        if (this.scrollToSpokenNode_ && spokenNode.state['offscreen']) {
            spokenNode.makeVisible();
        }
        const currentWord = this.prefsManager_.wordHighlightingEnabled() ?
            this.currentNodeWord_ :
            null;
        this.uiManager_.update(currentNodeGroup, spokenNode, currentWord, {
            showPanel: this.shouldShowNavigationControls_(),
            paused: this.isPaused_(),
            speechRateMultiplier: this.speechRateMultiplier_,
        });
    }
    /**
     * Shows a dialog to the user on first-run after enhanced voices update,
     * showing privacy disclaimer and asking if the user wants to turn on enhanced
     * network voices.
     *
     * @param callback Called back after user has confirmed or
     *     canceled in the dialog.
     */
    maybeShowEnhancedVoicesDialog_(callback) {
        if (!this.prefsManager_.enhancedVoicesDialogShown() &&
            this.prefsManager_.enhancedNetworkVoicesAllowed()) {
            // TODO(crbug.com/1230227): Style this dialog to match UX mocks.
            const title = chrome.i18n.getMessage('select_to_speak_natural_voice_dialog_title');
            const description = chrome.i18n.getMessage('select_to_speak_natural_voice_dialog_description');
            const cancelName = chrome.i18n.getMessage('select_to_speak_natural_voice_dialog_cancel');
            chrome.accessibilityPrivate.showConfirmationDialog(title, description, cancelName, confirm => {
                this.prefsManager_.setEnhancedNetworkVoicesFromDialog(confirm);
                if (callback !== undefined) {
                    callback();
                }
            });
        }
        else {
            // Flag not set or already shown, so we can continue the control flow
            // synchronously.
            if (callback !== undefined) {
                callback();
            }
        }
    }
    /**
     * Updates the currently highlighted node word based on the current text
     * and the character index of an event.
     * @param text The current text
     * @param charIndex The index of a current event in the text.
     * @param optStartIndex The index at which to start the
     *     highlight. This takes precedence over the charIndex.
     */
    updateNodeHighlight_(text, charIndex, optStartIndex) {
        if (charIndex >= text.length) {
            // No need to do work if we are at the end of the paragraph.
            return;
        }
        // Get the next word based on the event's charIndex.
        const nextWordStart = WordUtils.getNextWordStart(text, charIndex, this.currentNodeGroupItem_);
        // The |WordUtils.getNextWordEnd| will find the correct end based on the
        // trimmed text, so there is no need to provide additional input like
        // optStartIndex.
        const nextWordEnd = WordUtils.getNextWordEnd(text, optStartIndex === undefined ? nextWordStart : optStartIndex, this.currentNodeGroupItem_);
        // Map the next word into the node's index from the text.
        const nodeStart = optStartIndex === undefined ?
            nextWordStart - this.currentNodeGroupItem_.startChar :
            optStartIndex - this.currentNodeGroupItem_.startChar;
        const nodeEnd = Math.min(nextWordEnd - this.currentNodeGroupItem_.startChar, NodeUtils.nameLength(this.currentNodeGroupItem_.node));
        if ((this.currentNodeWord_ == null ||
            nodeStart >= this.currentNodeWord_.end) &&
            nodeStart <= nodeEnd) {
            // Only update the bounds if they have increased from the
            // previous node. Because tts may send multiple callbacks
            // for the end of one word and the beginning of the next,
            // checking that the current word has changed allows us to
            // reduce extra work.
            this.currentNodeWord_ = { 'start': nodeStart, 'end': nodeEnd };
            this.updateUi_();
        }
    }
    /**
     * @return Current speech rate.
     */
    getSpeechRate_() {
        // Multiply default speech rate with user-selected multiplier.
        const rate = this.prefsManager_.speechRate() * this.speechRateMultiplier_;
        // Then round to the nearest tenth (ex. 1.799999 becomes 1.8).
        return Math.round(rate * 10) / 10;
    }
    /**
     * @return Whether all given nodes support the navigation panel.
     */
    isNavigationPanelSupported_(nodes) {
        if (nodes.length === 0) {
            return true;
        }
        if (nodes.length === 1 && nodes[0] === nodes[0].root && nodes[0].parent &&
            nodes[0].parent.root &&
            nodes[0].parent.root.role === RoleType.DESKTOP) {
            // If the selected node is a root node within the desktop, such as a
            // a browser window, then do not show the navigation panel. There will
            // be no where for the user to navigate to. Also panel could be clipped
            // offscreen if the window is fullscreened.
            return false;
        }
        // Do not show panel on system UI. System UI can be problematic due to
        // auto-dismissing behavior (see http://crbug.com/1157148), but also
        // navigation controls do not work well for control-rich interfaces that are
        // light on text (and therefore no sentence and paragraph structures).
        return !nodes.some(n => n.root && n.root.role === RoleType.DESKTOP);
    }
    /**
     * @param keysPressed Which keys to pretend are currently pressed.
     */
    sendMockSelectToSpeakKeysPressedChanged(keysPressed) {
        this.inputHandler_.onKeysPressedChanged(new Set(keysPressed));
    }
    /**
     * Fires a mock mouse down event for testing.
     * @param type The event type.
     * @param mouseX The mouse x coordinate in global screen coordinates.
     * @param mouseY The mouse y coordinate in global screen coordinates.
     */
    fireMockMouseEvent(type, mouseX, mouseY) {
        this.inputHandler_.onMouseEvent(type, mouseX, mouseY);
    }
    /**
     * TODO(crbug.com/950391): Consider adding a metric for when voice switching
     * gets used.
     */
    shouldUseVoiceSwitching_() {
        return this.prefsManager_.voiceSwitchingEnabled();
    }
    /**
     * Used by C++ tests to ensure STS load is completed.
     * @param callback Callback for when desktop is loaded from
     * automation.
     */
    setOnLoadDesktopCallbackForTest(callback) {
        if (!this.desktop_) {
            this.onLoadDesktopCallbackForTest_ = callback;
            return;
        }
        // Desktop already loaded.
        callback();
    }
}
TestImportManager.exportForTesting(getGSuiteAppRoot);

// Copyright 2017 The Chromium Authors
// Use of this source code is governed by a BSD-style license that can be
// found in the LICENSE file.
// Needed for testing.
let selectToSpeak;
if (InstanceChecker.isActiveInstance()) {
    selectToSpeak = new SelectToSpeak();
    TestImportManager.exportForTesting(['selectToSpeak', selectToSpeak]);
}

export { selectToSpeak };
//# sourceMappingURL=select_to_speak_main.rollup.js.map
