// Copyright 2019 The Chromium Authors
// Use of this source code is governed by a BSD-style license that can be
// found in the LICENSE file.
import { RectUtil } from '/common/rect_util.js';
import { TestImportManager } from '/common/testing/test_import_manager.js';
import { Navigator } from '../navigator.js';
import { ActionResponse } from '../switch_access_constants.js';
import { BackButtonNode } from './back_button_node.js';
import { SAChildNode, SARootNode } from './switch_access_node.js';
var MenuAction = chrome.accessibilityPrivate.SwitchAccessMenuAction;
var RoleType = chrome.automation.RoleType;
/**
 * This class handles the grouping of nodes that are not grouped in the
 *     automation tree. They are defined by their parent and child nodes.
 * Ex: Nodes in the virtual keyboard have no intermediate grouping, but should
 *     be grouped by row.
 */
export class GroupNode extends SAChildNode {
    children_;
    containingNode_;
    /**
     * @param children The nodes that this group contains.
     *     Should not include the back button.
     * @param containingNode The automation node most closely containing the
     *     children.
     */
    constructor(children_, containingNode_) {
        super();
        this.children_ = children_;
        this.containingNode_ = containingNode_;
    }
    // ================= Getters and setters =================
    get actions() {
        return [MenuAction.DRILL_DOWN];
    }
    get automationNode() {
        return this.containingNode_;
    }
    get location() {
        // TODO(b/314203187): Not null asserted, check that this is correct.
        const childLocations = this.children_.filter(c => c.isValidAndVisible()).map(c => c.location);
        return RectUtil.unionAll(childLocations);
    }
    get role() {
        return RoleType.GROUP;
    }
    // ================= General methods =================
    asRootNode() {
        const root = new SARootNode(this.containingNode_);
        // Make a copy of the children array.
        const children = [...this.children_];
        children.push(new BackButtonNode(root));
        root.children = children;
        return root;
    }
    equals(other) {
        if (!(other instanceof GroupNode)) {
            return false;
        }
        if (other.children_.length !== this.children_.length) {
            return false;
        }
        for (let i = 0; i < this.children_.length; i++) {
            if (!other.children_[i].equals(this.children_[i])) {
                return false;
            }
        }
        return true;
    }
    isEquivalentTo(node) {
        if (node instanceof GroupNode) {
            return this.equals(node);
        }
        for (const child of this.children_) {
            if (child.isEquivalentTo(node)) {
                return true;
            }
        }
        return false;
    }
    isGroup() {
        return true;
    }
    isValidAndVisible() {
        for (const child of this.children_) {
            if (child.isValidAndVisible()) {
                return super.isValidAndVisible();
            }
        }
        return false;
    }
    performAction(action) {
        if (action === MenuAction.DRILL_DOWN) {
            Navigator.byItem.enterGroup();
            return ActionResponse.CLOSE_MENU;
        }
        return ActionResponse.NO_ACTION_TAKEN;
    }
    // ================= Static methods =================
    /** Assumes nodes are visually in rows. */
    static separateByRow(nodes, containingNode) {
        const result = [];
        for (let i = 0; i < nodes.length;) {
            const children = [];
            children.push(nodes[i]);
            i++;
            while (i < nodes.length &&
                RectUtil.sameRow(children[0].location, nodes[i].location)) {
                children.push(nodes[i]);
                i++;
            }
            result.push(new GroupNode(children, containingNode));
        }
        return result;
    }
}
TestImportManager.exportForTesting(GroupNode);
