import { TestImportManager } from '../../../../../../../../../../../../../../../../../../common/testing/test_import_manager.js';
import { AsyncUtil } from '../../../../../../../../../../../../../../../../../../common/async_util.js';
import { Flags, FlagName } from '../../../../../../../../../../../../../../../../../../common/flags.js';
import { KeepAlive } from '../../../../../../../../../../../../../../../../../../common/keep_alive.js';
import { InstanceChecker } from '../../../../../../../../../../../../../../../../../../common/mv3/instance_checker.js';
import { RectUtil } from '../../../../../../../../../../../../../../../../../../common/rect_util.js';
import { ArrayUtil } from '../../../../../../../../../../../../../../../../../../common/array_util.js';
import { EventHandler } from '../../../../../../../../../../../../../../../../../../common/event_handler.js';
import { EventGenerator } from '../../../../../../../../../../../../../../../../../../common/event_generator.js';
import { KeyCode } from '../../../../../../../../../../../../../../../../../../common/key_code.js';
import { AutomationPredicate } from '../../../../../../../../../../../../../../../../../../common/automation_predicate.js';
import { RepeatedEventHandler } from '../../../../../../../../../../../../../../../../../../common/repeated_event_handler.js';
import { constants } from '../../../../../../../../../../../../../../../../../../common/constants.js';
import { AutomationTreeWalker } from '../../../../../../../../../../../../../../../../../../common/tree_walker.js';
import { AutomationUtil } from '../../../../../../../../../../../../../../../../../../common/automation_util.js';
import { RepeatedTreeChangeHandler } from '../../../../../../../../../../../../../../../../../../common/repeated_tree_change_handler.js';
import { StringUtil } from '../../../../../../../../../../../../../../../../../../common/string_util.js';
import { Settings } from '../../../../../../../../../../../../../../../../../../common/settings.js';

// Copyright 2019 The Chromium Authors
// Use of this source code is governed by a BSD-style license that can be
// found in the LICENSE file.
/**
 * @fileoverview Constants used in Switch Access.
 */
/** When an action is performed, how the menu should respond. */
var ActionResponse;
(function (ActionResponse) {
    ActionResponse[ActionResponse["NO_ACTION_TAKEN"] = -1] = "NO_ACTION_TAKEN";
    ActionResponse[ActionResponse["REMAIN_OPEN"] = 0] = "REMAIN_OPEN";
    ActionResponse[ActionResponse["CLOSE_MENU"] = 1] = "CLOSE_MENU";
    ActionResponse[ActionResponse["EXIT_SUBMENU"] = 2] = "EXIT_SUBMENU";
    ActionResponse[ActionResponse["RELOAD_MENU"] = 3] = "RELOAD_MENU";
    ActionResponse[ActionResponse["OPEN_TEXT_NAVIGATION_MENU"] = 4] = "OPEN_TEXT_NAVIGATION_MENU";
})(ActionResponse || (ActionResponse = {}));
/**
 * The types of error or unexpected state that can be encountered by Switch
 * Access.
 * These values are persisted to logs and should not be renumbered or re-used.
 * See tools/metrics/histograms/enums.xml.
 */
var ErrorType;
(function (ErrorType) {
    ErrorType[ErrorType["UNKNOWN"] = 0] = "UNKNOWN";
    ErrorType[ErrorType["PREFERENCE_TYPE"] = 1] = "PREFERENCE_TYPE";
    ErrorType[ErrorType["UNTRANSLATED_STRING"] = 2] = "UNTRANSLATED_STRING";
    ErrorType[ErrorType["INVALID_COLOR"] = 3] = "INVALID_COLOR";
    ErrorType[ErrorType["NEXT_UNDEFINED"] = 4] = "NEXT_UNDEFINED";
    ErrorType[ErrorType["PREVIOUS_UNDEFINED"] = 5] = "PREVIOUS_UNDEFINED";
    ErrorType[ErrorType["NULL_CHILD"] = 6] = "NULL_CHILD";
    ErrorType[ErrorType["NO_CHILDREN"] = 7] = "NO_CHILDREN";
    ErrorType[ErrorType["MALFORMED_DESKTOP"] = 8] = "MALFORMED_DESKTOP";
    ErrorType[ErrorType["MISSING_LOCATION"] = 9] = "MISSING_LOCATION";
    ErrorType[ErrorType["MISSING_KEYBOARD"] = 10] = "MISSING_KEYBOARD";
    ErrorType[ErrorType["ROW_TOO_SHORT"] = 11] = "ROW_TOO_SHORT";
    ErrorType[ErrorType["MISSING_BASE_NODE"] = 12] = "MISSING_BASE_NODE";
    ErrorType[ErrorType["NEXT_INVALID"] = 13] = "NEXT_INVALID";
    ErrorType[ErrorType["PREVIOUS_INVALID"] = 14] = "PREVIOUS_INVALID";
    ErrorType[ErrorType["INVALID_SELECTION_BOUNDS"] = 15] = "INVALID_SELECTION_BOUNDS";
    ErrorType[ErrorType["UNINITIALIZED"] = 16] = "UNINITIALIZED";
    ErrorType[ErrorType["DUPLICATE_INITIALIZATION"] = 17] = "DUPLICATE_INITIALIZATION";
})(ErrorType || (ErrorType = {}));
/** The different types of menus and sub-menus that can be shown. */
var MenuType;
(function (MenuType) {
    MenuType[MenuType["MAIN_MENU"] = 0] = "MAIN_MENU";
    MenuType[MenuType["TEXT_NAVIGATION"] = 1] = "TEXT_NAVIGATION";
    MenuType[MenuType["POINT_SCAN_MENU"] = 2] = "POINT_SCAN_MENU";
})(MenuType || (MenuType = {}));
/**
 * The modes of interaction the user can select for how to interact with the
 * device.
 */
var Mode;
(function (Mode) {
    Mode[Mode["ITEM_SCAN"] = 0] = "ITEM_SCAN";
    Mode[Mode["POINT_SCAN"] = 1] = "POINT_SCAN";
})(Mode || (Mode = {}));
TestImportManager.exportForTesting(['Mode', Mode]);

// Copyright 2017 The Chromium Authors
// Use of this source code is governed by a BSD-style license that can be
// found in the LICENSE file.
/**
 * Class to handle auto-scan behavior.
 */
class AutoScanManager {
    intervalID_;
    isEnabled_ = false;
    /** Whether the current node is within the virtual keyboard. */
    inKeyboard_ = false;
    /** Auto-scan interval for the on-screen keyboard in milliseconds. */
    keyboardScanTime_ = NOT_INITIALIZED;
    /** Length of the auto-scan interval for most contexts, in milliseconds. */
    primaryScanTime_ = NOT_INITIALIZED;
    static instance;
    constructor() { }
    // ============== Static Methods ================
    static init() {
        if (AutoScanManager.instance) {
            throw SwitchAccess.error(ErrorType.DUPLICATE_INITIALIZATION, 'Cannot call AutoScanManager.init() more than once.');
        }
        AutoScanManager.instance = new AutoScanManager();
    }
    /** Restart auto-scan under current settings if it is currently running. */
    static restartIfRunning() {
        if (AutoScanManager.instance?.isRunning_()) {
            AutoScanManager.instance.stop_();
            AutoScanManager.instance.start_();
        }
    }
    /**
     * Stop auto-scan if it is currently running. Then, if |enabled| is true,
     * turn on auto-scan. Otherwise leave it off.
     */
    static setEnabled(enabled) {
        // TODO(b/314203187): Not null asserted, check that this is correct.
        if (AutoScanManager.instance.isRunning_()) {
            AutoScanManager.instance.stop_();
        }
        AutoScanManager.instance.isEnabled_ = enabled;
        if (enabled) {
            AutoScanManager.instance.start_();
        }
    }
    /** Sets whether the keyboard scan time is used. */
    static setInKeyboard(inKeyboard) {
        // TODO(b/314203187): Not null asserted, check that this is correct.
        AutoScanManager.instance.inKeyboard_ = inKeyboard;
    }
    /** Update this.keyboardScanTime_ to |scanTime|, in milliseconds. */
    static setKeyboardScanTime(scanTime) {
        // TODO(b/314203187): Not null asserted, check that this is correct.
        AutoScanManager.instance.keyboardScanTime_ = scanTime;
        if (AutoScanManager.instance.inKeyboard_) {
            AutoScanManager.restartIfRunning();
        }
    }
    /**
     * Update this.primaryScanTime_ to |scanTime|. Then, if auto-scan is currently
     * running, restart it.
     * @param scanTime Auto-scan interval time in milliseconds.
     */
    static setPrimaryScanTime(scanTime) {
        // TODO(b/314203187): Not null asserted, check that this is correct.
        AutoScanManager.instance.primaryScanTime_ = scanTime;
        AutoScanManager.restartIfRunning();
    }
    // ============== Private Methods ================
    /** Return true if auto-scan is currently running. Otherwise return false. */
    isRunning_() {
        return this.isEnabled_;
    }
    /**
     * Set the window to move to the next node at an interval in milliseconds
     * depending on where the user is navigating. Currently,
     * this.keyboardScanTime_ is used as the interval if the user is
     * navigating in the virtual keyboard, and this.primaryScanTime_ is used
     * otherwise. Does not do anything if AutoScanManager is already scanning.
     */
    start_() {
        if (this.primaryScanTime_ === NOT_INITIALIZED || this.intervalID_ ||
            SwitchAccess.mode === Mode.POINT_SCAN) {
            return;
        }
        let currentScanTime = this.primaryScanTime_;
        if (SwitchAccess.improvedTextInputEnabled() && this.inKeyboard_ &&
            this.keyboardScanTime_ !== NOT_INITIALIZED) {
            currentScanTime = this.keyboardScanTime_;
        }
        this.intervalID_ = setInterval(() => {
            if (SwitchAccess.mode === Mode.POINT_SCAN) {
                this.stop_();
                return;
            }
            Navigator.byItem.moveForward();
        }, currentScanTime);
    }
    /** Stop the window from moving to the next node at a fixed interval. */
    stop_() {
        clearInterval(this.intervalID_);
        this.intervalID_ = undefined;
    }
}
// Private to module.
/** Sentinel value that indicates an uninitialized scan time. */
const NOT_INITIALIZED = -1;
TestImportManager.exportForTesting(AutoScanManager);

// Copyright 2020 The Chromium Authors
// Use of this source code is governed by a BSD-style license that can be
// found in the LICENSE file.
/**
 * Saves computed values to avoid recalculating them repeatedly.
 *
 * Caches are single-use, and abandoned after the top-level question is answered
 * (e.g. what are all the interesting descendants of this node?)
 */
class SACache {
    isActionable = new Map();
    isGroup = new Map();
    isInterestingSubtree = new Map();
}
TestImportManager.exportForTesting(SACache);

// Copyright 2019 The Chromium Authors
// Use of this source code is governed by a BSD-style license that can be
// found in the LICENSE file.
var MenuAction$b = chrome.accessibilityPrivate.SwitchAccessMenuAction;
/**
 * This interface represents some object or group of objects on screen
 *     that Switch Access may be interested in interacting with.
 *
 * There is no guarantee of uniqueness; two distinct SAChildNodes may refer
 *     to the same object. However, it is expected that any pair of
 *     SAChildNodes referring to the same interesting object are equal
 *     (calling .equals() returns true).
 */
class SAChildNode {
    isFocused_ = false;
    next_ = null;
    previous_ = null;
    valid_ = true;
    // ================= Getters and setters =================
    get group() {
        return null;
    }
    set next(newVal) {
        this.next_ = newVal;
    }
    /** Returns the next node in pre-order traversal. */
    get next() {
        let next = this;
        while (true) {
            // TODO(b/314203187): Not null asserted, check that this is correct.
            next = next.next_;
            if (!next) {
                this.onInvalidNavigation_(ErrorType.NEXT_UNDEFINED, 'Next node must be set on all SAChildNodes before navigating');
            }
            if (this === next) {
                this.onInvalidNavigation_(ErrorType.NEXT_INVALID, 'No valid next node');
            }
            if (next.isValidAndVisible()) {
                return next;
            }
        }
    }
    set previous(newVal) {
        this.previous_ = newVal;
    }
    /** Returns the previous node in pre-order traversal. */
    get previous() {
        let previous = this;
        while (true) {
            // TODO(b/314203187): Not null asserted, check that this is correct.
            previous = previous.previous_;
            if (!previous) {
                this.onInvalidNavigation_(ErrorType.PREVIOUS_UNDEFINED, 'Previous node must be set on all SAChildNodes before navigating');
            }
            if (this === previous) {
                this.onInvalidNavigation_(ErrorType.PREVIOUS_INVALID, 'No valid previous node');
            }
            if (previous.isValidAndVisible()) {
                return previous;
            }
        }
    }
    // ================= General methods =================
    /** Performs the node's default action. */
    doDefaultAction() {
        if (!this.isFocused_) {
            return;
        }
        if (this.isGroup()) {
            this.performAction(MenuAction$b.DRILL_DOWN);
        }
        else {
            this.performAction(MenuAction$b.SELECT);
        }
    }
    /** Given a menu action, returns whether it can be performed on this node. */
    hasAction(action) {
        return this.actions.includes(action);
    }
    /** Returns whether the node is currently focused by Switch Access. */
    isFocused() {
        return this.isFocused_;
    }
    /**
     * Returns whether this node is still both valid and visible onscreen (e.g.
     *    has a location, and, if representing an AutomationNode, not hidden,
     *    not offscreen, not invisible).
     */
    isValidAndVisible() {
        return this.valid_ && Boolean(this.location);
    }
    /** Called when this node becomes the primary highlighted node. */
    onFocus() {
        this.isFocused_ = true;
        FocusRingManager.setFocusedNode(this);
    }
    /** Called when this node stops being the primary highlighted node. */
    onUnfocus() {
        this.isFocused_ = false;
    }
    // ================= Debug methods =================
    /** String-ifies the node (for debugging purposes). */
    debugString(wholeTree, prefix = '', currentNode = null) {
        if (this.isGroup() && wholeTree) {
            // TODO(b/314203187): Not null asserted, check that this is correct.
            return this.asRootNode().debugString(wholeTree, prefix + '  ', currentNode);
        }
        let str = this.constructor.name + ' role(' + this.role + ') ';
        if (this.automationNode.name) {
            str += 'name(' + this.automationNode.name + ') ';
        }
        const loc = this.location;
        if (loc) {
            str += 'loc(' + RectUtil.toString(loc) + ') ';
        }
        if (this.isGroup()) {
            str += '[isGroup]';
        }
        return str;
    }
    // ================= Private methods =================
    onInvalidNavigation_(error, message) {
        this.valid_ = false;
        throw SwitchAccess.error(error, message, true /* shouldRecover */);
    }
    /** @return Whether to ignore when computing the SARootNode's location. */
    ignoreWhenComputingUnionOfBoundingBoxes() {
        return false;
    }
}
/**
 * This class represents the root node of a Switch Access traversal group.
 */
class SARootNode {
    children_ = [];
    automationNode_;
    /**
     * @param autoNode The automation node that most closely contains all of
     * this node's children.
     */
    constructor(autoNode) {
        this.automationNode_ = autoNode;
    }
    // ================= Getters and setters =================
    /**
     * @return The automation node that most closely contains all of this node's
     * children.
     */
    get automationNode() {
        return this.automationNode_;
    }
    set children(newVal) {
        this.children_ = newVal;
        this.connectChildren_();
    }
    get children() {
        return this.children_;
    }
    get firstChild() {
        if (this.children_.length > 0) {
            return this.children_[0];
        }
        else {
            throw SwitchAccess.error(ErrorType.NO_CHILDREN, 'Root nodes must contain children.', true /* shouldRecover */);
        }
    }
    get lastChild() {
        if (this.children_.length > 0) {
            return this.children_[this.children_.length - 1];
        }
        else {
            throw SwitchAccess.error(ErrorType.NO_CHILDREN, 'Root nodes must contain children.', true /* shouldRecover */);
        }
    }
    get location() {
        const children = this.children_.filter(c => !c.ignoreWhenComputingUnionOfBoundingBoxes());
        const childLocations = children.map(c => c.location).filter(l => l);
        return RectUtil.unionAll(childLocations);
    }
    // ================= General methods =================
    equals(other) {
        if (!other) {
            return false;
        }
        if (this.children_.length !== other.children_.length) {
            return false;
        }
        let result = true;
        for (let i = 0; i < this.children_.length; i++) {
            if (!this.children_[i]) {
                console.error(SwitchAccess.error(ErrorType.NULL_CHILD, 'Child cannot be null.'));
                return false;
            }
            result = result && this.children_[i].equals(other.children_[i]);
        }
        return result;
    }
    /**
     * Looks for and returns the specified node within this node's children.
     * If no equivalent node is found, returns null.
     */
    findChild(node) {
        for (const child of this.children_) {
            if (child.isEquivalentTo(node)) {
                return child;
            }
        }
        return null;
    }
    isEquivalentTo(node) {
        if (node instanceof SARootNode) {
            return this.equals(node);
        }
        if (node instanceof SAChildNode) {
            return node.isEquivalentTo(this);
        }
        return false;
    }
    isValidGroup() {
        // Must have one interesting child whose location is important.
        return this.children_
            .filter((child) => !(child.ignoreWhenComputingUnionOfBoundingBoxes()) &&
            child.isValidAndVisible())
            .length >= 1;
    }
    firstValidChild() {
        const children = this.children_.filter(child => child.isValidAndVisible());
        return children.length > 0 ? children[0] : null;
    }
    /** Called when a group is set as the current group. */
    onFocus() { }
    /** Called when a group is no longer the current group. */
    onUnfocus() { }
    /** Called when a group is explicitly exited. */
    onExit() { }
    /** Called when a group should recalculate its children. */
    refreshChildren() {
        this.children =
            this.children.filter((child) => child.isValidAndVisible());
    }
    /** Called when the group's children may have changed. */
    refresh() { }
    // ================= Debug methods =================
    /**
     * String-ifies the node (for debugging purposes).
     * @param wholeTree Whether to recursively descend the tree
     * @param currentNode the currently focused node, to mark.
     */
    debugString(wholeTree = false, prefix = '', currentNode = null) {
        let str = 'Root: ' + this.constructor.name + ' ' + this.automationNode.role + ' ';
        if (this.automationNode.name) {
            str += 'name(' + this.automationNode.name + ') ';
        }
        const loc = this.location;
        if (loc) {
            str += 'loc(' + RectUtil.toString(loc) + ') ';
        }
        for (const child of this.children) {
            str += '\n' + prefix + ((child.equals(currentNode)) ? ' * ' : ' - ');
            str += child.debugString(wholeTree, prefix, currentNode);
        }
        return str;
    }
    // ================= Private methods =================
    /** Helper function to connect children. */
    connectChildren_() {
        if (this.children_.length < 1) {
            console.error(SwitchAccess.error(ErrorType.NO_CHILDREN, 'Root node must have at least 1 interesting child.'));
            return;
        }
        let previous = this.children_[this.children_.length - 1];
        for (let i = 0; i < this.children_.length; i++) {
            const current = this.children_[i];
            previous.next = current;
            current.previous = previous;
            previous = current;
        }
    }
}
TestImportManager.exportForTesting(SARootNode);

// Copyright 2017 The Chromium Authors
// Use of this source code is governed by a BSD-style license that can be
// found in the LICENSE file.
const StateType$2 = chrome.automation.StateType;
const Restriction = chrome.automation.Restriction;
const RoleType$8 = chrome.automation.RoleType;
const DefaultActionVerb = chrome.automation.DefaultActionVerb;
/**
 * Contains predicates for the chrome automation API. The following basic
 * predicates are available:
 *    - isActionable
 *    - isGroup
 *    - isInteresting
 *    - isInterestingSubtree
 *    - isVisible
 *    - isTextInput
 *
 * In addition to these basic predicates, there are also methods to get the
 * restrictions required by TreeWalker for specific traversal situations.
 */
var SwitchAccessPredicate;
(function (SwitchAccessPredicate) {
    SwitchAccessPredicate.GROUP_INTERESTING_CHILD_THRESHOLD = 2;
    /**
     * Returns true if |node| is actionable, meaning that a user can interact with
     * it in some way.
     */
    function isActionable(node, cache) {
        // TODO(b/314203187): Not null asserted, check that this is correct.
        if (cache.isActionable.has(node)) {
            return cache.isActionable.get(node);
        }
        const defaultActionVerb = node.defaultActionVerb;
        // Skip things that are offscreen or invisible.
        if (!SwitchAccessPredicate.isVisible(node)) {
            cache.isActionable.set(node, false);
            return false;
        }
        // Skip things that are disabled.
        if (node.restriction === Restriction.DISABLED) {
            cache.isActionable.set(node, false);
            return false;
        }
        // These web containers are not directly actionable.
        if (AutomationPredicate.structuralContainer(node)) {
            cache.isActionable.set(node, false);
            return false;
        }
        // Check various indicators that the node is actionable.
        const actionableRole = AutomationPredicate.roles([RoleType$8.BUTTON, RoleType$8.SLIDER, RoleType$8.TAB]);
        if (actionableRole(node)) {
            cache.isActionable.set(node, true);
            return true;
        }
        if (AutomationPredicate.comboBox(node) ||
            SwitchAccessPredicate.isTextInput(node)) {
            cache.isActionable.set(node, true);
            return true;
        }
        if (defaultActionVerb &&
            (defaultActionVerb === DefaultActionVerb.ACTIVATE ||
                defaultActionVerb === DefaultActionVerb.CHECK ||
                defaultActionVerb === DefaultActionVerb.OPEN ||
                defaultActionVerb === DefaultActionVerb.PRESS ||
                defaultActionVerb === DefaultActionVerb.SELECT ||
                defaultActionVerb === DefaultActionVerb.UNCHECK)) {
            cache.isActionable.set(node, true);
            return true;
        }
        if (node.role === RoleType$8.LIST_ITEM &&
            defaultActionVerb === DefaultActionVerb.CLICK) {
            cache.isActionable.set(node, true);
            return true;
        }
        // Focusable items should be surfaced as either groups or actionable. So
        // should menu items.
        // Current heuristic is to show as actionble any focusable item where no
        // child is an interesting subtree.
        // TODO(b/314203187): Not null asserted, check that this is correct.
        if (node.state[StateType$2.FOCUSABLE] ||
            node.role === RoleType$8.MENU_ITEM) {
            const result = !node.children.some((child) => SwitchAccessPredicate.isInterestingSubtree(child, cache));
            cache.isActionable.set(node, result);
            return result;
        }
        return false;
    }
    SwitchAccessPredicate.isActionable = isActionable;
    /**
     * Returns true if |node| is a group, meaning that the node has more than one
     * interesting descendant, and that its interesting descendants exist in more
     * than one subtree of its immediate children.
     *
     * Additionally, for |node| to be a group, it cannot have the same bounding
     * box as its scope.
     */
    function isGroup(node, scope, cache) {
        // If node is invalid (undefined or an undefined role), return false.
        if (!node || !node.role) {
            return false;
        }
        if (cache.isGroup.has(node)) {
            // TODO(b/314203187): Not null asserted, check that this is correct.
            return cache.isGroup.get(node);
        }
        const scopeEqualsNode = scope &&
            (scope instanceof SARootNode ? scope.isEquivalentTo(node) :
                scope === node);
        if (scope && !scopeEqualsNode &&
            RectUtil.equal(node.location, scope.location)) {
            cache.isGroup.set(node, false);
            return false;
        }
        // TODO(b/314203187): Not null asserted, check that this is correct.
        if (node.state[StateType$2.INVISIBLE]) {
            cache.isGroup.set(node, false);
            return false;
        }
        if (node.role === RoleType$8.KEYBOARD) {
            cache.isGroup.set(node, true);
            return true;
        }
        let interestingBranchesCount = SwitchAccessPredicate.isActionable(node, cache) ? 1 : 0;
        let child = node.firstChild;
        while (child) {
            if (SwitchAccessPredicate.isInterestingSubtree(child, cache)) {
                interestingBranchesCount += 1;
            }
            if (interestingBranchesCount >=
                SwitchAccessPredicate.GROUP_INTERESTING_CHILD_THRESHOLD) {
                cache.isGroup.set(node, true);
                return true;
            }
            child = child.nextSibling;
        }
        cache.isGroup.set(node, false);
        return false;
    }
    SwitchAccessPredicate.isGroup = isGroup;
    /**
     * Returns true if |node| is interesting for the user, meaning that |node|
     * is either actionable or a group.
     */
    function isInteresting(node, scope, cache) {
        cache = cache || new SACache();
        return SwitchAccessPredicate.isActionable(node, cache) ||
            SwitchAccessPredicate.isGroup(node, scope, cache);
    }
    SwitchAccessPredicate.isInteresting = isInteresting;
    /** Returns true if the element is visible to the user for any reason. */
    function isVisible(node) {
        // TODO(b/314203187): Not null asserted, check that this is correct.
        return Boolean(!node.state[StateType$2.OFFSCREEN] && node.location &&
            node.location.top >= 0 && node.location.left >= 0 &&
            !node.state[StateType$2.INVISIBLE]);
    }
    SwitchAccessPredicate.isVisible = isVisible;
    /**
     * Returns true if there is an interesting node in the subtree containing
     * |node| as its root (including |node| itself).
     *
     * This function does not call isInteresting directly, because that would
     * cause a loop (isInteresting calls isGroup, and isGroup calls
     * isInterestingSubtree).
     */
    function isInterestingSubtree(node, cache) {
        cache = cache || new SACache();
        if (cache.isInterestingSubtree.has(node)) {
            // TODO(b/314203187): Not null asserted, check that this is correct.
            return cache.isInterestingSubtree.get(node);
        }
        const result = SwitchAccessPredicate.isActionable(node, cache) ||
            node.children.some(child => SwitchAccessPredicate.isInterestingSubtree(child, cache));
        cache.isInterestingSubtree.set(node, result);
        return result;
    }
    SwitchAccessPredicate.isInterestingSubtree = isInterestingSubtree;
    /** Returns true if |node| is an element that contains editable text. */
    function isTextInput(node) {
        // TODO(b/314203187): Not null asserted, check that this is correct.
        return Boolean(node && node.state[StateType$2.EDITABLE]);
    }
    SwitchAccessPredicate.isTextInput = isTextInput;
    /** Returns true if |node| should be considered a window. */
    function isWindow(node) {
        return Boolean(node &&
            (node.role === RoleType$8.WINDOW ||
                (node.role === RoleType$8.CLIENT && node.parent &&
                    node.parent.role === RoleType$8.WINDOW)));
    }
    SwitchAccessPredicate.isWindow = isWindow;
    /**
     * Returns a Restrictions object ready to be passed to AutomationTreeWalker.
     */
    function restrictions(scope) {
        const cache = new SACache();
        return {
            leaf: SwitchAccessPredicate.leaf(scope, cache),
            root: SwitchAccessPredicate.root(scope),
            visit: SwitchAccessPredicate.visit(scope, cache),
        };
    }
    SwitchAccessPredicate.restrictions = restrictions;
    /**
     * Creates a function that confirms if |node| is a terminal leaf node of a
     * SwitchAccess scope tree when |scope| is the root.
     */
    function leaf(scope, cache) {
        return (node) => !SwitchAccessPredicate.isInterestingSubtree(node, cache) ||
            (scope !== node &&
                SwitchAccessPredicate.isInteresting(node, scope, cache));
    }
    SwitchAccessPredicate.leaf = leaf;
    /**
     * Creates a function that confirms if |node| is the root of a SwitchAccess
     * scope tree when |scope| is the root.
     */
    function root(scope) {
        return (node) => scope === node;
    }
    SwitchAccessPredicate.root = root;
    /**
     * Creates a function that determines whether |node| is to be visited in the
     * SwitchAccess scope tree with |scope| as the root.
     */
    function visit(scope, cache) {
        return (node) => node.role !== RoleType$8.DESKTOP &&
            SwitchAccessPredicate.isInteresting(node, scope, cache);
    }
    SwitchAccessPredicate.visit = visit;
})(SwitchAccessPredicate || (SwitchAccessPredicate = {}));
TestImportManager.exportForTesting(['SwitchAccessPredicate', SwitchAccessPredicate]);

// Copyright 2019 The Chromium Authors
// Use of this source code is governed by a BSD-style license that can be
// found in the LICENSE file.
const EventType$7 = chrome.automation.EventType;
var MenuAction$a = chrome.accessibilityPrivate.SwitchAccessMenuAction;
var RoleType$7 = chrome.automation.RoleType;
/**
 * This class handles the behavior of the back button.
 */
class BackButtonNode extends SAChildNode {
    /** The group that the back button is shown for. */
    group_;
    locationChangedHandler_;
    static automationNode_;
    static clickHandler_;
    static locationForTesting;
    constructor(group) {
        super();
        this.group_ = group;
    }
    // ================= Getters and setters =================
    get actions() {
        return [MenuAction$a.SELECT];
    }
    get automationNode() {
        return BackButtonNode.automationNode_;
    }
    get group() {
        return this.group_;
    }
    get location() {
        if (BackButtonNode.locationForTesting) {
            return BackButtonNode.locationForTesting;
        }
        if (this.automationNode) {
            return this.automationNode.location;
        }
        return undefined;
    }
    get role() {
        return RoleType$7.BUTTON;
    }
    // ================= General methods =================
    asRootNode() {
        return undefined;
    }
    equals(other) {
        return other instanceof BackButtonNode;
    }
    isEquivalentTo(node) {
        return node instanceof BackButtonNode || this.automationNode === node;
    }
    isGroup() {
        return false;
    }
    isValidAndVisible() {
        return this.group_.isValidGroup();
    }
    onFocus() {
        super.onFocus();
        chrome.accessibilityPrivate.updateSwitchAccessBubble(chrome.accessibilityPrivate.SwitchAccessBubble.BACK_BUTTON, true /* show */, this.group_.location);
        BackButtonNode.findAutomationNode_();
        this.locationChangedHandler_ = new RepeatedEventHandler(this.group_.automationNode, chrome.automation.EventType.LOCATION_CHANGED, () => FocusRingManager.setFocusedNode(this), { exactMatch: true, allAncestors: true });
    }
    onUnfocus() {
        super.onUnfocus();
        chrome.accessibilityPrivate.updateSwitchAccessBubble(chrome.accessibilityPrivate.SwitchAccessBubble.BACK_BUTTON, false /* show */);
        if (this.locationChangedHandler_) {
            this.locationChangedHandler_.stop();
        }
    }
    performAction(action) {
        if (action === MenuAction$a.SELECT && this.automationNode) {
            BackButtonNode.onClick_();
            return ActionResponse.CLOSE_MENU;
        }
        return ActionResponse.NO_ACTION_TAKEN;
    }
    ignoreWhenComputingUnionOfBoundingBoxes() {
        return true;
    }
    // ================= Debug methods =================
    debugString(wholeTree, prefix = '', currentNode = null) {
        if (!this.automationNode) {
            return 'BackButtonNode';
        }
        return super.debugString(wholeTree, prefix, currentNode);
    }
    // ================= Static methods =================
    /** Looks for the back button automation node. */
    static findAutomationNode_() {
        if (BackButtonNode.automationNode_ && BackButtonNode.automationNode_.role) {
            return;
        }
        SwitchAccess.findNodeMatching({
            role: RoleType$7.BUTTON,
            attributes: { className: 'SwitchAccessBackButtonView' },
        }, BackButtonNode.saveAutomationNode_);
    }
    /**
     * This function defines the behavior that should be taken when the back
     * button is pressed.
     */
    static onClick_() {
        if (MenuManager.isMenuOpen()) {
            ActionManager.exitCurrentMenu();
        }
        else {
            Navigator.byItem.exitGroupUnconditionally();
        }
    }
    /** Saves the back button automation node. */
    static saveAutomationNode_(automationNode) {
        BackButtonNode.automationNode_ = automationNode;
        if (BackButtonNode.clickHandler_) {
            BackButtonNode.clickHandler_.setNodes(automationNode);
        }
        else {
            BackButtonNode.clickHandler_ = new EventHandler(automationNode, EventType$7.CLICKED, BackButtonNode.onClick_);
        }
    }
}
TestImportManager.exportForTesting(BackButtonNode);

// Copyright 2019 The Chromium Authors
// Use of this source code is governed by a BSD-style license that can be
// found in the LICENSE file.
var AutomationActionType = chrome.automation.ActionType;
var EventType$6 = chrome.automation.EventType;
var MenuAction$9 = chrome.accessibilityPrivate.SwitchAccessMenuAction;
/**
 * This class handles interactions with an onscreen element based on a single
 * AutomationNode.
 */
class BasicNode extends SAChildNode {
    baseNode_;
    parent_;
    locationChangedHandler_;
    isActionable_;
    static creators_ = [];
    constructor(baseNode, parent) {
        super();
        this.baseNode_ = baseNode;
        this.parent_ = parent;
        this.isActionable_ = !this.isGroup() ||
            SwitchAccessPredicate.isActionable(baseNode, new SACache());
    }
    // ================= Getters and setters =================
    get actions() {
        const actions = [];
        if (this.isActionable_) {
            actions.push(MenuAction$9.SELECT);
        }
        if (this.isGroup()) {
            actions.push(MenuAction$9.DRILL_DOWN);
        }
        const ancestor = this.getScrollableAncestor_();
        // TODO(b/314203187): Not null asserted, check that this is correct.
        if (ancestor.scrollable) {
            if (ancestor.scrollX > ancestor.scrollXMin) {
                actions.push(MenuAction$9.SCROLL_LEFT);
            }
            if (ancestor.scrollX < ancestor.scrollXMax) {
                actions.push(MenuAction$9.SCROLL_RIGHT);
            }
            if (ancestor.scrollY > ancestor.scrollYMin) {
                actions.push(MenuAction$9.SCROLL_UP);
            }
            if (ancestor.scrollY < ancestor.scrollYMax) {
                actions.push(MenuAction$9.SCROLL_DOWN);
            }
        }
        // Coerce enums to string arrays for comparison.
        const menuActions = Object.values(MenuAction$9);
        const standardActions = this.baseNode_.standardActions.filter((action) => menuActions.includes(action));
        return actions.concat(standardActions);
    }
    get automationNode() {
        return this.baseNode_;
    }
    get location() {
        return this.baseNode_.location;
    }
    get role() {
        // TODO(b/314203187): Not null asserted, check that this is correct.
        return this.baseNode_.role;
    }
    // ================= General methods =================
    asRootNode() {
        if (!this.isGroup()) {
            return undefined;
        }
        return BasicRootNode.buildTree(this.baseNode_);
    }
    equals(rhs) {
        if (!rhs || !(rhs instanceof BasicNode)) {
            return false;
        }
        const other = rhs;
        return other.baseNode_ === this.baseNode_;
    }
    isEquivalentTo(node) {
        if (node instanceof BasicNode) {
            return this.baseNode_ === node.baseNode_;
        }
        if (node instanceof BasicRootNode) {
            return this.baseNode_ === node.automationNode;
        }
        if (node instanceof SAChildNode) {
            return node.isEquivalentTo(this);
        }
        return this.baseNode_ === node;
    }
    isGroup() {
        const cache = new SACache();
        return SwitchAccessPredicate.isGroup(this.baseNode_, this.parent_, cache);
    }
    isValidAndVisible() {
        // Nodes may have been deleted or orphaned.
        if (!this.baseNode_ || !this.baseNode_.role) {
            return false;
        }
        return SwitchAccessPredicate.isVisible(this.baseNode_) &&
            super.isValidAndVisible();
    }
    onFocus() {
        super.onFocus();
        this.locationChangedHandler_ = new RepeatedEventHandler(this.baseNode_, EventType$6.LOCATION_CHANGED, () => {
            if (this.isValidAndVisible()) {
                FocusRingManager.setFocusedNode(this);
            }
            else {
                Navigator.byItem.moveToValidNode();
            }
        }, { exactMatch: true, allAncestors: true });
    }
    onUnfocus() {
        super.onUnfocus();
        if (this.locationChangedHandler_) {
            this.locationChangedHandler_.stop();
        }
    }
    performAction(action) {
        let ancestor;
        switch (action) {
            case MenuAction$9.DRILL_DOWN:
                if (this.isGroup()) {
                    Navigator.byItem.enterGroup();
                    return ActionResponse.CLOSE_MENU;
                }
                // Should not happen.
                console.error('Action DRILL_DOWN received on non-group node.');
                return ActionResponse.NO_ACTION_TAKEN;
            case MenuAction$9.SELECT:
                this.baseNode_.doDefault();
                return ActionResponse.CLOSE_MENU;
            case MenuAction$9.SCROLL_DOWN:
                ancestor = this.getScrollableAncestor_();
                if (ancestor.scrollable) {
                    ancestor.scrollDown(() => { });
                }
                return ActionResponse.RELOAD_MENU;
            case MenuAction$9.SCROLL_UP:
                ancestor = this.getScrollableAncestor_();
                if (ancestor.scrollable) {
                    ancestor.scrollUp(() => { });
                }
                return ActionResponse.RELOAD_MENU;
            case MenuAction$9.SCROLL_RIGHT:
                ancestor = this.getScrollableAncestor_();
                if (ancestor.scrollable) {
                    ancestor.scrollRight(() => { });
                }
                return ActionResponse.RELOAD_MENU;
            case MenuAction$9.SCROLL_LEFT:
                ancestor = this.getScrollableAncestor_();
                if (ancestor.scrollable) {
                    ancestor.scrollLeft(() => { });
                }
                return ActionResponse.RELOAD_MENU;
            default:
                const actions = Object.values(AutomationActionType);
                const automationAction = actions.find((a) => a === action);
                if (automationAction) {
                    this.baseNode_.performStandardAction(automationAction);
                }
                return ActionResponse.CLOSE_MENU;
        }
    }
    // ================= Private methods =================
    getScrollableAncestor_() {
        let ancestor = this.baseNode_;
        while (!ancestor.scrollable && ancestor.parent) {
            ancestor = ancestor.parent;
        }
        return ancestor;
    }
    // ================= Static methods =================
    static create(baseNode, parent) {
        const item = BasicNode.creators.find((creator) => creator.predicate(baseNode));
        if (item) {
            return item.creator(baseNode, parent);
        }
        return new BasicNode(baseNode, parent);
    }
    static get creators() {
        return BasicNode.creators_;
    }
}
/**
 * This class handles constructing and traversing a group of onscreen elements
 * based on all the interesting descendants of a single AutomationNode.
 */
class BasicRootNode extends SARootNode {
    static builders_ = [];
    childrenChangedHandler_;
    invalidated_ = false;
    /**
     * WARNING: If you call this constructor, you must *explicitly* set children.
     *     Use the static function BasicRootNode.buildTree for most use cases.
     */
    constructor(baseNode) {
        super(baseNode);
    }
    // ================= Getters and setters =================
    get location() {
        return this.automationNode.location || super.location;
    }
    // ================= General methods =================
    equals(other) {
        if (!(other instanceof BasicRootNode)) {
            return false;
        }
        return super.equals(other) && this.automationNode === other.automationNode;
    }
    isEquivalentTo(node) {
        if (node instanceof BasicRootNode || node instanceof BasicNode) {
            return this.automationNode === node.automationNode;
        }
        if (node instanceof SAChildNode) {
            return node.isEquivalentTo(this);
        }
        return this.automationNode === node;
    }
    isValidGroup() {
        if (!this.automationNode.role) {
            // If the underlying automation node has been invalidated, return false.
            return false;
        }
        return !this.invalidated_ &&
            SwitchAccessPredicate.isVisible(this.automationNode) &&
            super.isValidGroup();
    }
    onFocus() {
        super.onFocus();
        this.childrenChangedHandler_ = new RepeatedEventHandler(this.automationNode, EventType$6.CHILDREN_CHANGED, event => {
            const cache = new SACache();
            if (SwitchAccessPredicate.isInterestingSubtree(event.target, cache)) {
                this.refresh();
            }
        });
    }
    onUnfocus() {
        super.onUnfocus();
        if (this.childrenChangedHandler_) {
            this.childrenChangedHandler_.stop();
        }
    }
    refreshChildren() {
        const childConstructor = (node) => BasicNode.create(node, this);
        try {
            BasicRootNode.findAndSetChildren(this, childConstructor);
        }
        catch (e) {
            this.invalidated_ = true;
        }
    }
    refresh() {
        // Find the currently focused child.
        let focusedChild = null;
        for (const child of this.children) {
            if (child.isFocused()) {
                focusedChild = child;
                break;
            }
        }
        // Update this BasicRootNode's children.
        this.refreshChildren();
        if (this.invalidated_) {
            this.onUnfocus();
            Navigator.byItem.moveToValidNode();
            return;
        }
        // Set the new instance of that child to be the focused node.
        if (focusedChild) {
            for (const child of this.children) {
                if (child.isEquivalentTo(focusedChild)) {
                    Navigator.byItem.forceFocusedNode(child);
                    return;
                }
            }
        }
        // If we didn't find a match, fall back and reset.
        Navigator.byItem.moveToValidNode();
    }
    // ================= Static methods =================
    static buildTree(rootNode) {
        const item = BasicRootNode.builders.find((builder) => builder.predicate(rootNode));
        if (item) {
            return item.builder(rootNode);
        }
        const root = new BasicRootNode(rootNode);
        const childConstructor = (node) => BasicNode.create(node, root);
        BasicRootNode.findAndSetChildren(root, childConstructor);
        return root;
    }
    /**
     * Helper function to connect tree elements, given the root node and a
     * constructor for the child type.
     * @param childConstructor Constructs a child node from an automation node.
     */
    static findAndSetChildren(root, childConstructor) {
        const interestingChildren = BasicRootNode.getInterestingChildren(root);
        const children = interestingChildren.map(childConstructor)
            .filter(child => child.isValidAndVisible());
        if (children.length < 1) {
            throw SwitchAccess.error(ErrorType.NO_CHILDREN, 'Root node must have at least 1 interesting child.', true /* shouldRecover */);
        }
        children.push(new BackButtonNode(root));
        root.children = children;
    }
    static getInterestingChildren(root) {
        if (root instanceof BasicRootNode) {
            root = root.automationNode;
        }
        if (root.children.length === 0) {
            return [];
        }
        const interestingChildren = [];
        const treeWalker = new AutomationTreeWalker(root, constants.Dir.FORWARD, SwitchAccessPredicate.restrictions(root));
        let node = treeWalker.next().node;
        while (node) {
            interestingChildren.push(node);
            node = treeWalker.next().node;
        }
        return interestingChildren;
    }
    static get builders() {
        return BasicRootNode.builders_;
    }
}
TestImportManager.exportForTesting(BasicNode, BasicRootNode);

// Copyright 2019 The Chromium Authors
// Use of this source code is governed by a BSD-style license that can be
// found in the LICENSE file.
var MenuAction$8 = chrome.accessibilityPrivate.SwitchAccessMenuAction;
var RoleType$6 = chrome.automation.RoleType;
/**
 * This class handles the grouping of nodes that are not grouped in the
 *     automation tree. They are defined by their parent and child nodes.
 * Ex: Nodes in the virtual keyboard have no intermediate grouping, but should
 *     be grouped by row.
 */
class GroupNode extends SAChildNode {
    children_;
    containingNode_;
    /**
     * @param children The nodes that this group contains.
     *     Should not include the back button.
     * @param containingNode The automation node most closely containing the
     *     children.
     */
    constructor(children_, containingNode_) {
        super();
        this.children_ = children_;
        this.containingNode_ = containingNode_;
    }
    // ================= Getters and setters =================
    get actions() {
        return [MenuAction$8.DRILL_DOWN];
    }
    get automationNode() {
        return this.containingNode_;
    }
    get location() {
        // TODO(b/314203187): Not null asserted, check that this is correct.
        const childLocations = this.children_.filter(c => c.isValidAndVisible()).map(c => c.location);
        return RectUtil.unionAll(childLocations);
    }
    get role() {
        return RoleType$6.GROUP;
    }
    // ================= General methods =================
    asRootNode() {
        const root = new SARootNode(this.containingNode_);
        // Make a copy of the children array.
        const children = [...this.children_];
        children.push(new BackButtonNode(root));
        root.children = children;
        return root;
    }
    equals(other) {
        if (!(other instanceof GroupNode)) {
            return false;
        }
        if (other.children_.length !== this.children_.length) {
            return false;
        }
        for (let i = 0; i < this.children_.length; i++) {
            if (!other.children_[i].equals(this.children_[i])) {
                return false;
            }
        }
        return true;
    }
    isEquivalentTo(node) {
        if (node instanceof GroupNode) {
            return this.equals(node);
        }
        for (const child of this.children_) {
            if (child.isEquivalentTo(node)) {
                return true;
            }
        }
        return false;
    }
    isGroup() {
        return true;
    }
    isValidAndVisible() {
        for (const child of this.children_) {
            if (child.isValidAndVisible()) {
                return super.isValidAndVisible();
            }
        }
        return false;
    }
    performAction(action) {
        if (action === MenuAction$8.DRILL_DOWN) {
            Navigator.byItem.enterGroup();
            return ActionResponse.CLOSE_MENU;
        }
        return ActionResponse.NO_ACTION_TAKEN;
    }
    // ================= Static methods =================
    /** Assumes nodes are visually in rows. */
    static separateByRow(nodes, containingNode) {
        const result = [];
        for (let i = 0; i < nodes.length;) {
            const children = [];
            children.push(nodes[i]);
            i++;
            while (i < nodes.length &&
                RectUtil.sameRow(children[0].location, nodes[i].location)) {
                children.push(nodes[i]);
                i++;
            }
            result.push(new GroupNode(children, containingNode));
        }
        return result;
    }
}
TestImportManager.exportForTesting(GroupNode);

// Copyright 2019 The Chromium Authors
// Use of this source code is governed by a BSD-style license that can be
// found in the LICENSE file.
const EventType$5 = chrome.automation.EventType;
var MenuAction$7 = chrome.accessibilityPrivate.SwitchAccessMenuAction;
const RoleType$5 = chrome.automation.RoleType;
/**
 * This class handles the behavior of keyboard nodes directly associated with a
 * single AutomationNode.
 */
class KeyboardNode extends BasicNode {
    static resetting = false;
    constructor(node, parent) {
        super(node, parent);
    }
    // ================= Getters and setters =================
    get actions() {
        return [MenuAction$7.SELECT];
    }
    // ================= General methods =================
    asRootNode() {
        return undefined;
    }
    isGroup() {
        return false;
    }
    isValidAndVisible() {
        if (super.isValidAndVisible()) {
            return true;
        }
        if (!KeyboardNode.resetting &&
            Navigator.byItem.currentGroupHasChild(this)) {
            // TODO(crbug/1130773): move this code to another location, if possible
            KeyboardNode.resetting = true;
            KeyboardRootNode.ignoreNextExit = true;
            Navigator.byItem.exitKeyboard().then(() => Navigator.byItem.enterKeyboard());
        }
        return false;
    }
    performAction(action) {
        if (action !== MenuAction$7.SELECT) {
            return ActionResponse.NO_ACTION_TAKEN;
        }
        const keyLocation = this.location;
        if (!keyLocation) {
            return ActionResponse.NO_ACTION_TAKEN;
        }
        // doDefault() does nothing on Virtual Keyboard buttons, so we must
        // simulate a mouse click.
        const center = RectUtil.center(keyLocation);
        EventGenerator.sendMouseClick(center.x, center.y, { delayMs: VK_KEY_PRESS_DURATION_MS });
        return ActionResponse.CLOSE_MENU;
    }
}
/**
 * This class handles the top-level Keyboard node, as well as the construction
 * of the Keyboard tree.
 */
class KeyboardRootNode extends BasicRootNode {
    static ignoreNextExit = false;
    static isVisible_ = false;
    static explicitStateChange_ = false;
    static object_;
    constructor(groupNode) {
        super(groupNode);
        KeyboardNode.resetting = false;
    }
    // ================= General methods =================
    isValidGroup() {
        // To ensure we can find the keyboard root node to appropriately respond to
        // visibility changes, never mark it as invalid.
        return true;
    }
    onExit() {
        if (KeyboardRootNode.ignoreNextExit) {
            KeyboardRootNode.ignoreNextExit = false;
            return;
        }
        // If the keyboard is currently visible, ignore the corresponding
        // state change.
        if (KeyboardRootNode.isVisible_) {
            KeyboardRootNode.explicitStateChange_ = true;
            chrome.accessibilityPrivate.setVirtualKeyboardVisible(false);
        }
        AutoScanManager.setInKeyboard(false);
    }
    refreshChildren() {
        KeyboardRootNode.findAndSetChildren_(this);
    }
    // ================= Static methods =================
    /** Creates the tree structure for the keyboard. */
    static buildTree() {
        KeyboardRootNode.loadKeyboard_();
        AutoScanManager.setInKeyboard(true);
        const keyboard = KeyboardRootNode.getKeyboardObject();
        if (!keyboard) {
            throw SwitchAccess.error(ErrorType.MISSING_KEYBOARD, 'Could not find keyboard in the automation tree', true /* shouldRecover */);
        }
        const root = new KeyboardRootNode(keyboard);
        KeyboardRootNode.findAndSetChildren_(root);
        return root;
    }
    /** Start listening for keyboard open/closed. */
    static startWatchingVisibility() {
        const keyboardObject = KeyboardRootNode.getKeyboardObject();
        if (!keyboardObject) {
            SwitchAccess.findNodeMatching({ role: RoleType$5.KEYBOARD }, KeyboardRootNode.startWatchingVisibility);
            return;
        }
        KeyboardRootNode.isVisible_ = KeyboardRootNode.isKeyboardVisible_();
        new EventHandler(keyboardObject, EventType$5.LOAD_COMPLETE, KeyboardRootNode.checkVisibilityChanged_)
            .start();
        new EventHandler(keyboardObject, EventType$5.STATE_CHANGED, KeyboardRootNode.checkVisibilityChanged_, { exactMatch: true })
            .start();
    }
    // ================= Private static methods =================
    static isKeyboardVisible_() {
        const keyboardObject = KeyboardRootNode.getKeyboardObject();
        return Boolean(keyboardObject && SwitchAccessPredicate.isVisible(keyboardObject) &&
            keyboardObject.find({ role: RoleType$5.ROOT_WEB_AREA }));
    }
    static checkVisibilityChanged_(_event) {
        const currentlyVisible = KeyboardRootNode.isKeyboardVisible_();
        if (currentlyVisible === KeyboardRootNode.isVisible_) {
            return;
        }
        KeyboardRootNode.isVisible_ = currentlyVisible;
        if (KeyboardRootNode.explicitStateChange_) {
            // When the user has explicitly shown / hidden the keyboard, do not
            // enter / exit the keyboard again to avoid looping / double-calls.
            KeyboardRootNode.explicitStateChange_ = false;
            return;
        }
        if (KeyboardRootNode.isVisible_) {
            Navigator.byItem.enterKeyboard();
        }
        else {
            Navigator.byItem.exitKeyboard();
        }
    }
    /** Helper function to connect tree elements, given the root node. */
    static findAndSetChildren_(root) {
        const childConstructor = (node) => new KeyboardNode(node, root);
        const interestingChildren = root.automationNode.findAll({ role: RoleType$5.BUTTON });
        const children = GroupNode.separateByRow(interestingChildren.map(childConstructor), root.automationNode);
        children.push(new BackButtonNode(root));
        root.children = children;
    }
    static getKeyboardObject() {
        if (!this.object_ || !this.object_.role) {
            this.object_ =
                Navigator.byItem.desktopNode.find({ role: RoleType$5.KEYBOARD });
        }
        return this.object_;
    }
    /** Loads the keyboard. */
    static loadKeyboard_() {
        if (KeyboardRootNode.isVisible_) {
            return;
        }
        chrome.accessibilityPrivate.setVirtualKeyboardVisible(true);
    }
}
BasicRootNode.builders.push({
    predicate: rootNode => rootNode.role === RoleType$5.KEYBOARD,
    builder: KeyboardRootNode.buildTree,
});
/**
 * The delay between keydown and keyup events on the virtual keyboard,
 * allowing the key press animation to display.
 */
const VK_KEY_PRESS_DURATION_MS = 100;
TestImportManager.exportForTesting(KeyboardNode, KeyboardRootNode);

// Copyright 2017 The Chromium Authors
// Use of this source code is governed by a BSD-style license that can be
// found in the LICENSE file.
const EventType$4 = chrome.automation.EventType;
const RoleType$4 = chrome.automation.RoleType;
let readyCallback;
const readyPromise = new Promise(resolve => readyCallback = resolve);
/**
 * The top-level class for the Switch Access accessibility feature. Handles
 * initialization and small matters that don't fit anywhere else in the
 * codebase.
 */
class SwitchAccess {
    static instance;
    static mode = Mode.ITEM_SCAN;
    constructor() { }
    static async init(desktop) {
        if (SwitchAccess.instance) {
            throw new Error('Cannot create two SwitchAccess.instances');
        }
        SwitchAccess.instance = new SwitchAccess();
        const currentFocus = await AsyncUtil.getFocus();
        await SwitchAccess.instance.waitForFocus_(desktop, currentFocus);
    }
    /** Starts Switch Access behavior. */
    static start() {
        KeyboardRootNode.startWatchingVisibility();
        Navigator.byItem.start();
        readyCallback();
    }
    static async ready() {
        return readyPromise;
    }
    /**
     * Returns whether or not the feature flag
     * for improved text input is enabled.
     */
    static improvedTextInputEnabled() {
        // TODO(b/314203187): Not null asserted, check that this is correct.
        return Flags.isEnabled(FlagName.SWITCH_ACCESS_TEXT);
    }
    /**
     * Helper function to robustly find a node fitting a given FindParams, even if
     * that node has not yet been created.
     * Used to find the menu and back button.
     */
    static findNodeMatching(findParams, foundCallback) {
        const desktop = Navigator.byItem.desktopNode;
        // First, check if the node is currently in the tree.
        let node = desktop.find(findParams);
        if (node) {
            foundCallback(node);
            return;
        }
        // If it's not currently in the tree, listen for changes to the desktop
        // tree.
        const eventHandler = new EventHandler(desktop, EventType$4.CHILDREN_CHANGED, (_evt) => { });
        const onEvent = (event) => {
            if (event.target.matches(findParams)) {
                // If the event target is the node we're looking for, we've found it.
                eventHandler.stop();
                foundCallback(event.target);
            }
            else if (event.target.children.length > 0) {
                // Otherwise, see if one of its children is the node we're looking for.
                node = event.target.find(findParams);
                if (node) {
                    eventHandler.stop();
                    foundCallback(node);
                }
            }
        };
        eventHandler.setCallback(onEvent);
        eventHandler.start();
    }
    /** Creates and records the specified error. */
    static error(errorType, errorString, shouldRecover = false) {
        if (shouldRecover) {
            setTimeout(Navigator.byItem.moveToValidNode.bind(Navigator.byItem), 0);
        }
        const errorTypeCountForUMA = Object.keys(ErrorType).length;
        chrome.metricsPrivate.recordEnumerationValue('Accessibility.CrosSwitchAccess.Error', errorType, errorTypeCountForUMA);
        return new Error(errorString);
    }
    async waitForFocus_(desktop, currentFocus) {
        return new Promise(resolve => {
            // Focus is available. Finish init without waiting for further events.
            // Disallow web view nodes, which indicate a root web area is still
            // loading and pending focus.
            if (currentFocus && currentFocus.role !== RoleType$4.WEB_VIEW) {
                resolve();
                return;
            }
            // Wait for the focus to be sent. If |currentFocus| was undefined, this is
            // guaranteed. Otherwise, also set a timed callback to ensure we do
            // eventually init.
            let callbackId = 0;
            const listener = (maybeEvent) => {
                if (maybeEvent && maybeEvent.target.role === RoleType$4.WEB_VIEW) {
                    return;
                }
                desktop.removeEventListener(EventType$4.FOCUS, listener, false);
                clearTimeout(callbackId);
                resolve();
            };
            desktop.addEventListener(EventType$4.FOCUS, listener, false);
            callbackId = setTimeout(listener, 5000);
        });
    }
}
TestImportManager.exportForTesting(SwitchAccess);

// Copyright 2019 The Chromium Authors
// Use of this source code is governed by a BSD-style license that can be
// found in the LICENSE file.
const EventType$3 = chrome.automation.EventType;
const MenuAction$6 = chrome.accessibilityPrivate.SwitchAccessMenuAction;
/**
 * Class to handle navigating text. Currently, only
 * navigation and selection in editable text fields is supported.
 */
class TextNavigationManager {
    static instance_;
    currentlySelecting_ = false;
    /** Keeps track of when there's a selection in the current node. */
    selectionExists_ = false;
    /** Keeps track of when the clipboard is empty. */
    clipboardHasData_ = false;
    selectionStartIndex_ = TextNavigationManager.NO_SELECT_INDEX;
    selectionStartObject_;
    selectionEndIndex_ = TextNavigationManager.NO_SELECT_INDEX;
    selectionEndObject_;
    selectionListener_;
    constructor() {
        this.selectionListener_ = new EventHandler([], EventType$3.TEXT_SELECTION_CHANGED, () => this.onNavChange_());
        if (SwitchAccess.improvedTextInputEnabled()) {
            chrome.clipboard.onClipboardDataChanged.addListener(() => this.updateClipboardHasData_());
        }
    }
    static get instance() {
        if (!TextNavigationManager.instance_) {
            TextNavigationManager.instance_ = new TextNavigationManager();
        }
        return TextNavigationManager.instance_;
    }
    // =============== Static Methods ==============
    /**
     * Returns if the selection start index is set in the current node.
     */
    static currentlySelecting() {
        const manager = TextNavigationManager.instance;
        return (manager.selectionStartIndex_ !==
            TextNavigationManager.NO_SELECT_INDEX &&
            manager.currentlySelecting_);
    }
    /**
     * Jumps to the beginning of the text field (does nothing
     * if already at the beginning).
     */
    static jumpToBeginning() {
        const manager = TextNavigationManager.instance;
        if (manager.currentlySelecting_) {
            manager.setupDynamicSelection_(false /* resetCursor */);
        }
        EventGenerator.sendKeyPress(KeyCode.HOME, { ctrl: true });
    }
    /**
     * Jumps to the end of the text field (does nothing if
     * already at the end).
     */
    static jumpToEnd() {
        const manager = TextNavigationManager.instance;
        if (manager.currentlySelecting_) {
            manager.setupDynamicSelection_(false /* resetCursor */);
        }
        EventGenerator.sendKeyPress(KeyCode.END, { ctrl: true });
    }
    /**
     * Moves the text caret one character back (does nothing
     * if there are no more characters preceding the current
     * location of the caret).
     */
    static moveBackwardOneChar() {
        const manager = TextNavigationManager.instance;
        if (manager.currentlySelecting_) {
            manager.setupDynamicSelection_(true /* resetCursor */);
        }
        EventGenerator.sendKeyPress(KeyCode.LEFT);
    }
    /**
     * Moves the text caret one word backwards (does nothing
     * if already at the beginning of the field). If the
     * text caret is in the middle of a word, moves the caret
     * to the beginning of that word.
     */
    static moveBackwardOneWord() {
        const manager = TextNavigationManager.instance;
        if (manager.currentlySelecting_) {
            manager.setupDynamicSelection_(false /* resetCursor */);
        }
        EventGenerator.sendKeyPress(KeyCode.LEFT, { ctrl: true });
    }
    /**
     * Moves the text caret one line down (does nothing
     * if there are no lines below the current location of
     * the caret).
     */
    static moveDownOneLine() {
        const manager = TextNavigationManager.instance;
        if (manager.currentlySelecting_) {
            manager.setupDynamicSelection_(true /* resetCursor */);
        }
        EventGenerator.sendKeyPress(KeyCode.DOWN);
    }
    /**
     * Moves the text caret one character forward (does nothing
     * if there are no more characters following the current
     * location of the caret).
     */
    static moveForwardOneChar() {
        const manager = TextNavigationManager.instance;
        if (manager.currentlySelecting_) {
            manager.setupDynamicSelection_(true /* resetCursor */);
        }
        EventGenerator.sendKeyPress(KeyCode.RIGHT);
    }
    /**
     * Moves the text caret one word forward (does nothing if
     * already at the end of the field). If the text caret is
     * in the middle of a word, moves the caret to the end of
     * that word.
     */
    static moveForwardOneWord() {
        const manager = TextNavigationManager.instance;
        if (manager.currentlySelecting_) {
            manager.setupDynamicSelection_(false /* resetCursor */);
        }
        EventGenerator.sendKeyPress(KeyCode.RIGHT, { ctrl: true });
    }
    /**
     * Moves the text caret one line up (does nothing
     * if there are no lines above the current location of
     * the caret).
     */
    static moveUpOneLine() {
        const manager = TextNavigationManager.instance;
        if (manager.currentlySelecting_) {
            manager.setupDynamicSelection_(true /* resetCursor */);
        }
        EventGenerator.sendKeyPress(KeyCode.UP);
    }
    /**
     * Reset the currentlySelecting variable to false, reset the selection
     * indices, and remove the listener on navigation.
     */
    static resetCurrentlySelecting() {
        const manager = TextNavigationManager.instance;
        manager.currentlySelecting_ = false;
        manager.manageNavigationListener_(false /** Removing listener */);
        manager.selectionStartIndex_ = TextNavigationManager.NO_SELECT_INDEX;
        manager.selectionEndIndex_ = TextNavigationManager.NO_SELECT_INDEX;
        if (manager.currentlySelecting_) {
            manager.setupDynamicSelection_(true /* resetCursor */);
        }
        EventGenerator.sendKeyPress(KeyCode.DOWN);
    }
    static get clipboardHasData() {
        return TextNavigationManager.instance.clipboardHasData_;
    }
    static get selectionExists() {
        return TextNavigationManager.instance.selectionExists_;
    }
    static set selectionExists(newVal) {
        TextNavigationManager.instance.selectionExists_ = newVal;
    }
    getSelEndIndex() {
        return this.selectionEndIndex_;
    }
    resetSelStartIndex() {
        this.selectionStartIndex_ = TextNavigationManager.NO_SELECT_INDEX;
    }
    getSelStartIndex() {
        return this.selectionStartIndex_;
    }
    setSelStartIndexAndNode(startIndex, textNode) {
        this.selectionStartIndex_ = startIndex;
        this.selectionStartObject_ = textNode;
    }
    /**
     * Sets the selectionStart variable based on the selection of the current
     * node. Also sets the currently selecting boolean to true.
     */
    static saveSelectStart() {
        const manager = TextNavigationManager.instance;
        chrome.automation.getFocus((focusedNode) => {
            manager.selectionStartObject_ = focusedNode;
            manager.selectionStartIndex_ = manager.getSelectionIndexFromNode_(
            // TODO(b/314203187): Not null asserted, check that this is correct.
            manager.selectionStartObject_, true /* We are getting the start index.*/);
            manager.currentlySelecting_ = true;
        });
    }
    // =============== Instance Methods ==============
    /**
     * Returns either the selection start index or the selection end index of the
     * node based on the getStart param.
     * @return selection start if getStart is true otherwise selection
     * end
     */
    getSelectionIndexFromNode_(node, getStart) {
        let indexFromNode = TextNavigationManager.NO_SELECT_INDEX;
        // TODO(b/314203187): Not null asserted, check that this is correct.
        if (getStart) {
            indexFromNode = node.textSelStart;
        }
        else {
            indexFromNode = node.textSelEnd;
        }
        if (indexFromNode === undefined) {
            return TextNavigationManager.NO_SELECT_INDEX;
        }
        return indexFromNode;
    }
    /** Adds or removes the selection listener. */
    manageNavigationListener_(addListener) {
        if (!this.selectionStartObject_) {
            return;
        }
        if (addListener) {
            this.selectionListener_.setNodes(this.selectionStartObject_);
            this.selectionListener_.start();
        }
        else {
            this.selectionListener_.stop();
        }
    }
    /**
     * Function to handle changes in the cursor position during selection.
     * This function will remove the selection listener and set the end of the
     * selection based on the new position.
     */
    onNavChange_() {
        this.manageNavigationListener_(false);
        if (this.currentlySelecting_) {
            TextNavigationManager.saveSelectEnd();
        }
    }
    /**
     * Sets the selectionEnd variable based on the selection of the current node.
     */
    static saveSelectEnd() {
        const manager = TextNavigationManager.instance;
        chrome.automation.getFocus(focusedNode => {
            manager.selectionEndObject_ = focusedNode;
            manager.selectionEndIndex_ = manager.getSelectionIndexFromNode_(manager.selectionEndObject_, false /*We are not getting the start index.*/);
            manager.saveSelection_();
        });
    }
    /** Sets the selection after verifying that the bounds are set. */
    saveSelection_() {
        if (this.selectionStartIndex_ === TextNavigationManager.NO_SELECT_INDEX ||
            this.selectionEndIndex_ === TextNavigationManager.NO_SELECT_INDEX) {
            console.error(SwitchAccess.error(ErrorType.INVALID_SELECTION_BOUNDS, 'Selection bounds are not set properly: ' +
                this.selectionStartIndex_ + ' ' + this.selectionEndIndex_));
        }
        else {
            this.setSelection_();
        }
    }
    /**
     * Sets up the cursor position and selection listener for dynamic selection.
     * If the needToResetCursor boolean is true, the function will move the cursor
     * to the end point of the selection before adding the event listener. If not,
     * it will simply add the listener.
     */
    setupDynamicSelection_(needToResetCursor) {
        /**
         * TODO(crbug.com/999400): Work on text selection dynamic highlight and
         * text selection implementation.
         */
        if (needToResetCursor) {
            if (TextNavigationManager.currentlySelecting() &&
                this.selectionEndIndex_ !== TextNavigationManager.NO_SELECT_INDEX) {
                // Move the cursor to the end of the existing selection.
                this.setSelection_();
            }
        }
        this.manageNavigationListener_(true /** Add the listener */);
    }
    /**
     * Sets the selection. If start and end object are equal, uses
     * AutomationNode.setSelection. Otherwise calls
     * chrome.automation.setDocumentSelection.
     */
    setSelection_() {
        if (this.selectionStartObject_ === this.selectionEndObject_) {
            // TODO(b/314203187): Not null asserted, check that this is correct.
            this.selectionStartObject_.setSelection(this.selectionStartIndex_, this.selectionEndIndex_);
        }
        else {
            chrome.automation.setDocumentSelection({
                anchorObject: this.selectionStartObject_,
                anchorOffset: this.selectionStartIndex_,
                focusObject: this.selectionEndObject_,
                focusOffset: this.selectionEndIndex_,
            });
        }
    }
    /*
     * TODO(rosalindag): Add functionality to catch when clipboardHasData_ needs
     * to be set to false.
     * Set the clipboardHasData variable to true and reload the menu.
     */
    updateClipboardHasData_() {
        this.clipboardHasData_ = true;
        const node = Navigator.byItem.currentNode;
        if (node.hasAction(MenuAction$6.PASTE)) {
            ActionManager.refreshMenuForNode(node);
        }
    }
}
(function (TextNavigationManager) {
    TextNavigationManager.NO_SELECT_INDEX = -1;
})(TextNavigationManager || (TextNavigationManager = {}));
TestImportManager.exportForTesting(TextNavigationManager);

// Copyright 2020 The Chromium Authors
// Use of this source code is governed by a BSD-style license that can be
// found in the LICENSE file.
var EventType$2 = chrome.automation.EventType;
var MenuAction$5 = chrome.accessibilityPrivate.SwitchAccessMenuAction;
var StateType$1 = chrome.automation.StateType;
/**
 * This class handles interactions with editable text fields.
 */
class EditableTextNode extends BasicNode {
    constructor(baseNode, parent) {
        super(baseNode, parent);
    }
    // ================= Getters and setters =================
    get actions() {
        const actions = super.actions;
        // The SELECT action is used to press buttons, etc. For text inputs, the
        // equivalent action is KEYBOARD, which focuses the input and opens the
        // keyboard.
        const selectIndex = actions.indexOf(MenuAction$5.SELECT);
        if (selectIndex >= 0) {
            actions.splice(selectIndex, 1);
        }
        actions.unshift(MenuAction$5.KEYBOARD, MenuAction$5.DICTATION);
        if (SwitchAccess.improvedTextInputEnabled()) {
            actions.push(MenuAction$5.MOVE_CURSOR, MenuAction$5.JUMP_TO_BEGINNING_OF_TEXT, MenuAction$5.JUMP_TO_END_OF_TEXT, MenuAction$5.MOVE_BACKWARD_ONE_CHAR_OF_TEXT, MenuAction$5.MOVE_FORWARD_ONE_CHAR_OF_TEXT, MenuAction$5.MOVE_BACKWARD_ONE_WORD_OF_TEXT, MenuAction$5.MOVE_FORWARD_ONE_WORD_OF_TEXT, MenuAction$5.MOVE_DOWN_ONE_LINE_OF_TEXT, MenuAction$5.MOVE_UP_ONE_LINE_OF_TEXT);
            actions.push(MenuAction$5.START_TEXT_SELECTION);
            if (TextNavigationManager.currentlySelecting()) {
                actions.push(MenuAction$5.END_TEXT_SELECTION);
            }
            if (TextNavigationManager.selectionExists) {
                actions.push(MenuAction$5.CUT, MenuAction$5.COPY);
            }
            if (TextNavigationManager.clipboardHasData) {
                actions.push(MenuAction$5.PASTE);
            }
        }
        return actions;
    }
    // ================= General methods =================
    doDefaultAction() {
        this.performAction(MenuAction$5.KEYBOARD);
    }
    performAction(action) {
        switch (action) {
            case MenuAction$5.KEYBOARD:
                Navigator.byItem.enterKeyboard();
                return ActionResponse.CLOSE_MENU;
            case MenuAction$5.DICTATION:
                // TODO(crbug.com/314203187): Not null asserted, check that this is
                // correct.
                if (this.automationNode.state[StateType$1.FOCUSED]) {
                    chrome.accessibilityPrivate.toggleDictation();
                }
                else {
                    new EventHandler(this.automationNode, EventType$2.FOCUS, () => chrome.accessibilityPrivate.toggleDictation(), { exactMatch: true, listenOnce: true })
                        .start();
                    this.automationNode.focus();
                }
                return ActionResponse.CLOSE_MENU;
            case MenuAction$5.MOVE_CURSOR:
                return ActionResponse.OPEN_TEXT_NAVIGATION_MENU;
            case MenuAction$5.CUT:
                EventGenerator.sendKeyPress(KeyCode.X, { ctrl: true });
                return ActionResponse.REMAIN_OPEN;
            case MenuAction$5.COPY:
                EventGenerator.sendKeyPress(KeyCode.C, { ctrl: true });
                return ActionResponse.REMAIN_OPEN;
            case MenuAction$5.PASTE:
                EventGenerator.sendKeyPress(KeyCode.V, { ctrl: true });
                return ActionResponse.REMAIN_OPEN;
            case MenuAction$5.START_TEXT_SELECTION:
                TextNavigationManager.saveSelectStart();
                return ActionResponse.OPEN_TEXT_NAVIGATION_MENU;
            case MenuAction$5.END_TEXT_SELECTION:
                TextNavigationManager.saveSelectEnd();
                return ActionResponse.EXIT_SUBMENU;
            case MenuAction$5.JUMP_TO_BEGINNING_OF_TEXT:
                TextNavigationManager.jumpToBeginning();
                return ActionResponse.REMAIN_OPEN;
            case MenuAction$5.JUMP_TO_END_OF_TEXT:
                TextNavigationManager.jumpToEnd();
                return ActionResponse.REMAIN_OPEN;
            case MenuAction$5.MOVE_BACKWARD_ONE_CHAR_OF_TEXT:
                TextNavigationManager.moveBackwardOneChar();
                return ActionResponse.REMAIN_OPEN;
            case MenuAction$5.MOVE_BACKWARD_ONE_WORD_OF_TEXT:
                TextNavigationManager.moveBackwardOneWord();
                return ActionResponse.REMAIN_OPEN;
            case MenuAction$5.MOVE_DOWN_ONE_LINE_OF_TEXT:
                TextNavigationManager.moveDownOneLine();
                return ActionResponse.REMAIN_OPEN;
            case MenuAction$5.MOVE_FORWARD_ONE_CHAR_OF_TEXT:
                TextNavigationManager.moveForwardOneChar();
                return ActionResponse.REMAIN_OPEN;
            case MenuAction$5.MOVE_UP_ONE_LINE_OF_TEXT:
                TextNavigationManager.moveUpOneLine();
                return ActionResponse.REMAIN_OPEN;
            case MenuAction$5.MOVE_FORWARD_ONE_WORD_OF_TEXT:
                TextNavigationManager.moveForwardOneWord();
                return ActionResponse.REMAIN_OPEN;
        }
        return super.performAction(action);
    }
}
BasicNode.creators.push({
    predicate: SwitchAccessPredicate.isTextInput,
    creator: (node, parentNode) => new EditableTextNode(node, parentNode),
});

// Copyright 2020 The Chromium Authors
// Use of this source code is governed by a BSD-style license that can be
// found in the LICENSE file.
var MenuAction$4 = chrome.accessibilityPrivate.SwitchAccessMenuAction;
var RoleType$3 = chrome.automation.RoleType;
/** This class handles interactions with sliders. */
class SliderNode extends BasicNode {
    isCustomSlider_ = true;
    constructor(baseNode, parent) {
        super(baseNode, parent);
    }
    onFocus() {
        super.onFocus();
        this.automationNode.focus();
    }
    performAction(action) {
        // Currently, custom sliders have no way to support increment/decrement via
        // the automation API. We handle this case by simulating left/right arrow
        // presses.
        if (this.isCustomSlider_) {
            if (action === MenuAction$4.INCREMENT) {
                EventGenerator.sendKeyPress(KeyCode.RIGHT);
                return ActionResponse.REMAIN_OPEN;
            }
            else if (action === MenuAction$4.DECREMENT) {
                EventGenerator.sendKeyPress(KeyCode.LEFT);
                return ActionResponse.REMAIN_OPEN;
            }
        }
        return super.performAction(action);
    }
}
BasicNode.creators.push({
    predicate: baseNode => baseNode.role === RoleType$3.SLIDER,
    creator: (node, parent) => new SliderNode(node, parent),
});

// Copyright 2020 The Chromium Authors
// Use of this source code is governed by a BSD-style license that can be
// found in the LICENSE file.
var MenuAction$3 = chrome.accessibilityPrivate.SwitchAccessMenuAction;
var RoleType$2 = chrome.automation.RoleType;
/**
 * This class handles the behavior of tab nodes at the top level (i.e. as
 * groups).
 */
class TabNode extends BasicNode {
    tabAsRoot_;
    /**
     * @param node The node in the automation tree
     * @param tabAsRoot A pre-calculated object for exploring the parts of tab
     * (i.e. choosing whether to open the tab or close it).
     */
    constructor(node, parent, tabAsRoot_) {
        super(node, parent);
        this.tabAsRoot_ = tabAsRoot_;
    }
    // ================= Getters and setters =================
    get actions() {
        return [MenuAction$3.DRILL_DOWN];
    }
    // ================= General methods =================
    asRootNode() {
        return this.tabAsRoot_;
    }
    isGroup() {
        return true;
    }
    performAction(action) {
        if (action !== MenuAction$3.DRILL_DOWN) {
            return ActionResponse.NO_ACTION_TAKEN;
        }
        Navigator.byItem.enterGroup();
        return ActionResponse.CLOSE_MENU;
    }
    // ================= Static methods =================
    static create(tabNode, parent) {
        const tabAsRoot = new BasicRootNode(tabNode);
        let closeButton;
        for (const child of tabNode.children) {
            if (child.role === RoleType$2.BUTTON) {
                closeButton = new BasicNode(child, tabAsRoot);
                break;
            }
        }
        if (!closeButton) {
            // Pinned tabs have no close button, and so can be treated as just
            // actionable.
            return new ActionableTabNode(tabNode, parent, null);
        }
        const tabToSelect = new ActionableTabNode(tabNode, tabAsRoot, closeButton);
        const backButton = new BackButtonNode(tabAsRoot);
        tabAsRoot.children = [tabToSelect, closeButton, backButton];
        return new TabNode(tabNode, parent, tabAsRoot);
    }
}
/** This class handles the behavior of tabs as actionable elements */
class ActionableTabNode extends BasicNode {
    closeButton_;
    constructor(node, parent, closeButton_) {
        super(node, parent);
        this.closeButton_ = closeButton_;
    }
    // ================= Getters and setters =================
    get actions() {
        return [MenuAction$3.SELECT];
    }
    get location() {
        if (!this.closeButton_) {
            return super.location;
        }
        return RectUtil.difference(super.location, this.closeButton_.location);
    }
    // ================= General methods =================
    asRootNode() {
        return undefined;
    }
    isGroup() {
        return false;
    }
}
// TODO(crbug.com/314203187): Not null asserted, check that this is correct.
BasicNode.creators.push({
    predicate: baseNode => baseNode.role === RoleType$2.TAB &&
        baseNode.root.role === RoleType$2.DESKTOP,
    creator: TabNode.create,
});

// Copyright 2020 The Chromium Authors
// Use of this source code is governed by a BSD-style license that can be
// found in the LICENSE file.
/**
 * This class handles interactions with the desktop automation node.
 */
class DesktopNode extends BasicRootNode {
    // ================= General methods =================
    equals(other) {
        // The underlying automation tree only has one desktop node, so all
        // DesktopNode instances are equal.
        return other instanceof DesktopNode;
    }
    isValidGroup() {
        return true;
    }
    refresh() {
        // Find the currently focused child.
        let focusedChild = null;
        for (const child of this.children) {
            if (child.isFocused()) {
                focusedChild = child;
                break;
            }
        }
        // Update this DesktopNode's children.
        const childConstructor = (node) => BasicNode.create(node, this);
        DesktopNode.findAndSetChildren(this, childConstructor);
        // Set the new instance of that child to be the focused node.
        for (const child of this.children) {
            if (child.isEquivalentTo(focusedChild)) {
                Navigator.byItem.forceFocusedNode(child);
                return;
            }
        }
        // If the previously focused node no longer exists, focus the first node in
        // the group.
        Navigator.byItem.forceFocusedNode(this.children[0]);
    }
    // ================= Static methods =================
    static build(desktop) {
        const root = new DesktopNode(desktop);
        const childConstructor = (autoNode) => BasicNode.create(autoNode, root);
        DesktopNode.findAndSetChildren(root, childConstructor);
        return root;
    }
    static findAndSetChildren(root, childConstructor) {
        const interestingChildren = BasicRootNode.getInterestingChildren(root);
        if (interestingChildren.length < 1) {
            // If the desktop node does not behave as expected, we have no basis for
            // recovering. Wait for the next user input.
            throw SwitchAccess.error(ErrorType.MALFORMED_DESKTOP, 'Desktop node must have at least 1 interesting child.', false /* shouldRecover */);
        }
        // TODO(crbug.com/40706137): Add hittest intervals to new children which are
        // SwitchAccessPredicate.isWindow to check whether those children are
        // occluded or visible. Remove any intervals on the previous window
        // children before reassigning root.children.
        root.children = interestingChildren.map(childConstructor);
    }
}
TestImportManager.exportForTesting(DesktopNode);

// Copyright 2020 The Chromium Authors
// Use of this source code is governed by a BSD-style license that can be
// found in the LICENSE file.
/** This class is a structure to store previous state for restoration. */
class FocusData {
    group;
    focus;
    /** |focus| Must be a child of |group|. */
    constructor(group, focus) {
        this.group = group;
        this.focus = focus;
    }
    isValid() {
        if (this.group.isValidGroup()) {
            // Ensure it is still valid. Some nodes may have been added
            // or removed since this was last used.
            this.group.refreshChildren();
        }
        return this.group.isValidGroup();
    }
}
/** This class handles saving and retrieving FocusData. */
class FocusHistory {
    dataStack = [];
    /**
     * Creates the restore data to get from the desktop node to the specified
     * automation node.
     * Erases the current history and replaces with the new data.
     * @return Whether the history was rebuilt from the given node.
     */
    buildFromAutomationNode(node) {
        if (!node.parent) {
            // No ancestors, cannot create stack.
            return false;
        }
        const cache = new SACache();
        // Create a list of ancestors.
        const ancestorStack = [node];
        while (node.parent) {
            ancestorStack.push(node.parent);
            node = node.parent;
        }
        // TODO(b/314203187): Not null asserted, check that this is correct.
        let group = DesktopNode.build(ancestorStack.pop());
        const firstAncestor = ancestorStack[ancestorStack.length - 1];
        if (!SwitchAccessPredicate.isInterestingSubtree(firstAncestor, cache)) {
            // If the topmost ancestor (other than the desktop) is entirely
            // uninteresting, we leave the history as is.
            return false;
        }
        const newDataStack = [];
        while (ancestorStack.length > 0) {
            const candidate = ancestorStack.pop();
            if (!SwitchAccessPredicate.isInteresting(candidate, group, cache)) {
                continue;
            }
            // TODO(b/314203187): Not null asserted, check that this is correct.
            const focus = group.findChild(candidate);
            if (!focus) {
                continue;
            }
            newDataStack.push(new FocusData(group, focus));
            // TODO(b/314203187): Not null asserted, check that this is correct.
            group = focus.asRootNode();
            if (!group) {
                break;
            }
        }
        if (newDataStack.length === 0) {
            return false;
        }
        this.dataStack = newDataStack;
        return true;
    }
    containsDataMatchingPredicate(predicate) {
        for (const data of this.dataStack) {
            if (predicate(data)) {
                return true;
            }
        }
        return false;
    }
    /**
     * Returns the most proximal restore data, but does not remove it from the
     * history.
     */
    peek() {
        return this.dataStack[this.dataStack.length - 1] || null;
    }
    retrieve() {
        let data = this.dataStack.pop();
        while (data && !data.isValid()) {
            data = this.dataStack.pop();
        }
        if (data) {
            return data;
        }
        // If we don't have any valid history entries, fallback to the desktop node.
        const desktop = new DesktopNode(Navigator.byItem.desktopNode);
        return new FocusData(desktop, desktop.firstChild);
    }
    save(data) {
        this.dataStack.push(data);
    }
    /** Support for this type being used in for..of loops. */
    [Symbol.iterator]() {
        return this.dataStack[Symbol.iterator]();
    }
}

// Copyright 2021 The Chromium Authors
// Use of this source code is governed by a BSD-style license that can be
// found in the LICENSE file.
class ItemNavigatorInterface {
}

// Copyright 2019 The Chromium Authors
// Use of this source code is governed by a BSD-style license that can be
// found in the LICENSE file.
/** This class represents the group rooted at a modal dialog. */
class ModalDialogRootNode extends BasicRootNode {
    onExit() {
        // To close a modal dialog, we need to send an escape key event.
        EventGenerator.sendKeyPress(KeyCode.ESCAPE);
    }
    /**
     * Creates the tree structure for a modal dialog.
     */
    static buildTree(dialogNode) {
        const root = new ModalDialogRootNode(dialogNode);
        const childConstructor = (node) => BasicNode.create(node, root);
        BasicRootNode.findAndSetChildren(root, childConstructor);
        return root;
    }
}

// Copyright 2017 The Chromium Authors
// Use of this source code is governed by a BSD-style license that can be
// found in the LICENSE file.
const EventType$1 = chrome.automation.EventType;
const RoleType$1 = chrome.automation.RoleType;
const TreeChangeObserverFilter = chrome.automation.TreeChangeObserverFilter;
const TreeChangeType = chrome.automation.TreeChangeType;
/** This class handles navigation amongst the elements onscreen. */
class ItemScanManager extends ItemNavigatorInterface {
    desktop_;
    group_;
    node_;
    history_;
    suspendedGroup_ = null;
    ignoreFocusInKeyboard_ = false;
    constructor(desktop) {
        super();
        this.desktop_ = desktop;
        this.group_ = DesktopNode.build(this.desktop_);
        // TODO(crbug.com/40706137): It is possible for the firstChild to be a
        // window which is occluded, for example if Switch Access is turned on
        // when the user has several browser windows opened. We should either
        // dynamically pick this.node_'s initial value based on an occlusion check,
        // or ensure that we move away from occluded children as quickly as soon
        // as they are detected using an interval set in DesktopNode.
        this.node_ = this.group_.firstChild;
        this.history_ = new FocusHistory();
    }
    // =============== ItemNavigatorInterface implementation ==============
    currentGroupHasChild(node) {
        return this.group_.children.includes(node);
    }
    enterGroup() {
        if (!this.node_.isGroup()) {
            return;
        }
        const newGroup = this.node_.asRootNode();
        if (newGroup) {
            this.history_.save(new FocusData(this.group_, this.node_));
            this.setGroup_(newGroup);
        }
    }
    enterKeyboard() {
        this.ignoreFocusInKeyboard_ = true;
        this.node_.automationNode.focus();
        const keyboard = KeyboardRootNode.buildTree();
        this.jumpTo_(keyboard);
    }
    exitGroupUnconditionally() {
        this.exitGroup_();
    }
    exitIfInGroup(node) {
        if (this.group_.isEquivalentTo(node)) {
            this.exitGroup_();
        }
    }
    async exitKeyboard() {
        this.ignoreFocusInKeyboard_ = false;
        const isKeyboard = (data) => data.group instanceof KeyboardRootNode;
        // If we are not in the keyboard, do nothing.
        if (!(this.group_ instanceof KeyboardRootNode) &&
            !this.history_.containsDataMatchingPredicate(isKeyboard)) {
            return;
        }
        while (this.history_.peek() !== null) {
            if (this.group_ instanceof KeyboardRootNode) {
                this.exitGroup_();
                break;
            }
            this.exitGroup_();
        }
        const focus = await AsyncUtil.getFocus();
        // First, try to move back to the focused node.
        if (focus) {
            this.moveTo_(focus);
        }
        else {
            // Otherwise, move to anything that's valid based on the above history.
            this.moveToValidNode();
        }
    }
    forceFocusedNode(node) {
        // Check if they are exactly the same instance. Checking contents
        // equality is not sufficient in case the node has been repopulated
        // after a refresh.
        if (this.node_ !== node) {
            this.setNode_(node);
        }
    }
    getTreeForDebugging(wholeTree = true) {
        if (!wholeTree) {
            console.log(this.group_.debugString(wholeTree));
            return this.group_;
        }
        const desktopRoot = DesktopNode.build(this.desktop_);
        console.log(desktopRoot.debugString(wholeTree, '', this.node_));
        return desktopRoot;
    }
    jumpTo(automationNode) {
        if (!automationNode) {
            return;
        }
        const node = BasicRootNode.buildTree(automationNode);
        this.jumpTo_(node, false /* shouldExitMenu */);
    }
    moveBackward() {
        if (this.node_.isValidAndVisible()) {
            this.tryMoving(this.node_.previous, node => node.previous, this.node_);
        }
        else {
            this.moveToValidNode();
        }
    }
    moveForward() {
        if (this.node_.isValidAndVisible()) {
            this.tryMoving(this.node_.next, node => node.next, this.node_);
        }
        else {
            this.moveToValidNode();
        }
    }
    async tryMoving(node, getNext, startingNode) {
        if (node === startingNode) {
            // This should only happen if the desktop contains exactly one interesting
            // child and all other children are windows which are occluded.
            // Unlikely to happen since we can always access the shelf.
            return;
        }
        if (!(node instanceof BasicNode)) {
            this.setNode_(node);
            return;
        }
        if (!SwitchAccessPredicate.isWindow(node.automationNode)) {
            this.setNode_(node);
            return;
        }
        const location = node.location;
        if (!location) {
            // Closure compiler doesn't realize we already checked isValidAndVisible
            // before calling tryMoving, so we need to explicitly check location here
            // so that RectUtil.center does not cause a closure error.
            this.moveToValidNode();
            return;
        }
        const center = RectUtil.center(location);
        // Check if the top center is visible as a proxy for occlusion. It's
        // possible that other parts of the window are occluded, but in Chrome we
        // can't drag windows off the top of the screen.
        const hitNode = await new Promise(resolve => this.desktop_.hitTestWithReply(center.x, location.top, resolve));
        if (AutomationUtil.isDescendantOf(hitNode, node.automationNode)) {
            this.setNode_(node);
        }
        else if (node.isValidAndVisible()) {
            this.tryMoving(getNext(node), getNext, startingNode);
        }
        else {
            this.moveToValidNode();
        }
    }
    moveToValidNode() {
        const nodeIsValid = this.node_.isValidAndVisible();
        const groupIsValid = this.group_.isValidGroup();
        if (nodeIsValid && groupIsValid) {
            return;
        }
        if (nodeIsValid && !(this.node_ instanceof BackButtonNode)) {
            // Our group has been invalidated. Move to this node to repair the
            // group stack.
            this.moveTo_(this.node_.automationNode);
            return;
        }
        const child = this.group_.firstValidChild();
        if (groupIsValid && child) {
            this.setNode_(child);
            return;
        }
        this.restoreFromHistory_();
        // Make sure the menu isn't open unless we're still in the menu.
        // TODO(b/314203187): Not null asserted, check that this is correct.
        if (!this.group_.isEquivalentTo(MenuManager.menuAutomationNode)) {
            ActionManager.exitAllMenus();
        }
    }
    restart() {
        const point = Navigator.byPoint.currentPoint;
        SwitchAccess.mode = Mode.ITEM_SCAN;
        this.desktop_.hitTestWithReply(point.x, point.y, node => this.moveTo_(node));
    }
    restoreSuspendedGroup() {
        if (this.suspendedGroup_) {
            // Clearing the focus rings avoids having them re-animate to the same
            // position.
            FocusRingManager.clearAll();
            this.history_.save(new FocusData(this.group_, this.node_));
            this.loadFromData_(this.suspendedGroup_);
        }
    }
    suspendCurrentGroup() {
        const data = new FocusData(this.group_, this.node_);
        this.exitGroup_();
        this.suspendedGroup_ = data;
    }
    get currentNode() {
        this.moveToValidNode();
        return this.node_;
    }
    get desktopNode() {
        return this.desktop_;
    }
    // =============== Event Handlers ==============
    /**
     * When focus shifts, move to the element. Find the closest interesting
     *     element to engage with.
     */
    onFocusChange_(event) {
        if (SwitchAccess.mode === Mode.POINT_SCAN) {
            return;
        }
        // Ignore focus changes from our own actions.
        if (event.eventFrom === 'action') {
            return;
        }
        // To be safe, let's ignore focus when we're in the SA menu or over the
        // keyboard.
        if (this.ignoreFocusInKeyboard_ ||
            this.group_ instanceof KeyboardRootNode || MenuManager.isMenuOpen()) {
            return;
        }
        if (this.node_.isEquivalentTo(event.target)) {
            return;
        }
        this.moveTo_(event.target);
    }
    /**
     * When scroll position changes, ensure that the focus ring is in the
     * correct place and that the focused node / node group are valid.
     */
    onScrollChange_() {
        if (SwitchAccess.mode === Mode.POINT_SCAN) {
            return;
        }
        if (this.node_.isValidAndVisible()) {
            // Update focus ring.
            FocusRingManager.setFocusedNode(this.node_);
        }
        this.group_.refresh();
        ActionManager.refreshMenuUnconditionally();
    }
    /** When a menu is opened, jump focus to the menu. */
    onModalDialog_(event) {
        if (SwitchAccess.mode === Mode.POINT_SCAN) {
            return;
        }
        const modalRoot = ModalDialogRootNode.buildTree(event.target);
        if (modalRoot.isValidGroup()) {
            this.jumpTo_(modalRoot);
        }
    }
    /**
     * When the automation tree changes, ensure the group and node we are
     * currently listening to are fresh. This is only called when the tree change
     * occurred on the node or group which are currently active.
     */
    onTreeChange_(treeChange) {
        if (SwitchAccess.mode === Mode.POINT_SCAN) {
            return;
        }
        if (treeChange.type === TreeChangeType.NODE_REMOVED) {
            this.group_.refresh();
            this.moveToValidNode();
        }
        else if (treeChange.type === TreeChangeType.SUBTREE_UPDATE_END) {
            this.group_.refresh();
        }
    }
    // =============== Private Methods ==============
    exitGroup_() {
        this.group_.onExit();
        this.restoreFromHistory_();
    }
    start() {
        chrome.automation.getFocus((focus) => {
            if (focus && this.history_.buildFromAutomationNode(focus)) {
                this.restoreFromHistory_();
            }
            else {
                this.group_.onFocus();
                this.node_.onFocus();
            }
        });
        new RepeatedEventHandler(this.desktop_, EventType$1.FOCUS, event => this.onFocusChange_(event));
        // ARC++ fires SCROLL_POSITION_CHANGED.
        new RepeatedEventHandler(this.desktop_, EventType$1.SCROLL_POSITION_CHANGED, () => this.onScrollChange_());
        // Web and Views use AXEventGenerator, which fires
        // separate horizontal and vertical events.
        new RepeatedEventHandler(this.desktop_, EventType$1.SCROLL_HORIZONTAL_POSITION_CHANGED, () => this.onScrollChange_());
        new RepeatedEventHandler(this.desktop_, EventType$1.SCROLL_VERTICAL_POSITION_CHANGED, () => this.onScrollChange_());
        new RepeatedTreeChangeHandler(TreeChangeObserverFilter.ALL_TREE_CHANGES, treeChange => this.onTreeChange_(treeChange), {
            predicate: treeChange => this.group_.findChild(treeChange.target) != null ||
                this.group_.isEquivalentTo(treeChange.target),
        });
        // The status tray fires a SHOW event when it opens.
        new EventHandler(this.desktop_, [EventType$1.MENU_START, EventType$1.SHOW], event => this.onModalDialog_(event))
            .start();
    }
    /**
     * Jumps Switch Access focus to a specified node, such as when opening a menu
     * or the keyboard. Does not modify the groups already in the group stack.
     */
    jumpTo_(group, shouldExitMenu = true) {
        if (shouldExitMenu) {
            ActionManager.exitAllMenus();
        }
        this.history_.save(new FocusData(this.group_, this.node_));
        this.setGroup_(group);
    }
    /**
     * Moves Switch Access focus to a specified node, based on a focus shift or
     *     tree change event. Reconstructs the group stack to center on that node.
     *
     * This is a "permanent" move, while |jumpTo_| is a "temporary" move.
     */
    moveTo_(automationNode) {
        ActionManager.exitAllMenus();
        if (this.history_.buildFromAutomationNode(automationNode)) {
            this.restoreFromHistory_();
        }
    }
    /** Restores the most proximal state that is still valid from the history. */
    restoreFromHistory_() {
        // retrieve() guarantees that the data's group is valid.
        this.loadFromData_(this.history_.retrieve());
    }
    /** Extracts the focus and group from save data. */
    loadFromData_(data) {
        if (!data.group.isValidGroup()) {
            return;
        }
        // |data.focus| may not be a child of |data.group| anymore since
        // |data.group| updates when retrieving the history record. So |data.focus|
        // should not be used as the preferred focus node. Instead, we should find
        // the equivalent node in the group's children.
        let focusTarget = null;
        for (const child of data.group.children) {
            if (child.isEquivalentTo(data.focus)) {
                focusTarget = child;
                break;
            }
        }
        if (focusTarget && focusTarget.isValidAndVisible()) {
            this.setGroup_(data.group, focusTarget);
        }
        else {
            this.setGroup_(data.group);
        }
    }
    /**
     * Set |this.group_| to |group|, and sets |this.node_| to either |opt_focus|
     * or |group.firstChild|.
     */
    setGroup_(group, focus) {
        // Clear the suspended group, as it's only valid in its original context.
        this.suspendedGroup_ = null;
        this.group_.onUnfocus();
        this.group_ = group;
        this.group_.onFocus();
        const node = focus || this.group_.firstValidChild();
        if (!node) {
            this.moveToValidNode();
            return;
        }
        // Check to see if the new node requires we try and focus a new window.
        chrome.automation.getFocus((currentAutomationFocus) => {
            const newAutomationNode = node.automationNode;
            if (!newAutomationNode || !currentAutomationFocus) {
                return;
            }
            // First, if the current focus is a descendant of the new node or vice
            // versa, then we're done here.
            if (AutomationUtil.isDescendantOf(currentAutomationFocus, newAutomationNode) ||
                AutomationUtil.isDescendantOf(newAutomationNode, currentAutomationFocus)) {
                return;
            }
            // The current focus and new node do not have one another in their
            // ancestry; try to focus an ancestor window of the new node. In
            // particular, the parenting aura::Window of the views::Widget.
            let widget = newAutomationNode;
            while (widget &&
                (widget.role !== RoleType$1.WINDOW || widget.className !== 'Widget')) {
                widget = widget.parent;
            }
            if (widget && widget.parent) {
                widget.parent.focus();
            }
        });
        this.setNode_(node);
    }
    /** Set |this.node_| to |node|, and update what is displayed onscreen. */
    setNode_(node) {
        if (!node.isValidAndVisible()) {
            this.moveToValidNode();
            return;
        }
        this.node_.onUnfocus();
        this.node_ = node;
        this.node_.onFocus();
        AutoScanManager.restartIfRunning();
    }
}
TestImportManager.exportForTesting(ItemScanManager);

// Copyright 2021 The Chromium Authors
// Use of this source code is governed by a BSD-style license that can be
// found in the LICENSE file.
var MenuAction$2 = chrome.accessibilityPrivate.SwitchAccessMenuAction;
var PointScanState = chrome.accessibilityPrivate.PointScanState;
class PointScanManager {
    point_ = { x: 0, y: 0 };
    pointListener_;
    constructor() {
        this.pointListener_ = point => this.handleOnPointScanSet_(point);
    }
    // ====== PointNavigatorInterface implementation =====
    get currentPoint() {
        return this.point_;
    }
    start() {
        FocusRingManager.clearAll();
        SwitchAccess.mode = Mode.POINT_SCAN;
        chrome.accessibilityPrivate.onPointScanSet.addListener(this.pointListener_);
        chrome.accessibilityPrivate.setPointScanState(PointScanState.START);
    }
    stop() {
        chrome.accessibilityPrivate.setPointScanState(PointScanState.STOP);
    }
    performMouseAction(action) {
        if (SwitchAccess.mode !== Mode.POINT_SCAN) {
            return;
        }
        if (action !== MenuAction$2.LEFT_CLICK && action !== MenuAction$2.RIGHT_CLICK) {
            return;
        }
        const params = {};
        if (action === MenuAction$2.RIGHT_CLICK) {
            params.mouseButton =
                chrome.accessibilityPrivate.SyntheticMouseEventButton.RIGHT;
        }
        EventGenerator.sendMouseClick(this.point_.x, this.point_.y, params);
        this.start();
    }
    // ============= Private Methods =============
    /**
     * Shows the point scan menu and sets the point scan position
     * coordinates.
     */
    handleOnPointScanSet_(point) {
        this.point_ = point;
        ActionManager.openMenu(MenuType.POINT_SCAN_MENU);
        chrome.accessibilityPrivate.onPointScanSet.removeListener(this.pointListener_);
    }
}

// Copyright 2020 The Chromium Authors
// Use of this source code is governed by a BSD-style license that can be
// found in the LICENSE file.
class Navigator {
    static itemManager_;
    static pointManager_;
    static initializeSingletonInstances(desktop) {
        Navigator.itemManager_ = new ItemScanManager(desktop);
        Navigator.pointManager_ = new PointScanManager();
    }
    static get byItem() {
        if (!Navigator.itemManager_) {
            throw SwitchAccess.error(ErrorType.UNINITIALIZED, 'Cannot access itemManager before Navigator.init()');
        }
        return Navigator.itemManager_;
    }
    static get byPoint() {
        if (!Navigator.pointManager_) {
            throw SwitchAccess.error(ErrorType.UNINITIALIZED, 'Cannot access pointManager before Navigator.init()');
        }
        return Navigator.pointManager_;
    }
}
TestImportManager.exportForTesting(Navigator);

// Copyright 2018 The Chromium Authors
// Use of this source code is governed by a BSD-style license that can be
// found in the LICENSE file.
var EventType = chrome.automation.EventType;
var MenuAction$1 = chrome.accessibilityPrivate.SwitchAccessMenuAction;
var RoleType = chrome.automation.RoleType;
var StateType = chrome.automation.StateType;
var SwitchAccessBubble = chrome.accessibilityPrivate.SwitchAccessBubble;
/**
 * Class to handle interactions with the Switch Access action menu, including
 * opening and closing the menu and setting its location / the actions to be
 * displayed.
 */
class MenuManager {
    displayedActions_ = null;
    displayedLocation_;
    isMenuOpen_ = false;
    menuAutomationNode_;
    clickHandler_;
    static instance;
    constructor() {
        this.clickHandler_ = new EventHandler([], EventType.CLICKED, (event) => this.onButtonClicked_(event));
    }
    static create() {
        if (MenuManager.instance) {
            throw new Error('Cannot instantiate more than one MenuManager');
        }
        MenuManager.instance = new MenuManager();
        return MenuManager.instance;
    }
    // ================= Static Methods ==================
    static isMenuOpen() {
        // TODO(b/314203187): Not nulls asserted, check that this is correct.
        return Boolean(MenuManager.instance) && MenuManager.instance.isMenuOpen_;
    }
    static get menuAutomationNode() {
        if (MenuManager.instance) {
            return MenuManager.instance.menuAutomationNode_;
        }
        return null;
    }
    // ================ Instance Methods =================
    /**
     * If multiple actions are available for the currently highlighted node,
     * opens the menu. Otherwise performs the node's default action.
     */
    open(actions, location) {
        if (!this.isMenuOpen_) {
            if (!location) {
                return;
            }
            this.displayedLocation_ = location;
        }
        if (ArrayUtil.contentsAreEqual(actions, this.displayedActions_ ?? undefined)) {
            return;
        }
        this.displayMenuWithActions_(actions);
    }
    /** Exits the menu. */
    close() {
        this.isMenuOpen_ = false;
        this.displayedActions_ = null;
        // To match the accessibilityPrivate function signature, displayedLocation_
        // has to be undefined rather than null.
        this.displayedLocation_ = undefined;
        Navigator.byItem.exitIfInGroup(this.menuAutomationNode_ ?? null);
        this.menuAutomationNode_ = null;
        chrome.accessibilityPrivate.updateSwitchAccessBubble(SwitchAccessBubble.MENU, false /* show */);
    }
    // ================= Private Methods ==================
    asAction_(actionString) {
        if (Object.values(MenuAction$1).includes(actionString)) {
            return actionString;
        }
        return null;
    }
    /**
     * Opens or reloads the menu for the current action node with the specified
     * actions.
     */
    displayMenuWithActions_(actions) {
        chrome.accessibilityPrivate.updateSwitchAccessBubble(SwitchAccessBubble.MENU, true /* show */, this.displayedLocation_, actions);
        this.isMenuOpen_ = true;
        this.findAndJumpToMenu_();
        this.displayedActions_ = actions;
    }
    /**
     * Searches the automation tree to find the node for the Switch Access menu.
     * If we've already found a node, and it's still valid, then jump to that
     * node.
     */
    findAndJumpToMenu_() {
        if (this.hasMenuNode_() && this.menuAutomationNode_) {
            this.jumpToMenu_(this.menuAutomationNode_);
            return;
        }
        SwitchAccess.findNodeMatching({
            role: RoleType.MENU,
            attributes: { className: 'SwitchAccessMenuView' },
        }, node => this.jumpToMenu_(node));
    }
    hasMenuNode_() {
        // TODO(b/314203187): Not null asserted, check that this is correct.
        return Boolean(this.menuAutomationNode_ && this.menuAutomationNode_.role &&
            !this.menuAutomationNode_.state[StateType.OFFSCREEN]);
    }
    /**
     * Saves the automation node representing the menu, adds all listeners, and
     * jumps to the node.
     */
    jumpToMenu_(node) {
        if (!this.isMenuOpen_) {
            return;
        }
        // If the menu hasn't fully loaded, wait for that before jumping.
        // TODO(b/314203187): Not null asserted, check that this is correct.
        if (node.children.length < 1 ||
            node.firstChild.state[StateType.OFFSCREEN]) {
            new EventHandler(node, [EventType.CHILDREN_CHANGED, EventType.LOCATION_CHANGED], () => this.jumpToMenu_(node), { listenOnce: true })
                .start();
            return;
        }
        this.menuAutomationNode_ = node;
        this.clickHandler_.setNodes(this.menuAutomationNode_);
        this.clickHandler_.start();
        Navigator.byItem.jumpTo(this.menuAutomationNode_);
    }
    /**
     * Listener for when buttons are clicked. Identifies the action to perform
     * and forwards the request to the action manager.
     */
    onButtonClicked_(event) {
        const selectedAction = this.asAction_(event.target.value);
        if (!this.isMenuOpen_ || !selectedAction) {
            return;
        }
        ActionManager.performAction(selectedAction);
    }
}

// Copyright 2019 The Chromium Authors
// Use of this source code is governed by a BSD-style license that can be
// found in the LICENSE file.
var FocusType = chrome.accessibilityPrivate.FocusType;
/** Class to handle focus rings. */
class FocusRingManager {
    observer_;
    /** A map of all the focus rings. */
    rings_;
    ringNodesForTesting_ = {
        [RingId.PRIMARY]: null,
        [RingId.PREVIEW]: null,
    };
    static instance_;
    constructor() {
        this.rings_ = this.createRings_();
    }
    static init() {
        if (FocusRingManager.instance_) {
            throw SwitchAccess.error(ErrorType.DUPLICATE_INITIALIZATION, 'Cannot initialize focus ring manager twice.');
        }
        FocusRingManager.instance_ = new FocusRingManager();
    }
    static get instance() {
        if (!FocusRingManager.instance_) {
            throw SwitchAccess.error(ErrorType.UNINITIALIZED, 'FocusRingManager cannot be accessed before being initialized');
        }
        return FocusRingManager.instance_;
    }
    /** Sets the focus ring color. */
    static setColor(color) {
        if (!COLOR_PATTERN.test(color)) {
            console.error(SwitchAccess.error(ErrorType.INVALID_COLOR, 'Problem setting focus ring color: ' + color + ' is not' +
                'a valid CSS color string.'));
            return;
        }
        FocusRingManager.instance.setColorValidated_(color);
    }
    /** Sets the primary and preview focus rings based on the provided node. */
    static setFocusedNode(node) {
        if (node.ignoreWhenComputingUnionOfBoundingBoxes()) {
            FocusRingManager.instance.setFocusedNodeIgnorePrimary_(node);
            return;
        }
        if (!node.location) {
            throw SwitchAccess.error(ErrorType.MISSING_LOCATION, 'Cannot set focus rings if node location is undefined', true /* shouldRecover */);
        }
        // If the primary node is a group, show its first child as the "preview"
        // focus.
        if (node.isGroup()) {
            // TODO(b/314203187): Not null asserted, check that this is correct.
            const firstChild = node.asRootNode().firstChild;
            FocusRingManager.instance.setFocusedNodeGroup_(node, firstChild);
            return;
        }
        FocusRingManager.instance.setFocusedNodeLeaf_(node);
    }
    /** Clears all focus rings. */
    static clearAll() {
        FocusRingManager.instance.clearAll_();
    }
    /**
     * Set an observer that will be called every time the focus rings
     * are updated. It will be called with two arguments: the node for
     * the primary ring, and the node for the preview ring. Either may
     * be null.
     */
    static setObserver(observer) {
        FocusRingManager.instance.observer_ = observer;
    }
    // ======== Private methods ========
    clearAll_() {
        this.forEachRing_(ring => ring.rects = []);
        this.updateNodesForTesting_(null, null);
        this.updateFocusRings_();
    }
    /** Creates the map of focus rings. */
    createRings_() {
        const primaryRing = {
            id: RingId.PRIMARY,
            rects: [],
            type: FocusType.SOLID,
            color: PRIMARY_COLOR,
            secondaryColor: OUTER_COLOR,
        };
        const previewRing = {
            id: RingId.PREVIEW,
            rects: [],
            type: FocusType.DASHED,
            color: PREVIEW_COLOR,
            secondaryColor: OUTER_COLOR,
        };
        return {
            [RingId.PRIMARY]: primaryRing,
            [RingId.PREVIEW]: previewRing,
        };
    }
    /** Calls a function for each focus ring. */
    forEachRing_(callback) {
        Object.values(this.rings_).forEach(ring => callback(ring));
    }
    /** Sets the focus ring color. Assumes the color has been validated. */
    setColorValidated_(color) {
        this.forEachRing_(ring => ring.color = color);
    }
    /**
     * Sets the primary focus ring to |node|, and the preview focus ring to
     * |firstChild|.
     */
    setFocusedNodeGroup_(group, firstChild) {
        // Clear the dashed ring between transitions, as the animation is
        // distracting.
        this.rings_[RingId.PREVIEW].rects = [];
        let focusRect = group.location;
        const childRect = firstChild ? firstChild.location : null;
        if (childRect) {
            // If the current element is not specialized in location handling, e.g.
            // the back button, the focus rect should expand to contain the child
            // rect.
            focusRect =
                RectUtil.expandToFitWithPadding(GROUP_BUFFER, focusRect, childRect);
            this.rings_[RingId.PREVIEW].rects = [childRect];
        }
        this.rings_[RingId.PRIMARY].rects = [focusRect];
        this.updateNodesForTesting_(group, firstChild);
        this.updateFocusRings_();
    }
    /**
     * Clears the primary focus ring and sets the preview focus ring based on the
     * provided node.
     */
    setFocusedNodeIgnorePrimary_(node) {
        // Nodes of this type, e.g. the back button node, handles setting its own
        // focus, as it has special requirements (a round focus ring that has no
        // gap with the edges of the view).
        this.rings_[RingId.PRIMARY].rects = [];
        // Clear the dashed ring between transitions, as the animation is
        // distracting.
        this.rings_[RingId.PREVIEW].rects = [];
        this.updateFocusRings_();
        // Show the preview focus ring unless the menu is open (it has a custom exit
        // button).
        if (!MenuManager.isMenuOpen()) {
            // TODO(b/314203187): Not null asserted, check that this is correct.
            this.rings_[RingId.PREVIEW].rects = [node.group.location];
        }
        this.updateNodesForTesting_(node, node.group);
        this.updateFocusRings_();
    }
    /** Sets the primary focus to |node| and clears the secondary focus. */
    setFocusedNodeLeaf_(node) {
        // TODO(b/314203187): Not nulls asserted, check these to make sure
        // this is correct.
        this.rings_[RingId.PRIMARY].rects = [node.location];
        this.rings_[RingId.PREVIEW].rects = [];
        this.updateNodesForTesting_(node, null);
        this.updateFocusRings_();
    }
    /**
     * Updates all focus rings to reflect new location, color, style, or other
     * changes. Enables observers to monitor what's focused.
     */
    updateFocusRings_() {
        if (SwitchAccess.mode === Mode.POINT_SCAN && !MenuManager.isMenuOpen()) {
            return;
        }
        const focusRings = Object.values(this.rings_);
        chrome.accessibilityPrivate.setFocusRings(focusRings, chrome.accessibilityPrivate.AssistiveTechnologyType.SWITCH_ACCESS);
    }
    /** Saves the primary/preview focus for testing. */
    updateNodesForTesting_(primary, preview) {
        // Keep track of the nodes associated with each focus ring for testing
        // purposes, since focus ring locations are not guaranteed to exactly match
        // node locations.
        this.ringNodesForTesting_[RingId.PRIMARY] = primary;
        this.ringNodesForTesting_[RingId.PREVIEW] = preview;
        const observer = FocusRingManager.instance.observer_;
        if (observer) {
            observer(primary, preview);
        }
    }
}
/**
 * Regex pattern to verify valid colors. Checks that the first character
 * is '#', followed by 3, 4, 6, or 8 valid hex characters, and no other
 * characters (ignoring case).
 */
const COLOR_PATTERN = /^#([0-9A-F]{3,4}|[0-9A-F]{6}|[0-9A-F]{8})$/i;
/**
 * The buffer (in dip) between a child's focus ring and its parent's focus
 * ring.
 */
const GROUP_BUFFER = 2;
/**
 * The focus ring IDs used by Switch Access.
 * Exported for testing.
 */
var RingId;
(function (RingId) {
    // The ID for the ring showing the user's current focus.
    RingId["PRIMARY"] = "primary";
    // The ID for the ring showing a preview of the next focus, if the user
    // selects the current element.
    RingId["PREVIEW"] = "preview";
})(RingId || (RingId = {}));
/** The secondary color for both rings. */
const OUTER_COLOR = '#174EA6'; // Google Blue 900
/** The inner color of the preview focus ring. */
const PREVIEW_COLOR = '#8AB4F880'; // Google Blue 300, 50% opacity
/** The inner color of the primary focus ring. */
const PRIMARY_COLOR = '#8AB4F8'; // Google Blue 300
TestImportManager.exportForTesting(FocusRingManager, ['RingId', RingId]);

// Copyright 2017 The Chromium Authors
// Use of this source code is governed by a BSD-style license that can be
// found in the LICENSE file.
/**
 * Class to record metrics for Switch Access.
 */
var SwitchAccessMetrics;
(function (SwitchAccessMetrics) {
    function recordMenuAction(menuAction) {
        const metricName = 'Accessibility.CrosSwitchAccess.MenuAction.' +
            StringUtil.toUpperCamelCase(menuAction);
        chrome.metricsPrivate.recordUserAction(metricName);
    }
    SwitchAccessMetrics.recordMenuAction = recordMenuAction;
})(SwitchAccessMetrics || (SwitchAccessMetrics = {}));

// Copyright 2020 The Chromium Authors
// Use of this source code is governed by a BSD-style license that can be
// found in the LICENSE file.
var MenuAction = chrome.accessibilityPrivate.SwitchAccessMenuAction;
/**
 * Class to handle performing actions with Switch Access, including determining
 * which actions are available in the given context.
 */
class ActionManager {
    /**
     * The node on which actions are currently being performed.
     * Null if the menu is closed.
     */
    actionNode_;
    menuManager_;
    menuStack_ = [];
    static instance;
    constructor() {
        this.menuManager_ = MenuManager.create();
    }
    static init() {
        if (ActionManager.instance) {
            throw SwitchAccess.error(ErrorType.DUPLICATE_INITIALIZATION, 'Cannot call ActionManager.init() more than once.');
        }
        ActionManager.instance = new ActionManager();
    }
    // ================= Static Methods ==================
    /**
     * Exits all of the open menus and unconditionally closes the menu window.
     */
    static exitAllMenus() {
        // TODO(b/314203187): Not null asserted, check that this is correct.
        ActionManager.instance.menuStack_ = [];
        ActionManager.instance.actionNode_ = null;
        ActionManager.instance.menuManager_.close();
        if (SwitchAccess.mode === Mode.POINT_SCAN) {
            Navigator.byPoint.start();
        }
        else {
            Navigator.byPoint.stop();
        }
    }
    /**
     * Exits the current menu. If there are no menus on the stack, closes the
     * menu.
     */
    static exitCurrentMenu() {
        // TODO(b/314203187): Not null asserted, check that this is correct.
        ActionManager.instance.menuStack_.pop();
        if (ActionManager.instance.menuStack_.length > 0) {
            ActionManager.instance.openCurrentMenu_();
        }
        else {
            ActionManager.exitAllMenus();
        }
    }
    /**
     * Handles what to do when the user presses 'select'.
     * If multiple actions are available for the currently highlighted node,
     * opens the action menu. Otherwise performs the node's default action.
     */
    static onSelect() {
        const node = Navigator.byItem.currentNode;
        if (MenuManager.isMenuOpen() || node.actions.length <= 1 ||
            !node.location) {
            node.doDefaultAction();
            return;
        }
        // TODO(b/314203187): Not null asserted, check that this is correct.
        ActionManager.instance.menuStack_ = [];
        ActionManager.instance.menuStack_.push(MenuType.MAIN_MENU);
        ActionManager.instance.actionNode_ = node;
        ActionManager.instance.openCurrentMenu_();
    }
    static openMenu(menu) {
        // TODO(b/314203187): Not null asserted, check that this is correct.
        ActionManager.instance.menuStack_.push(menu);
        ActionManager.instance.openCurrentMenu_();
    }
    /** Given the action to be performed, appropriately handles performing it. */
    static performAction(action) {
        SwitchAccessMetrics.recordMenuAction(action);
        switch (action) {
            // Global actions:
            case MenuAction.SETTINGS:
                chrome.accessibilityPrivate.openSettingsSubpage('manageAccessibility/switchAccess');
                ActionManager.exitCurrentMenu();
                break;
            case MenuAction.POINT_SCAN:
                ActionManager.exitCurrentMenu();
                Navigator.byPoint.start();
                break;
            case MenuAction.ITEM_SCAN:
                Navigator.byItem.restart();
                ActionManager.exitAllMenus();
                break;
            // Point scan actions:
            case MenuAction.LEFT_CLICK:
            case MenuAction.RIGHT_CLICK:
                // Exit menu, then click (so the action will hit the desired target,
                // instead of the menu).
                FocusRingManager.clearAll();
                ActionManager.exitCurrentMenu();
                Navigator.byPoint.performMouseAction(action);
                break;
            // Item scan actions:
            default:
                // TODO(b/314203187): Not null asserted, check that this is correct.
                ActionManager.instance.performActionOnCurrentNode_(action);
        }
    }
    /** Refreshes the current menu, if needed. */
    static refreshMenuUnconditionally() {
        if (!MenuManager.isMenuOpen()) {
            return;
        }
        // TODO(b/314203187): Not null asserted, check that this is correct.
        ActionManager.instance.openCurrentMenu_();
    }
    /**
     * Refreshes the current menu, if the current action node matches the node
     * provided.
     */
    static refreshMenuForNode(node) {
        // TODO(b/314203187): Not null asserted, check that this is correct.
        const actionNode = ActionManager.instance.actionNode_;
        if (actionNode && node.equals(actionNode)) {
            ActionManager.refreshMenuUnconditionally();
        }
    }
    // ================= Private Methods ==================
    /** Returns all possible actions for the provided menu type */
    actionsForType_(type) {
        switch (type) {
            case MenuType.MAIN_MENU:
                return [
                    MenuAction.COPY,
                    MenuAction.CUT,
                    MenuAction.DECREMENT,
                    MenuAction.DICTATION,
                    MenuAction.DRILL_DOWN,
                    MenuAction.INCREMENT,
                    MenuAction.KEYBOARD,
                    MenuAction.MOVE_CURSOR,
                    MenuAction.PASTE,
                    MenuAction.SCROLL_DOWN,
                    MenuAction.SCROLL_LEFT,
                    MenuAction.SCROLL_RIGHT,
                    MenuAction.SCROLL_UP,
                    MenuAction.SELECT,
                    MenuAction.START_TEXT_SELECTION,
                ];
            case MenuType.TEXT_NAVIGATION:
                return [
                    MenuAction.JUMP_TO_BEGINNING_OF_TEXT,
                    MenuAction.JUMP_TO_END_OF_TEXT,
                    MenuAction.MOVE_UP_ONE_LINE_OF_TEXT,
                    MenuAction.MOVE_DOWN_ONE_LINE_OF_TEXT,
                    MenuAction.MOVE_BACKWARD_ONE_WORD_OF_TEXT,
                    MenuAction.MOVE_FORWARD_ONE_WORD_OF_TEXT,
                    MenuAction.MOVE_BACKWARD_ONE_CHAR_OF_TEXT,
                    MenuAction.MOVE_FORWARD_ONE_CHAR_OF_TEXT,
                    MenuAction.END_TEXT_SELECTION,
                ];
            case MenuType.POINT_SCAN_MENU:
                return [
                    MenuAction.LEFT_CLICK,
                    MenuAction.RIGHT_CLICK,
                ];
            default:
                return [];
        }
    }
    addGlobalActions_(actions) {
        if (SwitchAccess.mode === Mode.POINT_SCAN) {
            actions.push(MenuAction.ITEM_SCAN);
        }
        else {
            actions.push(MenuAction.POINT_SCAN);
        }
        actions.push(MenuAction.SETTINGS);
        return actions;
    }
    get currentMenuType_() {
        return this.menuStack_[this.menuStack_.length - 1];
    }
    getActionsForCurrentMenuAndNode_() {
        if (this.currentMenuType_ === MenuType.POINT_SCAN_MENU) {
            let actions = this.actionsForType_(MenuType.POINT_SCAN_MENU);
            actions = this.addGlobalActions_(actions);
            return actions;
        }
        if (!this.actionNode_ || !this.actionNode_.isValidAndVisible()) {
            return [];
        }
        let actions = this.actionNode_.actions;
        const possibleActions = this.actionsForType_(this.currentMenuType_);
        actions = actions.filter(a => possibleActions.includes(a));
        if (this.currentMenuType_ === MenuType.MAIN_MENU) {
            actions = this.addGlobalActions_(actions);
        }
        return actions;
    }
    getLocationForCurrentMenuAndNode_() {
        if (this.currentMenuType_ === MenuType.POINT_SCAN_MENU) {
            return {
                left: Math.floor(Navigator.byPoint.currentPoint.x),
                top: Math.floor(Navigator.byPoint.currentPoint.y),
                width: 1,
                height: 1,
            };
        }
        if (this.actionNode_) {
            return this.actionNode_.location;
        }
        return undefined;
    }
    openCurrentMenu_() {
        const actions = this.getActionsForCurrentMenuAndNode_();
        const location = this.getLocationForCurrentMenuAndNode_();
        if (actions.length < 2) {
            ActionManager.exitCurrentMenu();
        }
        this.menuManager_.open(actions, location);
    }
    performActionOnCurrentNode_(action) {
        // TODO(b/314203187): Not null asserted, check that this is correct.
        if (!this.actionNode_.hasAction(action)) {
            ActionManager.refreshMenuUnconditionally();
            return;
        }
        // We exit the menu before asking the node to perform the action, because
        // having the menu on the group stack interferes with some actions. We do
        // not close the menu bubble until we receive the ActionResponse CLOSE_MENU.
        // If we receive a different response, we re-enter the menu.
        Navigator.byItem.suspendCurrentGroup();
        // TODO(b/314203187): Not null asserted, check that this is correct.
        const response = this.actionNode_.performAction(action);
        switch (response) {
            case ActionResponse.CLOSE_MENU:
                ActionManager.exitAllMenus();
                return;
            case ActionResponse.EXIT_SUBMENU:
                ActionManager.exitCurrentMenu();
                return;
            case ActionResponse.REMAIN_OPEN:
                Navigator.byItem.restoreSuspendedGroup();
                return;
            case ActionResponse.RELOAD_MENU:
                ActionManager.refreshMenuUnconditionally();
                return;
            case ActionResponse.OPEN_TEXT_NAVIGATION_MENU:
                if (SwitchAccess.improvedTextInputEnabled()) {
                    this.menuStack_.push(MenuType.TEXT_NAVIGATION);
                }
                this.openCurrentMenu_();
        }
    }
}
/** @type {ActionManager} */
ActionManager.instance;
TestImportManager.exportForTesting(ActionManager);

// Copyright 2017 The Chromium Authors
// Use of this source code is governed by a BSD-style license that can be
// found in the LICENSE file.
var Command = chrome.accessibilityPrivate.SwitchAccessCommand;
/** Runs user commands. */
class SACommands {
    static instance;
    commandMap_ = new Map([
        [Command.SELECT, () => ActionManager.onSelect()],
        [Command.NEXT, () => Navigator.byItem.moveForward()],
        [Command.PREVIOUS, () => Navigator.byItem.moveBackward()],
    ]);
    constructor() {
        chrome.accessibilityPrivate.onSwitchAccessCommand.addListener((command) => this.runCommand_(command));
    }
    static init() {
        if (SACommands.instance) {
            throw SwitchAccess.error(ErrorType.DUPLICATE_INITIALIZATION, 'Cannot create more than one SACommands instance.');
        }
        SACommands.instance = new SACommands();
    }
    runCommand_(command) {
        this.commandMap_.get(command)();
        AutoScanManager.restartIfRunning();
    }
}
TestImportManager.exportForTesting(SACommands);

// Copyright 2017 The Chromium Authors
// Use of this source code is governed by a BSD-style license that can be
// found in the LICENSE file.
/**
 * Class to manage user preferences.
 */
class SettingsManager {
    static async init() {
        await Settings.init(Object.values(Preference));
        Settings.addListener(Preference.AUTO_SCAN_ENABLED, (value) => AutoScanManager.setEnabled(value));
        Settings.addListener(Preference.AUTO_SCAN_TIME, (value) => AutoScanManager.setPrimaryScanTime(value));
        Settings.addListener(Preference.AUTO_SCAN_KEYBOARD_TIME, (value) => AutoScanManager.setKeyboardScanTime(value));
        if (!SettingsManager.settingsAreConfigured_()) {
            chrome.accessibilityPrivate.openSettingsSubpage('manageAccessibility/switchAccess');
        }
    }
    // =============== Private Methods ==============
    /**
     * Whether the current settings configuration is reasonably usable;
     * specifically, whether there is a way to select and a way to navigate.
     */
    static settingsAreConfigured_() {
        const selectPref = Settings.get(Preference.SELECT_DEVICE_KEY_CODES);
        const selectSet = selectPref ? Object.keys(selectPref).length : false;
        const nextPref = Settings.get(Preference.NEXT_DEVICE_KEY_CODES);
        const nextSet = nextPref ? Object.keys(nextPref).length : false;
        const previousPref = Settings.get(Preference.PREVIOUS_DEVICE_KEY_CODES);
        const previousSet = previousPref ? Object.keys(previousPref).length : false;
        const autoScanEnabled = Settings.get(Preference.AUTO_SCAN_ENABLED);
        if (!selectSet) {
            return false;
        }
        if (nextSet || previousSet) {
            return true;
        }
        return Boolean(autoScanEnabled);
    }
}
/** Preferences that are configurable in Switch Access. */
var Preference;
(function (Preference) {
    Preference["AUTO_SCAN_ENABLED"] = "settings.a11y.switch_access.auto_scan.enabled";
    Preference["AUTO_SCAN_TIME"] = "settings.a11y.switch_access.auto_scan.speed_ms";
    Preference["AUTO_SCAN_KEYBOARD_TIME"] = "settings.a11y.switch_access.auto_scan.keyboard.speed_ms";
    Preference["NEXT_DEVICE_KEY_CODES"] = "settings.a11y.switch_access.next.device_key_codes";
    Preference["PREVIOUS_DEVICE_KEY_CODES"] = "settings.a11y.switch_access.previous.device_key_codes";
    Preference["SELECT_DEVICE_KEY_CODES"] = "settings.a11y.switch_access.select.device_key_codes";
})(Preference || (Preference = {}));

// Copyright 2017 The Chromium Authors
// Use of this source code is governed by a BSD-style license that can be
// found in the LICENSE file.
async function initAll() {
    await InstanceChecker.closeExtraInstances();
    // Prevent this service worker from going to sleep.
    KeepAlive.init();
    await Flags.init();
    const desktop = await AsyncUtil.getDesktop();
    await SwitchAccess.init(desktop);
    // Navigator must be initialized before other classes.
    Navigator.initializeSingletonInstances(desktop);
    ActionManager.init();
    AutoScanManager.init();
    FocusRingManager.init();
    SACommands.init();
    SettingsManager.init();
    SwitchAccess.start();
}
initAll();
//# sourceMappingURL=switch_access_loader.rollup.js.map
