// Copyright 2019 The Chromium Authors
// Use of this source code is governed by a BSD-style license that can be
// found in the LICENSE file.
import { EventHandler } from '/common/event_handler.js';
import { RepeatedEventHandler } from '/common/repeated_event_handler.js';
import { TestImportManager } from '/common/testing/test_import_manager.js';
import { ActionManager } from '../action_manager.js';
import { FocusRingManager } from '../focus_ring_manager.js';
import { MenuManager } from '../menu_manager.js';
import { Navigator } from '../navigator.js';
import { SwitchAccess } from '../switch_access.js';
import { ActionResponse } from '../switch_access_constants.js';
import { SAChildNode } from './switch_access_node.js';
const EventType = chrome.automation.EventType;
var MenuAction = chrome.accessibilityPrivate.SwitchAccessMenuAction;
var RoleType = chrome.automation.RoleType;
/**
 * This class handles the behavior of the back button.
 */
export class BackButtonNode extends SAChildNode {
    /** The group that the back button is shown for. */
    group_;
    locationChangedHandler_;
    static automationNode_;
    static clickHandler_;
    static locationForTesting;
    constructor(group) {
        super();
        this.group_ = group;
    }
    // ================= Getters and setters =================
    get actions() {
        return [MenuAction.SELECT];
    }
    get automationNode() {
        return BackButtonNode.automationNode_;
    }
    get group() {
        return this.group_;
    }
    get location() {
        if (BackButtonNode.locationForTesting) {
            return BackButtonNode.locationForTesting;
        }
        if (this.automationNode) {
            return this.automationNode.location;
        }
        return undefined;
    }
    get role() {
        return RoleType.BUTTON;
    }
    // ================= General methods =================
    asRootNode() {
        return undefined;
    }
    equals(other) {
        return other instanceof BackButtonNode;
    }
    isEquivalentTo(node) {
        return node instanceof BackButtonNode || this.automationNode === node;
    }
    isGroup() {
        return false;
    }
    isValidAndVisible() {
        return this.group_.isValidGroup();
    }
    onFocus() {
        super.onFocus();
        chrome.accessibilityPrivate.updateSwitchAccessBubble(chrome.accessibilityPrivate.SwitchAccessBubble.BACK_BUTTON, true /* show */, this.group_.location);
        BackButtonNode.findAutomationNode_();
        this.locationChangedHandler_ = new RepeatedEventHandler(this.group_.automationNode, chrome.automation.EventType.LOCATION_CHANGED, () => FocusRingManager.setFocusedNode(this), { exactMatch: true, allAncestors: true });
    }
    onUnfocus() {
        super.onUnfocus();
        chrome.accessibilityPrivate.updateSwitchAccessBubble(chrome.accessibilityPrivate.SwitchAccessBubble.BACK_BUTTON, false /* show */);
        if (this.locationChangedHandler_) {
            this.locationChangedHandler_.stop();
        }
    }
    performAction(action) {
        if (action === MenuAction.SELECT && this.automationNode) {
            BackButtonNode.onClick_();
            return ActionResponse.CLOSE_MENU;
        }
        return ActionResponse.NO_ACTION_TAKEN;
    }
    ignoreWhenComputingUnionOfBoundingBoxes() {
        return true;
    }
    // ================= Debug methods =================
    debugString(wholeTree, prefix = '', currentNode = null) {
        if (!this.automationNode) {
            return 'BackButtonNode';
        }
        return super.debugString(wholeTree, prefix, currentNode);
    }
    // ================= Static methods =================
    /** Looks for the back button automation node. */
    static findAutomationNode_() {
        if (BackButtonNode.automationNode_ && BackButtonNode.automationNode_.role) {
            return;
        }
        SwitchAccess.findNodeMatching({
            role: RoleType.BUTTON,
            attributes: { className: 'SwitchAccessBackButtonView' },
        }, BackButtonNode.saveAutomationNode_);
    }
    /**
     * This function defines the behavior that should be taken when the back
     * button is pressed.
     */
    static onClick_() {
        if (MenuManager.isMenuOpen()) {
            ActionManager.exitCurrentMenu();
        }
        else {
            Navigator.byItem.exitGroupUnconditionally();
        }
    }
    /** Saves the back button automation node. */
    static saveAutomationNode_(automationNode) {
        BackButtonNode.automationNode_ = automationNode;
        if (BackButtonNode.clickHandler_) {
            BackButtonNode.clickHandler_.setNodes(automationNode);
        }
        else {
            BackButtonNode.clickHandler_ = new EventHandler(automationNode, EventType.CLICKED, BackButtonNode.onClick_);
        }
    }
}
TestImportManager.exportForTesting(BackButtonNode);
