// Copyright 2020 The Chromium Authors
// Use of this source code is governed by a BSD-style license that can be
// found in the LICENSE file.
import { RectUtil } from '/common/rect_util.js';
import { Navigator } from '../navigator.js';
import { ActionResponse } from '../switch_access_constants.js';
import { BackButtonNode } from './back_button_node.js';
import { BasicNode, BasicRootNode } from './basic_node.js';
var MenuAction = chrome.accessibilityPrivate.SwitchAccessMenuAction;
var RoleType = chrome.automation.RoleType;
/**
 * This class handles the behavior of tab nodes at the top level (i.e. as
 * groups).
 */
export class TabNode extends BasicNode {
    tabAsRoot_;
    /**
     * @param node The node in the automation tree
     * @param tabAsRoot A pre-calculated object for exploring the parts of tab
     * (i.e. choosing whether to open the tab or close it).
     */
    constructor(node, parent, tabAsRoot_) {
        super(node, parent);
        this.tabAsRoot_ = tabAsRoot_;
    }
    // ================= Getters and setters =================
    get actions() {
        return [MenuAction.DRILL_DOWN];
    }
    // ================= General methods =================
    asRootNode() {
        return this.tabAsRoot_;
    }
    isGroup() {
        return true;
    }
    performAction(action) {
        if (action !== MenuAction.DRILL_DOWN) {
            return ActionResponse.NO_ACTION_TAKEN;
        }
        Navigator.byItem.enterGroup();
        return ActionResponse.CLOSE_MENU;
    }
    // ================= Static methods =================
    static create(tabNode, parent) {
        const tabAsRoot = new BasicRootNode(tabNode);
        let closeButton;
        for (const child of tabNode.children) {
            if (child.role === RoleType.BUTTON) {
                closeButton = new BasicNode(child, tabAsRoot);
                break;
            }
        }
        if (!closeButton) {
            // Pinned tabs have no close button, and so can be treated as just
            // actionable.
            return new ActionableTabNode(tabNode, parent, null);
        }
        const tabToSelect = new ActionableTabNode(tabNode, tabAsRoot, closeButton);
        const backButton = new BackButtonNode(tabAsRoot);
        tabAsRoot.children = [tabToSelect, closeButton, backButton];
        return new TabNode(tabNode, parent, tabAsRoot);
    }
}
/** This class handles the behavior of tabs as actionable elements */
class ActionableTabNode extends BasicNode {
    closeButton_;
    constructor(node, parent, closeButton_) {
        super(node, parent);
        this.closeButton_ = closeButton_;
    }
    // ================= Getters and setters =================
    get actions() {
        return [MenuAction.SELECT];
    }
    get location() {
        if (!this.closeButton_) {
            return super.location;
        }
        return RectUtil.difference(super.location, this.closeButton_.location);
    }
    // ================= General methods =================
    asRootNode() {
        return undefined;
    }
    isGroup() {
        return false;
    }
}
// TODO(crbug.com/314203187): Not null asserted, check that this is correct.
BasicNode.creators.push({
    predicate: baseNode => baseNode.role === RoleType.TAB &&
        baseNode.root.role === RoleType.DESKTOP,
    creator: TabNode.create,
});
