// Copyright 2023 The Chromium Authors
// Use of this source code is governed by a BSD-style license that can be
// found in the LICENSE file.
import './error_dialog.js';
import 'chrome://resources/ash/common/cr_elements/cr_button/cr_button.js';
import 'chrome://resources/ash/common/cr_elements/cr_shared_vars.css.js';
import 'chrome://resources/polymer/v3_0/paper-progress/paper-progress.js';
import '/strings.m.js';
import 'chrome://resources/ash/common/cr.m.js';
import 'chrome://resources/ash/common/event_target.js';
import { assertNotReached } from 'chrome://resources/ash/common/assert.js';
import { loadTimeData } from 'chrome://resources/ash/common/load_time_data.m.js';
import { PolymerElement } from 'chrome://resources/polymer/v3_0/polymer/polymer_bundled.min.js';
import { getTemplate } from './app.html.js';
import { InstallResult } from './borealis_types.mojom-webui.js';
import { BrowserProxy } from './browser_proxy.js';
const State = {
    WELCOME: 'welcome',
    INSTALLING: 'installing',
    COMPLETED: 'completed',
};
/**
 * @fileoverview
 * Dialog to install borealis.
 */
export class BorealisInstallerAppElement extends PolymerElement {
    static get is() {
        return 'borealis-installer-app';
    }
    static get template() {
        return getTemplate();
    }
    static get properties() {
        return {
            state: {
                type: String,
                value: State.WELCOME,
            },
        };
    }
    constructor() {
        super();
        this.canceling = false;
        this.listenerIds = [];
        this.router = BrowserProxy.getInstance().callbackRouter;
    }
    ready() {
        super.ready();
        this.addEventListener('retry', this.onErrorRetry);
        this.addEventListener('cancel', this.onErrorCancel);
        this.addEventListener('storage', this.onOpenStorage);
    }
    connectedCallback() {
        super.connectedCallback();
        this.listenerIds.push(this.router.onProgressUpdate.addListener((progressFraction, progressLabel) => {
            // Multiply by 100 to get percentage.
            this.installerProgress = Math.round(progressFraction * 100);
            this.progressLabel = progressLabel;
        }), this.router.onInstallFinished.addListener((installResult) => {
            this.handleInstallResult(installResult);
        }), 
        // Called when the user closes the installer (e.g. from the window bar)
        this.router.requestClose.addListener(() => {
            this.cancelAndClose();
        }));
    }
    disconnectedCallback() {
        super.disconnectedCallback();
        this.listenerIds.forEach(id => this.router.removeListener(id));
    }
    onErrorRetry() {
        this.startInstall();
    }
    onErrorCancel() {
        this.cancelAndClose();
    }
    onOpenStorage() {
        BrowserProxy.getInstance().handler.openStoragePage();
        this.cancelAndClose();
    }
    handleInstallResult(installResult) {
        switch (installResult) {
            case InstallResult.kSuccess:
                this.state = State.COMPLETED;
                this.$.installLaunch.focus();
                break;
            case InstallResult.kCancelled:
                this.cancelAndClose();
                break;
            default:
                this.$.errorDialog.show(installResult);
        }
    }
    eq(value1, value2) {
        return value1 === value2;
    }
    getTitle() {
        let titleId = '';
        switch (this.state) {
            case State.WELCOME:
                titleId = 'confirmationTitle';
                break;
            case State.INSTALLING:
                titleId = 'ongoingTitle';
                break;
            case State.COMPLETED:
                titleId = 'finishedTitle';
                break;
            default:
                assertNotReached();
        }
        return loadTimeData.getString(titleId);
    }
    getMessage() {
        let messageId = '';
        switch (this.state) {
            case State.WELCOME:
                messageId = 'confirmationMessage';
                break;
            case State.INSTALLING:
                messageId = 'ongingMessage';
                break;
            case State.COMPLETED:
                messageId = 'finishedMessage';
                break;
            default:
                assertNotReached();
        }
        return loadTimeData.getString(messageId);
    }
    getProgressMessage() {
        return loadTimeData.getStringF('percent', this.installerProgress);
    }
    getProgressLabel() {
        return this.progressLabel;
    }
    shouldShowInstallOrLaunchButton(state) {
        return [State.WELCOME, State.COMPLETED].includes(state);
    }
    getInstallOrLaunchLabel(state) {
        if (state === State.COMPLETED) {
            return loadTimeData.getString('launch');
        }
        return loadTimeData.getString('install');
    }
    getCancelOrCloseLabel(state) {
        if (state === State.COMPLETED) {
            return loadTimeData.getString('close');
        }
        return loadTimeData.getString('cancel');
    }
    onCancelButtonClicked() {
        this.cancelAndClose();
    }
    cancelAndClose() {
        if (this.canceling) {
            return;
        }
        this.canceling = true;
        switch (this.state) {
            case State.INSTALLING:
                BrowserProxy.getInstance().handler.cancelInstall();
                break;
            case State.COMPLETED:
                BrowserProxy.getInstance().handler.shutDown();
                break;
            default:
                break;
        }
        this.closePage();
    }
    onInstallOrLaunchButtonClicked() {
        switch (this.state) {
            case State.WELCOME:
                // 'Install' button clicked.
                this.startInstall();
                break;
            case State.COMPLETED:
                // 'Open Steam' button clicked.
                BrowserProxy.getInstance().handler.launch();
                this.closePage();
                break;
        }
    }
    startInstall() {
        this.installerProgress = 0;
        this.progressLabel = '';
        this.state = State.INSTALLING;
        BrowserProxy.getInstance().handler.install();
    }
    closePage() {
        BrowserProxy.getInstance().handler.onPageClosed();
    }
}
customElements.define(BorealisInstallerAppElement.is, BorealisInstallerAppElement);
