// Copyright 2023 The Chromium Authors
// Use of this source code is governed by a BSD-style license that can be
// found in the LICENSE file.
import 'chrome://resources/polymer/v3_0/paper-ripple/paper-ripple.js';
import { getTemplate } from './file_handler_card.html.js';
/**
 * The BaseCardElement defines the base class for all the file handler and
 * accordion cards.
 */
export class BaseCardElement extends HTMLElement {
    constructor() {
        super();
        const shadowRoot = this.attachShadow({ mode: 'open' });
        shadowRoot.innerHTML = getTemplate();
        this.addStyles();
        this.addEventListener('keyup', this.onKeyUp.bind(this));
    }
    $(query) {
        return this.shadowRoot.querySelector(query);
    }
    addStyles() {
        this.$('#container')
            .classList.add('margin-top', 'round-top', 'round-bottom');
        this.tabIndex = 0;
    }
    onKeyUp(e) {
        if (e.key !== ' ' && e.key !== 'Enter') {
            return;
        }
        e.preventDefault();
        e.stopPropagation();
        this.click();
    }
}
/**
 * The AccordionTopCardElement defines the card at the top of the
 * "Other apps" accordion.
 */
export class AccordionTopCardElement extends BaseCardElement {
    constructor() {
        super(...arguments);
        this.expanded_ = false;
    }
    addStyles() {
        super.addStyles();
        this.$('#icon').style.display = 'none';
        this.$('#title').textContent = 'Other apps';
        this.$('#right-icon').classList.add('chevron');
        this.ariaExpanded = 'false';
        this.role = 'button';
    }
    toggleExpandedState() {
        this.expanded_ = !this.expanded_;
        if (this.expanded_) {
            this.$('#container').classList.add('separator-bottom');
            this.$('#container').classList.remove('round-bottom');
            this.$('#right-icon').setAttribute('expanded', '');
            this.ariaExpanded = 'true';
        }
        else {
            this.$('#container').classList.remove('separator-bottom');
            this.$('#container').classList.add('round-bottom');
            this.$('#right-icon').removeAttribute('expanded');
            this.ariaExpanded = 'false';
        }
        return this.expanded_;
    }
    get expanded() {
        return this.expanded_;
    }
}
/**
 * The FileHandlerCardElement defines the base class for the cloud provider and
 * local handler cards.
 */
export class FileHandlerCardElement extends BaseCardElement {
    constructor() {
        super();
        this.selected_ = false;
        this.ariaSelected = 'false';
        this.ariaCurrent = 'false';
        this.role = 'option';
    }
    updateSelection(selected) {
        this.selected_ = selected;
        if (this.selected_) {
            this.$('#card').setAttribute('selected', '');
            this.ariaSelected = 'true';
            this.ariaCurrent = 'true';
        }
        else {
            this.$('#card').removeAttribute('selected');
            this.ariaSelected = 'false';
            this.ariaCurrent = 'false';
        }
    }
    get selected() {
        return this.selected_;
    }
}
export var CloudProviderType;
(function (CloudProviderType) {
    CloudProviderType[CloudProviderType["NONE"] = 0] = "NONE";
    CloudProviderType[CloudProviderType["DRIVE"] = 1] = "DRIVE";
    CloudProviderType[CloudProviderType["ONE_DRIVE"] = 2] = "ONE_DRIVE";
})(CloudProviderType || (CloudProviderType = {}));
export class CloudProviderCardElement extends FileHandlerCardElement {
    constructor() {
        super(...arguments);
        this.type_ = CloudProviderType.NONE;
    }
    setParameters(type, name, description) {
        this.type_ = type;
        this.$('#title').textContent = name;
        this.$('#description').textContent = description;
    }
    setIconClass(className) {
        this.$('#icon').classList.add(className);
    }
    get type() {
        return this.type_;
    }
}
export class LocalHandlerCardElement extends FileHandlerCardElement {
    constructor() {
        super(...arguments);
        this.taskPosition_ = -1;
    }
    addStyles() { }
    setParameters(taskPosition, name) {
        this.taskPosition_ = taskPosition;
        this.$('#title').textContent = name;
    }
    setIconUrl(url) {
        this.$('#icon').setAttribute('style', 'background-image: url(' + url + ')');
    }
    show() {
        this.style.display = '';
        this.tabIndex = 0;
    }
    hide() {
        this.style.display = 'none';
        this.tabIndex = -1;
    }
    get taskPosition() {
        return this.taskPosition_;
    }
}
customElements.define('accordion-top-card', AccordionTopCardElement);
customElements.define('cloud-provider-card', CloudProviderCardElement);
customElements.define('local-handler-card', LocalHandlerCardElement);
