import { PageHandlerFactory, PageHandlerRemote } from './emoji_picker.mojom-webui.js';
import { EmojiSearch } from './emoji_search.mojom-webui.js';
import { NewWindowProxy } from './new_window_proxy.mojom-webui.js';
import { Status } from './tenor_types.mojom-webui.js';
const HELP_CENTRE_URL = 'https://support.google.com/chrome?p=palette';
export class EmojiPickerApiProxy {
    static { this.instance = null; }
    constructor() {
        this.handler = new PageHandlerRemote();
        this.newWindowProxy = NewWindowProxy.getRemote();
        // TODO(b/309343774): Once search is always on, remove function wrapper.
        this.searchProxy = () => EmojiSearch.getRemote();
        const factory = PageHandlerFactory.getRemote();
        factory.createPageHandler(this.handler.$.bindNewPipeAndPassReceiver());
    }
    showUi() {
        this.handler.showUI();
    }
    insertEmoji(emoji, isVariant, searchLength) {
        this.handler.insertEmoji(emoji, isVariant, searchLength);
    }
    insertGif(gif) {
        this.handler.insertGif(gif);
    }
    isIncognitoTextField() {
        return this.handler.isIncognitoTextField();
    }
    getFeatureList() {
        return this.handler.getFeatureList();
    }
    async getCategories() {
        const { gifCategories } = await this.handler.getCategories();
        return {
            gifCategories: gifCategories.map((category) => ({ name: category })),
        };
    }
    getFeaturedGifs(pos) {
        if (!navigator.onLine) {
            return Promise.resolve({
                status: Status.kNetError,
                featuredGifs: {
                    next: '',
                    results: [],
                },
            });
        }
        return this.handler.getFeaturedGifs(pos || null);
    }
    searchGifs(query, pos) {
        if (!navigator.onLine) {
            return Promise.resolve({
                status: Status.kNetError,
                searchGifs: {
                    next: '',
                    results: [],
                },
            });
        }
        // Avoid sending blank queries to the backend.
        if (query.trim().length === 0) {
            return Promise.resolve({
                status: Status.kHttpOk,
                searchGifs: {
                    next: '',
                    results: [],
                },
            });
        }
        return this.handler.searchGifs(query, pos || null);
    }
    searchEmoji(query) {
        // TODO(b/346457889): Add multilingual search for emoji picker.
        // For now assume English.
        return this.searchProxy().searchEmoji(query, ['en']);
    }
    /** @override */
    getGifsByIds(ids) {
        return this.handler.getGifsByIds(ids);
    }
    openHelpCentreArticle() {
        this.newWindowProxy.openUrl({
            url: HELP_CENTRE_URL,
        });
    }
    getInitialCategory() {
        return this.handler.getInitialCategory();
    }
    getInitialQuery() {
        return this.handler.getInitialQuery();
    }
    updateHistoryInPrefs(category, history) {
        this.handler.updateHistoryInPrefs(category, history);
    }
    updatePreferredVariantsInPrefs(preferredVariants) {
        this.handler.updatePreferredVariantsInPrefs(Object.keys(preferredVariants).map(base => ({
            'base': base,
            'variant': preferredVariants[base],
        })));
    }
    getHistoryFromPrefs(category) {
        return this.handler.getHistoryFromPrefs(category);
    }
    onUiFullyLoaded() {
        this.handler.onUiFullyLoaded();
    }
    convertTenorGifsToEmoji(gifs) {
        return gifs.results.map(({ id, url, previewSize, contentDescription, }) => ({
            base: {
                visualContent: {
                    id,
                    url,
                    previewSize,
                },
                name: contentDescription,
            },
            alternates: [],
        }));
    }
    static getInstance() {
        if (EmojiPickerApiProxy.instance === null) {
            EmojiPickerApiProxy.instance = new EmojiPickerApiProxy();
        }
        return EmojiPickerApiProxy.instance;
    }
    static setInstance(instance) {
        EmojiPickerApiProxy.instance = instance;
    }
}
