// Copyright 2024 The Chromium Authors
// Use of this source code is governed by a BSD-style license that can be
// found in the LICENSE file.
import { UnitLabel } from './unit_label.js';
/**
 * The ID of category for system trend.
 */
export var CategoryTypeEnum;
(function (CategoryTypeEnum) {
    CategoryTypeEnum["BATTERY"] = "Battery";
    CategoryTypeEnum["CPU_FREQUENCY"] = "CPU Frequency";
    CategoryTypeEnum["CPU_USAGE"] = "CPU Usage";
    CategoryTypeEnum["MEMORY"] = "Memory";
    CategoryTypeEnum["THERMAL"] = "Thermals";
    CategoryTypeEnum["ZRAM"] = "Zram";
    CategoryTypeEnum["CUSTOM"] = "Custom";
})(CategoryTypeEnum || (CategoryTypeEnum = {}));
/**
 * Controller for system trend page. Used to maintain the data and return the
 * data displayed in the line chart.
 */
export class SystemTrendController {
    constructor(element) {
        this.element = element;
        this.dataCollection = {
            battery: {
                dataList: [],
                selectedIndices: [],
                unitLabel: new UnitLabel([''], 1),
            },
            cpuFrequency: {
                dataList: [],
                selectedIndices: [],
                unitLabel: new UnitLabel(['kHz', 'MHz', 'GHz'], 1000),
            },
            cpuUsage: {
                dataList: [],
                selectedIndices: [],
                unitLabel: new UnitLabel(['%'], 1),
            },
            memory: {
                dataList: [],
                selectedIndices: [],
                unitLabel: new UnitLabel(['KB', 'MB', 'GB'], 1024),
            },
            thermal: {
                dataList: [],
                selectedIndices: [],
                unitLabel: new UnitLabel(['°C'], 1),
            },
            zram: {
                dataList: [],
                selectedIndices: [],
                unitLabel: new UnitLabel(['B', 'KB', 'MB', 'GB'], 1024),
            },
        };
    }
    setBatteryData(dataSeriesList) {
        this.dataCollection.battery.dataList = dataSeriesList;
        this.element.refreshData(CategoryTypeEnum.BATTERY);
    }
    setCpuFrequencyData(dataSeriesList) {
        this.dataCollection.cpuFrequency.dataList = dataSeriesList;
        if (dataSeriesList.length > 1) {
            // Select the first one and last one to include both big and little cores.
            this.dataCollection.cpuFrequency.selectedIndices =
                [0, dataSeriesList.length - 1];
        }
        this.element.refreshData(CategoryTypeEnum.CPU_FREQUENCY);
    }
    setCpuUsageData(dataSeriesList) {
        this.dataCollection.cpuUsage.dataList = dataSeriesList;
        // The first one is overall usage.
        this.dataCollection.cpuUsage.selectedIndices = [0];
        this.element.refreshData(CategoryTypeEnum.CPU_USAGE);
    }
    setMemoryData(dataSeriesList) {
        this.dataCollection.memory.dataList = dataSeriesList;
        // The first one is available memory.
        this.dataCollection.memory.selectedIndices = [0];
        this.element.refreshData(CategoryTypeEnum.MEMORY);
    }
    setThermalData(dataSeriesList) {
        this.dataCollection.thermal.dataList = dataSeriesList;
        this.element.refreshData(CategoryTypeEnum.THERMAL);
    }
    setZramData(dataSeriesList) {
        this.dataCollection.zram.dataList = dataSeriesList;
        // The first one is total used zram.
        this.dataCollection.zram.selectedIndices = [0];
        this.element.refreshData(CategoryTypeEnum.ZRAM);
    }
    setSelectedIndices(type, selectedIndices) {
        switch (type) {
            case CategoryTypeEnum.BATTERY:
                this.dataCollection.battery.selectedIndices = selectedIndices;
                break;
            case CategoryTypeEnum.CPU_FREQUENCY:
                this.dataCollection.cpuFrequency.selectedIndices = selectedIndices;
                break;
            case CategoryTypeEnum.CPU_USAGE:
                this.dataCollection.cpuUsage.selectedIndices = selectedIndices;
                break;
            case CategoryTypeEnum.MEMORY:
                this.dataCollection.memory.selectedIndices = selectedIndices;
                break;
            case CategoryTypeEnum.THERMAL:
                this.dataCollection.thermal.selectedIndices = selectedIndices;
                break;
            case CategoryTypeEnum.ZRAM:
                this.dataCollection.zram.selectedIndices = selectedIndices;
                break;
            case CategoryTypeEnum.CUSTOM:
                console.error('SystemTrendController: Got unexpected type.');
                break;
        }
    }
    /**
     * Get the required data for line chart.
     *
     * @param type - Type of displayed category.
     * @returns - List of `DataSeriesList` data. Except for custom category, we
     *            only return one element in the list for single source.
     */
    getData(type) {
        switch (type) {
            case CategoryTypeEnum.BATTERY:
                return [this.dataCollection.battery];
            case CategoryTypeEnum.CPU_FREQUENCY:
                return [this.dataCollection.cpuFrequency];
            case CategoryTypeEnum.CPU_USAGE:
                return [this.dataCollection.cpuUsage];
            case CategoryTypeEnum.MEMORY:
                return [this.dataCollection.memory];
            case CategoryTypeEnum.THERMAL:
                return [this.dataCollection.thermal];
            case CategoryTypeEnum.ZRAM:
                return [this.dataCollection.zram];
            case CategoryTypeEnum.CUSTOM:
                return this.getCustomData();
        }
    }
    getCustomData() {
        const output = [];
        const allData = [
            this.dataCollection.cpuUsage,
            this.dataCollection.cpuFrequency,
            this.dataCollection.memory,
            this.dataCollection.zram,
            this.dataCollection.battery,
            this.dataCollection.thermal,
        ];
        for (const data of allData) {
            if (data.selectedIndices.length === 0) {
                continue;
            }
            output.push({
                dataList: data.selectedIndices.map(i => data.dataList[i]),
                selectedIndices: data.selectedIndices,
                unitLabel: data.unitLabel,
            });
        }
        return output;
    }
}
