// Copyright 2024 The Chromium Authors
// Use of this source code is governed by a BSD-style license that can be
// found in the LICENSE file.
import { MAX_LABEL_VERTICAL_NUM, MIN_LABEL_VERTICAL_SPACING, TEXT_SIZE } from '../utils/line_chart_configs.js';
/**
 * A scalable label which can calculate the suitable unit and generate text
 * labels.
 */
export class UnitLabel {
    constructor(units, unitBase) {
        // The current max value for this label. To calculate the suitable units.
        this.maxValue = 0;
        // The cache of maxValue. See `setMaxValue()`.
        this.maxValueCache = 0;
        // The current suitable unit's index.
        this.currentUnitIdx = 0;
        // The generated text labels.
        this.labels = [];
        // The height of the label, in pixels.
        this.height = 0;
        // The maximum precision for the number of the label.
        this.precision = 2;
        // Vertical scale of line chart. The real value between two pixels.
        this.valueScale = 1;
        // True if the label need not be regenerated.
        this.isCache = false;
        this.units = units;
        if (units.length === 0) {
            console.error('UnitLabel: Length of units must greater than 0.');
        }
        this.unitBase = unitBase;
        if (unitBase <= 0) {
            console.error('UnitLabel: unitBase must greater than 0.');
        }
    }
    /**
     * Get the generated text labels.
     */
    getLabels() {
        this.updateLabelsAndScale();
        return this.labels;
    }
    /**
     * Get the vertical scale of line chart.
     */
    getValueScale() {
        this.updateLabelsAndScale();
        return this.valueScale;
    }
    /**
     * Get the current unit scale, which is a multiplier to convert displayed
     * value to real value (raw value).
     */
    getUnitScale() {
        return Math.pow(this.unitBase, this.currentUnitIdx);
    }
    /**
     * Get current suitable unit.
     */
    getUnitString() {
        return this.units[this.currentUnitIdx];
    }
    /**
     * Set the layout of the label.
     * @param height - The label height, in pixels.
     */
    setLayout(height) {
        if (this.height === height) {
            return;
        }
        this.height = height;
        this.isCache = false;
    }
    /**
     * Set the maximum value of the label. Decide the suitable unit by this value.
     */
    setMaxValue(maxValue) {
        if (this.maxValueCache === maxValue) {
            return;
        }
        this.maxValueCache = maxValue;
        const result = this.getSuitableUnit(maxValue);
        this.currentUnitIdx = result.unitIdx;
        this.maxValue = result.value;
        this.isCache = false;
    }
    /**
     * Find the suitable unit for the original value. If the value is greater than
     * `unitBase`, we will try to use a bigger unit.
     */
    getSuitableUnit(value) {
        let unitIdx = 0;
        while (unitIdx + 1 < this.units.length && value >= this.unitBase) {
            value /= this.unitBase;
            ++unitIdx;
        }
        return {
            unitIdx: unitIdx,
            value: value,
        };
    }
    /**
     * Update the labels and scale if the status is changed.
     */
    updateLabelsAndScale() {
        if (this.isCache) {
            return;
        }
        this.isCache = true;
        if (this.maxValue === 0) {
            return;
        }
        const result = this.getSuitableStepSize();
        const stepSize = result.stepSize;
        const stepSizePrecision = result.stepSizePrecision;
        const topLabelValue = this.getTopLabelValue(this.maxValue, stepSize);
        const unitStr = this.getUnitString();
        const labels = [];
        for (let value = topLabelValue; value >= 0; value -= stepSize) {
            const valueStr = value.toFixed(stepSizePrecision);
            const label = valueStr + ' ' + unitStr;
            labels.push(label);
        }
        this.labels = labels;
        const realTopValue = this.getRealValueWithCurrentUnit(topLabelValue);
        this.valueScale = realTopValue / this.height;
    }
    /**
     * Top label value is an exact multiple of `stepSize`.
     */
    getTopLabelValue(maxValue, stepSize) {
        return Math.ceil(maxValue / stepSize) * stepSize;
    }
    /**
     * Transform the value in the current suitable unit to the real value.
     */
    getRealValueWithCurrentUnit(value) {
        return value * this.getUnitScale();
    }
    /**
     * Find a step size to show a suitable amount of labels on screen. The step
     * size according to the `precision` of the label. The minimum step size of
     * the label is `10^(-percision)`.
     *
     * We will try 1 time, 2 times and 5 tims of the default step size. If they
     * are not suitable, we will reduce the precision and try again.
     */
    getSuitableStepSize() {
        const maxLabelNum = this.getMaxNumberOfLabel();
        let stepSize = Math.pow(10, -this.precision);
        // This number is for Number.toFixed. if precision is less than 0, it is set
        // to 0.
        let stepSizePrecision = Math.max(this.precision, 0);
        while (true) {
            if (this.getNumberOfLabelWithStepSize(stepSize) <= maxLabelNum) {
                break;
            }
            if (this.getNumberOfLabelWithStepSize(stepSize * 2) <= maxLabelNum) {
                stepSize *= 2;
                break;
            }
            if (this.getNumberOfLabelWithStepSize(stepSize * 5) <= maxLabelNum) {
                stepSize *= 5;
                break;
            }
            // Reduce the precision.
            stepSize *= 10;
            if (stepSizePrecision > 0) {
                --stepSizePrecision;
            }
        }
        return { stepSize: stepSize, stepSizePrecision: stepSizePrecision };
    }
    /**
     * Get the maximum number of equally spaced labels. `TEXT_SIZE` is doubled
     * because the top two labels are both drawn in the same gap.
     */
    getMaxNumberOfLabel() {
        const minLabelSpacing = 2 * TEXT_SIZE + MIN_LABEL_VERTICAL_SPACING;
        const maxLabelNum = 1 + Math.floor(this.height / minLabelSpacing);
        return Math.min(Math.max(maxLabelNum, 2), MAX_LABEL_VERTICAL_NUM);
    }
    /**
     * Get the number of labels with `stepSize`. Because we want the top of the
     * label to be an exact multiple of the `stepSize`, we use `Math.ceil() + 1`
     * to add an additional label above the `maxValue`.
     */
    getNumberOfLabelWithStepSize(stepSize) {
        return Math.ceil(this.maxValue / stepSize) + 1;
    }
}
