// Copyright 2024 The Chromium Authors
// Use of this source code is governed by a BSD-style license that can be
// found in the LICENSE file.
import { assert } from '//resources/js/assert.js';
/**
 * Helper class to calculate CPU usage.
 */
export class CpuUsageHelper {
    constructor() {
        // Last execution time for logical CPUs. The index for the first dimension is
        // physical CPU ID and the index for the second dimension is logical CPU ID.
        this.lastExecutionTime = [];
    }
    /**
     * Calculate the CPU usage from execution time.
     *
     * @returns - CPU usage for each logical CPU, or null when the value can not
     *            be determined. The index for the first dimension is physical CPU
     *            ID and the index for the second dimension is logical CPU ID.
     *
     *            Return null if the last execution time is not found.
     */
    getCpuUsage(cpu) {
        if (this.lastExecutionTime.length === 0) {
            for (const [physicalCpuId, physicalCpu] of cpu.physicalCpus.entries()) {
                this.lastExecutionTime[physicalCpuId] = [];
                for (const logicalCpu of physicalCpu.logicalCpus) {
                    this.lastExecutionTime[physicalCpuId].push(logicalCpu.executionTime);
                }
            }
            return null;
        }
        const output = [];
        for (const [physicalCpuId, physicalCpu] of cpu.physicalCpus.entries()) {
            output[physicalCpuId] = [];
            for (const [logicalCpuId, logicalCpu] of physicalCpu.logicalCpus
                .entries()) {
                const lastExecTime = this.lastExecutionTime[physicalCpuId][logicalCpuId];
                assert(lastExecTime !== undefined);
                output[physicalCpuId].push(this.getLogicalCpuUsage(logicalCpu.executionTime, lastExecTime));
                this.lastExecutionTime[physicalCpuId][logicalCpuId] =
                    logicalCpu.executionTime;
            }
        }
        return output;
    }
    getLogicalCpuUsage(currentExecTime, lastExecTime) {
        const user = this.getCpuTimeDiff(parseInt(currentExecTime.user), parseInt(lastExecTime.user));
        const system = this.getCpuTimeDiff(parseInt(currentExecTime.system), parseInt(lastExecTime.system));
        const idle = this.getCpuTimeDiff(parseInt(currentExecTime.idle), parseInt(lastExecTime.idle));
        if (user === null || system === null || idle === null) {
            return null;
        }
        const total = user + system + idle;
        if (total === 0) {
            return null;
        }
        return {
            userPercentage: user / total * 100,
            systemPercentage: system / total * 100,
            idlePercentage: idle / total * 100,
        };
    }
    getCpuTimeDiff(current, last) {
        if (current < last) {
            // The increment is negative when the execution time counter exceeds
            // maximum value.
            return null;
        }
        return current - last;
    }
}
