// Copyright 2024 The Chromium Authors
// Use of this source code is governed by a BSD-style license that can be
// found in the LICENSE file.
import { assert } from '//resources/js/assert.js';
/**
 * Get the average point from a range of points.
 */
function getAveragePoint(points) {
    const valueSum = points.reduce((acc, item) => acc + item.value, 0);
    const timeSum = points.reduce((acc, item) => acc + item.time, 0);
    return { value: valueSum / points.length, time: timeSum / points.length };
}
/**
 * The helper class to collect data points, record title and get required point
 * values for displaying on the line chart.
 */
export class DataSeries {
    constructor(title, category) {
        // All the data points of the data series. Sorted by time.
        this.dataPoints = [];
        // Whether the data is visible on the line chart.
        this.isVisible = true;
        // We will report the displayed points based on the cached data below. If the
        // cached data is the same, we don't need to calculate the points again.
        this.cachedStartTime = 0;
        this.cachedEndTime = 0;
        this.cachedStepSize = 0;
        // Used to display partial line chart on the canvas.
        this.displayedPoints = [];
        // The maximum value of `displayedPoints` value.
        this.maxValue = 0;
        this.displayedTitle = title;
        this.displayedCategory = category;
    }
    // Add a new data point to this data series. The time must be greater than the
    // time of the last data point in the data series.
    addDataPoint(value, time) {
        if (!isFinite(value) || !isFinite(time)) {
            console.warn('Add invalid data to DataSeries: value: ' + value +
                ', time: ' + time);
            return;
        }
        const length = this.dataPoints.length;
        if (length > 0 && this.dataPoints[length - 1].time > time) {
            console.warn('Add invalid time to DataSeries: ' + time +
                '. Time must be non-strictly increasing.');
            return;
        }
        this.dataPoints.push({ value: value, time: time });
    }
    // Control the visibility of data series.
    setVisible(isVisible) {
        this.isVisible = isVisible;
    }
    getVisible() {
        return this.isVisible;
    }
    getTitle() {
        return this.displayedTitle;
    }
    // Use full title when displaying custom category.
    getTitleForCustom() {
        return `${this.displayedTitle} (${this.displayedCategory})`;
    }
    getPoints() {
        return this.dataPoints;
    }
    /**
     * Get the displayed points to draw on line chart.
     *
     * @param startTime - The start time for the displayed part of chart.
     * @param endTime - The end time for the displayed part of chart.
     * @param stepSize - The step size in millisecond. We will only display one
     *                   point in one step.
     * @return - The displayed points.
     */
    getDisplayedPoints(startTime, endTime, stepSize) {
        if (!this.isVisible) {
            return [];
        }
        this.updateCachedData(startTime, endTime, stepSize);
        return this.displayedPoints;
    }
    // Get the maximum value of the displayed points. See `getDisplayedPoints()`
    // for details of arguments.
    getDisplayedMaxValue(startTime, endTime, stepSize) {
        if (!this.isVisible) {
            return 0;
        }
        this.updateCachedData(startTime, endTime, stepSize);
        return this.maxValue;
    }
    // Implementation of querying the displayed points and the max value.
    updateCachedData(startTime, endTime, stepSize) {
        if (this.cachedStartTime === startTime && this.cachedEndTime === endTime &&
            this.cachedStepSize === stepSize) {
            return;
        }
        this.displayedPoints =
            this.getDisplayedPointsInner(startTime, endTime, stepSize);
        this.maxValue = this.displayedPoints.reduce((maxValue, item) => Math.max(maxValue, item.value), 0);
        // Updated the cached value after the data is updated.
        this.cachedStartTime = startTime;
        this.cachedEndTime = endTime;
        this.cachedStepSize = stepSize;
    }
    getDisplayedPointsInner(startTime, endTime, stepSize) {
        const output = [];
        let pointsInStep = [];
        // Helper function to collect one point in the current step.
        const storeDisplayedPoint = () => {
            if (pointsInStep.length !== 0) {
                output.push(getAveragePoint(pointsInStep));
                pointsInStep = [];
            }
        };
        // To avoid shaking, we need to minus the offset (`startTime % stepSize`) to
        // keep every step the same under the same `stepSize`.
        // Also minus `stepSize` to collect one more point before `startTime` to
        // avoid showing blank at the start.
        let currentStepStart = startTime - startTime % stepSize - stepSize;
        let currentIndex = this.findLowerBoundPointIndex(currentStepStart);
        // Return empty point list as there is no visible point after `startTime`.
        if (currentIndex >= this.dataPoints.length) {
            return [];
        }
        // If there is no point in the step before `startTime`, we will collect one
        // more point before `currentIndex`.
        if (this.dataPoints[currentIndex].time > startTime && currentIndex >= 1) {
            output.push(this.dataPoints[currentIndex - 1]);
        }
        while (currentIndex < this.dataPoints.length) {
            // Collect one more point outside the visible time to avoid showing blank
            // at the end.
            if (output.length !== 0 && output[output.length - 1].time >= endTime) {
                break;
            }
            const currentPoint = this.dataPoints[currentIndex];
            // Time of current point should be greater than or equal to
            // `currentStepStart`. See `findLowerBoundPointIndex()` for more details.
            assert(currentPoint.time >= currentStepStart);
            // Collect the points in [`currentStepStart`, `currentStepEnd`).
            const currentStepEnd = currentStepStart + stepSize;
            if (currentPoint.time >= currentStepEnd) {
                storeDisplayedPoint();
                currentStepStart = currentStepEnd;
            }
            else {
                pointsInStep.push(currentPoint);
                currentIndex += 1;
            }
        }
        storeDisplayedPoint();
        return output;
    }
    /**
     * Find the minimum index of point which time is greater than or equal to
     * `time` by simple binary search.
     */
    findLowerBoundPointIndex(time) {
        let lower = 0;
        let upper = this.dataPoints.length;
        while (lower < upper) {
            const mid = Math.floor((lower + upper) / 2);
            const point = this.dataPoints[mid];
            if (time <= point.time) {
                upper = mid;
            }
            else {
                lower = mid + 1;
            }
        }
        return lower;
    }
    /**
     * Filter out data points which the `time` field is earlier than `startTime`.
     * Keep an additional buffer of data points to avoid modifying `dataPoints`
     * too frequently.
     *
     * @param startTime - The new start time. Data points which time before this
     *                    should be removed.
     */
    removeOutdatedData(startTime) {
        // Retain one hour more of data points as buffer so we only need to update
        // `dataPoints` every hour.
        const dataRetentionBuffer = 1 * 60 * 60 * 1000;
        // Find the index of the first data point within the buffer.
        const bufferStartIndex = this.findLowerBoundPointIndex(startTime - dataRetentionBuffer);
        // If there are points outside the buffer, remove them and points in buffer.
        if (bufferStartIndex > 0) {
            const newStartIndex = this.findLowerBoundPointIndex(startTime);
            assert(newStartIndex >= bufferStartIndex);
            this.dataPoints.splice(0, newStartIndex);
        }
    }
    /**
     * Get the statistics data for the data points within visible time frame.
     *
     * @param startTime - The visible start time.
     * @param endTime - The visible end time.
     * @returns - The statistics data of visible data points.
     */
    getLatestStatistics(startTime, endTime) {
        const output = {
            latest: 0,
            min: Number.POSITIVE_INFINITY,
            max: Number.NEGATIVE_INFINITY,
            average: 0,
        };
        let count = 0;
        const lowerBound = this.findLowerBoundPointIndex(startTime);
        for (let i = lowerBound; i < this.dataPoints.length; ++i) {
            if (this.dataPoints[i].time > endTime) {
                break;
            }
            output.latest = this.dataPoints[i].value;
            output.min = Math.min(this.dataPoints[i].value, output.min);
            output.max = Math.max(this.dataPoints[i].value, output.max);
            output.average += this.dataPoints[i].value;
            count += 1;
        }
        if (count !== 0) {
            output.average /= count;
        }
        return output;
    }
}
