// Copyright 2024 The Chromium Authors
// Use of this source code is governed by a BSD-style license that can be
// found in the LICENSE file.
import './info_card.js';
import { PolymerElement } from '//resources/polymer/v3_0/polymer/polymer_bundled.min.js';
import { toFixedFloat } from '../../utils/number_utils.js';
import { getTemplate } from './thermal_card.html.js';
/**
 * The value of temperature unit in selected menu.
 */
var TemperatureUnitEnum;
(function (TemperatureUnitEnum) {
    TemperatureUnitEnum["CELSIUS"] = "celsius";
    TemperatureUnitEnum["FAHRENHEIT"] = "fahrenheit";
})(TemperatureUnitEnum || (TemperatureUnitEnum = {}));
export class HealthdInternalsThermalCardElement extends PolymerElement {
    constructor() {
        super(...arguments);
        // Displayed temperature unit.
        this.temperatureUnit = TemperatureUnitEnum.CELSIUS;
    }
    static get is() {
        return 'healthd-internals-thermal-card';
    }
    static get template() {
        return getTemplate();
    }
    static get properties() {
        return {
            temperatureUnit: { type: String },
        };
    }
    connectedCallback() {
        super.connectedCallback();
        this.$.infoCard.appendCardRow('EC');
        this.$.infoCard.appendCardRow('SYSFS (thermal_zone)');
        this.$.infoCard.appendCardRow('CPU TEMP CHANNELS');
    }
    updateTelemetryData(data) {
        this.latestTelemetryData = data;
        this.refreshThermalCard();
    }
    updateExpanded(isExpanded) {
        this.$.infoCard.updateExpanded(isExpanded);
    }
    refreshThermalCard() {
        if (this.latestTelemetryData === undefined) {
            return;
        }
        const data = this.latestTelemetryData;
        this.$.infoCard.updateDisplayedInfo(0, this.filterThermalsBySource(data.thermals, 'EC'));
        this.$.infoCard.updateDisplayedInfo(1, this.filterThermalsBySource(data.thermals, 'SysFs'));
        const cpuTemperature = data.cpu.temperatureChannels.map(item => ({
            'Label': item.label,
            'Temperature': this.convertTemperature(item.temperatureCelsius),
        }));
        this.$.infoCard.updateDisplayedInfo(2, cpuTemperature);
    }
    filterThermalsBySource(thermals, targetSource) {
        return thermals.filter(item => item.source === targetSource)
            .map(item => ({
            'Name': item.name,
            'Temperature': this.convertTemperature(item.temperatureCelsius),
        }));
    }
    convertTemperature(temperatureCelsius) {
        switch (this.temperatureUnit) {
            case TemperatureUnitEnum.CELSIUS: {
                return `${toFixedFloat(temperatureCelsius, 4)} °C`;
            }
            case TemperatureUnitEnum.FAHRENHEIT: {
                return `${toFixedFloat(temperatureCelsius * 9 / 5 + 32, 4)} °F`;
            }
            default: {
                console.error('Unknown temperature unit: ', this.temperatureUnit);
                return 'N/A';
            }
        }
    }
    onTemperatureUnitChanged() {
        this.temperatureUnit =
            this.$.temperatureUnitSelector.value;
        this.refreshThermalCard();
    }
}
customElements.define(HealthdInternalsThermalCardElement.is, HealthdInternalsThermalCardElement);
