// Copyright 2024 The Chromium Authors
// Use of this source code is governed by a BSD-style license that can be
// found in the LICENSE file.
import { PolymerElement } from '//resources/polymer/v3_0/polymer/polymer_bundled.min.js';
import { getTemplate } from './scrollbar.html.js';
/**
 * Create two div blocks for displaying scrollbar, which is used to show the
 * position of line chart and to scroll the line chart.
 *
 * The width of outer div will be the same as visible chart width. And the width
 * of inner div will be the same as the width of whole chart.
 *
 * With the scrollbar, we can draw the visible part of line chart only instead
 * of the whole chart .
 */
export class HealthdInternalsLineChartScrollbarElement extends PolymerElement {
    constructor() {
        super(...arguments);
        // The range the scrollbar can scroll.
        this.scrollableRange = 0;
        // The current position of the scrollbar.
        this.currentPosition = 0;
        // The visible width of this scrollbar.
        this.visibleWidth = 0;
        // Whether the scrollbar is at the right edge.
        this.atRightEdge = true;
    }
    static get is() {
        return 'healthd-internals-line-chart-scrollbar';
    }
    static get template() {
        return getTemplate();
    }
    connectedCallback() {
        super.connectedCallback();
        this.$.outerDiv.addEventListener('scroll', () => this.onScroll());
    }
    // Return the height of scrollbar element.
    getHeight() {
        return this.$.outerDiv.offsetHeight;
    }
    // Position may be float point number because `scrollLeft` may be float point
    // number.
    getPosition() {
        return Math.round(this.currentPosition);
    }
    // Change the size of the outer div and update the scrollbar position.
    resize(width) {
        if (this.visibleWidth === width) {
            return;
        }
        this.visibleWidth = width;
        this.$.outerDiv.style.width = this.visibleWidth + 'px';
    }
    // Set the scrollable range to `range`. Use the inner div's width to control
    // the scrollable range. If position go out of range after range update, set
    // it to the boundary value.
    setScrollableRange(range) {
        if (this.scrollableRange === range) {
            return;
        }
        this.scrollableRange = range;
        this.$.innerDiv.style.width =
            (this.visibleWidth + this.scrollableRange) + 'px';
        if (this.scrollableRange < this.currentPosition) {
            this.currentPosition = this.scrollableRange;
            this.updateScrollbarPosition();
        }
    }
    // Set the scrollbar position to `position`. If the new position go out of
    // range, set it to the boundary value.
    setPosition(position) {
        const newPosition = Math.max(0, Math.min(position, this.scrollableRange));
        this.currentPosition = newPosition;
        this.updateScrollbarPosition();
    }
    // Return true if scrollbar is at the right edge of the chart.
    isScrolledToRightEdge() {
        return this.atRightEdge;
    }
    // Scroll the scrollbar to the right edge.
    scrollToRightEdge() {
        this.setPosition(this.scrollableRange);
    }
    // Update the scrollbar position.
    updateScrollbarPosition() {
        if (this.$.outerDiv.scrollLeft === this.currentPosition) {
            return;
        }
        this.$.outerDiv.scrollLeft = this.currentPosition;
    }
    // Scrolling event handler.
    onScroll() {
        const newPosition = this.$.outerDiv.scrollLeft;
        if (newPosition === this.currentPosition) {
            return;
        }
        this.currentPosition = newPosition;
        // `scrollLeft` may become a float point number even if we set it to some
        // integer value. If the distance to the right edge less than 2 pixels, we
        // consider that it is scrolled to the right edge.
        const scrollLeftErrorAmount = 2;
        this.atRightEdge =
            this.currentPosition + scrollLeftErrorAmount > this.scrollableRange;
        this.dispatchEvent(new CustomEvent('bar-scroll', { bubbles: true, composed: true }));
    }
}
customElements.define(HealthdInternalsLineChartScrollbarElement.is, HealthdInternalsLineChartScrollbarElement);
