// Copyright 2024 The Chromium Authors
// Use of this source code is governed by a BSD-style license that can be
// found in the LICENSE file.
import { PolymerElement } from '//resources/polymer/v3_0/polymer/polymer_bundled.min.js';
import { getAverageCpuUsage } from '../../utils/cpu_usage_utils.js';
import { getFormattedMemory, MemoryUnitEnum } from '../../utils/memory_utils.js';
import { toFixedFloat } from '../../utils/number_utils.js';
import { UiUpdateHelper } from '../../utils/ui_update_helper.js';
import { getTemplate } from './info.html.js';
export class HealthdInternalsInfoElement extends PolymerElement {
    constructor() {
        super(...arguments);
        // Latest raw data from healthd.
        this.healthdData = undefined;
        // Other latest data.
        this.cpuUsageData = undefined;
        this.zramData = undefined;
        // Displayed info.
        this.infoNumOfCpu = '0';
        this.infoCpuUsage = '0.00%';
        this.infoCpuKernel = '0.00%';
        this.infoCpuUser = '0.00%';
        this.infoCpuIdle = '0.00%';
        this.infoMemoryTotal = '0';
        this.infoMemoryUsed = '0';
        this.infoMemorySwapTotal = '0';
        this.infoMemorySwapUsed = '0';
        this.infoZramTotal = '0';
        this.infoZramOrig = '0';
        this.infoZramCompr = '0';
        this.infoZramComprRatio = 'N/A';
        this.infoZramSpaceRedu = '0.00%';
    }
    static get is() {
        return 'healthd-internals-info';
    }
    static get template() {
        return getTemplate();
    }
    static get properties() {
        return {
            infoNumOfCpu: { type: String },
            infoCpuUsage: { type: String },
            infoCpuKernel: { type: String },
            infoCpuUser: { type: String },
            infoCpuIdle: { type: String },
            infoMemoryTotal: { type: String },
            infoMemoryUsed: { type: String },
            infoMemorySwapTotal: { type: String },
            infoMemorySwapUsed: { type: String },
            infoZramTotal: { type: String },
            infoZramOrig: { type: String },
            infoZramCompr: { type: String },
            infoZramComprRatio: { type: String },
        };
    }
    connectedCallback() {
        super.connectedCallback();
        this.updateHelper = new UiUpdateHelper(() => {
            this.refreshInfoPage();
        });
    }
    updateTelemetryData(data) {
        const isInitilized = this.healthdData !== undefined;
        this.healthdData = data;
        if (!isInitilized) {
            // Display data as soon as we first receive it.
            this.renderHealthdData();
        }
    }
    updateCpuUsageData(physcialCpuUsage) {
        const isInitilized = this.cpuUsageData !== undefined;
        this.cpuUsageData = physcialCpuUsage;
        if (!isInitilized) {
            // Display data as soon as we first receive it.
            this.renderCpuUsageData();
        }
    }
    updateZramData(zram) {
        const isInitilized = this.zramData !== undefined;
        this.zramData = zram;
        if (!isInitilized) {
            // Display data as soon as we first receive it.
            this.renderZramData();
        }
    }
    updateVisibility(isVisible) {
        this.updateHelper.updateVisibility(isVisible);
    }
    updateUiUpdateInterval(intervalSeconds) {
        this.updateHelper.updateUiUpdateInterval(intervalSeconds);
    }
    refreshInfoPage() {
        this.renderHealthdData();
        this.renderCpuUsageData();
        this.renderZramData();
    }
    renderHealthdData() {
        if (this.healthdData === undefined) {
            return;
        }
        this.infoNumOfCpu = this.healthdData.cpu.numTotalThreads;
        const memory = this.healthdData.memory;
        const unit = MemoryUnitEnum.AUTO;
        const totalMemoryKib = parseInt(memory.totalMemoryKib);
        const usedMemoryKib = totalMemoryKib - parseInt(memory.availableMemoryKib);
        this.infoMemoryTotal = getFormattedMemory(unit, totalMemoryKib);
        this.infoMemoryUsed = getFormattedMemory(unit, usedMemoryKib);
        if (memory.totalSwapMemoryKib !== undefined &&
            memory.freeSwapMemoryKib !== undefined) {
            const totalSwapMemoryKib = parseInt(memory.totalSwapMemoryKib);
            const usedSwapMemoryKib = totalSwapMemoryKib - parseInt(memory.freeSwapMemoryKib);
            this.infoMemorySwapTotal = getFormattedMemory(unit, totalSwapMemoryKib);
            this.infoMemorySwapUsed = getFormattedMemory(unit, usedSwapMemoryKib);
        }
    }
    renderCpuUsageData() {
        if (this.cpuUsageData === undefined) {
            return;
        }
        const flattenCpuUsage = this.cpuUsageData.flat().filter(usage => usage !== null);
        const averageUsage = getAverageCpuUsage(flattenCpuUsage);
        const usagePercentage = averageUsage.systemPercentage + averageUsage.userPercentage;
        this.infoCpuUsage = `${toFixedFloat(usagePercentage, 2)}%`;
        this.infoCpuKernel = `${toFixedFloat(averageUsage.systemPercentage, 2)}%`;
        this.infoCpuUser = `${toFixedFloat(averageUsage.userPercentage, 2)}%`;
        this.infoCpuIdle = `${toFixedFloat(averageUsage.idlePercentage, 2)}%`;
    }
    renderZramData() {
        if (this.zramData === undefined) {
            return;
        }
        const totalUsedMemoryKib = parseInt(this.zramData.totalUsedMemory) / 1024;
        const originalDataSizeKib = parseInt(this.zramData.originalDataSize) / 1024;
        const compressedDataSizeKib = parseInt(this.zramData.compressedDataSize) / 1024;
        // In general, higher compression ratio means better compression.
        const compressionRatio = originalDataSizeKib / compressedDataSizeKib;
        const spaceReductionPercentage = (originalDataSizeKib - compressedDataSizeKib) / originalDataSizeKib *
            100;
        const unit = MemoryUnitEnum.AUTO;
        this.infoZramTotal = getFormattedMemory(unit, totalUsedMemoryKib);
        this.infoZramOrig = getFormattedMemory(unit, originalDataSizeKib);
        this.infoZramCompr = getFormattedMemory(unit, compressedDataSizeKib);
        this.infoZramComprRatio = toFixedFloat(compressionRatio, 2);
        this.infoZramSpaceRedu = `${toFixedFloat(spaceReductionPercentage, 2)}%`;
    }
}
customElements.define(HealthdInternalsInfoElement.is, HealthdInternalsInfoElement);
