// Copyright 2024 The Chromium Authors
// Use of this source code is governed by a BSD-style license that can be
// found in the LICENSE file.
import '//resources/ash/common/cr_elements/cr_button/cr_button.js';
import '../info_card/cpu_card.js';
import '../info_card/fan_card.js';
import '../info_card/memory_card.js';
import '../info_card/power_card.js';
import '../info_card/thermal_card.js';
import { PolymerElement } from '//resources/polymer/v3_0/polymer/polymer_bundled.min.js';
import { UiUpdateHelper } from '../../utils/ui_update_helper.js';
import { getTemplate } from './telemetry.html.js';
export class HealthdInternalsTelemetryElement extends PolymerElement {
    constructor() {
        super(...arguments);
        // Latest raw data from healthd.
        this.healthdData = undefined;
        // Other latest data.
        this.cpuUsageData = undefined;
        this.zramData = undefined;
        // The time that the telemetry data is last updated.
        this.lastUpdateTime = '';
    }
    static get is() {
        return 'healthd-internals-telemetry';
    }
    static get template() {
        return getTemplate();
    }
    static get properties() {
        return {
            lastUpdateTime: { type: String },
        };
    }
    connectedCallback() {
        super.connectedCallback();
        this.updateHelper = new UiUpdateHelper(() => {
            this.refreshTelemetryPage();
        });
    }
    updateTelemetryData(data) {
        const isInitilized = this.healthdData !== undefined;
        this.healthdData = data;
        if (!isInitilized) {
            // Display data as soon as we first receive it.
            this.refreshTelemetryPage();
        }
    }
    updateCpuUsageData(physcialCpuUsage) {
        const isInitilized = this.cpuUsageData !== undefined;
        this.cpuUsageData = physcialCpuUsage;
        if (!isInitilized) {
            // Display data as soon as we first receive it.
            this.refreshTelemetryPage();
        }
    }
    updateZramData(zram) {
        const isInitilized = this.zramData !== undefined;
        this.zramData = zram;
        if (!isInitilized) {
            // Display data as soon as we first receive it.
            this.refreshTelemetryPage();
        }
    }
    updateVisibility(isVisible) {
        this.updateHelper.updateVisibility(isVisible);
    }
    updateUiUpdateInterval(intervalSeconds) {
        this.updateHelper.updateUiUpdateInterval(intervalSeconds);
    }
    refreshTelemetryPage() {
        if (this.healthdData !== undefined) {
            this.$.cpuCard.updateTelemetryData(this.healthdData);
            this.$.fanCard.updateTelemetryData(this.healthdData);
            this.$.memoryCard.updateTelemetryData(this.healthdData);
            this.$.powerCard.updateTelemetryData(this.healthdData);
            this.$.thermalCard.updateTelemetryData(this.healthdData);
        }
        if (this.cpuUsageData !== undefined) {
            this.$.cpuCard.updateCpuUsageData(this.cpuUsageData);
        }
        if (this.zramData !== undefined) {
            this.$.memoryCard.updateZramData(this.zramData);
        }
        this.lastUpdateTime = new Date().toLocaleTimeString();
    }
    onExpandAllButtonClick() {
        this.updateCardsExpanded(true);
    }
    onCollapseAllButtonClicked() {
        this.updateCardsExpanded(false);
    }
    updateCardsExpanded(isExpanded) {
        this.$.cpuCard.updateExpanded(isExpanded);
        this.$.fanCard.updateExpanded(isExpanded);
        this.$.memoryCard.updateExpanded(isExpanded);
        this.$.powerCard.updateExpanded(isExpanded);
        this.$.thermalCard.updateExpanded(isExpanded);
    }
}
customElements.define(HealthdInternalsTelemetryElement.is, HealthdInternalsTelemetryElement);
