// Copyright 2024 The Chromium Authors
// Use of this source code is governed by a BSD-style license that can be
// found in the LICENSE file.
import { assert } from '//resources/js/assert.js';
import { dedupingMixin } from '//resources/polymer/v3_0/polymer/polymer_bundled.min.js';
import { Oobe } from '../../cr_ui.js';
import { OobeUiState } from '../display_manager_types.js';
import { OobeBaseMixin } from './oobe_base_mixin.js';
export const LoginScreenMixin = dedupingMixin((superClass) => {
    const superClassBase = OobeBaseMixin(superClass);
    class LoginScreenMixinInternal extends superClassBase {
        // List of methods exported to login.screenName.<method> API.
        // This is expected to be overridden by the Polymer object using this
        // mixin.
        get EXTERNAL_API() {
            return [];
        }
        /**
         * Initialize screen mixin.
         * @param screenName Name of created class (external api prefix).
         */
        initializeLoginScreen(screenName) {
            const api = {};
            for (const methodName of this.EXTERNAL_API) {
                assert((methodName in this) &&
                    typeof this[methodName] === 'function', 'External method "' + methodName + '" for screen "' +
                    screenName + '" is not a function or is undefined.');
                api[methodName] = (...args) => this[methodName](...args);
            }
            this.sendPrefix = 'login.' + screenName + '.userActed';
            this.registerScreenApi(screenName, api);
            Oobe.getInstance().registerScreen(this);
        }
        userActed(args) {
            if (this.sendPrefix === undefined) {
                console.error('LoginScreenMixin: sendPrefix is undefined');
                return;
            }
            if (typeof args === 'string') {
                args = [args];
            }
            chrome.send(this.sendPrefix, args);
        }
        /**
         * Returns UI state to be used when showing this screen. Default
         * implementation returns OobeUiState.HIDDEN.
         * @return The state of the OOBE UI.
         */
        // eslint-disable-next-line @typescript-eslint/naming-convention
        getOobeUIInitialState() {
            return OobeUiState.HIDDEN;
        }
        /**
         * Invoked for the currently active screen when screen localized data
         * needs to be updated.
         */
        updateLocalizedContent() { }
        /**
         * If defined, invoked when OOBE configuration is loaded.
         */
        updateOobeConfiguration(_configuration) { }
        /**
         * Register external screen API with login object.
         * Example:
         *    this.registerScreenApi('ScreenName', {
         *         foo() { console.log('foo'); },
         *     });
         *     login.ScreenName.foo(); // valid
         *
         * @param name Name of created class.
         * @param api Screen API.
         */
        registerScreenApi(name, api) {
            // TODO(b/260015147) - Improve this.
            if (globalThis.login === undefined) {
                globalThis.login = {};
            }
            assert('login' in globalThis);
            globalThis.login[name] = api;
        }
    }
    return LoginScreenMixinInternal;
});
