// Copyright 2024 The Chromium Authors
// Use of this source code is governed by a BSD-style license that can be
// found in the LICENSE file.
import '//resources/ash/common/cr_elements/cros_color_overrides.css.js';
import '//resources/ash/common/cr_elements/icons.html.js';
import { PolymerElement } from '//resources/polymer/v3_0/polymer/polymer_bundled.min.js';
import { getTemplate } from './oobe_categories_list.html.js';
const GENERATE_WEB_VIEW_CSS = (backgroundColor, iconColor) => {
    return {
        code: `svg {
      background-color: ` +
            backgroundColor + `;
      --oobe-jelly-icon-color: ` +
            iconColor + `;
    }`,
    };
};
const OobeCategoriesListBase = PolymerElement;
export class OobeCategoriesList extends OobeCategoriesListBase {
    static get is() {
        return 'oobe-categories-list';
    }
    static get template() {
        return getTemplate();
    }
    static get properties() {
        return {
            /**
             * List of categories to display.
             */
            categoriesList: {
                type: Array,
                value: [],
                notify: true,
            },
            /**
             * List of selected categories.
             */
            categoriesSelected: {
                type: Array,
                value: [],
            },
            /**
             * Number of selected categories.
             */
            selectedCategoriesCount: {
                type: Number,
                value: 0,
                notify: true,
            },
            /**
             * Number of loaded icons categories.
             */
            loadedIconsCount: {
                type: Number,
                value: 0,
            },
            /**
             * Number of dom repeat rendered items.
             */
            itemRendered: {
                type: Number,
                value: 0,
                observer: 'itemRenderedChanged',
            },
        };
    }
    /**
     * Initialize the list of categories.
     */
    init(categories) {
        this.categoriesList = categories;
        this.categoriesSelected = [];
        this.selectedCategoriesCount = 0;
        this.loadedIconsCount = 0;
        this.itemRendered = 0;
        this.categoriesList.forEach((category) => {
            if (category.selected) {
                this.selectedCategoriesCount++;
                this.categoriesSelected.push(category.categoryId);
            }
        });
    }
    reset() {
        this.categoriesList = [];
        this.categoriesSelected = [];
        this.selectedCategoriesCount = 0;
        this.loadedIconsCount = 0;
        this.itemRendered = 0;
    }
    itemRenderedChanged() {
        if (this.categoriesList.length !== 0 &&
            this.itemRendered === this.categoriesList.length &&
            this.loadedIconsCount === this.categoriesList.length) {
            this.setWebviewStyle();
            this.markCheckedUseCases();
            this.dispatchEvent(new CustomEvent('icons-loaded', { bubbles: true, composed: true }));
        }
    }
    /**
     * Return the list of selected categories.
     */
    getCategoriesSelected() {
        return this.categoriesSelected;
    }
    setWebviewStyle() {
        const iconWebviews = this.shadowRoot?.querySelectorAll('.category-icon');
        if (iconWebviews) {
            const BackgroundColor = getComputedStyle(document.body)
                .getPropertyValue('--cros-sys-primary_container');
            const iconColor = getComputedStyle(document.body)
                .getPropertyValue('--cros-sys-primary');
            for (const iconWebview of iconWebviews) {
                this.injectCss(iconWebview, BackgroundColor, iconColor);
            }
        }
    }
    markCheckedUseCases() {
        this.categoriesList.forEach((category) => {
            if (category.selected) {
                const element = this.shadowRoot?.querySelector(`#${this.getCategoryId(category.categoryId)}`);
                if (element) {
                    element.setAttribute('checked', 'true');
                }
            }
        });
    }
    getIconUrl(iconUrl) {
        return iconUrl;
    }
    onClick(e) {
        const clickedCategory = e.model.item;
        const previousSelectedState = clickedCategory.selected;
        const currentSelectedState = !previousSelectedState;
        const path = `categoriesList.${this.categoriesList.indexOf(clickedCategory)}.selected`;
        this.set(path, currentSelectedState);
        e.currentTarget
            ?.setAttribute('checked', String(currentSelectedState));
        if (currentSelectedState) {
            this.selectedCategoriesCount++;
            this.categoriesSelected.push(clickedCategory.categoryId);
        }
        else {
            this.selectedCategoriesCount--;
            this.categoriesSelected.splice(this.categoriesSelected.indexOf(clickedCategory.categoryId), 1);
        }
        this.notifyPath('categoriesList');
    }
    getCategoryId(categoryId) {
        return 'cr-button-' + categoryId;
    }
    getWebViewId(categoryId) {
        return 'webview-' + categoryId;
    }
    onIconLoaded() {
        this.loadedIconsCount += 1;
    }
    injectCss(webview, backgroundColor, iconColor) {
        webview.addEventListener('contentload', () => {
            webview.insertCSS(GENERATE_WEB_VIEW_CSS(backgroundColor, iconColor), () => {
                if (chrome.runtime.lastError) {
                    console.warn('Failed to insertCSS: ' + chrome.runtime.lastError.message);
                }
            });
        });
    }
}
customElements.define(OobeCategoriesList.is, OobeCategoriesList);
