// Copyright 2022 The Chromium Authors
// Use of this source code is governed by a BSD-style license that can be
// found in the LICENSE file.
import '//resources/ash/common/cr_elements/cros_color_overrides.css.js';
import '//resources/ash/common/cr_elements/icons.html.js';
import { PolymerElement } from '//resources/polymer/v3_0/polymer/polymer_bundled.min.js';
import { OobeI18nMixin } from './mixins/oobe_i18n_mixin.js';
import { getTemplate } from './oobe_screens_list.html.js';
const OobeScreensListBase = OobeI18nMixin(PolymerElement);
export class OobeScreensList extends OobeScreensListBase {
    static get is() {
        return 'oobe-screens-list';
    }
    static get template() {
        return getTemplate();
    }
    static get properties() {
        return {
            /**
             * List of screens to display.
             */
            screensList: {
                type: Array,
                value: [],
                notify: true,
            },
            /**
             * List of selected screens.
             */
            screensSelected: {
                type: Array,
                value: [],
            },
            /**
             * Number of selected screens.
             */
            selectedScreensCount: {
                type: Number,
                value: 0,
                notify: true,
            },
        };
    }
    /**
     * Initialize the list of screens.
     */
    init(screens) {
        this.screensList = screens;
        this.screensSelected = [];
        this.selectedScreensCount = 0;
    }
    /**
     * Return the list of selected screens.
     */
    getScreenSelected() {
        return this.screensSelected;
    }
    onClick(e) {
        const clickedScreen = e.model.item;
        const previousSelectedState = clickedScreen.selected;
        const currentSelectedState = !previousSelectedState;
        const path = `screensList.${this.screensList.indexOf(clickedScreen)}.selected`;
        this.set(path, currentSelectedState);
        e.currentTarget?.
            setAttribute('checked', String(currentSelectedState));
        if (currentSelectedState) {
            this.selectedScreensCount++;
            this.screensSelected.push(clickedScreen.screenId);
        }
        else {
            this.selectedScreensCount--;
            this.screensSelected.splice(this.screensSelected.indexOf(clickedScreen.screenId), 1);
        }
        this.notifyPath('screensList');
    }
    getSubtitle(locale, screenSubtitle, screenId) {
        if (screenSubtitle) {
            // display size screen is special case as the subtitle include directly
            // the percentage  and will be placed in the message placeholder.
            if (screenId === 'display-size') {
                return this.i18nDynamic(locale, 'choobeDisplaySizeSubtitle', screenSubtitle);
            }
            return this.i18nDynamic(locale, screenSubtitle);
        }
        return '';
    }
    isScreenDisabled(isRevisitable, isCompleted) {
        return !isRevisitable && isCompleted;
    }
    isSyncedIconHidden(isSynced, isCompleted, isSelected) {
        return !isSynced || isSelected || isCompleted;
    }
    isScreenVisited(isSelected, isCompleted) {
        return isCompleted && !isSelected;
    }
    getScreenId(screenId) {
        return 'cr-button-' + screenId;
    }
    focus() {
        const screens = this.shadowRoot?.querySelectorAll('.screen-item');
        if (!screens || screens.length < 1) {
            this.focus();
        }
        else {
            // Focus the first enabled screen in the list
            for (const screen of screens) {
                if (!this.isScreenDisabled(screen.hasAttribute('isRevisitable'), screen.hasAttribute('isCompleted'))) {
                    screen.focus();
                    return;
                }
            }
        }
    }
    getAriaLabelToggleButtons(locale, screenTitle, screenSubtitle, screenIsSynced, screenIsCompleted, screenId, screenIsSelected) {
        let ariaLabel = this.i18nDynamic(locale, screenTitle);
        if (screenSubtitle) {
            if (screenId === 'display-size') {
                ariaLabel = ariaLabel + '.' + screenSubtitle;
            }
            else {
                ariaLabel = ariaLabel + '.' + this.i18nDynamic(locale, screenSubtitle);
            }
        }
        if (!screenIsSelected && screenIsCompleted) {
            ariaLabel =
                ariaLabel + '.' + this.i18nDynamic(locale, 'choobeVisitedTile');
        }
        if (!screenIsSelected && !screenIsCompleted && screenIsSynced) {
            ariaLabel =
                ariaLabel + '.' + this.i18nDynamic(locale, 'choobeSyncedTile');
        }
        return ariaLabel;
    }
}
customElements.define(OobeScreensList.is, OobeScreensList);
