// Copyright 2024 The Chromium Authors
// Use of this source code is governed by a BSD-style license that can be
// found in the LICENSE file.
/**
 * @fileoverview Polymer element for perks discovery screen.
 */
import '//resources/polymer/v3_0/iron-icon/iron-icon.js';
import '../../components/dialogs/oobe_loading_dialog.js';
import { assert } from '//resources/js/assert.js';
import { PolymerElement } from '//resources/polymer/v3_0/polymer/polymer_bundled.min.js';
import { LoginScreenMixin } from '../../components/mixins/login_screen_mixin.js';
import { MultiStepMixin } from '../../components/mixins/multi_step_mixin.js';
import { OobeDialogHostMixin } from '../../components/mixins/oobe_dialog_host_mixin.js';
import { OobeI18nMixin } from '../../components/mixins/oobe_i18n_mixin.js';
import { getTemplate } from './perks_discovery.html.js';
export const PerksDiscoveryElementBase = OobeDialogHostMixin(LoginScreenMixin(MultiStepMixin(OobeI18nMixin(PolymerElement))));
const GENERATE_WEB_VIEW_CSS = () => {
    return {
        code: `svg {
          width: 100%;
          height: 100%;
          background-color: ` +
            getComputedStyle(document.body)
                .getPropertyValue('--cros-sys-app_base_shaded') +
            `;
          --cros-sys-illo-color1 :` +
            getComputedStyle(document.body)
                .getPropertyValue('--cros-sys-illo-color1') +
            `;
          --cros-sys-illo-color1-1 :` +
            getComputedStyle(document.body)
                .getPropertyValue('--cros-sys-illo-color1-1') +
            `;
          --cros-sys-illo-color1-2 :` +
            getComputedStyle(document.body)
                .getPropertyValue('--cros-sys-illo-color1-2') +
            `;
          --cros-sys-illo-color2 :` +
            getComputedStyle(document.body)
                .getPropertyValue('--cros-sys-illo-color2') +
            `;
          --cros-sys-illo-color3 :` +
            getComputedStyle(document.body)
                .getPropertyValue('--cros-sys-illo-color3') +
            `;
          --cros-sys-illo-color4 :` +
            getComputedStyle(document.body)
                .getPropertyValue('--cros-sys-illo-color4') +
            `;
          --cros-sys-illo-color5 :` +
            getComputedStyle(document.body)
                .getPropertyValue('--cros-sys-illo-color5') +
            `;
          --cros-sys-illo-color6 :` +
            getComputedStyle(document.body)
                .getPropertyValue('--cros-sys-illo-color6') +
            `;
          --cros-sys-illo-base :` +
            getComputedStyle(document.body)
                .getPropertyValue('--cros-sys-illo-base') +
            `;
          --cros-sys-illo-secondary :` +
            getComputedStyle(document.body)
                .getPropertyValue('--cros-sys-illo-secondary') +
            `;
          --cros-sys-illo-on_primary_container :` +
            getComputedStyle(document.body)
                .getPropertyValue('--cros-sys-illo-on_primary_container') +
            `;
          --cros-sys-illo-card-color1:` +
            getComputedStyle(document.body)
                .getPropertyValue('--cros-sys-illo-card-color1') +
            `;
          --cros-sys-illo-card-color2 :` +
            getComputedStyle(document.body)
                .getPropertyValue('--cros-sys-illo-card-color2') +
            `;
          --cros-sys-illo-card-color3 :` +
            getComputedStyle(document.body)
                .getPropertyValue('--cros-sys-illo-card-color3') +
            `;
          --cros-sys-illo-card-color4 :` +
            getComputedStyle(document.body)
                .getPropertyValue('--cros-sys-illo-card-color4') +
            `;
          --cros-sys-illo-card-color5 :` +
            getComputedStyle(document.body)
                .getPropertyValue('--cros-sys-illo-card-color5') +
            `;
          --cros-sys-illo-card-on_color1 :` +
            getComputedStyle(document.body)
                .getPropertyValue('--cros-sys-illo-card-on_color1') +
            `;
          --cros-sys-illo-card-on_color2 :` +
            getComputedStyle(document.body)
                .getPropertyValue('--cros-sys-illo-card-on_color2') +
            `;
          --cros-sys-illo-card-on_color3 :` +
            getComputedStyle(document.body)
                .getPropertyValue('--cros-sys-illo-card-on_color3') +
            `;
          --cros-sys-illo-card-on_color4 :` +
            getComputedStyle(document.body)
                .getPropertyValue('--cros-sys-illo-card-on_color4') +
            `;
          --cros-sys-illo-card-on_color5 :` +
            getComputedStyle(document.body)
                .getPropertyValue('--cros-sys-illo-card-on_color5') +
            `;
          --cros-sys-app_base_shaded :` +
            getComputedStyle(document.body)
                .getPropertyValue('--cros-sys-app_base_shaded') +
            `;
          --cros-sys-app_base :` +
            getComputedStyle(document.body)
                .getPropertyValue('--cros-sys-app_base') +
            `;
          --cros-sys-base_elevated :` +
            getComputedStyle(document.body)
                .getPropertyValue('--cros-sys-base_elevated') +
            `;
          --cros-sys-illo-analog :` +
            getComputedStyle(document.body)
                .getPropertyValue('--cros-sys-illo-analog') +
            `;
          --cros-sys-illo-muted :` +
            getComputedStyle(document.body)
                .getPropertyValue('--cros-sys-illo-muted') +
            `;
          --cros-sys-illo-complement :` +
            getComputedStyle(document.body)
                .getPropertyValue('--cros-sys-illo-complement') +
            `;
          --cros-sys-illo-on_gradient :` +
            getComputedStyle(document.body)
                .getPropertyValue('--cros-sys-illo-on_gradient') +
            `;
          --cros-sys-illo-analog_variant :` +
            getComputedStyle(document.body)
                .getPropertyValue('--cros-sys-illo-analog_variant') +
            `;
          --cros-sys-illo-muted_variant :` +
            getComputedStyle(document.body)
                .getPropertyValue('--cros-sys-illo-muted_variant') +
            `;
          --cros-sys-illo-complement_variant :` +
            getComputedStyle(document.body)
                .getPropertyValue('--cros-sys-illo-complement_variant') +
            `;
          --cros-sys-illo-on_gradient_variant :` +
            getComputedStyle(document.body)
                .getPropertyValue('--cros-sys-illo-on_gradient_variant') +
            `;
          --cros-sys-primary: ` +
            getComputedStyle(document.body).getPropertyValue('--cros-sys-primary') +
            `;
    }`,
    };
};
/**
 * Available user actions.
 */
var UserAction;
(function (UserAction) {
    UserAction["FINISHED"] = "finished";
    UserAction["LOADED"] = "loaded";
})(UserAction || (UserAction = {}));
var PerksDiscoveryStep;
(function (PerksDiscoveryStep) {
    PerksDiscoveryStep["LOADING"] = "loading";
    PerksDiscoveryStep["OVERVIEW"] = "overview";
})(PerksDiscoveryStep || (PerksDiscoveryStep = {}));
export class PerksDiscoveryElement extends PerksDiscoveryElementBase {
    static get is() {
        return 'perks-discovery-element';
    }
    static get template() {
        return getTemplate();
    }
    static get properties() {
        return {
            /**
             * List of perks to display.
             */
            perksList: {
                type: Array,
                value: [],
                notify: true,
            },
            /**
             * A map that stores a user's interest in various perks.
             */
            selectedPerks: {
                type: Object,
            },
            /**
             * Current perk displayed.
             */
            currentPerk: {
                type: Number,
                value: -1,
            },
            /**
             * Number of dom repeat icons rendered items.
             */
            itemIconsRendered: {
                type: Number,
                value: 0,
            },
            /**
             * Number of dom repeat illustration rendered items.
             */
            itemIllustrationsRendered: {
                type: Number,
                value: 0,
            },
        };
    }
    get UI_STEPS() {
        return PerksDiscoveryStep;
    }
    static get observers() {
        return [
            'itemContentRendered(itemIconsRendered, itemIllustrationsRendered)',
        ];
    }
    // eslint-disable-next-line @typescript-eslint/naming-convention
    defaultUIStep() {
        return PerksDiscoveryStep.LOADING;
    }
    ready() {
        super.ready();
        this.initializeLoginScreen('PerksDiscoveryScreenScreen');
    }
    get EXTERNAL_API() {
        return [
            'setPerksData',
            'setOverviewStep',
        ];
    }
    onBeforeHide() {
        super.onBeforeHide();
        this.perksList = [];
        this.currentPerk = -1;
        this.itemIconsRendered = 0;
        this.itemIllustrationsRendered = 0;
        if (this.selectedPerks) {
            this.selectedPerks.clear();
        }
    }
    isElementHidden(currentPerk, index) {
        return !(currentPerk === index);
    }
    setPerksData(perksData) {
        assert(perksData !== null);
        this.perksList = perksData;
        this.selectedPerks = new Set();
    }
    setOverviewStep() {
        this.currentPerk = 0;
        this.setUIStep(PerksDiscoveryStep.OVERVIEW);
    }
    itemContentRendered(itemIconsRendered, itemIllustrationsRendered) {
        if (this.perksList.length === 0) {
            return;
        }
        if (itemIconsRendered === this.perksList.length) {
            this.setIconsWebviewStyle();
        }
        if (itemIllustrationsRendered === this.perksList.length) {
            this.setIllustrationWebviewStyle();
        }
        if (itemIconsRendered === this.perksList.length &&
            itemIllustrationsRendered === this.perksList.length) {
            this.userActed(UserAction.LOADED);
        }
    }
    /**
     * Dynamically styles illustration webviews based on data from `perksList`.
     * It sets the width and height of each webview using corresponding values
     * from the `perksList` array, ensuring that illustrations are displayed with
     * their intended dimensions.
     */
    setIllustrationWebviewStyle() {
        const iconWebviews = this.shadowRoot?.querySelectorAll('.illustration');
        if (iconWebviews) {
            iconWebviews.forEach((iconWebview, index) => {
                iconWebview.style['width'] =
                    this.perksList[index].illustrationWidth;
                iconWebview.style['height'] =
                    this.perksList[index].illustrationHeight;
                this.injectCss(iconWebview);
            });
        }
    }
    /**
     * Sets the background color of category icon webviews to match the system's
     * base shaded color.
     */
    setIconsWebviewStyle() {
        const iconWebviews = this.shadowRoot?.querySelectorAll('.perk-icon');
        if (iconWebviews) {
            for (const iconWebview of iconWebviews) {
                this.injectCss(iconWebview);
            }
        }
    }
    /**
     * Injects CSS into a webview after its content has loaded.
     */
    injectCss(webview) {
        webview.addEventListener('contentload', () => {
            webview.insertCSS(GENERATE_WEB_VIEW_CSS(), () => {
                if (chrome.runtime.lastError) {
                    console.warn('Failed to insertCSS: ' + chrome.runtime.lastError.message);
                }
            });
        });
    }
    onBackClicked() {
        assert(this.currentPerk > 0);
        this.currentPerk--;
    }
    canGoBack(currentStep) {
        return currentStep > 0;
    }
    advanceToNextPerk() {
        if (this.currentPerk === this.perksList.length - 1) {
            this.userActed([UserAction.FINISHED, Array.from(this.selectedPerks)]);
            return;
        }
        this.currentPerk++;
    }
    onNotInterestedClicked() {
        this.selectedPerks.delete(this.perksList[this.currentPerk].perkId);
        this.advanceToNextPerk();
    }
    onInterestedClicked() {
        this.selectedPerks.add(this.perksList[this.currentPerk].perkId);
        this.advanceToNextPerk();
    }
    /**
     * Returns the title of the perk.
     */
    getCurrentPerkTitle(currentPerk) {
        if (currentPerk === -1) {
            return '';
        }
        assert(currentPerk >= 0 && currentPerk < this.perksList.length);
        return this.perksList[currentPerk].title;
    }
    /**
     * Returns the subtitle of the perk.
     */
    getCurrentPerkSubtitle(currentPerk) {
        if (currentPerk === -1) {
            return '';
        }
        assert(currentPerk >= 0 && currentPerk < this.perksList.length);
        return this.perksList[currentPerk].subtitle;
    }
    /**
     * Returns the additional description of the perk.
     */
    getCurrentPerkAdditionalText(currentPerk) {
        if (currentPerk === -1) {
            return '';
        }
        assert(currentPerk >= 0 && currentPerk < this.perksList.length);
        if (!this.perksList[currentPerk].additionalText) {
            return '';
        }
        return this.perksList[currentPerk].additionalText;
    }
    /**
     * Returns the primary_button_label of the perk.
     */
    getCurrentPerkPrimaryButtonLabel(currentPerk) {
        if (currentPerk === -1) {
            return '';
        }
        assert(currentPerk >= 0 && currentPerk < this.perksList.length);
        return this.perksList[currentPerk].primaryButtonLabel;
    }
    /**
     * Returns the secondary_button_label of the perk.
     */
    getCurrentPerkSecondaryButtonLabel(currentPerk) {
        if (currentPerk === -1) {
            return '';
        }
        assert(currentPerk >= 0 && currentPerk < this.perksList.length);
        return this.perksList[currentPerk].secondaryButtonLabel;
    }
}
customElements.define(PerksDiscoveryElement.is, PerksDiscoveryElement);
