// Copyright 2024 The Chromium Authors
// Use of this source code is governed by a BSD-style license that can be
// found in the LICENSE file.
/**
 * @fileoverview Polymer element for perosonalized recommend apps screen.
 */
import '//resources/polymer/v3_0/iron-icon/iron-icon.js';
import '../../components/dialogs/oobe_loading_dialog.js';
import '../../components/oobe_personalized_apps_list.js';
import { assert } from '//resources/js/assert.js';
import { PolymerElement } from '//resources/polymer/v3_0/polymer/polymer_bundled.min.js';
import { OobeUiState } from '../../components/display_manager_types.js';
import { LoginScreenMixin } from '../../components/mixins/login_screen_mixin.js';
import { MultiStepMixin } from '../../components/mixins/multi_step_mixin.js';
import { OobeDialogHostMixin } from '../../components/mixins/oobe_dialog_host_mixin.js';
import { OobeI18nMixin } from '../../components/mixins/oobe_i18n_mixin.js';
import { getTemplate } from './personalized_recommend_apps.html.js';
var PersonalizedAppsStep;
(function (PersonalizedAppsStep) {
    PersonalizedAppsStep["LOADING"] = "loading";
    PersonalizedAppsStep["OVERVIEW"] = "overview";
})(PersonalizedAppsStep || (PersonalizedAppsStep = {}));
/**
 * Available user actions.
 */
var UserAction;
(function (UserAction) {
    UserAction["SKIP"] = "skip";
    UserAction["NEXT"] = "next";
    UserAction["BACK"] = "back";
    UserAction["LOADED"] = "loaded";
})(UserAction || (UserAction = {}));
export const PersonalizedRecommedAppsElementBase = OobeDialogHostMixin(LoginScreenMixin(MultiStepMixin(OobeI18nMixin(PolymerElement))));
export class PersonalizedRecommedAppsElement extends PersonalizedRecommedAppsElementBase {
    static get is() {
        return 'personalized-apps-element';
    }
    static get template() {
        return getTemplate();
    }
    static get properties() {
        return {
            numberOfSelectedApps: {
                type: Number,
                value: 0,
            },
        };
    }
    get UI_STEPS() {
        return PersonalizedAppsStep;
    }
    // eslint-disable-next-line @typescript-eslint/naming-convention
    defaultUIStep() {
        return PersonalizedAppsStep.LOADING;
    }
    get EXTERNAL_API() {
        return [
            'setAppsAndUseCasesData',
            'setOverviewStep',
        ];
    }
    /**
     * Returns the control which should receive initial focus.
     */
    get defaultControl() {
        const categoriesDialog = this.shadowRoot?.querySelector('#personalizedRecommendDialog');
        if (categoriesDialog instanceof HTMLElement) {
            return categoriesDialog;
        }
        return null;
    }
    // eslint-disable-next-line @typescript-eslint/naming-convention
    getOobeUIInitialState() {
        return OobeUiState.ONBOARDING;
    }
    onBeforeShow() {
        super.onBeforeShow();
        this.setUIStep(PersonalizedAppsStep.LOADING);
    }
    onBeforeHide() {
        super.onBeforeHide();
        this.shadowRoot
            .querySelector('#categoriesAppsList').reset();
    }
    setAppsAndUseCasesData(categoriesData) {
        assert(categoriesData !== null);
        this.shadowRoot
            .querySelector('#categoriesAppsList').init(categoriesData);
    }
    setOverviewStep() {
        this.setUIStep(PersonalizedAppsStep.OVERVIEW);
        const categoriesAppsList = this.shadowRoot?.querySelector('#categoriesAppsList');
        if (categoriesAppsList instanceof HTMLElement) {
            categoriesAppsList.focus();
        }
    }
    /**
     * Handles event when contents in the webview is generated.
     */
    onFullyLoaded() {
        this.userActed(UserAction.LOADED);
    }
    ready() {
        super.ready();
        this.initializeLoginScreen('PersonalizedRecommendAppsScreen');
    }
    onNextClicked() {
        const appsSelected = this.shadowRoot
            .querySelector('#categoriesAppsList').getAppsSelected();
        this.userActed([UserAction.NEXT, appsSelected]);
    }
    onSkip() {
        this.userActed(UserAction.SKIP);
    }
    canProceed() {
        return this.numberOfSelectedApps > 0;
    }
    onBackClicked() {
        this.userActed(UserAction.BACK);
    }
}
customElements.define(PersonalizedRecommedAppsElement.is, PersonalizedRecommedAppsElement);
