// Copyright 2018 The Chromium Authors
// Use of this source code is governed by a BSD-style license that can be
// found in the LICENSE file.
/**
 * @fileoverview Polymer element for displaying network selection OOBE dialog.
 */
import '//resources/polymer/v3_0/iron-icon/iron-icon.js';
import '//resources/ash/common/network/network_list.js';
import '../../components/buttons/oobe_back_button.js';
import '../../components/buttons/oobe_next_button.js';
import '../../components/common_styles/oobe_common_styles.css.js';
import '../../components/common_styles/oobe_dialog_host_styles.css.js';
import '../../components/dialogs/oobe_adaptive_dialog.js';
import '../../components/dialogs/oobe_loading_dialog.js';
import { PolymerElement } from '//resources/polymer/v3_0/polymer/polymer_bundled.min.js';
import { assert } from 'chrome://resources/js/assert.js';
import { OobeAdaptiveDialog } from '../../components/dialogs/oobe_adaptive_dialog.js';
import { LoginScreenMixin } from '../../components/mixins/login_screen_mixin.js';
import { MultiStepMixin } from '../../components/mixins/multi_step_mixin.js';
import { OobeDialogHostMixin } from '../../components/mixins/oobe_dialog_host_mixin.js';
import { OobeI18nMixin } from '../../components/mixins/oobe_i18n_mixin.js';
import { NetworkSelectLogin } from '../../components/network_select_login.js';
import { getTemplate } from './oobe_network.html.js';
export var NetworkScreenStates;
(function (NetworkScreenStates) {
    NetworkScreenStates["DEFAULT"] = "default";
    // This state is only used for quick start flow, but might be extended to
    // the regular OOBE flow as well.
    NetworkScreenStates["QUICK_START_CONNECTING"] = "quick-start-connecting";
})(NetworkScreenStates || (NetworkScreenStates = {}));
const NetworkScreenBase = OobeDialogHostMixin(LoginScreenMixin(MultiStepMixin(OobeI18nMixin(PolymerElement))));
/**
 * @polymer
 */
class NetworkScreen extends NetworkScreenBase {
    static get is() {
        return 'oobe-network-element';
    }
    static get template() {
        return getTemplate();
    }
    static get properties() {
        return {
            /**
             * Network error message.
             */
            errorMessage: {
                type: String,
                value: '',
            },
            /**
             * Whether device is connected to the network.
             */
            isNetworkConnected: {
                type: Boolean,
                value: false,
            },
            /**
             * Controls if periodic background Wi-Fi scans are enabled to update the
             * list of available networks. It is enabled by default so that when user
             * gets to screen networks are already listed, but should be off when
             * user leaves the screen, as scanning can reduce effective bandwidth.
             */
            enableWifiScans: {
                type: Boolean,
                value: true,
            },
            /**
             * Whether Quick start feature is visible. If it's set the quick start
             * button will be shown in the network select login list as first item.
             */
            isQuickStartVisible: {
                type: Boolean,
                value: false,
            },
            // SSID (WiFi Network Name) used during the QuickStart step.
            ssid: {
                type: String,
                value: '',
            },
            // Whether the QuickStart subtitle should be shown while showing the
            // network list
            useQuickStartSubtitle: {
                type: Boolean,
                value: false,
            },
            // Whether to use a title and subtitle telling the user that there was
            // an error during QuickStart. This is only true when the QuickStart flow
            // is aborted while showing the network screen.
            useQuickStartWiFiErrorStrings: {
                type: Boolean,
                value: false,
            },
            // Whether the QuickStart 'Cancel' button is visible.
            quickStartCancelButtonVisible: {
                type: Boolean,
                value: true,
            },
        };
    }
    static get observers() {
        return [];
    }
    get EXTERNAL_API() {
        return ['setError', 'setQuickStartEntryPointVisibility'];
    }
    constructor() {
        super();
    }
    // eslint-disable-next-line @typescript-eslint/naming-convention
    defaultUIStep() {
        return NetworkScreenStates.DEFAULT;
    }
    get UI_STEPS() {
        return NetworkScreenStates;
    }
    getNetworkSelectLogin() {
        const networkSelectLogin = this.shadowRoot?.querySelector('#networkSelectLogin');
        assert(networkSelectLogin instanceof NetworkSelectLogin);
        return networkSelectLogin;
    }
    /**
     * Called when dialog is shown.
     * @param data Screen init payload.
     */
    onBeforeShow(data) {
        super.onBeforeShow(data);
        // Right now `ssid` is only set during quick start flow.
        if (data && 'ssid' in data && data['ssid']) {
            this.ssid = data['ssid'];
        }
        else {
            this.ssid = '';
        }
        if (this.ssid) {
            this.setUIStep(NetworkScreenStates.QUICK_START_CONNECTING);
            this.quickStartCancelButtonVisible = true;
            return;
        }
        this.useQuickStartSubtitle = data?.useQuickStartSubtitle ?? false;
        this.useQuickStartWiFiErrorStrings =
            data?.useQuickStartWiFiErrorStrings ?? false;
        this.setUIStep(NetworkScreenStates.DEFAULT);
        this.enableWifiScans = true;
        this.errorMessage = '';
        this.getNetworkSelectLogin().onBeforeShow();
        this.show();
    }
    /** Called when dialog is hidden. */
    onBeforeHide() {
        super.onBeforeHide();
        this.getNetworkSelectLogin().onBeforeHide();
        this.enableWifiScans = false;
        this.isQuickStartVisible = false;
    }
    ready() {
        super.ready();
        this.initializeLoginScreen('NetworkScreen');
        this.updateLocalizedContent();
    }
    getNetworkDialog() {
        const networkDialog = this.shadowRoot?.querySelector('#networkDialog');
        assert(networkDialog instanceof OobeAdaptiveDialog);
        return networkDialog;
    }
    /** Shows the dialog. */
    show() {
        this.getNetworkDialog().show();
    }
    focus() {
        this.getNetworkDialog().focus();
    }
    /** Updates localized elements of the UI. */
    updateLocalizedContent() {
        this.i18nUpdateLocale();
    }
    /**
     * Returns subtitle of the network dialog.
     */
    getSubtitleMessage(locale, errorMessage, useQuickStartSubtitle, useQuickStartWiFiErrorStrings) {
        if (errorMessage) {
            return errorMessage;
        }
        if (useQuickStartSubtitle) {
            return this.i18nDynamic(locale, 'quickStartNetworkNeededSubtitle');
        }
        if (useQuickStartWiFiErrorStrings) {
            return this.i18nDynamic(locale, 'networkScreenQuickStartWiFiErrorSubtitle');
        }
        return this.i18nDynamic(locale, 'networkSectionSubtitle');
    }
    /**
     * Sets the network error message.
     * @param message Message to be shown.
     */
    setError(message) {
        this.errorMessage = message;
        // Reset QuickStart WiFi error message
        this.useQuickStartWiFiErrorStrings = false;
    }
    setQuickStartEntryPointVisibility(visible) {
        this.isQuickStartVisible = visible;
    }
    /**
     * Returns element of the network list with the given name.
     * Used to simplify testing.
     */
    getNetworkListItemByNameForTest(name) {
        const item = this.getNetworkSelectLogin()
            ?.shadowRoot?.querySelector('#networkSelect')
            ?.getNetworkListItemByNameForTest(name);
        if (item !== undefined) {
            return item;
        }
        return null;
    }
    /**
     * Called after dialog is shown. Refreshes the list of the networks.
     */
    onShown() {
        const networkSelectLogin = this.getNetworkSelectLogin();
        networkSelectLogin.refresh();
        setTimeout(() => {
            if (this.isNetworkConnected) {
                const nextButton = this.shadowRoot?.querySelector('#nextButton');
                assert(nextButton instanceof HTMLElement);
                nextButton.focus();
            }
            else {
                networkSelectLogin.focus();
            }
        }, 300);
        // Timeout is a workaround to correctly propagate focus to
        // RendererFrameHostImpl see https://crbug.com/955129 for details.
    }
    /**
     * Quick start button click handler.
     */
    onQuickStartClicked() {
        this.userActed('activateQuickStart');
    }
    /**
     * Back button click handler.
     */
    onBackClicked() {
        this.userActed('back');
    }
    /**
     * Cancels ongoing connection with the phone for QuickStart.
     */
    onCancelClicked() {
        this.quickStartCancelButtonVisible = false;
        this.userActed('cancel');
    }
    /**
     * Called when the network setup is completed. Either by clicking on
     * already connected network in the list or by directly clicking on the
     * next button in the bottom of the screen.
     */
    onContinue() {
        this.userActed('continue');
    }
}
customElements.define(NetworkScreen.is, NetworkScreen);
