// Copyright 2022 The Chromium Authors
// Use of this source code is governed by a BSD-style license that can be
// found in the LICENSE file.
import './folder_selector.js';
import { BrowserProxy } from '../browser_proxy.js';
import { FOLDER_EXPANDED } from './folder_selector.js';
import { getTemplate } from './manage_mirrorsync.html.js';
/**
 * ManageMirrorSync represents the top level web component that tracks the
 * state for the chrome://manage-mirrorsync dialog.
 */
class ManageMirrorSync extends HTMLElement {
    /**
     * The <folder-selector> component on the page. Gets set when the
     * <manage-mirrorsync> components connects to the DOM.
     */
    folderSelector = null;
    /**
     * Bind the onSpecifyFolderSelection event listener to enable removal of it
     * once the first click has been registered.
     */
    onSpecifyFolderSelectionBound;
    constructor() {
        super();
        this.attachShadow({ mode: 'open' })
            .appendChild(getTemplate().content.cloneNode(true));
        this.onSpecifyFolderSelectionBound =
            this.onSpecifyFolderSelection.bind(this);
    }
    /**
     * Invoked when the <manage-mirrorsync> web component is attached to the DOM.
     */
    connectedCallback() {
        this.shadowRoot.getElementById('selected').addEventListener('click', this.onSpecifyFolderSelectionBound);
        this.folderSelector = this.shadowRoot.querySelector('folder-selector');
    }
    /**
     * Shows the folder-selector web component if the specific folder selection
     * option was chosen. Doesn't hide after initially shown.
     */
    async onSpecifyFolderSelection(event) {
        if (!event.currentTarget) {
            return;
        }
        const isChecked = event.currentTarget.checked;
        if (!isChecked) {
            return;
        }
        this.folderSelector?.addEventListener(FOLDER_EXPANDED, (event) => this.onFolderExpanded(event.detail));
        this.folderSelector?.toggleAttribute('hidden', false);
        this.shadowRoot.getElementById('selected').removeEventListener('click', this.onSpecifyFolderSelectionBound);
        // Kick off retrieving the child folders for the root path.
        this.onFolderExpanded('/');
    }
    /**
     * When a folder gets expanded, kick off the retrieval of the child folders.
     * This will call back into the <folder-selector> element with the data on
     * retrieval.
     */
    async onFolderExpanded(path) {
        const proxy = BrowserProxy.getInstance().handler;
        const childFolders = await proxy.getChildFolders({ path });
        const folderPaths = childFolders.paths.map(({ path }) => path);
        this.folderSelector?.addChildFolders(folderPaths);
    }
}
customElements.define('manage-mirrorsync', ManageMirrorSync);
//# sourceMappingURL=data:application/json;base64,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