// Copyright 2022 The Chromium Authors
// Use of this source code is governed by a BSD-style license that can be
// found in the LICENSE file.
/**
 * @fileoverview The 'nearby-onboarding-one-page' component handles the Nearby
 * Share onboarding flow. It is embedded in chrome://os-settings,
 * chrome://settings and as a standalone dialog via chrome://nearby.
 */
import 'chrome://resources/ash/common/cr_elements/cr_shared_style.css.js';
import 'chrome://resources/ash/common/cr_elements/cr_shared_vars.css.js';
import 'chrome://resources/ash/common/cr_elements/cr_icons.css.js';
import 'chrome://resources/ash/common/cr_elements/cr_input/cr_input.js';
import 'chrome://resources/polymer/v3_0/iron-icon/iron-icon.js';
import './nearby_page_template.js';
import 'chrome://resources/ash/common/cr_elements/cros_color_overrides.css.js';
import { I18nMixin } from 'chrome://resources/ash/common/cr_elements/i18n_mixin.js';
import { loadTimeData } from 'chrome://resources/js/load_time_data.js';
import { DeviceNameValidationResult, Visibility } from 'chrome://resources/mojo/chromeos/ash/services/nearby/public/mojom/nearby_share_settings.mojom-webui.js';
import { PolymerElement } from 'chrome://resources/polymer/v3_0/polymer/polymer_bundled.min.js';
import { getOnboardingEntryPoint, NearbyShareOnboardingEntryPoint, NearbyShareOnboardingFinalState, processOnePageOnboardingCancelledMetrics, processOnePageOnboardingCompleteMetrics, processOnePageOnboardingInitiatedMetrics, processOnePageOnboardingVisibilityButtonOnInitialPageClickedMetrics } from './nearby_metrics_logger.js';
import { getTemplate } from './nearby_onboarding_one_page.html.js';
import { getNearbyShareSettings } from './nearby_share_settings.js';
const NearbyOnboardingOnePageElementBase = I18nMixin(PolymerElement);
export class NearbyOnboardingOnePageElement extends NearbyOnboardingOnePageElementBase {
    static get is() {
        return 'nearby-onboarding-one-page';
    }
    static get template() {
        return getTemplate();
    }
    static get properties() {
        return {
            settings: {
                type: Object,
            },
            errorMessage: {
                type: String,
                value: '',
            },
            /**
             * Onboarding page entry point
             */
            entryPoint_: {
                type: NearbyShareOnboardingEntryPoint,
                value: NearbyShareOnboardingEntryPoint.MAX,
            },
            /**
             * Determines whether the QuickShareV2 flag is enabled.
             */
            isQuickShareV2Enabled_: {
                type: Boolean,
                value: () => loadTimeData.getBoolean('isQuickShareV2Enabled'),
            },
        };
    }
    ready() {
        super.ready();
        this.addEventListener('next', this.onNext_);
        this.addEventListener('close', this.onClose_);
        this.addEventListener('keydown', this.onKeydown_);
        this.addEventListener('view-enter-start', this.onViewEnterStart_);
    }
    onNext_() {
        this.finishOnboarding_();
    }
    onClose_() {
        processOnePageOnboardingCancelledMetrics(this.entryPoint_, NearbyShareOnboardingFinalState.INITIAL_PAGE);
        const onboardingCancelledEvent = new CustomEvent('onboarding-cancelled', {
            bubbles: true,
            composed: true,
        });
        this.dispatchEvent(onboardingCancelledEvent);
    }
    onKeydown_(e) {
        e.stopPropagation();
        if (e.key === 'Enter') {
            this.finishOnboarding_();
            e.preventDefault();
        }
    }
    onViewEnterStart_() {
        this.$.deviceName.focus();
        const url = new URL(document.URL);
        this.entryPoint_ = getOnboardingEntryPoint(url);
        processOnePageOnboardingInitiatedMetrics(this.entryPoint_);
    }
    async onDeviceNameInput_() {
        const result = await getNearbyShareSettings().validateDeviceName(this.$.deviceName.value);
        this.updateErrorMessage_(result.result);
    }
    async finishOnboarding_() {
        const result = await getNearbyShareSettings().setDeviceName(this.$.deviceName.value);
        this.updateErrorMessage_(result.result);
        if (result.result === DeviceNameValidationResult.kValid) {
            /**
             * TODO(crbug.com/1265562): remove this line once the old onboarding
             * is deprecated and default visibility is changed in
             * nearby_share_prefs.cc:kNearbySharingBackgroundVisibilityName
             */
            this.set('settings.visibility', this.getDefaultVisibility_());
            this.set('settings.isOnboardingComplete', true);
            this.set('settings.enabled', true);
            processOnePageOnboardingCompleteMetrics(this.entryPoint_, NearbyShareOnboardingFinalState.INITIAL_PAGE, this.getDefaultVisibility_());
            const onboardingCompleteEvent = new CustomEvent('onboarding-complete', {
                bubbles: true,
                composed: true,
            });
            this.dispatchEvent(onboardingCompleteEvent);
        }
    }
    /**
     * Switch to visibility selection page when the button is clicked
     */
    switchToVisibilitySelectionView_() {
        /**
         * TODO(crbug.com/1265562): remove this line once the old onboarding is
         * deprecated and default visibility is changed in
         * nearby_share_prefs.cc:kNearbySharingBackgroundVisibilityName
         */
        this.set('settings.visibility', this.getDefaultVisibility_());
        processOnePageOnboardingVisibilityButtonOnInitialPageClickedMetrics();
        const changePageEvent = new CustomEvent('change-page', { bubbles: true, composed: true, detail: { page: 'visibility' } });
        this.dispatchEvent(changePageEvent);
    }
    updateErrorMessage_(validationResult) {
        switch (validationResult) {
            case DeviceNameValidationResult.kErrorEmpty:
                this.errorMessage = this.i18n('nearbyShareDeviceNameEmptyError');
                break;
            case DeviceNameValidationResult.kErrorTooLong:
                this.errorMessage = this.i18n('nearbyShareDeviceNameTooLongError');
                break;
            case DeviceNameValidationResult.kErrorNotValidUtf8:
                this.errorMessage =
                    this.i18n('nearbyShareDeviceNameInvalidCharactersError');
                break;
            default:
                this.errorMessage = '';
                break;
        }
    }
    hasErrorMessage_(errorMessage) {
        return errorMessage !== '';
    }
    /**
     * Temporary workaround to set default visibility. Changing the
     * kNearbySharingBackgroundVisibilityName in nearby_share_prefs.cc results in
     * setting visibility selection to 'all contacts' in nearby_visibility_page in
     * existing onboarding workflow.
     *
     * TODO(crbug.com/1265562): remove this function once the old onboarding is
     * deprecated and default visibility is changed in
     * nearby_share_prefs.cc:kNearbySharingBackgroundVisibilityName
     */
    getDefaultVisibility_() {
        if (this.settings.visibility === Visibility.kUnknown) {
            return Visibility.kAllContacts;
        }
        return this.settings.visibility;
    }
    getVisibilitySelectionButtonText_() {
        const visibility = this.getDefaultVisibility_();
        if (this.isQuickShareV2Enabled_) {
            switch (visibility) {
                case Visibility.kAllContacts:
                    return this.i18n('nearbyShareContactVisiblityContactsButton');
                case Visibility
                    .kSelectedContacts: // Selected Contacts does not exist in Quick
                // Share v2. Your devices set instead.
                case Visibility.kYourDevices:
                    return this.i18n('nearbyShareContactVisibilityYourDevices');
                case Visibility.kNoOne:
                    return this.i18n('nearbyShareContactVisibilityNone');
                default:
                    return this.i18n('nearbyShareContactVisiblityContactsButton');
            }
        }
        switch (visibility) {
            case Visibility.kAllContacts:
                return this.i18n('nearbyShareContactVisiblityContactsButton');
            case Visibility.kSelectedContacts:
                return this.i18n('nearbyShareContactVisibilitySome');
            case Visibility.kYourDevices:
                return this.i18n('nearbyShareContactVisibilityYourDevices');
            case Visibility.kNoOne:
                return this.i18n('nearbyShareContactVisibilityNone');
            default:
                return this.i18n('nearbyShareContactVisiblityContactsButton');
        }
    }
    getVisibilitySelectionButtonIcon_() {
        const visibility = this.getDefaultVisibility_();
        if (this.isQuickShareV2Enabled_) {
            switch (visibility) {
                case Visibility.kAllContacts:
                    return 'contact-all';
                case Visibility
                    .kSelectedContacts: // Selected Contacts does not exist in Quick
                // Share v2. Your devices set instead.
                case Visibility.kYourDevices:
                    return 'your-devices';
                case Visibility.kNoOne:
                    return 'visibility-off';
                default:
                    return 'contact-all';
            }
        }
        switch (visibility) {
            case Visibility.kAllContacts:
                return 'contact-all';
            case Visibility.kSelectedContacts:
                return 'contact-group';
            case Visibility.kYourDevices:
                return 'your-devices';
            case Visibility.kNoOne:
                return 'visibility-off';
            default:
                return 'contact-all';
        }
    }
    /**
     * TODO(crbug.com/1265562): Add strings for other modes and switch based on
     * default visibility selection
     */
    getVisibilitySelectionButtonHelpText_() {
        return this.i18n('nearbyShareOnboardingPageDeviceVisibilityHelpAllContacts');
    }
}
customElements.define(NearbyOnboardingOnePageElement.is, NearbyOnboardingOnePageElement);
