// Copyright 2020 The Chromium Authors
// Use of this source code is governed by a BSD-style license that can be
// found in the LICENSE file.
/**
 * @fileoverview The 'nearby-visibility-page' component is part of the Nearby
 * Share onboarding flow. It allows users to setup their visibility preference
 * while enabling the feature for the first time.
 *
 * It is embedded in chrome://os-settings, chrome://settings and as a standalone
 * dialog via chrome://nearby.
 */
import 'chrome://resources/ash/common/cr_elements/cr_shared_style.css.js';
import 'chrome://resources/ash/common/cr_elements/cr_icons.css.js';
import './nearby_contact_visibility.js';
import { I18nMixin } from 'chrome://resources/ash/common/cr_elements/i18n_mixin.js';
import { PolymerElement } from 'chrome://resources/polymer/v3_0/polymer/polymer_bundled.min.js';
import { getOnboardingEntryPoint, NearbyShareOnboardingEntryPoint, NearbyShareOnboardingFinalState, processOnePageOnboardingCancelledMetrics, processOnePageOnboardingCompleteMetrics, processOnePageOnboardingManageContactsMetrics, processOnePageOnboardingVisibilityPageShownMetrics } from './nearby_metrics_logger.js';
import { getTemplate } from './nearby_visibility_page.html.js';
const NearbyVisibilityPageElementBase = I18nMixin(PolymerElement);
export class NearbyVisibilityPageElement extends NearbyVisibilityPageElementBase {
    static get is() {
        return 'nearby-visibility-page';
    }
    static get template() {
        return getTemplate();
    }
    static get properties() {
        return {
            settings: {
                type: Object,
                notify: true,
            },
            isVisibilitySelected_: {
                type: Boolean,
                notify: true,
            },
            /**
             * Onboarding page entry point
             */
            entryPoint_: {
                type: NearbyShareOnboardingEntryPoint,
                value: NearbyShareOnboardingEntryPoint.MAX,
            },
        };
    }
    ready() {
        super.ready();
        this.addEventListener('next', this.onNext_);
        this.addEventListener('manage-contacts', this.onManageContacts_);
        this.addEventListener('close', this.onClose_);
        this.addEventListener('view-enter-start', this.onVisibilityViewEnterStart_);
    }
    onNext_() {
        this.$.contactVisibility.saveVisibilityAndAllowedContacts();
        this.set('settings.isOnboardingComplete', true);
        this.set('settings.enabled', true);
        processOnePageOnboardingCompleteMetrics(this.entryPoint_, NearbyShareOnboardingFinalState.VISIBILITY_PAGE, this.$.contactVisibility.getSelectedVisibility());
        const onboardingCompleteEvent = new CustomEvent('onboarding-complete', {
            bubbles: true,
            composed: true,
        });
        this.dispatchEvent(onboardingCompleteEvent);
    }
    onClose_() {
        processOnePageOnboardingCancelledMetrics(this.entryPoint_, NearbyShareOnboardingFinalState.VISIBILITY_PAGE);
        const onboardingCancelledEvent = new CustomEvent('onboarding-cancelled', {
            bubbles: true,
            composed: true,
        });
        this.dispatchEvent(onboardingCancelledEvent);
    }
    onVisibilityViewEnterStart_() {
        processOnePageOnboardingVisibilityPageShownMetrics();
        const url = new URL(document.URL);
        this.entryPoint_ = getOnboardingEntryPoint(url);
    }
    onManageContacts_() {
        processOnePageOnboardingManageContactsMetrics();
        window.open(this.i18n('nearbyShareManageContactsUrl'), '_blank');
    }
}
customElements.define(NearbyVisibilityPageElement.is, NearbyVisibilityPageElement);
