// Copyright 2019 The Chromium Authors
// Use of this source code is governed by a BSD-style license that can be
// found in the LICENSE file.

import 'chrome://resources/ash/common/cr_elements/cr_button/cr_button.js';
import 'chrome://resources/ash/common/cr_elements/cr_dialog/cr_dialog.js';
import 'chrome://resources/ash/common/cr_elements/cr_input/cr_input.js';
import '/strings.m.js';

import {loadTimeData} from '//resources/ash/common/load_time_data.m.js';
import {assert} from 'chrome://resources/ash/common/assert.js';
import {I18nBehavior} from 'chrome://resources/ash/common/i18n_behavior.js';
import {SmbBrowserProxy, SmbBrowserProxyImpl} from 'chrome://resources/ash/common/smb_shares/smb_browser_proxy.js';
import {ColorChangeUpdater} from 'chrome://resources/cr_components/color_change_listener/colors_css_updater.js';
import {html, Polymer} from 'chrome://resources/polymer/v3_0/polymer/polymer_bundled.min.js';

/**
 * @fileoverview
 * 'smb-credentials-dialog' is used to update the credentials for a mounted
 * smb share.
 */
Polymer({
  is: 'smb-credentials-dialog',

  _template: html`<!--_html_template_start_-->
<style>
  cr-dialog::part(dialog) {
    background-color: var(--cros-sys-dialog_container);
    border-radius: 0;
    height: 100%;
    width: 100%;
  }

  cr-input {
    --cr-form-field-label-color: var(--cros-sys-on_surface);
    --cr-input-background-color: var(--cros-sys-input_field_on_base);
    --cr-input-border-radius: 8px;
    --cr-input-color: var(--cros-sys-on_surface);
    --cr-input-error-color: var(--cros-sys-error);
    --cr-input-error-display: none;
    --cr-input-focus-color: var(--cros-sys-primary);
    --cr-input-min-height: 36px;
    --cr-input-padding-end: 16px;
    --cr-input-padding-start: 16px;
    --cr-input-placeholder-color: var(--cros-sys-secondary);
    font: var(--cros-body-2-font);
    margin-bottom: var(--cr-form-field-bottom-spacing);
  }

  cr-button {
    --active-bg: transparent;
    --active-shadow: none;
    --active-shadow-action: none;
    --bg-action: var(--cros-sys-primary);
    --cr-button-height: 36px;
    --disabled-bg-action:
        var(--cros-sys-disabled_container);
    --disabled-bg: var(--cros-sys-disabled_container);;
    --disabled-text-color: var(--cros-sys-disabled);
    /* Use the default bg color as hover color because we rely on
       hoverBackground layer below.  */
    --hover-bg-action: var(--cros-sys-primary);
    --hover-bg-color: var(--cros-sys-primary_container);
    --ink-color: var(--cros-sys-ripple_primary);
    --ripple-opacity-action: 1;
    --ripple-opacity: 1;
    --text-color-action: var(--cros-sys-on_primary);
    --text-color: var(--cros-sys-on_primary_container);
    border: none;
    border-radius: 18px;
    font: var(--cros-button-2-font);
    position: relative;
  }

  cr-button.cancel-button {
    background-color: var(--cros-sys-primary_container);
  }

  cr-button.cancel-button:hover::part(hoverBackground) {
    background-color: var(--cros-sys-hover_on_subtle);
    display: block;
  }

  cr-button.action-button:hover::part(hoverBackground) {
    background-color: var(--cros-sys-hover_on_prominent);
    display: block;
  }

  :host-context(.focus-outline-visible) cr-button:focus {
    /* disable the focus shadow because we use outline below */
    box-shadow: none;
    outline: 2px solid var(--cros-sys-focus_ring);
    outline-offset: 2px;
  }

  .action-button:not(:active):hover {
    box-shadow: none;
  }

  [slot='title'] {
    --cr-dialog-title-slot-padding-bottom: 32px;
    --cr-dialog-title-slot-padding-end: 32px;
    --cr-dialog-title-slot-padding-start: 32px;
    --cr-dialog-title-slot-padding-top: 32px;
    --cr-primary-text-color: var(--cros-sys-on_surface);
    font: var(--cros-display-7-font);
  }

  [slot='body'] #password {
    margin-bottom: 0;
  }

  [slot='body'] {
    --cr-dialog-body-padding-horizontal: 32px;
    --cr-form-field-bottom-spacing: 8px;
  }

  [slot='button-container'] {
    --cr-dialog-button-container-padding-bottom: 28px;
    --cr-dialog-button-container-padding-horizontal: 32px;
    padding-top: 32px;
  }
</style>

<cr-dialog id="dialog">
  <div slot="title">[[i18n('smbCredentialsDialogTitle', sharePath_)]]</div>
  <div slot="body" spellcheck="false">
    <cr-input id="username" label="[[i18n('smbCredentialsUsername')]]"
          value="{{username_}}">
    </cr-input>
    <cr-input id="password" type="password"
        label="[[i18n('smbCredentialsPassword')]]" value="{{password_}}">
    </cr-input>
  </div>
  <div slot="button-container">
    <cr-button class="cancel-button" on-click="onCancelButtonClick_">
      [[i18n('cancel')]]
    </cr-button>
    <cr-button class="action-button" on-click="onSaveButtonClick_"
        disabled="[[!username_]]">
      [[i18n('save')]]
    </cr-button>
  </div>
</cr-dialog>
<!--_html_template_end_-->`,

  behaviors: [I18nBehavior],

  properties: {
    /** @private {string} */
    sharePath_: String,

    /** @private {string} */
    username_: String,

    /** @private {string} */
    password_: String,
  },

  /** @private {?SmbBrowserProxy} */
  browserProxy_: null,

  /** @override */
  created() {
    this.browserProxy_ = SmbBrowserProxyImpl.getInstance();

    /** @suppress {checkTypes} */
    (function() {
      ColorChangeUpdater.forDocument().start();
    })();
  },

  /** @override */
  attached() {
    const dialogArgs = chrome.getVariableValue('dialogArguments');
    assert(dialogArgs);
    const args = JSON.parse(dialogArgs);
    assert(args);
    assert(args.path);
    this.sharePath_ = args.path;

    this.$.dialog.showModal();
  },

  /** @private */
  onCancelButtonClick_() {
    chrome.send('dialogClose');
  },

  /** @private */
  onSaveButtonClick_() {
    this.browserProxy_.updateCredentials(this.username_, this.password_);
    chrome.send('dialogClose');
  },
});
