// Copyright 2015 The Chromium Authors
// Use of this source code is governed by a BSD-style license that can be
// found in the LICENSE file.
/**
 * @fileoverview
 * 'settings-downloads-page' is the settings page containing downloads
 * settings.
 */
import 'chrome://resources/cr_elements/cr_button/cr_button.js';
import 'chrome://resources/cr_elements/cr_shared_style.css.js';
import '../controls/controlled_button.js';
import '../controls/settings_toggle_button.js';
import '../settings_shared.css.js';
import { PrefsMixin } from '/shared/settings/prefs/prefs_mixin.js';
import { WebUiListenerMixin } from 'chrome://resources/cr_elements/web_ui_listener_mixin.js';
import { PolymerElement } from 'chrome://resources/polymer/v3_0/polymer/polymer_bundled.min.js';
import { loadTimeData } from '../i18n_setup.js';
import { getSearchManager } from '../search_settings.js';
import { DownloadsBrowserProxyImpl } from './downloads_browser_proxy.js';
import { getTemplate } from './downloads_page.html.js';
const SettingsDownloadsPageElementBase = WebUiListenerMixin(PrefsMixin(PolymerElement));
export class SettingsDownloadsPageElement extends SettingsDownloadsPageElementBase {
    static get is() {
        return 'settings-downloads-page';
    }
    static get template() {
        return getTemplate();
    }
    static get properties() {
        return {
            autoOpenDownloads_: {
                type: Boolean,
                value: false,
            },
            // 
            /**
             * The download location string that is suitable to display in the UI.
             */
            downloadLocation_: String,
            // 
            /**
             * Whether the user can toggle the option to display downloads when
             * they're done.
             */
            downloadBubblePartialViewControlledByPref_: {
                type: Boolean,
                value() {
                    return loadTimeData.getBoolean('downloadBubblePartialViewControlledByPref');
                },
            },
        };
    }
    // 
    static get observers() {
        return [
            'handleDownloadLocationChanged_(prefs.download.default_directory.value)',
        ];
    }
    browserProxy_ = DownloadsBrowserProxyImpl.getInstance();
    ready() {
        super.ready();
        this.addWebUiListener('auto-open-downloads-changed', (autoOpen) => {
            this.autoOpenDownloads_ = autoOpen;
        });
        this.browserProxy_.initializeDownloads();
    }
    selectDownloadLocation_() {
        this.browserProxy_.selectDownloadLocation();
    }
    // 
    handleDownloadLocationChanged_() {
        this.browserProxy_
            .getDownloadLocationText(this.getPref('download.default_directory').value)
            .then(text => {
            this.downloadLocation_ = text;
        });
    }
    // 
    onClearAutoOpenFileTypesClick_() {
        this.browserProxy_.resetAutoOpenFileTypes();
    }
    // SettingsPlugin implementation
    async searchContents(query) {
        const searchRequest = await getSearchManager().search(query, this);
        return searchRequest.getSearchResult();
    }
}
customElements.define(SettingsDownloadsPageElement.is, SettingsDownloadsPageElement);
