// Copyright 2015 The Chromium Authors
// Use of this source code is governed by a BSD-style license that can be
// found in the LICENSE file.
/**
 * @fileoverview
 * 'settings-people-page' is the settings page containing sign-in settings.
 */
import 'chrome://resources/cr_elements/cr_button/cr_button.js';
import 'chrome://resources/cr_elements/cr_icon_button/cr_icon_button.js';
import 'chrome://resources/cr_elements/cr_link_row/cr_link_row.js';
import 'chrome://resources/cr_elements/cr_toast/cr_toast.js';
import 'chrome://resources/cr_elements/icons.html.js';
import 'chrome://resources/cr_elements/policy/cr_policy_indicator.js';
import 'chrome://resources/cr_elements/cr_shared_style.css.js';
import 'chrome://resources/cr_elements/cr_shared_vars.css.js';
import '../controls/settings_toggle_button.js';
// 
import '../icons.html.js';
import '../settings_page/settings_section.js';
import '../settings_shared.css.js';
import { ProfileInfoBrowserProxyImpl } from '/shared/settings/people_page/profile_info_browser_proxy.js';
import { SignedInState, SyncBrowserProxyImpl } from '/shared/settings/people_page/sync_browser_proxy.js';
// 
import { convertImageSequenceToPng } from 'chrome://resources/ash/common/cr_picture/png.js';
import { WebUiListenerMixin } from 'chrome://resources/cr_elements/web_ui_listener_mixin.js';
import { assert } from 'chrome://resources/js/assert.js';
import { getImage } from 'chrome://resources/js/icon.js';
import { OpenWindowProxyImpl } from 'chrome://resources/js/open_window_proxy.js';
import { isChromeOS } from 'chrome://resources/js/platform.js';
import { PolymerElement } from 'chrome://resources/polymer/v3_0/polymer/polymer_bundled.min.js';
import { loadTimeData } from '../i18n_setup.js';
import { routes } from '../route.js';
import { Router } from '../router.js';
import { SettingsViewMixin } from '../settings_page/settings_view_mixin.js';
// 
// 
import { AccountManagerBrowserProxyImpl } from './account_manager_browser_proxy.js';
// 
import { getTemplate } from './people_page.html.js';
// 
// 
const SettingsPeoplePageElementBase = SettingsViewMixin(WebUiListenerMixin(PolymerElement));
// 
export class SettingsPeoplePageElement extends SettingsPeoplePageElementBase {
    static get is() {
        return 'settings-people-page';
    }
    static get template() {
        return getTemplate();
    }
    static get properties() {
        return {
            /**
             * Preferences state.
             */
            prefs: {
                type: Object,
                notify: true,
            },
            /**
             * This flag is used to conditionally show a set of new sign-in UIs to the
             * profiles that have been migrated to be consistent with the web
             * sign-ins.
             * TODO(tangltom): In the future when all profiles are completely
             * migrated, this should be removed, and UIs hidden behind it should
             * become default.
             */
            signinAllowed_: {
                type: Boolean,
                value() {
                    return loadTimeData.getBoolean('signinAllowed');
                },
            },
            /**
             * This property stores whether the profile is a Dasherless profiles,
             * which is associated with a non-Dasher account. Some UIs related to
             * sign in and sync service will be different because they are not
             * available for these profiles.
             */
            isDasherlessProfile_: {
                type: Boolean,
                value() {
                    return loadTimeData.getBoolean('isDasherlessProfile');
                },
            },
            // 
            /**
             * The current sync status, supplied by SyncBrowserProxy.
             */
            syncStatus: Object,
            /**
             * Authentication token provided by settings-lock-screen.
             */
            authToken_: {
                type: String,
                value: '',
            },
            /**
             * The currently selected profile icon URL. May be a data URL.
             */
            profileIconUrl_: String,
            /**
             * Whether the profile row is clickable. The behavior depends on the
             * platform.
             */
            isProfileActionable_: {
                type: Boolean,
                value() {
                    if (!isChromeOS) {
                        // Opens profile manager.
                        return true;
                    }
                    // Post-SplitSettings links out to account manager if it is available.
                    return loadTimeData.getBoolean('isAccountManagerEnabled');
                },
                readOnly: true,
            },
            /**
             * The current profile name.
             */
            profileName_: String,
            // 
        };
    }
    // 
    syncBrowserProxy_ = SyncBrowserProxyImpl.getInstance();
    connectedCallback() {
        super.connectedCallback();
        let useProfileNameAndIcon = true;
        // 
        if (loadTimeData.getBoolean('isAccountManagerEnabled')) {
            // If this is SplitSettings and we have the Google Account manager,
            // prefer the GAIA name and icon.
            useProfileNameAndIcon = false;
            this.addWebUiListener('accounts-changed', this.updateAccounts_.bind(this));
            this.updateAccounts_();
        }
        // 
        if (useProfileNameAndIcon) {
            ProfileInfoBrowserProxyImpl.getInstance().getProfileInfo().then(this.handleProfileInfo_.bind(this));
            this.addWebUiListener('profile-info-changed', this.handleProfileInfo_.bind(this));
        }
        this.syncBrowserProxy_.getSyncStatus().then(this.handleSyncStatus_.bind(this));
        this.addWebUiListener('sync-status-changed', this.handleSyncStatus_.bind(this));
        // 
    }
    // 
    /**
     * Handler for when the profile's icon and name is updated.
     */
    handleProfileInfo_(info) {
        this.profileName_ = info.name;
        /**
         * Extract first frame from image by creating a single frame PNG using
         * url as input if base64 encoded and potentially animated.
         */
        // 
        if (info.iconUrl.startsWith('data:image/png;base64')) {
            this.profileIconUrl_ = convertImageSequenceToPng([info.iconUrl]);
            return;
        }
        // 
        this.profileIconUrl_ = info.iconUrl;
    }
    // 
    async updateAccounts_() {
        const accounts = await AccountManagerBrowserProxyImpl.getInstance().getAccounts();
        // The user might not have any GAIA accounts (e.g. guest mode or Active
        // Directory). In these cases the profile row is hidden, so there's nothing
        // to do.
        if (accounts.length === 0) {
            return;
        }
        this.profileName_ = accounts[0].fullName;
        this.profileIconUrl_ = accounts[0].pic;
    }
    // 
    /**
     * Handler for when the sync state is pushed from the browser.
     */
    handleSyncStatus_(syncStatus) {
        // 
        this.syncStatus = syncStatus;
        // 
        // 
    }
    // 
    onProfileClick_() {
        // 
        if (loadTimeData.getBoolean('isAccountManagerEnabled')) {
            // Post-SplitSettings. The browser C++ code loads OS settings in a window.
            OpenWindowProxyImpl.getInstance().openUrl(loadTimeData.getString('osSettingsAccountsPageUrl'));
        }
        // 
        // 
    }
    // 
    onSyncClick_() {
        // Users can go to sync subpage regardless of sync status.
        Router.getInstance().navigateTo(routes.SYNC);
    }
    // 
    /**
     * Open URL for managing your Google Account.
     */
    openGoogleAccount_() {
        OpenWindowProxyImpl.getInstance().openUrl(loadTimeData.getString('googleAccountUrl'));
        chrome.metricsPrivate.recordUserAction('ManageGoogleAccount_Clicked');
    }
    /**
     * @return A CSS image-set for multiple scale factors.
     */
    getIconImageSet_(iconUrl) {
        return getImage(iconUrl);
    }
    isSyncing_() {
        return !!this.syncStatus &&
            this.syncStatus.signedInState === SignedInState.SYNCING;
    }
    // 
    // SettingsViewMixin implementation.
    getFocusConfig() {
        const map = new Map();
        if (routes.SYNC) {
            map.set(routes.SYNC.path, '#sync-setup');
        }
        // 
        return map;
    }
    // SettingsViewMixin implementation.
    getAssociatedControlFor(childViewId) {
        const ids = [
            'sync', 'syncControls',
            // 
        ];
        assert(ids.includes(childViewId));
        let triggerId = null;
        switch (childViewId) {
            case 'sync':
            case 'syncControls':
                triggerId = 'sync-setup';
                break;
            // 
        }
        assert(triggerId);
        const control = this.shadowRoot.querySelector(`#${triggerId}`);
        assert(control);
        return control;
    }
}
customElements.define(SettingsPeoplePageElement.is, SettingsPeoplePageElement);
