// Copyright 2025 The Chromium Authors
// Use of this source code is governed by a BSD-style license that can be
// found in the LICENSE file.
/**
 * @fileoverview
 * 'settings-account-card' component is a card that shows the user's account
 * name and picture with optional controls.
 * It's a slimmed down copy of
 * chrome/browser/resources/settings/people_page/people_page.ts
 */
import 'chrome://resources/cr_elements/cr_button/cr_button.js';
import 'chrome://resources/cr_elements/cr_icon_button/cr_icon_button.js';
import 'chrome://resources/cr_elements/cr_link_row/cr_link_row.js';
import 'chrome://resources/cr_elements/cr_shared_style.css.js';
import 'chrome://resources/cr_elements/cr_shared_vars.css.js';
import 'chrome://resources/cr_elements/icons.html.js';
import 'chrome://resources/cr_elements/policy/cr_policy_indicator.js';
import '../controls/settings_toggle_button.js';
// 
import '../icons.html.js';
import '../settings_shared.css.js';
import { ProfileInfoBrowserProxyImpl } from '/shared/settings/people_page/profile_info_browser_proxy.js';
import { SignedInState, SyncBrowserProxyImpl } from '/shared/settings/people_page/sync_browser_proxy.js';
import { WebUiListenerMixin } from 'chrome://resources/cr_elements/web_ui_listener_mixin.js';
import { getImage } from 'chrome://resources/js/icon.js';
import { OpenWindowProxyImpl } from 'chrome://resources/js/open_window_proxy.js';
import { PolymerElement } from 'chrome://resources/polymer/v3_0/polymer/polymer_bundled.min.js';
import { loadTimeData } from '../i18n_setup.js';
// clang-format off
// 
import { convertImageSequenceToPng } from 'chrome://resources/ash/common/cr_picture/png.js';
import { AccountManagerBrowserProxyImpl } from '../people_page/account_manager_browser_proxy.js';
// 
// clang-format on
import { getTemplate } from './account_card_element.html.js';
const SettingsAccountCardElementBase = WebUiListenerMixin(PolymerElement);
export class SettingsAccountCardElement extends SettingsAccountCardElementBase {
    static get is() {
        return 'settings-account-card';
    }
    static get template() {
        return getTemplate();
    }
    static get properties() {
        return {
            /**
             * Preferences state.
             */
            prefs: {
                type: Object,
                notify: true,
            },
            /**
             * This flag is used to conditionally show a set of new sign-in UIs to the
             * profiles that have been migrated to be consistent with the web
             * sign-ins.
             */
            signinAllowed_: {
                type: Boolean,
                value() {
                    return loadTimeData.getBoolean('signinAllowed');
                },
            },
            /**
             * The current sync status, supplied by SyncBrowserProxy.
             */
            syncStatus: Object,
            // 
            // 
            /**
             * The currently selected profile icon URL. May be a data URL.
             */
            profileIconUrl_: String,
            /**
             * The current profile name.
             */
            profileName_: String,
            /**
             * Whether the profile row is clickable. The behavior depends on the
             * platform.
             */
            isProfileActionable_: {
                type: Boolean,
                value: () => loadTimeData.getBoolean('isAccountManagerEnabled'),
                readOnly: true,
            },
            // 
        };
    }
    // 
    syncBrowserProxy_ = SyncBrowserProxyImpl.getInstance();
    connectedCallback() {
        super.connectedCallback();
        // 
        if (loadTimeData.getBoolean('isAccountManagerEnabled')) {
            // If this is SplitSettings and we have the Google Account manager,
            // prefer the GAIA name and icon.
            this.addWebUiListener('accounts-changed', this.updateAccounts_.bind(this));
            this.updateAccounts_();
        }
        else {
            ProfileInfoBrowserProxyImpl.getInstance().getProfileInfo().then(this.handleProfileInfo_.bind(this));
            this.addWebUiListener('profile-info-changed', this.handleProfileInfo_.bind(this));
        }
        // 
        this.syncBrowserProxy_.getSyncStatus().then(this.handleSyncStatus_.bind(this));
        this.addWebUiListener('sync-status-changed', this.handleSyncStatus_.bind(this));
        // 
    }
    // 
    /**
     * Handler for when the profile's icon and name is updated.
     */
    handleProfileInfo_(info) {
        this.profileName_ = info.name;
        /**
         * Extract first frame from image by creating a single frame PNG using
         * url as input if base64 encoded and potentially animated.
         */
        if (info.iconUrl.startsWith('data:image/png;base64')) {
            this.profileIconUrl_ = convertImageSequenceToPng([info.iconUrl]);
            return;
        }
        this.profileIconUrl_ = info.iconUrl;
    }
    async updateAccounts_() {
        const accounts = await AccountManagerBrowserProxyImpl.getInstance().getAccounts();
        // The user might not have any GAIA accounts (e.g. guest mode or Active
        // Directory). In these cases the profile row is hidden, so there's nothing
        // to do.
        if (accounts.length === 0) {
            return;
        }
        this.profileName_ = accounts[0].fullName;
        this.profileIconUrl_ = accounts[0].pic;
    }
    onProfileClick_() {
        if (loadTimeData.getBoolean('isAccountManagerEnabled')) {
            // Post-SplitSettings. The browser C++ code loads OS settings in a window.
            OpenWindowProxyImpl.getInstance().openUrl(loadTimeData.getString('osSettingsAccountsPageUrl'));
        }
    }
    // 
    /**
     * Handler for when the sync state is pushed from the browser.
     */
    handleSyncStatus_(syncStatus) {
        // 
        this.syncStatus = syncStatus;
        // 
        // 
    }
    // 
    /**
     * @return A CSS image-set for multiple scale factors.
     */
    getIconImageSet_(iconUrl) {
        return getImage(iconUrl);
    }
    isSyncing_() {
        return !!this.syncStatus &&
            this.syncStatus.signedInState === SignedInState.SYNCING;
    }
}
customElements.define(SettingsAccountCardElement.is, SettingsAccountCardElement);
