// Copyright 2024 The Chromium Authors
// Use of this source code is governed by a BSD-style license that can be
// found in the LICENSE file.
import '//resources/cr_elements/cr_icon_button/cr_icon_button.js';
import '//resources/cr_elements/cr_lazy_render/cr_lazy_render_lit.js';
import '//resources/cr_elements/cr_icon/cr_icon.js';
import '//resources/cr_elements/icons.html.js';
import '//resources/cr_elements/cr_action_menu/cr_action_menu.js';
import './language_menu.js';
import { WebUiListenerMixinLit } from '//resources/cr_elements/web_ui_listener_mixin_lit.js';
import { assert } from '//resources/js/assert.js';
import { loadTimeData } from '//resources/js/load_time_data.js';
import { CrLitElement } from '//resources/lit/v3_0/lit.rollup.js';
import { openMenu, spinnerDebounceTimeout, ToolbarEvent } from './common.js';
import { ReadAloudSettingsChange } from './metrics_browser_proxy.js';
import { ReadAnythingLogger } from './read_anything_logger.js';
// clang-format off
// 
// clang-format on
import { VoiceNotificationManager } from './read_aloud/voice_notification_manager.js';
import { areVoicesEqual, convertLangOrLocaleForVoicePackManager, isNatural, NotificationType } from './read_aloud/voice_language_conversions.js';
import { getCss } from './voice_selection_menu.css.js';
import { getHtml } from './voice_selection_menu.html.js';
const VoiceSelectionMenuElementBase = WebUiListenerMixinLit(CrLitElement);
export class VoiceSelectionMenuElement extends VoiceSelectionMenuElementBase {
    static get is() {
        return 'voice-selection-menu';
    }
    static get styles() {
        return getCss();
    }
    render() {
        return getHtml.bind(this)();
    }
    static get properties() {
        return {
            selectedVoice: { type: Object },
            availableVoices: { type: Array },
            enabledLangs: { type: Array },
            previewVoicePlaying: { type: Object },
            currentNotifications_: { type: Object },
            previewVoiceInitiated: { type: Object },
            localeToDisplayName: { type: Object },
            showLanguageMenuDialog_: { type: Boolean },
            downloadingMessages_: { type: Boolean },
            voiceGroups_: { type: Object },
        };
    }
    #selectedVoice_accessor_storage;
    get selectedVoice() { return this.#selectedVoice_accessor_storage; }
    set selectedVoice(value) { this.#selectedVoice_accessor_storage = value; }
    #localeToDisplayName_accessor_storage = {};
    get localeToDisplayName() { return this.#localeToDisplayName_accessor_storage; }
    set localeToDisplayName(value) { this.#localeToDisplayName_accessor_storage = value; }
    #previewVoicePlaying_accessor_storage = null;
    get previewVoicePlaying() { return this.#previewVoicePlaying_accessor_storage; }
    set previewVoicePlaying(value) { this.#previewVoicePlaying_accessor_storage = value; }
    #enabledLangs_accessor_storage = [];
    get enabledLangs() { return this.#enabledLangs_accessor_storage; }
    set enabledLangs(value) { this.#enabledLangs_accessor_storage = value; }
    #availableVoices_accessor_storage = [];
    get availableVoices() { return this.#availableVoices_accessor_storage; }
    set availableVoices(value) { this.#availableVoices_accessor_storage = value; }
    #currentNotifications__accessor_storage = {};
    // The current notifications that should be used in the voice menu.
    get currentNotifications_() { return this.#currentNotifications__accessor_storage; }
    set currentNotifications_(value) { this.#currentNotifications__accessor_storage = value; }
    #previewVoiceInitiated_accessor_storage = null;
    get previewVoiceInitiated() { return this.#previewVoiceInitiated_accessor_storage; }
    set previewVoiceInitiated(value) { this.#previewVoiceInitiated_accessor_storage = value; }
    errorMessages_ = [];
    #downloadingMessages__accessor_storage = [];
    get downloadingMessages_() { return this.#downloadingMessages__accessor_storage; }
    set downloadingMessages_(value) { this.#downloadingMessages__accessor_storage = value; }
    #voiceGroups__accessor_storage = [];
    get voiceGroups_() { return this.#voiceGroups__accessor_storage; }
    set voiceGroups_(value) { this.#voiceGroups__accessor_storage = value; }
    #showLanguageMenuDialog__accessor_storage = false;
    get showLanguageMenuDialog_() { return this.#showLanguageMenuDialog__accessor_storage; }
    set showLanguageMenuDialog_(value) { this.#showLanguageMenuDialog__accessor_storage = value; }
    spBodyPadding_ = Number.parseInt(window.getComputedStyle(document.body)
        .getPropertyValue('--sp-body-padding'), 10);
    logger_ = ReadAnythingLogger.getInstance();
    notificationManager_ = VoiceNotificationManager.getInstance();
    willUpdate(changedProperties) {
        super.willUpdate(changedProperties);
        if (changedProperties.has('previewVoicePlaying') &&
            (this.previewVoicePlaying !== this.previewVoiceInitiated)) {
            // When the preview stops, the voice is set to null in app.ts, so
            // we should update the preview voice to null here as well to clear the
            // voice.
            this.previewVoiceInitiated = this.previewVoicePlaying;
        }
        const changedPrivateProperties = changedProperties;
        if (changedProperties.has('selectedVoice') ||
            changedProperties.has('availableVoices') ||
            changedProperties.has('enabledLangs') ||
            changedPrivateProperties.has('previewVoiceInitiated') ||
            changedProperties.has('previewVoicePlaying') ||
            changedProperties.has('localeToDisplayName')) {
            this.voiceGroups_ = this.computeVoiceDropdown_();
        }
        if (changedPrivateProperties.has('currentNotifications_')) {
            this.errorMessages_ = this.computeErrorMessages_();
            this.downloadingMessages_ = this.computeDownloadingMessages_();
        }
    }
    notify(type, language) {
        if (!language) {
            return;
        }
        this.currentNotifications_ = {
            ...this.currentNotifications_,
            [language]: type,
        };
    }
    onVoiceSelectionMenuClick(targetElement) {
        this.notificationManager_.addListener(this);
        const menu = this.$.voiceSelectionMenu.get();
        openMenu(menu, targetElement, {
            minX: this.spBodyPadding_,
            maxX: document.body.clientWidth - this.spBodyPadding_,
        }, this.onMenuShown.bind(this));
    }
    onMenuShown() {
        this.fire(ToolbarEvent.VOICE_MENU_OPEN);
        const selectedItem = this.$.voiceSelectionMenu.get().querySelector('.item-hidden-false.check-mark');
        selectedItem?.scrollIntoViewIfNeeded();
    }
    voiceItemTabIndex_(groupIndex, voiceIndex) {
        return (groupIndex + voiceIndex) === 0 ? 0 : -1;
    }
    computeEnabledVoices_() {
        if (!this.availableVoices || !this.enabledLangs) {
            return [];
        }
        const enablesLangsLowerCase = new Set(this.enabledLangs.map(lang => lang.toLowerCase()));
        return this.availableVoices.filter(({ lang }) => enablesLangsLowerCase.has(lang.toLowerCase()));
    }
    getLangDisplayName(lang) {
        const langLower = lang.toLowerCase();
        return this.localeToDisplayName[langLower] || langLower;
    }
    computeVoiceDropdown_() {
        const enabledVoices = this.computeEnabledVoices_();
        if (!enabledVoices) {
            return [];
        }
        const languageToVoices = enabledVoices.reduce((languageToDropdownItems, voice) => {
            const dropdownItem = {
                title: this.getVoiceTitle_(voice),
                voice,
                id: this.stringToHtmlTestId_(voice.name),
                selected: areVoicesEqual(this.selectedVoice, voice),
                previewActuallyPlaying: areVoicesEqual(this.previewVoicePlaying, voice),
                previewInitiated: areVoicesEqual(this.previewVoiceInitiated, voice),
            };
            const lang = this.getLangDisplayName(voice.lang);
            if (languageToDropdownItems[lang]) {
                languageToDropdownItems[lang].push(dropdownItem);
            }
            else {
                languageToDropdownItems[lang] = [dropdownItem];
            }
            return languageToDropdownItems;
        }, {});
        for (const lang of Object.keys(languageToVoices)) {
            languageToVoices[lang].sort(voiceQualityRankComparator);
        }
        return Object.entries(languageToVoices).map(([language, voices,]) => ({ language, voices }));
    }
    getVoiceTitle_(voice) {
        let title = voice.name;
        // 
        return title;
    }
    // This ID does not ensure uniqueness and is just used for testing purposes.
    stringToHtmlTestId_(s) {
        return s.replace(/\s/g, '-').replace(/[()]/g, '');
    }
    onVoiceSelectClick_(e) {
        this.logger_.logSpeechSettingsChange(ReadAloudSettingsChange.VOICE_NAME_CHANGE);
        const selectedVoice = this.getVoiceItemForEvent_(e).voice;
        this.fire(ToolbarEvent.VOICE, { selectedVoice });
    }
    onVoicePreviewClick_(e) {
        // Because the preview button is layered onto the voice-selection button,
        // the onVoiceSelectClick_() listener is also subscribed to this event. This
        // line is to make sure that the voice-selection callback is not triggered.
        e.stopImmediatePropagation();
        const dropdownItem = this.getVoiceItemForEvent_(e);
        // Set a small timeout to ensure we're not showing the spinner too
        // frequently. If speech starts fairly quickly after the button is
        // pressed, there's no need for a spinner. This timeout should only be
        // set if the preview is starting, not when a preview is stopped.
        if (!dropdownItem.previewActuallyPlaying) {
            setTimeout(() => {
                this.previewVoiceInitiated = dropdownItem.voice;
            }, spinnerDebounceTimeout);
        }
        this.fire(ToolbarEvent.PLAY_PREVIEW, 
        // If preview is currently playing, we pass null to indicate the audio
        // should be paused.
        {
            previewVoice: dropdownItem.previewActuallyPlaying ? null : dropdownItem.voice,
        });
    }
    openLanguageMenu_() {
        this.showLanguageMenuDialog_ = true;
        this.fire(ToolbarEvent.LANGUAGE_MENU_OPEN);
    }
    onLanguageMenuClose_(event) {
        event.preventDefault();
        event.stopPropagation();
        this.showLanguageMenuDialog_ = false;
        this.fire(ToolbarEvent.LANGUAGE_MENU_CLOSE);
    }
    onClose_() {
        this.notificationManager_.removeListener(this);
        this.currentNotifications_ = {};
        this.fire(ToolbarEvent.VOICE_MENU_CLOSE);
    }
    shouldAllowPropagation_(e, currentElement) {
        // Always allow propagation for keys other than Tab.
        if (e.key !== 'Tab') {
            return true;
        }
        // If the shift key is not pressed with the tab, only allow propagation on
        // the language menu button.
        if (!e.shiftKey) {
            return currentElement.classList.contains('language-menu-button');
        }
        // In the case that shift is pressed, only allow propagation on the first
        // voice option.
        const targetIsVoiceOption = currentElement.classList.contains('dropdown-voice-selection-button');
        return targetIsVoiceOption &&
            Number.parseInt(currentElement.dataset['groupIndex']) === 0 &&
            Number.parseInt(currentElement.dataset['voiceIndex']) === 0;
    }
    onVoiceMenuKeyDown_(e) {
        const currentElement = e.target;
        assert(currentElement, 'no key target');
        // Allowing propagation on Tab closes the menu. We want to stop that
        // propagation unless we're tabbing forward on the last item or tabbing
        // backward on the first item.
        if (!this.shouldAllowPropagation_(e, currentElement)) {
            e.stopImmediatePropagation();
            return;
        }
        const targetIsVoiceOption = currentElement.classList.contains('dropdown-voice-selection-button');
        const targetIsPreviewButton = currentElement.id === 'preview-icon';
        // For voice options, only handle the right arrow - everything else is
        // default
        if (targetIsVoiceOption && !['ArrowRight'].includes(e.key)) {
            return;
        }
        // For a voice preview, handle up, down and left arrows
        if (targetIsPreviewButton &&
            !['ArrowLeft', 'ArrowUp', 'ArrowDown'].includes(e.key)) {
            return;
        }
        // When the menu first opens, the target is the whole menu.
        // In that case, use default behavior.
        if (!targetIsVoiceOption && !targetIsPreviewButton) {
            return;
        }
        e.preventDefault();
        if (targetIsVoiceOption) {
            // From a voice option, go to its preview button
            const visiblePreviewButton = currentElement.querySelector('#preview-icon');
            assert(visiblePreviewButton, 'can\'t find preview button');
            visiblePreviewButton.focus();
        }
        // This action is also handled by the menu itself
        // For left arrow, this takes us to the voice being previewed,
        // For up and down arrows this is combined with the default up/down
        // action, taking us to the next or previous voice.
        currentElement.parentElement.focus();
    }
    previewLabel_(previewPlaying) {
        if (previewPlaying) {
            return loadTimeData.getString('stopLabel');
        }
        else {
            return loadTimeData.getString('previewTooltip');
        }
    }
    hideSpinner_(voiceDropdown) {
        return !(voiceDropdown.previewInitiated &&
            !voiceDropdown.previewActuallyPlaying);
    }
    voiceLabel_(selected, voiceName) {
        const selectedPrefix = selected ? loadTimeData.getString('selected') : '';
        return selectedPrefix + ' ' +
            loadTimeData.getStringF('readingModeLanguageMenuItemLabel', voiceName);
    }
    shouldDisableButton_(voiceDropdown) {
        return (voiceDropdown.previewInitiated &&
            !voiceDropdown.previewActuallyPlaying);
    }
    previewIcon_(previewInitiated) {
        if (previewInitiated) {
            return 'read-anything-20:stop-circle';
        }
        else {
            return 'read-anything-20:play-circle';
        }
    }
    getVoiceItemForEvent_(e) {
        const groupIndex = Number.parseInt(e.currentTarget.dataset['groupIndex']);
        const voiceIndex = Number.parseInt(e.currentTarget.dataset['voiceIndex']);
        return this.voiceGroups_[groupIndex].voices[voiceIndex];
    }
    computeErrorMessages_() {
        const allocationErrors = this.computeMessages_(([_, notification]) => notification === NotificationType.NO_SPACE, 'readingModeVoiceMenuNoSpace');
        const noInternetErrors = this.computeMessages_(([_, notification]) => notification === NotificationType.NO_INTERNET, 'readingModeVoiceMenuNoInternet');
        return allocationErrors.concat(noInternetErrors);
    }
    computeDownloadingMessages_() {
        return this.computeMessages_(([_, notification]) => notification === NotificationType.DOWNLOADING, 'readingModeVoiceMenuDownloading');
    }
    computeMessages_(filterFn, message) {
        // We need to redeclare the type here otherwise the filterFn type
        // declaration doesn't work.
        const entries = Object.entries(this.currentNotifications_);
        return entries.filter(filterFn)
            .map(([lang, _]) => this.getDisplayNameForLocale(lang))
            .filter(possibleName => possibleName.length > 0)
            .map(displayName => loadTimeData.getStringF(message, displayName));
    }
    getDisplayNameForLocale(language) {
        const voicePackLang = convertLangOrLocaleForVoicePackManager(language);
        return voicePackLang ? chrome.readingMode.getDisplayNameForLocale(voicePackLang, voicePackLang) :
            '';
    }
}
function voiceQualityRankComparator(voice1, voice2) {
    if (isNatural(voice1.voice) && isNatural(voice2.voice)) {
        return 0;
    }
    if (!isNatural(voice1.voice) && !isNatural(voice2.voice)) {
        return 0;
    }
    // voice1 is a Natural voice and voice2 is not
    if (isNatural(voice1.voice)) {
        return -1;
    }
    // voice2 is a Natural voice and voice1 is not
    return 1;
}
customElements.define(VoiceSelectionMenuElement.is, VoiceSelectionMenuElement);
