"use strict";
import "../../ui/legacy/legacy.js";
import * as Common from "../../core/common/common.js";
import * as Host from "../../core/host/host.js";
import * as i18n from "../../core/i18n/i18n.js";
import * as Platform from "../../core/platform/platform.js";
import * as Root from "../../core/root/root.js";
import * as SDK from "../../core/sdk/sdk.js";
import * as Protocol from "../../generated/protocol.js";
import * as AiAssistanceModel from "../../models/ai_assistance/ai_assistance.js";
import * as Badges from "../../models/badges/badges.js";
import * as TextUtils from "../../models/text_utils/text_utils.js";
import * as Workspace from "../../models/workspace/workspace.js";
import * as Buttons from "../../ui/components/buttons/buttons.js";
import * as Snackbars from "../../ui/components/snackbars/snackbars.js";
import * as UI from "../../ui/legacy/legacy.js";
import * as Lit from "../../ui/lit/lit.js";
import * as VisualLogging from "../../ui/visual_logging/visual_logging.js";
import * as NetworkForward from "../network/forward/forward.js";
import * as NetworkPanel from "../network/network.js";
import * as TimelinePanel from "../timeline/timeline.js";
import aiAssistancePanelStyles from "./aiAssistancePanel.css.js";
import {
  ChatMessageEntity,
  ChatView,
  State as ChatViewState
} from "./components/ChatView.js";
import { ExploreWidget } from "./components/ExploreWidget.js";
import { MarkdownRendererWithCodeBlock } from "./components/MarkdownRendererWithCodeBlock.js";
import { PerformanceAgentMarkdownRenderer } from "./components/PerformanceAgentMarkdownRenderer.js";
import { isAiAssistancePatchingEnabled } from "./PatchWidget.js";
const { html } = Lit;
const AI_ASSISTANCE_SEND_FEEDBACK = "https://crbug.com/364805393";
const AI_ASSISTANCE_HELP = "https://developer.chrome.com/docs/devtools/ai-assistance";
const SCREENSHOT_QUALITY = 100;
const SHOW_LOADING_STATE_TIMEOUT = 100;
const JPEG_MIME_TYPE = "image/jpeg";
const UIStrings = {
  /**
   * @description AI assistance UI text creating a new chat.
   */
  newChat: "New chat",
  /**
   * @description AI assistance UI tooltip text for the help button.
   */
  help: "Help",
  /**
   * @description AI assistant UI tooltip text for the settings button (gear icon).
   */
  settings: "Settings",
  /**
   * @description AI assistant UI tooltip sending feedback.
   */
  sendFeedback: "Send feedback",
  /**
   * @description Announcement text for screen readers when a new chat is created.
   */
  newChatCreated: "New chat created",
  /**
   * @description Announcement text for screen readers when the chat is deleted.
   */
  chatDeleted: "Chat deleted",
  /**
   * @description AI assistance UI text creating selecting a history entry.
   */
  history: "History",
  /**
   * @description AI assistance UI text deleting the current chat session from local history.
   */
  deleteChat: "Delete local chat",
  /**
   * @description AI assistance UI text that deletes all local history entries.
   */
  clearChatHistory: "Clear local chats",
  /**
   *@description AI assistance UI text for the export conversation button.
   */
  exportConversation: "Export conversation",
  /**
   * @description AI assistance UI text explains that he user had no pas conversations.
   */
  noPastConversations: "No past conversations",
  /**
   * @description Placeholder text for an inactive text field. When active, it's used for the user's input to the GenAI assistance.
   */
  followTheSteps: "Follow the steps above to ask a question",
  /**
   * @description Disclaimer text right after the chat input.
   */
  inputDisclaimerForEmptyState: "This is an experimental AI feature and won't always get it right.",
  /**
   * @description The message shown in a toast when the response is copied to the clipboard.
   */
  responseCopiedToClipboard: "Response copied to clipboard"
};
const UIStringsNotTranslate = {
  /**
   * @description Announcement text for screen readers when the conversation starts.
   */
  answerLoading: "Answer loading",
  /**
   * @description Announcement text for screen readers when the answer comes.
   */
  answerReady: "Answer ready",
  /**
   * @description Placeholder text for the input shown when the conversation is blocked because a cross-origin context was selected.
   */
  crossOriginError: "To talk about data from another origin, start a new chat",
  /**
   * @description Placeholder text for the chat UI input.
   */
  inputPlaceholderForStyling: "Ask a question about the selected element",
  /**
   * @description Placeholder text for the chat UI input.
   */
  inputPlaceholderForNetwork: "Ask a question about the selected network request",
  /**
   * @description Placeholder text for the chat UI input.
   */
  inputPlaceholderForFile: "Ask a question about the selected file",
  /**
   * @description Placeholder text for the chat UI input.
   */
  inputPlaceholderForPerformanceWithNoRecording: "Record a performance trace and select an item to ask a question",
  /**
   * @description Placeholder text for the chat UI input when there is no context selected.
   */
  inputPlaceholderForStylingNoContext: "Select an element to ask a question",
  /**
   * @description Placeholder text for the chat UI input when there is no context selected.
   */
  inputPlaceholderForNetworkNoContext: "Select a network request to ask a question",
  /**
   * @description Placeholder text for the chat UI input when there is no context selected.
   */
  inputPlaceholderForFileNoContext: "Select a file to ask a question",
  /**
   * @description Placeholder text for the chat UI input.
   */
  inputPlaceholderForPerformanceTrace: "Ask a question about the selected performance trace",
  /**
   *@description Placeholder text for the chat UI input.
   */
  inputPlaceholderForPerformanceTraceNoContext: "Record or select a performance trace to ask a question",
  /**
   * @description Disclaimer text right after the chat input.
   */
  inputDisclaimerForStyling: "Chat messages and any data the inspected page can access via Web APIs are sent to Google and may be seen by human reviewers to improve this feature. This is an experimental AI feature and won\u2019t always get it right.",
  /**
   * @description Disclaimer text right after the chat input.
   */
  inputDisclaimerForStylingEnterpriseNoLogging: "Chat messages and any data the inspected page can access via Web APIs are sent to Google. The content you submit and that is generated by this feature will not be used to improve Google\u2019s AI models. This is an experimental AI feature and won\u2019t always get it right.",
  /**
   * @description Disclaimer text right after the chat input.
   */
  inputDisclaimerForNetwork: "Chat messages and the selected network request are sent to Google and may be seen by human reviewers to improve this feature. This is an experimental AI feature and won\u2019t always get it right.",
  /**
   * @description Disclaimer text right after the chat input.
   */
  inputDisclaimerForNetworkEnterpriseNoLogging: "Chat messages and the selected network request are sent to Google. The content you submit and that is generated by this feature will not be used to improve Google\u2019s AI models. This is an experimental AI feature and won\u2019t always get it right.",
  /**
   * @description Disclaimer text right after the chat input.
   */
  inputDisclaimerForFile: "Chat messages and the selected file are sent to Google and may be seen by human reviewers to improve this feature. This is an experimental AI feature and won't always get it right.",
  /**
   * @description Disclaimer text right after the chat input.
   */
  inputDisclaimerForFileEnterpriseNoLogging: "Chat messages and the selected file are sent to Google. The content you submit and that is generated by this feature will not be used to improve Google\u2019s AI models. This is an experimental AI feature and won\u2019t always get it right.",
  /**
   * @description Disclaimer text right after the chat input.
   */
  inputDisclaimerForPerformance: "Chat messages and trace data from your performance trace are sent to Google and may be seen by human reviewers to improve this feature. This is an experimental AI feature and won't always get it right.",
  /**
   * @description Disclaimer text right after the chat input.
   */
  inputDisclaimerForPerformanceEnterpriseNoLogging: "Chat messages and data from your performance trace are sent to Google. The content you submit and that is generated by this feature will not be used to improve Google\u2019s AI models. This is an experimental AI feature and won\u2019t always get it right.",
  /**
   * @description Message displayed in toast in case of any failures while taking a screenshot of the page.
   */
  screenshotFailureMessage: "Failed to take a screenshot. Please try again.",
  /**
   * @description Message displayed in toast in case of any failures while uploading an image file as input.
   */
  uploadImageFailureMessage: "Failed to upload image. Please try again."
};
const str_ = i18n.i18n.registerUIStrings("panels/ai_assistance/AiAssistancePanel.ts", UIStrings);
const i18nString = i18n.i18n.getLocalizedString.bind(void 0, str_);
const lockedString = i18n.i18n.lockedString;
function selectedElementFilter(maybeNode) {
  if (maybeNode) {
    return maybeNode.nodeType() === Node.ELEMENT_NODE ? maybeNode : null;
  }
  return null;
}
async function getEmptyStateSuggestions(context, conversation) {
  if (context) {
    const specialSuggestions = await context.getSuggestions();
    if (specialSuggestions) {
      return specialSuggestions;
    }
  }
  if (!conversation?.type || conversation.isReadOnly) {
    return [];
  }
  switch (conversation.type) {
    case AiAssistanceModel.AiHistoryStorage.ConversationType.STYLING:
      return [
        { title: "What can you help me with?", jslogContext: "styling-default" },
        { title: "Why isn\u2019t this element visible?", jslogContext: "styling-default" },
        { title: "How do I center this element?", jslogContext: "styling-default" }
      ];
    case AiAssistanceModel.AiHistoryStorage.ConversationType.FILE:
      return [
        { title: "What does this script do?", jslogContext: "file-default" },
        { title: "Is the script optimized for performance?", jslogContext: "file-default" },
        { title: "Does the script handle user input safely?", jslogContext: "file-default" }
      ];
    case AiAssistanceModel.AiHistoryStorage.ConversationType.NETWORK:
      return [
        { title: "Why is this network request taking so long?", jslogContext: "network-default" },
        { title: "Are there any security headers present?", jslogContext: "network-default" },
        { title: "Why is the request failing?", jslogContext: "network-default" }
      ];
    case AiAssistanceModel.AiHistoryStorage.ConversationType.PERFORMANCE: {
      return [
        { title: "What performance issues exist with my page?", jslogContext: "performance-default" }
      ];
    }
    default:
      Platform.assertNever(conversation.type, "Unknown conversation type");
  }
}
function getMarkdownRenderer(context, conversation) {
  if (context instanceof AiAssistanceModel.PerformanceAgent.PerformanceTraceContext) {
    if (!context.external) {
      const focus = context.getItem();
      return new PerformanceAgentMarkdownRenderer(
        focus.parsedTrace.data.Meta.mainFrameId,
        focus.lookupEvent.bind(focus)
      );
    }
  } else if (conversation?.type === AiAssistanceModel.AiHistoryStorage.ConversationType.PERFORMANCE) {
    return new PerformanceAgentMarkdownRenderer();
  }
  return new MarkdownRendererWithCodeBlock();
}
function toolbarView(input) {
  return html`
    <div class="toolbar-container" role="toolbar" jslog=${VisualLogging.toolbar()}>
      <devtools-toolbar class="freestyler-left-toolbar" role="presentation">
      ${input.showChatActions ? html`<devtools-button
          title=${i18nString(UIStrings.newChat)}
          aria-label=${i18nString(UIStrings.newChat)}
          .iconName=${"plus"}
          .jslogContext=${"freestyler.new-chat"}
          .variant=${Buttons.Button.Variant.TOOLBAR}
          @click=${input.onNewChatClick}></devtools-button>
        <div class="toolbar-divider"></div>
        <devtools-menu-button
          title=${i18nString(UIStrings.history)}
          aria-label=${i18nString(UIStrings.history)}
          .iconName=${"history"}
          .jslogContext=${"freestyler.history"}
          .populateMenuCall=${input.populateHistoryMenu}
        ></devtools-menu-button>` : Lit.nothing}
        ${input.showActiveConversationActions ? html`
          <devtools-button
              title=${i18nString(UIStrings.deleteChat)}
              aria-label=${i18nString(UIStrings.deleteChat)}
              .iconName=${"bin"}
              .jslogContext=${"freestyler.delete"}
              .variant=${Buttons.Button.Variant.TOOLBAR}
              @click=${input.onDeleteClick}>
          </devtools-button>
          <devtools-button
            title=${i18nString(UIStrings.exportConversation)}
            aria-label=${i18nString(UIStrings.exportConversation)}
            .iconName=${"download"}
            .disabled=${input.isLoading}
            .jslogContext=${"export-ai-conversation"}
            .variant=${Buttons.Button.Variant.TOOLBAR}
            @click=${input.onExportConversationClick}>
          </devtools-button>` : Lit.nothing}
      </devtools-toolbar>
      <devtools-toolbar class="freestyler-right-toolbar" role="presentation">
        <x-link
          class="toolbar-feedback-link devtools-link"
          title=${UIStrings.sendFeedback}
          href=${AI_ASSISTANCE_SEND_FEEDBACK}
          jslog=${VisualLogging.link().track({ click: true, keydown: "Enter|Space" }).context("freestyler.send-feedback")}
        >${UIStrings.sendFeedback}</x-link>
        <div class="toolbar-divider"></div>
        <devtools-button
          title=${i18nString(UIStrings.help)}
          aria-label=${i18nString(UIStrings.help)}
          .iconName=${"help"}
          .jslogContext=${"freestyler.help"}
          .variant=${Buttons.Button.Variant.TOOLBAR}
          @click=${input.onHelpClick}></devtools-button>
        <devtools-button
          title=${i18nString(UIStrings.settings)}
          aria-label=${i18nString(UIStrings.settings)}
          .iconName=${"gear"}
          .jslogContext=${"freestyler.settings"}
          .variant=${Buttons.Button.Variant.TOOLBAR}
          @click=${input.onSettingsClick}></devtools-button>
      </devtools-toolbar>
    </div>
  `;
}
function defaultView(input, output, target) {
  Lit.render(
    html`
      ${toolbarView(input)}
      <div class="ai-assistance-view-container">
        ${input.state !== ChatViewState.EXPLORE_VIEW ? html` <devtools-ai-chat-view
              .props=${input}
              ${Lit.Directives.ref((el) => {
      if (!el || !(el instanceof ChatView)) {
        return;
      }
      output.chatView = el;
    })}
            ></devtools-ai-chat-view>` : html`<devtools-widget
              class="explore"
              .widgetConfig=${UI.Widget.widgetConfig(ExploreWidget)}
            ></devtools-widget>`}
      </div>
    `,
    target
  );
}
function createNodeContext(node) {
  if (!node) {
    return null;
  }
  return new AiAssistanceModel.StylingAgent.NodeContext(node);
}
function createFileContext(file) {
  if (!file) {
    return null;
  }
  return new AiAssistanceModel.FileAgent.FileContext(file);
}
function createRequestContext(request) {
  if (!request) {
    return null;
  }
  const calculator = NetworkPanel.NetworkPanel.NetworkPanel.instance().networkLogView.timeCalculator();
  return new AiAssistanceModel.NetworkAgent.RequestContext(request, calculator);
}
function createPerformanceTraceContext(focus) {
  if (!focus) {
    return null;
  }
  return new AiAssistanceModel.PerformanceAgent.PerformanceTraceContext(focus);
}
function agentToConversationType(agent) {
  if (agent instanceof AiAssistanceModel.StylingAgent.StylingAgent) {
    return AiAssistanceModel.AiHistoryStorage.ConversationType.STYLING;
  }
  if (agent instanceof AiAssistanceModel.NetworkAgent.NetworkAgent) {
    return AiAssistanceModel.AiHistoryStorage.ConversationType.NETWORK;
  }
  if (agent instanceof AiAssistanceModel.FileAgent.FileAgent) {
    return AiAssistanceModel.AiHistoryStorage.ConversationType.FILE;
  }
  if (agent instanceof AiAssistanceModel.PerformanceAgent.PerformanceAgent) {
    return agent.getConversationType();
  }
  throw new Error("Provided agent does not have a corresponding conversation type");
}
let panelInstance;
export class AiAssistancePanel extends UI.Panel.Panel {
  constructor(view = defaultView, { aidaClient, aidaAvailability, syncInfo }) {
    super(AiAssistancePanel.panelName);
    this.view = view;
    this.registerRequiredCSS(aiAssistancePanelStyles);
    this.#aiAssistanceEnabledSetting = this.#getAiAssistanceEnabledSetting();
    this.#aidaClient = aidaClient;
    this.#aidaAvailability = aidaAvailability;
    this.#userInfo = {
      accountImage: syncInfo.accountImage,
      accountFullName: syncInfo.accountFullName
    };
    this.#conversationHandler = AiAssistanceModel.ConversationHandler.ConversationHandler.instance(
      { aidaClient: this.#aidaClient, aidaAvailability }
    );
    if (UI.ActionRegistry.ActionRegistry.instance().hasAction("elements.toggle-element-search")) {
      this.#toggleSearchElementAction = UI.ActionRegistry.ActionRegistry.instance().getAction("elements.toggle-element-search");
    }
    AiAssistanceModel.AiHistoryStorage.AiHistoryStorage.instance().addEventListener(
      AiAssistanceModel.AiHistoryStorage.Events.HISTORY_DELETED,
      this.#onHistoryDeleted,
      this
    );
  }
  static panelName = "freestyler";
  // NodeJS debugging does not have Elements panel, thus this action might not exist.
  #toggleSearchElementAction;
  #aidaClient;
  #viewOutput = {};
  #serverSideLoggingEnabled = isAiAssistanceServerSideLoggingEnabled();
  #aiAssistanceEnabledSetting;
  #changeManager = new AiAssistanceModel.ChangeManager.ChangeManager();
  #mutex = new Common.Mutex.Mutex();
  #conversationAgent;
  #conversation;
  #selectedFile = null;
  #selectedElement = null;
  #selectedPerformanceTrace = null;
  #selectedRequest = null;
  // Messages displayed in the `ChatView` component.
  #messages = [];
  // Indicates whether the new conversation context is blocked due to cross-origin restrictions.
  // This happens when the conversation's context has a different
  // origin than the selected context.
  #blockedByCrossOrigin = false;
  // Whether the UI should show loading or not.
  #isLoading = false;
  // Selected conversation context. The reason we keep this as a
  // state field rather than using `#getConversationContext` is that,
  // there is a case where the context differs from the selectedElement (or other selected context type).
  // Specifically, it allows restoring the previous context when a new selection is cross-origin.
  // See `#onContextSelectionChanged` for details.
  #selectedContext = null;
  // Stores the availability status of the `AidaClient` and the reason for unavailability, if any.
  #aidaAvailability;
  // Info of the currently logged in user.
  #userInfo;
  #imageInput;
  // Used to disable send button when there is not text input.
  #isTextInputEmpty = true;
  #timelinePanelInstance = null;
  #conversationHandler;
  #runAbortController = new AbortController();
  #getChatUiState() {
    const blockedByAge = Root.Runtime.hostConfig.aidaAvailability?.blockedByAge === true;
    if (this.#aidaAvailability !== Host.AidaClient.AidaAccessPreconditions.AVAILABLE) {
      return ChatViewState.CHAT_VIEW;
    }
    if (!this.#aiAssistanceEnabledSetting?.getIfNotDisabled() || blockedByAge) {
      return ChatViewState.CONSENT_VIEW;
    }
    if (this.#conversation?.type) {
      return ChatViewState.CHAT_VIEW;
    }
    return ChatViewState.EXPLORE_VIEW;
  }
  #getAiAssistanceEnabledSetting() {
    try {
      return Common.Settings.moduleSetting("ai-assistance-enabled");
    } catch {
      return;
    }
  }
  static async instance(opts = { forceNew: null }) {
    const { forceNew } = opts;
    if (!panelInstance || forceNew) {
      const aidaClient = new Host.AidaClient.AidaClient();
      const syncInfoPromise = new Promise(
        (resolve) => Host.InspectorFrontendHost.InspectorFrontendHostInstance.getSyncInformation(resolve)
      );
      const [aidaAvailability, syncInfo] = await Promise.all([Host.AidaClient.AidaClient.checkAccessPreconditions(), syncInfoPromise]);
      panelInstance = new AiAssistancePanel(defaultView, { aidaClient, aidaAvailability, syncInfo });
    }
    return panelInstance;
  }
  /**
   * Called when the TimelinePanel instance changes. We use this to listen to
   * the status of if the user is viewing a trace or not, and update the
   * placeholder text in the panel accordingly. We do this because if the user
   * has an active trace, we show different text than if they are viewing
   * the performance panel but have no trace imported.
   */
  #bindTimelineTraceListener() {
    const timelinePanel = UI.Context.Context.instance().flavor(TimelinePanel.TimelinePanel.TimelinePanel);
    if (timelinePanel === this.#timelinePanelInstance) {
      return;
    }
    this.#timelinePanelInstance?.removeEventListener(
      TimelinePanel.TimelinePanel.Events.IS_VIEWING_TRACE,
      this.requestUpdate,
      this
    );
    this.#timelinePanelInstance = timelinePanel;
    if (this.#timelinePanelInstance) {
      this.#timelinePanelInstance.addEventListener(
        TimelinePanel.TimelinePanel.Events.IS_VIEWING_TRACE,
        this.requestUpdate,
        this
      );
    }
  }
  // We select the default agent based on the open panels if
  // there isn't any active conversation.
  #selectDefaultAgentIfNeeded() {
    if (this.#conversationAgent && this.#conversation && !this.#conversation.isEmpty || this.#isLoading) {
      return;
    }
    const { hostConfig } = Root.Runtime;
    const viewManager = UI.ViewManager.ViewManager.instance();
    const isElementsPanelVisible = viewManager.isViewVisible("elements");
    const isNetworkPanelVisible = viewManager.isViewVisible("network");
    const isSourcesPanelVisible = viewManager.isViewVisible("sources");
    const isPerformancePanelVisible = viewManager.isViewVisible("timeline");
    let targetConversationType = void 0;
    if (isElementsPanelVisible && hostConfig.devToolsFreestyler?.enabled) {
      targetConversationType = AiAssistanceModel.AiHistoryStorage.ConversationType.STYLING;
    } else if (isNetworkPanelVisible && hostConfig.devToolsAiAssistanceNetworkAgent?.enabled) {
      targetConversationType = AiAssistanceModel.AiHistoryStorage.ConversationType.NETWORK;
    } else if (isSourcesPanelVisible && hostConfig.devToolsAiAssistanceFileAgent?.enabled) {
      targetConversationType = AiAssistanceModel.AiHistoryStorage.ConversationType.FILE;
    } else if (isPerformancePanelVisible && hostConfig.devToolsAiAssistancePerformanceAgent?.enabled) {
      targetConversationType = AiAssistanceModel.AiHistoryStorage.ConversationType.PERFORMANCE;
    }
    if (this.#conversation?.type === targetConversationType) {
      return;
    }
    const agent = targetConversationType ? this.#conversationHandler.createAgent(targetConversationType, this.#changeManager) : void 0;
    this.#updateConversationState({ agent });
  }
  #updateConversationState(opts) {
    if (this.#conversationAgent !== opts?.agent) {
      this.#cancel();
      this.#messages = [];
      this.#isLoading = false;
      this.#conversation?.archiveConversation();
      this.#conversationAgent = opts?.agent;
      if (opts?.agent) {
        this.#conversation = new AiAssistanceModel.AiHistoryStorage.Conversation(
          agentToConversationType(opts?.agent),
          [],
          opts?.agent.id,
          false
        );
      }
    }
    if (!opts?.agent) {
      this.#conversation = void 0;
      this.#messages = [];
      if (opts?.conversation) {
        this.#conversation = opts?.conversation;
      }
    }
    if (!this.#conversationAgent && !this.#conversation) {
      this.#selectDefaultAgentIfNeeded();
    }
    this.#onContextSelectionChanged();
    this.requestUpdate();
  }
  wasShown() {
    super.wasShown();
    this.#viewOutput.chatView?.restoreScrollPosition();
    this.#viewOutput.chatView?.focusTextInput();
    void this.#handleAidaAvailabilityChange();
    this.#selectedElement = createNodeContext(selectedElementFilter(UI.Context.Context.instance().flavor(SDK.DOMModel.DOMNode)));
    this.#selectedRequest = createRequestContext(UI.Context.Context.instance().flavor(SDK.NetworkRequest.NetworkRequest));
    this.#selectedPerformanceTrace = createPerformanceTraceContext(UI.Context.Context.instance().flavor(AiAssistanceModel.AIContext.AgentFocus));
    this.#selectedFile = createFileContext(UI.Context.Context.instance().flavor(Workspace.UISourceCode.UISourceCode));
    this.#updateConversationState({ agent: this.#conversationAgent });
    this.#aiAssistanceEnabledSetting?.addChangeListener(this.requestUpdate, this);
    Host.AidaClient.HostConfigTracker.instance().addEventListener(
      Host.AidaClient.Events.AIDA_AVAILABILITY_CHANGED,
      this.#handleAidaAvailabilityChange
    );
    this.#toggleSearchElementAction?.addEventListener(UI.ActionRegistration.Events.TOGGLED, this.requestUpdate, this);
    UI.Context.Context.instance().addFlavorChangeListener(SDK.DOMModel.DOMNode, this.#handleDOMNodeFlavorChange);
    UI.Context.Context.instance().addFlavorChangeListener(
      SDK.NetworkRequest.NetworkRequest,
      this.#handleNetworkRequestFlavorChange
    );
    UI.Context.Context.instance().addFlavorChangeListener(
      AiAssistanceModel.AIContext.AgentFocus,
      this.#handlePerformanceTraceFlavorChange
    );
    UI.Context.Context.instance().addFlavorChangeListener(
      Workspace.UISourceCode.UISourceCode,
      this.#handleUISourceCodeFlavorChange
    );
    UI.ViewManager.ViewManager.instance().addEventListener(
      UI.ViewManager.Events.VIEW_VISIBILITY_CHANGED,
      this.#selectDefaultAgentIfNeeded,
      this
    );
    SDK.TargetManager.TargetManager.instance().addModelListener(
      SDK.DOMModel.DOMModel,
      SDK.DOMModel.Events.AttrModified,
      this.#handleDOMNodeAttrChange,
      this
    );
    SDK.TargetManager.TargetManager.instance().addModelListener(
      SDK.DOMModel.DOMModel,
      SDK.DOMModel.Events.AttrRemoved,
      this.#handleDOMNodeAttrChange,
      this
    );
    SDK.TargetManager.TargetManager.instance().addModelListener(
      SDK.ResourceTreeModel.ResourceTreeModel,
      SDK.ResourceTreeModel.Events.PrimaryPageChanged,
      this.#onPrimaryPageChanged,
      this
    );
    UI.Context.Context.instance().addFlavorChangeListener(
      TimelinePanel.TimelinePanel.TimelinePanel,
      this.#bindTimelineTraceListener,
      this
    );
    this.#bindTimelineTraceListener();
    this.#selectDefaultAgentIfNeeded();
    Host.userMetrics.actionTaken(Host.UserMetrics.Action.AiAssistancePanelOpened);
  }
  willHide() {
    super.willHide();
    this.#aiAssistanceEnabledSetting?.removeChangeListener(this.requestUpdate, this);
    Host.AidaClient.HostConfigTracker.instance().removeEventListener(
      Host.AidaClient.Events.AIDA_AVAILABILITY_CHANGED,
      this.#handleAidaAvailabilityChange
    );
    this.#toggleSearchElementAction?.removeEventListener(
      UI.ActionRegistration.Events.TOGGLED,
      this.requestUpdate,
      this
    );
    UI.Context.Context.instance().removeFlavorChangeListener(SDK.DOMModel.DOMNode, this.#handleDOMNodeFlavorChange);
    UI.Context.Context.instance().removeFlavorChangeListener(
      SDK.NetworkRequest.NetworkRequest,
      this.#handleNetworkRequestFlavorChange
    );
    UI.Context.Context.instance().removeFlavorChangeListener(
      AiAssistanceModel.AIContext.AgentFocus,
      this.#handlePerformanceTraceFlavorChange
    );
    UI.Context.Context.instance().removeFlavorChangeListener(
      Workspace.UISourceCode.UISourceCode,
      this.#handleUISourceCodeFlavorChange
    );
    UI.ViewManager.ViewManager.instance().removeEventListener(
      UI.ViewManager.Events.VIEW_VISIBILITY_CHANGED,
      this.#selectDefaultAgentIfNeeded,
      this
    );
    UI.Context.Context.instance().removeFlavorChangeListener(
      TimelinePanel.TimelinePanel.TimelinePanel,
      this.#bindTimelineTraceListener,
      this
    );
    SDK.TargetManager.TargetManager.instance().removeModelListener(
      SDK.DOMModel.DOMModel,
      SDK.DOMModel.Events.AttrModified,
      this.#handleDOMNodeAttrChange,
      this
    );
    SDK.TargetManager.TargetManager.instance().removeModelListener(
      SDK.DOMModel.DOMModel,
      SDK.DOMModel.Events.AttrRemoved,
      this.#handleDOMNodeAttrChange,
      this
    );
    SDK.TargetManager.TargetManager.instance().removeModelListener(
      SDK.ResourceTreeModel.ResourceTreeModel,
      SDK.ResourceTreeModel.Events.PrimaryPageChanged,
      this.#onPrimaryPageChanged,
      this
    );
    if (this.#timelinePanelInstance) {
      this.#timelinePanelInstance.removeEventListener(
        TimelinePanel.TimelinePanel.Events.IS_VIEWING_TRACE,
        this.requestUpdate,
        this
      );
      this.#timelinePanelInstance = null;
    }
  }
  #handleAidaAvailabilityChange = async () => {
    const currentAidaAvailability = await Host.AidaClient.AidaClient.checkAccessPreconditions();
    if (currentAidaAvailability !== this.#aidaAvailability) {
      this.#aidaAvailability = currentAidaAvailability;
      const syncInfo = await new Promise(
        (resolve) => Host.InspectorFrontendHost.InspectorFrontendHostInstance.getSyncInformation(resolve)
      );
      this.#userInfo = {
        accountImage: syncInfo.accountImage,
        accountFullName: syncInfo.accountFullName
      };
      this.requestUpdate();
    }
  };
  #handleDOMNodeFlavorChange = (ev) => {
    if (this.#selectedElement?.getItem() === ev.data) {
      return;
    }
    this.#selectedElement = createNodeContext(selectedElementFilter(ev.data));
    this.#updateConversationState({ agent: this.#conversationAgent });
  };
  #handleDOMNodeAttrChange = (ev) => {
    if (this.#selectedElement?.getItem() === ev.data.node) {
      if (ev.data.name === "class" || ev.data.name === "id") {
        this.requestUpdate();
      }
    }
  };
  #handleNetworkRequestFlavorChange = (ev) => {
    if (this.#selectedRequest?.getItem() === ev.data) {
      return;
    }
    if (Boolean(ev.data)) {
      const calculator = NetworkPanel.NetworkPanel.NetworkPanel.instance().networkLogView.timeCalculator();
      this.#selectedRequest = new AiAssistanceModel.NetworkAgent.RequestContext(ev.data, calculator);
    } else {
      this.#selectedRequest = null;
    }
    this.#updateConversationState({ agent: this.#conversationAgent });
  };
  #handlePerformanceTraceFlavorChange = (ev) => {
    if (this.#selectedPerformanceTrace?.getItem() === ev.data) {
      return;
    }
    this.#selectedPerformanceTrace = Boolean(ev.data) ? new AiAssistanceModel.PerformanceAgent.PerformanceTraceContext(ev.data) : null;
    this.#updateConversationState({ agent: this.#conversationAgent });
  };
  #handleUISourceCodeFlavorChange = (ev) => {
    const newFile = ev.data;
    if (!newFile) {
      return;
    }
    if (this.#selectedFile?.getItem() === newFile) {
      return;
    }
    this.#selectedFile = new AiAssistanceModel.FileAgent.FileContext(ev.data);
    this.#updateConversationState({ agent: this.#conversationAgent });
  };
  #onPrimaryPageChanged() {
    if (!this.#imageInput) {
      return;
    }
    this.#imageInput = void 0;
    this.requestUpdate();
  }
  #getChangeSummary() {
    if (!isAiAssistancePatchingEnabled() || !this.#conversationAgent || this.#conversation?.isReadOnly) {
      return;
    }
    return this.#changeManager.formatChangesForPatching(
      this.#conversationAgent.id,
      /* includeSourceLocation= */
      true
    );
  }
  async performUpdate() {
    const emptyStateSuggestions = await getEmptyStateSuggestions(this.#selectedContext, this.#conversation);
    const markdownRenderer = getMarkdownRenderer(this.#selectedContext, this.#conversation);
    this.view(
      {
        state: this.#getChatUiState(),
        blockedByCrossOrigin: this.#blockedByCrossOrigin,
        aidaAvailability: this.#aidaAvailability,
        isLoading: this.#isLoading,
        messages: this.#messages,
        selectedContext: this.#selectedContext,
        conversationType: this.#conversation?.type,
        isReadOnly: this.#conversation?.isReadOnly ?? false,
        changeSummary: this.#getChangeSummary(),
        inspectElementToggled: this.#toggleSearchElementAction?.toggled() ?? false,
        userInfo: this.#userInfo,
        canShowFeedbackForm: this.#serverSideLoggingEnabled,
        multimodalInputEnabled: isAiAssistanceMultimodalInputEnabled() && this.#conversation?.type === AiAssistanceModel.AiHistoryStorage.ConversationType.STYLING,
        imageInput: this.#imageInput,
        showChatActions: this.#shouldShowChatActions(),
        showActiveConversationActions: Boolean(this.#conversation && !this.#conversation.isEmpty),
        isTextInputDisabled: this.#isTextInputDisabled(),
        emptyStateSuggestions,
        inputPlaceholder: this.#getChatInputPlaceholder(),
        disclaimerText: this.#getDisclaimerText(),
        isTextInputEmpty: this.#isTextInputEmpty,
        changeManager: this.#changeManager,
        uploadImageInputEnabled: isAiAssistanceMultimodalUploadInputEnabled() && this.#conversation?.type === AiAssistanceModel.AiHistoryStorage.ConversationType.STYLING,
        markdownRenderer,
        onNewChatClick: this.#handleNewChatRequest.bind(this),
        populateHistoryMenu: this.#populateHistoryMenu.bind(this),
        onDeleteClick: this.#onDeleteClicked.bind(this),
        onExportConversationClick: this.#onExportConversationClick.bind(this),
        onHelpClick: () => {
          UI.UIUtils.openInNewTab(AI_ASSISTANCE_HELP);
        },
        onSettingsClick: () => {
          void UI.ViewManager.ViewManager.instance().showView("chrome-ai");
        },
        onTextSubmit: async (text, imageInput, multimodalInputType) => {
          this.#imageInput = void 0;
          this.#isTextInputEmpty = true;
          Host.userMetrics.actionTaken(Host.UserMetrics.Action.AiAssistanceQuerySubmitted);
          await this.#startConversation(text, imageInput, multimodalInputType);
        },
        onInspectElementClick: this.#handleSelectElementClick.bind(this),
        onFeedbackSubmit: this.#handleFeedbackSubmit.bind(this),
        onCancelClick: this.#cancel.bind(this),
        onContextClick: this.#handleContextClick.bind(this),
        onNewConversation: this.#handleNewChatRequest.bind(this),
        onTakeScreenshot: isAiAssistanceMultimodalInputEnabled() ? this.#handleTakeScreenshot.bind(this) : void 0,
        onRemoveImageInput: isAiAssistanceMultimodalInputEnabled() ? this.#handleRemoveImageInput.bind(this) : void 0,
        onCopyResponseClick: this.#onCopyResponseClick.bind(this),
        onTextInputChange: this.#handleTextInputChange.bind(this),
        onLoadImage: isAiAssistanceMultimodalUploadInputEnabled() ? this.#handleLoadImage.bind(this) : void 0
      },
      this.#viewOutput,
      this.contentElement
    );
  }
  #onCopyResponseClick(message) {
    const markdown = getResponseMarkdown(message);
    if (markdown) {
      Host.InspectorFrontendHost.InspectorFrontendHostInstance.copyText(markdown);
      Snackbars.Snackbar.Snackbar.show({
        message: i18nString(UIStrings.responseCopiedToClipboard)
      });
    }
  }
  #handleSelectElementClick() {
    UI.Context.Context.instance().setFlavor(
      Common.ReturnToPanel.ReturnToPanelFlavor,
      new Common.ReturnToPanel.ReturnToPanelFlavor(this.panelName)
    );
    void this.#toggleSearchElementAction?.execute();
  }
  #isTextInputDisabled() {
    const aiAssistanceSetting = this.#aiAssistanceEnabledSetting?.getIfNotDisabled();
    const isBlockedByAge = Root.Runtime.hostConfig.aidaAvailability?.blockedByAge === true;
    if (!aiAssistanceSetting || isBlockedByAge) {
      return true;
    }
    const isAidaAvailable = this.#aidaAvailability === Host.AidaClient.AidaAccessPreconditions.AVAILABLE;
    if (!isAidaAvailable) {
      return true;
    }
    if (this.#blockedByCrossOrigin) {
      return true;
    }
    if (!this.#conversation || !this.#selectedContext) {
      return true;
    }
    return false;
  }
  #shouldShowChatActions() {
    const aiAssistanceSetting = this.#aiAssistanceEnabledSetting?.getIfNotDisabled();
    const isBlockedByAge = Root.Runtime.hostConfig.aidaAvailability?.blockedByAge === true;
    if (!aiAssistanceSetting || isBlockedByAge) {
      return false;
    }
    if (this.#aidaAvailability === Host.AidaClient.AidaAccessPreconditions.NO_ACCOUNT_EMAIL || this.#aidaAvailability === Host.AidaClient.AidaAccessPreconditions.SYNC_IS_PAUSED) {
      return false;
    }
    return true;
  }
  #getChatInputPlaceholder() {
    const state = this.#getChatUiState();
    if (state === ChatViewState.CONSENT_VIEW || !this.#conversation) {
      return i18nString(UIStrings.followTheSteps);
    }
    if (this.#blockedByCrossOrigin) {
      return lockedString(UIStringsNotTranslate.crossOriginError);
    }
    switch (this.#conversation.type) {
      case AiAssistanceModel.AiHistoryStorage.ConversationType.STYLING:
        return this.#selectedContext ? lockedString(UIStringsNotTranslate.inputPlaceholderForStyling) : lockedString(UIStringsNotTranslate.inputPlaceholderForStylingNoContext);
      case AiAssistanceModel.AiHistoryStorage.ConversationType.FILE:
        return this.#selectedContext ? lockedString(UIStringsNotTranslate.inputPlaceholderForFile) : lockedString(UIStringsNotTranslate.inputPlaceholderForFileNoContext);
      case AiAssistanceModel.AiHistoryStorage.ConversationType.NETWORK:
        return this.#selectedContext ? lockedString(UIStringsNotTranslate.inputPlaceholderForNetwork) : lockedString(UIStringsNotTranslate.inputPlaceholderForNetworkNoContext);
      case AiAssistanceModel.AiHistoryStorage.ConversationType.PERFORMANCE: {
        const perfPanel = UI.Context.Context.instance().flavor(TimelinePanel.TimelinePanel.TimelinePanel);
        if (perfPanel?.hasActiveTrace()) {
          return this.#selectedContext ? lockedString(UIStringsNotTranslate.inputPlaceholderForPerformanceTrace) : lockedString(UIStringsNotTranslate.inputPlaceholderForPerformanceTraceNoContext);
        }
        return lockedString(UIStringsNotTranslate.inputPlaceholderForPerformanceWithNoRecording);
      }
    }
  }
  #getDisclaimerText() {
    const state = this.#getChatUiState();
    if (state === ChatViewState.CONSENT_VIEW || !this.#conversation || this.#conversation.isReadOnly) {
      return i18nString(UIStrings.inputDisclaimerForEmptyState);
    }
    const noLogging = Root.Runtime.hostConfig.aidaAvailability?.enterprisePolicyValue === Root.Runtime.GenAiEnterprisePolicyValue.ALLOW_WITHOUT_LOGGING;
    switch (this.#conversation.type) {
      case AiAssistanceModel.AiHistoryStorage.ConversationType.STYLING:
        if (noLogging) {
          return lockedString(UIStringsNotTranslate.inputDisclaimerForStylingEnterpriseNoLogging);
        }
        return lockedString(UIStringsNotTranslate.inputDisclaimerForStyling);
      case AiAssistanceModel.AiHistoryStorage.ConversationType.FILE:
        if (noLogging) {
          return lockedString(UIStringsNotTranslate.inputDisclaimerForFileEnterpriseNoLogging);
        }
        return lockedString(UIStringsNotTranslate.inputDisclaimerForFile);
      case AiAssistanceModel.AiHistoryStorage.ConversationType.NETWORK:
        if (noLogging) {
          return lockedString(UIStringsNotTranslate.inputDisclaimerForNetworkEnterpriseNoLogging);
        }
        return lockedString(UIStringsNotTranslate.inputDisclaimerForNetwork);
      // It is deliberate that both Performance agents use the same disclaimer
      // text and this has been approved by Privacy.
      case AiAssistanceModel.AiHistoryStorage.ConversationType.PERFORMANCE:
        if (noLogging) {
          return lockedString(UIStringsNotTranslate.inputDisclaimerForPerformanceEnterpriseNoLogging);
        }
        return lockedString(UIStringsNotTranslate.inputDisclaimerForPerformance);
    }
  }
  #handleFeedbackSubmit(rpcId, rating, feedback) {
    void this.#aidaClient.registerClientEvent({
      corresponding_aida_rpc_global_id: rpcId,
      disable_user_content_logging: !this.#serverSideLoggingEnabled,
      do_conversation_client_event: {
        user_feedback: {
          sentiment: rating,
          user_input: {
            comment: feedback
          }
        }
      }
    });
  }
  #handleContextClick() {
    const context = this.#selectedContext;
    if (context instanceof AiAssistanceModel.NetworkAgent.RequestContext) {
      const requestLocation = NetworkForward.UIRequestLocation.UIRequestLocation.tab(
        context.getItem(),
        NetworkForward.UIRequestLocation.UIRequestTabs.HEADERS_COMPONENT
      );
      return Common.Revealer.reveal(requestLocation);
    }
    if (context instanceof AiAssistanceModel.FileAgent.FileContext) {
      return Common.Revealer.reveal(context.getItem().uiLocation(0, 0));
    }
    if (context instanceof AiAssistanceModel.PerformanceAgent.PerformanceTraceContext) {
      const focus = context.getItem();
      if (focus.callTree) {
        const event = focus.callTree.selectedNode?.event ?? focus.callTree.rootNode.event;
        const revealable = new SDK.TraceObject.RevealableEvent(event);
        return Common.Revealer.reveal(revealable);
      }
      if (focus.insight) {
        return Common.Revealer.reveal(focus.insight);
      }
    }
  }
  #canExecuteQuery() {
    const isBrandedBuild = Boolean(Root.Runtime.hostConfig.aidaAvailability?.enabled);
    const isBlockedByAge = Boolean(Root.Runtime.hostConfig.aidaAvailability?.blockedByAge);
    const isAidaAvailable = Boolean(this.#aidaAvailability === Host.AidaClient.AidaAccessPreconditions.AVAILABLE);
    const isUserOptedIn = Boolean(this.#aiAssistanceEnabledSetting?.getIfNotDisabled());
    return isBrandedBuild && isAidaAvailable && isUserOptedIn && !isBlockedByAge;
  }
  async handleAction(actionId, opts) {
    if (this.#isLoading && !opts?.["prompt"]) {
      this.#viewOutput.chatView?.focusTextInput();
      return;
    }
    let targetConversationType;
    switch (actionId) {
      case "freestyler.elements-floating-button": {
        Host.userMetrics.actionTaken(Host.UserMetrics.Action.AiAssistanceOpenedFromElementsPanelFloatingButton);
        targetConversationType = AiAssistanceModel.AiHistoryStorage.ConversationType.STYLING;
        break;
      }
      case "freestyler.element-panel-context": {
        Host.userMetrics.actionTaken(Host.UserMetrics.Action.AiAssistanceOpenedFromElementsPanel);
        targetConversationType = AiAssistanceModel.AiHistoryStorage.ConversationType.STYLING;
        break;
      }
      case "drjones.network-floating-button": {
        Host.userMetrics.actionTaken(Host.UserMetrics.Action.AiAssistanceOpenedFromNetworkPanelFloatingButton);
        targetConversationType = AiAssistanceModel.AiHistoryStorage.ConversationType.NETWORK;
        break;
      }
      case "drjones.network-panel-context": {
        Host.userMetrics.actionTaken(Host.UserMetrics.Action.AiAssistanceOpenedFromNetworkPanel);
        targetConversationType = AiAssistanceModel.AiHistoryStorage.ConversationType.NETWORK;
        break;
      }
      case "drjones.performance-panel-context": {
        Host.userMetrics.actionTaken(Host.UserMetrics.Action.AiAssistanceOpenedFromPerformancePanelCallTree);
        targetConversationType = AiAssistanceModel.AiHistoryStorage.ConversationType.PERFORMANCE;
        break;
      }
      case "drjones.sources-floating-button": {
        Host.userMetrics.actionTaken(Host.UserMetrics.Action.AiAssistanceOpenedFromSourcesPanelFloatingButton);
        targetConversationType = AiAssistanceModel.AiHistoryStorage.ConversationType.FILE;
        break;
      }
      case "drjones.sources-panel-context": {
        Host.userMetrics.actionTaken(Host.UserMetrics.Action.AiAssistanceOpenedFromSourcesPanel);
        targetConversationType = AiAssistanceModel.AiHistoryStorage.ConversationType.FILE;
        break;
      }
    }
    if (!targetConversationType) {
      return;
    }
    let agent = this.#conversationAgent;
    if (!this.#conversation || !this.#conversationAgent || this.#conversation.type !== targetConversationType || this.#conversation?.isEmpty) {
      agent = this.#conversationHandler.createAgent(targetConversationType, this.#changeManager);
    }
    this.#updateConversationState({ agent });
    const predefinedPrompt = opts?.["prompt"];
    if (predefinedPrompt && typeof predefinedPrompt === "string") {
      if (!this.#canExecuteQuery()) {
        return;
      }
      this.#imageInput = void 0;
      this.#isTextInputEmpty = true;
      Host.userMetrics.actionTaken(Host.UserMetrics.Action.AiAssistanceQuerySubmitted);
      if (this.#blockedByCrossOrigin) {
        this.#handleNewChatRequest();
      }
      await this.#startConversation(predefinedPrompt);
    } else {
      this.#viewOutput.chatView?.focusTextInput();
    }
  }
  #populateHistoryMenu(contextMenu) {
    const historicalConversations = AiAssistanceModel.AiHistoryStorage.AiHistoryStorage.instance().getHistory().map(
      (serializedConversation) => AiAssistanceModel.AiHistoryStorage.Conversation.fromSerializedConversation(serializedConversation)
    );
    for (const conversation of historicalConversations.reverse()) {
      if (conversation.isEmpty) {
        continue;
      }
      const title = conversation.title;
      if (!title) {
        continue;
      }
      contextMenu.defaultSection().appendCheckboxItem(title, () => {
        void this.#openHistoricConversation(conversation);
      }, { checked: this.#conversation === conversation, jslogContext: "freestyler.history-item" });
    }
    const historyEmpty = contextMenu.defaultSection().items.length === 0;
    if (historyEmpty) {
      contextMenu.defaultSection().appendItem(i18nString(UIStrings.noPastConversations), () => {
      }, {
        disabled: true
      });
    }
    contextMenu.footerSection().appendItem(
      i18nString(UIStrings.clearChatHistory),
      () => {
        void AiAssistanceModel.AiHistoryStorage.AiHistoryStorage.instance().deleteAll();
      },
      {
        disabled: historyEmpty
      }
    );
  }
  #onHistoryDeleted() {
    this.#updateConversationState();
  }
  #onDeleteClicked() {
    if (!this.#conversation) {
      return;
    }
    void AiAssistanceModel.AiHistoryStorage.AiHistoryStorage.instance().deleteHistoryEntry(this.#conversation.id);
    this.#updateConversationState();
    UI.ARIAUtils.LiveAnnouncer.alert(i18nString(UIStrings.chatDeleted));
  }
  async #onExportConversationClick() {
    if (!this.#conversation) {
      return;
    }
    const markdownContent = this.#conversation.getConversationMarkdown();
    const contentData = new TextUtils.ContentData.ContentData(markdownContent, false, "text/markdown");
    const titleFormatted = Platform.StringUtilities.toSnakeCase(this.#conversation.title || "");
    const prefix = "devtools_";
    const suffix = ".md";
    const maxTitleLength = 64 - prefix.length - suffix.length;
    let finalTitle = titleFormatted || "conversation";
    if (finalTitle.length > maxTitleLength) {
      finalTitle = finalTitle.substring(0, maxTitleLength);
    }
    const filename = `${prefix}${finalTitle}${suffix}`;
    await Workspace.FileManager.FileManager.instance().save(filename, contentData, true);
    Workspace.FileManager.FileManager.instance().close(filename);
  }
  async #openHistoricConversation(conversation) {
    if (this.#conversation === conversation) {
      return;
    }
    this.#updateConversationState({ conversation });
    await this.#doConversation(conversation.history);
  }
  #handleNewChatRequest() {
    this.#updateConversationState();
    UI.ARIAUtils.LiveAnnouncer.alert(i18nString(UIStrings.newChatCreated));
  }
  async #handleTakeScreenshot() {
    const mainTarget = SDK.TargetManager.TargetManager.instance().primaryPageTarget();
    if (!mainTarget) {
      throw new Error("Could not find main target");
    }
    const model = mainTarget.model(SDK.ScreenCaptureModel.ScreenCaptureModel);
    if (!model) {
      throw new Error("Could not find model");
    }
    const showLoadingTimeout = setTimeout(() => {
      this.#imageInput = { isLoading: true };
      this.requestUpdate();
    }, SHOW_LOADING_STATE_TIMEOUT);
    const bytes = await model.captureScreenshot(
      Protocol.Page.CaptureScreenshotRequestFormat.Jpeg,
      SCREENSHOT_QUALITY,
      SDK.ScreenCaptureModel.ScreenshotMode.FROM_VIEWPORT
    );
    clearTimeout(showLoadingTimeout);
    if (bytes) {
      this.#imageInput = {
        isLoading: false,
        data: bytes,
        mimeType: JPEG_MIME_TYPE,
        inputType: AiAssistanceModel.AiAgent.MultimodalInputType.SCREENSHOT
      };
      this.requestUpdate();
      void this.updateComplete.then(() => {
        this.#viewOutput.chatView?.focusTextInput();
      });
    } else {
      this.#imageInput = void 0;
      this.requestUpdate();
      Snackbars.Snackbar.Snackbar.show({
        message: lockedString(UIStringsNotTranslate.screenshotFailureMessage)
      });
    }
  }
  #handleRemoveImageInput() {
    this.#imageInput = void 0;
    this.requestUpdate();
    void this.updateComplete.then(() => {
      this.#viewOutput.chatView?.focusTextInput();
    });
  }
  #handleTextInputChange(value) {
    const disableSubmit = !value;
    if (disableSubmit !== this.#isTextInputEmpty) {
      this.#isTextInputEmpty = disableSubmit;
      void this.requestUpdate();
    }
  }
  async #handleLoadImage(file) {
    const showLoadingTimeout = setTimeout(() => {
      this.#imageInput = { isLoading: true };
      this.requestUpdate();
    }, SHOW_LOADING_STATE_TIMEOUT);
    const reader = new FileReader();
    let dataUrl;
    try {
      dataUrl = await new Promise((resolve, reject) => {
        reader.onload = () => {
          if (typeof reader.result === "string") {
            resolve(reader.result);
          } else {
            reject(new Error("FileReader result was not a string."));
          }
        };
        reader.readAsDataURL(file);
      });
    } catch {
      clearTimeout(showLoadingTimeout);
      this.#imageInput = void 0;
      this.requestUpdate();
      void this.updateComplete.then(() => {
        this.#viewOutput.chatView?.focusTextInput();
      });
      Snackbars.Snackbar.Snackbar.show({
        message: lockedString(UIStringsNotTranslate.uploadImageFailureMessage)
      });
      return;
    }
    clearTimeout(showLoadingTimeout);
    if (!dataUrl) {
      return;
    }
    const commaIndex = dataUrl.indexOf(",");
    const bytes = dataUrl.substring(commaIndex + 1);
    this.#imageInput = {
      isLoading: false,
      data: bytes,
      mimeType: file.type,
      inputType: AiAssistanceModel.AiAgent.MultimodalInputType.UPLOADED_IMAGE
    };
    this.requestUpdate();
    void this.updateComplete.then(() => {
      this.#viewOutput.chatView?.focusTextInput();
    });
  }
  #cancel() {
    this.#runAbortController.abort();
    this.#runAbortController = new AbortController();
  }
  #onContextSelectionChanged() {
    if (!this.#conversationAgent) {
      this.#blockedByCrossOrigin = false;
      return;
    }
    this.#selectedContext = this.#getConversationContext(this.#conversation);
    if (!this.#selectedContext) {
      this.#blockedByCrossOrigin = false;
      this.#viewOutput.chatView?.clearTextInput();
      return;
    }
    this.#blockedByCrossOrigin = !this.#selectedContext.isOriginAllowed(this.#conversationAgent.origin);
  }
  #getConversationContext(conversation) {
    if (!conversation) {
      return null;
    }
    let context;
    switch (conversation.type) {
      case AiAssistanceModel.AiHistoryStorage.ConversationType.STYLING:
        context = this.#selectedElement;
        break;
      case AiAssistanceModel.AiHistoryStorage.ConversationType.FILE:
        context = this.#selectedFile;
        break;
      case AiAssistanceModel.AiHistoryStorage.ConversationType.NETWORK:
        context = this.#selectedRequest;
        break;
      case AiAssistanceModel.AiHistoryStorage.ConversationType.PERFORMANCE:
        context = this.#selectedPerformanceTrace;
        break;
    }
    return context;
  }
  async #startConversation(text, imageInput, multimodalInputType) {
    if (!this.#conversationAgent) {
      return;
    }
    this.#cancel();
    const signal = this.#runAbortController.signal;
    const context = this.#getConversationContext(this.#conversation);
    if (context && !context.isOriginAllowed(this.#conversationAgent.origin)) {
      throw new Error("cross-origin context data should not be included");
    }
    if (this.#conversation?.isEmpty) {
      Badges.UserBadges.instance().recordAction(Badges.BadgeAction.STARTED_AI_CONVERSATION);
    }
    const image = isAiAssistanceMultimodalInputEnabled() ? imageInput : void 0;
    const imageId = image ? crypto.randomUUID() : void 0;
    const multimodalInput = image && imageId && multimodalInputType ? {
      input: image,
      id: imageId,
      type: multimodalInputType
    } : void 0;
    if (this.#conversation) {
      void VisualLogging.logFunctionCall(`start-conversation-${this.#conversation.type}`, "ui");
    }
    const generator = this.#conversationAgent.run(
      text,
      {
        signal,
        selected: context
      },
      multimodalInput
    );
    const generatorWithHistory = this.#conversationHandler.handleConversationWithHistory(generator, this.#conversation);
    await this.#doConversation(generatorWithHistory);
  }
  async #doConversation(items) {
    const release = await this.#mutex.acquire();
    try {
      let commitStep2 = function() {
        if (systemMessage.steps.at(-1) !== step) {
          systemMessage.steps.push(step);
        }
      };
      var commitStep = commitStep2;
      let systemMessage = {
        entity: ChatMessageEntity.MODEL,
        steps: []
      };
      let step = { isLoading: true };
      this.#isLoading = true;
      let announcedAnswerLoading = false;
      let announcedAnswerReady = false;
      for await (const data of items) {
        step.sideEffect = void 0;
        switch (data.type) {
          case AiAssistanceModel.AiAgent.ResponseType.USER_QUERY: {
            this.#messages.push({
              entity: ChatMessageEntity.USER,
              text: data.query,
              imageInput: data.imageInput
            });
            systemMessage = {
              entity: ChatMessageEntity.MODEL,
              steps: []
            };
            this.#messages.push(systemMessage);
            break;
          }
          case AiAssistanceModel.AiAgent.ResponseType.QUERYING: {
            step = { isLoading: true };
            if (!systemMessage.steps.length) {
              systemMessage.steps.push(step);
            }
            break;
          }
          case AiAssistanceModel.AiAgent.ResponseType.CONTEXT: {
            step.title = data.title;
            step.contextDetails = data.details;
            step.isLoading = false;
            commitStep2();
            break;
          }
          case AiAssistanceModel.AiAgent.ResponseType.TITLE: {
            step.title = data.title;
            commitStep2();
            break;
          }
          case AiAssistanceModel.AiAgent.ResponseType.THOUGHT: {
            step.isLoading = false;
            step.thought = data.thought;
            commitStep2();
            break;
          }
          case AiAssistanceModel.AiAgent.ResponseType.SUGGESTIONS: {
            systemMessage.suggestions = data.suggestions;
            break;
          }
          case AiAssistanceModel.AiAgent.ResponseType.SIDE_EFFECT: {
            step.isLoading = false;
            step.code ??= data.code;
            step.sideEffect = {
              onAnswer: (result) => {
                data.confirm(result);
                step.sideEffect = void 0;
                this.requestUpdate();
              }
            };
            commitStep2();
            break;
          }
          case AiAssistanceModel.AiAgent.ResponseType.ACTION: {
            step.isLoading = false;
            step.code ??= data.code;
            step.output ??= data.output;
            step.canceled = data.canceled;
            commitStep2();
            break;
          }
          case AiAssistanceModel.AiAgent.ResponseType.ANSWER: {
            systemMessage.suggestions ??= data.suggestions;
            systemMessage.answer = data.text;
            systemMessage.rpcId = data.rpcId;
            if (systemMessage.steps.length === 1 && systemMessage.steps[0].isLoading) {
              systemMessage.steps.pop();
            }
            step.isLoading = false;
            break;
          }
          case AiAssistanceModel.AiAgent.ResponseType.ERROR: {
            systemMessage.error = data.error;
            systemMessage.rpcId = void 0;
            const lastStep = systemMessage.steps.at(-1);
            if (lastStep) {
              if (data.error === AiAssistanceModel.AiAgent.ErrorType.ABORT) {
                lastStep.canceled = true;
              } else if (lastStep.isLoading) {
                systemMessage.steps.pop();
              }
            }
            if (data.error === AiAssistanceModel.AiAgent.ErrorType.BLOCK) {
              systemMessage.answer = void 0;
            }
          }
        }
        if (!this.#conversation?.isReadOnly) {
          this.requestUpdate();
          if (data.type === AiAssistanceModel.AiAgent.ResponseType.CONTEXT || data.type === AiAssistanceModel.AiAgent.ResponseType.SIDE_EFFECT) {
            this.#viewOutput.chatView?.scrollToBottom();
          }
          switch (data.type) {
            case AiAssistanceModel.AiAgent.ResponseType.CONTEXT:
              UI.ARIAUtils.LiveAnnouncer.status(data.title);
              break;
            case AiAssistanceModel.AiAgent.ResponseType.ANSWER: {
              if (!data.complete && !announcedAnswerLoading) {
                announcedAnswerLoading = true;
                UI.ARIAUtils.LiveAnnouncer.status(lockedString(UIStringsNotTranslate.answerLoading));
              } else if (data.complete && !announcedAnswerReady) {
                announcedAnswerReady = true;
                UI.ARIAUtils.LiveAnnouncer.status(lockedString(UIStringsNotTranslate.answerReady));
              }
            }
          }
        }
      }
      this.#isLoading = false;
      this.requestUpdate();
    } finally {
      release();
    }
  }
}
export function getResponseMarkdown(message) {
  const contentParts = ["## AI"];
  for (const step of message.steps) {
    if (step.title) {
      contentParts.push(`### ${step.title}`);
    }
    if (step.contextDetails) {
      contentParts.push(
        AiAssistanceModel.AiHistoryStorage.Conversation.generateContextDetailsMarkdown(step.contextDetails)
      );
    }
    if (step.thought) {
      contentParts.push(step.thought);
    }
    if (step.code) {
      contentParts.push(`**Code executed:**
\`\`\`
${step.code.trim()}
\`\`\``);
    }
    if (step.output) {
      contentParts.push(`**Data returned:**
\`\`\`
${step.output}
\`\`\``);
    }
  }
  if (message.answer) {
    contentParts.push(`### Answer

${message.answer}`);
  }
  return contentParts.join("\n\n");
}
export class ActionDelegate {
  handleAction(_context, actionId, opts) {
    switch (actionId) {
      case "freestyler.elements-floating-button":
      case "freestyler.element-panel-context":
      case "freestyler.main-menu":
      case "drjones.network-floating-button":
      case "drjones.network-panel-context":
      case "drjones.performance-panel-context":
      case "drjones.sources-floating-button":
      case "drjones.sources-panel-context": {
        void (async () => {
          const view = UI.ViewManager.ViewManager.instance().view(
            AiAssistancePanel.panelName
          );
          if (!view) {
            return;
          }
          await UI.ViewManager.ViewManager.instance().showView(
            AiAssistancePanel.panelName
          );
          const minDrawerSize = UI.InspectorView.InspectorView.instance().totalSize() / 4;
          if (UI.InspectorView.InspectorView.instance().drawerSize() < minDrawerSize) {
            UI.InspectorView.InspectorView.instance().setDrawerSize(minDrawerSize);
          }
          const widget = await view.widget();
          void widget.handleAction(actionId, opts);
        })();
        return true;
      }
    }
    return false;
  }
}
function isAiAssistanceMultimodalUploadInputEnabled() {
  return isAiAssistanceMultimodalInputEnabled() && Boolean(Root.Runtime.hostConfig.devToolsFreestyler?.multimodalUploadInput);
}
function isAiAssistanceMultimodalInputEnabled() {
  return Boolean(Root.Runtime.hostConfig.devToolsFreestyler?.multimodal);
}
function isAiAssistanceServerSideLoggingEnabled() {
  return !Root.Runtime.hostConfig.aidaAvailability?.disallowLogging;
}
//# sourceMappingURL=AiAssistancePanel.js.map
