"use strict";
import * as Common from "../../core/common/common.js";
import * as Root from "../../core/root/root.js";
import * as SDK from "../../core/sdk/sdk.js";
import * as UI from "../../ui/legacy/legacy.js";
import { ComputedStyleWidget } from "./ComputedStyleWidget.js";
import { StylesSidebarPane } from "./StylesSidebarPane.js";
export class ComputedStyleModel extends Common.ObjectWrapper.ObjectWrapper {
  #node;
  #cssModel;
  eventListeners;
  frameResizedTimer;
  computedStylePromise;
  currentTrackedNodeId;
  constructor() {
    super();
    this.#cssModel = null;
    this.eventListeners = [];
    this.#node = UI.Context.Context.instance().flavor(SDK.DOMModel.DOMNode);
    UI.Context.Context.instance().addFlavorChangeListener(SDK.DOMModel.DOMNode, this.onNodeChanged, this);
    UI.Context.Context.instance().addFlavorChangeListener(
      StylesSidebarPane,
      this.evaluateTrackingComputedStyleUpdatesForNode,
      this
    );
    UI.Context.Context.instance().addFlavorChangeListener(
      ComputedStyleWidget,
      this.evaluateTrackingComputedStyleUpdatesForNode,
      this
    );
  }
  dispose() {
    UI.Context.Context.instance().removeFlavorChangeListener(SDK.DOMModel.DOMNode, this.onNodeChanged, this);
    UI.Context.Context.instance().removeFlavorChangeListener(
      StylesSidebarPane,
      this.evaluateTrackingComputedStyleUpdatesForNode,
      this
    );
    UI.Context.Context.instance().removeFlavorChangeListener(
      ComputedStyleWidget,
      this.evaluateTrackingComputedStyleUpdatesForNode,
      this
    );
  }
  node() {
    return this.#node;
  }
  cssModel() {
    return this.#cssModel?.isEnabled() ? this.#cssModel : null;
  }
  // This is a debounced method because the user might be navigated from Styles tab to Computed Style tab and vice versa.
  // For that case, we want to only run this function once.
  evaluateTrackingComputedStyleUpdatesForNode = Common.Debouncer.debounce(() => {
    if (!this.#node) {
      if (this.currentTrackedNodeId) {
        void this.cssModel()?.trackComputedStyleUpdatesForNode(void 0);
        this.currentTrackedNodeId = void 0;
      }
      return;
    }
    const isComputedStyleWidgetVisible = Boolean(UI.Context.Context.instance().flavor(ComputedStyleWidget));
    const isStylesTabVisible = Boolean(UI.Context.Context.instance().flavor(StylesSidebarPane));
    const shouldTrackComputedStyleUpdates = isComputedStyleWidgetVisible || isStylesTabVisible && Root.Runtime.hostConfig.devToolsAnimationStylesInStylesTab?.enabled;
    if (!shouldTrackComputedStyleUpdates) {
      if (this.currentTrackedNodeId) {
        void this.cssModel()?.trackComputedStyleUpdatesForNode(void 0);
        this.currentTrackedNodeId = void 0;
      }
      return;
    }
    if (this.currentTrackedNodeId !== this.#node.id) {
      void this.cssModel()?.trackComputedStyleUpdatesForNode(this.#node.id);
      this.currentTrackedNodeId = this.#node.id;
    }
  }, 100);
  onNodeChanged(event) {
    this.#node = event.data;
    this.updateModel(this.#node ? this.#node.domModel().cssModel() : null);
    this.onCSSModelChanged(null);
    this.evaluateTrackingComputedStyleUpdatesForNode();
  }
  updateModel(cssModel) {
    if (this.#cssModel === cssModel) {
      return;
    }
    Common.EventTarget.removeEventListeners(this.eventListeners);
    this.#cssModel = cssModel;
    const domModel = cssModel ? cssModel.domModel() : null;
    const resourceTreeModel = cssModel ? cssModel.target().model(SDK.ResourceTreeModel.ResourceTreeModel) : null;
    if (cssModel && domModel && resourceTreeModel) {
      this.eventListeners = [
        cssModel.addEventListener(SDK.CSSModel.Events.StyleSheetAdded, this.onCSSModelChanged, this),
        cssModel.addEventListener(SDK.CSSModel.Events.StyleSheetRemoved, this.onCSSModelChanged, this),
        cssModel.addEventListener(SDK.CSSModel.Events.StyleSheetChanged, this.onCSSModelChanged, this),
        cssModel.addEventListener(SDK.CSSModel.Events.FontsUpdated, this.onCSSModelChanged, this),
        cssModel.addEventListener(SDK.CSSModel.Events.MediaQueryResultChanged, this.onCSSModelChanged, this),
        cssModel.addEventListener(SDK.CSSModel.Events.PseudoStateForced, this.onCSSModelChanged, this),
        cssModel.addEventListener(SDK.CSSModel.Events.StartingStylesStateForced, this.onCSSModelChanged, this),
        cssModel.addEventListener(SDK.CSSModel.Events.ModelWasEnabled, this.onCSSModelChanged, this),
        cssModel.addEventListener(SDK.CSSModel.Events.ComputedStyleUpdated, this.onComputedStyleChanged, this),
        domModel.addEventListener(SDK.DOMModel.Events.DOMMutated, this.onDOMModelChanged, this),
        resourceTreeModel.addEventListener(SDK.ResourceTreeModel.Events.FrameResized, this.onFrameResized, this)
      ];
    }
  }
  onCSSModelChanged(event) {
    delete this.computedStylePromise;
    this.dispatchEventToListeners("CSSModelChanged" /* CSS_MODEL_CHANGED */, event?.data ?? null);
  }
  onComputedStyleChanged(event) {
    delete this.computedStylePromise;
    if (event?.data && "nodeId" in event.data && event.data.nodeId !== this.#node?.id) {
      return;
    }
    this.dispatchEventToListeners("ComputedStyleChanged" /* COMPUTED_STYLE_CHANGED */);
  }
  onDOMModelChanged(event) {
    const node = event.data;
    if (!this.#node || this.#node !== node && node.parentNode !== this.#node.parentNode && !node.isAncestor(this.#node)) {
      return;
    }
    this.onCSSModelChanged(null);
  }
  onFrameResized() {
    function refreshContents() {
      this.onCSSModelChanged(null);
      delete this.frameResizedTimer;
    }
    if (this.frameResizedTimer) {
      clearTimeout(this.frameResizedTimer);
    }
    this.frameResizedTimer = window.setTimeout(refreshContents.bind(this), 100);
  }
  elementNode() {
    const node = this.node();
    if (!node) {
      return null;
    }
    return node.enclosingElementOrSelf();
  }
  async fetchComputedStyle() {
    const elementNode = this.elementNode();
    const cssModel = this.cssModel();
    if (!elementNode || !cssModel) {
      return null;
    }
    const nodeId = elementNode.id;
    if (!nodeId) {
      return null;
    }
    if (!this.computedStylePromise) {
      this.computedStylePromise = cssModel.getComputedStyle(nodeId).then(verifyOutdated.bind(this, elementNode));
    }
    return await this.computedStylePromise;
    function verifyOutdated(elementNode2, style) {
      return elementNode2 === this.elementNode() && style ? new ComputedStyle(elementNode2, style) : null;
    }
  }
}
export var Events = /* @__PURE__ */ ((Events2) => {
  Events2["CSS_MODEL_CHANGED"] = "CSSModelChanged";
  Events2["COMPUTED_STYLE_CHANGED"] = "ComputedStyleChanged";
  return Events2;
})(Events || {});
export class ComputedStyle {
  node;
  computedStyle;
  constructor(node, computedStyle) {
    this.node = node;
    this.computedStyle = computedStyle;
  }
}
//# sourceMappingURL=ComputedStyleModel.js.map
