"use strict";
import * as Common from "../../core/common/common.js";
import * as Platform from "../../core/platform/platform.js";
import * as TextUtils from "../../models/text_utils/text_utils.js";
import * as VisualLogging from "../visual_logging/visual_logging.js";
import * as ARIAUtils from "./ARIAUtils.js";
import { SuggestBox } from "./SuggestBox.js";
import textPromptStyles from "./textPrompt.css.js";
import { Tooltip } from "./Tooltip.js";
import { cloneCustomElement, ElementFocusRestorer } from "./UIUtils.js";
export class TextPromptElement extends HTMLElement {
  static observedAttributes = ["editing", "completions"];
  #shadow = this.attachShadow({ mode: "open" });
  #entrypoint = this.#shadow.createChild("span");
  #slot = this.#entrypoint.createChild("slot");
  #textPrompt = new TextPrompt();
  #completionTimeout = null;
  constructor() {
    super();
    this.#textPrompt.initialize(this.#willAutoComplete.bind(this));
  }
  attributeChangedCallback(name, oldValue, newValue) {
    if (oldValue === newValue || !this.isConnected) {
      return;
    }
    switch (name) {
      case "editing":
        if (newValue !== null && newValue !== "false" && oldValue === null) {
          this.#startEditing();
        } else {
          this.#stopEditing();
        }
        break;
      case "completions":
        if (this.#textPrompt.isSuggestBoxVisible()) {
          void this.#textPrompt.complete(
            /* force=*/
            true
          );
        }
        break;
    }
  }
  async #willAutoComplete(expression, filter, force) {
    if (!force) {
      this.dispatchEvent(new TextPromptElement.BeforeAutoCompleteEvent({ expression, filter }));
    }
    const listId = this.getAttribute("completions");
    if (!listId) {
      return [];
    }
    const datalist = this.getComponentRoot()?.querySelectorAll(`datalist#${listId} > option`);
    if (!datalist?.length) {
      return [];
    }
    filter = filter?.toLowerCase();
    return datalist.values().filter((option) => option.textContent.startsWith(filter ?? "")).map((option) => ({ text: option.textContent })).toArray();
  }
  #startEditing() {
    const placeholder = this.#entrypoint.createChild("span");
    placeholder.textContent = this.#slot.deepInnerText();
    this.#slot.remove();
    const proxy = this.#textPrompt.attachAndStartEditing(placeholder, (e) => this.#done(
      e,
      /* commit=*/
      true
    ));
    proxy.addEventListener("keydown", this.#editingValueKeyDown.bind(this));
    placeholder.getComponentSelection()?.selectAllChildren(placeholder);
  }
  #stopEditing() {
    this.#entrypoint.removeChildren();
    this.#entrypoint.appendChild(this.#slot);
    this.#textPrompt.detach();
  }
  connectedCallback() {
    if (this.hasAttribute("editing")) {
      this.attributeChangedCallback("editing", null, "");
    }
  }
  #done(e, commit) {
    const target = e.target;
    const text = target.textContent || "";
    if (commit) {
      this.dispatchEvent(new TextPromptElement.CommitEvent(text));
    } else {
      this.dispatchEvent(new TextPromptElement.CancelEvent());
    }
    e.consume();
  }
  #editingValueKeyDown(event) {
    if (event.handled || !(event instanceof KeyboardEvent)) {
      return;
    }
    if (event.key === "Enter") {
      this.#done(
        event,
        /* commit=*/
        true
      );
    } else if (Platform.KeyboardUtilities.isEscKey(event)) {
      this.#done(
        event,
        /* commit=*/
        false
      );
    }
  }
  set completionTimeout(timeout) {
    this.#completionTimeout = timeout;
    this.#textPrompt.setAutocompletionTimeout(timeout);
  }
  cloneNode() {
    const clone = cloneCustomElement(this);
    if (this.#completionTimeout !== null) {
      clone.completionTimeout = this.#completionTimeout;
    }
    return clone;
  }
}
((TextPromptElement2) => {
  class CommitEvent extends CustomEvent {
    constructor(detail) {
      super("commit", { detail });
    }
  }
  TextPromptElement2.CommitEvent = CommitEvent;
  class CancelEvent extends CustomEvent {
    constructor() {
      super("cancel");
    }
  }
  TextPromptElement2.CancelEvent = CancelEvent;
  class BeforeAutoCompleteEvent extends CustomEvent {
    constructor(detail) {
      super("beforeautocomplete", { detail });
    }
  }
  TextPromptElement2.BeforeAutoCompleteEvent = BeforeAutoCompleteEvent;
})(TextPromptElement || (TextPromptElement = {}));
customElements.define("devtools-prompt", TextPromptElement);
export class TextPrompt extends Common.ObjectWrapper.ObjectWrapper {
  proxyElement;
  proxyElementDisplay;
  autocompletionTimeout;
  #title;
  queryRange;
  previousText;
  currentSuggestion;
  completionRequestId;
  ghostTextElement;
  leftParenthesesIndices;
  loadCompletions;
  completionStopCharacters;
  usesSuggestionBuilder;
  #element;
  boundOnKeyDown;
  boundOnInput;
  boundOnMouseWheel;
  boundClearAutocomplete;
  boundOnBlur;
  contentElement;
  suggestBox;
  isEditing;
  focusRestorer;
  blurListener;
  oldTabIndex;
  completeTimeout;
  #disableDefaultSuggestionForEmptyInput;
  jslogContext = void 0;
  constructor() {
    super();
    this.proxyElementDisplay = "inline-block";
    this.autocompletionTimeout = DefaultAutocompletionTimeout;
    this.#title = "";
    this.queryRange = null;
    this.previousText = "";
    this.currentSuggestion = null;
    this.completionRequestId = 0;
    this.ghostTextElement = document.createElement("span");
    this.ghostTextElement.classList.add("auto-complete-text");
    this.ghostTextElement.setAttribute("contenteditable", "false");
    this.leftParenthesesIndices = [];
    ARIAUtils.setHidden(this.ghostTextElement, true);
  }
  initialize(completions, stopCharacters, usesSuggestionBuilder) {
    this.loadCompletions = completions;
    this.completionStopCharacters = stopCharacters || " =:[({;,!+-*/&|^<>.";
    this.usesSuggestionBuilder = usesSuggestionBuilder || false;
  }
  setAutocompletionTimeout(timeout) {
    this.autocompletionTimeout = timeout;
  }
  renderAsBlock() {
    this.proxyElementDisplay = "block";
  }
  /**
   * Clients should never attach any event listeners to the |element|. Instead,
   * they should use the result of this method to attach listeners for bubbling events.
   */
  attach(element) {
    return this.#attach(element);
  }
  /**
   * Clients should never attach any event listeners to the |element|. Instead,
   * they should use the result of this method to attach listeners for bubbling events
   * or the |blurListener| parameter to register a "blur" event listener on the |element|
   * (since the "blur" event does not bubble.)
   */
  attachAndStartEditing(element, blurListener) {
    const proxyElement = this.#attach(element);
    this.startEditing(blurListener);
    return proxyElement;
  }
  #attach(element) {
    if (this.proxyElement) {
      throw new Error("Cannot attach an attached TextPrompt");
    }
    this.#element = element;
    this.boundOnKeyDown = this.onKeyDown.bind(this);
    this.boundOnInput = this.onInput.bind(this);
    this.boundOnMouseWheel = this.onMouseWheel.bind(this);
    this.boundClearAutocomplete = this.clearAutocomplete.bind(this);
    this.boundOnBlur = this.onBlur.bind(this);
    this.proxyElement = element.ownerDocument.createElement("span");
    Platform.DOMUtilities.appendStyle(this.proxyElement, textPromptStyles);
    this.contentElement = this.proxyElement.createChild("div", "text-prompt-root");
    this.proxyElement.style.display = this.proxyElementDisplay;
    if (element.parentElement) {
      element.parentElement.insertBefore(this.proxyElement, element);
    }
    this.contentElement.appendChild(element);
    let jslog = VisualLogging.textField().track({
      keydown: "ArrowLeft|ArrowUp|PageUp|Home|PageDown|ArrowRight|ArrowDown|End|Space|Tab|Enter|Escape",
      change: true
    });
    if (this.jslogContext) {
      jslog = jslog.context(this.jslogContext);
    }
    if (!this.#element.hasAttribute("jslog")) {
      this.#element.setAttribute("jslog", `${jslog}`);
    }
    this.#element.classList.add("text-prompt");
    ARIAUtils.markAsTextBox(this.#element);
    ARIAUtils.setAutocomplete(this.#element, ARIAUtils.AutocompleteInteractionModel.BOTH);
    ARIAUtils.setHasPopup(this.#element, ARIAUtils.PopupRole.LIST_BOX);
    this.#element.setAttribute("contenteditable", "plaintext-only");
    this.element().addEventListener("keydown", this.boundOnKeyDown, false);
    this.#element.addEventListener("input", this.boundOnInput, false);
    this.#element.addEventListener("wheel", this.boundOnMouseWheel, false);
    this.#element.addEventListener("selectstart", this.boundClearAutocomplete, false);
    this.#element.addEventListener("blur", this.boundOnBlur, false);
    this.suggestBox = new SuggestBox(this, 20);
    if (this.#title) {
      Tooltip.install(this.proxyElement, this.#title);
    }
    return this.proxyElement;
  }
  element() {
    if (!this.#element) {
      throw new Error("Expected an already attached element!");
    }
    return this.#element;
  }
  detach() {
    this.removeFromElement();
    if (this.focusRestorer) {
      this.focusRestorer.restore();
    }
    if (this.proxyElement?.parentElement) {
      this.proxyElement.parentElement.insertBefore(this.element(), this.proxyElement);
      this.proxyElement.remove();
    }
    delete this.proxyElement;
    this.element().classList.remove("text-prompt");
    this.element().removeAttribute("contenteditable");
    this.element().removeAttribute("role");
    ARIAUtils.clearAutocomplete(this.element());
    ARIAUtils.setHasPopup(this.element(), ARIAUtils.PopupRole.FALSE);
  }
  textWithCurrentSuggestion() {
    const text = this.text();
    if (!this.queryRange || !this.currentSuggestion) {
      return text;
    }
    const suggestion = this.currentSuggestion.text;
    return text.substring(0, this.queryRange.startColumn) + suggestion + text.substring(this.queryRange.endColumn);
  }
  text() {
    let text = this.element().textContent || "";
    if (this.ghostTextElement.parentNode) {
      const addition = this.ghostTextElement.textContent || "";
      text = text.substring(0, text.length - addition.length);
    }
    return text;
  }
  setText(text) {
    this.clearAutocomplete();
    this.element().textContent = text;
    this.previousText = this.text();
    if (this.element().hasFocus()) {
      this.moveCaretToEndOfPrompt();
      this.element().scrollIntoView();
    }
  }
  setSelectedRange(startIndex, endIndex) {
    if (startIndex < 0) {
      throw new RangeError("Selected range start must be a nonnegative integer");
    }
    const textContent = this.element().textContent;
    const textContentLength = textContent ? textContent.length : 0;
    if (endIndex > textContentLength) {
      endIndex = textContentLength;
    }
    if (endIndex < startIndex) {
      endIndex = startIndex;
    }
    const textNode = this.element().childNodes[0];
    const range = new Range();
    range.setStart(textNode, startIndex);
    range.setEnd(textNode, endIndex);
    const selection = window.getSelection();
    if (selection) {
      selection.removeAllRanges();
      selection.addRange(range);
    }
  }
  focus() {
    this.element().focus();
  }
  title() {
    return this.#title;
  }
  setTitle(title) {
    this.#title = title;
    if (this.proxyElement) {
      Tooltip.install(this.proxyElement, title);
    }
  }
  setPlaceholder(placeholder, ariaPlaceholder) {
    if (placeholder) {
      this.element().setAttribute("data-placeholder", placeholder);
      ARIAUtils.setPlaceholder(this.element(), ariaPlaceholder || placeholder);
    } else {
      this.element().removeAttribute("data-placeholder");
      ARIAUtils.setPlaceholder(this.element(), null);
    }
  }
  setEnabled(enabled) {
    if (enabled) {
      this.element().setAttribute("contenteditable", "plaintext-only");
    } else {
      this.element().removeAttribute("contenteditable");
    }
    this.element().classList.toggle("disabled", !enabled);
  }
  removeFromElement() {
    this.clearAutocomplete();
    this.element().removeEventListener(
      "keydown",
      this.boundOnKeyDown,
      false
    );
    this.element().removeEventListener("input", this.boundOnInput, false);
    this.element().removeEventListener(
      "selectstart",
      this.boundClearAutocomplete,
      false
    );
    this.element().removeEventListener("blur", this.boundOnBlur, false);
    if (this.isEditing) {
      this.stopEditing();
    }
    if (this.suggestBox) {
      this.suggestBox.hide();
    }
  }
  startEditing(blurListener) {
    this.isEditing = true;
    if (this.contentElement) {
      this.contentElement.classList.add("text-prompt-editing");
    }
    this.focusRestorer = new ElementFocusRestorer(this.element());
    if (blurListener) {
      this.blurListener = blurListener;
      this.element().addEventListener("blur", this.blurListener, false);
    }
    this.oldTabIndex = this.element().tabIndex;
    if (this.element().tabIndex < 0) {
      this.element().tabIndex = 0;
    }
    if (!this.text()) {
      this.autoCompleteSoon();
    }
  }
  stopEditing() {
    this.element().tabIndex = this.oldTabIndex;
    if (this.blurListener) {
      this.element().removeEventListener("blur", this.blurListener, false);
    }
    if (this.contentElement) {
      this.contentElement.classList.remove("text-prompt-editing");
    }
    delete this.isEditing;
  }
  onMouseWheel(_event) {
  }
  onKeyDown(event) {
    let handled = false;
    if (this.isSuggestBoxVisible() && this.suggestBox?.keyPressed(event)) {
      void VisualLogging.logKeyDown(this.suggestBox.element, event);
      event.consume(true);
      return;
    }
    switch (event.key) {
      case "Tab":
        handled = this.tabKeyPressed(event);
        break;
      case "ArrowLeft":
      case "ArrowUp":
      case "PageUp":
      case "Home":
        this.clearAutocomplete();
        break;
      case "PageDown":
      case "ArrowRight":
      case "ArrowDown":
      case "End":
        if (this.isCaretAtEndOfPrompt()) {
          handled = this.acceptAutoComplete();
        } else {
          this.clearAutocomplete();
        }
        break;
      case "Escape":
        if (this.isSuggestBoxVisible() || this.currentSuggestion) {
          this.clearAutocomplete();
          handled = true;
        }
        break;
      case " ":
        if (event.ctrlKey && !event.metaKey && !event.altKey && !event.shiftKey) {
          this.autoCompleteSoon(true);
          handled = true;
        }
        break;
    }
    if (event.key === "Enter") {
      event.preventDefault();
    }
    if (handled) {
      event.consume(true);
    }
  }
  acceptSuggestionOnStopCharacters(key) {
    if (!this.currentSuggestion || !this.queryRange || key.length !== 1 || !this.completionStopCharacters?.includes(key)) {
      return false;
    }
    const query = this.text().substring(this.queryRange.startColumn, this.queryRange.endColumn);
    if (query && this.currentSuggestion.text.startsWith(query + key)) {
      this.queryRange.endColumn += 1;
      return this.acceptAutoComplete();
    }
    return false;
  }
  onInput(ev) {
    const event = ev;
    let text = this.text();
    const currentEntry = event.data;
    if (event.inputType === "insertFromPaste" && text.includes("\n")) {
      text = Platform.StringUtilities.stripLineBreaks(text);
      this.setText(text);
    }
    const caretPosition = this.getCaretPosition();
    if (currentEntry === ")" && caretPosition >= 0 && this.leftParenthesesIndices.length > 0) {
      const nextCharAtCaret = text[caretPosition];
      if (nextCharAtCaret === ")" && this.tryMatchingLeftParenthesis(caretPosition)) {
        text = text.substring(0, caretPosition) + text.substring(caretPosition + 1);
        this.setText(text);
        return;
      }
    }
    if (currentEntry && !this.acceptSuggestionOnStopCharacters(currentEntry)) {
      const hasCommonPrefix = text.startsWith(this.previousText) || this.previousText.startsWith(text);
      if (this.queryRange && hasCommonPrefix) {
        this.queryRange.endColumn += text.length - this.previousText.length;
      }
    }
    this.refreshGhostText();
    this.previousText = text;
    this.dispatchEventToListeners("TextChanged" /* TEXT_CHANGED */);
    this.autoCompleteSoon();
  }
  acceptAutoComplete() {
    let result = false;
    if (this.isSuggestBoxVisible() && this.suggestBox) {
      result = this.suggestBox.acceptSuggestion();
    }
    if (!result) {
      result = this.#acceptSuggestion();
    }
    if (this.usesSuggestionBuilder && result) {
      this.autoCompleteSoon();
    }
    return result;
  }
  clearAutocomplete() {
    const beforeText = this.textWithCurrentSuggestion();
    if (this.isSuggestBoxVisible() && this.suggestBox) {
      this.suggestBox.hide();
    }
    this.clearAutocompleteTimeout();
    this.queryRange = null;
    this.refreshGhostText();
    if (beforeText !== this.textWithCurrentSuggestion()) {
      this.dispatchEventToListeners("TextChanged" /* TEXT_CHANGED */);
    }
    this.currentSuggestion = null;
  }
  onBlur() {
    this.clearAutocomplete();
  }
  refreshGhostText() {
    if (this.currentSuggestion?.hideGhostText) {
      this.ghostTextElement.remove();
      return;
    }
    if (this.queryRange && this.currentSuggestion && this.isCaretAtEndOfPrompt() && this.currentSuggestion.text.startsWith(this.text().substring(this.queryRange.startColumn))) {
      this.ghostTextElement.textContent = this.currentSuggestion.text.substring(this.queryRange.endColumn - this.queryRange.startColumn);
      this.element().appendChild(this.ghostTextElement);
    } else {
      this.ghostTextElement.remove();
    }
  }
  clearAutocompleteTimeout() {
    if (this.completeTimeout) {
      clearTimeout(this.completeTimeout);
      delete this.completeTimeout;
    }
    this.completionRequestId++;
  }
  autoCompleteSoon(force) {
    const immediately = this.isSuggestBoxVisible() || force;
    if (!this.completeTimeout) {
      this.completeTimeout = window.setTimeout(this.complete.bind(this, force), immediately ? 0 : this.autocompletionTimeout);
    }
  }
  async complete(force) {
    this.clearAutocompleteTimeout();
    if (!this.element().isConnected) {
      return;
    }
    const selection = this.element().getComponentSelection();
    if (!selection || selection.rangeCount === 0) {
      return;
    }
    const selectionRange = selection.getRangeAt(0);
    let shouldExit;
    if (!force && !this.isCaretAtEndOfPrompt() && !this.isSuggestBoxVisible()) {
      shouldExit = true;
    } else if (!selection.isCollapsed) {
      shouldExit = true;
    }
    if (shouldExit) {
      this.clearAutocomplete();
      return;
    }
    const wordQueryRange = Platform.DOMUtilities.rangeOfWord(
      selectionRange.startContainer,
      selectionRange.startOffset,
      this.completionStopCharacters,
      this.element(),
      "backward"
    );
    const expressionRange = wordQueryRange.cloneRange();
    expressionRange.collapse(true);
    expressionRange.setStartBefore(this.element());
    const completionRequestId = ++this.completionRequestId;
    const completions = await this.loadCompletions.call(null, expressionRange.toString(), wordQueryRange.toString(), Boolean(force));
    this.completionsReady(completionRequestId, selection, wordQueryRange, Boolean(force), completions);
  }
  disableDefaultSuggestionForEmptyInput() {
    this.#disableDefaultSuggestionForEmptyInput = true;
  }
  boxForAnchorAtStart(selection, textRange) {
    const rangeCopy = selection.getRangeAt(0).cloneRange();
    const anchorElement = document.createElement("span");
    anchorElement.textContent = "\u200B";
    textRange.insertNode(anchorElement);
    const box = anchorElement.boxInWindow(window);
    anchorElement.remove();
    selection.removeAllRanges();
    selection.addRange(rangeCopy);
    return box;
  }
  additionalCompletions(_query) {
    return [];
  }
  completionsReady(completionRequestId, selection, originalWordQueryRange, force, completions) {
    if (this.completionRequestId !== completionRequestId) {
      return;
    }
    const query = originalWordQueryRange.toString();
    const store = /* @__PURE__ */ new Set();
    completions = completions.filter((item) => !store.has(item.text) && Boolean(store.add(item.text)));
    if (query || force) {
      if (query) {
        completions = completions.concat(this.additionalCompletions(query));
      } else {
        completions = this.additionalCompletions(query).concat(completions);
      }
    }
    if (!completions.length) {
      this.clearAutocomplete();
      return;
    }
    const selectionRange = selection.getRangeAt(0);
    const fullWordRange = document.createRange();
    fullWordRange.setStart(originalWordQueryRange.startContainer, originalWordQueryRange.startOffset);
    fullWordRange.setEnd(selectionRange.endContainer, selectionRange.endOffset);
    if (query + selectionRange.toString() !== fullWordRange.toString()) {
      return;
    }
    const beforeRange = document.createRange();
    beforeRange.setStart(this.element(), 0);
    beforeRange.setEnd(fullWordRange.startContainer, fullWordRange.startOffset);
    this.queryRange = new TextUtils.TextRange.TextRange(
      0,
      beforeRange.toString().length,
      0,
      beforeRange.toString().length + fullWordRange.toString().length
    );
    const shouldSelect = !this.#disableDefaultSuggestionForEmptyInput || Boolean(this.text());
    if (this.suggestBox) {
      this.suggestBox.updateSuggestions(
        this.boxForAnchorAtStart(selection, fullWordRange),
        completions,
        shouldSelect,
        !this.isCaretAtEndOfPrompt(),
        this.text()
      );
    }
  }
  applySuggestion(suggestion, isIntermediateSuggestion) {
    this.currentSuggestion = suggestion;
    this.refreshGhostText();
    if (isIntermediateSuggestion) {
      this.dispatchEventToListeners("TextChanged" /* TEXT_CHANGED */);
    }
  }
  acceptSuggestion() {
    this.#acceptSuggestion();
  }
  #acceptSuggestion() {
    if (!this.queryRange) {
      return false;
    }
    const suggestionLength = this.currentSuggestion ? this.currentSuggestion.text.length : 0;
    const selectionRange = this.currentSuggestion ? this.currentSuggestion.selectionRange : null;
    const endColumn = selectionRange ? selectionRange.endColumn : suggestionLength;
    const startColumn = selectionRange ? selectionRange.startColumn : suggestionLength;
    this.element().textContent = this.textWithCurrentSuggestion();
    this.setDOMSelection(this.queryRange.startColumn + startColumn, this.queryRange.startColumn + endColumn);
    this.updateLeftParenthesesIndices();
    this.clearAutocomplete();
    this.dispatchEventToListeners("TextChanged" /* TEXT_CHANGED */);
    return true;
  }
  ownerElement() {
    return this.element();
  }
  setDOMSelection(startColumn, endColumn) {
    this.element().normalize();
    const node = this.element().childNodes[0];
    if (!node || node === this.ghostTextElement) {
      return;
    }
    const range = document.createRange();
    range.setStart(node, startColumn);
    range.setEnd(node, endColumn);
    const selection = this.element().getComponentSelection();
    if (selection) {
      selection.removeAllRanges();
      selection.addRange(range);
    }
  }
  isSuggestBoxVisible() {
    return this.suggestBox?.visible() ?? false;
  }
  isCaretAtEndOfPrompt() {
    const selection = this.element().getComponentSelection();
    if (!selection || selection.rangeCount === 0 || !selection.isCollapsed) {
      return false;
    }
    const selectionRange = selection.getRangeAt(0);
    let node = selectionRange.startContainer;
    if (!node.isSelfOrDescendant(this.element())) {
      return false;
    }
    if (this.ghostTextElement.isAncestor(node)) {
      return true;
    }
    if (node.nodeType === Node.TEXT_NODE && selectionRange.startOffset < (node.nodeValue || "").length) {
      return false;
    }
    let foundNextText = false;
    while (node) {
      if (node.nodeType === Node.TEXT_NODE && node.nodeValue?.length) {
        if (foundNextText && !this.ghostTextElement.isAncestor(node)) {
          return false;
        }
        foundNextText = true;
      }
      node = node.traverseNextNode(this.#element);
    }
    return true;
  }
  moveCaretToEndOfPrompt() {
    const selection = this.element().getComponentSelection();
    const selectionRange = document.createRange();
    let container = this.element();
    while (container.lastChild) {
      container = container.lastChild;
    }
    let offset = 0;
    if (container.nodeType === Node.TEXT_NODE) {
      const textNode = container;
      offset = (textNode.textContent || "").length;
    }
    selectionRange.setStart(container, offset);
    selectionRange.setEnd(container, offset);
    if (selection) {
      selection.removeAllRanges();
      selection.addRange(selectionRange);
    }
  }
  /**
   * -1 if no caret can be found in text prompt
   */
  getCaretPosition() {
    if (!this.element().hasFocus()) {
      return -1;
    }
    const selection = this.element().getComponentSelection();
    if (!selection || selection.rangeCount === 0 || !selection.isCollapsed) {
      return -1;
    }
    const selectionRange = selection.getRangeAt(0);
    if (selectionRange.startOffset !== selectionRange.endOffset) {
      return -1;
    }
    return selectionRange.startOffset;
  }
  tabKeyPressed(_event) {
    return this.acceptAutoComplete();
  }
  /**
   * Try matching the most recent open parenthesis with the given right
   * parenthesis, and closes the matched left parenthesis if found.
   * Return the result of the matching.
   */
  tryMatchingLeftParenthesis(rightParenthesisIndex) {
    const leftParenthesesIndices = this.leftParenthesesIndices;
    if (leftParenthesesIndices.length === 0 || rightParenthesisIndex < 0) {
      return false;
    }
    for (let i = leftParenthesesIndices.length - 1; i >= 0; --i) {
      if (leftParenthesesIndices[i] < rightParenthesisIndex) {
        leftParenthesesIndices.splice(i, 1);
        return true;
      }
    }
    return false;
  }
  updateLeftParenthesesIndices() {
    const text = this.text();
    const leftParenthesesIndices = this.leftParenthesesIndices = [];
    for (let i = 0; i < text.length; ++i) {
      if (text[i] === "(") {
        leftParenthesesIndices.push(i);
      }
    }
  }
  suggestBoxForTest() {
    return this.suggestBox;
  }
}
const DefaultAutocompletionTimeout = 250;
export var Events = /* @__PURE__ */ ((Events2) => {
  Events2["TEXT_CHANGED"] = "TextChanged";
  return Events2;
})(Events || {});
//# sourceMappingURL=TextPrompt.js.map
