"use strict";
import { createChild, Overlay } from "./common.js";
import { buildPath, emptyBounds } from "./highlight_common.js";
export class SourceOrderOverlay extends Overlay {
  sourceOrderContainer;
  reset(resetData) {
    super.reset(resetData);
    this.sourceOrderContainer.textContent = "";
  }
  install() {
    this.document.body.classList.add("fill");
    const canvas = this.document.createElement("canvas");
    canvas.id = "canvas";
    canvas.classList.add("fill");
    this.document.body.append(canvas);
    const sourceOrderContainer = this.document.createElement("div");
    sourceOrderContainer.id = "source-order-container";
    this.document.body.append(sourceOrderContainer);
    this.sourceOrderContainer = sourceOrderContainer;
    this.setCanvas(canvas);
    super.install();
  }
  uninstall() {
    this.document.body.classList.remove("fill");
    this.document.body.innerHTML = "";
    super.uninstall();
  }
  drawSourceOrder(highlight) {
    const sourceOrder = highlight.sourceOrder || 0;
    const path = highlight.paths.slice().pop();
    if (!path) {
      throw new Error("No path provided");
    }
    this.context.save();
    const bounds = emptyBounds();
    const outlineColor = path.outlineColor;
    this.context.save();
    drawPath(this.context, path.path, outlineColor, Boolean(sourceOrder), bounds, this.emulationScaleFactor);
    this.context.restore();
    this.context.save();
    if (Boolean(sourceOrder)) {
      this.drawSourceOrderLabel(sourceOrder, outlineColor, bounds);
    }
    this.context.restore();
    return { bounds };
  }
  drawSourceOrderLabel(sourceOrder, color, bounds) {
    const sourceOrderContainer = this.sourceOrderContainer;
    const otherLabels = sourceOrderContainer.children;
    const labelContainer = createChild(sourceOrderContainer, "div", "source-order-label-container");
    labelContainer.style.color = color;
    labelContainer.textContent = String(sourceOrder);
    const labelHeight = labelContainer.offsetHeight;
    const labelWidth = labelContainer.offsetWidth;
    const labelType = getLabelType(bounds, labelHeight, labelWidth, otherLabels, this.canvasHeight);
    const labelPosition = getPositionFromLabelType(labelType, bounds, labelHeight);
    labelContainer.classList.add(labelType);
    labelContainer.style.top = labelPosition.contentTop + "px";
    labelContainer.style.left = labelPosition.contentLeft + "px";
  }
}
const MAX_CHILD_ELEMENTS_THRESHOLD = 300;
export const LabelTypes = {
  topCorner: "top-corner",
  aboveElement: "above-element",
  belowElement: "below-element",
  aboveElementWider: "above-element-wider",
  belowElementWider: "below-element-wider",
  bottomCornerWider: "bottom-corner-wider",
  bottomCornerTaller: "bottom-corner-taller",
  bottomCornerWiderTaller: "bottom-corner-wider-taller"
};
export function getPositionFromLabelType(positionType, bounds, labelHeight) {
  let contentTop = 0;
  switch (positionType) {
    case LabelTypes.topCorner:
      contentTop = bounds.minY;
      break;
    case LabelTypes.aboveElement:
    case LabelTypes.aboveElementWider:
      contentTop = bounds.minY - labelHeight;
      break;
    case LabelTypes.belowElement:
    case LabelTypes.belowElementWider:
      contentTop = bounds.maxY;
      break;
    case LabelTypes.bottomCornerWider:
    case LabelTypes.bottomCornerTaller:
    case LabelTypes.bottomCornerWiderTaller:
      contentTop = bounds.maxY - labelHeight;
      break;
  }
  return {
    contentTop,
    contentLeft: bounds.minX
  };
}
export function getLabelType(bounds, labelHeight, labelWidth, otherLabels, canvasHeight) {
  let labelType;
  const widerThanElement = bounds.minX + labelWidth > bounds.maxX;
  const tallerThanElement = bounds.minY + labelHeight > bounds.maxY;
  if (!widerThanElement && !tallerThanElement || otherLabels.length >= MAX_CHILD_ELEMENTS_THRESHOLD) {
    return LabelTypes.topCorner;
  }
  let overlaps = false;
  for (let i = 0; i < otherLabels.length; i++) {
    const currentLabel = otherLabels[i];
    const rect = currentLabel.getBoundingClientRect();
    if (currentLabel.style.top === "" && currentLabel.style.left === "") {
      continue;
    }
    const topOverlaps = bounds.minY - labelHeight <= rect.top + rect.height && bounds.minY - labelHeight >= rect.top;
    const bottomOverlaps = bounds.minY <= rect.top + rect.height && bounds.minY >= rect.top;
    const leftOverlaps = bounds.minX >= rect.left && bounds.minX <= rect.left + rect.width;
    const rightOverlaps = bounds.minX + labelWidth >= rect.left && bounds.minX + labelWidth <= rect.left + rect.width;
    const sideOverlaps = leftOverlaps || rightOverlaps;
    if (sideOverlaps && (topOverlaps || bottomOverlaps)) {
      overlaps = true;
      break;
    }
  }
  if (bounds.minY - labelHeight > 0 && !overlaps) {
    labelType = LabelTypes.aboveElement;
    if (widerThanElement) {
      labelType = LabelTypes.aboveElementWider;
    }
  } else if (bounds.maxY + labelHeight < canvasHeight) {
    labelType = LabelTypes.belowElement;
    if (widerThanElement) {
      labelType = LabelTypes.belowElementWider;
    }
  } else if (widerThanElement && tallerThanElement) {
    labelType = LabelTypes.bottomCornerWiderTaller;
  } else if (widerThanElement) {
    labelType = LabelTypes.bottomCornerWider;
  } else {
    labelType = LabelTypes.bottomCornerTaller;
  }
  return labelType;
}
function drawPath(context, commands, outlineColor, isChild, bounds, emulationScaleFactor) {
  context.save();
  const path = buildPath(commands, bounds, emulationScaleFactor);
  if (outlineColor) {
    context.strokeStyle = outlineColor;
    context.lineWidth = 2;
    if (!isChild) {
      context.setLineDash([3, 3]);
    }
    context.stroke(path);
  }
  context.restore();
  return path;
}
//# sourceMappingURL=tool_source_order.js.map
