// Copyright 2012 The Chromium Authors
// Use of this source code is governed by a BSD-style license that can be
// found in the LICENSE file.
class ByteReader{constructor(arrayBuffer,offset=0,length){this.pos_=0;this.seekStack_=[];this.littleEndian_=false;length=length||arrayBuffer.byteLength-offset;this.view_=new DataView(arrayBuffer,offset,length)}static validateRead(pos,size,end){if(pos<0||pos>=end){throw new Error("Invalid read position")}if(pos+size>end){throw new Error("Read past end of buffer")}}static readString(dataView,pos,size,end){ByteReader.validateRead(pos,size,end||dataView.byteLength);const codes=[];for(let i=0;i<size;++i){codes.push(dataView.getUint8(pos+i))}return String.fromCharCode.apply(null,codes)}static readNullTerminatedString(dataView,pos,size,end){ByteReader.validateRead(pos,size,end||dataView.byteLength);const codes=[];for(let i=0;i<size;++i){const code=dataView.getUint8(pos+i);if(code===0){break}codes.push(code)}return String.fromCharCode.apply(null,codes)}static readNullTerminatedStringUtf16(dataView,pos,bom,size,end){ByteReader.validateRead(pos,size,end||dataView.byteLength);let littleEndian=false;let start=0;if(bom){littleEndian=dataView.getUint8(pos)===255;start=2}const codes=[];for(let i=start;i<size;i+=2){const code=dataView.getUint16(pos+i,littleEndian);if(code===0){break}codes.push(code)}return String.fromCharCode.apply(null,codes)}static readBase64(dataView,pos,size,end){ByteReader.validateRead(pos,size,end||dataView.byteLength);const rv=[];const chars=[];let padding=0;for(let i=0;i<size;){let bits=dataView.getUint8(pos+i++)<<16;if(i<size){bits|=dataView.getUint8(pos+i++)<<8;if(i<size){bits|=dataView.getUint8(pos+i++)}else{padding=1}}else{padding=2}chars[3]=BASE64_ALPHABET[bits&63];chars[2]=BASE64_ALPHABET[bits>>6&63];chars[1]=BASE64_ALPHABET[bits>>12&63];chars[0]=BASE64_ALPHABET[bits>>18&63];rv.push.apply(rv,chars)}if(padding>0){rv[rv.length-1]="="}if(padding>1){rv[rv.length-2]="="}return rv.join("")}static readImage(dataView,pos,size,end){end=end||dataView.byteLength;ByteReader.validateRead(pos,size,end);const prefixToMime={"P":"png","ÿØ":"jpeg",BM:"bmp",GI:"gif"};const prefix=ByteReader.readString(dataView,pos,2,end);const mime=prefixToMime[prefix]||dataView.getUint16(pos,false).toString(16);const b64=ByteReader.readBase64(dataView,pos,size,end);return"data:image/"+mime+";base64,"+b64}canRead(size){return this.pos_+size<=this.view_.byteLength}eof(){return this.pos_>=this.view_.byteLength}bof(){return this.pos_<0}beof(){return this.pos_>=this.view_.byteLength||this.pos_<0}setByteOrder(order){this.littleEndian_=order===ByteOrder.LITTLE_ENDIAN}validateRead(size,end){if(typeof end==="undefined"){end=this.view_.byteLength}ByteReader.validateRead(this.pos_,size,end)}readScalar(width,signed,end){this.validateRead(width,end);const method=WIDTH_TO_DATA_VIEW_METHOD[width][signed?1:0];let rv;if(method==="getInt8"||method==="getUint8"){rv=this.view_[method](this.pos_)}else{rv=this.view_[method](this.pos_,this.littleEndian_)}this.pos_+=width;return rv}readString(size,end){const rv=ByteReader.readString(this.view_,this.pos_,size,end);this.pos_+=size;return rv}readNullTerminatedString(size,end){const rv=ByteReader.readNullTerminatedString(this.view_,this.pos_,size,end);this.pos_+=rv.length;if(rv.length<size){this.pos_++}return rv}readNullTerminatedStringUtf16(bom,size,end){const rv=ByteReader.readNullTerminatedStringUtf16(this.view_,this.pos_,bom,size,end);if(bom){this.pos_+=2}this.pos_+=rv.length;if(rv.length<size){this.pos_+=2}return rv}readBase64(size,end){const rv=ByteReader.readBase64(this.view_,this.pos_,size,end);this.pos_+=size;return rv}readImage(size,end){const rv=ByteReader.readImage(this.view_,this.pos_,size,end);this.pos_+=size;return rv}seek(pos,seekStart=SeekOrigin.SEEK_BEG,end){end=end||this.view_.byteLength;let newPos;if(seekStart===SeekOrigin.SEEK_CUR){newPos=this.pos_+pos}else if(seekStart===SeekOrigin.SEEK_END){newPos=end+pos}else{newPos=pos}if(newPos<0||newPos>this.view_.byteLength){throw new Error("Seek outside of buffer: "+(newPos-end))}this.pos_=newPos}pushSeek(pos,seekStart){const oldPos=this.pos_;this.seek(pos,seekStart);this.seekStack_.push(oldPos)}popSeek(){const lastSeek=this.seekStack_.pop();if(lastSeek!==undefined){this.seek(lastSeek)}}tell(){return this.pos_}}var ByteOrder;(function(ByteOrder){ByteOrder[ByteOrder["LITTLE_ENDIAN"]=0]="LITTLE_ENDIAN";ByteOrder[ByteOrder["BIG_ENDIAN"]=1]="BIG_ENDIAN"})(ByteOrder||(ByteOrder={}));var SeekOrigin;(function(SeekOrigin){SeekOrigin[SeekOrigin["SEEK_BEG"]=0]="SEEK_BEG";SeekOrigin[SeekOrigin["SEEK_CUR"]=1]="SEEK_CUR";SeekOrigin[SeekOrigin["SEEK_END"]=2]="SEEK_END"})(SeekOrigin||(SeekOrigin={}));const BASE64_ALPHABET="ABCDEFGHIJKLMNOPQRSTUVWXYZabcdefghijklmnopqrstuvwxyz0123456789+/".split("");const WIDTH_TO_DATA_VIEW_METHOD={1:["getUint8","getInt8"],2:["getUint16","getInt16"],4:["getUint32","getInt32"]};var ExifMark;(function(ExifMark){ExifMark[ExifMark["SOS"]=65498]="SOS";ExifMark[ExifMark["SOF"]=65472]="SOF";ExifMark[ExifMark["SOI"]=65496]="SOI";ExifMark[ExifMark["EOI"]=65497]="EOI";ExifMark[ExifMark["APP0"]=65504]="APP0";ExifMark[ExifMark["EXIF"]=65505]="EXIF"})(ExifMark||(ExifMark={}));var ExifAlign;(function(ExifAlign){ExifAlign[ExifAlign["LITTLE"]=18761]="LITTLE";ExifAlign[ExifAlign["BIG"]=19789]="BIG"})(ExifAlign||(ExifAlign={}));var ExifTag;(function(ExifTag){ExifTag[ExifTag["TIFF"]=42]="TIFF";ExifTag[ExifTag["GPSDATA"]=34853]="GPSDATA";ExifTag[ExifTag["EXIFDATA"]=34665]="EXIFDATA";ExifTag[ExifTag["JPG_THUMB_OFFSET"]=513]="JPG_THUMB_OFFSET";ExifTag[ExifTag["JPG_THUMB_LENGTH"]=514]="JPG_THUMB_LENGTH";ExifTag[ExifTag["IMAGE_WIDTH"]=256]="IMAGE_WIDTH";ExifTag[ExifTag["IMAGE_HEIGHT"]=257]="IMAGE_HEIGHT";ExifTag[ExifTag["COMPRESSION"]=258]="COMPRESSION";ExifTag[ExifTag["MAKE"]=271]="MAKE";ExifTag[ExifTag["MODEL"]=272]="MODEL";ExifTag[ExifTag["ORIENTATION"]=274]="ORIENTATION";ExifTag[ExifTag["MODIFIED_DATETIME"]=306]="MODIFIED_DATETIME";ExifTag[ExifTag["X_DIMENSION"]=40962]="X_DIMENSION";ExifTag[ExifTag["Y_DIMENSION"]=40963]="Y_DIMENSION";ExifTag[ExifTag["SOFTWARE"]=305]="SOFTWARE";ExifTag[ExifTag["APERTURE"]=33437]="APERTURE";ExifTag[ExifTag["EXPOSURE_TIME"]=33434]="EXPOSURE_TIME";ExifTag[ExifTag["ISO_SPEED"]=34855]="ISO_SPEED";ExifTag[ExifTag["FOCAL_LENGTH"]=37386]="FOCAL_LENGTH";ExifTag[ExifTag["DATETIME_ORIGINAL"]=36867]="DATETIME_ORIGINAL";ExifTag[ExifTag["CREATE_DATETIME"]=36868]="CREATE_DATETIME"})(ExifTag||(ExifTag={}));class MetadataParser{constructor(parent_,type,urlFilter){this.parent_=parent_;this.type=type;this.urlFilter=urlFilter;this.mimeType="unknown";this.verbose=parent_.verbose}error(...args){this.parent_.error.apply(this.parent_,args)}log(...args){this.parent_.log.apply(this.parent_,args)}vlog(...args){if(this.verbose){this.parent_.log.apply(this.parent_,args)}}createDefaultMetadata(){return{type:this.type,mimeType:this.mimeType}}static async readFileBytes(file,begin,end){return new ByteReader(await file.slice(begin,end).arrayBuffer())}}class ImageParser extends MetadataParser{constructor(parent,type,urlFilter){super(parent,type,urlFilter);this.mimeType="image/"+this.type}}class ExifParser extends ImageParser{constructor(parent){super(parent,"jpeg",/\.jpe?g$/i)}parse(file,metadata,callback,errorCallback){this.requestSlice(file,callback,errorCallback,metadata,0)}requestSlice(file,callback,errorCallback,metadata,filePos,length){length=Math.max(1024,length||0);const self=this;const reader=new FileReader;reader.onerror=errorCallback;reader.onload=()=>{self.parseSlice(file,callback,errorCallback,metadata,filePos,reader.result)};reader.readAsArrayBuffer(file.slice(filePos,filePos+length))}parseSlice(file,callback,errorCallback,metadata,filePos,buf){try{const br=new ByteReader(buf);if(!br.canRead(4)){throw new Error("Unexpected EOF @"+(filePos+buf.byteLength))}if(filePos===0){const firstMark=this.readMark(br);if(firstMark!==ExifMark.SOI){throw new Error("Invalid file header: "+firstMark.toString(16))}}const self=this;const reread=(offset,bytes)=>{self.requestSlice(file,callback,errorCallback,metadata,filePos+br.tell()+(offset||0),bytes)};while(true){if(!br.canRead(4)){reread();return}const mark=this.readMark(br);if(mark===ExifMark.SOS){throw new Error("SOS marker found before SOF")}const markLength=this.readMarkLength(br);const nextSectionStart=br.tell()+markLength;if(!br.canRead(markLength)){if(filePos+br.tell()+markLength>file.size){throw new Error("Invalid section length @"+(filePos+br.tell()-2))}reread(-4,markLength+4);return}if(mark===ExifMark.EXIF){this.parseExifSection(metadata,buf,br)}else if(ExifParser.isSof_(mark)){br.seek(1,SeekOrigin.SEEK_CUR);const height=br.readScalar(2);const width=br.readScalar(2);ExifParser.setImageSize(metadata,width,height);callback(metadata);return}br.seek(nextSectionStart,SeekOrigin.SEEK_BEG)}}catch(e){errorCallback(e.toString())}}static isSof_(mark){if((mark&~15)!==ExifMark.SOF){return false}const type=mark&15;return type!==4&&type!==8&&type!==12}parseExifSection(metadata,buf,br){const magic=br.readString(6);if(magic!=="Exif\0\0"){this.vlog("Invalid EXIF magic: "+magic+br.readString(100));return}br=new ByteReader(buf,br.tell());const order=br.readScalar(2);if(order===ExifAlign.LITTLE){br.setByteOrder(ByteOrder.LITTLE_ENDIAN)}else if(order!==ExifAlign.BIG){this.log("Invalid alignment value: "+order.toString(16));return}const tag=br.readScalar(2);if(tag!==ExifTag.TIFF){this.log("Invalid TIFF tag: "+tag.toString(16));return}metadata.littleEndian=order===ExifAlign.LITTLE;metadata.ifd={image:{},thumbnail:{}};let directoryOffset=br.readScalar(4);this.vlog("Read image directory");br.seek(directoryOffset);directoryOffset=this.readDirectory(br,metadata.ifd.image);metadata.imageTransform=this.parseOrientation(metadata.ifd.image);if(directoryOffset){this.vlog("Read thumbnail directory");br.seek(directoryOffset);this.readDirectory(br,metadata.ifd.thumbnail);metadata.thumbnailTransform=this.parseOrientation(metadata.ifd.thumbnail)||metadata.imageTransform}if(ExifTag.EXIFDATA in metadata.ifd.image){this.vlog("Read EXIF directory");directoryOffset=metadata.ifd.image[ExifTag.EXIFDATA].value;br.seek(directoryOffset);metadata.ifd.exif={};this.readDirectory(br,metadata.ifd.exif)}if(ExifTag.GPSDATA in metadata.ifd.image){this.vlog("Read GPS directory");directoryOffset=metadata.ifd.image[ExifTag.GPSDATA].value;br.seek(directoryOffset);metadata.ifd.gps={};this.readDirectory(br,metadata.ifd.gps)}if(ExifTag.JPG_THUMB_OFFSET in metadata.ifd.thumbnail&&ExifTag.JPG_THUMB_LENGTH in metadata.ifd.thumbnail){this.vlog("Read thumbnail image");br.seek(metadata.ifd.thumbnail[ExifTag.JPG_THUMB_OFFSET].value);metadata.thumbnailURL=br.readImage(metadata.ifd.thumbnail[ExifTag.JPG_THUMB_LENGTH].value)}else{this.vlog("Image has EXIF data, but no JPG thumbnail")}}static setImageSize(metadata,width,height){if(metadata.imageTransform&&metadata.imageTransform.rotate90){metadata.width=height;metadata.height=width}else{metadata.width=width;metadata.height=height}}readMark(br){return br.readScalar(2)}readMarkLength(br){return br.readScalar(2)-2}readDirectory(br,tags){const entryCount=br.readScalar(2);for(let i=0;i<entryCount;i++){const tagId=br.readScalar(2);const tag=tags[tagId]={id:tagId,format:0,componentCount:0,value:undefined};tag.format=br.readScalar(2);tag.componentCount=br.readScalar(4);this.readTagValue(br,tag)}return br.readScalar(4)}readTagValue(br,tag){const self=this;function safeRead(size,readFunction,signed){try{unsafeRead(size,readFunction,signed)}catch(ex){self.log("Error reading tag 0x"+tag.id.toString(16)+"/"+tag.format+", size "+tag.componentCount+"*"+size+" "+(ex.stack||"<no stack>")+": "+ex);tag.value=null}}function unsafeRead(size,readFunction,signed){const reader=readFunction||(size=>br.readScalar(size,signed));const totalSize=tag.componentCount*size;if(totalSize<1){tag.componentCount=1;tag.value=br.readScalar(4);return}if(totalSize>4){br.pushSeek(br.readScalar(4))}if(tag.componentCount===1){tag.value=reader(size)}else{tag.value=[];for(let i=0;i<tag.componentCount;i++){tag.value[i]=reader(size)}}if(totalSize>4){br.popSeek()}else if(totalSize<4){br.seek(4-totalSize,SeekOrigin.SEEK_CUR)}}switch(tag.format){case 1:case 7:safeRead(1);break;case 2:safeRead(1);if(tag.componentCount===0){tag.value=""}else if(tag.componentCount===1){tag.value=String.fromCharCode(tag.value)}else{tag.value=String.fromCharCode.apply(null,tag.value)}this.validateAndFixStringTag_(tag);break;case 3:safeRead(2);break;case 4:safeRead(4);break;case 9:safeRead(4,undefined,true);break;case 5:safeRead(8,(()=>[br.readScalar(4),br.readScalar(4)]));break;case 10:safeRead(8,(()=>[br.readScalar(4,true),br.readScalar(4,true)]));break;default:this.vlog("Unknown tag format 0x"+Number(tag.id).toString(16)+": "+tag.format);safeRead(4);break}this.vlog("Read tag: 0x"+tag.id.toString(16)+"/"+tag.format+": "+tag.value)}validateAndFixStringTag_(tag){if(tag.format===2){if(tag.value.charAt(tag.value.length-1)!=="\0"){tag.value+="\0";tag.componentCount=tag.value.length;this.vlog("Appended missing null character at the end of tag 0x"+tag.id.toString(16)+"/"+tag.format)}}}parseOrientation(ifd){if(ifd[ExifTag.ORIENTATION]){const index=(ifd[ExifTag.ORIENTATION].value||1)-1;return{scaleX:SCALEX[index],scaleY:SCALEY[index],rotate90:ROTATE90[index]}}return undefined}}const SCALEX=[1,-1,-1,1,1,1,-1,-1];const SCALEY=[1,1,-1,-1,-1,1,1,-1];const ROTATE90=[0,0,0,0,1,1,1,1];class Id3Parser extends MetadataParser{constructor(parent){super(parent,"id3",/\.(mp3)$/i)}static readSynchSafe_(reader,length){let rv=0;switch(length){case 4:rv=reader.readScalar(1,false)<<21;case 3:rv|=reader.readScalar(1,false)<<14;case 2:rv|=reader.readScalar(1,false)<<7;case 1:rv|=reader.readScalar(1,false)}return rv}static readUint24_(reader){return reader.readScalar(2,false)<<16|reader.readScalar(1,false)}readString_(reader,encoding,size){switch(encoding){case Id3Parser.V2.ENCODING.ISO_8859_1:return reader.readNullTerminatedString(size);case Id3Parser.V2.ENCODING.UTF_16:return reader.readNullTerminatedStringUtf16(true,size);case Id3Parser.V2.ENCODING.UTF_16BE:return reader.readNullTerminatedStringUtf16(false,size);case Id3Parser.V2.ENCODING.UTF_8:this.log("UTF8 encoding not supported, used ISO_8859_1 instead");return reader.readNullTerminatedString(size);default:{this.log("Unsupported encoding in ID3 tag: "+encoding);return""}}}readTextFrame_(reader,_majorVersion,frame,end){frame.encoding=reader.readScalar(1,false,end);frame.value=this.readString_(reader,frame.encoding,end-reader.tell())}readUserDefinedTextFrame_(reader,_majorVersion,frame,end){frame.encoding=reader.readScalar(1,false,end);frame.description=this.readString_(reader,frame.encoding,end-reader.tell());frame.value=this.readString_(reader,frame.encoding,end-reader.tell())}readPic_(reader,_majorVersion,frame,end){frame.encoding=reader.readScalar(1,false,end);frame.format=reader.readNullTerminatedString(3,end-reader.tell());frame.pictureType=reader.readScalar(1,false,end);frame.description=this.readString_(reader,frame.encoding,end-reader.tell());if(frame.format==="--\x3e"){frame.imageUrl=reader.readNullTerminatedString(end-reader.tell())}else{frame.imageUrl=reader.readImage(end-reader.tell())}}readApic_(reader,_majorVersion,frame,end){this.vlog("Extracting picture");frame.encoding=reader.readScalar(1,false,end);frame.mime=reader.readNullTerminatedString(end-reader.tell());frame.pictureType=reader.readScalar(1,false,end);frame.description=this.readString_(reader,frame.encoding,end-reader.tell());if(frame.mime==="--\x3e"){frame.imageUrl=reader.readNullTerminatedString(end-reader.tell())}else{frame.imageUrl=reader.readImage(end-reader.tell())}}readFrame_(reader,majorVersion){if(reader.eof()){return null}const frame={name:"",headerSize:0,size:0};reader.pushSeek(reader.tell(),SeekOrigin.SEEK_BEG);const position=reader.tell();frame.name=majorVersion===2?reader.readNullTerminatedString(3):reader.readNullTerminatedString(4);if(frame.name===""){return null}this.vlog("Found frame "+frame.name+" at position "+position);switch(majorVersion){case 2:frame.size=Id3Parser.readUint24_(reader);frame.headerSize=6;break;case 3:frame.size=reader.readScalar(4,false);frame.headerSize=10;frame.flags=reader.readScalar(2,false);break;case 4:frame.size=Id3Parser.readSynchSafe_(reader,4);frame.headerSize=10;frame.flags=reader.readScalar(2,false);break}this.vlog("Found frame ["+frame.name+"] with size ["+frame.size+"]");const handler=Id3Parser.V2.HANDLERS[frame.name];if(handler){handler.call(this,reader,majorVersion,frame,reader.tell()+frame.size)}else if(frame.name.charAt(0)==="T"||frame.name.charAt(0)==="W"){this.readTextFrame_(reader,majorVersion,frame,reader.tell()+frame.size)}reader.popSeek();reader.seek(frame.size+frame.headerSize,SeekOrigin.SEEK_CUR);return frame}async parseId3v1(file,metadata){const reader=await MetadataParser.readFileBytes(file,file.size-128,file.size);if(reader.readString(3)==="TAG"){this.vlog("id3v1 found");const title=reader.readNullTerminatedString(30).trim();if(title.length>0){metadata.title=title}reader.seek(3+30,SeekOrigin.SEEK_BEG);const artist=reader.readNullTerminatedString(30).trim();if(artist.length>0){metadata.artist=artist}reader.seek(3+30+30,SeekOrigin.SEEK_BEG);const album=reader.readNullTerminatedString(30).trim();if(album.length>0){metadata.album=album}}}async parseId3v2(file,metadata){let reader=await MetadataParser.readFileBytes(file,0,10);if(reader.readString(3)!=="ID3"){return}this.vlog("id3v2 found");const major=reader.readScalar(1,false);const minor=reader.readScalar(1,false);const flags=reader.readScalar(1,false);const size=Id3Parser.readSynchSafe_(reader,4);const id3v2=metadata.id3v2={majorVersion:major,minorVersion:minor,flags:flags,size:size,frames:{}};reader=await MetadataParser.readFileBytes(file,10,10+id3v2.size);if(id3v2.majorVersion>2&&(id3v2.flags&Id3Parser.V2.FLAG_EXTENDED_HEADER)!==0){if(id3v2.majorVersion===3){reader.seek(reader.readScalar(4,false)-4)}else if(id3v2.majorVersion===4){reader.seek(Id3Parser.readSynchSafe_(reader,4)-4)}}let frame;while(frame=this.readFrame_(reader,id3v2.majorVersion)){id3v2.frames[frame.name]=frame}if(id3v2.frames["APIC"]){metadata.thumbnailURL=id3v2.frames["APIC"].imageUrl}else if(id3v2.frames["PIC"]){metadata.thumbnailURL=id3v2.frames["PIC"].imageUrl}metadata.description=[];for(const[key,frame]of Object.entries(id3v2.frames)){const mappedKey=Id3Parser.V2.MAPPERS[key];if(mappedKey&&frame.value&&frame.value.trim().length>0){metadata.description.push({key:mappedKey,value:frame.value.trim()})}}function extract(propName,...tagNames){for(const tagName of tagNames){const tag=id3v2.frames[tagName];if(tag&&tag.value){metadata[propName]=tag.value;break}}}extract("album","TALB","TAL");extract("title","TIT2","TT2");extract("artist","TPE1","TP1");metadata.description.sort(((a,b)=>Id3Parser.METADATA_ORDER.indexOf(a.key)-Id3Parser.METADATA_ORDER.indexOf(b.key)))}parse(file,metadata,callback,onError){this.log("Starting id3 parser for "+file.name);Promise.all([this.parseId3v1(file,metadata),this.parseId3v2(file,metadata)]).then((()=>{callback(metadata)})).catch((e=>{onError(e.toString())}))}static{this.METADATA_ORDER=["ID3_TITLE","ID3_LEAD_PERFORMER","ID3_YEAR","ID3_ALBUM","ID3_TRACK_NUMBER","ID3_BPM","ID3_COMPOSER","ID3_DATE","ID3_PLAYLIST_DELAY","ID3_LYRICIST","ID3_FILE_TYPE","ID3_TIME","ID3_LENGTH","ID3_FILE_OWNER","ID3_BAND","ID3_COPYRIGHT","ID3_OFFICIAL_AUDIO_FILE_WEBPAGE","ID3_OFFICIAL_ARTIST","ID3_OFFICIAL_AUDIO_SOURCE_WEBPAGE","ID3_PUBLISHERS_OFFICIAL_WEBPAGE"]}static{this.V1={GENRES:["Blues","Classic Rock","Country","Dance","Disco","Funk","Grunge","Hip-Hop","Jazz","Metal","New Age","Oldies","Other","Pop","R&B","Rap","Reggae","Rock","Techno","Industrial","Alternative","Ska","Death Metal","Pranks","Soundtrack","Euro-Techno","Ambient","Trip-Hop","Vocal","Jazz+Funk","Fusion","Trance","Classical","Instrumental","Acid","House","Game","Sound Clip","Gospel","Noise","AlternRock","Bass","Soul","Punk","Space","Meditative","Instrumental Pop","Instrumental Rock","Ethnic","Gothic","Darkwave","Techno-Industrial","Electronic","Pop-Folk","Eurodance","Dream","Southern Rock","Comedy","Cult","Gangsta","Top 40","Christian Rap","Pop/Funk","Jungle","Native American","Cabaret","New Wave","Psychadelic","Rave","Showtunes","Trailer","Lo-Fi","Tribal","Acid Punk","Acid Jazz","Polka","Retro","Musical","Rock & Roll","Hard Rock","Folk","Folk-Rock","National Folk","Swing","Fast Fusion","Bebob","Latin","Revival","Celtic","Bluegrass","Avantgarde","Gothic Rock","Progressive Rock","Psychedelic Rock","Symphonic Rock","Slow Rock","Big Band","Chorus","Easy Listening","Acoustic","Humour","Speech","Chanson","Opera","Chamber Music","Sonata","Symphony","Booty Bass","Primus","Porn Groove","Satire","Slow Jam","Club","Tango","Samba","Folklore","Ballad","Power Ballad","Rhythmic Soul","Freestyle","Duet","Punk Rock","Drum Solo","A capella","Euro-House","Dance Hall","Goa","Drum & Bass","Club-House","Hardcore","Terror","Indie","BritPop","Negerpunk","Polsk Punk","Beat","Christian Gangsta Rap","Heavy Metal","Black Metal","Crossover","Contemporary Christian","Christian Rock","Merengue","Salsa","Thrash Metal","Anime","Jpop","Synthpop"]}}static{this.V2={FLAG_EXTENDED_HEADER:1<<5,ENCODING:{ISO_8859_1:0,UTF_16:1,UTF_16BE:2,UTF_8:3},HANDLERS:{TXX:Id3Parser.prototype.readUserDefinedTextFrame_,WXX:Id3Parser.prototype.readUserDefinedTextFrame_,TXXX:Id3Parser.prototype.readUserDefinedTextFrame_,WXXX:Id3Parser.prototype.readUserDefinedTextFrame_,PIC:Id3Parser.prototype.readPic_,APIC:Id3Parser.prototype.readApic_},MAPPERS:{TALB:"ID3_ALBUM",TBPM:"ID3_BPM",TCOM:"ID3_COMPOSER",TDAT:"ID3_DATE",TDLY:"ID3_PLAYLIST_DELAY",TEXT:"ID3_LYRICIST",TFLT:"ID3_FILE_TYPE",TIME:"ID3_TIME",TIT2:"ID3_TITLE",TLEN:"ID3_LENGTH",TOWN:"ID3_FILE_OWNER",TPE1:"ID3_LEAD_PERFORMER",TPE2:"ID3_BAND",TRCK:"ID3_TRACK_NUMBER",TYER:"ID3_YEAR",WCOP:"ID3_COPYRIGHT",WOAF:"ID3_OFFICIAL_AUDIO_FILE_WEBPAGE",WOAR:"ID3_OFFICIAL_ARTIST",WOAS:"ID3_OFFICIAL_AUDIO_SOURCE_WEBPAGE",WPUB:"ID3_PUBLISHERS_OFFICIAL_WEBPAGE"}}}}class SimpleImageParser extends ImageParser{constructor(parent,type,urlFilter,headerSize){super(parent,type,urlFilter);this.headerSize=headerSize}parse(file,metadata,callback,errorCallback){MetadataParser.readFileBytes(file,0,this.headerSize).then((byteReader=>{this.parseHeader(metadata,byteReader);callback(metadata)})).catch((e=>{errorCallback(e.toString())}))}}class PngParser extends SimpleImageParser{constructor(parent){super(parent,"png",/\.png$/i,24)}parseHeader(metadata,br){br.setByteOrder(ByteOrder.BIG_ENDIAN);const signature=br.readString(8);if(signature!=="PNG\r\n\n"){throw new Error("Invalid PNG signature: "+signature)}br.seek(12);const ihdr=br.readString(4);if(ihdr!=="IHDR"){throw new Error("Missing IHDR chunk")}metadata.width=br.readScalar(4);metadata.height=br.readScalar(4)}}class BmpParser extends SimpleImageParser{constructor(parent){super(parent,"bmp",/\.bmp$/i,28)}parseHeader(metadata,br){br.setByteOrder(ByteOrder.LITTLE_ENDIAN);const signature=br.readString(2);if(signature!=="BM"){throw new Error("Invalid BMP signature: "+signature)}br.seek(18);metadata.width=br.readScalar(4);metadata.height=br.readScalar(4)}}class GifParser extends SimpleImageParser{constructor(parent){super(parent,"gif",/\.Gif$/i,10)}parseHeader(metadata,br){br.setByteOrder(ByteOrder.LITTLE_ENDIAN);const signature=br.readString(6);if(!signature.match(/GIF8(7|9)a/)){throw new Error("Invalid GIF signature: "+signature)}metadata.width=br.readScalar(2);metadata.height=br.readScalar(2)}}class WebpParser extends SimpleImageParser{constructor(parent){super(parent,"webp",/\.webp$/i,30)}parseHeader(metadata,br){br.setByteOrder(ByteOrder.LITTLE_ENDIAN);const riffSignature=br.readString(4);if(riffSignature!=="RIFF"){throw new Error("Invalid RIFF signature: "+riffSignature)}br.seek(8);const webpSignature=br.readString(4);if(webpSignature!=="WEBP"){throw new Error("Invalid WEBP signature: "+webpSignature)}const chunkFormat=br.readString(4);switch(chunkFormat){case"VP8 ":br.seek(23);const lossySignature=br.readScalar(2)|br.readScalar(1)<<16;if(lossySignature!==2752925){throw new Error("Invalid VP8 lossy bitstream signature: "+lossySignature)}{const dimensionBits=br.readScalar(4);metadata.width=dimensionBits&16383;metadata.height=dimensionBits>>16&16383}break;case"VP8L":br.seek(20);const losslessSignature=br.readScalar(1);if(losslessSignature!==47){throw new Error("Invalid VP8 lossless bitstream signature: "+losslessSignature)}{const dimensionBits=br.readScalar(4);metadata.width=(dimensionBits&16383)+1;metadata.height=(dimensionBits>>14&16383)+1}break;case"VP8X":br.seek(24);metadata.width=(br.readScalar(2)|br.readScalar(1)<<16)+1;metadata.height=(br.readScalar(2)|br.readScalar(1)<<16)+1;break;default:throw new Error("Invalid chunk format: "+chunkFormat)}}}class IcoParser extends SimpleImageParser{constructor(parent){super(parent,"ico",/\.ico$/i,8)}parseHeader(metadata,byteReader){byteReader.setByteOrder(ByteOrder.LITTLE_ENDIAN);const signature=byteReader.readString(4);if(signature!=="\0\0\0"){throw new Error("Invalid ICO signature: "+signature)}byteReader.seek(2);metadata.width=byteReader.readScalar(1);metadata.height=byteReader.readScalar(1)}}class MpegParser extends MetadataParser{constructor(parent){super(parent,"mpeg",/\.(mp4|m4v|m4a|mpe?g4?)$/i);this.mimeType="video/mpeg"}static readAtomSize(br,end){const pos=br.tell();if(end){br.validateRead(end-pos)}const size=br.readScalar(4,false,end);if(size<MpegParser.HEADER_SIZE){throw new Error("atom too short ("+size+") @"+pos)}if(end&&pos+size>end){throw new Error("atom too long ("+size+">"+(end-pos)+") @"+pos)}return size}static readAtomName(br,end){return br.readString(4,end).toLowerCase()}static createRootParser(metadata){function findParentAtom(atom,name){for(;;){if(!atom.parent){return null}atom=atom.parent;if(atom.name===name){return atom}}}function parseFtyp(br,atom){metadata.mpegBrand=br.readString(4,atom.end)}function parseMvhd(br,atom){const version=br.readScalar(4,false,atom.end);const offset=version===0?8:16;br.seek(offset,SeekOrigin.SEEK_CUR);const timescale=br.readScalar(4,false,atom.end);const duration=br.readScalar(4,false,atom.end);metadata.duration=duration/timescale}function parseHdlr(br,atom){br.seek(8,SeekOrigin.SEEK_CUR);const type=br.readString(4,atom.end);const track=findParentAtom(atom,"trak");if(track){track.trackType=type}}function parseStsd(br,atom){const track=findParentAtom(atom,"trak");if(track&&track.trackType==="vide"){br.seek(40,SeekOrigin.SEEK_CUR);metadata.width=br.readScalar(2,false,atom.end);metadata.height=br.readScalar(2,false,atom.end)}}function parseDataString(name,br,atom){br.seek(8,SeekOrigin.SEEK_CUR);metadata[name]=br.readString(atom.end-br.tell(),atom.end)}function parseCovr(br,atom){br.seek(8,SeekOrigin.SEEK_CUR);metadata.thumbnailURL=br.readImage(atom.end-br.tell(),atom.end)}const parseMeta={ilst:{"©nam":{data:parseDataString.bind(null,"title")},"©alb":{data:parseDataString.bind(null,"album")},"©art":{data:parseDataString.bind(null,"artist")},covr:{data:parseCovr}},versioned:true};return{ftyp:parseFtyp,moov:{mvhd:parseMvhd,trak:{mdia:{hdlr:parseHdlr,minf:{stbl:{stsd:parseStsd}}},meta:parseMeta},udta:{meta:parseMeta},meta:parseMeta},meta:parseMeta}}parse(file,metadata,callback,onError){const rootParser=MpegParser.createRootParser(metadata);this.requestRead(rootParser,file,0,MpegParser.HEADER_SIZE,null,onError,callback.bind(null,metadata))}applyParser(parser,br,atom,filePos){if(this.verbose){let path=atom.name;for(let p=atom.parent;p&&p.name;p=p.parent){path=p.name+"."+path}let action;if(!parser){action="skipping "}else if(parser instanceof Function){action="parsing  "}else{action="recursing"}const start=atom.start-MpegParser.HEADER_SIZE;this.vlog(path+": "+"@"+(filePos+start)+":"+(atom.end-start),action)}if(parser instanceof Function){br.pushSeek(atom.start);parser(br,atom);br.popSeek()}else if(parser instanceof Object){if(parser.versioned){atom.start+=4}this.parseMpegAtomsInRange(parser,br,atom,filePos)}}parseMpegAtomsInRange(parser,br,parentAtom,filePos){let count=0;for(let offset=parentAtom.start;offset!==parentAtom.end;){if(count++>100){throw new Error("too many child atoms in "+parentAtom.name+" @"+offset)}br.seek(offset);const size=MpegParser.readAtomSize(br,parentAtom.end);const name=MpegParser.readAtomName(br,parentAtom.end);this.applyParser(parser[name],br,{start:offset+MpegParser.HEADER_SIZE,end:offset+size,name:name,parent:parentAtom},filePos);offset+=size}}requestRead(rootParser,file,filePos,size,name,onError,onSuccess){const self=this;const reader=new FileReader;reader.onerror=onError;reader.onload=_event=>{self.processTopLevelAtom(reader.result,rootParser,file,filePos,size,name,onError,onSuccess)};this.vlog("reading @"+filePos+":"+size);reader.readAsArrayBuffer(file.slice(filePos,filePos+size))}processTopLevelAtom(buf,rootParser,file,filePos,size,name,onError,onSuccess){try{const br=new ByteReader(buf);const atomEnd=size-MpegParser.HEADER_SIZE;const bufLength=buf.byteLength;if(bufLength!==atomEnd&&bufLength!==size){throw new Error("Read failure @"+filePos+", "+"requested "+size+", read "+bufLength)}if(name){this.applyParser(rootParser[name],br,{start:0,end:atomEnd,name:name},filePos)}filePos+=bufLength;if(bufLength===size){br.seek(-MpegParser.HEADER_SIZE,SeekOrigin.SEEK_END);let nextSize=MpegParser.readAtomSize(br);const nextName=MpegParser.readAtomName(br);if(!rootParser[nextName]){filePos+=nextSize-MpegParser.HEADER_SIZE;nextSize=MpegParser.HEADER_SIZE}this.requestRead(rootParser,file,filePos,nextSize,nextName,onError,onSuccess)}else{this.vlog("EOF @"+filePos);onSuccess()}}catch(e){onError(e.toString())}}static{this.HEADER_SIZE=8}}function isFileEntry(entry){return entry.isFile}class MetadataDispatcher{constructor(port_){this.port_=port_;this.verbose=false;this.messageHandlers_={init:this.init_.bind(this),request:this.request_.bind(this)};this.port_.onmessage=this.onMessage.bind(this);const patterns=[];this.parserInstances_=[];const parserClasses=[BmpParser,ExifParser,GifParser,IcoParser,Id3Parser,MpegParser,PngParser,WebpParser];for(const parserClass of parserClasses){const parser=new parserClass(this);this.parserInstances_.push(parser);patterns.push(parser.urlFilter.source)}this.parserRegexp_=new RegExp("("+patterns.join("|")+")","i")}init_(){this.postMessage("initialized",[this.parserRegexp_]);this.vlog("initialized with URL filter "+this.parserRegexp_)}request_(fileURL){try{this.processOneFile(fileURL,(metadata=>{this.postMessage("result",[fileURL,metadata])}))}catch(ex){this.error(fileURL,ex)}}error(...args){this.postMessage("error",args)}log(...args){this.postMessage("log",args)}vlog(...args){if(this.verbose){this.log(...args)}}postMessage(verb,args){this.port_.postMessage({verb:verb,arguments:args})}onMessage(event){const data=event.data;const handler=this.messageHandlers_[data.verb];if(handler instanceof Function){handler.apply(this,data.arguments)}else{this.log("Unknown message from client: "+data.verb,data)}}detectFormat_(fileURL){for(const parser of this.parserInstances_){if(fileURL.match(parser.urlFilter)){return parser}}return null}async processOneFile(fileURL,callback){const parser=this.detectFormat_(fileURL);if(!parser){this.error(fileURL,"detectFormat","unsupported format");return}const metadata=parser.createDefaultMetadata();const entry=await new Promise(((resolve,reject)=>globalThis.webkitResolveLocalFileSystemURL(fileURL,resolve,reject)));if(!isFileEntry(entry)){this.error(fileURL,"getEntry","url does not refer a file",metadata);return}const file=await new Promise(entry.file.bind(entry));metadata.fileSize=file.size;try{parser.parse(file,metadata,callback,(error=>this.error(fileURL,"parseContent",error)))}catch(e){this.error(fileURL,"parseContent",e.stack)}}}const global=self;if(global.constructor.name==="SharedWorkerGlobalScope"){global.addEventListener("connect",(e=>{const port=e.ports[0];new MetadataDispatcher(port);port.start()}))}else{new MetadataDispatcher(global)}